/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "Muscle4Worker.h"
#include "Muscle4TaskLocalStorage.h"

#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/DelegateEditors.h>
#include <core_api/Log.h>

//#define _CRTDBG_MAP_ALLOC
//#include <crtdbg.h>
//#include <stdlib.h>

/* TRANSLATOR GB2::LocalWorkflow::Muscle4Worker */

namespace GB2 {
namespace LocalWorkflow {

static const QString IN_PORT("in");
static const QString OUT_PORT("out");

const QString Muscle4WorkerFactory::ACTOR_ID("muscle4.align");
const QString MODE_ATTR("mode");
const QString STABLE_ATTR("fix-order");

static LogCategory log(ULOG_CAT_WD);

void Muscle4WorkerFactory::init() {
    QList<PortDescriptor*> p; QList<Attribute*> a;
    Descriptor ind(IN_PORT, Muscle4Worker::tr("Input MSA"), Muscle4Worker::tr("Multiple sequence alignment to be processed."));
    Descriptor oud(OUT_PORT, Muscle4Worker::tr("Multiple sequence alignment"), Muscle4Worker::tr("Result of alignment."));
    p << new PortDescriptor(ind, BioDataTypes::MULTIPLE_ALIGNMENT_TYPE(), true /*input*/);
    p << new PortDescriptor(oud, BioDataTypes::MULTIPLE_ALIGNMENT_TYPE(), false /*input*/, true /*multi*/);
    Descriptor mod(MODE_ATTR, Muscle4Worker::tr("Mode"), 
        Muscle4Worker::tr("Selector of preset configurations, that give you the choice of optimizing accuracy, speed, or some compromise between the two. The default favors accuracy."));
    Descriptor sd(STABLE_ATTR, Muscle4Worker::tr("Stable order"), 
        Muscle4Worker::tr("Do not rearrange aligned sequences (-stable switch of MUSCLE). "
        "<p>Otherwise, MUSCLE re-arranges sequences so that similar sequences are adjacent in the output file. This makes the alignment easier to evaluate by eye. "));
    a << new Attribute(mod, CoreDataTypes::NUM_TYPE(), false, 0);
    a << new Attribute(sd, CoreDataTypes::BOOL_TYPE(), false, true);
    
	Descriptor desc(ACTOR_ID, GB2::LocalWorkflow::Muscle4Worker::tr("MUSCLE 4 multiple alignment"), 
        Muscle4Worker::tr("MUSCLE is public domain multiple alignment software for protein and nucleotide sequences."
        "<p><dfn>MUSCLE stands for MUltiple Sequence Comparison by Log-Expectation.</dfn></p>"));
    ActorPrototype* proto = new BusActorPrototype(desc, p, a);

    QMap<QString, PropertyDelegate*> delegates;    

    {
        QVariantMap vm; 
       // vm[DefaultModePreset().name] = 0;
       // vm[LargeModePreset().name] = 1;
       // vm[RefineModePreset().name] = 2;
        delegates[MODE_ATTR] = new ComboBoxDelegate(vm);
    }

    proto->setEditor(new DelegateEditor(delegates));
    proto->setPrompter(new Muscle4Prompter());
    proto->setIconPath(":umuscle/images/muscle_16.png");
    WorkflowEnv::getProtoRegistry()->registerProto(BioActorLibrary::CATEGORY_ALIGNMENT(), proto);

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new Muscle4WorkerFactory());
}

QString Muscle4Prompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(IN_PORT));
    Actor* producer = input->getProducer(IN_PORT);
    QString producerName = producer ? tr(" from %1").arg(producer->getLabel()) : "";
    QString preset;
   // switch (getParameter(MODE_ATTR).toInt()) {
   //     case 0: preset = DefaultModePreset().name; break;
   //     case 1: preset = LargeModePreset().name; break;
   //     case 2: preset = RefineModePreset().name; break;
   // }

    QString doc = tr("For each MSA<u>%1</u>, build the alignment using <u>\"%2\" preset</u> and send it to output.")
        .arg(producerName).arg(preset);

    return doc;
}

void Muscle4Worker::init() {
    input = ports.value(IN_PORT);
    output = ports.value(OUT_PORT);
//    switch (actor->getParameter(MODE_ATTR)->value.toInt()) {
//        case 0: DefaultModePreset().apply(cfg); break;
//        case 1: LargeModePreset().apply(cfg); break;
//        case 2: RefineModePreset().apply(cfg); break;
//    }
    cfg.stableMode = actor->getParameter(STABLE_ATTR)->value.toBool();

}

bool Muscle4Worker::isReady() {
    return (input && input->hasMessage());
}

Task* Muscle4Worker::tick() {
    MAlignment msa = input->get().getData().value<MAlignment>();
    Task* t = new Muscle4Task(msa, cfg);
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

void Muscle4Worker::sl_taskFinished() {
    Muscle4Task* t = qobject_cast<Muscle4Task*>(sender());
    if (t->getState() != Task::State_Finished) return;
    assert(t->resultMA.isNormalized());
    QVariant v = qVariantFromValue<MAlignment>(t->resultMA);
    output->put(Message(BioDataTypes::MULTIPLE_ALIGNMENT_TYPE(), v));
    if (input->isEnded()) {
        output->setEnded();
    }
    log.info(tr("Aligned %1 with MUSCLE").arg(t->resultMA.getName()));
}

bool Muscle4Worker::isDone() {
    return !input || input->isEnded();
}

} //namespace LocalWorkflow
} //namespace GB2
