/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_api/Counter.h>
#include <document_format/PDBFormat.h>
#include <BALL/STRUCTURE/triangulatedSES.h>

#include "SES.h"

namespace GB2
{

// SolventExcludedSurface

SolventExcludedSurface::SolventExcludedSurface()
{
    GCOUNTER( cvar, tvar, "SolventExcludedSurface" );
}

void SolventExcludedSurface::calculate(const QList<SharedAtom>& atoms)
{
    BALL::Surface surface;
    {
        std::vector<BALL::TSphere3<double> > spheres;
        foreach(const SharedAtom a, atoms)
        {
            Vector3D coord=a->coord3d;
            double radius=PDBFormat::getAtomCovalentRadius(a->atomicNumber)+TOLERANCE;
            spheres.push_back(BALL::TSphere3<double>(BALL::TVector3<double>(coord.x,coord.y,coord.z),radius));
        }
        double probeRadius=1.4;
        for(int attempt=0;attempt < 10;attempt++)
        {
            BALL::ReducedSurface reducedSurface(spheres,probeRadius);
            reducedSurface.compute();
            BALL::SolventExcludedSurface solventExcludedSurface(&reducedSurface);
            solventExcludedSurface.compute();
            if(solventExcludedSurface.check())
            {
                BALL::TriangulatedSES triangulatedSES(&solventExcludedSurface,6);
                triangulatedSES.compute();
                triangulatedSES.exportSurface(surface);
                break;
            }
            probeRadius+=0.01;
        }
    }
    for(int faceIndex=0;faceIndex < surface.getNumberOfTriangles();faceIndex++)
    {
        const BALL::Surface::Triangle &triangle=surface.getTriangle(faceIndex);
        Face face;
        for(int coordIndex=0;coordIndex < 3;coordIndex++)
        {
            face.v[0][coordIndex]=surface.getVertex(triangle.v1)[coordIndex];
            face.v[1][coordIndex]=surface.getVertex(triangle.v2)[coordIndex];
            face.v[2][coordIndex]=surface.getVertex(triangle.v3)[coordIndex];
            face.n[0][coordIndex]=surface.getNormal(triangle.v1)[coordIndex];
            face.n[1][coordIndex]=surface.getNormal(triangle.v2)[coordIndex];
            face.n[2][coordIndex]=surface.getNormal(triangle.v3)[coordIndex];
        }
        faces.append(face);
    }
}

void SolventExcludedSurface::calculate(const BioStruct3D& bioStruct)
{
    Vector3D center = bioStruct.getCenter();
    float radius = bioStruct.getMaxDistFromCenter();
    GeodesicSphere sphere(center, radius);
    vertices = sphere.getVertices();
}

// SolventExcludedSurfaceFactory

MolecularSurface *SolventExcludedSurfaceFactory::createInstance() const
{
    return new SolventExcludedSurface();
}

} // namespace
