/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DocWorkers.h"
#include "CoreLib.h"
#include "GenericReadWorker.h"

#include <workflow/WorkflowEnv.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_support/QVariantUtils.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <core_api/Log.h>
#include <core_api/DocumentModel.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/GObjectUtils.h>
#include <gobjects/DNASequenceObject.h>
#include <gobjects/MAlignmentObject.h>
#include <gobjects/TextObject.h>
#include <gobjects/GObjectRelationRoles.h>
#include <datatype/DNASequence.h>
#include <datatype/AnnotationData.h>

#include <document_format/EMBLGenbankAbstractDocument.h>

namespace GB2 {
namespace LocalWorkflow {

static LogCategory log(ULOG_CAT_WD);
static int ct = 0;

void TextWriter::data2doc(Document* doc, const QVariantMap& data) {
    QStringList list = data.value(CoreLib::DATA_PORT_ID).toStringList();
    QString text = list.join("\n");
    TextObject* to = qobject_cast<TextObject*>(GObjectUtils::selectOne(doc->getObjects(), GObjectTypes::TEXT, UOF_LoadedOnly));
    if (!to) {
        to = new TextObject(text, QString("Text %1").arg(++ct));
        doc->addObject(to);
    } else {
        to->setText(to->getText() + "\n" + text);
    }
}

void TextReader::doc2data(Document* doc) {
    log.info(tr("Reading text from %1").arg(doc->getURL()));
    foreach(GObject* go, GObjectUtils::select(doc->getObjects(), GObjectTypes::TEXT, UOF_LoadedOnly)) {
        TextObject* txtObject = qobject_cast<TextObject*>(go);
        assert(txtObject);
        QVariantMap m; 
        m[CoreLib::DATA_PORT_ID] = txtObject->getText();
        m[CoreLib::URL_SLOT_ID] = doc->getURL();
        cache << Message(mtype, m);
    }
}

void FastaWriter::data2doc(Document* doc, const QVariantMap& data) {
    DNASequence seq = qVariantValue<DNASequence>(data.value(BioActorLibrary::SEQ_SLOT_ID));
    QString sequenceName = data.value(CoreLib::FASTA_HEADER_SLOT_ID).toString();
    if (sequenceName.isEmpty()) {
        sequenceName = seq.getName();
    } else {
        seq.info.insert(DNAInfo::FASTA_HDR, sequenceName);
    }
    if (sequenceName.isEmpty()) {
        sequenceName = QString("unknown sequence %1").arg(doc->getObjects().size());
    }
    if (seq.alphabet && seq.length() != 0 && !doc->findGObjectByName(sequenceName)) {
        log.trace(QString("Adding seq [%1] to FASTA doc %2").arg(sequenceName).arg(doc->getURL()));
        doc->addObject(new DNASequenceObject(sequenceName, seq));
    }
}

void FastaReader::doc2data(Document* doc) {
    log.info(tr("Reading FASTA from %1").arg(doc->getURL()));
    foreach(GObject* go, GObjectUtils::select(doc->getObjects(), GObjectTypes::DNA_SEQUENCE, UOF_LoadedOnly)) {
        DNASequenceObject* dnaObject = qobject_cast<DNASequenceObject*>(go);
        assert(dnaObject);
        QVariantMap qm;
        qm[BioActorLibrary::SEQ_SLOT_ID] = qVariantFromValue<DNASequence>(dnaObject->getDNASequence());
        QVariant hdr = dnaObject->getDNASequence().info.value(DNAInfo::FASTA_HDR);
        if (hdr.isNull()) {
            QString name = dnaObject->getDNASequence().getName();
            if (name.isEmpty()) {
                name = dnaObject->getGObjectName();
            }
            hdr = name;
        }

        qm[CoreLib::FASTA_HEADER_SLOT_ID] = hdr;
        qm[CoreLib::URL_SLOT_ID] = doc->getURL();
        cache << Message(mtype, qm);
        log.trace("Got FASTA seq: " + dnaObject->getGObjectName());
    }
}

void GenbankWriter::data2doc(Document* doc, const QVariantMap& data) {
    DNASequence seq = qVariantValue<DNASequence>(data.value(BioActorLibrary::SEQ_SLOT_ID));
    QString sequenceName = seq.getName(); //data.value(CoreLib::GENBANK_ACN_SLOT_ID).toString();
    QString annotationName;
    if (sequenceName.isEmpty()) {
        int num = doc->findGObjectByType(GObjectTypes::DNA_SEQUENCE).size();
        sequenceName = QString("unknown sequence %1").arg(num);
        annotationName = QString("unknown features %1").arg(num);
    } else {
        annotationName = sequenceName + " features";
    }
    
    QList<SharedAnnotationData> atl = QVariantUtils::var2ftl(data.value(BioActorLibrary::FEATURE_TABLE_SLOT_ID).toList());
    DNASequenceObject* dna = qobject_cast<DNASequenceObject*>(doc->findGObjectByName(sequenceName));
    if (!dna && !seq.isNull()) {
        doc->addObject(dna = new DNASequenceObject(sequenceName, seq));
        log.trace(QString("Adding seq [%1] to GB doc %2").arg(sequenceName).arg(doc->getURL()));
    }
    if (!atl.isEmpty()) {
        AnnotationTableObject* att = NULL;
        if (dna) {
            QList<GObject*> relAnns = GObjectUtils::findObjectsRelatedToObjectByRole(dna, GObjectTypes::ANNOTATION_TABLE, GObjectRelationRole::SEQUENCE, doc->getObjects(), UOF_LoadedOnly);
            att = relAnns.isEmpty() ? NULL : qobject_cast<AnnotationTableObject*>(relAnns.first());
        }
        if (!att) {
            doc->addObject(att = new AnnotationTableObject(annotationName));
            if (dna) {
                att->addObjectRelation(dna, GObjectRelationRole::SEQUENCE);
            }
            log.trace(QString("Adding features [%1] to GB doc %2").arg(annotationName).arg(doc->getURL()));
        }
        foreach(SharedAnnotationData sad, atl) {
            att->addAnnotation(new Annotation(sad), QString());
        }
    }
}

void GenbankReader::doc2data(Document* doc) {
    log.info(tr("Reading GENBANK data from %1").arg(doc->getURL()));
    foreach(GObject* go, GObjectUtils::select(doc->getObjects(), GObjectTypes::ANNOTATION_TABLE, UOF_LoadedOnly)) {
        AnnotationTableObject* att = qobject_cast<AnnotationTableObject*>(go);
        assert(att);
        QList<GObjectRelation> rlist = att->findRelatedObjectsByRole(GObjectRelationRole::SEQUENCE);
        DNASequenceObject* dna = NULL;
        if (!rlist.isEmpty()) {
            const GObjectRelation& r = rlist.first();
            if (r.ref.docUrl == doc->getURL()) {
                dna = qobject_cast<DNASequenceObject*>(doc->findGObjectByName(r.ref.objName));
            }
        }
        //TODO: 
        assert(dna);
        QVariant qdna = qVariantFromValue<DNASequence>(dna->getDNASequence());
        QList<SharedAnnotationData> l;
        foreach(Annotation* a, att->getAnnotations()) {
            l << a->data();
        }
        QVariantMap qm;
        qm[BioActorLibrary::SEQ_SLOT_ID] = qdna;
        qm[BioActorLibrary::FEATURE_TABLE_SLOT_ID] = qVariantFromValue<QList<SharedAnnotationData> >(l);
        //qm[CoreLib::GENBANK_ACN_SLOT_ID] = QVariant(dna->getGObjectName());
        qm[CoreLib::URL_SLOT_ID] = doc->getURL();
        cache << Message(mtype, qm);
    }
}

void MSAReader::doc2data(Document* doc) {
    log.info(tr("Reading MSA from %1 [%2]").arg(doc->getURL()).arg(doc->getDocumentFormatId()));
    foreach(GObject* go, GObjectUtils::select(doc->getObjects(), GObjectTypes::MULTIPLE_ALIGNMENT, UOF_LoadedOnly)) {
        MAlignmentObject* obj = qobject_cast<MAlignmentObject*>(go);
        assert(obj);
        const MAlignment& ma = obj->getMAlignment();
        assert(ma.isNormalized());
        QVariantMap m; 
        m[BioActorLibrary::MA_SLOT_ID] = qVariantFromValue<MAlignment>(ma);
        m[CoreLib::URL_SLOT_ID] = doc->getURL();
        cache << Message(mtype, m);
    }
}

void MSAWriter::data2doc(Document* doc, const QVariantMap& data) {
    MAlignment ma = data.value(BioActorLibrary::MA_SLOT_ID).value<MAlignment>();
    if (!ma.isNormalized()) {
        log.error(tr("Empty alignment passed for writing to %1").arg(doc->getURL()));
        return; //FIXME
    }
    if (ma.getName().isEmpty()) {
        ma.setName( QString(MA_OBJECT_NAME + "_%1").arg(ct++) );
    }
    doc->addObject(new MAlignmentObject(ma));
}


Worker* DataWorkerFactory::createWorker(Actor* a) {
    BaseWorker* w = NULL;
    if (CoreLib::READ_TEXT_ACTOR == a->getProto()->getId()) {
        TextReader* t = new TextReader(a);
        w = t;
    } 
    else if (CoreLib::READ_FASTA_ACTOR == a->getProto()->getId()) {
        w = new FastaReader(a);
    }
    else if (CoreLib::WRITE_TEXT_ACTOR == a->getProto()->getId()) {
        w = new TextWriter(a);
    } 
    else if (CoreLib::WRITE_FASTA_ACTOR == a->getProto()->getId()) {
        w = new FastaWriter(a);
    }
    else if (CoreLib::READ_GENBANK_ACTOR == a->getProto()->getId()) {
        w = new GenbankReader(a);
    }
    else if (CoreLib::WRITE_GENBANK_ACTOR == a->getProto()->getId()) {
        w = new GenbankWriter(a);
    }
    else if (CoreLib::READ_CLUSTAL_ACTOR == a->getProto()->getId()) {
        w = new MSAReader(a, BaseDocumentFormats::CLUSTAL_ALN);
    }
    else if (CoreLib::WRITE_CLUSTAL_ACTOR == a->getProto()->getId()) {
        w = new MSAWriter(a, BaseDocumentFormats::CLUSTAL_ALN);
    }
    else if (CoreLib::READ_STOCKHOLM_ACTOR == a->getProto()->getId()) {
        w = new MSAReader(a, BaseDocumentFormats::STOCKHOLM);
    }
    else if (CoreLib::WRITE_STOCKHOLM_ACTOR == a->getProto()->getId()) {
        w = new MSAWriter(a, BaseDocumentFormats::STOCKHOLM);
    }
    else if (CoreLib::GENERIC_MA_ACTOR == a->getProto()->getId()) {
        w = new GenericMSAReader(a);
    }
    else if (CoreLib::GENERIC_SEQ_ACTOR == a->getProto()->getId()) {
        w = new GenericSeqReader(a);
    }

    return w;    
}

void DataWorkerFactory::init() {
    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new DataWorkerFactory(CoreLib::READ_FASTA_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLib::WRITE_FASTA_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLib::READ_GENBANK_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLib::WRITE_GENBANK_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLib::READ_TEXT_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLib::WRITE_TEXT_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLib::GENERIC_SEQ_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLib::GENERIC_MA_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLib::READ_CLUSTAL_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLib::WRITE_CLUSTAL_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLib::READ_STOCKHOLM_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLib::WRITE_STOCKHOLM_ACTOR));
}

} // Workflow namespace
} // GB2 namespace
