/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DNAExportTask.h"

#include <core_api/DNAAlphabet.h>
#include <core_api/DocumentModel.h>
#include <core_api/DocumentFormats.h>
#include <core_api/IOAdapter.h>
#include <core_api/AppContext.h>
#include <core_api/DNATranslation.h>
#include <core_api/Counter.h>
#include <util_algorithm/MSAUtils.h>
#include <util_text/TextUtils.h>

#include <gobjects/DNASequenceObject.h>
#include <gobjects/MAlignmentObject.h>
#include <memory>

namespace GB2 {

DNAExportSequenceTask::DNAExportSequenceTask(const DNAExportTaskSettings& s) 
: Task("", TaskFlag_None), config(s)
{
    GCOUNTER( cvar, tvar, "DNAExportSequenceTask" );
    setTaskName(tr("Export sequence to '%1'").arg(QFileInfo(s.fileName).fileName()));
    setVerboseLogMode(true);

    assert(config.names.size()!=0);
    assert(config.names.size() == config.sequences.size());
    assert(config.names.size() == config.alphabets.size());
}

void DNAExportSequenceTask::run() {
    DocumentFormatRegistry* r = AppContext::getDocumentFormatRegistry();
    DocumentFormat* f = r->getFormatById(BaseDocumentFormats::PLAIN_FASTA);
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(config.fileName));
    std::auto_ptr<Document> doc(f->createNewDocument(iof, config.fileName));
    
    //prepare complement if needed
    QStringList             namesPre;
    QList<DNAAlphabet*>     alphabetsPre; 
    QList<QByteArray>       sequencesPre;
    QList<DNATranslation*>  aminoTTPre;

    for (int i=0; i < config.names.size(); i++) {
        if (config.strand == TriState_Yes || config.strand == TriState_Unknown) {
            namesPre.append(config.names[i]);
            alphabetsPre.append(config.alphabets[i]);
            sequencesPre.append(config.sequences[i]);
            aminoTTPre.append(config.aminoTranslations[i]);
        } 
        if (config.strand == TriState_No || config.strand == TriState_Unknown) { 
            DNATranslation* complTT = config.complTranslations[i];
            QByteArray seq;
            if (complTT == NULL) {
                stateInfo.setError( tr("compl_tt_not_found") );
                return;
            }

            char* data = config.sequences[i].data();
            int len = config.sequences[i].length();
            complTT->translate(data, len);
            TextUtils::reverse(data, len);
            namesPre.append(config.names[i] + "|rev-compl"); //TODO: check if unique!
            alphabetsPre.append(config.alphabets[i]);
            sequencesPre.append(config.sequences[i]);
            aminoTTPre.append(config.aminoTranslations[i]);
            config.sequences[i].clear(); //release resources
        }
    }

    //translate data if needed
    QStringList             names;
    QList<DNAAlphabet*>     alphabets; 
    QList<QByteArray>       sequences;
    for (int i=0; i < namesPre.size(); i++) {
        DNATranslation* aminoTT = aminoTTPre[i];
        if (aminoTT != NULL) {
            for (int j=0, nStrands = config.allAminoStrands ? 3 : 1; j < nStrands; j++) {
                QByteArray seq = sequencesPre[i];
                int len = (seq.length() - j)/3;
                QByteArray res(len, '\0');
                if (res.isNull() && len != 0) {
                    stateInfo.setError( tr("memory_overflow_error") );
                    return;
                }
                assert(aminoTT->isThree2One());
                aminoTT->translate(seq.constData() + j, seq.length() - j, res.data(), res.length());

                sequences.append(res);
                names.append(namesPre[i] + QString("|transl") + (nStrands == 1 ? QString("") : " " + QString::number(j))); //TODO: check if unique!
                alphabets.append(aminoTT->getDstAlphabet());
            }
        } else {
            names.append(namesPre[i]);
            alphabets.append(alphabetsPre[i]);
            sequences.append(sequencesPre[i]);
        }
        sequencesPre[i].clear(); //release resources
    }

    //process merge property and construct the object
    int n = names.size();
    assert(n > 0);
    if (config.merge) {
        int size = 0;
        for (int i=0; i<n; i++) {
            const QByteArray& seq = sequences[i];
            size += seq.size();
            if (i != n-1) {
                size+=config.mergeGap;
            }
        }
        QByteArray mergedSequence;
        mergedSequence.reserve(size);
        DNAAlphabet* al = alphabets[0];
        const QString& name = names[0];
        QByteArray gapSequence(config.mergeGap, al->getDefaultSymbol());
        for (int i=0; i < n; i++) {
            mergedSequence+=sequences[i];
            sequences[i].clear(); //release resource
            if (i != n-1) {
                mergedSequence+=gapSequence;
            }
        }
        doc->addObject(new DNASequenceObject(name, DNASequence(mergedSequence, al)));
    } else {
        for (int i=0; i < names.size(); i++) {
            DNAAlphabet* al = alphabets[i];
            const QString& name = names[i];
            const QByteArray& seq = sequences[i];
            doc->addObject(new DNASequenceObject(name, DNASequence(seq, al)));
        }
    }
    //store the document
    f->storeDocument(doc.get(), stateInfo);
}


DNAExportAlignmentTask::DNAExportAlignmentTask(const MAlignment& _ma, const QString& _fileName)
: Task("", TaskFlag_None), ma(_ma), fileName(_fileName)
{
    setTaskName(tr("Export alignment to '%1'").arg(QFileInfo(fileName).fileName()));
    setVerboseLogMode(true);

    assert(!ma.isEmpty());
}

void DNAExportAlignmentTask::run() {
    DocumentFormatRegistry* r = AppContext::getDocumentFormatRegistry();
    DocumentFormat* f = r->getFormatById(BaseDocumentFormats::CLUSTAL_ALN);
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(fileName));
    std::auto_ptr<Document> doc(f->createNewDocument(iof, fileName));
    assert(ma.isNormalized());
    doc->addObject(new MAlignmentObject(ma));
    f->storeDocument(doc.get(), stateInfo);
}


//////////////////////////////////////////////////////////////////////////
// export ali 2 sequence

ExportAlignment2Sequence::ExportAlignment2Sequence(const MAlignment& _ma, const QString& _url, bool _trimAli) 
: Task(tr("Export alignment to FASTA: %1").arg(_url), TaskFlag_None), ma(_ma), fastaURL(_url), trimAli(_trimAli)
{
    setVerboseLogMode(true);
}

void ExportAlignment2Sequence::run() {
    DocumentFormatRegistry* r = AppContext::getDocumentFormatRegistry();
    DocumentFormat* f = r->getFormatById(BaseDocumentFormats::PLAIN_FASTA);
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(fastaURL));
    std::auto_ptr<Document> doc(f->createNewDocument(iof, fastaURL));

    QList<DNASequence> lst = MSAUtils::ma2seq(ma, trimAli);
    QSet<QString> usedNames;
    foreach(const DNASequence& s, lst) {
        QString name = s.getName();
        if (usedNames.contains(name)) {
            name = TextUtils::variate(name, " ", usedNames, false, 1);
        }
        doc->addObject(new DNASequenceObject(name, s));
        usedNames.insert(name);
    }
    f->storeDocument(doc.get(), stateInfo);
}

}//namespace

