/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SchemaSerializer.h"
#include "QVariantUtils.h"

#include <workflow/WorkflowRegistry.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/Schema.h>

#include <QtXml/qdom.h>

//Q_DECLARE_METATYPE(GB2::Workflow::CfgMap)

namespace GB2 {
namespace Workflow {

const QString SchemaSerializer::WORKFLOW_DOC = "GB2WORKFLOW";

static const QString WORKFLOW_EL = "workflow";
static const QString DOMAIN_EL = "workflow";
static const QString PROCESS_EL = "process";
static const QString ITERATION_EL = "iteration";
static const QString PORT_EL = "port";
static const QString PARAMS_EL = "params";
static const QString DATAFLOW_EL = "dataflow";
static const QString ID_ATTR = "id";
static const QString NAME_ATTR = "name";
static const QString TYPE_ATTR = "type";
static const QString SRC_PORT_ATTR = "sprt";
static const QString SRC_PROC_ATTR = "sprc";
static const QString DST_PORT_ATTR = "dprt";
static const QString DST_PROC_ATTR = "dprc";

QDomElement SchemaSerializer::saveActor(const Actor* proc, QDomElement& proj) {
    QDomElement docElement = proj.ownerDocument().createElement(PROCESS_EL);
    docElement.setAttribute(ID_ATTR, proc->getId());
    docElement.setAttribute(TYPE_ATTR, proc->getProto()->getId());
    docElement.setAttribute(NAME_ATTR, proc->getLabel());
    saveConfiguration(*proc, docElement);
    proj.appendChild(docElement);
    return docElement;
}

Actor* SchemaSerializer::readActor(const QDomElement& procElement) {

//    const int id = procElement.attribute(ID_ATTR).toInt();
//    Q_UNUSED(id);

    const QString name = procElement.attribute(TYPE_ATTR);
    ActorPrototype* proto = WorkflowEnv::getProtoRegistry()->getProto(name);
    if (!proto) {
        return NULL;
    }

    Actor* proc = proto->createInstance();
    if (proc) {
        readConfiguration(proc, procElement);
        proc->setLabel(procElement.attribute(NAME_ATTR));
    }
    return proc;
}

QDomElement SchemaSerializer::saveLink(const Link* link, QDomElement& proj) {
    QDomElement docElement = proj.ownerDocument().createElement(DATAFLOW_EL);
    docElement.setAttribute(SRC_PORT_ATTR, link->source()->getId());
    docElement.setAttribute(SRC_PROC_ATTR, link->source()->owner()->getId());
    docElement.setAttribute(DST_PORT_ATTR, link->destination()->getId());
    docElement.setAttribute(DST_PROC_ATTR, link->destination()->owner()->getId());
    proj.appendChild(docElement);
    return docElement;
}

QDomElement SchemaSerializer::savePort(const Port* port, QDomElement& owner) {
    QDomElement el = owner.ownerDocument().createElement(PORT_EL);
    el.setAttribute(ID_ATTR, port->getId());
    saveConfiguration(*port, el);
    owner.appendChild(el);
    return el;
}


void SchemaSerializer::schema2xml(const Schema& schema, QDomDocument& xml) {
    QDomElement projectElement = xml.createElement(WORKFLOW_EL);
    xml.appendChild(projectElement);
    foreach(Actor* a, schema.procs) {
        QDomElement el = saveActor(a, projectElement);
        foreach(Port* p, a->getPorts()) {
            savePort(p, el);
        }
    }
    foreach(Link* l, schema.flows) {
        saveLink(l, projectElement);
    }
    QDomElement el = xml.createElement(DOMAIN_EL);
    el.setAttribute(NAME_ATTR, schema.domain);
    projectElement.appendChild(el);
}

void SchemaSerializer::saveIterations(const QList<Iteration>& lst, QDomElement& proj) {
    foreach(const Iteration& it, lst) {
        QDomElement el = proj.ownerDocument().createElement(ITERATION_EL);
        el.setAttribute(ID_ATTR, it.id);
        el.setAttribute(NAME_ATTR, it.name);
        QVariant v = qVariantFromValue<CfgMap>(it.cfg);
        el.appendChild(proj.ownerDocument().createTextNode(QVariantUtils::var2String(v)));
        proj.appendChild(el);
    }
}
void SchemaSerializer::readIterations(QList<Iteration>& lst, const QDomElement& proj, const QMap<ActorId, ActorId>& remapping) {
    QDomNodeList paramNodes = proj.elementsByTagName(ITERATION_EL);
    for(int i=0; i<paramNodes.size(); i++) {
        QDomElement el = paramNodes.item(i).toElement();
        if (el.isNull()) continue;
        Iteration it(el.attribute(NAME_ATTR));
        if (el.hasAttribute(ID_ATTR)) {
            it.id = el.attribute(ID_ATTR).toInt();
        }
        QVariant var = QVariantUtils::String2Var(el.text());
        if (qVariantCanConvert<CfgMap>(var)) {
            it.cfg = var.value<CfgMap>();
        } 
        if (qVariantCanConvert<IterationCfg>(var)) {
            IterationCfg tmp = var.value<IterationCfg>();
            QMapIterator<IterationCfgKey, QVariant> tit(tmp);
            while (tit.hasNext())
            {
                tit.next();
                it.cfg[tit.key().first].insert(tit.key().second, tit.value());
            }
        }
        it.remap(remapping);
        lst.append(it);
    }
}

void SchemaSerializer::saveConfiguration(const Configuration& cfg, QDomElement& owner) {
    QVariantMap qm;
    foreach (Attribute* a, cfg.getParameters()) {
        qm[a->getId()] = a->value;
    }
    QDomElement el = owner.ownerDocument().createElement(PARAMS_EL);
    owner.appendChild(el);
    el.appendChild(owner.ownerDocument().createTextNode(QVariantUtils::map2String(qm)));
}

void SchemaSerializer::readConfiguration(Configuration* cfg, const QDomElement& owner) {
    QDomNodeList paramNodes = owner.elementsByTagName(PARAMS_EL);
    for(int i=0; i<paramNodes.size(); i++) {
        const QVariantMap& qm = QVariantUtils::string2Map(paramNodes.item(i).toElement().text(), true);
        QMapIterator<QString, QVariant> it(qm);
        while (it.hasNext()) {
            it.next();
            cfg->setParameter(it.key(), it.value());
        }
    }
}

static const QString META_EL = "info";

void SchemaSerializer::saveMeta(const Workflow::Metadata* meta, QDomElement& proj){
    QDomElement el = proj.ownerDocument().createElement(META_EL);
    proj.appendChild(el);
    el.setAttribute(NAME_ATTR, meta->name);
    el.appendChild(proj.ownerDocument().createCDATASection(meta->comment));
}

QString SchemaSerializer::readMeta(Workflow::Metadata* meta, const QDomElement& proj) {
    QDomElement el = proj.elementsByTagName(META_EL).item(0).toElement();
    meta->name = el.attribute(NAME_ATTR);
    meta->comment = el.text();
    return el.isNull() ? tr("no metadata") : QString();
}


QString SchemaSerializer::xml2schema(const QDomElement& projectElement, Schema* schema, QMap<ActorId, ActorId>& idmap, bool stopIfError) {
    QMap<ActorId, Actor*> procMap;

    QDomElement domainEl = projectElement.elementsByTagName(DOMAIN_EL).item(0).toElement();
    if (!domainEl.isNull()) {
        schema->domain = domainEl.attribute(NAME_ATTR);
    }

    WProtoRegistry* registry = WorkflowEnv::getProtoRegistry();

    QDomNodeList procNodes = projectElement.elementsByTagName(PROCESS_EL);
    for(int i=0; i<procNodes.size(); i++) {
        QDomElement procElement = procNodes.item(i).toElement();
        if (projectElement.isNull()) {
            continue;
        }

        const ActorId id = str2aid(procElement.attribute(ID_ATTR));
        if (stopIfError && procMap.contains(id)) {
            return tr("Invalid content: duplicate process %1").arg(id);
        }

        const QString name = procElement.attribute(TYPE_ATTR);
        ActorPrototype* proto = registry->getProto(name);
        if (!proto) {
            if (stopIfError) {
                return tr("Invalid content: unknown process type %1").arg(name);
            } else {
                continue;
            }
        }
        Actor* proc = proto->createInstance();
        readConfiguration(proc, procElement);
        proc->setLabel(procElement.attribute(NAME_ATTR));
        procMap[id] = proc;
        schema->procs.append(proc);

        //read port params
        QDomNodeList nl = procElement.elementsByTagName(PORT_EL);
        for(int j=0; j<nl.size(); j++) {
            QDomElement el = nl.item(j).toElement();
            if (el.isNull()) continue;
            QString eid = el.attribute(ID_ATTR);
            Port* p = proc->getPort(eid);
            if (!p) {
                if (stopIfError) {
                    return tr("Invalid content: unknown port %1 requested for %2").arg(eid).arg(name);
                } else {
                    continue;
                }
            }
            readConfiguration(p, el);
        }
    }

    QMapIterator<ActorId, Actor*> it(procMap);
    while(it.hasNext()) {
        it.next();
        idmap[it.key()] = it.value()->getId();
    }
    foreach(Actor* a, procMap) {
        a->remap(idmap);
    }

    QDomNodeList flowNodes = projectElement.elementsByTagName(DATAFLOW_EL);
    for(int i=0; i<flowNodes.size(); i++) {
        QDomElement flowElement = flowNodes.item(i).toElement();
        if (flowElement.isNull()) {
            continue;
        }
        const ActorId inId = str2aid(flowElement.attribute(DST_PROC_ATTR));
        const ActorId outId = str2aid(flowElement.attribute(SRC_PROC_ATTR));

        if (!procMap.contains(inId)) {
            if (stopIfError) {
                return tr("Invalid content: no such process %1 to bind").arg(inId);
            } else {
                continue;
            }
        }
        if (!procMap.contains(outId)) {
            if (stopIfError) {
                return tr("Invalid content: no such process %1 to bind").arg(outId);
            } else {
                continue;
            }
        }
        QString inP = flowElement.attribute(DST_PORT_ATTR);
        QString outP = flowElement.attribute(SRC_PORT_ATTR);

        Port* input = procMap[inId]->getPort(inP);
        Port* output = procMap[outId]->getPort(outP);
        if ((!input || !output || !input->canBind(output))) {
            if (stopIfError) {
                return tr("Invalid content: cannot bind [%1 : %2] to [%3 : %4]").
                    arg(inId).arg(inP).arg(outId).arg(outP);
            }
        } else {
            Link* l = new Link(input, output);
            schema->flows.append(l);
        }
    }
    return QString();
}

QMap<ActorId, ActorId> SchemaSerializer::deepCopy(const Schema& from, Schema* to) {
    QDomDocument xml(WORKFLOW_DOC);
    schema2xml(from, xml);
    QMap<ActorId, ActorId> idmap;
    xml2schema(xml.documentElement(), to, idmap);
    to->deepCopy = true;
    return idmap;
}

}//namespace Workflow
}//namespace GB2
