/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtCore/QFileInfo>
#include <QtCore/QFile>
#include <QtCore/QUrl>
#include <QtNetwork>

#include <core_api/AppContext.h>
#include <core_api/AppSettings.h>
#include <core_api/UserApplicationsSettings.h>
#include <core_api/DocumentModel.h>
#include <core_api/IOAdapter.h>
#include <core_api/ProjectModel.h>
#include <core_api/Counter.h>
#include <document_format/DocumentFormatUtils.h>
#include <util_tasks/AddDocumentTask.h>
#include <util_tasks/CopyDataTask.h>
#include <util_tasks/LoadDocumentTask.h>


#include "LoadRemoteDocumentTask.h"

namespace GB2 {

LoadRemoteDocumentTask::LoadRemoteDocumentTask( const QString& fileUrl, const DocumentFormatId& formatId) : 
    Task("LoadRemoteDocument", TaskFlags_NR_FOSCOE | TaskFlag_MinimizeSubtaskErrorText), formatId(formatId), copyDataTask(NULL), loadDocumentTask(NULL)
{
    url_from = fileUrl;
    GCOUNTER( cvar, tvar, "LoadRemoteDocumentTask" );
}

void LoadRemoteDocumentTask::prepare()
{
    QUrl url(url_from);
    
    QFileInfo fileInfo(url.path());
    fileName = fileInfo.fileName();
    
    if (fileName.isEmpty()) {
        stateInfo.setError("Incorrect key identifier!");
        return;
    }
    
    // Check if the file has already been downloaded
    RecentlyDownloadedCache* cache = AppContext::getRecentlyDownloadedCache();
    if (cache->contains(fileName)) {
        fullPath = cache->getFullPath(fileName);
        if (initLoadDocumentTask()) {
            addSubTask(loadDocumentTask);
        }
        return;   
    }
    
    QString path = AppContext::getAppSettings()->getUserAppsSettings()->getDownloadDirPath();
    QDir dir;
    if (!dir.exists(path)) {
        // Creating directory if it doesn't exist
        bool res = dir.mkdir(path);
        if (!res) {
            stateInfo.setError(QString("Cannot create directory %1").arg(path));
            return;
        }
    }

    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::HTTP_FILE);
    IOAdapterFactory * iow = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
    fullPath = path + "/" + fileName;
    QList<Task*> tasks;
    copyDataTask = new CopyDataTask(iof, url_from, iow, fullPath);
    addSubTask(copyDataTask);
}

Task::ReportResult LoadRemoteDocumentTask::report()
{
    return ReportResult_Finished;
}

QList<Task*> LoadRemoteDocumentTask::onSubTaskFinished( Task* subTask )
{
    QList<Task*> subTasks;
    if (subTask->hasErrors()) {
        return subTasks;
    }
    if (subTask == copyDataTask) {
        if (initLoadDocumentTask()) {
            AppContext::getRecentlyDownloadedCache()->append(fullPath);
            subTasks.append(loadDocumentTask);
        }
    } else if (subTask == loadDocumentTask) {
        Project* proj = AppContext::getProject();
        if (proj == NULL) {
            subTasks.append(  AppContext::getProjectLoader()->openProjectTask(fullPath, false));
        } else {
            Document* doc = loadDocumentTask->getDocument();
            DocumentFormat* format = AppContext::getDocumentFormatRegistry()->getFormatById(formatId);
            IOAdapterFactory * iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
            Document* clonedDoc = new Document(format, iof, fullPath);
            clonedDoc->loadFrom(doc); // doc was loaded in a separate thread -> clone all GObjects
            assert(!clonedDoc->isTreeItemModified());
            assert(clonedDoc->isLoaded());
            subTasks.append(new AddDocumentTask(clonedDoc));
            subTasks.append(new LoadUnloadedDocumentAndOpenViewTask(clonedDoc));
            
        }
    }
    return subTasks;
}

bool LoadRemoteDocumentTask::initLoadDocumentTask(  )
{
    Q_ASSERT(!fullPath.isEmpty());
    
    // Check if the document has been loaded 
    Project* proj = AppContext::getProject();
    if (proj != NULL) {
        Document* foundDoc = proj->findDocumentByURL(fullPath);
        if (foundDoc != NULL) {
            addSubTask(new LoadUnloadedDocumentAndOpenViewTask(foundDoc));
            return false;
        }
    }
    
    // Detect format
    if (formatId.isEmpty()) {
        QList<DocumentFormat*> formats = DocumentFormatUtils::detectFormat(fullPath);
        if (formats.isEmpty()) {
            stateInfo.setError("Unknown file format!");
            return false;
        } else {
            formatId = formats.first()->getFormatId();
        }
    }
    IOAdapterFactory * iow = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
    loadDocumentTask =  new LoadDocumentTask(formatId, fullPath, iow);
    
    return true;
}
////////////////////////////////////////////////////////////////////////////////////////////////////


RecentlyDownloadedCache::RecentlyDownloadedCache()
{
    QStringList fileNames = AppContext::getAppSettings()->getUserAppsSettings()->getRecentlyDownloadedFileNames();
    foreach (const QString& path, fileNames) {
        QFileInfo info(path);
        if (info.exists()) {
            append(path);
        }
    }
}

void RecentlyDownloadedCache::append( const QString& fileName )
{
    QFileInfo info(fileName);
    urlMap.insert(info.fileName(), fileName);
}

QString RecentlyDownloadedCache::getFullPath( const QString& fileName )
{
    Q_ASSERT(urlMap.contains(fileName));
    return urlMap.value(fileName);
}

RecentlyDownloadedCache::~RecentlyDownloadedCache()
{
    //TODO: cache depends on AppSettings! get rid of this dependency!
    QStringList fileNames = urlMap.values();
    AppSettings* settings = AppContext::getAppSettings();
    Q_ASSERT(settings != NULL);
    UserAppsSettings* us = settings->getUserAppsSettings();
    Q_ASSERT(us != NULL);
    us->setRecentlyDownloadedFileNames(fileNames);
}
} //namespace
