/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MSAEditorTasks.h"
#include "MSAEditor.h"
#include "MSAEditorFactory.h"
#include "MSAEditorState.h"

#include <core_api/Log.h>
#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/DocumentModel.h>

#include <gobjects/MAlignmentObject.h>
#include <gobjects/UnloadedObject.h>
#include <gobjects/GObjectTypes.h>


#include <QtCore/QSet>

namespace GB2 {

/* TRANSLATOR GB2::MSAEditor */
/* TRANSLATOR GB2::ObjectViewTask */

static LogCategory log(ULOG_CAT_MSA);

//////////////////////////////////////////////////////////////////////////
/// open new view

OpenMSAEditorTask::OpenMSAEditorTask(MAlignmentObject* _obj) 
: ObjectViewTask(MSAEditorFactory::ID), msaObject(_obj)
{
	assert(!msaObject.isNull());
}

OpenMSAEditorTask::OpenMSAEditorTask(UnloadedObject* _obj) 
: ObjectViewTask(MSAEditorFactory::ID), unloadedReference(_obj)
{
    assert(_obj->getLoadedObjectType() == GObjectTypes::MULTIPLE_ALIGNMENT);
    documentsToLoad.append(_obj->getDocument());
}

OpenMSAEditorTask::OpenMSAEditorTask(Document* doc) 
: ObjectViewTask(MSAEditorFactory::ID), msaObject(NULL)
{
	assert(!doc->isLoaded());
    documentsToLoad.append(doc);
}

void OpenMSAEditorTask::open() {
	if (stateInfo.hasErrors() || (msaObject.isNull() && documentsToLoad.isEmpty())) {
		return;
	}
	if (msaObject.isNull()) {
		Document* doc = documentsToLoad.first();
        QList<GObject*> objects;
        if (unloadedReference.isValid()) {
            GObject* obj = doc->findGObjectByName(unloadedReference.objName);
            if (obj!=NULL && obj->getGObjectType() == GObjectTypes::MULTIPLE_ALIGNMENT) {
                msaObject = qobject_cast<MAlignmentObject*>(obj);
            }
        } else {
		    QList<GObject*> objects = doc->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT, UOF_LoadedAndUnloaded);
            msaObject = objects.isEmpty() ? NULL : qobject_cast<MAlignmentObject*>(objects.first());
        }
        if (msaObject.isNull()) {
            stateInfo.setError(tr("Multiple alignment object not found"));
            return;
        }
	}
	viewName = GObjectViewUtils::genUniqueViewName(msaObject->getDocument(), msaObject);
	log.details(tr("openining_msa_editor_for_object_%1").arg(msaObject->getGObjectName()));

	MSAEditor* v = new MSAEditor(viewName, msaObject);
	GObjectViewWindow* w = new GObjectViewWindow(v, viewName, false);
	MWMDIManager* mdiManager = 	AppContext::getMainWindow()->getMDIManager();
	mdiManager->addMDIWindow(w);

}

//////////////////////////////////////////////////////////////////////////
// open view from state


OpenSavedMSAEditorTask::OpenSavedMSAEditorTask(const QString& viewName, const QVariantMap& stateData) 
: ObjectViewTask(MSAEditorFactory::ID, viewName, stateData)
{
	MSAEditorState state(stateData);
	GObjectReference ref = state.getMSAObject();
	Document* doc = AppContext::getProject()->findDocumentByURL(ref.docUrl);
	if (doc == NULL) {
		stateIsIllegal = true;
        stateInfo.setError(ObjectViewTask::tr("document_not_found_%1").arg(ref.docUrl));
		return;
	}
	if (!doc->isLoaded()) {
		documentsToLoad.append(doc);
	}
	
}

void OpenSavedMSAEditorTask::open() {
    if (stateInfo.hasErrors()) {
        return;
    }
    MSAEditorState state(stateData);
    GObjectReference ref = state.getMSAObject();
    Document* doc = AppContext::getProject()->findDocumentByURL(ref.docUrl);
    if (doc == NULL) {
        stateIsIllegal = true;
        stateInfo.setError(ObjectViewTask::tr("document_not_found_%1").arg(ref.docUrl));
        return;
    }
    GObject* obj = doc->findGObjectByName(ref.objName);
    if (obj == NULL || obj->getGObjectType() != GObjectTypes::MULTIPLE_ALIGNMENT) {
        stateIsIllegal = true;
        stateInfo.setError(tr("dna_object_not_found_%1").arg(ref.objName));
        return;
    }
    MAlignmentObject* msaObject = qobject_cast<MAlignmentObject*>(obj);
    assert(msaObject!=NULL);
	
    MSAEditor* v = new MSAEditor(viewName, msaObject);
    GObjectViewWindow* w = new GObjectViewWindow(v, viewName, true);
    MWMDIManager* mdiManager = 	AppContext::getMainWindow()->getMDIManager();
    mdiManager->addMDIWindow(w);
	
    updateRanges(stateData, v);
}

void OpenSavedMSAEditorTask::updateRanges(const QVariantMap& stateData, MSAEditor* ctx) {
    Q_UNUSED(ctx);
	MSAEditorState state(stateData);
	int x = state.getX();
	if (x >= 0) {
		//todo:
	}

	int y = state.getY();
	if (y>=0) {
		//todo:
	}
}


//////////////////////////////////////////////////////////////////////////
// update
UpdateMSAEditorTask::UpdateMSAEditorTask(GObjectView* v, const QString& stateName, const QVariantMap& stateData) 
: ObjectViewTask(v, stateName, stateData)
{
}

void UpdateMSAEditorTask::update() {
	if (view.isNull() || view->getFactoryId() != MSAEditorFactory::ID) {
		return; //view was closed;
	}

	MSAEditor* msaView = qobject_cast<MSAEditor*>(view.data());
    assert(msaView!=NULL);
	
	OpenSavedMSAEditorTask::updateRanges(stateData, msaView);
}

} // namespace
