/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MSAEditorSequenceArea.h"
#include "MSAEditor.h"
#include "MSAColorScheme.h"

#include <core_api/Log.h>
#include <core_api/DNAAlphabet.h>
#include <core_api/AppContext.h>
#include <core_api/Settings.h>
#include <datatype/MAlignment.h>
#include <gobjects/MAlignmentObject.h>
#include <util_gui/GUIUtils.h>
#include <util_gui/PositionSelector.h>
#include <util_text/TextUtils.h>


#include <QtGui/QPainter>
#include <QtGui/QMouseEvent>
#include <QtGui/QClipboard>
#include <QtGui/QApplication>
#include <QtGui/QDialog>
#include <QtGui/QFontDialog>

namespace GB2 {

/* TRANSLATOR GB2::MSAEditor */

static LogCategory log(ULOG_CAT_MSA);

#define MIN_FONT_SIZE 6
#define MAX_FONT_SIZE 24

#define SETTINGS_ROOT QString("msaeditor/")

#define SETTINGS_FONT_FAMILY    "font_family"
#define SETTINGS_FONT_SIZE      "font_size"
#define SETTINGS_FONT_ITALIC    "font_italic"
#define SETTINGS_FONT_BOLD      "font_bold"
#define SETTINGS_COLOR_NUCL     "color_nucl"
#define SETTINGS_COLOR_AMINO    "color_amino"

#define DEFAULT_FONT_FAMILY "Verdana" 
#define DEFAULT_FONT_SIZE 10

MSAEditorSequenceArea::MSAEditorSequenceArea(MSAEditorUI* _ui, QScrollBar* hb, QScrollBar* vb) 
: editor(_ui->editor), ui(_ui), shBar(hb), svBar(vb) 
{
    setFocusPolicy(Qt::WheelFocus);

    cachedView = new QPixmap();
    completeRedraw = true;

    setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::MinimumExpanding);
    setMinimumSize(100, 100);
    startPos = 0;
    startSeq = 0;
    highlightSelection = false;

    Settings* s = AppContext::getSettings();
    seqFont.setFamily(s->getValue(SETTINGS_ROOT + SETTINGS_FONT_FAMILY, DEFAULT_FONT_FAMILY).toString());
    seqFont.setPointSize(s->getValue(SETTINGS_ROOT + SETTINGS_FONT_SIZE, DEFAULT_FONT_SIZE).toInt());
    seqFont.setItalic(s->getValue(SETTINGS_ROOT + SETTINGS_FONT_ITALIC, false).toBool());
    seqFont.setBold(s->getValue(SETTINGS_ROOT + SETTINGS_FONT_BOLD, false).toBool());
    updateSeqFontMetrics();
    
    delSymAction = new QAction(tr("del_sym"), this);
    delSymAction->setShortcut(QKeySequence(Qt::Key_Delete));
    delSymAction->setShortcutContext(Qt::WidgetShortcut);
    connect(delSymAction, SIGNAL(triggered()), SLOT(sl_delSym()));
    
    delColAction = new QAction(tr("del_col"), this);
    delColAction->setShortcut(QKeySequence(Qt::SHIFT| Qt::Key_Delete));
    delColAction->setShortcutContext(Qt::WidgetShortcut);
    connect(delColAction, SIGNAL(triggered()), SLOT(sl_delCol()));
    
    insSymAction = new QAction(tr("ins_sym"), this);
    insSymAction->setShortcut(QKeySequence(Qt::Key_Space));
    insSymAction->setShortcutContext(Qt::WidgetShortcut);
    connect(insSymAction, SIGNAL(triggered()), SLOT(sl_insSym()));
    
    insColAction = new QAction(tr("ins_col"), this);
    insColAction->setShortcut(QKeySequence(Qt::SHIFT| Qt::Key_Space));
    insColAction->setShortcutContext(Qt::WidgetShortcut);
    connect(insColAction, SIGNAL(triggered()), SLOT(sl_insCol()));
    
    gotoAction = new QAction(QIcon(":core/images/goto.png"), tr("goto_pos"), this);
    gotoAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_G));
    insColAction->setShortcutContext(Qt::WindowShortcut);
    connect(gotoAction, SIGNAL(triggered()), SLOT(sl_goto()));

    takeScreenshotAction = new QAction(QIcon(":core/images/cam2.png"), tr("Capture scren"), this);
    connect(takeScreenshotAction, SIGNAL(triggered()), ui, SLOT(sl_saveScreenshot()));

    increaseFontAction = new QAction(QIcon(":core/images/zoom_in.png"), tr("Increase Font"), this);
    connect(increaseFontAction, SIGNAL(triggered()), SLOT(sl_zoomIn()));
    
    decreseFontAction = new QAction(QIcon(":core/images/zoom_out.png"), tr("Decrease Font"), this);
    connect(decreseFontAction, SIGNAL(triggered()), SLOT(sl_zoomOut()));
    
    changeFontAction = new QAction(QIcon(":core/images/font.png"), tr("Change Font"), this);
    connect(changeFontAction, SIGNAL(triggered()), SLOT(sl_changeFont()));

    resetFontAction = new QAction(QIcon(":core/images/font_default.png"), tr("Reset Font"), this);
    connect(resetFontAction, SIGNAL(triggered()), SLOT(sl_resetFont()));

    removeGapColumnsAction = new QAction(QIcon(":core/images/msaed_remove_columns_with_gaps.png"), tr("Remove columns with gaps"), this);
    connect(removeGapColumnsAction, SIGNAL(triggered()), SLOT(sl_removeColumnsWithGaps()));
    
    removeAllGapsAction = new QAction(QIcon(":core/images/msaed_remove_all_gaps.png"), tr("Remove all gaps"), this);
    connect(removeAllGapsAction, SIGNAL(triggered()), SLOT(sl_removeAllGaps()));
    
    connect(editor->getMSAObject(), SIGNAL(si_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)), 
        SLOT(sl_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)));

    connect(editor, SIGNAL(si_buildStaticMenu(GObjectView*, QMenu*)), SLOT(sl_buildStaticMenu(GObjectView*, QMenu*)));
    connect(editor, SIGNAL(si_buildStaticToolbar(GObjectView*, QToolBar*)), SLOT(sl_buildStaticToolbar(GObjectView*, QToolBar*)));
    connect(editor, SIGNAL(si_buildPopupMenu(GObjectView* , QMenu*)), SLOT(sl_buildContextMenu(GObjectView*, QMenu*)));
    connect(editor->getMSAObject(), SIGNAL(si_lockedStateChanged()), SLOT(sl_lockedStateChanged()));

    prepareColorSchemeMenuActions();
    updateActions();
}

MSAEditorSequenceArea::~MSAEditorSequenceArea() {
    delete cachedView;
}

void MSAEditorSequenceArea::prepareColorSchemeMenuActions() {
    Settings* s = AppContext::getSettings();
    MAlignmentObject* maObj = editor->getMSAObject();
    DNAAlphabetType atype = maObj->getMAlignment().alphabet->getType();
    MSAColorSchemeRegistry* csr = AppContext::getMSAColorSchemeRegistry();
    QString csid = atype == DNAAlphabet_AMINO ? 
            s->getValue(SETTINGS_ROOT + SETTINGS_COLOR_AMINO, MSAColorScheme::UGENE_AMINO).toString()
          : s->getValue(SETTINGS_ROOT + SETTINGS_COLOR_NUCL, MSAColorScheme::UGENE_NUCL).toString();

    MSAColorSchemeFactory* csf = csr->getMSAColorSchemeFactoryById(csid);
    if (csf == NULL) {
        csf = csr->getMSAColorSchemeFactoryById(atype == DNAAlphabet_AMINO ? MSAColorScheme::UGENE_AMINO : MSAColorScheme::UGENE_NUCL);
    }
    assert(csf!=NULL);
    colorScheme = csf->create(this, maObj);

    QList<MSAColorSchemeFactory*> factories= csr->getMSAColorSchemes(atype);
    foreach(MSAColorSchemeFactory* f, factories) {
        QAction* action = new QAction(f->getName(), this);
        action->setCheckable(true);
        action->setChecked(f == csf);
        action->setData(f->getId());
        connect(action, SIGNAL(triggered()), SLOT(sl_changeColorScheme()));
        colorSchemeMenuActions.append(action);
    }
}

void MSAEditorSequenceArea::sl_changeColorScheme() {
    QAction* a = qobject_cast<QAction*>(sender());
    QString id = a->data().toString();
    MSAColorSchemeFactory* f = AppContext::getMSAColorSchemeRegistry()->getMSAColorSchemeFactoryById(id);
    delete colorScheme;
    colorScheme = f->create(this, ui->editor->getMSAObject());
    foreach(QAction* action, colorSchemeMenuActions) {
        action->setChecked(action == a);
    }
    if (f->getAlphabetType() == DNAAlphabet_AMINO) {
        AppContext::getSettings()->setValue(SETTINGS_ROOT + SETTINGS_COLOR_AMINO, id);
    } else {
        AppContext::getSettings()->setValue(SETTINGS_ROOT + SETTINGS_COLOR_NUCL, id);
    }

    completeRedraw = true;
    update();
}

void MSAEditorSequenceArea::updateSeqFontMetrics() {
    QFontMetrics seqFM(seqFont);
    seqCharWidth = seqFM.width('W') * 5 / 4;
    seqCharHeight = seqFM.height() * 5 / 4;
}

void MSAEditorSequenceArea::updateActions() {
    bool readOnly = editor->getMSAObject()->isStateLocked();
    
    delColAction->setEnabled(!readOnly);
    delSymAction->setEnabled(!readOnly);
    insColAction->setEnabled(!readOnly);
    insSymAction->setEnabled(!readOnly);
    removeGapColumnsAction->setEnabled(!readOnly);
    removeAllGapsAction->setEnabled(!readOnly);

    increaseFontAction->setEnabled(seqFont.pointSize() < MAX_FONT_SIZE);
    decreseFontAction->setEnabled(seqFont.pointSize() > MIN_FONT_SIZE);
    assert(checkState());
}



void MSAEditorSequenceArea::paintEvent(QPaintEvent *e) {
    drawAll();
    QWidget::paintEvent(e);
}

void MSAEditorSequenceArea::drawAll() {
    QSize s = size();
    if (cachedView->size() != s) {
        assert(completeRedraw);
        delete cachedView;
        cachedView = new QPixmap(s);
    }
    if (completeRedraw) {
        QPainter pCached(cachedView);
        drawContent(pCached);
        completeRedraw = false;
    }
    QPainter p(this);
    p.drawPixmap(0, 0, *cachedView);
    drawCursor(p);
    drawFocus(p);
}

void MSAEditorSequenceArea::drawContent(QPainter& p) {
    p.fillRect(cachedView->rect(), Qt::white);
    p.setFont(seqFont);
    
    //for every sequence in msa starting from first visible
    //draw it starting from startPos
    int firstVisibleSeq = getFirstVisibleSequence();
    int lastVisibleSeq  = getLastVisibleSequence(true);
    int lastPos = getLastVisibleBase(true);
    int w = width();
    int h = height();
    const MAlignment& msa = editor->getMSAObject()->getMAlignment();
    for (int seq = firstVisibleSeq; seq <= lastVisibleSeq; seq++) {
        const MAlignmentItem& item = msa.alignedSeqs[seq];
        LRegion baseYRange = getSequenceYRange(seq, true);

        //draw horizontal grid
        //p.drawLine(0, r.startPos, width(), r.startPos); p.drawLine(0, r.endPos(), width(), r.endPos());

        for (int pos = startPos; pos <= lastPos; pos++) {
            LRegion baseXRange = getBaseXRange(pos, true);
            QRect cr(baseXRange.startPos, baseYRange.startPos, baseXRange.len+1, baseYRange.len);
            assert(cr.left() < w && cr.top() < h); Q_UNUSED(w); Q_UNUSED(h);
            char c = msa.charAt(seq, pos);
            QColor color = colorScheme->getColor(seq, pos);
            if (color.isValid()) {
                p.fillRect(cr, color);
            }
            p.drawText(cr, Qt::AlignCenter, QString(c));
        }
    }
}

void MSAEditorSequenceArea::drawCursor(QPainter& p) {
    if (!isVisible(cursorPos, true)) {
        return;
    }
    LRegion xRange = getBaseXRange(cursorPos.x(), true);
    LRegion yRange = getSequenceYRange(cursorPos.y(), true);

    QPen pen(highlightSelection || hasFocus()? Qt::black : Qt::gray);
    pen.setStyle(Qt::DashLine);
    pen.setWidth(highlightSelection ? 2 : 1);
    p.setPen(pen);
    p.drawRect(xRange.startPos, yRange.startPos, xRange.len, yRange.len);
}

void MSAEditorSequenceArea::drawFocus(QPainter& p) {
    if (hasFocus()) {
        p.setPen(QPen(Qt::black, 1, Qt::DotLine));
        p.drawRect(0, 0, width()-1, height()-1);
    }
}


bool MSAEditorSequenceArea::isPosInRange(int p) const {
    return p >= 0 && p < editor->getAlignmentLen();
}

bool MSAEditorSequenceArea::isSeqInRange(int s) const {
    return s >= 0 && s < editor->getNumSequences();
}


bool MSAEditorSequenceArea::isPosVisible(int pos, bool countClipped) const {
    if (pos < getFirstVisibleBase() || pos > getLastVisibleBase(countClipped)) {
        return false;
    }
    return true;
}

bool MSAEditorSequenceArea::isSeqVisible(int seq, bool countClipped) const {
    if (seq < getFirstVisibleSequence() || seq > getLastVisibleSequence(countClipped)) {
        return false;
    }
    return true;
}


void MSAEditorSequenceArea::setFirstVisibleBase(int pos) {
    Q_ASSERT(isPosInRange(pos));
    if (pos == startPos) {
        return;
    }
    QPoint prev(startPos, startSeq);
    int aliLen = editor->getAlignmentLen();
    int effectiveFirst = qMin(aliLen - countWidthForBases(false), pos);
    startPos = qMax(0, effectiveFirst);

    updateHScrollBar();

    QPoint p(startPos, startSeq);
    emit si_startChanged(p,  prev);

    completeRedraw = true;
    update();
}

void MSAEditorSequenceArea::setFirstVisibleSequence(int seq) {
    Q_ASSERT(isSeqInRange(seq));
    if (seq == startSeq) {
        return;
    }
    QPoint prev(startPos, startSeq);

    int nSeq = editor->getNumSequences();
    int effectiveFirst = qMin(nSeq - countHeightForSequences(false), seq);
    startSeq = qMax(0, effectiveFirst);

    updateVScrollBar();

    QPoint p(startPos, startSeq);
    emit si_startChanged(p, prev);

    completeRedraw = true;
    update();
}


void MSAEditorSequenceArea::resizeEvent(QResizeEvent *e) {
    completeRedraw = true;
    validateRanges();
    QWidget::resizeEvent(e);
}

void MSAEditorSequenceArea::validateRanges() {
    //check x dimension
    int aliLen = editor->getAlignmentLen();
    int visibleBaseCount = countWidthForBases(false);
    if (visibleBaseCount > aliLen) {
        setFirstVisibleBase(0);
    } else if (startPos + visibleBaseCount > aliLen) {
        setFirstVisibleBase(aliLen - visibleBaseCount);
    }
    assert(startPos >= 0);
    assert((aliLen >= startPos + visibleBaseCount) || aliLen < visibleBaseCount);
    updateHScrollBar();

    //check y dimension
    int nSeqs = editor->getNumSequences();
    int visibleSequenceCount = countHeightForSequences(false);
    if (visibleSequenceCount > nSeqs) {
        setFirstVisibleSequence(0);
    } else if (startSeq + visibleSequenceCount > nSeqs) {
        setFirstVisibleSequence(nSeqs - visibleSequenceCount);
    }
    assert(startSeq >= 0);
    assert((nSeqs >= startSeq + visibleSequenceCount) || nSeqs < visibleSequenceCount);
    updateVScrollBar();

}

void MSAEditorSequenceArea::sl_onHScrollMoved(int pos) {
    assert(pos >=0 && pos <= editor->getAlignmentLen() - getNumVisibleBases(false));    
    setFirstVisibleBase(pos);
}

void MSAEditorSequenceArea::sl_onVScrollMoved(int seq) {
    assert(seq >=0 && seq <= editor->getNumSequences() - getNumVisibleSequences(false));    
    setFirstVisibleSequence(seq);
}

void MSAEditorSequenceArea::updateHScrollBar() {
    shBar->disconnect(this);

    int numVisibleBases = getNumVisibleBases(false);
    int alignmentLen = editor->getAlignmentLen();
    Q_ASSERT(numVisibleBases <= alignmentLen);

    shBar->setMinimum(0);
    shBar->setMaximum(alignmentLen - numVisibleBases);
    shBar->setSliderPosition(getFirstVisibleBase());

    shBar->setSingleStep(1);
    shBar->setPageStep(numVisibleBases);

    shBar->setDisabled(numVisibleBases == alignmentLen);

    connect(shBar, SIGNAL(valueChanged(int)), SLOT(sl_onHScrollMoved(int)));
}

void MSAEditorSequenceArea::updateVScrollBar() {
    svBar->disconnect(this);

    int numVisibleSequences = getNumVisibleSequences(false);
    int nSeqs = editor->getNumSequences();
    Q_ASSERT(numVisibleSequences <= nSeqs);

    svBar->setMinimum(0);
    svBar->setMaximum(nSeqs - numVisibleSequences);
    svBar->setSliderPosition(getFirstVisibleSequence());

    svBar->setSingleStep(1);
    svBar->setPageStep(numVisibleSequences);

    svBar->setDisabled(numVisibleSequences == nSeqs);

    connect(svBar, SIGNAL(valueChanged(int)), SLOT(sl_onVScrollMoved(int)));
}


int MSAEditorSequenceArea::countWidthForBases(bool countClipped) const {
    int seqAreaWidth = width();
    int nVisible = seqAreaWidth / seqCharWidth + (countClipped && (seqAreaWidth % seqCharWidth != 0) ? 1 : 0);
    return nVisible;
}

int MSAEditorSequenceArea::countHeightForSequences(bool countClipped) const {
    int seqAreaHeight = height();
    int nVisible = seqAreaHeight / seqCharHeight + (countClipped && (seqAreaHeight % seqCharHeight != 0) ? 1 : 0);
    return nVisible;
}

int MSAEditorSequenceArea::getNumVisibleBases(bool countClipped) const {
    int lastVisible = getLastVisibleBase(countClipped);
    assert(lastVisible >= startPos || (!countClipped && lastVisible + 1 == startPos /*1 symbol is visible & clipped*/));
    assert(lastVisible < editor->getAlignmentLen());
    int res = lastVisible - startPos + 1;
    return res;
}

int MSAEditorSequenceArea::getLastVisibleBase(bool countClipped) const {
    int nVisible = countWidthForBases(countClipped);
    int alignLen = editor->getAlignmentLen();
    int res = qBound(0, startPos + nVisible - 1, alignLen - 1);
    return res;
}

int MSAEditorSequenceArea::getLastVisibleSequence(bool countClipped) const {
    int nVisible = countHeightForSequences(countClipped);
    int numSeqs = editor->getNumSequences();
    int res = qBound(0, startSeq + nVisible - 1, numSeqs - 1);
    return res;
}

int MSAEditorSequenceArea::getNumVisibleSequences(bool countClipped) const {
    int lastVisible =  getLastVisibleSequence(countClipped);
    assert(lastVisible >= startSeq);
    assert(lastVisible < editor->getNumSequences());
    int res = lastVisible - startSeq + 1;
    return res;
}

int MSAEditorSequenceArea::getSequenceNumByY(int y) {
    int seqOffs = y / seqCharHeight;
    int seq = startSeq + seqOffs;
    if (seq >= editor->getNumSequences()) {
        return -1; 
    }
    return seq;
}

LRegion MSAEditorSequenceArea::getBaseXRange(int pos, bool useVirtualCoords) const {
    LRegion res(seqCharWidth * (pos - startPos), seqCharWidth);
    if (!useVirtualCoords) {
        int w = width();
        res = res.intersect(LRegion(0, w));
    }
    return res;
}

LRegion MSAEditorSequenceArea::getSequenceYRange(int seq, bool useVirtualCoords) const {
    LRegion res(seqCharHeight* (seq - startSeq), seqCharHeight);
    if (!useVirtualCoords) {
        int h = height();
        res = res.intersect(LRegion(0, h));
    }
    return res;
}



void MSAEditorSequenceArea::mousePressEvent(QMouseEvent *e) {
    if (!hasFocus()) {
        setFocus();
    }

    if (e->button() != Qt::LeftButton) {
        QWidget::mousePressEvent(e);
        return;
    }
    QPoint p = coordToPos(e->pos());
    if (p.x()!=-1 && p.y()!=-1) {
        setCursorPos(p);
    }
    QWidget::mousePressEvent(e);
}

void MSAEditorSequenceArea::keyPressEvent(QKeyEvent *e) {
    if (!hasFocus()) {
        return;
    }
    int key = e->key();
    bool shift = e->modifiers().testFlag(Qt::ShiftModifier);
    bool ctrl = e->modifiers().testFlag(Qt::ControlModifier);
    if (ctrl && (key == Qt::Key_Left || key == Qt::Key_Right || key == Qt::Key_Up || key == Qt::Key_Down)) {
        //remap to page_up/page_down
        shift = key == Qt::Key_Up || key == Qt::Key_Down;
        key =  (key == Qt::Key_Up || key == Qt::Key_Left) ? Qt::Key_PageUp : Qt::Key_PageDown;
    }
    //part of these keys are assigned to actions -> so them never passed to keyPressEvent (action handling has higher priority)
    switch(key) {
        case Qt::Key_Left:
            moveCursor(-1, 0);    
            break;
        case Qt::Key_Right:
            moveCursor(1, 0);    
            break;
        case Qt::Key_Up:
            moveCursor(0, -1);    
            break;
        case Qt::Key_Down:
            moveCursor(0, 1);    
            break;
        case Qt::Key_Home:
            if (shift) { //scroll namelist
                setFirstVisibleSequence(0);
                setCursorPos(QPoint(cursorPos.x(), 0));
            } else { //scroll sequence
                setFirstVisibleBase(0);
                setCursorPos(QPoint(0, cursorPos.y()));
            }
            break;
        case Qt::Key_End:
            if (shift) { //scroll namelist
                int n = editor->getNumSequences() - 1;
                setFirstVisibleSequence(n);
                setCursorPos(QPoint(cursorPos.x(), n));
            } else { //scroll sequence
                int n = editor->getAlignmentLen() - 1;
                setFirstVisibleBase(n);
                setCursorPos(QPoint(n, cursorPos.y()));
            }
            break;
        case Qt::Key_PageUp:
            if (shift) { //scroll namelist
                int nVis = getNumVisibleSequences(false);
                int fp = qMax(0, getFirstVisibleSequence() - nVis);
                int cp = qMax(0, cursorPos.y() - nVis);
                setFirstVisibleSequence(fp);
                setCursorPos(QPoint(cursorPos.x(), cp));
            } else { //scroll sequence
                int nVis = getNumVisibleBases(false);
                int fp = qMax(0, getFirstVisibleBase() - nVis);
                int cp = qMax(0, cursorPos.x() - nVis);
                setFirstVisibleBase(fp);
                setCursorPos(QPoint(cp, cursorPos.y()));
            }
            break;
        case Qt::Key_PageDown:
            if (shift) { //scroll namelist
                int nVis = getNumVisibleSequences(false);
                int nSeq = editor->getNumSequences();
                int fp = qMin(nSeq-1, getFirstVisibleSequence() + nVis);
                int cp = qMin(nSeq-1, cursorPos.y() + nVis);
                setFirstVisibleSequence(fp);
                setCursorPos(QPoint(cursorPos.x(), cp));
            } else { //scroll sequence
                int nVis = getNumVisibleBases(false);
                int len = editor->getAlignmentLen();
                int fp  = qMin(len-1, getFirstVisibleBase() + nVis);
                int cp  = qMin(len-1, cursorPos.x() + nVis);
                setFirstVisibleBase(fp);
                setCursorPos(QPoint(cp, cursorPos.y()));
            }
            break;
        case Qt::Key_Delete:
            del(cursorPos, shift);
            break;
        case Qt::Key_Backspace:
            if (cursorPos.x() > 0) {
                del(QPoint(cursorPos.x()-1, cursorPos.y()), shift);
            }
            break;
        case Qt::Key_Insert:
        case Qt::Key_Space:
            //printf("key!\n");
            ins(cursorPos, shift);
            break;
    }
    QWidget::keyPressEvent(e);
}

void MSAEditorSequenceArea::focusInEvent(QFocusEvent* fe) {
    QWidget::focusInEvent(fe);
    update();
}

void MSAEditorSequenceArea::focusOutEvent(QFocusEvent* fe) {
    QWidget::focusOutEvent(fe);
    update();
}


void MSAEditorSequenceArea::moveCursor(int dx, int dy) {
    QPoint p = cursorPos + QPoint(dx, dy);
    if (!isInRange(p)) {
        return;
    }   
    if (!isVisible(p, false)) {
        if (isVisible(cursorPos, true)) {
            if (dx != 0) { 
                setFirstVisibleBase(startPos + dx);
            } 
            if (dy!=0) {
                setFirstVisibleSequence(getFirstVisibleSequence()+dy);
            }
        } else {
            setFirstVisibleBase(p.x());
            setFirstVisibleSequence(p.y());
        }
    }
    setCursorPos(p);
}


int MSAEditorSequenceArea::coordToPos(int x) const {
    int y = getSequenceYRange(getFirstVisibleSequence(), false).startPos;
    return coordToPos(QPoint(x, y)).x();
}

QPoint MSAEditorSequenceArea::coordToPos(const QPoint& coord) const {
    QPoint res(-1, -1);
    //Y: row
    for (int i=getFirstVisibleSequence(), n = getLastVisibleSequence(true); i<=n; i++) {
        LRegion r = getSequenceYRange(i, false);
        if (r.contains(coord.y())) {
            res.setY(i);
            break;
        }
    }
    
    //X: position in sequence
    for (int i=getFirstVisibleBase(), n = getLastVisibleBase(true); i<=n; i++) {
        LRegion r = getBaseXRange(i, false);
        if (r.contains(coord.x())) {
            res.setX(i);
            break;
        }
    }
    return res;
}

void MSAEditorSequenceArea::setCursorPos(const QPoint& p) {
    assert(isInRange(p));
    if (p == cursorPos) {
        return;
    }
    
    bool up = isVisible(cursorPos, true) || isVisible(p, true);
    QPoint prev = cursorPos;
    cursorPos = p;
    
    emit si_cursorMoved(cursorPos, prev);
    
    if (up) {
        update();
    }
    highlightSelection = false;
    updateActions();
}

void MSAEditorSequenceArea::ins(const QPoint& p, bool columnMode) {
    assert(isInRange(p));
    MAlignmentObject* maObj = editor->getMSAObject();
    if (maObj->isStateLocked()) {
        return;
    }
    if (columnMode) {
        maObj->insertGap(p.x(), 1);
    } else {
        maObj->insertGap(p.y(), p.x(), 1);
    }
}

void MSAEditorSequenceArea::del(const QPoint& p, bool columnMode) {
    assert(isInRange(p));
    MAlignmentObject* maObj = editor->getMSAObject();
    if (maObj->isStateLocked()) {
        return;
    }
    if (columnMode) {
        maObj->deleteGap(p.x(), 1);
    } else {
        maObj->deleteGap(p.y(), p.x(), 1);
    }
}

void MSAEditorSequenceArea::sl_alignmentChanged(const MAlignment&, const MAlignmentModInfo&) {
    int aliLen = editor->getAlignmentLen();
    int nSeq = editor->getNumSequences();
    
    //todo: set in one method!
    setFirstVisibleBase(qBound(0, startPos, aliLen-countWidthForBases(false)));
    setFirstVisibleSequence(qBound(0, startSeq, nSeq - countHeightForSequences(false)));

    setCursorPos(qMin(cursorPos.x(), aliLen-1), qMin(cursorPos.y(), nSeq-1));

    updateHScrollBar();
    updateVScrollBar();

    completeRedraw = true;
    update();
}

void MSAEditorSequenceArea::sl_buildStaticToolbar(GObjectView*, QToolBar* t) {
    t->addAction(gotoAction);
    t->addAction(removeGapColumnsAction);
    t->addAction(removeAllGapsAction);
    t->addAction(takeScreenshotAction);
    t->addAction(increaseFontAction);
    t->addAction(decreseFontAction);
    t->addAction(changeFontAction);
    t->addAction(resetFontAction);
    t->addAction(ui->getUndoAction());
    t->addAction(ui->getRedoAction());
}

void MSAEditorSequenceArea::sl_buildStaticMenu(GObjectView* v, QMenu* m) {
    Q_UNUSED(v);
    buildMenu(m);
}

void MSAEditorSequenceArea::sl_buildContextMenu(GObjectView* v, QMenu* m) {
    Q_UNUSED(v);
    buildMenu(m);
}

void MSAEditorSequenceArea::buildMenu(QMenu* m) {
    QAction* copyMenuAction = GUIUtils::findAction(m->actions(), MSAE_MENU_COPY);
    m->insertAction(copyMenuAction, gotoAction);

    QMenu* editMenu = GUIUtils::findSubMenu(m, MSAE_MENU_EDIT);
    assert(editMenu!=NULL);
    
    QList<QAction*> actions; actions << insSymAction << insColAction 
        << delSymAction << delColAction 
        << removeGapColumnsAction << removeAllGapsAction;

    editMenu->insertActions(editMenu->isEmpty() ? NULL : editMenu->actions().first(), actions);

    QMenu* colorsSchemeMenu = new QMenu(tr("Colors"), m);
    colorsSchemeMenu->setIcon(QIcon(":core/images/color_wheel.png"));
    foreach(QAction* a, colorSchemeMenuActions) {
        colorsSchemeMenu->addAction(a);
    }
    m->insertMenu(GUIUtils::findAction(m->actions(), MSAE_MENU_EDIT), colorsSchemeMenu);
}


void MSAEditorSequenceArea::sl_delSym() {
    del(cursorPos, false);
}

void MSAEditorSequenceArea::sl_delCol() {
    del(cursorPos, true);
}

void MSAEditorSequenceArea::sl_insSym() {
    ins(cursorPos, false);
}

void MSAEditorSequenceArea::sl_insCol() {
    ins(cursorPos, true);
}

void MSAEditorSequenceArea::sl_goto() {
    QDialog dlg;
    dlg.setModal(true);
    dlg.setWindowTitle(tr("Go To"));
    int aliLen = editor->getAlignmentLen();
    PositionSelector* ps = new PositionSelector(&dlg, 1, aliLen, true);
    connect(ps, SIGNAL(si_positionChanged(int)), SLOT(sl_onPosChangeRequest(int)));
    dlg.exec();
    delete ps;
}

void MSAEditorSequenceArea::sl_onPosChangeRequest(int pos) {
    centerPos(pos-1);
    setCursorPos(pos-1);
}

void MSAEditorSequenceArea::sl_lockedStateChanged() {
    updateActions();
}

void MSAEditorSequenceArea::centerPos(const QPoint& pos) {
    assert(isInRange(pos));
    int newStartPos = qMax(0, pos.x() - getNumVisibleBases(false)/2);
    setFirstVisibleBase(newStartPos);

    int newStartSeq = qMax(0, pos.y() - getNumVisibleSequences(false)/2);
    setFirstVisibleSequence(newStartSeq);
}


void MSAEditorSequenceArea::centerPos(int pos) {
    centerPos(QPoint(pos, cursorPos.y()));
}


void MSAEditorSequenceArea::wheelEvent (QWheelEvent * we) {
    bool toMin = we->delta() > 0;
    if (we->modifiers() == 0) {
        shBar->triggerAction(toMin ? QAbstractSlider::SliderSingleStepSub : QAbstractSlider::SliderSingleStepAdd);
    }  else if (we->modifiers() & Qt::SHIFT) {
        svBar->triggerAction(toMin ? QAbstractSlider::SliderSingleStepSub : QAbstractSlider::SliderSingleStepAdd);
    }
    QWidget::wheelEvent(we);
}

void MSAEditorSequenceArea::sl_removeColumnsWithGaps() {
    MAlignmentObject* msa = editor->getMSAObject();
    assert(!msa->isStateLocked());
    MAlignment ma = msa->getMAlignment();
    QList<int> columnsWithGaps;
    for (int c = 0, nc = ma.getLength(); c < nc; c++) {
        bool onlyGaps = true;
        foreach(const MAlignmentItem& item, ma.alignedSeqs) {
            if (item.sequence[c] != MAlignment_GapChar) {
                onlyGaps = false;
                break;
            }
        }
        if (onlyGaps) {
            columnsWithGaps.append(c);
        }
    }
    if (columnsWithGaps.isEmpty()) {
        return;
    }
    foreach(int c, columnsWithGaps) {
        for (int i = 0, n = ma.getNumSequences(); i < n; i++) {
            MAlignmentItem& item = ma.alignedSeqs[i];
            item.sequence[c] = 0;
        }
    }

    QBitArray gapMap(256);
    gapMap[0] = true;
    for (int i = 0, n = ma.getNumSequences(); i < n; i++) {
        MAlignmentItem& item = ma.alignedSeqs[i];
        int newLen = TextUtils::remove(item.sequence.data(), item.sequence.length(), gapMap);
        assert(newLen == item.sequence.length() - columnsWithGaps.size());
        item.sequence.resize(newLen);
    }

    ma.normalizeModel();
    msa->setMAlignment(ma);
}

void MSAEditorSequenceArea::sl_removeAllGaps() {
    QBitArray gapMap(256);
    gapMap[MAlignment_GapChar] = true;
    
    MAlignmentObject* msa = editor->getMSAObject();
    assert(!msa->isStateLocked());
    MAlignment ma = msa->getMAlignment();
    bool changed = false;
    for (int i = 0, n = ma.getNumSequences(); i < n; i++) {
        MAlignmentItem& item = ma.alignedSeqs[i];
        int newLen = TextUtils::remove(item.sequence.data(), item.sequence.length(), gapMap);
        changed = changed || newLen != item.sequence.length();
        item.sequence.resize(newLen);
    }
    if (changed) {
        ma.normalizeModel();
        msa->setMAlignment(ma);
        setFirstVisibleBase(0);
        setFirstVisibleSequence(0);
    }
}

bool MSAEditorSequenceArea::checkState() const {
#ifdef _DEBUG
    int aliLen = editor->getMSAObject()->getMAlignment().getLength();
    int nSeqs = editor->getMSAObject()->getMAlignment().getNumSequences();
    
    assert(startPos >=0 && startSeq >=0);
    int lastPos = getLastVisibleBase(true);
    int lastSeq = getLastVisibleSequence(true);
    assert(lastPos < aliLen && lastSeq < nSeqs);
    
    int cx = cursorPos.x();
    int cy = cursorPos.y();
    assert(cx >= 0 && cy >= 0);
    assert(cx < aliLen && cy < nSeqs);
#endif
    return true;
}

static void saveFont(const QFont& f) {
    Settings* s = AppContext::getSettings();
    s->setValue(SETTINGS_ROOT + SETTINGS_FONT_FAMILY, f.family());
    s->setValue(SETTINGS_ROOT + SETTINGS_FONT_SIZE, f.pointSize());
    s->setValue(SETTINGS_ROOT + SETTINGS_FONT_ITALIC, f.italic());
    s->setValue(SETTINGS_ROOT + SETTINGS_FONT_BOLD, f.bold());
}

void MSAEditorSequenceArea::setFont(const QFont& f) {
    seqFont = f;
    seqFont.setPointSize(qBound(MIN_FONT_SIZE, f.pointSize(), MAX_FONT_SIZE));
    completeRedraw = true;
    updateSeqFontMetrics();
    validateRanges();
    updateActions();
    emit si_scaleChanged();
    update();
    saveFont(seqFont);
}

void MSAEditorSequenceArea::sl_changeFont() {
    bool ok = false;
    QFont f = QFontDialog::getFont(&ok, seqFont, this, tr("Select font for alignment"));
    if (!ok) {
        return;
    }
    setFont(f);
}

void MSAEditorSequenceArea::sl_zoomIn() {
    int pSize = seqFont.pointSize();
    if (pSize < MAX_FONT_SIZE) {
        seqFont.setPointSize(pSize+1);
        setFont(seqFont);
    }
}

void MSAEditorSequenceArea::sl_zoomOut() {
    int pSize = seqFont.pointSize();
    if (pSize > MIN_FONT_SIZE) {
        seqFont.setPointSize(pSize-1);
        setFont(seqFont);
    }
}

void MSAEditorSequenceArea::sl_resetFont() {
    QFont f(DEFAULT_FONT_FAMILY, DEFAULT_FONT_SIZE);
    setFont(f);
}
}//namespace
