/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SubstMatrixFile.h"

#include <assert.h>

#include <QFile>
#include <QString>
#include <QTextStream>
#include <QByteArray>

#include <core_api/AppContext.h>
#include <core_api/Log.h>
#include <core_api/SubstMatrixRegistry.h>
#include <util_smith_waterman/SubstMatrixFactory.h>

#include <QObject>

#define IS_WHITESPACE(c) (' ' == c || '\t' == c || '\v' == c || '\f' == c || '\r' == c)

#define IS_UPPER_ALPHA(c) (c >= 'A' && c <= 'Z')
#define IS_LOWER_ALPHA(c) (c >= 'a' && c <= 'z')

#define IS_NEWLINE_SYMBOL(c) ('\n' == c)
#define IS_COMMENT_SYMBOL(c) ('#'  == c)
#define IS_CORRECT_SYMBOL(c) (IS_UPPER_ALPHA(c) || IS_LOWER_ALPHA(c) || ('*' == c) || ('-' == c))


namespace GB2 {

static GB2::LogCategory log(ULOG_CAT_SW);

SubstMatrix* SubstMatrixFile::read(const QString& fileName) {
    QFile mtxFile(fileName);
    if (!mtxFile.open(QIODevice::ReadOnly | QIODevice::Text)) {
        return 0;
    }
    QTextStream inText(&mtxFile);
	
    int dim = 0;
    int currRow = 0;
    float* rawMatrix = 0;
    QByteArray mtxAlphabet;

    enum State {State_skipComment, State_getAlph, State_getMtx, State_fin, State_err};

    int state = State_skipComment;
    while (!inText.atEnd() && State_err != state && State_fin != state) {
        char ch = 0;
        inText >> ch;

        if (IS_WHITESPACE(ch)) {
            continue;
        }

        switch (state) {
            case (State_skipComment):
                if (IS_COMMENT_SYMBOL(ch)) {
                    inText.readLine();
                } else if (IS_CORRECT_SYMBOL(ch)) {
                    mtxAlphabet.append(ch);
                    state = State_getAlph;
                }
                break;
            case (State_getAlph):
                if (IS_CORRECT_SYMBOL(ch)) {
                    mtxAlphabet.append(ch);
                } else if (IS_NEWLINE_SYMBOL(ch)) {
                    dim = mtxAlphabet.length();
                    rawMatrix = new float[dim * dim];
                    state = State_getMtx;
                } else {
                    state = State_err;
                }
                break;
            case (State_getMtx):
                if (IS_CORRECT_SYMBOL(ch) && ch == mtxAlphabet[currRow]) {
                    for (int currColumn = 0; currColumn < dim; currColumn++) {
                        inText >> rawMatrix[currRow * dim + currColumn];
                    }
                    if (++currRow == dim) {
                        state = State_fin;
                    }
                    inText.readLine();
                    } else {
                    state = State_err;
                }
                break;
            case (State_fin):
                break;
            case (State_err):
                break;
            default:
                assert(false);
            break;
        }
    }
    if (State_fin != state) {
        if (0 != rawMatrix) {
            delete[] rawMatrix;
        }
        log.error(GB2::SubstMatrixRegistry::tr("Error parsing weight-matrix file: %1").arg(fileName));
        return 0;
    }
    SubstMatrix* mtx = SubstMatrixFactory::createSubstMatrix(mtxAlphabet,rawMatrix);
    delete[] rawMatrix;
    return mtx;
}

bool SubstMatrixFile::write(const SubstMatrix* mtx, const QString& fileName) {
    Q_UNUSED(mtx); Q_UNUSED(fileName);
    assert(false);
    return false;
}

} // namespace
