/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtGui/QApplication>
#include <QtCore/QtAlgorithms>
#include <QtCore/QFile>
#include <QtCore/QDir>

#include <assert.h>
#include <utility>
#include <memory>

#include <core_api/Settings.h>
#include <core_api/AppContext.h>
#include <core_api/ServiceTypes.h>
#include <core_api/Log.h>
#include "ScriptRegistry.h"

namespace GB2
{

/* TRANSLATOR GB2::ScriptRegistryService */    

static LogCategory log(ULOG_CAT_SCRIPTS);

static const char * SCRIPT_PROPERTY_NAME = "name";
static const char * SCRIPT_PROPERTY_TYPE = "type";
static const char * SCRIPT_PROPERTY_CONFIGURE = "configure";
static const char * SCRIPT_PROPERTY_SETUP = "setup";
static const char * SCRIPT_PROPERTY_MAIN = "main";
static const char * SCRIPT_PROPERTY_GLOBAL = "U_GLOBAL";

static QString loadTextFromUrl( const QString & url ) {
    QFile f( url );
    f.open( QIODevice::ReadOnly );
    if( f.isOpen() && f.isReadable() ) {
        QString res( f.readAll() );
        return res;
    }
    return QString();
}

Script::Script( const QString & _url ) : state(ScriptState_Unloaded), url(_url) {
}

bool Script::reload() {
    bool ret = false;
    text = loadTextFromUrl( url );
    std::auto_ptr<QScriptEngine> eng( new QScriptEngine() );
//    importExtensions( eng.get() );
    if( text.isEmpty() || !eng->canEvaluate(text) ) {
        state = ScriptState_Invalid;
    }
    else {
        eng->evaluate( text );
        name = getGlobal( eng.get() ).property( SCRIPT_PROPERTY_NAME ).toString();
        type = getGlobal( eng.get() ).property( SCRIPT_PROPERTY_TYPE ).toString();
        QScriptValue main = getGlobal( eng.get() ).property( SCRIPT_PROPERTY_MAIN );

        if( eng->hasUncaughtException() ) {
            state = ScriptState_Invalid;
            log.trace( eng->uncaughtException().toString() );
            log.trace( eng->uncaughtExceptionBacktrace().join("\n") );
        }

        ret = !name.isEmpty() && !type.isEmpty() && main.isFunction() && !eng->hasUncaughtException();
        if( !ret ) {
            state = ScriptState_Invalid;
        } else {
            state = ScriptState_Ready;
        }
    }
    return ret;
}

bool Script::init_engine( QScriptEngine * eng, QScriptValue * exception ) {
    assert( eng );
    if( isReady() ) {
//        importExtensions( eng );
        eng->evaluate( text );
        if( eng->hasUncaughtException() && exception ) {
            *exception = eng->uncaughtException();
            return false;
        }
        return true;
    }
    return false;
}

void Script::importExtensions( QScriptEngine * engine ) {
    assert( engine );
    engine->importExtension( "qt.core" );
    engine->importExtension( "qt.gui" );
    engine->importExtension( "qt.xml" );
}

QScriptValue Script::callMain( QScriptEngine * eng, QScriptValue * exception /* = QScriptValue */ ) {
    if( isReady() ) {
        return callFunction( eng, SCRIPT_PROPERTY_MAIN, exception );
    }
    return QScriptValue();
}

QScriptValue Script::callConfigure( QScriptEngine * eng, QScriptValue * exception ) {
    QScriptValue ret = callFunction( eng, SCRIPT_PROPERTY_CONFIGURE, exception );
    if( ret.isObject() && !eng->hasUncaughtException() ) {
        custom_props = eng->fromScriptValue<QVariantMap>( ret );
    }
    return ret;
}

QScriptValue Script::callSetup( QScriptEngine * eng, QScriptValue * exception ) {
    QScriptValueList settings;
    settings << eng->toScriptValue( custom_props );
    QScriptValue ret = callFunction( eng, SCRIPT_PROPERTY_SETUP, exception, &settings );
    return ret;
}

QScriptValue Script::callFunction( QScriptEngine * eng, const QString & name, QScriptValue * exception,
                                   const QScriptValueList * arguments ) {
    assert( eng );
    QScriptValue ret;
    QScriptValue func = getGlobal(eng).property( name );
    if( func.isFunction() ) {
        ret = arguments ? func.call( QScriptValue(), *arguments ) : func.call( QScriptValue(), QScriptValueList() );
        if( eng->hasUncaughtException() && exception ) {
            *exception = eng->uncaughtException();
        }
    }
    return ret;
}

QScriptValue Script::getGlobal( QScriptEngine * eng ) {
    return eng->globalObject().property( SCRIPT_PROPERTY_GLOBAL );
}

QString Script::getStateString() const {
    switch( state ) {
        case ScriptState_Invalid:   return ScriptRegistryService::tr("scr_state_invalid");
        case ScriptState_Unloaded:  return ScriptRegistryService::tr("scr_state_unloaded");
        case ScriptState_Ready:     return ScriptRegistryService::tr("scr_state_ready");
    }
    return QString();
}

void Script::saveCustomSettings( Settings * settings, const QString & path ) const {
    QMapIterator< QString, QVariant > it( custom_props );
    while( it.hasNext() ) {
        it.next();
        settings->setValue( path + it.key(), it.value() );
    }
}

void Script::loadCustomSettings( Settings * settings, const QString & path ) {
    QStringList keys = settings->getAllKeys( path );
    foreach( QString key, keys ) {
        custom_props[key] = settings->getValue( path + key );
    }
}

// ==============  ScriptRegistry ===============

const char * ScriptRegistryService::SETTINGS_SCRIPT_URLS = "script_registry/script_urls";
const char * ScriptRegistryService::SETTINGS_SCRIPT_CUSTOM_SETTINGS = "script_registry/custom_settings";

ScriptRegistryService::ScriptRegistryService() :
Service(Service_ScriptRegistry, tr("script_reg_service_name"), tr("script_reg_service_desc")) {
    QString path = QApplication::applicationDirPath();
    QApplication::addLibraryPath( path + "/plugins" );

    Settings * settings = AppContext::getSettings();
    QStringList urls = settings->getValue( SETTINGS_SCRIPT_URLS ).toStringList();
    if (urls.isEmpty()) {
        QDir scriptsDir = QDir::searchPaths( PATH_PREFIX_SCRIPTS ).first();

        QStringList nameFilter; nameFilter << "*.qs";
        QFileInfoList scripts = scriptsDir.entryInfoList(nameFilter, QDir::Files | QDir::Readable, QDir::Name);
        foreach(const QFileInfo& fi, scripts) {
            log.details(tr("adding_script_%1").arg(fi.absoluteFilePath()));
            urls.append(fi.absoluteFilePath());
        }
    } 
    int i = 0;
    foreach( QString url, urls ) {
        Script* sc = registerScript(url);
        if (sc!=NULL) {
            sc->loadCustomSettings( settings, QString(SETTINGS_SCRIPT_CUSTOM_SETTINGS) + "/" + QString::number(i++) + "/" );
        }
    }
}

ScriptRegistryService::~ScriptRegistryService() {
    QStringList urls;
    Settings * settings = AppContext::getSettings();
    int i = 0;
    foreach( Script * scr, scripts ) {
        scr->saveCustomSettings( settings, QString(SETTINGS_SCRIPT_CUSTOM_SETTINGS) + "/" + QString::number(i++) + "/" );
        urls << scr->getUrl();
    }
    AppContext::getSettings()->setValue( SETTINGS_SCRIPT_URLS, urls );
    qDeleteAll( scripts );
}

Script* ScriptRegistryService::registerScript( const QString& url) {
    if( findByUrl(url) ) {
        return NULL;
    }
    Script* script = new Script(url);
    scripts.push_back(script);
    return script;
}

void ScriptRegistryService::unregisterScript( Script * scr ) {
    int n = scripts.removeAll( scr );
    assert( 1 == n ); Q_UNUSED(n);
    delete scr;
}

QList< Script * > ScriptRegistryService::getScriptsByType( const QString & type ) const {
    QList< Script * > result;
    foreach( Script * scr, scripts ) {
        if( type == scr->getType() ) {
            result.push_back( scr );
        }
    }
    return result;
}

Script * ScriptRegistryService::getScript( const QString & name, const QString & type ) const {
    QList<Script *> typed_scripts = getScriptsByType( type );
    foreach( Script * scr, typed_scripts ) {
        if( name == scr->getName() ) {
            return scr;
        }
    }
    return 0;
}

void ScriptRegistryService::loadScripts() {
    foreach( Script * s, scripts ) {
        s->reload();
    }
}

Script * ScriptRegistryService::findByUrl( const QString & url ) {
    foreach( Script * scr, scripts ) {
        if( url == scr->getUrl() ) {
            return scr;
        }
    }
    return 0;
}

// ==============  ScriptRegistryEnableTask ===============

ScriptRegistryEnableTask::ScriptRegistryEnableTask( ScriptRegistryService * _srs ) :
Task( tr("sript_registry_enable_task"), TaskFlag_None), srs(_srs) {
}

void ScriptRegistryEnableTask::run() {
    srs->loadScripts();
}

} //namespace
