/***************************************************************************
 *   Copyright (C) 2008 by Alexey Balakin                                  *
 *   mathgl.abalakin@gmail.com                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <QTextEdit>
#include <QPrintDialog>
#include <QCloseEvent>
#include <QStatusBar>
#include <QMessageBox>
#include <QPrinter>
#include <QApplication>
#include <QPainter>
#include <QSplitter>
#include <QAction>
#include <QPixmap>
#include <QMdiArea>
#include <QMdiSubWindow>
#include <QDockWidget>
#include <mgl/mgl_parse.h>
//-----------------------------------------------------------------------------
#include "scriptwindow.h"
#include "qmglsyntax.h"
#include "qmglcanvas.h"
#include "finddialog.h"
#include "helpwindow.h"
#include "optiondialog.h"
#include "styledialog.h"
#include "propdialog.h"
#include "setupdialog.h"
#include "infodialog.h"
#include "graphwindow.h"
#include "infodialog.h"
#include "newcmddialog.h"
//-----------------------------------------------------------------------------
extern bool mglAutoExecute;
extern bool mglAutoSave;
extern mglParse parser;
ArgsDialog *args_dlg=0;
QString defFontFamily;
int defFontSize;
PropDialog *propDlg=0;
int ScriptWindow::num_wnd=0;
//-----------------------------------------------------------------------------
ScriptWindow::ScriptWindow(QWidget *wp) : QMainWindow(wp)
{
	setWindowTitle(tr("untitled - UDAV"));
	setAttribute(Qt::WA_DeleteOnClose);
	printer = new QPrinter;
	if(!args_dlg)	args_dlg = new ArgsDialog;
	findDialog = new FindDialog(this);
	optDialog = new OptionDialog(this);
	stlDialog = new StyleDialog(this);
	setupDlg = new SetupDialog(this);
	dataOpenDlg = new DataOpenDialog(this);
	newCmdDlg = new NewCmdDialog(this);

	connect(findDialog, SIGNAL(findText(const QString &, bool, bool)), this, SLOT(findText(const QString &, bool, bool)));
	connect(findDialog, SIGNAL(replText(const QString &, const QString &, bool, bool)), this, SLOT(replText(const QString &, const QString &, bool, bool)));
	connect(setupDlg, SIGNAL(putText(const QString &)), this, SLOT(animPutText(const QString &)));

	split = new QSplitter(this);
	edit = new QTextEdit(split);	edit->setAcceptRichText(false);
	connect(edit, SIGNAL(cursorPositionChanged()), this, SLOT(editPosChanged()));
	new QMGLSyntax(edit);
	defFontFamily = edit->fontFamily();
	defFontSize = int(edit->fontPointSize());

	info = new MemoryWindow(this);
	graph = new GraphWindow(this);	graph->mgl->textMGL = edit;
	connect(graph, SIGNAL(save()), this, SLOT(save()));
	connect(graph, SIGNAL(statusBarMessage(const QString &)), this, SLOT(statusBarMessage(const QString &)));
	connect(graph, SIGNAL(animPutText(const QString &)), this, SLOT(animPutText(const QString &)));
	connect(graph->mgl, SIGNAL(objChanged(int)), this, SLOT(setCursorPosition(int)));
	connect(graph->mgl, SIGNAL(posChanged(QString)), statusBar(), SLOT(showMessage(QString)));
	connect(graph->mgl, SIGNAL(refreshData()), info, SLOT(refresh()));

	mdi = new QMdiArea(split);
	mdi->addSubWindow(graph);	graph->setWindowState(Qt::WindowMaximized);
	mdi->addSubWindow(info);	info->setWindowState(Qt::WindowMaximized);
	hlp = new HelpWindow(this);	mdi->addSubWindow(hlp);
	hlp->setWindowState(Qt::WindowMaximized);
	mdi->setActiveSubWindow(graph);	mdi->setViewMode(QMdiArea::TabbedView);
	connect(mdi, SIGNAL(subWindowActivated(QMdiSubWindow *)), this, SLOT(subActivated(QMdiSubWindow *)));

	messWnd = new QDockWidget(tr("Messages and warnings"),this);
	mess = new QTextEdit(this);	messWnd->setWidget(mess);
	messWnd->setAllowedAreas(Qt::TopDockWidgetArea | Qt::BottomDockWidgetArea);
	addDockWidget(Qt::BottomDockWidgetArea, messWnd);
	messWnd->resize(size().width(), 0);
	graph->mgl->warnMGL = mess;	new MessSyntax(mess);
	connect(mess, SIGNAL(textChanged()), this, SLOT(warnChanged()));

	calcWnd = new QDockWidget(tr("Calculator"),this);
	calc = new CalcDialog();	calcWnd->setWidget(calc);
	calcWnd->setAllowedAreas(Qt::TopDockWidgetArea | Qt::BottomDockWidgetArea);
	addDockWidget(Qt::BottomDockWidgetArea, calcWnd);
	calcWnd->resize(size().width(), 200);
	connect(calc, SIGNAL(putNumber(QString)),this,SLOT(putText(QString)));

	makeMenu();
	setCentralWidget(split);
	setWindowIcon(QIcon(":/udav.png"));
	readSettings();
	if(!propDlg)	propDlg = new PropDialog;
	edit->setLineWrapMode(QTextEdit::NoWrap);
	connect(propDlg, SIGNAL(sizeChanged(int,int)), graph->mgl, SLOT(imgSize(int,int)));
	connect(edit,SIGNAL(textChanged()), this, SLOT(setAsterix()));
	edit->setFocus();	// TODO: not work right now :(
	statusBar()->showMessage(tr("Ready"), 2000);
	num_wnd++;
}
//-----------------------------------------------------------------------------
ScriptWindow::~ScriptWindow()	{	delete printer;	}
//-----------------------------------------------------------------------------
void ScriptWindow::statusBarMessage(const QString &txt)
{	statusBar()->showMessage(txt, 2000);	}
//-----------------------------------------------------------------------------
void ScriptWindow::printText()
{
	QPrintDialog printDlg(printer, this);
	if (printDlg.exec() == QDialog::Accepted)
	{
		statusBar()->showMessage(tr("Printing..."));
		edit->print(printer);
		statusBar()->showMessage(tr("Printing completed"), 2000);
	}
	else
		statusBar()->showMessage(tr("Printing aborted"), 2000);
}
//-----------------------------------------------------------------------------
void ScriptWindow::closeEvent(QCloseEvent* ce)
{
	bool ok=true;
	writeSettings();
	if(edit->document()->isModified())
		switch(QMessageBox::information(this, tr("UDAV"),
				tr("Do you want to save the changes to the document?"),
				QMessageBox::Yes, QMessageBox::No, QMessageBox::Cancel))
		{
			case QMessageBox::Yes:	save();	break;
			case QMessageBox::No:	break;
			default:	ok=false;	break;
		}
	if(ok)
	{
		num_wnd--;
		ce->accept();
		if(num_wnd==0)	QApplication::quit();
	}
	else	ce->ignore();
}
//-----------------------------------------------------------------------------
void ScriptWindow::find()
{
	findDialog->show();
	findDialog->raise();
	findDialog->activateWindow();
}
//-----------------------------------------------------------------------------
bool ScriptWindow::findText(const QString &str, bool cs, bool fw)
{
//	static int para=0, index=0;
	static QTextDocument::FindFlags f;
	static QString stri="";
	if(!str.isEmpty())
	{
		stri = str;
		f = QTextDocument::FindFlags();
		if(fw)	f = f|QTextDocument::FindBackward;
		if(cs)	f = f|QTextDocument::FindCaseSensitively;
	}
	bool res = edit->find(stri, f);
	if(!res)
		QMessageBox::information(this, tr("UDAV - find text"), tr("No string occurrence is found"));
	return res;
}
//-----------------------------------------------------------------------------
void ScriptWindow::replText(const QString &str, const QString &txt, bool cs, bool fw)
{
	static bool res=false;
	if(str.isEmpty())	{	res = false;	return;	}
	if(res)	edit->textCursor().insertText(txt);
	res = findText(str, cs, fw);
}
//-----------------------------------------------------------------------------
void ScriptWindow::showHelp()
{
	QString s = edit->textCursor().block().text(), dlm(" #;:\t");
	int i, n = s.length();
	for(i=0;i<n;i++)	if(dlm.contains(s[i]))	break;
	s.truncate(i);
//	s = s.section(' ',0);
	hlp->showHelp(s);
}
//-----------------------------------------------------------------------------
int mgl_cmd_cmp(const void *a, const void *b);
void ScriptWindow::editPosChanged()
{
	register int i, n, m;
	QString text = edit->textCursor().block().text(), dlm(" #;:\t");
	n = text.length();
	for(i=0;i<n;i++)	if(dlm.contains(text[i]))	break;
	text.truncate(i);	m = text.length();

	for(n=0;parser.Cmd[n].name[0];n++){};	// determine the number of symbols in parser
	mglCommand tst, *rts;
	wchar_t *s = new wchar_t[m+1];
	text.toWCharArray(s);	s[m]=0;	tst.name = s;
	rts = (mglCommand *)bsearch(&tst, parser.Cmd, n, sizeof(mglCommand), mgl_cmd_cmp);
	if(rts)
	{
		statusBar()->showMessage(QString::fromWCharArray(rts->desc)+": " + QString::fromWCharArray(rts->form));
	}
	else	statusBar()->showMessage(tr("Not recognized"));
	delete []s;
}
//-----------------------------------------------------------------------------
void ScriptWindow::addOptions()
{
	if(optDialog->exec()==QDialog::Accepted)
	{
		edit->moveCursor(QTextCursor::EndOfLine);
		edit->insertPlainText(optDialog->getOption());
	}
}
//-----------------------------------------------------------------------------
void ScriptWindow::addSetup()	{	setupDlg->exec();	}
//-----------------------------------------------------------------------------
void ScriptWindow::animPutText(const QString &s)
{	edit->moveCursor(QTextCursor::Start);	edit->insertPlainText(s);	}
//-----------------------------------------------------------------------------
void ScriptWindow::putText(const QString &txt)
{	edit->insertPlainText(txt);	}
//-----------------------------------------------------------------------------
void ScriptWindow::addStyle()
{
	if(stlDialog->exec()==QDialog::Accepted)
	{
		QString s = edit->textCursor().block().text();
		int i = s.indexOf(';');
		if(i<0)	edit->moveCursor(QTextCursor::EndOfLine);
		else
		{
			edit->moveCursor(QTextCursor::StartOfBlock);
			// foolish way :(
			for(;i>0;i--)	edit->moveCursor(QTextCursor::Left);
		}
		edit->insertPlainText(stlDialog->getStyle());
	}
}
//-----------------------------------------------------------------------------
void ScriptWindow::setEditorFont(QFont *f)
{	edit->setFont(f ? *f : QFont(defFontFamily, defFontSize));	}
//-----------------------------------------------------------------------------
void ScriptWindow::setEditPos(bool bottom)
{	split->setOrientation(bottom ? Qt::Vertical : Qt::Horizontal);	}
//-----------------------------------------------------------------------------
void ScriptWindow::properties()	{	propDlg->exec();	}
//-----------------------------------------------------------------------------
void ScriptWindow::setCursorPosition(int n)
{
	if(n<0)	return;
	edit->moveCursor(QTextCursor::Start);
	for(int i=0;i<n;i++)	edit->moveCursor(QTextCursor::NextBlock);
	edit->setFocus();
}
//-----------------------------------------------------------------------------
void ScriptWindow::warnChanged()
{
	if(mess->toPlainText().isEmpty())	return;
	messWnd->show();	ainfo->setChecked(true);
}
//-----------------------------------------------------------------------------
#ifndef USE_HDF5
void ScriptWindow::saveHDF5(const QString &fileName){}
void ScriptWindow::loadHDF5(const QString &fileName){}
//-----------------------------------------------------------------------------
#else
#include <hdf5.h>
void ScriptWindow::loadHDF5(const QString &fileName)
{
	// H5T_C_S1 - C string
	hid_t hf,hg,hd,hs,ht;
	hsize_t dims[3];
	long rank;
	hf = H5Fopen(fileName.toAscii().constData(), H5F_ACC_RDONLY, H5P_DEFAULT);
	if(!hf)	return;
	hg = H5Gopen(hf, "/");
	hsize_t num, nx, ny, nz, i;
	char name[256];
	H5Gget_num_objs(hg, &num);
	for(i=0;i<num;i++)		// TODO: add script loading
	{
		if(H5Gget_objtype_by_idx(hg, i)!=H5G_DATASET)	continue;
		H5Gget_objname_by_idx(hg, i, name, 256);
		hd = H5Dopen(hg,name);	hs = H5Dget_space(hd);
		ht = H5Dget_type(hd);
		rank = H5Sget_simple_extent_ndims(hs);
		if(H5Tget_class(ht)==H5T_STRING)	// load script
		{
			H5Sget_simple_extent_dims(hs,dims,0);
			char *buf = new char[dims[0]+1];
			H5Dread(hd, H5T_C_S1, H5S_ALL, H5S_ALL, H5P_DEFAULT, buf);
			buf[dims[0]]=0;		// to be sure :)
			edit->setText(buf);
			graph->animParseText(edit->toPlainText());
			setCurrentFile(fileName);
			delete []buf;
			statusBar()->showMessage(tr("Loaded document %1").arg(fileName), 2000);
			if(mglAutoExecute)	graph->mgl->execute();
		}
		else if(H5Tget_class(ht)==H5T_FLOAT || H5Tget_class(ht)==H5T_INTEGER)
		{
			for(int j=0;name[j];j++)	if(!isalnum(name[j]))	name[j]='_';
			mglVar *v = parser.AddVar(name);
			nx = ny = nz = 1;
			if(rank>0 && rank<=3)
			{
				H5Sget_simple_extent_dims(hs,dims,0);
				switch(rank)
				{
					case 1:	nx=dims[0];	break;
					case 2:	nx=dims[1];	ny=dims[0];	break;
					case 3:	nx=dims[2];	ny=dims[1];	nz=dims[0];	break;
				}
				v->d.Create(nx, ny, nz);
				H5Dread(hd, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, H5P_DEFAULT, v->d.a);
			}
		}
		H5Dclose(hd);	H5Sclose(hs);	H5Tclose(ht);
	}
	H5Gclose(hg);	H5Fclose(hf);
}
//-----------------------------------------------------------------------------
void ScriptWindow::saveHDF5(const QString &fileName)
{
	hid_t hf,hd,hs;
	hsize_t dims[3];
	long rank = 3;
	const char *fname = fileName.toAscii().constData();

	H5Eset_auto(0,0);
	hf = H5Fcreate(fname, H5F_ACC_TRUNC, H5P_DEFAULT, H5P_DEFAULT);
	if(hf<0)
	{
		statusBar()->showMessage(tr("Could not write to %1").arg(fileName), 2000);
		return;
	}
	{	// save script
		QString txt = edit->toPlainText();
		dims[0] = txt.length()+1;
		char *buf = new char[dims[0]+1];
		memcpy(buf, txt.toAscii().constData(), dims[0]);
		buf[dims[0]]=0;
		hs = H5Screate_simple(1, dims, 0);
		hd = H5Dcreate(hf, "mgl_script", H5T_C_S1, hs, H5P_DEFAULT);
		H5Dwrite(hd, H5T_C_S1, hs, hs, H5P_DEFAULT, buf);
		H5Dclose(hd);	H5Sclose(hs);
		delete []buf;
	}
	mglVar *v = parser.DataList;
	char name[256];
	while(v)
	{
		wcstombs(name,v->s,wcslen(v->s)+1);
		if(v->d.nz==1 && v->d.ny == 1)
		{	rank = 1;	dims[0] = v->d.nx;	}
		else if(v->d.nz==1)
		{	rank = 2;	dims[0] = v->d.ny;	dims[1] = v->d.nx;	}
		else
		{	rank = 3;	dims[0] = v->d.nz;	dims[1] = v->d.ny;	dims[2] = v->d.nx;	}
		hs = H5Screate_simple(rank, dims, 0);
		hd = H5Dcreate(hf, name, H5T_IEEE_F32LE, hs, H5P_DEFAULT);

		H5Dwrite(hd, H5T_NATIVE_FLOAT, hs, hs, H5P_DEFAULT, v->d.a);
		H5Dclose(hd);	H5Sclose(hs);
		v = v->next;
	}
	H5Fclose(hf);
	setCurrentFile(fileName);
	statusBar()->showMessage(tr("File %1 saved").arg(fileName), 2000);
	return;
}
//-----------------------------------------------------------------------------
#endif
