// OASISXMLCatalogReader.java - Read XML Catalog files

/*
 * The Apache Software License, Version 1.1
 *
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Xalan" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation and was
 * originally based on software copyright (c) 2001, International
 * Business Machines Corporation., http://www.ibm.com.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.xml.resolver.readers;

import java.lang.Integer;
import java.util.Stack;
import java.util.Vector;
import java.util.Enumeration;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.net.MalformedURLException;
import org.apache.xml.resolver.Catalog;
import org.apache.xml.resolver.CatalogEntry;
import org.apache.xml.resolver.CatalogException;
import org.apache.xml.resolver.readers.CatalogReader;
import org.apache.xml.resolver.helpers.PublicId;
import org.apache.xml.resolver.helpers.Debug;

import org.xml.sax.*;
import org.w3c.dom.*;

/**
 * <p>Parse OASIS Entity Resolution Technical Committee 
 * XML Catalog files.</p>
 *
 * @see Catalog
 *
 * @author Norman Walsh
 * <a href="mailto:Norman.Walsh@Sun.COM">Norman.Walsh@Sun.COM</a>
 *
 * @version 1.0
 */
public class OASISXMLCatalogReader extends SAXCatalogReader implements SAXCatalogParser {
  /** The catalog object needs to be stored by the object so that
     * SAX callbacks can use it.
     */
  protected Catalog catalog = null;

  /** The namespace name of OASIS ERTC catalogs */
  public static final String namespaceName = "urn:oasis:names:tc:entity:xmlns:xml:catalog";

  /** The namespace name of OASIS ERTC TR9401 catalog extension */
  public static final String tr9401NamespaceName = "urn:oasis:names:tc:entity:xmlns:tr9401:catalog";

  protected Stack baseURIStack = new Stack();
  protected Stack overrideStack = new Stack();
  protected Stack namespaceStack = new Stack();

  /** Set the current catalog. */
  public void setCatalog (Catalog catalog) {
    this.catalog = catalog;
  }

  /** Get the current catalog. */
  public Catalog getCatalog () {
    return catalog;
  }

  /**
   * <p>Are we in an extension namespace?</p>
   *
   * @returns true if the current stack of open namespaces includes
   *               an extension namespace.
   */
  protected boolean inExtensionNamespace() {
    boolean inExtension = false;

    Enumeration elements = namespaceStack.elements();
    while (!inExtension && elements.hasMoreElements()) {
      String ns = (String) elements.nextElement();
      if (ns == null) {
	inExtension = true;
      } else {
	inExtension = (!ns.equals(tr9401NamespaceName)
		       && !ns.equals(namespaceName));
      }
    }

    return inExtension;
  }

  // ----------------------------------------------------------------------
  // Implement the SAX ContentHandler interface

  /** <p>The SAX <code>setDocumentLocator</code> method does nothing.</p> */
  public void setDocumentLocator (Locator locator) {
    return;
  }

  /** <p>The SAX <code>startDocument</code> method does nothing.</p> */
  public void startDocument ()
    throws SAXException {
    baseURIStack.push(catalog.getCurrentBase());
    overrideStack.push(catalog.getDefaultOverride());
    return;
  }

  /** <p>The SAX <code>endDocument</code> method does nothing.</p> */
  public void endDocument ()
    throws SAXException {
    return;
  }

  /**
   * <p>The SAX <code>startElement</code> method recognizes elements
   * from the plain catalog format and instantiates CatalogEntry
   * objects for them.</p>
   *
   * @param namespaceURI The namespace name of the element.
   * @param localName The local name of the element.
   * @param qName The QName of the element.
   * @param atts The list of attributes on the element.
   *
   * @see CatalogEntry
   */
  public void startElement (String namespaceURI,
			    String localName,
			    String qName,
			    Attributes atts)
    throws SAXException {

    int entryType = -1;
    Vector entryArgs = new Vector();

    namespaceStack.push(namespaceURI);

    boolean inExtension = inExtensionNamespace();

    if (namespaceURI != null && namespaceName.equals(namespaceURI)
	&& !inExtension) {
      // This is an XML Catalog entry

      if (atts.getValue("xml:base") != null) {
	String baseURI = atts.getValue("xml:base");
	entryType = Catalog.BASE;
	entryArgs.add(baseURI);
	baseURIStack.push(baseURI);

	Debug.message(4, "xml:base", baseURI);

	try {
	  CatalogEntry ce = new CatalogEntry(entryType, entryArgs);
	  catalog.addEntry(ce);
	} catch (CatalogException cex) {
	  if (cex.getExceptionType() == CatalogException.INVALID_ENTRY_TYPE) {
	    Debug.message(1, "Invalid catalog entry type", localName);
	  } else if (cex.getExceptionType() == CatalogException.INVALID_ENTRY) {
	    Debug.message(1, "Invalid catalog entry (base)", localName);
	  }
	}

	entryType = -1;
	entryArgs = new Vector();

      } else {
	baseURIStack.push(baseURIStack.peek());
      }

      if ((localName.equals("catalog") || localName.equals("group"))
	  && atts.getValue("prefer") != null) {
	String override = atts.getValue("prefer");

	if (override.equals("public")) {
	  override = "yes";
	} else if (override.equals("system")) {
	  override = "no";
	} else {
	  Debug.message(1,
			"Invalid prefer: must be 'system' or 'public'",
			localName);
	  override = catalog.getDefaultOverride();
	}

	entryType = Catalog.OVERRIDE;
	entryArgs.add(override);
	overrideStack.push(override);

	Debug.message(4, "override", override);

	try {
	  CatalogEntry ce = new CatalogEntry(entryType, entryArgs);
	  catalog.addEntry(ce);
	} catch (CatalogException cex) {
	  if (cex.getExceptionType() == CatalogException.INVALID_ENTRY_TYPE) {
	    Debug.message(1, "Invalid catalog entry type", localName);
	  } else if (cex.getExceptionType() == CatalogException.INVALID_ENTRY) {
	    Debug.message(1, "Invalid catalog entry (override)", localName);
	  }
	}

	entryType = -1;
	entryArgs = new Vector();

      } else {
	overrideStack.push(overrideStack.peek());
      }

      if (localName.equals("delegatePublic")) {
	if (checkAttributes(atts, "publicIdStartString", "catalog")) {
	  entryType = Catalog.DELEGATE_PUBLIC;
	  entryArgs.add(atts.getValue("publicIdStartString"));
	  entryArgs.add(atts.getValue("catalog"));

	  Debug.message(4, "delegatePublic",
			PublicId.normalize(atts.getValue("publicIdStartString")),
			atts.getValue("catalog"));
	}
      } else if (localName.equals("delegateSystem")) {
	if (checkAttributes(atts, "systemIdStartString", "catalog")) {
	  entryType = Catalog.DELEGATE_SYSTEM;
	  entryArgs.add(atts.getValue("systemIdStartString"));
	  entryArgs.add(atts.getValue("catalog"));

	  Debug.message(4, "delegateSystem",
			atts.getValue("systemIdStartString"),
			atts.getValue("catalog"));
	}
      } else if (localName.equals("delegateURI")) {
	if (checkAttributes(atts, "uriStartString", "catalog")) {
	  entryType = Catalog.DELEGATE_URI;
	  entryArgs.add(atts.getValue("uriStartString"));
	  entryArgs.add(atts.getValue("catalog"));

	  Debug.message(4, "delegateURI",
			atts.getValue("uriStartString"),
			atts.getValue("catalog"));
	}
      } else if (localName.equals("rewriteSystem")) {
	if (checkAttributes(atts, "systemIdStartString", "rewritePrefix")) {
	  entryType = Catalog.REWRITE_SYSTEM;
	  entryArgs.add(atts.getValue("systemIdStartString"));
	  entryArgs.add(atts.getValue("rewritePrefix"));

	  Debug.message(4, "rewriteSystem",
			atts.getValue("systemIdStartString"),
			atts.getValue("rewritePrefix"));
	}
      } else if (localName.equals("rewriteURI")) {
	if (checkAttributes(atts, "uriStartString", "rewritePrefix")) {
	  entryType = Catalog.REWRITE_URI;
	  entryArgs.add(atts.getValue("uriStartString"));
	  entryArgs.add(atts.getValue("rewritePrefix"));

	  Debug.message(4, "rewriteURI",
			atts.getValue("uriStartString"),
			atts.getValue("rewritePrefix"));
	}
      } else if (localName.equals("nextCatalog")) {
	if (checkAttributes(atts, "catalog")) {
	  entryType = Catalog.CATALOG;
	  entryArgs.add(atts.getValue("catalog"));

	  Debug.message(4, "nextCatalog", atts.getValue("catalog"));
	}
      } else if (localName.equals("public")) {
	if (checkAttributes(atts, "publicId", "uri")) {
	  entryType = Catalog.PUBLIC;
	  entryArgs.add(atts.getValue("publicId"));
	  entryArgs.add(atts.getValue("uri"));

	  Debug.message(4, "public",
			PublicId.normalize(atts.getValue("publicId")),
			atts.getValue("uri"));
	}
      } else if (localName.equals("system")) {
	if (checkAttributes(atts, "systemId", "uri")) {
	  entryType = Catalog.SYSTEM;
	  entryArgs.add(atts.getValue("systemId"));
	  entryArgs.add(atts.getValue("uri"));

	  Debug.message(4, "system",
			atts.getValue("systemId"),
			atts.getValue("uri"));
	}
      } else if (localName.equals("uri")) {
	if (checkAttributes(atts, "name", "uri")) {
	  entryType = Catalog.URI;
	  entryArgs.add(atts.getValue("name"));
	  entryArgs.add(atts.getValue("uri"));

	  Debug.message(4, "uri",
			atts.getValue("name"),
			atts.getValue("uri"));
	}
      } else if (localName.equals("catalog")) {
	// nop, start of catalog
      } else if (localName.equals("group")) {
	// nop, a group
      } else {
	// This is equivalent to an invalid catalog entry type
	Debug.message(1, "Invalid catalog entry type", localName);
      }

      if (entryType >= 0) {
	try {
	  CatalogEntry ce = new CatalogEntry(entryType, entryArgs);
	  catalog.addEntry(ce);
	} catch (CatalogException cex) {
	  if (cex.getExceptionType() == CatalogException.INVALID_ENTRY_TYPE) {
	    Debug.message(1, "Invalid catalog entry type", localName);
	  } else if (cex.getExceptionType() == CatalogException.INVALID_ENTRY) {
	    Debug.message(1, "Invalid catalog entry", localName);
	  }
	}
      }
    }

    if (namespaceURI != null && tr9401NamespaceName.equals(namespaceURI)
	&& !inExtension) {
      // This is a TR9401 Catalog entry

      if (atts.getValue("xml:base") != null) {
	String baseURI = atts.getValue("xml:base");
	entryType = Catalog.BASE;
	entryArgs.add(baseURI);
	baseURIStack.push(baseURI);

	Debug.message(4, "xml:base", baseURI);

	try {
	  CatalogEntry ce = new CatalogEntry(entryType, entryArgs);
	  catalog.addEntry(ce);
	} catch (CatalogException cex) {
	  if (cex.getExceptionType() == CatalogException.INVALID_ENTRY_TYPE) {
	    Debug.message(1, "Invalid catalog entry type", localName);
	  } else if (cex.getExceptionType() == CatalogException.INVALID_ENTRY) {
	    Debug.message(1, "Invalid catalog entry (base)", localName);
	  }
	}

	entryType = -1;
	entryArgs = new Vector();

      } else {
	baseURIStack.push(baseURIStack.peek());
      }

      if (localName.equals("doctype")) {
	entryType = catalog.DOCTYPE;
	entryArgs.add(atts.getValue("name"));
	entryArgs.add(atts.getValue("uri"));
      } else if (localName.equals("document")) {
	entryType = catalog.DOCUMENT;
	entryArgs.add(atts.getValue("uri"));
      } else if (localName.equals("dtddecl")) {
	entryType = catalog.DTDDECL;
	entryArgs.add(atts.getValue("publicId"));
	entryArgs.add(atts.getValue("uri"));
      } else if (localName.equals("entity")) {
	entryType = Catalog.ENTITY;
	entryArgs.add(atts.getValue("name"));
	entryArgs.add(atts.getValue("uri"));
      } else if (localName.equals("linktype")) {
	entryType = Catalog.LINKTYPE;
	entryArgs.add(atts.getValue("name"));
	entryArgs.add(atts.getValue("uri"));
      } else if (localName.equals("notation")) {
	entryType = Catalog.NOTATION;
	entryArgs.add(atts.getValue("name"));
	entryArgs.add(atts.getValue("uri"));
      } else if (localName.equals("sgmldecl")) {
	entryType = Catalog.SGMLDECL;
	entryArgs.add(atts.getValue("uri"));
      } else {
	// This is equivalent to an invalid catalog entry type
	Debug.message(1, "Invalid catalog entry type", localName);
      }

      if (entryType >= 0) {
	try {
	  CatalogEntry ce = new CatalogEntry(entryType, entryArgs);
	  catalog.addEntry(ce);
	} catch (CatalogException cex) {
	  if (cex.getExceptionType() == CatalogException.INVALID_ENTRY_TYPE) {
	    Debug.message(1, "Invalid catalog entry type", localName);
	  } else if (cex.getExceptionType() == CatalogException.INVALID_ENTRY) {
	    Debug.message(1, "Invalid catalog entry", localName);
	  }
	}
      }
    }
  }

  public boolean checkAttributes (Attributes atts, String attName) {
    if (atts.getValue(attName) == null) {
      Debug.message(1, "Error: required attribute " + attName + " missing.");
      return false;
    } else {
      return true;
    }
  }

  public boolean checkAttributes (Attributes atts,
				  String attName1,
				  String attName2) {
    return checkAttributes(atts, attName1)
      && checkAttributes(atts, attName2);
  }

  /** <p>The SAX <code>endElement</code> method does nothing.</p> */
  public void endElement (String namespaceURI,
			  String localName,
			  String qName)
    throws SAXException {

    int entryType = -1;
    Vector entryArgs = new Vector();

    boolean inExtension = inExtensionNamespace();

    if (namespaceURI != null
	&& !inExtension
	&& (namespaceName.equals(namespaceURI)
	    || tr9401NamespaceName.equals(namespaceURI))) {

      String popURI = (String) baseURIStack.pop();
      String baseURI = (String) baseURIStack.peek();

      if (!baseURI.equals(popURI)) {
	entryType = catalog.BASE;
	entryArgs.add(baseURI);

	Debug.message(4, "(reset) xml:base", baseURI);

	try {
	  CatalogEntry ce = new CatalogEntry(entryType, entryArgs);
	  catalog.addEntry(ce);
	} catch (CatalogException cex) {
	  if (cex.getExceptionType() == CatalogException.INVALID_ENTRY_TYPE) {
	    Debug.message(1, "Invalid catalog entry type", localName);
	  } else if (cex.getExceptionType() == CatalogException.INVALID_ENTRY) {
	    Debug.message(1, "Invalid catalog entry (rbase)", localName);
	  }
	}
      }
    }

    if (namespaceURI != null && namespaceName.equals(namespaceURI)
	&& !inExtension) {
      if (localName.equals("catalog") || localName.equals("group")) {
	String popOverride = (String) overrideStack.pop();
	String override = (String) overrideStack.peek();

	if (!override.equals(popOverride)) {
	  entryType = catalog.OVERRIDE;
	  entryArgs.add(override);
	  overrideStack.push(override);

	  Debug.message(4, "(reset) override", override);

	  try {
	    CatalogEntry ce = new CatalogEntry(entryType, entryArgs);
	    catalog.addEntry(ce);
	  } catch (CatalogException cex) {
	    if (cex.getExceptionType() == CatalogException.INVALID_ENTRY_TYPE) {
	      Debug.message(1, "Invalid catalog entry type", localName);
	    } else if (cex.getExceptionType() == CatalogException.INVALID_ENTRY) {
	      Debug.message(1, "Invalid catalog entry (roverride)", localName);
	    }
	  }
	}
      }
    }

    namespaceStack.pop();

    return;
  }

  /** <p>The SAX <code>characters</code> method does nothing.</p> */
  public void characters (char ch[], int start, int length)
    throws SAXException {
    return;
  }

  /** <p>The SAX <code>ignorableWhitespace</code> method does nothing.</p> */
  public void ignorableWhitespace (char ch[], int start, int length)
    throws SAXException {
    return;
  }

  /** <p>The SAX <code>processingInstruction</code> method does nothing.</p> */
  public void processingInstruction (String target, String data)
    throws SAXException {
    return;
  }

  /** <p>The SAX <code>skippedEntity</code> method does nothing.</p> */
  public void skippedEntity (String name)
    throws SAXException {
    return;
  }

  /** <p>The SAX <code>startPrefixMapping</code> method does nothing.</p> */
  public void startPrefixMapping(String prefix, String uri)
    throws SAXException {
    return;
  }

  /** <p>The SAX <code>endPrefixMapping</code> method does nothing.</p> */
  public void endPrefixMapping(String prefix)
    throws SAXException {
    return;
  }

}
