//-*-c++-*-
/**
 Authors: David Auber, Romain Bourqui, Patrick Mary
 from the LaBRI Visualization Team
 Email : auber@tulip-software.org
 Last modification : 13/07/2007 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifndef Tulip_GLGRAPH_H
#define Tulip_GLGRAPH_H

#ifndef NDEBUG
#define TRACE_EXEC()	
//	cerr << __PRETTY_FUNCTION__ << endl;
#else
#define TRACE_EXEC()
#endif

#if (__GNUC__ < 3)
#include <hash_map>
#else
#include <ext/hash_map>
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <stdio.h>

#include <GL/gl.h>
#include <tulip/PluginContext.h>
#include <tulip/Observable.h>
#include <tulip/Coord.h>
#include <tulip/Color.h>
#include <tulip/Graph.h>
#include <tulip/Types.h>
#include <tulip/TemplateFactory.h>
#include <tulip/Reflect.h>
#include <tulip/MutableContainer.h>
#include <tulip/hash_string.h>
#include <tulip/tulipconf.h>
#include <tulip/Coord.h>
#include <tulip/Matrix.h>
#include <tulip/GlGraphRenderingParameters.h>

#include "tulip/Camera.h"
#include "tulip/ObservableGlGraph.h"

namespace tlp {
  class LayoutProperty;
  class DoubleProperty;
  class StringProperty;
  class IntegerProperty;
  class BooleanProperty;
  class SizeProperty;
  class ColorProperty;
  class GraphProperty;
  class PropertyManager;
  class Glyph;
  class OcclusionTest;
  class PluginLoader;
  class TextRenderer;
  class Coord;
  struct GraphExporter;

  enum DrawState { DRAWNODE = 0, DRAWEDGE = 1, DRAWSELECTEDNODELABELS = 2, 
		   DRAWSELECTEDEDGELABELS = 3, DRAWNODELABELS = 4 , 
		   DRAWEDGELABELS = 5 , DRAWEND = 6};
  
  enum EdgeShape {POLYLINESHAPE = 0, BEZIERSHAPE = 4, SPLINESHAPE = 8};
  
  /**
   * This widget enables an incremental 3D drawing of a Tulip Graph.
   *  It also includes 3D navigation capabilities and selection methods.
   *  All the displaying is done with OpenGl.
   */
  class TLP_GL_SCOPE GlGraph : public GraphObserver, public ObservableGlGraph {
  
  public:
    GlGraph();
    ~GlGraph();

    //==============================================================================
    // Rendering parameters
    //==============================================================================
    /**
     * Return the rendering parameters used for rendering
     */
    const GlGraphRenderingParameters& getRenderingParameters();
    /**
     * Set the rendering parameters used for rendering
     */
    void setRenderingParameters(const GlGraphRenderingParameters &parameter);
    /**
     * Initialize the camera in order to see the entire graph on the screen 
     */
    void centerScene();
    /**
     * Draw the entire graph. Warning : the open gl context must setup before to call
     * that function.
     */
    void draw();
    //==============================================================================
    // Data centralization in order to optimize memory using
    //==============================================================================
    /** activate Texture with name filename.
     * \param filename the image to be textured
     * \return true on success, false otherwise (format not supported, file not found,
     * not enough memory, ...)
    */
    bool activateTexture(const std::string &filename);
    /*
     * Desactivate texturing
     */
    void desactivateTexture();

    //=======================================================================
    // Navigation functions: GlraphNavigate.cpp
    //=======================================================================
    /** set translation to apply to the scene
     * \sa getSceneTranslation, setSceneTranslation
     */
    void translateCamera(const int x, const int y, const int z);
    /** set rotation to apply to the scene
     *  \sa getSceneRotation, setSceneRotation
     */
    void rotateScene(const int rotx, const int roty, const int rotz);
    /** zoom on the center of the screen
     *  \param step positive: zoom in, negative: zoom out
     */
    void zoom(const int step);
    /** \brief zoom to  screen coordinates
     *  a translation is performed while zooming. The farther the point (x,y) is
     *  from the center of the screen, the higher the translation is. The direction
     *  of the translation is reversed when switching from zoom in to zoom out.
     *  \sa zoom
     */
    void zoomXY(const int step, const int x, const int y);
    //=======================================================================
    //Selection of elements on the screen
    //=======================================================================
    /** \brief select nodes and edges in a region of the screen
     *
     *  select all nodes and edges lying in the area of the screen of given width and height,
     *  and with its upper-left corner at (x,y)
     *  \param sNode filled by the method with the nodes found in the region
     *  \param sEdge filled by the method with the edges found in the region
     */
    void doSelect(const int x, const int y, 
		  const int width, const int height, 
		  std::vector<node> &sNode, std::vector<edge> &sEdge);
    /** \brief select a node or edge at a point
     *  select either a node or edge at point (x,y)
     *  \param type tells what has been found: NODE, EDGE
     *  \return true if something has been found, false otherwise
     */
    bool doSelect(const int x, const int y, 
		  tlp::ElementType &type, 
		  node &, edge &);
    /** \brief select nodes in a region of the screen
     *  \param selected the nodes found
     *  \return true if at least one node has been found
     */
    bool doNodeSelect(const int x, const int y, 
		      const int width, const int height, 
		      std::vector<node> &selected, const bool ordered = true);
    /** select edges at a given point, ordered from the closest to the farthest
     *  \param selected the edges found
     *  \return true if at least one edge has been found
     */
    bool doEdgeSelect(const int x, const int y, 
		      const int width, const int height, 
		      std::vector<edge> &selected, const bool ordered = true);
    GLuint (*selectBuf)[4];
    void initDoSelect(const GLint x, const GLint y, 
		      GLint w, GLint h, 
		      const unsigned int nbPickableElements);
    void endSelect();
    //=======================================================================
    //Tools
    //=======================================================================
    //FIXME: mode outputEPS outside of GlGraph, return true/false if success/failure
    bool outputEPS(int size, int doSort, const char *filename);
    // SVG output of a GlGraph, return true/false if success/failure
    bool outputSVG(int size, const char* filename);
    /**
     * Compute coordinate from the 2D screen space to the 
     * 3D space.
     */
    Coord screenTo3DWorld(const Coord&);
    /**
     * Compute coordinate from the 3D space to the
     * 2D screen space.
     */
    Coord worldTo2DScreen(const Coord&);
    /** 
     *  Take a snapshot of the Open GL windows
     *  \return an array of dimension width*height*3 char (8bits per color RGB).
     *   The pointer has to be freed after (with free, not delete)
     **/
    unsigned char* getImage();
    /*
     * Return the graph used for the rendering, that function is
     * given for convenience, one should use setRendereignParameters
     * to change the graph used for rendering.
     */
    tlp::Graph* getGraph() const;

    
    //=========================================
    // Information on available plug-ins, edges style...
    //=========================================
    static std::string glyphName(int id);
    static int  glyphId(std::string name);
    static void loadPlugins(PluginLoader *plug=0);
    static std::string edgeShapeName(int id);
    static int edgeShapeId(std::string name);
    static const int edgeShapesCount;
    static int edgeShapeIds[];
    static std::string labelPositionName(int id);
    static int labelPositionId(std::string name);


    //FIXME: made public to allow plugins to access those properties
  public:
    ColorProperty    *elementColor;
    ColorProperty    *elementLabelColor;
    SizeProperty     *elementSize;
    IntegerProperty  *elementLabelPosition;

    IntegerProperty  *elementShape;
    DoubleProperty   *elementRotation;
    BooleanProperty  *elementSelected;
    StringProperty   *elementLabel;
    LayoutProperty   *elementLayout;
    GraphProperty    *elementGraph;
    StringProperty   *elementTexture;
    ColorProperty    *elementBorderColor;
    DoubleProperty   *elementBorderWidth;


  protected:
    tlp::GlGraphRenderingParameters _renderingParameters;

  private:
    static void loadGlyphPlugins();

    void addNode (Graph *, const node);
    void addEdge (Graph *, const edge);
    void delNode (Graph *, const node);
    void delEdge (Graph *, const edge);
    void destroy (Graph *);
    void drawPixmapFont(const std::string &str, const Color &col, const Coord &position,
			int relPos, bool selected, float width);
    void drawEdge(const Coord &startNode, const Coord &finalNode,
		  const Coord &startPoint, const Coord &endPoint, const LineType::RealType &bends,
		  const Color &startColor, const Color &endColor, const Size &size, int shape);
    void drawEdge(edge ite);
    unsigned int drawEdges(unsigned int , Iterator<edge> *, unsigned int = 0);
    void drawNode(node itv,unsigned int depth);
    unsigned int drawNodes(unsigned int , Iterator<node> *, unsigned int = 0);
    unsigned int drawNodeLabels(unsigned int , Iterator<node> *, bool, unsigned int =0);
    void drawNodeLabel(node itn, bool, unsigned int);
    void drawEdgeLabel(edge e, bool);
    unsigned int drawEdgeLabels(unsigned int number, Iterator<edge> *itE, bool, unsigned int =0);
    void drawMetaNode(node n,unsigned int depth, bool = false, bool = false);
    tlp::Matrix<float, 4> makeArrowMatrix(const Coord& , const Coord&);
    void initProxies();
    void deleteIterators();
    void initIterators();
    void makeNodeSelect(const int);
    void makeEdgeSelect(const int);
    void initProjection(bool reset = true);
    void initModelView();
    void initLights();
    void initGlParameter();
    void buildOrderedList();
    void buildDisplayLists();
    void deleteDisplayLists();
    // output SVG stuff
    void initMapsSVG(Graph*, GraphExporter*);
    void exportHeaderSVG(FILE*, GraphExporter*);
    void exportGraphSVG(FILE*, GLfloat*, Graph*, GraphExporter*, std::string);
    void exportBufferSVG(FILE*, GLfloat*, GLint, GraphExporter *, std::string);

    //we never have to get a copy of that GlGraph.
    //GlGraph objects should be only passed by reference or pointer
    GlGraph(const GlGraph &);

    Graph *_graph;

    stdext::hash_map<std::string, GLuint > texturesMap; //FIXME: make it static?
    tlp::Matrix<float, 4> modelviewMatrix;
    tlp::Matrix<float, 4> projectionMatrix;
    tlp::Matrix<float, 4> transformMatrix;

    //predefined display list
    GLuint selectionDL;
    GLuint arrowDL;
    //glyphs
    MutableContainer<Glyph *> glyphs;
    Iterator<node>* drawNodesIterator;
    Iterator<node>* drawLabelsIterator;
    Iterator<node>* drawSelectedLabelsIterator;
    Iterator<edge>* drawEdgesIterator;
    Iterator<edge>* drawEdgeLabelsIterator;
    Iterator<edge>* drawEdgeSelectedLabelsIterator;


    std::list<node> orderedNode;
    std::list<edge> orderedEdge;

    MutableContainer<bool> nodesRenderedAsPoint;

    //Occlusion testing
    OcclusionTest *occlusionTest;
    //Font renderer
    TextRenderer * fontRenderer;
  };
 
}

#endif // Tulip_GLGRAPH_H
