
INCLUDE(PackageArchProcessPackagesAndDirsLists)

INCLUDE(AddOptionAndDefine)
INCLUDE(AdvancedOption)
INCLUDE(AdvancedSet)
INCLUDE(AppendStringVar)
INCLUDE(CMakeBuildTypesList)
INCLUDE(FindListElement)
INCLUDE(GlobalNullSet)
INCLUDE(PrintNonemptyVar)
INCLUDE(PrintVar)
INCLUDE(RemoveGlobalDuplicates)


#
# Function that creates error message about missing/misspelled package.
#

FUNCTION(PACKAGE_ARCH_ABORT_ON_MISSING_PACKAGE   DEP_PKG  PACKAGE_NAME  DEP_PKG_LIST_NAME)
  MULTILINE_SET(ERRMSG
    "Error, the package '${DEP_PKG}' is listed as a dependency of the package"
    " '${PACKAGE_NAME}' in the list '${DEP_PKG_LIST_NAME}' but the package"
    " '${DEP_PKG}' is either not defined or is listed later in the package order."
    "  Check the spelling of '${DEP_PKG}' or see how it is listed in"
    " ${PROJECT_NAME}_PACKAGES_AND_DIRS_AND_CLASSIFICATIONS in relationship to"
    " '${PACKAGE_NAME}'.")
  MESSAGE(FATAL_ERROR ${ERRMSG})
ENDFUNCTION()


#
# Function that helps to set up backward package dependency lists
#

FUNCTION(PACKAGE_ARCH_SET_DEP_PACKAGES PACKAGE_NAME LIST_TYPE)

  #MESSAGE("\nPACKAGE_ARCH_SET_DEP_PACKAGES: ${PACKAGE_NAME} ${LIST_TYPE}")

  SET(PACKAGE_DEPS_LIST)

  FOREACH(DEP_PKG ${${LIST_TYPE}})
    FIND_LIST_ELEMENT( ${PROJECT_NAME}_PACKAGES ${DEP_PKG} DEP_PKG_FOUND)
    #PRINT_VAR(DEP_PKG_FOUND)
    IF (DEP_PKG_FOUND)
      LIST(APPEND PACKAGE_DEPS_LIST ${DEP_PKG})
    ELSE()
      IF (${PROJECT_NAME}_ASSERT_MISSING_PACKAGES)
        PACKAGE_ARCH_ABORT_ON_MISSING_PACKAGE(${DEP_PKG} ${ PACKAGE_NAME} ${DEP_PKG_LIST_NAME})
      ELSE()
        IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
          MESSAGE(
            "\n***"
            "\n*** WARNING: The package ${DEP_PKG} which is a dependent package of"
              " ${PACKAGE_NAME} being ignored because ${DEP_PKG} is missing!"
            "\n***\n" )
        ENDIF()
      ENDIF()
    ENDIF()
  ENDFOREACH()

  #PRINT_VAR(PACKAGE_DEPS_LIST)

  GLOBAL_SET(${PACKAGE_NAME}_${LIST_TYPE} ${PACKAGE_DEPS_LIST})

ENDFUNCTION()


#
# Macro that helps to set up forward package dependency lists
#

FUNCTION(PACKAGE_ARCH_APPEND_FORWARD_DEP_PACKAGES PACKAGE_NAME LIST_TYPE)
 
  #MESSAGE("\nPACKAGE_ARCH_APPEND_FORWARD_DEP_PACKAGES: ${PACKAGE_NAME} ${LIST_TYPE}")

  SET(DEP_PKG_LIST_NAME "${PACKAGE_NAME}_${LIST_TYPE}")

  #MESSAGE("DEP_PKG_LIST_NAME = ${DEP_PKG_LIST_NAME}")
  #MESSAGE("${DEP_PKG_LIST_NAME} = ${${DEP_PKG_LIST_NAME}}")

  ASSERT_DEFINED(${PROJECT_NAME}_ASSERT_MISSING_PACKAGES)
  FOREACH(DEP_PKG ${${DEP_PKG_LIST_NAME}})
    #MESSAGE("DEP_PKG = ${DEP_PKG}")
    SET(FWD_DEP_PKG_LIST_NAME "${DEP_PKG}_FORWARD_${LIST_TYPE}")
    #MESSAGE("FWD_DEP_PKG_LIST_NAME = ${FWD_DEP_PKG_LIST_NAME}")
    IF (NOT DEFINED ${FWD_DEP_PKG_LIST_NAME})
      IF (${PROJECT_NAME}_ASSERT_MISSING_PACKAGES)
        PACKAGE_ARCH_ABORT_ON_MISSING_PACKAGE(${DEP_PKG} ${ PACKAGE_NAME} ${DEP_PKG_LIST_NAME})
      ELSE()
        IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
          MESSAGE(
            "\n***"
            "\n*** WARNING: The package ${DEP_PKG} has forward dependent package"
              " ${PACKAGE_NAME}, but that dependency is being ignored because the package"
              " ${DEP_PKG} is missing!"
            "\n***\n" )
        ENDIF()
      ENDIF()
    ELSE()
      SET(${FWD_DEP_PKG_LIST_NAME} ${${FWD_DEP_PKG_LIST_NAME}} ${PACKAGE_NAME} PARENT_SCOPE)
    ENDIF()
  ENDFOREACH()

ENDFUNCTION()


#
# Macro that reads in package dependencies for a package and sets forward
# dependencies for packages already read in.
#
# Modifies the global variables:
#
#   ${PACKAGE_NAME}_LIB_REQUIRED_DEP_PACKAGES
#   ${PACKAGE_NAME}_LIB_OPTIONAL_DEP_PACKAGES
#   ${PACKAGE_NAME}_TEST_REQUIRED_DEP_PACKAGES
#   ${PACKAGE_NAME}_TEST_OPTIONAL_DEP_PACKAGES
#   ${PACKAGE_NAME}_FORWARD_LIB_REQUIRED_DEP_PACKAGES
#   ${PACKAGE_NAME}_FORWARD_LIB_OPTIONAL_DEP_PACKAGES
#   ${PACKAGE_NAME}_FORWARD_TEST_REQUIRED_DEP_PACKAGES
#   ${PACKAGE_NAME}_FORWARD_TEST_OPTIONAL_DEP_PACKAGES
#

MACRO(PACKAGE_ARCH_READ_PACKAGE_DEPENDENCIES PACKAGE_NAME PACKAGE_DIR)

  SET(${PACKAGE_NAME}_FORWARD_LIB_REQUIRED_DEP_PACKAGES "")
  SET(${PACKAGE_NAME}_FORWARD_LIB_OPTIONAL_DEP_PACKAGES "")
  SET(${PACKAGE_NAME}_FORWARD_TEST_REQUIRED_DEP_PACKAGES "")
  SET(${PACKAGE_NAME}_FORWARD_TEST_OPTIONAL_DEP_PACKAGES "")

  DECLARE_UNDEFINED(LIB_REQUIRED_DEP_PACKAGES)
  DECLARE_UNDEFINED(LIB_OPTIONAL_DEP_PACKAGES)
  DECLARE_UNDEFINED(TEST_REQUIRED_DEP_PACKAGES)
  DECLARE_UNDEFINED(TEST_OPTIONAL_DEP_PACKAGES)

  DECLARE_UNDEFINED(LIB_REQUIRED_DEP_TPLS "")
  DECLARE_UNDEFINED(LIB_OPTIONAL_DEP_TPLS "")
  DECLARE_UNDEFINED(TEST_REQUIRED_DEP_TPLS "")
  DECLARE_UNDEFINED(TEST_OPTIONAL_DEP_TPLS "")

  INCLUDE(packages/${PACKAGE_DIR}/cmake/Dependencies)

  ASSERT_DEFINED_PACKAGE_VAR(LIB_REQUIRED_DEP_PACKAGES ${PACKAGE_NAME})
  ASSERT_DEFINED_PACKAGE_VAR(LIB_OPTIONAL_DEP_PACKAGES ${PACKAGE_NAME})
  ASSERT_DEFINED_PACKAGE_VAR(TEST_REQUIRED_DEP_PACKAGES ${PACKAGE_NAME})
  ASSERT_DEFINED_PACKAGE_VAR(TEST_OPTIONAL_DEP_PACKAGES ${PACKAGE_NAME})

  ASSERT_DEFINED_PACKAGE_VAR(LIB_REQUIRED_DEP_TPLS ${PACKAGE_NAME})
  ASSERT_DEFINED_PACKAGE_VAR(LIB_OPTIONAL_DEP_TPLS ${PACKAGE_NAME})
  ASSERT_DEFINED_PACKAGE_VAR(TEST_REQUIRED_DEP_TPLS ${PACKAGE_NAME})
  ASSERT_DEFINED_PACKAGE_VAR(TEST_OPTIONAL_DEP_TPLS ${PACKAGE_NAME})

  PACKAGE_ARCH_SET_DEP_PACKAGES(${PACKAGE_NAME} LIB_REQUIRED_DEP_PACKAGES)
  PACKAGE_ARCH_SET_DEP_PACKAGES(${PACKAGE_NAME} LIB_OPTIONAL_DEP_PACKAGES)
  PACKAGE_ARCH_SET_DEP_PACKAGES(${PACKAGE_NAME} TEST_REQUIRED_DEP_PACKAGES)
  PACKAGE_ARCH_SET_DEP_PACKAGES(${PACKAGE_NAME} TEST_OPTIONAL_DEP_PACKAGES)

  SET(${PACKAGE_NAME}_LIB_REQUIRED_DEP_TPLS ${LIB_REQUIRED_DEP_TPLS})
  SET(${PACKAGE_NAME}_LIB_OPTIONAL_DEP_TPLS ${LIB_OPTIONAL_DEP_TPLS})
  SET(${PACKAGE_NAME}_TEST_REQUIRED_DEP_TPLS ${TEST_REQUIRED_DEP_TPLS})
  SET(${PACKAGE_NAME}_TEST_OPTIONAL_DEP_TPLS ${TEST_OPTIONAL_DEP_TPLS})

  PACKAGE_ARCH_APPEND_FORWARD_DEP_PACKAGES(${PACKAGE_NAME} LIB_REQUIRED_DEP_PACKAGES)
  PACKAGE_ARCH_APPEND_FORWARD_DEP_PACKAGES(${PACKAGE_NAME} LIB_OPTIONAL_DEP_PACKAGES)
  PACKAGE_ARCH_APPEND_FORWARD_DEP_PACKAGES(${PACKAGE_NAME} TEST_REQUIRED_DEP_PACKAGES)
  PACKAGE_ARCH_APPEND_FORWARD_DEP_PACKAGES(${PACKAGE_NAME} TEST_OPTIONAL_DEP_PACKAGES)

ENDMACRO()


#
# Macro that reads all the package dependencies
#

MACRO(PACKAGE_ARCH_READ_ALL_PACKAGE_DEPENDENCIES)
  
  FOREACH(PACKAGE_IDX RANGE ${${PROJECT_NAME}_LAST_PACKAGE_IDX})
    LIST(GET ${PROJECT_NAME}_PACKAGES ${PACKAGE_IDX} PACKAGE)
    LIST(GET ${PROJECT_NAME}_PACKAGE_DIRS ${PACKAGE_IDX} PACKAGE_DIR)
    PACKAGE_ARCH_READ_PACKAGE_DEPENDENCIES(${PACKAGE} ${PACKAGE_DIR})
    #PACKAGE_ARCH_ADD_OPTIONAL_PACKAGE_ENABLES(${PACKAGE})
  ENDFOREACH()
  
  ADVANCED_OPTION(${PROJECT_NAME}_DUMP_PACKAGE_DEPENDENCIES
    "Dump the package dependency information." OFF)
  
  IF (${PROJECT_NAME}_VERBOSE_CONFIGURE OR ${PROJECT_NAME}_DUMP_PACKAGE_DEPENDENCIES)
    MESSAGE("")
    MESSAGE("Printing package dependencies ...")
    MESSAGE("")
    PRINT_VAR(${PROJECT_NAME}_PACKAGES)
    MESSAGE("")
    FOREACH(PACKAGE ${${PROJECT_NAME}_PACKAGES})
      PACKAGE_ARCH_PRINT_PACKAGE_DEPENDENCIES(${PACKAGE})
      MESSAGE("")
    ENDFOREACH()
  ENDIF()

ENDMACRO()


#
# Macro that gets the current list of enables packages
#
# Accesses the global varaibles:
#
#   ${PROJECT_NAME}_PACKAGES
#   ${PROJECT_NAME}_ENABLE_${PACKAGE}
#
# where ${PACKAGE} is every package in ${LISTVAR}
#

MACRO(PACKAGE_ARCH_GET_ENABLED_LIST
  LISTVAR ENABLED_PREFIX ENABLED_FLAG INCLUDE_EMPTY
  ENABLED_LIST_OUT NUM_ENABLED_OUT
  )
  SET(${ENABLED_LIST_OUT} "")
  SET(${NUM_ENABLED_OUT} 0)
  FOREACH(ENTITY ${${LISTVAR}})
    SET(ENTITY_NAME ${ENABLED_PREFIX}_ENABLE_${ENTITY})
    ASSERT_DEFINED(${ENTITY_NAME})
    SET(INCLUDE_ENTITY FALSE)
    IF ("${ENTITY_NAME}" STREQUAL "${ENABLED_FLAG}")
      SET(INCLUDE_ENTITY TRUE)
    ELSEIF(INCLUDE_EMPTY AND "${ENTITY_NAME}" STREQUAL "")
      SET(INCLUDE_ENTITY TRUE)
    ENDIF()
    IF (INCLUDE_ENTITY)
      SET(${ENABLED_LIST_OUT} "${${ENABLED_LIST_OUT}} ${ENTITY}")
      MATH(EXPR ${NUM_ENABLED_OUT} "${${NUM_ENABLED_OUT}}+1")
    ENDIF()
  ENDFOREACH()
ENDMACRO()


#
# Function that prints the current set of enabled/disabled packages
#

FUNCTION(PACKAGE_ARCH_PRINT_ENABLED_PACKAGE_LIST DOCSTRING ENABLED_FLAG INCLUDE_EMPTY)
  PACKAGE_ARCH_GET_ENABLED_LIST( ${PROJECT_NAME}_PACKAGES ${PROJECT_NAME} ${ENABLED_FLAG}
    ${INCLUDE_EMPTY} ${PROJECT_NAME}_ENABLED_PACKAGES NUM_ENABLED)
  MESSAGE("${DOCSTRING}: ${${PROJECT_NAME}_ENABLED_PACKAGES} ${NUM_ENABLED}")
ENDFUNCTION()


#
# Function that prints the current set of enabled/disabled TPLs
#

FUNCTION(PACKAGE_ARCH_PRINT_ENABLED_TPL_LIST DOCSTRING ENABLED_FLAG INCLUDE_EMPTY)
  PACKAGE_ARCH_GET_ENABLED_LIST( ${PROJECT_NAME}_TPLS TPL ${ENABLED_FLAG}
    ${INCLUDE_EMPTY} ${PROJECT_NAME}_ENABLED_PACKAGES NUM_ENABLED)
  MESSAGE("${DOCSTRING}: ${${PROJECT_NAME}_ENABLED_PACKAGES} ${NUM_ENABLED}")
ENDFUNCTION()


#
# Function that sets a varaible to DECLARED-UNDEFINED
#

FUNCTION(DECLARE_UNDEFINED VAR)
  SET(${VAR} DECLARED-UNDEFINED PARENT_SCOPE)
ENDFUNCTION()


#
# Function that asserts that a package dependency variable is defined
# correctly
#

FUNCTION(ASSERT_DEFINED_PACKAGE_VAR PACKAGE_VAR PACKAGE_NAME)
  IF (${PACKAGE_VAR} STREQUAL DECLARED-UNDEFINED)
    MESSAGE(FATAL_ERROR
      "Error, the package variable ${PACKAGE_VAR} was not defined correctly for package ${PACKAGE_NAME}!"
      )
  ENDIF()
ENDFUNCTION()


#
# Private helper macros
#


FUNCTION(PACKAGE_ARCH_PRIVATE_PRINT_DISABLE
  ENABLE_BEING_DISABLED_VAR_NAME  PACKAGE_WITH_SOMETHING_BEING_DISABLED
  DEP_TYPE_STR  THING_DISALBED_TYPE  THING_DISABLED_NAME
  )
  IF (${ENABLE_BEING_DISABLED_VAR_NAME})
    MESSAGE(
      "\n***"
      "\n*** WARNING: Setting ${ENABLE_BEING_DISABLED_VAR_NAME}=OFF"
      " which was 'ON' because ${PACKAGE_WITH_SOMETHING_BEING_DISABLED} has"
      " a required ${DEP_TYPE_STR} dependence on disabled"
      " ${THING_DISALBED_TYPE} ${THING_DISABLED_NAME}!"
      "\n***\n"
      )
  ELSE()
    MESSAGE("-- "
      "Setting ${ENABLE_BEING_DISABLED_VAR_NAME}=OFF"
      " because ${PACKAGE_WITH_SOMETHING_BEING_DISABLED} has a required ${DEP_TYPE_STR}"
      " dependence on disabled ${THING_DISALBED_TYPE} ${THING_DISABLED_NAME}")
  ENDIF()
ENDFUNCTION()


MACRO(PACKAGE_ARCH_PRIVATE_DISABLE_TPL_REQUIRED_PACKAGE_ENABLE
  TPL_NAME PACKAGE_NAME LIBRARY_DEP
  )

  #MESSAGE("PACKAGE_ARCH_PRIVATE_DISABLE_TPL_REQUIRED_PACKAGE_ENABLE"
  #  " ${FORWARD_DEP_PACKAGE_NAME} ${LIBRARY_DEP}")  

  # Only turn off PACKAGE_NAME libraries and test/eamples if it
  # is currently enabled or could be enabled.

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}
     OR ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME} STREQUAL ""
     )

    IF ("${LIBRARY_DEP}" STREQUAL "TRUE")

      PACKAGE_ARCH_PRIVATE_PRINT_DISABLE(
        ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME} ${PACKAGE_NAME} "library"
        "TPL" ${TPL_NAME}
        )

      SET(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME} OFF)

    ELSE()

      SET(DEP_TYPE_STR "test/example")

      ASSERT_DEFINED(${PACKAGE_NAME}_ENABLE_TESTS)
      IF (${PACKAGE_NAME}_ENABLE_TESTS
        OR ${PACKAGE_NAME}_ENABLE_TESTS STREQUAL ""
        )
        PACKAGE_ARCH_PRIVATE_PRINT_DISABLE(
          ${PACKAGE_NAME}_ENABLE_TESTS ${PACKAGE_NAME} "${DEP_TYPE_STR}"
          "TPL" ${TPL_NAME}
          )
        SET(${PACKAGE_NAME}_ENABLE_TESTS OFF)
      ENDIF()

      ASSERT_DEFINED(${PACKAGE_NAME}_ENABLE_EXAMPLES)
      IF (${PACKAGE_NAME}_ENABLE_EXAMPLES
        OR ${PACKAGE_NAME}_ENABLE_EXAMPLES STREQUAL ""
        )
        PACKAGE_ARCH_PRIVATE_PRINT_DISABLE(
          ${PACKAGE_NAME}_ENABLE_EXAMPLES ${PACKAGE_NAME} "${DEP_TYPE_STR}"
          "TPL" ${TPL_NAME}
          )
        SET(${PACKAGE_NAME}_ENABLE_EXAMPLES OFF)
      ENDIF()

    ENDIF()

  ENDIF()
  
ENDMACRO()


FUNCTION(PACKAGE_ARCH_PRIVATE_PRINT_DISABLE_REQUIRED_PACKAGE_ENABLE
  PACKAGE_NAME  PACKAGE_ENABLE_SOMETHING_VAR_NAME  FORWARD_DEP_PACKAGE_NAME
  DEP_TYPE_STR
  )
  PACKAGE_ARCH_PRIVATE_PRINT_DISABLE(
    ${PACKAGE_ENABLE_SOMETHING_VAR_NAME} ${FORWARD_DEP_PACKAGE_NAME}
    "${DEP_TYPE_STR}" "package" ${PACKAGE_NAME} )
ENDFUNCTION()


MACRO(PACKAGE_ARCH_PRIVATE_DISABLE_REQUIRED_PACKAGE_ENABLES
  FORWARD_DEP_PACKAGE_NAME PACKAGE_NAME LIBRARY_DEP
  )

  #MESSAGE("PACKAGE_ARCH_PRIVATE_DISABLE_REQUIRED_PACKAGE_ENABLES"
  #  " ${FORWARD_DEP_PACKAGE_NAME} ${LIBRARY_DEP}")  

  # Only turn off FORWARD_DEP_PACKAGE libraries and test/eamples if it
  # is currently enabled or could be enabled

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME})
  IF (${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME}
     OR ${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME} STREQUAL ""
     )

    IF ("${LIBRARY_DEP}" STREQUAL "TRUE")

      PACKAGE_ARCH_PRIVATE_PRINT_DISABLE_REQUIRED_PACKAGE_ENABLE(
        ${PACKAGE_NAME} ${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME}
        ${FORWARD_DEP_PACKAGE_NAME} "library" )

      SET(${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME} OFF)

    ELSE()

      SET(DEP_TYPE_STR "test/example")

      #ASSERT_DEFINED(${FORWARD_DEP_PACKAGE_NAME}_ENABLE_TESTS)
      IF (${FORWARD_DEP_PACKAGE_NAME}_ENABLE_TESTS
        OR ${FORWARD_DEP_PACKAGE_NAME}_ENABLE_TESTS STREQUAL ""
        )
        PACKAGE_ARCH_PRIVATE_PRINT_DISABLE_REQUIRED_PACKAGE_ENABLE(
          ${PACKAGE_NAME} ${FORWARD_DEP_PACKAGE_NAME}_ENABLE_TESTS
          ${FORWARD_DEP_PACKAGE_NAME} "${DEP_TYPE_STR}" )
        SET(${FORWARD_DEP_PACKAGE_NAME}_ENABLE_TESTS OFF)
      ENDIF()
    
      #ASSERT_DEFINED(${FORWARD_DEP_PACKAGE_NAME}_ENABLE_EXAMPLES)
      IF (${FORWARD_DEP_PACKAGE_NAME}_ENABLE_EXAMPLES
        OR ${FORWARD_DEP_PACKAGE_NAME}_ENABLE_EXAMPLES STREQUAL ""
        )
        PACKAGE_ARCH_PRIVATE_PRINT_DISABLE_REQUIRED_PACKAGE_ENABLE(
          ${PACKAGE_NAME} ${FORWARD_DEP_PACKAGE_NAME}_ENABLE_EXAMPLES
          ${FORWARD_DEP_PACKAGE_NAME} "${DEP_TYPE_STR}" )
        SET(${FORWARD_DEP_PACKAGE_NAME}_ENABLE_EXAMPLES OFF)
      ENDIF()

    ENDIF()

  ENDIF()
  
ENDMACRO()


MACRO(PACKAGE_ARCH_PRIVATE_DISABLE_OPTIONAL_PACKAGE_ENABLES
  FORWARD_DEP_PACKAGE_NAME PACKAGE_NAME
  )

  #MESSAGE("PACKAGE_ARCH_PRIVATE_DISABLE_OPTIONAL_PACKAGE_ENABLES"
  #  " ${FORWARD_DEP_PACKAGE_NAME} ${PACKAGE_NAME}")
  #MESSAGE("-- " "${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME} = ${${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME}}")

  #ASSERT_DEFINED(${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME})
  IF (${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME} OR "${${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME}}" STREQUAL "")
    # Always disable the conditional enable but only print the message if the package is enabled.
    #MESSAGE("--  Disasble ${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME} ...")
    IF (${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME})
      MESSAGE("-- "
        "Setting ${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME}=OFF"
        " because ${FORWARD_DEP_PACKAGE_NAME} has an optional library dependence"
        " on disabled package ${PACKAGE_NAME}")
    ENDIF()
    SET(${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME} OFF)
  ENDIF()

ENDMACRO()



#
# Function that disables all packages and optional package TPL support
# for a disabled TPL.
#

MACRO(PACKAGE_ARCH_DISABLE_TPL_DEP_PACKAGES TPL_NAME)

  #MESSAGE("PACKAGE_ARCH_DISABLE_TPL_DEP_PACKAGES: ${TPL_NAME}")

  #PRINT_VAR(TPL_ENABLE_${TPL_NAME})
  IF ("${TPL_ENABLE_${TPL_NAME}}" STREQUAL "OFF")

    FOREACH(PACKAGE_NAME ${${PROJECT_NAME}_PACKAGES})

      #ASSERT_DEFINED(${PACKAGE_NAME}_LIB_REQUIRED_DEP_TPLS)
      FOREACH(CURR_TPL_NAME ${${PACKAGE_NAME}_LIB_REQUIRED_DEP_TPLS})
        IF (CURR_TPL_NAME STREQUAL ${TPL_NAME})
          PACKAGE_ARCH_PRIVATE_DISABLE_TPL_REQUIRED_PACKAGE_ENABLE(
            ${TPL_NAME} ${PACKAGE_NAME} TRUE )
        ENDIF()
      ENDFOREACH()

      #ASSERT_DEFINED(${PACKAGE_NAME}_TEST_REQUIRED_DEP_TPLS)
      FOREACH(CURR_TPL_NAME ${${PACKAGE_NAME}_TEST_REQUIRED_DEP_TPLS})
        IF (CURR_TPL_NAME STREQUAL ${TPL_NAME})
          PACKAGE_ARCH_PRIVATE_DISABLE_TPL_REQUIRED_PACKAGE_ENABLE(
            ${TPL_NAME} ${PACKAGE_NAME} FALSE )
        ENDIF()
      ENDFOREACH()

    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Function that disables all forward packages that depend on the given packages
#

MACRO(PACKAGE_ARCH_DISABLE_FORWARD_REQUIRED_DEP_PACKAGES PACKAGE_NAME)

  #MESSAGE("PACKAGE_ARCH_DISABLE_FORWARD_REQUIRED_DEP_PACKAGES: ${PACKAGE_NAME}")

  IF ("${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}}" STREQUAL "OFF")

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_LIB_REQUIRED_DEP_PACKAGES})
      PACKAGE_ARCH_PRIVATE_DISABLE_REQUIRED_PACKAGE_ENABLES(${FWD_DEP_PKG} ${PACKAGE_NAME} TRUE)
    ENDFOREACH()

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_LIB_OPTIONAL_DEP_PACKAGES})
      PACKAGE_ARCH_PRIVATE_DISABLE_OPTIONAL_PACKAGE_ENABLES(${FWD_DEP_PKG} ${PACKAGE_NAME})
    ENDFOREACH()

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_TEST_REQUIRED_DEP_PACKAGES})
      PACKAGE_ARCH_PRIVATE_DISABLE_REQUIRED_PACKAGE_ENABLES(${FWD_DEP_PKG} ${PACKAGE_NAME} FALSE)
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Macro that prints out dependencies for a package
#
# Does not modify the global state.
#

MACRO(PACKAGE_ARCH_PRINT_PACKAGE_DEPENDENCIES PACKAGE_NAME)

  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_LIB_REQUIRED_DEP_PACKAGES)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_LIB_OPTIONAL_DEP_PACKAGES)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_TEST_REQUIRED_DEP_PACKAGES)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_TEST_OPTIONAL_DEP_PACKAGES)

  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_FORWARD_LIB_REQUIRED_DEP_PACKAGES)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_FORWARD_LIB_OPTIONAL_DEP_PACKAGES)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_FORWARD_TEST_REQUIRED_DEP_PACKAGES)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_FORWARD_TEST_OPTIONAL_DEP_PACKAGES)

  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_LIB_REQUIRED_DEP_TPLS)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_LIB_OPTIONAL_DEP_TPLS)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_TEST_REQUIRED_DEP_TPLS)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_TEST_OPTIONAL_DEP_TPLS)

ENDMACRO()


#
# Private helper macros
#


MACRO(PACKAGE_ARCH_PRIVATE_ADD_OPTIONAL_PACKAGE_ENABLE PACKAGE_NAME  OPTIONAL_DEP_PACKAGE
  TYPE  SET_AS_CACHE_IN
  )

  #MESSAGE("\nPACKAGE_ARCH_PRIVATE_ADD_OPTIONAL_PACKAGE_ENABLE: ${PACKAGE_NAME} ${OPTIONAL_DEP_PACKAGE}")

  IF (SET_AS_CACHE_IN)

    MULTILINE_SET(DOCSTR
      "Enable optional ${TYPE} support in the package ${PACKAGE_NAME}"
      " for the package ${OPTIONAL_DEP_PACKAGE}."
      "  Set to 'ON', 'OFF', or leave empty"
      " to allow for other logic to decide."
      )
  
    SET_CACHE_ON_OFF_EMPTY( ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE} ""
      ${DOCSTR} )

  ELSE()

    IF (NOT DEFINED ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE})
      SET( ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE} "" )
    ENDIF()

  ENDIF()

ENDMACRO()


MACRO(PACKAGE_ARCH_PRIVATE_ADD_OPTIONAL_TPL_ENABLE PACKAGE_NAME OPTIONAL_DEP_TPL
  TYPE  SET_AS_CACHE_IN )

  IF (SET_AS_CACHE_IN)
  
    MULTILINE_SET(DOCSTR
      "Enable optional ${TYPE} support in the package ${PACKAGE_NAME}"
      " for the TPL ${OPTIONAL_DEP_TPL}."
      "  Set to 'ON', 'OFF', or leave empty"
      " to allow for other logic to decide."
      )
    
    SET_CACHE_ON_OFF_EMPTY( ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL} ""
      ${DOCSTR} )

  ELSE()

    IF (NOT DEFINED ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL})
      SET( ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL} "" )
    ENDIF()

  ENDIF()

ENDMACRO()


#
# Macro that enables optional package interdependencies
#

MACRO(PACKAGE_ARCH_ADD_OPTIONAL_PACKAGE_ENABLES PACKAGE_NAME)

  #MESSAGE("\nPACKAGE_ARCH_ADD_OPTIONAL_PACKAGE_ENABLES: ${PACKAGE_NAME}")

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
  SET(SET_AS_CACHE ${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}})

  IF (SET_AS_CACHE)

    MULTILINE_SET(DOCSTR
      "Build tests for the package ${PACKAGE_NAME}.  Set to 'ON', 'OFF', or leave empty ''"
       " to allow for other logic to decide."
       )
    SET_CACHE_ON_OFF_EMPTY( ${PACKAGE_NAME}_ENABLE_TESTS "" ${DOCSTR} )
    
    MULTILINE_SET(DOCSTR
      "Build examples for the package ${PACKAGE_NAME}.  Set to 'ON', 'OFF', or leave empty ''"
       " to allow for other logic to decide."
       )
    SET_CACHE_ON_OFF_EMPTY( ${PACKAGE_NAME}_ENABLE_EXAMPLES "" ${DOCSTR} )

  ELSE()

    IF (NOT DEFINED ${PACKAGE_NAME}_ENABLE_TESTS)
      SET( ${PACKAGE_NAME}_ENABLE_TESTS "" )
    ENDIF()
    IF (NOT DEFINED ${PACKAGE_NAME}_ENABLE_EXAMPLES)
      SET( ${PACKAGE_NAME}_ENABLE_EXAMPLES "" )
    ENDIF()

  ENDIF() 
 
  FOREACH(OPTIONAL_DEP_PACKAGE ${${PACKAGE_NAME}_LIB_OPTIONAL_DEP_PACKAGES})
    PACKAGE_ARCH_PRIVATE_ADD_OPTIONAL_PACKAGE_ENABLE(
      ${PACKAGE_NAME} ${OPTIONAL_DEP_PACKAGE} "library" "${SET_AS_CACHE}" )
  ENDFOREACH()
  
  FOREACH(OPTIONAL_DEP_PACKAGE ${${PACKAGE_NAME}_TEST_OPTIONAL_DEP_PACKAGES})
    PACKAGE_ARCH_PRIVATE_ADD_OPTIONAL_PACKAGE_ENABLE(
      ${PACKAGE_NAME} ${OPTIONAL_DEP_PACKAGE} "test" "${SET_AS_CACHE}" )
  ENDFOREACH()
  
  FOREACH(OPTIONAL_DEP_TPL ${${PACKAGE_NAME}_LIB_OPTIONAL_DEP_TPLS})
    PACKAGE_ARCH_PRIVATE_ADD_OPTIONAL_TPL_ENABLE(
      ${PACKAGE_NAME} ${OPTIONAL_DEP_TPL} "library" "${SET_AS_CACHE}" )
  ENDFOREACH()
  
  FOREACH(OPTIONAL_DEP_TPL ${${PACKAGE_NAME}_TEST_OPTIONAL_DEP_TPLS})
    PACKAGE_ARCH_PRIVATE_ADD_OPTIONAL_TPL_ENABLE(
      ${PACKAGE_NAME} ${OPTIONAL_DEP_TPL} "test" "${SET_AS_CACHE}" )
  ENDFOREACH()

ENDMACRO()


#
# Private helper macros
#

MACRO(PACKAGE_ARCH_PRIVATE_POSTPROCESS_OPTIONAL_PACKAGE_ENABLE PACKAGE_NAME OPTIONAL_DEP_PACKAGE)

  #MESSAGE("PACKAGE_ARCH_PRIVATE_POSTPROCESS_OPTIONAL_PACKAGE_ENABLE: ${PACKAGE_NAME} ${OPTIONAL_DEP_PACKAGE}")

  IF("${${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE}}" STREQUAL "")
    IF(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME} AND ${PROJECT_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE})
      MESSAGE("-- " "Setting ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE}=ON"
       " since ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=ON AND"
       " ${PROJECT_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE}=ON")
      SET(${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE} ON)
    ENDIF()
  ENDIF()

  STRING(TOUPPER ${PACKAGE_NAME} PACKAGE_NAME_UPPER)
  STRING(TOUPPER ${OPTIONAL_DEP_PACKAGE} OPTIONAL_DEP_PACKAGE_UPPER)
  SET(MACRO_DEFINE_NAME HAVE_${PACKAGE_NAME_UPPER}_${OPTIONAL_DEP_PACKAGE_UPPER})

  IF(${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE})
    SET(${MACRO_DEFINE_NAME} ON)
  ELSE()
    SET(${MACRO_DEFINE_NAME} OFF)
  ENDIF()

ENDMACRO()


MACRO(PACKAGE_ARCH_PRIVATE_POSTPROCESS_OPTIONAL_TPL_ENABLE PACKAGE_NAME OPTIONAL_DEP_TPL)

  #MESSAGE("PACKAGE_ARCH_PRIVATE_POSTPROCESS_OPTIONAL_TPL_ENABLE: ${PACKAGE_NAME} ${OPTIONAL_DEP_TPL}")

  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    IF (TPL_ENABLE_${OPTIONAL_DEP_TPL} STREQUAL OFF
      AND ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL}
      )
      MESSAGE(
        "\n***"
        "\n*** WARNING: Setting ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL}=OFF"
        " which was ON since TPL_ENABLE_${OPTIONAL_DEP_TPL}=OFF"
        "\n***\n"
        )
      SET(${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL} OFF)
    ELSEIF ("${${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL}}" STREQUAL ""
      AND TPL_ENABLE_${OPTIONAL_DEP_TPL}
      )
      MESSAGE("-- " "Setting ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL}=ON"
        " since TPL_ENABLE_${OPTIONAL_DEP_TPL}=ON")
      SET(${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL} ON)
    ENDIF()
  
    STRING(TOUPPER ${PACKAGE_NAME} PACKAGE_NAME_UPPER)
    STRING(TOUPPER ${OPTIONAL_DEP_TPL} OPTIONAL_DEP_TPL_UPPER)
    SET(MACRO_DEFINE_NAME HAVE_${PACKAGE_NAME_UPPER}_${OPTIONAL_DEP_TPL_UPPER})
  
    IF (${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL})
      SET(${MACRO_DEFINE_NAME} ON)
    ELSE()
      SET(${MACRO_DEFINE_NAME} OFF)
    ENDIF()

  ENDIF()

ENDMACRO()


#
# Macro that post-processes optional dependancies after all other
# dependencies have been worked out
#

MACRO(PACKAGE_ARCH_POSTPROCESS_OPTIONAL_PACKAGE_ENABLES PACKAGE_NAME)

  #MESSAGE("\nPACKAGE_ARCH_POSTPROCESS_OPTIONAL_PACKAGE_ENABLES: ${PACKAGE_NAME}")

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
  IF(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(OPTIONAL_DEP_PACKAGE ${${PACKAGE_NAME}_LIB_OPTIONAL_DEP_PACKAGES})
      PACKAGE_ARCH_PRIVATE_POSTPROCESS_OPTIONAL_PACKAGE_ENABLE(
        ${PACKAGE_NAME} ${OPTIONAL_DEP_PACKAGE} )
    ENDFOREACH()
  
    FOREACH(OPTIONAL_DEP_PACKAGE ${${PACKAGE_NAME}_TEST_OPTIONAL_DEP_PACKAGES})
      PACKAGE_ARCH_PRIVATE_POSTPROCESS_OPTIONAL_PACKAGE_ENABLE(
        ${PACKAGE_NAME} ${OPTIONAL_DEP_PACKAGE} )
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Macro that post-processes optional package TPL based on if the TPL
# has been enabled or not
#

MACRO(PACKAGE_ARCH_POSTPROCESS_OPTIONAL_TPL_ENABLES PACKAGE_NAME)

  #MESSAGE("\nPACKAGE_ARCH_ADD_OPTIONAL_TPL_ENABLES: ${PACKAGE_NAME}")

  FOREACH(OPTIONAL_DEP_TPL ${${PACKAGE_NAME}_LIB_OPTIONAL_DEP_TPLS})
    PACKAGE_ARCH_PRIVATE_POSTPROCESS_OPTIONAL_TPL_ENABLE(
      ${PACKAGE_NAME} ${OPTIONAL_DEP_TPL} )
  ENDFOREACH()

  FOREACH(OPTIONAL_DEP_TPL ${${PACKAGE_NAME}_TEST_OPTIONAL_DEP_TPLS})
    PACKAGE_ARCH_PRIVATE_POSTPROCESS_OPTIONAL_TPL_ENABLE(
      ${PACKAGE_NAME} ${OPTIONAL_DEP_TPL} )
  ENDFOREACH()

ENDMACRO()


#
# Set an individual package variable enable based on the global value
#

MACRO(PACKAGE_ARCH_SET_ALL_PACKAGES_PACKAGE_ENABLE_VARIABLE PACKAGE_ARCH_VAR PACKAGE_VAR)

  IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
    MESSAGE("")
    MESSAGE("PACKAGE_ARCH_SET_ALL_PACKAGES_PACKAGE_ENABLE_VARIABLE:")
    MESSAGE("-- " "${PACKAGE_ARCH_VAR} = ${${PACKAGE_ARCH_VAR}}")
    MESSAGE("-- " "${PACKAGE_VAR} = ${${PACKAGE_VAR}}")
  ENDIF()

  IF ("${${PACKAGE_VAR}}" STREQUAL "")
    IF (${PACKAGE_ARCH_VAR} STREQUAL "ON")
      MESSAGE("-- " "Setting ${PACKAGE_VAR}=ON")
      SET(${PACKAGE_VAR} ON)
    ELSEIF (PACKAGE_ARCH_VAR STREQUAL "OFF")
      MESSAGE("-- " "Setting ${PACKAGE_VAR}=OFF")
      SET(${PACKAGE_VAR} OFF)
    ELSE()
      IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
        MESSAGE("-- " "ELSE")
        # Otherwise, we will leave it up the the individual package
        # to decide?
      ENDIF()
    ENDIF()
  ELSE()
    IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
      MESSAGE("-- " "${PACKAGE_VAR} NOT DEFAULT")
    ENDIF()
  ENDIF()

  IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
    MESSAGE("-- " "${PACKAGE_VAR} = ${${PACKAGE_VAR}}")
  ENDIF()

ENDMACRO()


#
# Macro used to set ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME} based on
# ${PROJECT_NAME}_ENABLE_ALL_PACKAGES
#

MACRO(PACKAGE_ARCH_APPLY_ALL_PACKAGE_ENABLES PACKAGE_NAME)
  PACKAGE_ARCH_IMPLICIT_PACKAGE_ENABLE_IS_ALLOWED( ${PACKAGE_NAME}
    PROCESS_PACKAGE_ENABLE )
  IF (PROCESS_PACKAGE_ENABLE)
    PACKAGE_ARCH_SET_ALL_PACKAGES_PACKAGE_ENABLE_VARIABLE(
      ${PROJECT_NAME}_ENABLE_ALL_PACKAGES ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME} )
  ENDIF()
ENDMACRO()


#
# Macro used to set ${PACKAGE)_ENABLE_TESTS and ${PACKAGE)_ENABLE_EXAMPLES
# based on ${PROJECT_NAME}_ENABLE_ALL_PACKAGES
#

MACRO(PACKAGE_ARCH_APPLY_TEST_EXAMPLE_ENABLES PACKAGE_NAME)
  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
    PACKAGE_ARCH_SET_ALL_PACKAGES_PACKAGE_ENABLE_VARIABLE(
      ${PROJECT_NAME}_ENABLE_TESTS ${PACKAGE_NAME}_ENABLE_TESTS )
    PACKAGE_ARCH_SET_ALL_PACKAGES_PACKAGE_ENABLE_VARIABLE(
      ${PROJECT_NAME}_ENABLE_EXAMPLES ${PACKAGE_NAME}_ENABLE_EXAMPLES )
  ENDIF()
ENDMACRO()


#
# Private helper macro
#

MACRO(PACKAGE_ARCH_PRIVATE_ENABLE_FORWARD_PACKAGE  FORWARD_DEP_PACKAGE_NAME  PACKAGE_NAME)
  PACKAGE_ARCH_IMPLICIT_PACKAGE_ENABLE_IS_ALLOWED( ${FORWARD_DEP_PACKAGE_NAME}
    ALLOW_PACKAGE_ENABLE )
  #MESSAGE("PACKAGE_ARCH_PRIVATE_ENABLE_FORWARD_PACKAGE: "
  #  "${FORWARD_DEP_PACKAGE_NAME} ${PACKAGE_NAME} ${ALLOW_PACKAGE_ENABLE}") 
  # Enable the forward package if it is not already set to ON or OFF
  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME})
  IF(${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME} STREQUAL ""
    AND ALLOW_PACKAGE_ENABLE
    )
    MESSAGE("-- " "Setting ${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME}=ON"
      " because ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=ON")
    ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME})
    SET(${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME} ON)
  ENDIF()
ENDMACRO()


#
# Macro used to set ${PROJECT_NAME}_ENABLE_${FWD_PACKAGE_NAME)=ON for all optional
# and required forward library dependencies of the package ${PACKAGE_NAME}
#

MACRO(PACKAGE_ARCH_ENABLE_FORWARD_LIB_PACKAGE_ENABLES PACKAGE_NAME)

  #MESSAGE("\nPACKAGE_ARCH_ENABLE_FORWARD_PACKAGE_ENABLES ${PACKAGE_NAME}")
  #PRINT_VAR(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

  # Enable the forward packages if this package is enabled
  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_LIB_REQUIRED_DEP_PACKAGES})
      PACKAGE_ARCH_PRIVATE_ENABLE_FORWARD_PACKAGE(${FWD_DEP_PKG} ${PACKAGE_NAME})
    ENDFOREACH()

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_LIB_OPTIONAL_DEP_PACKAGES})
      PACKAGE_ARCH_PRIVATE_ENABLE_FORWARD_PACKAGE(${FWD_DEP_PKG} ${PACKAGE_NAME})
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Macro used to set ${PROJECT_NAME}_ENABLE_${FWD_PACKAGE_NAME)=ON for all optional
# and required forward test/example dependencies of the package ${PACKAGE_NAME}
#

MACRO(PACKAGE_ARCH_ENABLE_FORWARD_TEST_PACKAGE_ENABLES PACKAGE_NAME)

  #MESSAGE("\nPACKAGE_ARCH_ENABLE_FORWARD_PACKAGE_ENABLES ${PACKAGE_NAME}")
  #MESSAGE("-- " "${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}}")

  # Enable the forward packages if this package is enabled
  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_TEST_REQUIRED_DEP_PACKAGES})
      PACKAGE_ARCH_PRIVATE_ENABLE_FORWARD_PACKAGE(${FWD_DEP_PKG} ${PACKAGE_NAME})
    ENDFOREACH()

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_TEST_OPTIONAL_DEP_PACKAGES})
      PACKAGE_ARCH_PRIVATE_ENABLE_FORWARD_PACKAGE(${FWD_DEP_PKG} ${PACKAGE_NAME})
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Private helper macros
#


MACRO(PACKAGE_ARCH_PRIVATE_ENABLE_DEP_PACKAGE PACKAGE_NAME DEP_PACKAGE_NAME)
  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME})
  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_SECONDARY_STABLE_CODE)
  #PRINT_VAR(ENABLE_OVERRIDE)
  #PRINT_VAR(${PACKAGE_NAME}_ENABLE_${DEP_PACKAGE_NAME})
  IF (${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME} STREQUAL "")
    IF (ENABLE_OVERRIDE OR ${PACKAGE_NAME}_ENABLE_${DEP_PACKAGE_NAME})
      IF (${DEP_PACKAGE_NAME}_CLASSIFICATION STREQUAL PS OR
        (${DEP_PACKAGE_NAME}_CLASSIFICATION STREQUAL SS
          AND ${PROJECT_NAME}_ENABLE_SECONDARY_STABLE_CODE)
        )
        MESSAGE("-- " "Setting ${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME}=ON"
          " because ${PACKAGE_NAME}_ENABLE_${DEP_PACKAGE_NAME}=ON")
        ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME})
        SET(${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME} ON)
      ELSE()
        MESSAGE("-- " "WARNING: Not Setting ${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME}=ON"
          " even through ${PACKAGE_NAME}_ENABLE_${DEP_PACKAGE_NAME}=ON because"
          " ${PROJECT_NAME}_ENABLE_SECONDARY_STABLE_CODE=OFF" )
      ENDIF()
    ENDIF()
  ENDIF()
ENDMACRO()


MACRO(PACKAGE_ARCH_PRIVATE_ENABLE_REQUIRED_DEP_PACKAGE PACKAGE_NAME DEP_PACKAGE_NAME)
  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME})
  #PRINT_VAR(ENABLE_OVERRIDE)
  #PRINT_VAR(${PACKAGE_NAME}_ENABLE_${DEP_PACKAGE_NAME})
  IF (${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME} STREQUAL "")
    MESSAGE("-- " "Setting ${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME}=ON"
      " because ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=ON")
    ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME})
    SET(${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME} ON)
  ENDIF()
ENDMACRO()


MACRO(PACKAGE_ARCH_PRIVATE_ENABLE_DEP_TPL PACKAGE_NAME DEP_TPL_NAME)
  ASSERT_DEFINED(TPL_ENABLE_${DEP_TPL_NAME})
  IF(TPL_ENABLE_${DEP_TPL_NAME} STREQUAL "")
    MESSAGE("-- " "Setting TPL_ENABLE_${DEP_TPL_NAME}=ON because"
      " it is required by the enabled package ${PACKAGE_NAME}")
    ASSERT_DEFINED(TPL_ENABLE_${DEP_TPL_NAME})
    SET(TPL_ENABLE_${DEP_TPL_NAME} ON)
  ENDIF()
ENDMACRO()


MACRO(PACKAGE_ARCH_PRIVATE_ENABLE_OPTIONAL_DEP_TPL PACKAGE_NAME DEP_TPL_NAME)
  #ASSERT_DEFINED(${PACKAGE_NAME}_ENABLE_${DEP_TPL_NAME})
  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}
    AND ${PACKAGE_NAME}_ENABLE_${DEP_TPL_NAME}
    AND TPL_ENABLE_${DEP_TPL_NAME} STREQUAL ""
    )
    MESSAGE("-- " "Setting TPL_ENABLE_${DEP_TPL_NAME}=ON because"
      " ${PACKAGE_NAME}_ENABLE_${DEP_TPL_NAME}=ON")
    ASSERT_DEFINED(TPL_ENABLE_${DEP_TPL_NAME})
    SET(TPL_ENABLE_${DEP_TPL_NAME} ON)
  ENDIF()
ENDMACRO()


#
# Macro that enables the optional TPLs for given package
#

MACRO(PACKAGE_ARCH_ENABLE_OPTIONAL_TPLS PACKAGE_NAME)

  #MESSAGE("PACKAGE_ARCH_ENABLE_OPTIONAL_TPLS: ${PACKAGE_NAME}")
  #MESSAGE("-- " "${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}}")

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(DEP_TPL ${${PACKAGE_NAME}_LIB_OPTIONAL_DEP_TPLS})
      PACKAGE_ARCH_PRIVATE_ENABLE_OPTIONAL_DEP_TPL(${PACKAGE_NAME} ${DEP_TPL})
    ENDFOREACH()

    FOREACH(DEP_TPL ${${PACKAGE_NAME}_TEST_OPTIONAL_DEP_TPLS})
      PACKAGE_ARCH_PRIVATE_ENABLE_OPTIONAL_DEP_TPL(${PACKAGE_NAME} ${DEP_TPL})
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Macro that sets the optional packages for given package
#
# Here I have to enable the required packages too or the logic just does no
# work as expected.
#

MACRO(PACKAGE_ARCH_ENABLE_OPTIONAL_PACKAGES PACKAGE_NAME)

  #MESSAGE("PACKAGE_ARCH_ENABLE_OPTIONAL_PACKAGE_ENABLES: ${PACKAGE_NAME}")
  #MESSAGE("-- " "${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}}")

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

  #PRINT_VAR(ENABLE_ALL_DEP_PKGS)

  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(DEP_PKG ${${PACKAGE_NAME}_LIB_REQUIRED_DEP_PACKAGES})
      SET(ENABLE_OVERRIDE ${ENABLE_ALL_DEP_PKGS})
      PACKAGE_ARCH_PRIVATE_ENABLE_DEP_PACKAGE(${PACKAGE_NAME} ${DEP_PKG})
    ENDFOREACH()

    FOREACH(DEP_PKG ${${PACKAGE_NAME}_LIB_OPTIONAL_DEP_PACKAGES})
      SET(ENABLE_OVERRIDE ${ENABLE_ALL_DEP_PKGS})
      PACKAGE_ARCH_PRIVATE_ENABLE_DEP_PACKAGE(${PACKAGE_NAME} ${DEP_PKG})
    ENDFOREACH()

    FOREACH(DEP_PKG ${${PACKAGE_NAME}_TEST_REQUIRED_DEP_PACKAGES})
      SET(ENABLE_OVERRIDE ${ENABLE_ALL_DEP_PKGS})
      PACKAGE_ARCH_PRIVATE_ENABLE_DEP_PACKAGE(${PACKAGE_NAME} ${DEP_PKG})
    ENDFOREACH()

    FOREACH(DEP_PKG ${${PACKAGE_NAME}_TEST_OPTIONAL_DEP_PACKAGES})
      SET(ENABLE_OVERRIDE ${ENABLE_ALL_DEP_PKGS})
      PACKAGE_ARCH_PRIVATE_ENABLE_DEP_PACKAGE(${PACKAGE_NAME} ${DEP_PKG})
    ENDFOREACH()

    # 2008/02/13: rabartl: Above, I had to set the varaible ENABLE_OVERRIDE
    # and then call the function because it was not being passed
    # as an argument as was zero on the other side.  This is very
    # strange but you have to do what you have to do!

  ENDIF()

ENDMACRO()


#
# Macro that sets the required packages for given package
#

MACRO(PACKAGE_ARCH_ENABLE_REQUIRED_PACKAGES PACKAGE_NAME)

  #MESSAGE("PACKAGE_ARCH_ENABLE_REQUIRED_PACKAGE_ENABLES: ${PACKAGE_NAME}")
  #MESSAGE("-- " "${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}}")

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(DEP_PKG ${${PACKAGE_NAME}_LIB_REQUIRED_DEP_PACKAGES})
      PACKAGE_ARCH_PRIVATE_ENABLE_REQUIRED_DEP_PACKAGE(${PACKAGE_NAME} ${DEP_PKG} ON)
    ENDFOREACH()

    FOREACH(DEP_PKG ${${PACKAGE_NAME}_TEST_REQUIRED_DEP_PACKAGES})
      PACKAGE_ARCH_PRIVATE_ENABLE_REQUIRED_DEP_PACKAGE(${PACKAGE_NAME} ${DEP_PKG} ON)
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Macro that sets the required TPLs for given package
#

MACRO(PACKAGE_ARCH_ENABLE_REQUIRED_TPLS PACKAGE_NAME)

  #MESSAGE("PACKAGE_ARCH_ENABLE_REQUIRED_TPL_ENABLES: ${PACKAGE_NAME}")
  #MESSAGE("-- " "${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}}")

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(DEP_TPL ${${PACKAGE_NAME}_LIB_REQUIRED_DEP_TPLS})
      PACKAGE_ARCH_PRIVATE_ENABLE_DEP_TPL(${PACKAGE_NAME} ${DEP_TPL})
    ENDFOREACH()

    FOREACH(DEP_TPL ${${PACKAGE_NAME}_TEST_REQUIRED_DEP_TPLS})
      PACKAGE_ARCH_PRIVATE_ENABLE_DEP_TPL(${PACKAGE_NAME} ${DEP_TPL})
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Macro that adjusts all of the package enables from what the user input
# to the final set that will be used to enable packages
#

MACRO(PACKAGE_ARCH_ADJUST_PACKAGE_ENABLES  DO_PROCESS_MPI_ENABLES)

  IF (${PROJECT_NAME}_UNENABLE_ENABLED_PACKAGES)
    MESSAGE("")
    MESSAGE("Setting to empty '' all package enables on reqeust ...")
    MESSAGE("")
    FOREACH(PACKAGE ${${PROJECT_NAME}_PACKAGES})
      SET_CACHE_ON_OFF_EMPTY(${PROJECT_NAME}_ENABLE_${PACKAGE} ""
        "Forced to empty '' by ${PROJECT_NAME}_UNENABLE_ENABLED_PACKAGES=OFF" FORCE)
      SET(${PROJECT_NAME}_ENABLE_${PACKAGE} "")
      #PRINT_VAR(${PROJECT_NAME}_ENABLE_${PACKAGE})
    ENDFOREACH()
    ADVANCED_SET(${PROJECT_NAME}_UNENABLE_ENABLED_PACKAGES OFF CACHE BOOL
      "Forced to FALSE after use" FORCE)
  ENDIF()

  MESSAGE("")
  MESSAGE("Disabling all packages that have a required dependency"
    " on disabled TPLs and optional package TPL support based on TPL_ENABLE_<TPL>=OFF ...")
  MESSAGE("")
  FOREACH(TPL ${${PROJECT_NAME}_TPLS})
    PACKAGE_ARCH_DISABLE_TPL_DEP_PACKAGES(${TPL})
  ENDFOREACH()

  MESSAGE("")
  MESSAGE("Disabling forward required packages and optional intra-package"
    " support that have a dependancy on disabled packages"
    " ${PROJECT_NAME}_ENABLE_<PACKAGE>=OFF ...")
  MESSAGE("")
  FOREACH(PACKAGE ${${PROJECT_NAME}_PACKAGES})
    PACKAGE_ARCH_DISABLE_FORWARD_REQUIRED_DEP_PACKAGES(${PACKAGE})
  ENDFOREACH()
  
  IF (${PROJECT_NAME}_ENABLE_ALL_PACKAGES)
    MESSAGE("")
    MESSAGE("Enabling all packages that are not currently disabled because of"
      " ${PROJECT_NAME}_ENABLE_ALL_PACKAGES=ON ...")
    MESSAGE("")
    FOREACH(PACKAGE ${${PROJECT_NAME}_PACKAGES})
      PACKAGE_ARCH_APPLY_ALL_PACKAGE_ENABLES(${PACKAGE})
    ENDFOREACH()
  ENDIF()
  
  IF (${PROJECT_NAME}_ENABLE_ALL_FORWARD_DEP_PACKAGES)
    MESSAGE("")
    MESSAGE("Sweep forward enabling all forward library dependent packages because"
      " ${PROJECT_NAME}_ENABLE_ALL_FORWARD_DEP_PACKAGES=ON ...")
    MESSAGE("")
    FOREACH(PACKAGE ${${PROJECT_NAME}_PACKAGES})
      PACKAGE_ARCH_ENABLE_FORWARD_LIB_PACKAGE_ENABLES(${PACKAGE})
    ENDFOREACH()
    MESSAGE("")
    MESSAGE("Sweep backward enabling all forward test dependent packages because"
      " ${PROJECT_NAME}_ENABLE_ALL_FORWARD_DEP_PACKAGES=ON ...")
    MESSAGE("")
    FOREACH(PACKAGE ${${PROJECT_NAME}_REVERSE_PACKAGES})
      PACKAGE_ARCH_ENABLE_FORWARD_TEST_PACKAGE_ENABLES(${PACKAGE})
    ENDFOREACH()
    # NOTE: Above, we want to sweep backward to enable test-dependent packages
    # because we don't want to enable pacakge Z just because pacakge Y was enabled
    # because it had a test-only dependency on package X.  Sweeping backwards through
    # the packages makes sure this does not happen.
    SET(${PROJECT_NAME}_ENABLE_ALL_OPTIONAL_PACKAGES ON)
  ENDIF()
  
  IF (${PROJECT_NAME}_ENABLE_TESTS OR ${PROJECT_NAME}_ENABLE_EXAMPLES)
    MESSAGE("")
    MESSAGE("Enabling all tests and/or examples that have not been"
      " explicitly disabled because ${PROJECT_NAME}_ENABLE_[TESTS,EXAMPLES]=ON ...")
    MESSAGE("")
    FOREACH(PACKAGE ${${PROJECT_NAME}_PACKAGES})
      PACKAGE_ARCH_APPLY_TEST_EXAMPLE_ENABLES(${PACKAGE})
    ENDFOREACH()
  ENDIF()
  # NOTE: Above, we enable tests and examples here, before the remaining required
  # packages so that we don't enable tests that don't need to be enabled based
  # on the use of the option ${PROJECT_NAME}_ENABLE_ALL_FORWARD_DEP_PACKAGES.

  IF (${PROJECT_NAME}_ENABLE_ALL_OPTIONAL_PACKAGES)
    MESSAGE("")
    MESSAGE("Enabling all optional packages for current set of enabled"
      " packages because ${PROJECT_NAME}_ENABLE_ALL_OPTIONAL_PACKAGES=ON ...")
    MESSAGE("")
    FOREACH(PACKAGE ${${PROJECT_NAME}_REVERSE_PACKAGES})
      SET(ENABLE_ALL_DEP_PKGS ON) # Hack to get around CMake bug?
      PACKAGE_ARCH_ENABLE_OPTIONAL_PACKAGES(${PACKAGE})
    ENDFOREACH()
    # NOTE: Above, we have to loop through the packages backward to enable all the
    # packages that feed into these packages.
    # NOTE Above, we don't have to enable the required packages because that will
    # come next
  ELSE()
    MESSAGE("")
    MESSAGE("Enabling all packages that are not disabled based on"
      " <PACKAGE>_ENABLE_<DEPPACKAGE>=ON  ...")
    MESSAGE("")
    FOREACH(PACKAGE ${${PROJECT_NAME}_REVERSE_PACKAGES})
      SET(ENABLE_ALL_DEP_PKGS OFF) # Hack to get around CMake bug?
      PACKAGE_ARCH_ENABLE_OPTIONAL_PACKAGES(${PACKAGE})
    ENDFOREACH()
    # NOTE: Aaove, we loop backwards through the packages to pick up as many
    # package enables as we can.  Also, we do this after the tests/examples are 
    # enabled.
  ENDIF()
  
  MESSAGE("")
  MESSAGE("Enabling all remaining required packages for the current set"
    " of enabled packages ...")
  MESSAGE("")
  FOREACH(PACKAGE ${${PROJECT_NAME}_REVERSE_PACKAGES})
    PACKAGE_ARCH_ENABLE_REQUIRED_PACKAGES(${PACKAGE})
  ENDFOREACH()
  
  MESSAGE("")
  MESSAGE("Enabling all optional intra-package enables <PACKAGE>_ENABLE_<DEPPACKAGE>"
    " that are not currently disabled if both sets of packages are enabled ...")
  MESSAGE("")
  FOREACH(PACKAGE ${${PROJECT_NAME}_PACKAGES})
    PACKAGE_ARCH_POSTPROCESS_OPTIONAL_PACKAGE_ENABLES(${PACKAGE})
  ENDFOREACH()

  ASSERT_DEFINED(DO_PROCESS_MPI_ENABLES)
  IF (DO_PROCESS_MPI_ENABLES)

    MESSAGE("")
    MESSAGE("Enabling all remaining required TPLs for current set of"
      " enabled packages ...")
    MESSAGE("")
    FOREACH(PACKAGE ${${PROJECT_NAME}_PACKAGES})
      PACKAGE_ARCH_ENABLE_REQUIRED_TPLS(${PACKAGE})
    ENDFOREACH()
  
    MESSAGE("")
    MESSAGE("Enabling all optional package TPL support for currently"
      " enabled TPLs ...")
    MESSAGE("")
    FOREACH(PACKAGE ${${PROJECT_NAME}_PACKAGES})
      PACKAGE_ARCH_POSTPROCESS_OPTIONAL_TPL_ENABLES(${PACKAGE})
    ENDFOREACH()
  
    MESSAGE("")
    MESSAGE("Enabling TPLs based on <PACKAGE>_ENABLE_<TPL>=ON if TPL is not explicitly disabled ...")
    MESSAGE("")
    FOREACH(PACKAGE ${${PROJECT_NAME}_PACKAGES})
      PACKAGE_ARCH_ENABLE_OPTIONAL_TPLS(${PACKAGE})
    ENDFOREACH()
    # NOTE: We need to do this after the above optional package TPL support
    # logic so that the TPL will be turned on for this package only as requested
    # in bug 4298.

    MESSAGE("")
    MESSAGE("Set cache entries for optional packages/TPLs and tests/examples for packages actually enabled ...")
    MESSAGE("")
    FOREACH(PACKAGE ${${PROJECT_NAME}_PACKAGES})
      PACKAGE_ARCH_ADD_OPTIONAL_PACKAGE_ENABLES(${PACKAGE})
    ENDFOREACH()

  ENDIF()

ENDMACRO()
