#!/usr/bin/env python

#****************************************************************************
# treeformats.py, provides non-GUI base classes for stroing node format info
#
# TreeLine, an information storage program
# Copyright (C) 2005, Douglas W. Bell
#
# This is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License, Version 2.  This program is
# distributed in the hope that it will be useful, but WITTHOUT ANY WARRANTY.
#****************************************************************************

import nodeformat, globalref, treedoc

class TreeFormats(list):
    """A list to store node formats in the doc"""
    rootFormatDefault = _('ROOT', 'root format default name')
    formatDefault = _('DEFAULT', 'default format name')
    fieldDefault = _('Name', 'default field name')
    textFieldName = _('Text', 'text field name')
    linkFieldName = _('Link', 'link field name')
    def __init__(self, initList=[], setDefaults=False):
        list.__init__(self, initList)
        self.derivedDict = {}
        self.hasConditionals = False
        if setDefaults:
            self.extend([nodeformat.NodeFormat(TreeFormats.rootFormatDefault, \
                                               {}, TreeFormats.fieldDefault), \
                         nodeformat.NodeFormat(TreeFormats.formatDefault, \
                                               {}, TreeFormats.fieldDefault)])
            self[0].childType = TreeFormats.formatDefault

    def __getslice__(self, i, j):
        """Modified to copy object with slice"""
        return TreeFormats(list.__getslice__(self, i, j))

    def nameList(self, excludeFileInfo=False):
        """Return list of names of format items"""
        formats = self[:]
        if excludeFileInfo and globalref.docRef.fileInfoItem.nodeFormat \
                               in formats:
            formats.remove(globalref.docRef.fileInfoItem.nodeFormat)
        return [form.name for form in formats]

    def findFormat(self, name):
        """Return format matching name or None"""
        try:
            return self[self.nameList().index(name)]
        except ValueError:
            return None

    def findDefaultFormat(self):
        """Return default or non-root format"""
        dflt = self.findFormat(TreeFormats.formatDefault)
        if not dflt:
            for format in self:
                if format.name != TreeFormats.rootFormatDefault:
                    return format
            return self[0]
        return dflt

    def addIfMissing(self, format):
        """Add format to list if not a duplicate"""
        if format not in self:
            self.append(format)

    def removeQuiet(self, format):
        """Remove from list if there"""
        try:
            self.remove(format)
        except ValueError:
            pass

    def renameFields(self, nameDict):
        """Rename data fields for all items, doesn't change node format"""
        for format in self:
            if format.genericType in nameDict:
                nameDict[format.name] = nameDict[format.genericType]
        for item in globalref.docRef.root.descendantGen():
            for oldName, newName in nameDict.get(item.nodeFormat.name, []):
                if oldName in item.data:
                    item.data[newName] = item.data[oldName]
                    del item.data[oldName]

    def updateAllLineFields(self):
        """Re-find fields to update format lines for any changes
           in the fieldLists"""
        for format in self:
            format.updateLineFields()
        globalref.docRef.fileInfoItem.nodeFormat.updateLineFields()

    def update(self, updateList, renameDict):
        """Update formats to be equiv to updateList,
           changed names are in renameDict"""
        for key in renameDict.keys():
            format = self.findFormat(key)
            if format:
                format.name = renameDict[key]
        newList = []
        for updateFormat in updateList:
            format = self.findFormat(updateFormat.name)
            if format:
                format.duplicateSettings(updateFormat)
                newList.append(format)
            else:
                newList.append(updateFormat)
        self[:] = newList

    def updateAutoChoices(self):
        """Update auto menu choices for all AutoChoice fields"""
        autoFields = {}
        for format in self:
            fields = format.findAutoChoiceFields()
            if fields:
                autoFields[format.name] = fields
        if autoFields:
            for item in globalref.docRef.root.descendantGen():
                for field in autoFields.get(item.nodeFormat.name, []):
                    field.addChoice(item.data.get(field.name, ''))
            for fieldList in autoFields.values():
                for field in fieldList:
                    field.sortChoices()

    def updateDerivedTypes(self):
        """Update fields for all derived types,
           update the dict containing lists of type families and
           update fields used in conditionals"""
        self.hasConditionals = False
        self.derivedDict = {}
        for format in self:
            if format.genericType:
                format.updateFromGeneric(self)
                generic = self.findFormat(format.genericType)
                if format.genericType in self.derivedDict:
                    self.derivedDict[format.genericType].append(format)
                else:
                    self.derivedDict[format.genericType] = [generic, format]
            if format.conditional:
                self.hasConditionals = True
                format.conditional.setupFields(format)

    def configCopy(self, fileRef, password=''):
        """Copy the configuration from another TreeLine file"""
        if hasattr(fileRef, 'read'):
            fileName = unicode(fileRef.name, treedoc.TreeDoc.localEncoding)
        else:
            fileName = fileRef
        origDocRef = globalref.docRef
        refDoc = treedoc.TreeDoc()
        if password:
            refDoc.setPassword(fileName, password)
        try:
            refDoc.readFile(fileRef)
        except:
            globalref.docRef = origDocRef
            raise
        globalref.docRef = origDocRef
        origDocRef.undoStore.addFormatUndo(self, \
                                           origDocRef.fileInfoItem.nodeFormat, \
                                           {}, {})
        for newFormat in refDoc.treeFormats:
            format = self.findFormat(newFormat.name)
            if format:
                format.duplicateSettings(newFormat)
            else:
                self.append(newFormat)
        self.updateDerivedTypes()
        globalref.docRef.modified = True
        globalref.updateViewAll()
