/**
 * Copyright (C) 2007-2012 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 432 $
 * $Date: 2012-03-08 22:17:25 +0800 (Thu, 08 Mar 2012) $
 */
#include "ThumbHSL.hpp"

#include "wx/wx.h"
#include "wx/gbsizer.h"

using namespace indii;

const float ThumbHSL::SPOW = 2.5f;

ThumbHSL::ThumbHSL(wxWindow *parent, const unsigned int cluster) :
    wxPanel(parent), model(NULL), cluster(cluster) {
  static const int PADDING = 4;
      
  wxGridBagSizer* sizer = new wxGridBagSizer(0,0);
  wxGridBagSizer* slideSizer = new wxGridBagSizer(0,0);

  thumb = new ThumbImage(this, cluster);
  
  slideH = new wxSlider(this, ID_SLIDE_H, 0, -128, 128, wxDefaultPosition,
      wxSize(256,-1), wxSL_HORIZONTAL);
  slideS = new wxSlider(this, ID_SLIDE_S, -255, -255, 255, wxDefaultPosition,
      wxSize(256,-1), wxSL_HORIZONTAL);
  slideL = new wxSlider(this, ID_SLIDE_L, 0, -255, 255, wxDefaultPosition,
      wxSize(256,-1), wxSL_HORIZONTAL);

  spectH = new SpectrumImage(this, cluster, SpectrumImage::H);
  spectS = new SpectrumImage(this, cluster, SpectrumImage::S);
  spectL = new SpectrumImage(this, cluster, SpectrumImage::L);

  sizer->Add(thumb, wxGBPosition(0,0), wxDefaultSpan,
      wxALIGN_TOP|wxALL, PADDING);
      
  slideSizer->Add(spectH, wxGBPosition(1,0), wxDefaultSpan, wxALIGN_TOP); 
  slideSizer->Add(spectS, wxGBPosition(3,0), wxDefaultSpan, wxALIGN_TOP);
  slideSizer->Add(spectL, wxGBPosition(5,0), wxDefaultSpan, wxALIGN_TOP);
      
  slideSizer->Add(slideH, wxGBPosition(0,0), wxDefaultSpan,
      wxALIGN_TOP|wxEXPAND|wxTOP, PADDING);  
  slideSizer->Add(slideS, wxGBPosition(2,0), wxDefaultSpan,
      wxALIGN_TOP|wxEXPAND|wxTOP, PADDING);
  slideSizer->Add(slideL, wxGBPosition(4,0), wxDefaultSpan,
      wxALIGN_TOP|wxEXPAND|wxTOP, PADDING);
      
  sizer->Add(slideSizer, wxGBPosition(0,1), wxDefaultSpan,
      wxALIGN_CENTRE_VERTICAL);
      
  //sizer->SetFlexibleDirection(wxHORIZONTAL);
  //sizer->AddGrowableCol(2);
  SetSizerAndFit(sizer);
  //unsigned naturalWidth = this->GetSize().GetWidth();
  //unsigned naturalHeight = this->GetSize().GetHeight();
  //SetMinSize(wxSize(naturalWidth, naturalHeight));
  //SetBestSize(wxSize(naturalWidth, naturalHeight));
}

void ThumbHSL::setModel(ClusterModel* model) {
  this->model = model;
  ignore();
  thumb->setModel(model);
  spectH->setModel(model);
  spectS->setModel(model);
  spectL->setModel(model);

  if (model != NULL) {
    if (cluster < model->getNumClusters()) {
      if (!model->hasLight(cluster)) {
	slideH->Disable();
	spectH->Disable();
	slideS->Disable();
	spectS->Disable();
      } else if (!model->hasSat(cluster)) {
	slideH->Disable();
	spectH->Disable();
      }          
    }

    watch(model);
    notifyNumClustersChange();
    notifySaturationDecayChange();
    notifySaturationSoftnessChange();
  }

  Show(model != NULL && cluster < model->getNumClusters());
}

void ThumbHSL::notifyNumClustersChange() {
  if (cluster < model->getNumClusters()) {
    notifyHueChange(cluster);
    notifySatChange(cluster);
    notifyLightChange(cluster);
  }
}

void ThumbHSL::notifySaturationDecayChange() {
//  if (cluster < model->getNumClusters()) {
//    Refresh();
//  }
}

void ThumbHSL::notifySaturationSoftnessChange() {
//  if (cluster < model->getNumClusters()) {
//    Refresh();
//  }
}

void ThumbHSL::notifyClusterChange(const unsigned int i) {
  notifyHueChange(i);
  notifySatChange(i);
  notifyLightChange(i);
}

void ThumbHSL::notifyHueChange(const unsigned i) {
  if (i == cluster) {
    int val = static_cast<int>(model->getHue(cluster)*256.0f);
    if (slideH->GetValue() != val) {
      slideH->SetValue(val);
    }
  }
}

void ThumbHSL::notifySatChange(const unsigned i) {
  if (i == cluster) {
	  float s = model->getSat(cluster);
    float sign = (s < 0.0f) ? -1.0f : 1.0f;
    s = sign*255.0f*std::pow(std::abs(s), 1.0f/SPOW);
    int val = static_cast<int>(s);
    if (slideS->GetValue() != val) {
      slideS->SetValue(val);
    }
    if (!model->hasSat(cluster)) {
      slideH->Disable();
      spectH->Disable();
    } else if (model->hasLight(cluster)) {
      slideH->Enable();
      spectH->Enable();
    }
    spectH->Refresh();
  }
}

void ThumbHSL::notifyLightChange(const unsigned i) {
  if (i == cluster) {
    float l = model->getLight(cluster);
    int val = static_cast<int>(l);
    if (slideL->GetValue() != val) {
      slideL->SetValue(val);
    }
    if (!model->hasLight(cluster)) {
      slideH->Disable();
      spectH->Disable();
      slideS->Disable();
      spectS->Disable();
    } else {
      if (model->hasSat(cluster)) {
				slideH->Enable();
				spectH->Enable();
      }
      slideS->Enable();
      spectS->Enable();
    }
    spectH->Refresh();
    spectS->Refresh();
  }
}

void ThumbHSL::OnSlideH(wxScrollEvent& evt) {
  float h;
  h = slideH->GetValue()/256.0f;
  h = ColourSpace::bound(-0.5f, h, 0.5f);
  model->setHue(cluster, h);
}

void ThumbHSL::OnSlideS(wxScrollEvent& evt) {
  float s = slideS->GetValue()/255.0f;
	float sign = (s < 0.0f) ? -1.0f : 1.0f;
  s = sign*std::pow(std::abs(s), SPOW);
  s = ColourSpace::bound(-1.0f, s, 1.0f);
  model->setSat(cluster, s);
}

void ThumbHSL::OnSlideL(wxScrollEvent& evt) {
  float l = slideL->GetValue();
  l = ColourSpace::bound(-255.0f, l, 255.0f);
  model->setLight(cluster, l);
}

BEGIN_EVENT_TABLE(ThumbHSL, wxPanel)
EVT_COMMAND_SCROLL(ID_SLIDE_H, ThumbHSL::OnSlideH)
EVT_COMMAND_SCROLL(ID_SLIDE_S, ThumbHSL::OnSlideS)
EVT_COMMAND_SCROLL(ID_SLIDE_L, ThumbHSL::OnSlideL)
END_EVENT_TABLE()
