/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2012
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tickr.h"
#include "tickr_socket.h"

#ifdef G_OS_WIN32
extern FILE	*stdout_fp, *stderr_fp;
#endif

/*
 *can use IPv4 or IPv6
 */
#ifndef G_OS_WIN32
static void *get_in_addr(struct sockaddr *s_a)
{
	if (s_a->sa_family == AF_INET)
		return &(((struct sockaddr_in *)s_a)->sin_addr);
	else
		return &(((struct sockaddr_in6 *)s_a)->sin6_addr);
}
#endif

/*
 * open stream socket in non-blocking mode and connect to host
 * return socket fd (> 0) / SOCK_CREATE_ERROR (-1 on Linux) if error
 */
sockt connect_to_host(const char *host, const char *portnum_str)
{
	sockt			sock;
#ifndef G_OS_WIN32
	char			ipa_str[INET6_ADDRSTRLEN];
#else
	u_long			i_mode = 1;	/* != 0 to enable non-blocking mode */
#endif
	struct addrinfo		hints, *server_info, *ptr;
	fd_set			read_set, write_set;
	struct timeval		timeout;
	int			s_opt_value;
	socklen_t		s_opt_len = sizeof(sockt);
	int			i;

	/* addrinfo stuff */
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
/*#ifdef VERBOSE_OUTPUT
	FPRINTF_FFLUSH_3(STD_OUT, "Resolving %s ... ", host);
	FPRINTF_FFLUSH_3(STD_OUT, "Resolving %s ...\n", host);
#endif*/
	if ((i = getaddrinfo(host, portnum_str, &hints, &server_info)) != 0) {
		if (USE_PROXY_FLAG) {
#ifndef G_OS_WIN32
			WARNING(FALSE, 4, "getaddrinfo() error: ", host, ": ", gai_strerror(i));
#else
			WARNING(FALSE, 4, "getaddrinfo() error: ", host, ": ", sock_error_message());
#endif
		} else {
#ifndef G_OS_WIN32
			FPRINTF_FFLUSH_4(STD_ERR, "getaddrinfo() error: %s: %s\n", host, gai_strerror(i));
#else
			FPRINTF_FFLUSH_4(STD_ERR, "getaddrinfo() error: %s: %s\n", host, sock_error_message());
#endif
		}
		return -1;
	}
/*#ifdef VERBOSE_OUTPUT
	FPRINTF_FFLUSH_2(STD_OUT, "Done\n");
#endif*/
	/* we get a list */
	for (ptr = server_info; ptr != NULL; ptr = ptr->ai_next) {
		/* create socket */
		if ((sock = socket(ptr->ai_family, ptr->ai_socktype, ptr->ai_protocol)) == SOCK_CREATE_ERROR) {
			FPRINTF_FFLUSH_3(STD_ERR, "Error: %s\n", sock_error_message());
			continue;
		}
		/* set socket in non-blocking mode */
#ifndef G_OS_WIN32
		if ((i = fcntl(sock, F_GETFL, 0)) == SOCK_FUNC_ERROR) {
			FPRINTF_FFLUSH_3(STD_ERR, "fcntl() error: %s\n", sock_error_message());
			CLOSE_SOCK(sock);
			break;
		} else if (fcntl(sock, F_SETFL, i | O_NONBLOCK) == SOCK_FUNC_ERROR) {
			FPRINTF_FFLUSH_3(STD_ERR, "fcntl() error: %s\n", sock_error_message());
			CLOSE_SOCK(sock);
			break;
		}
#else
		if (ioctlsocket(sock, FIONBIO, &i_mode) == SOCK_FUNC_ERROR) {
			FPRINTF_FFLUSH_3(STD_ERR, "ioctlsocket() error %s\n", sock_error_message());
			CLOSE_SOCK(sock);
			break;
		}
#endif
		/* get IP addr from server_info */
#ifdef VERBOSE_OUTPUT
#ifndef G_OS_WIN32
		inet_ntop(ptr->ai_family, get_in_addr((struct sockaddr *)ptr->ai_addr),
			ipa_str, sizeof(ipa_str));
		/*FPRINTF_FFLUSH_4(STD_OUT, "Connecting to %s (%s) ...\n", ipa_str, host);*/
		FPRINTF_FFLUSH_4(STD_OUT, "Connecting to %s (%s) ... ", ipa_str, host);
#else
		/* available only on vista and above but we don't really need that (do we?) so disabled
		InetNtop(ptr->ai_family, get_in_addr((struct sockaddr *)ptr->ai_addr), s, sizeof(s));*/
		/*FPRINTF_FFLUSH_3(STD_OUT, "Connecting to %s ...\n", host);*/
		FPRINTF_FFLUSH_3(STD_OUT, "Connecting to %s ... ", host);
#endif
#endif
		/* connect */
		if ((i = connect(sock, ptr->ai_addr, ptr->ai_addrlen)) == SOCK_FUNC_ERROR &&
#ifndef G_OS_WIN32
		 		errno == EINPROGRESS) {
#else
				WSAGetLastError() == WSAEWOULDBLOCK) {
#endif
			/* as socket is in non-blocking mode, we must use select() */
			FD_ZERO(&read_set);
			FD_ZERO(&write_set);
			FD_SET(sock, &read_set);
			FD_SET(sock, &write_set);
			timeout.tv_sec = CONNECT_TIMEOUT;
			timeout.tv_usec = 0;
			if ((i = select(sock + 1, &read_set, &write_set, NULL, &timeout)) == SOCK_FUNC_ERROR) {
				FPRINTF_FFLUSH_3(STD_ERR, "select() error: %s\n", sock_error_message());
			} else if (i == 0){
				FPRINTF_FFLUSH_2(STD_ERR, "Timed out\n");
			} else if (FD_ISSET(sock, &read_set) || FD_ISSET(sock, &write_set)) {
				if (getsockopt(sock, SOL_SOCKET, SO_ERROR,
						(void *)(&s_opt_value), &s_opt_len) == SOCK_FUNC_ERROR) {
					FPRINTF_FFLUSH_3(STD_ERR, "getsockopt() error: %s\n", sock_error_message());
				} else if (s_opt_value == 0) {
#ifdef VERBOSE_OUTPUT
					FPRINTF_FFLUSH_2(STD_OUT, "OK\n");
#endif
					freeaddrinfo(server_info);
					return sock;
				}
#ifndef G_OS_WIN32
				FPRINTF_FFLUSH_3(STD_ERR, "getsockopt(): %s\n", strerror(s_opt_value));
#else
				FPRINTF_FFLUSH_3(STD_ERR, "getsockopt(): %s\n", win32_error_msg(s_opt_value));
#endif
				CLOSE_SOCK(sock);
				break;
			}
			CLOSE_SOCK(sock);
			break;
		} else if (i == 0) {
#ifdef VERBOSE_OUTPUT
			FPRINTF_FFLUSH_2(STD_OUT, "OK\n");
#endif
			freeaddrinfo(server_info);
			return sock;
		} else {
			FPRINTF_FFLUSH_3(STD_ERR, "connect() error: %s\n", sock_error_message());
			CLOSE_SOCK(sock);
		}
	}
	freeaddrinfo(server_info);
	return -1;
}

int writable_data_is_available_on_socket(sockt sock)
{
	fd_set		write_set;
	struct timeval	timeout;
	int		i;

	FD_ZERO(&write_set);
	FD_SET(sock, &write_set);
	timeout.tv_sec = SEND_RECV_TIMEOUT_SEC;
	timeout.tv_usec = SEND_RECV_TIMEOUT_USEC;
	if ((i = select(sock + 1, NULL, &write_set, NULL, &timeout)) == SOCK_FUNC_ERROR) {
		FPRINTF_FFLUSH_3(STD_ERR, "select() error: %s\n", sock_error_message());
		return SELECT_ERROR;
	} else if (i == 0) {
		return SELECT_TIMED_OUT;
	} else {
		if (FD_ISSET(sock, &write_set))
			return SELECT_TRUE;
		else
			return SELECT_FALSE;
	}
}

int readable_data_is_available_on_socket(sockt sock)
{
	fd_set		read_set;
	struct timeval	timeout;
	int		i;

	FD_ZERO(&read_set);
	FD_SET(sock, &read_set);
	timeout.tv_sec = SEND_RECV_TIMEOUT_SEC;
	timeout.tv_usec = SEND_RECV_TIMEOUT_USEC;
	if ((i = select(sock + 1, &read_set, NULL, NULL, &timeout)) == SOCK_FUNC_ERROR) {
		FPRINTF_FFLUSH_3(STD_ERR, "select() error: %s\n", sock_error_message());
		return SELECT_ERROR;
	} else if (i == 0) {
		return SELECT_TIMED_OUT;
	} else {
		if (FD_ISSET(sock, &read_set))
			return SELECT_TRUE;
		else
			return SELECT_FALSE;
	}
}

/*
 * return n bytes sent or SOCK_FUNC_ERROR (-1 on Linux) if error (connection
 * closed by server or ?)
 */
int send_full(sockt sock, const char *str)
{
	int	len = strlen(str), i, j = 0;

	while (writable_data_is_available_on_socket(sock) == SELECT_TRUE) {
		if ((i = send(sock, str + j, len, 0)) != SOCK_FUNC_ERROR) {
			if (i > 0) {
				j += i;
				len -= i;
				if (len == 0)
					break;
			} else {
				/* something to do? */
			}
		} else {
			j = SOCK_FUNC_ERROR;
#ifndef G_OS_WIN32
			if (errno == EPIPE) {
#else
			if ((i = WSAGetLastError()) == WSAECONNRESET || i == WSAECONNABORTED ||\
					i == WSAESHUTDOWN) {
#endif
#ifdef VERBOSE_OUTPUT
				FPRINTF_FFLUSH_2(STD_ERR, "Connection closed by server\n");
#endif
			} else {
				FPRINTF_FFLUSH_3(STD_ERR, "send() error: %s\n", sock_error_message());
			}
			break;
		}
	}
	return j;
}

/*
 * return response = recv_full(socket, &bytes_received, &status) or NULL if error
 * -> status = SOCK_OK, CONNECTION_CLOSED_BY_SERVER, RECV_ERROR or SOCK_SHOULD_BE_CLOSED
 * -> allocate memory for response (must be freed afterwards with free2() if != NULL)
 */
char *recv_full(sockt sock, int *bytes_received, int *status)
{
	char	*response, *full_response;
	int	i;

	*bytes_received = 0;
	*status = RECV_ERROR;
	response = malloc2(RECV_CHUNK_LEN + 1);
	response[0] = '\0';
	full_response = l_str_new(response);
	while (readable_data_is_available_on_socket(sock) == SELECT_TRUE) {
		if ((i = recv(sock, response, RECV_CHUNK_LEN, 0)) != SOCK_FUNC_ERROR) {
			if (i > 0) {
				response[MIN(i, RECV_CHUNK_LEN)] = '\0';
				full_response = l_str_cat(full_response, response);
				*bytes_received += i;
				*status = SOCK_OK;
			} else if (i == 0) {
				*status = CONNECTION_CLOSED_BY_SERVER;
#ifdef VERBOSE_OUTPUT
				FPRINTF_FFLUSH_2(STD_ERR, "Connection closed by server\n");
#endif
				break;
			}
		} else {
#ifndef G_OS_WIN32
			*status = RECV_ERROR;
			l_str_free(full_response);
			full_response = NULL;
#ifdef VERBOSE_OUTPUT
			FPRINTF_FFLUSH_3(STD_ERR, "recv() error: %s\n", sock_error_message());
#endif
#else
			if ((i = WSAGetLastError()) == WSAECONNRESET || i == WSAECONNABORTED ||\
					i == WSAESHUTDOWN) {
				*status = SOCK_SHOULD_BE_CLOSED;
#ifdef VERBOSE_OUTPUT
				FPRINTF_FFLUSH_2(STD_ERR, "Connection closed by server\n");
#endif
			} else {
				*status = RECV_ERROR;
				l_str_free(full_response);
				full_response = NULL;
#ifdef VERBOSE_OUTPUT
				FPRINTF_FFLUSH_3(STD_ERR, "recv() error: %s\n", win32_error_msg(i));
#endif
			}
#endif
			break;
		}
	}
	free2(response);
	return full_response;
}

#ifdef G_OS_WIN32
const char *win32_error_msg(int i)
{
	static char	str[1024];
	gchar		*win32_str;

	win32_str = g_win32_error_message(i);
	str_n_cpy(str, (const char *)win32_str, 1023);
	g_free(win32_str);
	return (const char *)str;
}
#endif

const char *sock_error_message()
{
	static char	str[1024];

#ifndef G_OS_WIN32
	str_n_cpy(str, strerror(errno), 1023);
#else
	str_n_cpy(str, win32_error_msg(WSAGetLastError()), 1023);
#endif
	return (const char *)str;
}
