/***************************************************************************
                             th-job.c
                             --------
    begin                : Tue Nov 09 2004
    copyright            : (C) 2004 by Tim-Philipp Mller
    email                : tim centricular net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "th-job.h"
#include "th-marshal.h"
#include "th-utils.h"

#include <string.h>
#include <unistd.h>

#include <gst/gst.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gtk/gtk.h>

#include <glib/gi18n.h>

#define DEFAULT_PICTURE_SIZE    TH_PICTURE_SIZE_MEDIUM_LARGE

#define DEFAULT_AUDIO_BITRATE   128  /* kbps */

#define TH_JOB_ERROR  (g_quark_from_static_string("th-job-error-quark"))

/* check free disk space every X seconds */
#define DISK_SPACE_CHECK_INTERVAL  15

/* pause if less than this many MB of
 *  free disk space are left */
#define MIN_FREE_DISK_SPACE   5.0

#define th_atomic_uint_get(p_int)   ((guint)g_atomic_int_get((gint *)(p_int)))
#define th_atomic_uint_set(p_int,i) g_atomic_int_set((gint *)(p_int),(gint)(i))
#define th_atomic_uint_inc(p_int)   g_atomic_int_inc((gint *)(p_int))

enum
{
	PROP_DEVICE = 1,
	PROP_OUTPUT_FN,
	PROP_TITLE_NUM,
	PROP_TITLE_LENGTH,
	PROP_NUM_CHAPTERS,
	PROP_PICTURE_SIZE,
	PROP_VIDEO_BITRATE,         /* in kbps */
	PROP_AUDIO_BITRATE,         /* in kbps */
	PROP_AUDIO_STREAM_ID,
	PROP_AUDIO_ALL,             /* Flag whether to record all streams */
	PROP_NUM_SUB_STREAMS,
	PROP_TARGET_OUTPUT_SIZE,    /* in MB   */
	PROP_TARGET_QUALITY,
	PROP_BYTES_WRITTEN,         /* bytes written, accessed from multiple threads          */
	PROP_CURRENT_FRAME,
	PROP_NUM_FRAMES_ENCODED,    /* number of frames encoded so far                        */
	PROP_TOTAL_FRAMES_ENCODED,  /* total number of frames encoded so far (can't be reset) */
	PROP_WEIGHT,                /* some imaginative 'weight' for total progress report)   */
	PROP_CROP_TOP,
	PROP_CROP_LEFT,
	PROP_CROP_RIGHT,
	PROP_CROP_BOTTOM,
	PROP_PICTURE_SIZE_X,
	PROP_PICTURE_SIZE_Y,
	PROP_TITLE_ID,
	PROP_PIPELINE_PAUSED,
	PROP_IS_MAIN_TITLE,
	PROP_TITLE_TAG,              /* tag title string   */
	PROP_COMMENT_TAG             /* tag comment string */
};

enum
{
	OUT_OF_DISK_SPACE,
	LAST_SIGNAL
};

struct _ThJobPrivate
{
	gchar         *device;       /* device            */
	gchar         *output_fn;
	guint          title_num;
	guint          title_length; /* length in seconds */
	guint          num_chapters;

	gchar         *title_id; /* unique ID string identifying disc and title */

	GList         *sub_streams;   /* list of ThJobSubStream structs   */

	guint          video_bitrate;
	guint          audio_bitrate;
	
	guint          audio_stream_id; /* logical audio stream selected */
	guint          sub_stream_id;   /* logical subpicture stream selected
	                                 * (G_MAXUINT = NONE) */
	gboolean       audio_all; /* Set if all audiostreams should be recorded */

	guint          crop_left, crop_right;
	guint          crop_top, crop_bottom;

	gdouble        target_output_size;
	guint          target_quality; /* 0 = use target output size */

	ThPictureSize  picture_size;   /* scale factor from original after cropping */
	guint          size_x, size_y; /* input picture width / height              */

	GstElement    *pipeline;
	
	const gchar   *audio_pad_name;   /* interned string */
	const gchar   *video_pad_name;   /* interned string */
	const gchar   *audio_decoder;

	guint          bytes_total;
	guint          num_frames;        /* ATOMIC access */
	guint          num_frames_total;  /* ATOMIC access */

	GMainLoop     *loop;

	GstBuffer     *current_buf;
	guint          current_buf_timestamp; /* in milliseconds! */

	gdouble        pixel_aspect_ratio; /* set when pipeline is playing */

	GError        *err;
	gboolean       got_eos;
	gboolean       got_cancelled;
	
	gboolean       is_main_title;

	/* tag */
	gchar         *title_tag;
	gchar         *comment_tag;
};

typedef struct
{
	guint  sid;
	gchar *lang;
	gchar *pad_name;
} ThJobSubStream;

static void             job_class_init       (ThJobClass *klass);

static void             job_instance_init    (ThJob *j);

static void             job_finalize         (GObject *object);

static GdkPixbuf       *job_get_snapshot_from_current_buf (ThJob *j);

static void             job_set_video_bitrate_from_file_size (ThJob *j, gdouble targetsize_megabyte);

static gdouble          job_get_output_size_estimate (ThJob *j, guint video_bitrate);

static guint            job_get_total_audio_bitrate (ThJob *j);

static gdouble          job_calculate_weight (ThJob *j);


/* variables */

static GObjectClass    *j_parent_class;           /* NULL */

static guint            j_signals[LAST_SIGNAL];   /* all 0 */

/***************************************************************************
 *
 *   job_get_audio_stream_from_aid
 *
 ***************************************************************************/

static ThJobAudioStream *
job_get_audio_stream_from_aid (ThJob *j, guint aid)
{
	guint i;

	for (i = 0; i < j->audio_streams->len; ++i) {
		ThJobAudioStream *as;

		as = &g_array_index (j->audio_streams, ThJobAudioStream, i);
		if (as->aid == aid)
			return as;
	}

	return NULL;
}

/***************************************************************************
 *
 *   job_set_property
 *
 ***************************************************************************/

static void
job_set_property (GObject      *object,
                  guint         param_id,
                  const GValue *value,
                  GParamSpec   *pspec)
{
	gboolean  need_size_est = FALSE; /* FIXME: this is unused */
	ThJob    *j = (ThJob*) object;

	g_return_if_fail (TH_IS_JOB (j));
	
	switch (param_id)
	{
		case PROP_DEVICE:
			j->priv->device = g_value_dup_string (value);
			break;

		case PROP_TITLE_NUM:
			j->priv->title_num = g_value_get_uint (value);
			break;

		case PROP_NUM_CHAPTERS:
			j->priv->num_chapters = g_value_get_uint (value);
			break;

		case PROP_TITLE_LENGTH:
			j->priv->title_length = g_value_get_uint (value);
			need_size_est = TRUE;
			break;

		case PROP_OUTPUT_FN:
			j->priv->output_fn = g_value_dup_string (value);
			break;

		case PROP_PICTURE_SIZE:
			j->priv->picture_size = g_value_get_uint (value);
			break;

		case PROP_TARGET_OUTPUT_SIZE:
			job_set_video_bitrate_from_file_size (j, g_value_get_double (value));
			th_log ("New video bitrate: %u kbps (for target size %.1fM)", 
			        j->priv->video_bitrate, j->priv->target_output_size);
			j->priv->target_quality = 0;
			break;

		case PROP_TARGET_QUALITY:
			j->priv->target_quality = g_value_get_uint (value);
			break;

		case PROP_VIDEO_BITRATE:
			j->priv->video_bitrate = g_value_get_uint (value);
			need_size_est = TRUE;
			break;

		case PROP_AUDIO_BITRATE:
			j->priv->audio_bitrate = g_value_get_uint (value);
			need_size_est = TRUE;
			break;
		
		case PROP_AUDIO_STREAM_ID:
		{
			ThJobAudioStream *as;
			guint             new_aid;
			/* If you set a stream ID, you don't want all audio */
			j->priv->audio_all = FALSE;
			new_aid = g_value_get_uint (value);

			as = job_get_audio_stream_from_aid (j, new_aid);
			if (as != NULL)
			{
				j->priv->audio_stream_id = new_aid;
				th_log ("[%u] Selected Audio Stream: %s (aid = %u, pad = %s)\n", 
				        j->priv->title_num + 1, 
				        (as) ? as->lang : "??", 
				        new_aid, as->pad_name);
			}
			else
			{
				g_warning ("Unknown audio stream ID %u for job %p\n", new_aid, j);
			}
			break;
		}
		case PROP_AUDIO_ALL:
		{
			j->priv->audio_all = g_value_get_boolean(value);
			break;
		}
		case PROP_NUM_FRAMES_ENCODED:
		{
			th_atomic_uint_set (&j->priv->num_frames,
			                    g_value_get_uint (value));

			return; /* sic! avoid size estimate update */
		}
		
		case PROP_CROP_TOP:
			j->priv->crop_top = g_value_get_uint (value);
			break;

		case PROP_CROP_LEFT:
			j->priv->crop_left = g_value_get_uint (value);
			break;
		
		case PROP_CROP_RIGHT:
			j->priv->crop_right = g_value_get_uint (value);
			break;
		
		case PROP_CROP_BOTTOM:
			j->priv->crop_bottom = g_value_get_uint (value);
			break;

		case PROP_PICTURE_SIZE_X:
			j->priv->size_x = g_value_get_uint (value);
			break;
		
		case PROP_PICTURE_SIZE_Y:
			j->priv->size_y = g_value_get_uint (value);
			break;
		
		case PROP_TITLE_ID:
			j->priv->title_id = g_value_dup_string (value);
			break;
		
		case PROP_PIPELINE_PAUSED:
		{
			if (j->priv->pipeline)
			{
				GstState state;
				gboolean set_paused; 

				set_paused = g_value_get_boolean (value);
				gst_element_get_state (j->priv->pipeline, &state, NULL, 0);

				if (set_paused && state == GST_STATE_PLAYING)
				{
					gst_element_set_state (j->priv->pipeline, GST_STATE_PAUSED);
					th_log ("Set encoding pipeline to PAUSED.\n");
				}
				else if (!set_paused && state == GST_STATE_PAUSED)
				{
					gst_element_set_state (j->priv->pipeline, GST_STATE_PLAYING);
					th_log ("Set encoding pipeline to PLAYING.\n");
				}
			}
		}
		break;

		case PROP_IS_MAIN_TITLE:
			j->priv->is_main_title = g_value_get_boolean (value);
			break;

		case PROP_TITLE_TAG:
			g_free (j->priv->title_tag);
			j->priv->title_tag = g_value_dup_string (value);
			break;

		case PROP_COMMENT_TAG:
			g_free (j->priv->comment_tag);
			j->priv->comment_tag = g_value_dup_string (value);
			break;
		
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
	}
}

/***************************************************************************
 *
 *   job_get_property
 *
 ***************************************************************************/

static void
job_get_property (GObject      *object,
                  guint         param_id,
                  GValue       *value,
                  GParamSpec   *pspec)
{
	ThJob *j = (ThJob*) object;

	g_return_if_fail (TH_IS_JOB (j));
	
	switch (param_id)
	{
		case PROP_DEVICE:
			g_value_set_string (value, j->priv->device);
			break;

		case PROP_TITLE_NUM:
			g_value_set_uint (value, j->priv->title_num);
			break;

		case PROP_NUM_CHAPTERS:
			g_value_set_uint (value, j->priv->num_chapters);
			break;

		case PROP_TITLE_LENGTH:
			g_value_set_uint (value, j->priv->title_length);
			break;

		case PROP_OUTPUT_FN:
			g_value_set_string (value, j->priv->output_fn);
			break;
		
		case PROP_PICTURE_SIZE:
			g_value_set_uint (value, j->priv->picture_size);
			break;

		case PROP_TARGET_OUTPUT_SIZE:
			g_value_set_double (value, j->priv->target_output_size);
			break;
		
		case PROP_TARGET_QUALITY:
			g_value_set_uint (value, j->priv->target_quality);
			break;
		
		case PROP_VIDEO_BITRATE:
			g_value_set_uint (value, j->priv->video_bitrate);
			break;

		case PROP_AUDIO_BITRATE:
			g_value_set_uint (value, j->priv->audio_bitrate);
			break;
		
		case PROP_AUDIO_STREAM_ID:
			g_value_set_uint (value, j->priv->audio_stream_id);
			break;

		case PROP_AUDIO_ALL:
			g_value_set_boolean (value, j->priv->audio_all);
			break;

		case PROP_NUM_SUB_STREAMS:
			g_value_set_uint (value, g_list_length (j->priv->sub_streams));
			break;

		case PROP_BYTES_WRITTEN:
		{
			guint v = (guint) g_atomic_int_get ((gint*) &j->priv->bytes_total);
			g_value_set_uint (value, v);
			break;
		}

		case PROP_CURRENT_FRAME:
		{
			GdkPixbuf *pixbuf;
			
			pixbuf = job_get_snapshot_from_current_buf (j);
			
			g_value_take_object (value, pixbuf);
			
			if (pixbuf)
			{
				guint pos_ms;
				 
				pos_ms = (guint) g_atomic_int_get ((gint*) &j->priv->current_buf_timestamp);
				
				g_object_set_data (G_OBJECT (pixbuf), "timestamp", GUINT_TO_POINTER (pos_ms));
			}
		}
		break;
		
		case PROP_NUM_FRAMES_ENCODED:
		{
			guint v = th_atomic_uint_get (&j->priv->num_frames);
			g_value_set_uint (value, v);
			break;
		}
		
		case PROP_TOTAL_FRAMES_ENCODED:
		{
			guint v = th_atomic_uint_get (&j->priv->num_frames_total);
			g_value_set_uint (value, v);
			break;
		}
		
		case PROP_WEIGHT:
			g_value_set_double (value, job_calculate_weight (j));
			break;
		
		case PROP_CROP_TOP:
			g_value_set_uint (value, j->priv->crop_top);
			break;

		case PROP_CROP_LEFT:
			g_value_set_uint (value, j->priv->crop_left);
			break;
		
		case PROP_CROP_RIGHT:
			g_value_set_uint (value, j->priv->crop_right);
			break;
		
		case PROP_CROP_BOTTOM:
			g_value_set_uint (value, j->priv->crop_bottom);
			break;
		
		case PROP_PICTURE_SIZE_X:
			g_value_set_uint (value, j->priv->size_x);
			break;
		
		case PROP_PICTURE_SIZE_Y:
			g_value_set_uint (value, j->priv->size_y);
			break;
		
		case PROP_TITLE_ID:
			g_value_set_string (value, j->priv->title_id);
			break;

		case PROP_PIPELINE_PAUSED:
		{
			GstState state = GST_STATE_NULL;

			if (j->priv->pipeline != NULL) {
				gst_element_get_state (j->priv->pipeline, &state, NULL, 0);
			}

			g_value_set_boolean (value, state == GST_STATE_PAUSED);
			break;
		}
		
		case PROP_IS_MAIN_TITLE:
			g_value_set_boolean (value, j->priv->is_main_title);
			break;
		
		case PROP_TITLE_TAG:
			g_value_set_string (value, j->priv->title_tag);
			break;
		
		case PROP_COMMENT_TAG:
			g_value_set_string (value, j->priv->comment_tag);
			break;
		
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
	}
}

/***************************************************************************
 *
 *   job_class_init
 *
 ***************************************************************************/

static void
job_class_init (ThJobClass *klass)
{
	GObjectClass  *object_class; 

	object_class = G_OBJECT_CLASS (klass);
	
	j_parent_class = g_type_class_peek_parent (klass);

	object_class->finalize     = job_finalize;
	object_class->set_property = job_set_property;
	object_class->get_property = job_get_property;

	g_object_class_install_property (object_class, PROP_DEVICE,
	                                 g_param_spec_string ("device", 
	                                                      "device", 
	                                                      "device", 
	                                                      NULL, 
	                                                      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
	
	g_object_class_install_property (object_class, PROP_TITLE_NUM,
	                                 g_param_spec_uint ("title-num", 
	                                                    "title-num", 
	                                                    "title-num", 
	                                                    0, 99, 0, 
	                                                    G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property (object_class, PROP_NUM_CHAPTERS,
	                                 g_param_spec_uint ("num-chapters", 
	                                                    "num-chapters", 
	                                                    "num-chapters", 
	                                                    0, 999, 0, 
	                                                    G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property (object_class, PROP_TITLE_LENGTH,
	                                 g_param_spec_uint ("title-length", 
	                                                    "title-length", 
	                                                    "title-length", 
	                                                    0, G_MAXUINT, 0, 
	                                                    G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
	
	g_object_class_install_property (object_class, PROP_OUTPUT_FN,
	                                 g_param_spec_string ("output-fn", 
	                                                      "output-fn", 
	                                                      "output-fn", 
	                                                      NULL, 
	                                                      G_PARAM_READWRITE));
	
	g_object_class_install_property (object_class, PROP_PICTURE_SIZE,
	                                 g_param_spec_uint ("picture-size", 
	                                                    "picture-size", 
	                                                    "picture-size", 
	                                                    0, TH_PICTURE_SIZE_FULL, 
	                                                    DEFAULT_PICTURE_SIZE, 
	                                                    G_PARAM_READWRITE));

	g_object_class_install_property (object_class, PROP_VIDEO_BITRATE,
	                                 g_param_spec_uint ("video-bitrate", 
	                                                    "Video Bitrate in kbps", 
	                                                    "Video Bitrate in kbps", 
	                                                    1, G_MAXUINT, 500,
	                                                    G_PARAM_READWRITE));
	
	g_object_class_install_property (object_class, PROP_AUDIO_BITRATE,
	                                 g_param_spec_uint ("audio-bitrate", 
	                                                    "Audio Bitrate in kbps", 
	                                                    "Audio Bitrate in kbps", 
	                                                    1, 512, 128, 
	                                                    G_PARAM_READWRITE));
	
	g_object_class_install_property (object_class, PROP_AUDIO_STREAM_ID,
	                                 g_param_spec_uint ("audio-stream-id", 
	                                                    "Which Audio Stream to Encode", 
	                                                    "Which Audio Stream to Encode", 
	                                                    0, G_MAXUINT, G_MAXUINT, 
	                                                    G_PARAM_READWRITE));

	g_object_class_install_property (object_class, PROP_AUDIO_ALL,
	                                 g_param_spec_boolean ("audio-all", 
	                                                    "Whether to encode all streams", 
	                                                    "Whether to encode all streams", 
	                                                    FALSE, 
	                                                    G_PARAM_READWRITE));

	/* FIXME: get rid of this useless property and replace with direct
	 * structure access instead */
	g_object_class_install_property (object_class, PROP_NUM_SUB_STREAMS,
	                                 g_param_spec_uint ("num-sub-streams", 
	                                                    "Number of Subpicture Streams", 
	                                                    "Number of Subpicture Streams", 
	                                                    0, 32, 0, 
	                                                    G_PARAM_READABLE));

	g_object_class_install_property (object_class, PROP_TARGET_OUTPUT_SIZE,
	                                 g_param_spec_double ("target-output-size", 
	                                                      "Target Output Size in MB", 
	                                                      "Target Output Size in MB", 
	                                                      0.1, G_MAXDOUBLE, 695.0, 
	                                                      G_PARAM_READWRITE));

	g_object_class_install_property (object_class, PROP_TARGET_QUALITY,
	                                 g_param_spec_uint ("target-quality", 
	                                                    "Target Quality (0=use file size)", 
	                                                    "Target Quality (0=use file size)", 
	                                                    0, 63, 0, 
	                                                    G_PARAM_READWRITE));

	g_object_class_install_property (object_class, PROP_BYTES_WRITTEN, 
	                                 g_param_spec_uint ("bytes-written", 
	                                                    "Number of bytes written",
	                                                    "Number of bytes written", 
	                                                    0, G_MAXUINT, 0, G_PARAM_READABLE));

	g_object_class_install_property (object_class, PROP_CURRENT_FRAME, 
	                                 g_param_spec_object ("current-frame", 
	                                                      "current-frame",
	                                                      "current-frame", 
	                                                      GDK_TYPE_PIXBUF, G_PARAM_READABLE));

	g_object_class_install_property (object_class, PROP_NUM_FRAMES_ENCODED, 
	                                 g_param_spec_uint ("num-frames-encoded", 
	                                                    "num-frames-encoded",
	                                                    "num-frames-encoded", 
	                                                    0, G_MAXUINT, 0, G_PARAM_READWRITE));

	g_object_class_install_property (object_class, PROP_TOTAL_FRAMES_ENCODED, 
	                                 g_param_spec_uint ("total-frames-encoded", 
	                                                    "total-frames-encoded",
	                                                    "total-frames-encoded", 
	                                                    0, G_MAXUINT, 0, G_PARAM_READABLE));
	
	g_object_class_install_property (object_class, PROP_WEIGHT, 
	                                 g_param_spec_double ("weight", 
	                                                      "weight",
	                                                      "weight", 
	                                                       0.0, G_MAXDOUBLE, 0.0, G_PARAM_READABLE));

	g_object_class_install_property (object_class, PROP_CROP_TOP, 
	                                 g_param_spec_uint ("crop-top", 
	                                                    "top cropping",
	                                                    "top cropping", 
	                                                    0, G_MAXUINT, 0, G_PARAM_READWRITE));

	g_object_class_install_property (object_class, PROP_CROP_LEFT, 
	                                 g_param_spec_uint ("crop-left", 
	                                                    "left cropping",
	                                                    "left cropping", 
	                                                    0, G_MAXUINT, 0, G_PARAM_READWRITE));

	g_object_class_install_property (object_class, PROP_CROP_RIGHT, 
	                                 g_param_spec_uint ("crop-right", 
	                                                    "right cropping",
	                                                    "right cropping", 
	                                                    0, G_MAXUINT, 0, G_PARAM_READWRITE));

	g_object_class_install_property (object_class, PROP_CROP_BOTTOM, 
	                                 g_param_spec_uint ("crop-bottom", 
	                                                    "bottom cropping",
	                                                    "bottom cropping", 
	                                                    0, G_MAXUINT, 0, G_PARAM_READWRITE));

	g_object_class_install_property (object_class, PROP_PICTURE_SIZE_X, 
	                                 g_param_spec_uint ("picture-size-x", 
	                                                    "Raw picture width input",
	                                                    "Raw picture width of input", 
	                                                    0, G_MAXUINT, 0, 
	                                                    G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
	
	g_object_class_install_property (object_class, PROP_PICTURE_SIZE_Y, 
	                                 g_param_spec_uint ("picture-size-y", 
	                                                    "Raw picture height of input",
	                                                    "Raw picture height of input", 
	                                                    0, G_MAXUINT, 0,
	                                                    G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property (object_class, PROP_TITLE_ID,
	                                 g_param_spec_string ("title-id", 
	                                                      "title-id", 
	                                                      "title-id", 
	                                                      NULL, 
	                                                      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
	
	g_object_class_install_property (object_class, PROP_PIPELINE_PAUSED, 
	                                 g_param_spec_boolean ("pipeline-paused", 
	                                                       "pipeline-paused",
	                                                       "pipeline-paused", 
	                                                       FALSE,
	                                                       G_PARAM_READWRITE));

	g_object_class_install_property (object_class, PROP_IS_MAIN_TITLE, 
	                                 g_param_spec_boolean ("is-main-title", 
	                                                       "is-main-title",
	                                                       "is-main-title", 
	                                                       FALSE,
	                                                       G_PARAM_READWRITE));
	
	g_object_class_install_property (object_class, PROP_TITLE_TAG,
	                                 g_param_spec_string ("title-tag", 
	                                                      "title-tag", 
	                                                      "title-tag", 
	                                                      NULL, 
	                                                      G_PARAM_READWRITE));
	
	g_object_class_install_property (object_class, PROP_COMMENT_TAG,
	                                 g_param_spec_string ("comment-tag", 
	                                                      "comment-tag", 
	                                                      "comment-tag", 
	                                                      NULL, 
	                                                      G_PARAM_READWRITE));
	
	j_signals[OUT_OF_DISK_SPACE] = g_signal_new ("out-of-disk-space",
	                                             G_TYPE_FROM_CLASS (object_class),
	                                             G_SIGNAL_RUN_FIRST,
	                                             G_STRUCT_OFFSET (ThJobClass, out_of_disk_space),
	                                             NULL, NULL,
	                                             th_marshal_VOID__VOID,
	                                             G_TYPE_NONE, 0);
}

/***************************************************************************
 *
 *   job_instance_init
 *
 ***************************************************************************/

static void
job_instance_init (ThJob *j)
{
	j->priv = g_new0 (ThJobPrivate, 1);
	
	j->priv->audio_stream_id = G_MAXUINT;
	j->priv->sub_stream_id = G_MAXUINT;

	j->audio_streams = g_array_new (TRUE, TRUE, sizeof (ThJobAudioStream));
}

/***************************************************************************
 *
 *   job_finalize
 *
 ***************************************************************************/

static void
job_finalize (GObject *object)
{
	ThJob *j = (ThJob*) object;

	th_log ("ThJob: finalize (%p)\n", j);

	if (j->priv->pipeline)
	{
		gst_object_ref (GST_OBJECT (j->priv->pipeline));
		gst_element_set_state (j->priv->pipeline, GST_STATE_NULL);
		gst_object_unref (GST_OBJECT (j->priv->pipeline));
		gst_object_unref (GST_OBJECT (j->priv->pipeline));
	}

	memset (j->priv, 0xab, sizeof (ThJobPrivate));
	g_free (j->priv);
	j->priv = NULL;

	g_array_free (j->audio_streams, TRUE);

	/* chain up */
	j_parent_class->finalize (object);
}


/***************************************************************************
 *
 *   th_job_get_type
 *
 ***************************************************************************/

GType
th_job_get_type (void)
{
	static GType type; /* 0 */

	if (type == 0)
	{
		static GTypeInfo info =
		{
			sizeof (ThJobClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) job_class_init,
			NULL, NULL,
			sizeof (ThJob),
			0,
			(GInstanceInitFunc) job_instance_init
		};

		type = g_type_register_static (G_TYPE_OBJECT, "ThJob", &info, 0);
	}

	return type;
}


/***************************************************************************
 *
 *   job_migrate_old_config_files
 *
 *   Moves config cache files from ~/.thoggen/config-cache/XXX.xml
 *    to ~/.cache/thoggen/title-config-XXX.xml
 *
 ***************************************************************************/

static void
job_migrate_old_config_files (ThJob *job)
{
	const gchar *entry;
	gchar       *dirname;
	GDir        *dir;

	g_return_if_fail (job->priv->title_id != NULL);

	dirname = g_strdup_printf ("%s/.thoggen/config-cache", g_get_home_dir());

	dir = g_dir_open (dirname, 0, NULL);

	/* dir probably doesn't exist, or isn't accessible */
	if (dir == NULL)
		goto done;

	while ((entry = g_dir_read_name (dir)))
	{
		if (strlen (entry) > 4 && g_str_has_suffix (entry, ".xml"))
		{
			gchar *old_path, *buf;
			gsize  buflen;

			old_path = g_strdup_printf ("%s/.thoggen/config-cache/%s", g_get_home_dir(), entry);

			if (g_file_get_contents (old_path, &buf, &buflen, NULL))
			{
				gchar *new_fn = g_strdup_printf ("title-config-%s", entry);
				gchar *new_path = th_get_user_cache_fn (new_fn);
				FILE  *f = fopen (new_path, "w");
				if (f)
				{
					th_log ("Moving '%s' => '%s' ...\n", old_path, new_path);
					if (fwrite (buf, buflen, 1, f) == 1)
					{
						unlink (old_path);
					}
					fclose (f);
				}

				g_free (new_path);
				g_free (new_fn);
			}

			g_free (old_path);
		}
	}

	g_dir_close (dir);

done:

	/* remove ~/.thoggen/config-cache dir if empty */
	rmdir (dirname);
	g_free (dirname);

	/* remove ~/.thoggen dir if empty */
	dirname = g_strdup_printf ("%s/.thoggen", g_get_home_dir());
	rmdir (dirname);
	g_free (dirname);
}

/***************************************************************************
 *
 *   job_get_config_filename
 *
 ***************************************************************************/

static gchar *
job_get_config_filename (ThJob *job, GError **err)
{
	static gboolean  done_migration; /* FALSE */
	gchar           *fn, *fullpath;

	if (!done_migration)
	{
		job_migrate_old_config_files (job);
		done_migration = TRUE;
	}

	g_return_val_if_fail (job->priv->title_id != NULL, NULL);

	fn = g_strdup_printf ("title-config-%s.xml", job->priv->title_id);

	fullpath = th_get_user_cache_fn (fn);

	g_free (fn);

	return fullpath;
}

/***************************************************************************
 *
 *   th_job_save_config
 *
 ***************************************************************************/

gboolean
th_job_save_config (ThJob *job)
{
	const gchar *props[] = { "crop-top",  "crop-left", "crop-right", "crop-bottom",
	                         "picture-size", "audio-bitrate", "target-output-size",
	                         "target-quality", "output-fn", "title-tag", "comment-tag" };
	GString *xml;
	guint    n;
	
	g_return_val_if_fail (TH_IS_JOB (job), FALSE);
	
	xml = g_string_new (_("\n<!-- This file is not important. Feel free to delete it at any time. -->\n\n"));
	g_string_append (xml, "<thoggen>\n");
	g_string_append_printf (xml, " <job id='%s'>\n", job->priv->title_id);

	for (n = 0;  n < G_N_ELEMENTS (props);  ++n)
	{
		GParamSpec *pspec;
		
		pspec = g_object_class_find_property (G_OBJECT_GET_CLASS (job), props[n]);
		if (pspec)
		{
			GValue val = { 0, };
			gchar *valstr = NULL;
		
			g_value_init (&val, pspec->value_type);
			
			g_object_get_property (G_OBJECT (job), props[n], &val);
		
			if (G_VALUE_HOLDS_UINT (&val))
			{
				 valstr = g_strdup_printf ("%u", g_value_get_uint (&val));
			}
			else if (G_VALUE_HOLDS_DOUBLE (&val))
			{
				 valstr = g_strdup_printf ("%.3f", g_value_get_double (&val));
			}
			else if (G_VALUE_HOLDS_STRING (&val))
			{
				valstr = g_value_dup_string (&val);
			}
			else
			{
				g_warning ("%s: cannot serialise ThJob property '%s'. Unexpected type '%s'\n", 
				           G_STRLOC, props[n], g_type_name (pspec->value_type));
			}

			if (valstr)
			{
				gchar *val_escaped;

				val_escaped = g_markup_escape_text (valstr, -1);
				g_string_append_printf (xml, "  <%s>%s</%s>\n", props[n], val_escaped, props[n]);
				g_free (val_escaped);
				g_free (valstr);
				valstr = NULL;
			}

			g_value_unset (&val);
		}
	}
	
	g_string_append (xml, " </job>\n</thoggen>\n\n");

	{
		gchar *fn;

		if ((fn = job_get_config_filename (job, NULL)))
		{
			FILE *f = fopen (fn, "w");
			if (f)
			{
				if (fwrite (xml->str, xml->len, 1, f) != 1)
				{
					g_warning ("Failed to save job configuration to file '%s': %s\n", 
					           fn, g_strerror (errno));
				}
				fclose (f);
			}
			g_free (fn);
		}
	}
	
	g_string_free (xml, TRUE);

	return TRUE;
}

/***************************************************************************
 *
 *   job_try_load_config_parse_tag_text
 *
 ***************************************************************************/

static void
job_try_load_config_parse_tag_text (GMarkupParseContext *ctx,
                                    const gchar         *txt, 
                                    gsize                len, 
                                    gpointer             data, 
                                    GError             **err)
{
	const gchar *prop_name;
	GParamSpec  *pspec;
	ThJob       *j = TH_JOB (data);

	prop_name = g_markup_parse_context_get_element (ctx);
	if (g_str_equal (prop_name, "thoggen") 
	 || g_str_equal (prop_name, "job"))
		return;

	pspec = g_object_class_find_property (G_OBJECT_GET_CLASS (j), prop_name);
	if (pspec == NULL)
		return;

	if (pspec->value_type == G_TYPE_UINT)
	{
		g_object_set (j, prop_name, atoi (txt), NULL);
	}
	else if (pspec->value_type == G_TYPE_DOUBLE)
	{
		gfloat d = 0.0;
		if (sscanf (txt, "%f", &d) == 1) /* eeek, locale dependent */
		{
			g_object_set (j, prop_name, (gdouble) d, NULL);
		}
		else
		{
			g_warning ("%s: cannot deserialise ThJob property '%s'. '%s' does not look like a double value\n", 
			           G_STRLOC, prop_name, txt);
		}
	}
	else if (pspec->value_type == G_TYPE_STRING)
	{
		gchar *s = g_strndup (txt, len);
		if (!g_str_equal (prop_name, "output-fn"))
		{
			g_object_set (j, prop_name, s, NULL);
		}
		else if (s != NULL)
		{
			/* if it's the output filename, only restore the
			 *  previously saved one, if the directory still
			 *  exists */
			gchar *dir = g_path_get_dirname (s);
			if (g_file_test (dir, G_FILE_TEST_IS_DIR))
				g_object_set (j, "output-fn", s, NULL);
			g_free (dir);
		}
		g_free (s);
	}
	else
	{
		g_warning ("%s: cannot deserialise ThJob property '%s'. Unexpected type '%s'\n", 
		           G_STRLOC, prop_name, g_type_name (pspec->value_type));
	}
}

/***************************************************************************
 *
 *   job_try_load_config
 *
 ***************************************************************************/

static void
job_try_load_config (ThJob *job)
{
	GMarkupParser parser = { NULL, NULL, job_try_load_config_parse_tag_text, NULL, NULL };
	GMarkupParseContext *ctx;
	GError *err = NULL;
	gchar  *fn, *xml;
	gsize   len;

	fn = job_get_config_filename (job, NULL);
	if (fn == NULL)
		return;
	
	if (!g_file_get_contents (fn, &xml, &len, NULL))
	{
		g_free (fn);
		return;
	}
	
	ctx = g_markup_parse_context_new (&parser, 0, job, NULL);
	if (!g_markup_parse_context_parse (ctx, xml, (gssize) len, &err))
	{
		g_warning ("Error parsing config cache file '%s': %s\n", fn, err->message);
		g_error_free (err);
	}
	g_markup_parse_context_free (ctx);
	g_free (fn);
	g_free (xml);
}

/***************************************************************************
 *
 *   th_job_new
 *
 ***************************************************************************/

ThJob *
th_job_new (const gchar *device, 
            const gchar *title_id,
            guint        title,
            guint        len_secs, 
            guint        num_chapters, 
            guint        raw_size_x, 
            guint        raw_size_y)
{
	ThJob *j;

	g_return_val_if_fail (device != NULL, NULL);
	g_return_val_if_fail (title_id != NULL, NULL);
	g_return_val_if_fail (len_secs > 0, NULL);
	
	j = (ThJob *) g_object_new (TH_TYPE_JOB, 
	                            "device", device,
	                            "title-num", title,
	                            "title-length", len_secs,
	                            "num-chapters", num_chapters,
	                            "picture-size", DEFAULT_PICTURE_SIZE,
	                            "audio-bitrate", DEFAULT_AUDIO_BITRATE,
	                            "picture-size-x", raw_size_x,
	                            "picture-size-y", raw_size_y,
	                            "title-id", title_id,
	                            NULL);

	/* No, there isn't any logic to the values below */
	
	if (len_secs < 15*60) /* shorter than 15 mins? => 6 MB / minute */
	{
		job_set_video_bitrate_from_file_size (j, 6.0 * len_secs / 60.0);
	}
	else if (len_secs < 35*60) /* 15-35 mins? => 8 MB / minute */
	{
		job_set_video_bitrate_from_file_size (j, 8.0 * len_secs / 60.0);
	}
	else if (len_secs < 60*60) /* 35-60 mins? => 7 MB / minute */
	{
		job_set_video_bitrate_from_file_size (j, 7.0 * len_secs / 60.0);
	}
	else /* > 1 hour? => 1 CD size */
	{
		job_set_video_bitrate_from_file_size (j, 695.0);
	}
		
	job_try_load_config (j);

	return j;
}

/***************************************************************************
 *
 *   th_job_get_effective_picture_size
 *
 *   TODO: this function needs to take cropping into account later
 *
 ***************************************************************************/

gboolean
th_job_get_effective_picture_size (ThJob         *job, 
                                   ThPictureSize  picsize, 
                                   guint         *p_width, 
                                   guint         *p_height)
{
	guint hcrop, vcrop;

	const struct { guint mult; guint div; } table[] = 
	{
		{ 1, 8 }, { 1, 4 }, { 1, 2 }, { 2, 3 }, { 3, 4 }, { 5, 6 }, { 1, 1 }
	};

	g_return_val_if_fail (TH_IS_JOB (job), FALSE);
	g_return_val_if_fail (p_width != NULL, FALSE);
	g_return_val_if_fail (p_height != NULL, FALSE);
	g_return_val_if_fail (picsize <= TH_PICTURE_SIZE_FULL, FALSE);

	hcrop = GST_ROUND_UP_2 (job->priv->crop_left) + GST_ROUND_UP_2 (job->priv->crop_right);
	vcrop = GST_ROUND_UP_2 (job->priv->crop_top) + GST_ROUND_UP_2 (job->priv->crop_bottom);

	*p_width = GST_ROUND_UP_4 (((job->priv->size_x - hcrop) * table[picsize].mult) / table[picsize].div);
	*p_height = GST_ROUND_UP_4 (((job->priv->size_y - vcrop) * table[picsize].mult) / table[picsize].div);

	return TRUE;
}

/***************************************************************************
 *
 *   job_get_output_size_estimate
 *
 ***************************************************************************/

static gdouble
job_get_output_size_estimate (ThJob *j, guint video_bitrate)
{
	gdouble  total_bitrate_kbps, total_size_kbit, est_output_size;
	
	total_bitrate_kbps = (gdouble) video_bitrate;
	total_bitrate_kbps +=  (gdouble) job_get_total_audio_bitrate (j);
	
	/* Add 1.75% overhead for the Ogg encapsulation (ca. 1-2% apparently) */
	total_bitrate_kbps = total_bitrate_kbps * 1.0175;
	
	total_size_kbit = total_bitrate_kbps * (gdouble) j->priv->title_length;

	est_output_size = total_size_kbit / (8.0 * 1024.0);

	/* g_print ("Estimated Output Size: %.2fM  with video @ %u kbps, audio @ %u kbps, Length = %u:%02uh\n", 
	         est_output_size, video_bitrate, audio_bitrate,
	         length / 3600, (length % 3600) / 60); */
	
	return est_output_size;
}

/***************************************************************************
 *
 *   job_get_total_audio_bitrate
 *
 *   Returns the sum of the bitrates of all (selected) audio streams
 *
 ***************************************************************************/

static guint
job_get_total_audio_bitrate (ThJob * job)
{
  guint num_audio_streams;

  /* when we support audio bitrate configuration and audio encoder selection
   * etc. we need to do something more sophisticated here */
  if (job->priv->audio_all) {
    num_audio_streams = job->audio_streams->len;
  } else {
    num_audio_streams = 1;
  }

  return num_audio_streams * job->priv->audio_bitrate;
}

/***************************************************************************
 *
 *   job_set_video_bitrate_from_file_size
 *
 ***************************************************************************/

static void
job_set_video_bitrate_from_file_size (ThJob *j, gdouble targetsize_megabyte)
{
	gdouble kilobits, kbps_total, kbps_video;

	g_return_if_fail (TH_IS_JOB (j));
	g_return_if_fail (targetsize_megabyte > 0.0);

	kilobits = targetsize_megabyte * 1024.0 * 8.0;
	
	kbps_total = kilobits / (gdouble) j->priv->title_length;
	
	/* Subtract 1.75% overhead for the Ogg encapsulation 
	 * (ca. 1-2% apparently; 1.2% in my test for a full movie) */
	kbps_total = kbps_total * 0.9825;
	
	kbps_video = kbps_total - (gdouble) job_get_total_audio_bitrate (j);

	j->priv->target_output_size = targetsize_megabyte;
	
	g_object_set (j, "video-bitrate", (guint) kbps_video, NULL);
}


/***************************************************************************
 *
 *   job_error_cb
 *
 ***************************************************************************/

static void
job_error_cb (ThJob *j, GstMessage *msg, GstBus *bus)
{
	GError *err = NULL;
	gchar *dbgmsg;

	gst_message_parse_error (msg, &err, &dbgmsg);

	th_log ("========= ERROR from element %s =========",
	        GST_OBJECT_NAME (msg->src));

	th_log ("%s (%s)\n", (err) ? err->message : "(null error)", dbgmsg);
	j->priv->err = err;
	g_main_loop_quit (j->priv->loop);
	g_free (dbgmsg);
	err = NULL;
}

/***************************************************************************
 *
 *   job_eos_cb
 *
 ***************************************************************************/

static void
job_eos_cb (ThJob *j, GstMessage *msg, GstBus *bus)
{
	th_log ("========= EOS =========");
	j->priv->got_eos = TRUE;
	g_main_loop_quit (j->priv->loop);
}

/***************************************************************************
 *
 *   job_output_identity_handoff_cb
 *
 ***************************************************************************/
#include <sys/time.h>
#include <sys/resource.h>
#include <errno.h>

static void
job_output_identity_handoff_cb (ThJob *j, GstBuffer *buf, GstElement *fakesink)
{
	g_atomic_int_add ((gint*) &j->priv->bytes_total, (gint)GST_BUFFER_SIZE(buf));
}

/***************************************************************************
 *
 *   job_video_caps_notify_cb
 *
 ***************************************************************************/

static void
job_video_caps_notify_cb (ThJob *j, GObject *obj, GParamSpec *pspec, GstElement *video_enc)
{
	const GstCaps *caps;
	const GValue  *par_val;
	GstStructure  *structure;
	gdouble        n, d;
	
	if (!GST_IS_PAD (obj) || !GST_PAD_IS_SINK (GST_PAD (obj)))
		return;

	j->priv->pixel_aspect_ratio = 0.0;

	caps = gst_pad_get_negotiated_caps (GST_PAD (obj));

	if (caps == NULL)
		return;

	g_assert (GST_CAPS_IS_SIMPLE (caps));
	
	structure = gst_caps_get_structure (caps, 0);
	g_assert (structure != NULL);

	par_val = gst_structure_get_value (structure, "pixel-aspect-ratio");
	
	if (par_val == NULL)
		return;

	g_assert (G_IS_VALUE (par_val) && GST_VALUE_HOLDS_FRACTION (par_val));
	
	n = (gdouble) gst_value_get_fraction_numerator (par_val);
	d = (gdouble) gst_value_get_fraction_denominator (par_val);
	j->priv->pixel_aspect_ratio = n / d;

	th_log ("pixel-aspect-ratio = %.0f/%0.f = %.3f", n, d, j->priv->pixel_aspect_ratio);
}

/***************************************************************************
 *
 *   job_print_benchmark_stats
 *
 ***************************************************************************/

static void
job_print_benchmark_stats (ThJob *j)
{
	struct rusage  u;
	guint          v;

	v = th_atomic_uint_get (&j->priv->num_frames_total);
	
	if ((v % 1000) > 0)
		return;

	if (getrusage (RUSAGE_SELF, &u) == 0)
	{
		gdouble usrtime, systime;

		usrtime = (gdouble) u.ru_utime.tv_sec;
		systime = (gdouble) u.ru_stime.tv_sec;

		th_log ("Encoded %u frames.", v);
		th_log ("  User time     = %4.0f secs (%.2f msecs/frame)",
		         usrtime, (usrtime * 1000.0) / (gdouble) v);
		th_log ("  User+Sys time = %4.0f secs (%.2f msecs/frame)", 
		         usrtime + systime, ((usrtime + systime) * 1000.0) / (gdouble) v);
	}
	else
	{
		th_log ("Encoded %u frames. getrusage() failed: %s", v, g_strerror (errno));
	}
}

/***************************************************************************
 *
 *   job_video_probe_cb
 *
 *   Called whenever a buffer arrives at the sink pad of the video encoder
 *
 ***************************************************************************/

static gboolean
job_video_probe_cb (GstPad *sink_pad, GstBuffer *buf, ThJob *j)
{
	g_assert (buf != NULL);
	
	th_atomic_uint_inc (&j->priv->num_frames);
	th_atomic_uint_inc (&j->priv->num_frames_total);

	if (GST_BUFFER_TIMESTAMP_IS_VALID (buf)) {
		GstBuffer **curbuf_addr = &j->priv->current_buf;
		guint ms;

		ms = GST_BUFFER_TIMESTAMP (buf) / (1000*1000);

		gst_buffer_replace ((GstMiniObject **) curbuf_addr, buf);
		th_atomic_uint_set (&j->priv->current_buf_timestamp, ms);
	}

	/* print some stats */
	job_print_benchmark_stats (j);
	
	return TRUE; /* do not remove data from stream */
}

/***************************************************************************
 *
 *   job_create_pipeline_setup_queues
 *
 ***************************************************************************/

static void
job_create_pipeline_setup_queues (ThJob * j)
{
  guint i;

  /* configure audio in queues */
  for (i = 0; i < j->audio_streams->len; ++i) {
    ThJobAudioStream *as;

    as = &g_array_index (j->audio_streams, ThJobAudioStream, i);

    /* FIXME: use as->selected once the config dialog gets fixed up */
    if (j->priv->audio_all || j->priv->audio_stream_id == as->aid) {
      gchar enc_name[32];

      g_snprintf (enc_name, sizeof (enc_name), "q-a-in-%u", as->aid);

      /* unset/disable all limits apart from the bytes limit */
      th_bin_set_child_properties (GST_BIN (j->priv->pipeline), enc_name,
          "max-size-buffers", (guint) 0,
          "max-size-bytes", (guint) 10 * 1024 * 1024,
         "max-size-time", (guint64) 0,
          NULL);
    }
  }

  /* configure video in queue */
  th_bin_set_child_properties (GST_BIN (j->priv->pipeline), "q-v-in",
      "max-size-buffers", (guint) 0,
      "max-size-bytes", (guint) 10 * 1024 * 1024,
      "max-size-time", (guint64) 0,
      NULL);
}


/***************************************************************************
 *
 *   job_probe_pads_new_pad_cb
 *
 ***************************************************************************/

static void
job_probe_pads_new_pad_cb (ThJob *j, GstPad *new_pad, GstElement *dvddemux)
{
	GstPadLinkReturn  linkret;
	ThJobAudioStream *as = NULL;
	const gchar      *pad_name;
	GstElement       *queue, *fakesink, *pipeline;
	GstPad           *queue_sink_pad;

	pipeline = GST_ELEMENT_PARENT (dvddemux);

	pad_name = GST_OBJECT_NAME (new_pad);

	as = job_get_audio_stream_from_aid (j, j->priv->audio_stream_id);

	/* FIXME: sanity check, remove again */
	{
	  ThJobAudioStream *as2;
	  guint aid = G_MAXUINT;

	  if (sscanf (pad_name, "audio_%u", &aid) == 1) {
	    as2 = job_get_audio_stream_from_aid (j, aid);
	    if (as2) {
	      th_log ("Mapped caps %s => decoder %s",
	              gst_caps_to_string (gst_pad_get_caps (new_pad)),
	              as2->decoder);
	    }
	  }
	}

	if (j->priv->video_pad_name == NULL  &&  g_str_has_prefix (pad_name, "video_"))
	{
		j->priv->video_pad_name = g_intern_string (pad_name);
		th_log ("dvddemux::new-pad '%s' (using)", pad_name);
		goto hook_up;
	}
	else if (j->priv->audio_pad_name == NULL
	      && g_str_has_prefix (pad_name, "audio_")
	      && as != NULL && g_str_equal (pad_name, as->pad_name))
	{
		const gchar *mime = NULL;
		GstCaps *caps;
		
		j->priv->audio_pad_name = g_intern_string (pad_name);
		j->priv->audio_decoder = "decodebin";
		
		caps = gst_pad_get_caps (new_pad);
		if (caps)
		{
			GstStructure *s = gst_caps_get_structure (caps, 0);
			if (s)
			{
				mime = gst_structure_get_name (s);
				if (gst_structure_has_name (s, "audio/x-ac3"))
					j->priv->audio_decoder = "a52dec";
				else if (gst_structure_has_name (s, "audio/x-dts"))
					j->priv->audio_decoder = "dtsdec";
				else if (gst_structure_has_name (s, "audio/x-lpcm"))
					j->priv->audio_decoder = "dvdlpcmdec";
				else if (gst_structure_has_name (s, "audio/mpeg"))
					j->priv->audio_decoder = "mad";
			}
			gst_caps_unref (caps);
		}
		
		th_log ("dvddemux::new-pad '%s' (using) (%s) (%s)", 
		         pad_name, j->priv->audio_decoder, (mime) ? mime : "???");
		goto hook_up;
	}
	
	th_log ("dvddemux::new-pad '%s' (skipping)", pad_name);
	return;

hook_up:

	queue = gst_element_factory_make ("queue", NULL);
	fakesink = gst_element_factory_make ("fakesink", NULL);
	gst_bin_add (GST_BIN (pipeline), queue);
	gst_bin_add (GST_BIN (pipeline), fakesink);
	if (!gst_element_link (queue, fakesink))
      g_warning ("failed to link queue <=> fakesink at %s", G_STRLOC);
	gst_element_set_state (fakesink, GST_STATE_PAUSED);
	queue_sink_pad = gst_element_get_pad (queue, "sink");
	linkret = gst_pad_link (new_pad, queue_sink_pad);
	gst_object_unref (queue_sink_pad);
	g_return_if_fail (GST_PAD_LINK_SUCCESSFUL (linkret));
	gst_element_set_state (queue, GST_STATE_PAUSED);

	fakesink = gst_bin_get_by_name (GST_BIN (pipeline), "dummyfakesink");
	if (fakesink)
	{
		th_log ("REMOVING DUMMY FAKESINK"); /* FIXME */
		gst_element_set_state (fakesink, GST_STATE_NULL);
		gst_bin_remove (GST_BIN (pipeline), fakesink);
		gst_object_unref (fakesink);
	}
}

/***************************************************************************
 *
 *   job_probe_pads
 *
 *   FIXME: this entire probing thing is very sucky, we should find
 *          a more elegant solution
 *
 ***************************************************************************/

static gboolean
job_probe_pads (ThJob *j, GError **err)
{
	GstElement *p, *demux;
	gulong      sigid;

	th_log ("Probing dvddemux audio and video pads ...");

	p = gst_parse_launch ("dvdreadsrc name=src ! dvddemux name=demux "
                          " fakesink name=dummyfakesink", err);
	if ((err && *err) || p == NULL)
	{
		if (p)
			gst_object_unref (GST_OBJECT (p));
		return FALSE;
	}

	th_bin_set_child_properties (GST_BIN (p), "src",
	              "device", j->priv->device,
	              "title", j->priv->title_num + 1,
	              NULL);
	
	demux = gst_bin_get_by_name (GST_BIN (p), "demux");

	sigid = g_signal_connect_swapped (demux, 
	                                  "pad-added",
	                                  G_CALLBACK (job_probe_pads_new_pad_cb),
	                                  j);

	gst_object_unref (demux);

	gst_element_set_state (p, GST_STATE_PAUSED);

	gst_element_get_state (p, NULL, NULL, 30 * GST_SECOND);	

	gst_element_set_state (p, GST_STATE_NULL);
	
	gst_object_unref (GST_OBJECT (p));

	if (j->priv->audio_pad_name == NULL  ||  j->priv->video_pad_name == NULL)
	{
		if (j->priv->audio_pad_name == NULL && j->priv->video_pad_name == NULL)
		{
			g_set_error (err, TH_JOB_ERROR, 0,
			             _("Failed to find video stream and audio stream."));
		}
		else if (j->priv->video_pad_name == NULL)
		{
			g_set_error (err, TH_JOB_ERROR, 0, 
			             _("Failed to find video stream."));
		}
		else /* j->priv->audio_pad_name == NULL */
		{
			g_set_error (err, TH_JOB_ERROR, 0, 
			             _("Failed to find audio stream with ID %d."), 
			             j->priv->audio_stream_id);
		}

		j->priv->audio_pad_name = NULL;
		j->priv->video_pad_name = NULL;

		return FALSE;
	}

	th_log ("Found audio and video pads.");

	return TRUE;
}

/***************************************************************************
 *
 *   job_check_free_disk_space
 *
 ***************************************************************************/

static gboolean
job_check_free_disk_space (ThJob *j)
{
	gboolean paused;

	g_object_get (j, "pipeline-paused", &paused, NULL);

	if (!paused)
	{
		gdouble  free_size;
		gchar   *out_dir;
	
		out_dir = g_path_get_dirname (j->priv->output_fn);
		free_size = th_utils_get_free_space_from_path (out_dir);

		/* pause when there's less than 2 MB of free space */
		if (free_size >= 0.0  &&  free_size < MIN_FREE_DISK_SPACE)
		{
			th_log ("Only %.1f MB of free disk space in %s!", free_size, out_dir);
			g_signal_emit (j, j_signals[OUT_OF_DISK_SPACE], 0);
			g_object_set (j, "pipeline-paused", TRUE, NULL);
		}
		
		g_free (out_dir);
	}
	
	return TRUE; /* call us again */
}

static void
job_pipeline_set_tags_on_encoder (ThJob *j, GstElement *element,
    const gchar *lang)
{
	if (!GST_IS_TAG_SETTER (element))
		return;

	if (j->priv->title_tag && *j->priv->title_tag)
	{
		gst_tag_setter_add_tags (GST_TAG_SETTER (element), 
		                         GST_TAG_MERGE_REPLACE, 
		                         GST_TAG_TITLE, j->priv->title_tag, 
		                         NULL);
	}

	if (j->priv->comment_tag && *j->priv->comment_tag)
	{
		gst_tag_setter_add_tags (GST_TAG_SETTER (element), 
		                         GST_TAG_MERGE_APPEND, 
		                         GST_TAG_COMMENT, j->priv->comment_tag, 
		                         NULL);
	}

	gst_tag_setter_add_tags (GST_TAG_SETTER (element), 
	                         GST_TAG_MERGE_APPEND, 
	                         GST_TAG_COMMENT, "Created with thoggen: http://thoggen.net", 
	                         NULL);

	if (lang != NULL && *lang != '\0' && g_utf8_validate (lang, -1, NULL)) {
	  gst_tag_setter_add_tags (GST_TAG_SETTER (element),
	      GST_TAG_MERGE_APPEND, GST_TAG_LANGUAGE_CODE, lang, NULL);
	}
}

/***************************************************************************
 *
 *   job_create_pipeline_setup_tagging
 *
 ***************************************************************************/

static void
job_create_pipeline_setup_tagging (ThJob *j, GstBin *pipeline)
{
	GstElement *tagsetter;
	guint i;

	/* FIXME: maybe do this by recursing into the pipeline
	 * instead of picking out the elements by name? */
	tagsetter = gst_bin_get_by_name (pipeline, "mux");
	job_pipeline_set_tags_on_encoder (j, tagsetter, NULL);
	gst_object_unref (tagsetter);

	for (i = 0; i < j->audio_streams->len; ++i) {
	  ThJobAudioStream *as;
	  gchar enc_name[32];

	  as = &g_array_index (j->audio_streams, ThJobAudioStream, i);
	  g_snprintf (enc_name, sizeof (enc_name), "a-encoder-%u", as->aid);
	  tagsetter = gst_bin_get_by_name (pipeline, enc_name);
	  if (tagsetter) {
	    job_pipeline_set_tags_on_encoder (j, tagsetter, as->lang);
            gst_object_unref (tagsetter);
	  }
	}

	tagsetter = gst_bin_get_by_name (pipeline, "v-encoder");
	job_pipeline_set_tags_on_encoder (j, tagsetter, NULL);
	gst_object_unref (tagsetter);
}

/***************************************************************************
 *
 *   job_create_pipeline_setup_video_encoder
 *
 ***************************************************************************/

static void
job_create_pipeline_setup_video_encoder (ThJob *j)
{
	GstElement *video_enc;
	GstPad     *sink_pad;
	
	video_enc = gst_bin_get_by_name (GST_BIN (j->priv->pipeline), "v-encoder");
	
	/* theoraenc - keyframe values suggested in a post by David Kuehling 
	 *  on 13 December 2004, 10:04h GMT on the theora mailing list 
	 *
	 *  "keyframe-freq", 3*64,
	 *  "keyframe-force", 4*64,
	 *  "sharpness", 2,
	 */
	if (j->priv->target_quality == 0)
	{
		th_log ("Using target video bitrate of %u.", j->priv->video_bitrate);
		
		g_object_set (video_enc, 
		              "quick", FALSE, 
		              "bitrate", j->priv->video_bitrate,
		              NULL);
	}
	else
	{
		th_log ("Using target video quality of %u.", j->priv->target_quality);

		g_object_set (video_enc, 
		              "quick", FALSE, 
		              "quality", j->priv->target_quality, 
		              NULL);
	}

	/* only sharpen a little bit for now */
	g_object_set (video_enc, "sharpness", 1, NULL);

	sink_pad = gst_element_get_pad (video_enc, "sink");
	g_return_if_fail (GST_IS_PAD (sink_pad));

	gst_pad_add_buffer_probe (sink_pad, G_CALLBACK (job_video_probe_cb), j);

	g_signal_connect_swapped (video_enc, 
	                          "deep-notify::caps",
	                          G_CALLBACK (job_video_caps_notify_cb),
	                          j);

	gst_object_unref (video_enc);
}

static void
job_create_pipeline_setup_audio_encoders (ThJob * j)
{
  guint i;

  for (i = 0; i < j->audio_streams->len; ++i) {
    ThJobAudioStream *as;

    as = &g_array_index (j->audio_streams, ThJobAudioStream, i);

    /* FIXME: fix up configuration dialog so we can use as->selected in future */
    if (j->priv->audio_all || j->priv->audio_stream_id == as->aid) {
      gchar enc_name[32];

      g_snprintf (enc_name, sizeof (enc_name), "a-encoder-%u", as->aid);

      /* configure vorbisenc */
      th_bin_set_child_properties (GST_BIN (j->priv->pipeline), enc_name,
          "bitrate", j->priv->audio_bitrate * 1000, "managed", TRUE,  NULL);
    }
  }
}

/***************************************************************************
 *
 *   job_create_pipeline
 *
 ***************************************************************************/

static gboolean
job_create_pipeline (ThJob *j, GError **err)
{
	const gchar *deinterlacer, *scale_method;
	GstElement  *pipe, *out_identity;
	GstBus      *bus;
	GString     *pipe_desc;
	guint        width, height;

	if (gst_default_registry_check_feature_version ("ffdeinterlace", 0, 10, 2))
	  deinterlacer = "ffdeinterlace";
	else
	  deinterlacer = "thdeinterlace";

	th_log ("Using %s for deinterlacing", deinterlacer);

	/* FIXME: always use 4-tap once we depend on -base 0.10.17 */
	if (gst_default_registry_check_feature_version ("videoscale", 0, 10, 17))
	  scale_method = "4-tap";
	else
	  scale_method = "bilinear";

	th_log ("Scaling method: %s", scale_method);

	if (!th_job_get_effective_picture_size (j, j->priv->picture_size, &width, &height))
		g_return_val_if_reached (FALSE);
	
	if (!job_probe_pads (j, err))
		return FALSE;

	th_log ("Size %u = %ux%u (cropping=%uL,%uR,%uT,%uB)",
	        j->priv->picture_size, width, height, j->priv->crop_left,
	        j->priv->crop_right, j->priv->crop_top, j->priv->crop_bottom);

	pipe_desc = g_string_new ("");

	/* muxer and filesink */
	g_string_append (pipe_desc,
	      "    oggmux name=mux                                           "
	      "  ! identity name=outidentity                                 "
	      "  ! filesink name=filesink                                    "
	      "                                                              ");

	/* source and demuxer */
	g_string_append (pipe_desc,
	      "    dvdreadsrc name=src                                       "
	      "  ! dvddemux name=demux                                       "
	      "                                                              ");

	/* audio transcoder(s) */
	{
	  guint i;

	  for (i = 0; i < j->audio_streams->len; ++i) {
	    ThJobAudioStream *as;

	    as = &g_array_index (j->audio_streams, ThJobAudioStream, i);

	    if (j->priv->audio_all || j->priv->audio_stream_id == as->aid) {
	      g_string_append_printf (pipe_desc,
	          "          demux.%s                                        "
	          "    ! queue name=q-a-in-%u                                "
	          "    ! %s                                                  "
	          "    ! audioconvert                                        "
	          "    ! audioresample                                       "
	          "    ! audio/x-raw-float,rate=44100,channels=2             "
	          "    ! vorbisenc name=a-encoder-%u                         "
	          "    ! mux.                                                "
	          "                                                          ",
	          as->pad_name, as->aid, j->priv->audio_decoder, as->aid);

	      th_log ("Adding audio stream %u, pad = %s\n", i, as->pad_name);
	    }
	  }
	}

	/* video transcoder  */	
	g_string_append_printf (pipe_desc,
	      "      demux.%s                                                "
	      "    ! queue name=q-v-in                                       "
	      "    ! mpeg2dec                                                "
	      "    ! thparsetter pixel-aspect-ratio=%u/%u                    "
	      "    ! %s                                                      "
	      "    ! videorate                                               "
	      "    ! video/x-raw-yuv,framerate=(fraction)%u/%u               "
	      "    ! videobox left=%u right=%u top=%u bottom=%u              "
	      "    ! videoscale method=%s                              "
	      "    ! queue max-size-buffers=5                                "
	      "    ! video/x-raw-yuv,width=%u,height=%u                      "
	      "    ! theoraenc name=v-encoder                                "
	      "    ! mux.                                                    "
	      "                                                              ",
	      j->priv->video_pad_name,
	      j->video_info.aspect_num, j->video_info.aspect_denom,
	      deinterlacer,
	      j->video_info.fps_num, j->video_info.fps_denom,
	      j->priv->crop_left, j->priv->crop_right, 
	      j->priv->crop_top, j->priv->crop_bottom,
	      scale_method,
	      width, height);
	
	pipe = gst_parse_launch (pipe_desc->str, err);
	g_string_free (pipe_desc, TRUE);
	pipe_desc = NULL;
	
	if ((err && *err) || pipe == NULL)
	{
		if (pipe)
			gst_object_unref (GST_OBJECT (pipe));
		return FALSE;
	}
	
	j->priv->pipeline = pipe;

	job_create_pipeline_setup_queues (j);

	{
		/* to avoid 'dereferencing type-punned pointer' compiler warnings */
		GstElement **pipeline_addr = &j->priv->pipeline;
	
		g_object_add_weak_pointer (G_OBJECT (j->priv->pipeline), 
		                           (gpointer*) pipeline_addr);

	}
	
	th_bin_set_child_properties (GST_BIN (pipe), "src", 
	              "device", j->priv->device,
	              "title", j->priv->title_num + 1,
	              NULL);

	th_bin_set_child_properties (GST_BIN (pipe), "filesink",
	              "location", j->priv->output_fn,
	              NULL);

	job_create_pipeline_setup_video_encoder (j);

	job_create_pipeline_setup_audio_encoders (j);

	/* oggmux */
	th_bin_set_child_properties (GST_BIN (pipe), "mux",
	              "max-delay", (guint64) GST_SECOND/2,
	              "max-page-delay", (guint64) GST_SECOND/2, 
	              NULL);

	job_create_pipeline_setup_tagging (j, GST_BIN (j->priv->pipeline));

	bus = gst_element_get_bus (j->priv->pipeline);

	g_signal_connect_swapped (bus, "message::eos",
	                          G_CALLBACK (job_eos_cb),
	                          j);

	g_signal_connect_swapped (bus, "message::error",
	                          G_CALLBACK (job_error_cb),
	                          j);

	gst_bus_add_signal_watch (bus);

	gst_object_unref (bus);

	out_identity = gst_bin_get_by_name (GST_BIN (pipe), "outidentity");

	g_signal_connect_swapped (out_identity, 
	                          "handoff",
	                          G_CALLBACK (job_output_identity_handoff_cb),
	                          j);

	gst_object_unref (out_identity);

	gst_element_set_state (j->priv->pipeline, GST_STATE_PLAYING);
	
	return TRUE;
}
 
/***************************************************************************
 *
 *   th_job_run
 *
 *   Returns FALSE if the job got cancelled via th_job_cancel(), or an error
 *    occured, otherwise it returns TRUE.
 *
 ***************************************************************************/

gboolean
th_job_run (ThJob *j, GError **err)
{
	gulong checkid;

	g_return_val_if_fail (TH_IS_JOB (j), FALSE);
	g_return_val_if_fail (j->priv->pipeline == NULL, FALSE);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	j->priv->err = NULL;

	j->priv->bytes_total = 0;
	j->priv->num_frames = 0;
	j->priv->num_frames_total = 0;
	
	if (j->priv->current_buf)
	{
		gst_buffer_unref (j->priv->current_buf);
		j->priv->current_buf = NULL;
	}

	if (!job_create_pipeline (j, err))
		return FALSE;

	th_job_save_config (j);
	
	j->priv->loop = g_main_loop_new (NULL, FALSE);

	checkid = g_timeout_add (DISK_SPACE_CHECK_INTERVAL*1000, 
	                         (GSourceFunc) job_check_free_disk_space, 
	                         j);
	
	g_main_loop_run (j->priv->loop);
	
	g_source_remove (checkid);

	if (j->priv->pipeline)
	{
		gst_object_ref (GST_OBJECT (j->priv->pipeline));
		gst_element_set_state (j->priv->pipeline, GST_STATE_NULL);
		gst_object_unref (GST_OBJECT (j->priv->pipeline));
		gst_object_unref (GST_OBJECT (j->priv->pipeline));
		j->priv->pipeline = NULL;
	}

	j->priv->audio_pad_name = NULL;
	j->priv->video_pad_name = NULL;

	if (j->priv->err)
	{
		if (err)
			*err = j->priv->err;
		else
			g_error_free (j->priv->err);
		
		j->priv->err = NULL;
		
		return FALSE;
	}

	return (!j->priv->got_cancelled);
}

/***************************************************************************
 *
 *   th_job_cancel
 *
 ***************************************************************************/

void
th_job_cancel (ThJob *j, gboolean remove_partial_file)
{
	g_return_if_fail (TH_IS_JOB (j));
	
	j->priv->got_cancelled = TRUE;
	
	if (remove_partial_file)
	{
		if (unlink (j->priv->output_fn) != 0 && errno != ENOENT)
		{
			g_warning ("Couldn't delete partial file '%s': %s\n", 
			           j->priv->output_fn, g_strerror (errno));
		}
	}

	g_main_loop_quit (j->priv->loop);

	if (j->priv->pipeline == NULL)
	{
		; /* got an error */
	}
}



/***************************************************************************
 *
 *   job_get_snapshot_from_current_buf
 *
 *   needs to be thread safe (accessed from main GUI thread)
 *
 ***************************************************************************/

static GdkPixbuf *
job_get_snapshot_from_current_buf (ThJob *j)
{
	GstBuffer **job_current_buf_addr = &j->priv->current_buf;
	GstBuffer *curbuf;
	GdkPixbuf *pixbuf, *pixbuf_stretched;
	guint      width, height;
	
	if (!th_job_get_effective_picture_size (j, j->priv->picture_size, &width, &height))
		g_return_val_if_reached (NULL);
	
	/* steal the buffer and take ownership of it, so that it can't 
	 *   get unreffed from the other thread before we have a 
	 *   chance to ref it (is this necessary?) */
	do
	{
		curbuf = g_atomic_pointer_get ((gpointer *) job_current_buf_addr);
	}
	while (!g_atomic_pointer_compare_and_exchange ((gpointer *) job_current_buf_addr, curbuf, NULL));

	if (curbuf == NULL)
		return NULL;
	
	pixbuf = th_pixbuf_from_yuv_i420_buffer (curbuf, width, height);

	if (!g_atomic_pointer_compare_and_exchange ((gpointer*) job_current_buf_addr, NULL, curbuf))
		gst_buffer_unref (curbuf);

	/* return it as is if we don't have the PAR */
	if (j->priv->pixel_aspect_ratio == 0.0)
		return pixbuf;
	
	/* don't stretch unless it's really noticeable */
	if (j->priv->pixel_aspect_ratio > 0.90 && j->priv->pixel_aspect_ratio < 1.10)
		return pixbuf;

	pixbuf_stretched = gdk_pixbuf_scale_simple (pixbuf, 
	        (gint) ((gdouble) width * j->priv->pixel_aspect_ratio),
	        height,
	        GDK_INTERP_BILINEAR);

	g_object_unref (pixbuf);

	return pixbuf_stretched;
}

/***************************************************************************
 *
 *   job_calculate_weight
 *
 ***************************************************************************/

static gdouble
job_calculate_weight (ThJob *j)
{
	guint width, height;

	if (!th_job_get_effective_picture_size (j, j->priv->picture_size, &width, &height))
		g_return_val_if_reached (384.0 * 288.0 * (gdouble) j->priv->title_length);

	return (gdouble) width * (gdouble) height * (gdouble) j->priv->title_length;
}


/***************************************************************************
 *
 *   th_job_set_filename_from_disc_title
 *
 ***************************************************************************/

void
th_job_set_filename_from_disc_title (ThJob *j, const gchar *disc_title)
{
	const gchar *homedir;
	gchar       *fn;

	g_return_if_fail (TH_IS_JOB (j));

	homedir = g_get_home_dir();

	if (disc_title && !g_str_equal(disc_title,"(no title)") && !g_str_equal(disc_title,_("(no title)")))
	{
		gchar *disc_title_lc;
		 
		disc_title_lc = g_utf8_strdown (disc_title, -1);
		g_strdelimit (disc_title_lc, "\\/ ", '-');
		
		fn = g_strdup_printf ("%s/%s-%02u.ogv", homedir, disc_title_lc, j->priv->title_num + 1);
		
		g_free (disc_title_lc);
	}
	else
	{
		/* strip title number from id (as counted from 0) */
		gchar *id = g_strdup (j->priv->title_id);
		
		if (id && strchr (id, '-'))
			*strchr (id, '-') = 0x00;

		fn = g_strdup_printf ("%s/dvd-%s-%02u.ogv", homedir, (id) ? id : "", j->priv->title_num + 1);
		
		g_free (id);
	}

	g_object_set (j, "output-fn", fn, NULL);
	
	g_free (fn);
}

/***************************************************************************
 *
 *   th_job_add_audio_stream
 *
 ***************************************************************************/

void
th_job_add_audio_stream (ThJob *j, guint aid, const gchar *pad_name,
    const gchar *lang, const gchar *desc, const gchar *decoder)
{
	ThJobAudioStream *as;
	const gchar      *cur_lang;

	g_return_if_fail (TH_IS_JOB (j));
	g_return_if_fail (aid <= 99);
	g_return_if_fail (lang != NULL);
	g_return_if_fail (desc != NULL);
	g_return_if_fail (pad_name != NULL);
	g_return_if_fail (decoder != NULL);
	
	as = g_new0 (ThJobAudioStream, 1);
	
	as->aid      = aid;
	as->lang     = g_intern_string (lang);
	as->desc     = g_intern_string (desc);
	as->pad_name = g_intern_string (pad_name);
	as->decoder  = g_intern_string (decoder);

	g_array_append_vals (j->audio_streams, as, 1);
	
	/* Get current locale; if none set, assume English */
	cur_lang = g_getenv ("LANG");
	if (cur_lang == NULL)
		cur_lang = g_getenv ("LC_ALL");
	if (cur_lang == NULL || g_str_equal (cur_lang, "C") || strlen (cur_lang) < 2)
		cur_lang = "en_GB";

	/* If stream has same language as locale, or we have no
	 *  default audio stream set yet, use this one as default */
	if ((as->lang && strncmp (cur_lang, as->lang, 2) == 0)
	 || j->priv->audio_stream_id == G_MAXUINT)
	{
		j->priv->audio_stream_id = aid;
		g_object_notify (G_OBJECT (j), "audio-stream-id");
		th_log ("[%u] Default Audio Stream: %s (aid = %u, pad = %s)", 
		        j->priv->title_num + 1, as->lang, aid, as->pad_name);
	}
}

/***************************************************************************
 *
 *   th_job_get_audio_stream
 *
 ***************************************************************************/

gboolean
th_job_get_audio_stream (ThJob *j, guint n, guint *p_aid, const gchar **p_lang, const gchar **p_desc)
{
	ThJobAudioStream *as;

	g_return_val_if_fail (TH_IS_JOB (j), FALSE);

	if (n >= j->audio_streams->len)
		return FALSE;

	as = &g_array_index (j->audio_streams, ThJobAudioStream, n);

	if (p_aid)
		*p_aid = as->aid;

	if (p_lang)
		*p_lang = as->lang;
	
	if (p_desc)
		*p_desc = as->desc;

	return TRUE;
}


/***************************************************************************
 *
 *   th_job_add_subpicture_stream
 *
 ***************************************************************************/

void
th_job_add_subpicture_stream (ThJob * j, guint sid, const gchar * pad_name,
    const gchar * lang)
{
	ThJobSubStream *ss;

	g_return_if_fail (TH_IS_JOB (j));
	g_return_if_fail (sid <= 99);
	g_return_if_fail (lang != NULL);
	g_return_if_fail (pad_name != NULL);
	
	ss = g_new0 (ThJobSubStream, 1);
	
	ss->sid      = sid;
	ss->lang     = g_strdup (lang);
	ss->pad_name = g_strdup (pad_name);

	j->priv->sub_streams = g_list_append (j->priv->sub_streams, ss);
	
	g_object_notify (G_OBJECT (j), "num-sub-streams");
}

/***************************************************************************
 *
 *   th_job_get_subpicture_stream
 *
 ***************************************************************************/

gboolean
th_job_get_subpicture_stream (ThJob * j, guint n, guint * p_sid,
    const gchar ** p_lang)
{
	ThJobSubStream *ss;

	g_return_val_if_fail (TH_IS_JOB (j), FALSE);
	
	ss = (ThJobSubStream*) g_list_nth_data (j->priv->sub_streams, n);
	if (ss == NULL)
		return FALSE;
	
	if (p_sid)
		*p_sid = ss->sid;

	if (p_lang)
		*p_lang = ss->lang;

	return TRUE;
}

/***************************************************************************
 *
 *   th_job_get_file_size_range
 *
 ***************************************************************************/

void
th_job_get_file_size_range (ThJob *job, gdouble *p_min_size, gdouble *p_max_size)
{
	g_return_if_fail (TH_IS_JOB (job));

	if (p_min_size)
		*p_min_size = job_get_output_size_estimate (job, 1);
	
	/* theoraenc currently only accepts video bitrates up to 2000 kbps (0.8.7) */
	if (p_max_size)
		*p_max_size = job_get_output_size_estimate (job, 2000);
}


/***************************************************************************
 *
 *   th_job_get_default_title
 *
 ***************************************************************************/

gchar *
th_job_get_default_title (ThJob *j)
{
	g_return_val_if_fail (TH_IS_JOB (j), NULL);

	if (j->priv->is_main_title)
		return g_strdup_printf (_("Title %u - Main Feature"), j->priv->title_num + 1);

	return g_strdup_printf (_("Title %u"), j->priv->title_num + 1);
}


