/*
    Theseus - maximum likelihood superpositioning of macromolecular structures

    Copyright (C) 2004-2007 Douglas L. Theobald

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the:

    Free Software Foundation, Inc.,
    59 Temple Place, Suite 330,
    Boston, MA  02111-1307  USA

    -/_|:|_|_\-
*/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <float.h>
#include "DLTmath.h"
#include "statistics.h"
#include "gamma_dist.h"
#include "invgamma_dist.h"


/* The Inverse Gamma distribution of order a > 0 is defined by:

   p(x) dx = {(b^c)/Gamma(c)} x^-{1+c} e^{-b/x} dx

   0 <= x < +inf
   scale parameter b > 0
   shape parameter c > 0.

   Mode = b/(c+1)
*/
double
invgamma_dev(const double b, const double c)
{
    return(1.0 / gamma_dev(1.0/b, c));
}


double
invgamma_pdf(const double x, const double b, const double c)
{
    double          p;

    if (x <= 0.0)
    {
        return (0.0);
    }
    else 
    {
        p = c*log(b) - lgamma(c) - (1.0+c)*log(x) - (b/x);
        return (exp(p));
    }
}


double
invgamma_lnpdf(const double x, const double b, const double c)
{
    if (x <= 0.0)
        return (-DBL_MAX);
    else 
        return (c*log(b) - lgamma(c) - (1.0+c)*log(x) - (b/x));
}


double
invgamma_cdf(const double x, const double b, const double c)
{
    return(InGammaQ(c, b/x));
}


double
invgamma_sdf(const double x, const double b, const double c)
{
    return(1.0 - invgamma_cdf(x, b, c));
}


double
invgamma_int(const double x, const double y, const double b, const double c)
{
/*     printf("\n~~~~~~~~% 12.6e % 12.6e", */
/*            integrate_qsimp(invgamma_pdf, b, c, x, y), */
/*            invgamma_sdf(x, b, c) - invgamma_sdf(y, b, c)); */
    return(invgamma_cdf(y, b, c) - invgamma_cdf(x, b, c));
}


/* This is correct, but I haven't proven it. */
double
invgamma_ent(const double b, const double c)
{
/*     printf("\n**********logL: % 10.6e % 10.6e % 10.6e % 10.6e % 10.6e", */
/*            lgamma(c), log(b), -(1.0 + c) * s_PolyGamma(c, 1), c, */
/*            -(lgamma(c) + log(b) - (1.0 +c) * s_PolyGamma(c, 1) + c)); */
    return(-(lgamma(c) + log(b) - (1.0 + c) * s_PolyGamma(c, 1) + c));
}


double
invgamma_logL(const double *data, const int num, const double b, const double c)
{
    int             i;
    double          logsum, invsum, logL;

    logsum = invsum = 0.0;
    for (i = 0; i < num; ++i)
    {
        logsum += log(data[i]);
        invsum += 1.0 / data[i];
    }

    logL = -(1.0 + c) * logsum - b * invsum + num * c * log(b) - num * lgamma(c);

    return(logL);
}


/* We must find the root of:

       F1 = ln(b) - digamma(c) - 1/N \Sum ln(x_i) = 0

   where the first derivative with repect to c (dF1/dc) is:

       F1' = -trigamma(c)
*/
static void
evalinvgammaEM(const double lnave, const double b, const double c,
               double *fx, double *dfx)
{
    *fx = log(b) - s_PolyGamma(c, 1) - lnave;
    *dfx = -s_PolyGamma(c, 2);
}


/* Fit a gamma distribution by maximum likelihood.
   Uses Newton-Raphson and E-M algorithm.
   Alternates between N-R for c (using current b)
   and ML for b (based on current c).
   Global convergence is probably poorer too.
   Takes more iterations than the pure N-R ML method below.
*/
double
invgamma_EMfit(const double *data, const int num, double *b, double *c, double *logL)
{
    double          lnave, harmave, ave, var, fx = 0.0, dfx = 1.0;
    double          guess_b, guess_c;
    int             i;
    double          tol = 1e-9;
    int             maxit = 300;

    lnave = harmave = ave = 0.0;
    for (i = 0; i < num; ++i)
    {
        if (data[i] < 0.0)
        {
            fprintf(stderr, "\n ERROR345: inverse gamma distributed data must be >= 0.0 ");
            return(-1.0);
        }
        else if(data[i] == 0.0)
            continue;
        else
        {
            ave += data[i];
            lnave += log(data[i]);
            harmave += (1.0 / data[i]);
        }
    }
    ave /= (double) num;
    lnave /= (double) num;
    harmave = (double) num / harmave;

    var = 0.0;
    for (i = 0; i < num; ++i)
        var += mysquare(data[i] - ave);
    var /= (double) num - 1;

    guess_b = *b = ave / var;
    guess_c = *c = var / mysquare(ave);

    guess_c = *c = 1.0;
    guess_b = *b = *c * harmave;

    /* Maximum likelihood fit.
       Uses Newton-Raphson to find ML estimate of c.
       We must find the root of:
    
           F1 = ln(b) - digamma(c) - 1/N \Sum ln(x_i) = 0
    
       where the first derivative with repect to c (dF1/dc) is:
    
           F1' = -trigamma(c)
    */
    printf("            b             c            fx           dfx @\n");
    for (i = 0; i < maxit; ++i)
    {
        printf("% 10.6e % 10.6e % 10.6e % 10.6e % 10.6e\n",
               *b, *c, fx, dfx, fx/dfx);
        evalinvgammaEM(lnave, *b, *c, &fx, &dfx);

        if (fabs(fx) < tol)
            break; /* success */

        *c -= (fx / dfx); /* Newton-Raphson correction */

        if (*c <= 0.0)
            *c = 0.1;

        if (*b <= 0.0)
            *b = 0.1;

        /* plug in result to get b */
        *b = *c * harmave;
    }
    printf("iter = %d\n", i);
    /* invgamma_logL(*b, *c); */
    fflush(NULL);

    if (i == maxit)
    {
        SCREAMS("no convergence!!!!!!!!!!!!!!!!");
        *b = guess_b;
        *c = guess_c;
    }

    return(chi_sqr_adapt(data, num, 0, logL, *b, *c, invgamma_pdf, invgamma_lnpdf, invgamma_int));
}


double
invgamma_fit_safe0(const double *data, const int num, double *b, double *c, double *logL)
{
    int             i, j;
    double         *x = malloc(num * sizeof(double));
    double         *y = malloc(num * sizeof(double));
    double          chi2;

    j = 0;
    for (i = 0; i < num; ++i)
    {
        if (data[i] > 0.0)
        {
            x[j] = 1.0 / data[i];
            y[j] = data[i];
            ++j;
        }
    }

    gamma_fit_no_stats(x, j, b, c);

    *b = 1.0 / *b;

    chi2 = chi_sqr_adapt(y, j, 0, logL, *b, *c, invgamma_pdf, invgamma_lnpdf, invgamma_int);

    free(x);
    free(y);

    return(chi2);
}


double
invgamma_fit(const double *data, const int num, double *b, double *c, double *logL)
{
    int             i, j;
    double         *x = malloc(num * sizeof(double));
    double          chi2;

    j = 0;
    for (i = 0; i < num; ++i)
    {
        if (data[i] > 0.0)
        {
            x[j] = 1.0 / data[i];
            ++j;
        }
        else
        {
            fprintf(stderr, "\n WARNING345: Inverse gamma distributed data must be > 0.0 ");
            fprintf(stderr, "\n WARNING345: Continuing but omitting data[%d] = %e \n\n", i, data[i]);
        }
    }

    if (j < 2)
    {
        fprintf(stderr, "\n ERROR346: Number of data points > 0.0 is only %d ", j);
        fprintf(stderr, "\n ERROR346: Aborting calculation \n\n");
        exit(EXIT_FAILURE);
    }

    gamma_fit_no_stats(x, j, b, c);

    *b = 1.0 / *b;

    chi2 = chi_sqr_adapt(data, num, 0, logL, *b, *c, invgamma_pdf, invgamma_lnpdf, invgamma_int);

    free(x);

    return(chi2);
}


double
invgamma_bayes_fit(const double *data, const int num, double *b, double *c, double *logL)
{
    int             i, j;
    double         *x = malloc(num * sizeof(double));
    double          chi2;

    j = 0;
    for (i = 0; i < num; ++i)
    {
        if (data[i] > 0.0)
        {
            x[j] = 1.0 / data[i];
            ++j;
        }
        else
        {
            fprintf(stderr, "\n WARNING345: Inverse gamma distributed data must be > 0.0 ");
            fprintf(stderr, "\n WARNING345: Continuing but omitting data[%d] = %e \n\n", i, data[i]);
        }
    }

    if (j < 2)
    {
        fprintf(stderr, "\n ERROR346: Number of data points > 0.0 is only %d ", j);
        fprintf(stderr, "\n ERROR346: Aborting calculation \n\n");
        exit(EXIT_FAILURE);
    }

    gamma_bayes_fit_no_stats(x, j, b, c);

    *b = 1.0 / *b;

    chi2 = chi_sqr_adapt(data, num, 0, logL, *b, *c, invgamma_pdf, invgamma_lnpdf, invgamma_int);

    free(x);

    return(chi2);
}


double
invgamma1_fit(const double *data, const int num, double *b, double *nullp, double *logL)
{
    int             i, j;
    double         *x = malloc(num * sizeof(double));
    double          chi2;

    j = 0;
    for (i = 0; i < num; ++i)
    {
        if (data[i] > 0.0)
        {
            x[j] = 1.0 / data[i];
            ++j;
        }
    }

    gamma1_fit(x, j, b, nullp, logL);

    *b = 1.0 / *b;

    chi2 = chi_sqr_adapt(x, j, 0, logL, *b, 1.0, invgamma_pdf, invgamma_lnpdf, invgamma_int);

    free(x);

    return(chi2);
}


double
invgamma_fit_guess(const double *data, const int num, double *b, double *c, double *logL)
{
    int             i;
    double          invb;
    double         *x = malloc(num * sizeof(double));

    for (i = 0; i < num; ++i)
        x[i] = 1.0 / data[i];

    gamma_fit_guess(x, num, &invb, c, logL);
/*     if (fabs(invb) < DBL_EPSILON) */
/*         *b = DBL_MAX; */
/*     else */
        *b = 1.0 / invb;

    free(x);

    return(chi_sqr_adapt(data, num, 0, logL, *b, *c, invgamma_pdf, invgamma_lnpdf, invgamma_int));
}


double
invgamma_minc_fit(const double *data, const int num, double *b, double *c, const double minc, double *logL)
{
    int             i;
    double          invb;
    double         *x = malloc(num * sizeof(double));

    for (i = 0; i < num; ++i)
        x[i] = 1.0/data[i];

    gamma_minc_fit(x, num, &invb, c, minc, logL);
    *b = 1.0 / invb;

    free(x);

    return(chi_sqr_adapt(data, num, 0, logL, *b, *c, invgamma_pdf, invgamma_lnpdf, invgamma_int));
}


double
invgamma_minc_opt_fit(const double *data, const int num, double *b, double *c, const double minc, double *logL)
{
    int             i;
    double          invb;
    double         *x = malloc(num * sizeof(double));

    for (i = 0; i < num; ++i)
        x[i] = 1.0/data[i];

    gamma_minc_opt_fit(x, num, &invb, c, minc, logL);
    *b = 1.0 / invb;

    free(x);

    return(chi_sqr_adapt(data, num, 0, logL, *b, *c, invgamma_pdf, invgamma_lnpdf, invgamma_int));
}


static void
evalinvgammaML(const double lnave, const double harmave, const double c,
               double *fx, double *dfx);

/* We must find the root of:

       F1 = ln((N*c)/\Sum(1/x_i)) - digamma(c) - 1/N \Sum ln(x_i) = 0

   where the first derivative with repect to c (dF1/dc) is:

       F1' =  1/c - trigamma(c)
*/
static void
evalinvgammaML(const double lnave, const double harmave, const double c,
               double *fx, double *dfx)
{
    *fx = log(c * harmave) - s_PolyGamma(c, 1) - lnave;
    *dfx = 1.0/c - s_PolyGamma(c, 2);
}


/* Fit an inverse gamma distribution by maximum likelihood.
*/
double
invgamma_BROKEN_fit(const double *data, const int num, double *b, double *c, double *logL)
{
    double          lnave, harmave, aveinv, invvar, x;
    double         *invx = malloc(num * sizeof(double));
    double          fx = 0.0, dfx = 1.0, fxdfx = 0.0;
    int             i, skip;
    int             maxit = 300;
    double          tol = 1e-9;

/*     double tmp = invgamma_transfit(data, num, b, c, logL); */
/*     printf("\n %-12s %11.3e %11.3e %9.2f %12.3f %14.1f %14.1f %14.1f %6.3f", */
/*            "invg_trans", *b, *c, tmp, *logL/num, *logL, *logL, *logL, 1.0); */

    lnave = harmave = 0.0;
    skip = 0;
    for (i = 0; i < num; ++i)
    {
        x = data[i];
        if (x > 0.0)
        {
            invx[i] = 1.0/x;
            lnave += log(x);      /* logarithmic average */
            harmave += (invx[i]); /* harmonic average */
        }
        else if(x == 0.0)
        {
            invx[i] = 0.0;
            ++skip;
            continue;
        }
        else
        {
            fprintf(stderr, "\n ERROR345: inverse gamma distributed data must be >= 0.0 ");
            return(-1.0);
        }
    }
    lnave /= (double) (num - skip);
    aveinv = harmave / (double) (num - skip);
    harmave = (double) (num - skip) / harmave;

    /* MMEs are:
           E(x) = b/(c-1)               if c > 1
           Var(x) = b^2/((c-1)^2(c-2))  if c > 2
       The MLE of b:
           b = c * Ave(x)_harmonic
       Using the first and third eqns above gives:
           c = E(x) / (E(x) - Ave(x)_harmonic)
       This is always positive as E(x) > Ave(x)_harmonic.
       In fact it is always >= 1.0.
    */

    invvar = 0.0;
    for (i = 0; i < num; ++i)
        invvar += mysquare(invx[i] - aveinv);
    invvar /= (double) (num - 1 - skip);

    /* MMEs from the regular gamma distribution (using inverse transformed data) */
    *c = aveinv * aveinv / invvar;
    *b = *c * harmave; /* this is actually the MLE given known c */

    /* Newton-Raphson to find ML estimate of c.
       We must find the root of:

           F1 = ln((N*c)/\Sum(1/x_i)) - digamma(c) - 1/N \Sum ln(x_i) = 0

       where the first derivative with repect to c (dF1/dc) is:

           F1' = 1/c - trigamma(c)
    */
/*     printf("\n            b             c            fx           dfx"); */
/*     printf("\n% 10.6e % 10.6e % 10.6e % 10.6e % 10.6e", */
/*            *b, *c, fx, dfx, fx/dfx); */
    for (i = 0; i < maxit; ++i)
    {
        evalinvgammaML(lnave, harmave, *c, &fx, &dfx);

        fxdfx = fx/dfx;

        if (fabs(fx) < tol && fabs(fxdfx) < tol)
            break; /* success */

        *c -= fxdfx; /* Newton-Raphson correction */

        if (*c <= 0.0)
            *c = 0.1;

        /* plug in result to get b */
        *b = *c * harmave;
/*         printf("\n% 10.6e % 10.6e % 10.6e % 10.6e % 10.6e", */
/*                *b, *c, fx, dfx, fx/dfx); */
    }
/*     printf("\niter = %d", i); */
    /* invgamma_logL(*b, *c); */

    if (i == maxit)
        printf("\n\n ERROR543: Newton-Raphson ML fit in invgamma_fit() did not converge.\n");

    free(invx);

    return(chi_sqr_adapt(data, num, 0, logL, *b, *c, invgamma_pdf, invgamma_lnpdf, invgamma_int));
}


static void
evalinvgammaMLmode(const double lnave, const double invave, const double b,
                   const double mode, double *fx, double *dfx)
{
/*     double          invmode = 1.0 / mode; */

/*     *fx = invmode */
/*           + (invmode * log(b)) */
/*           - (invmode * s_PolyGamma(b * invmode - 1.0, 1)) */
/*           - (invmode * lnave) */
/*           - invave; */

    double term = b/mode - 1.0;

    *fx = term/b - invave - lnave/mode + log(b)/mode - s_PolyGamma(term, 1) / mode;
    *dfx = -term/(b*b) + 2.0/(b*mode) - s_PolyGamma(term, 2)/(mode*mode);

/*     *fx = ((1.0 + log(b) - s_PolyGamma(b/mode - 1.0, 1) - lnave) / mode) - invave - (1.0 / b); */
/*  */
/*     *dfx = ((1.0 / b) - (s_PolyGamma(b/mode - 1.0, 2) / mode)) / mode + (1.0 / (b*b)); */
}


double
invgamma_mode_fit(const double *data, const int num, double *b, double *c, const double mode, double *logL)
{
    double          lnave, ave, invave, x;
    double          fx = 0.0, dfx = 1.0, fxdfx = 0.0;
    int             i;
    int             maxit = 40;
    double          tol = 1e-9;
    /* double          mode = 2.0; */

    ave = lnave = invave = 0.0;
    for (i = 0; i < num; ++i)
    {
        x = data[i];
        if (x > 0.0)
        {
            ave += x;
            lnave += log(x); /* logarithmic average */
            invave += 1.0/x; /* inverse average */
        }
        else
        {
            fprintf(stderr, "\n ERROR345: inverse gamma distributed data must be > 0.0 ");
            return(-1.0);
        }
    }

    ave /= num;
    lnave /= num;
    invave /= num;

    if (ave > mode)
        *b = 2.0 * mode * ave / (ave - mode);
    else
        *b = 1.0;

    if (*b > mode)
        *c = (*b - mode) / mode;
    else
        *c = 1.0;

    for (i = 0; i < maxit; ++i)
    {
        printf("% 10.6e % 10.6e % 10.6e % 10.6e % 10.6e\n",
               *b, *c, fx, dfx, fxdfx);

        evalinvgammaMLmode(lnave, invave, *b, mode, &fx, &dfx);

        fxdfx = fx/dfx;

        if (fabs(fx) < tol && fabs(fxdfx) < tol)
            break; /* success */

        *b -= fxdfx; /* Newton-Raphson correction */

        if (*b <= 0.0)
            *b = 0.1;

        /* plug in result to get c */
        if (*b > mode)
            *c = (*b - mode) / mode;
        else
            *c = 1.0;
    }

    printf("iter = %d\n", i);
    /* invgamma_logL(*b, *c); */

    if (i == maxit)
        printf("\n\n ERROR543: Newton-Raphson ML fit in invgamma_fit() did not converge.\n");

    return(chi_sqr_adapt(data, num, 0, logL, *b, *c, invgamma_pdf, invgamma_lnpdf, invgamma_int));
}



/* We must find the root of:

       F1 = log(c) + 1 - digamma(c) - \sum{log(x)}/K - \sum{1/x}/K = 0

   where the first derivative with repect to c (dF1/dc) is:

       F1' =  1/c - trigamma(c)
*/
static void
evalinvgamma_eq_bc_ML(const double lnave, const double invave, const double c,
               double *fx, double *dfx)
{
    *fx = log(c) + 1.0 - s_PolyGamma(c, 1) - lnave - invave;
    *dfx = 1.0/c - s_PolyGamma(c, 2);
}


/* Fit an inverse gamma distribution by maximum likelihood,
   assuming b = c.
*/
double
invgamma_eq_bc_fit(const double *data, const int num, double *b, double *c, double *logL, int init)
{
    double          lnave, invave, invvar, x;
    double          fx = 0.0, dfx = 1.0, fxdfx = 0.0;
    int             i, skip;
    int             maxit = 300;
    double          tol = 1e-9;

/*     double tmp = invgamma_transfit(data, num, b, c, logL); */
/*     printf("\n %-12s %11.3e %11.3e %9.2f %12.3f %14.1f %14.1f %14.1f %6.3f", */
/*            "invg_trans", *b, *c, tmp, *logL/num, *logL, *logL, *logL, 1.0); */

    lnave = invave = 0.0;
    skip = 0;
    for (i = 0; i < num; ++i)
    {
        x = data[i];
        if (x > 0.0)
        {
            lnave += log(x); /* logarithmic average */
            invave += 1.0/x; /* harmonic average */
        }
        else if(x == 0.0)
        {
            ++skip;
            continue;
        }
        else
        {
            fprintf(stderr, "\n ERROR345: inverse gamma distributed data must be > 0.0 ");
            return(-1.0);
        }
    }

    lnave /= (num - skip);
    invave /= (num - skip);

    /* MMEs are:
           E(x) = b/(c-1)               if c > 1
           Var(x) = b^2/((c-1)^2(c-2))  if c > 2
       The MLE of b:
           b = c * Ave(x)_harmonic
       Using the first and third eqns above gives:
           c = E(x) / (E(x) - Ave(x)_harmonic)
       This is always positive as E(x) > Ave(x)_harmonic.
       In fact it is always >= 1.0.
    */

    if (init == 0)
    {
        invvar = 0.0;
        for (i = 0; i < num; ++i)
            invvar += mysquare(1.0/data[i] - invave);
        invvar /= (double) (num - 1 - skip);

        /* MMEs from the regular gamma distribution (using inverse transformed data) */
        *c = invave * invave  / invvar;
        *b = *c/*  = 1.0 */;
    }
    else /* use the values passed as initial guesses */
    {
        *c = *b;
    }

	if (!isfinite(*b) || !isfinite(*c))
	{
		printf("\n ERROR04: b(%e) or c(%e) parameter in invgamma_eq_bc_fit() not finite\n",
			   *b, *c);
		fflush(NULL);
		exit(EXIT_FAILURE);
	}

    /* Newton-Raphson to find ML estimate of c.
       We must find the root of:

           F1 = log(c) + 1 - digamma(c) - \sum{log(x)}/K - \sum{1/x}/K = 0

       where the first derivative with repect to c (dF1/dc) is:

           F1' = 1/c - trigamma(c)
    */
    printf("            b             c            fx           dfx ~\n");
    printf("% 10.6e % 10.6e % 10.6e % 10.6e % 10.6e\n",
           *b, *c, fx, dfx, fx/dfx);

    for (i = 0; i < maxit; ++i)
    {
        evalinvgamma_eq_bc_ML(lnave, invave, *c, &fx, &dfx);

		if (!isfinite(*c))
		{
			printf("\n ERROR05: c(%e) parameter in invgamma_eq_bc_fit() not finite\n",
				   *c);
            printf("\n%3d: % 10.6e % 10.6e % 10.6e % 10.6e % 10.6e\n\n",
                   i, *b, *c, fx, dfx, fx/dfx);
			fflush(NULL);
			exit(EXIT_FAILURE);
		}

        fxdfx = fx/dfx;

        if (fabs(fx) < tol || fabs(fxdfx) < tol)
            break; /* success */

        *c -= fxdfx; /* Newton-Raphson correction */

        if (*c <= 0.0)
            *c = 0.5 * (fxdfx + *c);

        /* plug in result to get b */
        *b = *c;
        printf("%3d: % 10.6e % 10.6e % 10.6e % 10.6e % 10.6e\n",
               i, *b, *c, fx, dfx, fx/dfx);
    }
/*     printf("\niter = %d", i); */
    /* invgamma_logL(*b, *c); */

    if (i == maxit)
        printf("\n\n ERROR543: Newton-Raphson ML fit in invgamma_eq_bc_fit() did not converge.\n");

    return(chi_sqr_adapt(data, num, 0, logL, *b, *c, invgamma_pdf, invgamma_lnpdf, invgamma_int));
}
