/* spawg -  Simple Previewer in the World (Antialiased) with GTK
 *
 *  Copyright (C) 1999-2001  Hirotsugu Kakugawa. All rights reserved. 
 *  See "COPYING" for distribution of this software. 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
  
/*
 * History
 *
 *  5 Sep 1999  First implementation. Tested with GTK+ 1.2.3.
 * 21 Sep 1999  Added controlling text commands from standard input.
 * 14 May 2001  Enhanced user commands.
 *
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#  include <sys/types.h>
#endif
#if HAVE_STRING_H
#  include <string.h>
#endif
#if HAVE_STRINGS_H
#  include <strings.h>
#endif
#include <ctype.h>

#include <gtk/gtk.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "libdvi29.h"
#include "defs.h"

#define AA      4        /* must be integer value and be greater than 1 */
#define N_AA   (AA*AA+1)

#define CTL(ch)  ((ch)-0x40)

int              param_dpi           = -1;
char            *param_kpathsea_mode = NULL;
char            *param_vflibcap      = VFLIBCAP;
double           param_shrink        = (double)SHRINK;
char            *param_paper         = "A4"; 
int              param_orient        = 0;
char            *param_foregr        = COLOR_FG;
char            *param_backgr        = COLOR_BG;
unsigned int     param_win_size_w    = WIN_WIDTH;
unsigned int     param_win_size_h    = WIN_HEIGHT;
int              param_scrollbar_policy = GTK_POLICY_AUTOMATIC;
int              param_key_ctl          = 1;
int              param_stdin_ctl        = 0;
int              param_initial_page     = 1;

#define WINDOW_DELTA   50
int              param_win_delta     = WINDOW_DELTA;


char            *dvi_file  = NULL; 
DVI              dvi       = NULL;
DVI_DEVICE       dev       = NULL;
DVI_FRAME_BUFFER framebuff = NULL;
DVI_PROPERTY     dvi_props = NULL;
int              page, ui_cmd, page_state, status;
int              paper_w, paper_h;
int              ofs_x, ofs_y;
int              paper_aa_w, paper_aa_h;
int              x_pos, y_pos;
int              with_position   = 0;
GtkWidget       *window          = NULL;
GtkWidget       *win_scrolled    = NULL;
GtkWidget       *win_drawing     = NULL;
GdkPixmap       *back_pixmap_aa  = NULL;
GdkGC          **color_gcs       = NULL;
GdkColor         color_fg, color_bg;
guint            ctl_chan;

char    **paper_list = NULL;
char    **mode_list  = NULL;

#define EXIT_MSG1(a1)   {fprintf(stderr,a1); fprintf(stderr,"\n"); exit(1);}
#define EXIT_MSG2(a1,a2){fprintf(stderr,a1,a2); fprintf(stderr,"\n"); exit(1);}

#define PAGE_EMPTY        0
#define PAGE_DRAWING      1
#define PAGE_DRAW_FINISH  2
#define STAT_RUNNING      0
#define STAT_MAINLOOP     1
#define CMD_NONE          0
#define CMD_DRAW_PAGE     1

#define RELOAD_WAIT     1000  /* mSec */
#define RELOAD_RETRY       5 

char  *parse_args(int argc, char **argv);
void  usage(void);
int   window_size(char *paper, int ori);
int   open_dvi(char *file, int start_page,char**);
void  draw_page(void);
void  reload_dvi(void);
void  clear_page(void);
void  resize_window(void);
void  resize_window_delta(int dx, int dy);
DVI_DEVICE   create_dev(void);
void  DEV_put_bitmap(DVI_DEVICE,DVI,DVI_BITMAP,int,long,long,long,long);
void  DEV_put_rectangle(DVI_DEVICE,DVI,long,long,long,long);
int   DEV_poll(DVI_DEVICE dev, DVI dvi, int poll_type);
void  aa_put(long pos_x, long pos_y, long w, long h);
void  gui_init(void);
gint  event_key_press(GtkWidget *w, GdkEventKey *ev);
gint  event_configure_win(GtkWidget *w, GdkEventConfigure *ev);
gint  event_configure(GtkWidget *w, GdkEventConfigure *ev);
gint  event_expose(GtkWidget *w, GdkEventExpose *ev);
gint  event_button_press(GtkWidget *w, GdkEventButton *ev);
gint  event_button_release(GtkWidget *w, GdkEventButton *ev);
gint  event_motion_notify(GtkWidget *w, GdkEventMotion *ev);
void  preview_window_scroll_delta(int dx, int dy);
void  event_command_input(gpointer data, gint fd, GdkInputCondition cond);
void        run_stdin_cmmand(int argc, char **argv);
static int  tokenize_string(char **argv, int nargv, char *cmdline);
static int  readline_buffering(int fd, char *s, int n, int rf);
void   show_device_list(void);
void   show_paper_list(void);


int
main(int argc, char **argv)
{
  char  params[1024], *file, *errmsg;

  DVI_SETUP();
  paper_list = DVI_get_paper_size_list();
  mode_list  = DVI_get_device_mode_list();
  gtk_init(&argc, &argv);
  file = parse_args(argc, argv);
  if ((param_stdin_ctl == 0) && (file == NULL))
    EXIT_MSG1("No DVI file.\n");
  sprintf(params, "TeX_DPI=%d, TeX_KPATHSEA_MODE=%s, TeX_KPATHSEA_PROGRAM=%s",
	  param_dpi, param_kpathsea_mode, "xgdvi");
  if ((DVI_INIT(param_vflibcap, params) < 0) || (create_dev() == NULL))
    EXIT_MSG1("Failed to initialize DVI reader.\n");
  gui_init();
  open_dvi(file, param_initial_page, &errmsg);
  if (errmsg != NULL)
    EXIT_MSG2("%s\n", errmsg);
  if ((param_stdin_ctl == 0) && (dvi == NULL))
    exit(0);
  if (param_stdin_ctl == 1){
    ctl_chan = gtk_input_add_full(0, GDK_INPUT_READ, 
				  event_command_input, NULL, NULL, NULL);
  }
  status = STAT_MAINLOOP;
  gtk_main();

  return 0;
}

char*
parse_args(int argc, char **argv)
{
  char   *file;
  int     m;
  double  d, mag = 1.0;

  file = NULL;
  for (--argc, argv++; argc > 0;  argc--, argv++){
    if ((strcmp(*argv, "-m") == 0) && (argc > 1)){
      argc--; argv++;
      mag = atof(*argv);
    } else if ((strcmp(*argv, "-g") == 0)  && (argc > 2)){
      argc--;  argv++; 
      if ((param_win_size_w = atoi(*argv)) < 10)
	param_win_size_w = WIN_WIDTH;
      argc--;  argv++; 
      if ((param_win_size_h = atoi(*argv)) < 10)
	param_win_size_h = WIN_HEIGHT;
    } else if (strcmp(*argv, "-l") == 0){
      param_orient = 1;
    } else if (strcmp(*argv, "-p") == 0){
      argc--;  argv++;
      param_initial_page = atoi(*argv);
    } else if ((strcmp(*argv, "-v") == 0)  && (argc > 1)){
      argc--;  argv++;
      param_vflibcap = *argv;
    } else if (DVI_parse_device_mode(&argv[0][1], NULL, NULL, NULL) >= 0){
      param_kpathsea_mode = &argv[0][1];
    } else if ((strcmp(*argv, "-mode") == 0) && (argc > 1)){
      argc--;  argv++;
      param_kpathsea_mode = *argv;
    } else if ((strcmp(*argv, "-dpi") == 0) && (argc > 1)){
      argc--;  argv++;
      param_dpi = atoi(*argv);
    } else if (DVI_parse_paper_size(&argv[0][1], NULL, NULL) >= 0){
      param_paper = &argv[0][1];
    } else if ((strcmp(*argv, "-fg") == 0) && (argc > 1)){
      argc--;  argv++;
      param_foregr = *argv;
    } else if ((strcmp(*argv, "-bg") == 0) && (argc > 1)){
      argc--;  argv++;
      param_backgr = *argv;
    } else if (strcmp(*argv, "-scrollbar-auto") == 0){
      param_scrollbar_policy = GTK_POLICY_AUTOMATIC;
    } else if (strcmp(*argv, "-scrollbar-always") == 0){
      param_scrollbar_policy = GTK_POLICY_ALWAYS;
    } else if (strcmp(*argv, "-stdin-ctl") == 0){
      param_stdin_ctl = 1;
    } else if (strcmp(*argv, "-no-key-ctl") == 0){
      param_stdin_ctl = 1;
      param_key_ctl = 0;
    } else if (strcmp(argv[0], "-geometry") == 0){
      m = XParseGeometry(argv[1], &x_pos, &y_pos, 
			 &param_win_size_w, &param_win_size_h);
      if (((m & XValue) != 0) || ((m & YValue) != 0))
	with_position = 1;
      if ((m & XNegative) != 0)
	x_pos = -x_pos;
      if ((m & YNegative) != 0)
	y_pos = -y_pos;
      argc--;  argv++;
    } else if (**argv != '-'){
      file = *argv;
    } else if ((strcmp(*argv, "-h") == 0)
	       || (strcmp(*argv, "-help") == 0)
	       || (strcmp(*argv, "--help") == 0)){
      usage();
    } else if (strcmp(*argv, "-paper-list") == 0){
      show_paper_list();
      exit(0);
    } else if (strcmp(*argv, "-mode-list") == 0){
      show_device_list();
      exit(0);
    } else 
      EXIT_MSG2("Unknow option: %s\n", *argv);
  }
  if (mag < MAG_MIN)
    EXIT_MSG1("Maginification is too small.");
  if (MAG_MAX < mag)
    EXIT_MSG1("Maginification is too large.");
  if (param_kpathsea_mode == NULL){
    param_kpathsea_mode = KPATHSEA_MODE;
    if (param_dpi <= 0)
      param_dpi = DPI;
  } else {
    if (DVI_parse_device_mode(param_kpathsea_mode, &d, NULL, NULL) >= 0)
      param_dpi = d;
    else if (param_dpi <= 0)
      param_dpi = DPI;
  }
  param_shrink = ((double)SHRINK * param_dpi) / (mag * DPI);
  if (window_size(param_paper, param_orient) < 0)
    EXIT_MSG2("Unknown paper size: %s\n", param_paper);
  return  file;
}

void
usage(void)
{
  printf("SPAWG %s - A Previewer for TeX DVI files using GTK+.\n",
	 SPAWG_VERSION);
  printf("Usage: spawg [PaperSize] [Options] DVI-FILE\n");
  printf("PaperSize:\n");
  printf("  -a1,..,-a7,-b1,..,-b7,-letter,-us,-legal\n");
  printf("  -l      Landscape mode\n");
  printf("Options:  (Each value enclosed by [ ] is the default.)\n");
  printf("  -m M         Magnification (%.2f<=M<=%.2f) [1.0]\n", 
	 (double)MAG_MIN, (double)MAG_MAX);
  printf("  -p PAGE      Inital page number to display\n");
  printf("  -g W H       Window size [%d %d]\n", WIN_WIDTH, WIN_HEIGHT);
  printf("  -v VFLIBCAP  A path name of vflibcap [%s]\n", VFLIBCAP);
  printf("  -cx, -sparcptr, -ljfour, .... -MODE\n");
  printf("  -mode MODE   Device mode name for kpathsea [%s]\n", KPATHSEA_MODE);
  printf("  -dpi DPI     Device resolution for font selection [%d]\n", DPI);
  printf("  -fg COLOR    Foreground color [%s]\n", COLOR_FG);
  printf("  -bg COLOR    Background color [%s]\n", COLOR_BG);
  printf("  -scrollbar-auto    Show and hide scrollbar automatically\n");
  printf("  -scrollbar-always  Show scrollbar always\n");
  printf("  -stdin-ctl         Control spawg by text commands from stdin\n");
  printf("  -no-key-ctl        Disable control spawg by key\n");
  printf("  -paper-list  Show list of paper sizes\n");
  printf("  -mode-list   Show device mode names\n");
  printf("Key operations:");
  printf("\n\
  C-f, C-b   Move view port forward, backward, resp.\n\
  C-p, C-n   Move view port up, down, resp.\n\
  C-a, C-e   Go to beginning, end of line, resp.\n\
  C-v   Next page                     v     Previous page \n\
  <     Go to the first page          >     Go to the last page \n\
  ,     Go to next 10 page            .     Go to previous 10 page \n\
  ]     Magnify document              [     Shrink document\n\
  }     Enlarge window                {     Shrink window \n\
  0     Enlarge window vertically     9     Shrink window vertically \n\
  q     Quit                          o     Swap paper orientation\n");
  exit(0);
}

void
show_paper_list(void)
{
  char  **p;
  int   i;
  
  if ((p = paper_list) == NULL){
    printf("Paper size list is not found.\n");
  } else {
    i = 0;
    printf("Paper Size: \n");
    while (*p != NULL){
      printf("%s%s", *p, (*(p+1) == NULL) ? "\n" : ", ");
      i++;
      p++;
    }
  }
}

void
show_device_list(void)
{
  char **p, *desc;
  int  i;
  
  if ((p = mode_list) == NULL){
    printf("Device mode list is not found.\n");
  } else {
    i = 0;
    printf("Device Mode Names: \n");
    printf("   Mode Name     Description\n");
    while (*p != NULL){
      DVI_parse_device_mode(*p, NULL, NULL, &desc);
      printf("%12s     %s\n", *p, desc);
      i++;
      p++;
    }
  }
}

int
window_size(char *paper, int ori)
{
  double  pw, ph;

  if (DVI_parse_paper_size(paper, &pw, &ph) < 0)
    return -1;

  paper_w = ((ori == 0)?pw:ph) * AA * param_dpi / param_shrink;
  paper_h = ((ori == 0)?ph:pw) * AA * param_dpi / param_shrink;
  paper_aa_w = (paper_w + AA - 1) / AA;
  paper_aa_h = (paper_h + AA - 1) / AA;
  ofs_x = OFFSET_X * AA * param_dpi / param_shrink;
  ofs_y = OFFSET_Y * AA * param_dpi / param_shrink;
  return 0;
}

int
open_dvi(char *file, int start_page, char **errmsg)
{
  char  *junk;

  if (errmsg == NULL)
    errmsg = &junk;
  *errmsg = NULL;

  ui_cmd     = CMD_NONE;
  page_state = PAGE_DRAW_FINISH;

  if (dvi != NULL){
    DVI_DISPOSE(dvi, dev);
    dvi = NULL;
  }

  if (file == NULL){
    if (param_stdin_ctl == 1){
      return 0;
    } else if (dvi_file == NULL){
      *errmsg = "No DVI File.";
      return -1;
    }
  } else {
    if (dvi_file != NULL){
      free(dvi_file);
      dvi_file = NULL;
    }
    dvi_file = DVI_find_dvi_path(file);
  }
  if (dvi_file == NULL){
    *errmsg = "Cannot read.";
    return -1;
  }

  dvi_props = DVI_PROPERTY_ALLOC_DEFAULT();
  DVI_PROPERTY_SET(dvi_props, DVI_PROP_DELAYED_FONT_OPEN);
#if 1
  DVI_PROPERTY_SET(dvi_props, DVI_PROP_ASYNC_GS_INVOCATION);
#else
  DVI_PROPERTY_SET(dvi_props, DVI_PROP_SKIP_SPECIALS);
#endif
  DVI_PROPERTY_SET(dvi_props, DVI_PROP_INCREMENTAL_EPS_DISPLAY);
  if ((dvi = DVI_CREATE(dev, dvi_file, dvi_props)) == NULL){
    *errmsg = "Not a DVI file.";
    return -1;
  } 
  if (DVI_OPEN_FONT(dvi, dev) < 0){
    DVI_DISPOSE(dvi, dev);
    dvi = NULL;
    *errmsg = "Cannot open all fonts.";
    return -1;
  }

  if (file != NULL){
    if (start_page > dvi->pages)
      start_page = dvi->pages;
    if (start_page < 1)
      start_page = 1;
    page = start_page;
  } else {
    if (page > dvi->pages)
      page = dvi->pages;
  }
  ui_cmd     = CMD_DRAW_PAGE;
  page_state = PAGE_EMPTY;
  *errmsg = NULL;

  return 0;
}

int
goto_page(int p)
{ 
  int  v = 0;

  clear_page();
  if (dvi == NULL){
    ui_cmd = CMD_NONE;
    return  v;
  }

  if ((1 <= p) && (p <= dvi->pages)){
    if (p != page){
      page = p;
      ui_cmd = CMD_DRAW_PAGE;
      v = 1;
    }
  }
  return  v;
}

void
draw_page(void)
{
  if ((status != STAT_MAINLOOP) || (dvi == NULL))
    return;
  status = STAT_RUNNING;
  while (ui_cmd == CMD_DRAW_PAGE){
    if (DVI_FILE_MODIFIED(dvi, dvi_file) == 1){
      reload_dvi();
    }
    clear_page();
    ui_cmd = CMD_NONE;
    page_state = PAGE_DRAWING;
    if (DVI_DRAW_PAGE(dvi, dev, page, param_shrink/AA) 
	!= DVI_DRAW_INTERRUPTED){
      page_state = PAGE_DRAW_FINISH;
      ui_cmd = CMD_NONE;
    }
  }
  status = STAT_MAINLOOP;
}

void
reload_dvi(void)
{
  int   r;

  do {
    DVI_DISPOSE(dvi, dev);
    dvi = NULL;
    r = 0;
    while ((dvi = DVI_CREATE(dev, dvi_file, dvi_props)) == NULL){
      usleep(RELOAD_WAIT*1000);
      r++;
      if (r > RELOAD_RETRY)
	return;
    }
    usleep(100*1000);
    if ((DVI_FILE_MODIFIED(dvi, dvi_file) == 0)
	&& (DVI_OPEN_FONT(dvi, dev) < 0))
      fprintf(stderr, "Cannot find all fonts.\n");
  } while (DVI_FILE_MODIFIED(dvi, dvi_file) == 1);

  if (dvi->pages < page)
    page = dvi->pages;
}

void
clear_page(void)
{
  gdk_draw_rectangle(back_pixmap_aa, color_gcs[0], TRUE, 0, 0,
		     paper_aa_w, paper_aa_h);
  gdk_draw_pixmap(win_drawing->window, 
		  win_drawing->style->fg_gc[GTK_WIDGET_STATE(win_drawing)], 
		  back_pixmap_aa, 0, 0, 0, 0, paper_aa_w, paper_aa_h);
  DVI_fb_clear(framebuff);
  gtk_widget_draw(window, NULL);
  gdk_flush();
  page_state = PAGE_EMPTY;
}

void
resize_window(void)
{
  if (window_size(param_paper, param_orient) < 0)
    return;
  gtk_drawing_area_size(GTK_DRAWING_AREA(win_drawing), paper_aa_w, paper_aa_h);
  gtk_widget_queue_resize(win_drawing);
  if (back_pixmap_aa != NULL)
    gdk_pixmap_unref(back_pixmap_aa);
  back_pixmap_aa = gdk_pixmap_new(win_drawing->window, 
				  paper_aa_w, paper_aa_h, -1);
  if (framebuff != NULL){
    DVI_fb_dispose(framebuff);
    framebuff = NULL;
  }
  if ((framebuff = DVI_fb_create((long)paper_aa_w * AA, 
				 (long)paper_aa_h * AA)) == NULL){
    fprintf(stderr, "No memory for frame buffer.\n");
    exit(1);
  }
  DVI_fb_clear(framebuff);
}



DVI_DEVICE
create_dev(void)
{
  if ((dev = DVI_DEVICE_ALLOC()) == NULL)
    return NULL;
  dev->h_dpi = dev->v_dpi   = param_dpi;
  dev->device_polling       = DEV_poll;
  dev->put_bitmap           = DEV_put_bitmap;
  dev->put_rectangle        = DEV_put_rectangle;
  dev->message_advice       = NULL;
  return dev;
}

void
DEV_put_bitmap(DVI_DEVICE dev, DVI dvi, DVI_BITMAP bm, int font_id, 
	       long key2, long code_point, long pos_x, long pos_y)
{
  DVI_fb_put_bitmap(framebuff, bm, pos_x + ofs_x, pos_y + ofs_y);
  aa_put(pos_x + ofs_x, pos_y + ofs_y, bm->width, bm->height);
}

void
DEV_put_rectangle(DVI_DEVICE dev, DVI dvi, 
		  long pos_x, long pos_y, long w, long h)
{
  DVI_fb_put_rectangle(framebuff, pos_x + ofs_x, pos_y + ofs_y, w, h);
  aa_put(pos_x + ofs_x, pos_y + ofs_y, w, h);
}

void
aa_put(long pos_x, long pos_y, long w, long h)
{
  long            aabw, aabh, x, y, pos_x_aa, pos_y_aa;
  unsigned short *aa_buff;

  if (pos_x < 0){
    w += pos_x;
    pos_x = 0;
  }
  if (pos_y < 0){
    h += pos_y;
    pos_y = 0;
  }
  if (pos_x + w >= framebuff->width)
    w = framebuff->width - pos_x - 1;
  if (pos_y + h >= framebuff->height)
    h = framebuff->height - pos_y -1;
  if ((w <= 0) || (h <= 0))
    return;

  if ((aa_buff = DVI_fb_antialias(framebuff, AA, 
				  pos_x, pos_y, w, h, &aabw, &aabh)) == NULL)
    return;
  pos_x_aa = pos_x / AA;
  pos_y_aa = pos_y / AA;
  for (y = 0; y < aabh; y++){
    for (x = 0; x < aabw; x++){
      gdk_draw_point(back_pixmap_aa, color_gcs[aa_buff[aabw * y + x]], 
		     pos_x_aa + x, pos_y_aa + y);
    }
  }
  gdk_draw_pixmap(win_drawing->window, color_gcs[N_AA-1], back_pixmap_aa, 
		  pos_x_aa, pos_y_aa, pos_x_aa, pos_y_aa, aabw, aabh);
  free(aa_buff);
  aa_buff = NULL;
}

int
DEV_poll(DVI_DEVICE dev, DVI dvi, int poll_type)
{
  static int  t = 0;

  if (poll_type != DVI_POLL_PAGE)
    return 0;

  if ((--t) > 0)
    return 0;
  t = POLLING_INTERVAL;
  while (gtk_events_pending()){
    gtk_main_iteration();
  }
  if (ui_cmd != CMD_NONE)
    return 1;
  return 0;
}



void
gui_init(void)
{
  int  hints;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), "spawg");
  gtk_widget_set_events(window, GDK_EXPOSURE_MASK);
  gtk_signal_connect(GTK_OBJECT(window), "configure_event",
		     (GtkSignalFunc)event_configure_win, NULL);
  gtk_signal_connect(GTK_OBJECT(window), "key_press_event", 
		     GTK_SIGNAL_FUNC(event_key_press), NULL);
  gtk_signal_connect(GTK_OBJECT(window), "destroy", 
		     GTK_SIGNAL_FUNC(gtk_main_quit), NULL);
  gtk_container_set_border_width(GTK_CONTAINER(window), 2);
  gtk_window_set_default_size(GTK_WINDOW(window), 
			      param_win_size_w, param_win_size_h);
  gtk_window_set_policy(GTK_WINDOW(window), 1, 1, 1);
  win_drawing = gtk_drawing_area_new();
  gtk_drawing_area_size(GTK_DRAWING_AREA(win_drawing), paper_aa_w, paper_aa_h);
  win_scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(win_scrolled),
				 param_scrollbar_policy, 
				 param_scrollbar_policy);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(win_scrolled),
					win_drawing);
  gtk_widget_set_events(win_drawing, 
			(GDK_EXPOSURE_MASK
			 | GDK_POINTER_MOTION_MASK 
			 | GDK_POINTER_MOTION_HINT_MASK 
			 | GDK_BUTTON_PRESS_MASK 
			 | GDK_BUTTON_RELEASE_MASK));
  gtk_signal_connect(GTK_OBJECT(win_drawing), "configure_event",
		     (GtkSignalFunc)event_configure, NULL);
  gtk_signal_connect(GTK_OBJECT(win_drawing), "expose_event",
		     GTK_SIGNAL_FUNC(event_expose), NULL);
  gtk_signal_connect(GTK_OBJECT(win_drawing), "button_press_event",
		     GTK_SIGNAL_FUNC(event_button_press), NULL);
  gtk_signal_connect(GTK_OBJECT(win_drawing), "button_release_event",
		     GTK_SIGNAL_FUNC(event_button_release), NULL);
  gtk_signal_connect(GTK_OBJECT(win_drawing), "motion_notify_event",
		     GTK_SIGNAL_FUNC(event_motion_notify), NULL);
  gtk_container_add(GTK_CONTAINER(window), win_scrolled);

  gtk_widget_show(win_drawing);
  gtk_widget_show(win_scrolled);
  gtk_widget_show(window);

  hints = 0;
  if (with_position == 1)
    hints = (hints | GDK_HINT_POS);
#if 0
      printf("** <%d,%d> %d\n", x_pos, y_pos, with_position);
#endif
  gdk_window_set_hints(GTK_WIDGET(window)->window, 
		       x_pos, y_pos, 
		       param_win_size_w, param_win_size_h,
		       param_win_size_w, param_win_size_h,
		       hints);
}

gint
event_key_press(GtkWidget *w, GdkEventKey *ev)
{
  double  s;
  int     opage, k;

  if (param_key_ctl == 0)
    return TRUE;

  if (ev-> length != 1)
    return TRUE;

  k = (int)(ev->string[0]);
  switch (k){
  default:
    break;
  case CTL('Q'): case 'q': 
    gtk_main_quit();
    break;
  case ' ': case CTL('V'): case 'b': case 'v': case 0x7f:
  case '<': case '>':  case ',':  case '.': 
    opage = page;
    switch (k){
    case ' ': case CTL('V'): 
      page = (page < dvi->pages) ? (page + 1) : page; break;
    case 'b': case 'v': case 0x7f:
      page = (1 < page) ? (page - 1) : page; break;
    case '<':  page = 1; break;
    case '>':  page = dvi->pages; break;
    case ',':  page = (1 <= page-10)          ? (page-10): 1;          break;
    case '.':  page = (page+10 <= dvi->pages) ? (page+10): dvi->pages; break;
    }
    if (page != opage)
      ui_cmd = CMD_DRAW_PAGE;
    break;
  case CTL('R'): case 'r': 
    reload_dvi();
    clear_page();
    ui_cmd = CMD_DRAW_PAGE;
    break;
  case CTL('P'): case 'k': 
    preview_window_scroll_delta(0, (int)(-0.03*paper_aa_h));
    break;
  case CTL('N'): case 'j': 
    preview_window_scroll_delta(0, (int)(+0.03*paper_aa_h));
    break;
  case CTL('B'): 
    preview_window_scroll_delta((int)(-0.03*paper_aa_h), 0);
    break;
  case CTL('F'): 
    preview_window_scroll_delta((int)(+0.03*paper_aa_h), 0);
    break;
  case CTL('A'): 
    preview_window_scroll_delta((int)(-1*paper_aa_w), 0);
    break;
  case CTL('E'): 
    preview_window_scroll_delta((int)(1*paper_aa_w), 0);
    break;
  case '^': 
    preview_window_scroll_delta(0, (int)(-1*paper_aa_h));
    break;
  case '_': 
    preview_window_scroll_delta(0, (int)(1*paper_aa_h));
    break;
  case '{': 
    resize_window_delta(-param_win_delta, -param_win_delta);
    break;
  case '}':  
    resize_window_delta(param_win_delta, param_win_delta);
    break;
  case '9': 
    resize_window_delta(0, -param_win_delta);
    break;
  case '0': 
    resize_window_delta(0, param_win_delta);
    break;
  case '(': 
    resize_window_delta(-param_win_delta, 0);
    break;
  case ')': 
    resize_window_delta(param_win_delta, 0);
    break;
  case '[': case ']': case 'o':
    s = param_shrink;
    switch (ev->keyval){
    case '[':  s = s*10/9; break;
    case ']':  s = s*9/10; break;
    case 'o':  param_orient = (1 - param_orient); break;
    }
    if ((1 <= s) && (s <= 128)){
      ui_cmd = CMD_DRAW_PAGE;
      param_shrink = s;
      resize_window();
    }
    break;
  }

  if (ui_cmd == CMD_DRAW_PAGE){
    draw_page();
  }
  return TRUE;
}

void
resize_window_delta(int dx, int dy)
{
  param_win_size_w += dx;
  if (param_win_size_w < 100)
    param_win_size_w = 100;
  param_win_size_h += dy;
  if (param_win_size_h < 100)
    param_win_size_h = 100;
  gtk_window_set_default_size(GTK_WINDOW(window),
			      param_win_size_w, param_win_size_h);
  gtk_widget_queue_resize(window);
}

gint
event_configure_win(GtkWidget *w, GdkEventConfigure *ev)
{
#if 0
  printf("** configure event: window <%d,%d> %dx%d\n",
	 ev->x, ev->y, ev->width, ev->height);
#endif
  param_win_size_w = ev->width;
  param_win_size_h = ev->height;
  return  FALSE;
}

gint
event_configure(GtkWidget *w, GdkEventConfigure *ev)
{
  GdkColormap   *cmap;
  GdkColor       col;
  double         r, g, b;
  int            i;

#if 0
  printf("** configure event: drawing <%d,%d> %dx%d\n",
	 ev->x, ev->y, ev->width, ev->height);
#endif

  if (color_gcs == NULL){
    color_gcs = g_new(GdkGC*, N_AA);
    if (color_gcs == NULL){
      fprintf(stderr, "No memory\n");
      exit(1);
    }
    cmap = gdk_window_get_colormap(win_drawing->window);
    if (    (gdk_color_parse(param_foregr, &color_fg) == FALSE)
	 || (gdk_color_parse(param_backgr, &color_bg) == FALSE) ){
      color_fg.red = color_fg.green = color_fg.blue = (guint16)0;
      color_bg.red = color_bg.green = color_bg.blue = (guint16)65535;
    } else {
      gdk_color_alloc(cmap, &color_fg);
      gdk_color_alloc(cmap, &color_bg);
    }
    r = (double)(color_fg.red   - color_bg.red)   / (double)(N_AA-1);
    g = (double)(color_fg.green - color_bg.green) / (double)(N_AA-1);
    b = (double)(color_fg.blue  - color_bg.blue)  / (double)(N_AA-1);
    for (i = 0; i < N_AA; i++){
      color_gcs[i] = gdk_gc_new(win_drawing->window);
      col.red   = (guint16) ((double)color_bg.red   + r * i);
      col.green = (guint16) ((double)color_bg.green + g * i);
      col.blue  = (guint16) ((double)color_bg.blue  + b * i);
      gdk_color_alloc(cmap, &col);
      gdk_gc_set_foreground(color_gcs[i], &col);
    }
  }

  if (back_pixmap_aa != NULL)
    gdk_pixmap_unref(back_pixmap_aa);
  back_pixmap_aa = gdk_pixmap_new(win_drawing->window, 
				  paper_aa_w, paper_aa_h, -1);
  gdk_draw_rectangle(back_pixmap_aa, color_gcs[0], TRUE, 0, 0, 
		     paper_aa_w, paper_aa_h);

  if (framebuff != NULL){
    DVI_fb_dispose(framebuff);
    framebuff = NULL;
  }
  if ((framebuff = DVI_fb_create((long)paper_aa_w * AA, 
				 (long)paper_aa_h * AA)) == NULL){
    fprintf(stderr, "No memory for frame buffer.\n");
    exit(1);
  }
  DVI_fb_clear(framebuff);

  page_state = PAGE_EMPTY;
  ui_cmd = CMD_DRAW_PAGE;
  return  FALSE;
}

gint
event_expose(GtkWidget *w, GdkEventExpose *ev)
{
  if (back_pixmap_aa == NULL)
    return FALSE;
  /*printf("expose: status=%d, page_state=%d\n", status, page_state);*/
  gdk_draw_pixmap(win_drawing->window, 
		  win_drawing->style->fg_gc[GTK_WIDGET_STATE(win_drawing)], 
		  back_pixmap_aa, ev->area.x, ev->area.y, 
		  ev->area.x, ev->area.y, ev->area.width, ev->area.height);
  if (status == STAT_MAINLOOP){
    draw_page();
  }
  return FALSE;
}


static int  button_pressed_x = 0;
static int  button_pressed_y = 0;

gint
event_button_press(GtkWidget *w, GdkEventButton *ev)
{
  if (ev->button != 1)
    return TRUE;    

  button_pressed_x = ev->x;
  button_pressed_y = ev->y;

  return TRUE;
}

gint
event_motion_notify(GtkWidget *w, GdkEventMotion *ev)
{
  int  dx, dy, x, y;
  GdkModifierType state;

  if (ev->is_hint != 0){
    gdk_window_get_pointer(ev->window, &x, &y, &state);
  } else {
    x = ev->x;
    y = ev->y;
    state = ev->state;
  }

  if ((state & GDK_BUTTON1_MASK) == 0)
    return TRUE;

  dx = button_pressed_x - x;
  dy = button_pressed_y - y;

  if ((dx*dx + dy*dy) < 10){   /* Movement is too small. */
    return TRUE;
  }

  preview_window_scroll_delta(dx, dy);

  gdk_window_get_pointer(ev->window, &x, &y, &state);   /* important */
  button_pressed_x = x;
  button_pressed_y = y;

  return TRUE;
}

gint
event_button_release(GtkWidget *w, GdkEventButton *ev)
{
  int  dx, dy;

  if (ev->button != 1)
    return TRUE;

  dx = button_pressed_x - ev->x;
  dy = button_pressed_y - ev->y;
  preview_window_scroll_delta(dx, dy);
  button_pressed_x = ev->x;
  button_pressed_y = ev->y;

  return TRUE;
}


void
preview_window_scroll_delta(int dx, int dy)
{
  int  new_x, new_y;
  GtkAdjustment *hadj, *vadj;

  hadj = gtk_scrolled_window_get_hadjustment((GtkScrolledWindow*)win_scrolled);
  vadj = gtk_scrolled_window_get_vadjustment((GtkScrolledWindow*)win_scrolled);

  new_x = hadj->value + dx;
  if (new_x < hadj->lower)
    new_x = hadj->lower;
  if (hadj->upper - hadj->page_size < new_x)
    new_x = hadj->upper - hadj->page_size;

  new_y = vadj->value + dy;
  if (new_y < vadj->lower)
    new_y = vadj->lower;
  if (vadj->upper - vadj->page_size < new_y)
    new_y = vadj->upper - vadj->page_size;

  hadj->value = new_x;
  vadj->value = new_y;
  gtk_adjustment_value_changed(hadj);
  gtk_adjustment_value_changed(vadj);
}



void
event_command_input(gpointer data, gint fd, GdkInputCondition cond)
{
  char  buff[BUFSIZ], *argv[32], *p;
  int  argc, j;

  j = readline_buffering(fd, buff, BUFSIZ, 1);
  while (j > 0){
    if ((argc = tokenize_string(argv, 32, buff)) == 0)
      break;
    for (p = argv[0]; *p != '\0'; p++)
      *p = tolower(*p);
    run_stdin_cmmand(argc, argv);
    j = readline_buffering(fd, buff, BUFSIZ, 0);
  }
  return;
}

void 
run_stdin_cmmand(int argc, char **argv)
{
  static char  paper[256];
  static int   start_page = 1;
  char  resp[256], *response, *errmsg; 
  int      need_resp; 
  double   s;

  errmsg = NULL;
  response = NULL;
  need_resp = 0;

  if ((strcmp(argv[0], "quit") == 0)
      || (strcmp(argv[0], "q") == 0)){
    gtk_main_quit();
    exit(0);

  } else if (strcmp(argv[0], "file") == 0){
    if (argc > 1){
      if (open_dvi(argv[1], start_page, &errmsg) >= 0)
	draw_page();
    } else {
      errmsg = "No file name";
    }
    start_page = 1;

  } else if (strcmp(argv[0], "reload") == 0){
    if (open_dvi(NULL, 0, &errmsg) >= 0)
      draw_page();

  } else if ((strcmp(argv[0], "page-next") == 0)
	     || (strcmp(argv[0], "next") == 0)){
    if ((dvi != NULL) && (page < dvi->pages))
      goto_page(page + 1); 
    else
      errmsg = "Can't go to the page";

  } else if ((strcmp(argv[0], "page-prev") == 0)
	     || (strcmp(argv[0], "prev") == 0)){
    if ((dvi != NULL) && (1 < page))
      goto_page(page - 1);
    else
      errmsg = "Can't go to the page";

  } else if ((strcmp(argv[0], "page-begin") == 0)
	     || (strcmp(argv[0], "<") == 0)){
    if (dvi != NULL)
      goto_page(1); 
    else
      errmsg = "Can't go to the page";

  } else if ((strcmp(argv[0], "page-end") == 0)
	     || (strcmp(argv[0], ">") == 0)){
    if (dvi != NULL)
      goto_page(dvi->pages); 
    else
      errmsg = "Can't go to the page";

  } else if ((strcmp(argv[0], "goto-page") == 0)
	     || (strcmp(argv[0], "p") == 0)){
    if ((argc > 1) && (dvi != NULL) && (atoi(argv[1]) > 0))
      goto_page(atoi(argv[1])); 
    else
      errmsg = "Can't go to the page";
    
  } else if (strcmp(argv[0], "start-page") == 0){
    if ((argc > 1) && (atoi(argv[1]) > 0))
	start_page = atoi(argv[1]);
    else
      start_page = 1;

  } else if (strcmp(argv[0], "total-pages") == 0){
    need_resp = 1;
    if (dvi != NULL)
      sprintf(resp, "%d", dvi->pages);
    else 
      sprintf(resp, "%d", 0);
    response = resp;

  } else if (strcmp(argv[0], "current-page") == 0){
    need_resp = 1;
    if (dvi != NULL)
      sprintf(resp, "%d", page);
    else 
      sprintf(resp, "%d", 0);
    response = resp;

  } else if ((strcmp(argv[0], "zoom-in") == 0)
	     || (strcmp(argv[0], "]") == 0)){
    s = param_shrink * 8/10;
    if ((1 <= s) && (s <= 128)){
      ui_cmd = CMD_DRAW_PAGE;
      param_shrink = s;
      resize_window();
    }

  } else if ((strcmp(argv[0], "zoom-out") == 0)
	     || (strcmp(argv[0], "[") == 0)){
    s = param_shrink * 10/8;
    if ((1 <= s) && (s <= 128)){
      ui_cmd = CMD_DRAW_PAGE;
      param_shrink = s;
      resize_window();
    }

  } else if (strcmp(argv[0], "paper-size") == 0){
    if ((argc > 1) && (window_size(argv[1], param_orient) >= 0)){
      strcpy(paper, argv[1]);
      param_paper = paper;
      resize_window();
    } else {
      errmsg = "Unknown paper size";
    }

  } else if (strcmp(argv[0], "orientation") == 0){
    if (argc > 1){
      param_orient = (strcmp(argv[1], "portrait") == 0) ? 0 : 1;
      resize_window();
    }

  } else if (strcmp(argv[0], "help") == 0){
    response = "Usage:\n"
      "    quit\n"
      "    start-page PAGE\n"
      "    file FILENAME\n"
      "    reload \n"
      "    total-pages\n"
      "    goto-page PAGE\n"
      "    page-next\n"
      "    page-prev\n"
      "    page-begin\n"
      "    page-end\n"
      "    current-page\n"
      "    zoom-in\n"
      "    zoom-out\n"
      "    paper-size PAPERNAME\n"
      "    orientation ORIENT  (portrait, landscape)\n"
      "    help\n";
  } else {
    errmsg = "Unknow command";
  }

  if (ui_cmd == CMD_DRAW_PAGE){
    draw_page();
  }

  if (need_resp == 1){
    if (errmsg != NULL){
      printf("ERROR\n%s\n.\n", errmsg);
    } else {
      if (response == NULL)
	printf("OK\n.\n");
      else
	printf("OK\n%s\n.\n", response);
    }
  }
}


static int
readline_buffering(int fd, char *s, int n, int rf)
{
  static int   in = 0;
  static int   out = 0;
  static int   room = BUFSIZ-1;
  static char  buff[BUFSIZ];
  char subbuff[BUFSIZ+1];
  int  line, rn, i;

  if (n == 0)
    return 0;

  if ((rf == 1) && (room > 0)){
    if ((rn = read(fd, subbuff, room)) <= 0){
      gtk_input_remove(ctl_chan);
    } else {
      for (i = 0; i < rn; i++){
	buff[in] = subbuff[i];
	in = (in + 1) % BUFSIZ;
	room--;
      }
    }
  }

#if 0
  printf("[");
  for (i = out; i != in; i = (i+1) % BUFSIZ){
    if (isprint(buff[i]))
      printf("%c", buff[i]);
    else
      printf("\\0x%02x", buff[i]);
  }
  printf("]\n");
#endif

  line = 0;
  for (i = out; i != in; i = (i+1) % BUFSIZ){
    if (buff[i] == '\n'){
      line = 1;
      break;
    }
  }

  if (line == 0){
    i = 0;
  } else {
    for (i = 0; out != in; i++){
      if (i == n-1)
	break;
      s[i] = buff[out];
      out = (out+1) % BUFSIZ ;
      room++;
      if (s[i] == '\n')
	break;
    }
  }
  s[i] = '\0';

  return i;
}

static int
tokenize_string(char **argv, int nargv, char *cmdline)
{
  int  argc, idx;

  idx = 0;
  for (argc = 0; argc < nargv-1; ){
    while (isspace(cmdline[idx]))
      idx++;
    if (cmdline[idx] == '\0')
      break;
    argv[argc] = &cmdline[idx];
    argc++;
    while (!isspace(cmdline[idx]) && (cmdline[idx] != '\0'))
      idx++;
    if (cmdline[idx] == '\0')
      break;
    cmdline[idx++] = '\0';
  }
  argv[argc] = NULL;

  return argc;
}
