/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 *
 * ti-dlg-add-members.c:
 * Dialog to add members to a Channel.Interface.Group
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "dlg-add-members.h"
#include "util.h"
#include "config.h"
#include "handle-list-editor.h"

#include <glade/glade.h>

G_DEFINE_TYPE (TIDlgAddMembers, ti_dlg_add_members, G_TYPE_OBJECT);

/**
 * Instance private data.
 */
struct _TIDlgAddMembersPrivate {
    gboolean disposed;

    GtkWindow *parent;

    GladeXML *glade_xml;

    TIHandleListEditor *handle_list_editor;

    GtkWidget *dialog;

    GtkTextBuffer *textbuffer_message;
};
typedef struct _TIDlgAddMembersPrivate TIDlgAddMembersPrivate;

#define TI_DLG_ADD_MEMBERS_GET_PRIVATE(object) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_DLG_ADD_MEMBERS, \
                                TIDlgAddMembersPrivate))

static gchar *_ti_dlg_add_members_get_message_text (TIDlgAddMembers *);
static void _ti_dlg_add_members_setup (TIDlgAddMembers *);

static void
ti_dlg_add_members_dispose (GObject *object)
{
  TIDlgAddMembers *self = TI_DLG_ADD_MEMBERS (object);
  TIDlgAddMembersPrivate *priv = TI_DLG_ADD_MEMBERS_GET_PRIVATE (self);

  if (priv->disposed)
    return;

  priv->disposed = TRUE;

  if (priv->glade_xml != NULL)
    {
      g_object_unref (priv->glade_xml);
      priv->glade_xml = NULL;
    }

  G_OBJECT_CLASS (ti_dlg_add_members_parent_class)->dispose (object);
}

static void
ti_dlg_add_members_class_init (TIDlgAddMembersClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->dispose = ti_dlg_add_members_dispose;
  g_type_class_add_private (klass, sizeof (TIDlgAddMembersPrivate));
}

static void
ti_dlg_add_members_init (TIDlgAddMembers *self)
{
  TIDlgAddMembersPrivate *priv = TI_DLG_ADD_MEMBERS_GET_PRIVATE (self);

  priv->disposed = FALSE;
}

TIDlgAddMembers *
ti_dlg_add_members_new (GtkWindow *parent,
                        TIHandleMapper *handle_mapper)
{
  TIDlgAddMembers *self = NULL;
  TIDlgAddMembersPrivate *priv = NULL;
  gchar *glade_file_path = NULL;

  self = g_object_new (TI_TYPE_DLG_ADD_MEMBERS, NULL);

  priv = TI_DLG_ADD_MEMBERS_GET_PRIVATE (self);
  priv->parent = parent;
  priv->handle_list_editor = TI_HANDLE_LIST_EDITOR (ti_handle_list_editor_new
      (handle_mapper));

  glade_file_path = ti_get_glade_path ("dlg-add-members.xml");
  priv->glade_xml = glade_xml_new (glade_file_path, NULL, NULL);
  if (priv->glade_xml == NULL)
    {
      g_critical ("Error loading glade file \"%s\".", glade_file_path);
      g_object_unref (self);
      self = NULL;
      goto CLEAN_UP;
    }

  _ti_dlg_add_members_setup (self);

  gtk_window_set_transient_for (GTK_WINDOW (priv->dialog),
      GTK_WINDOW (priv->parent));
  gtk_window_set_position (GTK_WINDOW (priv->dialog),
      GTK_WIN_POS_CENTER_ON_PARENT);

  glade_xml_signal_connect_data (priv->glade_xml, "button_ok_clicked",
      G_CALLBACK (ti_ok_button_clicked), priv->dialog);
  glade_xml_signal_connect_data (priv->glade_xml, "button_cancel_clicked",
      G_CALLBACK (ti_cancel_button_clicked), priv->dialog);

CLEAN_UP:
  g_free (glade_file_path);

  return self;
}

gboolean
ti_dlg_add_members_run (TIDlgAddMembers *self,
                        GArray *in_handles,
                        GArray **out_handles,
                        gchar **message)
{
  TIDlgAddMembersPrivate *priv = TI_DLG_ADD_MEMBERS_GET_PRIVATE (self);
  gint result;

  g_return_val_if_fail (out_handles != NULL, FALSE);
  g_return_val_if_fail (message != NULL, FALSE);

  ti_handle_list_editor_set (priv->handle_list_editor, in_handles);

  gtk_widget_show_all (priv->dialog);
  result = gtk_dialog_run (GTK_DIALOG (priv->dialog));
  gtk_widget_hide_all (priv->dialog);

  *message = _ti_dlg_add_members_get_message_text (self);

  *out_handles = ti_handle_list_editor_get (priv->handle_list_editor);

  if (*out_handles == NULL)
    {
      return FALSE;
    }
  else
    {
      return (result == GTK_RESPONSE_OK);
    }
}

static gchar *
_ti_dlg_add_members_get_message_text (TIDlgAddMembers *self)
{
  TIDlgAddMembersPrivate *priv = TI_DLG_ADD_MEMBERS_GET_PRIVATE (self);
  GtkTextIter start_iter;
  GtkTextIter end_iter;

  gtk_text_buffer_get_start_iter (priv->textbuffer_message, &start_iter);
  gtk_text_buffer_get_end_iter (priv->textbuffer_message, &end_iter);

  return gtk_text_buffer_get_text (priv->textbuffer_message, &start_iter,
      &end_iter, FALSE);
}

static void
_ti_dlg_add_members_setup (TIDlgAddMembers *self)
{
  TIDlgAddMembersPrivate *priv = TI_DLG_ADD_MEMBERS_GET_PRIVATE (self);
  GtkWidget *widget;

  priv->dialog = glade_xml_get_widget (priv->glade_xml,
      "dialog_add_members");
  g_assert (GTK_IS_DIALOG (priv->dialog));

  widget = glade_xml_get_widget (priv->glade_xml,
      "container_handle_list_editor");
  g_assert (widget != NULL && GTK_IS_CONTAINER (widget));
  gtk_container_add (GTK_CONTAINER (widget),
      GTK_WIDGET (priv->handle_list_editor));

  widget = glade_xml_get_widget (priv->glade_xml, "textview_message");
  g_assert (widget != NULL && GTK_IS_TEXT_VIEW (widget));
  priv->textbuffer_message = gtk_text_view_get_buffer (GTK_TEXT_VIEW (widget));
}
