/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-page.c:
 * Abstract class for all TIPage* classes.
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-page.h"
#include "ti-page-priv.h"
#include "ti-config.h"

#include <glade/glade.h>

G_DEFINE_TYPE (TIPage, ti_page, G_TYPE_OBJECT);


/**
 * Drop all references to other objects.
 */
static void
ti_page_dispose (GObject *object)
{
    TIPage *page = TI_PAGE (object);
    TIPagePrivate *priv = TI_PAGE_GET_PRIVATE (page);

    if (priv->disposed)
        return;
    else
        priv->disposed = TRUE;

    if (priv->parent_notebook == NULL)
    {
        g_object_unref (priv->parent_notebook);
        priv->parent_notebook = NULL;
    }

    if (priv->page != NULL)
    {
        g_object_unref (priv->page);
        priv->page = NULL;
    }

    if (priv->page_label != NULL)
    {
        g_object_unref (priv->page_label);
        priv->page_label = NULL;
    }

    G_OBJECT_CLASS (ti_page_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_page_class_init (TIPageClass *ti_page_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_page_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_page_dispose;

    ti_page_class->setup_page = NULL;
    ti_page_class->restart_page = NULL;
		
	/* Add private */
	g_type_class_add_private (ti_page_class, sizeof (TIPagePrivate));
}

/**
 * Instance initialization.
 */
static void
ti_page_init (TIPage *ti_page)
{
    TIPagePrivate *priv = TI_PAGE_GET_PRIVATE (ti_page);

    priv->disposed = FALSE;
    priv->parent_notebook = NULL;
    priv->page = NULL;
    priv->page_label = NULL;
}

/**
 * Loads a new instance.
 */
void
_ti_page_new (TIPage** page, GtkNotebook* parent_notebook, const gchar* glade_file_name) 
{
    TIPagePrivate* priv;
    guint result;
    GladeXML* glade_xml;
    gchar* glade_file_path = NULL;
    TIPageClass* klass = NULL;

    g_assert (parent_notebook != NULL);
    g_assert (GTK_IS_NOTEBOOK (parent_notebook));
    g_assert (page != NULL);
    g_assert (*page != NULL);

    klass = TI_PAGE_GET_CLASS (*page);

    priv = TI_PAGE_GET_PRIVATE (*page);

    priv->parent_notebook = parent_notebook;
    g_object_ref (parent_notebook);
    

    glade_file_path = g_strdup_printf ("%s%s", TI_DATA_DIR_PREFIX, glade_file_name);
    glade_xml = glade_xml_new (glade_file_path, NULL, NULL);
    if (glade_xml == NULL)
    {
        g_critical ("Error loading glade file \"%s\".", glade_file_path);
        g_object_unref (*page);
        *page = NULL;
        goto CLEAN_UP;
    }

    priv->page = glade_xml_get_widget (glade_xml, "page");
    g_assert (priv->page != NULL);
    g_object_ref (priv->page);
    
    priv->page_label = glade_xml_get_widget (glade_xml, "page_label");
    g_assert (priv->page_label != NULL);
    g_object_ref (priv->page_label);

    g_assert (klass->setup_page != NULL);
    klass->setup_page (*page, glade_xml);

    if (klass->restart_page != NULL)
    {
        klass->restart_page (*page);
    }

    /* Get rid of the glade file and its holder windows. */
    gtk_widget_unparent (priv->page);
    gtk_widget_unparent (priv->page_label);
    
    g_object_unref (glade_xml);
    glade_xml = NULL;        

    /* Append page to notebook */
    result = gtk_notebook_append_page (parent_notebook, priv->page, priv->page_label);
    if (result == -1)
    {
        g_critical ("Unable to append page to notebook.");
        g_object_unref (*page);
        *page = NULL;
        goto CLEAN_UP;
    }

    CLEAN_UP:
    g_free (glade_file_path);
}

/**
 * Detach from Notebook
 */
void ti_page_detach_from_notebook (TIPage* page)
{
    g_assert (page != NULL && TI_IS_PAGE (page));
    TIPagePrivate* priv = TI_PAGE_GET_PRIVATE (page);
    gint page_num;

    if (priv->parent_notebook == NULL)
        return;

    page_num = gtk_notebook_page_num (priv->parent_notebook, priv->page);
    g_assert (page_num != -1);

    gtk_notebook_remove_page (priv->parent_notebook, page_num);

    g_object_unref (priv->parent_notebook);
    priv->parent_notebook = NULL;
}

/**
 * Attach to Notebook
 */
void ti_page_attach_to_notebook (TIPage* page, GtkNotebook* notebook)
{
    g_assert (page != NULL && TI_IS_PAGE (page));
    TIPagePrivate* priv = TI_PAGE_GET_PRIVATE (page);
    TIPageClass* klass = TI_PAGE_GET_CLASS (page);

    g_assert (notebook != NULL);
    g_assert (GTK_IS_NOTEBOOK (notebook));

    // Detach from current notebook, if there is one.
    ti_page_detach_from_notebook (page);

    priv->parent_notebook = notebook;
    g_object_ref (notebook);

    gtk_notebook_append_page (notebook, priv->page, priv->page_label);

    if (klass->restart_page != NULL)
    {
        klass->restart_page (page);
    }
}
