[comment {-*- tcl -*- doctools manpage}]
[manpage_begin smtpd n 1.4]
[copyright {Pat Thoyts <patthoyts@users.sourceforge.net>}]
[moddesc   {Tcl SMTP Server Package}]
[titledesc {Tcl SMTP server implementation}]
[category  Networking]
[require Tcl 8.3]
[require smtpd [opt 1.4]]
[description]
[para]

The [package smtpd] package provides a simple Tcl-only server library
for the Simple Mail Transfer Protocol as described in

RFC  821 ([uri http://www.rfc-editor.org/rfc/rfc821.txt]) and
RFC 2821 ([uri http://www.rfc-editor.org/rfc/rfc2821.txt]).

By default the server will bind to the default network address and the
standard SMTP port (25).

[para]

This package was designed to permit testing of Mail User Agent code
from a developers workstation. [emph "It does not attempt to deliver \
mail to your mailbox." ] Instead users of this package are expected to
write a procedure that will be called when mail arrives. Once this
procedure returns, the server has nothing further to do with the mail.

[section SECURITY]

On Unix platforms binding to the SMTP port requires root privileges. I
would not recommend running any script-based server as root unless
there is some method for dropping root privileges immediately after
the socket is bound. Under Windows platforms, it is not necessary to
have root or administrator privileges to bind low numbered
sockets. However, security on these platforms is weak anyway.

[para]

In short, this code should probably not be used as a permanently
running Mail Transfer Agent on an Internet connected server, even
though we are careful not to evaluate remote user input. There are
many other well tested and security audited programs that can be used
as mail servers for internet connected hosts.

[section COMMANDS]

[list_begin definitions]

[call [cmd ::smtpd::start] [opt [arg myaddr]] [opt [arg port]]]

Start the service listening on [arg port] or the default port 25. If
[arg myaddr] is given as a domain-style name or numerical
dotted-quad IP address then the server socket will be bound to that
network interface. By default the server is bound to all network
interfaces. For example:

[para]

[example {
  set sock [::smtpd::start [info hostname] 0]
}]

[para]

will bind to the hosts internet interface on the first available port.

[para]

At present the package only supports a single instance of a SMTP
server. This could be changed if required at the cost of making the
package a little more complicated to read. If there is a good reason
for running multiple SMTP services then it will only be necessary to
fix the [var options] array and the [var ::smtpd::stopped] variable
usage.

[para]

As the server code uses [cmd fileevent](n) handlers to process the
input on sockets you will need to run the event loop. This means
either you should be running from within [syscmd wish](1) or you
should [cmd vwait](n) on the [var ::smtpd::stopped] variable which is
set when the server is stopped.

[call [cmd ::smtpd::stop]]

Halt the server and release the listening socket. If the server has
not been started then this command does nothing.

The [var ::smtpd::stopped] variable is set for use with

[cmd vwait](n).

[para]

It should be noted that stopping the server does not disconnect any
currently active sessions as these are operating over an independent
channel. Only explicitly tracking and closing these sessions, or
exiting the server process will close down all the running
sessions. This is similar to the usual unix daemon practice where the
server performs a [syscmd fork](2) and the client session continues on
the child process.

[call [cmd ::smptd::configure] [opt "[arg option] [arg value]"] [opt "[arg option] [arg value] [arg ...]"]]

Set configuration options for the SMTP server. Most values are the
name of a callback procedure to be called at various points in the
SMTP protocol. See the [sectref CALLBACKS] section for details of the
procedures.

[list_begin definitions]

[def "[option -validate_host] [arg proc]"]

Callback to authenticate new connections based on the ip-address of
the client.

[def "[option -validate_sender] [arg proc]"]

Callback to authenticate new connections based on the senders email
address.

[def "[option -validate_recipient] [arg proc]"]

Callback to validate and authorize a recipient email address

[def "[option -deliverMIME] [arg proc]"]

Callback used to deliver mail as a mime token created by the tcllib
[package mime] package.

[def "[option -deliver] [arg proc]"]

Callback used to deliver email. This option has no effect if
the [option -deliverMIME] option has been set.

[list_end]

[call [cmd ::smtpd::cget] [opt [arg option]]]

If no [arg option] is specified the command will return a list of all
options and their current values. If an option is specified it will
return the value of that option.

[list_end]

[section CALLBACKS]

[list_begin definitions]
[def "[cmd validate_host] callback"]

This procedure is called with the clients ip address as soon as a
connection request has been accepted and before any protocol commands
are processed. If you wish to deny access to a specific host then an
error should be returned by this callback. For example:

[para]
[example {
 proc validate_host {ipnum} {
    if {[string match "192.168.1.*" $ipnum]} {
       error "go away!"
    }
 }
}]
[para]

If access is denied the client will receive a standard message that
includes the text of your error, such as:

[para]
[example {
 550 Access denied: I hate you.
}]
[para]

As per the SMTP protocol, the connection is not closed but we wait for
the client to send a QUIT command. Any other commands cause a

[const {503 Bad Sequence}] error.

[def "[cmd validate_sender] callback"]

The validate_sender callback is called with the senders mail address
during processing of a MAIL command to allow you to accept or reject
mail based upon the declared sender. To reject mail you should throw
an error. For example, to reject mail from user "denied":

[para]
[example {
 proc validate_sender {address} {
    eval array set addr [mime::parseaddress $address]
    if {[string match "denied" $addr(local)]} {
         error "mailbox $addr(local) denied"
    }
    return    
 }
}]

[para]

The content of any error message will not be passed back to the client.

[def "[cmd validate_recipient] callback"]

The validate_recipient callback is similar to the validate_sender
callback and permits you to verify a local mailbox and accept mail for
a local user address during RCPT command handling. To reject mail,
throw an error as above. The error message is ignored.

[def "[cmd deliverMIME] callback"]]

The deliverMIME callback is called once a mail message has been
successfully passed to the server. A mime token is constructed from
the sender, recipients and data and the users procedure it called with
this single argument. When the call returns, the mime token is cleaned
up so if the user wishes to preserve the data she must make a copy.

[para]
[example {
 proc deliverMIME {token} {
     set sender [lindex [mime::getheader $token From] 0]
     set recipients [lindex [mime::getheader $token To] 0]
     set mail "From $sender [clock format [clock seconds]]"
     append mail "\n" [mime::buildmessage $token]
     puts $mail
 }
}]

[def "[cmd deliver] callback"]

The deliver callback is called once a mail message has been
successfully passed to the server and there is no -deliverMIME option
set. The procedure is called with the sender, a list of recipients and
the text of the mail as a list of lines. For example:

[para]
[example {
 proc deliver {sender recipients data} {
    set mail "From $sender \
               [clock format [clock seconds]]"
    append mail "\n" [join $data "\n"]
    puts "$mail"
 }
}]
[para]

Note that the DATA command will return an error if no sender or
recipient has yet been defined.

[list_end]

[section VARIABLES]

[list_begin definitions]

[def [var ::smtpd::stopped]]

This variable is set to [const true] during the [cmd ::smtpd::stop]
command to permit the use of the [cmd vwait](n) command.

[comment ::smtpd::postmaster]
[comment {The e-mail address of the person that is the contact for the server.}]

[list_end]

[section AUTHOR]

Written by Pat Thoyts [uri mailto:patthoyts@users.sourceforge.net].

[section LICENSE]

This software is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the file
[file license.terms] for more details.

[section {BUGS, IDEAS, FEEDBACK}]

This document, and the package it describes, will undoubtedly contain
bugs and other problems.

Please report such in the category [emph smtpd] of the
[uri {http://sourceforge.net/tracker/?group_id=12883} {Tcllib SF Trackers}].

Please also report any ideas for enhancements you may have for either
package and/or documentation.


[keywords smtpd smtp services {rfc 821} {rfc 2821} vwait socket]
[manpage_end]
