// Copyright (C) 2009-2009 Martin Sandve Alnæs
// Licensed under the GNU LGPL Version 3.0.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program solves a 1D nonlinear hyperelasticity model.

#include <iostream>
#include <dolfin.h>
#include "generated_code/HyperElasticity1D.h"

using namespace dolfin;
using namespace HyperElasticity1D;


const double eps = DOLFIN_EPS;
const double K_VALUE = 1.0; 
const double B_VALUE = 0.1;

class Weight: public Expression
{
public:
  double value; 
  void eval(Array<double>& values, const Array<double>& x) const
  {
    values[0] = value;  
  }
};

class Source: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    const double dx = x[0] - 0.5;
    const double b = B_VALUE;
    const double K = K_VALUE;
    // Matching solution u = sin(x) - x:
    values[0] = -K*exp((1.0/4.0)*pow( pow(cos(x[0]),2.0)-1.0,2.0)*b)*( pow(cos(x[0]),2.0)-1.0)*sin(x[0])*b+-2.0*K*exp((1.0/4.0)*pow( pow(cos(x[0]),2.0)-1.0,2.0)*b)*pow(cos(x[0]),2.0)*sin(x[0])*b-K*exp((1.0/4.0)*pow( pow(cos(x[0]),2.0)-1.0,2.0)*b)*pow(cos(x[0]),2.0)*pow( pow(cos(x[0]),2.0)-1.0,2.0)*sin(x[0])*(b*b);
  }
};

class Solution: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    values[0] = sin(x[0]) - x[0];
  }
};

class DirichletBoundary: public SubDomain
{
  bool inside(const Array<double>& x, bool on_boundary) const
  {
    return x[0] < DOLFIN_EPS || x[0] > 1.0-DOLFIN_EPS;
  }
};

int main()
{
    // Geometry
    unsigned n = 20;
    UnitInterval mesh(n);
    
    // Function spaces
    Form_J::TrialSpace V(mesh);
    CoefficientSpace_K C(mesh);
    
    // Coefficient functions
    Weight K;
    K.value = K_VALUE;
    Weight b;
    b.value = B_VALUE;
    Source g;
    Solution usol;
    
    // Solution function
    Function u(V);
    
    // Forms
    Form_J  a(V, V);
    Form_F  L(V);
    
    // Attach coefficients to forms
    a.u = u;
    a.K = K;
    a.b = b;
    L.u = u;
    L.K = K;
    L.b = b;
    L.g = g;
    
    // Boundary conditions
    DirichletBoundary boundary;
    DirichletBC bc(V, usol, boundary);
    
    // Solve variational problem
    VariationalProblem problem(a, L, bc, true);
    problem.solve(u);
    
    // Compute F at u
    Vector F;
    problem.F(F, u.vector());
    cout << "Norm of F = " << F.norm("l2") << endl;
    cout << "Min  of F = " << F.min() << endl;
    cout << "Max  of F = " << F.max() << endl;
    
    // Compute e_h = usol_h - e
    Function usol_h(V);
    usol_h = usol; 
    Function e_h(V);
    e_h.vector() = usol_h.vector();
    e_h.vector() -= u.vector();
    cout << "Norm of e_h = " << e_h.vector().norm("l2") << endl;
    cout << "Min  of e_h = " << e_h.vector().min() << endl;
    cout << "Max  of e_h = " << e_h.vector().max() << endl;
    
    // Write solutions to file
    File ufile("u.pvd");
    ufile << u;
    File usolfile("usol.pvd");
    usolfile << usol_h;
    File efile("e.pvd");
    efile << e_h;
    
    //plot(u);
    //plot(usol_h);
}
