// Copyright (C) 2009-2009 Martin Sandve Alnæs
// Licensed under the GNU LGPL Version 3.0.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program solves Poissons equation in 3D.

#include <cmath>
#include <dolfin.h>
#include "generated_code/Poisson3D.h"

using std::sqrt;
using std::sin;
using std::cos;
using std::acos;
using namespace dolfin;
using namespace Poisson3D;

const double a = 1.0;
const double b = 1.0;
const double k = 1.0;
const double Pi = acos(-1);

class Source: public Function
{
public:
  Source(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    double dz = x[2] - 0.5;
    values[0] =  b * 3.0 * (Pi*Pi)*(k*k) * cos(Pi*k*x[0])*cos(x[1]*Pi*k)*cos(Pi*k*x[2])
              +  a * 3.0 * (Pi*Pi)*(k*k) * sin(x[1]*Pi*k)*sin(Pi*k*x[2])*sin(Pi*k*x[0]);
  }
};

class BoundarySource: public Function
{
public:
  BoundarySource(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    double dz = x[2] - 0.5;

    values[0] = 0.0;

    bool gx0 = x[0] < DOLFIN_EPS;
    bool gx1 = x[0] > 1.0-DOLFIN_EPS;
    bool gy0 = x[1] < DOLFIN_EPS;
    bool gy1 = x[1] > 1.0-DOLFIN_EPS;
    bool gz0 = x[2] < DOLFIN_EPS;
    bool gz1 = x[2] > 1.0-DOLFIN_EPS;

    if     (gx0) values[0] = -sin(x[1]*Pi*k)*sin(Pi*k*x[2])*Pi*k*a;
    else if(gx1) values[0] =  sin(x[1]*Pi*k)*sin(Pi*k*x[2])*Pi*k*cos(Pi*k)*a-sin(Pi*k)*b*Pi*k*cos(x[1]*Pi*k)*cos(Pi*k*x[2]);
    else if(gy0) values[0] = -sin(Pi*k*x[2])*Pi*k*sin(Pi*k*x[0])*a;
    else if(gy1) values[0] =  sin(Pi*k*x[2])*Pi*k*cos(Pi*k)*sin(Pi*k*x[0])*a-cos(Pi*k*x[0])*sin(Pi*k)*b*Pi*k*cos(Pi*k*x[2]);
    else if(gz0) values[0] = -sin(x[1]*Pi*k)*Pi*k*sin(Pi*k*x[0])*a;
    else if(gz1) values[0] =  sin(x[1]*Pi*k)*Pi*k*cos(Pi*k)*sin(Pi*k*x[0])*a-cos(Pi*k*x[0])*sin(Pi*k)*b*Pi*k*cos(x[1]*Pi*k);
  }
};

class Solution: public Function
{
public:
  Solution(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    values[0] =  b * cos(Pi*k*x[0])*cos(x[1]*Pi*k)*cos(Pi*k*x[2]) + a * sin(x[1]*Pi*k)*sin(Pi*k*x[2])*sin(Pi*k*x[0]);
  }
};

class EntireBoundary: public SubDomain
{
  bool inside(const double* x, bool on_boundary) const
  {
    return on_boundary;
  }
};

class NeumannBoundary: public SubDomain
{
  bool inside(const double* x, bool on_boundary) const
  {
    if(!on_boundary)
        return false;

    bool x0 = x[0] < DOLFIN_EPS;
    bool x1 = x[0] > 1.0-DOLFIN_EPS;
    bool y0 = x[1] < DOLFIN_EPS;
    bool y1 = x[1] > 1.0-DOLFIN_EPS;
    bool z0 = x[2] < DOLFIN_EPS;
    bool z1 = x[2] > 1.0-DOLFIN_EPS;
    
    // With a=0, b=1, k=1, n=30, we got:
    //return x0 || y0 || z0;                             // 0.00109, same as do-nothing on same sides
    //return x0 || y0 || z0 || x1;                       // 0.00152
    //return x0 || y0 || z0 || x1 || y1;                 // 0.00186
    //return x0 || y0 || z0 || x1 || y1 || z1;           // 0.00205 (no dbc at all)
    
    return z0 || z1 || y0 || y1;
  }
};

int main()
{
    // Geometry
    message("Mesh");
    unsigned n = 30;
    UnitCube mesh(n, n, n);

    // Boundaries
    message("Boundaries");
    unsigned neumann_boundary_marker   = 0;
    unsigned dirichlet_boundary_marker = 1;
    unsigned do_nothing_boundary_marker = 2;

    MeshFunction<unsigned> boundaries(mesh, mesh.geometry().dim()-1);

    // First set all DBC:
    EntireBoundary entire_boundary;
    entire_boundary.mark(boundaries, dirichlet_boundary_marker);

    // Then override with Neumann where we want it:
    NeumannBoundary neumann_boundary;
    neumann_boundary.mark(boundaries, neumann_boundary_marker);

    // Function spaces
    message("Function spaces");
    BilinearForm::TrialSpace V(mesh);

    // Forms
    message("Forms");
    BilinearForm a(V, V);
    LinearForm L(V);

    // Coefficient functions
    message("Functions");
    Source f(V);
    BoundarySource g(V);
    Solution usol(V);

    // Solution function
    Function u(V);

    // Attach functions to forms
    L.f = f;
    L.g = g;

    // Setup boundary conditions
    message("Boundary conditions");
    DirichletBC bc(V, usol, boundaries, dirichlet_boundary_marker);

    std::vector<BoundaryCondition*> bcs;
    bcs.push_back(&bc);

    // Create variational problem (a, L, bcs, cell_domains, exterior_facet_domains, interior_facet_domains, nonlinear)
    message("Variational problem");
    VariationalProblem problem(a, L, bcs, 0, &boundaries, 0, false);

    // Solve!
    message("Solving");
    //problem.set("linear solver", "direct");
    problem.set("linear solver", "iterative");
    problem.set("symmetric", true);
    problem.solve(u);

    // Interpolate exact solution
    Function usol_h(V);
    usol.interpolate(usol_h.vector(), V);

    Function e(V);
    e.vector() = u.vector();
    e.vector() -= usol_h.vector();
    
    int N = u.vector().size();
    cout << endl;
    cout << "==========================" << endl;
    cout << "Norm of e = " << e.vector().norm() / sqrt(N) << endl;
    cout << "Min  of e = " << e.vector().min() << endl;
    cout << "Max  of e = " << e.vector().max() << endl;
    cout << "==========================" << endl;

    // Write results to file
    message("Writing to file");
    File ufile("u.pvd");
    ufile << u;
    File usolfile("usol.pvd");
    usolfile << usol_h;
    File efile("e.pvd");
    efile << e;

    //plot(u);
    
    return 0;
}

