/***************************************************************************
 *   Copyright (C) 2004 by Fred Schaettgen                                 *
 *   kde.sch@ttgen.net                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "clockpaintview.h"
#include <kmessagebox.h>
#include <algorithm>
#include <qtimer.h>
#include <qdir.h>
#include <kjs/interpreter.h>
#include <kdebug.h>
#include "ecmafunc.h"
#include <qfile.h>
#include <qtextstream.h>
#include <klocale.h>
#include <kapplication.h>
#include <kstandarddirs.h>
#include "prefs.h"
#include <qtooltip.h>
#include <qcolor.h>
#include <kdirwatch.h>
#include <knotifyclient.h>
//#include "../styleclock.h"
#include "clockappinterface.h"
#include "debugdialog.h"
#include <kdialogbase.h>
#include <qslider.h>
#include <qlineedit.h>

using namespace std;

const QString resourceDirName = "styleclock";

ClockPaintView::ClockPaintView(QWidget*parent, ClockAppInterface* iface, Prefs* prefs) :
    QGLWidget(parent),
    prefs(prefs),
    envSeconds(0),
    envMinutes(0),
    envHours(0)
{   
    this->owner = iface;
    interpreter = NULL;
    
    dirWatch = NULL;
    debugMode = false;
    debugDialog = NULL;
    debugDialogFrame = NULL;
    
    defineLayerFunc = new ECMAFunc(this);
    connect(defineLayerFunc, SIGNAL(call(KJS::ExecState*, KJS::Object&, 
        const KJS::List&, KJS::Value&)), this, SLOT(ecmaSlotDefineLayer( 
            KJS::ExecState*, KJS::Object&, const KJS::List&, KJS::Value&)));
    getColorFunc = new ECMAFunc(this);
    connect(getColorFunc, SIGNAL(call(KJS::ExecState*, KJS::Object&, 
        const KJS::List&, KJS::Value&)), this, SLOT(ecmaSlotGetColor( 
            KJS::ExecState*, KJS::Object&, const KJS::List&, KJS::Value&)));
    
    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(slotTimeout()));
    
    globalObject = KJS::Object(new KJS::ObjectImp());
    interpreter = new KJS::Interpreter(globalObject);
    reloadSettings();
}

ClockPaintView::~ClockPaintView()
{
    delete interpreter;
}

int ClockPaintView::loadTexture(QString filename)
{
    QImage buf;
    buf.setAlphaBuffer(true);
    if ( ! buf.load(filename) ) {
        KNotifyClient::event(KApplication::kApplication()->mainWidget()->winId(), 
            "FileOpenError", i18n("Failed to load image %1").arg(filename)); 
        kdDebug() << i18n("Failed to load image %1").arg(filename) << endl;
        return 0;
    }
        
    GLuint texName;
    glGenTextures(1, &texName);
    glBindTexture(GL_TEXTURE_2D, texName);
    
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
    
    int wOld, hOld;
    int mipmapLevel = 0;
    do {
        QImage tex1;
        wOld = buf.width();
        hOld = buf.height();
        tex1 = QGLWidget::convertToGLFormat(buf);
        
        glTexImage2D(GL_TEXTURE_2D, mipmapLevel, 4, tex1.width(), tex1.height(), 0,
            GL_RGBA, GL_UNSIGNED_BYTE, tex1.bits() );
        buf = buf.smoothScale(max(1, buf.width()/2), max(1, buf.height()/2));
        ++mipmapLevel;
    
    } while (wOld != buf.width() && hOld != buf.height());
    return texName;
} 

void ClockPaintView::initializeGL()
{
    // Set up the rendering context, define display lists etc.:
    glClearColor( 1.0, 0.0, 0.0, 0.0 );
    glDisable(GL_DEPTH_TEST);
    glEnable(GL_BLEND);
    glDisable(GL_DITHER);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable(GL_TEXTURE_2D);

    glGenTextures(1, &bgTexName);
    //loadTexture("/usr/share/icons/crystalsvg/128x128/apps/clock.png");
}

void ClockPaintView::resizeGL( int w, int h )
{
    // setup viewport, projection etc.:
    this->screenWidth = w;
    this->screenHeight = h;
    glViewport( 0, 0, (GLint)w, (GLint)h );
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    float left = 0;
    float right = w;
    float top = 0;
    float bottom = h;
    float near = 0;
    float far = 1;
    glOrtho(left, right, bottom, top, near, far);
}

void ClockPaintView::paintGL()
{
    // Get the background pixmap
        
    QPixmap bgPixmap(screenWidth, screenHeight);
    bgPixmap.fill(QColor(255,0,0));
    bgPixmap.fill(parentWidget(), 0, 0);
    double bgOrigSizeX = bgPixmap.width();
    double bgOrigSizeY = bgPixmap.height();
    int bgExpandedSizeX = 1;
    int bgExpandedSizeY = 1;
    while (bgExpandedSizeX < bgOrigSizeX) bgExpandedSizeX *= 2; 
    while (bgExpandedSizeY < bgOrigSizeY) bgExpandedSizeY *= 2; 
    double bgTexRight = bgOrigSizeX / bgExpandedSizeX;
    double bgTexBottom = bgOrigSizeY / bgExpandedSizeY;
    bgPixmap.resize(bgExpandedSizeX, bgExpandedSizeY);
    //bgPixmap.fill(QColor(128,129,130));
    QImage bgImage = bgPixmap.convertToImage();
    bgImage = bgImage.convertDepth(32, Qt::ThresholdDither);
    bgImage.setAlphaBuffer(true);
    QImage bgGlImage = QGLWidget::convertToGLFormat(bgImage);
        
    glBindTexture(GL_TEXTURE_2D, bgTexName);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
    glTexImage2D(GL_TEXTURE_2D, 0, 4, bgGlImage.width(), bgGlImage.height(), 0,
            GL_RGBA, GL_UNSIGNED_BYTE, bgGlImage.bits() );
            
    // draw the scene:
    glMatrixMode(GL_MODELVIEW);
    //QColor bgColor = KApplication::palette().active().background();
    //glColor3d(bgColor.red()/256.0, bgColor.green()/256.0, bgColor.blue()/256.0);
    glBindTexture(GL_TEXTURE_2D, bgTexName);
    glEnable(GL_TEXTURE_2D);
    glColor4f(1, 1, 1, 1);
    glBegin( GL_QUADS );
    glTexCoord2f(bgTexRight, 1.0);
    glVertex3f(screenWidth, 0., 0);
    glTexCoord2f(0, 1.0);
    glVertex3f(0. , 0., 0);
    glTexCoord2f(0, 1.0-bgTexBottom);
    glVertex3f(0., screenHeight,0);
    glTexCoord2f(bgTexRight, 1.0-bgTexBottom);
    glVertex3f(screenWidth, screenHeight, 0);
    glEnd();
    // Get rid of the background texture
        
    
    if (interpreter == NULL) return;
    
    list<DisplayListItem>::const_iterator it;
    KJS::ExecState* state = interpreter->globalExec();
    
    for (it = displayList.begin(); it != displayList.end(); ++it) {
        
        DisplayListItem item = *it;
        double moveX = item.properties.get(state, "moveX").toNumber(state);
        double moveY = item.properties.get(state, "moveY").toNumber(state);
        double anchorX = item.properties.get(state, "anchorX").toNumber(state);
        double anchorY = item.properties.get(state, "anchorY").toNumber(state);
        double rotate = item.properties.get(state, "rotate").toNumber(state);
        double scaleX = item.properties.get(state, "scaleX").toNumber(state);
        double scaleY = item.properties.get(state, "scaleY").toNumber(state);
        double texX = item.properties.get(state, "texX").toNumber(state);
        double texY = item.properties.get(state, "texY").toNumber(state);
        double texW = item.properties.get(state, "texW").toNumber(state);
        double texH = item.properties.get(state, "texH").toNumber(state);
        KJS::Object col = item.properties.get(state, "color").toObject(state);
        /*kdDebug() << QString("Move(%1/%2) ").arg(moveX).arg(moveY) <<
            QString("Move(%1/%2) ").arg(moveX).arg(moveY) <<
            QString("Anchor(%1/%2) ").arg(anchorX).arg(anchorY) << 
            QString("Rotate(%1) ").arg(rotate) <<
            QString("Scale(%1/%2) ").arg(scaleX).arg(scaleY)<<  endl;*/
        
        //kdDebug() << "Drawing item " << item.imageName << endl;
        int textureName;
        if (imageMap.find(item.imageName) != imageMap.end()) {
            textureName = imageMap[item.imageName];
        }
        else {
            if (dirWatch) {
                kdDebug() << "Watching file " << expandFilename(item.imageName) << endl;
                dirWatch->addFile(expandFilename(item.imageName));
            }
            textureName = loadTexture(expandFilename(item.imageName));
            imageMap[item.imageName] = textureName;
        }
        //kdDebug() << item.imageName << " " << textureName << endl;
        glPushMatrix();
        
        glTranslatef(moveX, moveY, 0);
                
        glTranslatef((anchorX-texX)*scaleX/texW, (anchorY-texY)*scaleY/texH, 0);
        glRotatef(rotate, 0, 0, 1);
        glTranslatef(-(anchorX-texX)*scaleX/texW, -(anchorY-texY)*scaleY/texH, 0);
        glScalef(scaleX, scaleY, 1);
        
        glBindTexture(GL_TEXTURE_2D, textureName);
        glEnable(GL_TEXTURE_2D);
        
        glColor4f(col.get(state, "r").toNumber(state),
            col.get(state, "g").toNumber(state),
            col.get(state, "b").toNumber(state),
            1.0);
        
        glBegin( GL_QUADS );
        glTexCoord2f(texX, 1-texY);
        glVertex3f(0, 0, 0);
        glTexCoord2f(texX+texW, 1-texY);
        glVertex3f(1, 0, 0);
        glTexCoord2f(texX+texW, 1-(texY+texH));
        glVertex3f(1, 1, 0);
        glTexCoord2f(texX, 1-(texY+texH));
        glVertex3f(0, 1,0);
        glEnd();
        glPopMatrix();
    }
}

int ClockPaintView::heightForWidth(int width)
{
    kdDebug() << "Calling heightForWidth(" << width << ")" << endl;
    if (interpreter == NULL) return width;
    
    executeScript(false);
    
    KJS::ExecState* state = interpreter->globalExec();
    KJS::Object globalObject = interpreter->globalObject();
    if (globalObject.hasProperty(state, "heightForWidth")) {
        kdDebug() << "Found heightForWidth property" << endl;
        KJS::Object h4wObj = globalObject.get(state, "heightForWidth").toObject(state);
        if (h4wObj.isNull() == false && h4wObj.implementsCall()) {
            KJS::List argList;
            argList.append(KJS::Number(width));
            KJS::Object thisObj(new KJS::ObjectImp());
            KJS::Value h = h4wObj.call(state, thisObj, argList);
            kdDebug() << "Height for width is " << h.toInteger(state) << endl;
            return max(8,min(256,h.toInteger(state)));   
        }
    }
    return width;
}

int ClockPaintView::widthForHeight(int height)
{
    kdDebug() << "Calling widthForHeight(" << height << ")" << endl;
    if (interpreter == NULL) return height;
    
    executeScript(false);
    
    KJS::ExecState* state = interpreter->globalExec();
    KJS::Object globalObject = interpreter->globalObject();
    if (globalObject.hasProperty(state, "widthForHeight")) {
        kdDebug() << "Found widthForHeight property" << endl;
        KJS::Object w4hObj = globalObject.get(state, "widthForHeight").toObject(state);
        if (w4hObj.isNull() == false && w4hObj.implementsCall()) {
            KJS::List argList;
            argList.append(KJS::Number(height));
            KJS::Object thisObj(new KJS::ObjectImp());
            KJS::Value w = w4hObj.call(state, thisObj, argList);
            kdDebug() << "Width for height is " << w.toInteger(state) << endl;
            return max(8,min(256,w.toInteger(state)));   
        }
    }
    return height;
}

void ClockPaintView::executeScript(bool defineSize)
{
    if (interpreter == NULL) return;
    
    KJS::ExecState* state = interpreter->globalExec();
    
    // Updating the script state variables
    QTime curTime = QTime::currentTime();
    QDate curDate = QDate::currentDate();
    
    if (debugDialog != NULL && debugDialog->isVisible()) {
        int h = debugDialog->hourSlider->value();
        int m = debugDialog->minuteSlider->value();
        int s = debugDialog->secondSlider->value();
        curTime = QTime(h, m, s);
    }
        
    globalObject.put(state, "widthForHeight", KJS::Null());
    globalObject.put(state, "heightForWidth", KJS::Null());
    globalObject.put(state, "second", KJS::Number(curTime.second()));
    globalObject.put(state, "minute", KJS::Number(curTime.minute()));
    globalObject.put(state, "hour", KJS::Number(curTime.hour()));
    globalObject.put(state, "use12HourFormat",
        KJS::Boolean(KGlobal::locale()->use12Clock()));
    
    QDateTime alarmTime = owner->getAlarmTime();
    if (alarmTime.isValid() && alarmTime > QDateTime::currentDateTime()) {
        globalObject.put(state, "alarmActive", KJS::Boolean(true));
        globalObject.put(state, "alarmSecond", KJS::Number(alarmTime.time().second()));
        globalObject.put(state, "alarmMinute", KJS::Number(alarmTime.time().minute()));
        globalObject.put(state, "alarmHour", KJS::Number(alarmTime.time().hour()));
    }
    else {
        globalObject.put(state, "alarmActive", KJS::Boolean(false));
    }
    
    globalObject.put(state, "month", KJS::Number(curDate.month()));
    globalObject.put(state, "year", KJS::Number(curDate.year()));
    globalObject.put(state, "day", KJS::Number(curDate.day()));
    globalObject.put(state, "dayOfWeek", KJS::Number(curDate.dayOfWeek()));
        
    globalObject.put(state, "screenHeight",KJS::Number(
        defineSize ? screenHeight : 0));
    globalObject.put(state, "screenWidth",KJS::Number(
        defineSize ? screenWidth : 0));    
    globalObject.put(state, "showSeconds", KJS::Boolean(showSeconds));
    globalObject.put(state, "showDate", KJS::Boolean(showDate));
    
    KJS::Object sysCols(new KJS::ObjectImp());
    globalObject.put(state, "systemColors", sysCols);
    sysCols.put(state, "activeTitleBar", createColorObject(state, 0, 0.5, 1));
    
    displayList.clear();
    KJS::Completion c = interpreter->evaluate((const char*)(script.utf8()));
    QString msg;
    switch (c.complType()) {
    case KJS::Normal:
        msg = "Normal";
        break;
    case KJS::Break:
        msg = "Break";
        break;
    case KJS::Continue:
        msg = "continue";
        break;
    case KJS::ReturnValue:
        msg = "ReturnValue";
        break;
    case KJS::Throw:
        msg = QString("Throw:%1 %2").arg(c.target().ascii())
            .arg(c.value().toString(state).ascii());
        break;
    default:
        msg = "Unknown result code";
    }
    //KMessageBox::information(this, msg, "Script result");
    if (msg!="Normal") {
        KNotifyClient::event(winId(), "ScriptError", 
            i18n("Script error: %1").arg(msg));
        kdDebug() << i18n("Script error: %1").arg(msg) << endl;
    }

}

void ClockPaintView::slotTimeout()
{
    QTime curTime = QTime::currentTime();
    if ((showSeconds == true && lastRedrawTime.second() != curTime.second()) ||         
        lastRedrawTime.isValid() == false || 
        lastRedrawTime.minute() != curTime.minute()) 
    {
        updateClock();    
        QDate curDate = QDate::currentDate();
        QString dateStr = curDate.toString();
        if (lastDateString != dateStr) {
            QToolTip::remove(this);
            QToolTip::add(this, dateStr);
            lastDateString = dateStr;
        }
        lastRedrawTime = curTime;   
    }
}

void ClockPaintView::updateClock()
{
    executeScript(true);            
    repaint();
}

QString ClockPaintView::expandFilename(const QString& s)
{
    KStandardDirs *dirs = KApplication::kApplication()->dirs();
    return dirs->findResource("data", QString("%1/themes/%2/%3")
        .arg(resourceDirName).arg(themeName).arg(s));
}

QStringList ClockPaintView::getAvailableThemes()
{
    KStandardDirs *dirs = KApplication::kApplication()->dirs();
    QStringList themeList;
    QStringList themeDirs;
    themeDirs = dirs->findDirs("data", QString("%1/themes").arg(resourceDirName));
    for (uint nDir=0; nDir < themeDirs.size(); ++nDir) {
        kdDebug() << "Searching for themes in " << themeDirs[nDir] << endl;
        QDir curDir(themeDirs[nDir]);
        QStringList themes = curDir.entryList(QDir::Dirs | QDir::Readable, QDir::Name);
        for (uint nTheme=0; nTheme < themes.size(); ++nTheme) {
            kdDebug() << "Found theme " << themes[nTheme] << endl;
            if (themes[nTheme].startsWith(".")) continue;
            if (themeList.contains(themes[nTheme]) == 0) {
                themeList.push_back(themes[nTheme]);
            }
        }
    }
    themeList.sort();
    return themeList;
}


void ClockPaintView::setTheme(const QString& s)
{
    timer->stop();
    if (dirWatch) dirWatch->stopScan();
    delete dirWatch;
    themeName = s;
    if (debugMode) {
        dirWatch = new KDirWatch(this);
        connect(dirWatch, SIGNAL(dirty(const QString&)), this, SLOT(slotThemeModified())); 
    }
    else dirWatch = NULL;
    
    if (dirWatch) {
        kdDebug() << "Watching directory " << expandFilename("") << endl;
        dirWatch->addDir(expandFilename(""));
    }
    displayList.clear();
    
    // Remove the old cached images
    std::map<QString, int>::iterator it;
    for (it = imageMap.begin(); it != imageMap.end(); ++it) {
        GLuint texture = it->second;
        glDeleteTextures(1, &texture);
    }
    imageMap.clear();
    
    //globalObject = KJS::Object(new KJS::ObjectImp());
    //delete interpreter;
    //interpreter = new KJS::Interpreter(globalObject);
    KJS::ExecState* state = interpreter->globalExec();
    
    globalObject.put(state, KJS::Identifier("defineLayer"), 
        KJS::Object(defineLayerFunc->getObjectImp()));
    globalObject.put(state, KJS::Identifier("getColor"), 
        KJS::Object(getColorFunc->getObjectImp()));
    
    QString scriptName = expandFilename("main.js");
    if (scriptName.length() > 0) {
        QFile f(scriptName);
        f.open(IO_ReadOnly);
        QTextStream stream(&f);
        script = stream.read();
    }
    else {
        KNotifyClient::event(KApplication::kApplication()->mainWidget()->winId(),
            "FileOpenError", 
            i18n("Failed to load script %1 for theme %2").arg(scriptName).arg(themeName)); 
        kdDebug() << i18n("Failed to load script %1 for theme %2")    
            .arg(scriptName).arg(themeName) << endl;
        script = "";
        return;
    }

    updateClock();
    timer->start(200);
}

void ClockPaintView::slotThemeModified() 
{
    QTimer::singleShot(0, this, SLOT(slotThemeModifiedAndWait()));
}

    
void ClockPaintView::slotThemeModifiedAndWait() 
{
    kdDebug() << "Theme modified! Reloading.." << endl;
    setTheme(themeName);
}

void ClockPaintView::ecmaSlotDefineLayer(KJS::ExecState */*exec*/, KJS::Object &/*obj*/, 
    const KJS::List &args, KJS::Value &ret)
{
    KJS::ExecState* state = interpreter->globalExec();
    int numArgs = args.size();
   
    // Default values for the parameters
    //item.moveX = item.moveY = item.anchorX = item.anchorY = item.rotate = 0;
    //item.scaleX = item.scaleY = 1;
    
    if (numArgs != 1) {
        kdError() << "Wrong number of arguments for js function defineLayer(filename)." 
            << endl;
        ret = KJS::Undefined();
        return;
    }
    
    QString imageName = QString(args[0].toString(state).ascii());
    DisplayListItem item(imageName);
    ret = item.properties;
    displayList.push_back(item);
    
    // Setting default values for the ecmascript object
    KJS::Object& props = item.properties;
    props.put(state, "moveX", KJS::Number(0));
    props.put(state, "moveY", KJS::Number(0));
    props.put(state, "anchorX", KJS::Number(0));
    props.put(state, "anchorY", KJS::Number(0));
    props.put(state, "rotate", KJS::Number(0));
    props.put(state, "scaleX", KJS::Number(1));
    props.put(state, "scaleY", KJS::Number(1));
    props.put(state, "texX", KJS::Number(0.0));
    props.put(state, "texY",KJS::Number(0.0));
    props.put(state, "texW",KJS::Number(1.0));
    props.put(state, "texH",KJS::Number(1.0));
    props.put(state, "color", createColorObject(state, 1, 1, 1));
    
    //kdDebug() << QString("called ecmaSlotDefineLayer(\"%1\")").arg(imageName) << endl;
}

std::map<QString, QString> ClockPaintView::currentThemeProperties()
{
    return themeProperties;
}

void ClockPaintView::setProperty(QString theme, QString property, QString val)
{
    kdDebug() << "setProperty "<< theme << " " << property << " " << val << endl;
    QStringList props = prefs->properties();
    QStringList newProps;
    bool bFound = false;
    for (uint n=0; n<props.size(); ++n) {
        if (props[n].startsWith(theme+"/"+property+"/")) {
             newProps.append(theme+"/"+property+"/"+val);
             kdDebug() << "property replaced" << endl;
             bFound = true;
        }
        else newProps.append(props[n]);
    }
    if (bFound == false) {
        newProps.append(theme+"/"+property+"/"+val);
    }
    prefs->setProperties(newProps);
}

void ClockPaintView::ecmaSlotGetColor(KJS::ExecState */*exec*/, KJS::Object &/*obj*/, 
    const KJS::List &args, KJS::Value &ret)
{
    KJS::ExecState* state = interpreter->globalExec();
    int numArgs = args.size();
   
    if (numArgs != 4) {
        kdError() << "Wrong number of arguments for getColor(name,r,g,b)" << endl;
        ret = KJS::Undefined();
        return;
    }
    
    QString colorName = QString(args[0].toString(state).ascii());
    double r = args[1].toNumber(state);
    double g = args[2].toNumber(state);
    double b = args[3].toNumber(state);
    KJS::Object retObj = KJS::Object(new KJS::ObjectImp());
    
    QString colorStr = QColor(int(r*255), int(g*255), int(b*255)).name();
    if (themeProperties[colorName] != QString()) {
        colorStr = themeProperties[colorName];
    }
    QColor color(colorStr);
    
    retObj.put(state, "r", KJS::Number(color.red()/255.0));
    retObj.put(state, "g", KJS::Number(color.green()/255.0));
    retObj.put(state, "b", KJS::Number(color.blue()/255.0));
    ret = retObj;
}

KJS::Object ClockPaintView::createColorObject(KJS::ExecState* state, double r, double g, double b)
{
    KJS::Object obj(new KJS::ObjectImp());
    obj.put(state, "r", KJS::Number(r));
    obj.put(state, "g", KJS::Number(g));
    obj.put(state, "b", KJS::Number(b));
    return obj;
}

ClockPaintView::DisplayListItem::DisplayListItem(QString imageName)
{
    this->imageName = imageName;
    properties = KJS::Object(new KJS::ObjectImp());
}

void ClockPaintView::reloadSettings()
{
    kdDebug() << "ClockPaintView::reloadSettings" << endl;
    
    if (prefs->theme() != themeName) {
        kdDebug() << "ClockPaintView: setting theme to " << prefs->theme() << endl;
        setTheme(prefs->theme());
    }
    
    // Update the theme properties
    themeProperties.clear();
    QStringList propList = prefs->properties();
    for (uint n=0; n<propList.size(); ++n) {
        QStringList propItems = QStringList::split("/", propList[n]);
        kdDebug() << "Check property override: " << propItems[0] 
            << " " << propItems[1] << endl;
        if (propItems[0] == themeName) {
            themeProperties[propItems[1]] = propItems[2];
        }
    }
    updateClock();
    
    showSeconds = prefs->showSeconds();
    showDate = prefs->showDate();
    emit sizeSuggestionChanged();
}

QString ClockPaintView::currentTheme()
{
    return themeName;
}

ClockPaintView::DisplayListItem::~DisplayListItem()
{
    //delete properties.imp();
}

void ClockPaintView::setDebugMode(bool b)
{
    debugMode = b;
    if (debugMode == true) {
        if (debugDialog == NULL) {
            debugDialogFrame = new KDialogBase(this, "debugdialog", 
                false, i18n("debug"), KDialogBase::Close );
            debugDialog = new DebugDialog(debugDialogFrame);
            debugDialogFrame->setMainWidget(debugDialog);
            connect(debugDialog->hourSlider, SIGNAL(valueChanged(int)),\
                this, SLOT(slotDebugDialogChanged()));
            connect(debugDialog->minuteSlider, SIGNAL(valueChanged(int)),\
                this, SLOT(slotDebugDialogChanged()));
            connect(debugDialog->secondSlider, SIGNAL(valueChanged(int)),\
                this, SLOT(slotDebugDialogChanged()));
        }
        debugDialogFrame->show();
        slotDebugDialogChanged();
    }
    else {
        debugDialogFrame->hide();
        slotDebugDialogChanged();
    }
    setTheme(themeName);
}

void ClockPaintView::slotDebugDialogChanged()
{
    debugDialog->hourEdit->setText(QString::number(debugDialog->hourSlider->value()));
    debugDialog->minuteEdit->setText(QString::number(debugDialog->minuteSlider->value()));
    debugDialog->secondEdit->setText(QString::number(debugDialog->secondSlider->value()));
    updateClock();
}


#include "clockpaintview.moc"
