#include <kglobal.h>
#include <klocale.h>
#include <kconfig.h>
#include <kapplication.h>
#include <kmessagebox.h>
#include <qhbox.h>
//#include <kconfigdialogmanager.h>
#include <kconfigdialog.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <kdebug.h>
#include <ktimewidget.h>
#include <qtimer.h>
#include <vector>
#include <algorithm>

#include "datepicker.h"
#include "alarmdlg.h"
#include "settimer.h"
#include "setalarm.h"
#include "styleconfigdlg.h"
#include "styleclockwidget/prefs.h"
#include "styleclock.h"
#include "styleclockwidget/clockpaintview.h"

const int themeBase = 2000;
const int alarmBase = 3000;
const int timerBase = 4000;
const int menuIndexThemes = 1;

StyleClock::StyleClock(const QString& configFile, Type type, int actions, QWidget *parent, const char *name)
    : DCOPObject("styleclock"), KPanelApplet(configFile, type, actions, parent, name)
{
    datePicker = NULL;
    alarmDlg = NULL;
    
    // Get the current application configuration handle
    this->configFile = configFile;
    ksConfig = config();
    prefs = new Prefs(sharedConfig());
    prefs->readConfig();
    kdDebug() << "Current alarm time " << prefs->currentAlarmTime().toString() << endl;
    kdDebug() << "Current time " << QDateTime::currentDateTime().toString() << endl;
    if (prefs->currentAlarmTime() > QDateTime::currentDateTime()) {
        kdDebug() << "Setting alarm time" << endl;
        alarmTime = prefs->currentAlarmTime();
    }
    
    if (kapp->authorizeKAction("kicker_rmb")) {
        contextMenu = new KPopupMenu(this);
        themeMenu = new KPopupMenu(contextMenu);
        alarmMenu = new KPopupMenu(contextMenu);
        timerMenu = new KPopupMenu(contextMenu);
        connect(themeMenu, SIGNAL(aboutToShow()), this, SLOT(slotAboutToShowThemeMenu()));
        connect(themeMenu, SIGNAL(activated(int)), this, SLOT(slotThemesMenuClicked(int)));
        
        connect(alarmMenu, SIGNAL(aboutToShow()), this, SLOT(slotAboutToShowAlarmMenu()));
        connect(alarmMenu, SIGNAL(activated(int)), this, SLOT(slotAlarmMenuClicked(int)));
        connect(timerMenu, SIGNAL(aboutToShow()), this, SLOT(slotAboutToShowTimerMenu()));
        connect(timerMenu, SIGNAL(activated(int)), this, SLOT(slotTimerMenuClicked(int)));
        connect(contextMenu, SIGNAL(aboutToShow()), this, SLOT(slotAboutToShowContextMenu()));
        
        contextMenu->insertTitle(SmallIcon("clock"), i18n("Clock"), 0);
        contextMenu->insertItem(SmallIcon("kalarm"), i18n("Alarm clock"), alarmMenu);
        contextMenu->insertItem(SmallIcon("kalarm"), i18n("Countdown timer"), timerMenu);
        cancelAlarmItemIndex = contextMenu->insertItem(SmallIcon("stop"), 
            i18n("Cancel alarm"), this, SLOT(slotCancelAlarm()));
        contextMenu->insertItem(SmallIcon("style"), i18n("Themes"), 
            themeMenu, menuIndexThemes);
        contextMenu->insertItem(SmallIcon("configure"), i18n("Configure..."), 
            this, SLOT(slotShowPreferences()));

        setCustomMenu(contextMenu);
    }
    
    mainView = new ClockPaintView(this, this, prefs);
    connect(mainView, SIGNAL(sizeSuggestionChanged()), 
        this, SLOT(slotClockSizeSuggestionChanged()));
    //mainView->setPreferredWidth(32);
    //mainView->setPreferredHeight(32);
    //mainView->show();
    mainView->show();
    if (QGLFormat::hasOpenGL() == false) {
        KMessageBox::detailedSorry(this, 
            i18n("Styleclock needs OpenGL support, but your system doesn't support it."),
            i18n("QGLFormat::hasOpenGL() returned false, indication no OpenGL support for Qt"),
            i18n("StyleClock"));
    }
}


StyleClock::~StyleClock()
{
    delete prefs; 
    prefs = NULL;
}


void StyleClock::about()
{
    KMessageBox::information(0, i18n("This is an about box"));
}


void StyleClock::help()
{
    KMessageBox::information(0, i18n("This is a help box"));
}

void StyleClock::slotShowPreferences()
{
    preferences();
}

void StyleClock::preferences()
{
    KConfigDialog *dialog = dynamic_cast<KConfigDialog*>(
        KConfigDialog::exists(configFile.local8Bit()));
    
        
    if (!dialog)
    {
        dialog = new StyleConfigDialog(this, prefs, mainView);
        connect(dialog, SIGNAL(settingsChanged()), this, SLOT(slotApplySettings()));
    }

    dialog->show();
}

int StyleClock::widthForHeight(int height) const
{
    kdDebug() << "StyleClock::widthForHeight " << height << endl;
    if (mainView == NULL) return height;
    return mainView->widthForHeight(height);
}

int StyleClock::heightForWidth(int width) const
{
    kdDebug() << "StyleClock::heightForWidth " << width << endl;
    if (mainView == NULL) return width;
    return mainView->heightForWidth(width);
}

void StyleClock::resizeEvent(QResizeEvent *e)
{
    if (mainView == NULL) return;
    mainView->resize(e->size().width(), e->size().height());
    mainView->updateClock();
}

void StyleClock::slotApplySettings()
{
    mainView->reloadSettings();
}

void StyleClock::updateThemeMenu()
{
    themeMenu->clear();
    QStringList themeList = mainView->getAvailableThemes();
    for (uint n=0; n<themeList.count(); ++n) {
        int i = themeMenu->insertItem(themeList[n], themeBase+n);    
        if (themeList[n] == mainView->currentTheme()) {
            themeMenu->setItemChecked(i, true);
        }
    }
}

void StyleClock::slotThemesMenuClicked(int n)
{
    QStringList themeList = mainView->getAvailableThemes();
    int index = n-themeBase;
    if (index>=0 && index<int(themeList.count())) {
        prefs->setTheme(themeList[index]);
        prefs->writeConfig();
        mainView->reloadSettings();
    }
}

void StyleClock::slotAboutToShowThemeMenu()
{
    updateThemeMenu();
}

void StyleClock::slotAboutToShowAlarmMenu()
{
    alarmMenu->clear();
    alarmTimeList.clear();
    struct NumInt { 
        int interval;
        int num;
    };
    NumInt intervals[] = {{5*60, 3}, {15*60, 2}, {30*60, 2}, {60*60, 1}};
    const int numIntervals = 4;
    
    alarmMenu->insertTitle(0, i18n("Alarm"));
    alarmMenu->insertItem(i18n("Custom time..."), this, SLOT(slotCustomAlarm()));
    alarmMenu->insertSeparator();
    
    uint t = QDateTime::currentDateTime().toTime_t();
    int index = 0;
    for (int n=0; n<numIntervals; ++n) {
        NumInt& i = intervals[n];
        while (i.num > 0) {
            t = (t/(i.interval)+1) * i.interval;
            QDateTime dt; 
            dt.setTime_t(t);
            alarmTimeList.push_back(dt);
            alarmMenu->insertItem(dt.time().toString(), alarmBase+index);
            --i.num;
            ++index;
        }
    }
    
}

QString StyleClock::secsToString(int t)
{
    QString s;
    if (t%60 != 0) {
        s = i18n("%1 sec").arg(t%60);
    }
    if ((t%3600)/60 != 0) {
        s = i18n("%1 min ").arg((t%3600)/60)+s;
    }
    if (t >= 3600) {
        s = i18n("%1 h ").arg(t/3600)+s;
    }
    return s;
}

void StyleClock::slotAboutToShowTimerMenu()
{
    timerMenu->clear();
    timerTimeList.clear();
    struct Int { 
        int interval;
    };
    Int intervals[] = {3*60, 5*60, 10*60, 15*60, 30*60, 60*60};
    const int numIntervals = 6;
    
    timerMenu->insertTitle(0, i18n("Countdown"));
    timerMenu->insertItem(i18n("Custom time..."), this, SLOT(slotCustomTimer()));
    timerMenu->insertSeparator();

    QValueList<int> historyList = prefs->lastCountdownHistory();
    std::vector<int> history;
    for (uint n=0; n<historyList.size(); ++n) history.push_back(historyList[n]);
    std::sort(history.begin(), history.end());
        
    uint t;
    int index = 0;
    int historyPos = 0;
    for (int n=0; n<numIntervals; ++n) {
        t = intervals[n].interval;
        while (historyPos < int(history.size()) && history[historyPos] <= int(t)) {
            if (history[historyPos] < int(t)) {
                timerTimeList.push_back(history[historyPos]);
                timerMenu->insertItem(SmallIcon("flag"),                     
                    secsToString(history[historyPos]), timerBase+index);
                ++index;
            }
            ++historyPos;
        }
        timerTimeList.push_back(t);
        timerMenu->insertItem(secsToString(t), timerBase+index);
        ++index;
    }
}

void StyleClock::slotAboutToShowContextMenu()
{
    QString txt = i18n("Cancel alarm");
    if (alarmTime.isValid()) {
        txt += QString(" (at %1)").arg(alarmTime.time().toString());
    }
    contextMenu->changeTitle(0, SmallIcon("clock"), 
        i18n("Clock - %1").arg(QTime::currentTime().toString()));
    contextMenu->changeItem(cancelAlarmItemIndex, txt);
    contextMenu->setItemEnabled(cancelAlarmItemIndex, alarmTime.isValid());
}


void StyleClock::slotTimerMenuClicked(int n)
{
    int timerSelection = n-timerBase;
    if (timerSelection >= 0 && timerSelection < int(timerTimeList.size())) {
        showStartTimerDialog(timerTimeList[timerSelection]);    
    }
}

void StyleClock::slotAlarmMenuClicked(int n)
{
    int alarmSelection = n-alarmBase;
    if (alarmSelection >= 0 && alarmSelection < int(alarmTimeList.size())) {
        showStartAlarmDialog(alarmTimeList[alarmSelection]);    
    }
}

void StyleClock::slotCustomAlarm() 
{
    showStartAlarmDialog(QDateTime::currentDateTime());
}

void StyleClock::slotCustomTimer()
{
    showStartTimerDialog(prefs->lastCountdownSecs());
}

void StyleClock::slotCancelAlarm()
{
    setAlarmTime(QDateTime());
}

void StyleClock::showStartAlarmDialog(QDateTime initialTime)
{
    KDialogBase dlg(this, "alarmdlg", true, i18n("alarm time"), 
        KDialogBase::Ok | KDialogBase::Cancel);
    AlarmDialog alarmDlg(&dlg);
    dlg.setMainWidget(&alarmDlg);
    alarmDlg.timeWidget->setTime(initialTime.time());
    if (dlg.exec() == QDialog::Accepted) {
        QDateTime t = QDateTime(QDate::currentDate(), alarmDlg.timeWidget->time());
        if (t <= QDateTime::currentDateTime()) {
                t = t.addDays(1);
        }
        setAlarmTime(t);
    }    
}

void StyleClock::showStartTimerDialog(int seconds)
{
    KDialogBase dlg(this, "alarmdlg", true, i18n("alarm time"), 
        KDialogBase::Ok | KDialogBase::Cancel);
    TimerDialog timerDlg(&dlg);
    dlg.setMainWidget(&timerDlg);
    timerDlg.timeWidget->setTime(QTime().addSecs(seconds));
    if (dlg.exec() == QDialog::Accepted) {
        int t = QTime().secsTo(timerDlg.timeWidget->time());
        prefs->setLastCountdownSecs(t);
        QValueList<int> countdownSecList = prefs->lastCountdownHistory();
        if (countdownSecList.contains(t) == 0) countdownSecList.append(t);
        while (countdownSecList.size() > 3) {
            countdownSecList.pop_front();
        }
        prefs->setLastCountdownHistory(countdownSecList);
        prefs->writeConfig();
        setAlarmTime(QDateTime::currentDateTime().addSecs(t));
    }    
}

void StyleClock::setAlarmTime(QDateTime t, bool bAsk)
{
    bool canSetAlarm = true;
    if (alarmTime.isValid() && bAsk==true) {
        if (t.isValid()) {
            if (KMessageBox::warningContinueCancel(this, 
                i18n("You have set an alarm already. It will be cancelled if you continue."),
                i18n("Warning")) != KMessageBox::Continue) 
            {
                canSetAlarm = false;    
            }
        }
        else {
            if (KMessageBox::questionYesNo(this, 
                i18n("Do you really want to cancel the alarm?"),
                i18n("Alarm")) != KMessageBox::Yes)
            {
                canSetAlarm = false;    
            }
        }
    }
    if (canSetAlarm) {
        alarmTime = QDateTime::currentDateTime();
        alarmTime = t;
        prefs->setCurrentAlarmTime(alarmTime);
        prefs->writeConfig();
        int timeout = 1000*(QDateTime::currentDateTime().secsTo(alarmTime)+1);
        kdDebug() << "Set alarm time to " << timeout << endl;
        QTimer::singleShot(timeout, this, SLOT(slotAlarm()));
    }
    mainView->updateClock();
}

void StyleClock::mousePressEvent(QMouseEvent *e)
{
    if (e->button() == Qt::RightButton) {
        contextMenu->popup(mapToGlobal(e->pos()));    
    }
    else if (e->button() == Qt::LeftButton) {
        toggleCalendar();
    }
}

void StyleClock::slotAlarm()
{
    kdDebug() << "slotAlarm" << endl;
    if (alarmTime.isValid() == false) return;
    setAlarmTime(QDateTime(), false);
    if (alarmDlg == NULL) {
        alarmDlg = new AlarmDlg(this);
    }
    kdDebug() << "show Alarm" << endl;
    alarmDlg->alarm();
}

void StyleClock::slotClockSizeSuggestionChanged()
{
    kdDebug() << "StyleClock::slotClockSizeSuggestionChanged()" << endl;
    emit updateLayout();
}

// Borrowed from KDE's native clock applet
void StyleClock::toggleCalendar()
{
    if (datePicker /*&& !_disableCalendar*/) {
        // calls slotCalendarDeleted which does the cleanup for us
        datePicker->close();
        return;
    }
    if (datePicker /*|| _disableCalendar*/)
        return;
    datePicker = new DatePicker(this, QDate::currentDate(), prefs);
    connect(datePicker, SIGNAL(destroyed()), this, SLOT(slotDatePickerDeleted()));

    // Add 28 px. to width poperly as said in API
    int w = datePicker->sizeHint().width() + 28;
    int h = datePicker->sizeHint().height();

    QPoint c = mapToGlobal(QPoint(0,0));

    // some extra spacing is included if aligned on a desktop edge
    switch (position()) {
        case KPanelApplet::pLeft:
            c.setX(c.x()+width()+2);
            break;
      case KPanelApplet::pRight:
            c.setX(c.x()-w-2);
            break;
        case KPanelApplet::pTop:
            c.setY(c.y()+height()+2);
            break;
        case KPanelApplet::pBottom:
            c.setY(c.y()-h-2);
            break;
    }

    // make calendar fully visible
    QRect deskR = KGlobalSettings::desktopGeometry(c);

    if (c.y()+h > deskR.bottom())
        c.setY(deskR.bottom()-h-1);
    if (c.x()+w > deskR.right())
        c.setX(deskR.right()-w-1);

    datePicker->move(c);
    datePicker->show();
}

void StyleClock::debugMode(bool b)
{
    mainView->setDebugMode(b);
}


void StyleClock::slotDatePickerDeleted()
{
    datePicker = NULL;
}

QDateTime StyleClock::getAlarmTime()
{
    return alarmTime;
}

extern "C"
{
    KPanelApplet* init( QWidget *parent, const QString configFile)
    {
        KGlobal::locale()->insertCatalogue("styleclock");
        return new StyleClock(configFile, KPanelApplet::Normal,
                             KPanelApplet::About | KPanelApplet::Help | KPanelApplet::Preferences,
                             parent, "styleclock");
    }
}
