package org.kohsuke.stapler.compression;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.MessageFormat;
import java.util.Date;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.GZIPOutputStream;

import static org.kohsuke.stapler.Stapler.*;

/**
 * Pimps up {@link HttpServletResponse} so that it understands "Content-Encoding: gzip" and compress the response.
 * 
 * <p>
 * When exceptions are processed within web applications, different unrelated parts of the webapp can end up calling
 * {@link HttpServletResponse#getOutputStream()}. This fundamentally doesn't work with the notion that the application
 * needs to process "Content-Encoding:gzip" on its own. Such app has to maintain {@link GZIPOutputStream} on its own,
 * since {@link HttpServletResponse} doesn't know that its output is written through a compressed stream.
 *
 * <p>
 * Another place this break-down can be seen is when a servlet throws an exception that the container handles.
 * It tries to render an error page, but of course it wouldn't know that "Content-Encoding:gzip" is set, so it
 * will fail to write in the correct format.
 * 
 * <p>
 * The only way to properly fix this is to make {@link HttpServletResponse} smart enough that it returns
 * the compression-transparent stream from {@link HttpServletResponse#getOutputStream()} (and it would also
 * have to process the exception thrown from the app.) This filter does exactly that.
 *
 * @author Kohsuke Kawaguchi
 */
public class CompressionFilter implements Filter {
    public void init(FilterConfig filterConfig) throws ServletException {
    }

    public void doFilter(ServletRequest _req, ServletResponse _rsp, FilterChain filterChain) throws IOException, ServletException {
        _req.setAttribute(CompressionFilter.class.getName(),true);
        CompressionServletResponse rsp = new CompressionServletResponse(((HttpServletResponse) _rsp));
        try {
            filterChain.doFilter(_req, rsp);
        } catch (IOException e) {
            if (DISABLED)   throw e;
            reportException(e,rsp);
        } catch (ServletException e) {
            if (DISABLED)   throw e;
            reportException(e,rsp);
        }
    }

    private void reportException(Exception e, HttpServletResponse rsp) throws IOException {
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        e.printStackTrace(pw);
        pw.close();

        rsp.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        rsp.setContentType("text/html");
        PrintWriter w = rsp.getWriter();
        w.print(MessageFormat.format("<html><head><title>Error {0}</title></head>\n" +
                "<body bgcolor=#fff><h1>Status Code: {0}</h1>Exception: {1}<br>Stacktrace: <pre>{2}</pre><br><hr>\n" +
                "<i>Generated by Stapler at {3}</i></body></html>",
                HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
                escape(e.getMessage()),
                escape(sw.toString()),
                new Date().toString()
        ));
        w.close();
        
        LOGGER.log(Level.WARNING, "Untrapped servlet exception", e);
    }

    public void destroy() {
    }

    /**
     * Is this request already wrapped into {@link CompressionFilter}?
     */
    public static boolean has(ServletRequest req) {
        return req.getAttribute(CompressionFilter.class.getName())!=null;
    }
    
    public static boolean DISABLED = Boolean.getBoolean(CompressionFilter.class.getName()+".disabled");

    private static final Logger LOGGER = Logger.getLogger("UntrappedServletException");
}
