/* iparray.c - sparse array for holding IPv4 addresses
**
** I first used a sparse array for mapping RGB triplets to colormap values,
** when quantizing a 24 bit image down to 8 bits.  The array pretends to be
** a full char[256][256][256] 3-D array, but behind the scenes it only
** allocated the parts of the array that actually got used.  The full
** array would have taken up 16 MB, which at the time was way too much.
** These days 16 MB would be considered only slightly large, so that
** application for the data structure is obsolete.
**
** To represent IPv4 addresses, however, requires a 4-D array,
** char[256][256][256][256].  The full version of this would require
** 4 GB and that is still way too much, so a sparse array is still
** useful here.
**
** Also, it is fairly easy to add subnetting to the data structure,
** so that you can add class-C (/24), class-B (/16), or even entire
** class-A (/8) networks to the array without having to allocate
** space for all the individual addresses.  Plus I added code to handle
** net widths that are not on 8-bit boundaries, by doing recursive
** calls to the next lower 8-bit case.  And also some code to handle
** /0, the entire internet, because why not.
**
**
** Copyright  2004 by Jef Poskanzer <jef@mail.acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
**
** For commentary on this license please see http://www.acme.com/license.html
*/

#include <stdlib.h>
#include <string.h>

#include "iparray.h"


typedef unsigned char**** real_iparray;


/* Clever trick to use pointers to sometimes represent pointers and
** sometimes represent small integers.  What we do is allocate a
** hunk of memory with as many bytes as the largest integer we want
** to store, and then never use that memory.  Instead, any pointers
** into that chunk are the integers, offset by the initial address
** of the chunk.
*/
static char* reserved = (char*) 0;
#define VAL_LIMIT 4096
#define IS_VAL( x ) ( (char*) (x) >= reserved && (char*) (x) < reserved + VAL_LIMIT )
#define IS_PTR( x ) ( ! IS_VAL( x ) )
#define IS_EMPTY( x ) ( (x) == 0 )
#define MAKE_VAL( x ) ( reserved + ( (x) < VAL_LIMIT ? (x) : VAL_LIMIT - 1 ) )
#define GET_VAL( x ) ( (char*) (x) - reserved )


iparray
iparray_new( void )
    {
    real_iparray ripa;
    int o;

    if ( reserved == (char*) 0 )
	{
	reserved = (char*) malloc( VAL_LIMIT );
	if ( reserved == (char*) 0 )
	    return (real_iparray) 0;
	}

    ripa = (real_iparray) malloc( 256 * sizeof(unsigned char***) );
    if ( ripa == (real_iparray) 0 )
	return (real_iparray) 0;
    for ( o = 0; o < 256; ++o )
	ripa[o] = (unsigned char***) 0;
    return ripa;
    }


void
iparray_clear( iparray ipa )
    {
    real_iparray ripa = (real_iparray) ipa;
    int a, b, c;
    unsigned char*** ta;
    unsigned char** tb;
    unsigned char* tc;

    for ( a = 0; a < 256; ++a )
	{
	ta = ripa[a];
	ripa[a] = (unsigned char***) 0;
	if ( IS_EMPTY( ta ) )
	    continue;
	if ( IS_VAL( ta ) )
	    continue;
	for ( b = 0; b < 256; ++b )
	    {
	    tb = ta[b];
	    if ( IS_EMPTY( tb ) )
		continue;
	    if ( IS_VAL( tb ) )
		continue;
	    for ( c = 0; c < 256; ++c )
		{
		tc = tb[c];
		if ( IS_EMPTY( tc ) )
		    continue;
		if ( IS_VAL( tc ) )
		    continue;
		free( (void*) tc );
		}
	    free( (void*) tb );
	    }
	free( (void*) ta );
	}
    }


void
iparray_delete( iparray ipa )
    {
    iparray_clear( ipa );
    free( (void*) ipa );
    }


void
iparray_fini( void )
    {
    if ( reserved != (char*) 0 )
	{
	free( (void*) reserved );
	reserved = (char*) 0;
	}
    }


int
iparray_parse_octets( char* str, octets* ip )
    {
    char* endptr;

    ip->a = strtol( str, &endptr, 10 );
    if ( endptr == str || *endptr != '.' )
	return 0;
    str = endptr + 1;
    ip->b = strtol( str, &endptr, 10 );
    if ( endptr == str || *endptr != '.' )
	return 0;
    str = endptr + 1;
    ip->c = strtol( str, &endptr, 10 );
    if ( endptr == str || *endptr != '.' )
	return 0;
    str = endptr + 1;
    ip->d = strtol( str, &endptr, 10 );
    if ( endptr == str )
	return 0;
    if ( *endptr == '/' )
	{
	str = endptr + 1;
	ip->w = strtol( str, &endptr, 10 );
	if ( endptr == str )
	    return 0;
	if ( ip->w > 32 )
	    return 0;
	}
    else
	ip->w = 32;

    return 1;
    }


int
iparray_incr( iparray ipa, const octets ip )
    {
    real_iparray ripa = (real_iparray) ipa;
    octets ip1, ip2;
    unsigned char*** ta;
    unsigned char** tb;
    unsigned char* tc;
    unsigned char td;
    int o, o2, v;

    switch ( ip.w )
	{
	case 8: case 16: case 24: case 32: break;
	default:
	/* We handle non-8-bit network widths (and width 0, the entire
	** internet) via recursive binary decomposition.
	*/
	ip1 = ip2 = ip;
	ip1.w = ip2.w = ip.w + 1;
	switch ( ip.w )
	    {
	    case  0: ip1.a |= 0x80; ip2.a &= 0x7f; break;
	    case  1: ip1.a |= 0x40; ip2.a &= 0xbf; break;
	    case  2: ip1.a |= 0x20; ip2.a &= 0xdf; break;
	    case  3: ip1.a |= 0x10; ip2.a &= 0xef; break;
	    case  4: ip1.a |= 0x08; ip2.a &= 0xf7; break;
	    case  5: ip1.a |= 0x04; ip2.a &= 0xfb; break;
	    case  6: ip1.a |= 0x02; ip2.a &= 0xfd; break;
	    case  7: ip1.a |= 0x01; ip2.a &= 0xfe; break;
	    case  9: ip1.b |= 0x40; ip2.b &= 0xbf; break;
	    case 10: ip1.b |= 0x20; ip2.b &= 0xdf; break;
	    case 11: ip1.b |= 0x10; ip2.b &= 0xef; break;
	    case 12: ip1.b |= 0x08; ip2.b &= 0xf7; break;
	    case 13: ip1.b |= 0x04; ip2.b &= 0xfb; break;
	    case 14: ip1.b |= 0x02; ip2.b &= 0xfd; break;
	    case 15: ip1.b |= 0x01; ip2.b &= 0xfe; break;
	    case 17: ip1.c |= 0x40; ip2.c &= 0xbf; break;
	    case 18: ip1.c |= 0x20; ip2.c &= 0xdf; break;
	    case 19: ip1.c |= 0x10; ip2.c &= 0xef; break;
	    case 20: ip1.c |= 0x08; ip2.c &= 0xf7; break;
	    case 21: ip1.c |= 0x04; ip2.c &= 0xfb; break;
	    case 22: ip1.c |= 0x02; ip2.c &= 0xfd; break;
	    case 23: ip1.c |= 0x01; ip2.c &= 0xfe; break;
	    case 25: ip1.d |= 0x40; ip2.d &= 0xbf; break;
	    case 26: ip1.d |= 0x20; ip2.d &= 0xdf; break;
	    case 27: ip1.d |= 0x10; ip2.d &= 0xef; break;
	    case 28: ip1.d |= 0x08; ip2.d &= 0xf7; break;
	    case 29: ip1.d |= 0x04; ip2.d &= 0xfb; break;
	    case 30: ip1.d |= 0x02; ip2.d &= 0xfd; break;
	    case 31: ip1.d |= 0x01; ip2.d &= 0xfe; break;
	    }
	return iparray_incr( ipa, ip1 ) && iparray_incr( ipa, ip2 );
	}

    ta = ripa[ip.a];
    if ( ip.w == 8 )
	{
	/* The IP address is a class-A net. */
	if ( IS_EMPTY( ta ) )
	    {
	    /* This array position is empty. */
	    ripa[ip.a] = (unsigned char***) MAKE_VAL( 1 );
	    return 1;
	    }
	else if ( IS_VAL( ta ) )
	    {
	    /* This array position already has a class-A net entry. */
	    ripa[ip.a] = (unsigned char***) MAKE_VAL( GET_VAL( ta ) + 1 );
	    return 1;
	    }
	else
	    {
	    /* We want to increment the whole class-A net, but the
	    ** array entry has already been expanded out into
	    ** lower-level addresses.  Sum up everything from here
	    ** down, free it, and make a net entry.
	    */
	    v = iparray_get( ipa, ip );
	    ripa[ip.a] = (unsigned char***) MAKE_VAL( v + 1 );
	    for ( o = 0; o < 256; ++o )
		{
		tb = ta[o];
		if ( IS_EMPTY( tb ) )
		    continue;
		if ( IS_VAL( tb ) )
		    continue;
		for ( o2 = 0; o2 < 256; ++o2 )
		    {
		    tc = tb[o2];
		    if ( IS_EMPTY( tc ) )
			continue;
		    if ( IS_VAL( tc ) )
			continue;
		    free( (void*) tc );
		    }
		free( (void*) tb );
		}
	    free( (void*) ta );
	    return 1;
	    }
	}
    if ( IS_EMPTY( ta ) )
	{
	/* Allocate a new entry. */
	ta = (unsigned char***) malloc( sizeof(unsigned char**) * 256 );
	if ( ta == (unsigned char***) 0 )
	    return 0;
	for ( o = 0; o < 256; ++o )
	    ta[o] = (unsigned char**) 0;
	ripa[ip.a] = ta;
	}
    if ( IS_VAL( ta ) )
	/* The array entry is a class-A net, so just increment it. */
	ripa[ip.a] = (unsigned char***) MAKE_VAL( GET_VAL( ta ) + 1 );
    else
	{
	tb = ta[ip.b];
	if ( ip.w == 16 )
	    {
	    /* The IP address is a class-B net. */
	    if ( IS_EMPTY( tb ) )
		{
		/* This array position is empty. */
		ta[ip.b] = (unsigned char**) MAKE_VAL( 1 );
		return 1;
		}
	    else if ( IS_VAL( tb ) )
		{
		/* This array position already has a class-B net entry. */
		ta[ip.b] = (unsigned char**) MAKE_VAL( GET_VAL( tb ) + 1 );
		return 1;
		}
	    else
		{
		/* We want to increment the whole class-B net, but the
		** array entry has already been expanded out into
		** lower-level addresses.  Sum up everything from here
		** down, free it, and make a net entry.
		*/
		v = iparray_get( ipa, ip );
		ta[ip.b] = (unsigned char**) MAKE_VAL( v + 1 );
		for ( o = 0; o < 256; ++o )
		    {
		    tc = tb[o];
		    if ( IS_EMPTY( tc ) )
			continue;
		    if ( IS_VAL( tc ) )
			continue;
		    free( (void*) tc );
		    }
		free( (void*) tb );
		return 1;
		}
	    }
	if ( IS_EMPTY( tb ) )
	    {
	    /* Allocate a new entry. */
	    tb = (unsigned char**) malloc( sizeof(unsigned char*) * 256 );
	    if ( tb == (unsigned char**) 0 )
		return 0;
	    for ( o = 0; o < 256; ++o )
		tb[o] = (unsigned char*) 0;
	    ta[ip.b] = tb;
	    }
	if ( IS_VAL( tb ) )
	    /* The array entry is a class-B net, so just increment it. */
	    ta[ip.b] = (unsigned char**) MAKE_VAL( GET_VAL( tb ) + 1 );
	else
	    {
	    tc = tb[ip.c];
	    if ( ip.w == 24 )
		{
		/* The IP address is a class-C net. */
		if ( IS_EMPTY( tc ) )
		    {
		    /* This array position is empty. */
		    tb[ip.c] = (unsigned char*) MAKE_VAL( 1 );
		    return 1;
		    }
		else if ( IS_VAL( tc ) )
		    {
		    /* This array position already has a class-C net entry. */
		    tb[ip.c] = (unsigned char*) MAKE_VAL( GET_VAL( tc ) + 1 );
		    return 1;
		    }
		else
		    {
		    /* We want to increment the whole class-C net, but the
		    ** array entry has already been expanded out into
		    ** lower-level addresses.  Sum up everything from here
		    ** down, free it, and make a net entry.
		    */
		    v = iparray_get( ipa, ip );
		    tb[ip.c] = (unsigned char*) MAKE_VAL( v + 1 );
		    free( (void*) tc );
		    return 1;
		    }
		}
	    if ( IS_EMPTY( tc ) )
		{
		/* Allocate a new entry. */
		tc = (unsigned char*) malloc( sizeof(unsigned char) * 256 );
		if ( tc == (unsigned char*) 0 )
		    return 0;
		for ( o = 0; o < 256; ++o )
		    tc[o] = (unsigned char) 0;
		tb[ip.c] = tc;
		}
	    if ( IS_VAL( tc ) )
		/* The array entry is a class-C net, so just increment it. */
		tb[ip.c] = (unsigned char*) MAKE_VAL( GET_VAL( tc ) + 1 );
	    else
		{
		td = tc[ip.d];
		/* Don't overflow. */
		if ( td < 255 )
		    tc[ip.d] = td + 1;
		}
	    }
	}
    return 1;
    }


int
iparray_get( const iparray ipa, const octets ip )
    {
    real_iparray ripa = (real_iparray) ipa;
    octets ip1, ip2;
    unsigned char*** ta;
    unsigned char** tb;
    unsigned char* tc;
    int oa, ob, oc, od;
    int r;

    switch ( ip.w )
	{
	case 0: case 8: case 16: case 24: case 32: break;
	default:
	/* We handle non-8-bit network widths via recursive binary
	** decomposition.
	*/
	ip1 = ip2 = ip;
	ip1.w = ip2.w = ip.w + 1;
	switch ( ip.w )
	    {
	    case  1: ip1.a |= 0x40; ip2.a &= 0xbf; break;
	    case  2: ip1.a |= 0x20; ip2.a &= 0xdf; break;
	    case  3: ip1.a |= 0x10; ip2.a &= 0xef; break;
	    case  4: ip1.a |= 0x08; ip2.a &= 0xf7; break;
	    case  5: ip1.a |= 0x04; ip2.a &= 0xfb; break;
	    case  6: ip1.a |= 0x02; ip2.a &= 0xfd; break;
	    case  7: ip1.a |= 0x01; ip2.a &= 0xfe; break;
	    case  9: ip1.b |= 0x40; ip2.b &= 0xbf; break;
	    case 10: ip1.b |= 0x20; ip2.b &= 0xdf; break;
	    case 11: ip1.b |= 0x10; ip2.b &= 0xef; break;
	    case 12: ip1.b |= 0x08; ip2.b &= 0xf7; break;
	    case 13: ip1.b |= 0x04; ip2.b &= 0xfb; break;
	    case 14: ip1.b |= 0x02; ip2.b &= 0xfd; break;
	    case 15: ip1.b |= 0x01; ip2.b &= 0xfe; break;
	    case 17: ip1.c |= 0x40; ip2.c &= 0xbf; break;
	    case 18: ip1.c |= 0x20; ip2.c &= 0xdf; break;
	    case 19: ip1.c |= 0x10; ip2.c &= 0xef; break;
	    case 20: ip1.c |= 0x08; ip2.c &= 0xf7; break;
	    case 21: ip1.c |= 0x04; ip2.c &= 0xfb; break;
	    case 22: ip1.c |= 0x02; ip2.c &= 0xfd; break;
	    case 23: ip1.c |= 0x01; ip2.c &= 0xfe; break;
	    case 25: ip1.d |= 0x40; ip2.d &= 0xbf; break;
	    case 26: ip1.d |= 0x20; ip2.d &= 0xdf; break;
	    case 27: ip1.d |= 0x10; ip2.d &= 0xef; break;
	    case 28: ip1.d |= 0x08; ip2.d &= 0xf7; break;
	    case 29: ip1.d |= 0x04; ip2.d &= 0xfb; break;
	    case 30: ip1.d |= 0x02; ip2.d &= 0xfd; break;
	    case 31: ip1.d |= 0x01; ip2.d &= 0xfe; break;
	    }
	return iparray_get( ipa, ip1 ) + iparray_get( ipa, ip2 );
	}

    if ( ip.w == 0 )
	{
	/* Sum up /0 (i.e. the entire internet - not typically a reasonable
	** thing to do).
	*/
	r = 0;
	for ( oa = 0; oa < 256; ++oa )
	    {
	    ta = ripa[oa];
	    if ( IS_EMPTY( ta ) )
		continue;
	    else if ( IS_VAL( ta ) )
		r += GET_VAL( ta );
	    else
		{
		for ( ob = 0; ob < 256; ++ob )
		    {
		    tb = ta[ob];
		    if ( IS_EMPTY( tb ) )
			continue;
		    else if ( IS_VAL( tb ) )
			r += GET_VAL( tb );
		    else
			{
			for ( oc = 0; oc < 256; ++oc )
			    {
			    tc = tb[oc];
			    if ( IS_EMPTY( tc ) )
				continue;
			    else if ( IS_VAL( tc ) )
				r += GET_VAL( tc );
			    else
				for ( od = 0; od < 256; ++od )
				    r += tc[od];
			    }
			}
		    }
		}
	    }
	return r;
	}
    ta = ripa[ip.a];
    if ( IS_EMPTY( ta ) )
	return 0;
    if ( IS_VAL( ta ) )
	return GET_VAL( ta );
    if ( ip.w == 8 )
	{
	/* Sum up /8. */
	r = 0;
	for ( ob = 0; ob < 256; ++ob )
	    {
	    tb = ta[ob];
	    if ( IS_EMPTY( tb ) )
		continue;
	    else if ( IS_VAL( tb ) )
		r += GET_VAL( tb );
	    else
		{
		for ( oc = 0; oc < 256; ++oc )
		    {
		    tc = tb[oc];
		    if ( IS_EMPTY( tc ) )
			continue;
		    else if ( IS_VAL( tc ) )
			r += GET_VAL( tc );
		    else
			for ( od = 0; od < 256; ++od )
			    r += tc[od];
		    }
		}
	    }
	return r;
	}
    tb = ta[ip.b];
    if ( IS_EMPTY( tb ) )
	return 0;
    if ( IS_VAL( tb ) )
	return GET_VAL( tb );
    if ( ip.w == 16 )
	{
	/* Sum up /16. */
	r = 0;
	for ( oc = 0; oc < 256; ++oc )
	    {
	    tc = tb[oc];
	    if ( IS_EMPTY( tc ) )
		continue;
	    else if ( IS_VAL( tc ))
		r += GET_VAL( tc );
	    else
		for ( od = 0; od < 256; ++od )
		    r += tc[od];
	    }
	return r;
	}
    tc = tb[ip.c];
    if ( IS_EMPTY( tc ) )
	return 0;
    if ( IS_VAL( tc ) )
	return GET_VAL( tc );
    if ( ip.w == 24 )
	{
	/* Sum up /24. */
	r = 0;
	for ( od = 0; od < 256; ++od )
	    r += tc[od];
	return r;
	}
    return (int) tc[ip.d];
    }
