/*
/ TableTree.cpp
/ tree control to show tables, columns, indices and triggers
/
/ version 1.2, 2008 October 9
/
/ Author: Sandro Furieri a-furieri@lqt.it
/
/ Copyright (C) 2008  Alessandro Furieri
/
/    This program is free software: you can redistribute it and/or modify
/    it under the terms of the GNU General Public License as published by
/    the Free Software Foundation, either version 3 of the License, or
/    (at your option) any later version.
/
/    This program is distributed in the hope that it will be useful,
/    but WITHOUT ANY WARRANTY; without even the implied warranty of
/    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/    GNU General Public License for more details.
/
/    You should have received a copy of the GNU General Public License
/    along with this program.  If not, see <http://www.gnu.org/licenses/>.
/
*/

#include "Classdef.h"

#include "wx/filename.h"
#include "wx/imaglist.h"

//
// ICONs in XPM format [universally portable]
//
#include "icons/db.xpm"
#include "icons/table.xpm"
#include "icons/geotable.xpm"
#include "icons/vtable.xpm"
#include "icons/view.xpm"
#include "icons/pkey.xpm"
#include "icons/column.xpm"
#include "icons/index.xpm"
#include "icons/trigger.xpm"
#include "icons/geometry.xpm"
#include "icons/spatialidx.xpm"
#include "icons/mbrcache.xpm"
#include "icons/kill_spindex.xpm"
#include "icons/dumpshp.xpm"

MyTableTree::MyTableTree(MyFrame * parent, wxWindowID id):wxTreeCtrl(parent, id)
{
//
// constructor: TREE control to show DB objects
//
  MainFrame = parent;
  Root = AddRoot(wxT("no current DB"));
// setting up icons 
  Images = new wxImageList(16, 16, true);
  wxIcon icons[12];
  icons[0] = wxIcon(db_xpm);
  icons[1] = wxIcon(table_xpm);
  icons[2] = wxIcon(pkey_xpm);
  icons[3] = wxIcon(column_xpm);
  icons[4] = wxIcon(index_xpm);
  icons[5] = wxIcon(trigger_xpm);
  icons[6] = wxIcon(geometry_xpm);
  icons[7] = wxIcon(spatialidx_xpm);
  icons[8] = wxIcon(vtable_xpm);
  icons[9] = wxIcon(view_xpm);
  icons[10] = wxIcon(geotable_xpm);
  icons[11] = wxIcon(mbrcache_xpm);
  Images->Add(icons[0]);
  Images->Add(icons[1]);
  Images->Add(icons[2]);
  Images->Add(icons[3]);
  Images->Add(icons[4]);
  Images->Add(icons[5]);
  Images->Add(icons[6]);
  Images->Add(icons[7]);
  Images->Add(icons[8]);
  Images->Add(icons[9]);
  Images->Add(icons[10]);
  Images->Add(icons[11]);
  SetImageList(Images);
  SetItemImage(Root, 0);
// setting up event handlers 
  Connect(wxID_ANY, wxEVT_COMMAND_TREE_SEL_CHANGED,
          (wxObjectEventFunction) & MyTableTree::OnSelChanged);
  Connect(wxID_ANY, wxEVT_COMMAND_TREE_ITEM_RIGHT_CLICK,
          (wxObjectEventFunction) & MyTableTree::OnRightClick);
  Connect(Tree_NewTable, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdNewTable);
  Connect(Tree_NewView, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdNewView);
  Connect(Tree_NewIndex, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdNewIndex);
  Connect(Tree_NewTrigger, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdNewTrigger);
  Connect(Tree_NewColumn, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdNewColumn);
  Connect(Tree_Show, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdShow);
  Connect(Tree_Drop, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdDrop);
  Connect(Tree_Rename, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdRename);
  Connect(Tree_Select, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdSelect);
  Connect(Tree_Refresh, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdRefresh);
  Connect(Tree_Recover, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdRecover);
  Connect(Tree_ShowSql, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdShowSql);
  Connect(Tree_SpatialIndex, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdSpatialIndex);
  Connect(Tree_MbrCache, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdMbrCache);
  Connect(Tree_RebuildTriggers, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdRebuildTriggers);
  Connect(Tree_CheckGeometry, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdCheckGeometry);
  Connect(Tree_SetSrid, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdSetSrid);
  Connect(Tree_DumpShp, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdDumpShp);
  Connect(Tree_DumpTxtTab, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdDumpTxtTab);
  Connect(Tree_DumpCsv, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdDumpCsv);
  Connect(Tree_DumpHtml, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdDumpHtml);
  Connect(Tree_Edit, wxEVT_COMMAND_MENU_SELECTED,
          (wxObjectEventFunction) & MyTableTree::OnCmdEdit);
}

void MyTableTree::AddTable(wxString & tableName, bool virtualTable)
{
//
// appends a table to the TREE list
  MyTableInfo list;
  MyColumnInfo *col;
  MyIndexInfo *idx;
  MyTriggerInfo *trgr;
  wxTreeItemId item;
  wxTreeItemId item2;
  wxString columnInfo;
  wxString indexInfo;
  wxString triggerInfo;
  int icon = 1;
  if (virtualTable == true)
    icon = 8;
  item = AppendItem(Root, tableName, icon);
  if (virtualTable == true)
    SetItemData(item, (wxTreeItemData *) (new MyObject(MY_VTABLE, tableName)));
  else
    SetItemData(item, (wxTreeItemData *) (new MyObject(MY_TABLE, tableName)));
  MainFrame->GetTableColumns(tableName, &list);
  MainFrame->GetTableIndices(tableName, &list);
  MainFrame->GetTableTriggers(tableName, &list);
  col = list.GetFirstColumn();
  while (col)
    {
      int icon;
      if (col->IsPrimaryKey() == true)
        icon = 2;
      else
        {
          if (col->IsGeometry() == true)
            {
              if (virtualTable == false)
                SetItemImage(item, 10);
              if (col->IsGeometryIndex() == true)
                icon = 7;
              else if (col->IsMbrCache() == true)
                icon = 11;
              else
                icon = 6;
          } else
            icon = 3;
        }
      columnInfo = col->GetName();
      item2 = AppendItem(item, columnInfo, icon);
      if (col->IsGeometry() == true)
        {
          if (col->IsGeometryIndex() == true)
            SetItemData(item2,
                        (wxTreeItemData *) (new
                                            MyObject(MY_GEOMETRY_INDEX,
                                                     tableName,
                                                     col->GetName())));
          else if (col->IsMbrCache() == true)
            SetItemData(item2,
                        (wxTreeItemData *) (new
                                            MyObject(MY_GEOMETRY_CACHED,
                                                     tableName,
                                                     col->GetName())));
          else
            SetItemData(item2,
                        (wxTreeItemData *) (new
                                            MyObject(MY_GEOMETRY, tableName,
                                                     col->GetName())));
      } else
        SetItemData(item2,
                    (wxTreeItemData *) (new
                                        MyObject(MY_COLUMN, tableName,
                                                 col->GetName())));
      col = col->GetNext();
    }
  idx = list.GetFirstIndex();
  while (idx)
    {
      indexInfo = idx->GetName();
      item2 = AppendItem(item, indexInfo, 4);
      SetItemData(item2,
                  (wxTreeItemData *) (new MyObject(MY_INDEX, idx->GetName())));
      idx = idx->GetNext();
    }
  trgr = list.GetFirstTrigger();
  while (trgr)
    {
      triggerInfo = trgr->GetName();
      item2 = AppendItem(item, triggerInfo, 5);
      SetItemData(item2,
                  (wxTreeItemData *) (new
                                      MyObject(MY_TRIGGER, trgr->GetName())));
      trgr = trgr->GetNext();
    }
}

void MyTableTree::AddView(wxString & viewName)
{
//
// appends a view to the TREE list
  MyViewInfo list;
  MyColumnInfo *col;
  wxTreeItemId item;
  wxTreeItemId item2;
  wxString columnInfo;
  item = AppendItem(Root, viewName, 9);
  SetItemData(item, (wxTreeItemData *) (new MyObject(MY_VIEW, viewName)));
  MainFrame->GetViewColumns(viewName, &list);
  col = list.GetFirst();
  while (col)
    {
      columnInfo = col->GetName();
      item2 = AppendItem(item, columnInfo, 3);
      SetItemData(item2,
                  (wxTreeItemData *) (new
                                      MyObject(MY_COLUMN, viewName,
                                               col->GetName())));
      col = col->GetNext();
    }
}

void MyTableTree::OnSelChanged(wxTreeEvent & event)
{
//
// selecting some node [mouse action]
//
  wxTreeItemId item = event.GetItem();
  if (item == Root)
    return;
  MyObject *obj = (MyObject *) GetItemData(item);
  if (obj == NULL)
    return;
  CurrentItem = item;
}

void MyTableTree::OnRightClick(wxTreeEvent & event)
{
//
// right click on some node [mouse action]
//
  wxMenu *menu = new wxMenu();
  wxMenuItem *menuItem;
  wxString title;
  bool table = false;
  bool canEdit = false;
  bool view = false;
  bool column = false;
  bool geometry = false;
  bool geometry_index = false;
  bool geometry_cached = false;
  bool index = false;
  bool trigger = false;
  bool metadata = MainFrame->CheckMetadata();
  if (MainFrame->IsConnected() == false)
    return;
  wxTreeItemId item = event.GetItem();
  SelectItem(item);
  wxPoint pt = event.GetPoint();
  if (item == Root)
    {
      CurrentItem = NULL;
      menuItem = new wxMenuItem(menu, Tree_Refresh, wxT("&Refresh"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem = new wxMenuItem(menu, Tree_NewTable, wxT("Create New &Table"));
      menu->Append(menuItem);
      menuItem = new wxMenuItem(menu, Tree_NewView, wxT("Create New &View"));
      menu->Append(menuItem);
      PopupMenu(menu, pt);
      return;
    }
  MyObject *obj = (MyObject *) GetItemData(item);
  if (obj == NULL)
    {
      CurrentItem = NULL;
      menuItem = new wxMenuItem(menu, Tree_Refresh, wxT("&Refresh"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem = new wxMenuItem(menu, Tree_NewTable, wxT("Create New &Table"));
      menu->Append(menuItem);
      menuItem = new wxMenuItem(menu, Tree_NewView, wxT("Create New &View"));
      menu->Append(menuItem);
      PopupMenu(menu, pt);
      return;
    }
  switch (obj->GetType())
    {
      case MY_VTABLE:
      case MY_TABLE:
        table = true;
        break;
      case MY_VIEW:
        view = true;
        break;
      case MY_COLUMN:
        column = true;
        break;
      case MY_GEOMETRY:
        geometry = true;
        break;
      case MY_GEOMETRY_INDEX:
        geometry_index = true;
        break;
      case MY_GEOMETRY_CACHED:
        geometry_cached = true;
        break;
      case MY_INDEX:
        index = true;
        break;
      case MY_TRIGGER:
        trigger = true;
        break;
    };
  if (obj->GetType() == MY_TABLE)
    canEdit = true;
  CurrentItem = item;
  menuItem = new wxMenuItem(menu, Tree_Refresh, wxT("&Refresh"));
  menu->Append(menuItem);
  if (table == true)
    {
      wxString title = wxT("Table: \"") + obj->GetName() + wxT("\"");
      menu->SetTitle(title);
      menu->AppendSeparator();
      menuItem = new wxMenuItem(menu, Tree_NewTable, wxT("Create New &Table"));
      menu->Append(menuItem);
      menuItem = new wxMenuItem(menu, Tree_NewView, wxT("Create New &View"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      if (canEdit == true)
        {
          menuItem = new wxMenuItem(menu, Tree_Edit, wxT("&Edit table rows"));
          menu->Append(menuItem);
      } else
        {
          menuItem = new wxMenuItem(menu, Tree_Select, wxT("&Query table"));
          menu->Append(menuItem);
        }
      menuItem = new wxMenuItem(menu, Tree_Show, wxT("&Show columns"));
      menu->Append(menuItem);
      menuItem =
        new wxMenuItem(menu, Tree_ShowSql, wxT("&Show CREATE statement"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem = new wxMenuItem(menu, Tree_NewColumn, wxT("Add New &Column"));
      menu->Append(menuItem);
      menuItem = new wxMenuItem(menu, Tree_Rename, wxT("&Rename table"));
      menu->Append(menuItem);
      menuItem = new wxMenuItem(menu, Tree_Drop, wxT("&Drop table"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem = new wxMenuItem(menu, Tree_NewIndex, wxT("Create New &Index"));
      menu->Append(menuItem);
      menuItem =
        new wxMenuItem(menu, Tree_NewTrigger, wxT("Create New &Trigger"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem =
        new wxMenuItem(menu, Tree_DumpTxtTab, wxT("Export as &Txt/Tab"));
      menu->Append(menuItem);
      menuItem = new wxMenuItem(menu, Tree_DumpCsv, wxT("Export as &CSV"));
      menu->Append(menuItem);
      menuItem = new wxMenuItem(menu, Tree_DumpHtml, wxT("Export as &HTML"));
      menu->Append(menuItem);
    }
  if (view == true)
    {
      wxString title = wxT("View: \"") + obj->GetName() + wxT("\"");
      menu->SetTitle(title);
      menu->AppendSeparator();
      menuItem = new wxMenuItem(menu, Tree_NewTable, wxT("Create New &Table"));
      menu->Append(menuItem);
      menuItem = new wxMenuItem(menu, Tree_NewView, wxT("Create New &View"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem = new wxMenuItem(menu, Tree_Select, wxT("&Query view"));
      menu->Append(menuItem);
      menuItem = new wxMenuItem(menu, Tree_Show, wxT("&Show columns"));
      menu->Append(menuItem);
      menuItem =
        new wxMenuItem(menu, Tree_ShowSql, wxT("&Show CREATE statement"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem = new wxMenuItem(menu, Tree_Drop, wxT("&Drop view"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem =
        new wxMenuItem(menu, Tree_DumpTxtTab, wxT("Export as &Txt/Tab"));
      menu->Append(menuItem);
      menuItem = new wxMenuItem(menu, Tree_DumpCsv, wxT("Export as &CSV"));
      menu->Append(menuItem);
      menuItem = new wxMenuItem(menu, Tree_DumpHtml, wxT("Export as &HTML"));
      menu->Append(menuItem);
    }
  if (column == true)
    {
      wxString title =
        wxT("Column: \"") + obj->GetName() + wxT("\".\"") + obj->GetColumn() +
        wxT("\"");
      menu->SetTitle(title);
      menuItem =
        new wxMenuItem(menu, Tree_CheckGeometry, wxT("&Check geometries"));
      menu->Append(menuItem);
      menuItem = new wxMenuItem(menu, Tree_SetSrid, wxT("&Set SRID"));
      menu->Append(menuItem);
      if (metadata == true)
        {
          menu->AppendSeparator();
          menuItem =
            new wxMenuItem(menu, Tree_Recover, wxT("&Recover geometry column"));
          menu->Append(menuItem);
        }
    }
  if (geometry == true)
    {
      wxString title =
        wxT("Column: \"") + obj->GetName() + wxT("\".\"") + obj->GetColumn() +
        wxT("\"");
      menu->SetTitle(title);
      menuItem = new wxMenuItem(menu, Tree_Show, wxT("&Show Spatial Metadata"));
      menu->Append(menuItem);
      menuItem =
        new wxMenuItem(menu, Tree_CheckGeometry, wxT("&Check geometries"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem =
        new wxMenuItem(menu, Tree_SpatialIndex, wxT("&Build Spatial Index"));
      menuItem->SetBitmap(wxBitmap(spatialidx_xpm));
      menu->Append(menuItem);
      menuItem = new wxMenuItem(menu, Tree_MbrCache, wxT("Build &MBR cache"));
      menuItem->SetBitmap(wxBitmap(mbrcache_xpm));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem =
        new wxMenuItem(menu, Tree_RebuildTriggers,
                       wxT("Rebuild Geometry &Triggers"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem =
        new wxMenuItem(menu, Tree_DumpShp, wxT("Export as &Shapefile"));
      menuItem->SetBitmap(wxBitmap(dumpshp_xpm));
      menu->Append(menuItem);
    }
  if (geometry_index == true)
    {
      wxString title =
        wxT("Column: \"") + obj->GetName() + wxT("\".\"") + obj->GetColumn() +
        wxT("\"");
      menu->SetTitle(title);
      menuItem = new wxMenuItem(menu, Tree_Show, wxT("&Show Spatial Metadata"));
      menu->Append(menuItem);
      menuItem =
        new wxMenuItem(menu, Tree_CheckGeometry, wxT("&Check geometries"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem =
        new wxMenuItem(menu, Tree_SpatialIndex, wxT("&Remove Spatial Index"));
      menuItem->SetBitmap(wxBitmap(kill_spindex_xpm));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem =
        new wxMenuItem(menu, Tree_RebuildTriggers,
                       wxT("Rebuild Geometry &Triggers"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem =
        new wxMenuItem(menu, Tree_DumpShp, wxT("Export as &Shapefile"));
      menuItem->SetBitmap(wxBitmap(dumpshp_xpm));
      menu->Append(menuItem);
    }
  if (geometry_cached == true)
    {
      wxString title =
        wxT("Column: \"") + obj->GetName() + wxT("\".\"") + obj->GetColumn() +
        wxT("\"");
      menu->SetTitle(title);
      menuItem = new wxMenuItem(menu, Tree_Show, wxT("&Show Spatial Metadata"));
      menu->Append(menuItem);
      menuItem =
        new wxMenuItem(menu, Tree_CheckGeometry, wxT("&Check geometries"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem = new wxMenuItem(menu, Tree_MbrCache, wxT("&Remove MBR cache"));
      menuItem->SetBitmap(wxBitmap(kill_spindex_xpm));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem =
        new wxMenuItem(menu, Tree_RebuildTriggers,
                       wxT("Rebuild Geometry &Triggers"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem =
        new wxMenuItem(menu, Tree_DumpShp, wxT("Export as &Shapefile"));
      menuItem->SetBitmap(wxBitmap(dumpshp_xpm));
      menu->Append(menuItem);
    }
  if (index == true)
    {
      wxString title = wxT("Index: \"") + obj->GetName() + wxT("\"");
      menu->SetTitle(title);
      menu->AppendSeparator();
      menuItem = new wxMenuItem(menu, Tree_Show, wxT("&Show index"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem = new wxMenuItem(menu, Tree_Drop, wxT("&Drop index"));
      menu->Append(menuItem);
    }
  if (trigger == true)
    {
      wxString title = wxT("Trigger: \"") + obj->GetName() + wxT("\"");
      menu->SetTitle(title);
      menu->AppendSeparator();
      menuItem = new wxMenuItem(menu, Tree_Show, wxT("&Show trigger"));
      menu->Append(menuItem);
      menu->AppendSeparator();
      menuItem = new wxMenuItem(menu, Tree_Drop, wxT("&Drop trigger"));
      menu->Append(menuItem);
    }
  PopupMenu(menu, pt);
}

void MyTableTree::OnCmdNewTable(wxCommandEvent & event)
{
//
// menu event - new table creation required
//
  wxString sql;
  sql = wxT("CREATE TABLE ...table-name... (\n");
  sql += wxT("...column1,\n...column2,\n...columnN)");
  MainFrame->SetSql(sql, false);
}

void MyTableTree::OnCmdNewView(wxCommandEvent & event)
{
//
// menu event - new view creation required
//
  wxString sql;
  sql = wxT("CREATE VIEW ...view-name... AS\n");
  sql += wxT("SELECT ...sql-select-statement...");
  MainFrame->SetSql(sql, false);
}

void MyTableTree::OnCmdNewIndex(wxCommandEvent & event)
{
//
// menu event - new index creation required
//
  wxString sql;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_TABLE)
    {
      sql = wxT("CREATE [ UNIQUE ] INDEX ...index-name...\nON ");
      sql += obj->GetName();
      sql += wxT("\n(\n...column1, column2, columnN...\n)");
      MainFrame->SetSql(sql, false);
    }
}

void MyTableTree::OnCmdNewTrigger(wxCommandEvent & event)
{
//
// menu event - new trigger creation required
//
  wxString sql;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_TABLE)
    {
      sql = wxT("CREATE TRIGGER ...trigger-name...\n[ BEFORE | AFTER ]\n");
      sql += wxT("[ INSERT | UPDATE | DELETE ]\nON ");
      sql += obj->GetName();
      sql += wxT("\n...sql-statement...");
      MainFrame->SetSql(sql, false);
    }
}

void MyTableTree::OnCmdNewColumn(wxCommandEvent & event)
{
//
// menu event - new column creation required
//
  wxString sql;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_TABLE)
    {
      sql = wxT("ALTER TABLE \"");
      sql += obj->GetName();
      sql += wxT("\"\nADD COLUMN ...column-name column-type...");
      MainFrame->SetSql(sql, false);
    }
}

void MyTableTree::OnCmdSelect(wxCommandEvent & event)
{
//
// menu event - examining table rows required
//
  wxString sql;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  sql = wxT("SELECT * FROM \"");
  sql += obj->GetName() + wxT("\"");
  MainFrame->SetSql(sql, true);
}

void MyTableTree::OnCmdShow(wxCommandEvent & event)
{
//
// menu event - examining full infos required
//
  wxString sql;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_TABLE || obj->GetType() == MY_VTABLE
      || obj->GetType() == MY_VIEW)
    {
      sql = wxT("PRAGMA table_info(\"");
      sql += obj->GetName();
      sql += wxT("\")");
    }
  if (obj->GetType() == MY_GEOMETRY || obj->GetType() == MY_GEOMETRY_INDEX
      || obj->GetType() == MY_GEOMETRY_CACHED)
    {
      sql = wxT("SELECT * FROM geom_cols_ref_sys\n");
      sql += wxT("WHERE f_table_name = '");
      sql += obj->GetName();
      sql += wxT("'\nAND f_geometry_column = '");
      sql += obj->GetColumn();
      sql += wxT("'");
    }
  if (obj->GetType() == MY_INDEX)
    {
      sql = wxT("PRAGMA index_info(\"");
      sql += obj->GetName();
      sql += wxT("\")");
    }
  if (obj->GetType() == MY_TRIGGER)
    {
      sql =
        wxT
        ("SELECT sql FROM sqlite_master\nWHERE type = 'trigger' AND name = '");
      sql += obj->GetName();
      sql += wxT("'");
    }
  if (sql.Len() < 1)
    return;
  MainFrame->SetSql(sql, true);
}

void MyTableTree::OnCmdDrop(wxCommandEvent & event)
{
//
// menu event - dropping some object required
//
  wxString sql;
  wxString msg;
  char *errMsg = NULL;
  int ret;
  sqlite3 *sqlite = MainFrame->GetSqlite();
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_TABLE || obj->GetType() == MY_VTABLE)
    {
      bool metadata = MainFrame->CheckMetadata();
      if (metadata == true)
        {
          // SpatialMetadata exists; we must check for geometry_columns and SpatialIndices as well
          char *errMsg = NULL;
          int ret;
          int i;
          char **results;
          int rows;
          int columns;
          sqlite3 *sqlite = MainFrame->GetSqlite();
          sql = wxT("SELECT f_geometry_column, spatial_index_enabled ");
          sql += wxT("FROM geometry_columns WHERE f_table_name = '");
          sql += obj->GetName();
          sql += wxT("'");
          ret =
            sqlite3_get_table(sqlite, sql.ToUTF8(), &results, &rows, &columns,
                              &errMsg);
          if (ret != SQLITE_OK)
            {
              wxMessageBox(wxT("SQLite SQL error: ") +
                           wxString::FromUTF8(errMsg), wxT("spatialite-gui"),
                           wxOK | wxICON_ERROR, this);
              sqlite3_free(errMsg);
              return;
            }
          sql = wxT("BEGIN; ");
          if (rows < 1)
            ;
          else
            {
              for (i = 1; i <= rows; i++)
                {
                  if (results[(i * columns) + 0])
                    {
                      const char *column = results[(i * columns) + 0];
                      wxString geo_column = wxString::FromUTF8(column);
                      if (atoi(results[(i * columns) + 1]))
                        {
                          // dropping the SpatialIndex
                          sql += wxT("DROP TABLE IF EXISTS \"idx_");
                          sql += obj->GetName();
                          sql += wxT("_");
                          sql += geo_column;
                          sql += wxT("\"; ");
                          // dropping the MBR cache
                          sql += wxT("DROP TABLE IF EXISTS \"cache_");
                          sql += obj->GetName();
                          sql += wxT("_");
                          sql += geo_column;
                          sql += wxT("\"; ");
                        }
                      // deleting from geometry_columns
                      sql +=
                        wxT
                        ("DELETE FROM geometry_columns\n    WHERE f_table_name = '");
                      sql += obj->GetName();
                      sql += wxT("'\n        AND f_geometry_column = '");
                      sql += geo_column;
                      sql += wxT("'; ");
                    }
                }
            }
          sqlite3_free_table(results);
          sql += wxT("DROP TABLE IF EXISTS \"");
          sql += obj->GetName();
          sql += wxT("\"; COMMIT;");
      } else
        {
          // there are no SpatialMetadata at all; we'll simply try to drop the table
          sql = wxT("BEGIN; ");
          sql += wxT("DROP TABLE IF EXISTS \"");
          sql += obj->GetName();
          sql += wxT("\"; COMMIT;");
        }
      msg = wxT("Do you really intend to drop the Table named: \"");
      msg += obj->GetName();
      msg += wxT("\"\n[and any other related object] ?");
    }
  if (obj->GetType() == MY_VIEW)
    {
      msg = wxT("Do you really intend to drop the View named: \"");
      msg += obj->GetName();
      msg += wxT("\" ?");
      sql = wxT("BEGIN; DROP VIEW IF EXISTS \"");
      sql += obj->GetName();
      sql += wxT("\"; COMMIT;");
    }
  if (obj->GetType() == MY_INDEX)
    {
      sql = wxT("BEGIN; DROP INDEX IF EXISTS \"");
      sql += obj->GetName();
      sql += wxT("\"; COMMIT;");
    }
  if (obj->GetType() == MY_TRIGGER)
    {
      msg = wxT("Do you really intend to drop the Index named: \"");
      msg += obj->GetName();
      msg += wxT("\" ?");
      sql = wxT("BEGIN; DROP TRIGGER IF EXISTS \"");
      sql += obj->GetName();
      sql += wxT("\"; COMMIT;");
    }
  if (sql.Len() < 1)
    return;
  wxMessageDialog confirm(this, msg, wxT("Confirming DROP"),
                          wxOK | wxCANCEL | wxICON_QUESTION);
  ret = confirm.ShowModal();
  if (ret != wxID_OK)
    return;
  ::wxBeginBusyCursor();
  ret = sqlite3_exec(sqlite, sql.ToUTF8(), NULL, NULL, &errMsg);
  if (ret != SQLITE_OK)
    {
      wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      sqlite3_free(errMsg);
      ::wxEndBusyCursor();
      goto rollback;
    }
  ::wxEndBusyCursor();
  wxMessageBox(wxT("Selected object '") + obj->GetName() +
               wxT("' was successfully removed"), wxT("spatialite-gui"),
               wxOK | wxICON_INFORMATION, this);
  MainFrame->InitTableTree();
  return;
rollback:
  ret = sqlite3_exec(sqlite, "ROLLBACK", NULL, NULL, &errMsg);
  if (ret != SQLITE_OK)
    {
      wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      sqlite3_free(errMsg);
      ::wxEndBusyCursor();
      return;
    }
  ::wxEndBusyCursor();
  wxMessageBox(wxT
               ("An error occurred\n\na ROLLBACK was automatically performed"),
               wxT("spatialite-gui"), wxOK | wxICON_WARNING, this);
}

void MyTableTree::OnCmdRename(wxCommandEvent & event)
{
//
// menu event - table renaming required
//
  wxString sql;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_TABLE || obj->GetType() == MY_VTABLE)
    {
      sql = wxT("ALTER TABLE \"");
      sql += obj->GetName();
      sql += wxT("\"\nRENAME TO ...new-table-name...");
      MainFrame->SetSql(sql, false);
    }
}

void MyTableTree::OnCmdRefresh(wxCommandEvent & event)
{
//
// menu event - refreshing the Tree
//
  MainFrame->InitTableTree();
}

void MyTableTree::OnCmdRecover(wxCommandEvent & event)
{
//
// menu event - Recover Geometry
//
  char *errMsg = NULL;
  int ret;
  wxString sql;
  int i;
  char **results;
  int rows;
  int columns;
  int retval = 0;
  int srid = -1;
  char dummy[128];
  wxString type;
  RecoverDialog dlg;
  sqlite3 *sqlite = MainFrame->GetSqlite();
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_COLUMN)
    {
      // trying to recover a geometry column
      if (MainFrame->CheckMetadata() == false)
        {
          wxMessageBox(wxT
                       ("Missing Spatial Metadata tables\n\ntry to run the 'init_spatialite.sql' script ..."),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          return;
        }
      dlg.Create(MainFrame, obj->GetName(), obj->GetColumn());
      ret = dlg.ShowModal();
      if (ret == wxID_OK)
        {
          srid = dlg.GetSrid();
          type = dlg.GetType();
          ::wxBeginBusyCursor();
          ret = sqlite3_exec(sqlite, "BEGIN", NULL, NULL, &errMsg);
          if (ret != SQLITE_OK)
            {
              wxMessageBox(wxT("SQLite SQL error: ") +
                           wxString::FromUTF8(errMsg), wxT("spatialite-gui"),
                           wxOK | wxICON_ERROR, this);
              sqlite3_free(errMsg);
              ::wxEndBusyCursor();
              return;
            }
          sql = wxT("SELECT RecoverGeometryColumn('");
          sql += obj->GetName();
          sql += wxT("', '");
          sql += obj->GetColumn();
          sprintf(dummy, "', %d, '", srid);
          sql += wxString::FromUTF8(dummy);
          sql += type;
          sql += wxT("', 2)");
          ret =
            sqlite3_get_table(sqlite, sql.ToUTF8(), &results, &rows, &columns,
                              &errMsg);
          if (ret != SQLITE_OK)
            {
              wxMessageBox(wxT("SQLite SQL error: ") +
                           wxString::FromUTF8(errMsg), wxT("spatialite-gui"),
                           wxOK | wxICON_ERROR, this);
              sqlite3_free(errMsg);
              goto rollback;
            }
          if (rows < 1)
            ;
          else
            {
              for (i = 1; i <= rows; i++)
                {
                  if (results[(i * columns) + 0])
                    retval = atoi(results[(i * columns) + 0]);
                }
            }
          sqlite3_free_table(results);
          if (!retval)
            goto rollback;
          ret = sqlite3_exec(sqlite, "COMMIT", NULL, NULL, &errMsg);
          if (ret != SQLITE_OK)
            {
              wxMessageBox(wxT("SQLite SQL error: ") +
                           wxString::FromUTF8(errMsg), wxT("spatialite-gui"),
                           wxOK | wxICON_ERROR, this);
              sqlite3_free(errMsg);
              ::wxEndBusyCursor();
              return;
            }
          ::wxEndBusyCursor();
          wxMessageBox(wxT("Geometry column \"") + obj->GetName() +
                       wxT(".") + obj->GetColumn() +
                       wxT("\" was successfully recovered"),
                       wxT("spatialite-gui"), wxOK | wxICON_INFORMATION, this);
        }
    }
  return;
rollback:
  ret = sqlite3_exec(sqlite, "ROLLBACK", NULL, NULL, &errMsg);
  if (ret != SQLITE_OK)
    {
      wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      sqlite3_free(errMsg);
      ::wxEndBusyCursor();
      return;
    }
  ::wxEndBusyCursor();
  wxMessageBox(wxT
               ("Geometry column doesn't satisfies required constraints\n\na ROLLBACK was automatically performed"),
               wxT("spatialite-gui"), wxOK | wxICON_WARNING, this);
}

void MyTableTree::OnCmdShowSql(wxCommandEvent & event)
{
//
// menu event - Showing CREATE TABLE statement
//
  wxString sql;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_TABLE || obj->GetType() == MY_VTABLE)
    {
      sql =
        wxT("SELECT sql FROM sqlite_master\nWHERE type = 'table' AND name = '");
      sql += obj->GetName();
      sql += wxT("'");
      MainFrame->SetSql(sql, true);
    }
  if (obj->GetType() == MY_VIEW)
    {
      sql =
        wxT("SELECT sql FROM sqlite_master\nWHERE type = 'view' AND name = '");
      sql += obj->GetName();
      sql += wxT("'");
      MainFrame->SetSql(sql, true);
    }
}

void MyTableTree::OnCmdSpatialIndex(wxCommandEvent & event)
{
//
// menu event - Spatial Index creation-destruction
//
  char *errMsg = NULL;
  int ret;
  wxString sql;
  wxString msg;
  int i;
  char **results;
  int rows;
  int columns;
  int retval = 0;
  sqlite3 *sqlite = MainFrame->GetSqlite();
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_GEOMETRY)
    {
      // creating the Spatial Index
      ::wxBeginBusyCursor();
      ret = sqlite3_exec(sqlite, "BEGIN", NULL, NULL, &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          ::wxEndBusyCursor();
          return;
        }
      sql = wxT("SELECT CreateSpatialIndex('");
      sql += obj->GetName();
      sql += wxT("', '");
      sql += obj->GetColumn();
      sql += wxT("')");
      ret =
        sqlite3_get_table(sqlite, sql.ToUTF8(), &results, &rows, &columns,
                          &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          goto rollback;
        }
      if (rows < 1)
        ;
      else
        {
          for (i = 1; i <= rows; i++)
            {
              if (results[(i * columns) + 0])
                retval = atoi(results[(i * columns) + 0]);
            }
        }
      sqlite3_free_table(results);
      if (!retval)
        goto rollback;
      ret = sqlite3_exec(sqlite, "COMMIT", NULL, NULL, &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          ::wxEndBusyCursor();
          return;
        }
      ::wxEndBusyCursor();
      wxMessageBox(wxT("Spatial Index \"idx_") + obj->GetName() +
                   wxT("_") + obj->GetColumn() +
                   wxT("\" was successfully created"), wxT("spatialite-gui"),
                   wxOK | wxICON_INFORMATION, this);
      MainFrame->InitTableTree();
  } else if (obj->GetType() == MY_GEOMETRY_INDEX)
    {
      // dropping the Spatial Index
      ::wxBeginBusyCursor();
      ret = sqlite3_exec(sqlite, "BEGIN", NULL, NULL, &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          ::wxEndBusyCursor();
          return;
        }
      sql = wxT("SELECT DisableSpatialIndex('");
      sql += obj->GetName();
      sql += wxT("', '");
      sql += obj->GetColumn();
      sql += wxT("')");
      msg = wxT("Do you really intend to delete the SpatialIndex\n");
      msg += wxT("on column \"");
      msg += obj->GetName();
      msg += wxT("\".\"");
      msg += obj->GetColumn();
      msg += wxT("\" ?");
      wxMessageDialog confirm(this, msg, wxT("Confirming Delete Spatial Index"),
                              wxOK | wxCANCEL | wxICON_QUESTION);
      ret = confirm.ShowModal();
      if (ret != wxID_OK)
        return;
      ret =
        sqlite3_get_table(sqlite, sql.ToUTF8(), &results, &rows, &columns,
                          &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          goto rollback;
        }
      if (rows < 1)
        ;
      else
        {
          for (i = 1; i <= rows; i++)
            {
              if (results[(i * columns) + 0])
                retval = atoi(results[(i * columns) + 0]);
            }
        }
      sqlite3_free_table(results);
      if (!retval)
        goto rollback;
      sql = wxT("DROP TABLE IF EXISTS \"idx_");
      sql += obj->GetName();
      sql += wxT("_");
      sql += obj->GetColumn();
      sql += wxT("\"");
      ret = sqlite3_exec(sqlite, sql.ToUTF8(), NULL, NULL, &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          ::wxEndBusyCursor();
          return;
        }
      ret = sqlite3_exec(sqlite, "COMMIT", NULL, NULL, &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          ::wxEndBusyCursor();
          return;
        }
      ::wxEndBusyCursor();
      wxMessageBox(wxT("Spatial Index idx_") + obj->GetName() +
                   wxT("_") + obj->GetColumn() +
                   wxT(" was successfully removed"), wxT("spatialite-gui"),
                   wxOK | wxICON_INFORMATION, this);
      MainFrame->InitTableTree();
    }
  return;
rollback:
  ret = sqlite3_exec(sqlite, "ROLLBACK", NULL, NULL, &errMsg);
  if (ret != SQLITE_OK)
    {
      wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      sqlite3_free(errMsg);
      ::wxEndBusyCursor();
      return;
    }
  ::wxEndBusyCursor();
  wxMessageBox(wxT
               ("An error occurred\n\na ROLLBACK was automatically performed"),
               wxT("spatialite-gui"), wxOK | wxICON_WARNING, this);
}

void MyTableTree::OnCmdMbrCache(wxCommandEvent & event)
{
//
// menu event - MBR cache creation-destruction
//
  char *errMsg = NULL;
  int ret;
  wxString sql;
  wxString msg;
  int i;
  char **results;
  int rows;
  int columns;
  int retval = 0;
  sqlite3 *sqlite = MainFrame->GetSqlite();
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_GEOMETRY)
    {
      // creating the MBR cache
      ::wxBeginBusyCursor();
      ret = sqlite3_exec(sqlite, "BEGIN", NULL, NULL, &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          ::wxEndBusyCursor();
          return;
        }
      sql = wxT("SELECT CreateMbrCache('");
      sql += obj->GetName();
      sql += wxT("', '");
      sql += obj->GetColumn();
      sql += wxT("')");
      ret =
        sqlite3_get_table(sqlite, sql.ToUTF8(), &results, &rows, &columns,
                          &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          goto rollback;
        }
      if (rows < 1)
        ;
      else
        {
          for (i = 1; i <= rows; i++)
            {
              if (results[(i * columns) + 0])
                retval = atoi(results[(i * columns) + 0]);
            }
        }
      sqlite3_free_table(results);
      if (!retval)
        goto rollback;
      ret = sqlite3_exec(sqlite, "COMMIT", NULL, NULL, &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          ::wxEndBusyCursor();
          return;
        }
      ::wxEndBusyCursor();
      wxMessageBox(wxT("MBR cache \"cache_") + obj->GetName() +
                   wxT("_") + obj->GetColumn() +
                   wxT("\" was successfully created"), wxT("spatialite-gui"),
                   wxOK | wxICON_INFORMATION, this);
      MainFrame->InitTableTree();
  } else if (obj->GetType() == MY_GEOMETRY_CACHED)
    {
      // dropping the MBR cache
      ::wxBeginBusyCursor();
      ret = sqlite3_exec(sqlite, "BEGIN", NULL, NULL, &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          ::wxEndBusyCursor();
          return;
        }
      sql = wxT("SELECT DisableSpatialIndex('");
      sql += obj->GetName();
      sql += wxT("', '");
      sql += obj->GetColumn();
      sql += wxT("')");
      msg = wxT("Do you really intend to delete the MBR cache\n");
      msg += wxT("on column \"");
      msg += obj->GetName();
      msg += wxT("\".\"");
      msg += obj->GetColumn();
      msg += wxT("\" ?");
      wxMessageDialog confirm(this, msg, wxT("Confirming Delete MBR cache"),
                              wxOK | wxCANCEL | wxICON_QUESTION);
      ret = confirm.ShowModal();
      if (ret != wxID_OK)
        return;
      ret =
        sqlite3_get_table(sqlite, sql.ToUTF8(), &results, &rows, &columns,
                          &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          goto rollback;
        }
      if (rows < 1)
        ;
      else
        {
          for (i = 1; i <= rows; i++)
            {
              if (results[(i * columns) + 0])
                retval = atoi(results[(i * columns) + 0]);
            }
        }
      sqlite3_free_table(results);
      if (!retval)
        goto rollback;
      sql = wxT("DROP TABLE IF EXISTS \"cache_");
      sql += obj->GetName();
      sql += wxT("_");
      sql += obj->GetColumn();
      sql += wxT("\"");
      ret = sqlite3_exec(sqlite, sql.ToUTF8(), NULL, NULL, &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          ::wxEndBusyCursor();
          return;
        }
      ret = sqlite3_exec(sqlite, "COMMIT", NULL, NULL, &errMsg);
      if (ret != SQLITE_OK)
        {
          wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                       wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
          sqlite3_free(errMsg);
          ::wxEndBusyCursor();
          return;
        }
      ::wxEndBusyCursor();
      wxMessageBox(wxT("MBR cache \"cache_") + obj->GetName() +
                   wxT("_") + obj->GetColumn() +
                   wxT("\" was successfully removed"), wxT("spatialite-gui"),
                   wxOK | wxICON_INFORMATION, this);
      MainFrame->InitTableTree();
    }
  return;
rollback:
  ret = sqlite3_exec(sqlite, "ROLLBACK", NULL, NULL, &errMsg);
  if (ret != SQLITE_OK)
    {
      wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      sqlite3_free(errMsg);
      ::wxEndBusyCursor();
      return;
    }
  ::wxEndBusyCursor();
  wxMessageBox(wxT
               ("An error occurred\n\na ROLLBACK was automatically performed"),
               wxT("spatialite-gui"), wxOK | wxICON_WARNING, this);
}

void MyTableTree::OnCmdRebuildTriggers(wxCommandEvent & event)
{
//
// menu event - rebuilding Geometry Triggers
//
  wxString sql;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_COLUMN || obj->GetType() == MY_GEOMETRY
      || obj->GetType() == MY_GEOMETRY_INDEX
      || obj->GetType() == MY_GEOMETRY_CACHED)
    {
      sql = wxT("SELECT RebuildGeometryTriggers('");
      sql += obj->GetName();
      sql += wxT("', '");
      sql += obj->GetColumn();
      sql += wxT("')");
      MainFrame->SetSql(sql, true);
    }
}

void MyTableTree::OnCmdCheckGeometry(wxCommandEvent & event)
{
//
// menu event - checking geometries
//
  wxString sql;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_COLUMN || obj->GetType() == MY_GEOMETRY
      || obj->GetType() == MY_GEOMETRY_INDEX
      || obj->GetType() == MY_GEOMETRY_CACHED)
    {
      sql = wxT("SELECT Count(*), GeometryType(\"");
      sql += obj->GetColumn();
      sql += wxT("\"), Srid(\"");
      sql += obj->GetColumn();
      sql += wxT("\")\nFROM \"");
      sql += obj->GetName();
      sql += wxT("\"\nGROUP BY 2, 3");
      MainFrame->SetSql(sql, true);
    }
}

void MyTableTree::OnCmdSetSrid(wxCommandEvent & event)
{
//
// menu event - setting SRID for geometries
//
  SetSridDialog dlg;
  wxString sql;
  int srid;
  int oldSrid;
  int ret;
  char dummy[128];
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_COLUMN)
    {
      dlg.Create(MainFrame, obj->GetName(), obj->GetColumn());
      ret = dlg.ShowModal();
      if (ret == wxID_OK)
        {
          srid = dlg.GetSrid();
          oldSrid = dlg.GetOldSrid();
          sql = wxT("UPDATE ");
          sql += obj->GetName();
          sql += wxT(" SET ");
          sql += obj->GetColumn();
          sql += wxT(" = SetSrid(");
          sql += obj->GetColumn();
          sprintf(dummy, ", %d)", srid);
          sql += wxString::FromUTF8(dummy);
          sql += wxT("\nWHERE Srid(");
          sql += obj->GetColumn();
          sprintf(dummy, ") = %d", oldSrid);
          sql += wxString::FromUTF8(dummy);
          MainFrame->SetSql(sql, true);
        }
    }
}

void MyTableTree::OnCmdDumpShp(wxCommandEvent & event)
{
//
// menu event - dumping as Shapefile
//
  int ret;
  wxString path;
  wxString lastDir;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  if (obj->GetType() == MY_GEOMETRY || obj->GetType() == MY_GEOMETRY_INDEX
      || obj->GetType() == MY_GEOMETRY_CACHED)
    {
      wxFileDialog *fileDialog = new wxFileDialog(this, wxT("Dump Shapefile"),
                                                  wxT(""), wxT("shapefile.shp"),
                                                  wxT
                                                  ("Shapefile (*.shp)|*.shp|All files (*.*)|*.*"),
                                                  wxFD_SAVE |
                                                  wxFD_OVERWRITE_PROMPT,
                                                  wxDefaultPosition,
                                                  wxDefaultSize,
                                                  wxT("filedlg"));
      lastDir = MainFrame->GetLastDirectory();
      if (lastDir.Len() >= 1)
        fileDialog->SetDirectory(lastDir);
      ret = fileDialog->ShowModal();
      if (ret == wxID_OK)
        {

          wxFileName file(fileDialog->GetPath());
          path = file.GetPath();
          path += file.GetPathSeparator();
          path += file.GetName();
          lastDir = file.GetPath();
          if (MainFrame->IsSetAskCharset() == false)
            {
              // using the default output charset
              MainFrame->SetLastDirectory(lastDir);
              ::wxBeginBusyCursor();
              MainFrame->DumpShapefile(path, obj->GetName(), obj->GetColumn(),
                                       MainFrame->GetDefaultCharset());
              ::wxEndBusyCursor();
          } else
            {
              // asking the charset to be used
              DumpShpDialog dlg;
              dlg.Create(MainFrame, path, obj->GetName(), obj->GetColumn(),
                         MainFrame->GetDefaultCharset());
              ret = dlg.ShowModal();
              if (ret == wxID_OK)
                {
                  MainFrame->SetLastDirectory(lastDir);
                  ::wxBeginBusyCursor();
                  MainFrame->DumpShapefile(path, obj->GetName(),
                                           obj->GetColumn(), dlg.GetCharset());
                  ::wxEndBusyCursor();
                }
            }
        }
    }
}

void MyTableTree::OnCmdDumpTxtTab(wxCommandEvent & event)
{
//
// menu event - dumping as TxtTab
//
  int ret;
  wxString path;
  wxString lastDir;
  wxString target;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  wxFileDialog *fileDialog = new wxFileDialog(this, wxT("Dump Txt/Tab file"),
                                              wxT(""), wxT("table.txt"),
                                              wxT
                                              ("File Txt/Tab (*.txt)|*.txt|All files (*.*)|*.*"),
                                              wxFD_SAVE | wxFD_OVERWRITE_PROMPT,
                                              wxDefaultPosition,
                                              wxDefaultSize, wxT("filedlg"));
  lastDir = MainFrame->GetLastDirectory();
  if (lastDir.Len() >= 1)
    fileDialog->SetDirectory(lastDir);
  ret = fileDialog->ShowModal();
  if (ret == wxID_OK)
    {
      wxFileName file(fileDialog->GetPath());
      path = file.GetPath();
      path += file.GetPathSeparator();
      path += file.GetName();
      path += wxT(".txt");
      lastDir = file.GetPath();
      if (MainFrame->IsSetAskCharset() == false)
        {
          // using the default output charset
          MainFrame->SetLastDirectory(lastDir);
          ::wxBeginBusyCursor();
          MainFrame->DumpTxtTab(path, obj->GetName(),
                                MainFrame->GetDefaultCharset());
          ::wxEndBusyCursor();
      } else
        {
          // asking the charset to be used
          DumpTxtDialog dlg;
          target = wxT("TXT / TAB");
          dlg.Create(MainFrame, path, target, MainFrame->GetDefaultCharset());
          ret = dlg.ShowModal();
          if (ret == wxID_OK)
            {
              MainFrame->SetLastDirectory(lastDir);
              ::wxBeginBusyCursor();
              MainFrame->DumpTxtTab(path, obj->GetName(), dlg.GetCharset());
              ::wxEndBusyCursor();
            }
        }
    }
}

void MyTableTree::OnCmdDumpCsv(wxCommandEvent & event)
{
//
// menu event - dumping as CSV
//
  int ret;
  wxString path;
  wxString lastDir;
  wxString target;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  wxFileDialog *fileDialog = new wxFileDialog(this, wxT("Dump CSV file"),
                                              wxT(""), wxT("table.csv"),
                                              wxT
                                              ("File CSV (*.csv)|*.csv|All files (*.*)|*.*"),
                                              wxFD_SAVE | wxFD_OVERWRITE_PROMPT,
                                              wxDefaultPosition,
                                              wxDefaultSize,
                                              wxT("filedlg"));
  lastDir = MainFrame->GetLastDirectory();
  if (lastDir.Len() >= 1)
    fileDialog->SetDirectory(lastDir);
  ret = fileDialog->ShowModal();
  if (ret == wxID_OK)
    {
      wxFileName file(fileDialog->GetPath());
      path = file.GetPath();
      path += file.GetPathSeparator();
      path += file.GetName();
      path += wxT(".csv");
      lastDir = file.GetPath();
      if (MainFrame->IsSetAskCharset() == false)
        {
          // using the default output charset
          MainFrame->SetLastDirectory(lastDir);
          ::wxBeginBusyCursor();
          MainFrame->DumpCsv(path, obj->GetName(),
                             MainFrame->GetDefaultCharset());
          ::wxEndBusyCursor();
      } else
        {
          // asking the charset to be used
          DumpTxtDialog dlg;
          target = wxT("CSV");
          dlg.Create(MainFrame, path, target, MainFrame->GetDefaultCharset());
          ret = dlg.ShowModal();
          if (ret == wxID_OK)
            {
              MainFrame->SetLastDirectory(lastDir);
              ::wxBeginBusyCursor();
              MainFrame->DumpCsv(path, obj->GetName(), dlg.GetCharset());
              ::wxEndBusyCursor();
            }
        }
    }
}

void MyTableTree::OnCmdDumpHtml(wxCommandEvent & event)
{
//
// menu event - dumping as Html
//
  int ret;
  wxString path;
  wxString lastDir;
  wxString target;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  wxFileDialog *fileDialog = new wxFileDialog(this, wxT("Dump HTML file"),
                                              wxT(""), wxT("table.html"),
                                              wxT
                                              ("HTML web page (*.html)|*.html|All files (*.*)|*.*"),
                                              wxFD_SAVE | wxFD_OVERWRITE_PROMPT,
                                              wxDefaultPosition,
                                              wxDefaultSize,
                                              wxT("filedlg"));
  lastDir = MainFrame->GetLastDirectory();
  if (lastDir.Len() >= 1)
    fileDialog->SetDirectory(lastDir);
  ret = fileDialog->ShowModal();
  if (ret == wxID_OK)
    {
      wxFileName file(fileDialog->GetPath());
      path = file.GetPath();
      path += file.GetPathSeparator();
      path += file.GetName();
      path += wxT(".html");
      lastDir = file.GetPath();
      if (MainFrame->IsSetAskCharset() == false)
        {
          // using the default output charset
          MainFrame->SetLastDirectory(lastDir);
          ::wxBeginBusyCursor();
          MainFrame->DumpHtml(path, obj->GetName(),
                              MainFrame->GetDefaultCharset());
          ::wxEndBusyCursor();
      } else
        {
          // asking the charset to be used
          DumpTxtDialog dlg;
          target = wxT("HTML");
          dlg.Create(MainFrame, path, target, MainFrame->GetDefaultCharset());
          ret = dlg.ShowModal();
          if (ret == wxID_OK)
            {
              MainFrame->SetLastDirectory(lastDir);
              ::wxBeginBusyCursor();
              MainFrame->DumpHtml(path, obj->GetName(), dlg.GetCharset());
              ::wxEndBusyCursor();
            }
        }
    }
}

void MyTableTree::OnCmdEdit(wxCommandEvent & event)
{
//
// menu event - editing row valuew
//
  char **results;
  int rows;
  int columns;
  int i;
  char *errMsg = NULL;
  wxString sql;
  char *column;
  char *type;
  int pk = 0;
  int pb = 0;
  int primaryKeys[1024];
  int blobCols[1024];
  for (i = 0; i < 1024; i++)
    {
      primaryKeys[i] = -1;
      blobCols[i] = -1;
    }
  primaryKeys[pk++] = 0;
  MyObject *obj = (MyObject *) GetItemData(CurrentItem);
  if (obj == NULL)
    return;
  sql = wxT("PRAGMA table_info(\"");
  sql += obj->GetName();
  sql += wxT("\")");
  int ret = sqlite3_get_table(MainFrame->GetSqlite(), sql.ToUTF8(), &results,
                              &rows, &columns, &errMsg);
  if (ret != SQLITE_OK)
    {
      wxMessageBox(wxT("SQLite SQL error: ") + wxString::FromUTF8(errMsg),
                   wxT("spatialite-gui"), wxOK | wxICON_ERROR, this);
      sqlite3_free(errMsg);
      return;
    }
  sql = wxT("");
  if (rows < 1)
    ;
  else
    {
      sql = wxT("SELECT ROWID");
      for (i = 1; i <= rows; i++)
        {
          column = results[(i * columns) + 1];
          sql += wxT(", \"");
          sql += wxString::FromUTF8(column);
          sql += wxT("\"");
          type = results[(i * columns) + 2];
          if (strcasecmp(type, "BLOB") == 0)
            blobCols[pb++] = i;
          if (atoi(results[(i * columns) + 5]) == 0)
            ;
          else
            primaryKeys[pk++] = i;
        }
    }
  sqlite3_free_table(results);
  if (sql.Len() < 1)
    return;
  sql += wxT("\nFROM \"");
  sql += obj->GetName();
  sql += wxT("\"\nORDER BY ROWID");
  MainFrame->EditTable(sql, primaryKeys, blobCols, obj->GetName());
}
