-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Procedure to analyse a .vlg file                                            --
--                                                                            --
--------------------------------------------------------------------------------

with SPARK_XML;
with Date_Time;

separate (VCS)
procedure AnalyseVictorLogFile
  (Filename           : in     E_Strings.T;
   VC_File_Date_Time  : in     E_Strings.T;
   SIV_File_Date_Time : in     E_Strings.T;
   VLG_File_Date_Time :    out E_Strings.T;
   Error_In_VLG_File  :    out Boolean;
   File_Error         :    out E_Strings.T)
is

   --  String and error conctants.
   Str_File_Corrupt   : constant String := "ViCToR log file corrupt: Could not parse timestamp";
   Str_Cannot_Open    : constant String := "Cannot open ViCToR log file";
   Str_VCG_Is_Newer   : constant String := "VCG file is newer than the VCT file";
   Str_SIV_Is_Newer   : constant String := "SIV file is newer than the VCT file";
   Str_Bad_Timestamps : constant String := "Malformed timestamps";

   Timestamp_Mark : constant String := "Date: ";

   File_Status     : SPARK_IO.File_Status;
   Victor_Log_File : SPARK_IO.File_Type;
   Line_Read       : E_Strings.T;

begin -- AnalyseVictorLogFile
   Error_In_VLG_File := False;
   File_Error        := E_Strings.Empty_String;
   Victor_Log_File   := SPARK_IO.Null_File;

   --  Open the ViCToR log file.
   E_Strings.Open
     (File         => Victor_Log_File,
      Mode_Of_File => SPARK_IO.In_File,
      Name_Of_File => Filename,
      Form_Of_File => "",
      Status       => File_Status);
   if File_Status /= SPARK_IO.Ok then
      Error_In_VLG_File := True;
      File_Error        := E_Strings.Copy_String (Str_Cannot_Open);
      FatalErrors.Process (FatalErrors.Could_Not_Open_Input_File, E_Strings.Empty_String);
   end if;

   --  The timestamp is on the first line.
   E_Strings.Get_Line (File  => Victor_Log_File,
                       E_Str => Line_Read);
   Line_Read := E_Strings.Trim (Line_Read);

   --# assert True;

   --  We have three options: A timestamp, no timestamp or a corrupt file
   if E_Strings.Starts_With (Line_Read, Timestamp_Mark)
     and then E_Strings.Get_Length (Line_Read) >= Timestamp_Mark'Length + 1 then
      VLG_File_Date_Time :=
        E_Strings.Section
        (E_Str     => Line_Read,
         Start_Pos => Timestamp_Mark'Length + 1,
         Length    => E_Strings.Get_Length (Line_Read) - Timestamp_Mark'Length);
      --# assert True;

      --  Since we've got a timestamp, we should compare it with the
      --  other timestamps, if we can.
      if not (E_Strings.Is_Empty (VC_File_Date_Time) or E_Strings.Starts_With (VC_File_Date_Time, "Unknown")) then
         case Date_Time.Compare_Timestamps (VC_File_Date_Time, VLG_File_Date_Time) is
            when Date_Time.A_Less_Than_B | Date_Time.A_Equals_B =>
               null;
            when Date_Time.A_Greater_Than_B =>
               Error_In_VLG_File := True;
               File_Error        := E_Strings.Copy_String (Str_VCG_Is_Newer);
            when Date_Time.Malformed_Timestamps =>
               Error_In_VLG_File := True;
               File_Error        := E_Strings.Copy_String (Str_Bad_Timestamps);
         end case;
      end if;
      --# assert True;
      if not (E_Strings.Is_Empty (SIV_File_Date_Time) or E_Strings.Starts_With (SIV_File_Date_Time, "Unknown")) then
         case Date_Time.Compare_Timestamps (SIV_File_Date_Time, VLG_File_Date_Time) is
            when Date_Time.A_Less_Than_B | Date_Time.A_Equals_B =>
               null;
            when Date_Time.A_Greater_Than_B =>
               Error_In_VLG_File := True;
               File_Error        := E_Strings.Copy_String (Str_SIV_Is_Newer);
            when Date_Time.Malformed_Timestamps =>
               Error_In_VLG_File := True;
               File_Error        := E_Strings.Copy_String (Str_Bad_Timestamps);
         end case;
      end if;
   elsif E_Strings.Is_Empty (Line_Read) then
      VLG_File_Date_Time := E_Strings.Copy_String (Unknown_VCT_Date);
   else
      VLG_File_Date_Time := E_Strings.Empty_String;
      Error_In_VLG_File  := True;
      File_Error         := E_Strings.Copy_String (Str_File_Corrupt);
   end if;

   --# assert True;

   --  Finally, close the ViCToR log file.
   --# accept F, 10, File_Status, "We don't care anymore since we've got everything we came for." &
   --#        F, 10, Victor_Log_File, "Same as above.";
   SPARK_IO.Close (Victor_Log_File, File_Status);
   --# end accept;
end AnalyseVictorLogFile;
