-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Ada.Characters.Handling;
with CommandLineData;
with ErrorHandler;
with SystemErrors;

use type CommandLineData.Language_Profiles;

package body SparkLex
--# own Curr_Line is LineManager.Curr_Line;
is

   End_Of_Text       : constant Character := Ada.Characters.Latin_1.ETX;
   End_Of_Line       : constant Character := Ada.Characters.Latin_1.CR;
   End_Of_Annotation : constant Character := Ada.Characters.Latin_1.EM;
   Line_Too_Long     : constant Character := Ada.Characters.Latin_1.ESC;

   --# inherit Ada.Characters.Latin_1,
   --#         CommandLineData,
   --#         Dictionary,
   --#         ErrorHandler,
   --#         E_Strings,
   --#         LexTokenManager,
   --#         SparkLex,
   --#         SPARK_IO,
   --#         SystemErrors;
   package LineManager
   --# own Curr_Line : Program_Line;
   is
      --++++ Type Program_Line and own variable Curr_Line are made visible to ++++--
      --++++ avoid excessive copying of the line contents by other modules. ++++--
      ----------------------------------------------------------------------------

      type Program_Line is record
         Context                                              : SparkLex.Program_Context;
         Anno_Context                                         : SparkLex.Annotation_Context;
         Line_No                                              : LexTokenManager.Line_Numbers;
         Line_Length, Last_Token_Pos, Curr_Pos, Lookahead_Pos : SparkLex.Line_Lengths;
         Conts                                                : SparkLex.Line;
      end record;

      Curr_Line : Program_Line;  -- By SPARK Rules, read only outside LineManager.
                                 ------------------------------------------------------------------------------

      procedure Clear_Line;
      --# global out Curr_Line;
      --# derives Curr_Line from ;

      procedure Copy_Out_Line (Line : out Program_Line);
      --# global in Curr_Line;
      --# derives Line from Curr_Line;

      procedure Copy_In_Line (Line : in Program_Line);
      --# global out Curr_Line;
      --# derives Curr_Line from Line;

      procedure Record_Curr_Pos;
      --# global in out Curr_Line;
      --# derives Curr_Line from *;

      procedure Reset_Curr_Pos;
      --# global in out Curr_Line;
      --# derives Curr_Line from *;

      procedure Inspect_Char (Ch : out Character);
      --# global in Curr_Line;
      --# derives Ch from Curr_Line;

      procedure Accept_Char;
      --# global in out Curr_Line;
      --# derives Curr_Line from *;

      procedure Lookahead_Char (Ch : out Character);
      --# global in out Curr_Line;
      --# derives Ch,
      --#         Curr_Line from Curr_Line;

      procedure Accept_Lookahead;
      --# global in out Curr_Line;
      --# derives Curr_Line from *;

      procedure Reject_Lookahead;
      --# global in out Curr_Line;
      --# derives Curr_Line from *;

      procedure Next_Sig_Char (Prog_Text : in SPARK_IO.File_Type);
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out Curr_Line;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives Curr_Line,
      --#         ErrorHandler.Error_Context,
      --#         LexTokenManager.State,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Curr_Line,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Prog_Text,
      --#                                         SPARK_IO.File_Sys;

      procedure Set_Context (New_Context : in SparkLex.Program_Context);
      --# global in out Curr_Line;
      --# derives Curr_Line from *,
      --#                        New_Context;

      procedure Set_Anno_Context (New_Context : in SparkLex.Annotation_Context);
      --# global in out Curr_Line;
      --# derives Curr_Line from *,
      --#                        New_Context;
      pragma Unreferenced (Set_Anno_Context); -- not used at present

   end LineManager;

   No_Of_RW      : constant Natural  := 120;
   Min_RW_Length : constant Positive := 1;
   Max_RW_Length : constant Positive := 20;

   type Anno_Type is (Start_Anno, Proof_Anno, Hide_Anno, Other_Anno, No_Anno);

   subtype RW_Length is Positive range 1 .. Max_RW_Length;
   subtype RW_Index is Natural range 0 .. No_Of_RW - 1;

   subtype Res_Word is String (RW_Length);
   type RWPair is record
      Word  : Res_Word;
      Token : SPSymbols.SPTerminal;
   end record;
   type RWList is array (RW_Index) of RWPair;

   -- The table of reserved words MUST be listed in alphabetical order.
   RW : constant RWList :=
     RWList'
     (RWPair'(Res_Word'("abort               "), SPSymbols.RWabort),
      RWPair'(Res_Word'("abs                 "), SPSymbols.RWabs),
      RWPair'(Res_Word'("abstract            "), SPSymbols.RWabstract),
      RWPair'(Res_Word'("accept              "), SPSymbols.RWaccept),
      RWPair'(Res_Word'("access              "), SPSymbols.RWaccess),
      RWPair'(Res_Word'("aliased             "), SPSymbols.RWaliased),
      RWPair'(Res_Word'("all                 "), SPSymbols.RWall),
      RWPair'(Res_Word'("and                 "), SPSymbols.RWand),
      RWPair'(Res_Word'("are_interchangeable "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("array               "), SPSymbols.RWarray),
      RWPair'(Res_Word'("as                  "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("assert              "), SPSymbols.RWassert),
      RWPair'(Res_Word'("assume              "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("at                  "), SPSymbols.RWat),

      RWPair'(Res_Word'("begin               "), SPSymbols.RWbegin),
      RWPair'(Res_Word'("body                "), SPSymbols.RWbody),

      RWPair'(Res_Word'("case                "), SPSymbols.RWcase),
      RWPair'(Res_Word'("check               "), SPSymbols.RWcheck),
      RWPair'(Res_Word'("const               "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("constant            "), SPSymbols.RWconstant),

      RWPair'(Res_Word'("declare             "), SPSymbols.RWdeclare),
      RWPair'(Res_Word'("delay               "), SPSymbols.RWdelay),
      RWPair'(Res_Word'("delta               "), SPSymbols.RWdelta),
      RWPair'(Res_Word'("derives             "), SPSymbols.RWderives),
      RWPair'(Res_Word'("digits              "), SPSymbols.RWdigits),
      RWPair'(Res_Word'("div                 "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("do                  "), SPSymbols.RWdo),

      RWPair'(Res_Word'("element             "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("else                "), SPSymbols.RWelse),
      RWPair'(Res_Word'("elsif               "), SPSymbols.RWelsif),
      RWPair'(Res_Word'("end                 "), SPSymbols.RWend),
      RWPair'(Res_Word'("entry               "), SPSymbols.RWentry),
      RWPair'(Res_Word'("exception           "), SPSymbols.RWexception),
      RWPair'(Res_Word'("exit                "), SPSymbols.RWexit),

      RWPair'(Res_Word'("finish              "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("first               "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("for                 "), SPSymbols.RWfor),
      RWPair'(Res_Word'("for_all             "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("for_some            "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("from                "), SPSymbols.RWfrom),
      RWPair'(Res_Word'("function            "), SPSymbols.RWfunction),

      RWPair'(Res_Word'("generic             "), SPSymbols.RWgeneric),
      RWPair'(Res_Word'("global              "), SPSymbols.RWglobal),
      RWPair'(Res_Word'("goal                "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("goto                "), SPSymbols.RWgoto),

      RWPair'(Res_Word'("hide                "), SPSymbols.RWhide),

      RWPair'(Res_Word'("if                  "), SPSymbols.RWif),
      RWPair'(Res_Word'("in                  "), SPSymbols.RWin),
      RWPair'(Res_Word'("inherit             "), SPSymbols.RWinherit),
      RWPair'(Res_Word'("initializes         "), SPSymbols.RWinitializes),
      RWPair'(Res_Word'("interface           "), SPSymbols.RWinterface),
      RWPair'(Res_Word'("is                  "), SPSymbols.RWis),

      RWPair'(Res_Word'("last                "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("limited             "), SPSymbols.RWlimited),
      RWPair'(Res_Word'("loop                "), SPSymbols.RWloop),

      RWPair'(Res_Word'("main_program        "), SPSymbols.RWmain_program),
      RWPair'(Res_Word'("may_be_deduced      "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("may_be_deduced_from "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("may_be_replaced_by  "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("mod                 "), SPSymbols.RWmod),

      RWPair'(Res_Word'("new                 "), SPSymbols.RWnew),
      RWPair'(Res_Word'("nonfirst            "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("nonlast             "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("not                 "), SPSymbols.RWnot),
      RWPair'(Res_Word'("not_in              "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("null                "), SPSymbols.RWnull),

      RWPair'(Res_Word'("odd                 "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("of                  "), SPSymbols.RWof),
      RWPair'(Res_Word'("or                  "), SPSymbols.RWor),
      RWPair'(Res_Word'("others              "), SPSymbols.RWothers),
      RWPair'(Res_Word'("out                 "), SPSymbols.RWout),
      RWPair'(Res_Word'("overriding          "), SPSymbols.RWoverriding),
      RWPair'(Res_Word'("own                 "), SPSymbols.RWown),

      RWPair'(Res_Word'("package             "), SPSymbols.RWpackage),
      RWPair'(Res_Word'("pending             "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("post                "), SPSymbols.RWpost),
      RWPair'(Res_Word'("pragma              "), SPSymbols.RWpragma),
      RWPair'(Res_Word'("pre                 "), SPSymbols.RWpre),
      RWPair'(Res_Word'("pred                "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("private             "), SPSymbols.RWprivate),
      RWPair'(Res_Word'("procedure           "), SPSymbols.RWprocedure),
      RWPair'(Res_Word'("proof               "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("protected           "), SPSymbols.RWprotected),

      RWPair'(Res_Word'("raise               "), SPSymbols.RWraise),
      RWPair'(Res_Word'("range               "), SPSymbols.RWrange),
      RWPair'(Res_Word'("real                "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("record              "), SPSymbols.RWrecord),
      RWPair'(Res_Word'("rem                 "), SPSymbols.RWrem),
      RWPair'(Res_Word'("renames             "), SPSymbols.RWrenames),
      RWPair'(Res_Word'("requeue             "), SPSymbols.RWrequeue),
      RWPair'(Res_Word'("requires            "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("return              "), SPSymbols.RWreturn),
      RWPair'(Res_Word'("reverse             "), SPSymbols.RWreverse),

      RWPair'(Res_Word'("save                "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("select              "), SPSymbols.RWselect),
      RWPair'(Res_Word'("separate            "), SPSymbols.RWseparate),
      RWPair'(Res_Word'("sequence            "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("set                 "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("some                "), SPSymbols.RWsome),
      RWPair'(Res_Word'("sqr                 "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("start               "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("strict_subset_of    "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("subset_of           "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("subtype             "), SPSymbols.RWsubtype),
      RWPair'(Res_Word'("succ                "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("synchronized        "), SPSymbols.RWsynchronized),

      RWPair'(Res_Word'("tagged              "), SPSymbols.RWtagged),
      RWPair'(Res_Word'("task                "), SPSymbols.RWtask),
      RWPair'(Res_Word'("terminate           "), SPSymbols.RWterminate),
      RWPair'(Res_Word'("then                "), SPSymbols.RWthen),
      RWPair'(Res_Word'("type                "), SPSymbols.RWtype),

      RWPair'(Res_Word'("until               "), SPSymbols.RWuntil),
      RWPair'(Res_Word'("update              "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("use                 "), SPSymbols.RWuse),

      RWPair'(Res_Word'("var                 "), SPSymbols.predefined_FDL_identifier),

      RWPair'(Res_Word'("when                "), SPSymbols.RWwhen),
      RWPair'(Res_Word'("where               "), SPSymbols.predefined_FDL_identifier),
      RWPair'(Res_Word'("while               "), SPSymbols.RWwhile),
      RWPair'(Res_Word'("with                "), SPSymbols.RWwith),

      RWPair'(Res_Word'("xor                 "), SPSymbols.RWxor));

         No_Of_FDL_RW : constant Natural := 35;
         subtype FDL_RW_Index is Natural range 0 .. No_Of_FDL_RW - 1;
         type FDL_RWList is array (FDL_RW_Index) of Res_Word;

         FDL_RW : constant FDL_RWList :=
           FDL_RWList'
           (Res_Word'("are_interchangeable "),
            Res_Word'("as                  "),
            Res_Word'("assume              "),

            Res_Word'("const               "),

            Res_Word'("div                 "),

            Res_Word'("element             "),

            Res_Word'("finish              "),
            Res_Word'("first               "),
            Res_Word'("for_all             "),
            Res_Word'("for_some            "),

            Res_Word'("goal                "),
            Res_Word'("last                "),

            Res_Word'("may_be_deduced      "),
            Res_Word'("may_be_deduced_from "),
            Res_Word'("may_be_replaced_by  "),

            Res_Word'("nonfirst            "),
            Res_Word'("nonlast             "),
            Res_Word'("not_in              "),

            Res_Word'("odd                 "),

            Res_Word'("pending             "),
            Res_Word'("pred                "),
            Res_Word'("proof               "),

            Res_Word'("real                "),
            Res_Word'("requires            "),

            Res_Word'("save                "),
            Res_Word'("sequence            "),
            Res_Word'("set                 "),
            Res_Word'("sqr                 "),
            Res_Word'("start               "),
            Res_Word'("strict_subset_of    "),
            Res_Word'("subset_of           "),
            Res_Word'("succ                "),

            Res_Word'("update              "),

            Res_Word'("var                 "),

            Res_Word'("where               "));

         subtype Offset is Integer range 0 .. 3;
         type Possible_Prefixes is (Field_Prefix, Update_Prefix);
         type Prefix_Strings is array (Offset) of Character;
         type Prefix_Tables is array (Possible_Prefixes) of Prefix_Strings;
         Prefix_Table : constant Prefix_Tables :=
           Prefix_Tables'(Field_Prefix  => Prefix_Strings'('f', 'l', 'd', '_'),
                          Update_Prefix => Prefix_Strings'('u', 'p', 'f', '_'));

         package body LineManager is separate;

         --++++  The following functions are defined as per Ada LRM Chapter 2. ++++--
         -------------------------------------------------------------------------------

         -- Note a "Special Character" here to the SPARK Lexer is NOT the
         -- same as Ada95 LRM's Ada.Characters.Handling.Is_Special
         function Special_Character (Ch : Character) return Boolean is
            Result : Boolean;
         begin
            case Ch is
               when '"' | '#' | '&' | ''' | '(' | ')' | '*' | '+' | ',' | '-' | '.' | '/' | ':' | ';' | '<' | '=' | '>' | '_' | '|' =>
                  Result := True;
               when others =>
                  Result := False;
            end case;
            return Result;
         end Special_Character;

         function Format_Effector (Ch : Character) return Boolean is
            Result : Boolean;
         begin
            case Ch is
               when Ada.Characters.Latin_1.HT |
                 Ada.Characters.Latin_1.VT |
                 Ada.Characters.Latin_1.CR |
                 Ada.Characters.Latin_1.LF |
                 Ada.Characters.Latin_1.FF =>
                  Result := True;
               when others =>
                  Result := False;
            end case;
            return Result;
         end Format_Effector;

         function Other_Special_Character (Ch : Character) return Boolean is
            Result : Boolean;
         begin
            case Ch is
               when '!' | '$' | '%' | '?' | '@' | '[' | '\' | ']' | '^' | '`' | '{' | '}' | '~' =>
                  Result := True;
               when others =>
                  Result := False;
            end case;
            return Result;
         end Other_Special_Character;

         -- Note a "Basic Character" here to the SPARK Lexer is NOT the
         -- same as Ada95 LRM's Ada.Characters.Handling.Is_Basic
         function Basic_Graphic_Character (Ch : Character) return Boolean is
         begin
            return Ada.Characters.Handling.Is_Upper (Ch)
              or else Ada.Characters.Handling.Is_Digit (Ch)
              or else Special_Character (Ch => Ch)
              or else Ch = ' ';
         end Basic_Graphic_Character;

         -- Note a "Graphic Character" here to the SPARK Lexer is NOT the
         -- same as Ada95 LRM's Ada.Characters.Handling.Is_Graphic
         function Graphic_Character (Ch : Character) return Boolean is
         begin
            return Basic_Graphic_Character (Ch => Ch)
              or else Ada.Characters.Handling.Is_Lower (Ch)
              or else Other_Special_Character (Ch => Ch);
         end Graphic_Character;

         function Separator (Ch : Character) return Boolean is
            Result : Boolean;
         begin
            case Ch is
               when ' '                       |
                 Ada.Characters.Latin_1.HT |
                 Ada.Characters.Latin_1.VT |
                 Ada.Characters.Latin_1.CR |
                 Ada.Characters.Latin_1.LF |
                 Ada.Characters.Latin_1.FF =>
                  Result := True;
               when others =>
                  Result := False;
            end case;
            return Result;
         end Separator;

         function Simple_Delimiter (Ch : Character) return Boolean is
            Result : Boolean;
         begin
            case Ch is
               when '&' |
                 ''' |
                 '(' |
                 ')' |
                 '*' |
                 '+' |
                 ',' |
                 '-' |
                 '.' |
                 '/' |
                 ':' |
                 ';' |
                 '<' |
                 '=' |
                 '>' |
                 '|' |
                 '[' |
                 ']' |
                 '@' |
                 '~' |
                 '%' =>
                  Result := True;
               when others =>
                  Result := False;
            end case;
            return Result;
         end Simple_Delimiter;

         function Letter_Or_Digit (Ch : Character) return Boolean is
         begin
            return Ada.Characters.Handling.Is_Letter (Ch) or else Ada.Characters.Handling.Is_Digit (Ch);
         end Letter_Or_Digit;

         function Extended_Digit (Ch : Character) return Boolean is
         begin
            -- We consider an extended digit to be a decimal digit or
            -- any letter, so...
            return Letter_Or_Digit (Ch => Ch);
         end Extended_Digit;

         -------------------------------------------------------------------------------

         procedure Clear_Line_Context
         --# global out LineManager.Curr_Line;
         --# derives LineManager.Curr_Line from ;
         is
         begin
            LineManager.Clear_Line;
         end Clear_Line_Context;

         procedure Store_Line_Context (File_Line : out Line_Context)
         --# global in LineManager.Curr_Line;
         --# derives File_Line from LineManager.Curr_Line;
         is
            Program_Line : LineManager.Program_Line;
         begin
            LineManager.Copy_Out_Line (Program_Line);
            File_Line.Context := Program_Line.Context;
            -- field-by-field update of record will give flow error here
            File_Line.Anno_Context   := Program_Line.Anno_Context;
            File_Line.Line_No        := Program_Line.Line_No;
            File_Line.Line_Length    := Program_Line.Line_Length;
            File_Line.Conts          := Program_Line.Conts;
            File_Line.Last_Token_Pos := Program_Line.Last_Token_Pos;
            File_Line.Curr_Pos       := Program_Line.Curr_Pos;
            File_Line.Lookahead_Pos  := Program_Line.Lookahead_Pos;
         end Store_Line_Context;

         procedure Restore_Line_Context (File_Line : in Line_Context)
         --# global out LineManager.Curr_Line;
         --# derives LineManager.Curr_Line from File_Line;
         is
            Program_Line : LineManager.Program_Line;
         begin
            Program_Line.Context := File_Line.Context;
            -- field-by-field update of record will give flow error here
            Program_Line.Anno_Context   := File_Line.Anno_Context;
            Program_Line.Line_No        := File_Line.Line_No;
            Program_Line.Conts          := File_Line.Conts;
            Program_Line.Line_Length    := File_Line.Line_Length;
            Program_Line.Last_Token_Pos := File_Line.Last_Token_Pos;
            Program_Line.Curr_Pos       := File_Line.Curr_Pos;
            Program_Line.Lookahead_Pos  := File_Line.Lookahead_Pos;
            LineManager.Copy_In_Line (Program_Line);
         end Restore_Line_Context;

         ---------------------------------------------------------------------------

         procedure Check_Reserved
           (Start_Pos, End_Pos : in     E_Strings.Lengths;
            Look_Ahead         : in     Boolean;
            Token              :    out SPSymbols.SPTerminal)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.State;
         --#        in     LineManager.Curr_Line;
         --#        in out ErrorHandler.Error_Context;
         --#        in out SPARK_IO.File_Sys;
         --# derives ErrorHandler.Error_Context,
         --#         SPARK_IO.File_Sys          from CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         End_Pos,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         LineManager.Curr_Line,
         --#                                         Look_Ahead,
         --#                                         SPARK_IO.File_Sys,
         --#                                         Start_Pos &
         --#         Token                      from CommandLineData.Content,
         --#                                         End_Pos,
         --#                                         LineManager.Curr_Line,
         --#                                         Start_Pos;
         is
            type Cmp_Res is (CEQ, CLT, CGT);
            Leng    : Natural;
            Ix      : RW_Index;
            IL, IU  : Natural;
            RW_Ix   : Res_Word;
            Result  : Cmp_Res;
            L_Token : SPSymbols.SPTerminal;

            -----------------------------------------

            procedure Comp_RW (R_Word             : in     Res_Word;
                               Start_Pos, End_Pos : in     E_Strings.Lengths;
                               Result             :    out Cmp_Res)
            --# global in LineManager.Curr_Line;
            --# derives Result from End_Pos,
            --#                     LineManager.Curr_Line,
            --#                     R_Word,
            --#                     Start_Pos;
            is
               Ch1, Ch2    : Character;
               LX          : E_Strings.Lengths;
               RW_X        : RW_Index;
               Comp_Result : Cmp_Res;
            begin
               RW_X := 1;
               LX   := Start_Pos;

               loop
                  Ch1 := R_Word (RW_X);

                  -- Reserved words in lower case.
                  Ch2 := Ada.Characters.Handling.To_Lower (LineManager.Curr_Line.Conts (LX));

                  if Ch1 < Ch2 then
                     Comp_Result := CLT;
                  elsif Ch1 > Ch2 then
                     Comp_Result := CGT;
                  elsif LX = End_Pos then
                     if RW_X = Max_RW_Length or else R_Word (RW_X + 1) = ' ' then
                        Comp_Result := CEQ;
                     else
                        Comp_Result := CGT;
                     end if;
                  else
                     Comp_Result := CEQ;
                  end if;
                  exit when Comp_Result /= CEQ or else LX = End_Pos;
                  LX   := LX + 1;
                  RW_X := RW_X + 1;
               end loop;
               Result := Comp_Result;
            end Comp_RW;

            function Check_FLD_Or_UPF return  SPSymbols.SPTerminal
            --# global in Leng;
            --#        in LineManager.Curr_Line;
            --#        in Start_Pos;
            is
               Prefix_Sort : Possible_Prefixes;
               Result      : SPSymbols.SPTerminal;

               ----------------------------------------------

               procedure Check_Rest_Of_Prefix
               --# global in     LineManager.Curr_Line;
               --#        in     Prefix_Sort;
               --#        in     Start_Pos;
               --#           out Result;
               --# derives Result from LineManager.Curr_Line,
               --#                     Prefix_Sort,
               --#                     Start_Pos;
               is
                  Ptr : Offset;
               begin
                  Ptr := 1;
                  loop
                     if Ada.Characters.Handling.To_Lower (LineManager.Curr_Line.Conts (Start_Pos + Ptr)) /=
                       Prefix_Table (Prefix_Sort) (Ptr) then
                        Result := SPSymbols.identifier;
                        exit;
                     end if;

                     if Ptr = Offset'Last then
                        Result := SPSymbols.predefined_FDL_identifier;
                        exit;
                     end if;

                     Ptr := Ptr + 1;
                  end loop;
               end Check_Rest_Of_Prefix;

               ----------------------------------------------

            begin  --Check_FLD_Or_UPF
               Result := SPSymbols.identifier;
               if Leng >= 5 then -- minimum length a valid fld_ or upf_ could be
                  if Ada.Characters.Handling.To_Lower (LineManager.Curr_Line.Conts (Start_Pos)) = 'f' then

                     Prefix_Sort := Field_Prefix;
                     Check_Rest_Of_Prefix;
                  elsif Ada.Characters.Handling.To_Lower (LineManager.Curr_Line.Conts (Start_Pos)) = 'u' then

                     Prefix_Sort := Update_Prefix;
                     Check_Rest_Of_Prefix;
                  else
                     Result := SPSymbols.identifier;
                  end if;
               end if;
               return Result;
            end Check_FLD_Or_UPF;

            -----------------------------------------------------------

            function Convert_FDL (Token : SPSymbols.SPTerminal) return SPSymbols.SPTerminal
            --# global in CommandLineData.Content;
            is
               Result : SPSymbols.SPTerminal;
            begin
               if Token = SPSymbols.predefined_FDL_identifier and then not CommandLineData.Content.FDL_Reserved then
                  Result := SPSymbols.identifier;
               else
                  Result := Token;
               end if;
               return Result;
            end Convert_FDL;

            -----------------------------------------------------------

            function Convert_Reserved (Token : SPSymbols.SPTerminal) return SPSymbols.SPTerminal
            --# global in CommandLineData.Content;
            is
               Result : SPSymbols.SPTerminal;
            begin
               if Token = SPSymbols.RWsome and then not CommandLineData.Content.FDL_Reserved then
                  Result := SPSymbols.identifier;
               else
                  Result := Token;
               end if;
               return Result;
            end Convert_Reserved;

            -----------------------------------------------------------

            procedure Convert95_And_2005_Reserved (Token           : in     SPSymbols.SPTerminal;
                                                   Converted_Token :    out SPSymbols.SPTerminal)
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     LexTokenManager.State;
            --#        in     LineManager.Curr_Line;
            --#        in     Look_Ahead;
            --#        in     Start_Pos;
            --#        in out ErrorHandler.Error_Context;
            --#        in out SPARK_IO.File_Sys;
            --# derives Converted_Token            from CommandLineData.Content,
            --#                                         Token &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         LexTokenManager.State,
            --#                                         LineManager.Curr_Line,
            --#                                         Look_Ahead,
            --#                                         SPARK_IO.File_Sys,
            --#                                         Start_Pos,
            --#                                         Token;
            is
               Result : SPSymbols.SPTerminal;

               --------------------------------------------------------

               -- Returns true if Token is a new 95 reserved word.
               function SPARK95_Reserved (Token : SPSymbols.SPTerminal) return Boolean
               --# global in CommandLineData.Content;
               is
               begin
                  -- In SPARK83 mode, "aliased", "protected", "requeue", "tagged",
                  -- "until" are all identifiers, not reserved words.  Additionally,
                  -- "abstract" is an identifier UNLESS -fdl is active, in which case
                  -- it remains a reserved word
                  return Token = SPSymbols.RWaliased or
                    Token = SPSymbols.RWprotected or
                    Token = SPSymbols.RWrequeue or
                    Token = SPSymbols.RWtagged or
                    Token = SPSymbols.RWuntil or
                    (Token = SPSymbols.RWabstract and not CommandLineData.Content.FDL_Reserved);
               end SPARK95_Reserved;

               --------------------------------------------------------

               --------------------------------------------------------

               -- Returns true if Token is a new 2005 reserved word.
               function SPARK2005_Reserved (Token : SPSymbols.SPTerminal) return Boolean is
               begin
                  return (Token = SPSymbols.RWinterface or Token = SPSymbols.RWoverriding or Token = SPSymbols.RWsynchronized);
               end SPARK2005_Reserved;

               --------------------------------------------------------

            begin -- Convert95_And_2005_Reserved
               case CommandLineData.Content.Language_Profile is
                  when CommandLineData.SPARK83 =>
                     if SPARK95_Reserved (Token => Token) or SPARK2005_Reserved (Token => Token) then
                        Result := SPSymbols.identifier;
                     else
                        Result := Token;
                     end if;
                  when CommandLineData.SPARK95 =>
                     if SPARK2005_Reserved (Token => Token) then
                        Result := SPSymbols.identifier;

                        if not Look_Ahead then

                           -- Raise a warning if the identifier is an Ada2005
                           -- reserve word. To remove duplicate warning messages,
                           -- lexical analyser only raises the warning if it is
                           -- not looking ahead.

                           ErrorHandler.Semantic_Warning
                             (7,
                              LexTokenManager.Token_Position'(Start_Line_No => LineManager.Curr_Line.Line_No,
                                                              Start_Pos     => Start_Pos),
                              LexTokenManager.Null_String);
                        end if;
                     else
                        Result := Token;
                     end if;
                  when CommandLineData.SPARK2005 =>
                     Result := Token;
               end case;

               Converted_Token := Result;
            end Convert95_And_2005_Reserved;

            -----------------------------------------------------------

         begin -- Check_Reserved
            Leng := (End_Pos - Start_Pos) + 1;
            if Leng >= Min_RW_Length and Leng <= Max_RW_Length then
               IL := RW_Index'First;
               IU := RW_Index'Last;

               loop
                  Comp_RW (R_Word    => RW (IL).Word,
                           Start_Pos => Start_Pos,
                           End_Pos   => End_Pos,
                           Result    => Result);
                  if Result = CGT then
                     exit;
                  end if;
                  Comp_RW (R_Word    => RW (IU).Word,
                           Start_Pos => Start_Pos,
                           End_Pos   => End_Pos,
                           Result    => Result);
                  if Result = CLT then
                     exit;
                  end if;

                  Ix    := (IL + IU) / 2;
                  RW_Ix := RW (Ix).Word;
                  Comp_RW (R_Word    => RW_Ix,
                           Start_Pos => Start_Pos,
                           End_Pos   => End_Pos,
                           Result    => Result);
                  case Result is
                     when CEQ =>
                        null;
                     when CGT =>
                        IU := Ix - 1;
                     when CLT =>
                        IL := Ix + 1;
                  end case;
                  exit when Result = CEQ or else IL > IU;
               end loop;

               if Result = CEQ then
                  --# accept F, 501, Ix, "Ix always defined on this path";
                  L_Token := RW (Ix).Token;
                  --# end accept;
               else
                  L_Token := Check_FLD_Or_UPF;
               end if;
            else
               L_Token := Check_FLD_Or_UPF;
            end if;

            Convert95_And_2005_Reserved (Token           => Convert_Reserved (Token => Convert_FDL (Token => L_Token)),
                                         Converted_Token => Token);

            --# accept F, 602, SPARK_IO.File_Sys, Ix, "Ix always defined here" &
            --#        F, 602, ErrorHandler.Error_Context, Ix, "Ix always defined here" &
            --#        F, 602, Token, Ix, "Ix always defined here";
         end Check_Reserved;

         function Check_FDL_RW (Ex_Str : E_Strings.T) return Boolean is
            type Cmp_Res is (CEQ, CLT, CGT);
            Is_FDL_RW : Boolean := True;
            Ix        : FDL_RW_Index;
            FDL_RW_Ix : Res_Word;
            IL, IU    : Natural;
            Result    : Cmp_Res;

            procedure Comp_FDL_RW (R_Word : in     Res_Word;
                                   Ex_Str : in     E_Strings.T;
                                   Result :    out Cmp_Res)
            --# derives Result from Ex_Str,
            --#                     R_Word;
            is
               Ch1, Ch2    : Character;
               LX          : E_Strings.Lengths;
               RW_X        : RW_Index;
               Comp_Result : Cmp_Res;
            begin
               RW_X := 1;
               LX   := 1;

               loop
                  Ch1 := R_Word (RW_X);

                  -- Reserved words in lower case.
                  Ch2 := Ada.Characters.Handling.To_Lower (E_Strings.Get_Element (E_Str => Ex_Str,
                                                                                  Pos   => LX));

                  if Ch1 < Ch2 then
                     Comp_Result := CLT;
                  elsif Ch1 > Ch2 then
                     Comp_Result := CGT;
                  elsif LX = E_Strings.Get_Length (E_Str => Ex_Str) then
                     if RW_X = Max_RW_Length or else R_Word (RW_X + 1) = ' ' then
                        Comp_Result := CEQ;
                     else
                        Comp_Result := CGT;
                     end if;
                  else
                     Comp_Result := CEQ;
                  end if;
                  exit when Comp_Result /= CEQ or else LX = E_Strings.Get_Length (E_Str => Ex_Str);
                  LX   := LX + 1;
                  RW_X := RW_X + 1;
               end loop;
               Result := Comp_Result;
            end Comp_FDL_RW;

         begin
            if E_Strings.Get_Length (E_Str => Ex_Str) > Max_RW_Length
              or else E_Strings.Get_Length (E_Str => Ex_Str) < Min_RW_Length then
               Is_FDL_RW := False;
            else
               IL := FDL_RW_Index'First;
               IU := FDL_RW_Index'Last;
               loop
                  Comp_FDL_RW (R_Word => FDL_RW (IL),
                               Ex_Str => Ex_Str,
                               Result => Result);
                  if Result = CGT then
                     exit;
                  end if;
                  Comp_FDL_RW (R_Word => FDL_RW (IU),
                               Ex_Str => Ex_Str,
                               Result => Result);
                  if Result = CLT then
                     exit;
                  end if;

                  Ix        := (IL + IU) / 2;
                  FDL_RW_Ix := FDL_RW (Ix);
                  Comp_FDL_RW (R_Word => FDL_RW_Ix,
                               Ex_Str => Ex_Str,
                               Result => Result);
                  case Result is
                     when CEQ =>
                        null;
                     when CGT =>
                        IU := Ix - 1;
                     when CLT =>
                        IL := Ix + 1;
                  end case;
                  exit when Result = CEQ or else IL > IU;
               end loop;

               if Result /= CEQ then
                  Is_FDL_RW := False;
               end if;
            end if;

            return Is_FDL_RW;
         end Check_FDL_RW;

         ---------------------------------------------------------------------------

         --  Returns the type of the annotation starting at the lookahead position in
         --  the line buffer.
         procedure Check_Anno_Type (Unfinished_Anno : in     Boolean;
                                    Anno_Kind       :    out Anno_Type)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.State;
         --#        in     LineManager.Curr_Line;
         --#        in out ErrorHandler.Error_Context;
         --#        in out SPARK_IO.File_Sys;
         --# derives Anno_Kind                  from CommandLineData.Content,
         --#                                         LineManager.Curr_Line,
         --#                                         Unfinished_Anno &
         --#         ErrorHandler.Error_Context,
         --#         SPARK_IO.File_Sys          from CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         LineManager.Curr_Line,
         --#                                         SPARK_IO.File_Sys;
         is
            type Element_Type is (End_Of_Line_Or_Text, Non_Character, RW_Or_Ident);

            Unused, Start_Posn, End_Posn : E_Strings.Lengths;
            Anno_Token                   : SPSymbols.SPTerminal;
            Next_Element                 : Element_Type;

            procedure Check_Next_Element
              (Start_Posn   : in     E_Strings.Lengths;
               End_Posn     :    out E_Strings.Lengths;
               Next_Element :    out Element_Type;
               Symbol       :    out SPSymbols.SPTerminal)
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     LexTokenManager.State;
            --#        in     LineManager.Curr_Line;
            --#        in out ErrorHandler.Error_Context;
            --#        in out SPARK_IO.File_Sys;
            --# derives End_Posn,
            --#         Next_Element               from LineManager.Curr_Line,
            --#                                         Start_Posn &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         LexTokenManager.State,
            --#                                         LineManager.Curr_Line,
            --#                                         SPARK_IO.File_Sys,
            --#                                         Start_Posn &
            --#         Symbol                     from CommandLineData.Content,
            --#                                         LineManager.Curr_Line,
            --#                                         Start_Posn;
            is
               Ch                               : Character;
               Local_End_Posn, Local_Start_Posn : E_Strings.Lengths;
            begin
               Local_Start_Posn := Start_Posn;
               loop
                  Ch := LineManager.Curr_Line.Conts (Local_Start_Posn);
                  exit when Ch = End_Of_Line or Ch = End_Of_Text or not Separator (Ch => Ch);
                  Local_Start_Posn := Local_Start_Posn + 1;
               end loop;
               Local_End_Posn := Local_Start_Posn;
               if Ch = End_Of_Line or Ch = End_Of_Text then
                  Next_Element := End_Of_Line_Or_Text;
                  Symbol       := SPSymbols.illegal_id; -- not used
               elsif not Ada.Characters.Handling.Is_Letter (Ch) then
                  Next_Element := Non_Character;
                  Symbol       := SPSymbols.illegal_id;  -- not used
               else
                  Next_Element := RW_Or_Ident;

                  -- Scan the next identifier, but then see if it's a reserved word.
                  while Letter_Or_Digit (Ch => LineManager.Curr_Line.Conts (Local_End_Posn)) or
                    LineManager.Curr_Line.Conts (Local_End_Posn) = '_' loop

                     Local_End_Posn := Local_End_Posn + 1;
                  end loop;

                  Local_End_Posn := Local_End_Posn - 1;
                  Check_Reserved (Start_Pos  => Local_Start_Posn,
                                  End_Pos    => Local_End_Posn,
                                  Look_Ahead => True,
                                  Token      => Symbol);
               end if;
               End_Posn := Local_End_Posn;

            end Check_Next_Element;

         begin
            Start_Posn := LineManager.Curr_Line.Lookahead_Pos + 1;

            Check_Next_Element (Start_Posn   => Start_Posn,
                                End_Posn     => End_Posn,
                                Next_Element => Next_Element,
                                Symbol       => Anno_Token);

            if Next_Element = End_Of_Line_Or_Text then
               Anno_Kind := No_Anno;
            elsif Next_Element = Non_Character then
               Anno_Kind := Other_Anno;
            else
               case Anno_Token is
                  when SPSymbols.RWmain_program |
                    SPSymbols.RWinherit      |
                    SPSymbols.RWown          |
                    SPSymbols.RWinitializes  |
                    SPSymbols.RWglobal       |
                    SPSymbols.RWderives      |
                    SPSymbols.RWdeclare      |
                    SPSymbols.RWpre          |
                    SPSymbols.RWpost         =>
                     Anno_Kind := Start_Anno;

                  when SPSymbols.RWreturn =>
                     if (Unfinished_Anno) then
                        -- Still in an annotation so it's not the start of a new one
                        Anno_Kind := Other_Anno;
                     else
                        -- New annotation
                        Anno_Kind := Start_Anno;
                     end if;
                  when SPSymbols.RWassert   |
                    SPSymbols.RWcheck    |
                    SPSymbols.RWtype     |
                    SPSymbols.RWsubtype  |
                    SPSymbols.RWfunction |
                    SPSymbols.RWaccept   |
                    SPSymbols.RWend      =>

                     Anno_Kind := Proof_Anno;

                  when SPSymbols.RWfor =>
                     -- do a second look ahead to check for "some" or "all"
                     Start_Posn := End_Posn + 1;
                     --# accept F, 10, Unused, "Unused not referenced here";
                     Check_Next_Element
                       (Start_Posn   => Start_Posn,
                        End_Posn     => Unused,
                        Next_Element => Next_Element,
                        Symbol       => Anno_Token);
                     --# end accept;
                     if Next_Element = RW_Or_Ident and then (Anno_Token = SPSymbols.RWsome or Anno_Token = SPSymbols.RWall) then
                        Anno_Kind := Other_Anno;
                     else
                        Anno_Kind := Proof_Anno;
                     end if;
                  when SPSymbols.RWhide =>
                     Anno_Kind := Hide_Anno;
                  when others =>
                     -- When a proof constant declaration occurs
                     -- interpreting --# as proof context is
                     -- handled by Hyph_Intro.
                     Anno_Kind := Other_Anno;
               end case;
            end if;
            --# accept F, 33, Unused, "Unused not referenced here";
         end Check_Anno_Type;

         -- Main implementation of the lexical analyser, common to both
         -- the Examiner and SPARKFormat.
         procedure Lex
           (Prog_Text    : in     SPARK_IO.File_Type;
            Allow_Dollar : in     Boolean;
            Token        :    out SPSymbols.SPTerminal;
            Lex_Val      :    out LexTokenManager.Lex_Value;
            Punct_Token  :    out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in out ErrorHandler.Error_Context;
         --#        in out LexTokenManager.State;
         --#        in out LineManager.Curr_Line;
         --#        in out SPARK_IO.File_Sys;
         --# derives ErrorHandler.Error_Context,
         --#         LexTokenManager.State,
         --#         Lex_Val,
         --#         LineManager.Curr_Line,
         --#         Punct_Token,
         --#         SPARK_IO.File_Sys,
         --#         Token                      from Allow_Dollar,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         LineManager.Curr_Line,
         --#                                         Prog_Text,
         --#                                         SPARK_IO.File_Sys;
            is separate;

         procedure Examiner_Lex
           (Prog_Text   : in     SPARK_IO.File_Type;
            Token       :    out SPSymbols.SPTerminal;
            Lex_Val     :    out LexTokenManager.Lex_Value;
            Punct_Token :    out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in out ErrorHandler.Error_Context;
         --#        in out LexTokenManager.State;
         --#        in out LineManager.Curr_Line;
         --#        in out SPARK_IO.File_Sys;
         --# derives ErrorHandler.Error_Context,
         --#         LexTokenManager.State,
         --#         Lex_Val,
         --#         LineManager.Curr_Line,
         --#         Punct_Token,
         --#         SPARK_IO.File_Sys,
         --#         Token                      from CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         LineManager.Curr_Line,
         --#                                         Prog_Text,
         --#                                         SPARK_IO.File_Sys;
         is
         begin
            Lex (Prog_Text    => Prog_Text,
                 Allow_Dollar => False,
                 Token        => Token,
                 Lex_Val      => Lex_Val,
                 Punct_Token  => Punct_Token);
         end Examiner_Lex;

         procedure SPARK_Format_Lex
           (Prog_Text   : in     SPARK_IO.File_Type;
            Token       :    out SPSymbols.SPTerminal;
            Lex_Val     :    out LexTokenManager.Lex_Value;
            Punct_Token :    out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in out ErrorHandler.Error_Context;
         --#        in out LexTokenManager.State;
         --#        in out LineManager.Curr_Line;
         --#        in out SPARK_IO.File_Sys;
         --# derives ErrorHandler.Error_Context,
         --#         LexTokenManager.State,
         --#         Lex_Val,
         --#         LineManager.Curr_Line,
         --#         Punct_Token,
         --#         SPARK_IO.File_Sys,
         --#         Token                      from CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         LineManager.Curr_Line,
         --#                                         Prog_Text,
         --#                                         SPARK_IO.File_Sys;
         is
         begin
            -- For SPARKFormat, we allow identifiers to begin with a '$'
            -- character so that SPARKFormat doesn't mess up annotations
            -- containing GNATPREP symbols
            Lex (Prog_Text    => Prog_Text,
                 Allow_Dollar => True,
                 Token        => Token,
                 Lex_Val      => Lex_Val,
                 Punct_Token  => Punct_Token);
         end SPARK_Format_Lex;

         function Similar_Tokens (Token1, Token2 : SPSymbols.SPTerminal) return Boolean is
            type Token_Type is (
                                Delimiter,
                                Reserved_Word,
                                Id,
                                Number,
                                Chars,
                                Other_Token);

            Token_Type1, Token_Type2 : Token_Type;
            Result                   : Boolean;

            function Type_Of_Token (Token : SPSymbols.SPTerminal) return Token_Type is
               Token_Kind : Token_Type;
            begin
               case Token is
                  when SPSymbols.ampersand         |
                    SPSymbols.apostrophe        |
                    SPSymbols.left_paren        |
                    SPSymbols.right_paren       |
                    SPSymbols.multiply          |
                    SPSymbols.plus              |
                    SPSymbols.comma             |
                    SPSymbols.minus             |
                    SPSymbols.point             |
                    SPSymbols.divide            |
                    SPSymbols.colon             |
                    SPSymbols.semicolon         |
                    SPSymbols.less_than         |
                    SPSymbols.equals            |
                    SPSymbols.greater_than      |
                    SPSymbols.vertical_bar      |
                    SPSymbols.arrow             |
                    SPSymbols.double_dot        |
                    SPSymbols.double_star       |
                    SPSymbols.becomes           |
                    SPSymbols.not_equal         |
                    SPSymbols.greater_or_equal  |
                    SPSymbols.less_or_equal     |
                    SPSymbols.left_label_paren  |
                    SPSymbols.right_label_paren |
                    SPSymbols.box               |
                    SPSymbols.implies           |
                    SPSymbols.is_equivalent_to  |
                    SPSymbols.tilde             |
                    SPSymbols.square_open       |
                    SPSymbols.square_close      |
                    SPSymbols.percent           =>
                     Token_Kind := Delimiter;
                  when SPSymbols.integer_number | SPSymbols.real_number | SPSymbols.based_integer | SPSymbols.based_real =>
                     Token_Kind := Number;
                  when SPSymbols.character_literal | SPSymbols.string_literal =>
                     Token_Kind := Chars;
                  when SPSymbols.identifier =>
                     Token_Kind := Id;
                  when SPSymbols.RWabort        |
                    SPSymbols.RWabs          |
                    SPSymbols.RWabstract     |
                    SPSymbols.RWaccept       |
                    SPSymbols.RWaccess       |
                    SPSymbols.RWaliased      |
                    SPSymbols.RWall          |
                    SPSymbols.RWand          |
                    SPSymbols.RWandthen      |
                    SPSymbols.RWany          |
                    SPSymbols.RWarray        |
                    SPSymbols.RWassert       |
                    SPSymbols.RWat           |
                    SPSymbols.RWbegin        |
                    SPSymbols.RWbody         |
                    SPSymbols.RWcase         |
                    SPSymbols.RWcheck        |
                    SPSymbols.RWconstant     |
                    SPSymbols.RWdeclare      |
                    SPSymbols.RWdelay        |
                    SPSymbols.RWdelta        |
                    SPSymbols.RWderives      |
                    SPSymbols.RWdigits       |
                    SPSymbols.RWdo           |
                    SPSymbols.RWelse         |
                    SPSymbols.RWelsif        |
                    SPSymbols.RWend          |
                    SPSymbols.RWentry        |
                    SPSymbols.RWexception    |
                    SPSymbols.RWexit         |
                    SPSymbols.RWfor          |
                    SPSymbols.RWforall       |
                    SPSymbols.RWforsome      |
                    SPSymbols.RWfrom         |
                    SPSymbols.RWfunction     |
                    SPSymbols.RWgeneric      |
                    SPSymbols.RWglobal       |
                    SPSymbols.RWgoto         |
                    SPSymbols.RWhide         |
                    SPSymbols.RWif           |
                    SPSymbols.RWin           |
                    SPSymbols.RWinherit      |
                    SPSymbols.RWinitializes  |
                    SPSymbols.RWis           |
                    SPSymbols.RWlimited      |
                    SPSymbols.RWloop         |
                    SPSymbols.RWmain_program |
                    SPSymbols.RWmod          |
                    SPSymbols.RWnew          |
                    SPSymbols.RWnot          |
                    SPSymbols.RWnotin        |
                    SPSymbols.RWnull         |
                    SPSymbols.RWof           |
                    SPSymbols.RWor           |
                    SPSymbols.RWorelse       |
                    SPSymbols.RWothers       |
                    SPSymbols.RWout          |
                    SPSymbols.RWown          |
                    SPSymbols.RWpackage      |
                    SPSymbols.RWpost         |
                    SPSymbols.RWpragma       |
                    SPSymbols.RWpre          |
                    SPSymbols.RWprivate      |
                    SPSymbols.RWprotected    |
                    SPSymbols.RWprocedure    |
                    SPSymbols.RWraise        |
                    SPSymbols.RWrange        |
                    SPSymbols.RWrecord       |
                    SPSymbols.RWrequeue      |
                    SPSymbols.RWrem          |
                    SPSymbols.RWrenames      |
                    SPSymbols.RWreturn       |
                    SPSymbols.RWreverse      |
                    SPSymbols.RWselect       |
                    SPSymbols.RWseparate     |
                    SPSymbols.RWsome         |
                    SPSymbols.RWsubtype      |
                    SPSymbols.RWtagged       |
                    SPSymbols.RWtask         |
                    SPSymbols.RWterminate    |
                    SPSymbols.RWthen         |
                    SPSymbols.RWtype         |
                    SPSymbols.RWuntil        |
                    SPSymbols.RWuse          |
                    SPSymbols.RWwhen         |
                    SPSymbols.RWwhile        |
                    SPSymbols.RWwith         |
                    SPSymbols.RWxor          |
                    -- Ada2005 reserved words
                    SPSymbols.RWinterface    |
                    SPSymbols.RWoverriding   |
                    SPSymbols.RWsynchronized =>
                     Token_Kind := Reserved_Word;
                  when others =>
                     Token_Kind := Other_Token;
               end case;
               return Token_Kind;
            end Type_Of_Token;

         begin
            Token_Type1 := Type_Of_Token (Token => Token1);
            Token_Type2 := Type_Of_Token (Token => Token2);
            case Token_Type1 is
               when Delimiter =>
                  case Token_Type2 is
                     when Delimiter =>
                        Result := True;
                     when others =>
                        Result := False;
                  end case;
               when Number =>
                  case Token_Type2 is
                     when Number =>
                        Result := True;
                     when others =>
                        Result := False;
                  end case;
               when Chars =>
                  case Token_Type2 is
                     when Chars =>
                        Result := True;
                     when others =>
                        Result := False;
                  end case;
               when Id =>
                  case Token_Type2 is
                     when Id =>
                        Result := True;
                     when Reserved_Word =>
                        Result := True;
                     when others =>
                        Result := False;
                  end case;
               when Reserved_Word =>
                  case Token_Type2 is
                     when Id =>
                        Result := True;
                     when Reserved_Word =>
                        Result := True;
                     when others =>
                        Result := False;
                  end case;
               when others =>
                  Result := False;
            end case;
            return Result;
         end Similar_Tokens;

end SparkLex;
