-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure Wf_Store_List (Node    : in out STree.SyntaxNode;
                         Scope   : in     Dictionary.Scopes;
                         E_Stack : in out ExpStack.ExpStackType) is
   Exp_Result, Type_Info : Exp_Record;
   Sym                   : Dictionary.Symbol;
   Error_Found           : Boolean := False;

   --------------------------------------------------------------

   procedure Chain_Up_To_Store (Node : in out STree.SyntaxNode)
   --# global in STree.Table;
   --# derives Node from *,
   --#                   STree.Table;
   is
   begin
      while Syntax_Node_Type (Node => Node) = SPSymbols.store_list loop
         -- ASSUME Node = store_list
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Node) = SPSymbols.store_list,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = store_list in Chain_Up_To_Store");
         Node := Parent_Node (Current_Node => Node);
      end loop;
      -- ASSUME Node = store
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.store,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = store in Chain_Up_To_Store");
   end Chain_Up_To_Store;

   --------------------------------------------------------------

   function Expression_Location (Node : STree.SyntaxNode) return LexTokenManager.Token_Position
   --# global in STree.Table;
   is
      Exp_Loc : LexTokenManager.Token_Position;
   begin
      -- ASSUME Node = store_list
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.store_list,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = store_list in Expression_Location");
      if Syntax_Node_Type (Node => Child_Node (Current_Node => Node)) = SPSymbols.annotation_expression then
         -- ASSUME Child_Node (Current_Node => Node) = annotation_expression
         Exp_Loc := Node_Position (Node => Node);
      elsif Syntax_Node_Type (Node => Child_Node (Current_Node => Node)) = SPSymbols.store_list then
         -- ASSUME Child_Node (Current_Node => Node) = store_list
         Exp_Loc := Node_Position (Node => Next_Sibling (Current_Node => Child_Node (Current_Node => Node)));
      else
         Exp_Loc := LexTokenManager.Null_Token_Position;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Child_Node (Current_Node => Node) = store_list OR annotation_expression in Expression_Location");
      end if;
      return Exp_Loc;
   end Expression_Location;

begin -- Wf_Store_List

   -- ASSUME Node = store_list
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.store_list,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = store_list in Wf_Store_List");

   ExpStack.Pop (Exp_Result, E_Stack);
   ExpStack.Pop (Type_Info, E_Stack);
   Sym := Type_Info.Other_Symbol;

   -- we must be dealing with an array update because Down_Wf_Store_List
   -- prunes at store_list node for records and so this procedure will
   -- never be called with a record (or any other illegal type)

   if Type_Info.Param_Count = Dictionary.GetNumberOfDimensions (Type_Info.Type_Symbol) then
      -- too many index expressions found
      Type_Info := UnknownSymbolRecord;
      ErrorHandler.Semantic_Error
        (Err_Num   => 93,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Node),
         Id_Str    => Dictionary.GetSimpleName (Sym));
      Chain_Up_To_Store (Node => Node);
   else -- still counting index expressions
      Type_Info.Param_Count := Type_Info.Param_Count + 1;
      if not Dictionary.CompatibleTypes
        (Scope,
         Dictionary.GetArrayIndex (Type_Info.Type_Symbol, Type_Info.Param_Count),
         Exp_Result.Type_Symbol) then
         Error_Found := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 38,
            Reference => ErrorHandler.No_Reference,
            Position  => Expression_Location (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   end if;

   Type_Info.Errors_In_Expression := Error_Found or else Type_Info.Errors_In_Expression or else Exp_Result.Errors_In_Expression;

   ExpStack.Push (Type_Info, E_Stack);

   -- ASSUME Node = store_list OR store
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.store_list
        or else Syntax_Node_Type (Node => Node) = SPSymbols.store,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = store_list OR store in Wf_Store_List");
end Wf_Store_List;
