-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure Wf_Procedure_Annotation
  (Node          : in STree.SyntaxNode;
   Current_Scope : in Dictionary.Scopes;
   Subprog_Sym   : in Dictionary.Symbol;
   First_Seen    : in Boolean) is
   Next_Node         : STree.SyntaxNode;
   Glob_Def_Err      : Boolean;
   Dependency_Found  : Boolean := False;
   Report_Node_Pos   : LexTokenManager.Token_Position;
   Global_Has_Errors : Boolean := False;

   -- look up table: if First_Seen then we are dealing with Abstract spec else Refined
   type Which_Abstractions is array (Boolean) of Dictionary.Abstractions;
   Which_Abstraction : constant Which_Abstractions :=
     Which_Abstractions'(False => Dictionary.IsRefined,
                         True  => Dictionary.IsAbstract);

begin
   -- ASSUME Node = procedure_annotation
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.procedure_annotation,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = procedure_annotation in Wf_Procedure_Annotation");

   Report_Node_Pos := Node_Position (Node => Parent_Node (Current_Node => Node));
   Glob_Def_Err    := False;
   Next_Node       := Child_Node (Current_Node => Node);
   -- ASSUME Next_Node = moded_global_definition OR dependency_relation OR declare_annotation OR procedure_constraint
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Next_Node) = SPSymbols.moded_global_definition
        or else Syntax_Node_Type (Node => Next_Node) = SPSymbols.dependency_relation
        or else Syntax_Node_Type (Node => Next_Node) = SPSymbols.declare_annotation
        or else Syntax_Node_Type (Node => Next_Node) = SPSymbols.procedure_constraint,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Next_Node = moded_global_definition OR dependency_relation OR declare_annotation OR procedure_constraint in Wf_Procedure_Annotation");
   if not (First_Seen or else Syntax_Node_Type (Node => Next_Node) = SPSymbols.moded_global_definition) then
      Glob_Def_Err      := True;
      Global_Has_Errors := True;
      ErrorHandler.Semantic_Error
        (Err_Num   => 87,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Parent_Node (Current_Node => Node)),
         Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
   end if;

   while Next_Node /= STree.NullNode loop
      if Syntax_Node_Type (Node => Next_Node) = SPSymbols.moded_global_definition then
         -- ASSUME Next_Node = moded_global_definition
         Report_Node_Pos := Node_Position (Node => Next_Node);
         wf_global_definition (Next_Node, Current_Scope, Subprog_Sym, First_Seen,
                               --to get
                               Glob_Def_Err);
         Global_Has_Errors := Global_Has_Errors or else Glob_Def_Err;
      elsif Syntax_Node_Type (Node => Next_Node) = SPSymbols.dependency_relation then
         -- ASSUME Next_Node = dependency_relation
         Dependency_Found := True;
         wf_dependency_relation (Next_Node, Current_Scope, Subprog_Sym, First_Seen, Glob_Def_Err);
      elsif Syntax_Node_Type (Node => Next_Node) = SPSymbols.declare_annotation then
         -- ASSUME Next_Node = declare_annotation
         Wf_Declare_Annotation (Node         => Next_Node,
                                Scope        => Current_Scope,
                                Task_Or_Proc => Subprog_Sym,
                                First_Seen   => First_Seen);
      elsif Syntax_Node_Type (Node => Next_Node) /= SPSymbols.procedure_constraint then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Next_Node = moded_global_definition OR dependency_relation OR declare_annotation OR procedure_constraint in Wf_Procedure_Annotation");
      end if;
      Next_Node := Next_Sibling (Next_Node);
   end loop;

   AddDerivesStreamEffects (Node        => Node,
                            Subprog_Sym => Subprog_Sym,
                            Abstraction => Which_Abstraction (First_Seen));

   -- mark subprogram as having incorrect signature if necessary
   if Global_Has_Errors then
      Dictionary.SetSubprogramSignatureNotWellformed (Which_Abstraction (First_Seen), Subprog_Sym);
   end if;

   if not Dependency_Found
     and then (CommandLineData.Content.Language_Profile = CommandLineData.SPARK83 or CommandLineData.Content.Do_Information_Flow)
   then
      Dictionary.SetSubprogramSignatureNotWellformed (Which_Abstraction (First_Seen), Subprog_Sym);
      ErrorHandler.Semantic_Error
        (Err_Num   => 501,
         Reference => ErrorHandler.No_Reference,
         Position  => Report_Node_Pos,
         Id_Str    => LexTokenManager.Null_String);
   end if;

end Wf_Procedure_Annotation;
