-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure wf_entry_body (Node      : in     STree.SyntaxNode;
                         Scope     : in out Dictionary.Scopes;
                         Next_Node :    out STree.SyntaxNode) is
   --Grammar:
   --       entry_body
   --            |
   --       entry spec --- identifier --- procedure_annotation --- subprogram_implementation
   --            |         (^the guard Boolean)
   --            |
   --       identifier --- formal_part
   --                          |
   --                      formal_part_rep
   --
   -- Actions:
   -- (1) First identifier must be name of (sole) entry declared in spec
   -- (2) Second identifier must be Boolean and must be protected element
   -- (3) If valid, add body, set up a local scope
   -- (4) wff annotation; note FirstSeen is False by definition; however, second anno may not be needed
   -- (5) Allow main tree walk to continue in new scope
   -- (6) Check end designator matches if not hidden

   EntrySym, GuardSym : Dictionary.Symbol;
   Entry_Spec_Node, Formal_Part_Node, Ident_Node, Guard_Node, Anno_Node, Subprogram_Implementation_Node, Pragma_Rep_Node, End_Node :
     STree.SyntaxNode;
   Hidden : Hidden_Class;

   -- check whether a second anno is needed, if it is present, and process it if necessary
   procedure CheckAnnotation (Anno_Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     EntrySym;
   --#        in     Node;
   --#        in     Scope;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         STree.Table                from *,
   --#                                         Anno_Node,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         EntrySym,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         Anno_Node,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         EntrySym,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Anno_Node,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         EntrySym,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State                  from *,
   --#                                         Anno_Node,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         EntrySym,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      ConstraintNode : STree.SyntaxNode;

      -- A second annotation is only needed if the abstract global anno contains
      -- the implicitly-declared "own variable" that shares the name of the type.
      function RequiresSecondAnnotation return Boolean
      --# global in Dictionary.Dict;
      --#        in EntrySym;
      is
         Result : Boolean := False;
         OwnVar : Dictionary.Symbol;
         It     : Dictionary.Iterator;
      begin
         OwnVar := Dictionary.GetProtectedTypeOwnVariable (Dictionary.GetRegion (Dictionary.GetScope (EntrySym)));
         It     := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, EntrySym);
         while not Dictionary.IsNullIterator (It) loop
            Result := Dictionary.CurrentSymbol (It) = OwnVar;
            exit when Result;

            It := Dictionary.NextSymbol (It);
         end loop;
         return Result;
      end RequiresSecondAnnotation;

      function HasSecondAnnotation return Boolean
      --# global in Anno_Node;
      --#        in STree.Table;
      is
      begin
         return Syntax_Node_Type (Node => Child_Node (Anno_Node)) = SPSymbols.moded_global_definition
           or else Syntax_Node_Type (Node => Child_Node (Anno_Node)) = SPSymbols.dependency_relation;
      end HasSecondAnnotation;

   begin -- CheckAnnotation
      if HasSecondAnnotation then
         if RequiresSecondAnnotation then
            -- wanted and present so process it
            Wf_Procedure_Annotation (Node          => Anno_Node,
                                     Current_Scope => Scope,
                                     Subprog_Sym   => EntrySym,
                                     First_Seen    => False);

            -- check for and handle second, concrete constraint
            ConstraintNode := Last_Sibling_Of (Start_Node => Child_Node (Anno_Node));
            if Child_Node (ConstraintNode) /= STree.NullNode then
               -- a constraint is present, so process it
               wf_procedure_constraint (Node      => ConstraintNode,
                                        Scope     => Dictionary.LocalScope (EntrySym),
                                        FirstSeen => False);
            end if;

            -- If not performing full information flow analysis then
            -- synthesize the "full" dependency clause using moded globals
            if not CommandLineData.Content.Do_Information_Flow then
               CreateFullSubProgDependency (Node, EntrySym, Dictionary.IsRefined);
            end if;

         else -- anno found but not needed
            ErrorHandler.Semantic_Error
              (Err_Num   => 155,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Anno_Node),
               Id_Str    => Dictionary.GetSimpleName (EntrySym));
            Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, EntrySym);
         end if;
      else -- no anno
         if RequiresSecondAnnotation then
            -- anno missing
            ErrorHandler.Semantic_Error
              (Err_Num   => 87,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => Dictionary.GetSimpleName (EntrySym));
            Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsRefined, EntrySym);
         end if;
      end if;
   end CheckAnnotation;

begin -- wf_entry_body

   -- ASSUME Node = entry_body
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.entry_body,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = entry_body in Wf_Entry_Body");
   Entry_Spec_Node := Child_Node (Current_Node => Node);
   -- ASSUME Entry_Spec_Node = entry_specification
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Entry_Spec_Node) = SPSymbols.entry_specification,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Entry_Spec_Node = entry_specification in Wf_Entry_Body");
   Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Entry_Spec_Node));
   -- ASSUME Ident_Node = identifier
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Ident_Node = identifier in Wf_Entry_Body");
   Formal_Part_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Entry_Spec_Node));
   Guard_Node       := Next_Sibling (Entry_Spec_Node);
   -- ASSUME Guard_Node = identifier
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Guard_Node) = SPSymbols.identifier,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Guard_Node = identifier in Wf_Entry_Body");
   Anno_Node := Next_Sibling (Guard_Node);
   -- ASSUME Anno_Node = procedure_annotation
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Anno_Node) = SPSymbols.procedure_annotation,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Anno_Node = procedure_annotation in Wf_Entry_Body");
   Subprogram_Implementation_Node := Next_Sibling (Anno_Node);
   -- ASSUME Subprogram_Implementation_Node = subprogram_implementation
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Subprogram_Implementation_Node) = SPSymbols.subprogram_implementation,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Subprogram_Implementation_Node = subprogram_implementation in Wf_Entry_Body");
   Pragma_Rep_Node := Child_Node (Subprogram_Implementation_Node);
   -- ASSUME Pragma_Rep_Node = pragma_rep
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Pragma_Rep_Node) = SPSymbols.pragma_rep,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Pragma_Rep_Node = pragma_rep in Wf_Entry_Body");
   End_Node := Last_Sibling_Of (Start_Node => Pragma_Rep_Node);

   Hidden := Body_Hidden_Class (Subprogram_Implementation_Node);

   EntrySym :=
     Dictionary.LookupItem (Name              => Node_Lex_String (Node => Ident_Node),
                            Scope             => Scope,
                            Context           => Dictionary.ProgramContext,
                            Full_Package_Name => False);
   -- Check that EntrySym is an entry declared in the spec.  Since we are looking up an identifier
   -- not a full, dotted name we can't find any other entry by mistake so a simple check is all that
   -- is needed.
   if Dictionary.IsEntry (EntrySym) then

      -- ASSUME Formal_Part_Node = formal_part OR NULL
      if Syntax_Node_Type (Node => Formal_Part_Node) = SPSymbols.formal_part then
         -- ASSUME Formal_Part_Node = formal_part
         STree.Set_Node_Lex_String (Sym  => EntrySym,
                                    Node => Ident_Node);
         Wf_Formal_Part
           (Node             => Formal_Part_Node,
            Current_Scope    => Scope,
            Subprog_Sym      => EntrySym,
            First_Occurrence => False,
            Context          => Dictionary.ProgramContext);
      elsif Formal_Part_Node = STree.NullNode then
         -- ASSUME Formal_Part_Node = NULL
         if Dictionary.GetNumberOfSubprogramParameters (EntrySym) /= 0 then
            ErrorHandler.Semantic_Error
              (Err_Num   => 152,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => Dictionary.GetSimpleName (EntrySym));
         else
            STree.Set_Node_Lex_String (Sym  => EntrySym,
                                       Node => Ident_Node);
         end if;
      else
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Formal_Part_Node = formal_part OR NULL in Wf_Entry_Body");
      end if;
      -- ok so far
      -- now check that the Guard is valid
      GuardSym :=
        Dictionary.LookupItem
        (Name              => Node_Lex_String (Node => Guard_Node),
         Scope             => Scope,
         Context           => Dictionary.ProgramContext,
         Full_Package_Name => False);
      if Dictionary.IsVariable (GuardSym)
        and then Dictionary.IsRefinement (Dictionary.GetProtectedTypeOwnVariable (Dictionary.GetRegion (Scope)), GuardSym)
        and then Dictionary.IsBooleanTypeMark (Dictionary.GetType (GuardSym)) then
         -- Guard is a protected element of type Boolean, which is OK

         -- store it for use in VCG
         Dictionary.SetSubprogramEntryBarrier (EntrySym, GuardSym);
         STree.Set_Node_Lex_String (Sym  => GuardSym,
                                    Node => Guard_Node);
         -- The entry is valid so far, it may be hidden or it may have a real sequence of statements
         if Hidden = All_Hidden then
            Dictionary.AddBody
              (CompilationUnit => EntrySym,
               Comp_Unit       => ContextManager.Ops.Current_Unit,
               TheBody         => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                       End_Position   => Node_Position (Node => Node)),
               Hidden          => True);
            ErrorHandler.Hidden_Text
              (Position => Node_Position (Node => End_Node),
               Unit_Str => Node_Lex_String (Node => Ident_Node),
               Unit_Typ => SPSymbols.subprogram_implementation);
            Next_Node := STree.NullNode; -- prune tree walk on hidden part
         else
            Dictionary.AddBody
              (CompilationUnit => EntrySym,
               Comp_Unit       => ContextManager.Ops.Current_Unit,
               TheBody         => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                       End_Position   => Node_Position (Node => Node)),
               Hidden          => False);

            -- check annotation
            CheckAnnotation (Anno_Node);

            if Hidden = Handler_Hidden then
               ErrorHandler.Hidden_Handler
                 (Position => Node_Position (Node => End_Node),
                  Unit_Str => Node_Lex_String (Node => Ident_Node),
                  Unit_Typ => SPSymbols.entry_body);
            end if;

            -- set up scope for rest of tree walk
            Scope := Dictionary.LocalScope (EntrySym);

            --set up next node for rest of tree walk
            Next_Node := Subprogram_Implementation_Node;
         end if;
      else
         -- Guard is not a protected element or is not Boolean
         ErrorHandler.Semantic_Error
           (Err_Num   => 994,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Guard_Node),
            Id_Str    => Node_Lex_String (Node => Guard_Node));
         Next_Node := STree.NullNode; -- prune tree walk on error

      end if;
   else
      -- not a valid Entry
      ErrorHandler.Semantic_Error
        (Err_Num   => 995,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Ident_Node),
         Id_Str    => Node_Lex_String (Node => Ident_Node));
      Next_Node := STree.NullNode; -- prune tree walk on error
   end if;

   -- check closing identifier
   if Syntax_Node_Type (Node => End_Node) = SPSymbols.designator then
      if LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Node_Lex_String (Node => Ident_Node),
         Lex_Str2 => Node_Lex_String (Node => Child_Node (End_Node))) /=
        LexTokenManager.Str_Eq then
         ErrorHandler.Semantic_Error
           (Err_Num   => 58,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => End_Node),
            Id_Str    => Node_Lex_String (Node => Ident_Node));
      end if;
   end if;
end wf_entry_body;
