-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with SLI;

separate (Sem.CompUnit)
procedure wf_dependency_relation
  (Node         : in STree.SyntaxNode;
   CurrentScope : in Dictionary.Scopes;
   SubprogSym   : in Dictionary.Symbol;
   FirstSeen    : in Boolean;
   GlobDefErr   : in Boolean) is
   It                                                           : STree.Iterator;
   NextNode                                                     : STree.SyntaxNode;
   Position                                                     : LexTokenManager.Token_Position;
   Abstraction                                                  : Dictionary.Abstractions;
   Error, SemanticErrorFound                                    : Boolean;
   ExportList, ImportList, ExportsInRelation, ImportsInRelation : SeqAlgebra.Seq;
   Null_Import_Node                                             : STree.SyntaxNode;  -- location of "derives null..."

   ------------------------------------------------------------------------

   procedure wf_dependency_clause
     (TheHeap     : in out Heap.HeapRecord;
      Node        : in     STree.SyntaxNode;
      SubprogSym  : in     Dictionary.Symbol;
      Abstraction : in     Dictionary.Abstractions;
      DoIFA       : in     Boolean;
      ErrorFound  :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     CurrentScope;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out ExportList;
   --#           out ImportList;
   --# derives Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         DoIFA,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         SubprogSym,
   --#                                         TheHeap &
   --#         ErrorFound                 from Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         DoIFA,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         SubprogSym,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         DoIFA,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         SubprogSym,
   --#                                         TheHeap &
   --#         ExportList,
   --#         ImportList                 from TheHeap;
   is
      Star_Node, Import_List_Node, Export_List_Node : STree.SyntaxNode;
      FirstValidExport                              : Dictionary.Symbol;
      LocalError                                    : Boolean;
      Star_Found                                    : Boolean;
      Null_List                                     : Boolean;

      -------------------------------------------------------------

      procedure ValidExportListAdd (Sym : in Dictionary.Symbol)
      --# global in     ExportList;
      --#        in out Statistics.TableUsage;
      --#        in out TheHeap;
      --# derives Statistics.TableUsage,
      --#         TheHeap               from *,
      --#                                    ExportList,
      --#                                    Sym,
      --#                                    TheHeap;
      is
      begin
         SeqAlgebra.AddMember (TheHeap, ExportList, Natural (Dictionary.SymbolRef (Sym)));
      end ValidExportListAdd;

      ---------------------------

      procedure GetFirstValidExport (Sym : out Dictionary.Symbol)
      --# global in ExportList;
      --#        in TheHeap;
      --# derives Sym from ExportList,
      --#                  TheHeap;
      is
         Member : SeqAlgebra.MemberOfSeq;
      begin
         Member := SeqAlgebra.FirstMember (TheHeap, ExportList);
         if SeqAlgebra.IsNullMember (Member) then
            Sym := Dictionary.NullSymbol;
         else
            Sym :=
              Dictionary.ConvertSymbolRef
              (ExaminerConstants.RefType (SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                                                      M        => Member)));
         end if;
      end GetFirstValidExport;

      -----------------------------------------------------------------------

      procedure UniqueExport
        (ExportSym  : in out Dictionary.Symbol;
         ErrorPos   : in     LexTokenManager.Token_Position;
         SubprogSym : in     Dictionary.Symbol)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LocalError;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from Abstraction,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         ErrorPos,
      --#                                         ExportSym,
      --#                                         LexTokenManager.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         SubprogSym &
      --#         ExportSym,
      --#         LocalError                 from *,
      --#                                         Abstraction,
      --#                                         Dictionary.Dict,
      --#                                         ExportSym,
      --#                                         SubprogSym;
      is
         NextExport : Dictionary.Iterator;

      begin
         NextExport := Dictionary.FirstExport (Abstraction, SubprogSym);
         while NextExport /= Dictionary.NullIterator loop
            if ExportSym = Dictionary.CurrentSymbol (NextExport) then
               LocalError := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 159,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Id_Str    => Dictionary.GetSimpleName (ExportSym));
               ExportSym := Dictionary.NullSymbol;
               exit;
            end if;
            NextExport := Dictionary.NextSymbol (NextExport);
         end loop;
      end UniqueExport;

      -----------------------------------------------------------------------

      -- Uses an explicit import list.  The import is passed in,
      -- if it is already in the import list an error is reported and a null symbol
      -- returned; else it is added to the list.  Imports by virtue of * never
      -- appear in the list and do not give rise to duplicate errors.

      procedure UniqueImport (ImportSym : in out Dictionary.Symbol;
                              ErrorPos  : in     LexTokenManager.Token_Position)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     ImportList;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LocalError;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out TheHeap;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         ErrorPos,
      --#                                         ImportList,
      --#                                         ImportSym,
      --#                                         LexTokenManager.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         TheHeap &
      --#         ImportSym,
      --#         LocalError,
      --#         Statistics.TableUsage,
      --#         TheHeap                    from *,
      --#                                         ImportList,
      --#                                         ImportSym,
      --#                                         TheHeap;
      is
         ImportRep : Natural;

      begin
         ImportRep := Natural (Dictionary.SymbolRef (ImportSym));
         if SeqAlgebra.IsMember (TheHeap, ImportList, ImportRep) then
            LocalError := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 160,
               Reference => ErrorHandler.No_Reference,
               Position  => ErrorPos,
               Id_Str    => Dictionary.GetSimpleName (ImportSym));
            ImportSym := Dictionary.NullSymbol;
         else
            SeqAlgebra.AddMember (TheHeap, ImportList, ImportRep);
         end if;
      end UniqueImport;

      -----------------------------------------------------------------------

      -- Similar to UniqueImport but this checks that the import has not appeared as
      -- an import anywhere in the dependency relation whereas UniqueImport only checks
      -- it is not duplicated in the current clause. CheckNotImportedBefore is for use
      -- in handling "derives null from ..." cases where the imports must not have
      -- been imported before.
      procedure CheckNotImportedBefore (ImportSym : in out Dictionary.Symbol;
                                        ErrorPos  : in     LexTokenManager.Token_Position)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     ImportList;
      --#        in     LexTokenManager.State;
      --#        in     SubprogSym;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LocalError;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out TheHeap;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from Abstraction,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         ErrorPos,
      --#                                         ImportSym,
      --#                                         LexTokenManager.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         SubprogSym &
      --#         ImportSym,
      --#         LocalError                 from *,
      --#                                         Abstraction,
      --#                                         Dictionary.Dict,
      --#                                         ImportSym,
      --#                                         SubprogSym &
      --#         Statistics.TableUsage,
      --#         TheHeap                    from *,
      --#                                         Abstraction,
      --#                                         Dictionary.Dict,
      --#                                         ImportList,
      --#                                         ImportSym,
      --#                                         SubprogSym,
      --#                                         TheHeap;
      is
         It : Dictionary.Iterator;
      begin --CheckNotImportedBefore
         It := Dictionary.FirstImport (Abstraction, SubprogSym);
         while not Dictionary.IsNullIterator (It) loop
            if Dictionary.CurrentSymbol (It) = ImportSym then --duplicate
               LocalError := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 160,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Id_Str    => Dictionary.GetSimpleName (ImportSym));
               ImportSym := Dictionary.NullSymbol;
            else
               SeqAlgebra.AddMember (TheHeap, ImportList, Natural (Dictionary.SymbolRef (ImportSym)));
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      end CheckNotImportedBefore;

      -----------------------------------------------------------------------

      procedure CheckExportMode
        (ExportSym  : in Dictionary.Symbol;
         SubprogSym : in Dictionary.Symbol;
         ErrorPos   : in LexTokenManager.Token_Position)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     CurrentScope;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LocalError;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from Abstraction,
      --#                                         CommandLineData.Content,
      --#                                         CurrentScope,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         ErrorPos,
      --#                                         ExportSym,
      --#                                         LexTokenManager.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         SubprogSym &
      --#         LocalError                 from *,
      --#                                         Abstraction,
      --#                                         Dictionary.Dict,
      --#                                         ExportSym,
      --#                                         SubprogSym;
      --#

      is
         Mode : Dictionary.Modes;

      begin
         if Dictionary.IsFormalParameter (SubprogSym, ExportSym) then
            Mode := Dictionary.GetSubprogramParameterMode (ExportSym);
            if Mode = Dictionary.InMode or Mode = Dictionary.DefaultMode then
               LocalError := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 161,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Id_Str    => Dictionary.GetSimpleName (ExportSym));
            end if;

         else -- check mode, only "IN" is unacceptable
            Mode := Dictionary.GetGlobalMode (Abstraction, SubprogSym, ExportSym);
            if Mode = Dictionary.InMode then
               LocalError := True;
               ErrorHandler.Semantic_Error_Sym
                 (Err_Num   => 502,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Sym       => ExportSym,
                  Scope     => CurrentScope);
            end if;
         end if;
      end CheckExportMode;

      -----------------------------------------------------------------------

      -- We check that imports are of the correct mode and (at the
      -- main progam level) that they are initialized.  If the import list
      -- includes a * self reference we do not need to make these checks where the
      -- import being checked is also an export since the checks will already have
      -- been made at the * node.  This function is used by CheckImportMode and
      -- CheckImportInit because the check needed is the same in both cases.

      function ExportImportedByStar (ImportSym  : Dictionary.Symbol;
                                     Star_Found : Boolean) return Boolean
      --# global in ExportList;
      --#        in TheHeap;
      is
      begin
         return Star_Found and then SeqAlgebra.IsMember (TheHeap, ExportList, Natural (Dictionary.SymbolRef (ImportSym)));
      end ExportImportedByStar;

      -----------------------------------------------------------------------

      -- ignore checks if the import is an export which has already been
      -- imported by use of the * abbreviation; this is because checks will already
      -- have been carried out at the * node.

      procedure CheckImportMode
        (ImportSym  : in Dictionary.Symbol;
         SubprogSym : in Dictionary.Symbol;
         ErrorPos   : in LexTokenManager.Token_Position;
         Star_Found : in Boolean)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     CurrentScope;
      --#        in     Dictionary.Dict;
      --#        in     ExportList;
      --#        in     LexTokenManager.State;
      --#        in     TheHeap;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LocalError;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from Abstraction,
      --#                                         CommandLineData.Content,
      --#                                         CurrentScope,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         ErrorPos,
      --#                                         ExportList,
      --#                                         ImportSym,
      --#                                         LexTokenManager.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         Star_Found,
      --#                                         SubprogSym,
      --#                                         TheHeap &
      --#         LocalError                 from *,
      --#                                         Abstraction,
      --#                                         Dictionary.Dict,
      --#                                         ExportList,
      --#                                         ImportSym,
      --#                                         Star_Found,
      --#                                         SubprogSym,
      --#                                         TheHeap;
      is

      begin
         if not ExportImportedByStar (ImportSym, Star_Found) then
            if Dictionary.IsFormalParameter (SubprogSym, ImportSym) then
               if Dictionary.GetSubprogramParameterMode (ImportSym) = Dictionary.OutMode then
                  LocalError := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 162,
                     Reference => ErrorHandler.No_Reference,
                     Position  => ErrorPos,
                     Id_Str    => Dictionary.GetSimpleName (ImportSym));
               end if;

            elsif -- check mode, only "OUT" is unacceptable
              Dictionary.GetGlobalMode (Abstraction, SubprogSym, ImportSym) = Dictionary.OutMode then
               LocalError := True;
               ErrorHandler.Semantic_Error_Sym
                 (Err_Num   => 503,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Sym       => ImportSym,
                  Scope     => CurrentScope);
            end if;
            -- check here that we are not trying to import a mode out stream variable
            if Dictionary.GetOwnVariableOrConstituentMode (ImportSym) = Dictionary.OutMode then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 714,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Id_Str    => Dictionary.GetSimpleName (ImportSym));
               LocalError := True;
            end if;
         end if;
      end CheckImportMode;

      -----------------------------------------------------------------------
      -- ignore checks if the import is an export which has already been
      -- imported by use of the * abbreviation; this is because checks will already
      -- have been carried out at the * node.

      procedure CheckImportInit
        (ImportNode            : in STree.SyntaxNode;
         ImportSym, SubprogSym : in Dictionary.Symbol;
         Star_Found            : in Boolean)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     CurrentScope;
      --#        in     Dictionary.Dict;
      --#        in     ExportList;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in     TheHeap;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LocalError;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from Abstraction,
      --#                                         CommandLineData.Content,
      --#                                         CurrentScope,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         ExportList,
      --#                                         ImportNode,
      --#                                         ImportSym,
      --#                                         LexTokenManager.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         Star_Found,
      --#                                         STree.Table,
      --#                                         SubprogSym,
      --#                                         TheHeap &
      --#         LocalError                 from *,
      --#                                         Abstraction,
      --#                                         Dictionary.Dict,
      --#                                         ExportList,
      --#                                         ImportSym,
      --#                                         Star_Found,
      --#                                         SubprogSym,
      --#                                         TheHeap;
      is

      begin
         if not ExportImportedByStar (ImportSym, Star_Found) then

            if Dictionary.IsTaskType (SubprogSym)
              and then Dictionary.IsOwnVariable (ImportSym)
              and then not Dictionary.GetOwnVariableProtected (ImportSym)
              and then Dictionary.GetOwnVariableMode (ImportSym) /= Dictionary.InMode
              and then Dictionary.PartitionElaborationPolicyIsConcurrent then
               -- This is an unprotected import to a task that is not of mode in.
               -- The elaboration policy is concurrent and hence we cannot
               -- guarantee the import is initialised before it is used.
               LocalError := True;
               ErrorHandler.Semantic_Error_Sym
                 (Err_Num   => 958,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => ImportNode),
                  Sym       => ImportSym,
                  Scope     => CurrentScope);

            elsif (Dictionary.IsMainProgram (SubprogSym) or Dictionary.IsTaskType (SubprogSym)) then
               if Dictionary.IsGlobalVariable (Abstraction, SubprogSym, ImportSym)
                 and then Dictionary.IsOwnVariable (ImportSym)
                 and then not Dictionary.OwnVariableIsInitialized (ImportSym)
                 and then Dictionary.GetOwnVariableOrConstituentMode (ImportSym) = Dictionary.DefaultMode then
                  LocalError := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 167,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => ImportNode),
                     Id_Str    => Dictionary.GetSimpleName (ImportSym));
               end if;
            end if;
         end if;
      end CheckImportInit;

      -----------------------------------------------------------------------

      procedure DoExportList (Node       : in STree.SyntaxNode;
                              SubprogSym : in Dictionary.Symbol)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     CurrentScope;
      --#        in     ExportList;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LocalError;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#        in out TheHeap;
      --# derives Dictionary.Dict,
      --#         LocalError,
      --#         STree.Table                from *,
      --#                                         Abstraction,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         SubprogSym &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from Abstraction,
      --#                                         CommandLineData.Content,
      --#                                         CurrentScope,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         SubprogSym &
      --#         Statistics.TableUsage,
      --#         TheHeap                    from *,
      --#                                         Abstraction,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ExportList,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         SubprogSym,
      --#                                         TheHeap;
      is
         Sym        : Dictionary.Symbol;
         It         : STree.Iterator;
         ExportNode : STree.SyntaxNode;
         Unused     : Boolean;
      begin
         It := Find_First_Node (Node_Kind    => SPSymbols.entire_variable,
                                From_Root    => Node,
                                In_Direction => STree.Down);

         while not STree.IsNull (It) loop

            ExportNode := Get_Node (It => It);

            --# accept Flow, 10, Unused, "Expected ineffective assignment";
            wf_entire_variable (ExportNode, Dictionary.LocalScope (SubprogSym), In_Derives_Export_List,
                                --to get
                                Sym, Unused);
            --# end accept;
            LocalError := LocalError or (Sym = Dictionary.NullSymbol);
            if Sym /= Dictionary.NullSymbol then
               UniqueExport (Sym, Node_Position (Node => ExportNode), SubprogSym);
            end if;

            -- check that the export is not a stream variable of mode in
            if Sym /= Dictionary.NullSymbol then
               if Dictionary.GetOwnVariableOrConstituentMode (Sym) = Dictionary.InMode then
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 713,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => ExportNode),
                     Id_Str    => Dictionary.GetSimpleName (Sym));
                  LocalError := True;
                  Sym        := Dictionary.NullSymbol;
               end if;
            end if;

            if Dictionary.IsTaskType (SubprogSym)
              and then Dictionary.IsOwnVariable (Sym)
              and then not Dictionary.GetOwnVariableProtected (Sym)
              and then Dictionary.OwnVariableIsInitialized (Sym)
              and then Dictionary.PartitionElaborationPolicyIsConcurrent then
               -- This export is defined during elaboration as well by this task.
               -- The order is non-deterministic. Hence we must prohibit the export
               -- being initialised during elaboration.
               ErrorHandler.Semantic_Error_Sym
                 (Err_Num   => 959,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => ExportNode),
                  Sym       => Sym,
                  Scope     => CurrentScope);
            end if;

            if Sym /= Dictionary.NullSymbol then
               CheckExportMode (Sym, SubprogSym, Node_Position (Node => ExportNode));
               Dictionary.AddExport
                 (Abstraction     => Abstraction,
                  TheProcedure    => SubprogSym,
                  TheExport       => Sym,
                  ExportReference => Dictionary.Location'(Start_Position => Node_Position (Node => ExportNode),
                                                          End_Position   => Node_Position (Node => ExportNode)),
                  Annotation      => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                          End_Position   => Node_Position (Node => Node)));
               ValidExportListAdd (Sym);
            end if;
            It := STree.NextNode (It);
         end loop;

         --# accept Flow, 33, Unused, "Expected to be neither referenced nor exported";
      end DoExportList;

      -----------------------------------------------------------------------

      procedure Do_Import_List (Node      : in STree.SyntaxNode;
                                Null_List : in Boolean)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     CurrentScope;
      --#        in     DoIFA;
      --#        in     ExportList;
      --#        in     ImportList;
      --#        in     LexTokenManager.State;
      --#        in     Star_Found;
      --#        in     SubprogSym;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LocalError;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#        in out TheHeap;
      --#           out FirstValidExport;
      --# derives Dictionary.Dict,
      --#         LocalError,
      --#         Statistics.TableUsage,
      --#         STree.Table,
      --#         TheHeap                    from *,
      --#                                         Abstraction,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         DoIFA,
      --#                                         ExportList,
      --#                                         ImportList,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         Null_List,
      --#                                         Star_Found,
      --#                                         STree.Table,
      --#                                         SubprogSym,
      --#                                         TheHeap &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from Abstraction,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         CurrentScope,
      --#                                         Dictionary.Dict,
      --#                                         DoIFA,
      --#                                         ErrorHandler.Error_Context,
      --#                                         ExportList,
      --#                                         ImportList,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         Null_List,
      --#                                         SPARK_IO.File_Sys,
      --#                                         Star_Found,
      --#                                         STree.Table,
      --#                                         SubprogSym,
      --#                                         TheHeap &
      --#         FirstValidExport           from ExportList,
      --#                                         TheHeap;
      is
         Sym        : Dictionary.Symbol;
         It         : STree.Iterator;
         ImportNode : STree.SyntaxNode;
         Unused     : Boolean;

         --------------------------------------------------------------------
         procedure AddDependencies
           (Abstraction     : in Dictionary.Abstractions;
            TheProcedure    : in Dictionary.Symbol;
            FirstExport     : in Dictionary.Symbol;
            TheImport       : in Dictionary.Symbol;
            ImportReference : in Dictionary.Location)
         --# global in     ContextManager.Ops.Unit_Stack;
         --#        in     ExportList;
         --#        in     LexTokenManager.State;
         --#        in     Star_Found;
         --#        in     TheHeap;
         --#        in out Dictionary.Dict;
         --#        in out SPARK_IO.File_Sys;
         --# derives Dictionary.Dict   from *,
         --#                                Abstraction,
         --#                                ContextManager.Ops.Unit_Stack,
         --#                                ExportList,
         --#                                FirstExport,
         --#                                ImportReference,
         --#                                Star_Found,
         --#                                TheHeap,
         --#                                TheImport,
         --#                                TheProcedure &
         --#         SPARK_IO.File_Sys from *,
         --#                                Abstraction,
         --#                                ContextManager.Ops.Unit_Stack,
         --#                                Dictionary.Dict,
         --#                                ExportList,
         --#                                FirstExport,
         --#                                ImportReference,
         --#                                LexTokenManager.State,
         --#                                Star_Found,
         --#                                TheHeap,
         --#                                TheImport,
         --#                                TheProcedure;
         is
            Member        : SeqAlgebra.MemberOfSeq;
            CurrentExport : Dictionary.Symbol;

         begin
            if Star_Found then
               Member := SeqAlgebra.FirstMember (TheHeap, ExportList);
               while not SeqAlgebra.IsNullMember (Member) loop
                  CurrentExport :=
                    Dictionary.ConvertSymbolRef
                    (ExaminerConstants.RefType (SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                                                            M        => Member)));
                  if TheImport /= CurrentExport then    -- don't add if already added by *
                     Dictionary.AddDependency
                       (Abstraction     => Abstraction,
                        Comp_Unit       => ContextManager.Ops.Current_Unit,
                        TheProcedure    => TheProcedure,
                        TheExport       => CurrentExport,
                        TheImport       => TheImport,
                        ImportReference => ImportReference);
                  end if;
                  Member := SeqAlgebra.NextMember (TheHeap, Member);
               end loop;

            else --only need to copy it to first export (CopyImports will do rest)
               Dictionary.AddDependency
                 (Abstraction     => Abstraction,
                  Comp_Unit       => ContextManager.Ops.Current_Unit,
                  TheProcedure    => TheProcedure,
                  TheExport       => FirstExport,
                  TheImport       => TheImport,
                  ImportReference => ImportReference);
            end if;
         end AddDependencies;

         -------------------

      begin -- Do_Import_List
         GetFirstValidExport (FirstValidExport); -- NullSymbol if doing a null derives

         It := Find_First_Node (Node_Kind    => SPSymbols.entire_variable,
                                From_Root    => Node,
                                In_Direction => STree.Down);

         while not STree.IsNull (It) loop

            ImportNode := Get_Node (It => It);
            --# accept Flow, 10, Unused, "Expected ineffective assignment";
            wf_entire_variable (ImportNode, Dictionary.LocalScope (SubprogSym), In_Derives_Import_List,
                                --to get
                                Sym, Unused);
            --# end accept;
            LocalError := LocalError or (Sym = Dictionary.NullSymbol);
            if Sym /= Dictionary.NullSymbol and then FirstValidExport /= Dictionary.NullSymbol then
               -- also covers Null_List case
               UniqueImport (Sym, Node_Position (Node => ImportNode));
            end if;

            -- new check, similar to UniqueImport except that for a null import clause
            -- we require that the variable has not appeared as an import anywhere else rather
            -- than just in the current clause
            if Null_List and then Sym /= Dictionary.NullSymbol then
               CheckNotImportedBefore (Sym, Node_Position (Node => ImportNode));
            end if;

            if Sym /= Dictionary.NullSymbol then  -- we have valid import so far
               CheckImportMode (Sym, SubprogSym, Node_Position (Node => ImportNode), Star_Found);
               CheckImportInit (ImportNode, Sym, SubprogSym, Star_Found);

               -- One of two things must happen to the valid import we have at this point.
               -- If we are doing IFA and we have valid export to associated it with we do so;
               -- however, if we are not doing IFA or there is no valid export we just
               -- mark the import in the dictionary as being an import usign the force call
               --# accept Flow, 41, "Expected stable expression";
               if DoIFA
                 and then --  stable index OK
                 FirstValidExport /= Dictionary.NullSymbol then
                  --# end accept;
                  AddDependencies
                    (Abstraction,
                     SubprogSym,
                     FirstValidExport,
                     Sym,
                     Dictionary.Location'(Start_Position => Node_Position (Node => ImportNode),
                                          End_Position   => Node_Position (Node => ImportNode)));
               else
                  -- No valid export (including null derives case) to associate with  or else we
                  -- are just doing DFA.  In either case mark as import to avoid
                  -- knock-on errors
                  Dictionary.ForceImport
                    (Abstraction,
                     SubprogSym,
                     Sym,
                     Dictionary.Location'(Start_Position => Node_Position (Node => ImportNode),
                                          End_Position   => Node_Position (Node => ImportNode)));
               end if;
            end if;
            It := STree.NextNode (It);
         end loop;
         --# accept Flow, 33, Unused, "Expected to be neither referenced nor exported";
      end Do_Import_List;

      -----------------------------------------------------------------------

      procedure DoSelfReferences (Star_Node  : in STree.SyntaxNode;
                                  SubprogSym : in Dictionary.Symbol)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     CurrentScope;
      --#        in     ExportList;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in     TheHeap;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LocalError;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dictionary.Dict,
      --#         LocalError                 from *,
      --#                                         Abstraction,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ExportList,
      --#                                         Star_Node,
      --#                                         STree.Table,
      --#                                         SubprogSym,
      --#                                         TheHeap &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from Abstraction,
      --#                                         CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         CurrentScope,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         ExportList,
      --#                                         LexTokenManager.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         Star_Node,
      --#                                         STree.Table,
      --#                                         SubprogSym,
      --#                                         TheHeap;
      is
         Member    : SeqAlgebra.MemberOfSeq;
         TheExport : Dictionary.Symbol;

         ------------------------------------------------------------------

      begin --DoSelfReferences
         Member := SeqAlgebra.FirstMember (TheHeap, ExportList);
         while not SeqAlgebra.IsNullMember (Member) loop
            TheExport :=
              Dictionary.ConvertSymbolRef
              (ExaminerConstants.RefType (SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                                                      M        => Member)));
            CheckImportMode (TheExport, SubprogSym, Node_Position (Node => Star_Node), False);
            CheckImportInit (Star_Node, TheExport, SubprogSym, False);
            Dictionary.AddDependency
              (Abstraction     => Abstraction,
               Comp_Unit       => ContextManager.Ops.Current_Unit,
               TheProcedure    => SubprogSym,
               TheExport       => TheExport,
               TheImport       => TheExport,
               ImportReference => Dictionary.Location'(Start_Position => Node_Position (Node => Star_Node),
                                                       End_Position   => Node_Position (Node => Star_Node)));
            Member := SeqAlgebra.NextMember (TheHeap, Member);
         end loop;
      end DoSelfReferences;

      -----------------------------------------------------------------------

      -- This procedure does checks on * node when no IFA selected
      procedure AppendSelfReferences (Star_Node  : in STree.SyntaxNode;
                                      SubprogSym : in Dictionary.Symbol)
      --# global in     Abstraction;
      --#        in     CommandLineData.Content;
      --#        in     CurrentScope;
      --#        in     Dictionary.Dict;
      --#        in     ExportList;
      --#        in     ImportList;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LocalError;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out TheHeap;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from Abstraction,
      --#                                         CommandLineData.Content,
      --#                                         CurrentScope,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         ExportList,
      --#                                         LexTokenManager.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         Star_Node,
      --#                                         STree.Table,
      --#                                         SubprogSym,
      --#                                         TheHeap &
      --#         LocalError                 from *,
      --#                                         Abstraction,
      --#                                         Dictionary.Dict,
      --#                                         ExportList,
      --#                                         SubprogSym,
      --#                                         TheHeap &
      --#         Statistics.TableUsage,
      --#         TheHeap                    from *,
      --#                                         ExportList,
      --#                                         ImportList,
      --#                                         TheHeap;
      is
         Member    : SeqAlgebra.MemberOfSeq;
         TheExport : Dictionary.Symbol;

      begin --AppendSelfReferences
         Member := SeqAlgebra.FirstMember (TheHeap, ExportList);
         while not SeqAlgebra.IsNullMember (Member) loop
            TheExport :=
              Dictionary.ConvertSymbolRef
              (ExaminerConstants.RefType (SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                                                      M        => Member)));
            CheckImportMode (TheExport, SubprogSym, Node_Position (Node => Star_Node), False);
            CheckImportInit (Star_Node, TheExport, SubprogSym, False);
            Member := SeqAlgebra.NextMember (TheHeap, Member);
         end loop;
         SeqAlgebra.AugmentSeq (TheHeap, ImportList, ExportList);
      end AppendSelfReferences;

      -----------------------------------------------------------------------

      procedure CopyImports (SubprogSym : in Dictionary.Symbol)
      --# global in     Abstraction;
      --#        in     ExportList;
      --#        in     FirstValidExport;
      --#        in     TheHeap;
      --#        in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              ExportList,
      --#                              FirstValidExport,
      --#                              SubprogSym,
      --#                              TheHeap;
      --#
      --#
      --#
      --#
      --#
      is
         Member : SeqAlgebra.MemberOfSeq;

      begin
         Member := SeqAlgebra.FirstMember (TheHeap, ExportList);
         if not SeqAlgebra.IsNullMember (Member) then
            --there is at leat one valid export, we want to loop through rest
            loop
               Member := SeqAlgebra.NextMember (TheHeap, Member);
               exit when SeqAlgebra.IsNullMember (Member);

               Dictionary.CopyDependencyList
                 (Abstraction,
                  SubprogSym,
                  FirstValidExport,
                  Dictionary.ConvertSymbolRef
                    (ExaminerConstants.RefType (SeqAlgebra.Value_Of_Member (The_Heap => TheHeap,
                                                                            M        => Member))));
            end loop;
         end if;
      end CopyImports;

      -----------------------------------------------------------------------

      procedure CheckStar (Node               : in     STree.SyntaxNode;
                           Found              :    out Boolean;
                           Star_Node, ImpNode :    out STree.SyntaxNode)
      --# global in STree.Table;
      --# derives Found,
      --#         ImpNode,
      --#         Star_Node from Node,
      --#                        STree.Table;
      is
         ImpNodeLocal : STree.SyntaxNode;
      begin
         ImpNodeLocal := Next_Sibling (Child_Node (Node));
         ImpNode      := ImpNodeLocal;
         if Syntax_Node_Type (Node => Child_Node (ImpNodeLocal)) = SPSymbols.multiply then
            Found     := True;
            Star_Node := Child_Node (ImpNodeLocal);
         else
            Found     := False;
            Star_Node := STree.NullNode;
         end if;
      end CheckStar;

   begin  -- wf_dependency_clause

      --  For each exported variable E : E is a visible entire
      --  variable; occurrence of E as an export is unique (Section
      --  6.1.2 Rule 6); mode of E is in out or out (Section 6.1.2
      --  Rule 2); for each imported variable I : I is a visible
      --  entire variable; occurrence of I as an import is unique
      --  (Section 6.1.2 Rule 6); mode of I is in or in out (Section
      --  6.1.2 Rule 1); if dependency clause belongs to main_program,
      --  then I is initialized by the package of which it is an own
      --  variable;

      SeqAlgebra.CreateSeq (TheHeap, ExportList);
      SeqAlgebra.CreateSeq (TheHeap, ImportList);
      LocalError := False;

      -- ASSUME Node = dependency_clause OR null_import_list
      if Syntax_Node_Type (Node => Node) = SPSymbols.null_import_list then
         -- ASSUME Node = null_import_list

         -- we are dealing with " derives null from ..." so there are no stars to worry
         -- about and we don't have to go looking for the node holding the list of imports
         -- because that is what has been passed in
         Null_List        := True;
         Star_Found       := False;
         Star_Node        := STree.NullNode; -- not used but removes apparent DFA
         Import_List_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Node));
         -- ASSUME Import_List_Node = dependency_clause_optrep
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Import_List_Node) = SPSymbols.dependency_clause_optrep,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Import_List_Node = dependency_clause_optrep in Wf_Dependency_Clause");
         -- Add NullVariable as an export to provide a data sink for the imported items
         Dictionary.AddGlobalVariable
           (Abstraction       => Abstraction,
            Subprogram        => SubprogSym,
            Variable          => Dictionary.GetNullVariable,
            Mode              => Dictionary.OutMode,
            PrefixNeeded      => False,
            Comp_Unit         => ContextManager.Ops.Current_Unit,
            VariableReference => Dictionary.Location'(Start_Position => Node_Position (Node => Import_List_Node),
                                                      End_Position   => Node_Position (Node => Import_List_Node)));
         Dictionary.AddExport
           (Abstraction     => Abstraction,
            TheProcedure    => SubprogSym,
            TheExport       => Dictionary.GetNullVariable,
            ExportReference => Dictionary.Location'(Start_Position => Node_Position (Node => Import_List_Node),
                                                    End_Position   => Node_Position (Node => Import_List_Node)),
            Annotation      => Dictionary.Location'(Start_Position => Node_Position (Node => Import_List_Node),
                                                    End_Position   => Node_Position (Node => Import_List_Node)));
         ValidExportListAdd (Dictionary.GetNullVariable);
      elsif Syntax_Node_Type (Node => Node) = SPSymbols.dependency_clause then
         -- ASSUME Node = dependency_clause

         Null_List := False;
         -- it's an ordinary dependency clause
         Export_List_Node := Child_Node (Node);
         -- ASSUME Export_List_Node = dependency_clause_optrep
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Export_List_Node) = SPSymbols.dependency_clause_optrep,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Export_List_Node = dependency_clause_optrep in Wf_Dependency_Clause");
         CheckStar (Node,
                    --to get
                    Star_Found, Star_Node, Import_List_Node);
         DoExportList (Export_List_Node, SubprogSym);
      else
         Null_List        := False;
         Star_Found       := False;
         Star_Node        := STree.NullNode;
         Import_List_Node := STree.NullNode;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = dependency_clause OR null_import_list in Wf_Dependency_Clause");
      end if;

      --# accept Flow, 10, FirstValidExport, "Expected ineffective assignment";
      if DoIFA then --DoIFA can only be False here in SPARK 83 DFA mode
         if Star_Found then
            DoSelfReferences (Star_Node, SubprogSym);
            Do_Import_List (Node      => Import_List_Node,
                            Null_List => Null_List); -- ineffective FirstValidExport OK
         else
            Do_Import_List (Node      => Import_List_Node,
                            Null_List => Null_List);
            CopyImports (SubprogSym);
         end if;
      else --not doing IFA
         Do_Import_List (Node      => Import_List_Node,
                         Null_List => Null_List);  -- ineffective FirstValidExport OK
         if Star_Found then
            AppendSelfReferences (Star_Node, SubprogSym);
         end if;
      end if;
      ErrorFound := LocalError;
   end wf_dependency_clause;

   ----------------------------------------------------------------------------

   function PositionToReportError (Node : STree.SyntaxNode) return LexTokenManager.Token_Position
   --# global in STree.Table;
   is
   begin
      return Node_Position (Node => STree.FindLastItemInDependencyRelation (Node));
   end PositionToReportError;

   -------------------------------------------------------------------------

   procedure ParamsAreImportsOrExports (SubprogSym : in Dictionary.Symbol;
                                        ErrorPos   : in LexTokenManager.Token_Position)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SemanticErrorFound;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorPos,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         SubprogSym &
   --#         SemanticErrorFound         from *,
   --#                                         Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         SubprogSym;
   is
      NextParam : Dictionary.Iterator;
      ParamSym  : Dictionary.Symbol;

   begin
      NextParam := Dictionary.FirstSubprogramParameter (SubprogSym);
      while NextParam /= Dictionary.NullIterator loop
         ParamSym := Dictionary.CurrentSymbol (NextParam);
         if not Dictionary.IsImport (Abstraction, SubprogSym, ParamSym) and
           not Dictionary.IsExport (Abstraction, SubprogSym, ParamSym) then
            SemanticErrorFound := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 200,
               Reference => ErrorHandler.No_Reference,
               Position  => ErrorPos,
               Id_Str    => Dictionary.GetSimpleName (ParamSym));
         end if;

         -- test to ensure that parameters of mode IN OUT are imported
         if Dictionary.GetSubprogramParameterMode (ParamSym) = Dictionary.InOutMode then
            if not Dictionary.IsImport (Abstraction, SubprogSym, ParamSym) then
               --# accept Flow, 41, "Expected stable expression";
               if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83 then
                  --# end accept;
                  SemanticErrorFound := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 504,
                     Reference => ErrorHandler.No_Reference,
                     Position  => ErrorPos,
                     Id_Str    => Dictionary.GetSimpleName (ParamSym));

               elsif Dictionary.TypeIsScalar (Dictionary.GetType (ParamSym)) then
                  SemanticErrorFound := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 338,
                     Reference => ErrorHandler.No_Reference,
                     Position  => ErrorPos,
                     Id_Str    => Dictionary.GetSimpleName (ParamSym));
               end if;
            end if; --end of checks that in out is an import

            -- check that in out is exported
            if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83
              and then not Dictionary.IsExport (Abstraction, SubprogSym, ParamSym) then
               SemanticErrorFound := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 506,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Id_Str    => Dictionary.GetSimpleName (ParamSym));
            end if;
         end if;
         NextParam := Dictionary.NextSymbol (NextParam);
      end loop;
   end ParamsAreImportsOrExports;

   ---------------------------------------------------------------------------

   procedure GlobalsAreImportsOrExports (SubprogSym : in Dictionary.Symbol;
                                         ErrorPos   : in LexTokenManager.Token_Position)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     CurrentScope;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SemanticErrorFound;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorPos,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         SubprogSym &
   --#         SemanticErrorFound         from *,
   --#                                         Abstraction,
   --#                                         Dictionary.Dict,
   --#                                         SubprogSym;
   --#
   is
      NextGlobal : Dictionary.Iterator;
      GlobalSym  : Dictionary.Symbol;

   begin
      NextGlobal := Dictionary.FirstGlobalVariable (Abstraction, SubprogSym);
      while NextGlobal /= Dictionary.NullIterator loop
         GlobalSym := Dictionary.CurrentSymbol (NextGlobal);
         if not Dictionary.IsImport (Abstraction, SubprogSym, GlobalSym)
           and then not Dictionary.IsExport (Abstraction, SubprogSym, GlobalSym) then
            SemanticErrorFound := True;
            ErrorHandler.Semantic_Error_Sym
              (Err_Num   => 201,
               Reference => ErrorHandler.No_Reference,
               Position  => ErrorPos,
               Sym       => GlobalSym,
               Scope     => CurrentScope);
         end if;
         -- check that globals with mode in out are both imported and exported.
         if Dictionary.GetGlobalMode (Abstraction, SubprogSym, GlobalSym) = Dictionary.InOutMode then
            if not Dictionary.IsImport (Abstraction, SubprogSym, GlobalSym) then
               SemanticErrorFound := True;
               ErrorHandler.Semantic_Error_Sym
                 (Err_Num   => 505,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Sym       => GlobalSym,
                  Scope     => CurrentScope);
            end if;
            if not Dictionary.IsExport (Abstraction, SubprogSym, GlobalSym) then
               SemanticErrorFound := True;
               ErrorHandler.Semantic_Error_Sym
                 (Err_Num   => 507,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Sym       => GlobalSym,
                  Scope     => CurrentScope);
            end if;
         end if;
         NextGlobal := Dictionary.NextSymbol (NextGlobal);
      end loop;
   end GlobalsAreImportsOrExports;

   ---------------------------------------------------------------------------

   procedure CheckInformationFlowPolicy (SubprogSym : in Dictionary.Symbol;
                                         Position   : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     CurrentScope;
   --#        in     Dictionary.Dict;
   --#        in     FirstSeen;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FirstSeen,
   --#                                         LexTokenManager.State,
   --#                                         Position,
   --#                                         SPARK_IO.File_Sys,
   --#                                         SubprogSym;
   is
      ExportIt  : Dictionary.Iterator;
      ImportIt  : Dictionary.Iterator;
      TheExport : Dictionary.Symbol;
      TheImport : Dictionary.Symbol;

      type FSToAbsTable is array (Boolean) of Dictionary.Abstractions;
      ToAbs : constant FSToAbsTable := FSToAbsTable'(False => Dictionary.IsRefined,
                                                     True  => Dictionary.IsAbstract);
   begin
      -- Iterate over all the Exports of this subprogram...
      ExportIt := Dictionary.FirstExport (ToAbs (FirstSeen), SubprogSym);
      while ExportIt /= Dictionary.NullIterator loop
         TheExport := Dictionary.CurrentSymbol (ExportIt);

         ImportIt := Dictionary.FirstDependency (ToAbs (FirstSeen), SubprogSym, TheExport);

         -- For each export, iterate over all the imports...
         while ImportIt /= Dictionary.NullIterator loop

            TheImport := Dictionary.CurrentSymbol (ImportIt);

            -- ...and see if each relation violates the current information
            -- flow policy.
            if Dictionary.RelationViolatesInfoFlowPolicy (TheExport, TheImport) then

               ErrorHandler.Dependency_Error
                 (Err_Type       => ErrorHandler.Policy_Violation,
                  Position       => Position,
                  Import_Var_Sym => TheImport,
                  Export_Var_Sym => TheExport,
                  Scope          => CurrentScope);
            end if;

            ImportIt := Dictionary.NextSymbol (ImportIt);
         end loop;
         ExportIt := Dictionary.NextSymbol (ExportIt);
      end loop;
   end CheckInformationFlowPolicy;

begin -- wf_dependency_relation

   -- ASSUME Node = dependency_relation
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.dependency_relation,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = dependency_relation in Wf_Dependency_Relation");
   Position := PositionToReportError (Node);
   if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83
     and then not CommandLineData.Content.Do_Information_Flow then -- derives not used in this case, don't process, just issue note
      ErrorHandler.Semantic_Note (Err_Num  => 1,
                                  Position => Position,
                                  Id_Str   => LexTokenManager.Null_String);
   else --proceed as usual (even if doing DFA in 83 mode)
      if FirstSeen then
         Abstraction := Dictionary.IsAbstract;
      else
         Abstraction := Dictionary.IsRefined;
      end if;

      SeqAlgebra.CreateSeq (TheHeap, ExportsInRelation);
      SeqAlgebra.CreateSeq (TheHeap, ImportsInRelation);

      SemanticErrorFound := GlobDefErr;
      Dictionary.AddDependencyRelation
        (Abstraction,
         SubprogSym,
         Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                              End_Position   => Node_Position (Node => Node)));

      It := Find_First_Node (Node_Kind    => SPSymbols.dependency_clause,
                             From_Root    => Node,
                             In_Direction => STree.Down);

      while not STree.IsNull (It) loop

         NextNode := Get_Node (It => It);
         -- globals ImportList, ExportList are created in wf_dependency_clause
         wf_dependency_clause (TheHeap, NextNode, SubprogSym, Abstraction, CommandLineData.Content.Do_Information_Flow, Error);
         SemanticErrorFound := SemanticErrorFound or Error;

         -- add in the imports and exports from this clause to the total imports/exports
         --# accept Flow, 41, "Expected stable expression";
         if not CommandLineData.Content.Do_Information_Flow then
            --# end accept;
            SeqAlgebra.AugmentSeq (TheHeap, ExportsInRelation, ExportList);
            SeqAlgebra.AugmentSeq (TheHeap, ImportsInRelation, ImportList);
         end if;
         SeqAlgebra.DisposeOfSeq (TheHeap, ExportList);
         SeqAlgebra.DisposeOfSeq (TheHeap, ImportList);
         It := STree.NextNode (It);
      end loop;

      -- Process any import list of the form "derives null from x, y, z;"
      Null_Import_Node := Child_Node (Current_Node => Child_Node (Current_Node => Node));
      -- ASSUME Null_Import_Node = dependency_relation_rep OR null_import_list OR NULL
      if Syntax_Node_Type (Node => Null_Import_Node) = SPSymbols.dependency_relation_rep then
         Null_Import_Node := Next_Sibling (Current_Node => Next_Sibling (Current_Node => Null_Import_Node));
      elsif Null_Import_Node /= STree.NullNode
        and then Syntax_Node_Type (Node => Null_Import_Node) /= SPSymbols.null_import_list then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Null_Import_Node = dependency_relation_rep OR null_import_list OR NULL in Wf_Dependency_Relation");
      end if;
      -- ASSUME Null_Import_Node = null_import_list OR NULL
      -- if there is a null import clause then Null_Import_Node is now pointing at it
      if Syntax_Node_Type (Node => Null_Import_Node) = SPSymbols.null_import_list then
         wf_dependency_clause
           (TheHeap,
            Null_Import_Node,
            SubprogSym,
            Abstraction,
            CommandLineData.Content.Do_Information_Flow,
            Error);
         SemanticErrorFound := SemanticErrorFound or Error;

         -- add in the imports and exports from this clause to the total imports/exports
         if not CommandLineData.Content.Do_Information_Flow then  -- stable index OK
            SeqAlgebra.AugmentSeq (TheHeap, ExportsInRelation, ExportList);
            SeqAlgebra.AugmentSeq (TheHeap, ImportsInRelation, ImportList);
         end if;
         SeqAlgebra.DisposeOfSeq (TheHeap, ExportList);
         SeqAlgebra.DisposeOfSeq (TheHeap, ImportList);
      elsif Null_Import_Node /= STree.NullNode then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Null_Import_Node = null_import_list OR NULL in Wf_Dependency_Relation");
      end if;

      -- make all exports depend on all imports here
      if not CommandLineData.Content.Do_Information_Flow then
         ErrorHandler.Semantic_Note (Err_Num  => 2,
                                     Position => Position,
                                     Id_Str   => LexTokenManager.Null_String);
         CreateFullDependency
           (Node        => Node,
            SubprogSym  => SubprogSym,
            Abstraction => Abstraction,
            ImportList  => ImportsInRelation,
            ExportList  => ExportsInRelation);
      end if;

      SeqAlgebra.DisposeOfSeq (TheHeap, ExportsInRelation);
      SeqAlgebra.DisposeOfSeq (TheHeap, ImportsInRelation);

      ParamsAreImportsOrExports (SubprogSym, Position);
      GlobalsAreImportsOrExports (SubprogSym, Position);

      --do consistency checks if second anno and no semantic errors found and
      --second global annotation is not missing completely
      if not (FirstSeen or
                SemanticErrorFound or
                (Syntax_Node_Type (Node => Child_Node (Parent_Node (Current_Node => Node))) = SPSymbols.dependency_relation)) then
         CheckDerivesConsistency (SubprogSym, Position);
      end if;

      if not SemanticErrorFound then
         CheckInformationFlowPolicy (SubprogSym, Position);
      end if;

      -- mark subprogram as having incorrect signature if necessary
      if SemanticErrorFound then
         Dictionary.SetSubprogramSignatureNotWellformed (Abstraction, SubprogSym);
      end if;
   end if;

   if ErrorHandler.Generate_SLI then
      SLI.Generate_Xref_Derives
        (Comp_Unit   => ContextManager.Ops.Current_Unit,
         Parse_Tree  => Node,
         Scope       => Dictionary.LocalScope (SubprogSym),
         Subprog_Sym => SubprogSym);
   end if;

end wf_dependency_relation;
