-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

-- Overview:
-- Checks a with clause for Sem starting at node with_clause.
-- Directly capable of raising errors for: undeclared item in with list,
-- duplicate item in with list or withing of something which is not a
-- package.
--
-- NB.  In present form permits with for something not inherited; this is
--      necessary for withing something to be used solely in hidden part
--      (eg. text_io by spark_io).  However, we wish to issue a
--      semantic warning in such circumstances.
--      It is also necessary to with something not inherited in the case
--      where an inherit cannot be placed; for example where a package
--      body withs a private child package.
--------------------------------------------------------------------------------

separate (Sem.CompUnit.Wf_Context_Clause)
procedure With_Clause (Node     : in STree.SyntaxNode;
                       Comp_Sym : in Dictionary.Symbol;
                       Scope    : in Dictionary.Scopes) is
   It : STree.Iterator;

   -----------------------------

   procedure Process_Dotted_Simple_Name
     (Node     : in STree.SyntaxNode;
      Comp_Sym : in Dictionary.Symbol;
      Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
      Prefix_Sym         : Dictionary.Symbol := Dictionary.NullSymbol;
      Current_Sym        : Dictionary.Symbol;
      Current_Node       : STree.SyntaxNode;
      Explicit_Duplicate : Boolean;
      Withing_Descendent : Boolean           := False;
      Discard            : Boolean;
      Lib_Sym            : Dictionary.Symbol;
      Search_String      : LexTokenManager.Lex_String;

      ----------------------------

      function Dotted_Identifier_Found (Node : STree.SyntaxNode) return Boolean
      --# global in STree.Table;
      is
      begin
         -- ASSUME Node = dotted_simple_name
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Node) = SPSymbols.dotted_simple_name,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = inherit_clause in Dotted_Identifier_Found");
         return Syntax_Node_Type (Node => Child_Node (Current_Node => Node)) = SPSymbols.dotted_simple_name;
      end Dotted_Identifier_Found;

      -------------------

      function Is_Last_Identifier_Node (Node : STree.SyntaxNode) return Boolean
      --# global in STree.Table;
      is
      begin
         -- ASSUME Node = identifier
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Node) = SPSymbols.identifier,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = identifier in Is_Last_Identifier_Node");

         return Syntax_Node_Type (Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Node))) /=
           SPSymbols.dotted_simple_name;
      end Is_Last_Identifier_Node;

      --------------------

      function Look_Up
        (Prefix            : in Dictionary.Symbol;
         Str               : in LexTokenManager.Lex_String;
         Scope             : in Dictionary.Scopes;
         Full_Package_Name : in Boolean)
        return   Dictionary.Symbol
      --# global in CommandLineData.Content;
      --#        in Dictionary.Dict;
      --#        in LexTokenManager.State;
      is
         Sym : Dictionary.Symbol;
      begin
         if Prefix = Dictionary.NullSymbol then
            Sym := Dictionary.LookupItem (Name              => Str,
                                          Scope             => Scope,
                                          Context           => Dictionary.ProofContext,
                                          Full_Package_Name => Full_Package_Name);
         else
            Sym :=
              Dictionary.LookupSelectedItem
              (Prefix   => Prefix,
               Selector => Str,
               Scope    => Scope,
               Context  => Dictionary.ProofContext);
         end if;
         return Sym;
      end Look_Up;

      --------------------

      procedure Check_Descendent
        (Sym                : in out Dictionary.Symbol;
         Lib_Sym            : in     Dictionary.Symbol;
         Withing_Descendent :    out Boolean)
      --# global in Dictionary.Dict;
      --# derives Sym,
      --#         Withing_Descendent from Dictionary.Dict,
      --#                                 Lib_Sym,
      --#                                 Sym;
      is
      begin
         if Dictionary.GetPackageOwner (Sym) /= Lib_Sym then
            Sym                := Dictionary.NullSymbol;
            Withing_Descendent := False;
         else
            Withing_Descendent := True;
         end if;
      end Check_Descendent;

   begin -- Process_Dotted_Simple_Name

      -- ASSUME Node = dotted_simple_name
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.dotted_simple_name,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = inherit_clause in Process_Dotted_Simple_Name");

      if Dotted_Identifier_Found (Node => Node) and then CommandLineData.Content.Language_Profile = CommandLineData.SPARK83 then
         ErrorHandler.Semantic_Error
           (Err_Num   => 610,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
      else
         Lib_Sym      := Dictionary.GetLibraryPackage (Dictionary.VisibleScope (Comp_Sym));
         Current_Node := Last_Child_Of (Start_Node => Node);
         loop
            -- ASSUME Current_Node = identifier
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Node => Current_Node) = SPSymbols.identifier,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Current_Node = identifier in Process_Dotted_Simple_Name");
            -- look up in global scope first:
            Search_String := Node_Lex_String (Node => Current_Node);
            Current_Sym   := Look_Up (Prefix            => Prefix_Sym,
                                      Str               => Search_String,
                                      Scope             => Dictionary.GlobalScope,
                                      Full_Package_Name => True);

            if Current_Sym /= Dictionary.NullSymbol
              and then Current_Sym /= Lib_Sym
              and then Dictionary.IsPackage (Current_Sym) then
               -- package exists and is not self
               -- if necessary, check inherited by looking up in current scope
               if Dictionary.IsProperDescendent (Current_Sym, Lib_Sym) then
                  -- withing a descendent
                  Check_Descendent (Sym                => Current_Sym,
                                    Lib_Sym            => Lib_Sym,
                                    Withing_Descendent => Withing_Descendent);
               elsif Dictionary.IsProperDescendent (Lib_Sym, Current_Sym) then
                  -- withing an ancestor
                  if Is_Last_Identifier_Node (Node => Current_Node) then
                     Current_Sym := Look_Up (Prefix            => Dictionary.NullSymbol,
                                             Str               => Search_String,
                                             Scope             => Scope,
                                             Full_Package_Name => False);
                  end if;
               elsif Prefix_Sym /= Dictionary.NullSymbol and then Dictionary.IsProperDescendent (Lib_Sym, Prefix_Sym) then
                  -- withing child of ancestor
                  Current_Sym := Look_Up (Prefix            => Dictionary.NullSymbol,
                                          Str               => Search_String,
                                          Scope             => Scope,
                                          Full_Package_Name => False);
               else
                  Current_Sym := Look_Up (Prefix            => Prefix_Sym,
                                          Str               => Search_String,
                                          Scope             => Scope,
                                          Full_Package_Name => True);
               end if;
            end if;

            if Current_Sym /= Dictionary.NullSymbol
              and then not Dictionary.IsPackage (Current_Sym)
              and then not Dictionary.Is_Generic_Subprogram (The_Symbol => Current_Sym) then
               -- can't be inherited
               ErrorHandler.Semantic_Error
                 (Err_Num   => 18,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Current_Node),
                  Id_Str    => Search_String);
               exit;
            end if;

            if Is_Last_Identifier_Node (Node => Current_Node) and then Current_Sym = Lib_Sym then
               -- trying to with self (or enclosing package)
               ErrorHandler.Semantic_Error
                 (Err_Num   => 132,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Current_Node),
                  Id_Str    => Search_String);
               exit;
            end if;

            -- extra check for private root packages,
            -- which cannot be with'd by specs of public packages:
            if Current_Sym /= Dictionary.NullSymbol
              and then Dictionary.IsPackage (Current_Sym)
              and then -- guard for precon of next line
              Dictionary.IsPrivatePackage (Current_Sym)
              and then Dictionary.GetPackageParent (Current_Sym) = Dictionary.NullSymbol
              and then Dictionary.IsVisibleScope (Scope)
              and then Dictionary.GetPackageOwner (Lib_Sym) = Dictionary.NullSymbol
              and then not Dictionary.IsPrivatePackage (Dictionary.GetRootPackage (Lib_Sym)) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 616,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Current_Node),
                  Id_Str    => Search_String);
               exit;
            end if;

            -- Look up will find generic functions because they have an associated
            -- proof function in proof context.  We want to exclude them unless they have
            -- also been inherited.
            if Dictionary.Is_Generic_Subprogram (The_Symbol => Current_Sym)
              and then not Dictionary.IsInherited (Current_Sym, Comp_Sym) then
               Current_Sym := Dictionary.NullSymbol;
            end if;

            if Current_Sym = Dictionary.NullSymbol then
               if CommandLineData.Ravenscar_Selected
                 and then LexTokenManager.Lex_String_Case_Insensitive_Compare
                 (Lex_Str1 => Search_String,
                  Lex_Str2 => LexTokenManager.Ada_Token) /=
                 LexTokenManager.Str_Eq
                 and then LexTokenManager.Lex_String_Case_Insensitive_Compare
                 (Lex_Str1 => Search_String,
                  Lex_Str2 => LexTokenManager.System_Token) /=
                 LexTokenManager.Str_Eq then
                  -- stronger warning for uninherited withs of non-predefined packages in Ravenscar
                  ErrorHandler.Semantic_Warning
                    (Err_Num  => 391,
                     Position => Node_Position (Node => Current_Node),
                     Id_Str   => Search_String);
               else
                  ErrorHandler.Semantic_Warning
                    (Err_Num  => 1,
                     Position => Node_Position (Node => Current_Node),
                     Id_Str   => Search_String);
               end if;
               exit;
            end if;

            -- check sym found is not a local redeclaration
            if not Dictionary.IsGlobalScope (Dictionary.GetScope (Current_Sym)) then
               -- This semantic error has not been checked with new error number
               -- because unable to find test case which causes the error.
               ErrorHandler.Semantic_Error
                 (Err_Num   => 133,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Current_Node),
                  Id_Str    => Search_String);
               exit;
            end if;

            -- there is something to add because symbol is not null
            Dictionary.AddWithReference
              (Scope            => Scope,
               ThePackage       => Current_Sym,
               Explicit         => Is_Last_Identifier_Node (Node => Current_Node),
               Comp_Unit        => ContextManager.Ops.Current_Unit,
               PackageReference => Dictionary.Location'(Start_Position => Node_Position (Node => Current_Node),
                                                        End_Position   => Node_Position (Node => Current_Node)),
               AlreadyPresent   => Explicit_Duplicate);

            -- handle the case of a with for a descendent package, which
            -- causes problems with visibility for nested child packages.
            -- only relevant for private child packages, as public child
            -- packages will not have been found by the earlier lookup

            -- Add a 'fake inherit' as well as the 'with'
            if Withing_Descendent then
               --# accept Flow, 10, Discard, "Expected ineffective assignment";
               Dictionary.AddInheritsReference
                 (CompilationUnit  => Comp_Sym,
                  ThePackage       => Current_Sym,
                  Explicit         => False,
                  Comp_Unit        => ContextManager.Ops.Current_Unit,
                  PackageReference => Dictionary.Location'(Start_Position => Node_Position (Node => Current_Node),
                                                           End_Position   => Node_Position (Node => Current_Node)),
                  AlreadyPresent   => Discard);
               --# end accept;
               -- can never be explicitly duplicated, as is only ever implicit...
            end if;

            if Explicit_Duplicate then
               ErrorHandler.Semantic_Error_Sym
                 (Err_Num   => 191,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Node),
                  Sym       => Current_Sym,
                  Scope     => Dictionary.GlobalScope);
            end if;
            STree.Set_Node_Lex_String (Sym  => Current_Sym,
                                       Node => Current_Node);
            exit when Is_Last_Identifier_Node (Node => Current_Node);

            Prefix_Sym := Current_Sym; -- ready for next lookup

            Current_Node := Next_Sibling (Current_Node => Parent_Node (Current_Node => Current_Node));
         end loop;
      end if;
      --# accept Flow, 33, Discard, "Expected to be neither referenced nor exported";
   end Process_Dotted_Simple_Name;

begin -- With_Clause

   -- ASSUME Node = with_clause
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.with_clause,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = with_clause in With_Clause");

   Dictionary.AddWithClause
     (Scope,
      Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                           End_Position   => Node_Position (Node => Node)));

   It := Find_First_Node (Node_Kind    => SPSymbols.dotted_simple_name,
                          From_Root    => Node,
                          In_Direction => STree.Down);
   while not STree.IsNull (It) loop
      -- ASSUME It = dotted_simple_name
      Process_Dotted_Simple_Name (Node     => Get_Node (It => It),
                                  Comp_Sym => Comp_Sym,
                                  Scope    => Scope);
      It := STree.NextNode (It);
   end loop;

end With_Clause;
