-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure wf_attribute_designator
  (Node         : in     STree.SyntaxNode;
   Scope        : in     Dictionary.Scopes;
   EStack       : in out ExpStack.ExpStackType;
   IsAnnotation : in     Boolean;
   RefVar       : in     SeqAlgebra.Seq) is
   ATT_LOOKUP : constant Annotation_Symbol_Table :=
     Annotation_Symbol_Table'(False => SPSymbols.attribute_designator,
                              True  => SPSymbols.annotation_attribute_designator);
   EXP_LOOKUP : constant Annotation_Symbol_Table :=
     Annotation_Symbol_Table'(False => SPSymbols.expression,
                              True  => SPSymbols.annotation_expression);

   IdentNode, ArgExpNode         : STree.SyntaxNode;
   SecondArgExpNode              : STree.SyntaxNode;
   TypeSoFar, ArgumentExpression : Exp_Record;
   SecondArgumentExpression      : Exp_Record;
   ArgumentFound                 : Boolean;
   SecondArgumentFound           : Boolean;
   IdentStr                      : LexTokenManager.Lex_String;
   Val, UnusedVal, RHSval        : Maths.Value;
   BaseFound                     : Boolean;
   OkSoFar                       : Boolean;
   PrefixKind                    : Dictionary.PrefixSort;
   PrefixType                    : Dictionary.Symbol;
   VCGtype                       : Dictionary.Symbol;

   ----------------------------------------------------------------------

   procedure GetPrefix (Prefix :    out Exp_Record;
                        Kind   :    out Dictionary.PrefixSort;
                        EStack : in out ExpStack.ExpStackType)
   --# global in     BaseFound;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IdentNode;
   --#        in     LexTokenManager.State;
   --#        in     Node;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from BaseFound,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         IdentNode,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from * &
   --#         Kind                       from BaseFound,
   --#                                         Dictionary.Dict,
   --#                                         EStack &
   --#         Prefix                     from BaseFound,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         Scope;
   is
      Result : Exp_Record;
   begin
      Kind := Dictionary.AType;
      ExpStack.Pop (Result, EStack); --this is type of prefix expression
      if Result.Sort = Is_Unknown then
         Result := UnknownTypeRecord;
      elsif Result.Sort = Is_Type_Mark then
         Result.Is_Static   := Dictionary.IsStatic (Result.Type_Symbol, Scope);
         Result.Is_Constant := True;
         Result.Is_ARange   := False;
         if BaseFound then
            if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83
              and then not Dictionary.TypeIsScalar (Result.Type_Symbol) then
               Result := UnknownTypeRecord;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 96,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Node),
                  Id_Str    => LexTokenManager.Base_Token);
            end if;
            Kind := Dictionary.ABaseType;
         end if;
      elsif Dictionary.IsObject (Result.Other_Symbol) or else Dictionary.IsRecordSubcomponent (Result.Other_Symbol) then
         if Dictionary.IsUniversalIntegerType (Result.Type_Symbol)
           or else Dictionary.IsUniversalRealType (Result.Type_Symbol) then
            --its a named number and not a proper object
            Result := UnknownTypeRecord;
            ErrorHandler.Semantic_Error
              (Err_Num   => 31,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => IdentNode),
               Id_Str    => LexTokenManager.Null_String);
         else
            Result.Is_Static             := Dictionary.IsStatic (Result.Type_Symbol, Scope);
            Result.Is_Constant           := Dictionary.IsConstant (Result.Type_Symbol);
            Result.Is_ARange             := False;
            Result.Variable_Symbol       := Dictionary.NullSymbol;
            Result.Is_AVariable          := False;
            Result.Is_An_Entire_Variable := False;
            Kind                         := Dictionary.AnObject;
         end if;
      elsif Result = UnknownTypeRecord then
         null;
      else
         Result := UnknownTypeRecord;
         ErrorHandler.Semantic_Error
           (Err_Num   => 31,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => IdentNode),
            Id_Str    => LexTokenManager.Null_String);
      end if;
      Prefix := Result;
   end GetPrefix;

   ----------------------------------------------------------------------

   -- this function should gradually wither away as attributes are implemented
   function NotYetImplemented (Str : LexTokenManager.Lex_String) return Boolean
   --# global in LexTokenManager.State;
   is
   begin
      return (LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                   Lex_Str2 => LexTokenManager.Adjacent_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Compose_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Copy_Sign_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Leading_Part_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Remainder_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Scaling_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Exponent_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Fraction_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Machine_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Model_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Rounding_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Truncation_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Machine_Rounding_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Unbiased_Rounding_Token) =
                LexTokenManager.Str_Eq);
   end NotYetImplemented;

   ----------------------------------------------------------------------

   -- identifies special attributes that are part of the pre/post annotation
   -- language only
   function IsProofAttribute (Str : LexTokenManager.Lex_String) return Boolean
   --# global in LexTokenManager.State;
   is
   begin
      return (LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                   Lex_Str2 => LexTokenManager.Tail_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Append_Token) =
                LexTokenManager.Str_Eq);
   end IsProofAttribute;

   ----------------------------------------------------------------------

   function ProofAttributeIsVisible
     (IdentStr   : LexTokenManager.Lex_String;
      PrefixKind : Dictionary.PrefixSort;
      PrefixSym  : Dictionary.Symbol)
     return       Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   is
   begin
      return PrefixKind = Dictionary.AnObject
        and then ((LexTokenManager.Lex_String_Case_Insensitive_Compare
                     (Lex_Str1 => IdentStr,
                      Lex_Str2 => LexTokenManager.Tail_Token) =
                     LexTokenManager.Str_Eq
                     and then Dictionary.GetOwnVariableOrConstituentMode (PrefixSym) = Dictionary.InMode)
                  or else (LexTokenManager.Lex_String_Case_Insensitive_Compare
                             (Lex_Str1 => IdentStr,
                              Lex_Str2 => LexTokenManager.Append_Token) =
                             LexTokenManager.Str_Eq
                             and then Dictionary.GetOwnVariableOrConstituentMode (PrefixSym) = Dictionary.OutMode));
   end ProofAttributeIsVisible;

   ----------------------------------------------------------------------

   function NeverTakesArguments (Str : LexTokenManager.Lex_String) return Boolean
   --# global in LexTokenManager.State;
   is
   begin
      return not (LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                       Lex_Str2 => LexTokenManager.First_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Last_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Length_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Pos_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Pred_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Range_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Succ_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Val_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Min_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Max_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Tail_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Ceiling_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Floor_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Append_Token) =
                    LexTokenManager.Str_Eq
                    or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Str,
                     Lex_Str2 => LexTokenManager.Mod_Token) =
                    LexTokenManager.Str_Eq);
   end NeverTakesArguments;

   ----------------------------------------------------------------------

   function AlwaysTakesOneArgument (Str : LexTokenManager.Lex_String) return Boolean
   --# global in LexTokenManager.State;
   is
   begin
      return (LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                   Lex_Str2 => LexTokenManager.Pos_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                             Lex_Str2 => LexTokenManager.Pred_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                             Lex_Str2 => LexTokenManager.Succ_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                             Lex_Str2 => LexTokenManager.Val_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Floor_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Ceiling_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                             Lex_Str2 => LexTokenManager.Tail_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                             Lex_Str2 => LexTokenManager.Mod_Token) =
                LexTokenManager.Str_Eq);
   end AlwaysTakesOneArgument;

   ----------------------------------------------------------------------

   function AlwaysTakesTwoArguments (Str : LexTokenManager.Lex_String) return Boolean
   --# global in LexTokenManager.State;
   is
   begin
      return (LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                   Lex_Str2 => LexTokenManager.Min_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                             Lex_Str2 => LexTokenManager.Max_Token) =
                LexTokenManager.Str_Eq
                or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                (Lex_Str1 => Str,
                 Lex_Str2 => LexTokenManager.Append_Token) =
                LexTokenManager.Str_Eq);
   end AlwaysTakesTwoArguments;

   ----------------------------------------------------------------------

   function ArgumentTypeCorrect (Str                 : LexTokenManager.Lex_String;
                                 PrefixType, ArgType : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   --#        in Scope;
   is
      Result : Boolean;
   begin
      if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                              Lex_Str2 => LexTokenManager.Val_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Mod_Token) =
        LexTokenManager.Str_Eq then
         Result := Dictionary.IsIntegerTypeMark (ArgType, Scope)
           or else Dictionary.IsModularTypeMark (ArgType, Scope)
           or else Dictionary.IsUnknownTypeMark (ArgType);
      else
         Result := Dictionary.CompatibleTypes (Scope, PrefixType, ArgType);
      end if;
      return Result;
   end ArgumentTypeCorrect;

   ----------------------------------------------------------------------

   function AttributeConsideredStatic (Str : LexTokenManager.Lex_String) return Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   --#        in Scope;
   --#        in TypeSoFar;
   is
      Result : Boolean;
   begin
      if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                              Lex_Str2 => LexTokenManager.Aft_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Base_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Delta_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Digits_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Emax_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Epsilon_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.First_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Fore_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Large_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Last_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Machine_Emax_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Machine_Emin_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Machine_Mantissa_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Machine_Overflows_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Machine_Radix_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Machine_Rounds_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Mantissa_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Pred_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Pos_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Safe_Emax_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Safe_Large_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Safe_Small_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Small_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Succ_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Val_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Component_Size_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Denorm_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Model_Emin_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Model_Epsilon_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Model_Mantissa_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Model_Small_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Safe_First_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Safe_Last_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Str,
         Lex_Str2 => LexTokenManager.Signed_Zeros_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Min_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Max_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Modulus_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Floor_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Ceiling_Token) =
        LexTokenManager.Str_Eq
        or else LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                     Lex_Str2 => LexTokenManager.Mod_Token) =
        LexTokenManager.Str_Eq then
         Result := True;

      elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Str,
                                                                 Lex_Str2 => LexTokenManager.Size_Token) =
        LexTokenManager.Str_Eq then
         -- 'Size is static only for a prefix that denotes a static
         -- scalar subtype
         Result := Dictionary.IsScalarType (TypeSoFar.Type_Symbol, Scope);
      else
         Result := False;
      end if;

      return Result;
   end AttributeConsideredStatic;

   ----------------------------------------------------------------------

   function CheckStatic return Boolean
   --# global in ArgumentExpression;
   --#        in ArgumentFound;
   --#        in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in IdentStr;
   --#        in LexTokenManager.State;
   --#        in PrefixType;
   --#        in Scope;
   --#        in TypeSoFar;
   is
      Result : Boolean;

      function IsStaticArrayAttribute return Boolean
      --# global in CommandLineData.Content;
      --#        in Dictionary.Dict;
      --#        in IdentStr;
      --#        in LexTokenManager.State;
      --#        in PrefixType;
      is
      begin
         return CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83
           and then (LexTokenManager.Lex_String_Case_Insensitive_Compare
                       (Lex_Str1 => IdentStr,
                        Lex_Str2 => LexTokenManager.First_Token) =
                       LexTokenManager.Str_Eq
                       or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                       (Lex_Str1 => IdentStr,
                        Lex_Str2 => LexTokenManager.Last_Token) =
                       LexTokenManager.Str_Eq
                       or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                       (Lex_Str1 => IdentStr,
                        Lex_Str2 => LexTokenManager.Length_Token) =
                       LexTokenManager.Str_Eq)
           and then Dictionary.TypeIsArray (PrefixType)
           and then not Dictionary.IsUnconstrainedArrayType (PrefixType);
      end IsStaticArrayAttribute;

   begin
      Result := (TypeSoFar.Is_Static and then AttributeConsideredStatic (IdentStr)) or else IsStaticArrayAttribute;
      if Result and then ArgumentFound then
         Result := ArgumentExpression.Is_Static;
      end if;
      return Result;
   end CheckStatic;

   ----------------------------------------------------------------------

   function CheckConstant return Boolean
   --# global in ArgumentExpression;
   --#        in ArgumentFound;
   --#        in Dictionary.Dict;
   --#        in IdentStr;
   --#        in LexTokenManager.State;
   --#        in Scope;
   --#        in TypeSoFar;
   is
      Result : Boolean;
   begin
      Result := TypeSoFar.Is_Constant or else Dictionary.IsConstrainedArrayType (TypeSoFar.Type_Symbol, Scope);
      if Result then
         if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => IdentStr,
                                                                 Lex_Str2 => LexTokenManager.Base_Token) =
           LexTokenManager.Str_Eq then
            Result := not Dictionary.IsUnconstrainedArrayType (Dictionary.GetRootType (TypeSoFar.Type_Symbol));
         elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => IdentStr,
                                                                    Lex_Str2 => LexTokenManager.Size_Token) =
           LexTokenManager.Str_Eq then
            -- S'Size is only considered to be constant/static for
            -- scalar types
            Result := Dictionary.IsScalarType (TypeSoFar.Type_Symbol, Scope);
         end if;
      end if;
      if Result and then ArgumentFound then
         Result := ArgumentExpression.Is_Constant;
      end if;

      return Result;
   end CheckConstant;

   ----------------------------------------------------------------------

   function CheckRange return Boolean
   --# global in IdentStr;
   --#        in LexTokenManager.State;
   is
   begin
      return LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => IdentStr,
                                                                  Lex_Str2 => LexTokenManager.Range_Token) =
        LexTokenManager.Str_Eq;
   end CheckRange;

   ----------------------------------------------------------------------

   procedure BasicChecks (Ok : out Boolean)
   --# global in     ArgExpNode;
   --#        in     ArgumentFound;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IdentNode;
   --#        in     IdentStr;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in     Node;
   --#        in     SecondArgExpNode;
   --#        in     SecondArgumentFound;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TypeSoFar;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from ArgExpNode,
   --#                                         ArgumentFound,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         IdentNode,
   --#                                         IdentStr,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SecondArgExpNode,
   --#                                         SecondArgumentFound,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TypeSoFar &
   --#         Ok                         from ArgumentFound,
   --#                                         CommandLineData.Content,
   --#                                         IdentStr,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         SecondArgumentFound,
   --#                                         TypeSoFar &
   --#         TypeSoFar                  from *,
   --#                                         ArgumentFound,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         IdentStr,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         SecondArgumentFound;
   is
   begin
      Ok := False;
      if not LexTokenManager.Is_Attribute_Token (Tok      => IdentStr,
                                                 Language => CommandLineData.Content.Language_Profile) then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.Semantic_Error
           (Err_Num   => 54,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => IdentNode),
            Id_Str    => IdentStr);

      elsif NotYetImplemented (IdentStr) then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.Semantic_Error
           (Err_Num   => 30,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => IdentNode),
            Id_Str    => IdentStr);

      elsif IsProofAttribute (IdentStr) and then not IsAnnotation then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.Semantic_Error
           (Err_Num   => 54,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => IdentNode),
            Id_Str    => IdentStr);

      elsif ArgumentFound and then NeverTakesArguments (IdentStr) then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.Semantic_Error
           (Err_Num   => 55,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ArgExpNode),
            Id_Str    => IdentStr);

      elsif not ArgumentFound and then (AlwaysTakesOneArgument (IdentStr) or else (AlwaysTakesTwoArguments (IdentStr))) then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.Semantic_Error
           (Err_Num   => 56,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => IdentNode),
            Id_Str    => LexTokenManager.Null_String);

      elsif not SecondArgumentFound and then (AlwaysTakesTwoArguments (IdentStr)) then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.Semantic_Error
           (Err_Num   => 56,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ArgExpNode),
            Id_Str    => LexTokenManager.Null_String);

      elsif SecondArgumentFound and then AlwaysTakesOneArgument (IdentStr) then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.Semantic_Error
           (Err_Num   => 49,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => SecondArgExpNode),
            Id_Str    => IdentStr);

         --check that prefix of Pred, Succ, Pos, Val is typemark
      elsif AlwaysTakesOneArgument (IdentStr)
        and then not IsProofAttribute (IdentStr)
        and then -- don't want 'Tail to trip this test
        TypeSoFar.Sort = Is_Object then
         ErrorHandler.Semantic_Error
           (Err_Num   => 63,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Parent_Node (Current_Node => Node)),
            Id_Str    => Dictionary.GetSimpleName (TypeSoFar.Other_Symbol));
         TypeSoFar := UnknownTypeRecord;

      else
         Ok := True;
      end if;
   end BasicChecks;

   ----------------------------------------------------------------------

   procedure BaseChecks (Continue : out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IdentNode;
   --#        in     IdentStr;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in     Node;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TypeSoFar;
   --# derives Continue                   from IdentStr,
   --#                                         LexTokenManager.State &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         IdentNode,
   --#                                         IdentStr,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TypeSoFar &
   --#         TypeSoFar                  from *,
   --#                                         Dictionary.Dict,
   --#                                         IdentStr,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
   is
   begin
      if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => IdentStr,
                                                              Lex_Str2 => LexTokenManager.Base_Token) =
        LexTokenManager.Str_Eq then
         Continue := False; --whatever happens we don't want to do any more checks
         if Syntax_Node_Type (Node => Parent_Node (Current_Node => Node)) /= ATT_LOOKUP (IsAnnotation)
           or else Syntax_Node_Type (Node => Child_Node (Node)) /= SPSymbols.attribute_ident then
            TypeSoFar := UnknownTypeRecord;
            if IsAnnotation then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 97,
                  Reference => 2,
                  Position  => Node_Position (Node => Node),
                  Id_Str    => LexTokenManager.Null_String);
            else
               ErrorHandler.Semantic_Error
                 (Err_Num   => 97,
                  Reference => 1,
                  Position  => Node_Position (Node => Node),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
         elsif TypeSoFar.Sort /= Is_Type_Mark then
            TypeSoFar := UnknownTypeRecord;
            ErrorHandler.Semantic_Error
              (Err_Num   => 96,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => IdentNode),
               Id_Str    => IdentStr);
         end if;
      else
         Continue := True;
      end if;
   end BaseChecks;

   -----------------------------------------------------------------------------

   procedure ProcessArrayAttribute
   --# global in     ArgExpNode;
   --#        in     ArgumentExpression;
   --#        in     ArgumentFound;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IdentStr;
   --#        in     LexTokenManager.State;
   --#        in     PrefixType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out OkSoFar;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TypeSoFar;
   --#        in out VCGtype;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from ArgExpNode,
   --#                                         ArgumentExpression,
   --#                                         ArgumentFound,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TypeSoFar &
   --#         OkSoFar                    from *,
   --#                                         ArgumentExpression,
   --#                                         ArgumentFound,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         TypeSoFar &
   --#         TypeSoFar                  from *,
   --#                                         ArgumentExpression,
   --#                                         ArgumentFound,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         IdentStr,
   --#                                         LexTokenManager.State,
   --#                                         PrefixType,
   --#                                         Scope &
   --#         VCGtype                    from *,
   --#                                         ArgumentExpression,
   --#                                         ArgumentFound,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         IdentStr,
   --#                                         LexTokenManager.State,
   --#                                         TypeSoFar;
   --#
   is
      IndexNumber : Integer;
      Continue    : Boolean;

      procedure ProcessArgument (Ok : out Boolean)
      -- 430 annotation completed
      --# global in     ArgExpNode;
      --#        in     ArgumentExpression;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     IdentStr;
      --#        in     LexTokenManager.State;
      --#        in     Scope;
      --#        in     STree.Table;
      --#        in out ErrorHandler.Error_Context;
      --#        in out IndexNumber;
      --#        in out OkSoFar;
      --#        in out SPARK_IO.File_Sys;
      --#        in out TypeSoFar;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from ArgExpNode,
      --#                                         ArgumentExpression,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Scope,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         TypeSoFar &
      --#         IndexNumber                from *,
      --#                                         ArgumentExpression,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict &
      --#         Ok                         from ArgumentExpression,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         TypeSoFar &
      --#         OkSoFar                    from *,
      --#                                         ArgumentExpression,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         TypeSoFar &
      --#         TypeSoFar                  from *,
      --#                                         ArgumentExpression,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         IdentStr,
      --#                                         LexTokenManager.State;
      is
         Err : Maths.ErrorCode;

         ---------------------------------------------------------
         procedure SetIllegalResult
         --# global in     Dictionary.Dict;
         --#        in     IdentStr;
         --#        in     LexTokenManager.State;
         --#           out OkSoFar;
         --#           out TypeSoFar;
         --# derives OkSoFar   from  &
         --#         TypeSoFar from Dictionary.Dict,
         --#                        IdentStr,
         --#                        LexTokenManager.State;
         is
         begin
            OkSoFar             := False;
            TypeSoFar           := UnknownTypeRecord;
            TypeSoFar.Is_ARange := CheckRange;
         end SetIllegalResult;

      begin --ProcessArgument
         Ok := False;
         -- Ada83 LRM says arg N must be Universal Integer.
         -- Ada95 LRM says arg N must be Universal Integer or any integer
         --   (signed or modular) type.
         if not (Dictionary.IsUniversalIntegerType (ArgumentExpression.Type_Symbol) or
                   Dictionary.IsUnknownTypeMark (ArgumentExpression.Type_Symbol) or
                   ((Dictionary.TypeIsInteger (ArgumentExpression.Type_Symbol) or
                       Dictionary.TypeIsModular (ArgumentExpression.Type_Symbol)) and
                      CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83)) then
            SetIllegalResult;
            ErrorHandler.Semantic_Error
              (Err_Num   => 38,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => ArgExpNode),
               Id_Str    => LexTokenManager.Null_String);
         elsif not ArgumentExpression.Is_Static then
            SetIllegalResult;
            ErrorHandler.Semantic_Error
              (Err_Num   => 36,
               Reference => 1,
               Position  => Node_Position (Node => ArgExpNode),
               Id_Str    => LexTokenManager.Null_String);
         else --we have a static expression of the correct type
            Maths.ValueToInteger (ArgumentExpression.Value,
                                  --to get
                                  IndexNumber, Err);
            if Err = Maths.NoError then
               if IndexNumber > 0 and then IndexNumber <= Dictionary.GetNumberOfDimensions (TypeSoFar.Type_Symbol) then
                  Ok := True;
               else  --number out of range
                  ErrorHandler.Semantic_Error_Sym
                    (Err_Num   => 403,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => ArgExpNode),
                     Sym       => TypeSoFar.Type_Symbol,
                     Scope     => Scope);
                  SetIllegalResult;
               end if;
            else    --maths conversion error
               SetIllegalResult;
            end if;
         end if;
      end ProcessArgument;

   begin --ProcessArrayAttribute
      Continue    := True;
      IndexNumber := 1; --default value if no argument found
      if ArgumentFound then
         ProcessArgument (Continue);
      end if;
      if Continue then
         -- Set suitable symbol to be planted in the syntax tree for use by VCG
         if LexTokenManager.Lex_String_Case_Insensitive_Compare
           (Lex_Str1 => IdentStr,
            Lex_Str2 => LexTokenManager.Component_Size_Token) =
           LexTokenManager.Str_Eq then
            -- For component_size we want the type of the array for use by the VCG
            VCGtype := Dictionary.GetRootType (TypeSoFar.Type_Symbol);
         elsif Dictionary.IsUnconstrainedArrayType (TypeSoFar.Type_Symbol) then
            -- For unconstrained arrays, obtain the implcitly declared constraint symbol for the array object
            VCGtype := Dictionary.GetSubprogramParameterConstraint (TypeSoFar.Other_Symbol, IndexNumber);
         else
            -- For constrained arrays then obtain appropriate index for the array type; this is what the VCG needs

            -- TypeSoFar.TypeSymbol might denote a full-range subtype, so jump down to the
            -- first constrained subtype before proceeding...
            TypeSoFar.Type_Symbol := Dictionary.GetFirstConstrainedSubtype (TypeSoFar.Type_Symbol);
            VCGtype               := Dictionary.GetArrayIndex (TypeSoFar.Type_Symbol, IndexNumber);
         end if;

         TypeSoFar.Is_Static    := CheckStatic;
         TypeSoFar.Is_Constant  := CheckConstant;
         TypeSoFar.Is_ARange    := CheckRange;
         TypeSoFar.Type_Symbol  := Dictionary.GetArrayAttributeType (IdentStr, TypeSoFar.Type_Symbol, IndexNumber);
         TypeSoFar.Other_Symbol := Dictionary.NullSymbol;
         TypeSoFar.Sort         := Type_Result;
      end if;

   end ProcessArrayAttribute;

   -----------------------------------------------------------------------------

begin -- wf_attribute_designator

   --node = attribute_designator
   ArgumentExpression       := Null_Exp_Record;
   SecondArgumentExpression := Null_Exp_Record;

   BaseFound := False;
   IdentNode := Child_Node (Node);
   if Syntax_Node_Type (Node => IdentNode) = ATT_LOOKUP (IsAnnotation) then
      IdentNode := Next_Sibling (IdentNode);
      BaseFound := True;
   end if;
   IdentStr         := Node_Lex_String (Node => IdentNode);
   ArgExpNode       := Child_Node (Next_Sibling (IdentNode));
   SecondArgExpNode := Next_Sibling (ArgExpNode);

   --# assert True;

   -- look for second argument
   if Syntax_Node_Type (Node => SecondArgExpNode) = EXP_LOOKUP (IsAnnotation) then
      SecondArgumentFound := True;
      ExpStack.Pop (SecondArgumentExpression, EStack);
   else
      SecondArgumentFound := False;
   end if;

   --# assert True;

   -- look for first argument
   if Syntax_Node_Type (Node => ArgExpNode) = EXP_LOOKUP (IsAnnotation) then
      ArgumentFound := True;
      ExpStack.Pop (ArgumentExpression, EStack);
   else
      ArgumentFound := False;
   end if;

   GetPrefix (TypeSoFar, PrefixKind, EStack);
   PrefixType := TypeSoFar.Type_Symbol;
   VCGtype    := PrefixType;

   -- if clause to add prefix variable to reference list

   --# assert True;

   -- X'Valid _IS_ considered a read of X for flow analysis.
   if (not IsAnnotation)
     and then Dictionary.IsVariableOrSubcomponent (TypeSoFar.Other_Symbol)
     and then (Dictionary.IsUnconstrainedArrayType (TypeSoFar.Type_Symbol)
                 or else LexTokenManager.Lex_String_Case_Insensitive_Compare
                 (Lex_Str1 => IdentStr,
                  Lex_Str2 => LexTokenManager.Valid_Token) =
                 LexTokenManager.Str_Eq) then
      SeqAlgebra.AddMember (TheHeap, RefVar, Natural (Dictionary.SymbolRef (TypeSoFar.Other_Symbol)));
   end if;

   --# assert True;

   BasicChecks (OkSoFar);
   if OkSoFar then
      BaseChecks (OkSoFar);
   end if;

   --# assert True;

   -- any attempt to use proof attributes 'Tail and 'Append in program context
   -- will have been trapped by basic checks which will have set OkSoFar to False.
   -- We can therefore treat type and argument checking of these proof attributes
   -- normally from here on because if they are being checked then they must be
   -- being used in a valid context

   if OkSoFar then
      if Dictionary.AttributeIsVisibleButObselete (IdentStr, PrefixKind, TypeSoFar.Type_Symbol, Scope) then
         ErrorHandler.Semantic_Warning (Err_Num  => 310,
                                        Position => Node_Position (Node => IdentNode),
                                        Id_Str   => IdentStr);
      end if;

      if not (Dictionary.AttributeIsVisible (IdentStr, PrefixKind, TypeSoFar.Type_Symbol, Scope)
                or else ProofAttributeIsVisible (IdentStr, PrefixKind, TypeSoFar.Other_Symbol)) then
         TypeSoFar := UnknownTypeRecord;
         OkSoFar   := False;
         ErrorHandler.Semantic_Error
           (Err_Num   => 96,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => IdentNode),
            Id_Str    => IdentStr);
      elsif Dictionary.IsArrayAttribute (IdentStr, TypeSoFar.Type_Symbol) then
         if SecondArgumentFound then
            -- must be error, array attributes take a maximum of one argument
            TypeSoFar := UnknownTypeRecord;
            OkSoFar   := False;
            ErrorHandler.Semantic_Error
              (Err_Num   => 49,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => SecondArgExpNode),
               Id_Str    => IdentStr);
         else -- zero or one expression provided
            ProcessArrayAttribute;
         end if;

      elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => IdentStr,
                                                                 Lex_Str2 => LexTokenManager.Access_Token) =
        LexTokenManager.Str_Eq then
         -- 'Access only allowed if subject is aliased.  We could roll this into AttributeIsVisible
         -- but this would make the error unclear, so we do a special check here
         if Dictionary.VariableIsAliased (TypeSoFar.Other_Symbol) then
            -- valid application
            TypeSoFar.Is_Static       := False;
            TypeSoFar.Is_Constant     := True;
            TypeSoFar.Is_ARange       := False;
            TypeSoFar.Type_Symbol     := Dictionary.GetScalarAttributeType (IdentStr, TypeSoFar.Type_Symbol);
            TypeSoFar.Variable_Symbol := TypeSoFar.Other_Symbol;
            TypeSoFar.Other_Symbol    := Dictionary.NullSymbol;
            TypeSoFar.Sort            := Type_Result;
            -- note we preserve OtherSymbol in VariableSymbol for use in wf_discriminant_constraint
         else
            OkSoFar := False;
            ErrorHandler.Semantic_Error_Sym
              (Err_Num   => 895,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Sym       => TypeSoFar.Other_Symbol,
               Scope     => Scope);
         end if;

      else --an "ordinary" attribute
         if ArgumentFound then
            if not (AlwaysTakesOneArgument (IdentStr) or else AlwaysTakesTwoArguments (IdentStr)) then
               TypeSoFar := UnknownTypeRecord;
               OkSoFar   := False;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 55,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => ArgExpNode),
                  Id_Str    => IdentStr);

            elsif ArgumentTypeCorrect (IdentStr, TypeSoFar.Type_Symbol, ArgumentExpression.Type_Symbol) then   -- first argument
                                                                                                               --type
                                                                                                               --is ok, is there a
                                                                                                               --second
               if not SecondArgumentFound
                 or else ArgumentTypeCorrect (IdentStr, TypeSoFar.Type_Symbol, SecondArgumentExpression.Type_Symbol) then  --either
                                                                                                                           --no
                                                                                                                           --second
                                                                                                                           --arg or
                                                                                                                           --it
                                                                                                                           --type
                                                                                                                           --checks
                                                                                                                           --ok
                  TypeSoFar.Is_Static    := CheckStatic;
                  TypeSoFar.Is_Constant  := CheckConstant;
                  TypeSoFar.Is_ARange    := CheckRange;
                  TypeSoFar.Type_Symbol  := Dictionary.GetScalarAttributeType (IdentStr, TypeSoFar.Type_Symbol);
                  TypeSoFar.Other_Symbol := Dictionary.NullSymbol;
                  TypeSoFar.Sort         := Type_Result;

               else -- second argument type wrong
                  TypeSoFar := UnknownTypeRecord;
                  OkSoFar   := False;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 38,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => SecondArgExpNode),
                     Id_Str    => LexTokenManager.Null_String);
               end if;

            else -- first argument type wrong
               TypeSoFar := UnknownTypeRecord;
               OkSoFar   := False;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 38,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => ArgExpNode),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
         else --no argument found so just set up result
            TypeSoFar.Is_Static    := CheckStatic;
            TypeSoFar.Is_Constant  := CheckConstant;
            TypeSoFar.Is_ARange    := CheckRange;
            TypeSoFar.Type_Symbol  := Dictionary.GetScalarAttributeType (IdentStr, TypeSoFar.Type_Symbol);
            TypeSoFar.Other_Symbol := Dictionary.NullSymbol;
            TypeSoFar.Sort         := Type_Result;
         end if;
      end if;
   end if;

   --# assert True;
   --if Dict has returned a null symbol for the attribute type then convert it to
   --the unknown type symbol
   if TypeSoFar.Type_Symbol = Dictionary.NullSymbol then
      TypeSoFar.Type_Symbol := Dictionary.GetUnknownTypeMark;
   end if;

   --# assert True;

   --# accept Flow, 10, UnusedVal, "Expected ineffective assignment" &
   --#        Flow, 33, UnusedVal, "Expected ineffective assignment";
   if OkSoFar then
      if ArgumentFound then
         if SecondArgumentFound then
            -- we could statically evaluate Max and Min here but since use of these functions
            -- with two static arguments seems unlikely it has been left for now.  However,
            -- we must check that any static argument is in type range.
            Val := ArgumentExpression.Value;
            ConstraintCheck
              (Val,
               UnusedVal,
               IsAnnotation,
               Dictionary.GetRootType (PrefixType),
               Node_Position (Node => ArgExpNode));

            Val := SecondArgumentExpression.Value;
            ConstraintCheck
              (Val,
               UnusedVal,
               IsAnnotation,
               Dictionary.GetRootType (PrefixType),
               Node_Position (Node => SecondArgExpNode));

         else -- just one argument found
            Val := ArgumentExpression.Value;
         end if;

      else -- no arguments found
         Val := Maths.NoValue;
      end if;
      -- constraint checking of arguments to attributes other than Min/Max done as part
      -- of CalcAttribute
      CalcAttribute (Node, IdentStr, -- the attribute name
                     PrefixKind, -- object, type or basetype
                     PrefixType, BaseFound, IsAnnotation,
                     -- using and to get
                     Val, RHSval);
      TypeSoFar.Value     := Val;
      TypeSoFar.Range_RHS := RHSval;
   end if;

   ExpStack.Push (TypeSoFar, EStack);

   --# assert True;
   STree.AddNodeSymbol (Node, VCGtype);

end wf_attribute_designator;
