-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

-- Overview: Called to check validity of a
-- arange node.  Replaces calls to StaticARange, BaseTypeARange and
-- CheckTypeARange
----------------------------------------------------------------------------

separate (Sem.CompUnit)
procedure wf_arange
  (Node         : in     STree.SyntaxNode;
   Scope        : in     Dictionary.Scopes;
   EStack       : in out ExpStack.ExpStackType;
   IsAnnotation : in     Boolean) is
   ATTRIB_LOOKUP : constant Annotation_Symbol_Table :=
     Annotation_Symbol_Table'(False => SPSymbols.attribute,
                              True  => SPSymbols.annotation_attribute);

   NextNode            : STree.SyntaxNode;
   Left, Right, Result : Exp_Record;
   LeftType, RightType : Dictionary.Symbol;

   ---------------------------------------------------------------

   procedure IntegerImplicitTypeConversion (LeftType, RightType : in out Dictionary.Symbol;
                                            Scope               : in     Dictionary.Scopes)
   --# global in Dictionary.Dict;
   --# derives LeftType,
   --#         RightType from Dictionary.Dict,
   --#                        LeftType,
   --#                        RightType,
   --#                        Scope;
   is

   begin
      if Dictionary.IsUniversalIntegerType (LeftType) then
         if Dictionary.IsIntegerTypeMark (RightType, Scope) or Dictionary.IsModularTypeMark (RightType, Scope) then
            LeftType := RightType;
         end if;

      elsif Dictionary.IsUniversalIntegerType (RightType) then
         if Dictionary.IsIntegerTypeMark (LeftType, Scope) or Dictionary.IsModularTypeMark (LeftType, Scope) then
            RightType := LeftType;
         end if;

      end if;
   end IntegerImplicitTypeConversion;

   ---------------------------------------------------------------

   procedure RealImplicitTypeConversion (LeftType, RightType : in out Dictionary.Symbol;
                                         Scope               : in     Dictionary.Scopes)
   --# global in Dictionary.Dict;
   --# derives LeftType,
   --#         RightType from Dictionary.Dict,
   --#                        LeftType,
   --#                        RightType,
   --#                        Scope;
   is

   begin
      if Dictionary.IsUniversalRealType (LeftType) then
         if Dictionary.IsRealTypeMark (RightType, Scope) then
            LeftType := RightType;
         end if;

      elsif Dictionary.IsUniversalRealType (RightType) then
         if Dictionary.IsRealTypeMark (LeftType, Scope) then
            RightType := LeftType;
         end if;

      end if;
   end RealImplicitTypeConversion;

   ---------------------------------------------------------------

   function RangeIsEmpty (Left, Right : Maths.Value) return Boolean is
      Unused      : Maths.ErrorCode;
      MathsResult : Maths.Value;
      FuncResult  : Boolean;
   begin
      --# accept Flow, 10, Unused, "Expected ineffective assignment" &
      --#        Flow, 33, Unused, "Expected to be neither referenced nor exported";
      Maths.Lesser (Right, Left,
                    --to get
                    MathsResult, Unused);  --not used because it can only be ok or type mismatch
      Maths.ValueToBool (MathsResult,
                         --to get
                         FuncResult, Unused);
      return FuncResult;
   end RangeIsEmpty;

   ---------------------------------------------------------------

begin --wf_arange
   NextNode := Child_Node (Node);
   if Syntax_Node_Type (Node => NextNode) = ATTRIB_LOOKUP (IsAnnotation) then
      ExpStack.Pop (Result, EStack);
      if not Result.Is_ARange then
         Result.Is_ARange            := True;
         Result.Errors_In_Expression := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 98,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => NextNode),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   else -- explicit range of the form "Left .. Right"
      ExpStack.Pop (Right, EStack);
      ExpStack.Pop (Left, EStack);

      Result := NullTypeRecord; --safety: we may not set all fields below

      -- In this case neither "Left" nor "Right" can themselves denote a Range.
      -- The following two checks prevent cases such as
      --   S'First .. S'Range
      --   S'Range .. S'Last
      --   S'Range .. S'Range
      -- which are all illegal.  We check both Left and Right separately so
      -- that two errors are issued for the latter case.

      if Left.Is_ARange then
         Result := UnknownTypeRecord;
         ErrorHandler.Semantic_Error
           (Err_Num   => 114,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => NextNode),
            Id_Str    => LexTokenManager.Null_String);
      end if;

      if Right.Is_ARange then
         Result := UnknownTypeRecord;
         ErrorHandler.Semantic_Error
           (Err_Num   => 114,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Next_Sibling (NextNode)),
            Id_Str    => LexTokenManager.Null_String);
      end if;

      if not Left.Is_ARange and not Right.Is_ARange then

         -- Neither Left nor Right is a Range, so we can proceed...
         Result.Is_Constant := Left.Is_Constant and Right.Is_Constant;
         Result.Is_Static   := Left.Is_Static and Right.Is_Static;
         Result.Is_ARange   := True;
         LeftType           := Dictionary.GetRootType (Left.Type_Symbol);
         RightType          := Dictionary.GetRootType (Right.Type_Symbol);
         IntegerImplicitTypeConversion (LeftType, RightType, Scope);
         RealImplicitTypeConversion (LeftType, RightType, Scope);
         if LeftType /= RightType then
            Result := UnknownTypeRecord;
            ErrorHandler.Semantic_Error
              (Err_Num   => 42,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Next_Sibling (NextNode)),
               Id_Str    => LexTokenManager.Null_String);
         elsif not (Dictionary.IsScalarType (LeftType, Scope) or else Dictionary.IsUnknownTypeMark (LeftType)) then
            Result := UnknownTypeRecord;
            ErrorHandler.Semantic_Error
              (Err_Num   => 44,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => LexTokenManager.Null_String);
         else
            Result.Type_Symbol := LeftType;
            Result.Value       := Left.Value;
            Result.Range_RHS   := Right.Value;

            -- check that static range is non empty
            if (not IsAnnotation) and then RangeIsEmpty (Left.Value, Right.Value) then
               Result.Value     := Maths.NoValue;
               Result.Range_RHS := Maths.NoValue;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 409,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Node),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
         end if;
      end if;

      Result.Errors_In_Expression := Result.Errors_In_Expression or Left.Errors_In_Expression or Right.Errors_In_Expression;
   end if;

   ExpStack.Push (Result, EStack);

end wf_arange;
