-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure wf_ancestor_part
  (Node         : in out STree.SyntaxNode;
   Scope        : in     Dictionary.Scopes;
   EStack       : in out ExpStack.ExpStackType;
   HeapParam    : in out Lists.List_Heap;
   IsAnnotation : in     Boolean) is
   QualifyingExpression, AncestorPart : Exp_Record;
   Ptr                                : Lists.List;

   --------------------------------------------------------------------

   function HasInterveningPrivateExtensions (RootType, ExtendedType : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   --#        in Scope;
   -- pre Dictionary.IsAnExtensionOf (RootType, ExtendedType);
   is
      Result        : Boolean := False;
      CurrentRecord : Dictionary.Symbol;
   begin
      CurrentRecord := ExtendedType;
      loop
         -- follow chain of Inherit field pointers
         CurrentRecord := Dictionary.CurrentSymbol (Dictionary.FirstRecordComponent (CurrentRecord));
         exit when CurrentRecord = Dictionary.NullSymbol; -- root record is a null record

         CurrentRecord := Dictionary.GetType (CurrentRecord);
         exit when CurrentRecord = RootType;

         exit when not Dictionary.TypeIsTagged (CurrentRecord); -- all fields checked - false result exit

         if Dictionary.IsPrivateType (CurrentRecord, Scope) then
            Result := True;
            exit;        -- true result exit
         end if;
      end loop;
      return Result;
   end HasInterveningPrivateExtensions;

   --------------------------------------------------------------------

   function IsNullAgregate return Boolean
   --# global in Node;
   --#        in STree.Table;
   is
   begin
      return Syntax_Node_Type (Node => Next_Sibling (Node)) = SPSymbols.RWnull;
   end IsNullAgregate;

   --------------------------------------------------------------------

   function FieldsNeedDefining (RootType, ExtendedType : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return Dictionary.GetNumberOfComponents (ExtendedType) > Dictionary.GetNumberOfComponents (RootType);
   end FieldsNeedDefining;

   --------------------------------------------------------------------

   function AssociationIsNamed return Boolean
   --# global in IsAnnotation;
   --#        in Node;
   --#        in STree.Table;
   is
      NAM_ASS_LOOKUP : constant Annotation_Symbol_Table :=
        Annotation_Symbol_Table'
        (False => SPSymbols.named_record_component_association,
         True  => SPSymbols.annotation_named_record_component_association);
   begin
      return Syntax_Node_Type (Node => Child_Node (Next_Sibling (Node))) = NAM_ASS_LOOKUP (IsAnnotation);
   end AssociationIsNamed;

   --------------------------------------------------------------------
   -- Assume:
   -- 1. Node is [annotation_]ancestor_part going up
   -- 2. TOS is result of walking ancestor part expression
   -- 3. 2nd TOS is result of walking qualifier
   --
   -- Check:
   -- 1. Qualifier is an extended tagged type
   -- 2. Ancestor part represents an ancestor of the extended type
   -- 3. There are no private extensions between the ancestor and the qualifier
   -- 4. If the aggregate part is null record then there are no new components
   --    between ancestor part and the qualifier.
   --

begin -- wf_ancestor_part
   ExpStack.Pop (AncestorPart, EStack);
   ExpStack.Pop (QualifyingExpression, EStack);
   -- seed syntax tree with type of ancestor for use in VCG
   STree.AddNodeSymbol (Node, AncestorPart.Type_Symbol);

   if QualifyingExpression.Sort = Is_Type_Mark then
      -- Correctly formed qualifying expression so carry on with rest of checks
      QualifyingExpression.Is_Constant := True;
      if not Dictionary.TypeIsExtendedTagged (QualifyingExpression.Type_Symbol) then
         ExpStack.Push (UnknownTypeRecord, EStack);
         ErrorHandler.Semantic_Error_Sym
           (Err_Num   => 835,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Node))),
            Sym       => QualifyingExpression.Type_Symbol,
            Scope     => Scope);
         -- move up so as to prevent walk of rest of illegal aggregate
         Node := Parent_Node (Current_Node => Node);

      elsif not Dictionary.IsAnExtensionOf (AncestorPart.Type_Symbol, QualifyingExpression.Type_Symbol) then
         ExpStack.Push (UnknownTypeRecord, EStack);
         ErrorHandler.Semantic_Error_Sym
           (Err_Num   => 836,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Sym       => QualifyingExpression.Type_Symbol,
            Scope     => Scope);
         -- move up so as to prevent walk of rest of illegal aggregate
         Node := Parent_Node (Current_Node => Node);
      elsif HasInterveningPrivateExtensions
        (RootType     => AncestorPart.Type_Symbol,
         ExtendedType => QualifyingExpression.Type_Symbol) then
         ExpStack.Push (UnknownTypeRecord, EStack);
         ErrorHandler.Semantic_Error_Sym
           (Err_Num   => 837,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Sym       => QualifyingExpression.Type_Symbol,
            Scope     => Scope);
         -- move up so as to prevent walk of rest of illegal aggregate
         Node := Parent_Node (Current_Node => Node);

      elsif IsNullAgregate then
         if FieldsNeedDefining (RootType     => AncestorPart.Type_Symbol,
                                ExtendedType => QualifyingExpression.Type_Symbol) then
            ExpStack.Push (UnknownTypeRecord, EStack);
            ErrorHandler.Semantic_Error_Sym
              (Err_Num   => 838,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Sym       => QualifyingExpression.Type_Symbol,
               Scope     => Scope);
            -- move up so as to prevent walk of rest of illegal aggregate
            Node := Parent_Node (Current_Node => Node);

         else -- valid null record case, this means aggregate is complete
            QualifyingExpression.Sort         := Type_Result;
            QualifyingExpression.Is_Constant  := True;
            QualifyingExpression.Param_Count  := 0;
            QualifyingExpression.Param_List   := Lists.Null_List;
            QualifyingExpression.Other_Symbol := Dictionary.NullSymbol;
            QualifyingExpression.Is_ARange    := False;
            QualifyingExpression.Is_Static    := False;
            ExpStack.Push (QualifyingExpression, EStack);
            -- move up to qualified expression node since aggregate is complete
            Node := Parent_Node (Current_Node => Parent_Node (Current_Node => Node));
         end if;
      else
         -- Valid non-null case
         -- Set up either counters, for positional, or check lists, for named association
         if AssociationIsNamed then
            CreateNameList (Ptr, HeapParam);
            QualifyingExpression.Param_List := Ptr;
            -- to check whether a field is correctly part of the record but NOT already in
            -- the ancestor part we need access to the ancestor symbol when processing
            -- the associations.  For this reason we put the ancestor symbol into the
            -- OtherSymbol field of the record representing the aggregate
            QualifyingExpression.Other_Symbol := AncestorPart.Type_Symbol;
            ExpStack.Push (QualifyingExpression, EStack);
            -- walk continues at Next_Sibling (Node) -  record_component_association
         else
            --positional association
            -- when we check the expressions we want to match those fields NOT included
            -- in the ancestor part so we set the field counter thus:
            QualifyingExpression.Param_Count := Dictionary.GetNumberOfComponents (AncestorPart.Type_Symbol);
            ExpStack.Push (QualifyingExpression, EStack);
            -- walk continues at Next_Sibling (Node) -  record_component_association
         end if;
      end if;
   else -- qualifying prefix is not a type mark
      ExpStack.Push (UnknownTypeRecord, EStack);
      ErrorHandler.Semantic_Error
        (Err_Num   => 95,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Node))),
         Id_Str    => LexTokenManager.Null_String);
      -- move up so as to prevent walk of rest of illegal aggregate
      Node := Parent_Node (Current_Node => Node);
   end if;
end wf_ancestor_part;
