-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
function SimpleExpressionTypeFromContext
  (ExpNode : in STree.SyntaxNode;
   TStack  : in TypeContextStack.TStackType)
  return    Dictionary.Symbol
is
   NewContextType : Dictionary.Symbol;
   Parent         : STree.SyntaxNode;
   GrandParent    : STree.SyntaxNode;
begin
   -- Assume NodeType(ExpNode) = simple_expression or
   --        NodeType(ExpNode) = annotation_simple_expression

   -- The determination of the type context for a simple_expression depends
   -- on the parent (and possibly the grandparent) node in the syntax tree.
   -- The possible parent nodes (as determined from the grammar) are as follows,
   -- and fall into 3 groups:
   --
   --  Group 1 - parent nodes where a change of context might be needed
   --      arange
   --      annotation_arange
   --      aggregate_choice
   --      annotation_aggregate_choice
   --      case_choice
   --      relation
   --      annotation_relation
   --
   -- Group 2 - parent nodes where the simple_expression appears in a universal
   --           context, and the context is supplied by whoever called WalkExpression,
   --           so no change in context needed.
   --      Modular_Type_Definition
   --      Floating_Accuracy_Definition
   --      Fixed_Accuracy_Definition
   --
   -- Group 3 - Simple_Expressions appearing in rep. clauses, which are not analysed
   --           at present, so no change in context needed.
   --      Attribute_Definition_Clause
   --      Mod_Clause
   --      Component_Clause
   --      At_Clause

   Parent := Parent_Node (Current_Node => ExpNode);

   case Syntax_Node_Type (Node => Parent) is

      --------------------------
      -- Group 1 Parent nodes --
      --------------------------

      when SPSymbols.arange | SPSymbols.annotation_arange =>

         -- For [annotation_]arange, the new context depends on the
         -- grandparent node as well.  Possible grandparent nodes are:
         --   relation
         --   annotation_relation
         --   loop_parameter_specification
         --   range_constraint
         --   annotation_range_constraint
         --   quantified_expression
         --   component_clause

         GrandParent := Parent_Node (Current_Node => Parent);

         case Syntax_Node_Type (Node => GrandParent) is
            when SPSymbols.relation | SPSymbols.annotation_relation =>
               -- Must be a membership test like "A in B .. C" or
               -- Context is lost here, since "in" is defined for all types.
               NewContextType := Dictionary.GetUnknownTypeMark;

            when SPSymbols.loop_parameter_specification =>
               -- Context for the loop range is passed in from wf_loop_param,
               -- so no change is needed here.
               NewContextType := TypeContextStack.Top (TStack);

            when SPSymbols.range_constraint | SPSymbols.annotation_range_constraint =>
               -- These nodes have their own special function for determining context,
               -- so no change here.
               NewContextType := TypeContextStack.Top (TStack);

            when SPSymbols.quantified_expression =>

               -- Down_wf_quantifier plants the quantified variable's symbol
               -- in the Identifier node below the quantified_expression node, so we
               -- can grab that and look up its type.
               NewContextType := Dictionary.GetType (STree.NodeSymbol (Next_Sibling (Child_Node (GrandParent))));

            when SPSymbols.component_clause =>
               -- Part of a rep. clause, so no change
               NewContextType := TypeContextStack.Top (TStack);

            when others =>
               -- Must be an error resulting from an invalid syntax tree,
               -- but we need to push something so...
               NewContextType := TypeContextStack.Top (TStack);
         end case;

      when SPSymbols.aggregate_choice | SPSymbols.annotation_aggregate_choice =>

         -- For a named aggregate choice, the required index type is
         -- always on top of the aggregate stack, so...
         NewContextType := AggregateStack.TopTypeSym;

      when SPSymbols.case_choice =>

         -- The correct type for the context is passed into WalkExpression
         -- from wf_case_choice, so no change required here.
         NewContextType := TypeContextStack.Top (TStack);

      when SPSymbols.relation | SPSymbols.annotation_relation =>

         if Next_Sibling (Child_Node (Parent)) = STree.NullNode then
            -- This relation has no operator, so preserve context
            NewContextType := TypeContextStack.Top (TStack);
         else
            -- This relation has an operator, so context is lost
            NewContextType := Dictionary.GetUnknownTypeMark;
         end if;

         --------------------------
         -- Group 2 Parent nodes --
         --------------------------

      when SPSymbols.modular_type_definition | SPSymbols.floating_accuracy_definition | SPSymbols.fixed_accuracy_definition =>
         -- No change in context here.
         NewContextType := TypeContextStack.Top (TStack);

         --------------------------
         -- Group 3 Parent nodes --
         --------------------------

      when SPSymbols.attribute_definition_clause | SPSymbols.mod_clause | SPSymbols.component_clause | SPSymbols.at_clause =>
         -- No change in context here.
         NewContextType := TypeContextStack.Top (TStack);

      when others =>
         -- Must be an error, which will be caught elsewhere,
         -- but we need to push something so...
         NewContextType := TypeContextStack.Top (TStack);
   end case;

   return NewContextType;

end SimpleExpressionTypeFromContext;
