-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Dictionary)
function BinaryOperatorIsVisible
  (Name  : SPSymbols.SPSymbol;
   Left  : Symbol;
   Right : Symbol;
   Scope : Scopes)
  return  Boolean
is

   --------------------------------------------------------------------------------

   function IsDirectlyVisible
     (Name  : SPSymbols.SPSymbol;
      Left  : Symbol;
      Right : Symbol;
      Scope : Scopes)
     return  Boolean
   --# global in CommandLineData.Content;
   --#        in Dict;
   is
      IsVisible : Boolean;

      --------------------------------------------------------------------------------

      function CheckDeclarationsAreVisible (Inner, Outer : Scopes) return Boolean
      --# global in CommandLineData.Content;
      --#        in Dict;
      is
         CurrentScope : Scopes;
         PackSym      : Symbol;
         IsVisible    : Boolean;
      begin
         case CommandLineData.Content.Language_Profile is
            when CommandLineData.SPARK83 =>

               IsVisible := InSameImmediatePackage (Inner, Outer);

            when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>

               CurrentScope := Inner;
               loop
                  IsVisible := InSameImmediatePackage (CurrentScope, Outer);
                  exit when IsVisible;
                  CurrentScope := GetEnclosingScope (CurrentScope);
                  exit when CurrentScope = GlobalScope;
               end loop;
               if not IsVisible then
                  PackSym := GetLibraryPackage (Inner);
                  if PackSym /= GetPredefinedPackageStandard then
                     loop
                        PackSym := GetPackageParent (PackSym);
                        exit when PackSym = NullSymbol;
                        IsVisible := InSameImmediatePackage (VisibleScope (PackSym), Outer);
                        exit when IsVisible;
                     end loop;
                  end if;
               end if;
         end case;
         return IsVisible;
      end CheckDeclarationsAreVisible;

      --------------------------------------------------------------------------------

      function EqualityOperatorsAreVisible (Left, Right : Symbol;
                                            Scope       : Scopes) return Boolean
      --# global in CommandLineData.Content;
      --#        in Dict;
      is
         Operand   : Symbol;
         IsVisible : Boolean;
      begin

         if TypeIsUnknown (Left) then
            Operand := Right;
         else
            Operand := Left;
         end if;

         if DefinedInPackageStandard (Operand) or else TypeIsUnknown (Operand) then
            IsVisible := True;
         else
            IsVisible := CheckDeclarationsAreVisible (Scope, GetScope (Operand)) and then not TypeIsLimited (Operand, Scope);
         end if;

         return IsVisible;

      end EqualityOperatorsAreVisible;

      --------------------------------------------------------------------------------

      function SymmetricOperatorsAreVisible (Left, Right : Symbol;
                                             Scope       : Scopes) return Boolean
      --# global in CommandLineData.Content;
      --#        in Dict;
      is
         Operand   : Symbol;
         IsVisible : Boolean;
      begin

         if TypeIsUnknown (Left) then
            Operand := Right;
         else
            Operand := Left;
         end if;

         if DefinedInPackageStandard (Operand) or else TypeIsUnknown (Operand) then
            IsVisible := True;
         else
            IsVisible := CheckDeclarationsAreVisible (Scope, GetScope (Operand)) and then not TypeIsPrivateHere (Operand, Scope);
         end if;

         return IsVisible;

      end SymmetricOperatorsAreVisible;

      --------------------------------------------------------------------------------

      function AsymmetricOperatorsAreVisible
        (Left  : Symbol;
         Right : Symbol;
         Scope : Scopes)
        return  Boolean
      --# global in CommandLineData.Content;
      --#        in Dict;
      is
         IsVisible : Boolean;
      begin

         if DefinedInPackageStandard (Left) or else TypeIsUnknown (Left) then
            if DefinedInPackageStandard (Right) or else TypeIsUnknown (Right) then
               IsVisible := True;
            else
               IsVisible := CheckDeclarationsAreVisible (Scope, GetScope (Right)) and then not TypeIsPrivateHere (Right, Scope);
            end if;
         elsif DefinedInPackageStandard (Right) or else TypeIsUnknown (Right) then
            IsVisible := CheckDeclarationsAreVisible (Scope, GetScope (Left)) and then not TypeIsPrivateHere (Left, Scope);
         else
            IsVisible := CheckDeclarationsAreVisible (Scope, GetScope (Left))
              and then not TypeIsPrivateHere (Left, Scope)
              and then CheckDeclarationsAreVisible (Scope, GetScope (Right))
              and then not TypeIsPrivateHere (Right, Scope);
         end if;

         return IsVisible;

      end AsymmetricOperatorsAreVisible;

      --------------------------------------------------------------------------------

   begin
      if Name = SPSymbols.equals or else Name = SPSymbols.not_equal then
         IsVisible := EqualityOperatorsAreVisible (Left, Right, Scope);
      elsif Name = SPSymbols.multiply
        and then (TypeIsFixedPoint (Left) or else TypeIsUnknown (Left))
        and then (TypeIsFixedPoint (Right) or else TypeIsUnknown (Right)) then
         IsVisible := True;
      elsif Name = SPSymbols.divide
        and then (TypeIsFixedPoint (Left) or else TypeIsUnknown (Left))
        and then (TypeIsFixedPoint (Right) or else TypeIsUnknown (Right)) then
         IsVisible := True;
      elsif Left = Right then
         IsVisible := SymmetricOperatorsAreVisible (Left, Right, Scope);
      else
         IsVisible := AsymmetricOperatorsAreVisible (Left, Right, Scope);
      end if;

      return IsVisible;

   end IsDirectlyVisible;

   --------------------------------------------------------------------------------

   function BinaryOperatorIsRenamed
     (Name  : SPSymbols.SPSymbol;
      Left  : Symbol;
      Right : Symbol;
      Scope : Scopes)
     return  Boolean
   --# global in CommandLineData.Content;
   --#        in Dict;
   is
      NameToLookFor : SPSymbols.SPSymbol;

      --------------------------------------------------------------------------------

      function IsExplicitlyRenamed
        (Name  : SPSymbols.SPSymbol;
         Left  : Symbol;
         Right : Symbol;
         Scope : Scopes)
        return  Boolean
      --# global in CommandLineData.Content;
      --#        in Dict;
      is
         Current : Scopes;
         Region  : Symbol;
         Found   : Boolean;

         --------------------------------------------------------------------------------

         function IsRenamedInThisScope
           (Name  : SPSymbols.SPSymbol;
            Left  : Symbol;
            Right : Symbol;
            Scope : Scopes)
           return  Boolean
         --# global in Dict;
         is
            Region : Symbol;
            Found  : Boolean;

            --------------------------------------------------------------------------------

            function SearchRenamingDeclarations
              (Name         : SPSymbols.SPSymbol;
               Left         : Symbol;
               Right        : Symbol;
               Declarations : Symbol)
              return         Boolean
            --# global in Dict;
            is
               Current : Symbol;
               Item    : Symbol;
            begin

               Current := Declarations;

               loop
                  exit when Current = NullSymbol;
                  Item := RawDict.GetDeclarationItem (Current);
                  exit when RawDict.GetSymbolDiscriminant (Item) = OperatorSymbol
                    and then RawDict.GetOperatorName (Item) = Name
                    and then RawDict.GetOperatorLeftOperand (Item) = Left
                    and then RawDict.GetOperatorRightOperand (Item) = Right;
                  Current := RawDict.GetNextDeclaration (Current);
               end loop;

               return Current /= NullSymbol;

            end SearchRenamingDeclarations;

            --------------------------------------------------------------------------------

            function IsRenamedInThisPackage
              (Name  : SPSymbols.SPSymbol;
               Left  : Symbol;
               Right : Symbol;
               Scope : Scopes)
              return  Boolean
            --# global in Dict;
            is
               ThisPackage : Symbol;
               Found       : Boolean;

               --------------------------------------------------------------------------------

               function IsRenamedInThisPackageSpecification
                 (Name        : SPSymbols.SPSymbol;
                  Left        : Symbol;
                  Right       : Symbol;
                  ThisPackage : Symbol)
                 return        Boolean
               --# global in Dict;
               is
               begin
                  return SearchRenamingDeclarations
                    (Name,
                     Left,
                     Right,
                     RawDict.GetPackageVisibleRenamingDeclarations (ThisPackage));
               end IsRenamedInThisPackageSpecification;

               --------------------------------------------------------------------------------

               function IsRenamedInThisPackageBody
                 (Name        : SPSymbols.SPSymbol;
                  Left        : Symbol;
                  Right       : Symbol;
                  ThisPackage : Symbol)
                 return        Boolean
               --# global in Dict;
               is
               begin
                  return SearchRenamingDeclarations (Name, Left, Right, RawDict.GetPackageLocalRenamingDeclarations (ThisPackage));
               end IsRenamedInThisPackageBody;

               --------------------------------------------------------------------------------

            begin

               ThisPackage := GetRegion (Scope);

               case Scope.TypeOfScope is
                  when Visible | Privat =>
                     Found := IsRenamedInThisPackageSpecification (Name, Left, Right, ThisPackage);
                  when Local =>
                     Found := IsRenamedInThisPackageBody (Name, Left, Right, ThisPackage)
                       or else IsRenamedInThisPackageSpecification (Name, Left, Right, ThisPackage);
               end case;

               return Found;

            end IsRenamedInThisPackage;

            --------------------------------------------------------------------------------

            function IsRenamedInThisSubprogram
              (Name       : SPSymbols.SPSymbol;
               Left       : Symbol;
               Right      : Symbol;
               Subprogram : Symbol)
              return       Boolean
            --# global in Dict;
            is
            begin
               return SearchRenamingDeclarations (Name, Left, Right, RawDict.GetSubprogramRenamingDeclarations (Subprogram));
            end IsRenamedInThisSubprogram;

            --------------------------------------------------------------------------------

         begin

            Region := GetRegion (Scope);

            case RawDict.GetSymbolDiscriminant (Region) is
               when PackageSymbol =>
                  Found := IsRenamedInThisPackage (Name, Left, Right, Scope);
               when SubprogramSymbol =>
                  Found := IsRenamedInThisSubprogram (Name, Left, Right, Region);
               when others =>
                  Found := False;
            end case;

            return Found;

         end IsRenamedInThisScope;

         --------------------------------------------------------------------------------

      begin

         Current := Scope;

         loop
            Found := IsRenamedInThisScope (Name, Left, Right, Current);
            exit when Found;
            Region := GetRegion (Current);
            exit when IsPackage (Region) and CommandLineData.Content.Language_Profile = CommandLineData.SPARK83;
            exit when IsMainProgram (Region);
            Current := GetEnclosingScope (Current);
            exit when Current = GlobalScope;
         end loop;

         if Current = GlobalScope then
            Region := GetLibraryPackage (Scope);
            if Region /= GetPredefinedPackageStandard then
               loop
                  Region := GetPackageParent (Region);
                  exit when Region = NullSymbol;
                  Found := IsRenamedInThisScope (Name, Left, Right, VisibleScope (Region));
                  exit when Found;
               end loop;
            end if;
         end if;

         return Found;

      end IsExplicitlyRenamed;

      --------------------------------------------------------------------------------

   begin

      if Name = SPSymbols.not_equal then
         NameToLookFor := SPSymbols.equals;
      else
         NameToLookFor := Name;
      end if;

      return IsExplicitlyRenamed (NameToLookFor, Left, Right, Scope);

   end BinaryOperatorIsRenamed;

   --------------------------------------------------------------------------------

   function TypeIsUsed (TypeSym : Symbol;
                        Scope   : Scopes) return Boolean
   --# global in CommandLineData.Content;
   --#        in Dict;
   is
      Current : Scopes;
      Region  : Symbol;
      Found   : Boolean;

      ------------------

      function IsUsedInThisScope (TypeSym : Symbol;
                                  Scope   : Scopes) return Boolean
      --# global in Dict;
      is
         Region : Symbol;
         Found  : Boolean;

         ------------------

         function IsUsedInThisPackage (TypeSym : Symbol;
                                       Scope   : Scopes) return Boolean
         --# global in Dict;
         is
            Found : Boolean;

         begin --IsUsedInThisPackage
            case Scope.TypeOfScope is
               when Visible =>
                  Found := IsUsedLocally (TypeSym, Scope);
               when Privat =>
                  Found := IsUsedLocally (TypeSym, VisibleScope (GetRegion (Scope)));
               when Local =>
                  Found := IsUsedLocally (TypeSym, Scope) or else IsUsedLocally (TypeSym, VisibleScope (GetRegion (Scope)));
            end case;
            return Found;
         end IsUsedInThisPackage;

         ------------------

      begin --IsUsedInThisScope
         Region := GetRegion (Scope);
         case RawDict.GetSymbolDiscriminant (Region) is
            when PackageSymbol =>
               Found := IsUsedInThisPackage (TypeSym, Scope);
            when SubprogramSymbol =>
               Found := IsUsedLocally (TypeSym, Scope);
            when TypeSymbol => -- Task or Protected
               Found := IsUsedLocally (TypeSym, Scope);
            when others =>
               Found := False;
         end case;

         return Found;

      end IsUsedInThisScope;

      ------------------

   begin --TypeIsUsed
      case CommandLineData.Content.Language_Profile is
         when CommandLineData.SPARK83 =>

            Found := False;

         when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>

            if DefinedInPackageStandard (TypeSym) then  --can't possible be in a use type clause
               Found := False;

            else
               Current := Scope;

               loop
                  Found := IsUsedInThisScope (TypeSym, Current);
                  exit when Found;
                  Region := GetRegion (Current);

                  exit when IsMainProgram (Region);
                  Current := GetEnclosingScope (Current);
                  exit when Current = GlobalScope;
               end loop;

               if Current = GlobalScope then
                  Region := GetLibraryPackage (Scope);
                  if Region /= GetPredefinedPackageStandard then
                     loop
                        Region := GetPackageParent (Region);
                        exit when Region = NullSymbol;
                        Found := IsUsedInThisScope (TypeSym, VisibleScope (Region));
                        exit when Found;
                     end loop;
                  end if;
               end if;
            end if;
      end case;
      return Found;
   end TypeIsUsed;

   --------------------------------------------------------------------------------

   function IsPotentiallyUseVisible
     (OpName  : SPSymbols.SPSymbol;
      TypeSym : Symbol;
      Scope   : Scopes)
     return    Boolean
   --# global in CommandLineData.Content;
   --#        in Dict;
   is
   begin
      -- The time types defined in package Ada.Real_Time are private but export
      -- several binary operators ("+", "-", "*", "/" and the relational ops).
      -- So, the original code of "not TypeIsPrivateHere" is replaced by a call
      -- to TypeExportsOperators.
      return OpName = SPSymbols.equals or else OpName = SPSymbols.not_equal or else TypeExportsOperators (TypeSym, Scope);
   end IsPotentiallyUseVisible;

   --------------------------------------------------------------------------------

begin
   return IsDirectlyVisible (Name, Left, Right, Scope)
     or else (TypeIsUsed (Left, Scope) and then IsPotentiallyUseVisible (Name, Left, Scope))
     or else (TypeIsUsed (Right, Scope) and then IsPotentiallyUseVisible (Name, Right, Scope))
     or else BinaryOperatorIsRenamed (Name, Left, Right, Scope);
end BinaryOperatorIsVisible;
