<?php
/**
 * Vacation_Driver_plesk implements the Vacation_Driver API for Plesk
 * control panel servers.
 *
 * Plesk 8.1 or later is required.
 *
 * $Horde: vacation/lib/Driver/plesk.php,v 1.10.2.2 2009/01/06 15:28:08 jan Exp $
 *
 * Copyright 2008-2009 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSD). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @link    http://swdn.swsoft.com/en/library/plesk/
 * @author  Jan Schneider <jan@horde.org>
 * @package Vacation
 * @since   Vacation 3.1
 */

class Vacation_Driver_plesk extends Vacation_Driver {

    /**
     * The curl resource handler
     *
     * @var resource
     */
    var $_curl;

    /**
     * The Plesk domain id of the current realm.
     *
     * @var integer
     */
    var $_domain_id;

    /**
     * The current vacation details.
     *
     * @var array
     */
    var $_details = null;

    /**
     * Returns a parameter value.
     *
     * @param string $param  The parameter to check in.
     *
     * @return mixed  The parameter value, or null if not found.
     */
    function getParam($param)
    {
        if ($param == 'hordeauth') {
            return 'full';
        }
        return parent::getParam($param);
    }

    /**
     * Setup vacation notices for a user.
     *
     * @param string $password The password for the user.
     * @param string $message  The text of the vacation notice.
     * @param string $subject  The subject of the vacation notice.
     * @param string $from     The From: address of the vacation notice.
     * @param string $alias    Alias email address -- Not yet implemented in
     *                         backends.
     */
    function setVacation($password, $message, $subject, $from, $alias = '')
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig($password), 'PEAR_Error')) {
            return $checked;
        }

        // Query the server.
        @list($user,) = explode('@', $this->_user);
        $request = '<mail><update><set><filter><domain_id>'
            . $this->_domain_id . '</domain_id><mailname><name>'
            . htmlspecialchars($user)
            . '</name><autoresponders><enabled>true</enabled>'
            . '<autoresponder><name>Horde</name><enabled>true</enabled>'
            . '<subject>'
            . htmlspecialchars(String::convertCharset($subject, NLS::getCharset(), 'UTF-8'), ENT_COMPAT, 'UTF-8')
            . '</subject><text>'
            . htmlspecialchars(String::convertCharset($message, NLS::getCharset(), 'UTF-8'), ENT_COMPAT, 'UTF-8')
            . '</text><ans_freq>1</ans_freq></autoresponder></autoresponders></mailname></filter>'
            . '</set></update></mail>';
        $result = $this->_request($password, $request);

        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if (isset($result['mail']['update']['set']['result'])) {
            $result = $result['mail']['update']['set']['result'];
        } else {
            $result = false;
        }
        if (isset($result['status']['_']) &&
            $result['status']['_'] == 'error') {
            return PEAR::raiseError(sprintf(_("Cannot set vacation notice for mail user %s: %s"), $this->_user, $result['errtext']['_']));
        }
        if (!isset($result['status']['_']) ||
            $result['status']['_'] != 'ok' ||
            empty($result)) {
            return PEAR::raiseError(sprintf(_("Cannot set vacation notice for mail user %s."), $this->_user));
        }
    }

    /**
     * Removes any existing vacation notices.
     *
     * @param string $password  The password of the user.
     */
    function unsetVacation($password)
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig($password), 'PEAR_Error')) {
            return $checked;
        }

        // Query the server.
        @list($user,) = explode('@', $this->_user);
        $request = '<mail><update><set><filter><domain_id>'
            . $this->_domain_id . '</domain_id><mailname><name>'
            . htmlspecialchars($user)
            . '</name><autoresponders><enabled>false</enabled>'
            . '<autoresponder><name>Horde</name><enabled>false</enabled>'
            . '</autoresponder></autoresponders></mailname></filter>'
            . '</set></update></mail>';
        $result = $this->_request($password, $request);

        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if (isset($result['mail']['update']['set']['result'])) {
            $result = $result['mail']['update']['set']['result'];
        } else {
            $result = false;
        }
        if (isset($result['status']['_']) &&
            $result['status']['_'] == 'error') {
            return PEAR::raiseError(sprintf(_("Cannot remove vacation notice for mail user %s: %s"), $this->_user, $result['errtext']['_']));
        }
        if (!isset($result['status']['_']) ||
            $result['status']['_'] != 'ok' ||
            empty($result)) {
            return PEAR::raiseError(sprintf(_("Cannot remove vacation notice for mail user %s."), $this->_user));
        }
    }

    /**
     * Retrieves current vacation From: address.
     *
     * @param string $password  The password for user.
     *
     * @return string  The current or default vacation From: address.
     */
    function currentFrom($password)
    {
        return $this->_user;
    }

    /**
     * Retrieves the current vacation details for the user.
     *
     * @param string $password  The password for user.
     *
     * @return mixed  Vacation details, or PEAR_Error.
     */
    function _getUserDetails($password)
    {
        if (!is_null($this->_details)) {
            return $this->_details;
        }

        if (is_a($checked = $this->_checkConfig($password),
                 'PEAR_Error')) {
            return $checked;
        }
        @list($user,) = explode('@', $this->_user);

        // Query the server.
        $request = '<mail><get_info><filter><domain_id>'
            . $this->_domain_id . '</domain_id><name>'
            . htmlspecialchars($user)
            . '</name></filter><autoresponders/></get_info></mail>';
        $result = $this->_request($password, $request);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if (isset($result['mail']['get_info']['result']['status']['_']) &&
            $result['mail']['get_info']['result']['status']['_'] == 'error') {
            return PEAR::raiseError(sprintf(_("Cannot retrieve information about mail user %s: %s"), $this->_user, $result['mail']['get_info']['result']['errtext']['_']));
        } elseif (!isset($result['mail']['get_info']['result']['status']['_']) ||
                  $result['mail']['get_info']['result']['status']['_'] != 'ok' ||
                  !isset($result['mail']['get_info']['result']['mailname'])) {
            return PEAR::raiseError(sprintf(_("Cannot retrieve information about mail user %s."), $this->_user));
        }

        if (!isset($result['mail']['get_info']['result']['mailname']['autoresponders'])) {
            return array('vacation' => false);
        }

        $details = $result['mail']['get_info']['result']['mailname']['autoresponders']['autoresponder'];
        if (key($details) === 0) {
            $details = $details[0];
        }
        $this->_details['message'] = String::convertCharset($details['text']['_'], 'UTF-8');
        $this->_details['subject'] = String::convertCharset($details['subject']['_'], 'UTF-8');
        $this->_details['vacation'] = $result['mail']['get_info']['result']['mailname']['autoresponders']['enabled']['_'] == 'true' && $details['enabled']['_'] == 'true' ? 'y' :'n';

        return $this->_details;
    }

    /**
     * Checks if the realm has a specific configuration. If not, tries to fall
     * back on the default configuration. If still not a valid configuration
     * then returns an error.
     *
     * @param string $password  The password for the user.
     */
    function _checkConfig($password)
    {
        if (!(@include_once 'Horde/DOM.php')) {
            return PEAR::raiseError(_("The Plesk driver requires the Horde_DOM package from Horde 3.2 or later. See http://pear.horde.org/index.php?package=Horde_DOM"));
        }

        // If no realm passed in, or no host config for the realm passed in,
        // then we fall back to the default realm
        if (empty($this->_params[$this->_realm]['host'])) {
            $this->_realm = 'default';
        }

        if (!isset($this->_domain_id)) {
            @list(, $domain) = explode('@', $this->_user, 2);
            $request = '<domain><get><filter><domain_name>'
                . htmlspecialchars($domain)
                . '</domain_name></filter><dataset><gen_info/></dataset></get></domain>';
            $result = $this->_request($password, $request);
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }
            if (isset($result['domain']['get']['result']['status']['_']) &&
                $result['domain']['get']['result']['status']['_'] == 'ok' &&
                isset($result['domain']['get']['result']['id']['_'])) {
                $this->_domain_id = $result['domain']['get']['result']['id']['_'];
            } elseif (isset($result['domain']['get']['result']['status']['_']) &&
                      $result['domain']['get']['result']['status']['_'] == 'error') {
                return PEAR::raiseError(sprintf(_("Cannot retrieve domain ID of domain %s: %s"), $domain, $result['domain']['get']['result']['errtext']['_']));
            } else {
                return PEAR::raiseError(sprintf(_("Cannot retrieve domain ID of domain %s."), $domain));
            }
        }

        // If still no host/port, then we have a misconfigured module.
        if (empty($this->_params[$this->_realm]['host']) ||
            empty($this->_params[$this->_realm]['user']) ||
            empty($this->_params[$this->_realm]['pass'])) {
            return PEAR::raiseError(_("The vacation application is not properly configured."));
        }
    }

    /**
     * Connects to the Plesk RPC API server and sends a request.
     *
     * @param string $password  The password to connect with.
     * @param string $packet    The XML fragment for the request.
     *
     * @return boolean  True on success, PEAR_Error otherwise.
     */
    function _request($password, $packet)
    {
        if (!$this->_curl) {
            $url = 'https://' . $this->_params[$this->_realm]['host']
                . ':8443/enterprise/control/agent.php';
            $headers = array(
                'HTTP_AUTH_LOGIN: ' . $this->_params[$this->_realm]['user'],
                'HTTP_AUTH_PASSWD: ' . $this->_params[$this->_realm]['pass'],
                'Content-Type: text/xml');
            $this->_curl = curl_init();
            curl_setopt($this->_curl, CURLOPT_SSL_VERIFYHOST, 0);
            curl_setopt($this->_curl, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($this->_curl, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($this->_curl, CURLOPT_URL, $url);
            curl_setopt($this->_curl, CURLOPT_RETURNTRANSFER, true);
        }

        $content = '<?xml version="1.0" encoding="' . NLS::getCharset()
            . '"?><packet version="1.4.2.0">' . $packet . '</packet>';
        curl_setopt($this->_curl, CURLOPT_POSTFIELDS, $content);
        $retval = curl_exec($this->_curl);
        if ($retval === false) {
            return PEAR::raiseError(curl_error($this->_curl));
        }

        $doc = Horde_DOM_Document::factory(
            array('xml' => $retval,
                  'options' => HORDE_DOM_LOAD_REMOVE_BLANKS));
        $result = array();
        $this->_parseResponse($doc->root(), $result);
        if (isset($result['packet']['system']['status']['_']) &&
            $result['packet']['system']['status']['_'] == 'error') {
            return PEAR::raiseError($result['packet']['system']['errtext']['_']);
        }

        return $result['packet'];
    }

    /**
     * Parses the XML response body of the Plesk API call into a hash.
     *
     * @param Horde_DOM_Node $node  A DOM node object.
     * @param array $array          The result hash.
     */
    function _parseResponse($node, &$array)
    {
        $name = $node->node_name();
        $element = array();
        if (isset($array[$name])) {
            $array[$name] = array($array[$name]);
            $array[$name][] = &$element;
        } else {
            $array[$name] = &$element;
        }
        $array[$name] = array();
        if ($node->has_child_nodes()) {
            for ($child = $node->first_child();
                 $child;
                 $child = $child->next_sibling())  {
                if ($child->node_type() == XML_TEXT_NODE) {
                    $array[$name]['_'] = $child->node_value();
                } else {
                    $this->_parseResponse($child, $array[$name]);
                }
            }
        }
    }

}
