<?php

require_once 'SOAP/Client.php';

/**
 * Forwards_Driver_soap:: implements the Forwards_Driver API for SOAP
 * services.
 *
 * $Horde: forwards/lib/Driver/soap.php,v 1.8.2.4 2009/01/06 15:22:46 jan Exp $
 *
 * Copyright 2004-2009 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSDL). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Marc Jauvin <marc@r4l.com>
 * @author  Jan Schneider <jan@horde.org>
 * @since   Forwards 3.0
 * @package Forwards
 */
class Forwards_Driver_soap extends Forwards_Driver {

    /**
     * The information that was retrieved from the SOAP server.
     *
     * @var array
     */
    var $_current_details;

    /**
     * Is the SOAP client initialized?
     *
     * @var boolean
     */
    var $_initialised = false;

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     */
    function enableForwarding($password, $target, $keeplocal = false)
    {
        // _initialise() will die with Horde::fatal() upon failure.
        $this->_initialise();

        // Build username.
        $account = $this->_buildUsername();

        $data[$this->_params['username_varname']] = $account;
        if ($password && $this->_params['password_varname']) {
            $data[$this->_params['password_varname']] = $password;
        }
        $data[$this->_params['target_varname']] = $target;
        $data[$this->_params['keeplocal_varname']] = ($keeplocal === 'on') ? 1 : 0;

        $args = new SOAP_Value('args', 'array', $data);

        $result = $this->_client->call($this->_params['fwd_enable_func'], $args,
                                       $this->_params['soap_uri']);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $password  The password of the user.
     */
    function disableForwarding($password)
    {
        // _initialise() will die with Horde::fatal() upon failure.
        $this->_initialise();

        // Build username.
        $account = $this->_buildUsername();

        $data[$this->_params['username_varname']] = $account;
        if ($password && $this->_params['password_varname']) {
            $data[$this->_params['password_varname']] = $password;
        }

        $args = new SOAP_Value('args', 'array', $data);

        $result = $this->_client->call($this->_params['fwd_disable_func'],
                                       $args, $this->_params['soap_uri']);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @param string $password  The password of the user.
     *
     * @return mixed  Returns 'Y' if forwarding is enabled for the user, 'N' if
     *                forwarding is currently disabled, false if the status
     *                cannot be determined, and PEAR_Error on error.
     */
    function isEnabledForwarding($password)
    {
        // Get current details.
        $current_details = $this->_getUserDetails($password);
        if ($current_details === false ||
            is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        // Check forwarding flag.
        return (bool)$current_details[$this->_params['target_varname']];
    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($password)
    {
        // Get current details.
        $current_details = $this->_getUserDetails($password);
        if ($current_details === false ||
            is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        return (bool)$current_details[$this->_params['keeplocal_varname']];
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, false if no
     *                 forwarding is set, or PEAR_Error on error.
     */
    function currentTarget($password)
    {
        $current_details = $this->_getUserDetails($password);
        if ($current_details === false ||
            is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        if ($current_details[$this->_params['target_varname']]) {
            return $current_details[$this->_params['target_varname']];
        }
    }

    /**
     * Builds a username based on presence of realm.
     *
     * @return string  Fully qualified username.
     */
    function _buildUsername()
    {
        if ($this->_realm === 'default' ||
            $this->_realm == '') {
            return $this->_user;
        } else {
            return $this->_user . '@' . $this->_realm;
        }
    }

    /**
     * Retrieves user details from the backend.
     *
     * @param string $password  The password of the user.
     *
     * @return mixed  SOAP result resource or PEAR_Error.
     */
    function _getUserDetails($password)
    {
        // _initialise() will die with Horde::fatal() upon failure.
        $this->_initialise();

        if ($this->_current_details) {
            return $this->_current_details;
        }

        // Build username.
        $account = $this->_buildUsername();

        $data[$this->_params['username_varname']] = $account;
        $args = new SOAP_Value('args', 'array', $data);

        $result = $this->_client->call($this->_params['fwd_fetch_func'], $args,
                                       $this->_params['soap_uri']);

        if (is_a($result, 'PEAR_Error')) {
            $this->_current_details = array();
            return $result;
        }

        $this->_current_details = get_object_vars($result);
        return $this->_current_details;
    }

    /**
     * Initialises the SOAP connection object.
     */
    function _initialise()
    {
        if ($this->_initialised) {
            return;
        }

        Horde::assertDriverConfig($this->_params, 'server',
                                  array('soap_proxy',
                                        'soap_uri',
                                        'fwd_fetch_func',
                                        'fwd_enable_func',
                                        'fwd_disable_func',
                                        'username_varname',
                                        'target_varname',
                                        'keeplocal_varname'),
                                  'SOAP initialisation');

        $this->_client = new SOAP_Client($this->_params['soap_proxy']);
        $this->_client->setOpt('curl', CURLOPT_FAILONERROR, false);
        $this->_client->setOpt('curl', CURLOPT_SSL_VERIFYPEER, false);

        $this->_initialised = true;
    }

}
