<?php
/**
 * Forwards_Driver_mdaemon:: implements the Forwards_Driver API for the
 * Mdaemon mail servers.
 *
 * $Horde: forwards/lib/Driver/mdaemon.php,v 1.17.2.4 2009/01/06 15:22:46 jan Exp $
 *
 * Copyright 2001-2009 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSDL). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @author  Jan Schneider <jan@horde.org>
 * @since   Forwards 2.1
 * @package Forwards
 */
class Forwards_Driver_mdaemon extends Forwards_Driver {

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     */
    function enableForwarding($password, $target, $keeplocal = false)
    {
        if (!is_dir($this->_params[$this->_realm]['location'])) {
            return PEAR::raiseError(_("Mdaemon path not found"));
        }

        // Update forward target list.
        $forward_file = $this->_params[$this->_realm]['location'] . '/forward.dat';
        if ($fp = fopen($forward_file, 'rb')) {
            $current = fread($fp, filesize($forward_file));
            fclose($fp);
            if ($fp = fopen($forward_file, 'wb')) {
                fwrite($fp, '[' . $this->_user . '@' . $this->_realm. "]\nAddress=$target\n$current");
                fclose($fp);
            } else {
                return PEAR::raiseError(sprintf(_("Cannot open %s."), $forward_file));
            }
        } else {
            return PEAR::raiseError(sprintf(_("Cannot open %s."), $forward_file));
        }

        // Create lock file.
        $fp = fopen($this->_params[$this->_realm]['location'] . '/edituser.lck', 'w');
        if (!$fp) {
            return PEAR::raiseError(_("Cannot create lockfile."));
        } else {
            fclose($fp);
        }

        // Get current details.
        $current_details = $this->_getUserDetails();
        if (is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        // Set forwarding flag.
        $new_details = substr_replace($current_details, 'Y', 216, 1);

        // Set retain copy flag.
        if ($keeplocal) {
            $new_details = substr_replace($new_details, 'Y', 219, 1);
        } else {
            $new_details = substr_replace($new_details, 'N', 219, 1);
        }

        // Create semaphore file.
        $fp = fopen($this->_params[$this->_realm]['location'] . '/edituser.sem', 'wb');
        if (!$fp) {
            return PEAR::raiseError(_("Not able to create semaphore file"));
        } else {
            fwrite($fp, $this->_user . '@' . $this->_realm . ', ' . $new_details);
            fclose($fp);
        }

        // Remove lock file.
        @unlink($this->_params[$this->_realm]['location'] . '/edituser.lck');

        return true;
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $password  The password of the user.
     */
    function disableForwarding($password)
    {
        if (!is_dir($this->_params[$this->_realm]['location'])) {
            return PEAR::raiseError(_("Mdaemon path not found"));
        }

        // Create lock file.
        $fp = fopen($this->_params[$this->_realm]['location'] . '/edituser.lck', 'w');
        if (!$fp) {
            return PEAR::raiseError(_("Cannot create lockfile."));
        } else {
            fclose($fp);
        }

        // Get current details.
        $current_details = $this->_getUserDetails();
        if (is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        // Set forwarding flag.
        $new_details = substr_replace($current_details, 'N', 216, 1);

        // Create semaphore file.
        $fp = fopen($this->_params[$this->_realm]['location'] . '/edituser.sem', 'wb');
        if (!$fp) {
            return PEAR::raiseError(_("Not able to create semaphore file"));
        } else {
            fwrite($fp, $this->_user . '@' . $this->_realm . ', ' . $new_details);
            fclose($fp);
        }

        // Remove lock file.
        @unlink($this->_params[$this->_realm]['location'] . '/edituser.lck');

        return true;
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @param string $password  The password of the user.
     *
     * @return mixed  Returns 'Y' if forwarding is enabled for the user, 'N' if
     *                forwarding is currently disabled, false if the status
     *                cannot be determined, and PEAR_Error on error.
     */
    function isEnabledForwarding($password)
    {
        // Get current details.
        $current_details = $this->_getUserDetails();
        if (is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        // Check forwarding flag.
        if (substr($current_details, 216, 1) == 'Y') {
            return 'Y';
        }

        return false;
    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($password)
    {
        // Get current details.
        $current_details = $this->_getUserDetails();
        if (is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        // Check retain copy flag.
        return substr($current_details, 219, 1) == 'Y';
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, false if no
     *                 forwarding is set, or PEAR_Error on error.
     */
    function currentTarget($password)
    {
        $searchString = '[' . $this->_user . '@' . $this->_realm . ']';

        $fp = fopen($this->_params[$this->_realm]['location'] . '/forward.dat', 'rb');
        if (!$fp) {
            return PEAR::raiseError(
                sprintf(_("Cannot open %s."),
                        $this->_params[$this->_realm]['location'] . '/forward.dat'));
        }

        while (!feof($fp)) {
            $line = fgets($fp, 4096);
            if (strpos($line, $searchString) === 0) {
                $line = fgets($fp, 4096);
                fclose($fp);
                return substr($line, 8);
            }
        }
        fclose($fp);
    }

    /**
     * Retrieves user details from userlist.dat
     *
     * @return string  Line from userlist.dat, or false.
     */
    function _getUserDetails()
    {
        $searchString = str_pad($this->_realm, 45) . str_pad($this->_user, 30);

        $fp = fopen($this->_params[$this->_realm]['location'] . '/userlist.dat', 'rb');
        if (!$fp) {
            return PEAR::raiseError(
                sprintf(_("Cannot open %s."),
                        $this->_params[$this->_realm]['location'] . '/userlist.dat'));
        }

        while (!feof ($fp)) {
            $line = fgets($fp, 4096);
            if (strpos($line, $searchString) === 0) {
                fclose($fp);
                return $line;
            }
        }

        fclose($fp);
    }

}
