<?php
/**
 * Forwards_Driver_forwards:: implements the Forwards_Driver API for ftp
 * driven dot-forward compliant mail servers.
 *
 * $Horde: forwards/lib/Driver/forwards.php,v 1.36.2.5 2009/01/06 15:22:46 jan Exp $
 *
 * Copyright 2001-2009 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSDL). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Eric Rostetter <eric.rostetter@physics.utexas.edu>
 * @author  Jan Schneider <jan@horde.org>
 * @since   Forwards 2.1
 * @package Forwards
 */
class Forwards_Driver_forwards extends Forwards_Driver {

    /**
     * The name of the forwards file.
     *
     * @var string
     */
    var $_forwardsFile = '.forward';

    /**
     * The FTP stream we open via the VFS class.
     *
     * @var VFS_ftp
     */
    var $_vfs;

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     */
    function enableForwarding($password, $target, $keeplocal = false)
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Connect to the server.
        if (is_a($connected = $this->_connect($password), 'PEAR_Error')) {
            return $connected;
        }

        return $this->_createForwardsFile($target, $keeplocal);
    }

    /**
     * Creates the main forwards file.
     *
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     */
    function _createForwardsFile($target, $keeplocal = false)
    {
        // Create the forwarding information
        if (is_a($address = $this->_makeEmailAddress($target), 'PEAR_Error')) {
            return $address;
        }

        // Append the user if they want to save a copy to themselves...
        if ($keeplocal == 'on') {
            $address = $address . ', \\' . $this->_user;
        }

        // Set the forward
        if (is_a($status = $this->_vfs->writeData('', '.forward', $address), 'PEAR_Error')) {
            return $status;
        }

        // Try to change the permissions, but ignore any errors
        $this->_vfs->changePermissions('', '.forward', '0600');
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $password  The password of the user.
     */
    function disableForwarding($password)
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Connect to the server.
        if (is_a($connected = $this->_connect($password), 'PEAR_Error')) {
            return $connected;
        }

        if (is_a($status = $this->_vfs->deleteFile('', $this->_forwardsFile), 'PEAR_Error')) {
            $status->message .= '  '
                . _("Maybe you didn't have a vacation notice installed?");
            return $status;
        }
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @TODO FIXME: This function is implemented poorly, and should be
     * rewritten.
     *
     * @param string $password  The password of the user.
     *
     * @return mixed  Returns 'Y' if forwarding is enabled for the user, 'N' if
     *                forwarding is currently disabled, false if the status
     *                cannot be determined, and PEAR_Error on error.
     */
    function isEnabledForwarding($password)
    {
        $yn = $this->currentTarget($password);
        if (is_a($yn, 'PEAR_Error')) {
            return $yn;
        }
        if ($yn) {
            return 'Y';
        }
        return false;
    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($password)
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Connect to the server.
        if (is_a($connected = $this->_connect($password), 'PEAR_Error')) {
            return $connected;
        }

        if (is_a($status = $this->_vfs->read('', '.forward'), 'PEAR_Error')) {
            return $status;
        }

        return preg_match('/(^|, *)\\\\' . preg_quote($this->_user, '/') . '( *,|$)/',
                          $status);
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, false if no
     *                 forwarding is set, or PEAR_Error on error.
     */
    function currentTarget($password)
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Connect to the server.
        if (is_a($connected = $this->_connect($password), 'PEAR_Error')) {
            return $connected;
        }

        if (!$this->_vfs->exists('', $this->_forwardsFile)) {
            return false;
        }
        if (is_a($status = $this->_vfs->read('', $this->_forwardsFile), 'PEAR_Error')) {
            return $status;
        }

        return $this->_parseForwardsFile($status);
    }

    /**
     * Parses the target information out of the forwards file content.
     *
     * @param string $content  The file content.
     *
     * @return string  The forwards target.
     */
    function _parseForwardsFile($content)
    {
        return preg_replace('/ *(^|, *)\\\\' . preg_quote($this->_user, '/') . '( *, *|$)/',
                            '', $content);
    }

    /**
     * Checks if the realm has a specific configuration.
     *
     * If not, try to fall back on the default configuration. If still not a
     * valid configuration then return an error.
     */
    function _checkConfig()
    {
        // If no host config for the realm, then we fall back to the default
        // realm.
        if (empty($this->_params[$this->_realm]['host'])) {
            $this->_realm = 'default';
        }

        // If still no host/port, then we have a misconfigured module.
        if (empty($this->_params[$this->_realm]['host']) ||
            empty($this->_params[$this->_realm]['port'])) {
            return PEAR::raiseError(_("The module is not properly configured!"));
        }
    }

    /**
     * Connects to the FTP server.
     *
     * @param string $password  The password to connect with.
     */
    function _connect($password)
    {
        if ($this->_vfs) {
            return;
        }

        // Build the FTP array to pass to VFS.
        $args = array('hostspec' => $this->_params[$this->_realm]['host'],
                      'port' => $this->_params[$this->_realm]['port'],
                      'pasv' => $this->_params[$this->_realm]['pasv'],
                      'ssl' => $this->_params[$this->_realm]['ssl'],
                      'username' => $this->_user,
                      'password' => $password);

        // Create the VFS ftp driver.
        require_once 'VFS.php';
        $vfs = &VFS::singleton('ftp', $args);
        if (is_a($vfs, 'PEAR_Error')) {
            return $vfs;
        }
        $this->_vfs = &$vfs;

        // Try to login with the username/password, no realm.
        return $this->_vfs->checkCredentials();
    }

}
