;;; R5RS test suite
;;;
;;; sources include 
;;;   clisp test suite
;;;   Paul Dietz's CL test suite
;;;   R Kelsey, W Clinger, and J Rees r5rs.html (and r6rs.html)
;;;   A Jaffer's r4rstest.scm (the inspiration for this...)
;;;   guile test suite
;;;   gauche test suite
;;;   gambit test suite
;;;   sacla test suite
;;;   Kent Dybvig's "The Scheme Programming Language"
;;;   Brad Lucier (who also pointed out many bugs)
;;;   numtst.c
;;;   GSL tests
;;;   Abramowitz and Stegun, "Handbook of Mathematical Functions"
;;;   Weisstein, "Encyclopedia of Mathematics"
;;;   the arprec package of David Bailey et al
;;;   Maxima, William Schelter et al
;;;   H Cohen, "A Course in Computational Algebraic Number Theory"
;;;   various mailing lists and websites (see individual cases below)


(define with-bignums (provided? 'gmp))                         ; scheme integer has any number of bits
(define with-bigfloats (provided? 'gmp))                       ; scheme real has any number of bits
(define with-bignum-function (defined? 'bignum))               ;   this is a function that turns its string arg into a bignum
(define with-64-bit-ints #t)                                   ; scheme integer has at least 64 bits
(define with-hyperbolic-functions #t)                          ; sinh et al
(define with-char-ops-with-more-than-2-args #t)                ; char<? et al restricted to 2 args?
(define with-string-ops-with-more-than-2-args #t)              ; string<? et al restricted to 2 args?
(define with-relational-ops-that-require-at-least-2-args #t)   ; < et al min args = 2?
(define with-generic-modulo #t)                                ; #f is the Scheme norm
(define with-list-set! #t)                                     ; test list-set!
(define with-reverse! #t)                                      ; test reverse!
(define with-open-input-string-and-friends #t)                 ; string IO, as well as file
(define with-delay #f)                                         ; delay and force 
(define with-bitwise-functions #t)                             ; logand|or|xor|ior, ash
(define with-hash-tables #t)                                   ; make-hash-table and friends
(define with-keywords #t)                                      ; make-keyword, keyword->symbol etc
(define with-gensym #t)                                        ; gensym with optional string arg (prefix)
(define with-format #t)                                        ; simple format tests
(define with-define* #t)                                       ; this tests s7's version of define*
(define with-procedure-arity #t)                               ; procedure-arity and other s7-specific stuff
(define with-error-data #f)                                    ; collect numerical error info and report at end
(define with-the-bug-finding-machine #t)                       ; run the machine (this variable can be set to the number of tries)
					                       ;   the default number of tries is 10000
					                       ;   the machine needs format and random
(define with-error-checks #t)                                  ; include checks for bad arg types etc
(define with-large-powers #t)                                  ; for Gauche
(define with-rationalize #t)                                   ; test rationalize
(define with-full-fledged-random #t)                           ; random exists and can take any kind of numeric args, also make-random-state
(define with-sort! #t)                                         ; sort! exists for both lists and vectors (needs random)
(define with-generic-length #t)                                ; length accepts things other than lists
(define with-vector-for-each #t)                               ; vector-for-each and vector-map are defined


(define our-pi 3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582231725359408128481117450284102701938521105559644622948954930382)


;;; --------------------------------------------------------------------------------

; we're assuming call/cc is defined
;    (define call/cc call-with-current-continuation)
; we also assume there's a "catch" macro trapping all errors
;   if you're not expecting any errors, catch can be:
;   (define (catch tag l1 l2) (l1))

(if (and (defined? 'current-time) ; in Snd
	 (defined? 'mus-rand-seed))
    (set! (mus-rand-seed) (current-time)))


(define (ok? tst result expected)
  (if (not (equal? result expected))
      (begin
	(display tst)
	(display " got ")
	(display result)
	(display " but expected ")
	(display expected)
	(newline) (newline)
	)))


(defmacro test (tst expected) ;(display tst) (newline)
  `(let ((result (catch #t (lambda () ,tst) (lambda args 'error))))
     (ok? ',tst result ,expected)))


;;; the error limits below are pretty expansive in some cases, so with-error-data
;;;   tries to keep a record of the worst case error for each operator.  error-data
;;;   is a list: '(#(op worst-error worst-error-case) ...). 

(define error-data '())

(define (op-error op result expected)

  (define (conjugate n) 
    (make-rectangular (real-part n) (- (imag-part n))))

  (if (and (real? result)
	   (real? expected))
      (/ (abs (- result expected)) (max 1.0 (abs expected)))
      (case op
	((acosh)
	 (/ (magnitude (- (cosh result) (cosh expected)))
	    (max 0.001 (magnitude (cosh expected)))))
	((asin)
	 (/ (min (magnitude (- (sin result) (sin expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (* 10 (magnitude (sin expected))))))
	((acos)
	 (/ (min (magnitude (- (cos result) (cos expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (magnitude (cos expected)))))
	((asinh)
	 (/ (magnitude (- (sinh result) (sinh expected)))
	    (max 0.001 (magnitude (sinh expected)))))
	((atanh)
	 (/ (min (magnitude (- (tanh result) (tanh expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (magnitude (tanh expected)))))
	((atan)
	 (/ (min (magnitude (- (tan result) (tan expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (magnitude (tan expected)))))
	((cosh)
	 (/ (min (magnitude (- result expected))
		 (magnitude (- result (- expected))))
	    (max 0.001 (magnitude expected))))
	(else (/ (magnitude (- result expected)) (max 0.001 (magnitude expected)))))))


(define (check-error tst result expected)
  (if (and (number? result)
	   (number? expected))
      (let ((err (op-error (car tst) result expected)))
	(if (> err 0.0)
	    (letrec ((find-if (lambda (pred l)
				(cond ((null? l) #f)
				      ((pred (car l)) (car l))
				      (else (find-if pred (cdr l)))))))
	      (let* ((op (car tst))
		     (err-op (find-if (lambda (n) (eq? op (vector-ref n 0))) error-data)))
		(if err-op
		    (if (> err (vector-ref err-op 1))
			(begin
			  (vector-set! err-op 1 err)
			  (vector-set! err-op 2 tst)
			  (vector-set! err-op 3 result)
			  (vector-set! err-op 4 expected)))
		    (set! error-data (cons (vector op err tst result expected) error-data)))))))))


(define (types-consistent? n)
  (not (or (and (integer? n) 
	       (or (not (= (denominator n) 1))
		   (not (= n (numerator n)))
		   (not (= (imag-part n) 0))
		   (not (= (floor n) (ceiling n) (truncate n) (round n) n))
		   (not (= n (real-part n)))))
	  (and (rational? n)
	       (not (integer? n))
	       (or (not (= (imag-part n) 0))
		   (= (denominator n) 1)
		   (= (denominator n) 0)
		   (not (= n (real-part n)))
		   (not (= n (/ (numerator n) (denominator n))))))
	  (and (real? n)
	       (not (rational? n))
	       (or (not (= (imag-part n) 0))
		   (not (= n (real-part n)))))
	  (and (complex? n) 
	       (not (real? n))
	       (or (= (imag-part n) 0)
		   (not (= n (+ (real-part n) (* 0+i (imag-part n))))))))))


(define (our-nan? x)
  (or (and (real? x)
	   (not (= 1 (+ (if (positive? x) 1 0)
			(if (negative? x) 1 0)
			(if (zero? x) 1 0)))))
      (and (integer? x)
	   (not (= 1 (+ (if (even? x) 1 0)
			(if (odd? x) 1 0)))))
      (not (= 1 (+ (if (zero? x) 1 0)
		   (if (> (magnitude x) 0) 1 0))))
      (let ((type (+ (if (integer? x) 1 0)
		     (if (rational? x) 2 0)
		     (if (real? x) 4 0)
		     (if (complex? x) 8 0))))
	(and (not (= type 8))
	     (not (= type 12))
	     (not (= type 14))
	     (not (= type 15))))
      (not (zero? (- x x))))) ; inf probably


(define (number-ok? tst result expected)
  (if (and with-error-data
	   (number? result)
	   (number? expected))
      (check-error tst result expected))

  ;; (number? +nan.0) returns #t in Guile and Gauche -- this strikes me as perverse

  (if (not (eq? result expected))
      (if (or (and (not (number? expected))
		   (not (eq? result expected)))
	      (and (number? expected)
		   (or (not (number? result))
		       (our-nan? result)))
	      (and (rational? expected)
		   (exact? expected)
		   (rational? result)
		   (exact? result)
		   (not (= result expected)))
	      (and (or (and (rational? expected) 
			    (exact? expected))
		       (and (rational? result)
			    (exact? result)))
		   (real? expected)
		   (real? result)
		   (> (abs (- result expected)) 1.0e-12))
	      (> (op-error (car tst) result expected) 1.0e-6)
	      (and (number? result)
		   (not (types-consistent? result))))
	  (begin
	    (display tst)
	    (display " got ")
	    (display result)
	    (if (and (rational? result) (not (rational? expected)))
		(begin
		  (display " (")
		  (display (exact->inexact result))
		  (display ")" )))
	    (display " but expected ")
	    (display expected)

	    (if (defined? 'format) 
		(begin
		  (if (and (not (number? expected))
			   (not (eq? result expected)))
		      (format #t ", (eq? ~A ~A) -> #f" result expected)
		      (if (and (number? expected)
			       (or (not (number? result))
				   (our-nan? result)))
			  (begin
			    (if (not (number? result))
				(format #t ", (number? ~A) but not (number? ~A)" expected result)
				(format #t ", (number? ~A) but (nan? ~A)" expected result)))
			  (if (and (rational? expected)
				   (exact? expected)
				   (rational? result)
				   (exact? result)
				   (not (= result expected)))
			      (format #t ", exact results but not (= ~A ~A): ~A" expected result (= result expected))
			      (if (and (or (and (rational? expected) 
						(exact? expected))
					   (and (rational? result)
						(exact? result)))
				       (real? expected)
				       (real? result)
				       (> (abs (- result expected)) 1.0e-12))
				  (format #t ", rational results but diff > 1e-12: ~A" (> (abs (- result expected)) 1.0e-12))
				  (if (< (op-error (car tst) result expected) 1.0e-6)
				      (let ((n result))
					(format #t ", result not internally consistent")
					(if (and (integer? n) 
						 (or (not (= (denominator n) 1))
						     (not (= n (numerator n)))
						     (not (= (imag-part n) 0))
						     (not (= (floor n) (ceiling n) (truncate n) (round n) n))
						     (not (= n (real-part n)))))
					    (format #t ", ~A integer but den: ~A, num: ~A, imag: ~A, real: ~A, floors: ~A ~A ~A ~A"
						    n (denominator n) (numerator n) (imag-part n) (real-part n)
						    (floor n) (ceiling n) (truncate n) (round n))
					    (if (and (rational? n)
						     (not (integer? n))
						     (or (not (= (imag-part n) 0))
							 (= (denominator n) 1)
							 (= (denominator n) 0)
							 (not (= n (real-part n)))
							 (not (= n (/ (numerator n) (denominator n))))))
						(format #t ", ~A ratio but imag: ~A, den: ~A, real: ~A, ~A/~A=~A"
							n (imag-part n) (denominator n) (real-part n) 
							(numerator n) (denominator n) (exact->inexact (/ (numerator n) (denominator n))))
						(if (and (real? n)
							 (not (rational? n))
							 (or (not (= (imag-part n) 0))
							     (not (= n (real-part n)))))
						    (format #t ", ~A real but rational: ~A, imag: ~A, real: ~A"
							    n (rational? n) (imag-part n) (real-part n))
						    (format #t ", ~A complex but real? ~A, imag: ~A, ~A+~A=~A"
							    n (real? n) (imag-part n) (real-part n) (imag-part n)
							    (+ (real-part n) (* 0+i (imag-part n)))))))))))))))
	    (newline) (newline)))))

(defmacro num-test (tst expected) ;(display tst) (newline)
  `(let ((result (catch #t (lambda () ,tst) (lambda args 'error))))
     (number-ok? ',tst result ,expected)
     (if with-bignum-function
	 (letrec ((bigify (lambda (lst)
			    (if (pair? lst)
				(cons (if (number? (car lst))
					  (list 'bignum (number->string (car lst)))
					  (bigify (car lst)))
				      (bigify (cdr lst)))
				lst))))

	   (let* ((big-test (bigify ',tst)))
	     (let ((big-result (catch #t (lambda () (eval big-test)) (lambda args 'error))))
	       (number-ok? big-test big-result ,expected)))
	   ))))



;;; --------------------------------------------------------------------------------
;;; GENERIC STUFF
;;; --------------------------------------------------------------------------------

(test (eq? 'a 3) #f)
(test (eq? #t 't) #f)
(test (eq? "abs" 'abc) #f)
(test (eq? "hi" '(hi)) #f)
(test (eq? "()" '()) #f)
(test (eq? #\a #\b) #f)
(test (eq? #t #t) #t)
(test (eq? #f #f) #t)
(test (eq? #f #t) #f)
(test (eq? (null? '()) #t) #t)
(test (eq? (null? '(a)) #f) #t)
(test (eq? (cdr '(a)) '()) #t)
(test (eq? 'a 'a) #t)
(test (eq? 'a 'b) #f)
(test (eq? 'a (string->symbol "a")) #t)
(test (eq? '(a) '(b)) #f)
(test (let ((x '(a . b))) (eq? x x)) #t)
(test (let ((x (cons 'a 'b))) (eq? x x)) #t)
(test (eq? (cons 'a 'b) (cons 'a 'b)) #f)
(test (eq? "abc" "cba") #f)
(test (let ((x "hi")) (eq? x x)) #t)
(test (eq? (string #\h #\i) (string #\h #\i)) #f)
(test (eq? '#(a) '#(b)) #f)
(test (let ((x (vector 'a))) (eq? x x)) #t)
(test (eq? (vector 'a) (vector 'a)) #f)
(test (eq? car car) #t)
(test (eq? car cdr) #f)
(test (let ((x (lambda () 1))) (eq? x x)) #t)
(test (eq? 'abc 'abc) #t)
(test (eq? eq? eq?) #t)
(test (eq? (if #f 1) 1) #f)
;(test (eq? call/cc call-with-current-continuation) #t)
; guile accepts (eq? #t #t #t)
(test (eq? 3/4 3) #f)
(test (eq? '() '()) #t)
(test (eq? '() (list)) #t)

(let ((things (vector #t #f #\space '() "" 0 1 3/4 1+i 1.5 '(1 .2) '#() (vector) (vector 1) (list 1) 'f 't #\t)))
  (do ((i 0 (+ i 1)))
      ((= i (- (vector-length things) 1)))
    (do ((j (+ i 1) (+ j 1)))
	((= j (vector-length things)))
      (if (eq? (vector-ref things i) (vector-ref things j))
	  (begin
	    (display "(eq? ") (display (vector-ref things i)) (display " ") (display (vector-ref things j)) (display ") -> #t?") (newline))))))



(test (eqv? 'a 3) #f)
(test (eqv? #t 't) #f)
(test (eqv? "abs" 'abc) #f)
(test (eqv? "hi" '(hi)) #f)
(test (eqv? "()" '()) #f)
(test (eqv? #\a #\b) #f)
(test (eqv? #\a #\a) #t)
(test (let ((x (string-ref "hi" 0))) (eqv? x x)) #t)
(test (eqv? #t #t) #t)
(test (eqv? #f #f) #t)
(test (eqv? #f #t) #f)
(test (eqv? (null? '()) #t) #t)
(test (eqv? (null? '(a)) #f) #t)
(test (eqv? (cdr '(a)) '()) #t)
(test (eqv? 'a 'a) #t)
(test (eqv? 'a 'b) #f)
(test (eqv? 'a (string->symbol "a")) #t)
(test (eqv? '(a) '(b)) #f)
(test (let ((x '(a . b))) (eqv? x x)) #t)
(test (let ((x (cons 'a 'b))) (eqv? x x)) #t)
(test (eqv? (cons 'a 'b) (cons 'a 'b)) #f)
(test (eqv? "abc" "cba") #f)
(test (let ((x "hi")) (eqv? x x)) #t)
(test (eqv? (string #\h #\i) (string #\h #\i)) #f)
(test (eqv? '#(a) '#(b)) #f)
(test (let ((x (vector 'a))) (eqv? x x)) #t)
(test (eqv? (vector 'a) (vector 'a)) #f)
(test (eqv? car car) #t)
(test (eqv? car cdr) #f)
(test (let ((x (lambda () 1))) (eqv? x x)) #t)
(test (eqv? 9/2 9/2) #t)
;(test (eqv? 3.4 (+ 3.0 0.4)) #t) ; can be fooled
(test (let ((x 3.141)) (eqv? x x)) #t)
(test (eqv? (cons 'a 'b) (cons 'a 'c)) #f)
(test (eqv? eqv? eqv?) #t)
;(test (let ((quote -)) (eqv? '1 1)) #f)
(test (eqv? '#(1) '#(1)) #f)
(test (eqv? '(1) '(1)) #f)
(test (eqv? '() '()) #t)
(test (eqv? '() (list)) #t)

(let ((things (vector #t #f #\space '() "" 0 1 3/4 1+i 1.5 '(1 .2) '#() (vector) (vector 1) (list 1) 'f 't #\t)))
  (do ((i 0 (+ i 1)))
      ((= i (- (vector-length things) 1)))
    (do ((j (+ i 1) (+ j 1)))
	((= j (vector-length things)))
      (if (eqv? (vector-ref things i) (vector-ref things j))
	  (begin
	    (display "(eqv? ") (display (vector-ref things i)) (display " ") (display (vector-ref things j)) (display ") -> #t?") (newline))))))



(test (equal? 'a 3) #f)
(test (equal? #t 't) #f)
(test (equal? "abs" 'abc) #f)
(test (equal? "hi" '(hi)) #f)
(test (equal? "()" '()) #f)
(test (equal? '(()) '(() . ())) #t)
(test (equal? #\a #\b) #f)
(test (equal? #\a #\a) #t)
(test (let ((x (string-ref "hi" 0))) (equal? x x)) #t)
(test (equal? #t #t) #t)
(test (equal? #f #f) #t)
(test (equal? #f #t) #f)
(test (equal? (null? '()) #t) #t)
(test (equal? (null? '(a)) #f) #t)
(test (equal? (cdr '(a)) '()) #t)
(test (equal? 'a 'a) #t)
(test (equal? 'a 'b) #f)
(test (equal? 'a (string->symbol "a")) #t)
(test (equal? '(a) '(b)) #f)
(test (equal? '(a) '(a)) #t)
(test (let ((x '(a . b))) (equal? x x)) #t)
(test (let ((x (cons 'a 'b))) (equal? x x)) #t)
(test (equal? (cons 'a 'b) (cons 'a 'b)) #t)
(test (equal? "abc" "cba") #f)
(test (equal? "abc" "abc") #t)
(test (let ((x "hi")) (equal? x x)) #t)
(test (equal? (string #\h #\i) (string #\h #\i)) #t)
(test (equal? '#(a) '#(b)) #f)
(test (equal? '#(a) '#(a)) #t)
(test (let ((x (vector 'a))) (equal? x x)) #t)
(test (equal? (vector 'a) (vector 'a)) #t)
(test (equal? '#(1 2) (vector 1 2)) #t)
(test (equal? '#(1.0 2/3) (vector 1.0 2/3)) #t)
(test (equal? '#(1 2) (vector 1 2.0)) #f) ; 2 not equal 2.0!
(test (equal? '(1 . 2) (cons 1 2)) #t)
(test (equal? '#(1 "hi" #\a) (vector 1 "hi" #\a)) #t)
(test (equal? '#((1 . 2)) (vector (cons 1 2))) #t)
(test (equal? '#(1 "hi" #\a (1 . 2)) (vector 1 "hi" #\a (cons 1 2))) #t)
(test (equal? '#(#f hi (1 2) 1 "hi" #\a (1 . 2)) (vector #f 'hi (list 1 2) 1 "hi" #\a (cons 1 2))) #t)
(test (equal? '#(#(1) #(1)) (vector (vector 1) (vector 1))) #t)
(test (equal? '#(()) (vector '())) #t)
(test (equal? '#("hi" "ho") (vector "hi" '"ho")) #t)
(test (equal? (list 1 "hi" #\a) '(1 "hi" #\a)) #t)
(test (equal? (list 1.0 2/3) '(1.0 2/3)) #t)
(test (equal? (list 1 2) '(1 2.0)) #f)
(test (equal? '#(1.0+1.0i) (vector 1.0+1.0i)) #t)
(test (equal? (list 1.0+1.0i) '(1.0+1.0i)) #t)
(test (equal? '((())) (list (list (list)))) #t)
(test (equal? car car) #t)
(test (equal? car cdr) #f)
(test (let ((x (lambda () 1))) (equal? x x)) #t)
(test (equal? 9/2 9/2) #t)
;(test (equal? 3.4 (+ 3.0 0.4)) #t)
(test (let ((x 3.141)) (equal? x x)) #t)
(test (equal? 3 3) #t)
(test (equal? 3 3.0) #f)
(test (equal? 3.0 3.0) #t)
(test (equal? 3-4i 3-4i) #t)
(test (equal? (string #\c) "c") #t)
(test (equal? equal? equal?) #t)
(test (equal? (cons 1 (cons 2 3)) '(1 2 . 3)) #t)
(test (equal? '() '()) #t)
(test (equal? '() (list)) #t)

(test (equal? "asd""asd") #t) ; is this the norm?
(let ((streq (lambda (a b) (equal? a b)))) (test (streq "asd""asd") #t))

(let ((things (vector #t #f #\space '() "" 0 1 3/4 1+i 1.5 '(1 .2) '#() (vector 1) (list 1) 'f 't #\t)))
  (do ((i 0 (+ i 1)))
      ((= i (- (vector-length things) 1)))
    (do ((j (+ i 1) (+ j 1)))
	((= j (vector-length things)))
      (if (equal? (vector-ref things i) (vector-ref things j))
	  (begin
	    (display "(equal? ") (display (vector-ref things i)) (display " ") (display (vector-ref things j)) (display ") -> #t?") (newline))))))




(test (boolean? #f) #t)
(test (boolean? #t) #t)
(test (boolean? 0) #f)
(test (boolean? 1) #f)
(test (boolean? "") #f)
(test (boolean? #\0) #f)
(test (boolean? '()) #f)
(test (boolean? '#()) #f)
(test (boolean? 't) #f)
(test (boolean? (list)) #f)

(for-each
 (lambda (arg)
   (if (boolean? arg)
       (begin
	 (display "(boolean? ") (display arg) (display ") returned #t?") (newline))))
 (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1))))



(test (not #f) #t)
(test (not #t) #f)
(test (not (not #t)) #t)
(test (not 0) #f)
(test (not 1) #f)
(test (not '()) #f)
(test (not 't) #f)
(test (not (list)) #f)
(test (not (list 3)) #f)
(test (not 'nil) #f)
(test (not not) #f)

(for-each
 (lambda (arg)
   (if (not arg)
       (begin
	 (display "(not ") (display arg) (display ") returned #t?") (newline))))
 (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1))))



(test (symbol? 't) #t)
(test (symbol? "t") #f)
(test (symbol? '(t)) #f)
(test (symbol? #t) #f)
(test (symbol? 4) #f)
(test (symbol? 'foo) #t)
(test (symbol? (car '(a b))) #t)
(test (symbol? 'nil) #t)
(test (symbol? '()) #f)
(test (symbol? #f) #f)
(test (symbol? 'car) #t)
(test (symbol? '#f) #f)
(test (symbol? 'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789) #t) ;M Gran
(test (symbol? (vector-ref '#(1 a 34) 1)) #t)

(for-each
 (lambda (arg)
   (if (symbol? arg)
       (begin
	 (display "(symbol? ") (display arg) (display ") returned #t?") (newline))))
 (list "hi" (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1))))



(test (procedure? car) #t)
(test (procedure? procedure?) #t)
(test (procedure? 'car) #f)
(test (procedure? (lambda (x) x)) #t)
(test (procedure? '(lambda (x) x)) #f)
(test (call/cc procedure?) #t) ; ??
(test (let ((a (lambda (x) x)))	(procedure? a)) #t)
(test (letrec ((a (lambda () (procedure? a)))) (a)) #t)
(test (let ((a 1)) (let ((a (lambda () (procedure? a)))) (a))) #f)

(for-each
 (lambda (arg)
   (if (procedure? arg)
       (begin
	 (display "(procedure? ") (display arg) (display ") returned #t?") (newline))))
 (list "hi" (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f))




;;; --------------------------------------------------------------------------------
;;; CHARACTERS
;;; --------------------------------------------------------------------------------

(test (eqv? '#\  #\space) #t)
(test (eqv? #\space '#\space) #t)

(test (char? #\a) #t)
(test (char? #\() #t)
(test (char? #\space) #t)
(test (char? '#\newline) #t)
(test (char? #\1) #t)
(test (char? #\$) #t)
(test (char? #\.) #t)
(test (char? #\\) #t)
(test (char? #\)) #t)
(test (char? #\%) #t)
(test (char? '#\space) #t)
(test (char? '#\ ) #t)
(test (char? '#\newline) #t)
(test (char? '#\a) #t)
(test (char? '#\8) #t)
(test (char? #\-) #t)
(test (char? #\n) #t)
(test (char? #\() #t)
(test (char? #e1) #f)
(test (char? #\#) #t)
(test (char? #\x) #t)
(test (char? #\o) #t)
(test (char? #\b) #t)
(test (char? #b101) #f)
(test (char? #o73) #f)
(test (char? #x73) #f)
(test (char? 'a) #f)
(test (char? 97) #f)
(test (char? "a") #f)
(test (char? (string-ref "hi" 0)) #t)
(test (char? (string-ref (make-string 1) 0)) #t)
(test (char? #\") #t)
(test (char? #\') #t)
(test (char? #\`) #t)
(test (char? #\@) #t)

(for-each
 (lambda (arg)
   (if (char? arg)
       (begin
	 (display "(char? ") (display arg) (display ") returned #t?") (newline))))
 (list "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #f #t (lambda (a) (+ a 1))))

(do ((i 0 (+ i 1)))
    ((= i 256))
   (if (not (char? (integer->char i)))
       (begin
	 (display "(char? (integer->char ") (display i) (display ")) returned #f?") (newline))))

(num-test (let ((str (make-string 258 #\space)))
	    (do ((i 1 (+ i 1)))
		((= i 256))
	      (string-set! str i (integer->char i)))
	    (string-set! str 257 (integer->char 0))
	    (string-length str))
	  258)


(let ((a-to-z (list #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\x #\y #\z))
      (cap-a-to-z (list #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\X #\Y #\Z))
      (mixed-a-to-z (list #\a #\B #\c #\D #\e #\F #\g #\H #\I #\j #\K #\L #\m #\n #\O #\p #\Q #\R #\s #\t #\U #\v #\X #\y #\Z))
      (digits (list #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))


  (test (char-upper-case? #\a) #f)
  (test (char-upper-case? #\A) #t)

  (for-each
   (lambda (arg)
     (if (not (char-upper-case? arg))
	 (begin
	   (display "(char-upper-case? ") (display arg) (display ") returned #f?") (newline))))
   cap-a-to-z)

  (for-each
   (lambda (arg)
     (if (char-upper-case? arg)
	 (begin
	   (display "(char-upper-case? ") (display arg) (display ") returned #t?") (newline))))
   a-to-z)

  ;; non-alpha chars are "unspecified" here


  (test (char-lower-case? #\A) #f)
  (test (char-lower-case? #\a) #t)

  (for-each
   (lambda (arg)
     (if (not (char-lower-case? arg))
	 (begin
	   (display "(char-lower-case? ") (display arg) (display ") returned #f?") (newline))))
   a-to-z)

  (for-each
   (lambda (arg)
     (if (char-lower-case? arg)
	 (begin
	   (display "(char-lower-case? ") (display arg) (display ") returned #t?") (newline))))
   cap-a-to-z)


  (test (char-upcase #\A) #\A)
  (test (char-upcase #\a) #\A)
  (test (char-upcase #\?) #\?)
  (test (char-upcase #\$) #\$)
  (test (char-upcase #\.) #\.)
  (test (char-upcase #\\) #\\)
  (test (char-upcase #\5) #\5)
  (test (char-upcase #\)) #\))
  (test (char-upcase #\%) #\%)
  (test (char-upcase #\0) #\0)
  (test (char-upcase #\_) #\_)
  (test (char-upcase #\space) #\space)
  (test (char-upcase #\newline) #\newline)

  (for-each
   (lambda (arg1 arg2)
     (if (not (char=? (char-upcase arg1) arg2))
	 (begin
	   (display "(char-upcase ") (display arg1) (display ") returned ") (display arg2) (display "?") (newline))))
   a-to-z
   cap-a-to-z)

  (do ((i 1 (+ i 1)))
      ((= i 128))
    (if (and (not (char=? (integer->char i) (char-upcase (integer->char i))))
	     (not (char-alphabetic? (integer->char i))))
	(begin 
	  (display "(char-upcase ") (display (integer->char i)) (display ") -> ")
	  (display (integer->char i)) (display ", but not alphabetic?")
	  (newline))))

  (test (char-downcase #\A) #\a)
  (test (char-downcase #\a) #\a)
  (test (char-downcase #\?) #\?)
  (test (char-downcase #\$) #\$)
  (test (char-downcase #\.) #\.)
  (test (char-downcase #\_) #\_)
  (test (char-downcase #\\) #\\)
  (test (char-downcase #\5) #\5)
  (test (char-downcase #\)) #\))
  (test (char-downcase #\%) #\%)
  (test (char-downcase #\0) #\0)
  (test (char-downcase #\space) #\space)

  (for-each
   (lambda (arg1 arg2)
     (if (not (char=? (char-downcase arg1) arg2))
	 (begin
	   (display "(char-downcase ") (display arg1) (display ") returned ") (display arg2) (display "?") (newline))))
   cap-a-to-z
   a-to-z)


  (test (char-numeric? #\a) #f)
  (test (char-numeric? #\5) #t)
  (test (char-numeric? #\A) #f)
  (test (char-numeric? #\z) #f)
  (test (char-numeric? #\Z) #f)
  (test (char-numeric? #\0) #t)
  (test (char-numeric? #\9) #t)
  (test (char-numeric? #\space) #f)
  (test (char-numeric? #\;) #f)
  (test (char-numeric? #\.) #f)
  (test (char-numeric? #\-) #f)

  (for-each
   (lambda (arg)
     (if (char-numeric? arg)
	 (begin
	   (display "(char-numeric? ") (display arg) (display ") returned #t?") (newline))))
   cap-a-to-z)

  (for-each
   (lambda (arg)
     (if (char-numeric? arg)
	 (begin
	   (display "(char-numeric? ") (display arg) (display ") returned #t?") (newline))))
   a-to-z)


  (test (char-whitespace? #\a) #f)
  (test (char-whitespace? #\A) #f)
  (test (char-whitespace? #\z) #f)
  (test (char-whitespace? #\Z) #f)
  (test (char-whitespace? #\0) #f)
  (test (char-whitespace? #\9) #f)
  (test (char-whitespace? #\space) #t)
  (test (char-whitespace? #\newline) #t)
  (test (char-whitespace? #\;) #f)

  (for-each
   (lambda (arg)
     (if (char-whitespace? arg)
	 (begin
	   (display "(char-whitespace? ") (display arg) (display ") returned #t?") (newline))))
   mixed-a-to-z)

  (for-each
   (lambda (arg)
     (if (char-whitespace? arg)
	 (begin
	   (display "(char-whitespace? ") (display arg) (display ") returned #t?") (newline))))
   digits)


  (test (char-alphabetic? #\a) #t)
  (test (char-alphabetic? #\$) #f)
  (test (char-alphabetic? #\A) #t)
  (test (char-alphabetic? #\z) #t)
  (test (char-alphabetic? #\Z) #t)
  (test (char-alphabetic? #\0) #f)
  (test (char-alphabetic? #\9) #f)
  (test (char-alphabetic? #\space) #f)
  (test (char-alphabetic? #\;) #f)
  (test (char-alphabetic? #\.) #f)
  (test (char-alphabetic? #\-) #f)
  (test (char-alphabetic? #\_) #f)
  (test (char-alphabetic? #\^) #f)
  (test (char-alphabetic? #\[) #f)

  (for-each
   (lambda (arg)
     (if (char-alphabetic? arg)
	 (begin
	   (display "(char-alphabetic? ") (display arg) (display ") returned #t?") (newline))))
   digits)

  (for-each
   (lambda (arg)
     (if (not (char-alphabetic? arg))
	 (begin
	   (display "(char-alphabetic? ") (display arg) (display ") returned #f?") (newline))))
   mixed-a-to-z)

  (test 
   (let ((unhappy '()))
     (do ((i 0 (+ i 1))) 
	 ((= i 256)) 
       (let* ((ch (integer->char i))
	      (chu (char-upcase ch))
	      (chd (char-downcase ch)))
	 (if (or (and (not (char=? ch chu))
		      (not (char=? ch (char-downcase chu))))
		 (and (not (char=? ch chd))
		      (not (char=? ch (char-upcase chd))))
		 (and (not (char=? ch chd))
		      (not (char=? ch chu)))
		 (not (char-ci=? chu chd))
		 (not (char-ci=? ch chu))
		 (and (char-alphabetic? ch)
		      (or (not (char-alphabetic? chd))
			  (not (char-alphabetic? chu))))
		 (and (char-numeric? ch)
		      (or (not (char-numeric? chd))
			  (not (char-numeric? chu))))
		 (and (char-whitespace? ch)
		      (or (not (char-whitespace? chd))
			  (not (char-whitespace? chu))))
		 (and (char-alphabetic? ch)
		      (char-whitespace? ch))
		 (and (char-numeric? ch)
		      (char-whitespace? ch))
		 (and (char-alphabetic? ch)
		      (char-numeric? ch)))
	     ;; there are characters that are alphabetic but the result of char-upcase is not an upper-case character
	     ;; 223 for example, or 186 for lower case
	     (set! unhappy (cons (format #f "~C: ~C ~C (~D)~%" ch chu chd i) unhappy)))))
     unhappy)
   '())


  (test (char=? #\d #\d) #t)
  (test (char=? #\A #\a) #f)
  (test (char=? #\d #\x) #f)
  (test (char=? #\d #\D) #f)
  (test (char=? #\a #\a) #t)
  (test (char=? #\A #\B) #f)
  (test (char=? #\a #\b) #f)
  (test (char=? #\9 #\0) #f)
  (test (char=? #\A #\A) #t)
  (test (char=? #\  #\space) #t)
  (let ((i (char->integer #\space)))
    (test (char=? (integer->char i) #\space) #t))
  (test (char=? (integer->char (char->integer #\")) #\") #t)

  (if with-char-ops-with-more-than-2-args (begin
    (test (char=? #\d #\d #\d #\d) #t)
    (test (char=? #\d #\d #\x #\d) #f)
    (test (char=? #\d #\y #\x #\c) #f)
    (test (apply char=? cap-a-to-z) #f)
    (test (apply char=? mixed-a-to-z) #f)
    (test (apply char=? digits) #f)
    (test (char=? #\d #\c #\d) #f)))


  (test (char<? #\z #\0) #f)
  (test (char<? #\d #\x) #t)
  (test (char<? #\d #\d) #f)
  (test (char<? #\d #\x) #t)
  (test (char<? #\A #\B) #t)
  (test (char<? #\a #\b) #t)
  (test (char<? #\9 #\0) #f)
  (test (char<? #\A #\A) #f)
  (test (char<? #\space #\space) #f)

  (if with-char-ops-with-more-than-2-args (begin
    (test (char<? #\a #\e #\y #\z) #t)
    (test (char<? #\a #\e #\e #\y) #f)
    (test (apply char<? a-to-z) #t)
    (test (apply char<? cap-a-to-z) #t)
    (test (apply char<? mixed-a-to-z) #f)
    (test (apply char<? digits) #t)
    (test (apply char<? (reverse a-to-z)) #f)
    (test (apply char<? (reverse cap-a-to-z)) #f)
    (test (apply char<? (reverse mixed-a-to-z)) #f)
    (test (apply char<? (reverse digits)) #f)
    (test (char<? #\b #\c #\a) #f)
    (test (char<? #\B #\B #\A) #f)
    (test (char<? #\b #\c #\e) #t)))


  (test (char<=? #\d #\x) #t)
  (test (char<=? #\d #\d) #t)
  (if with-char-ops-with-more-than-2-args (begin
    (test (char<=? #\a #\e #\y #\z) #t)
    (test (char<=? #\a #\e #\e #\y) #t)))
  (test (char<=? #\A #\B) #t)
  (test (char<=? #\a #\b) #t)
  (test (char<=? #\9 #\0) #f)
  (test (char<=? #\A #\A) #t)
  (test (char<=? #\space #\space) #t)

  (if with-char-ops-with-more-than-2-args (begin
    (test (char<=? #\a #\e #\y #\z) #t)
    (test (char<=? #\a #\e #\e #\y) #t)
    (test (char<=? #\e #\e #\d #\y) #f)
    (test (apply char<=? a-to-z) #t)
    (test (apply char<=? cap-a-to-z) #t)
    (test (apply char<=? mixed-a-to-z) #f)
    (test (apply char<=? digits) #t)
    (test (apply char<=? (reverse a-to-z)) #f)
    (test (apply char<=? (reverse cap-a-to-z)) #f)
    (test (apply char<=? (reverse mixed-a-to-z)) #f)
    (test (apply char<=? (reverse digits)) #f)
    (test (char<=? #\b #\c #\a) #f)
    (test (char<=? #\B #\B #\A) #f)
    (test (char<=? #\b #\c #\e) #t)))
  
  
  (test (char>? #\e #\d) #t)
  (test (char>? #\z #\a) #t)
  (test (char>? #\A #\B) #f)
  (test (char>? #\a #\b) #f)
  (test (char>? #\9 #\0) #t)
  (test (char>? #\A #\A) #f)
  (test (char>? #\space #\space) #f)

  (if with-char-ops-with-more-than-2-args (begin
    (test (char>? #\d #\c #\b #\a) #t)
    (test (char>? #\d #\d #\c #\a) #f)
    (test (char>? #\e #\d #\b #\c #\a) #f)
    (test (apply char>? a-to-z) #f)
    (test (apply char>? cap-a-to-z) #f)
    (test (apply char>? mixed-a-to-z) #f)
    (test (apply char>? digits) #f)
    (test (apply char>? (reverse a-to-z)) #t)
    (test (apply char>? (reverse cap-a-to-z)) #t)
    (test (apply char>? (reverse mixed-a-to-z)) #f)
    (test (apply char>? (reverse digits)) #t)
    (test (char>? #\d #\c #\a) #t)
    (test (char>? #\d #\c #\c) #f)
    (test (char>? #\B #\B #\C) #f)
    (test (char>? #\b #\c #\e) #f)))


  (test (char>=? #\e #\d) #t)
  (test (char>=? #\A #\B) #f)
  (test (char>=? #\a #\b) #f)
  (test (char>=? #\9 #\0) #t)
  (test (char>=? #\A #\A) #t)
  (test (char>=? #\space #\space) #t)

  (if with-char-ops-with-more-than-2-args (begin
    (test (char>=? #\d #\c #\b #\a) #t)
    (test (char>=? #\d #\d #\c #\a) #t)
    (test (char>=? #\e #\d #\b #\c #\a) #f)
    (test (apply char>=? a-to-z) #f)
    (test (apply char>=? cap-a-to-z) #f)
    (test (apply char>=? mixed-a-to-z) #f)
    (test (apply char>=? digits) #f)
    (test (apply char>=? (reverse a-to-z)) #t)
    (test (apply char>=? (reverse cap-a-to-z)) #t)
    (test (apply char>=? (reverse mixed-a-to-z)) #f)
    (test (apply char>=? (reverse digits)) #t)
    (test (char>=? #\d #\c #\a) #t)
    (test (char>=? #\d #\c #\c) #t)
    (test (char>=? #\B #\B #\C) #f)
    (test (char>=? #\b #\c #\e) #f)))


  (test (char-ci=? #\A #\B) #f)
  (test (char-ci=? #\a #\B) #f)
  (test (char-ci=? #\A #\b) #f)
  (test (char-ci=? #\a #\b) #f)
  (test (char-ci=? #\9 #\0) #f)
  (test (char-ci=? #\A #\A) #t)
  (test (char-ci=? #\A #\a) #t)
  (test (char-ci=? #\a #\A) #t)
  (test (char-ci=? #\space #\space) #t)

  (if with-char-ops-with-more-than-2-args (begin
    (test (char-ci=? #\d #\D #\d #\d) #t)
    (test (char-ci=? #\d #\d #\X #\d) #f)
    (test (char-ci=? #\d #\Y #\x #\c) #f)
    (test (apply char-ci=? cap-a-to-z) #f)
    (test (apply char-ci=? mixed-a-to-z) #f)
    (test (apply char-ci=? digits) #f)
    (test (char-ci=? #\d #\c #\d) #f)))

  
  (test (char-ci<? #\A #\B) #t)
  (test (char-ci<? #\a #\B) #t)
  (test (char-ci<? #\A #\b) #t)
  (test (char-ci<? #\a #\b) #t)
  (test (char-ci<? #\9 #\0) #f)
  (test (char-ci<? #\0 #\9) #t)
  (test (char-ci<? #\A #\A) #f)
  (test (char-ci<? #\A #\a) #f)
  (test (char-ci<? #\Y #\_) #t)
  (test (char-ci<? #\\ #\J) #f)
  (test (char-ci<? #\_ #\e) #f)
  (test (char-ci<? #\t #\_) #t)
  (test (char-ci<? #\a #\]) #t)
  (test (char-ci<? #\z #\^) #t)


;;; this tries them all:
;(do ((i 0 (+ i 1)))
;    ((= i 128))
;  (do ((k 0 (+ k 1)))
;      ((= k 128))
;    (let ((c1 (integer->char i))
;	  (c2 (integer->char k)))
;      (for-each
;       (lambda (op1 op2)
;	 (if (not (eq? (op1 c1 c2) (op2 (string c1) (string c2))))
;	     (format #t "(~A|~A ~A ~A) -> ~A|~A~%" op1 op2 c1 c2 (op1 c1 c2) (op2 (string c1) (string c2)))))
;       (list char=? char<? char<=? char>? char>=? char-ci=? char-ci<? char-ci<=? char-ci>? char-ci>=?)
;       (list string=? string<? string<=? string>? string>=? string-ci=? string-ci<? string-ci<=? string-ci>? string-ci>=?)))))


  (if with-char-ops-with-more-than-2-args (begin
    (test (char-ci<? #\d #\D #\d #\d) #f)
    (test (char-ci<? #\d #\d #\X #\d) #f)
    (test (char-ci<? #\d #\Y #\x #\c) #f)
    (test (apply char-ci<? cap-a-to-z) #t)
    (test (apply char-ci<? mixed-a-to-z) #t)
    (test (apply char-ci<? digits) #t)
    (test (char-ci<? #\d #\c #\d) #f)
    (test (char-ci<? #\b #\c #\a) #f)
    (test (char-ci<? #\b #\C #\e) #t)
    (test (char-ci<? #\3 #\? #\Z #\[) #t)
    ))

  
  (test (char-ci>? #\A #\B) #f)
  (test (char-ci>? #\a #\B) #f)
  (test (char-ci>? #\A #\b) #f)
  (test (char-ci>? #\a #\b) #f)
  (test (char-ci>? #\9 #\0) #t)
  (test (char-ci>? #\A #\A) #f)
  (test (char-ci>? #\A #\a) #f)
  (test (char-ci>? #\^ #\a) #t)
  (test (char-ci>? #\_ #\e) #t)
  (test (char-ci>? #\[ #\S) #t)
  (test (char-ci>? #\\ #\l) #t)
  (test (char-ci>? #\t #\_) #f)
  (test (char-ci>? #\a #\]) #f)
  (test (char-ci>? #\z #\^) #f)
  (test (char-ci>? #\] #\X) #t)

  (if with-char-ops-with-more-than-2-args (begin
    (test (char-ci>? #\d #\D #\d #\d) #f)
    (test (char-ci>? #\d #\d #\X #\d) #f)
    (test (char-ci>? #\d #\Y #\x #\c) #f)
    (test (apply char-ci>? cap-a-to-z) #f)
    (test (apply char-ci>? mixed-a-to-z) #f)
    (test (apply char-ci>? (reverse mixed-a-to-z)) #t)
    (test (apply char-ci>? digits) #f)
    (test (char-ci>? #\d #\c #\d) #f)
    (test (char-ci>? #\b #\c #\a) #f)
    (test (char-ci>? #\d #\C #\a) #t)))

  
  (test (char-ci<=? #\A #\B) #t)
  (test (char-ci<=? #\a #\B) #t)
  (test (char-ci<=? #\A #\b) #t)
  (test (char-ci<=? #\a #\b) #t)
  (test (char-ci<=? #\9 #\0) #f)
  (test (char-ci<=? #\A #\A) #t)
  (test (char-ci<=? #\A #\a) #t)
  (test (char-ci<=? #\` #\H) #f)
  (test (char-ci<=? #\[ #\m) #f)
  (test (char-ci<=? #\j #\`) #t)
  (test (char-ci<=? #\\ #\E) #f)
  (test (char-ci<=? #\t #\_) #t)
  (test (char-ci<=? #\a #\]) #t)
  (test (char-ci<=? #\z #\^) #t)

  (if with-char-ops-with-more-than-2-args (begin
    (test (char-ci<=? #\d #\D #\d #\d) #t)
    (test (char-ci<=? #\d #\d #\X #\d) #f)
    (test (char-ci<=? #\d #\Y #\x #\c) #f)
    (test (apply char-ci<=? cap-a-to-z) #t)
    (test (apply char-ci<=? mixed-a-to-z) #t)
    (test (apply char-ci<=? digits) #t)
    (test (char-ci<=? #\d #\c #\d) #f)
    (test (char-ci<=? #\b #\c #\a) #f)
    (test (char-ci<=? #\b #\c #\C) #t)
    (test (char-ci<=? #\b #\C #\e) #t)))

  
  (test (char-ci>=? #\A #\B) #f)
  (test (char-ci>=? #\a #\B) #f)
  (test (char-ci>=? #\A #\b) #f)
  (test (char-ci>=? #\a #\b) #f)
  (test (char-ci>=? #\9 #\0) #t)
  (test (char-ci>=? #\A #\A) #t)
  (test (char-ci>=? #\A #\a) #t)
  (test (char-ci>=? #\Y #\_) #f)
  (test (char-ci>=? #\` #\S) #t)
  (test (char-ci>=? #\[ #\Y) #t)
  (test (char-ci>=? #\t #\_) #f)
  (test (char-ci>=? #\a #\]) #f)
  (test (char-ci>=? #\z #\^) #f)

  (if with-char-ops-with-more-than-2-args (begin
    (test (char-ci>=? #\d #\D #\d #\d) #t)
    (test (char-ci>=? #\d #\d #\X #\d) #f)
    (test (char-ci>=? #\d #\Y #\x #\c) #f)
    (test (apply char-ci>=? cap-a-to-z) #f)
    (test (apply char-ci>=? mixed-a-to-z) #f)
    (test (apply char-ci>=? (reverse mixed-a-to-z)) #t)
    (test (apply char-ci>=? (reverse mixed-a-to-z)) #t)
    (test (apply char-ci>=? digits) #f)
    (test (char-ci>=? #\d #\c #\d) #f)
    (test (char-ci>=? #\b #\c #\a) #f)
    (test (char-ci>=? #\d #\D #\a) #t)
    (test (char-ci>=? #\\ #\J #\+) #t)
    ))

  ) ; end let with a-to-z


(test (integer->char (char->integer #\.)) #\.)
(test (integer->char (char->integer #\A)) #\A)
(test (integer->char (char->integer #\a)) #\a)
(test (integer->char (char->integer #\space)) #\space)




;;; --------------------------------------------------------------------------------
;;; STRINGS
;;; --------------------------------------------------------------------------------

(test (string? "abc") #t)
(test (string? ':+*/-) #f)
(test (string? "das ist einer der teststrings") #t)
(test (string? '(das ist natuerlich falsch)) #f)
(test (string? "aaaaaa") #t)
(test (string? #\a) #f)
(test (string? "\"\\\"") #t)

(for-each
 (lambda (arg)
   (test (string? arg) #f))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (lambda (a) (+ a 1))))


(test (string=? "foo" "foo") #t)
(test (string=? "foo" "FOO") #f)
(test (string=? "foo" "bar") #f)
(test (string=? "FOO" "FOO") #t)
(test (string=? "A" "B") #f)
(test (string=? "a" "b") #f)
(test (string=? "9" "0") #f)
(test (string=? "A" "A") #t)
(test (string=? "" "") #t)
(test (string=? (string #\newline) (string #\newline)) #t)
(if with-string-ops-with-more-than-2-args (begin
  (test (string=? "A" "B" "a") #f)
  (test (string=? "A" "A" "a") #f)
  (test (string=? "A" "A" "A") #t)
  (test (string=? "foo" "foo" "foo") #t)
  (test (string=? "foo" "foo" "") #f)
  (test (string=? "foo" "foo" "fOo") #f)))

(test (let ((str (string #\" #\1 #\\ #\2 #\")))	(string=? str "\"1\\2\"")) #t)
(test (let ((str (string #\\ #\\ #\\)))	(string=? str "\\\\\\")) #t)
(test (let ((str (string #\")))	(string=? str "\"")) #t)
(test (let ((str (string #\\ #\"))) (string=? str "\\\"")) #t)
(test (let ((str (string #\space #\? #\)))) (string=? str " ?)")) #t)
(test (let ((str (string #\# #\\ #\t))) (string=? str "#\\t")) #t)
(test (let ((str (string #\# #\\ #\t))) (string-length str)) 3)


(test (string<? "aaaa" "aaab") #t)
(test (string<? "aaaa" "aaaaa") #t)
(test (string<? "" "abcdefgh") #t)
(test (string<? "a" "abcdefgh") #t)
(test (string<? "abc" "abcdefgh") #t)
(test (string<? "cabc" "abcdefgh") #f)
(test (string<? "abcdefgh" "abcdefgh") #f)
(test (string<? "xyzabc" "abcdefgh") #f)
(test (string<? "abc" "xyzabcdefgh") #t)
(test (string<? "abcdefgh" "") #f)
(test (string<? "abcdefgh" "a") #f)
(test (string<? "abcdefgh" "abc") #f)
(test (string<? "abcdefgh" "cabc") #t)
(test (string<? "abcdefgh" "xyzabc") #t)
(test (string<? "xyzabcdefgh" "abc") #f)
(test (string<? "abcdef" "bcdefgh") #t)
(test (string<? "" "abcdefgh") #t)
(test (string<? "" "") #f)
(test (string<? "A" "B") #t)
(test (string<? "a" "b") #t)
(test (string<? "9" "0") #f)
(test (string<? "A" "A") #f)
(if with-string-ops-with-more-than-2-args (begin
  (test (string<? "A" "B" "A") #f)
  (test (string<? "A" "A" "B") #f)
  (test (string<? "A" "A" "A") #f)
  (test (string<? "B" "B" "C") #f)
  (test (string<? "foo" "foo" "foo") #f)
  (test (string<? "foo" "foo" "") #f)
  (test (string<? "foo" "foo" "fOo") #f)))

(test (string>? "aaab" "aaaa") #t)
(test (string>? "aaaaa" "aaaa") #t)
(test (string>? "" "abcdefgh") #f)
(test (string>? "a" "abcdefgh") #f)
(test (string>? "abc" "abcdefgh") #f)
(test (string>? "cabc" "abcdefgh") #t)
(test (string>? "abcdefgh" "abcdefgh") #f)
(test (string>? "xyzabc" "abcdefgh") #t)
(test (string>? "abc" "xyzabcdefgh") #f)
(test (string>? "abcdefgh" "") #t)
(test (string>? "abcdefgh" "a") #t)
(test (string>? "abcdefgh" "abc") #t)
(test (string>? "abcdefgh" "cabc") #f)
(test (string>? "abcdefgh" "xyzabc") #f)
(test (string>? "xyzabcdefgh" "abc") #t)
(test (string>? "abcde" "bc") #f)
(test (string>? "bcdef" "abcde") #t)
(test (string>? "bcdef" "abcdef") #t)
(test (string>? "" "") #f)
(test (string>? "A" "B") #f)
(test (string>? "a" "b") #f)
(test (string>? "9" "0") #t)
(test (string>? "A" "A") #f)
(if with-string-ops-with-more-than-2-args (begin
  (test (string>? "A" "B" "a") #f)
  (test (string>? "C" "B" "A") #t)
  (test (string>? "A" "A" "A") #f)
  (test (string>? "B" "B" "A") #f)
  (test (string>? "foo" "foo" "foo") #f)
  (test (string>? "foo" "foo" "") #f)
  (test (string>? "foo" "foo" "fOo") #f)))



(test (string<=? "aaa" "aaaa") #t)
(test (string<=? "aaaaa" "aaaa") #f)
(test (string<=? "a" "abcdefgh") #t)
(test (string<=? "abc" "abcdefgh") #t)
(test (string<=? "aaabce" "aaabcdefgh") #f)
(test (string<=? "cabc" "abcdefgh") #f)
(test (string<=? "abcdefgh" "abcdefgh") #t)
(test (string<=? "xyzabc" "abcdefgh") #f)
(test (string<=? "abc" "xyzabcdefgh") #t)
(test (string<=? "abcdefgh" "") #f)
(test (string<=? "abcdefgh" "a") #f)
(test (string<=? "abcdefgh" "abc") #f)
(test (string<=? "abcdefgh" "cabc") #t)
(test (string<=? "abcdefgh" "xyzabc") #t)
(test (string<=? "xyzabcdefgh" "abc") #f)
(test (string<=? "abcdef" "bcdefgh") #t)
(test (string<=? "" "") #t)
(test (string<=? "A" "B") #t)
(test (string<=? "a" "b") #t)
(test (string<=? "9" "0") #f)
(test (string<=? "A" "A") #t)
(if with-string-ops-with-more-than-2-args (begin
  (test (string<=? "A" "B" "C") #t)
  (test (string<=? "C" "B" "A") #f)
  (test (string<=? "A" "B" "B") #t)
  (test (string<=? "A" "A" "A") #t)
  (test (string<=? "B" "B" "A") #f)
  (test (string<=? "foo" "foo" "foo") #t)
  (test (string<=? "foo" "foo" "") #f)
  (test (string<=? "foo" "foo" "fooo") #t)))


(test (string>=? "aaaaa" "aaaa") #t)
(test (string>=? "aaaa" "aaaa") #t)
(test (string>=? "aaa" "aaaa") #f)
(test (string>=? "" "abcdefgh") #f)
(test (string>=? "a" "abcdefgh") #f)
(test (string>=? "abc" "abcdefgh") #f)
(test (string>=? "cabc" "abcdefgh") #t)
(test (string>=? "abcdefgh" "abcdefgh") #t)
(test (string>=? "xyzabc" "abcdefgh") #t)
(test (string>=? "abc" "xyzabcdefgh") #f)
(test (string>=? "abcdefgh" "") #t)
(test (string>=? "abcdefgh" "a") #t)
(test (string>=? "abcdefgh" "abc") #t)
(test (string>=? "abcdefgh" "cabc") #f)
(test (string>=? "abcdefgh" "xyzabc") #f)
(test (string>=? "xyzabcdefgh" "abc") #t)
(test (string>=? "bcdef" "abcdef") #t)
(test (string>=? "A" "B") #f)
(test (string>=? "a" "b") #f)
(test (string>=? "9" "0") #t)
(test (string>=? "A" "A") #t)
(test (string>=? "" "") #t)
(if with-string-ops-with-more-than-2-args (begin
  (test (string>=? "A" "B" "C") #f)
  (test (string>=? "C" "B" "A") #t)
  (test (string>=? "C" "B" "B") #t)
  (test (string>=? "A" "B" "B") #f)
  (test (string>=? "A" "A" "A") #t)
  (test (string>=? "B" "B" "A") #t)
  (test (string>=? "B" "B" "C") #f)
  (test (string>=? "foo" "foo" "foo") #t)
  (test (string>=? "foo" "foo" "") #t)
  (test (string>=? "foo" "foo" "fo") #t)))


(test (string-ci=? "A" "B") #f)
(test (string-ci=? "a" "B") #f)
(test (string-ci=? "A" "b") #f)
(test (string-ci=? "a" "b") #f)
(test (string-ci=? "9" "0") #f)
(test (string-ci=? "A" "A") #t)
(test (string-ci=? "A" "a") #t)
(test (string-ci=? "" "") #t)
(test (string-ci=? "aaaa" "AAAA") #t)
(test (string-ci=? "aaaa" "Aaaa") #t)
(if with-string-ops-with-more-than-2-args (begin
  (test (string-ci=? "A" "B" "a") #f)
  (test (string-ci=? "A" "A" "a") #t)
  (test (string-ci=? "A" "A" "a") #t)
  (test (string-ci=? "foo" "foo" "foo") #t)
  (test (string-ci=? "foo" "foo" "") #f)
  (test (string-ci=? "foo" "Foo" "fOo") #t)))


(test (string-ci<? "a" "Aa") #t)
(test (string-ci<? "A" "B") #t)
(test (string-ci<? "a" "B") #t)
(test (string-ci<? "A" "b") #t)
(test (string-ci<? "a" "b") #t)
(test (string-ci<? "9" "0") #f)
(test (string-ci<? "0" "9") #t)
(test (string-ci<? "A" "A") #f)
(test (string-ci<? "A" "a") #f)
(test (string-ci<? "" "") #f)

(test (string-ci<? "t" "_") #t)
(test (string-ci<? "a" "]") #t)
(test (string-ci<? "z" "^") #t)
(test (string-ci<? "]4.jVKo\\\\^:\\A9Z4" "MImKA[mNv1`") #f)

(if with-string-ops-with-more-than-2-args (begin
  (test (string-ci<? "A" "B" "A") #f)
  (test (string-ci<? "A" "A" "B") #f)
  (test (string-ci<? "A" "A" "A") #f)
  (test (string-ci<? "B" "B" "C") #f)
  (test (string-ci<? "B" "b" "C") #f)
  (test (string-ci<? "foo" "foo" "foo") #f)
  (test (string-ci<? "foo" "foo" "") #f)
  (test (string-ci<? "foo" "foo" "fOo") #f)
  (test (string-ci<? "34ZsfQD<obff33FBPFl" "7o" "9l7OM" "FC?M63=" "rLM5*J") #t)
  (test (string-ci<? "NX7" "-;h>P" "DMhk3Bg") #f)
  (test (string-ci<? "+\\mZl" "bE7\\e(HaW5CDXbPi@U_" "B_") #t)
  ))


(test (string-ci>? "Aaa" "AA") #t)
(test (string-ci>? "A" "B") #f)
(test (string-ci>? "a" "B") #f)
(test (string-ci>? "A" "b") #f)
(test (string-ci>? "a" "b") #f)
(test (string-ci>? "9" "0") #t)
(test (string-ci>? "A" "A") #f)
(test (string-ci>? "A" "a") #f)
(test (string-ci>? "" "") #f)
(test (string-ci>? "Z" "DjNTl0") #t)
(test (string-ci>? "2399dt7BVN[,A" "^KHboHV") #f)

(test (string-ci>? "t" "_") #f)
(test (string-ci>? "a" "]") #f)
(test (string-ci>? "z" "^") #f)
(test (string-ci>? "R*95oG.k;?" "`2?J6LBbLG^alB[fMD") #f)
(test (string-ci>? "]" "X") #t)

(if with-string-ops-with-more-than-2-args (begin
  (test (string-ci>? "A" "B" "a") #f)
  (test (string-ci>? "C" "b" "A") #t)
  (test (string-ci>? "a" "A" "A") #f)
  (test (string-ci>? "B" "B" "A") #f)
  (test (string-ci>? "foo" "foo" "foo") #f)
  (test (string-ci>? "foo" "foo" "") #f)
  (test (string-ci>? "foo" "foo" "fOo") #f)
  (test (string-ci>? "ZNiuEa@/V" "KGbKliYMY" "9=69q3ica" ":]") #f)
  (test (string-ci>? "^" "aN@di;iEO" "7*9q6uPmX9)PaY,6J" "15vH") #t)
  ))

 
(test (string-ci<=? "A" "B") #t)
(test (string-ci<=? "a" "B") #t)
(test (string-ci<=? "A" "b") #t)
(test (string-ci<=? "a" "b") #t)
(test (string-ci<=? "9" "0") #f)
(test (string-ci<=? "A" "A") #t)
(test (string-ci<=? "A" "a") #t)
(test (string-ci<=? "" "") #t)
(test (string-ci<=? ":LPC`" ",O0>affA?(") #f)

(test (string-ci<=? "t" "_") #t)
(test (string-ci<=? "a" "]") #t)
(test (string-ci<=? "z" "^") #t)
(test (string-ci<=? "G888E>beF)*mwCNnagP" "`2uTd?h") #t)

(if with-string-ops-with-more-than-2-args (begin
  (test (string-ci<=? "A" "b" "C") #t)
  (test (string-ci<=? "c" "B" "A") #f)
  (test (string-ci<=? "A" "B" "B") #t)
  (test (string-ci<=? "a" "A" "A") #t)
  (test (string-ci<=? "B" "b" "A") #f)
  (test (string-ci<=? "foo" "foo" "foo") #t)
  (test (string-ci<=? "foo" "foo" "") #f)
  (test (string-ci<=? "FOO" "fOo" "fooo") #t)
  (test (string-ci<=? "78mdL82*" "EFaCrIdm@_D+" "eMu\\@dSSY") #t)
  (test (string-ci<=? "`5pNuFc3PM<rNs" "e\\Su_raVNk6HD" "vXnuN7?S0?S(w+M?p") #f)
  ))


(test (string-ci>=? "A" "B") #f)
(test (string-ci>=? "a" "B") #f)
(test (string-ci>=? "A" "b") #f)
(test (string-ci>=? "a" "b") #f)
(test (string-ci>=? "9" "0") #t)
(test (string-ci>=? "A" "A") #t)
(test (string-ci>=? "A" "a") #t)
(test (string-ci>=? "" "") #t)
(test (string-ci>=? "5d7?[o[:hop=ktv;9)" "p^r9;TAXO=^") #f)

(test (string-ci>=? "t" "_") #f)
(test (string-ci>=? "a" "]") #f)
(test (string-ci>=? "z" "^") #f)
(test (string-ci>=? "jBS" "`<+s[[:`l") #f)

(if with-string-ops-with-more-than-2-args (begin
  (test (string-ci>=? "A" "b" "C") #f)
  (test (string-ci>=? "C" "B" "A") #t)
  (test (string-ci>=? "C" "B" "b") #t)
  (test (string-ci>=? "a" "B" "B") #f)
  (test (string-ci>=? "A" "A" "A") #t)
  (test (string-ci>=? "B" "B" "A") #t)
  (test (string-ci>=? "B" "b" "C") #f)
  (test (string-ci>=? "foo" "foo" "foo") #t)
  (test (string-ci>=? "foo" "foo" "") #t)
  (test (string-ci>=? "foo" "foo" "fo") #t)
  (test (string-ci>=? "tF?8`Sa" "NIkMd7" "f`" "1td-Z?teE" "-ik1SK)hh)Nq].>") #t)
  (test (string-ci>=? "Z6a8P" "^/VpmWwt):?o[a9\\_N" "8[^h)<KX?[utsc") #f)
  ))



(test (string-length "abc") 3)
(test (string-length "") 0)
(test (string-length (string)) 0)
(test (string-length "\"\\\"") 3)
(test (string-length (string #\newline)) 1)
(test (string-length "hi there") 8)
(test (string-length "\"") 1)
(test (string-length "\\") 1)
(test (string-length "\n") 1)
(test (string-length (make-string 100 #\a)) 100)
(test (string-length "1\\2") 3)
(test (string-length "1\\") 2)
(test (string-length "hi\\") 3)
(test (string-length "\\\\\\\"") 4)

(test (string=? (string) "") #t)
(test (string=? (string) (make-string 0)) #t)
(test (string=? (string-copy (string)) (make-string 0)) #t)
(test (string=? "" (make-string 0)) #t)
(test (string=? "" (string-append)) #t)



(test (string) "")
(test (string #\a #\b #\c) "abc")
(test (string #\a) "a")



(test (make-string 0) "")
(test (make-string 3 #\a) "aaa")
(test (make-string 0 #\a) "")
(test (make-string 3 #\space) "   ")
(test (let ((hi (make-string 3 #\newline))) (string-length hi)) 3)



(test (string-ref "abcdef-dg1ndh" 0) #\a)
(test (string-ref "abcdef-dg1ndh" 1) #\b)
(test (string-ref "abcdef-dg1ndh" 6) #\-)
(test (string-ref "\"\\\"" 1) #\\)
(test (string-ref "\"\\\"" 2) #\")



(test (let ((hi (string-copy "hi"))) (string-set! hi 0 #\H) hi) "Hi")
(test (let ((hi (string-copy "hi"))) (string-set! hi 1 #\H) hi) "hH")
(test (let ((hi (string-copy "\"\\\""))) (string-set! hi 0 #\a) hi) "a\\\"")
(test (let ((hi (string-copy "\"\\\""))) (string-set! hi 1 #\a) hi) "\"a\"")
(test (let ((hi (string #\a #\newline #\b))) (string-set! hi 1 #\c) hi) "acb")

(let ((str (make-string 10 #\x)))
  (string-set! str 3 (integer->char 0))
  (test (string=? str "xxx") #t)
  (test (char=? (string-ref str 4) #\x) #t)
  (string-set! str 4 #\a)
  (test (string=? str "xxx") #t)
  (test (char=? (string-ref str 4) #\a) #t)
  (string-set! str 3 #\x)
  (test (string=? str "xxxxaxxxxx") #t))



(test (substring "ab" 0 0) "")
(test (substring "ab" 1 1) "")
(test (substring "ab" 2 2) "")
(test (substring "ab" 0 1) "a")
(test (substring "ab" 1 2) "b")
(test (substring "ab" 0 2) "ab")
(test (substring "hi there" 3 6) "the")
(test (substring "hi there" 0 (string-length "hi there")) "hi there")
(test (substring "" 0 0) "")
(test (let ((str "012345"))
	(let ((str1 (substring str 2 4)))
	  (string-set! str1 1 #\x))
	(string=? str "012345"))
      #t)


(test (string-append "hi" "ho") "hiho")
(test (string-append "hi") "hi")
(test (string-append "hi" "") "hi")
(test (string-append "hi" "" "ho") "hiho")
(test (string-append "" "hi") "hi")
(test (string-append) "")
(test (string-append "a" (string-append (string-append "b" "c") "d") "e") "abcde")
(test (string-append "a" "b" "c" "d" "e") "abcde")
(test (string-append (string-append) (string-append (string-append))) "")
(test (let ((hi "hi")) (let ((ho (string-append hi))) (eq? hi ho))) #f)
(test (let ((hi "hi")) (let ((ho (string-append hi))) (string-set! ho 0 #\a) hi)) "hi")
(test (let ((hi "hi")) (set! hi (string-append hi hi hi hi)) hi) "hihihihi")

(num-test (letrec ((hi (lambda (str n)
			 (if (= n 0)
			     str
			     (hi (string-append str "a") (- n 1))))))
	    (string-length (hi "" 100)))
	  100)

(test (let* ((str "hiho")
	     (str1 "ha")
	     (str2 (string-append str1 str)))
	(string-set! str2 1 #\x)
	(string-set! str2 4 #\x)
	(and (string=? str "hiho")
	     (string=? str1 "ha")
	     (string=? str2 "hxhixo")))
      #t)
(test (let* ((str (string-copy "hiho"))
	     (str1 (string-copy "ha"))
	     (str2 (string-append str1 str)))
	(string-set! str1 1 #\x)
	(string-set! str 2 #\x)
	(and (string=? str "hixo")
	     (string=? str1 "hx")
	     (string=? str2 "hahiho")))
      #t)



(test (let ((hi (string-copy "hi"))) (string-fill! hi #\s) hi) "ss")
(test (let ((hi (string-copy ""))) (string-fill! hi #\x) hi) "")
(test (let ((str (make-string 0))) (string-fill! str #\a) str) "")
(test (let ((hi (make-string 8 (integer->char 0)))) (string-fill! hi #\a) hi) "aaaaaaaa") ; is this result widely accepted?

(if with-vector-for-each
    (begin
      (test (let ((str (make-string 4 #\x))
		  (ctr 0))
	      (string-for-each
	       (lambda (c)
		 (string-set! str ctr c)
		 (set! ctr (+ ctr 1)))
	       "1234")
	      str)
	    "1234")

      (test (let ((str (make-string 8 #\x))
		  (ctr 0))
	      (string-for-each
	       (lambda (c1 c2)
		 (string-set! str ctr c1)
		 (string-set! str (+ ctr 1) c2)
		 (set! ctr (+ ctr 2)))
	       "1234"
	       "hiho")
	      str)
	    "1h2i3h4o")
      ))

(test (string-copy "ab") "ab")
(test (string-copy "") "")
(test (string-copy "\"\\\"") "\"\\\"")
(test (let ((hi "abc")) (eq? hi (string-copy hi))) #f)
(test (let ((hi (string-copy (make-string 8 (integer->char 0))))) (string-fill! hi #\a) hi) "aaaaaaaa") ; is this result widely accepted?
(test (string-copy (string-copy (string-copy "a"))) "a")
(test (string-copy (string-copy (string-copy ""))) "")



(test (string->list "abc") (list #\a #\b #\c))
(test (string->list "") '())
;(test (string->list (make-string 3 (integer->char 0))) '()) ; others return a list of #\nul or #\null
(test (string->list (list->string (list #\a #\b #\c))) (list #\a #\b #\c))
(test (string->list (list->string '())) '())
(test (list->string (string->list "abc")) "abc")
(test (list->string (string->list "hi there")) "hi there")
(test (list->string (string->list "&*#%^@%$)~@")) "&*#%^@%$)~@")
(test (list->string (string->list "")) "")
(test (let* ((str "abc")
	     (lst (string->list str)))
	(and (string=? str "abc")
	     (equal? lst (list #\a #\b #\c))))
      #t)
(test (list->string '()) "")

(test (list->string (list #\a #\b #\c)) "abc")
(test (list->string (list)) "")

(test (list->string (list #\" #\# #\")) "\"#\"")
(test (list->string (list #\\ #\\ #\# #\\ #\# #\#)) "\\\\#\\##")
(test (list->string (list #\' #\' #\` #\")) '"''`\"")
(test (string #\" #\# #\") "\"#\"")
(test (string #\\ #\\ #\# #\\ #\# #\#) "\\\\#\\##")
(test (string #\' #\' #\` #\") '"''`\"")
;;; some schemes accept \' and other such sequences in a string, but the spec only mentions \\ and \"


(test (symbol->string 'hi) "hi")
(test (symbol->string (string->symbol "()")) "()")
(test (string->symbol (symbol->string 'hi)) 'hi)
(test (eq? (string->symbol "hi") 'hi) #t)
(test (eq? (string->symbol "hi") (string->symbol "hi")) #t)

(test (string->symbol "hi") 'hi)

(test (let ((str (symbol->string 'hi)))
	(catch #t (lambda () (string-set! str 1 #\x)) (lambda args 'error)) ; can be disallowed
	(symbol->string 'hi))
      "hi")

(test (symbol->string 'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789)
      "sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789")
(test (string->symbol "sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789")
      'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789)
(test (let ((sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 32))
	(+ sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 1))
      33)

(test (symbol->string (string->symbol "hi there")) "hi there")
(test (symbol->string (string->symbol "Hi There")) "Hi There")
(test (symbol->string (string->symbol "HI THERE")) "HI THERE")
(test (symbol->string (string->symbol "")) "")
(test (symbol? (string->symbol "(weird name for a symbol!)")) #t)      



;;; --------------------------------------------------------------------------------
;;; LISTS
;;; --------------------------------------------------------------------------------

(test (cons 'a '()) '(a))
(test (cons '(a) '(b c d)) '((a) b c d))
(test (cons "a" '(b c)) '("a" b c))
(test (cons 'a 3) '(a . 3))
(test (cons '(a b) 'c) '((a b) . c))
(test (cons '() '()) '(()))
(test (cons '() 1) '(() . 1))
(test (cons 1 2) '(1 . 2))
(test (cons 1 '()) '(1))
(test (cons '() 2) '(() . 2))
(test (cons 1 (cons 2 (cons 3 (cons 4 '())))) '(1 2 3 4))
(test (cons 'a 'b) '(a . b))
(test (cons 'a (cons 'b (cons 'c '()))) '(a b c))
(test (cons 'a (list 'b 'c 'd)) '(a b c d))
(test (cons 'a (cons 'b (cons 'c 'd))) '(a b c . d))
(test '(a b c d e) '(a . (b . (c . (d . (e . ()))))))
(test (cons (cons 1 2) (cons 3 4)) '((1 . 2) 3 . 4))
(test (list (cons 1 2) (cons 3 4)) '((1 . 2) (3 . 4)))
(test (cons (cons 1 (cons 2 3)) 4) '((1 . (2 . 3)) . 4))
(test (cons (cons 1 (cons 2 '())) (cons 1 2)) '((1 2) . (1 . 2)))
(test (let ((lst (list 1 2))) (list (apply cons lst) lst)) '((1 . 2) (1 2)))
(test (let ((lst (list 1 2))) (list lst (apply cons lst))) '((1 2) (1 . 2)))
(test (cdadr (let ((lst (list 1 2))) (list (apply cons lst) lst))) '(2))

(test (car (list 1 2 3)) 1)
(test (car (cons 1 2)) 1)
(test (car (list 1)) 1)
(test (car '(1 2 3)) 1)
(test (car '(1)) 1)
(test (car '(1 . 2)) 1)
(test (car '((1 2) 3)) '(1 2))
(test (car '(((1 . 2) . 3) 4)) '((1 . 2) . 3))
(test (car (list (list) (list 1 2))) '())
(test (car '(a b c)) 'a)
(test (car '((a) b c d)) '(a))
(test (car (reverse (list 1 2 3 4))) 4)
(test (car (list 'a 'b 'c 'd 'e 'f 'g)) 'a)
(test (car '(a b c d e f g)) 'a)
(test (car '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((((1 2 3) 4) 5) (6 7)))
(test (car '(a)) 'a)
(test (car '(1 ^ 2)) 1)
(test (car '(1 .. 2)) 1)
(test (car ''foo) 'quote)
(test (car '(1 2 . 3)) 1)

(for-each
 (lambda (arg)
   (if (not (equal? (car (cons arg '())) arg))
       (begin
	 (display "(car '(") (display arg) (display ")) returned ") (display (car (cons arg '()))) (display "?") (newline))))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (lambda (a) (+ a 1))))


(test (cdr (list 1 2 3)) '(2 3))
(test (cdr (cons 1 2)) 2)
(test (cdr (list 1)) '())
(test (cdr '(1 2 3)) '(2 3))
(test (cdr '(1)) '())
(test (cdr '(1 . 2)) 2)
(test (cdr '((1 2) 3)) '(3))
(test (cdr '(((1 . 2) . 3) 4)) '(4))
(test (cdr (list (list) (list 1 2))) '((1 2)))
(test (cdr '(a b c)) '(b c))
(test (cdr '((a) b c d)) '(b c d))
(test (equal? (cdr (reverse (list 1 2 3 4))) 4) #f)
(test (equal? (cdr (list 'a 'b 'c 'd 'e 'f 'g)) 'a) #f)
(test (cdr '((((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f) g)) '(g))
(test (cdr '(a)) '())
(test (cdr '(a b c d e f g)) '(b c d e f g))
(test (cdr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((((u v w) x) y) ((q w e) r) (a b c) e f g))
(test (cdr ''foo) '(foo))
(test (cdr (cons (cons 1 2) (cons 3 4))) '(3 . 4))
(test (cdr '(1 2 . 3)) '(2 . 3))

(for-each
 (lambda (arg)
   (if (not (equal? (cdr (cons '() arg)) arg))
       (begin
	 (display "(cdr '(() ") (display arg) (display ")) returned ") (display (cdr (cons '() arg))) (display "?") (newline))))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (lambda (a) (+ a 1))))

(define (cons-r a b n) (if (= 0 n) (cons a b) (cons (cons-r (+ a 1) (+ b 1) (- n 1)) (cons-r (- a 1) (- b 1) (- n 1)))))
(define (list-r a b n) (if (= 0 n) (list a b) (list (list-r (+ a 1) (+ b 1) (- n 1)) (list-r (- a 1) (- b 1) (- n 1)))))

(define lists (list (list 1 2 3)
		    (cons 1 2)
		    (list 1)
		    (list)
		    (list (list 1 2) (list 3 4))
		    (list (list 1 2) 3)
		    '(1 . 2)
		    '(a b c)
		    '((a) b (c))
		    '((1 2) (3 4))
		    '((1 2 3) (4 5 6) (7 8 9))
		    '(((1) (2) (3)) ((4) (5) (6)) ((7) (8) (9)))
		    '((((1 123) (2 124) (3 125) (4 126)) ((5) (6) (7) (8)) ((9) (10) (11) (12)) ((13) (14) (15) (16)))
		      (((21 127) (22 128) (23 129) (24 130)) ((25) (26) (27) (28)) ((29) (30) (31) (32)) ((33) (34) (35) (36)))
		      (((41 131) (42 132) (43 133) (44 134)) ((45) (46) (47) (48)) ((49) (50) (51) (52)) ((53) (54) (55) (56)))
		      (((61 135) (62 136) (63 137) (64 138)) ((65) (66) (67) (68)) ((69) (70) (71) (72)) ((73) (74) (75) (76)))
		      321)
		    (cons 1 (cons 2 (cons 3 4)))
		    (cons (cons 2 (cons 3 4)) 5)
		    (cons '() 1)
		    (cons 1 '())
		    (cons '() '())
		    (list 1 2 (cons 3 4) 5 (list (list 6) 7))
		    (cons-r 0 0 4)
		    (cons-r 0 0 5)
		    (cons-r 0 0 10)
		    (list-r 0 0 3)
		    (list-r 0 0 7)
		    (list-r 0 0 11)
		    ''a
		    ))

(define (caar-1 x) (car (car x)))
(define (cadr-1 x) (car (cdr x)))
(define (cdar-1 x) (cdr (car x)))
(define (cddr-1 x) (cdr (cdr x)))
(define (caaar-1 x) (car (car (car x))))
(define (caadr-1 x) (car (car (cdr x))))
(define (cadar-1 x) (car (cdr (car x))))
(define (caddr-1 x) (car (cdr (cdr x))))
(define (cdaar-1 x) (cdr (car (car x))))
(define (cdadr-1 x) (cdr (car (cdr x))))
(define (cddar-1 x) (cdr (cdr (car x))))
(define (cdddr-1 x) (cdr (cdr (cdr x))))
(define (caaaar-1 x) (car (car (car (car x)))))
(define (caaadr-1 x) (car (car (car (cdr x)))))
(define (caadar-1 x) (car (car (cdr (car x)))))
(define (caaddr-1 x) (car (car (cdr (cdr x)))))
(define (cadaar-1 x) (car (cdr (car (car x)))))
(define (cadadr-1 x) (car (cdr (car (cdr x)))))
(define (caddar-1 x) (car (cdr (cdr (car x)))))
(define (cadddr-1 x) (car (cdr (cdr (cdr x)))))
(define (cdaaar-1 x) (cdr (car (car (car x)))))
(define (cdaadr-1 x) (cdr (car (car (cdr x)))))
(define (cdadar-1 x) (cdr (car (cdr (car x)))))
(define (cdaddr-1 x) (cdr (car (cdr (cdr x)))))
(define (cddaar-1 x) (cdr (cdr (car (car x)))))
(define (cddadr-1 x) (cdr (cdr (car (cdr x)))))
(define (cdddar-1 x) (cdr (cdr (cdr (car x)))))
(define (cddddr-1 x) (cdr (cdr (cdr (cdr x)))))

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (begin
	      (display "(") (display name) (display " ") (display lst) (display ")) returned ") (display val1) (display " ") (display val2) (newline)))))
    lists))
 (list 'caar 'cadr 'cdar 'cddr 'caaar 'caadr 'cadar 'cdaar 'caddr 'cdddr 'cdadr 'cddar 
       'caaaar 'caaadr 'caadar 'cadaar 'caaddr 'cadddr 'cadadr 'caddar 'cdaaar 
       'cdaadr 'cdadar 'cddaar 'cdaddr 'cddddr 'cddadr 'cdddar)

 (list caar cadr cdar cddr caaar caadr cadar cdaar caddr cdddr cdadr cddar 
       caaaar caaadr caadar cadaar caaddr cadddr cadadr caddar cdaaar 
       cdaadr cdadar cddaar cdaddr cddddr cddadr cdddar)

 (list caar-1 cadr-1 cdar-1 cddr-1 caaar-1 caadr-1 cadar-1 cdaar-1 caddr-1 cdddr-1 cdadr-1 cddar-1 
       caaaar-1 caaadr-1 caadar-1 cadaar-1 caaddr-1 cadddr-1 cadadr-1 caddar-1 cdaaar-1 
       cdaadr-1 cdadar-1 cddaar-1 cdaddr-1 cddddr-1 cddadr-1 cdddar-1))



(test (equal? (cadr (list 'a 'b 'c 'd 'e 'f 'g)) 'b) #t)
(test (equal? (cddr (list 'a 'b 'c 'd 'e 'f 'g)) '(c d e f g)) #t)
(test (equal? (caddr (list 'a 'b 'c 'd 'e 'f 'g)) 'c) #t)
(test (equal? (cdddr (list 'a 'b 'c 'd 'e 'f 'g)) '(d e f g)) #t)
(test (equal? (cadddr (list 'a 'b 'c 'd 'e 'f 'g)) 'd) #t)
(test (equal? (cddddr (list 'a 'b 'c 'd 'e 'f 'g)) '(e f g)) #t)
(test (equal? (caadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '((u v w) x)) #t)
(test (equal? (cadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(6 7)) #t)
(test (equal? (cdaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(5)) #t)
(test (equal? (cdadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(y)) #t)
(test (equal? (cddar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '()) #t)
(test (equal? (caaaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(1 2 3)) #t)
(test (equal? (caadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 6) #t)
(test (equal? (caaddr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(q w e)) #t)
(test (equal? (cadaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 5) #t)
(test (equal? (cadadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 'y) #t)
(test (equal? (caddar (list (list (list (list (list 1 2 3) 4) 5) 1 6 (list 5 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 6) #t)
(test (equal? (cadddr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(a b c)) #t)
(test (equal? (cdaaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(4)) #t)
(test (equal? (cdaadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(x)) #t)
(test (equal? (cdadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(7)) #t)

(test (caar '((a) b c d e f g)) 'a)
(test (cadr '(a b c d e f g)) 'b)
(test (cdar '((a b) c d e f g)) '(b))
(test (cddr '(a b c d e f g)) '(c d e f g))
(test (caaar '(((a)) b c d e f g)) 'a)
(test (caadr '(a (b) c d e f g)) 'b)
(test (cadar '((a b) c d e f g)) 'b)
(test (caddr '(a b c d e f g)) 'c)
(test (cdaar '(((a b)) c d e f g)) '(b))
(test (cdadr '(a (b c) d e f g)) '(c))
(test (cddar '((a b c) d e f g)) '(c))
(test (cdddr '(a b c d e f g)) '(d e f g))
(test (caaaar '((((a))) b c d e f g)) 'a)
(test (caaadr '(a ((b)) c d e f g)) 'b)
(test (caadar '((a (b)) c d e f g)) 'b)
(test (caaddr '(a b (c) d e f g)) 'c)
(test (cadaar '(((a b)) c d e f g)) 'b)
(test (cadadr '(a (b c) d e f g)) 'c)
(test (caddar '((a b c) d e f g)) 'c)
(test (cadddr '(a b c d e f g)) 'd)
(test (cdaaar '((((a b))) c d e f g)) '(b))
(test (cdaadr '(a ((b c)) d e f g)) '(c))
(test (cdadar '((a (b c)) d e f g)) '(c))
(test (cdaddr '(a b (c d) e f g)) '(d))
(test (cddaar '(((a b c)) d e f g)) '(c))
(test (cddadr '(a (b c d) e f g)) '(d))
(test (cdddar '((a b c d) e f g)) '(d))
(test (cddddr '(a b c d e f g)) '(e f g))
(test (cadr '(1 2 . 3)) 2)
(test (cddr '(1 2 . 3)) 3)

;; sacla
(test (caar '((a) b c)) 'a)
(test (cadr '(a b c)) 'b)
(test (cdar '((a . aa) b c)) 'aa)
(test (cddr '(a b . c)) 'c)
(test (caaar '(((a)) b c)) 'a)
(test (caadr '(a (b) c)) 'b)
(test (cadar '((a aa) b c)) 'aa)
(test (caddr '(a b c)) 'c)
(test (cdaar '(((a . aa)) b c)) 'aa)
(test (cdadr '(a (b . bb) c)) 'bb)
(test (cddar '((a aa . aaa) b c)) 'aaa)
(test (cdddr '(a b c . d)) 'd)
(test (caaaar '((((a))) b c)) 'a)
(test (caaadr '(a ((b)) c)) 'b)
(test (caadar '((a (aa)) b c)) 'aa)
(test (caaddr '(a b (c))) 'c)
(test (cadaar '(((a aa)) b c)) 'aa)
(test (cadadr '(a (b bb) c)) 'bb)
(test (caddar '((a aa aaa) b c)) 'aaa)
(test (cadddr '(a b c d)) 'd)
(test (cdaaar '((((a . aa))) b c)) 'aa)
(test (cdaadr '(a ((b . bb)) c)) 'bb)
(test (cdadar '((a (aa . aaa)) b c)) 'aaa)
(test (cdaddr '(a b (c . cc))) 'cc)
(test (cddaar '(((a aa . aaa)) b c)) 'aaa)
(test (cddadr '(a (b bb . bbb) c)) 'bbb)
(test (cdddar '((a aa aaa . aaaa) b c)) 'aaaa)
(test (cddddr '(a b c d . e)) 'e)

(test (caar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((1 2 3) 4) 5))
(test (cadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((u v w) x) y))
(test (cdar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((6 7)))
(test (cddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((q w e) r) (a b c) e f g))
(test (caaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((1 2 3) 4))
(test (caadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((u v w) x))
(test (cadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(6 7))
(test (caddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((q w e) r))
(test (cdaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(5))
(test (cdadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(y))
(test (cddar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '())
(test (cdddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((a b c) e f g))
(test (caaaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(1 2 3))
(test (caaadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(u v w))
(test (caadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 6)
(test (caaddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(q w e))
(test (cadaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 5)
(test (cadadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 'y)
(test (cadddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(a b c))
(test (cdaaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(4))
(test (cdaadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(x))
(test (cdadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(7))
(test (cdaddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(r))
(test (cddaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '())
(test (cddadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '())
(test (cddddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(e f g))

(test (cadr '(a b c d e f g)) 'b)
(test (cddr '(a b c d e f g)) '(c d e f g))
(test (caddr '(a b c d e f g)) 'c)
(test (cdddr '(a b c d e f g)) '(d e f g))
(test (cadddr '(a b c d e f g)) 'd)
(test (cddddr '(a b c d e f g)) '(e f g))

(test (caar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((a . b) c . d))
(test (caar '(((a . b) c . d) (e . f) g . h)) '(a . b))
(test (caar '((a . b) c . d)) 'a)
(test (cadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((i . j) k . l))
(test (cadr '(((a . b) c . d) (e . f) g . h)) '(e . f))
(test (cadr '((a . b) c . d)) 'c)
(test (cdar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((e . f) g . h))
(test (cdar '(((a . b) c . d) (e . f) g . h)) '(c . d))
(test (cdar '((a . b) c . d)) 'b)
(test (cddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((m . n) o . p))
(test (cddr '(((a . b) c . d) (e . f) g . h)) '(g . h))
(test (cddr '((a . b) c . d)) 'd)
(test (caaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(a . b))
(test (caaar '(((a . b) c . d) (e . f) g . h)) 'a)
(test (caadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(i . j))
(test (caadr '(((a . b) c . d) (e . f) g . h)) 'e)
(test (cddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(g . h))
(test (cddar '(((a . b) c . d) (e . f) g . h)) 'd)
(test (cdddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(o . p))
(test (cdddr '(((a . b) c . d) (e . f) g . h)) 'h)
(test (caaaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'a)
(test (caaadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'i)
(test (caddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'g)
(test (cadddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'o)
(test (cdaaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'b)
(test (cdaadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'j)
(test (cdddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'h)
(test (cddddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'p)

(test (cadr ''foo) 'foo)

(test (caar '((a) b c)) 'a)
(test (cadr '(a b c)) 'b)
(test (cdar '((a . aa) b c)) 'aa)
(test (cddr '(a b . c)) 'c)
(test (caaar '(((a)) b c)) 'a)
(test (caadr '(a (b) c)) 'b)
(test (cadar '((a aa) b c)) 'aa)
(test (caddr '(a b c)) 'c)
(test (cdaar '(((a . aa)) b c)) 'aa)
(test (cdadr '(a (b . bb) c)) 'bb)
(test (cddar '((a aa . aaa) b c)) 'aaa)
(test (cdddr '(a b c . d)) 'd)
(test (caaaar '((((a))) b c)) 'a)
(test (caaadr '(a ((b)) c)) 'b)
(test (caadar '((a (aa)) b c)) 'aa)
(test (caaddr '(a b (c))) 'c)
(test (cadaar '(((a aa)) b c)) 'aa)
(test (cadadr '(a (b bb) c)) 'bb)
(test (caddar '((a aa aaa) b c)) 'aaa)
(test (cadddr '(a b c d)) 'd)
(test (cdaaar '((((a . aa))) b c)) 'aa)
(test (cdaadr '(a ((b . bb)) c)) 'bb)
(test (cdadar '((a (aa . aaa)) b c)) 'aaa)
(test (cdaddr '(a b (c . cc))) 'cc)
(test (cddaar '(((a aa . aaa)) b c)) 'aaa)
(test (cddadr '(a (b bb . bbb) c)) 'bbb)
(test (cdddar '((a aa aaa . aaaa) b c)) 'aaaa)
(test (cddddr '(a b c d . e)) 'e)




(test (length (list 'a 'b 'c 'd 'e 'f)) 6)
(test (length (list 'a 'b 'c 'd)) 4)
(test (length (list 'a (list 'b 'c) 'd)) 3)
(test (length '()) 0)
(test (length '(this-that)) 1)
(test (length '(this - that)) 3)
(test (length '(a b)) 2)
(test (length '(a b c)) 3)
(test (length '(a (b) (c d e))) 3)
(test (length (list 1 (cons 1 2))) 2)
(test (length (list 1 (cons 1 '()))) 2)


(test (reverse '(a b c d)) '(d c b a))
(test (reverse '(a b c))  '(c b a))
(test (reverse '(a (b c) d (e (f))))  '((e (f)) d (b c) a))
(test (reverse '()) '())
(test (reverse (list 1 2 3)) '(3 2 1))
(test (reverse (list 1)) '(1))
(test (reverse (list)) (list))
(test (reverse '(1 2 3)) (list 3 2 1))
(test (reverse '(1)) '(1))
(test (reverse '((1 2) 3)) '(3 (1 2)))
(test (reverse '(((1 . 2) . 3) 4)) '(4 ((1 . 2) . 3)))
(test (reverse (list (list) (list 1 2))) '((1 2) ()))
(test (reverse '((a) b c d)) '(d c b (a)))
(test (reverse (reverse (list 1 2 3 4))) (list 1 2 3 4))
(test (reverse ''foo) '(foo quote))
(test (let ((x (list 1 2 3 4)))
	(let ((y (reverse x)))
	  (and (equal? x (list 1 2 3 4))
	       (equal? y (list 4 3 2 1)))))
      #t)
(test (letrec ((hi (lambda (lst n)
	       (if (= n 0)
		   lst
		   (hi (reverse lst) (- n 1))))))
	(hi (list 1 2 3) 100))
      (list 1 2 3))
(test (let ((var (list 1 2 3))) (reverse (cdr var)) var) (list 1 2 3))
(test (let ((var '(1 2 3))) (reverse (cdr var)) var) '(1 2 3))
(test (let ((var (list 1 (list 2 3)))) (reverse (cdr var)) var) (list 1 (list 2 3)))
(test (let ((var '(1 (2 3)))) (reverse (cdr var)) var) '(1 (2 3)))
(test (let ((var (list (list 1 2) (list 3 4 5)))) (reverse (car var)) var) '((1 2) (3 4 5)))
(test (let ((x '(1 2 3))) (list (reverse x) x)) '((3 2 1) (1 2 3)))

(for-each
 (lambda (lst)
   (if (list? lst)
       (if (not (equal? lst (reverse (reverse lst))))
	   (begin
	     (display "(reverse (reverse ") (display lst) (display ")) returned ") (display (reverse (reverse lst))) (newline)))))
 lists)

(for-each
 (lambda (lst)
   (if (list? lst)
       (if (not (equal? lst (reverse (reverse (reverse (reverse lst))))))
	   (begin
	     (display "(reverse...(4x) ") (display lst) (display ")) returned ") (display (reverse (reverse (reverse (reverse lst))))) (newline)))))
 lists)



(test (pair? 'a) #f)
(test (pair? '()) #f)
(test (pair? '(a b c)) #t)
(test (pair? (cons 1 2)) #t)
(test (pair? ''()) #t)
(test (pair? #f) #f)
(test (pair? (make-vector 6)) #f)
(test (pair? #t) #f)
(test (pair? '(a . b)) #t)
(test (pair? '#(a b))  #f)
(test (pair? (list 1 2)) #t)
(test (pair? (list)) #f)
(test (pair? ''foo) #t)
(test (pair? (list 'a 'b 'c 'd 'e 'f)) #t)
(test (pair? '(this-that)) #t)
(test (pair? '(this - that)) #t)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (pair? x) #t))
(test (pair? (list 1 (cons 1 2))) #t)
(test (pair? (list 1 (cons 1 '()))) #t)
(test (pair? (cons 1 '())) #t)
(test (pair? (cons '() '())) #t)
(test (pair? (cons '() 1)) #t)
(test (pair? (list (list))) #t)
(test (pair? '(())) #t)
(test (pair? (cons 1 (cons 2 3))) #t)

(for-each
 (lambda (arg)
   (if (pair? arg)
       (begin
	 (display "(pair? ") (display arg) (display ") returned #t?") (newline))))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (lambda (a) (+ a 1))))



(test (list? 'a) #f)
(test (list? '()) #t)
(test (list? '(a b c)) #t)
(test (list? (cons 1 2)) #f)
(test (list? ''()) #t)
(test (list? #f) #f)
(test (list? (make-vector 6)) #f)
(test (list? #t) #f)
(test (list? '(a . b)) #f)
(test (list? '#(a b))  #f)
(test (list? (list 1 2)) #t)
(test (list? (list)) #t)
(test (list? ''foo) #t)
(test (list? (list 'a 'b 'c 'd 'e 'f)) #t)
(test (list? '(this-that)) #t)
(test (list? '(this - that)) #t)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (list? x) #f))
(test (list? (list 1 (cons 1 2))) #t)
(test (list? (list 1 (cons 1 '()))) #t)
(test (list? (cons 1 '())) #t)
(test (list? (cons '() '())) #t)
(test (list? (cons '() 1)) #f)
(test (list? (list (list))) #t)
(test (list? '(())) #t)
(test (list? '(1 2 . 3)) #f)
(test (list? (cons 1 (cons 2 3))) #f)
(test (list? '(1 . ())) #t)

(for-each
 (lambda (arg)
   (if (list? arg)
       (begin
	 (display "(list? ") (display arg) (display ") returned #t?") (newline))))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (lambda (a) (+ a 1))))



(test (null? 'a) '#f)
(test (null? '()) #t)
(test (null? '(a b c)) #f)
(test (null? (cons 1 2)) #f)
(test (null? ''()) #f)
(test (null? #f) #f)
(test (null? (make-vector 6)) #f)
(test (null? #t) #f)
(test (null? '(a . b)) #f)
(test (null? '#(a b))  #f)
(test (null? (list 1 2)) #f)
(test (null? (list)) #t)
(test (null? ''foo) #f)
(test (null? (list 'a 'b 'c 'd 'e 'f)) #f)
(test (null? '(this-that)) #f)
(test (null? '(this - that)) #f)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (null? x) #f))
(test (null? (list 1 (cons 1 2))) #f)
(test (null? (list 1 (cons 1 '()))) #f)
(test (null? (cons 1 '())) #f)
(test (null? (cons '() '())) #f)
(test (null? (cons '() 1)) #f)
(test (null? (list (list))) #f)
(test (null? '(())) #f)
(test (null? '#()) #f)

(for-each
 (lambda (arg)
   (if (null? arg)
       (begin
	 (display "(null? ") (display arg) (display ") returned #t?") (newline))))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (lambda (a) (+ a 1))))


(if with-reverse! (begin
(test (reverse! '(a b c d)) '(d c b a))
(test (reverse! '(a b c))  '(c b a))
(test (reverse! '(a (b c) d (e (f))))  '((e (f)) d (b c) a))
(test (reverse! '()) '())
(test (reverse! (list 1 2 3)) '(3 2 1))
(test (reverse! (list 1)) '(1))
(test (reverse! (list)) (list))
(test (reverse! '(1 2 3)) (list 3 2 1))
(test (reverse! '(1)) '(1))
(test (reverse! '((1 2) 3)) '(3 (1 2)))
(test (reverse! '(((1 . 2) . 3) 4)) '(4 ((1 . 2) . 3)))
(test (reverse! (list (list) (list 1 2))) '((1 2) ()))
(test (reverse! '((a) b c d)) '(d c b (a)))
(test (reverse! (reverse! (list 1 2 3 4))) (list 1 2 3 4))
(test (reverse! ''foo) '(foo quote))
(test (reverse (reverse! (list 1 2 3))) (list 1 2 3))
(test (reverse (reverse! (reverse! (reverse (list 1 2 3))))) (list 1 2 3))
))


(test (let ((x (cons 1 2))) (set-car! x 3) x) (cons 3 2))
(test (let ((x (list 1 2))) (set-car! x 3) x) (list 3 2))
(test (let ((x (list (list 1 2) 3))) (set-car! x 22) x) (list 22 3))
(test (let ((x (cons 1 2))) (set-car! x '()) x) (cons '() 2))
(test (let ((x (list 1 (list 2 3 4)))) (set-car! x (list 5 (list 6))) x) (list (list 5 (list 6)) (list 2 3 4)))
(test (let ((x '(((1) 2) (3)))) (set-car! x '((2) 1)) x) '(((2) 1) (3)))
(test (let ((x ''foo)) (set-car! x "hi") x) (list "hi" 'foo))
(test (let ((x '((1 . 2) . 3))) (set-car! x 4) x) '(4 . 3))
(test (let ((x '(1 . 2))) (set-car! x (cdr x)) x) '(2 . 2))
(test (let ((x '(1 . 2))) (set-car! x x) (list? x)) #f)
(test (let ((x (list 1))) (set-car! x '()) x) '(()))
(test (let ((x '(((1 2) . 3) 4))) (set-car! x 1) x) '(1 4))
(test (let ((lst (cons 1 (cons 2 3)))) (set-car! (cdr lst) 4) lst) (cons 1 (cons 4 3)))
(test (let ((lst (cons 1 (cons 2 3)))) (set-car! lst 4) lst) (cons 4 (cons 2 3)))

;(set-car! '(1 . 2) 3)  ??


(test (let ((x (cons 1 2))) (set-cdr! x 3) x) (cons 1 3))
(test (let ((x (list 1 2))) (set-cdr! x 3) x) (cons 1 3))
(test (let ((x (list (list 1 2) 3))) (set-cdr! x 22) x) '((1 2) . 22))
(test (let ((x (cons 1 2))) (set-cdr! x '()) x) (list 1))
(test (let ((x (list 1 (list 2 3 4)))) (set-cdr! x (list 5 (list 6))) x) '(1 5 (6)))
(test (let ((x '(((1) 2) (3)))) (set-cdr! x '((2) 1)) x) '(((1) 2) (2) 1))
(test (let ((x ''foo)) (set-cdr! x "hi") x) (cons 'quote "hi"))
(test (let ((x '((1 . 2) . 3))) (set-cdr! x 4) x) '((1 . 2) . 4))
(test (let ((x '(1 . 2))) (set-cdr! x (cdr x)) x) '(1 . 2))
(test (let ((x '(1 . 2))) (set-cdr! x x) (list? x)) #f)
(test (let ((x (list 1))) (set-cdr! x '()) x) (list 1))
(test (let ((x '(1 . (2 . (3 (4 5)))))) (set-cdr! x 4) x) '(1 . 4))
(test (let ((lst (cons 1 (cons 2 3)))) (set-cdr! (cdr lst) 4) lst) (cons 1 (cons 2 4)))


(test (list-ref (list 1 2) 1) 2)
(test (list-ref '(a b c d) 2) 'c)
(test (list-ref (cons 1 2) 0) 1) ; !!
(test (list-ref ''foo 0) 'quote)
(test (list-ref '((1 2) (3 4)) 1) '(3 4))
(test (list-ref (list-ref (list (list 1 2) (list 3 4)) 1) 1) 4)
(test (let ((x (list 1 2 3))) (list-ref x (list-ref x 1))) 3)
; (list-ref '(1 2 . 3) 0) -- why is this acceptable?

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (begin
	      (display "(") (display name) (display " ") (display lst) (display ")) returned ") (display val1) (display " ") (display val2) (newline)))))
    lists))
 (list 'list-ref:0 'list-ref:1 'list-ref:2 'list-ref:3)
 (list car cadr caddr cadddr)
 (list (lambda (l) (list-ref l 0)) (lambda (l) (list-ref l 1)) (lambda (l) (list-ref l 2)) (lambda (l) (list-ref l 3))))

(for-each
 (lambda (arg)
   (test (list-ref (list 1 arg) 1) arg))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (lambda (a) (+ a 1))))

(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 0)) 1)
(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 1)) 1)
(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 100)) 1)



(if with-list-set! (begin
(test (let ((x (list 1))) (list-set! x 0 2) x) (list 2))
(test (let ((x (cons 1 2))) (list-set! x 0 3) x) '(3 . 2))
(test (let ((x '((1) 2))) (list-set! x 0 1) x) '(1 2))
(test (let ((x '(1 2))) (list-set! x 1 (list 3 4)) x) '(1 (3 4)))
(test (let ((x ''foo)) (list-set! x 0 "hi") x ) '("hi" foo))
(test (let ((x (list 1 2))) (list-set! x 0 x) (list? x)) #t)
(test (let ((x (list 1 2))) (list-set! x 1 x) (list? x)) #t)

(for-each
 (lambda (arg)
   (test (let ((x (list 1 2))) (list-set! x 0 arg) (list-ref x 0)) arg))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (lambda (a) (+ a 1))))
))



(test (let ((tree1 (list 1 (list 1 2) (list (list 1 2 3)) (list (list (list 1 2 3 4)))))) tree1) '(1 (1 2) ((1 2 3)) (((1 2 3 4)))))
(test (let ((tree2 (list "one" (list "one" "two") (list (list "one" "two" "three"))))) tree2) '("one" ("one" "two") (("one" "two" "three"))))
(test (let ((tree1 (list 1 (list 1 2) (list 1 2 3) (list 1 2 3 4)))) tree1) '(1 (1 2) (1 2 3) (1 2 3 4)))
(test (let ((tree1 (list 1 (list 1 2))) (tree2 (list 1 (list 1 2)))) tree2) '(1 (1 2)))
(test (let ((tree1 (list 1 (list 1 2))) (tree2 (list 1 (list 1 2)))) (eqv? tree1 tree2)) #f)
(test (let ((tree1 (list ''a (list ''b ''c))) (tree2 (list ''a (list ''b ''c)))) tree2) '('a ('b 'c)))
(test (let ((lst (list 1 (list 2 3)))) lst) '(1 (2 3)))
(test (let* ((lst (list 1 (list 2 3))) (slst lst)) slst) '(1 (2 3)))
(test (list 1) '(1))
(test (let ((a 1)) (list a 2)) '(1 2))
(test (let ((a 1)) (list 'a '2)) '(a 2))
(test (let ((a 1)) (list 'a 2)) '(a 2))
(test (list) '())
(test (let ((a (list 1 2))) a) '(1 2))
(test (let ((a (list 1 2))) (list 3 4 'a (car (cons 'b 'c)) (+ 6 -2))) '(3 4 a b 4))
(test (list) '())
(test (length (list quote do map call/cc lambda define if begin set! let let* cond and or for-each)) 15)


(test (list-tail '(1 2 3) 0) '(1 2 3))
(test (list-tail '(1 2 3) 2) '(3))
(test (list-tail '(1 2 3) 3) '())
(test (list-tail '(1 2 3 . 4) 2) '(3 . 4))
(test (list-tail '(1 2 3 . 4) 3) 4)
(test (let ((x (list 1 2 3))) (eq? (list-tail x 2) (cddr x))) #t)
(test (list-tail '() 0) '())
(test (list-tail (list 1 2) 2) '())
(test (list-tail (cons 1 2) 0) '(1 . 2))
(test (list-tail (cons 1 2) 1) 2)
(test (list-tail ''foo 1) '(foo))
(test (list-tail '((1 2) (3 4)) 1) '((3 4)))
(test (list-tail (list-tail (list-tail '(1 2 3 4) 1) 1) 1) '(4))

(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 0) x))
(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 1) (cdr x)))
(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 100) x))

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (begin
	      (display "(") (display name) (display " ") (display lst) (display ")) returned ") (display val1) (display " ") (display val2) (newline)))))
    lists))
 (list 'list-tail:0 'list-tail:1 'list-tail:2 'list-tail:3 'list-tail:4)
 (list (lambda (l) l) cdr cddr cdddr cddddr)
 (list (lambda (l) (list-tail l 0)) (lambda (l) (list-tail l 1)) (lambda (l) (list-tail l 2)) (lambda (l) (list-tail l 3)) (lambda (l) (list-tail l 4))))



(let ((e '((a 1) (b 2) (c 3))))
  (test (assq 'a e) '(a 1))
  (test (assq 'b e) '(b 2))
  (test (assq 'd e) #f))
(test (assq (list 'a) '(((a)) ((b)) ((c))))  #f)

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assq #t e) (list #t 1))
    (test (assq #f e) (list #f 2))
    (test (assq 'a e) (list 'a 3))
    (test (assq xcons e) (list xcons 4))
    (test (assq xvect e) (list xvect 5))
    (test (assq xlambda e) (list xlambda 6))
    (test (assq xstr e) (list xstr 7))
    (test (assq car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3))))
  (test (assq 1+i e) #f)
  (test (assq 3.0 e) #f)
  (test (assq 5/3 e) #f))



(let ((e '((a 1) (b 2) (c 3))))
  (test (assv 'a e) '(a 1))
  (test (assv 'b e) '(b 2))
  (test (assv 'd e) #f))
(test (assv (list 'a) '(((a)) ((b)) ((c))))  #f)

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assv #t e) (list #t 1))
    (test (assv #f e) (list #f 2))
    (test (assv 'a e) (list 'a 3))
    (test (assv xcons e) (list xcons 4))
    (test (assv xvect e) (list xvect 5))
    (test (assv xlambda e) (list xlambda 6))
    (test (assv xstr e) (list xstr 7))
    (test (assv car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3) (#\a 4) ("hiho" 5))))
  (test (assv 1+i e) '(1+i 1))
  (test (assv 3.0 e) '(3.0 2))
  (test (assv 5/3 e) '(5/3 3))
  (test (assv #\a e) '(#\a 4))
  (test (assv "hiho" e) #f))

(let ((e '(((a) 1) (#(a) 2) ("c" 3))))
  (test (assv '(a) e) #f)
  (test (assv '#(a) e) #f)
  (test (assv (string #\c) e) #f))

(let ((lst '((2 . a) (3 . b))))
  (set-cdr! (assv 3 lst) 'c)
  (test lst '((2 . a) (3 . c))))


(let ((e '((a 1) (b 2) (c 3))))
  (test (assoc 'a e) '(a 1))
  (test (assoc 'b e) '(b 2))
  (test (assoc 'd e) #f))
(test (assoc (list 'a) '(((a)) ((b)) ((c))))  '((a)))

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assoc #t e) (list #t 1))
    (test (assoc #f e) (list #f 2))
    (test (assoc 'a e) (list 'a 3))
    (test (assoc xcons e) (list xcons 4))
    (test (assoc xvect e) (list xvect 5))
    (test (assoc xlambda e) (list xlambda 6))
    (test (assoc xstr e) (list xstr 7))
    (test (assoc car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3) (#\a 4) ("hiho" 5))))
  (test (assoc 1+i e) '(1+i 1))
  (test (assoc 3.0 e) '(3.0 2))
  (test (assoc 5/3 e) '(5/3 3))
  (test (assoc #\a e) '(#\a 4))
  (test (assoc "hiho" e) '("hiho" 5)))

(let ((e '(((a) 1) (#(a) 2) ("c" 3))))
  (test (assoc '(a) e) '((a) 1))
  (test (assoc '#(a) e) '(#(a) 2))
  (test (assoc (string #\c) e) '("c" 3)))

(test (assoc 'a '((b c) (a u) (a i))) '(a u))
(test (assoc 'a '((b c) ((a) u) (a i))) '(a i))
(test (assoc (list 'a) '(((a)) ((b)) ((c))))  '((a)))
(test (assoc 5 '((2 3) (5 7) (11 13))) '(5 7))
(test (assoc 'key '()) #f)
(test (assoc 'key '(() ())) #f)
(test (assoc '() '()) #f)

(test (append '(a b c) '()) '(a b c))
(test (append '() '(a b c)) '(a b c))
(test (append '(a b) '(c d)) '(a b c d))
(test (append '(a b) 'c) '(a b . c))
(test (equal? (append (list 'a 'b 'c) (list 'd 'e 'f) '() '(g)) '(a b c d e f g)) #t)
(test (append (list 'a 'b 'c) (list 'd 'e 'f) '() (list 'g)) '(a b c d e f g))
(test (append (list 'a 'b 'c) 'd) '(a b c . d))
(test (append '() '()) '())
(test (append '() (list 'a 'b 'c)) '(a b c))
(test (append) '())
(test (append 'a) 'a)
(test (append '(x) '(y))  '(x y))
(test (append '(a) '(b c d)) '(a b c d))
(test (append '(a (b)) '((c)))  '(a (b) (c)))
(test (append '(a b) '(c . d))  '(a b c . d))
(test (append '() 'a)  'a)
(test (append '(a b) (append (append '(c)) '(e) 'f)) '(a b c e . f))
(test (append ''foo 'foo) '(quote foo . foo))
(test (append '() (cons 1 2)) '(1 . 2))
(test (append '() '() '()) '())
(test (append (cons 1 2)) '(1 . 2))

(test (append #f) #f)
(test (append '() #f) #f)
(test (append '(1 2) #f) '(1 2 . #f))
(test (append '() '() #f) #f)
(test (append '() '(1 2) #f) '(1 2 . #f))
(test (append '(1 2) '() #f) '(1 2 . #f))
(test (append '(1 2) '(3 4) #f) '(1 2 3 4 . #f))
(test (append '() '() '() #f) #f)
(test (append '(1 2) '(3 4) '(5 6) #f) '(1 2 3 4 5 6 . #f))

(test (append 0) 0) ; is this correct?
(test (append '() 0) 0)
(test (append '() '() 0) 0)
(test (let* ((x '(1 2 3)) (y (append x '()))) (eq? x y)) #f) ; check that append returns a new list
(test (let* ((x '(1 2 3)) (y (append x '()))) (equal? x y)) #t)
(test (let* ((x (list 1 2 3)) (y (append x (list)))) (eq? x y)) #f) 
(test (append '(1) 2) '(1 . 2))


(test (memq 'a '(a b c)) '(a b c))
(test (memq 'b '(a b c)) '(b c))
(test (memq 'a '(b c d)) #f)
(test (memq (list 'a) '(b (a) c))  #f)
(test (memq 'a '(b a c a d a)) '(a c a d a))
(let ((v (vector 'a))) (test (memq v (list 'a 1.2 v "hi")) (list v "hi")))
(test (memq #f '(1 a #t "hi" #f 2)) '(#f 2))
(test (memq eq? (list 2 eqv? 1 eq?)) (list eq?))
(test (memq eq? (list 2 eqv? 2)) #f)


(test (memv 101 '(100 101 102)) '(101 102))
(test (memv 3.4 '(1.2 2.3 3.4 4.5)) '(3.4 4.5))
(test (memv 3.4 '(1.3 2.5 3.7 4.9)) #f)
(let ((ls (list 'a 'b 'c)))
  (set-car! (memv 'b ls) 'z)
  (test ls '(a z c)))


(test (member (list 'a) '(b (a) c)) '((a) c))
(test (member "b" '("a" "c" "b")) '("b"))
(test (member 1 '(3 2 1 4)) '(1 4))
(test (member car (list abs car modulo)) (list car modulo))
(test (member do (list quote map do)) (list do))
(test (member 5/2 (list 1/3 2/4 5/2)) '(5/2))
(test (member 'a '(a b c d)) '(a b c d))
(test (member 'b '(a b c d)) '(b c d))
(test (member 'c '(a b c d)) '(c d))
(test (member 'd '(a b c d)) '(d))
(test (member 'e '(a b c d)) #f)




;;; --------------------------------------------------------------------------------
;;; VECTORS
;;; --------------------------------------------------------------------------------

(test (vector? (make-vector 6)) #t)
(test (vector? (make-vector 6 #\a)) #t)
(test (vector? (make-vector 0)) #t)
;; (test (vector? #*1011) #f)
(test (vector? '#(0 (2 2 2 2) "Anna")) #t)
(test (vector? '#()) #t)
(test (vector? '#("hi")) #t)
(test (vector? (vector 1)) #t)
(test (let ((v (vector 1 2 3))) (vector? v)) #t)

(for-each
 (lambda (arg)
   (test (vector? arg) #f))
 (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs 3.14 3/4 1.0+1.0i #t (lambda (a) (+ a 1))))


(test (let ((v (make-vector 3 #f))) (and (vector? v) (= (vector-length v) 3) (eq? (vector-ref v 1) #f))) #t)
(test (let ((v (make-vector 1 1))) (and (vector? v) (= (vector-length v) 1) (vector-ref v 0))) 1)
(test (let ((v (make-vector 0 1))) (and (vector? v) (= (vector-length v) 0))) #t)
(test (do ((vec (make-vector 5)) (i 0 (+ i 1))) ((= i 5) vec) (vector-set! vec i i)) '#(0 1 2 3 4))
(test (let ((v (make-vector 5))) (for-each (lambda (i) (vector-set! v i (* i i))) '(0 1 2 3 4)) v) '#(0 1 4 9 16))
(test (make-vector 2 'hi) '#(hi hi))
(test (make-vector 0) '#())
(test (make-vector 0 'hi) '#())
(test (make-vector 3 (make-vector 1 'hi)) '#(#(hi) #(hi) #(hi)))
(test (make-vector 3 '#(hi)) '#(#(hi) #(hi) #(hi)))
(test (make-vector 3 (list)) '#(() () ()))
(test (make-vector 3 (make-vector 1 (make-vector 1 'hi))) '#(#(#(hi)) #(#(hi)) #(#(hi))))

(for-each
 (lambda (arg)
   (test (vector-ref (make-vector 1 arg) 0) arg))
 (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))

(test (vector 1 2 3) '#(1 2 3))
(test (vector 1 '(2) 3) '#(1 (2) 3))
(test (vector) '#())
(test (vector (vector (vector))) '#(#(#())))
(test (vector (vector) (vector) (vector)) '#(#() #() #()))
(test (vector (list)) '#(()))
(test '#(1 #\a "hi" hi) (vector 1 #\a "hi" 'hi))
(test (let ((v (make-vector 4 "hi")))
	(vector-set! v 0 1)
	(vector-set! v 1 #\a)
	(vector-set! v 3 'hi)
	v)
      '#(1 #\a "hi" hi))
(let ((x 34))
  (test (vector x 'x) '#(34 x)))

(for-each
 (lambda (arg)
   (test (vector-ref (vector arg) 0) arg))
 (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))



(test (vector->list '#(0)) (list 0))
(test (vector->list (vector)) '())
(test (vector->list '#(a b c)) '(a b c))
(test (vector->list '#(#(0) #(1))) '(#(0) #(1)))
(test (vector? (list-ref (let ((v (vector 1 2))) (vector-set! v 1 v) (vector->list v)) 1)) #t)

(test (list->vector '()) '#())
(test (list->vector '(a b c)) '#(a b c))
(test (list->vector (list (list 1 2) (list 3 4))) '#((1 2) (3 4)))
(test (list->vector ''foo) '#(quote foo))
(test (list->vector (list)) '#())
(test (list->vector (list 1)) '#(1))
(test (list->vector (list (list))) '#(()))
(test (list->vector (list 1 #\a "hi" 'hi)) '#(1 #\a "hi" hi))

(for-each
 (lambda (arg)
   (if (list? arg)
       (test (vector->list (list->vector arg)) arg)))
 lists)

(test (list->vector (vector->list (vector))) '#())
(test (list->vector (vector->list (vector 1))) '#(1))
(test (vector->list (list->vector (list))) '())
(test (vector->list (list->vector (list 1))) '(1))



(test (vector-length (vector)) 0)
(test (vector-length (vector 1)) 1)
(test (vector-length (make-vector 128)) 128)
(test (vector-length '#(a b c d e f)) 6)
(test (vector-length '#()) 0)
(test (vector-length (vector #\a (list 1 2) (vector 1 2))) 3)
(test (vector-length '#(#(#(hi)) #(#(hi)) #(#(hi)))) 3)
(test (vector-length (vector 1 2 3 4)) 4)
(test (vector-length (let ((v (vector 1 2))) (vector-set! v 1 v) v)) 2)
(test (vector-length (let ((v (vector 1 2))) (vector-set! v 1 v) (vector-ref v 1))) 2)


(test (vector-ref '#(1 1 2 3 5 8 13 21) 5) 8)
(test (vector-ref '#(1 1 2 3 5 8 13 21) (let ((i (round (* 2 (acos -1))))) (if (inexact? i) (inexact->exact i)  i))) 13)
(test (let ((v (make-vector 1 0))) (vector-ref v 0)) 0)
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref v 1)) (list 2))
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref v 2)) '#(#\a #\a #\a))
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref (vector-ref v 2) 1)) #\a)
(test (vector-ref '#(a b c) 1) 'b)
(test (vector-ref '#(()) 0) '())
(test (vector-ref '#(#()) 0) '#())
(test (vector-ref (vector-ref (vector-ref '#(1 (2) #(3 (4) #(5))) 2) 2) 0) 5)
(test (let ((v (vector 1 2))) (vector-set! v 1 v) (eq? (vector-ref v 1) v)) #t)


(test (let ((vec (vector 0 '(2 2 2 2) "Anna"))) (vector-set! vec 1 '("Sue" "Sue")) vec) '#(0 ("Sue" "Sue") "Anna"))
(test (let ((v (vector 1 2 3))) (vector-set! v 1 32) v) '#(1 32 3))
(let ((v (make-vector 8 #f)))
  (for-each
   (lambda (arg)
     (vector-set! v 1 arg)
     (test (vector-ref v 1) arg))
   (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1)))))
(test (let ((v (vector 1 2 3))) (vector-set! v 1 0) v) '#(1 0 3))
(test (let ((v (vector #f))) (vector-set! v 0 (vector)) v) '#(#()))
(test (let ((v (vector 1 (list 2) (vector 1 2 3)))) (vector-set! (vector-ref v 2) 0 21) v) '#(1 (2) #(21 2 3)))



(test (let ((v (vector 1 2 3))) (vector-fill! v 0) v) '#(0 0 0))
(test (let ((v (vector))) (vector-fill! v #f) v) '#())
(let ((v (make-vector 8 #f)))
  (for-each
   (lambda (arg)
     (vector-fill! v arg)
     (test (vector-ref v 1) arg))
   (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1)))))

;(let ((str "hi") (v (make-vector 3))) (vector-fill! v str) (string-set! (vector-ref v 0) 1 #\a) str) -> mutable string error
;(let ((lst (list 1 2)) (v (make-vector 3))) (vector-fill! v lst) (list-set! (vector-ref v 0) 1 #\a) lst) -> '(1 #\a)


(if with-vector-for-each
    (begin
      (test (let ((sum 0)) (vector-for-each (lambda (n) (set! sum (+ sum n))) (vector 1 2 3)) sum) 6)
      (test (let ((sum 0)) (vector-for-each (lambda (n m) (set! sum (+ sum n (- m)))) (vector 1 2 3) (vector 4 5 6)) sum) -9)
      (test (let () (vector-for-each (lambda (n) (error "oops")) (vector)) #f) #f)
      (test (let ((sum 0)) (vector-for-each (lambda (n m p) (set! sum (+ sum n (- m) (* 2 p)))) (vector 1 2 3) (vector 4 5 6) (vector 6 7 8)) sum) 33)
      (test (let ((sum 0)) (vector-for-each (lambda (n) (vector-for-each (lambda (m) (set! sum (+ sum (* m n)))) (vector 1 2 3))) (vector 4 5 6)) sum) 90)
      (test (call/cc (lambda (return) (vector-for-each (lambda (n) (return "oops")) (vector 1 2 3)))) "oops")
      (test (call/cc (lambda (return) (vector-for-each (lambda (n) (if (even? n) (return n))) (vector 1 3 8 7 9 10)))) 8)


      (test (vector-map (lambda (n) (+ 1 n)) (vector 1 2 3)) '#(2 3 4))
      (test (vector-map (lambda (n m) (- n m)) (vector 1 2 3) (vector 4 5 6)) '#(-3 -3 -3))
      (test (vector-map (lambda (n m p) (+ n m p)) (vector 1 2 3) (vector 4 5 6) (vector 6 7 8)) '#(11 14 17))
      (test (vector-map (lambda (n) (vector-map (lambda (m) (* m n)) (vector 1 2 3))) (vector 4 5 6)) '#(#(4 8 12) #(5 10 15) #(6 12 18)))
      (test (call/cc (lambda (return) (vector-map (lambda (n) (return "oops")) (vector 1 2 3)))) "oops")
      (test (call/cc (lambda (return) (vector-map (lambda (n) (if (even? n) (return n))) (vector 1 3 8 7 9 10)))) 8)
      ))


(if (provided? 'multidimensional-vectors)
    (let ((v1 (make-vector 3 1)))
      (num-test (v1 1) 1)
      (set! (v1 1) 2)
      (num-test (v1 1) 2)
      (let ((i0 0)
	    (i2 2))
	(num-test (v1 i0) 1)
	(num-test (vector-ref v1 i2) 1)
	(set! (v1 i0) 0)
	(num-test (v1 0) 0)
	(set! (v1 i0) i2)
	(num-test (v1 i0) i2))
      (test (vector-dimensions v1) '(3))
      (set! v1 (make-vector '(3 2)))
      (test (vector-dimensions v1) '(3 2))
      (vector-set! v1 1 1 0)
      (num-test (vector-ref v1 1 1) 0)
      (let ((i0 1)
	    (i1 1)
	    (i2 32))
	(set! (v1 i0 i1) i2)
	(num-test (vector-ref v1 1 1) 32)
	(num-test (v1 i0 i1) i2)
	(vector-set! v1 0 1 3)
	(num-test (v1 0 1) 3)
	(num-test (v1 1 1) 32))
      (set! v1 (make-vector '(2 4 3) 1))
      (test (vector-dimensions v1) '(2 4 3))      
      (num-test (vector-ref v1 1 1 1) 1)
      (vector-set! v1 0 0 0 32)
      (num-test (v1 0 0 0) 32)
      (set! (v1 0 1 1) 3)
      (num-test (v1 0 1 1) 3)

      (if with-vector-for-each
	  (test (let ((v1 (make-vector '(3 2) 1))
		      (v2 (make-vector '(3 2) 2))
		      (sum 0))
		  (vector-for-each (lambda (n m) (set! sum (+ sum n m))) v1 v2)
		  sum)
		18))

      (test (equal? (make-vector 0) (make-vector '(0))) #t)
      (test (make-vector (1 . 2) "hi") 'error)))


(set! lists '())



;;; --------------------------------------------------------------------------------
;;; PORTS
;;; --------------------------------------------------------------------------------

(define start-input-port (current-input-port))
(define start-output-port (current-output-port))

(test (input-port? (current-input-port)) #t)

(for-each
 (lambda (arg)
   (if (input-port? arg)
       (begin
	 (display "(input-port? ") (display arg) (display ") returned #t?") (newline))))
 (list "hi" #f (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f))

(test (call-with-input-file "s7test.scm" input-port?) #t)
(if (not (eq? start-input-port (current-input-port)))
    (begin (display "call-with-input-file did not restore current-input-port? ") (display start-input-port) (display " ") (display (current-input-port)) (newline)))

(test (let ((this-file (open-input-file "s7test.scm"))) (let ((res (input-port? this-file))) (close-input-port this-file) res)) #t)
(if (not (eq? start-input-port (current-input-port)))
    (begin (display "open-input-file clobbered current-input-port? ") (display start-input-port) (display " ") (display (current-input-port)) (newline)))

(if with-open-input-string-and-friends
    (test (call-with-input-string "(+ 1 2)" input-port?) #t))
(if with-open-input-string-and-friends
    (test (let ((this-file (open-input-string "(+ 1 2)"))) (let ((res (input-port? this-file))) (close-input-port this-file) res)) #t))

(test (+ 100 (call-with-input-string "123" (lambda (p) (values (read p) 1)))) 224)


(test (output-port? (current-output-port)) #t)
(write-char #\space (current-output-port))
(write " " (current-output-port))
(newline (current-output-port))


(for-each
 (lambda (arg)
   (if (output-port? arg)
       (begin
	 (display "(output-port? ") (display arg) (display ") returned #t?") (newline))))
 (list "hi" #f (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f))

(test (call-with-output-file "tmp1.r5rs" output-port?) #t)
(if (not (eq? start-output-port (current-output-port)))
    (begin (display "call-with-output-file did not restore current-output-port? ") (display start-output-port) (display " ") (display (current-output-port)) (newline)))

(test (let ((this-file (open-output-file "tmp1.r5rs"))) (let ((res (output-port? this-file))) (close-output-port this-file) res)) #t)
(if (not (eq? start-output-port (current-output-port)))
    (begin (display "open-output-file clobbered current-output-port? ") (display start-output-port) (display " ") (display (current-output-port)) (newline)))

(if with-open-input-string-and-friends
    (test (let ((val #f)) (call-with-output-string (lambda (p) (set! val (output-port? p)))) val) #t))
(if with-open-input-string-and-friends
    (test (let ((res #f)) (let ((this-file (open-output-string))) (set! res (output-port? this-file)) (close-output-port this-file) res)) #t))



(for-each
 (lambda (arg)
   (if (eof-object? arg)
       (begin
	 (display "(eof-object? ") (display arg) (display ") returned #t?") (newline))))
 (list "hi" -1 #\a 1 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #f #t (lambda (a) (+ a 1))))

(call-with-output-file "tmp1.r5rs" (lambda (p) (display "3.14" p)))
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (read p) (let ((val (read p))) (eof-object? val)))) #t)


(test (call-with-input-file "tmp1.r5rs" (lambda (p) (read-char p))) #\3)
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (peek-char p))) #\3)
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (peek-char p) (read-char p))) #\3)
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (list->string (list (read-char p) (read-char p) (read-char p) (read-char p))))) "3.14")
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (list->string (list (read-char p) (peek-char p) (read-char p) (read-char p) (peek-char p) (read-char p))))) "3..144")

(for-each
 (lambda (arg)
   (call-with-output-file "tmp1.r5rs" (lambda (p) (write arg p)))
   (test (call-with-input-file "tmp1.r5rs" (lambda (p) (read p))) arg))
 (list "hi" -1 #\a 1 'a-symbol (make-vector 3 0) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) (cons 1 2)))

;;; r4rstest
(let* ((write-test-obj '(#t #f a () 9739 -3 . #((test) "te \" \" st" "" test #() b c)))
       (load-test-obj (list 'define 'foo (list 'quote write-test-obj))))

  (define (check-test-file name)
    (let ((val (call-with-input-file
		   name
		 (lambda (test-file)
		   (test (read test-file) load-test-obj)
		   (test (eof-object? (peek-char test-file)) #t)
		   (test (eof-object? (read-char test-file)) #t)
		   (input-port? test-file)))))
      (if (not (eq? val #t))
	  (begin (display "input-port? in call-with-input-file? returned ") (display val) (newline)))))

  (test (call-with-output-file
	    "tmp1.r5rs"
	  (lambda (test-file)
	    (write-char #\; test-file)
	    (display #\; test-file)
	    (display ";" test-file)
	    (write write-test-obj test-file)
	    (newline test-file)
	    (write load-test-obj test-file)
	    (output-port? test-file))) #t)
  (check-test-file "tmp1.r5rs")

  (let ((test-file (open-output-file "tmp2.r5rs")))
    (write-char #\; test-file)
    (display #\; test-file)
    (display ";" test-file)
    (write write-test-obj test-file)
    (newline test-file)
    (write load-test-obj test-file)
    (test (output-port? test-file) #t)
    (close-output-port test-file)
    (check-test-file "tmp2.r5rs")))


(call-with-output-file "tmp1.r5rs" (lambda (p) (display "3.14" p)))
(test (with-input-from-file "tmp1.r5rs" (lambda () (read))) 3.14)
(if (not (eq? start-input-port (current-input-port)))
    (begin (display "with-input-from-file did not restore current-input-port? ") (display start-input-port) (display " ") (display (current-input-port)) (newline)))

(test (with-input-from-file "tmp1.r5rs" (lambda () (eq? (current-input-port) start-input-port))) #f)

(test (with-output-to-file "tmp1.r5rs" (lambda () (eq? (current-output-port) start-output-port))) #f)
(if (not (eq? start-output-port (current-output-port)))
    (begin (display "with-output-to-file did not restore current-output-port? ") (display start-output-port) (display " ") (display (current-output-port)) (newline)))


(let ((newly-found-sonnet-probably-by-shakespeare 
        "This is the story, a sad tale but true \
        Of a programmer who had far too little to do.\
        One day as he sat in his hut swilling stew, \
        He cried \"CLM takes forever, it's stuck in a slough!,\
        Its C code is slow, too slow by a few.\
        Why, with just a small effort, say one line or two,\
        It could outpace a no-op, you could scarcely say 'boo'\"!\
        So he sat in his kitchen and worked like a dog.\
        He typed and he typed 'til his mind was a fog. \
        Now 6000 lines later, what wonders we see!  \
        CLM is much faster, and faster still it will be!\
        In fact, for most cases, C beats the DSP!  \
        But bummed is our coder; he grumbles at night.  \
        That DSP code took him a year to write.  \
        He was paid many dollars, and spent them with glee,\
        But his employer might mutter, this result were he to see."))
  (call-with-output-file "tmp1.r5rs"
    (lambda (p)
      (write newly-found-sonnet-probably-by-shakespeare p)))
  (let ((sonnet (with-input-from-file "tmp1.r5rs"
		  (lambda ()
		    (read)))))
    (if (or (not (string? sonnet))
	    (not (string=? sonnet newly-found-sonnet-probably-by-shakespeare)))
	(begin (display "write/read long string returned: ") (display sonnet) (newline))))

  (let ((file (open-output-file "tmp1.r5rs")))
    (let ((len (string-length newly-found-sonnet-probably-by-shakespeare)))
      (write-char #\" file)
      (do ((i 0 (+ i 1)))
	  ((= i len))
	(let ((chr (string-ref newly-found-sonnet-probably-by-shakespeare i)))
	  (if (char=? chr #\")
	      (write-char #\\ file))
	  (write-char chr file)))
      (write-char #\" file)
      (close-output-port file)))
  (let ((file (open-input-file "tmp1.r5rs")))
    (let ((sonnet (read file)))
      (close-input-port file)
      (if (or (not (string? sonnet))
	      (not (string=? sonnet newly-found-sonnet-probably-by-shakespeare)))
	  (begin (display "write-char/read long string returned: ") (display sonnet) (newline))))))

(let ((file (open-output-file "tmp1.r5rs")))
  (for-each
   (lambda (arg)
     (write arg file)
     (write-char #\space file))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))
  (close-output-port file))
(let ((file (open-input-file "tmp1.r5rs")))
  (for-each
   (lambda (arg)
     (let ((val (read file)))
       (if (not (equal? val arg))
	   (begin (display "read/write ") (display arg) (display " returned ") (display val) (newline)))))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))
  (close-input-port file))

(with-output-to-file "tmp1.r5rs"
  (lambda ()
    (write lists)))
(let ((val (with-input-from-file "tmp1.r5rs"
	     (lambda ()
	       (read)))))
  (if (not (equal? val lists))
      (begin (display "read/write lists returned ") (display val) (newline))))

(if with-open-input-string-and-friends
    (if (not (string=? "" (with-output-to-string (lambda () (display "")))))
	(begin (display "with-output-to-string null string?"))))

(if with-open-input-string-and-friends
    (let ((str (with-output-to-string
		 (lambda ()
		   (with-input-from-string "hiho123"
		     (lambda ()
		       (do ((c (read-char) (read-char)))
			   ((eof-object? c))
			 (display c))))))))
      (if (not (string=? str "hiho123"))
	  (begin (display "with string ports: \"") (display str) (display "\"?") (newline)))))

(if with-open-input-string-and-friends
    (begin
      (if (not (eof-object? (with-input-from-string "" (lambda () (read-char)))))
	  (begin (display ";input from null string not #<eof>?") (newline))
	  (let ((EOF (with-input-from-string "" (lambda () (read-char)))))
	    (if (not (eq? (with-input-from-string "" (lambda () (read-char)))
			  (with-input-from-string "" (lambda () (read-char)))))
		(begin (display "#<eof> is not eq? to itself?") (newline)))
	    (if (char? EOF)
		(do ((c 0 (+ c 1)))
		    ((= c 256))
		  (if (char=? EOF (integer->char c))
		      (begin
			(display "#<eof> is char=? to ") (display (integer->char c)) (newline)))))))))

(test (+ 100 (call-with-output-file "tmp.r5rs" (lambda (p) (write "1" p) (values 1 2)))) 103)
(test (+ 100 (with-output-to-file "tmp.r5rs" (lambda () (write "2") (values 1 2)))) 103)

(if with-open-input-string-and-friends
    (let ((str (with-output-to-string
		 (lambda ()
		   (with-input-from-string "hiho123"
		     (lambda ()
		       (do ((c (read-char) (read-char)))
			   ((or (not (char-ready?))
				(eof-object? c)))
			 (display c))))))))
      (if (not (string=? str "hiho123"))
	  (begin (display "with string ports: \"") (display str) (display "\"?") (newline)))))

(if with-open-input-string-and-friends
    (let ((str (with-output-to-string
		 (lambda ()
		   (with-input-from-string ""
		     (lambda ()
		       (do ((c (read-char) (read-char)))
			   ((eof-object? c))
			 (display c))))))))
      (if (not (string=? str ""))
	  (begin (display "with string ports and null string: \"") (display str) (display "\"?") (newline)))))

(if with-open-input-string-and-friends
    (let ((str (with-output-to-string ; this is from the guile-user mailing list, I think -- don't know who wrote it
		 (lambda ()
		   (with-input-from-string "A2B5E3426FG0ZYW3210PQ89R."
		     (lambda ()
		       (call/cc
			(lambda (hlt)
			  (define (nextchar)
			    (let ((c (read-char)))
			      (if (and (char? c) 
				       (char=? c #\space))
				  (nextchar) 
				  c)))
			  
			  (define inx
			    (lambda()
			      (let in1 ()
				(let ((c (nextchar)))
				  (if (char-numeric? c)
				      (let ((r (nextchar)))
					(let out*n ((n (- (char->integer c) (char->integer #\0))))
					  (out r)
					  (if (not (zero? n))
					      (out*n (- n 1)))))
				      (out c))
				  (in1)))))
			  
			  (define (move-char c)
			    (write-char c)
			    (if (char=? c #\.)
				(begin (hlt))))
			  
			  (define outx
			    (lambda()
			      (let out1 ()
				(let h1 ((n 16))
				  (move-char (in))
				  (move-char (in))
				  (move-char (in))
				  (if (= n 1)
				      (begin (out1))
				      (begin (write-char #\space) (h1 (- n 1))) )))))
			  
			  (define (in)
			    (call/cc (lambda(return)
				       (set! outx return)
				       (inx))))
			  
			  (define (out c)
			    (call/cc (lambda(return) 
				       (set! inx return)
				       (outx c))))
			  (outx)))))))))
      (if (not (string=? str "ABB BEE EEE E44 446 66F GZY W22 220 0PQ 999 999 999 R."))
	  (format #t "call/cc with-input-from-string str: ~A~%" str))))
    
(let ((badfile "tmp1.r5rs"))
  (let ((p (open-output-file badfile)))
    (close-output-port p))
  (load badfile))

(if with-procedure-arity
    (let ((loadit "tmp1.r5rs"))
      (let ((p (open-output-file loadit)))
	(display "(define s7test-var 314) (define (s7test-func) 314) (define-macro (s7test-mac a) `(+ ,a 2))" p)
	(newline p)
	(close-output-port p)
	(load loadit)
	(test (= s7test-var 314) #t)
	(test (s7test-func) 314)
	(test (s7test-mac 1) 3)
	(set! p (open-output-file loadit)) ; hopefully this starts a new file
	(display "(define s7test-var 3) (define (s7test-func) 3) (define-macro (s7test-mac a) `(+ ,a 1))" p)
	(newline p)
	(close-output-port p)
	(load loadit)
	(test (= s7test-var 3) #t)
	(test (s7test-func) 3)
	(test (s7test-mac 1) 2)
	)))

(if with-open-input-string-and-friends
    (test (+ 100 (with-input-from-string "123" (lambda () (values (read) 1)))) 224))

    
    
;;; --------------------------------------------------------------------------------
;;; CONTROL OPS
;;; --------------------------------------------------------------------------------
    
(define control-ops (list lambda define quote if begin set! let let* letrec cond case and or do
			  call/cc eval apply for-each map values call-with-values dynamic-wind
			  quasiquote))
(for-each
 (lambda (op)
   (if (not (eq? op op))
       (begin (display op) (display " not eq? to itself?") (newline))))
 control-ops)

(for-each
 (lambda (op)
   (if (not (eqv? op op))
       (begin (display op) (display " not eqv? to itself?") (newline))))
 control-ops)

(for-each
 (lambda (op)
   (if (not (equal? op op))
       (begin (display op) (display " not equal? to itself?") (newline))))
 control-ops)

(define question-ops (list boolean? eof-object? string? 
		           number? integer? real? rational? complex? char?
			   list? vector? pair? null?))

(for-each
 (lambda (ques)
   (for-each
    (lambda (op)
      (if (ques op)
	  (begin (display ques) (display " ") (display op) (display " returned #t?") (newline))))
    control-ops))
 question-ops)


(for-each 
 (lambda (s)
   (if (not (symbol? s))
       (begin (display "(symbol? ") (display s) (display " returned #f?") (newline))))
 '(+ - ... !.. $.+ %.- &.! *.: /:. <-. =. >. ?. ~. _. ^.))



;;; -------- if --------

(test ((if #f + *) 3 4) 12)
(test (if (> 3 2) 'yes 'no) 'yes)
(test (if (> 2 3) 'yes 'no) 'no)
(test (if (> 3 2) (- 3 2) (+ 3 2)) 1)
(test (if (> 3 2) 1) 1)
(test (if '() 1 2) 1)
(test (if 't 1 2) 1)
(test (if #t 1 2) 1)
(test (if '#() 1 2) 1)
(test (if 1 2 3) 2)
(test (if 0 2 3) 2)
(test (if (list) 2 3) 2)
(test (if "" 2 3) 2)

(test (let ((a #t) (b #f) (c #t) (d #f)) (if (if (if (if d d c) d b) d a) 'a 'd)) 'a)
(test (let ((a #t) (b #f) (c #t) (d #f)) (if a (if b (if c (if d d c) c) 'b) 'a)) 'b)
;(test (let ((a #t) (b #f) (c #t) (d #f)) (((if a if 'gad) c if 'gad) (not d) 'a 'gad)) 'a)
(test (let ((a #t) (b #f) (c #t) (d #f)) (if b (if a (if d 'gad) 'gad) (if d 'gad 'a))) 'a)

(let ((a #t))
  (for-each
   (lambda (arg)
     (test (if a arg 'gad) arg))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2))))

(let ((a #t))
  (for-each
   (lambda (arg)
     (test (if (not a) 'gad arg) arg))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2))))

(test (let ((ctr 0) (a #t)) (if a (let ((b ctr)) (set! ctr (+ ctr 1)) (list b ctr)) (let ((c ctr)) (set! ctr (+ ctr 100)) (list c ctr)))) (list 0 1))

(test (if if if if) if)
;(test (((if if if) if if) if if 'gad) if)
(test (if if (if if if) if) if)
;(test (let ((car if)) (car #t 0 1)) 0)
;(test ((car (list if)) #t 0 1) 0)
(test (symbol->string 'if) "if")
(test (if (and if (if if if)) if 'gad) if)
;(test (let ((if #t)) (or if)) #t)
;(test (let ((if +)) (if 1 2 3)) 6)
(test (let ((ctr 0)) (if (let () (set! ctr (+ ctr 1)) (= ctr 1)) 0 1)) 0)
(test (let ((ctr 0)) (if (let () (set! ctr (+ ctr 1)) (if (= ctr 1) (> 3 2) (< 3 2))) 0 1)) 0)
(test (        if (> 3 2) 1 2) 1)
(test (let ((alist (list (list map 1) (list car 2) (list if 3) (list do 4)))) (assoc if alist)) (list if 3))
(test (let ((alist (list (list map 1) (list car 2) (list if 3) (list do 4)))) (assv if alist)) (list if 3))
(test (let ((alist (list (list map 1) (list car 2) (list if 3) (list do 4)))) (assq if alist)) (list if 3))
(test (let ((alist (list map car if do))) (member if alist)) (list if do))
(test (let ((alist (list map car if do))) (memv if alist)) (list if do))
(test (let ((alist (list map car if do))) (memq if alist)) (list if do))
;(test ((vector-ref (vector if) 0) #t 1 2) 1)
;(test ((vector-ref (make-vector 1 if) 0) #t 1 2) 1)
(test ((if #t + -) 3 4) 7)
(test (list (if 0 1 2)) (list 1))
;(test ((car (list if map)) #f 1 2) 2)
(test (let ((ctr 0)) (if (= ctr 0) (let () (set! ctr (+ ctr 1)) (if (= ctr 1) 2 3)) (let () (set! ctr (+ ctr 1)) (if (= ctr 1) 4 5)))) 2)
(test (let ((x (cons 1 2))) (set-cdr! x x) (if x 1 2)) 1)
(test (let ((ctr 0)) (if (let ((ctr 123)) (set! ctr (+ ctr 1)) (= ctr 124)) (let () (set! ctr (+ ctr 100)) ctr) (let () (set! ctr (+ ctr 1000)) ctr)) ctr) 100)
(test (if (let ((if 3)) (> 2 if)) 4 5) 5)

(test (let ((ctr 0)) (call/cc (lambda (exit) (if (> 3 2) (let () (exit ctr) (set! ctr 100) ctr) #f)))) 0)
(test (let ((ctr 0)) (call/cc (lambda (exit) (if (< 3 2) #f (let () (exit ctr) (set! ctr 100) ctr))))) 0)
(test (let ((ctr 0)) (call/cc (lambda (exit) (if (let () (exit ctr) (set! ctr 100) ctr) 123 321)))) 0)
(test (let ((ctr 0)) (if (> 3 2) (call/cc (lambda (exit) (set! ctr (+ ctr 1)) (exit ctr))) #f) ctr) 1)

(test (let ((ctr 0))
	(do ((x 0 (+ x 1)))
	    ((= x 12))
	  (if (> x 0)
	      (if (> x 1)
		  (if (> x 2)
		      (if (> x 3)
			  (if (> x 4)
			      (if (> x 5)
				  (if (> x 6)
				      (if (> x 7)
					  (if (> x 8)
					      (if (> x 9)
						  (if (> x 10)
						      (set! ctr (+ ctr 1000))
						      (set! ctr (- ctr 1)))
						  (set! ctr (- ctr 2)))
					      (set! ctr (- ctr 3)))
					  (set! ctr (- ctr 4)))
				      (set! ctr (- ctr 5)))
				  (set! ctr (- ctr 6)))
			      (set! ctr (- ctr 7)))
			  (set! ctr (- ctr 8)))
		      (set! ctr (- ctr 9)))
		  (set! ctr (- ctr 10)))
	      (set! ctr (- ctr 11))))
	ctr)
      934)

(test (let ((ctr 0))
	(do ((x 0 (+ x 1)))
	    ((= x 12))
	  (if (> x 0)
	      (if (> x 1)
		  (if (> x 2)
		      (if (> x 3)
			  (if (> x 4)
			      (if (> x 5)
				  (if (> x 6)
				      (if (> x 7)
					  (if (> x 8)
					      (if (> x 9)
						  (if (> x 10)
						      (set! ctr (+ ctr 1000))
						      (set! ctr (- ctr 1)))
						  (set! ctr (- ctr 2)))
					      (set! ctr (- ctr 3)))
					  (set! ctr (- ctr 4))))))))
		  (set! ctr (- ctr 10)))
	      (set! ctr (- ctr 11))))
	ctr)
      969)




;;; -------- quote --------

(test (quote a) 'a)
(test 'a (quote a))
(test '1 1)
(test '(+ 2 3) '(+ 2 3))
(test '"hi" "hi")
(test '#\a #\a)
(test '#f #f)
(test '#t #t)
(test '#b1 1)
(test '() '())
(test (+ '1 '2) 3)
(test (+ '1 '2) '3)
(test (+ ' 1 '   2) '    3)
(test (char? '#\a) #t)
(test (string? '"hi") #t)
(test (boolean? '#t) #t)
(test (if '#f 2 3) 3)
(test (if '#t 2 3) 2)
(test (vector? '#()) #t)
(test (char? (quote #\a)) #t)
(test (string? (quote "hi")) #t)
(test (boolean? (quote #t)) #t)
(test (if (quote #f) 2 3) 3)
(test (if (quote #t) 2 3) 2)
(test (vector? (quote #())) #t)
(test (+ (quote 1) (quote 2)) (quote 3))
(test (list? (quote ())) #t)
(test (pair? (quote (1 . 2))) #t)
(test (+ '1.0 '2.0) 3.0)
(test (+ '1/2 '3/2) 2)
(test (+ '1.0+1.0i '-2.0) -1.0+1.0i)
(test (let ((hi 2)) (equal? hi 'hi)) #f)
(test ''1 (quote (quote 1)))
(test ''a (quote (quote a)))
(test (symbol? '#f) #f)
(test ''quote (quote (quote quote)))
(test (+ (cadr ''3) (cadadr '''4) (cadr (cadr (cadr ''''5)))) 12)

;(test (eq? '() ()) #t) ; not sure about this -- Gauche, SCM, stklos say #t; Guile says error; clisp, cmucl, and sbcl say T




;;; -------- for-each --------

(test (let ((v (make-vector 5))) (for-each (lambda (i) (vector-set! v i (* i i))) '(0 1 2 3 4)) v) '#(0 1 4 9 16))
(test (let ((ctr 0) (v (make-vector 5))) (for-each (lambda (i) (vector-set! v ctr (* i i)) (set! ctr (+ ctr 1))) '(0 1 2 3 4)) v) '#(0 1 4 9 16))
(for-each (lambda (x) (display "for-each should not have called this")) '())
(test (let ((ctr 0)) (for-each (lambda (x y) (if (= x y) (set! ctr (+ ctr 1)))) '(1 2 3 4 5 6) '(2 3 3 4 7 6)) ctr) 3)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(0 1) '(2 3) '(4 5)) ctr) 15)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(1) '(3) '(5)) ctr) 9)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '() '() '()) ctr) 0)
(test (let () (for-each abs '(1 2)) 1) 1)
(test (let ((ctr 0)) (for-each (lambda (a) (for-each (lambda (b) (set! ctr (+ ctr 1))) '(0 1))) '(2 3 4)) ctr) 6)
(test (let ((sum 0)) (for-each (lambda args (set! sum (+ sum (apply + args)))) '(0 1 2) '(2 1 0) '(3 4 5) '(5 4 3) '(6 7 8) '(8 7 6)) sum) 72)
(test (let ((sum 0)) (for-each (lambda (a b . args) (set! sum (+ sum a b (apply + args)))) '(0 1 2) '(2 1 0) '(3 4 5) '(5 4 3) '(6 7 8) '(8 7 6)) sum) 72)
(test (let ((sum 0)) (for-each (lambda (a b . args) (set! sum (+ sum a b (apply + args)))) '(0 1 2) '(2 1 0)) sum) 6)
(test (let () (for-each + '(0 1 2) '(2 1 0)) 0) 0)
(test (let ((d 0))
	(for-each (let ((a 0))
		    (for-each (lambda (b) (set! a (+ a b))) (list 1 2))
		    (lambda (c) (set! d (+ d c a))))
		  (list 3 4 5))
	d)
      21)
(test (let ((d 0))
	(for-each (lambda (c)
		    (let ((a 0))
		      (for-each (lambda (b) (set! a (+ a b))) (list 1 2))
		      (set! d (+ d a c))))
		  (list 3 4 5))
	d)
      21)

(test (let ((ctr 0)) 
	(let ((val (call/cc 
		    (lambda (exit) 
		      (for-each (lambda (a) 
				  (if (> a 3) (exit a)) 
				  (set! ctr (+ ctr 1))) 
				(list 0 1 2 3 4 5)))))) 
	  (list ctr val)))
      (list 4 4))

(test (call-with-current-continuation
       (lambda (exit)
	 (for-each 
	  (lambda (x) 
	    (if (negative? x) (exit x)))
	  '(54 0 37 -3 245 19))
	 #t))
      -3)

(test (let ((ctr 0)
	    (cont #f)
	    (lst '()))
	(let ((val (call/cc 
		    (lambda (exit) 
		      (for-each (lambda (a) 
				  (if (and (not cont) (= a 2))
				      (exit a)) 
				  (if (and cont (= a 5)) 
				      (exit a))
				  (call/cc (lambda (c) (set! cont c)))
				  (set! lst (cons ctr lst))
				  (set! ctr (+ ctr 1)))
				(list 0 1 2 3 4 5)))))) 
	  (if (< val 5)
	      (cont))
	  (list ctr val lst)))
      (list 5 5 (list 4 3 2 1 0)))

(test (let ((lst '())) 
	(for-each (lambda (a) (set! lst (cons a lst))) 
		  (let ((lst '())) 
		    (for-each (lambda (b) (set! lst (cons b lst))) 
			      (list 1 2 3)) 
		    lst)) 
	lst) 
      (list 1 2 3))

;;; this is an infinite loop?
; (let ((cont #f)) (call/cc (lambda (x) (set! cont x))) (for-each cont (list 1 2 3)))
(test (call/cc (lambda (x) (for-each x (list 1 2 3)))) 1) ; map also gives 1 ... perhaps not actually legal?

(test (let ((ctr 0))
	(for-each 
	 (lambda (x)
	   (for-each
	    (lambda (x y)
	      (for-each 
	       (lambda (x y z)
		 (set! ctr (+ x y z)))
	       (list x (+ x 1))
	       (list y (+ y 2))
	       (list (+ x y) (- x y))))
	    (list (+ x 3) (+ x 4) (+ x 5))
	    (list (- x 3) (- x 4) (- x 5))))
	 (list 1 2 3 4 5))
	ctr)
      23)
					      
	  



;;; -------- map --------

(test (map cadr '((a b) (d e) (g h))) '(b e h))
(test (map (lambda (n) (expt n n)) '(1 2 3 4 5)) '(1 4 27 256 3125))
(test (map + '(1 2 3) '(4 5 6)) '(5 7 9))

(test (apply vector (map (lambda (i) (* i i)) '(0 1 2 3 4))) '#(0 1 4 9 16))
(map (lambda (x) (display "map should not have called this")) '())
(test (let ((ctr 0)) (map (lambda (x y) (if (= x y) (set! ctr (+ ctr 1))) ctr) '(1 2 3 4 5 6) '(2 3 3 4 7 6))) (list 0 0 1 2 2 3))
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '(0 1) '(2 3) '(4 5))) (list 6 15))
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '(1) '(3) '(5))) (list 9))
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '() '() '())) '())
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1 2)) (list 2 4))
(test (map abs '(1 -2)) (list 1 2))
(test (map + '(0 1 2) '(2 1 0) '(3 4 5) '(5 4 3) '(6 7 8) '(8 7 6)) (list 24 24 24))
(test (map (lambda (a) (cons a (map (lambda (b) (+ b 1)) (list 0 1 2)))) (list 3 4 5)) '((3 1 2 3) (4 1 2 3) (5 1 2 3)))
(test (map (lambda (a) (+ a 1)) (map (lambda (b) (+ b 1)) (map (lambda (c) (+ c 1)) (list 0 1 2)))) '(3 4 5))
(test (map (lambda args (apply + args)) '(0 1 2) '(3 4 5) '(6 7 8) '(9 10 11) '(12 13 14)) '(30 35 40))
(test (map (lambda (a b . args) (+ a b (apply + args))) '(0 1 2) '(3 4 5) '(6 7 8) '(9 10 11) '(12 13 14)) '(30 35 40))
(test (map (lambda (a b . args) (+ a b (apply + args))) '(0 1 2) '(3 4 5)) '(3 5 7))

(test (let ((d 0))
	(map (let ((a 0))
	       (map (lambda (b) (set! a (+ a b))) (list 1 2))
	       (lambda (c) (set! d (+ d c a)) d))
	     (list 3 4 5)))
      (list 6 13 21))
(test (let ((d 0))
	(map (lambda (c)
	       (let ((a 0))
		 (map (lambda (b) (set! a (+ a b))) (list 1 2))
		 (set! d (+ d a c))
		 d))
	     (list 3 4 5)))
      (list 6 13 21))

(test (let ((ctr 0))
	(let ((val (call/cc 
		    (lambda (exit) 
		      (map (lambda (a) 
			     (if (> a 3) (exit a)) 
			     (set! ctr (+ ctr 1))
			     ctr)
			   (list 0 1 2 3 4 5))))))
	  (list ctr val)))
      (list 4 4))

(test (call-with-current-continuation
       (lambda (exit)
	 (map 
	  (lambda (x) 
	    (if (negative? x) (exit x))
	    x)
	  '(54 0 37 -3 245 19))))
      -3)

(test (let ((ctr 0)
	    (cont #f)
	    (lst '()))
	(let ((val (call/cc 
		    (lambda (exit) 
		      (map (lambda (a) 
			     (if (and (not cont) (= a 2))
				 (exit a)) 
			     (if (and cont (= a 5)) 
				 (exit a))
			     (call/cc (lambda (c) (set! cont c)))
			     (set! lst (cons ctr lst))
			     (set! ctr (+ ctr 1))
			     ctr)
			   (list 0 1 2 3 4 5))))))
	  (if (< val 5)
	      (cont))
	  (list ctr val lst)))
      (list 5 5 (list 4 3 2 1 0)))

(test (map (lambda (a) a) (map (lambda (b) b) (list 1 2 3))) (list 1 2 3))




;;; --------- do --------

(test (do () (#t 1)) 1)
(for-each
 (lambda (arg)
   (test (do () (#t arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (do ((i arg)) (#t i)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))

(test (do ((i 0 (+ i 1))) ((= i 3) #f)) #f)
(test (do ((i 0 (+ i 1))) ((= i 3) i)) 3)
(test (do ((vec (make-vector 5)) (i 0 (+ i 1))) ((= i 5) vec) (vector-set! vec i i)) '#(0 1 2 3 4))
(test (let ((x '(1 3 5 7 9))) (do ((x x (cdr x)) (sum 0 (+ sum (car x))))  ((null? x) sum))) 25)
(test (do ((i 4 (- i 1)) (a 1 (* a i))) ((zero? i) a)) 24)
	   
;(test (do () (() ()) ()) '()) ; ?? -- is '() the same as ()? -- scheme bboard sez not necessarily
(test (do () ('() '())) '())

(test (let ((x 0) (y 0)) (set! y (do () (#t (set! x 32) 123))) (list x y)) (list 32 123))
(test (let ((i 32)) (do ((i 0 (+ i 1)) (j i (+ j 1))) ((> j 33) i))) 2)
(test (let ((i 0)) (do () ((> i 1)) (set! i (+ i 1))) i) 2)
(test (let ((i 0) (j 0)) (do ((k #\a)) (#t i) (set! i (char->integer k)) (set! j (+ j i)))) 0)
(test (let ((i 0) (j 0)) (do ((k #\a)) ((> i 1) j) (set! i (char->integer k)) (set! j (+ j i)))) (char->integer #\a))
(test (let ((x 0)) (do ((i 0 (+ i 2)) (j 1 (* j 2))) ((= i 4) x) (set! x (+ x i j)))) 5)
(test (let ((sum 0)) (do ((lst '(1 2 3 4) (cdr lst))) ((null? lst) sum) (set! sum (+ sum (car lst))))) 10)
(test (do ((i 0 (+ 1 i))) ((= i 4) (do ((i 0 (+ i 2))) ((= i 10) i)))) 10)
(test (let ((i 0)) (do ((i 1 (+ i 1))) ((= i 3) i))) 3)
(test (let ((j 0)) (do ((i 0 (+ i 1))) ((= i 3) (+ i j)) (do ((j 0 (+ j i 1))) ((> j 3) j)))) 3)
(test (let ((add1 (lambda (a) (+ a 1)))) (do ((i 0 (add1 i))) ((= i 10) (add1 i)))) 11)
(test (do ((i 0 (do ((j 0 (+ j 1))) ((= j i) (+ i 1))))) ((= i 3) i)) 3)
(test (do ((i 0 (do ((i 0 (+ i 1))) ((= i 3) i)))) ((= i 3) i)) 3)
(test (let ((i 123)) (do ((i 0 (+ i 1)) (j i (+ j i))) ((> j 200) i))) 13)
(test (do ((i 0 (+ i 1))) ((> i 3) i) (set! i (* i 10))) 11)
(test (do ((i 123) (j 0 (+ j i))) ((= j 246) i)) 123)
(test (do ((i 123 i) (j 0 (+ j i))) ((= j 246) i)) 123)
(test (do ((i 0 i)) (i i)) 0)
(test (do ((i 1 i)) (i i (+ i i) (+ i i i))) 3)
(test (do ((i 1)) (#t 1) 123) 1)
(test (do ((i 0 (+ i j)) (j 0 (+ j 1))) (#t 1)) 1)
(test (do ((i 0 j) (j 0 (+ j 1))) ((= j 3) i)) 2) ; uh, lessee... lexical scoping...
(test (do ((i 1 j) (j 0 k) (k 0 m) (m 0 (+ i j k))) ((> m 10) (list i j k m))) (list 4 5 8 11))
(test (do ((do 1 (+ do do))) ((> do 3) do)) 4)
(test (do ((do 1 do) (j do do)) (do do)) 1)
(test (do ((do do do)) (do do)) do)
(test (do ((do do do)) (do do do)) do) ; ok ok!
(test (let ((i 10) (j 11) (k 12)) (do ((i i j) (j j k) (k k m) (m (+ i j k) (+ i j k))) ((> m 100) (list i j k m)))) (list 33 56 78 122))
(test (do ((i 0 (let () (set! j 3) (+ i 1))) (j 0 (+ j 1))) ((= i 3) j)) 4)
(test (let ((i 0)) (do () ((= i 3) (* i 2)) (set! i (+ i 1)))) 6)
(num-test (do ((i 0 (- i 1))) ((= i -3) i)) -3)
(num-test (do ((i 1/2 (+ i 1/2))) ((> i 2) i)) 5/2)
(num-test (do ((i 0.0 (+ i 0.1))) ((>= i 0.9999) i)) 1.0)
(num-test (do ((i 0 (- i 1/2))) ((< i -2) i)) -5/2)
(num-test (do ((i 0+i (+ i 0+i))) ((> (magnitude i) 2) i)) 0+3i)

(test (do () (/ 0)) 0)
(test (do () (+)) '())
(test (do () (+ +) *) +)

(if with-bignums
    (begin
      (num-test (do ((i 24444516448431392447461 (+ i 1))
		     (j 0 (+ j 1)))
		    ((>= i 24444516448431392447471) j))
		10)
      (num-test (do ((i 0 (+ i 24444516448431392447461))
		     (j 0 (+ j 1)))
		    ((>= i 244445164484313924474610) j))
		10)
      (num-test (do ((i 4096 (* i 2))
		     (j 0 (+ j 1)))
		    ((= i 4722366482869645213696) j))
		60)))

(num-test (do ((x (list 1 2 3) (cdr x)) (j -1)) ((null? x) j) (set! j (car x))) 3)

(test (let ((x 0)) 
	(do ((i 0 (+ i 1)))
	    ((= i (do ((j 0 (+ j 1))) ((= j 2) (+ j 1)))))
	  (set! x (+ x i)))
	x)
      3)
(test (let ((x 0)) 
	(do ((i 0 (+ i (do ((j 0 (+ j 1))) ((= j 2) 1)))))
	    ((= i 3) x)
	  (set! x (+ x i))))
      3)
(test (let ((x 0)) 
	(do ((i 0 (+ i (do ((j 0 (+ j 1))) ((= j 2) 1)))))
	    ((= i 3) (do ((j 0 (+ j 1))) ((= j 5) x) (set! x j)))
	  (set! x (+ x i))))
      4)

(test (call/cc (lambda (exit) (do ((i 0 (+ i 1))) ((= i 100) i) (if (= i 2) (exit 321))))) 321)
(test (call/cc (lambda (exit) (do ((i 0 (if (= i 3) (exit 321) (+ i 1)))) ((= i 100) i)))) 321)
(test (call/cc (lambda (exit) (do ((i 0 (+ i 1))) ((= i 10) (exit 321))))) 321)
(test (call/cc (lambda (exit) (do ((i 0 (+ i 1))) ((= i 10) i) (if (= i -2) (exit 321))))) 10)
(test (do ((x 0 (+ x 1)) (y 0 (call/cc (lambda (c) c)))) ((> x 5) x) #f) 6)
(test (let ((happy #f)) (do ((i 0 (+ i 1))) (happy happy) (if (> i 3) (set! happy i)))) 4)

(test (+ (do ((i 0 (+ i 1))) ((= i 3) i)) (do ((j 0 (+ j 1))) ((= j 4) j))) 7)
(test (let ((do 1) (map 2) (for-each 3) (quote 4)) (+ do map for-each quote)) 10)

(test (let ((cont #f)
	    (j 0)
	    (k 0))
	(call/cc (lambda (exit) 
		   (do ((i 0 (+ i 1))) 
		       ((= i 100) i) 
		     (set! j i)
		     (call/cc (lambda (r) (set! cont r)))
		     (if (= j 2) (exit))
		     (set! k i))))
	(if (= j 2)
	    (begin
	      (set! j 3)
	      (cont))
	    (list j k)))
      (list 99 99))

(test (call/cc (lambda (r) (do () (#f) (r 1)))) 1)
(test (let ((hi (lambda (x) (+ x 1)))) (do ((i 0 (hi i))) ((= i 3) i))) 3)
(test (do ((i 0 (+ i 1))) (list 1) ((= i 3) #t)) 1) ; a typo originally -- Guile and Gauche are happy with it
(test (do () (1 2) 3) 2)

;; from sacla tests
(test (let ((rev (lambda (list)
		   (do ((x list (cdr x))
			(reverse '() (cons (car x) reverse)))
		       ((null? x) reverse)))))
	(and (null? (rev '()))
	     (equal? (rev '(0 1 2 3 4)) '(4 3 2 1 0))))
      #t)

(test (let ((nrev (lambda (list)
		    (do ((f1st (if (null? list) '() (cdr list)) (if (null? f1st) '() (cdr f1st)))
			 (s2nd list f1st)
			 (t3rd '() s2nd))
			((null? s2nd) t3rd)
		      (set-cdr! s2nd t3rd)))))
	(and (null? (nrev '()))
	     (equal? (nrev (list 0 1 2 3 4)) '(4 3 2 1 0))))
      #t)

(test (do ((temp-one 1 (+ temp-one 1))
	     (temp-two 0 (- temp-two 1)))
	    ((> (- temp-one temp-two) 5) temp-one))
	4)

(test (do ((temp-one 1 (+ temp-one 1))
	     (temp-two 0 (+ temp-one 1)))     
	 ((= 3 temp-two) temp-one))
	3)

(let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
  (test (do ((i 0 (+ 1 i))
	       (n #f)
	       (j 9 (- j 1)))
	      ((>= i j) vec)
	    (set! n (vector-ref vec i))
	    (vector-set! vec i (vector-ref vec j))
	    (vector-set! vec j n))
	  '#(9 8 7 6 5 4 3 2 1 0)))




;;; -------- set! --------

(test (let ((a 1)) (set! a 2) a) 2)
(for-each
 (lambda (arg)
   (test (let ((a 0)) (set! a arg) a) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))

(test (let ((a 1)) (call/cc (lambda (r) (set! a (let () (if (= a 1) (r 123)) 321)))) a) 1)
(test (let ((a (lambda (b) (+ b 1)))) (set! a (lambda (b) (+ b 2))) (a 3)) 5)
(test (let ((a (lambda (x) (set! x 3) x))) (a 1)) 3)



;;; -------- or --------

(test (or (= 2 2) (> 2 1)) #t)
(test (or (= 2 2) (< 2 1)) #t)
(test (or #f #f #f) #f)
(test (or) #f)
(test (or (memq 'b '(a b c)) (+ 3 0)) '(b c))
(test (or 3 9) 3)
(test (or #f 3 asdf) 3) ; "evaluation stops immediately"
(test (or 3 (/ 1 0) (display "or is about to exit!") (exit)) 3)

(for-each
 (lambda (arg)
   (test (or arg) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call-with-input-file "s7test.scm"
	(lambda (p)
	  (let ((loc 0))
	    (let loop ((val (read p)))
	      (or (eof-object? val)
		  (> loc 10000) ; try to avoid the read-error stuff
		  (begin
		    (set! loc (+ 1 loc))
		    (loop (read p)))))
	    (> loc 10000))))
      #t)

(test (or (and (or (> 3 2) (> 3 4)) (> 2 3)) 4) 4)
(test (or or) or)
(test (or (or (or))) #f)
(test (or (or (or) (and))) #t)
(test (let ((a 1)) (or (let () (set! a 2) #f) (= a 1) (let () (set! a 3) #f) (and (= a 3) a) (let () (set! a 4) #f) a)) 3)
(test (or (let ((or #t)) or)) #t)
(test (or '#f '()) '())
(test (call/cc (lambda (r) (or #f (> 3 2) (r 123) 321))) #t)
(test (call/cc (lambda (r) (or #f (< 3 2) (r 123) 321))) 123)
(test (+ (or #f (not (null? '())) 3) (or (zero? 1) 2)) 5)
(test (or 0) 0)
(test (if (or) 1 2) 2)



;;; -------- and --------

(test (and (= 2 2) (> 2 1)) #t)
(test (and (= 2 2) (< 2 1)) #f)
(test (and 1 2 'c '(f g)) '(f g))
(test (and) #t)
(test (and 3) 3)
(test (and (memq 'b '(a b c)) (+ 3 0)) 3)
(test (and 3 9) 9)
(test (and #f 3 asdf) #f) ; "evaluation stops immediately"
(test (and 3 (zero? 1) (/ 1 0) (display "and is about to exit!") (exit)) #f)
(test (if (and) 1 2) 1)
(test (if (+) 1 2) 1)
(test (if (*) 1 2) 1)

(for-each
 (lambda (arg)
   (test (and arg) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call-with-input-file "s7test.scm"
	(lambda (p)
	  (let ((loc 0))
	    (let loop ((val (read p)))
	      (and (not (eof-object? val))
		   (< loc 10000)
		   (begin
		     (set! loc (+ 1 loc))
		     (loop (read p)))))
	    (>= loc 10000))))
      #t)

(test (and (or (and (> 3 2) (> 3 4)) (> 2 3)) 4) #f)
(test (and and) and)
(test (and (and (and))) #t)
(test (and (and (and (and (or))))) #f)
(test (let ((a 1)) (and (let () (set! a 2) #t) (= a 1) (let () (set! a 3) #f) (and (= a 3) a) (let () (set! a 4) #f) a)) #f)
(test (and (let ((and #t)) and)) #t)
(test (and '#t '()) '())
(test (call/cc (lambda (r) (and #t (> 3 2) (r 123) 321))) 123)
(test (call/cc (lambda (r) (and #t (< 3 2) (r 123) 321))) #f)
(test (+ (and (null? '()) 3) (and (zero? 0) 2)) 5)




;;; -------- cond --------

(test (cond ('a)) 'a)
(test (cond (3)) 3)
(test (cond (#f 'a) ('b)) 'b)
(test (cond (#t 'a) (#t 'b)) 'a)
(test (cond ((> 3 2) 'greater) ((< 3 2) 'less)) 'greater)
(test (cond ((> 3 3) 'greater) ((< 3 3) 'less)  (else 'equal)) 'equal)
(test (cond ((assv 'b '((a 1) (b 2))) => cadr)  (else #f)) 2)
(test (cond (#f 2) (else 5)) 5)
(test (cond (1 2) (else 5)) 2)
(test (cond (1 => (lambda (x) (+ x 2))) (else 8)) 3)
(test (cond ((+ 1 2))) 3)
(test (cond ((zero? 1) 123) ((= 1 1) 321)) 321)
(test (cond ('() 1)) 1)
;(test (cond (1 2) '()) 2)
(test (cond (1 2 3)) 3)
(test (cond (1 2) (3 4)) 2)
(test (cond ((= 1 2) 3) ((+ 3 4))) 7)
(test (cond ((= 1 1) (abs -1) (+ 2 3) (* 10 2)) (else 123)) 20)
(test (let ((a 1)) (cond ((= a 1) (set! a 2) (+ a 3)))) 5)
(test (let ((a 1)) (cond ((= a 2) (+ a 2)) (else (set! a 3) (+ a 3)))) 6)
(test (cond ((= 1 1))) #t)
(test (cond ((= 1 2) #f) (#t)) #t)
(test (cond ((+ 1 2))) 3)
(test (cond ((cons 1 2))) '(1 . 2))
(test (cond (#f #t) ((string-append "hi" "ho"))) "hiho")
(test (cond ('() 3) (#t 4)) 3)
(test (cond ((list) 3) (#t 4)) 3)
;;; (cond (1 1) (asdf 3)) -- should this be an error?
(test (cond (+ 0)) 0)

(for-each
 (lambda (arg)
   (test (cond ((or arg) => (lambda (x) x))) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))


(test (cond ((+ 1 2) => (lambda (x) (+ 1 x)))) 4)
(test (cond ((cons 1 2) => car)) 1)
; (cond ((values 1 2) => +)) -- seems like it ought to work
(test (cond ((values -1) => abs)) 1)

(test (cond (else 1)) 1)
(test (call/cc (lambda (r) (cond ((r 4) 3) (else 1)))) 4)
(test (cond ((cond (#t 1)))) 1)

(for-each
 (lambda (arg)
   (test (cond (#t arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (cond (arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (cond (#f 1) (else arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (cond (arg => (lambda (x) x))) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (let ((=> 3) (cond 4)) (+ => cond)) 7)
(test (cond (cond 'cond)) 'cond)

(test (and (defined? 'else) (boolean? else)) #f)



;;; -------- case --------

(test (case (* 2 3) ((2 3 5 7) 'prime) ((1 4 6 8 9) 'composite))  'composite)
(test (case (car '(c d)) ((a e i o u) 'vowel) ((w y) 'semivowel) (else 'consonant)) 'consonant)
(test (case 3.1 ((1.3 2.4) 1) ((4.1 3.1 5.4) 2) (else 3)) 2)
(test (case 3/2 ((3/4 1/2) 1) ((3/2) 2) (else 3)) 2)
(test (case 3 ((1) 1 2 3) ((2) 2 3 4) ((3) 3 4 5)) 5)
(test (case 1+i ((1) 1) ((1/2) 1/2) ((1.0) 1.0) ((1+i) 1+i)) 1+i)
(test (case 'abs ((car cdr) 1) ((+ cond) 2) ((abs) 3) (else 4)) 3)
(test (case #\a ((#\b) 1) ((#\a) 2) ((#\c) 3)) 2)
(test (case (boolean? 1) ((#t) 2) ((#f) 1) (else 0)) 1)
(test (case 1 ((1 2 3) (case 2 ((1 2 3) 3)))) 3)
(test (case 1 ((1 2) 1) ((3.14 2/3) 2)) 1)
(test (case 1 ((1 2) 1) ((#\a) 2)) 1)
(test (case 1 ((1 2) 1) ((#\a) 2) ((car cdr) 3) ((#f #t) 4)) 1)
(test (case #f ((1 2) 1) ((#\a) 2) ((car cdr) 3) ((#f #t) 4)) 4)
(test (case 1 ((#t) 2) ((#f) 1) (else 0)) 0)
(test (let ((x 1)) (case x ((x) "hi") (else "ho"))) "ho")
(test (let ((x 1)) (case x ((1) "hi") (else "ho"))) "hi")
(test (let ((x 1)) (case x (('x) "hi") (else "ho"))) "ho")
(test (let ((x 1)) (case 'x ((x) "hi") (else "ho"))) "hi")

(test (let ((x 1)) (case (+ 1 x) ((0 "hi" #f) 3/4) ((#\a 1+3i '(1 . 2)) "3") ((-1 'hi 2 2.0) #\f))) #\f)
(test (case (case 1 ((0 2) 3) (else 2)) ((0 1) 2) ((4 2) 3) (else 45)) 3)
(test (case 3/4 ((0 1.0 5/6) 1) (("hi" 'hi 3/4) 2) (else 3)) 2)
(test (case (case (+ 1 2) (else 3)) ((3) (case (+ 2 2) ((2 3) 32) ((4) 33) ((5) 0)))) 33)
(test (let ((x 1)) (case x ((0) (set! x 12)) ((2) (set! x 32))) x) 1)

(test (case 1 (else #f)) #f)
(test (case 1 ((1 2) (let ((case 3)) (+ case 1))) ((3 4) 0)) 4)

(for-each
 (lambda (arg)
   (test (case 1 ((0) 'gad) ((1 2 3) arg) (else 'gad)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (case arg ((0) 'gad) ((1 2 3) arg) (else 'gad)) 'gad))
 (list "hi" -1 #\a 0 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call/cc (lambda (r) (case 1 ((1) (r 123) #t) (else #f)))) 123)
(test (call/cc (lambda (r) (case 1 ((0) 0) (else (r 123) #f)))) 123)

(test (case '() ((1) 1) ('() 2)) 2)
(test (case (list) ((1) 1) ('() 2)) 2)
(test (case '() ((1) 1) ((()) 2)) 2)
(test (case (list) ((1) 1) ((()) 2)) 2)



;;; -------- lambda --------

(test (procedure? (lambda (x) x)) #t)
(test ((lambda (x) (+ x x)) 4) 8)
(test (let ((reverse-subtract (lambda (x y) (- y x)))) (reverse-subtract 7 10)) 3)
(test (let ((add4 (let ((x 4)) (lambda (y) (+ x y))))) (add4 6)) 10)
(test ((lambda x x) 3 4 5 6) (list 3 4 5 6))
(test ((lambda (x y . z) z) 3 4 5 6) (list 5 6))
(test ((lambda (a b c d e f) (+ a b c d e f)) 1 2 3 4 5 6) 21)
(test (let ((foo (lambda () 9))) (+ (foo) 1)) 10)
(test (let ((a 1)) (let ((f (lambda (x) (set! a x) a))) (let ((c (f 123))) (list c a)))) (list 123 123))
(test (let ((a 1) (b (lambda (a) a))) (b 3)) 3)
(test (let ((ctr 0)) (letrec ((f (lambda (x) (if (> x 0) (begin (set! ctr (+ ctr 1)) (f (- x 1))) 0)))) (f 10) ctr)) 10)
(test (let ((f (lambda (x) (car x)))) (f '(4 5 6))) 4)
(test ((lambda () ((lambda (x y) ((lambda (z) (* (car z) (cdr z))) (cons x y))) 3 4))) 12)
(test (let ((ctr 0)) (define (f) (set! ctr (+ ctr 1)) ctr) (let ((x (f))) (let ((y (f))) (list x y ctr)))) (list 1 2 2))

(test (let ((x 5)) (define foo (lambda (y) (bar x y))) (define bar (lambda (a b) (+ (* a b) a))) (foo (+ x 3))) 45)
(test (let ((x 5)) (letrec ((foo (lambda (y) (bar x y))) (bar (lambda (a b) (+ (* a b) a)))) (foo (+ x 3)))) 45)
(num-test (let () (define compose (lambda (f g) (lambda args (f (apply g args))))) ((compose sqrt *) 12 75))  30.0)
(test (let ((f (lambda () (lambda (x y) (+ x y))))) ((f) 1 2)) 3)
(test ((lambda (x) (define y 4) (+ x y)) 1) 5)
(test ((lambda () (define (y x) (+ x 1)) (y 1))) 2)
(test ((lambda (x) 123 (let ((a (+ x 1))) a)) 2) 3)
(test ((lambda (x) "documentation" (let ((a (+ x 1))) a)) 2) 3)
(test ((lambda (x) (x 1)) (lambda (y) (+ y 1))) 2)
(test (let ((a 1)) (let ((b (lambda (x) (define y 1) (define z 2) (define a 3) (+ x y z a)))) (b a))) 7)
(test ((lambda (f x) (f x x)) + 11) 22)
(test ((lambda () (+ 2 3))) 5)

(test (letrec ((f (lambda (x) (g x)))
	       (g (lambda (x) x)))
	(let ((top (f 1)))
	  (set! g (lambda (x) (- x)))
	  (+ top (f 1))))
      0)

(for-each
 (lambda (arg)
   (test ((lambda (x) x) arg) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((list-length
       (lambda (obj)
	 (call-with-current-continuation
	  (lambda (return)
	    (letrec ((r (lambda (obj) (cond ((null? obj) 0)
					    ((pair? obj) (+ (r (cdr obj)) 1))
					    (else (return #f))))))
	      (r obj)))))))
  (test (list-length '(1 2 3 4)) 4)
  (test (list-length '(a b . c)) #f))

(test (let ((samples (vector 0 1 2 3 4 5 6 7 8 9 10)))
	(let ((make-scaler 
	       (lambda (start end)
		 (letrec ((ctr start)
			  (us (lambda (them)
				(vector-set! samples ctr (* 2 (vector-ref samples ctr)))
				(set! ctr (+ ctr 2))
				(if (<= ctr end)
				    (them us)))))
		   us))))
	  ((make-scaler 0 11)
	   (make-scaler 1 11))) 
	samples)
      (vector 0 2 4 6 8 10 12 14 16 18 20))

(test ((lambda (x . y) y) 1 2 '(3 . 4)) '(2 (3 . 4)))
(test ((lambda (x . y) y) 1) '())
(test ((lambda x x) '()) '(()))
(test ((lambda x x)) '())        ; ??
(test ((lambda (x) x) '()) '())
(test (let ((lambda 4)) (+ lambda 1)) 5)
(test ((lambda (x) (+ x ((lambda (x) (+ x 1)) 2))) 3) 6)
(test ((lambda (x) (define y 1) (+ x y)) 2) 3)
(test ((lambda (a) "this is a doc string" a) 1) 1)
;;; ideally ((lambda (a) "hiho" (define x 1) x) 1) -> 1 but I'm not sure it's r5rs-ish
(test (let ((g (lambda () '3))) (= (g) 3)) #t)

(test ((lambda lambda lambda) 'x) '(x))
;(test ((lambda (begin) (begin 1 2 3)) (lambda lambda lambda)) '(1 2 3))

(test (let ((funcs (make-vector 3 #f)))
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (vector-set! funcs i (lambda () (+ i 1))))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((i 1))
	(let ((func1 (lambda () i)))
	  (let ((i 2))
	    (let ((func2 (lambda () i)))
	      (+ (func1) (func2))))))
      3)

(test (let ((funcs (make-vector 3 #f)))
	(map
	 (lambda (i)
	   (vector-set! funcs i (lambda () (+ i 1))))
	 (list 0 1 2))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((func #f))
	(define (func1 x)
	  (set! func (lambda () (+ x 1))))
	(func1 1)
	(+ (func)
	   (let ()
	     (func1 2)
	     (func))))
      5)

(test (((lambda (x) (lambda () (+ x 1))) 32)) 33)

(test (let ((func #f))
	(define (func1 x)
	  (set! func (lambda () (string-append x "-"))))
	(func1 "hi")
	(string-append (func)
		       (let ()
			 (func1 "ho")
			 (func))))
      "hi-ho-")

(test (let ((func1 #f)
	    (func2 #f))
	(let ((x 1))
	  (set! func1 (lambda () x))
	  (set! func2 (lambda (y) (set! x y) y)))
	(+ (func1)
	   (let ()
	     (func2 32)
	     (func1))))
      33)

(test (let ((funcs (make-vector 3)))
	(let ((hi (lambda (a) (vector-set! funcs (- a 1) (lambda () a)))))
	  (hi 1) (hi 2) (hi 3)
	  (+ ((vector-ref funcs 0))
	     ((vector-ref funcs 1))
	     ((vector-ref funcs 2)))))
      6)

(test (let ((hi (lambda (a) (+ a 1)))
	    (ho (lambda (a) (a 32))))
	(+ (hi (hi (hi 1)))
	   (ho hi)))
      37)

(test ((if (> 3 2) + -) 3 2) 5)
(test (let ((op +)) (op 3 2)) 5)
(test (((lambda () +)) 3 2) 5)
(test ((car (cons + -)) 3 2) 5)
(test ((do ((i 0 (+ i 1))) ((= i 3) +) ) 3 2) 5)
(test (((lambda (x) x) (lambda (x) x)) 3) 3)
(test ((((lambda (x) x) (lambda (x) x)) (lambda (x) x)) 3) 3)
(test (((lambda (x) (lambda (y) x)) 3) 4) 3)
(test (((lambda (x) (lambda (x) x)) 3) 4) 4)
(test (let ((x 32)) (((lambda (x) (lambda (y) x)) 3) x)) 3)
(test ((call/cc (lambda (return) (return +))) 3 2) 5)
(test ((call-with-values (lambda () (values +)) (lambda (x) x)) 3 2) 5)
(test ((case '+ ((+) +)) 3 2) 5)
(test ((case '+ ((-) -) (else +)) 3 2) 5)
(test ((call/cc (lambda (return) (dynamic-wind (lambda () #f) (lambda () (return +)) (lambda () #f)))) 3 2) 5)
(test (+ 1 ((call/cc (lambda (return) (dynamic-wind (lambda () #f) (lambda () (return +)) (lambda () #f)))) 3 2) 2) 8)


;;; -------- begin --------

(test (let () (begin) #f) #f)
(test (let () (begin (begin (begin (begin)))) #f) #f)
(test (let () (begin (define x 2) (define y 1)) (+ x y)) 3)
(test (let () (begin (define x 0)) (begin (set! x 5) (+ x 1)))  6)
(test (let () (begin (define first car)) (first '(1 2))) 1)
(test (let () (begin (define x 3)) (begin (set! x 4) (+ x x))) 8)

(if (equal? (begin 1) 1)
    (begin
      (test (let () (begin (define x 0)) (set! x (begin (begin 5))) (begin ((begin +) (begin x) (begin (begin 1))))) 6)      

      (test (let ((x 5))
	      (begin (begin (begin)
			    (begin (begin (begin) (define foo (lambda (y) (bar x y)))
					  (begin)))
			    (begin))
		     (begin)
		     (begin)
		     (begin (define bar (lambda (a b) (+ (* a b) a))))
		     (begin))
	      (begin)
	      (begin (foo (+ x 3))))
	    45)

      (for-each
       (lambda (arg)
	 (test (begin arg) arg))
       (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

      (test (if (= 1 1) (begin 2) (begin 3)) 2)
      ))

(test (let ((begin 3)) (+ begin 1)) 4)
(test ((lambda (x) (begin (set! x 1) (let ((a x)) (+ a 1)))) 2) 2)
;;; apparently these can be considered errors or not (guile says error, stklos and gauche do not)
;(test (begin (define x 0) (+ x 1)) 1)
;(test ((lambda () (begin (define x 0) (+ x 1)))) 1)
;(test (let ((f (lambda () (begin (define x 0) (+ x 1))))) (f)) 1)

(test ((lambda () (begin (define x 0)) (+ x 1))) 1)
(test (let ((f (lambda () (begin (define x 0)) (+ x 1)))) (f)) 1)
(test (let ((x 32)) (begin (define x 3)) x) 3)
(test ((lambda (x) (begin (define x 3)) x) 32) 3)
(test (let* ((x 32) (y x)) (define x 3) y) 32)

(test (let ((z 0)) (begin (define x 32)) (begin (define y x)) (set! z y) z) 32) ; so begin is like let*? -- guile uses letrec here = error
(test (let ((z 0)) (begin (define x 32) (define y x)) (set! z y) z) 32)         ; similarly here
;;; I can't find anything in r5rs.html that mandates letrec here, or that says it's in error

; (test (begin (define b 1) (begin (define a b) (define b 3)) a) 1) ; ?? if a 1st => "b unbound", but guile returns #f
; (begin (begin (define a 1) (begin (define a b) (define b 3))) a) ; guile says 3, s7 says error
;;; begin is a mess...



;;; -------- apply --------

(test (apply (lambda (a b) (+ a b)) (list 3 4)) 7)
(test (apply + 10 (list 3 4)) 17)
(test (apply list '()) '())
(test (apply + '(1 2)) 3)
(test (apply - '(1 2)) -1)
(test (apply max 3 5 '(2 7 3)) 7)
(test (apply cons '((+ 2 3) 4)) '((+ 2 3) . 4))
(test (apply + '()) 0)
(test (apply + (list 3 4)) 7)
(test (apply + '()) 0)
(test (apply + 2 '(3)) 5)
(test (apply + 2 3 '()) 5)
(test (apply + '(2 3)) 5)
(test (apply list 1 '(2 3)) (list 1 2 3))
(test (apply apply (list list 1 2 '(3))) (list 1 2 3))
(test (vector? (apply make-vector '(1))) #t)
(test (apply make-vector '(1 1)) '#(1))
;(test (let* ((x '(1 2 3)) (y (apply list x))) (not (eq? x y))) #t) ; is this standard?
(test (apply min '(1 2 3 5 4 0 9)) 0)
(test (apply min 1 2 4 3 '(4 0 9)) 0)
(test (apply vector 1 2 '(3)) '#(1 2 3))
(test (apply (lambda (x . y) x) (list 1 2 3)) 1)
(test (apply * (list 2 (apply + 1 2 '(3)))) 12)
(test (apply (if (> 3 2) + -) '(3 2)) 5)

(for-each
 (lambda (arg)
   (test (apply (lambda (x) x) (list arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (apply cadadr (list '''4)) 4)
(test (apply string-ref "hi" '(0)) #\h)
(test (let ((x (string-copy "hi"))) (apply string-set! x 0 '(#\c)) x) "ci")
(test (apply apply (list + '(3  2))) 5)
(test (apply apply apply apply (list (list (list + '(3  2))))) 5)
(test (apply + 1 2 (list 3 4)) 10)



;;; -------- define --------
;;;
;;; trying to avoid top-level definitions here

(let ()
  (define x 2)
  (test (+ x 1) 3)
  (set! x 4)
  (test (+ x 1) 5)
  (let ()
    (define (tprint x) #t)
    (test (tprint 56) #t)
    (let ()
      (define first car)
      (test (first '(1 2)) 1)
      (let ()
	(define foo (lambda () (define x 5) x))
	(test (foo) 5)
	(let ()
	  (define (foo x) ((lambda () (define x 5) x)) x)
	  (test (foo 88) 88))))))


(test (letrec ((foo (lambda (arg) (or arg (and (procedure? foo) (foo 99)))))) (define bar (foo #f)) (foo #f)) 99)
(test (letrec ((foo 77) (bar #f) (retfoo (lambda () foo))) (define baz (retfoo)) (retfoo)) 77)

(test (let () (define .. 1) ..) 1)

(test (let () (define (hi a) (+ a 1)) (hi 2)) 3)
(test (let () (define (hi a . b) (+ a (cadr b) 1)) (hi 2 3 4)) 7)
(test (let () (define (hi) 1) (hi)) 1)
(test (let () (define (hi . a) (apply + a)) (hi 1 2 3)) 6)

(for-each
 (lambda (arg)
   (test (let () (define x arg) x) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test ((lambda (x) (define (hi a) (+ a 1)) (hi x)) 1) 2)
(test (let ((x 2)) (define f (lambda (y) (+ y x))) (f 3)) 5)
(begin (define r5rstest-plus (lambda (x y) (+ x y))) (define r5rstest-x 32))
(test (r5rstest-plus r5rstest-x 3) 35)

(test (let () (define (asdf a) (define (asdf a) (+ a 1)) (+ a (asdf a))) (asdf 4)) 9)
(test (let ((asdf 1)) (define (asdf a) (define (asdf a) (+ a 1)) (+ a (asdf a))) (asdf 4)) 9)
(test (let () (define (a1 a) (define (a2 a) (define (a3 a) (define (a4 a) (+ a 1)) (+ (a4 a) 1)) (+ (a3 a) 1)) (+ (a2 a) 1)) (a1 0)) 4)





;;; -------- values, call-with-values --------

(test (call-with-values (lambda () (values 1 2 3)) +) 6)
(test (call-with-values (lambda () (values 4 5)) (lambda (a b) b))  5)
(test (call-with-values (lambda () (values 4 5)) (lambda (a b) (+ a b))) 9)
(test (call-with-values * -) -1) ; right...
(test (values 1) 1)
(test (call-with-values (lambda () (values 1 2 3 4)) list) (list 1 2 3 4))
(test (+ (values 1) (values 2)) 3)
(test (+ (values '1) (values '2)) 3)
(test (if (values #t) 1 2) 1)
(test (if (values '#t) 1 2) 1)
(test (call-with-values (lambda () 4) (lambda (x) x)) 4)
(test (let () (values 1 2 3) 4) 4)
(test (apply + (values '())) 0)
(test (+ (values 1 2 3)) 6)
(test (let ((f (lambda () (values 1 2 3)))) (+ (f))) 6)
(num-test (log (values 8 2)) 3)
(test (* (values 2 (values 3 4))) 24)
(test (* (values (+ (values 1 2)) (- (values 3 4)))) -3)
(test (list (values 1 2) (values 3) 4) '(1 2 3 4))
(test (let ((f1 (lambda (x) (values x (+ x 1)))) (f2 (lambda () (values 2)))) (+ (f1 3) (* 2 (f2)))) 11)
(test (+ (let () (values 1 2)) 3) 6)
(test (let () (values 1 2) 4) 4)
(test (let () + (values 1 2) 4) 4)
(test (string-ref (values "hiho" 2)) #\h)
(test (vector-ref (values (vector 1 2 3)) 1) 2)
(test (+ (values (+ 1 (values 2 3)) 4) 5 (values 6) (values 7 8 (+ (values 9 10) 11))) 66)
(test (+ (if (values) (values 1 2) (values 3 4)) (if (null? (values)) (values 5 6) (values 7 8))) 14)
(test (+ (cond (#f (values 1 2)) (#t (values 3 4))) 5) 12)
(test (apply + (list (values 1 2))) 3)
(test (apply + (list ((lambda (n) (values n (+ n 1))) 1))) 3)
(test (+ (do ((i 0 (+ i 1))) ((= i 3) (values i (+ i 1))))) 7)
(if with-open-input-string-and-friends
    (test (+ (with-input-from-string "(values 1 2 3)" (lambda () (read))) 2) 8))
(test (< (values 1 2 3)) #t)

(test (let ((sum 0)) (for-each (lambda (n m p) (set! sum (+ sum n m p))) (values (list 1 2 3) (list 4 5 6) (list 7 8 9)))) 45)
(test (map (lambda (n m p) (+ n m p)) (values (list 1 2 3) (list 4 5 6) (list 7 8 9))) '(12 15 18))
(test (string-append (values "123" "4" "5") "6" (values "78" "90")) "1234567890")
(test (+ (dynamic-wind (lambda () #f) (lambda () (values 1 2 3)) (lambda () #f)) 4) 10)

(for-each
 (lambda (arg)
   (test (values arg) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (call-with-values (lambda () (values arg arg)) (lambda (a b) b)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call-with-values (lambda () (values "hi" 1 3/2 'a)) (lambda (a b c d) (+ b c))) 5/2)
;(test (call-with-values values (lambda arg arg)) '())
(test (string-ref (values "hi") 1) #\i)
(test ((lambda (a b) (+ a b)) ((lambda () (values 1 2)))) 3)

(test (list (letrec ((split (lambda (ls)
			      (if (or (null? ls) (null? (cdr ls)))
				  (values ls '())
				  (call-with-values
				      (lambda () (split (cddr ls)))
				    (lambda (odds evens)
				      (values (cons (car ls) odds)
					      (cons (cadr ls) evens))))))))
	      (split '(a b c d e f))))
      '((a c e) (b d f)))

(test (call-with-values (lambda () (call/cc (lambda (k) (k 2 3)))) (lambda (x y) (list x y))) '(2 3))
(test (+ (call/cc (lambda (return) (return (values 1 2 3)))) 4) 10)

(test (let ((values 3)) (+ 2 values)) 5)
(test (let ((a (values 1))) a) 1)

(test (call-with-values (lambda () 2) (lambda (x) x)) 2)
(test (call-with-values (lambda () -1) abs) 1)
(test (call-with-values (lambda () (values -1)) abs) 1)
(test (call-with-values (lambda () (values -1)) (lambda (a) (abs a))) 1)

(test (call-with-values 
	  (lambda ()
	    (values
	     (call-with-values (lambda () (values 1 2 3)) +)
	     (call-with-values (lambda () (values 1 2 3 4)) *)))
	(lambda (a b)
	  (- a b)))
      -18)

(test (call-with-values 
	  (lambda ()
	    (values
	     (call-with-values (lambda () (values 1 2 3)) +)
	     (call-with-values (lambda () (values 1 2 3 4)) *)))
	(lambda (a b)
	  (+ (* a (call-with-values (lambda () (values 1 2 3)) +))
	     (* b (call-with-values (lambda () (values 1 2 3 4)) *)))))
      612)




;;; -------- let, let*, letrec --------

(test (let ((x 2) (y 3)) (* x y)) 6)
(test (let ((x 32)) (let ((x 3) (y x)) y)) 32)
(test (let ((x 32)) (let* ((x 3) (y x)) y)) 3)
(test (let ((x 2) (y 3)) (let ((x 7) (z (+ x y))) (* z x))) 35)
(test (let ((x 2) (y 3)) (let* ((x 7)  (z (+ x y))) (* z x))) 70)
(test (letrec ((even? (lambda (n)  (if (zero? n) #t (odd? (- n 1))))) (odd? (lambda (n)  (if (zero? n) #f (even? (- n 1)))))) (even? 88))  #t)
(test (let loop ((numbers '(3 -2 1 6 -5)) 
		 (nonneg '()) 
		 (neg '())) 
	(cond ((null? numbers) 
	       (list nonneg neg)) 
	      ((>= (car numbers) 0)  
	       (loop (cdr numbers) (cons (car numbers) nonneg)  neg)) 
	      ((< (car numbers) 0)  
	       (loop (cdr numbers) nonneg (cons (car numbers) neg))))) 
      '((6 1 3) (-5 -2)))

(test (let ((x 3)) (define x 5) x) 5)
(test (let* () (define x 8) x) 8)
(test (letrec () (define x 9) x) 9)
(test (letrec ((x 3)) (define x 10) x) 10)
(test (let foo () 1) 1)
(test (let ((f -)) (let f ((n (f 1))) n)) -1)
(test (let () 1 2 3 4) 4)

(test (let ((x 1)) (let ((x 32) (y x)) y)) 1)
(test (let ((x 1)) (letrec ((y (if #f x 1)) (x 32)) 1)) 1)
(test (let ((x 1)) (letrec ((y (lambda () (+ 1 x))) (x 32)) (y))) 33) ; good grief! -- (let ((x 1)) (letrec ((y (* 0 x)) (x 32)) y))
(test (let* ((x 1) (f (letrec ((y (lambda () (+ 1 x))) (x 32)) y))) (f)) 33)
(test (letrec ((x 1) (y (let ((x 2)) x))) (+ x y)) 3)
(test (letrec ((f (lambda () (+ x 3))) (x 2)) (f)) 5)
(test (let* ((x 1) (x 2)) x) 2)
(test (let* ((x 1) (y x)) y) 1)
(test (let ((x 1)) (let ((x 32) (y x)) (+ x y))) 33)
(test (let ((x 1)) (let* ((x 32) (y x)) (+ x y))) 64)
(test (let ((x 'a) (y '(b c))) (cons x y)) '(a b c))
(test (let ((x 0) (y 1)) (let ((x y) (y x)) (list x y))) (list 1 0))
(test (let ((x 0) (y 1)) (let* ((x y) (y x)) (list x y))) (list 1 1))
(test (letrec ((sum (lambda (x) (if (zero? x) 0 (+ x (sum (- x 1))))))) (sum 5)) 15)
(test (let ((divisors (lambda (n) (let f ((i 2)) (cond ((>= i n) '()) ((integer? (/ n i)) (cons i (f (+ i 1)))) (else (f (+ i 1)))))))) (divisors 32)) '(2 4 8 16))
(test (let ((a -1)) (let loop () (if (not (positive? a)) (begin (set! a (+ a 1)) (loop)))) a) 1)
(test (let* ((let 3) (x let)) (+ x let)) 6)
(test (let () (let () (let () '()))) '())
(test (let ((x 1)) (let ((y 0)) (begin (let ((x (* 2 x))) (set! y x))) y)) 2)
(test (let* ((x 1) (x (+ x 1)) (x (+ x 2))) x) 4)
(test (let ((.. 2) (.... 4) (..... +)) (..... .. ....)) 6)

(test (let () (begin (define x 1)) x) 1)
(test (let ((y 1)) (begin (define x 1)) (+ x y)) 2)
(test (let ((: 0)) (- :)) 0)

(test ((let ((x 2))
	 (let ((x 3))
	   (lambda (arg) (+ arg x))))
       1)
      4)

(test ((let ((x 2))
	 (define (inner arg) (+ arg x))
	 (let ((x 32))
	   (lambda (arg) (inner (+ arg x)))))
       1)
      35)

(test ((let ((inner (lambda (arg) (+ arg 1))))
	 (let ((inner (lambda (arg) (inner (+ arg 2)))))
	   inner))
       3)
      6)

(test ((let ()
	 (define (inner arg) (+ arg 1))
	 (let ((inner (lambda (arg) (inner (+ arg 2)))))
	   inner))
       3)
      6)

(test ((let ((x 11))
	 (define (inner arg) (+ arg x))
	 (let ((inner (lambda (arg) (inner (+ (* 2 arg) x)))))
	   inner))
       3)
      28)

(test ((let ((x 11))
	 (define (inner arg) (+ arg x))
	 (let ((x 2))
	   (lambda (arg) (inner (+ (* 2 arg) x)))))
       3)
      19)

(test (let ((f1 (lambda (arg) (+ arg 1))))
	(let ((f1 (lambda (arg) (f1 (+ arg 2)))))
	  (f1 1)))
      4)

(test (let ((f1 (lambda (arg) (+ arg 1))))
	(let* ((f1 (lambda (arg) (f1 (+ arg 2)))))
	  (f1 1)))
      4)

(test (let ((f1 (lambda (arg) (+ arg 1))))
	(let* ((x 32)
	       (f1 (lambda (arg) (f1 (+ x arg)))))
	  (f1 1)))
      34)

(test ((let ((x 11))
	 (define (inner arg) (+ arg x))
	 (let ((x 2)
	       (inner (lambda (arg) (inner (+ (* 2 arg) x)))))
	   inner))
       3)
      28)

(test ((let ((x 11))
	 (define (inner arg) (+ arg x))
	 (let* ((x 2)
		(inner (lambda (arg) (inner (+ (* 2 arg) x)))))
	   inner))
       3)
      19)

(test (let ((x 1))
	(let* ((f1 (lambda (arg) (+ x arg)))
	       (x 32))
	  (f1 1)))
      2)

(test (let ((inner (lambda (arg) (+ arg 1))))
	(let ((inner (lambda (arg) (+ (inner arg) 1))))
	  (inner 1)))
      3)
(test (let ((inner (lambda (arg) (+ arg 1))))
	(let* ((inner (lambda (arg) (+ (inner arg) 1))))
	  (inner 1)))
      3)

(test (let ((caller #f)) (let ((inner (lambda (arg) (+ arg 1)))) (set! caller inner)) (caller 1)) 2)
(test (let ((caller #f)) (let ((x 11)) (define (inner arg) (+ arg x)) (set! caller inner)) (caller 1)) 12)

(test (let ((caller #f)) 
	(let ((x 11)) 
	  (define (inner arg) 
	    (+ arg x)) 
	  (let ((y 12))
	    (let ((inner (lambda (arg) 
			   (+ (inner x) y arg)))) ; 11 + 11 + 12 + arg
	      (set! caller inner))))
	(caller 1))
      35)

(test (let ((caller #f)) 
	(let ((x 11)) 
	  (define (inner arg) 
	    (+ arg x)) 
	  (let* ((y 12) 
		 (inner (lambda (arg) 
			  (+ (inner x) y arg)))) ; 11 + 11 + 12 + arg
	    (set! caller inner))) 
	(caller 1))
      35)


(test (let* ((f1 3) (f1 4)) f1) 4)
(test (let ((f1 (lambda () 4))) (define (f1) 3) (f1)) 3)

(test (let ((j -1)
	    (k 0))
	(do ((i 0 (+ i j))
	     (j 1))
	    ((= i 3) k)
	  (set! k (+ k i))))
      3)

(test (let ((j (lambda () -1))
	    (k 0))
	(do ((i 0 (+ i (j)))
	     (j (lambda () 1)))
	    ((= i 3) k)
	  (set! k (+ k i))))
      3)

(test (let ((j (lambda () 0))
	    (k 0))
	(do ((i (j) (j))
	     (j (lambda () 1) (lambda () (+ i 1))))
	    ((= i 3) k)
	  (set! k (+ k i))))
      6) ; or is it 3?

(test (let ((k 0)) (do ((i 0 (+ i 1)) (j 0 (+ j i))) ((= i 3) k) (set! k (+ k j)))) 1)

(test (let ((j (lambda () 0))
	    (i 2)
	    (k 0))
	(do ((i (j) (j))
	     (j (lambda () i) (lambda () (+ i 1))))
	    ((= i 3) k)
	  (set! k (+ k i))))
      3) ; or 2?

(test (let ((f #f))
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (let ()
	    (define (x) i)
	    (if (= i 1) (set! f x))))
	(f))
      1)

(let ((x 123))
  (define (hi b) (+ b x))
  (let ((x 321))
    (test (hi 1) 124)
    (set! x 322)
    (test (hi 1) 124))
  (set! x 124)
  (test (hi 1) 125)
  (let ((x 321)
	(y (hi 1)))
    (test y 125))
  (let* ((x 321)
	 (y (hi 1)))
    (test y 125))
  (test (hi 1) 125))

(test (let ((x 123)) (begin (define x 0)) x) 0) ; this strikes me as weird, since (let ((x 123) (x 0)) x) is illegal, so...
(test (let ((x 123)) (begin (define (hi a) (+ x a)) (define x 0)) (hi 1)) 1) ; is non-lexical reference?

(for-each
 (lambda (arg)
   (test (let ((x arg)) x) arg))
 (list "hi" -1 #\a "" '() '#() (current-output-port) 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t abs (list 1 2 3) '(1 . 2)))

(test (let ((x 1)) (= 1 (let ((y 2)) (set! x y) x)) (+ x 1)) 3)
(test (let ((x 1)) (let ((xx (lambda (a) (set! x a) a))) (= 1 (xx 2))) (+ x 1)) 3)
(test (let ((x 32)) (begin (define x 123) (define (hi a) (+ a 1))) (hi x)) 124)
(test (let () (begin (define x 123) (define (hi a) (+ a 1))) (hi x)) 124)


;(let ((initial-chars "aA!$%&*/:<=>?^_~")
;      (subsequent-chars "9aA!$%&*+-./:<=>?@^_~")
;      (ctr 0))
;  (display (format #f "(let ("))
;  (do ((i 0 (+ i 1)))
;      ((= i (string-length initial-chars)))
;    (display (format #f "(~A ~D) " (string (string-ref initial-chars i)) ctr))
;    (set! ctr (+ ctr 1)))
;
;  (do ((i 0 (+ i 1)))
;      ((= i (string-length initial-chars)))
;    (do ((k 0 (+ k 1)))
;	((= k (string-length subsequent-chars)))
;      (display (format #f "(~A ~D) " (string (string-ref initial-chars i) (string-ref subsequent-chars k)) ctr))
;      (set! ctr (+ ctr 1))))
;
;  (display (format #f ")~%  (+ "))
;  (do ((i 0 (+ i 1)))
;      ((= i (string-length initial-chars)))
;    (display (format #f "~A " (string (string-ref initial-chars i)))))
;
;  (do ((i 0 (+ i 1)))
;      ((= i (string-length initial-chars)))
;    (do ((k 0 (+ k 1)))
;	((= k (string-length subsequent-chars)))
;      (display (format #f "~A " (string (string-ref initial-chars i) (string-ref subsequent-chars k))))))
;
;  (display (format #f "))~%")))

(num-test (let ((a 0) (A 1) (! 2) ($ 3) (% 4) (& 5) (* 6) (/ 7) (: 8) (< 9) (= 10) (> 11) (? 12) (^ 13) (_ 14) (~ 15) (a9 16) (aa 17) (aA 18) (a! 19) (a$ 20) (a% 21) (a& 22) (a* 23) (a+ 24) (a- 25) (a. 26) (a/ 27) (a: 28) (a< 29) (a= 30) (a> 31) (a? 32) (a@ 33) (a^ 34) (a_ 35) (a~ 36) (A9 37) (Aa 38) (AA 39) (A! 40) (A$ 41) (A% 42) (A& 43) (A* 44) (A+ 45) (A- 46) (A. 47) (A/ 48) (A: 49) (A< 50) (A= 51) (A> 52) (A? 53) (A@ 54) (A^ 55) (A_ 56) (A~ 57) (!9 58) (!a 59) (!A 60) (!! 61) (!$ 62) (!% 63) (!& 64) (!* 65) (!+ 66) (!- 67) (!. 68) (!/ 69) (!: 70) (!< 71) (!= 72) (!> 73) (!? 74) (!@ 75) (!^ 76) (!_ 77) (!~ 78) ($9 79) ($a 80) ($A 81) ($! 82) ($$ 83) ($% 84) ($& 85) ($* 86) ($+ 87) ($- 88) ($. 89) ($/ 90) ($: 91) ($< 92) ($= 93) ($> 94) ($? 95) ($@ 96) ($^ 97) ($_ 98) ($~ 99) (%9 100) (%a 101) (%A 102) (%! 103) (%$ 104) (%% 105) (%& 106) (%* 107) (%+ 108) (%- 109) (%. 110) (%/ 111) (%: 112) (%< 113) (%= 114) (%> 115) (%? 116) (%@ 117) (%^ 118) (%_ 119) (%~ 120) (&9 121) (&a 122) (&A 123) (&! 124) (&$ 125) (&% 126) (&& 127) (&* 128) (&+ 129) (&- 130) (&. 131) (&/ 132) (&: 133) (&< 134) (&= 135) (&> 136) (&? 137) (&@ 138) (&^ 139) (&_ 140) (&~ 141) (*9 142) (*a 143) (*A 144) (*! 145) (*$ 146) (*% 147) (*& 148) (** 149) (*+ 150) (*- 151) (*. 152) (*/ 153) (*: 154) (*< 155) (*= 156) (*> 157) (*? 158) (*@ 159) (*^ 160) (*_ 161) (*~ 162) (/9 163) (/a 164) (/A 165) (/! 166) (/$ 167) (/% 168) (/& 169) (/* 170) (/+ 171) (/- 172) (/. 173) (// 174) (/: 175) (/< 176) (/= 177) (/> 178) (/? 179) (/@ 180) (/^ 181) (/_ 182) (/~ 183) (:9 184) (ca 185) (CA 186) (:! 187) (:$ 188) (:% 189) (:& 190) (:* 191) (:+ 192) (:- 193) (:. 194) (:/ 195) (cc 196) (:< 197) (:= 198) (:> 199) (:? 200) (:@ 201) (:^ 202) (:_ 203) (:~ 204) (<9 205) (<a 206) (<A 207) (<! 208) (<$ 209) (<% 210) (<& 211) (<* 212) (<+ 213) (<- 214) (<. 215) (</ 216) (<: 217) (<< 218) (<= 219) (<> 220) (<? 221) (<@ 222) (<^ 223) (<_ 224) (<~ 225) (=9 226) (=a 227) (=A 228) (=! 229) (=$ 230) (=% 231) (=& 232) (=* 233) (=+ 234) (=- 235) (=. 236) (=/ 237) (=: 238) (=< 239) (== 240) (=> 241) (=? 242) (=@ 243) (=^ 244) (=_ 245) (=~ 246) (>9 247) (>a 248) (>A 249) (>! 250) (>$ 251) (>% 252) (>& 253) (>* 254) (>+ 255) (>- 256) (>. 257) (>/ 258) (>: 259) (>< 260) (>= 261) (>> 262) (>? 263) (>@ 264) (>^ 265) (>_ 266) (>~ 267) (?9 268) (?a 269) (?A 270) (?! 271) (?$ 272) (?% 273) (?& 274) (?* 275) (?+ 276) (?- 277) (?. 278) (?/ 279) (?: 280) (?< 281) (?= 282) (?> 283) (?? 284) (?@ 285) (?^ 286) (?_ 287) (?~ 288) (^9 289) (^a 290) (^A 291) (^! 292) (^$ 293) (^% 294) (^& 295) (^* 296) (^+ 297) (^- 298) (^. 299) (^/ 300) (^: 301) (^< 302) (^= 303) (^> 304) (^? 305) (^@ 306) (^^ 307) (^_ 308) (^~ 309) (_9 310) (_a 311) (_A 312) (_! 313) (_$ 314) (_% 315) (_& 316) (_* 317) (_+ 318) (_- 319) (_. 320) (_/ 321) (_: 322) (_< 323) (_= 324) (_> 325) (_? 326) (_@ 327) (_^ 328) (__ 329) (_~ 330) (~9 331) (~a 332) (~A 333) (~! 334) (~$ 335) (~% 336) (~& 337) (~* 338) (~+ 339) (~- 340) (~. 341) (~/ 342) (~: 343) (~< 344) (~= 345) (~> 346) (~? 347) (~@ 348) (~^ 349) (~_ 350) (~~ 351) )
  (+ a A ! $ % & * / : < = > ? ^ _ ~ a9 aa aA a! a$ a% a& a* a+ a- a. a/ a: a< a= a> a? a@ a^ a_ a~ A9 Aa AA A! A$ A% A& A* A+ A- A. A/ A: A< A= A> A? A@ A^ A_ A~ !9 !a !A !! !$ !% !& !* !+ !- !. !/ !: !< != !> !? !@ !^ !_ !~ $9 $a $A $! $$ $% $& $* $+ $- $. $/ $: $< $= $> $? $@ $^ $_ $~ %9 %a %A %! %$ %% %& %* %+ %- %. %/ %: %< %= %> %? %@ %^ %_ %~ &9 &a &A &! &$ &% && &* &+ &- &. &/ &: &< &= &> &? &@ &^ &_ &~ *9 *a *A *! *$ *% *& ** *+ *- *. */ *: *< *= *> *? *@ *^ *_ *~ /9 /a /A /! /$ /% /& /* /+ /- /. // /: /< /= /> /? /@ /^ /_ /~ :9 ca CA :! :$ :% :& :* :+ :- :. :/ cc :< := :> :? :@ :^ :_ :~ <9 <a <A <! <$ <% <& <* <+ <- <. </ <: << <= <> <? <@ <^ <_ <~ =9 =a =A =! =$ =% =& =* =+ =- =. =/ =: =< == => =? =@ =^ =_ =~ >9 >a >A >! >$ >% >& >* >+ >- >. >/ >: >< >= >> >? >@ >^ >_ >~ ?9 ?a ?A ?! ?$ ?% ?& ?* ?+ ?- ?. ?/ ?: ?< ?= ?> ?? ?@ ?^ ?_ ?~ ^9 ^a ^A ^! ^$ ^% ^& ^* ^+ ^- ^. ^/ ^: ^< ^= ^> ^? ^@ ^^ ^_ ^~ _9 _a _A _! _$ _% _& _* _+ _- _. _/ _: _< _= _> _? _@ _^ __ _~ ~9 ~a ~A ~! ~$ ~% ~& ~* ~+ ~- ~. ~/ ~: ~< ~= ~> ~? ~@ ~^ ~_ ~~ ))
	  61776)

(test (let ()(+ (let ((x 0) (y 1) (z 2) )(+ x y (let ((x 3) )(+ x (let ()(+ (let ()
        (+ (let ((x 0) (y 1) (z 2) )(+ x y z (let ((x 3) )(+ x (let ((x 4) (y 5) (z 6) )
        (+ x y z (let ()(+ (let ((x 7) )(+ x (let ()(+ (let ((x 8) (y 9) )
        (+ x (let ((x 10) (y 11) (z 12) )(+ x  ))))))))))))))))))))))))))
      50)
(test  (let* ((x 0) (y x) )(+ x y (let ()(+ (let ((x 2) )(+ x (let ()(+ (let ((x 4) )
          (+ x (let ((x 5) )(+ x (let ((x 6) (y x) (z y) )(+ x (let ((x 7) (y x) )
          (+ x (let ((x 8) (y x) )(+ x y (let ((x 9) (y x) (z y) )(+ x ))))))))))))))))))))
       48)
(test (let* ((x 0) (y x) )(+ x y (let* ()(+ (let* ((x 2) )(+ x (let* ()(+ (let* ((x 4) )
        (+ x (let* ((x 5) )(+ x (let* ((x 6) (y x) (z y) )(+ x (let* ((x 7) (y x) )
        (+ x (let* ((x 8) (y x) )(+ x y (let* ((x 9) (y x) (z y) )(+ x ))))))))))))))))))))
      49)

(test (let func ((a 1) (b 2)) (set! b a) (if (> b 0) (func (- a 1) b)) b) 1)
(test (let func ((a 1) (b 2)) (set! b a) (if (> b 0) (func (- a 1) b) b)) 0)
(test (let loop ((numbers '(3 -2 1 6 -5))
           (nonneg '())
           (neg '()))
  (cond ((null? numbers) (list nonneg neg))
        ((>= (car numbers) 0)
         (loop (cdr numbers)
               (cons (car numbers) nonneg)
               neg))
        ((< (car numbers) 0)
         (loop (cdr numbers)
               nonneg
               (cons (car numbers) neg)))))   
      '((6 1 3) (-5 -2)))
(test (let ((b '(1 2 3)))
  (let* ((a b)
         (b (cons 0 a)))
    (let b ((a b))
      (if (null? a)
           'done
            (b (cdr a))))))
      'done)
(test (let lp ((x 1000))
            (if (positive? x)
                (lp (- x 1))
                x))
      0)
(test (let func ((a 1) (b 2) (c 3)) (+ a b c (if (> a 1) (func (- a 1) (- b 1) (- c 1)) 0))) 6)
(test (let func ((a 1) (b 2) (c 3)) (+ a b c (if (> a 0) (func (- a 1) (- b 1) (- c 1)) 0))) 9)
(test (let func () 1) 1)
(test (let ((a 1)) (let func () (if (> a 1) (begin (set! a (- a 1)) (func)) 0))) 0)
(test (let func1 ((a 1)) (+ (let func2 ((a 2)) a) a)) 3)
(test (let func1 ((a 1)) (+ (if (> a 0) (func1 (- a 1)) (let func2 ((a 2)) (if (> a 0) (func2 (- a 1)) 0))) a)) 1)
(test (let func ((a (let func ((a 1)) a))) a) 1)




;;; -------- call/cc --------
;;;
;;; some of these were originally from Al Petrovsky, Scott G Miller, Matthias Radestock, J H Brown, Dorai Sitaram, 
;;;   and probably others.

(let* ((next-leaf-generator (lambda (obj eot)
			     (letrec ((return #f)
				      (cont (lambda (x)
					      (recur obj)
					      (set! cont (lambda (x) (return eot)))
					      (cont #f)))
				      (recur (lambda (obj)
					       (if (pair? obj)
						   (for-each recur obj)
						   (call-with-current-continuation
						    (lambda (c)
						      (set! cont c)
						      (return obj)))))))
			       (lambda () (call-with-current-continuation
					   (lambda (ret) (set! return ret) (cont #f)))))))
      (leaf-eq? (lambda (x y)
		  (let* ((eot (list 'eot))
			 (xf (next-leaf-generator x eot))
			 (yf (next-leaf-generator y eot)))
		    (letrec ((loop (lambda (x y)
				     (cond ((not (eq? x y)) #f)
					   ((eq? eot x) #t)
					   (else (loop (xf) (yf)))))))
		      (loop (xf) (yf)))))))

  (test (leaf-eq? '(a (b (c))) '((a) b c)) #t)
  (test (leaf-eq? '(a (b (c))) '((a) b c d)) #f))

(test (let ((r #f)
	    (a #f)
	    (b #f)
	    (c #f)
	    (i 0))
	(let () 
	  (set! r (+ 1 (+ 2 (+ 3 (call/cc (lambda (k) (set! a k) 4))))
		     (+ 5 (+ 6 (call/cc (lambda (k) (set! b k) 7))))))
	  (if (not c) 
	      (set! c a))
	  (set! i (+ i 1))
	  (case i
	    ((1) (a 5))
	    ((2) (b 8))
	    ((3) (a 6))
	    ((4) (c 4)))
	  r))
      28)

(test (let ((r #f)
	    (a #f)
	    (b #f)
	    (c #f)
	    (i 0))
	(let () 
	  (set! r (+ 1 (+ 2 (+ 3 (call/cc (lambda (k) (set! a k) 4))))
		     (+ 5 (+ 6 (call/cc (lambda (k) (set! b k) 7))))))
	  (if (not c) 
	      (set! c a))
	  (set! i (+ i 1))
	  (case i
	    ((1) (b 8))
	    ((2) (a 5))
	    ((3) (b 7))
	    ((4) (c 4)))
	  r))
      28)

 (test (let ((k1 #f)
	     (k2 #f)
	     (k3 #f)
	     (state 0))
	 (define (identity x) x)
	 (define (fn)
	   ((identity (if (= state 0)
			  (call/cc (lambda (k) (set! k1 k) +))
			  +))
	    (identity (if (= state 0)
			  (call/cc (lambda (k) (set! k2 k) 1))
			  1))
	    (identity (if (= state 0)
			  (call/cc (lambda (k) (set! k3 k) 2))
			  2))))
	 (define (check states)
	   (set! state 0)
	   (let* ((res '())
		  (r (fn)))
	     (set! res (cons r res))
	     (if (null? states)
		 res
		 (begin (set! state (car states))
			(set! states (cdr states))
			(case state
			  ((1) (k3 4))
			  ((2) (k2 2))
			  ((3) (k1 -)))))))
	 (map check '((1 2 3) (1 3 2) (2 1 3) (2 3 1) (3 1 2) (3 2 1))))
       '((-1 4 5 3) (4 -1 5 3) (-1 5 4 3) (5 -1 4 3) (4 5 -1 3) (5 4 -1 3)))

(test (let ((x '())
	    (y 0))
	(call/cc 
	 (lambda (escape)
	   (let* ((yin ((lambda (foo) 
			  (set! x (cons y x))
			  (if (= y 10)
			      (escape x)
			      (begin
				(set! y 0)
				foo)))
			(call/cc (lambda (bar) bar))))
		  (yang ((lambda (foo) 
			   (set! y (+ y 1))
			   foo)
			 (call/cc (lambda (baz) baz)))))
	     (yin yang)))))
      '(10 9 8 7 6 5 4 3 2 1 0))

(test (let ((c #f))
	(let ((r '()))
	  (let ((w (let ((v 1))
		     (set! v (+ (call-with-current-continuation
				 (lambda (c0) (set! c c0) v))
				v))
		     (set! r (cons v r))
		     v)))
	    (if (<= w 1024) (c w) r))))
      '(2048 1024 512 256 128 64 32 16 8 4 2))

(test (let ((cc #f)
	    (r '()))
	(let ((s (list 1 2 3 4 (call/cc (lambda (c) (set! cc c) 5)) 6 7 8)))
	  (if (null? r)
	      (begin (set! r s) (cc -1))
	      (list r s))))
      '((1 2 3 4 5 6 7 8) (1 2 3 4 -1 6 7 8)))

(test (let ((count 0))
        (let ((first-time? #t)
              (k (call/cc values)))
          (if first-time?
              (begin
                (set! first-time? #f)
                (set! count (+ count 1))
                (k values))
              (void)))
        count)
      2)

(test (procedure? (call/cc call/cc)) #t)
(test (call/cc (lambda (c) (0 (c 1)))) 1)
(test (call/cc (lambda (k) (k "foo"))) "foo")
(test (call/cc (lambda (k) "foo")) "foo")
(test (call/cc (lambda (k) (k "foo") "oops")) "foo")
(test (call/cc (lambda (return) (catch #t (lambda () (error 'hi "")) (lambda args (return "oops"))))) "oops")
(test (call/cc (lambda (return) (catch #t (lambda () (return 1)) (lambda args (return "oops"))))) 1)
(test (catch #t (lambda () (call/cc (lambda (return) (return "oops")))) (lambda arg 1)) "oops")
(test (call/cc (if (< 2 1) (lambda (return) (return 1)) (lambda (return) (return 2) 3))) 2)
(test (call/cc (let ((a 1)) (lambda (return) (set! a (+ a 1)) (return a)))) 2)
(test (call/cc (lambda (return) (let ((hi return)) (hi 2) 3))) 2)
(test (let () (define (hi) (call/cc func)) (define (func a) (a 1)) (hi)) 1)

(test (let ((listindex (lambda (e l)
			 (call/cc (lambda (not_found)
				    (letrec ((loop 
					      (lambda (l)
						(cond
						 ((null? l) (not_found #f))
						 ((equal? e (car l)) 0)
						 (else (+ 1 (loop (cdr l))))))))
				      (loop l)))))))
	(listindex 1 '(0 3 2 4 8)))
      #f)

(test (let ((product (lambda (li)
		       (call/cc (lambda (break)
				  (let loop ((l li))
				    (cond
				     ((null? l) 1)
				     ((= (car l) 0) (break 0))
				     (else (* (car l) (loop (cdr l)))))))))))
	(product '(1 2 3 0 4 5 6)))
      0)

(test (let ((lst '()))
	((call/cc
	  (lambda (goto)
	    (letrec ((start (lambda () (set! lst (cons "start" lst)) (goto next)))
		     (next  (lambda () (set! lst (cons "next" lst))  (goto last)))
		     (last  (lambda () (set! lst (cons "last" lst)) (reverse lst))))
	      start)))))
      '("start" "next" "last"))

(test (let ((cont #f))
	(letrec ((x (call-with-current-continuation (lambda (c) (set! cont c) 0)))
		 (y (call-with-current-continuation (lambda (c) (set! cont c) 0))))
	  (if cont
	      (let ((c cont))
		(set! cont #f)
		(set! x 1)
		(set! y 1)
		(c 0))
	      (+ x y))))
      0)

(test (letrec ((x (call-with-current-continuation
		   (lambda (c)
		     (list #t c)))))
	(if (car x)
	    ((cadr x) (list #f (lambda () x)))
	    (eq? x ((cadr x)))))
      #t)

(test (call/cc (lambda (c) (0 (c 1)))) 1)

(test (let ((member (lambda (x ls)
		      (call/cc
		       (lambda (break)
			 (do ((ls ls (cdr ls)))
			     ((null? ls) #f)
			   (if (equal? x (car ls))
			       (break ls))))))))
	(list (member 'd '(a b c))
	      (member 'b '(a b c))))
      '(#f (b c)))

(test (+ 2 (call/cc (lambda (k) (* 5 (k 4))))) 6)

(test (let ((x (call/cc (lambda (k) k))))
	(x (lambda (y) "hi")))
      "hi")

(test (((call/cc (lambda (k) k)) (lambda (x) x)) "hi") "hi")

(test (let ((return #f)
	    (lst '()))
	(let ((val (+ 1 (call/cc 
			 (lambda (cont) 
			   (set! return cont) 
			   1)))))
	  (set! lst (cons val lst)))
	(if (= (length lst) 1)
	    (return 10)
	    (if (= (length lst) 2)
		(return 20)))
	(reverse lst))
      '(2 11 21))

(test (let ((r1 #f)
	    (r2 #f)
	    (lst '()))
	(define (somefunc x y)
	  (+ (* 2 (expt x 2)) (* 3 y) 1))
	(let ((val (somefunc (call/cc
			      (lambda (c1)
				(set! r1 c1)
				(c1 1)))
			     (call/cc
			      (lambda (c2)
				(set! r2 c2)
				(c2 1))))))
	  (set! lst (cons val lst)))
	(if (= (length lst) 1)
	    (r1 2)
	    (if (= (length lst) 2)
		(r2 3)))
	(reverse lst))
      '(6 12 18))

(let ((tree->generator
       (lambda (tree)
	 (let ((caller '*))
	   (letrec
	       ((generate-leaves
		 (lambda ()
		   (let loop ((tree tree))
		     (cond ((null? tree) 'skip)
			   ((pair? tree)
			    (loop (car tree))
			    (loop (cdr tree)))
			   (else
			    (call/cc
			     (lambda (rest-of-tree)
			       (set! generate-leaves
				     (lambda ()
				       (rest-of-tree 'resume)))
			       (caller tree))))))
		   (caller '()))))
	     (lambda ()
	       (call/cc
		(lambda (k)
		  (set! caller k)
		  (generate-leaves)))))))))
  (let ((same-fringe? 
	 (lambda (tree1 tree2)
	   (let ((gen1 (tree->generator tree1))
		 (gen2 (tree->generator tree2)))
	     (let loop ()
	       (let ((leaf1 (gen1))
		     (leaf2 (gen2)))
		 (if (eqv? leaf1 leaf2)
		     (if (null? leaf1) #t (loop))
		     #f)))))))

    (test (same-fringe? '(1 (2 3)) '((1 2) 3)) #t)
    (test (same-fringe? '(1 2 3) '(1 (3 2))) #f)))


(for-each
 (lambda (arg)
   (test (let ((ctr 0))
	   (let ((val (call/cc (lambda (exit)
				 (do ((i 0 (+ i 1)))
				     ((= i 10) 'gad)
				   (set! ctr (+ ctr 1))
				   (if (= i 1)
				       (exit arg)))))))
	     (and (equal? val arg)
		  (= ctr 2))))
	 #t))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (let ((ctr 0))
	   (let ((val (call/cc (lambda (exit)
				 (do ((i 0 (+ i 1)))
				     ((= i 10) arg)
				   (set! ctr (+ ctr 1))
				   (if (= i 11)
				       (exit 'gad)))))))
	     (and (equal? val arg)
		  (= ctr 10))))
	 #t))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (let ((c #f)
	    (r (string-copy "testing-hiho")))
	(let ((v (call/cc (lambda (c0) (set! c c0) (list #\a 0)))))
	  (let ((chr (car v))
		(index (cadr v)))
	    (string-set! r index chr)
	    (set! index (+ index 1))
	    (if (<= index 8) 
		(c (list (integer->char (+ 1 (char->integer chr))) index)) 
		r))))
      "abcdefghiiho")

(test (let ((x 0)
	    (again #f))
	(call/cc (lambda (r) (set! again r)))
	(set! x (+ x 1))
	(if (< x 3) (again))
	x)
      3)

(test (let* ((x 0)
	     (again #f)
	     (func (lambda (r) (set! again r))))
	(call/cc func)
	(set! x (+ x 1))
	(if (< x 3) (again))
	x)
      3)

(test (let* ((x 0)
	     (again #f))
	(call/cc (let ()
		   (lambda (r) (set! again r))))
	(set! x (+ x 1))
	(if (< x 3) (again))
	x)
      3)

(test (let ((x 0)
	    (xx 0))
	(let ((cont #f))
	  (call/cc (lambda (c) (set! xx x) (set! cont c)))
	  (set! x (+ x 1))
	  (if (< x 3)	(cont))
	  xx))
      0)

(test (call/cc procedure?) #t)
(test (procedure? (call/cc (lambda (a) a))) #t)

(for-each
 (lambda (arg)
   (test (call/cc (lambda (a) arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((a (call/cc (lambda (a) a))))
  (test (eq? a a) #t)
  (test (eqv? a a) #t)
  (test (equal? a a) #t)
  (for-each
   (lambda (ques)
     (if (ques a)
	 (begin (display ques) (display " ") (display a) (display " returned #t?") (newline))))
   question-ops))

(test (let ((conts (make-vector 4 #f)))
	(let ((lst '()))
	  (set! lst (cons (+ (call/cc (lambda (a) (vector-set! conts 0 a) 0))
			     (call/cc (lambda (a) (vector-set! conts 1 a) 0))
			     (call/cc (lambda (a) (vector-set! conts 2 a) 0))
			     (call/cc (lambda (a) (vector-set! conts 3 a) 0)))
			  lst))
	  (let ((len (length lst)))
	    (if (< len 4)
		((vector-ref conts (- len 1)) (+ len 1))
		(reverse lst)))))
      '(0 2 5 9))

(test (let ((conts '()))
	(let ((lst '()))
	  (set! lst (cons (+ (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) 1))
			     (* (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) 1))
				(+ (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) 1))
				   (* (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) 1)) 2))))
			  lst))
	  (let ((len (length lst)))
	    (if (<= len 4)
		((list-ref conts (- len 1)) (+ len 1))
		(reverse lst)))))
      ; (+ 1 (* 1 (+ 1 (* 1 2)))) to start
      ; (+ 1 ...          2     )
      ; (+ 1 ...     3    [1]   )
      ; (+ 1 ...4    [1]        )
      ; (+ 5   [1]              )
      '(4 6 6 13 8))

(test (let ((conts (make-vector 4 #f)))
	(let ((lst '()))
	  (set! lst (cons (+ (call/cc (lambda (a) (if (not (vector-ref conts 0)) (vector-set! conts 0 a)) 0))
			     (call/cc (lambda (a) (if (not (vector-ref conts 1)) (vector-set! conts 1 a)) 0))
			     (call/cc (lambda (a) (if (not (vector-ref conts 2)) (vector-set! conts 2 a)) 0))
			     (call/cc (lambda (a) (if (not (vector-ref conts 3)) (vector-set! conts 3 a)) 0)))
			  lst))
	  (let ((len (length lst)))
	    (if (< len 4)
		((vector-ref conts (- len 1)) (+ len 1))
		(reverse lst)))))
      '(0 2 3 4))

(test (let ((conts '()))
	(let ((lst '()))
	  (set! lst (cons (+ (if (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) #f)) 1 0)
			     (* (if (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) #f)) 2 1)
				(+ (if (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) #f)) 1 0)
				   (* (if (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) #f)) 2 1) 2))))
			  lst))
	  (let ((len (length lst)))
	    (if (<= len 4)
		((list-ref conts (- len 1)) #t)
		(reverse lst)))))
      ; (+ 0 (* 1 (+ 0 (* 1 2)))) to start
      ; (+ 0 ...          2     )
      ; (+ 0 ...     1   [1]    )
      ; (+ 0 ...2   [0]         )
      ; (+ 1   [1]              )
      '(2 4 3 4 3))

(test (let ((call/cc 2)) (+ call/cc 1)) 3)


(let ((r5rs-ratify (lambda (ux err)
		     (if (= ux 0.0) 
			 0
			 (let ((tt 1) 
			       (a1 0) 
			       (b2 0) 
			       (a2 1) 
			       (b1 1) 
			       (a 0)  
			       (b 0)
			       (ctr 0)
			       (x (/ 1 ux)))
			   (call-with-current-continuation
			    (lambda (return)
			      (do ()
				  (#f)
				(set! a (+ (* a1 tt) a2)) 
				(set! b (+ (* tt b1) b2))
					;(display (format #f "~A ~A~%" a (- b a)))
				(if (or (<= (abs (- ux (/ a b))) err)
					(> ctr 1000))
				    (return (/ a b)))
				(set! ctr (+ 1 ctr))
				(if (= x tt) (return))
				(set! x (/ 1 (- x tt))) 
				(set! tt (inexact->exact (floor x)))
				(set! a2 a1) 
				(set! b2 b1) 
				(set! a1 a) 
				(set! b1 b)))))))))
  
  (test (r5rs-ratify (/ (log 2.0) (log 3.0)) 1/10000000) 665/1054)
  (if (positive? 2147483648)
      (test (r5rs-ratify (/ (log 2.0) (log 3.0)) 1/100000000000) 190537/301994)))

(for-each
 (lambda (arg)
   (test (let ((ctr 0)) 
	   (let ((val (call/cc 
		       (lambda (exit) 
			 (for-each (lambda (a) 
				     (if (equal? a arg) (exit arg))
				     (set! ctr (+ ctr 1))) 
				   (list 0 1 2 3 arg 5)))))) 
	     (list ctr val)))
	 (list 4 arg)))
 (list "hi" -1 #\a 11 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t '(1 . 2)))

(test (+ 2 (call/cc (lambda (rtn) (+ 1 (let () (begin (define x (+ 1 (rtn 3)))) x))))) 5)


;;; others from stackoverflow.com Paul Hollingsworth etc:

(test (procedure? (call/cc (lambda (k) k))) #t)
(test (call/cc (lambda (k) (+ 56 (k 3)))) 3)
(test (apply
       (lambda (k i) 
	 (if (> i 5) 
	     i 
	     (k (list k (* 2 i)))))
       (call/cc (lambda (k) (list k 1))))
      8)
(test (apply
       (lambda (k i n) (if (= i 0) n (k (list k (- i 1) (* i n)))))
       (call/cc (lambda (k) (list k 6 1))))
      720)
(test (let* ((ka (call/cc (lambda (k) `(,k 1)))) (k (car ka)) (a (cadr ka)))
	(if (< a 5) (k `(,k ,(* 2 a))) a))
      8)

(test (apply (lambda (k i n) (if (eq? i 0) n (k (list k (- i 1) (* i n))))) (call/cc (lambda (k) (list k 6 1)))) 720)
(test ((call/cc (lambda (k) k)) (lambda (x) 5)) 5)

(let ()
  (define (generate-one-element-at-a-time a-list)
    (define (generator)
      (call/cc control-state)) 
    (define (control-state return)
      (for-each 
       (lambda (an-element-from-a-list)
	 (set! return
	       (call/cc
		(lambda (resume-here)
		  (set! control-state resume-here)
		  (return an-element-from-a-list)))))
       a-list)
      (return 'you-fell-off-the-end-of-the-list))
    generator)
  (let ((gen (generate-one-element-at-a-time (list 3 2 1))))
    (test (gen) 3)
    (test (gen) 2)
    (test (gen) 1)
    (test (gen) 'you-fell-off-the-end-of-the-list)))

;;; from Ferguson and Duego "call with current continuation patterns"
(test (let ()
	(define count-to-n
	  (lambda (n)
	    (let ((receiver 
		   (lambda (exit-procedure)
		     (let ((count 0))
		       (letrec ((infinite-loop
				 (lambda ()
				   (if (= count n)
				       (exit-procedure count)
				       (begin
					 (set! count (+ count 1))
					 (infinite-loop))))))
			 (infinite-loop))))))
	      (call/cc receiver))))
	(count-to-n 10))
      10)

(test (let ()
	(define product-list
	  (lambda (nums)
	    (let ((receiver
		   (lambda (exit-on-zero)
		     (letrec ((product
			       (lambda (nums)
				 (cond ((null? nums) 1)
				       ((zero? (car nums)) (exit-on-zero 0))
				       (else (* (car nums)
						(product (cdr nums))))))))
		       (product nums)))))
	      (call/cc receiver))))
	(product-list '(1 2 3 0 4 5)))
      0)

(test (let ()
	(define product-list
	  (lambda (nums)
	    (let ((receiver
		   (lambda (exit-on-zero)
		     (letrec ((product
			       (lambda (nums) 
				 (cond ((null? nums) 1)
				       ((number? (car nums))
					(if (zero? (car nums))
					    (exit-on-zero 0)
					    (* (car nums)
					       (product (cdr nums)))))
				       (else (* (product (car nums))
						(product (cdr nums))))))))
		       (product nums)))))
	      (call/cc receiver))))
	(product-list '(1 2 (3 4) ((5)))))
      120)




;;; -------- dynamic-wind --------

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (dynamic-wind
			(lambda () (set! ctr1 (+ ctr1 1)))
			(lambda () (set! ctr2 (+ ctr2 1)) ctr2)
			(lambda () (set! ctr3 (+ ctr3 1))))))
	  (= ctr1 ctr2 ctr3 ctr4 1)))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (call/cc (lambda (exit)
			       (dynamic-wind
				   (lambda () (set! ctr1 (+ ctr1 1)))
				   (lambda () (exit ctr2) (set! ctr2 (+ ctr2 1)) ctr2)
				   (lambda () (set! ctr3 (+ ctr3 1)) 123))))))
	  (and (= ctr1 ctr3 1)
	       (= ctr2 ctr4 0))))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (call/cc (lambda (exit)
			       (dynamic-wind
				   (lambda () (exit ctr1) (set! ctr1 (+ ctr1 1)))
				   (lambda () (set! ctr2 (+ ctr2 1)) ctr2)
				   (lambda () (set! ctr3 (+ ctr3 1))))))))
	  (= ctr1 ctr2 ctr3 ctr4 0)))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (call/cc (lambda (exit)
			       (dynamic-wind
				   (lambda () (set! ctr1 (+ ctr1 1)))
				   (lambda () (set! ctr2 (+ ctr2 1)) ctr2)
				   (lambda () (exit ctr3) (set! ctr3 (+ ctr3 1))))))))
	  (and (= ctr1 ctr2 1)
	       (= ctr3 ctr4 0))))
      #t)

(test (let ((path '())  
	    (c #f)) 
	(let ((add (lambda (s)  
		     (set! path (cons s path))))) 
	  (dynamic-wind  
	      (lambda () (add 'connect))  
	      (lambda () (add (call-with-current-continuation  
			       (lambda (c0) (set! c c0) 'talk1))))  
	      (lambda () (add 'disconnect))) 
	  (if (< (length path) 4) 
	      (c 'talk2) 
	      (reverse path)))) 
      '(connect talk1 disconnect  connect talk2 disconnect))


(for-each
 (lambda (arg)
   (test (dynamic-wind (lambda () #f) (lambda () arg) (lambda () #f)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (dynamic-wind (lambda () #f) (lambda () #f) (lambda () #f)) #f)
(test (+ 1 (dynamic-wind (lambda () #f) (lambda () (values 2 3 4)) (lambda () #f)) 5) 15)

(test (let ((identity (lambda (a) a)))
        (let ((x '())
              (c #f))
          (dynamic-wind
	      (lambda () (set! x (cons 'a x)))
	      (lambda ()
		(dynamic-wind
		    (lambda () (set! x (cons 'b x)))
		    (lambda ()
		      (dynamic-wind
			  (lambda () (set! x (cons 'c x)))
			  (lambda () (set! c (call/cc identity)))
			  (lambda () (set! x (cons 'd x)))))
		    (lambda () (set! x (cons 'e x))))
		(dynamic-wind
		    (lambda () (set! x (cons 'f x)))
		    (lambda () (if c (c #f)))
		    (lambda () (set! x (cons 'g x)))))
	      (lambda () (set! x (cons 'h x))))
          (reverse x)))
      '(a b c d e f g b c d e f g h))

(test (list (dynamic-wind 
		(lambda () #f)
		(lambda () (values 'a 'b 'c))
		(lambda () #f)))
      (list 'a 'b 'c))

(test (let ((dynamic-wind 1)) (+ dynamic-wind 2)) 3)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((val (dynamic-wind
		       (lambda () #f)
		       (lambda ()
			 (set! ctr1 1)
			 (call/cc
			  (lambda (exit)
			    (exit 123)
			    (set! ctr2 2)
			    321)))
		       (lambda ()
			 (set! ctr3 3)))))
	  (and (= ctr1 1) (= ctr2 0) (= ctr3 3) (= val 123))))
      #t)

(test (let ((ctr1 0))
	(let ((val (dynamic-wind
		       (let ((a 1))
			 (lambda ()
			   (set! ctr1 a)))
		       (let ((a 10))
			 (lambda ()
			   (set! ctr1 (+ ctr1 a))
			   ctr1))
		       (let ((a 100))
			 (lambda ()
			   (set! ctr1 (+ ctr1 a)))))))
	  (and (= ctr1 111) (= val 11))))
      #t)

(test (let ((ctr1 0))
	(let ((val (+ 3 (dynamic-wind
			    (let ((a 1))
			      (lambda ()
				(set! ctr1 a)))
			    (let ((a 10))
			      (lambda ()
				(set! ctr1 (+ ctr1 a))
				ctr1))
			    (let ((a 100))
			      (lambda ()
				(set! ctr1 (+ ctr1 a)))))
		      1000)))
	  (and (= ctr1 111) (= val 1014))))
      #t)

(test (let ((n 0))
	(call-with-current-continuation
	 (lambda (k)
	   (dynamic-wind
	       (lambda ()
		 (set! n (+ n 1))
		 (k))
	       (lambda ()
		 (set! n (+ n 2)))
	       (lambda ()
		 (set! n (+ n 4))))))
	n)
      1)

(test (let ((n 0))
	(call-with-current-continuation
	 (lambda (k)
	   (dynamic-wind
	       (lambda () #f)
	       (lambda ()
		 (dynamic-wind
		     (lambda () #f)
		     (lambda ()
		       (set! n (+ n 1))
		       (k))
		     (lambda ()
		       (set! n (+ n 2))
		       ;(k)
		       )))
	       (lambda ()
		 (set! n (+ n 4))))))
	n)
      7)

(test (let ((n 0))
	(call-with-current-continuation
	 (lambda (k)
	   (dynamic-wind
	       (lambda () #f)
	       (lambda ()
		 (dynamic-wind
		     (lambda () #f)
		     (lambda ()
		       (dynamic-wind
			   (lambda () #f)
			   (lambda ()
			     (set! n (+ n 1))
			     (k))
			   (lambda ()
			     (if (= n 1)
				 (set! n (+ n 2))))))
		     (lambda ()
		       (if (= n 3)
			   (set! n (+ n 4))))))
	       (lambda ()
		 (if (= n 7)
		     (set! n (+ n 8)))))))
	n)
      15)



;;; -------- delay and force --------

(if with-delay
    (begin

      (test (let ((count 0))
	      (let ((p1 (delay (* 2 3 (let () (set! count (+ count 1)) count)))))
		(let ((val1 (force p1))
		      (val2 (force p1))
		      (val3 (force p1)))
		  (and (= val1 val2 val3 6)
		       (= count 1)))))
	    #t)

      (test (let ((stream-car (lambda (s) (car (force s))))
		  (stream-cdr (lambda (s) (cdr (force s))))
		  (counters (let next ((n 1)) (delay (cons n (next (+ n 1)))))))
	      (let* ((val1 (stream-car counters))
		     (val2 (stream-car (stream-cdr counters))))
		(letrec ((stream-add (lambda (s1 s2)
				       (delay (cons 
					       (+ (stream-car s1) (stream-car s2))
					       (stream-add (stream-cdr s1) (stream-cdr s2)))))))
		  (let ((even-counters (stream-add counters counters)))
		    (let* ((val3 (stream-car even-counters))
			   (val4 (stream-car (stream-cdr even-counters))))
		      (list val1 val2 val3 val4))))))
	    (list 1 2 2 4))

      (test (force (delay (+ 1 2))) 3)
      (test (let ((p (delay (+ 1 2)))) (list (force p) (force p))) (list 3 3))
      (test (letrec ((a-stream (letrec ((next (lambda (n)
						(cons n (delay (next (+ n 1)))))))
				 (next 0)))
		     (head car)
		     (tail (lambda (stream) (force (cdr stream)))))
	      (head (tail (tail a-stream))))
	    2)

      (letrec ((count 0)
	       (p (delay (begin (set! count (+ count 1))
				(if (> count x)
				    count
				    (force p)))))
	       (x 5))
	(test (force p) 6)
	(set! x 10)
	(test (force p) 6))

      (test (let ((generate (lambda (use-it)
			      (let loop ((i 0))
				(if (< i 10) (begin (use-it i) (loop (+ i 1)))))))
		  (generator->lazy-list (lambda (generator)
					  (delay
					    (call/cc (lambda (k-main)
						       (generator 
							(lambda (e)
							  (call/cc (lambda (k-reenter)
								     (k-main (cons e 
										   (delay 
										     (call/cc (lambda (k-new-main)
												(set! k-main k-new-main)
												(k-reenter #f))))))))))
						       (k-main '()))))))
		  (fnull? (lambda (x) (null? (force x))))
		  (fcar (lambda (x) (car (force x))))
		  (fcdr (lambda (x) (cdr (force x)))))
	      (letrec ((lazy-list->list (lambda (lz)
					  (if (fnull? lz) '()
					      (cons (fcar lz) (lazy-list->list (fcdr lz)))))))
		(lazy-list->list (generator->lazy-list generate))))
	    '(0 1 2 3 4 5 6 7 8 9))

      (test (let* ((x 1)
		   (p (delay (+ x 1))))
	      (force p)
	      (set! x (+ x 1))
	      (force p))
	    2)

      (test (let* ((x 1) 
		   (p #f))
	      (let* ((x 2))
		(set! p (delay (+ x 1))))
	      (force p))
	    3)

      (test (letrec ((count 0)
		     (x 5)
		     (p (delay (begin (set! count (+ count 1))
				      (if (> count x)
					  count
					  (force p))))))
	      (force p)
	      (set! x 10)
	      (force p))
	    6)
      
      (test (let ((count 0))
	      (define s (delay (begin (set! count (+ count 1)) 1)))
	      (+ (force s) (force s))
	      count)
	    1)

      (test (let ()
	      (define f
		(let ((first? #t))
		  (delay
		    (if first?
			(begin
			  (set! first? #f)
			  (force f))
			'second))))
	      (force f))
	    'second)

      (test (let ()
	      (define q
		(let ((count 5))
		  (define (get-count) count)
		  (define p (delay (if (<= count 0)
				       count
				       (begin (set! count (- count 1))
					      (force p)
					      (set! count (+ count 2))
					      count))))
		  (list get-count p)))
	      (let* ((get-count (car q))
		     (p (cadr q))
		     (a (get-count))
		     (b (force p))
		     (c (get-count)))
		(list a b c)))
	    (list 5 0 10))

      (for-each
       (lambda (arg)
	 (test (force (delay arg)) arg))
       (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

      (test
       (let ()
	 ;; j bartlett
	 
	 (define (square x) (* x x))
	 
	 (define (calculate-statistics-1 the-series)
	   (let* ((size (length the-series))
		  (sum (apply + the-series))
		  (mean (/ sum size))
		  (variance (let* ((variance-list (map (lambda (x) (square (- x mean))) the-series)))
			      (/ (apply + variance-list) size)))
		  (standard-deviation (sqrt variance)))
	     (vector mean variance standard-deviation)))
	 
	 (define (calculate-statistics the-series)
	   (let* ((size (delay (length the-series)))
		  (mean (delay (/ (apply + the-series) (force size))))
		  (variance (delay (let* ((variance-list (map (lambda (x) (square (- x (force mean)))) the-series)))
				     (/ (apply + variance-list) (force size)))))
		  (standard-deviation (delay (sqrt (force variance)))))
	     (vector mean variance standard-deviation)))
	 
	 (let* ((stats1 (calculate-statistics-1 '(2 6 4 3 7 4 3 6 7 8 43 4 3 2 36 75 3)))
		(stats (calculate-statistics '(2 6 4 3 7 4 3 6 7 8 43 4 3 2 36 75 3)))
		(mean (force (vector-ref stats 0)))
		(variance (force (vector-ref stats 1)))
		(stddev (force (vector-ref stats 2))))
	   (and (equal? (vector-ref stats1 0) mean)
		(equal? (vector-ref stats1 1) variance)
		(< (abs (- (vector-ref stats1 2) stddev)) 1.0e-6))))
       #t)

      (test 
       (let ()
	 ;; j bartlett
	 (define (flatten x)
	   (define (does-flatten x)
	     (if (not (pair? x)) x
		 (cond
		  ((null? (car x)) (does-flatten (cdr x)))
		  ((not (pair? (car x)))
		   (cons (car x) (delay (does-flatten (cdr x)))))
		  (else
		   (does-flatten
		    (cons (caar x) (cons (cdar x) (cdr x))))))))
	   (delay (does-flatten x)))
	 
	 (let ((lst '((1 2) 3 ((4) 5)))
	       (newlst '()))
	   (do ((i 0 (+ i 1))
		(p (force (flatten lst)) (force (cdr p))))
	       ((= i 5) (reverse newlst))
	     (set! newlst (cons (car p) newlst)))))
       (list 1 2 3 4 5))

      
      ))


    
    
;;; -------- quasiquote --------

(test `(1 2 3) '(1 2 3))
(test `() '())
(test `(list ,(+ 1 2) 4)  '(list 3 4))
(test `(1 ,@(list 1 2) 4) '(1 1 2 4))
(test `#(1 ,@(list 1 2) 4) '#(1 1 2 4))
(test `(a ,(+ 1 2) ,@(map abs '(4 -5 6)) b) '(a 3 4 5 6 b))
(if (eqv? 2 (sqrt 4))
    (test `#(10 5 ,(sqrt 4) ,@(map sqrt '(16 9)) 8) '#(10 5 2 4 3 8))) ; inexactness foolishness
(test `(a `(b ,(+ 1 2) ,(foo ,(+ 1 3) d) e) f) '(a `(b ,(+ 1 2) ,(foo 4 d) e) f))
(test (let ((name1 'x) (name2 'y)) `(a `(b ,,name1 ,',name2 d) e)) '(a `(b ,x ,'y d) e))
(test `(,@'() . foo) 'foo)
(test `(1 2 ,(* 9 9) 3 4) '(1 2 81 3 4))
(test `(1 ,(+ 1 1) 3) '(1 2 3))                     
;#; `(,(+ 1 2)) -> infinite loop?
(test `(,'a . ,'b) (cons 'a 'b))

;; from gauche
(let ((quasi0 99)
      (quasi1 101)
      (quasi2 '(a b))
      (quasi3 '(c d)))
  (test `,quasi0 99)
  (test `,quasi1 101)
  (test `(,(cons 1 2)) '((1 . 2)))
  (test `(,(cons 1 2) 3) '((1 . 2) 3))
  (test `(,quasi0 3) '(99 3))
  (test `(3 ,quasi0) '(3 99))
  (test `(,(+ quasi0 1) 3) '(100 3))
  (test `(3 ,(+ quasi0 1)) '(3 100))
  (test `(,quasi1 3) '(101 3))
  (test `(3 ,quasi1) '(3 101))
  (test `(,(+ quasi1 1) 3) '(102 3))
  (test `(3 ,(+ quasi1 1)) '(3 102))
  (test `(1 ,@(list 2 3) 4) '(1 2 3 4))
  (test `(1 2 ,@(list 3 4)) '(1 2 3 4))
  (test `(,@quasi2 ,@quasi3) '(a b c d))
  (test `(1 2 . ,(list 3 4)) '(1 2 3 4))
  (test `(,@quasi2 . ,quasi3) '(a b c d))
  (test `#(,(cons 1 2) 3) '#((1 . 2) 3))
  (test `#(,quasi0 3) '#(99 3))
  (test `#(,(+ quasi0 1) 3) '#(100 3))
  (test `#(3 ,quasi1) '#(3 101))
  (test `#(3 ,(+ quasi1 1)) '#(3 102))
  (test `#(1 ,@(list 2 3) 4) '#(1 2 3 4))
  (test `#(1 2 ,@(list 3 4)) '#(1 2 3 4))
  (test `#(,@quasi2 ,@quasi3) '#(a b c d))
  (test `#(,@quasi2 ,quasi3) '#(a b (c d)))
  (test `#(,quasi2  ,@quasi3) '#((a b) c d))
  (test `#() '#())
  (test `#(,@(list)) '#())
  (test `(,@(list 1 2) ,@(list 1 2)) '(1 2 1 2))
  (test `(,@(list 1 2) a ,@(list 1 2)) '(1 2 a 1 2))
  (test `(a ,@(list 1 2) ,@(list 1 2)) '(a 1 2 1 2))
  (test `(,@(list 1 2) ,@(list 1 2) a) '(1 2 1 2 a))
  (test `(,@(list 1 2) ,@(list 1 2) a b) '(1 2 1 2 a b))
  (test `(,@(list 1 2) ,@(list 1 2) . a) '(1 2 1 2 . a))
  (test `(,@(list 1 2) ,@(list 1 2) . ,(cons 1 2)) '(1 2 1 2 1 . 2))
  (test `(,@(list 1 2) ,@(list 1 2) . ,quasi2) '(1 2 1 2 a b))
  (test `(,@(list 1 2) ,@(list 1 2) a . ,(cons 1 2)) '(1 2 1 2 a 1 . 2))
  (test `(,@(list 1 2) ,@(list 1 2) a . ,quasi3) '(1 2 1 2 a c d))
  (test `#(,@(list 1 2) ,@(list 1 2)) '#(1 2 1 2))
  (test `#(,@(list 1 2) a ,@(list 1 2)) '#(1 2 a 1 2))
  (test `#(a ,@(list 1 2) ,@(list 1 2)) '#(a 1 2 1 2))
  (test `#(,@(list 1 2) ,@(list 1 2) a) '#(1 2 1 2 a))
  (test `#(,@(list 1 2) ,@(list 1 2) a b) '#(1 2 1 2 a b))
  (test `(1 `(1 ,2 ,,(+ 1 2)) 1) '(1 `(1 ,2 ,3) 1))
  (test `(1 `(1 ,,quasi0 ,,quasi1) 1) '(1 `(1 ,99 ,101) 1))
  (test `(1 `(1 ,@2 ,@,(list 1 2))) '(1 `(1 ,@2 ,@(1 2))))
  (test `(1 `(1 ,@,quasi2 ,@,quasi3)) '(1 `(1 ,@(a b) ,@(c d))))
  (test `(1 `(1 ,(,@quasi2 x) ,(y ,@quasi3))) '(1 `(1 ,(a b x) ,(y c d))))
  (test `#(1 `(1 ,2 ,,(+ 1 2)) 1) '#(1 `(1 ,2 ,3) 1))
  (test `#(1 `(1 ,,quasi0 ,,quasi1) 1) '#(1 `(1 ,99 ,101) 1))
  (test `#(1 `(1 ,@2 ,@,(list 1 2))) '#(1 `(1 ,@2 ,@(1 2))))
  (test `#(1 `(1 ,@,quasi2 ,@,quasi3)) '#(1 `(1 ,@(a b) ,@(c d))))
  (test `#(1 `(1 ,(,@quasi2 x) ,(y ,@quasi3))) '#(1 `(1 ,(a b x) ,(y c d))))
  (test `(1 `#(1 ,(,@quasi2 x) ,(y ,@quasi3))) '(1 `#(1 ,(a b x) ,(y c d)))))

(test (let ((hi (lambda (a) `(+ 1 ,a))))
	(hi 2))
      '(+ 1 2))

(test (let ((hi (lambda (a) `(+ 1 ,@a))))
	(hi (list 2 3)))
      '(+ 1 2 3))

(test (let ((hi (lambda (a) `(let ((b ,a)) ,(+ 1 a)))))
	(hi 3))
      '(let ((b 3)) 4))

(test (let ((x '(a b c)))
	`(x ,x ,@x foo ,(cadr x) bar ,(cdr x) baz ,@(cdr x)))
      '(x (a b c) a b c foo b bar (b c) baz b c))

(test (let ((x '(a b c)))
	`(,(car `(,x))))
      '((a b c)))

(test (let ((x '(a b c)))
	`(,@(car `(,x))))
      '(a b c))

(test (let ((x '(a b c)))
	`(,(car `(,@x))))
      '(a))

(test (let ((x '(a b c)))
	(cadadr ``,,x))
      '(a b c))

(test (let ((x '(a b c)))
	`,(car `,x))
      'a)

(test (let ((x '(2 3)))
	`(1 ,@x 4))
      '(1 2 3 4))

(test (let ((x '(2 3)))
	`#(9 ,@x 9))
      '#(9 2 3 9))

(test `#(1 ,(/ 12 2)) '#(1 6))
(test ((lambda () `#(1 ,(/ 12 2)))) '#(1 6))

(test (let ((x '(2 3)))
	`(1 ,@(map (lambda (a) (+ a 1)) x)))
      '(1 3 4))

;;; these are from the scheme bboard
(test (let ((x '(1 2 3))) `(0 . ,x)) '(0 1 2 3))
(test (let ((x '(1 2 3))) `(0 ,x)) '(0 (1 2 3)))
(test (let ((x '(1 2 3))) `#(0 ,x)) '#(0 (1 2 3)))
;(test (let ((x '(1 2 3))) `#(0 . ,x)) '#(0 1 2 3))



(if with-hash-tables
    (begin

      (let ((ht (make-hash-table)))
	(test (hash-table? ht) #t)
	(test (let () (hash-table-set! ht 'key 3.14) (hash-table-ref ht 'key)) 3.14)
	(test (let () (hash-table-set! ht "ky" 3.14) (hash-table-ref ht "ky")) 3.14)
	(test (let () (hash-table-set! ht 123 "hiho") (hash-table-ref ht 123)) "hiho")
	(test (let () (hash-table-set! ht 3.14 "hi") (hash-table-ref ht 3.14)) "hi")
	(test (let () (hash-table-set! ht our-pi "hiho") (hash-table-ref ht our-pi)) "hiho")
	(test (hash-table-ref ht "123") #f)
	(for-each
	 (lambda (arg)
	   (test (let () (hash-table-set! ht 'key arg) (hash-table-ref ht 'key)) arg))
	 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2))))
      
      (let ((ht (make-hash-table 277)))
	(test (hash-table? ht) #t)
	(test (hash-table-size ht) 277)
	(test (let () (hash-table-set! ht 'key 3.14) (hash-table-ref ht 'key)) 3.14)
	(test (let () (hash-table-set! ht "ky" 3.14) (hash-table-ref ht "ky")) 3.14)
	(for-each
	 (lambda (arg)
	   (test (let () (hash-table-set! ht 'key arg) (hash-table-ref ht 'key)) arg))
	 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2))))
      
      (for-each
       (lambda (arg)
	 (test (hash-table? arg) #f))
       (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))
      
      (test (hash-table? (make-vector 3 '())) #f)

      (let ((ht (make-hash-table)))	
	(test (hash-table-ref ht 'not-a-key) #f)
	(test (hash-table-ref ht "not-a-key") #f)
	(hash-table-set! ht 'key 3/4)
	(hash-table-set! ht "key" "hi")
	(test (hash-table-ref ht "key") "hi")
	(test (hash-table-ref ht 'key) "hi")
	
	(hash-table-set! ht 'asd 'hiho)
	(test (hash-table-ref ht 'asd) 'hiho))
      
      (let ((ht1 (make-hash-table 653))
	    (ht2 (make-hash-table 277)))
	(hash-table-set! ht1 'key 'hiho)
	(hash-table-set! ht2 (hash-table-ref ht1 'key) 3.14)
	(test (hash-table-size ht1) 653)
	(test (hash-table-ref ht2 'hiho) 3.14)
	(test (hash-table-ref ht2 (hash-table-ref ht1 'key)) 3.14))
      
      ))


(if with-keywords
    (begin

      (for-each
       (lambda (arg)
	 (test (keyword? arg) #f))
       (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))
      
      (let ((kw (make-keyword "hiho")))
	(test (keyword? kw) #t)
	(test (keyword->symbol kw) 'hiho)
	(test (symbol->keyword 'hiho) kw)
	(test (keyword->symbol (symbol->keyword 'key)) 'key)
	(test (symbol->keyword (keyword->symbol (make-keyword "hi"))) :hi)
	(test (keyword? :a-key) #t)
	(test (keyword? ':a-key) #t)
	(test (keyword? ':a-key:) #t)
	(test (keyword? 'a-key:) #t)
	(test (symbol? (keyword->symbol :hi)) #t)
	(test (keyword? (keyword->symbol :hi)) #f)
	(test (symbol? (symbol->keyword 'hi)) #t)
	(test (equal? kw :hiho) #t)
	(test ((lambda (arg) (keyword? arg)) :hiho) #t)
	(test ((lambda (arg) (keyword? arg)) 'hiho) #f)
	(test ((lambda (arg) (keyword? arg)) kw) #t)
	(test ((lambda (arg) (keyword? arg)) (symbol->keyword 'hiho)) #t)
	(test (make-keyword "3") :3)
	(test (keyword? :3) #t)
	(test (keyword? ':3) #t)
	(test (keyword? '3) #f)
	(test (symbol->string (keyword->symbol hi:)) "hi")
	(test (symbol->string (keyword->symbol :hi)) "hi")
	(test (make-keyword ":") ::))
      
      ))


(if with-sort!
    (begin

      (test (equal? (sort! (list 3 4 8 2 0 1 5 9 7 6) <) (list 0 1 2 3 4 5 6 7 8 9)) #t)
      (test (equal? (sort! (list 3 4 8 2 0 1 5 9 7 6) (lambda (a b) (< a b))) (list 0 1 2 3 4 5 6 7 8 9)) #t)
      (test (equal? (sort! (list) <) '()) #t)
      (test (equal? (sort! (list 1) <) '(1)) #t)
      (test (equal? (sort! (list 1 1 1) <) '(1 1 1)) #t)
      (test (equal? (sort! (list 0 1 2 3 4 5 6 7 8 9) <) '(0 1 2 3 4 5 6 7 8 9)) #t)
      (test (equal? (sort! (list #\a #\l #\o #\h #\a) char<?) '(#\a #\a #\h #\l #\o)) #t)
      (test (equal? (sort! (list "tic" "tac" "toe") string<?) '("tac" "tic" "toe")) #t)
      (test (equal? (sort! (list 3 4 8 2 0 1 5 9 7 6) >) (reverse (list 0 1 2 3 4 5 6 7 8 9))) #t)
      (test (equal? (sort! '((3 . 1) (2 . 8) (5 . 9) (4 . 7) (6 . 0)) (lambda (a b) (< (car a) (car b)))) '((2 . 8) (3 . 1) (4 . 7) (5 . 9) (6 . 0))) #t)
      (test (equal? (sort! '((3 . 1) (2 . 8) (5 . 9) (4 . 7) (6 . 0)) (lambda (a b) (< (cdr a) (cdr b)))) '((6 . 0) (3 . 1) (4 . 7) (2 . 8) (5 . 9))) #t)
      (test (equal? (sort! (list (list 1 2) (list 4 3 2) (list) (list 1 2 3 4)) (lambda (a b) (>= (length a) (length b)))) '((1 2 3 4) (4 3 2) (1 2) ())) #t)
      (test (equal? (sort! '((1 2 3) (4 5 6) (7 8 9)) (lambda (a b) (> (car a) (car b)))) '((7 8 9) (4 5 6) (1 2 3))) #t)
      (test (equal? (sort! (list #\b #\A #\B #\a #\c #\C) char<?) '(#\A #\B #\C #\a #\b #\c)) #t)
      (test (equal? (sort! (list (list 'u 2) (list 'i 1) (list 'a 7) (list 'k 3) (list 'c 4) (list 'b 6))
			   (lambda (a b) (< (cadr a) (cadr b))))
		    '((i 1) (u 2) (k 3) (c 4) (b 6) (a 7)))
	    #t)
      (test (equal? (sort! (sort! '(1 2 3) >) <) '(1 2 3)) #t)

      (test (equal? (sort! (vector 3 4 8 2 0 1 5 9 7 6) <) (vector 0 1 2 3 4 5 6 7 8 9)) #t)
      (test (equal? (sort! '#() <) '#()) #t)

      (test (call/cc (lambda (return) (sort! '(1 2 3) (lambda (a b) (return "oops"))))) "oops")
      
      (let ((v (make-vector 10000)))
	(do ((i 0 (+ i 1)))
	    ((= i 10000))
	  (vector-set! v i (random 100.0)))
	(sort! v <)
	(call/cc
	 (lambda (return)
	   (do ((i 0 (+ i 1)))
	       ((< i 9999))
	     (if (not (< (vector-ref v i) (vector-ref v (+ i 1))))
		 (return #f)))
	   #t)))

      (let ((v '()))
	(do ((i 0 (+ i 1)))
	    ((= i 10000))
	  (set! v (cons (random 100.0) v)))
	(set! v (sort! v >))
	(call/cc
	 (lambda (return)
	   (let ((val (car v)))
	     (do ((lst (cdr v) (cdr lst)))
		 ((null? (cdr lst)))
	       (if (not (> val (car lst)))
		   (return #f))
	       (set! val (car lst))))
	   #t)))

      ))


(if with-gensym
    (begin

      (test (symbol? (gensym)) #t)
      (test (symbol? (gensym "temp")) #t)
      (test (eq? (gensym) (gensym)) #f)
      (test (eqv? (gensym) (gensym)) #f)
      (test (equal? (gensym) (gensym)) #f)
      (test (keyword? (gensym)) #f)
      
      (let ((sym (gensym)))
	(test (eval `(let ((,sym 32)) (+ ,sym 1))) 33))

      (let ((sym1 (gensym))
	    (sym2 (gensym)))
	(test (eval `(let ((,sym1 32) (,sym2 1)) (+ ,sym1 ,sym2))) 33))

      (test (let ((hi (gensym))) (eq? hi (string->symbol (symbol->string hi)))) #t)

      (if with-reverse!
	  (begin

	    ;; from guile-user I think
	    ;; (block LABEL FORMS...)
	    ;;
	    ;; Execute FORMS.  Within FORMS, a lexical binding named LABEL is
	    ;; visible that contains an escape function for the block.  Calling
	    ;; the function in LABEL with a single argument will immediatly stop
	    ;; the execution of FORMS and return the argument as the value of the
	    ;; block.  If the function in LABEL is not invoked, the value of the
	    ;; block is the value of the last form in FORMS.
	    
	    (define-macro (block label . forms)
	      `(let ((body (lambda (,label) ,@forms))
		     (tag (gensym "return-")))
		 (catch tag
			(lambda () (body (lambda (val) (error tag val))))
			(lambda (tag val) val))))
	    
	    ;; (with-return FORMS...)
	    ;;
	    ;; Equivalent to (block return FORMS...)
	    
	    (define-macro (with-return . forms)
	      `(block return ,@forms))
	    
	    ;; (tagbody TAGS-AND-FORMS...)
	    ;;
	    ;; TAGS-AND-FORMS is a list of either tags or forms.  A TAG is a
	    ;; symbol while a FORM is everything else.  Normally, the FORMS are
	    ;; executed sequentially.  However, control can be transferred to the
	    ;; forms following a TAG by invoking the tag as a function.  That is,
	    ;; within the FORMS, there is a lexical binding for each TAG with the
	    ;; symbol that is the tag as its name.  The bindings carry functions
	    ;; that will execute the FORMS following the respective TAG.
	    ;;
	    ;; The value of a tagbody is always `#f'.
	    
	    (define (transform-tagbody forms)
	      (let ((start-tag (gensym "start-"))
		    (block-tag (gensym "block-")))
		(let loop ((cur-tag start-tag)
			   (cur-code '())
			   (tags-and-code '())
			   (forms forms))
		  (cond
		   ((null? forms)
		    `(block ,block-tag
			    (letrec ,(reverse! (cons (list cur-tag `(lambda ()
								      ,@(reverse! 
									 (cons `(,block-tag #f)
									       cur-code))))
						     tags-and-code))
			      (,start-tag))))
		   ((symbol? (car forms))
		    (loop (car forms)
			  '()
			  (cons (list cur-tag `(lambda ()
						 ,@(reverse! (cons `(,(car forms))
								   cur-code))))
				tags-and-code)
			  (cdr forms)))
		   (else
		    (loop cur-tag
			  (cons (car forms) cur-code)
			  tags-and-code
			  (cdr forms)))))))
	    
	    (define-macro (tagbody . forms)
	      (transform-tagbody forms))
	    
	    (define (first_even l)
	      (with-return
	       (tagbody
		continue
		(if (not (not (null? l)))
		    (break))
		(let ((e (car l)))
		  (if (not (number? e))
		      (break))
		  (if (even? e)
		      (return e))
		  (set! l (cdr l)))
		(continue)
		break)
	       (return #f)))
	    
	    (let ((val (first_even '(1 3 5 6 7 8 9))))
	      (if (not (equal? val (list 6)))
		  (begin (display "first_even (tagbody, gensym, reverse!): '") (display val) (newline))))
	    
	    ))
      ))


(if with-format
    (begin 
      
      (test (format #f "hiho") "hiho")
      (test (format #f "") "")
      (test (format #f "a") "a")
      
      (test (format #f "~~") "~")
      (test (format #f "~~~~") "~~")
      (test (format #f "a~~") "a~")
      (test (format #f "~~a") "~a")
      
      (test (format #f "hiho~%ha") (string-append "hiho" (string #\newline) "ha"))
      (test (format #f "~%") (string #\newline))
      (test (format #f "~%ha") (string-append (string #\newline) "ha"))
      (test (format #f "hiho~%") (string-append "hiho" (string #\newline)))
      
      (for-each
       (lambda (arg res)
	 (let ((val (catch #t (lambda () (format #f "~A" arg)) (lambda args 'error))))
	   (if (or (not (string? val))
		   (not (string=? val res)))
	       (begin (display "(format #f \"~A\" ") (display arg) 
		      (display " returned \"") (display val) 
		      (display "\" but expected \"") (display res) (display "\"") 
		      (newline)))))
       (list "hiho"  -1  #\a  1   #f   #t  '#(1 2 3)   3.14   3/4  1.5+1.5i '()  '#(())  (list 1 2 3) '(1 . 2) 'hi)
       (list "hiho" "-1" "a" "1" "#f" "#t" "#(1 2 3)" "3.14" "3/4" "1.5+1.5i"   "()" "#(())" "(1 2 3)"    "(1 . 2)" "hi"))
      
      (test (format #f "hi ~A ho" 1) "hi 1 ho")
      (test (format #f "hi ~a ho" 1) "hi 1 ho")
      (test (format #f "~a~A~a" 1 2 3) "123")
      (test (format #f "~a~~~a" 1 3) "1~3")
      (test (format #f "~a~%~a" 1 3) (string-append "1" (string #\newline) "3"))
      
      (for-each
       (lambda (arg res)
	 (let ((val (catch #t (lambda () (format #f "~S" arg)) (lambda args 'error))))
	   (if (or (not (string? val))
		   (not (string=? val res)))
	       (begin (display "(format #f \"~S\" ") (display arg) 
		      (display " returned \"") (display val) 
		      (display "\" but expected \"") (display res) (display "\"") 
		      (newline)))))
       (list "hiho"  -1  #\a  1   #f   #t  '#(1 2 3)   3.14   3/4  1.5+1.5i '()  '#(())  (list 1 2 3) '(1 . 2) 'hi)
       (list "\"hiho\"" "-1" "#\\a" "1" "#f" "#t" "#(1 2 3)" "3.14" "3/4" "1.5+1.5i"   "()" "#(())" "(1 2 3)"    "(1 . 2)" "hi"))
      
      (test (format #f "hi ~S ho" 1) "hi 1 ho")
      (test (format #f "hi ~S ho" "abc") "hi \"abc\" ho")
      (test (format #f "~s~a" #\a #\b) "#\\ab")
      (test (format #f "~C~c~C" #\a #\b #\c) "abc")

      (test (format #f "~{~A~}" '(1 2 3)) "123")
      (test (format #f "asb~{~A ~}asb" '(1 2 3 4)) "asb1 2 3 4 asb")
      (test (format #f "asb~{~A ~A.~}asb" '(1 2 3 4)) "asb1 2.3 4.asb")
      (test (format #f ".~{~A~}." '()) "..")
      
      (test (format #f "~{~A ~A ~}" '(1 "hi" 2 "ho")) "1 hi 2 ho ")
      (test (format #f "~{.~{+~A+~}.~}" (list (list 1 2 3) (list 4 5 6))) ".+1++2++3+..+4++5++6+.")
      (test (format #f "~{~s ~}" '(fred jerry jill)) "fred jerry jill ")
      (test (format #f "~{~s~^ ~}" '(fred jerry jill)) "fred jerry jill")
      (test (format #f "~{~s~^~^ ~}" '(fred jerry jill)) "fred jerry jill")
      (test (format #f "~{.~{~A~}+~{~A~}~}" '((1 2) (3 4 5) (6 7 8) (9))) ".12+345.678+9")
      (test (format #f "~{.~{+~{-~A~}~}~}" '(((1 2) (3 4 5)))) ".+-1-2+-3-4-5")
      (test (format #f "~{.~{+~{-~A~}~}~}" '(((1 2) (3 4 5)) ((6) (7 8 9)))) ".+-1-2+-3-4-5.+-6+-7-8-9")
      
      (test (format #f "~A ~* ~A" 1 2 3) "1  3")
      (test (format #f "~*" 1) "")
      (test (format #f "~{~* ~}" '(1 2 3)) "   ")
      
      (test (format #f "this is a ~
             sentence") "this is a sentence")
      
      ;; ~nT handling is a mess -- what are the defaults?  which is column 1? do we space up to or up to and including?
      
      (test (format #f "asdh~20Thiho") "asdh                hiho")
      (test (format #f "asdh~2Thiho") "asdhhiho")
      (test (format #f "a~Tb") "ab")
      (test (format #f "0123456~4,8Tb") "0123456     b")
;      (test (format #f "XXX~%0123456~4,8Tb") (string-append "XXX" (string #\newline) "0123456    b")) ; clearly wrong...
      (test (format #f "0123456~0,8Tb") "0123456 b")
;      (test (format #f "0123456~10,8Tb") "0123456           b")
      (test (format #f "0123456~1,0Tb") "0123456b")
      (test (format #f "0123456~1,Tb") "0123456b")
      (test (format #f "0123456~1,Tb") "0123456b")
      (test (format #f "0123456~,Tb") "0123456b")
;      (test (format #f "0123456~7,10Tb") "0123456          b") 
;      (test (format #f "0123456~8,10tb") "0123456           b")
      (test (format #f "0123456~3,12tb") "0123456        b")
      
;      (test (format #f "~40TX") "                                       X")
;      (test (format #f "X~,8TX~,8TX") "X       X       X")
      (test (format #f "X~8,TX~8,TX") "X       XX")
;      (test (format #f "X~8,10TX~8,10TX") "X                 X         X")
      (test (format #f "X~8,0TX~8,0TX") "X       XX")
      (test (format #f "X~0,8TX~0,8TX") "X       X       X")
;      (test (format #f "X~1,8TX~1,8TX") "X        X       X")
;      (test (format #f "X~,8TX~,8TX") "X       X       X")
      (test (format #f "X~TX~TX") "XXX") ; clisp and sbcl say "X X X" here and similar differences elsewhere
      (test (format #f "X~0,0TX~0,0TX") "XXX")
      (test (format #f "X~0,TX~0,TX") "XXX")
      (test (format #f "X~,0TX~,0TX") "XXX")
      
      (test (format #f "~D" 123) "123")
      (test (format #f "~X" 123) "7b")
      (test (format #f "~B" 123) "1111011")
      (test (format #f "~O" 123) "173")
      
      (test (format #f "~10D" 123) "       123")
      (test (format #f "~10X" 123) "        7b")
      (test (format #f "~10B" 123) "   1111011")
      (test (format #f "~10O" 123) "       173")
      
      (test (format #f "~D" -123) "-123")
      (test (format #f "~X" -123) "-7b")
      (test (format #f "~B" -123) "-1111011")
      (test (format #f "~O" -123) "-173")
      
      (test (format #f "~10D" -123) "      -123")
      (test (format #f "~10X" -123) "       -7b")
      (test (format #f "~10B" -123) "  -1111011")
      (test (format #f "~10O" -123) "      -173")
      
      (test (format #f "~d" 123) "123")
      (test (format #f "~x" 123) "7b")
      (test (format #f "~b" 123) "1111011")
      (test (format #f "~o" 123) "173")
      
      (test (format #f "~10d" 123) "       123")
      (test (format #f "~10x" 123) "        7b")
      (test (format #f "~10b" 123) "   1111011")
      (test (format #f "~10o" 123) "       173")
      
      (test (format #f "~d" -123) "-123")
      (test (format #f "~x" -123) "-7b")
      (test (format #f "~b" -123) "-1111011")
      (test (format #f "~o" -123) "-173")
      
      (test (format #f "~10d" -123) "      -123")
      (test (format #f "~10x" -123) "       -7b")
      (test (format #f "~10b" -123) "  -1111011")
      (test (format #f "~10o" -123) "      -173")
      
      (if (and (defined? 'most-positive-fixnum)
	       (= most-positive-fixnum 9223372036854775807))
	  (begin
	    
	    (test (format #f "~D" most-positive-fixnum) "9223372036854775807")
	    (test (format #f "~D" (+ 1 most-negative-fixnum)) "-9223372036854775807")
	    
	    (test (format #f "~X" most-positive-fixnum) "7fffffffffffffff")
	    (test (format #f "~X" (+ 1 most-negative-fixnum)) "-7fffffffffffffff")
	    
	    (test (format #f "~O" most-positive-fixnum) "777777777777777777777")
	    (test (format #f "~O" (+ 1 most-negative-fixnum)) "-777777777777777777777")
	    
	    (test (format #f "~B" most-positive-fixnum) "111111111111111111111111111111111111111111111111111111111111111")
	    (test (format #f "~B" (+ 1 most-negative-fixnum)) "-111111111111111111111111111111111111111111111111111111111111111")

	    (num-test (inexact->exact most-positive-fixnum) most-positive-fixnum)
	    
	    ))

      (if (and (defined? 'most-positive-fixnum)
	       (= most-positive-fixnum 2147483647))
	  (begin
	    
	    (test (format #f "~D" most-positive-fixnum) "2147483647")
	    (test (format #f "~D" (+ 1 most-negative-fixnum)) "-2147483647")
	    
	    (test (format #f "~X" most-positive-fixnum) "7fffffff")
	    (test (format #f "~X" (+ 1 most-negative-fixnum)) "-7fffffff")
	    
	    (test (format #f "~O" most-positive-fixnum) "17777777777")
	    (test (format #f "~O" (+ 1 most-negative-fixnum)) "-17777777777")
	    
	    (test (format #f "~B" most-positive-fixnum) "1111111111111111111111111111111")
	    (test (format #f "~B" (+ 1 most-negative-fixnum)) "-1111111111111111111111111111111")

	    (num-test (inexact->exact most-positive-fixnum) most-positive-fixnum)
	    
	    ))
      
      (test (format #f "~0D" 123) "123")
      (test (format #f "~0X" 123) "7b")
      (test (format #f "~0B" 123) "1111011")
      (test (format #f "~0O" 123) "173")

      (call-with-output-file "tmp1.r5rs" (lambda (p) (format p "this ~A ~C test ~D" "is" #\a 3)))
      (let ((res (call-with-input-file "tmp1.r5rs" (lambda (p) (read-line p)))))
	(if (not (string=? res "this is a test 3"))
	    (begin 
	      (display "call-with-input-file + format to \"tmp1.r5rs\" ... expected \"this is a test 3\", but got \"")
	      (display res) (display "\"?") (newline))))

      (if with-open-input-string-and-friends
	  (let ((val (format #f "line 1~%line 2~%line 3")))
	    (with-input-from-string val
	      (lambda ()
		(let ((line1 (read-line)))
		  (test (string=? line1 "line 1") #t))
		(let ((line2 (read-line)))
		  (test (string=? line2 "line 2") #t))
		(let ((line3 (read-line)))
		  (test (string=? line3 "line 3") #t))
		(let ((eof (read-line)))
		  (test (eof-object? eof) #t))))))

      (if with-open-input-string-and-friends
	  (let ((val (format #f "line 1~%line 2~%line 3")))
	    (call-with-input-string val
	      (lambda (p)
		(let ((line1 (read-line p #t)))
		  (test (string=? line1 (string-append "line 1" (string #\newline))) #t))
		(let ((line2 (read-line p #t)))
		  (test (string=? line2 (string-append "line 2" (string #\newline))) #t))
		(let ((line3 (read-line p #t)))
		  (test (string=? line3 "line 3") #t))
		(let ((eof (read-line p #t)))
		  (test (eof-object? eof) #t))))))
      
      (if with-open-input-string-and-friends
	  (let ((res #f)) 
	    (let ((this-file (open-output-string))) 
	      (format this-file "this ~A ~C test ~D" "is" #\a 3)
	      (set! res (get-output-string this-file))
	      (close-output-port this-file))
	    (if (not (string=? res "this is a test 3"))
		(begin 
		  (display "open-output-string + format ... expected \"this is a test 3\", but got \"")
		  (display res) (display "\"?") (newline)))))

      (if with-open-input-string-and-friends
	  (test (call/cc (lambda (return) (let ((val (format #f "line 1~%line 2~%line 3")))
					    (call-with-input-string val
					      (lambda (p) (return "oops"))))))
		"oops"))

      (format #t "format #t: ~D" 1)
      (format (current-output-port) " output-port: ~D! (this is testing output ports)~%" 2)
      ;; for float formats, assume s7 for now -- use our-pi and most-positive-fixnum
      ;; (format with 18 digits is enough to tell what s7_Double is via built-in pi)

      ;; from slib/formatst.scm
      (test (string=? (format #f "abc") "abc") #t)
      (test (string=? (format #f "~a" 10) "10") #t)
      (test (string=? (format #f "~a" -1.2) "-1.2") #t)
      (test (string=? (format #f "~a" 'a) "a") #t)
      (test (string=? (format #f "~a" #t) "#t") #t)
      (test (string=? (format #f "~a" #f) "#f") #t)
      (test (string=? (format #f "~a" "abc") "abc") #t)
      (test (string=? (format #f "~a" '#(1 2 3)) "#(1 2 3)") #t)
      (test (string=? (format #f "~a" '()) "()") #t)
      (test (string=? (format #f "~a" '(a)) "(a)") #t)
      (test (string=? (format #f "~a" '(a b)) "(a b)") #t)
      (test (string=? (format #f "~a" '(a (b c) d)) "(a (b c) d)") #t)
      (test (string=? (format #f "~a" '(a . b)) "(a . b)") #t)
      (test (string=? (format #f "~a ~a" 10 20) "10 20") #t)
      (test (string=? (format #f "~a abc ~a def" 10 20) "10 abc 20 def") #t)
      (test (string=? (format #f "~d" 100) "100") #t)
      (test (string=? (format #f "~x" 100) "64") #t)
      (test (string=? (format #f "~o" 100) "144") #t)
      (test (string=? (format #f "~b" 100) "1100100") #t)
      (test (string=? (format #f "~10d" 100) "       100") #t)
      (test (string=? (format #f "~10,'*d" 100) "*******100") #t)
      (test (string=? (format #f "~c" #\a) "a") #t)
      (test (string=? (format #f "~~~~") "~~") #t)
      (test (string=? (format #f "~s" "abc") "\"abc\"") #t)
      (test (string=? (format #f "~s" "abc \\ abc") "\"abc \\\\ abc\"") #t)
      (test (string=? (format #f "~a" "abc \\ abc") "abc \\ abc") #t)
      (test (string=? (format #f "~s" "abc \" abc") "\"abc \\\" abc\"") #t)
      (test (string=? (format #f "~a" "abc \" abc") "abc \" abc") #t)
      (test (string=? (format #f "~s" #\space) "#\\space") #t)
      (test (string=? (format #f "~s" #\newline) "#\\newline") #t)
      (test (string=? (format #f "~s" #\a) "#\\a") #t)
      (test (string=? (format #f "~a" '(a "b" c)) "(a \"b\" c)") #t)
      (test (string=? (format #f "abc~
         123") "abc123") #t)
      (test (string=? (format #f "abc~
123") "abc123") #t)
      (test (string=? (format #f "abc~
") "abc") #t)
      (test (string=? (format #f "~{ ~a ~}" '(a b c)) " a  b  c ") #t)
      (test (string=? (format #f "~{ ~a ~}" '()) "") #t)
      (test (string=? (format #f "~{ ~a,~a ~}" '(a 1 b 2 c 3)) " a,1  b,2  c,3 ") #t)
      (test (string=? (format #f "abc ~^ xyz") "abc ") #t)

      ))


(if with-define*
    (begin

      (let ((hi (lambda* (a) a)))
	(test (hi 1) 1)
	(test (hi) #f)          ; all args are optional
	(test (hi :a 32) 32)    ; all args are keywords
	(test (hi 1 2) 'error)  ; extra args
	
	(for-each
	 (lambda (arg)
	   (test (hi arg) arg)
	   (test (hi :a arg) arg))
	 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2)))

	(test (hi :b 1) 'error))

      (let ((hi (lambda* ((a 1)) a)))
	(test (hi 2) 2)
	(test (hi) 1)
	(test (hi :a 2) 2)

	(for-each
	 (lambda (arg)
	   (test (hi arg) arg)
	   (test (hi :a arg) arg))
	 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2))))

      (let ((hi (lambda* (a (b "hi")) (list a b))))
	(test (hi) (list #f "hi"))
	(test (hi 1) (list 1 "hi"))
	(test (hi 1 2) (list 1 2))
	(test (hi :b 1) (list #f 1))
	(test (hi :a 1) (list 1 "hi"))
	(test (hi 1 :b 2) (list 1 2))
	(test (hi :b 3 :a 1) (list 1 3))
	(test (hi :a 3 :b 1) (list 3 1))
	(test (hi 1 :a 3) (list 3 "hi"))
	(test (hi 1 2 :a 3) 'error) ; trailing (extra) args
	(test (hi :a 2 :c 1) 'error)
	(test (hi 1 :c 2) 'error)

	(for-each
	 (lambda (arg)
	   (test (hi :a 1 :b arg) (list 1 arg))
	   (test (hi :a arg) (list arg "hi"))
	   (test (hi :b arg) (list #f arg))
	   (test (hi arg arg) (list arg arg)))
	 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2))))

      (let ((hi (lambda* (a :key (b 3) :optional c) (list a b c))))
	(test (hi) (list #f 3 #f))
	(test (hi 1) (list 1 3 #f))
	(test (hi :c 32) (list #f 3 32))
	(test (hi :c 32 :b 43 :a 54) (list 54 43 32))
	(test (hi 1 2 3) (list 1 2 3))
	(test (hi :b 32) (list #f 32 #f))
	(test (hi 1 2 :c 32) (list 1 2 32)))

      (let ((hi (lambda* (a :rest b) (list a b))))
	(test (hi 1 2 3) (list 1 (list 2 3)))
	(test (hi) (list #f #f))
	(test (hi :a 2) (list 2 #f))
	(test (hi :b 3) (list #f 3)))

      (let ((hi (lambda* (a :rest b :rest c) (list a b c))))
	(test (hi 1 2 3 4 5) (list 1 (list 2 3 4 5) (list 3 4 5))))

      (let ((hi (lambda* ((a 3) :key (b #t) :optional (c our-pi) :rest d) (list a b c d))))
	(test (hi) (list 3 #t our-pi #f))
	(test (hi 1 2 3 4) (list 1 2 3 (list 4))))

      (let ((hi (lambda* ((a 'hi)) (equal? a 'hi))))
	(test (hi) #t)
	(test (hi 1) #f)
	(test (hi 'hi) #t)
	(test (hi :a 1) #f))

      (let* ((x 32)
	     (hi (lambda* (a (b x)) (list a b))))
	(test (hi) (list #f 32))
	(test (hi :a 1) (list 1 32)))

      (let ((hi (lambda* (a . b) (list a b))))
	(test (hi 1 2 3) (list 1 (list 2 3)))
	(test (hi) (list #f #f))
	(test (hi :a 2) (list 2 #f))
	(test (hi :b 3) (list #f 3)))
	
      (let ((hi (lambda* ((a 0.0) :optional (b 0.0)) (+ a b))))
	(num-test (hi 1.0) 1.0)
	(num-test (hi 1.0 2.0) 3.0)
	(num-test (hi) 0.0)
	(num-test (+ (hi) (hi 1.0) (hi 1.0 2.0)) 4.0)
	(num-test (+ (hi 1.0) (hi) (hi 1.0 2.0)) 4.0)
	(num-test (+ (hi 1.0) (hi 1.0 2.0) (hi)) 4.0)
	(num-test (+ (hi 1.0 2.0) (hi) (hi 1.0)) 4.0))

      (let ((hi (lambda* ((a 3)) a)))
	(test :hi (hi :hi)))
      (let ((hi (lambda* ((a 3) (b 0)) a)))
	(test :hi (hi :hi))
	(test 0 (hi 0 :hi)))

      (test (let ((hi (lambda*))) (hi)) 'error)
      (test (let ((hi (lambda* #f))) (hi)) 'error)
      (test (let ((hi (lambda* "hi" #f))) (hi)) 'error)
      (test (let ((hi (lambda* ("hi") #f))) (hi)) 'error)
      (test (let ((hi (lambda* (a 0.0) a))) (hi)) 'error)
      (test (let ((hi (lambda* (a . 0.0) a))) (hi)) 'error)
      (test (let ((hi (lambda* ((a . 0.0)) a))) (hi)) 'error)
      (test (let ((hi (lambda* ((a 0.0 "hi")) a))) (hi)) 'error)
      (test (let ((hi (lambda* ((a 0.0 . "hi")) a))) (hi)) 'error)
      (test (let ((hi (lambda* ((a)) a))) (hi)) 'error)
      (test (let ((hi (lambda* (a 0.0) (b 0.0) (+ a b)))) (hi)) 'error)

      ;; make sure #: also works
      (let ((hi (lambda* (a #:optional (b 1)) (+ a b))))
	(num-test (hi 1 2) 3)
	(num-test (hi 1 :b 2) 3)
	(num-test (hi 1) 2)
	(num-test (hi 1 #:b 2) 3))

      ))

(if with-procedure-arity ; s7-specific stuff that needs exercise
    (begin

      (test (procedure-arity car) '(1 0 #f))
      (test (procedure-arity 'car) '(1 0 #f))
      (test (procedure-arity +) '(0 0 #t))
      (test (procedure-arity '+) '(0 0 #t))
      (test (procedure-arity log) '(1 1 #f))
      (test (procedure-arity '/) '(1 0 #t))
      ;(test (procedure-arity vector-set!) '(3 0 #f)) ; can be '(3 0 #t)
      (test (let ((hi (lambda () 1))) (procedure-arity hi)) '(0 0 #f))
      (test (let ((hi (lambda (a) 1))) (procedure-arity hi)) '(1 0 #f))
      (test (let ((hi (lambda (a b) 1))) (procedure-arity hi)) '(2 0 #f))
      (test (let ((hi (lambda (a . b) 1))) (procedure-arity hi)) '(1 0 #t))
      (test (let ((hi (lambda a 1))) (procedure-arity hi)) '(0 0 #t))

      (test (let () (define (hi) 1) (procedure-arity hi)) '(0 0 #f))
      (test (let () (define (hi a) a) (procedure-arity hi)) '(1 0 #f))
      (test (let () (define* (hi a) a) (procedure-arity hi)) '(0 1 #f))
      (test (let () (define* (hi a . b) a) (procedure-arity hi)) '(0 1 #t))
      (test (let () (define* (hi (a 1) (b 2)) a) (procedure-arity hi)) '(0 2 #f))
      (test (let ((hi (lambda* (a) 1))) (procedure-arity hi)) '(0 1 #f))
      (test (call/cc (lambda (func) (procedure-arity func))) '(0 0 #t))

      (for-each
       (lambda (arg)
	 (test (procedure-arity arg) 'error))
       (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

      (test (string=? (let () (define (hi) "this is a string" 1) (procedure-documentation hi)) "this is a string") #t)

      (for-each
       (lambda (arg)
	 (test (procedure-environment arg) 'error))
       (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

      (for-each
       (lambda (arg)
	 (test (continuation? arg) #f))
       (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2) "hi" (lambda () 1)))

      (test (let ((cont #f)) 
	      (and (call/cc (lambda (x) (set! cont x) (continuation? x)))
		   (continuation? cont)))
	    #t)

      (test (string? (s7-version)) #t)
      (test (eval-string "(+ 1 2)") 3)
      (test (eval '(+ 1 2)) 3)
      (test (eval-string (string-append "(list 1 2 3)" (string #\newline) (string #\newline))) (list 1 2 3))
      (eval-string (string-append "(define evalstr_1 32)" (string #\newline) "(define evalstr_2 2)"))
      (test (eval-string "(+ evalstr_1 evalstr_2)") 34)
      (eval-string (string-append "(set! evalstr_1 3)" "(set! evalstr_2 12)"))
      (test (eval-string "(+ evalstr_1 evalstr_2)") 15)

      (test (+ (eval `(values 1 2 3)) 4) 10)
      (test (+ (eval-string "(values 1 2 3)") 4) 10)
      (test (+ 1 (eval-string "(+ 2 3)") 4) 10)

      (test (string=? (procedure-documentation abs) "(abs x) returns the absolute value of the real number x") #t)
      (test (string=? (procedure-documentation 'abs) "(abs x) returns the absolute value of the real number x") #t)
      (test (let ((hi (lambda (x) "this is a test" (+ x 1)))) 
	      (list (hi 1) (procedure-documentation hi)))
	    (list 2 "this is a test"))
      
      (for-each
       (lambda (arg)
	 (test (procedure-documentation arg) 'error))
       (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

      (test (let ((hi (lambda (x) (+ x 1)))) (procedure-source hi)) '(lambda (x) (+ x 1)))

      (for-each
       (lambda (arg)
	 (test (procedure-source arg) 'error))
       (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

      (test (make-list 0) '())
      (test (make-list 0 123) '())
      (test (make-list 1) '(#f))
      (test (make-list 1 123) '(123))
      (test (make-list 1 '()) '(()))
      (test (make-list 2) '(#f #f))
      (test (make-list 2 1) '(1 1))
      (test (make-list -1) 'error)

      (for-each
       (lambda (arg)
	 (test (make-list arg) 'error))
       (list #\a '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

      (test (let () (defmacro hiho (a) `(+ ,a 1)) (hiho 3)) 4)
      (test (let () (defmacro hiho () `(+ 3 1)) (hiho)) 4)
      (test (let () (defmacro hiho () `(+ 3 1)) (hiho 1)) 'error)
      (test (let () (defmacro hi (a) `(+ ,@a)) (hi (1 2 3))) 6)
      (test (let () (defmacro hi (a) `(+ ,a 1) #f) (hi 2)) #f)

      (test (let () (define-macro (hiho a) `(+ ,a 1)) (hiho 3)) 4)
      (test (let () (define-macro (hiho) `(+ 3 1)) (hiho)) 4)
      (test (let () (define-macro (hiho) `(+ 3 1)) (hiho 1)) 'error)
      (test (let () (define-macro (hi a) `(+ ,@a)) (hi (1 2 3))) 6)
      (test (let () (define-macro (hi a) `(+ ,a 1) #f) (hi 2)) #f)
      (test (let () (define-macro (mac1 a) `',a) (equal? (mac1 (+ 1 2)) '(+ 1 2))) #t)


      (define-macro* (_mac1_) `(+ 1 2))
      (test (_mac1_) 3)
      (define-macro* (_mac2_ a) `(+ ,a 2))
      (test (_mac2_ 1) 3)
      (test (_mac2_ :a 2) 4)
      (define-macro* (_mac3_ (a 1)) `(+ ,a 2))
      (test (_mac3_) 3)
      (test (_mac3_ 3) 5)
      (test (_mac3_ :a 0) 2)
      (define-macro* (_mac4_ (a 1) (b 2)) `(+ ,a ,b))
      (test (_mac4_) 3)
      (test (_mac4_ :b 3) 4)
      (test (_mac4_ 2 :b 3) 5)
      (test (_mac4_ :b 10 :a 12) 22)
      (test (_mac4_ :a 4) 6)
      
      
      (defmacro* _mac11_ () `(+ 1 2))
      (test (_mac11_) 3)
      (defmacro* _mac12_ (a) `(+ ,a 2))
      (test (_mac12_ 1) 3)
      (test (_mac12_ :a 2) 4)
      (defmacro* _mac13_ ((a 1)) `(+ ,a 2))
      (test (_mac13_) 3)
      (test (_mac13_ 3) 5)
      (test (_mac13_ :a 0) 2)
      (defmacro* _mac14_ ((a 1) (b 2)) `(+ ,a ,b))
      (test (_mac14_) 3)
      (test (_mac14_ :b 3) 4)
      (test (_mac14_ 2 :b 3) 5)
      (test (_mac14_ :b 10 :a 12) 22)
      (test (_mac14_ :a 4) 6)
      
      (begin
	(define-macro (hi a) `(+ ,a 1))
	(test (hi 2) 3)
	(let ()
	  (define (ho b) (+ 1 (hi b)))
	  (test (ho 1) 3))
	(let ((hi 32))
	  (test (+ hi 1) 33))
	(letrec ((hi (lambda (a) (if (= a 0) 0 (+ 2 (hi (- a 1)))))))
	  (test (hi 3) 6))
	(test (equal? '(hi 1) (quote (hi 1))) #t)
	(test (list? '(hi 1)) #t)
	(test (list? '(((hi 1)))) #t)
	(test (equal? (vector (hi 1)) '#(2)) #t)
	(test (symbol? (vector-ref '#(hi) 0)) #t))

      (test (let () (define-constant __c1__ 32) __c1__) 32)
      (test (let () __c1__) 'error)
      (test (let ((__c1__ 3)) __c1__) 'error)
      (test (let* ((__c1__ 3)) __c1__) 'error)
      (test (letrec ((__c1__ 3)) __c1__) 'error)
      (test (let () (define (__c1__ a) a) (__c1__ 3)) 'error)
      (test (let () (set! __c1__ 3)) 'error)
      (test (constant? '__c1__) #t)
      (test (constant? 'abs) #f)
      (test (constant? '*features*) #f)

      (test (defined? 'pi) #t)
      (test (defined? 'pi (global-environment)) #t)
      (test (defined? 'abs (global-environment)) #t)
      (test (defined? 'abs (current-environment)) #t)
      (test (let ((__c2__ 32)) (defined? '__c2__)) #t)
      (test (let ((__c2__ 32)) (defined? '__c2__ (current-environment))) #t)
      (test (let ((__c2__ 32)) (defined? '__c3__ (current-environment))) #f)
      (test (let ((__c2__ 32)) (defined? '__c2__ (global-environment))) #f)
      (test (let ((__c2__ 32)) (defined? '__c3__ (global-environment))) #f)

      (test (call-with-exit (lambda (c) (0 (c 1)))) 1)
      (test (call-with-exit (lambda (k) (k "foo"))) "foo")
      (test (call-with-exit (lambda (k) "foo")) "foo")
      (test (call-with-exit (lambda (k) (k "foo") "oops")) "foo")
      (test (let ((memb (lambda (x ls)
			  (call-with-exit
			   (lambda (break)
			     (do ((ls ls (cdr ls)))
				 ((null? ls) #f)
			       (if (equal? x (car ls))
				   (break ls))))))))
	      (list (memb 'd '(a b c))
		    (memb 'b '(a b c))))
	    '(#f (b c)))

      (test (string=? (let ((hi (lambda (b) (+ b 1)))) (object->string hi)) "hi") #t)
      (test (string=? (object->string 32) "32") #t)
      (test (string=? (object->string 32.5) "32.5") #t)
      (test (string=? (object->string 32/5) "32/5") #t)
      (test (string=? (object->string "hiho") "\"hiho\"") #t)
      (test (string=? (object->string 'symb) "symb") #t)
      (test (string=? (object->string (list 1 2 3)) "(1 2 3)") #t)
      (test (string=? (object->string (cons 1 2)) "(1 . 2)") #t)
      (test (string=? (object->string '#(1 2 3)) "#(1 2 3)") #t)
      (test (string=? (object->string +) "+") #t)

      (test (let ((l (list 1 2))) 
	      (list-set! l 0 l) 
	      (string=? (object->string l) "([circular list] 2)")) 
	    #t)
      (test (let ((lst (cons 1 2))) 
	      (set-cdr! lst lst)
	      (string=? (object->string lst) "[circular list]"))
	    #t)
      (test (let ((lst (cons 1 2))) 
	      (set-car! lst lst)
	      (string=? (object->string lst) "([circular list] . 2)"))
	    #t)
      (test (let ((lst (cons (cons 1 2) 3))) 
	      (set-car! (car lst) lst)
	      (string=? (object->string lst) "(([circular list] . 2) . 3)"))
	    #t)
      (test (let ((v (vector 1 2))) 
	      (vector-set! v 0 v) 
	      (string=? (object->string v) "#([circular vector] 2)")) 
	    #t)
      (test (let* ((l1 (list 1 2)) (l2 (list l1))) 
	      (list-set! l1 0 l1) 
	      (string=? (object->string l2) "(([circular list] 2))")) 
	    #t)
      (test (let* ((v1 (vector 1 2)) (v2 (vector v1))) 
	      (vector-set! v1 1 v1) 
	      (string=? (object->string v2) "#(#(1 [circular vector]))")) 
	    #t)
      (test (let ((v1 (make-vector 3 1))) 
	      (vector-set! v1 0 (cons 3 v1)) 
	      (string=? (object->string v1) "#((3 . [circular vector]) 1 1)")) 
	    #t)
      (test (let ((h1 (make-hash-table 11))) 
	      (hash-table-set! h1 'hi h1)
	      (string=? (object->string h1) "#(() () () () ((\"hi\" . [circular hash-table])) () () () () () ())"))
	    #t)
		 

      (test (let* ((l1 (list 1 2))
		   (v1 (vector 1 2))
		   (l2 (list 1 l1 2))
		   (v2 (vector l1 v1 l2)))
	      (vector-set! v1 0 v2)
	      (list-set! l1 1 l2)
	      (string=? (object->string v2) "#((1 (1 [circular list] 2)) #([circular vector] 2) (1 (1 [circular list]) 2))"))
	    #t)

      (test (multiple-value-bind (a b) (values 1 2) (+ a b)) 3)
      (test (multiple-value-bind (a) 1 a) 1)
      (test (multiple-value-bind (a . rest) (values 1 2 3) (+ a (apply + rest))) 6)
      (test (multiple-value-bind a (values 1 2 3) a) '(1 2 3))

      (test (let ((a 1)
		  (b 2))
	      (multiple-value-set! (a b) (values 32 64))
	      (+ a b))
	    96)

      (test (call-with-input-file "tmp1.r5rs" (lambda (p) (integer->char (read-byte p)))) #\t)

      (if (and (defined? 'provided?)
	       (provided? 'threads))
	  (begin

	    (test (let ((ctr 0))
		    (let ((t1 (make-thread (lambda () (set! ctr (+ ctr 1))))))
		      (join-thread t1))
		    ctr)
		  1)
	    
	    (test (let ((ctr 0))
		    (let ((t1 (make-thread (lambda () (set! ctr (+ ctr 1))))))
		      (join-thread t1)
		      (thread? t1)))
		  #t)
	    
	    (test (let ((ctr 0)
			(loc (make-thread-variable)))
		    (let ((t1 (make-thread (lambda () (set! (loc) (+ ctr 1)) (set! ctr (loc))))))
		      (join-thread t1)
		      ctr))
		  1)
	    
	    (test (let ((ctr 0)
			(loc (make-thread-variable)))
		    (let ((t1 (make-thread (lambda () (set! (loc) (+ ctr 1)) (set! ctr (thread-variable? loc))))))
		      (join-thread t1)
		      ctr))
		  #t)
	    
	    (test (let ((ctr 0)
			(lock (make-lock)))
		    (let ((t1 (make-thread (lambda () (grab-lock lock) (set! ctr (+ ctr 1)) (release-lock lock)))))
		      (join-thread t1))
		    ctr)
		  1)
	    
	    (test (let ((ctr 0)
			(lock (make-lock)))
		    (let ((t1 (make-thread (lambda () (grab-lock lock) (set! ctr (lock? lock)) (release-lock lock)))))
		      (join-thread t1))
		    ctr)
		  #t)
	    
	    (test (let ((ctr 0)
			(lock (make-lock)))
		    (let ((t1 (make-thread (lambda () (grab-lock lock) (set! ctr (+ ctr 1)) (release-lock lock))))
			  (t2 (make-thread (lambda () (grab-lock lock) (set! ctr (+ ctr 1)) (release-lock lock)))))
		      (join-thread t1)
		      (join-thread t2))
		    ctr)
		  2)
	    
	    (test (let ((ctr 0)
			(lock (make-lock)))
		    (let ((threads '()))
		      (do ((i 0 (+ 1 i)))
			  ((= i 8))
			(let ((t1 (make-thread (lambda () (grab-lock lock) (set! ctr (+ ctr 1)) (release-lock lock)))))
			  (set! threads (cons t1 threads))))
		      (for-each
		       (lambda (tn)
			 (join-thread tn))
		       threads))
		    ctr)
		  8)
	    
	    (test (let ((ctr 0)
			(ctr1 0)
			(ctr2 0)
			(lock (make-lock))
			(var (make-thread-variable)))
		    (let ((t1 (make-thread (lambda () (grab-lock lock) (set! ctr (+ ctr 1)) (set! (var) ctr) (release-lock lock) (set! ctr1 (var)))))
			  (t2 (make-thread (lambda () (grab-lock lock) (set! ctr (+ ctr 1)) (set! (var) ctr) (release-lock lock) (set! ctr2 (var))))))
		      (join-thread t1)
		      (join-thread t2))
		    (and (= ctr 2)
			 (= (+ ctr1 ctr2) 3)))
		  #t)

	    (for-each
	     (lambda (arg)
	       (test (thread? arg) #f))
	     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" abs '#(()) (list 1 2 3) '(1 . 2) (lambda () 1)))
	    
	    (for-each
	     (lambda (arg)
	       (test (lock? arg) #f))
	     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" abs '#(()) (list 1 2 3) '(1 . 2) (lambda () 1)))
	    
	    (for-each
	     (lambda (arg)
	       (test (thread-variable? arg) #f))
	     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" abs '#(()) (list 1 2 3) '(1 . 2) (lambda () 1)))
	    
	    (for-each
	     (lambda (arg)
	       (test (make-thread arg) 'error))
	     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" '#(()) (list 1 2 3) '(1 . 2)))
	    
	    (for-each
	     (lambda (arg)
	       (test (grab-lock arg) 'error))
	     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" '#(()) (list 1 2 3) '(1 . 2)))
	    
	    (for-each
	     (lambda (arg)
	       (test (release-lock arg) 'error))
	     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" '#(()) (list 1 2 3) '(1 . 2)))))
	    
      (let ((v (make-vector '(2 3) 0)))
	(num-test (vector-length v) 6)
	(test (vector-dimensions v) '(2 3))
	(num-test (v 0 0) 0)
	(num-test (v 1 2) 0)
	(test (v 2 2) 'error)
	(test (v 2 -1) 'error)
	(test (v 2 0) 'error)
	(set! (v 0 1) 1)
	(num-test (v 0 1) 1)
	(num-test (v 1 0) 0)
	(set! (v 1 2) 2)
	(num-test (v 1 2) 2)
	(test (set! (v 2 2) 32) 'error)
	(test (set! (v 1 -1) 0) 'error)
	(test (set! (v 2 0) 0) 'error)
	(num-test (vector-ref v 0 1) 1)
	(num-test (vector-ref v 1 2) 2)
	(test (vector-ref v 2 2) 'error)
	(test (vector-ref v 1 -1) 'error)
	(vector-set! v 1 1 64)
	(num-test (vector-ref v 1 1) 64)
	(num-test (vector-ref v 0 0) 0)
	(test (vector-ref v 1 2 3) 'error)
	(test (vector-set! v 1 2 3 4) 'error)
	(test (v 1 1 1) 'error)
	(test (set! (v 1 1 1) 1) 'error))
      
      (let ((v1 (make-vector '(3 2) 0))
	    (v2 (make-vector '(2 3) 0))
	    (v3 (make-vector '(2 3 4) 0))
	    (v4 (make-vector 6 0))
	    (v5 (make-vector '(2 3) 0)))
	(test (equal? v1 v2) #f)
	(test (equal? v1 v3) #f)
	(test (equal? v1 v4) #f)
	(test (equal? v2 v2) #t)
	(test (equal? v3 v2) #f)
	(test (equal? v4 v2) #f)
	(test (equal? v5 v2) #t)
	(test (equal? v4 v3) #f)
	(test (vector-dimensions v3) '(2 3 4))
	(test (vector-dimensions v4) '(6))
	(num-test (v3 1 2 3) 0)
	(set! (v3 1 2 3) 32)
	(num-test (v3 1 2 3) 32)
	(num-test (vector-length v3) 24)
	(num-test (vector-ref v3 1 2 3) 32)
	(vector-set! v3 1 2 3 -32)
	(num-test (v3 1 2 3) -32)
	(test (v3 1 2) 'error)
	(test (set! (v3 1 2) 3) 'error)
	(test (vector-ref v3 1 2) 'error)
	(test (vector-set! v3 1 2 32) 'error))

      (test (#(1 2) 1) 2)
      (test (#(1 2) 1 2) 'error)
      
      (test (constant? pi) #t)
      (test (constant? 'pi) #t) ; take that, Clisp!
      (test (constant? 12345) #t)
      (test (constant? 3.14) #t)
      (test (constant? :asdf) #t) 
      (test (constant? "hi") #t) 
      (test (constant? #\a) #t) 
      (test (constant? #f) #t) 
      (test (constant? #t) #t) 
      (test (constant? '()) #t) 
      (test (constant? '(a)) #t) 
      (test (constant? '*features*) #f)
      (test (let ((a 3)) (constant? 'a)) #f)
      (test (constant? 'abs) #f)
      (test (constant? abs) #t)

      (test (apply "hi" '(1 2)) 'error)
      (test ("hi" 1 2) 'error)
      (test (apply '(1 2) '(1 2)) 'error)
      (test ((list 1 2 3) 1 2) 'error)

      (test (apply "hi" '(1)) #\i)
      (test ("hi" 1) #\i)
      (test (apply '(1 2) '(1)) 2)
      (test ((list 1 2 3) 1) 2)
      
      (test (let ((pi 3)) pi) 'error)
      ;;   or ... (let ((:asdf 3)) :asdf) and worse (let ((:key 1)) :key) or even worse (let ((:3 1)) 1)
      (test (let ((x_x_x 32)) (let () (define-constant x_x_x 3) x_x_x) (set! x_x_x 31) x_x_x) 'error)
      

      (test (with-environment (current-environment) (let ((x 1)) x)) 1)

      (test (let ((x 12))
	      (let ((e (current-environment)))
		(let ((x 32))
		  (with-environment e (* x 2)))))
	    24)

      (test (let ()
	      (define (make-environment . initial-bindings)
		(cons initial-bindings (global-environment)))
	      (with-environment (make-environment '(x . 2) '(y . 4))
		(+ x y)))
	    6)

      (test (let ((*features* 123))
	      (let ((e (global-environment)))
		(with-environment e (list? *features*))))
	    #t)

      (test (let ()
        (define-macro (mac a b) 
          `(with-environment 
             (cons (list (cons 'a ,a) 
                         (cons 'b ,b))
                   (global-environment))
             (+ a b)))
	(let ((+ -))
	  (mac 1 2)))
	    3)

      (test (let ((local 123))
	      (define pws-test (make-procedure-with-setter
				(lambda () local)
				(lambda (val) (set! local val))))
	      (pws-test))
	    123)

      (test (call/cc (lambda (return) (let ((local 123))
					(define pws-test (make-procedure-with-setter
							  (lambda () (return "oops"))
							  (lambda (val) (set! local val))))
					(pws-test))))
	    "oops")
      (test (call/cc (lambda (return) (let ((local 123))
					(define pws-test (make-procedure-with-setter
							  (lambda () 123)
							  (lambda (val) (return "oops"))))
					(set! (pws-test) 1))))
	    "oops")

      (test (let ((local 123))
	      (define pws-test (make-procedure-with-setter
				(lambda () local)
				(lambda (val) (set! local val))))
	      (set! (pws-test) 321)
	      (pws-test))
	    321)

      (if (defined? 'open-encapsulator) 
	  (begin
	    ;; encapsulator tests
	    (let ((gi 32)
		  (gstr (make-string 8 #\x))
		  (gx 3.14)
		  (gcp 1.5+i)
		  (gl (list 1 2 3))
		  (gv (vector 1 2 3))
		  (gc (cons 1 2))
		  (gr 3/4)
		  (gf (lambda () 123)))
	      (encapsulate
	       (set! gi 3)
	       (string-set! gstr 3 #\A)
	       (set! gx 2.1718)
	       (set! gcp -1.0-i)
	       (list-set! gl 1 32)
	       (set! gc "not a cons")
	       (set! gr (list 3/4))
	       (set! gi 1) ; see if duplicates intrude
	       (set! gstr "hiho")
	       (set! gf #f))
	      (test gi 32)
	      (test gstr "xxxxxxxx")
	      (test gx 3.14)
	      (test gcp 1.5+i)
	      (test gl (list 1 2 3))
	      (test gv (vector 1 2 3))
	      (test gc (cons 1 2))
	      (test gr 3/4)
	      (test (gf) 123))
	    
	    (let ((gsym 123))
	      (let ((e (open-encapsulator)))
		(set! gsym 321)
		(test (e 'gsym) 123)
		(test (encapsulator? e) #t)
		(test (encapsulator? 123) #f)
		(test (encapsulator-bindings e) (list (cons 'gsym 123)))
		(e)
		(test gsym 123)
		(close-encapsulator e)))
	    
	    (let ((g1 1))
	      (let ((e1 (open-encapsulator)))
		(let ((g2 2))
		  (let ((e2 (open-encapsulator)))
		    (let ((g3 3))
		      (set! g2 32)
		      (set! g1 43)
		      (set! g3 54)
		      (test (encapsulator-bindings e1) (list (cons 'g1 1)))
		      (test (encapsulator-bindings e2) (list (cons 'g1 1) (cons 'g2 2))))
		    (test (equal? e1 e2) #f)
		    (test (equal? e1 e1) #t)
		    (e1)
		    (close-encapsulator e2))
		  (test g2 32)
		  (test g1 1)
		  (close-encapsulator e1))))

	    (let ((g1 321))
	      (define (set-g1 val) (set! g1 val))
	      (let ((e (open-encapsulator)))
		(set-g1 1)
		(e)
		(test g1 321)
		(set-g1 2)
		(test g1 2)
		(e)
		(test g1 321)
		(close-encapsulator e)))

	    ))
		
      (for-each
       (lambda (arg)
	 (test (port-filename arg) 'error))
       (list "hi" -1 #\a 1 0 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t '() (list 1 2 3) '(1 . 2)))

      (for-each
       (lambda (arg)
	 (test (port-line-number arg) 'error))
       (list "hi" -1 #\a 1 0 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t '() (list 1 2 3) '(1 . 2)))

      ;; generic length/copy/fill!
      (test (length (list 1 2)) 2)
      (test (length "hiho") 4)
      (test (length (vector 1 2)) 2)
      (test (length (make-hash-table 7)) 7)
      (test (length '()) 0)
      
      (test (copy 3) 3)
      (test (copy 3/4) 3/4)
      (test (copy "hi") "hi")
      (test (copy (list 1 2 3)) (list 1 2 3))
      (test (copy (vector 0.0)) (vector 0.0))
      (test (copy #\f) #\f)
      (test (copy (list 1 (list 2 3))) (list 1 (list 2 3)))
      (test (copy (cons 1 2)) (cons 1 2))
      (test (copy '(1 2 . 3)) '(1 2 . 3))
      
      (if (not (provided? 'gmp))
	  (let ((r1 (make-random-state 1234)))
	    (random 1.0 r1)
	    (let ((r2 (copy r1)))
	      (let ((v1 (random 1.0 r1))
		    (v2 (random 1.0 r2)))
		(test (= v1 v2) #t)
		(let ((v3 (random 1.0 r1)))
		  (random 1.0 r1)
		  (random 1.0 r1)
		  (let ((v4 (random 1.0 r2)))
		    (test (= v3 v4) #t)))))))

      (if (provided? 'gmp)
	  (let ((i (copy (bignum "1")))
		(r (copy (bignum "3/4")))
		(f (copy (bignum "1.5")))
		(c (copy (bignum "1.0+1.0i"))))
	    (test (= i (bignum "1")) #t)
	    (test (= r (bignum "3/4")) #t)
	    (test (= f (bignum "1.5")) #t)
	    (test (= c (bignum "1.0+1.0i")) #t)))

      (let ((str (string #\1 #\2 #\3)))
	(fill! str #\x)
	(test str "xxx"))
      (let ((v (vector 1 2 3)))
	(fill! v 0.0)
	(test v (vector 0.0 0.0 0.0)))
      (let ((lst (list 1 2 (list (list 3) 4))))
	(fill! lst 100)
	(test lst '(100 100 ((100) 100))))
      (let ((cn (cons 1 2)))
	(fill! cn 100)
	(test cn (cons 100 100)))

      ;; generic for-each/map
      (test (let ((sum 0)) (for-each (lambda (n) (set! sum (+ sum n))) (vector 1 2 3)) sum) 6)      
      (test (map (lambda (n) (+ n 1)) (vector 1 2 3)) '#(2 3 4))

      ;; try some applicable stuff
      (test (let ((lst (list 1 2 3)))
	      (set! (lst 1) 32)
	      (list (lst 0) (lst 1)))
	    (list 1 32))

      (test (let ((hash (make-hash-table)))
	      (set! (hash 'hi) 32)
	      (hash 'hi))
	    32)

      (test (let ((str (string #\1 #\2 #\3)))
	      (set! (str 1) #\a)
	      (str 1))
	    #\a)

      (test (let ((v (vector 1 2 3)))
	      (set! (v 1) 0)
	      (v 1))
	    0)

      (let ()
	(define (hiho a) __func__)
	(test (or (equal? (hiho 1) 'hiho)
		  (equal? (car (hiho 1)) 'hiho))
	      #t))

      (num-test (log 8 2) 3)
      (num-test (log -1 -1) 1.0)
      (num-test (log 1 1) 'error)
      (num-test (log 1 -1) 0.0)
      (num-test (log 1.5 -1) 0-0.12906355241341i)

      (test (random 0 #t) 'error)
      (test (random 0.0 #(1 2)) 'error)

      ))




;;; --------------------------------------------------------------------------------
;;; NUMBERS
;;; --------------------------------------------------------------------------------


;; -------- sin
(num-test (sin 0) 0.0)
(num-test (sin 1) 0.84147098480790)
(num-test (sin -1) -0.84147098480790)
(num-test (sin 2) 0.90929742682568)
(num-test (sin -2) -0.90929742682568)
(num-test (sin 0/1) 0.0)
(num-test (sin 0/2) 0.0)
(num-test (sin 0/3) 0.0)
(num-test (sin 0/10) 0.0)
(num-test (sin 0/1234) 0.0)
(num-test (sin 0/1234000000) 0.0)
(num-test (sin 0/500029) 0.0)
(num-test (sin 0/362880) 0.0)
(num-test (sin 1/1) 0.84147098480790)
(num-test (sin -1/1) -0.84147098480790)
(num-test (sin 1/2) 0.47942553860420)
(num-test (sin -1/2) -0.47942553860420)
(num-test (sin 1/3) 0.32719469679615)
(num-test (sin -1/3) -0.32719469679615)
(num-test (sin 1/10) 0.09983341664683)
(num-test (sin -1/10) -0.09983341664683)
(num-test (sin 1/1234) 0.00081037268278)
(num-test (sin -1/1234) -0.00081037268278)
(num-test (sin 1/1234000000) 0.00000000081037)
(num-test (sin -1/1234000000) -0.00000000081037)
(num-test (sin 1/500029) 0.00000199988401)
(num-test (sin -1/500029) -0.00000199988401)
(num-test (sin 1/362880) 0.00000275573192)
(num-test (sin -1/362880) -0.00000275573192)
(num-test (sin 2/1) 0.90929742682568)
(num-test (sin -2/1) -0.90929742682568)
(num-test (sin 2/2) 0.84147098480790)
(num-test (sin -2/2) -0.84147098480790)
(num-test (sin 2/3) 0.61836980306974)
(num-test (sin -2/3) -0.61836980306974)
(num-test (sin 2/10) 0.19866933079506)
(num-test (sin -2/10) -0.19866933079506)
(num-test (sin 2/1234) 0.00162074483338)
(num-test (sin -2/1234) -0.00162074483338)
(num-test (sin 2/1234000000) 0.00000000162075)
(num-test (sin -2/1234000000) -0.00000000162075)
(num-test (sin 2/500029) 0.00000399976801)
(num-test (sin -2/500029) -0.00000399976801)
(num-test (sin 2/362880) 0.00000551146384)
(num-test (sin -2/362880) -0.00000551146384)
(num-test (sin 3/2) 0.99749498660405)
(num-test (sin -3/2) -0.99749498660405)
(num-test (sin 3/3) 0.84147098480790)
(num-test (sin -3/3) -0.84147098480790)
(num-test (sin 3/10) 0.29552020666134)
(num-test (sin -3/10) -0.29552020666134)
(num-test (sin 3/1234) 0.00243111591964)
(num-test (sin -3/1234) -0.00243111591964)
(num-test (sin 3/1234000000) 0.00000000243112)
(num-test (sin -3/1234000000) -0.00000000243112)
(num-test (sin 3/500029) 0.00000599965202)
(num-test (sin -3/500029) -0.00000599965202)
(num-test (sin 3/362880) 0.00000826719577)
(num-test (sin -3/362880) -0.00000826719577)
(num-test (sin 10/10) 0.84147098480790)
(num-test (sin -10/10) -0.84147098480790)
(num-test (sin 10/1234) 0.00810363901920)
(num-test (sin -10/1234) -0.00810363901920)
(num-test (sin 10/1234000000) 0.00000000810373)
(num-test (sin -10/1234000000) -0.00000000810373)
(num-test (sin 10/500029) 0.00001999884007)
(num-test (sin -10/500029) -0.00001999884007)
(num-test (sin 10/362880) 0.00002755731922)
(num-test (sin -10/362880) -0.00002755731922)
(num-test (sin 1234/1234) 0.84147098480790)
(num-test (sin -1234/1234) -0.84147098480790)
(num-test (sin 1234/1234000000) 0.00000100000000)
(num-test (sin -1234/1234000000) -0.00000100000000)
(num-test (sin 1234/500029) 0.00246785435930)
(num-test (sin -1234/500029) -0.00246785435930)
(num-test (sin 1234/362880) 0.00340056663826)
(num-test (sin -1234/362880) -0.00340056663826)
(num-test (sin 1234000000/1234000000) 0.84147098480790)
(num-test (sin -1234000000/1234000000) -0.84147098480790)
(num-test (sin 500029/1234000000) 0.00040520987546)
(num-test (sin -500029/1234000000) -0.00040520987546)
(num-test (sin 500029/500029) 0.84147098480790)
(num-test (sin -500029/500029) -0.84147098480790)
(num-test (sin 500029/362880) 0.98146191370834)
(num-test (sin -500029/362880) -0.98146191370834)
(num-test (sin 362880/1234000000) 0.00029406806707)
(num-test (sin -362880/1234000000) -0.00029406806707)
(num-test (sin 362880/500029) 0.66367262572318)
(num-test (sin -362880/500029) -0.66367262572318)
(num-test (sin 362880/362880) 0.84147098480790)
(num-test (sin -362880/362880) -0.84147098480790)
(num-test (sin 0.0) 0.0)
(num-test (sin 0.00000001) 0.00000001)
(num-test (sin -0.00000001) -0.00000001)
(num-test (sin 1.0) 0.84147098480790)
(num-test (sin -1.0) -0.84147098480790)
(num-test (sin 0.0+0.0i) 0.0)
(num-test (sin -0.0+0.0i) 0.0)
(num-test (sin 0.0-0.0i) 0.0)
(num-test (sin -0.0-0.0i) -0.0)
(num-test (sin 0.0+0.00000001i) 0.0+0.00000001i)
(num-test (sin -0.0+0.00000001i) 0.0+0.00000001i)
(num-test (sin 0.0-0.00000001i) 0.0-0.00000001i)
(num-test (sin -0.0-0.00000001i) -0.0-0.00000001i)
(num-test (sin 0.0+1.0i) 0.0+1.17520119364380i)
(num-test (sin -0.0+1.0i) 0.0+1.17520119364380i)
(num-test (sin 0.0-1.0i) 0.0-1.17520119364380i)
(num-test (sin -0.0-1.0i) -0.0-1.17520119364380i)
(num-test (sin 0.0+3.14159265358979i) 0.0+11.54873935725775i)
(num-test (sin -0.0+3.14159265358979i) 0.0+11.54873935725775i)
(num-test (sin 0.0-3.14159265358979i) 0.0-11.54873935725775i)
(num-test (sin -0.0-3.14159265358979i) -0.0-11.54873935725775i)
(num-test (sin 0.0+2.71828182845905i) 0.0+7.54413710281697i)
(num-test (sin -0.0+2.71828182845905i) 0.0+7.54413710281697i)
(num-test (sin 0.0-2.71828182845905i) 0.0-7.54413710281697i)
(num-test (sin -0.0-2.71828182845905i) -0.0-7.54413710281697i)
(num-test (sin 0.00000001+0.0i) 0.00000001)
(num-test (sin -0.00000001+0.0i) -0.00000001)
(num-test (sin 0.00000001-0.0i) 0.00000001)
(num-test (sin -0.00000001-0.0i) -0.00000001)
(num-test (sin 0.00000001+0.00000001i) 0.00000001+0.00000001i)
(num-test (sin -0.00000001+0.00000001i) -0.00000001+0.00000001i)
(num-test (sin 0.00000001-0.00000001i) 0.00000001-0.00000001i)
(num-test (sin -0.00000001-0.00000001i) -0.00000001-0.00000001i)
(num-test (sin 0.00000001+1.0i) 0.00000001543081+1.17520119364380i)
(num-test (sin -0.00000001+1.0i) -0.00000001543081+1.17520119364380i)
(num-test (sin 0.00000001-1.0i) 0.00000001543081-1.17520119364380i)
(num-test (sin -0.00000001-1.0i) -0.00000001543081-1.17520119364380i)
(num-test (sin 0.00000001+3.14159265358979i) 0.00000011591953+11.54873935725775i)
(num-test (sin -0.00000001+3.14159265358979i) -0.00000011591953+11.54873935725775i)
(num-test (sin 0.00000001-3.14159265358979i) 0.00000011591953-11.54873935725775i)
(num-test (sin -0.00000001-3.14159265358979i) -0.00000011591953-11.54873935725775i)
(num-test (sin 0.00000001+2.71828182845905i) 0.00000007610125+7.54413710281697i)
(num-test (sin -0.00000001+2.71828182845905i) -0.00000007610125+7.54413710281697i)
(num-test (sin 0.00000001-2.71828182845905i) 0.00000007610125-7.54413710281697i)
(num-test (sin -0.00000001-2.71828182845905i) -0.00000007610125-7.54413710281697i)
(num-test (sin 1.0+0.0i) 0.84147098480790)
(num-test (sin -1.0+0.0i) -0.84147098480790)
(num-test (sin 1.0-0.0i) 0.84147098480790)
(num-test (sin -1.0-0.0i) -0.84147098480790)
(num-test (sin 1.0+0.00000001i) 0.84147098480790+0.00000000540302i)
(num-test (sin -1.0+0.00000001i) -0.84147098480790+0.00000000540302i)
(num-test (sin 1.0-0.00000001i) 0.84147098480790-0.00000000540302i)
(num-test (sin -1.0-0.00000001i) -0.84147098480790-0.00000000540302i)
(num-test (sin 1.0+1.0i) 1.29845758141598+0.63496391478474i)
(num-test (sin -1.0+1.0i) -1.29845758141598+0.63496391478474i)
(num-test (sin 1.0-1.0i) 1.29845758141598-0.63496391478474i)
(num-test (sin -1.0-1.0i) -1.29845758141598-0.63496391478474i)
(num-test (sin 1.0+3.14159265358979i) 9.75429233860021+6.23981050459650i)
(num-test (sin -1.0+3.14159265358979i) -9.75429233860021+6.23981050459650i)
(num-test (sin 1.0-3.14159265358979i) 9.75429233860021-6.23981050459650i)
(num-test (sin -1.0-3.14159265358979i) -9.75429233860021-6.23981050459650i)
(num-test (sin 1.0+2.71828182845905i) 6.40369949494148+4.07611467243740i)
(num-test (sin -1.0+2.71828182845905i) -6.40369949494148+4.07611467243740i)
(num-test (sin 1.0-2.71828182845905i) 6.40369949494148-4.07611467243740i)
(num-test (sin -1.0-2.71828182845905i) -6.40369949494148-4.07611467243740i)
(num-test (sin 10/3) -0.1905679628754527) 
(num-test (sin 1234/3) 0.213644699569724) 
(num-test (sin 1234/10) -0.7693905459455223) 
(num-test (sin 1234000000/3) 9.98585468017658e-1)
(num-test (sin 1234000000/500029) -0.9907886154453116) 
(num-test (sin 1234000000/362880) .9798963032808383) 
(num-test (sin 500029/2) 0.270047165973401) 
(num-test (sin 500029/3) 7.610322596690986e-1)
(num-test (sin 500029/10) .9665258739436294) 
(num-test (sin 500029/1234) .05553717596791147) 
(num-test (sin 362880/1234) -0.9463147870254296) 
(num-test (sin our-pi) -6.982889851335445E-15) 
(num-test (sin 2.71828182845905) .4107812905029501) 
(num-test (sin 3.14159265358979+0.0i) -6.982889851335445E-15) 
(num-test (sin 3.14159265358979+0.00000001i) -6.982889851335445E-15-1.0E-8i) 
(num-test (sin 3.14159265358979+1.0i) -1.077516210464362E-14-1.175201193643801i) 
(num-test (sin 3.14159265358979+3.14159265358979i) -8.094533288479446E-14-11.54873935725783i) 
(num-test (sin 3.14159265358979+2.71828182845905i) -5.314066559815525E-14-7.54413710281663i) 
(num-test (sin 2.71828182845905+0.0i) .4107812905029501) 
(num-test (sin 2.71828182845905+0.00000001i) .4107812905029501-9.117339147869465E-9i) 
(num-test (sin 2.71828182845905+1.0i) .6338686545195173-1.071470784943156i) 
(num-test (sin 2.71828182845905+3.14159265358979i) 4.761757525968664-10.52937734504676i) 
(num-test (sin 2.71828182845905+2.71828182845905i) 3.126097025348496-6.878245654440458i) 
(num-test (sin 1234.0+0.00000001i) .6019276547624973-7.985506235875843E-9i) 
(num-test (sin 1234.0+3.14159265358979i) 6.977517249251167-9.222253015388718i) 
(num-test (sin 1234.0+2.71828182845905i) 4.58074477716391-6.024375387884452i) 
(num-test (sin 1234000000.0+0.00000001i) -0.9872932128398908+1.5890913089022285E-9i) 
(num-test (sin 1234000000.0+3.14159265358979i) -11.44465679247962+1.835200134139553i) 
(num-test (sin 1234000000.0+2.71828182845905i) -7.513424898263172+1.198832270325275i) 
(num-test (sin 1234.0+12.0i) 48983.30495194942-64983.97008730317i)
(num-test (sin -3.45266983001243932001e-04+0.0e+00i) -3.4526697614140534807e-4+0.0i)
(num-test (sin 3.45266983001243932001e-04+0.0e+00i) 3.4526697614140534807e-4+0.0i)
(num-test (sin -3.45266983001243932001e-04+1.19209289550781250e-07i) -3.4526697614140780134e-4+1.1920928244535424533e-7i)
(num-test (sin -3.45266983001243932001e-04-1.19209289550781250e-07i) -3.4526697614140780134e-4-1.1920928244535424533e-7i)
(num-test (sin 3.45266983001243932001e-04+1.19209289550781250e-07i) 3.4526697614140780134e-4+1.1920928244535424533e-7i)
(num-test (sin 3.45266983001243932001e-04-1.19209289550781250e-07i) 3.4526697614140780134e-4-1.1920928244535424533e-7i)
(num-test (sin -3.45266983001243932001e-04+5.0e-01i) -3.8933200722534065172e-4+5.2109527443404709209e-1i)
(num-test (sin -3.45266983001243932001e-04-5.0e-01i) -3.8933200722534065172e-4-5.2109527443404709209e-1i)
(num-test (sin 3.45266983001243932001e-04+5.0e-01i) 3.8933200722534065172e-4+5.2109527443404709209e-1i)
(num-test (sin 3.45266983001243932001e-04-5.0e-01i) 3.8933200722534065172e-4-5.2109527443404709209e-1i)
(num-test (sin -3.45266983001243932001e-04+1.0e+00i) -5.3277478472501939236e-4+1.1752011235963524660e0i)
(num-test (sin -3.45266983001243932001e-04-1.0e+00i) -5.3277478472501939236e-4-1.1752011235963524660e0i)
(num-test (sin 3.45266983001243932001e-04+1.0e+00i) 5.3277478472501939236e-4+1.1752011235963524660e0i)
(num-test (sin 3.45266983001243932001e-04-1.0e+00i) 5.3277478472501939236e-4-1.1752011235963524660e0i)
(num-test (sin -3.45266983001243932001e-04+2.0e+00i) -1.2989619299126701883e-3+3.6268601916692946556e0i)
(num-test (sin -3.45266983001243932001e-04-2.0e+00i) -1.2989619299126701883e-3-3.6268601916692946556e0i)
(num-test (sin 3.45266983001243932001e-04+2.0e+00i) 1.2989619299126701883e-3+3.6268601916692946556e0i)
(num-test (sin 3.45266983001243932001e-04-2.0e+00i) 1.2989619299126701883e-3-3.6268601916692946556e0i)
(num-test (sin 1.57045105981189525579e+00+0.0e+00i) 9.9999994039535581669e-1+0.0i)
(num-test (sin -1.57045105981189525579e+00+0.0e+00i) -9.9999994039535581669e-1+0.0i)
(num-test (sin 1.57045105981189525579e+00+1.19209289550781250e-07i) 9.9999994039536292211e-1+4.1159030931177815679e-11i)
(num-test (sin 1.57045105981189525579e+00-1.19209289550781250e-07i) 9.9999994039536292211e-1-4.1159030931177815679e-11i)
(num-test (sin -1.57045105981189525579e+00+1.19209289550781250e-07i) -9.9999994039536292211e-1+4.1159030931177815679e-11i)
(num-test (sin -1.57045105981189525579e+00-1.19209289550781250e-07i) -9.9999994039536292211e-1-4.1159030931177815679e-11i)
(num-test (sin 1.57045105981189525579e+00+5.0e-01i) 1.1276258979946363572e0+1.7991700040937027667e-4i)
(num-test (sin 1.57045105981189525579e+00-5.0e-01i) 1.1276258979946363572e0-1.7991700040937027667e-4i)
(num-test (sin -1.57045105981189525579e+00+5.0e-01i) -1.1276258979946363572e0+1.7991700040937027667e-4i)
(num-test (sin -1.57045105981189525579e+00-5.0e-01i) -1.1276258979946363572e0-1.7991700040937027667e-4i)
(num-test (sin 1.57045105981189525579e+00+1.0e+00i) 1.5430805428404715942e0+4.0575816248730593018e-4i)
(num-test (sin 1.57045105981189525579e+00-1.0e+00i) 1.5430805428404715942e0-4.0575816248730593018e-4i)
(num-test (sin -1.57045105981189525579e+00+1.0e+00i) -1.5430805428404715942e0+4.0575816248730593018e-4i)
(num-test (sin -1.57045105981189525579e+00-1.0e+00i) -1.5430805428404715942e0-4.0575816248730593018e-4i)
(num-test (sin 1.57045105981189525579e+00+2.0e+00i) 3.7621954668392959445e0+1.2522351259047577385e-3i)
(num-test (sin 1.57045105981189525579e+00-2.0e+00i) 3.7621954668392959445e0-1.2522351259047577385e-3i)
(num-test (sin -1.57045105981189525579e+00+2.0e+00i) -3.7621954668392959445e0+1.2522351259047577385e-3i)
(num-test (sin -1.57045105981189525579e+00-2.0e+00i) -3.7621954668392959445e0-1.2522351259047577385e-3i)
(num-test (sin 1.57114159377789786021e+00+0.0e+00i) 9.9999994039535581673e-1+0.0i)
(num-test (sin -1.57114159377789786021e+00+0.0e+00i) -9.9999994039535581673e-1+0.0i)
(num-test (sin 1.57114159377789786021e+00+1.19209289550781250e-07i) 9.9999994039536292216e-1-4.1159030931163216752e-11i)
(num-test (sin 1.57114159377789786021e+00-1.19209289550781250e-07i) 9.9999994039536292216e-1+4.1159030931163216752e-11i)
(num-test (sin -1.57114159377789786021e+00+1.19209289550781250e-07i) -9.9999994039536292216e-1-4.1159030931163216752e-11i)
(num-test (sin -1.57114159377789786021e+00-1.19209289550781250e-07i) -9.9999994039536292216e-1+4.1159030931163216752e-11i)
(num-test (sin 1.57114159377789786021e+00+5.0e-01i) 1.1276258979946363573e0-1.7991700040930646090e-4i)
(num-test (sin 1.57114159377789786021e+00-5.0e-01i) 1.1276258979946363573e0+1.7991700040930646090e-4i)
(num-test (sin -1.57114159377789786021e+00+5.0e-01i) -1.1276258979946363573e0-1.7991700040930646090e-4i)
(num-test (sin -1.57114159377789786021e+00-5.0e-01i) -1.1276258979946363573e0+1.7991700040930646090e-4i)
(num-test (sin 1.57114159377789786021e+00+1.0e+00i) 1.5430805428404715942e0-4.0575816248716200955e-4i)
(num-test (sin 1.57114159377789786021e+00-1.0e+00i) 1.5430805428404715942e0+4.0575816248716200955e-4i)
(num-test (sin -1.57114159377789786021e+00+1.0e+00i) -1.5430805428404715942e0-4.0575816248716200955e-4i)
(num-test (sin -1.57114159377789786021e+00-1.0e+00i) -1.5430805428404715942e0+4.0575816248716200955e-4i)
(num-test (sin 1.57114159377789786021e+00+2.0e+00i) 3.7621954668392959447e0-1.2522351259043135762e-3i)
(num-test (sin 1.57114159377789786021e+00-2.0e+00i) 3.7621954668392959447e0+1.2522351259043135762e-3i)
(num-test (sin -1.57114159377789786021e+00+2.0e+00i) -3.7621954668392959447e0-1.2522351259043135762e-3i)
(num-test (sin -1.57114159377789786021e+00-2.0e+00i) -3.7621954668392959447e0+1.2522351259043135762e-3i)
(num-test (sin 3.14124738660679181379e+00+0.0e+00i) 3.4526697614158608860e-4+0.0i)
(num-test (sin -3.14124738660679181379e+00+0.0e+00i) -3.4526697614158608860e-4+0.0i)
(num-test (sin 3.14124738660679181379e+00+1.19209289550781250e-07i) 3.4526697614158854187e-4-1.1920928244535424532e-7i)
(num-test (sin 3.14124738660679181379e+00-1.19209289550781250e-07i) 3.4526697614158854187e-4+1.1920928244535424532e-7i)
(num-test (sin -3.14124738660679181379e+00+1.19209289550781250e-07i) -3.4526697614158854187e-4-1.1920928244535424532e-7i)
(num-test (sin -3.14124738660679181379e+00-1.19209289550781250e-07i) -3.4526697614158854187e-4+1.1920928244535424532e-7i)
(num-test (sin 3.14124738660679181379e+00+5.0e-01i) 3.8933200722554445944e-4-5.2109527443404709206e-1i)
(num-test (sin 3.14124738660679181379e+00-5.0e-01i) 3.8933200722554445944e-4+5.2109527443404709206e-1i)
(num-test (sin -3.14124738660679181379e+00+5.0e-01i) -3.8933200722554445944e-4-5.2109527443404709206e-1i)
(num-test (sin -3.14124738660679181379e+00-5.0e-01i) -3.8933200722554445944e-4+5.2109527443404709206e-1i)
(num-test (sin 3.14124738660679181379e+00+1.0e+00i) 5.3277478472529828958e-4-1.1752011235963524659e0i)
(num-test (sin 3.14124738660679181379e+00-1.0e+00i) 5.3277478472529828958e-4+1.1752011235963524659e0i)
(num-test (sin -3.14124738660679181379e+00+1.0e+00i) -5.3277478472529828958e-4-1.1752011235963524659e0i)
(num-test (sin -3.14124738660679181379e+00-1.0e+00i) -5.3277478472529828958e-4+1.1752011235963524659e0i)
(num-test (sin 3.14124738660679181379e+00+2.0e+00i) 1.2989619299133501696e-3-3.6268601916692946553e0i)
(num-test (sin 3.14124738660679181379e+00-2.0e+00i) 1.2989619299133501696e-3+3.6268601916692946553e0i)
(num-test (sin -3.14124738660679181379e+00+2.0e+00i) -1.2989619299133501696e-3-3.6268601916692946553e0i)
(num-test (sin -3.14124738660679181379e+00-2.0e+00i) -1.2989619299133501696e-3+3.6268601916692946553e0i)
(num-test (sin 3.14193792057279441821e+00+0.0e+00i) -3.4526697614134115926e-4+0.0i)
(num-test (sin -3.14193792057279441821e+00+0.0e+00i) 3.4526697614134115926e-4+0.0i)
(num-test (sin 3.14193792057279441821e+00+1.19209289550781250e-07i) -3.4526697614134361253e-4-1.1920928244535424533e-7i)
(num-test (sin 3.14193792057279441821e+00-1.19209289550781250e-07i) -3.4526697614134361253e-4+1.1920928244535424533e-7i)
(num-test (sin -3.14193792057279441821e+00+1.19209289550781250e-07i) 3.4526697614134361253e-4-1.1920928244535424533e-7i)
(num-test (sin -3.14193792057279441821e+00-1.19209289550781250e-07i) 3.4526697614134361253e-4+1.1920928244535424533e-7i)
(num-test (sin 3.14193792057279441821e+00+5.0e-01i) -3.8933200722526827075e-4-5.2109527443404709211e-1i)
(num-test (sin 3.14193792057279441821e+00-5.0e-01i) -3.8933200722526827075e-4+5.2109527443404709211e-1i)
(num-test (sin -3.14193792057279441821e+00+5.0e-01i) 3.8933200722526827075e-4-5.2109527443404709211e-1i)
(num-test (sin -3.14193792057279441821e+00-5.0e-01i) 3.8933200722526827075e-4+5.2109527443404709211e-1i)
(num-test (sin 3.14193792057279441821e+00+1.0e+00i) -5.3277478472492034385e-4-1.1752011235963524660e0i)
(num-test (sin 3.14193792057279441821e+00-1.0e+00i) -5.3277478472492034385e-4+1.1752011235963524660e0i)
(num-test (sin -3.14193792057279441821e+00+1.0e+00i) 5.3277478472492034385e-4-1.1752011235963524660e0i)
(num-test (sin -3.14193792057279441821e+00-1.0e+00i) 5.3277478472492034385e-4+1.1752011235963524660e0i)
(num-test (sin 3.14193792057279441821e+00+2.0e+00i) -1.2989619299124286975e-3-3.6268601916692946556e0i)
(num-test (sin 3.14193792057279441821e+00-2.0e+00i) -1.2989619299124286975e-3+3.6268601916692946556e0i)
(num-test (sin -3.14193792057279441821e+00+2.0e+00i) 1.2989619299124286975e-3-3.6268601916692946556e0i)
(num-test (sin -3.14193792057279441821e+00-2.0e+00i) 1.2989619299124286975e-3+3.6268601916692946556e0i)
(num-test (sin 4.71204371340168837179e+00+0.0e+00i) -9.9999994039535581664e-1+0.0i)
(num-test (sin -4.71204371340168837179e+00+0.0e+00i) 9.9999994039535581664e-1+0.0i)
(num-test (sin 4.71204371340168837179e+00+1.19209289550781250e-07i) -9.9999994039536292207e-1-4.1159030931192414605e-11i)
(num-test (sin 4.71204371340168837179e+00-1.19209289550781250e-07i) -9.9999994039536292207e-1+4.1159030931192414605e-11i)
(num-test (sin -4.71204371340168837179e+00+1.19209289550781250e-07i) 9.9999994039536292207e-1-4.1159030931192414605e-11i)
(num-test (sin -4.71204371340168837179e+00-1.19209289550781250e-07i) 9.9999994039536292207e-1+4.1159030931192414605e-11i)
(num-test (sin 4.71204371340168837179e+00+5.0e-01i) -1.1276258979946363572e0-1.7991700040943409243e-4i)
(num-test (sin 4.71204371340168837179e+00-5.0e-01i) -1.1276258979946363572e0+1.7991700040943409243e-4i)
(num-test (sin -4.71204371340168837179e+00+5.0e-01i) 1.1276258979946363572e0-1.7991700040943409243e-4i)
(num-test (sin -4.71204371340168837179e+00-5.0e-01i) 1.1276258979946363572e0+1.7991700040943409243e-4i)
(num-test (sin 4.71204371340168837179e+00+1.0e+00i) -1.5430805428404715941e0-4.0575816248744985081e-4i)
(num-test (sin 4.71204371340168837179e+00-1.0e+00i) -1.5430805428404715941e0+4.0575816248744985081e-4i)
(num-test (sin -4.71204371340168837179e+00+1.0e+00i) 1.5430805428404715941e0-4.0575816248744985081e-4i)
(num-test (sin -4.71204371340168837179e+00-1.0e+00i) 1.5430805428404715941e0+4.0575816248744985081e-4i)
(num-test (sin 4.71204371340168837179e+00+2.0e+00i) -3.7621954668392959444e0-1.2522351259052019007e-3i)
(num-test (sin 4.71204371340168837179e+00-2.0e+00i) -3.7621954668392959444e0+1.2522351259052019007e-3i)
(num-test (sin -4.71204371340168837179e+00+2.0e+00i) 3.7621954668392959444e0-1.2522351259052019007e-3i)
(num-test (sin -4.71204371340168837179e+00-2.0e+00i) 3.7621954668392959444e0+1.2522351259052019007e-3i)
(num-test (sin 4.71273424736769097620e+00+0.0e+00i) -9.9999994039535581677e-1+0.0i)
(num-test (sin -4.71273424736769097620e+00+0.0e+00i) 9.9999994039535581677e-1+0.0i)
(num-test (sin 4.71273424736769097620e+00+1.19209289550781250e-07i) -9.9999994039536292220e-1+4.1159030931148617825e-11i)
(num-test (sin 4.71273424736769097620e+00-1.19209289550781250e-07i) -9.9999994039536292220e-1-4.1159030931148617825e-11i)
(num-test (sin -4.71273424736769097620e+00+1.19209289550781250e-07i) 9.9999994039536292220e-1+4.1159030931148617825e-11i)
(num-test (sin -4.71273424736769097620e+00-1.19209289550781250e-07i) 9.9999994039536292220e-1-4.1159030931148617825e-11i)
(num-test (sin 4.71273424736769097620e+00+5.0e-01i) -1.1276258979946363573e0+1.7991700040924264514e-4i)
(num-test (sin 4.71273424736769097620e+00-5.0e-01i) -1.1276258979946363573e0-1.7991700040924264514e-4i)
(num-test (sin -4.71273424736769097620e+00+5.0e-01i) 1.1276258979946363573e0+1.7991700040924264514e-4i)
(num-test (sin -4.71273424736769097620e+00-5.0e-01i) 1.1276258979946363573e0-1.7991700040924264514e-4i)
(num-test (sin 4.71273424736769097620e+00+1.0e+00i) -1.5430805428404715943e0+4.0575816248701808892e-4i)
(num-test (sin 4.71273424736769097620e+00-1.0e+00i) -1.5430805428404715943e0-4.0575816248701808892e-4i)
(num-test (sin -4.71273424736769097620e+00+1.0e+00i) 1.5430805428404715943e0+4.0575816248701808892e-4i)
(num-test (sin -4.71273424736769097620e+00-1.0e+00i) 1.5430805428404715943e0-4.0575816248701808892e-4i)
(num-test (sin 4.71273424736769097620e+00+2.0e+00i) -3.7621954668392959448e0+1.2522351259038694139e-3i)
(num-test (sin 4.71273424736769097620e+00-2.0e+00i) -3.7621954668392959448e0-1.2522351259038694139e-3i)
(num-test (sin -4.71273424736769097620e+00+2.0e+00i) 3.7621954668392959448e0+1.2522351259038694139e-3i)
(num-test (sin -4.71273424736769097620e+00-2.0e+00i) 3.7621954668392959448e0-1.2522351259038694139e-3i)
(num-test (sin 6.28284004019658492979e+00+0.0e+00i) -3.4526697614170855328e-4+0.0i)
(num-test (sin -6.28284004019658492979e+00+0.0e+00i) 3.4526697614170855328e-4+0.0i)
(num-test (sin 6.28284004019658492979e+00+1.19209289550781250e-07i) -3.4526697614171100655e-4+1.1920928244535424532e-7i)
(num-test (sin 6.28284004019658492979e+00-1.19209289550781250e-07i) -3.4526697614171100655e-4-1.1920928244535424532e-7i)
(num-test (sin -6.28284004019658492979e+00+1.19209289550781250e-07i) 3.4526697614171100655e-4+1.1920928244535424532e-7i)
(num-test (sin -6.28284004019658492979e+00-1.19209289550781250e-07i) 3.4526697614171100655e-4-1.1920928244535424532e-7i)
(num-test (sin 6.28284004019658492979e+00+5.0e-01i) -3.8933200722568255379e-4+5.2109527443404709204e-1i)
(num-test (sin 6.28284004019658492979e+00-5.0e-01i) -3.8933200722568255379e-4-5.2109527443404709204e-1i)
(num-test (sin -6.28284004019658492979e+00+5.0e-01i) 3.8933200722568255379e-4+5.2109527443404709204e-1i)
(num-test (sin -6.28284004019658492979e+00-5.0e-01i) 3.8933200722568255379e-4-5.2109527443404709204e-1i)
(num-test (sin 6.28284004019658492979e+00+1.0e+00i) -5.3277478472548726245e-4+1.1752011235963524659e0i)
(num-test (sin 6.28284004019658492979e+00-1.0e+00i) -5.3277478472548726245e-4-1.1752011235963524659e0i)
(num-test (sin -6.28284004019658492979e+00+1.0e+00i) 5.3277478472548726245e-4+1.1752011235963524659e0i)
(num-test (sin -6.28284004019658492979e+00-1.0e+00i) 5.3277478472548726245e-4-1.1752011235963524659e0i)
(num-test (sin 6.28284004019658492979e+00+2.0e+00i) -1.2989619299138109057e-3+3.6268601916692946552e0i)
(num-test (sin 6.28284004019658492979e+00-2.0e+00i) -1.2989619299138109057e-3-3.6268601916692946552e0i)
(num-test (sin -6.28284004019658492979e+00+2.0e+00i) 1.2989619299138109057e-3+3.6268601916692946552e0i)
(num-test (sin -6.28284004019658492979e+00-2.0e+00i) 1.2989619299138109057e-3-3.6268601916692946552e0i)
(num-test (sin 6.28353057416258753420e+00+0.0e+00i) 3.4526697614121869459e-4+0.0i)
(num-test (sin -6.28353057416258753420e+00+0.0e+00i) -3.4526697614121869459e-4+0.0i)
(num-test (sin 6.28353057416258753420e+00+1.19209289550781250e-07i) 3.4526697614122114786e-4+1.1920928244535424534e-7i)
(num-test (sin 6.28353057416258753420e+00-1.19209289550781250e-07i) 3.4526697614122114786e-4-1.1920928244535424534e-7i)
(num-test (sin -6.28353057416258753420e+00+1.19209289550781250e-07i) -3.4526697614122114786e-4+1.1920928244535424534e-7i)
(num-test (sin -6.28353057416258753420e+00-1.19209289550781250e-07i) -3.4526697614122114786e-4-1.1920928244535424534e-7i)
(num-test (sin 6.28353057416258753420e+00+5.0e-01i) 3.8933200722513017641e-4+5.2109527443404709213e-1i)
(num-test (sin 6.28353057416258753420e+00-5.0e-01i) 3.8933200722513017641e-4-5.2109527443404709213e-1i)
(num-test (sin -6.28353057416258753420e+00+5.0e-01i) -3.8933200722513017641e-4+5.2109527443404709213e-1i)
(num-test (sin -6.28353057416258753420e+00-5.0e-01i) -3.8933200722513017641e-4-5.2109527443404709213e-1i)
(num-test (sin 6.28353057416258753420e+00+1.0e+00i) 5.3277478472473137099e-4+1.1752011235963524661e0i)
(num-test (sin 6.28353057416258753420e+00-1.0e+00i) 5.3277478472473137099e-4-1.1752011235963524661e0i)
(num-test (sin -6.28353057416258753420e+00+1.0e+00i) -5.3277478472473137099e-4+1.1752011235963524661e0i)
(num-test (sin -6.28353057416258753420e+00-1.0e+00i) -5.3277478472473137099e-4-1.1752011235963524661e0i)
(num-test (sin 6.28353057416258753420e+00+2.0e+00i) 1.2989619299119679614e-3+3.6268601916692946558e0i)
(num-test (sin 6.28353057416258753420e+00-2.0e+00i) 1.2989619299119679614e-3-3.6268601916692946558e0i)
(num-test (sin -6.28353057416258753420e+00+2.0e+00i) -1.2989619299119679614e-3+3.6268601916692946558e0i)
(num-test (sin -6.28353057416258753420e+00-2.0e+00i) -1.2989619299119679614e-3-3.6268601916692946558e0i)
(num-test (sin 9.42443269378637893396e+00+0.0e+00i) 3.4526697614094283958e-4+0.0i)
(num-test (sin -9.42443269378637893396e+00+0.0e+00i) -3.4526697614094283958e-4+0.0i)
(num-test (sin 9.42443269378637893396e+00+1.19209289550781250e-07i) 3.4526697614094529285e-4-1.1920928244535424535e-7i)
(num-test (sin 9.42443269378637893396e+00-1.19209289550781250e-07i) 3.4526697614094529285e-4+1.1920928244535424535e-7i)
(num-test (sin -9.42443269378637893396e+00+1.19209289550781250e-07i) -3.4526697614094529285e-4-1.1920928244535424535e-7i)
(num-test (sin -9.42443269378637893396e+00-1.19209289550781250e-07i) -3.4526697614094529285e-4+1.1920928244535424535e-7i)
(num-test (sin 9.42443269378637893396e+00+5.0e-01i) 3.8933200722481911514e-4-5.2109527443404709218e-1i)
(num-test (sin 9.42443269378637893396e+00-5.0e-01i) 3.8933200722481911514e-4+5.2109527443404709218e-1i)
(num-test (sin -9.42443269378637893396e+00+5.0e-01i) -3.8933200722481911514e-4-5.2109527443404709218e-1i)
(num-test (sin -9.42443269378637893396e+00-5.0e-01i) -3.8933200722481911514e-4+5.2109527443404709218e-1i)
(num-test (sin 9.42443269378637893396e+00+1.0e+00i) 5.3277478472430570447e-4-1.1752011235963524662e0i)
(num-test (sin 9.42443269378637893396e+00-1.0e+00i) 5.3277478472430570447e-4+1.1752011235963524662e0i)
(num-test (sin -9.42443269378637893396e+00+1.0e+00i) -5.3277478472430570447e-4-1.1752011235963524662e0i)
(num-test (sin -9.42443269378637893396e+00-1.0e+00i) -5.3277478472430570447e-4+1.1752011235963524662e0i)
(num-test (sin 9.42443269378637893396e+00+2.0e+00i) 1.2989619299109301409e-3-3.6268601916692946561e0i)
(num-test (sin 9.42443269378637893396e+00-2.0e+00i) 1.2989619299109301409e-3+3.6268601916692946561e0i)
(num-test (sin -9.42443269378637893396e+00+2.0e+00i) -1.2989619299109301409e-3-3.6268601916692946561e0i)
(num-test (sin -9.42443269378637893396e+00-2.0e+00i) -1.2989619299109301409e-3+3.6268601916692946561e0i)
(num-test (sin 9.42512322775237976202e+00+0.0e+00i) -3.4526697614020805155e-4+0.0i)
(num-test (sin -9.42512322775237976202e+00+0.0e+00i) 3.4526697614020805155e-4+0.0i)
(num-test (sin 9.42512322775237976202e+00+1.19209289550781250e-07i) -3.4526697614021050482e-4-1.1920928244535424538e-7i)
(num-test (sin 9.42512322775237976202e+00-1.19209289550781250e-07i) -3.4526697614021050482e-4+1.1920928244535424538e-7i)
(num-test (sin -9.42512322775237976202e+00+1.19209289550781250e-07i) 3.4526697614021050482e-4-1.1920928244535424538e-7i)
(num-test (sin -9.42512322775237976202e+00-1.19209289550781250e-07i) 3.4526697614021050482e-4+1.1920928244535424538e-7i)
(num-test (sin 9.42512322775237976202e+00+5.0e-01i) -3.8933200722399054908e-4-5.2109527443404709231e-1i)
(num-test (sin 9.42512322775237976202e+00-5.0e-01i) -3.8933200722399054908e-4+5.2109527443404709231e-1i)
(num-test (sin -9.42512322775237976202e+00+5.0e-01i) 3.8933200722399054908e-4-5.2109527443404709231e-1i)
(num-test (sin -9.42512322775237976202e+00-5.0e-01i) 3.8933200722399054908e-4+5.2109527443404709231e-1i)
(num-test (sin 9.42512322775237976202e+00+1.0e+00i) -5.3277478472317186729e-4-1.1752011235963524665e0i)
(num-test (sin 9.42512322775237976202e+00-1.0e+00i) -5.3277478472317186729e-4+1.1752011235963524665e0i)
(num-test (sin -9.42512322775237976202e+00+1.0e+00i) 5.3277478472317186729e-4-1.1752011235963524665e0i)
(num-test (sin -9.42512322775237976202e+00-1.0e+00i) 5.3277478472317186729e-4+1.1752011235963524665e0i)
(num-test (sin 9.42512322775237976202e+00+2.0e+00i) -1.2989619299081657245e-3-3.6268601916692946571e0i)
(num-test (sin 9.42512322775237976202e+00-2.0e+00i) -1.2989619299081657245e-3+3.6268601916692946571e0i)
(num-test (sin -9.42512322775237976202e+00+2.0e+00i) 1.2989619299081657245e-3-3.6268601916692946571e0i)
(num-test (sin -9.42512322775237976202e+00-2.0e+00i) 1.2989619299081657245e-3+3.6268601916692946571e0i)

(let ((val1 (sin (/ our-pi 60)))
      (val2 (* 1/16 (- (* (+ (sqrt 6) (sqrt 2)) (- (sqrt 5) 1)) (* 2 (- (sqrt 3) 1) (sqrt (+ 5 (sqrt 5))))))))
  (num-test (- val1 val2) 0.0))

(let ((val1 (sin (/ (* 4 our-pi) 15)))
      (val2 (* 1/8 (+ (sqrt (+ 10 (* 2 (sqrt 5)))) (sqrt 15) (- (sqrt 3))))))
    (num-test (- val1 val2) 0.0))


;; -------- cos
(num-test (cos 0) 1.0)
(num-test (cos 1) 0.54030230586814)
(num-test (cos 2) -0.41614683654714)
(num-test (cos 3) -0.98999249660045)
(num-test (cos 0/1) 1.0)
(num-test (cos 0/2) 1.0)
(num-test (cos 0/3) 1.0)
(num-test (cos 0/10) 1.0)
(num-test (cos 0/1234) 1.0)
(num-test (cos 0/1234000000) 1.0)
(num-test (cos 0/500029) 1.0)
(num-test (cos 0/362880) 1.0)
(num-test (cos 1/1) 0.54030230586814)
(num-test (cos 1/2) 0.87758256189037)
(num-test (cos -1/2) 0.87758256189037)
(num-test (cos 1/3) 0.94495694631474)
(num-test (cos -1/3) 0.94495694631474)
(num-test (cos 1/10) 0.99500416527803)
(num-test (cos -1/10) 0.99500416527803)
(num-test (cos 1/1234) 0.99999967164800)
(num-test (cos -1/1234) 0.99999967164800)
(num-test (cos 1/1234000000) 1.0)
(num-test (cos -1/1234000000) 1.0)
(num-test (cos 1/500029) 0.99999999999800)
(num-test (cos -1/500029) 0.99999999999800)
(num-test (cos 1/362880) 0.99999999999620)
(num-test (cos -1/362880) 0.99999999999620)
(num-test (cos 2/1) -0.41614683654714)
(num-test (cos 2/2) 0.54030230586814)
(num-test (cos 2/3) 0.78588726077695)
(num-test (cos 2/10) 0.98006657784124)
(num-test (cos -2/10) 0.98006657784124)
(num-test (cos 2/1234) 0.99999868659223)
(num-test (cos -2/1234) 0.99999868659223)
(num-test (cos 2/1234000000) 1.0)
(num-test (cos -2/1234000000) 1.0)
(num-test (cos 2/500029) 0.99999999999200)
(num-test (cos -2/500029) 0.99999999999200)
(num-test (cos 2/362880) 0.99999999998481)
(num-test (cos -2/362880) 0.99999999998481)
(num-test (cos 3/1) -0.98999249660045)
(num-test (cos 3/2) 0.07073720166770)
(num-test (cos 3/3) 0.54030230586814)
(num-test (cos 3/10) 0.95533648912561)
(num-test (cos -3/10) 0.95533648912561)
(num-test (cos 3/1234) 0.99999704483333)
(num-test (cos -3/1234) 0.99999704483333)
(num-test (cos 3/1234000000) 1.0)
(num-test (cos -3/1234000000) 1.0)
(num-test (cos 3/500029) 0.99999999998200)
(num-test (cos -3/500029) 0.99999999998200)
(num-test (cos 3/362880) 0.99999999996583)
(num-test (cos -3/362880) 0.99999999996583)
(num-test (cos 10/3) -0.98167400471108)
(num-test (cos 10/10) 0.54030230586814)
(num-test (cos 10/1234) 0.99996716497825)
(num-test (cos -10/1234) 0.99996716497825)
(num-test (cos 10/1234000000) 1.0)
(num-test (cos -10/1234000000) 1.0)
(num-test (cos 10/500029) 0.99999999980002)
(num-test (cos -10/500029) 0.99999999980002)
(num-test (cos 10/362880) 0.99999999962030)
(num-test (cos -10/362880) 0.99999999962030)
(num-test (cos 1234/1234) 0.54030230586814)
(num-test (cos 1234/1234000000) 0.99999999999950)
(num-test (cos -1234/1234000000) 0.99999999999950)
(num-test (cos 1234/500029) 0.99999695484279)
(num-test (cos -1234/500029) 0.99999695484279)
(num-test (cos 1234/362880) 0.99999421805655)
(num-test (cos -1234/362880) 0.99999421805655)
(num-test (cos 1234000000/1234000000) 0.54030230586814)
(num-test (cos 500029/1234000000) 0.99999991790247)
(num-test (cos -500029/1234000000) 0.99999991790247)
(num-test (cos 500029/500029) 0.54030230586814)
(num-test (cos 500029/362880) 0.19165727729458)
(num-test (cos 362880/1234000000) 0.99999995676199)
(num-test (cos -362880/1234000000) 0.99999995676199)
(num-test (cos 362880/500029) 0.74802315864263)
(num-test (cos 362880/362880) 0.54030230586814)
(num-test (cos 0.0) 1.0)
(num-test (cos 0.00000001) 1.0)
(num-test (cos -0.00000001) 1.0)
(num-test (cos 1.0) 0.54030230586814)
(num-test (cos our-pi) -1.0)
(num-test (cos 2.71828182845905) -0.91173391478697)
(num-test (cos 0.0+0.0i) 1.0)
(num-test (cos -0.0+0.0i) 1.0)
(num-test (cos 0.0-0.0i) 1.0)
(num-test (cos -0.0-0.0i) 1.0)
(num-test (cos 0.0+0.00000001i) 1.0)
(num-test (cos -0.0+0.00000001i) 1.0)
(num-test (cos 0.0-0.00000001i) 1.0)
(num-test (cos -0.0-0.00000001i) 1.0)
(num-test (cos 0.0+1.0i) 1.54308063481524)
(num-test (cos -0.0+1.0i) 1.54308063481524)
(num-test (cos 0.0-1.0i) 1.54308063481524)
(num-test (cos 0.0+3.14159265358979i) 11.59195327552152)
(num-test (cos -0.0+3.14159265358979i) 11.59195327552152)
(num-test (cos 0.0-3.14159265358979i) 11.59195327552152)
(num-test (cos 0.0+2.71828182845905i) 7.61012513866229)
(num-test (cos -0.0+2.71828182845905i) 7.61012513866229)
(num-test (cos 0.0-2.71828182845905i) 7.61012513866229)
(num-test (cos 0.00000001+0.0i) 1.0)
(num-test (cos -0.00000001+0.0i) 1.0)
(num-test (cos 0.00000001-0.0i) 1.0)
(num-test (cos -0.00000001-0.0i) 1.0)
(num-test (cos 0.00000001+0.00000001i) 1.0-1e-16i) ; maxima
(num-test (cos -0.00000001+0.00000001i) 1.0+1e-16i)
(num-test (cos 0.00000001-0.00000001i) 1.0+1e-16i)
(num-test (cos -0.00000001-0.00000001i) 1.0-1e-16i)
(num-test (cos 0.00000001+1.0i) 1.54308063481524-0.00000001175201i)
(num-test (cos 0.00000001-1.0i) 1.54308063481524+0.00000001175201i)
(num-test (cos 0.00000001+3.14159265358979i) 11.59195327552152-0.00000011548739i)
(num-test (cos 0.00000001-3.14159265358979i) 11.59195327552152+0.00000011548739i)
(num-test (cos 0.00000001+2.71828182845905i) 7.61012513866229-0.00000007544137i)
(num-test (cos 0.00000001-2.71828182845905i) 7.61012513866229+0.00000007544137i)
(num-test (cos 1.0+0.0i) 0.54030230586814)
(num-test (cos 1.0-0.0i) 0.54030230586814)
(num-test (cos 1.0+0.00000001i) 0.54030230586814-0.00000000841471i)
(num-test (cos 1.0-0.00000001i) 0.54030230586814+0.00000000841471i)
(num-test (cos 1.0+1.0i) 0.83373002513115-0.98889770576287i)
(num-test (cos 1.0-1.0i) 0.83373002513115+0.98889770576287i)
(num-test (cos 1.0+3.14159265358979i) 6.26315908428001-9.71792908024139i)
(num-test (cos 1.0-3.14159265358979i) 6.26315908428001+9.71792908024139i)
(num-test (cos 1.0+2.71828182845905i) 4.11176816036433-6.34817247743319i)
(num-test (cos 1.0-2.71828182845905i) 4.11176816036433+6.34817247743319i)
(num-test (cos 3.14159265358979+0.0i) -1.0)
(num-test (cos 3.14159265358979-0.0i) -1.0)
(num-test (cos 3.14159265358979+0.00000001i) -1.0-3.23121723694911E-23i) ; maxima
(num-test (cos 3.14159265358979-0.00000001i) -1.0+3.23121723694911E-23i) ; maxima
(num-test (cos 3.14159265358979+1.0i) -1.54308063481524-0.0i)
(num-test (cos 3.14159265358979-1.0i) -1.54308063481524+0.0i)
(num-test (cos 3.14159265358979+3.14159265358979i) -11.5919532755216+8.064357485351393E-14i) ; maxima
(num-test (cos 3.14159265358979-3.14159265358979i) -11.59195327552152+3.73164856762037E-14i)
(num-test (cos 3.14159265358979+2.71828182845905i) -7.61012513866229-2.437674584452965E-14i)
(num-test (cos 3.14159265358979-2.71828182845905i) -7.61012513866229+2.437674584452965E-14i)
(num-test (cos 2.71828182845905+0.0i) -0.91173391478697)
(num-test (cos 2.71828182845905-0.0i) -0.91173391478697)
(num-test (cos 2.71828182845905+0.00000001i) -0.91173391478697-0.00000000410781i)
(num-test (cos 2.71828182845905-0.00000001i) -0.91173391478697+0.00000000410781i)
(num-test (cos 2.71828182845905+1.0i) -1.40687894801206-0.48275066292556i)
(num-test (cos 2.71828182845905-1.0i) -1.40687894801206+0.48275066292556i)
(num-test (cos 2.71828182835905+3.14159265358979i) -10.56877693991882-4.74400605685607i)
(num-test (cos 2.71828182845905-3.14159265358979i) -10.56877693991882+4.74400605685607i)
(num-test (cos 2.71828182845905+2.71828182845905i) -6.93840918469126-3.09899037482603i)
(num-test (cos 2.71828182845905-2.71828182845905i) -6.93840918469126+3.09899037482603i)
(num-test (cos -2/3) .7858872607769459) 
(num-test (cos -3/2) 0.0707372016677029) 
(num-test (cos -10/3) -0.9816740047110853) 
(num-test (cos 1234/3) -0.9769114301438807) 
(num-test (cos 1234/10) -0.6387786688749486) 
(num-test (cos 1234000000/3) -5.317013319482049e-2)
(num-test (cos 1234000000/500029) .1354175745756898) 
(num-test (cos 1234000000/362880) .1995074806029773) 
(num-test (cos 500029/2) 0.962847094896035) 
(num-test (cos 500029/3) -0.6487140328750399) 
(num-test (cos 500029/10) .2565691622107056) 
(num-test (cos 500029/1234) -0.9984566200318916) 
(num-test (cos -500029/362880) .1916572772946199) 
(num-test (cos 362880/1234) .3232465372699541) 
(num-test (cos -362880/500029) .7480231586426291) 
(num-test (cos -3.14159265358979) -1.0) 
(num-test (cos -2.71828182845905) -0.9117339147869464) 
(num-test (cos 0.0+3.14159265358979i) 11.5919532755216) 
(num-test (cos 0.0+2.71828182845905i) 7.610125138661946) 
(num-test (cos 0.00000001+1.0i) 1.543080634815244-1.1752011936438014E-8i) 
(num-test (cos 0.00000001+3.14159265358979i) 11.5919532755216-1.154873935725783E-7i) 
(num-test (cos 0.00000001+2.71828182845905i) 7.610125138661945-7.54413710281663E-8i) 
(num-test (cos 1.0+0.00000001i) .5403023058681398-8.414709848078964E-9i) 
(num-test (cos 1.0+3.14159265358979i) 6.263159084280057-9.71792908024146i) 
(num-test (cos 1.0+2.71828182845905i) 4.111768160364146-6.348172477432901i) 
(num-test (cos 3.14159265358979+0.0i) -1.0) 
(num-test (cos 3.14159265358979+0.00000001i) -1.0+6.982889851335445E-23i) 
(num-test (cos 3.14159265358979+1.0i) -1.543080634815244+8.206300488372603E-15i) 
(num-test (cos 3.14159265358979+3.14159265358979i) -11.5919532755216+8.064357485351393E-14i) 
(num-test (cos 3.14159265358979+2.71828182845905i) -7.610125138661946+5.267987841234144E-14i) 
(num-test (cos 2.71828182845905+0.0i) -0.9117339147869464) 
(num-test (cos 2.71828182845905+0.00000001i) -0.9117339147869464-4.1078129050295015E-9i) 
(num-test (cos 2.71828182845905+1.0i) -1.406878948012029-0.4827506629256081i) 
(num-test (cos 2.71828182845905+3.14159265358979i) -10.56877693991868-4.744006056856582i) 
(num-test (cos 2.71828182845905+2.71828182845905i) -6.93840918469081-3.098990374826203i) 
(num-test (cos 1234.0+0.00000001i) -0.7985506235875843-6.019276547624973E-9i) 
(num-test (cos 1234.0+3.14159265358979i) -9.256761516765916-6.951505596777556i) 
(num-test (cos 1234.0+2.71828182845905i) -6.077070175058048-4.541024753505155i) 
(num-test (cos 1234000000.0+0.00000001i) +0.1589091308902228+9.872932128398908E-9i) 
(num-test (cos 1234000000.0+3.14159265358979i) +1.84206722033321+11.40199198427758i) 
(num-test (cos 1234000000.0+2.71828182845905i) +1.209318371750606+7.448275358344457i) 
(num-test (cos 1234.0+12.0i) -64983.97009220963-48983.30494825104i)
(num-test (cos -3.45266983001243932001e-04+0.0e+00i) 9.9999994039535581673e-1+0.0i)
(num-test (cos 3.45266983001243932001e-04+0.0e+00i) 9.9999994039535581673e-1+0.0i)
(num-test (cos -3.45266983001243932001e-04+1.19209289550781250e-07i) 9.9999994039536292216e-1+4.1159030931163569191e-11i)
(num-test (cos -3.45266983001243932001e-04-1.19209289550781250e-07i) 9.9999994039536292216e-1-4.1159030931163569191e-11i)
(num-test (cos 3.45266983001243932001e-04+1.19209289550781250e-07i) 9.9999994039536292216e-1-4.1159030931163569191e-11i)
(num-test (cos 3.45266983001243932001e-04-1.19209289550781250e-07i) 9.9999994039536292216e-1+4.1159030931163569191e-11i)
(num-test (cos -3.45266983001243932001e-04+5.0e-01i) 1.1276258979946363573e0+1.7991700040930800151e-4i)
(num-test (cos -3.45266983001243932001e-04-5.0e-01i) 1.1276258979946363573e0-1.7991700040930800151e-4i)
(num-test (cos 3.45266983001243932001e-04+5.0e-01i) 1.1276258979946363573e0-1.7991700040930800151e-4i)
(num-test (cos 3.45266983001243932001e-04-5.0e-01i) 1.1276258979946363573e0+1.7991700040930800151e-4i)
(num-test (cos -3.45266983001243932001e-04+1.0e+00i) 1.5430805428404715942e0+4.057581624871654840e-4i)
(num-test (cos -3.45266983001243932001e-04-1.0e+00i) 1.5430805428404715942e0-4.057581624871654840e-4i)
(num-test (cos 3.45266983001243932001e-04+1.0e+00i) 1.5430805428404715942e0-4.057581624871654840e-4i)
(num-test (cos 3.45266983001243932001e-04-1.0e+00i) 1.5430805428404715942e0+4.057581624871654840e-4i)
(num-test (cos -3.45266983001243932001e-04+2.0e+00i) 3.7621954668392959447e0+1.2522351259043242989e-3i)
(num-test (cos -3.45266983001243932001e-04-2.0e+00i) 3.7621954668392959447e0-1.2522351259043242989e-3i)
(num-test (cos 3.45266983001243932001e-04+2.0e+00i) 3.7621954668392959447e0-1.2522351259043242989e-3i)
(num-test (cos 3.45266983001243932001e-04-2.0e+00i) 3.7621954668392959447e0+1.2522351259043242989e-3i)
(num-test (cos 1.57045105981189525579e+00+0.0e+00i) 3.4526697614152485627e-4+0.0i)
(num-test (cos -1.57045105981189525579e+00+0.0e+00i) 3.4526697614152485627e-4+0.0i)
(num-test (cos 1.57045105981189525579e+00+1.19209289550781250e-07i) 3.4526697614152730954e-4-1.1920928244535424532e-7i)
(num-test (cos 1.57045105981189525579e+00-1.19209289550781250e-07i) 3.4526697614152730954e-4+1.1920928244535424532e-7i)
(num-test (cos -1.57045105981189525579e+00+1.19209289550781250e-07i) 3.4526697614152730954e-4+1.1920928244535424532e-7i)
(num-test (cos -1.57045105981189525579e+00-1.19209289550781250e-07i) 3.4526697614152730954e-4-1.1920928244535424532e-7i)
(num-test (cos 1.57045105981189525579e+00+5.0e-01i) 3.8933200722547541227e-4-5.2109527443404709207e-1i)
(num-test (cos 1.57045105981189525579e+00-5.0e-01i) 3.8933200722547541227e-4+5.2109527443404709207e-1i)
(num-test (cos -1.57045105981189525579e+00+5.0e-01i) 3.8933200722547541227e-4+5.2109527443404709207e-1i)
(num-test (cos -1.57045105981189525579e+00-5.0e-01i) 3.8933200722547541227e-4-5.2109527443404709207e-1i)
(num-test (cos 1.57045105981189525579e+00+1.0e+00i) 5.3277478472520380315e-4-1.1752011235963524659e0i)
(num-test (cos 1.57045105981189525579e+00-1.0e+00i) 5.3277478472520380315e-4+1.1752011235963524659e0i)
(num-test (cos -1.57045105981189525579e+00+1.0e+00i) 5.3277478472520380315e-4+1.1752011235963524659e0i)
(num-test (cos -1.57045105981189525579e+00-1.0e+00i) 5.3277478472520380315e-4-1.1752011235963524659e0i)
(num-test (cos 1.57045105981189525579e+00+2.0e+00i) 1.2989619299131198016e-3-3.6268601916692946554e0i)
(num-test (cos 1.57045105981189525579e+00-2.0e+00i) 1.2989619299131198016e-3+3.6268601916692946554e0i)
(num-test (cos -1.57045105981189525579e+00+2.0e+00i) 1.2989619299131198016e-3+3.6268601916692946554e0i)
(num-test (cos -1.57045105981189525579e+00-2.0e+00i) 1.2989619299131198016e-3-3.6268601916692946554e0i)
(num-test (cos 1.57114159377789786021e+00+0.0e+00i) -3.4526697614140239160e-4+0.0i)
(num-test (cos -1.57114159377789786021e+00+0.0e+00i) -3.4526697614140239160e-4+0.0i)
(num-test (cos 1.57114159377789786021e+00+1.19209289550781250e-07i) -3.4526697614140484486e-4-1.1920928244535424533e-7i)
(num-test (cos 1.57114159377789786021e+00-1.19209289550781250e-07i) -3.4526697614140484486e-4+1.1920928244535424533e-7i)
(num-test (cos -1.57114159377789786021e+00+1.19209289550781250e-07i) -3.4526697614140484486e-4+1.1920928244535424533e-7i)
(num-test (cos -1.57114159377789786021e+00-1.19209289550781250e-07i) -3.4526697614140484486e-4-1.1920928244535424533e-7i)
(num-test (cos 1.57114159377789786021e+00+5.0e-01i) -3.8933200722533731792e-4-5.2109527443404709209e-1i)
(num-test (cos 1.57114159377789786021e+00-5.0e-01i) -3.8933200722533731792e-4+5.2109527443404709209e-1i)
(num-test (cos -1.57114159377789786021e+00+5.0e-01i) -3.8933200722533731792e-4+5.2109527443404709209e-1i)
(num-test (cos -1.57114159377789786021e+00-5.0e-01i) -3.8933200722533731792e-4-5.2109527443404709209e-1i)
(num-test (cos 1.57114159377789786021e+00+1.0e+00i) -5.3277478472501483029e-4-1.1752011235963524660e0i)
(num-test (cos 1.57114159377789786021e+00-1.0e+00i) -5.3277478472501483029e-4+1.1752011235963524660e0i)
(num-test (cos -1.57114159377789786021e+00+1.0e+00i) -5.3277478472501483029e-4+1.1752011235963524660e0i)
(num-test (cos -1.57114159377789786021e+00-1.0e+00i) -5.3277478472501483029e-4-1.1752011235963524660e0i)
(num-test (cos 1.57114159377789786021e+00+2.0e+00i) -1.2989619299126590655e-3-3.6268601916692946556e0i)
(num-test (cos 1.57114159377789786021e+00-2.0e+00i) -1.2989619299126590655e-3+3.6268601916692946556e0i)
(num-test (cos -1.57114159377789786021e+00+2.0e+00i) -1.2989619299126590655e-3+3.6268601916692946556e0i)
(num-test (cos -1.57114159377789786021e+00-2.0e+00i) -1.2989619299126590655e-3-3.6268601916692946556e0i)
(num-test (cos 3.14124738660679181379e+00+0.0e+00i) -9.9999994039535581667e-1+0.0i)
(num-test (cos -3.14124738660679181379e+00+0.0e+00i) -9.9999994039535581667e-1+0.0i)
(num-test (cos 3.14124738660679181379e+00+1.19209289550781250e-07i) -9.9999994039536292209e-1-4.1159030931185115142e-11i)
(num-test (cos 3.14124738660679181379e+00-1.19209289550781250e-07i) -9.9999994039536292209e-1+4.1159030931185115142e-11i)
(num-test (cos -3.14124738660679181379e+00+1.19209289550781250e-07i) -9.9999994039536292209e-1+4.1159030931185115142e-11i)
(num-test (cos -3.14124738660679181379e+00-1.19209289550781250e-07i) -9.9999994039536292209e-1-4.1159030931185115142e-11i)
(num-test (cos 3.14124738660679181379e+00+5.0e-01i) -1.1276258979946363572e0-1.7991700040940218455e-4i)
(num-test (cos 3.14124738660679181379e+00-5.0e-01i) -1.1276258979946363572e0+1.7991700040940218455e-4i)
(num-test (cos -3.14124738660679181379e+00+5.0e-01i) -1.1276258979946363572e0+1.7991700040940218455e-4i)
(num-test (cos -3.14124738660679181379e+00-5.0e-01i) -1.1276258979946363572e0-1.7991700040940218455e-4i)
(num-test (cos 3.14124738660679181379e+00+1.0e+00i) -1.5430805428404715941e0-4.0575816248737789049e-4i)
(num-test (cos 3.14124738660679181379e+00-1.0e+00i) -1.5430805428404715941e0+4.0575816248737789049e-4i)
(num-test (cos -3.14124738660679181379e+00+1.0e+00i) -1.5430805428404715941e0+4.0575816248737789049e-4i)
(num-test (cos -3.14124738660679181379e+00-1.0e+00i) -1.5430805428404715941e0-4.0575816248737789049e-4i)
(num-test (cos 3.14124738660679181379e+00+2.0e+00i) -3.7621954668392959444e0-1.2522351259049798196e-3i)
(num-test (cos 3.14124738660679181379e+00-2.0e+00i) -3.7621954668392959444e0+1.2522351259049798196e-3i)
(num-test (cos -3.14124738660679181379e+00+2.0e+00i) -3.7621954668392959444e0+1.2522351259049798196e-3i)
(num-test (cos -3.14124738660679181379e+00-2.0e+00i) -3.7621954668392959444e0-1.2522351259049798196e-3i)
(num-test (cos 3.14193792057279441821e+00+0.0e+00i) -9.9999994039535581675e-1+0.0i)
(num-test (cos -3.14193792057279441821e+00+0.0e+00i) -9.9999994039535581675e-1+0.0i)
(num-test (cos 3.14193792057279441821e+00+1.19209289550781250e-07i) -9.9999994039536292218e-1+4.1159030931155917289e-11i)
(num-test (cos 3.14193792057279441821e+00-1.19209289550781250e-07i) -9.9999994039536292218e-1-4.1159030931155917289e-11i)
(num-test (cos -3.14193792057279441821e+00+1.19209289550781250e-07i) -9.9999994039536292218e-1-4.1159030931155917289e-11i)
(num-test (cos -3.14193792057279441821e+00-1.19209289550781250e-07i) -9.9999994039536292218e-1+4.1159030931155917289e-11i)
(num-test (cos 3.14193792057279441821e+00+5.0e-01i) -1.1276258979946363573e0+1.7991700040927455302e-4i)
(num-test (cos 3.14193792057279441821e+00-5.0e-01i) -1.1276258979946363573e0-1.7991700040927455302e-4i)
(num-test (cos -3.14193792057279441821e+00+5.0e-01i) -1.1276258979946363573e0-1.7991700040927455302e-4i)
(num-test (cos -3.14193792057279441821e+00-5.0e-01i) -1.1276258979946363573e0+1.7991700040927455302e-4i)
(num-test (cos 3.14193792057279441821e+00+1.0e+00i) -1.5430805428404715943e0+4.0575816248709004923e-4i)
(num-test (cos 3.14193792057279441821e+00-1.0e+00i) -1.5430805428404715943e0-4.0575816248709004923e-4i)
(num-test (cos -3.14193792057279441821e+00+1.0e+00i) -1.5430805428404715943e0-4.0575816248709004923e-4i)
(num-test (cos -3.14193792057279441821e+00-1.0e+00i) -1.5430805428404715943e0+4.0575816248709004923e-4i)
(num-test (cos 3.14193792057279441821e+00+2.0e+00i) -3.7621954668392959448e0+1.2522351259040914950e-3i)
(num-test (cos 3.14193792057279441821e+00-2.0e+00i) -3.7621954668392959448e0-1.2522351259040914950e-3i)
(num-test (cos -3.14193792057279441821e+00+2.0e+00i) -3.7621954668392959448e0-1.2522351259040914950e-3i)
(num-test (cos -3.14193792057279441821e+00-2.0e+00i) -3.7621954668392959448e0+1.2522351259040914950e-3i)
(num-test (cos 4.71204371340168837179e+00+0.0e+00i) -3.4526697614164732094e-4+0.0i)
(num-test (cos -4.71204371340168837179e+00+0.0e+00i) -3.4526697614164732094e-4+0.0i)
(num-test (cos 4.71204371340168837179e+00+1.19209289550781250e-07i) -3.4526697614164977421e-4+1.1920928244535424532e-7i)
(num-test (cos 4.71204371340168837179e+00-1.19209289550781250e-07i) -3.4526697614164977421e-4-1.1920928244535424532e-7i)
(num-test (cos -4.71204371340168837179e+00+1.19209289550781250e-07i) -3.4526697614164977421e-4-1.1920928244535424532e-7i)
(num-test (cos -4.71204371340168837179e+00-1.19209289550781250e-07i) -3.4526697614164977421e-4+1.1920928244535424532e-7i)
(num-test (cos 4.71204371340168837179e+00+5.0e-01i) -3.8933200722561350661e-4+5.2109527443404709205e-1i)
(num-test (cos 4.71204371340168837179e+00-5.0e-01i) -3.8933200722561350661e-4-5.2109527443404709205e-1i)
(num-test (cos -4.71204371340168837179e+00+5.0e-01i) -3.8933200722561350661e-4-5.2109527443404709205e-1i)
(num-test (cos -4.71204371340168837179e+00-5.0e-01i) -3.8933200722561350661e-4+5.2109527443404709205e-1i)
(num-test (cos 4.71204371340168837179e+00+1.0e+00i) -5.3277478472539277601e-4+1.1752011235963524659e0i)
(num-test (cos 4.71204371340168837179e+00-1.0e+00i) -5.3277478472539277601e-4-1.1752011235963524659e0i)
(num-test (cos -4.71204371340168837179e+00+1.0e+00i) -5.3277478472539277601e-4-1.1752011235963524659e0i)
(num-test (cos -4.71204371340168837179e+00-1.0e+00i) -5.3277478472539277601e-4+1.1752011235963524659e0i)
(num-test (cos 4.71204371340168837179e+00+2.0e+00i) -1.2989619299135805376e-3+3.6268601916692946552e0i)
(num-test (cos 4.71204371340168837179e+00-2.0e+00i) -1.2989619299135805376e-3-3.6268601916692946552e0i)
(num-test (cos -4.71204371340168837179e+00+2.0e+00i) -1.2989619299135805376e-3-3.6268601916692946552e0i)
(num-test (cos -4.71204371340168837179e+00-2.0e+00i) -1.2989619299135805376e-3+3.6268601916692946552e0i)
(num-test (cos 4.71273424736769097620e+00+0.0e+00i) 3.4526697614127992692e-4+0.0i)
(num-test (cos -4.71273424736769097620e+00+0.0e+00i) 3.4526697614127992692e-4+0.0i)
(num-test (cos 4.71273424736769097620e+00+1.19209289550781250e-07i) 3.4526697614128238019e-4+1.1920928244535424533e-7i)
(num-test (cos 4.71273424736769097620e+00-1.19209289550781250e-07i) 3.4526697614128238019e-4-1.1920928244535424533e-7i)
(num-test (cos -4.71273424736769097620e+00+1.19209289550781250e-07i) 3.4526697614128238019e-4-1.1920928244535424533e-7i)
(num-test (cos -4.71273424736769097620e+00-1.19209289550781250e-07i) 3.4526697614128238019e-4+1.1920928244535424533e-7i)
(num-test (cos 4.71273424736769097620e+00+5.0e-01i) 3.8933200722519922358e-4+5.2109527443404709212e-1i)
(num-test (cos 4.71273424736769097620e+00-5.0e-01i) 3.8933200722519922358e-4-5.2109527443404709212e-1i)
(num-test (cos -4.71273424736769097620e+00+5.0e-01i) 3.8933200722519922358e-4-5.2109527443404709212e-1i)
(num-test (cos -4.71273424736769097620e+00-5.0e-01i) 3.8933200722519922358e-4+5.2109527443404709212e-1i)
(num-test (cos 4.71273424736769097620e+00+1.0e+00i) 5.3277478472482585742e-4+1.1752011235963524660e0i)
(num-test (cos 4.71273424736769097620e+00-1.0e+00i) 5.3277478472482585742e-4-1.1752011235963524660e0i)
(num-test (cos -4.71273424736769097620e+00+1.0e+00i) 5.3277478472482585742e-4-1.1752011235963524660e0i)
(num-test (cos -4.71273424736769097620e+00-1.0e+00i) 5.3277478472482585742e-4+1.1752011235963524660e0i)
(num-test (cos 4.71273424736769097620e+00+2.0e+00i) 1.2989619299121983294e-3+3.6268601916692946557e0i)
(num-test (cos 4.71273424736769097620e+00-2.0e+00i) 1.2989619299121983294e-3-3.6268601916692946557e0i)
(num-test (cos -4.71273424736769097620e+00+2.0e+00i) 1.2989619299121983294e-3-3.6268601916692946557e0i)
(num-test (cos -4.71273424736769097620e+00-2.0e+00i) 1.2989619299121983294e-3+3.6268601916692946557e0i)
(num-test (cos 6.28284004019658492979e+00+0.0e+00i) 9.9999994039535581662e-1+0.0i)
(num-test (cos -6.28284004019658492979e+00+0.0e+00i) 9.9999994039535581662e-1+0.0i)
(num-test (cos 6.28284004019658492979e+00+1.19209289550781250e-07i) 9.9999994039536292205e-1+4.1159030931199714069e-11i)
(num-test (cos 6.28284004019658492979e+00-1.19209289550781250e-07i) 9.9999994039536292205e-1-4.1159030931199714069e-11i)
(num-test (cos -6.28284004019658492979e+00+1.19209289550781250e-07i) 9.9999994039536292205e-1-4.1159030931199714069e-11i)
(num-test (cos -6.28284004019658492979e+00-1.19209289550781250e-07i) 9.9999994039536292205e-1+4.1159030931199714069e-11i)
(num-test (cos 6.28284004019658492979e+00+5.0e-01i) 1.1276258979946363572e0+1.7991700040946600032e-4i)
(num-test (cos 6.28284004019658492979e+00-5.0e-01i) 1.1276258979946363572e0-1.7991700040946600032e-4i)
(num-test (cos -6.28284004019658492979e+00+5.0e-01i) 1.1276258979946363572e0-1.7991700040946600032e-4i)
(num-test (cos -6.28284004019658492979e+00-5.0e-01i) 1.1276258979946363572e0+1.7991700040946600032e-4i)
(num-test (cos 6.28284004019658492979e+00+1.0e+00i) 1.5430805428404715941e0+4.0575816248752181112e-4i)
(num-test (cos 6.28284004019658492979e+00-1.0e+00i) 1.5430805428404715941e0-4.0575816248752181112e-4i)
(num-test (cos -6.28284004019658492979e+00+1.0e+00i) 1.5430805428404715941e0-4.0575816248752181112e-4i)
(num-test (cos -6.28284004019658492979e+00-1.0e+00i) 1.5430805428404715941e0+4.0575816248752181112e-4i)
(num-test (cos 6.28284004019658492979e+00+2.0e+00i) 3.7621954668392959443e0+1.2522351259054239819e-3i)
(num-test (cos 6.28284004019658492979e+00-2.0e+00i) 3.7621954668392959443e0-1.2522351259054239819e-3i)
(num-test (cos -6.28284004019658492979e+00+2.0e+00i) 3.7621954668392959443e0-1.2522351259054239819e-3i)
(num-test (cos -6.28284004019658492979e+00-2.0e+00i) 3.7621954668392959443e0+1.2522351259054239819e-3i)
(num-test (cos 6.28353057416258753420e+00+0.0e+00i) 9.9999994039535581679e-1+0.0i)
(num-test (cos -6.28353057416258753420e+00+0.0e+00i) 9.9999994039535581679e-1+0.0i)
(num-test (cos 6.28353057416258753420e+00+1.19209289550781250e-07i) 9.9999994039536292222e-1-4.1159030931141318362e-11i)
(num-test (cos 6.28353057416258753420e+00-1.19209289550781250e-07i) 9.9999994039536292222e-1+4.1159030931141318362e-11i)
(num-test (cos -6.28353057416258753420e+00+1.19209289550781250e-07i) 9.9999994039536292222e-1+4.1159030931141318362e-11i)
(num-test (cos -6.28353057416258753420e+00-1.19209289550781250e-07i) 9.9999994039536292222e-1-4.1159030931141318362e-11i)
(num-test (cos 6.28353057416258753420e+00+5.0e-01i) 1.1276258979946363574e0-1.7991700040921073725e-4i)
(num-test (cos 6.28353057416258753420e+00-5.0e-01i) 1.1276258979946363574e0+1.7991700040921073725e-4i)
(num-test (cos -6.28353057416258753420e+00+5.0e-01i) 1.1276258979946363574e0+1.7991700040921073725e-4i)
(num-test (cos -6.28353057416258753420e+00-5.0e-01i) 1.1276258979946363574e0-1.7991700040921073725e-4i)
(num-test (cos 6.28353057416258753420e+00+1.0e+00i) 1.5430805428404715943e0-4.0575816248694612861e-4i)
(num-test (cos 6.28353057416258753420e+00-1.0e+00i) 1.5430805428404715943e0+4.0575816248694612861e-4i)
(num-test (cos -6.28353057416258753420e+00+1.0e+00i) 1.5430805428404715943e0+4.0575816248694612861e-4i)
(num-test (cos -6.28353057416258753420e+00-1.0e+00i) 1.5430805428404715943e0-4.0575816248694612861e-4i)
(num-test (cos 6.28353057416258753420e+00+2.0e+00i) 3.7621954668392959449e0-1.2522351259036473328e-3i)
(num-test (cos 6.28353057416258753420e+00-2.0e+00i) 3.7621954668392959449e0+1.2522351259036473328e-3i)
(num-test (cos -6.28353057416258753420e+00+2.0e+00i) 3.7621954668392959449e0+1.2522351259036473328e-3i)
(num-test (cos -6.28353057416258753420e+00-2.0e+00i) 3.7621954668392959449e0-1.2522351259036473328e-3i)
(num-test (cos 9.42443269378637893396e+00+0.0e+00i) -9.9999994039535581689e-1+0.0i)
(num-test (cos -9.42443269378637893396e+00+0.0e+00i) -9.9999994039535581689e-1+0.0i)
(num-test (cos 9.42443269378637893396e+00+1.19209289550781250e-07i) -9.9999994039536292231e-1-4.1159030931108433883e-11i)
(num-test (cos 9.42443269378637893396e+00-1.19209289550781250e-07i) -9.9999994039536292231e-1+4.1159030931108433883e-11i)
(num-test (cos -9.42443269378637893396e+00+1.19209289550781250e-07i) -9.9999994039536292231e-1+4.1159030931108433883e-11i)
(num-test (cos -9.42443269378637893396e+00-1.19209289550781250e-07i) -9.9999994039536292231e-1-4.1159030931108433883e-11i)
(num-test (cos 9.42443269378637893396e+00+5.0e-01i) -1.1276258979946363575e0-1.7991700040906699050e-4i)
(num-test (cos 9.42443269378637893396e+00-5.0e-01i) -1.1276258979946363575e0+1.7991700040906699050e-4i)
(num-test (cos -9.42443269378637893396e+00+5.0e-01i) -1.1276258979946363575e0+1.7991700040906699050e-4i)
(num-test (cos -9.42443269378637893396e+00-5.0e-01i) -1.1276258979946363575e0-1.7991700040906699050e-4i)
(num-test (cos 9.42443269378637893396e+00+1.0e+00i) -1.5430805428404715945e0-4.0575816248662194348e-4i)
(num-test (cos 9.42443269378637893396e+00-1.0e+00i) -1.5430805428404715945e0+4.0575816248662194348e-4i)
(num-test (cos -9.42443269378637893396e+00+1.0e+00i) -1.5430805428404715945e0+4.0575816248662194348e-4i)
(num-test (cos -9.42443269378637893396e+00-1.0e+00i) -1.5430805428404715945e0-4.0575816248662194348e-4i)
(num-test (cos 9.42443269378637893396e+00+2.0e+00i) -3.7621954668392959453e0-1.2522351259026468452e-3i)
(num-test (cos 9.42443269378637893396e+00-2.0e+00i) -3.7621954668392959453e0+1.2522351259026468452e-3i)
(num-test (cos -9.42443269378637893396e+00+2.0e+00i) -3.7621954668392959453e0+1.2522351259026468452e-3i)
(num-test (cos -9.42443269378637893396e+00-2.0e+00i) -3.7621954668392959453e0-1.2522351259026468452e-3i)
(num-test (cos 9.42512322775237976202e+00+0.0e+00i) -9.9999994039535581714e-1+0.0i)
(num-test (cos -9.42512322775237976202e+00+0.0e+00i) -9.9999994039535581714e-1+0.0i)
(num-test (cos 9.42512322775237976202e+00+1.19209289550781250e-07i) -9.9999994039536292257e-1+4.1159030931020840323e-11i)
(num-test (cos 9.42512322775237976202e+00-1.19209289550781250e-07i) -9.9999994039536292257e-1-4.1159030931020840323e-11i)
(num-test (cos -9.42512322775237976202e+00+1.19209289550781250e-07i) -9.9999994039536292257e-1-4.1159030931020840323e-11i)
(num-test (cos -9.42512322775237976202e+00-1.19209289550781250e-07i) -9.9999994039536292257e-1+4.1159030931020840323e-11i)
(num-test (cos 9.42512322775237976202e+00+5.0e-01i) -1.1276258979946363577e0+1.7991700040868409591e-4i)
(num-test (cos 9.42512322775237976202e+00-5.0e-01i) -1.1276258979946363577e0-1.7991700040868409591e-4i)
(num-test (cos -9.42512322775237976202e+00+5.0e-01i) -1.1276258979946363577e0-1.7991700040868409591e-4i)
(num-test (cos -9.42512322775237976202e+00-5.0e-01i) -1.1276258979946363577e0+1.7991700040868409591e-4i)
(num-test (cos 9.42512322775237976202e+00+1.0e+00i) -1.5430805428404715949e0+4.0575816248575841970e-4i)
(num-test (cos 9.42512322775237976202e+00-1.0e+00i) -1.5430805428404715949e0-4.0575816248575841970e-4i)
(num-test (cos -9.42512322775237976202e+00+1.0e+00i) -1.5430805428404715949e0-4.0575816248575841970e-4i)
(num-test (cos -9.42512322775237976202e+00-1.0e+00i) -1.5430805428404715949e0+4.0575816248575841970e-4i)
(num-test (cos 9.42512322775237976202e+00+2.0e+00i) -3.7621954668392959462e0+1.2522351258999818715e-3i)
(num-test (cos 9.42512322775237976202e+00-2.0e+00i) -3.7621954668392959462e0-1.2522351258999818715e-3i)
(num-test (cos -9.42512322775237976202e+00+2.0e+00i) -3.7621954668392959462e0-1.2522351258999818715e-3i)
(num-test (cos -9.42512322775237976202e+00-2.0e+00i) -3.7621954668392959462e0+1.2522351258999818715e-3i)
(num-test (cos 0) 1.0)


;; -------- tan
(num-test (tan 0) 0.0)
(num-test (tan 1) 1.55740772465490)
(num-test (tan -1) -1.55740772465490)
(num-test (tan 0/1) 0.0)
(num-test (tan 0/2) 0.0)
(num-test (tan 0/3) 0.0)
(num-test (tan 0/10) 0.0)
(num-test (tan 0/1234) 0.0)
(num-test (tan 0/1234000000) 0.0)
(num-test (tan 0/500029) 0.0)
(num-test (tan 0/362880) 0.0)
(num-test (tan 1/1) 1.55740772465490)
(num-test (tan -1/1) -1.55740772465490)
(num-test (tan 1/2) 0.54630248984379)
(num-test (tan -1/2) -0.54630248984379)
(num-test (tan 1/3) 0.34625354951058)
(num-test (tan -1/3) -0.34625354951058)
(num-test (tan 1/10) 0.10033467208545)
(num-test (tan -1/10) -0.10033467208545)
(num-test (tan 1/1234) 0.00081037294887)
(num-test (tan -1/1234) -0.00081037294887)
(num-test (tan 1/1234000000) 0.00000000081037)
(num-test (tan -1/1234000000) -0.00000000081037)
(num-test (tan 1/500029) 0.00000199988401)
(num-test (tan -1/500029) -0.00000199988401)
(num-test (tan 1/362880) 0.00000275573192)
(num-test (tan -1/362880) -0.00000275573192)
(num-test (tan 2/2) 1.55740772465490)
(num-test (tan -2/2) -1.55740772465490)
(num-test (tan 2/3) 0.78684288947298)
(num-test (tan -2/3) -0.78684288947298)
(num-test (tan 2/10) 0.20271003550867)
(num-test (tan -2/10) -0.20271003550867)
(num-test (tan 2/1234) 0.00162074696208)
(num-test (tan -2/1234) -0.00162074696208)
(num-test (tan 2/1234000000) 0.00000000162075)
(num-test (tan -2/1234000000) -0.00000000162075)
(num-test (tan 2/500029) 0.00000399976801)
(num-test (tan -2/500029) -0.00000399976801)
(num-test (tan 2/362880) 0.00000551146384)
(num-test (tan -2/362880) -0.00000551146384)
(num-test (tan 3/2) 14.10141994717166)
(num-test (tan -3/2) -14.10141994717166)
(num-test (tan 3/3) 1.55740772465490)
(num-test (tan -3/3) -1.55740772465490)
(num-test (tan 3/10) 0.30933624960962)
(num-test (tan -3/10) -0.30933624960962)
(num-test (tan 3/1234) 0.00243112310401)
(num-test (tan -3/1234) -0.00243112310401)
(num-test (tan 3/1234000000) 0.00000000243112)
(num-test (tan -3/1234000000) -0.00000000243112)
(num-test (tan 3/500029) 0.00000599965202)
(num-test (tan -3/500029) -0.00000599965202)
(num-test (tan 3/362880) 0.00000826719577)
(num-test (tan -3/362880) -0.00000826719577)
(num-test (tan 10/10) 1.55740772465490)
(num-test (tan -10/10) -1.55740772465490)
(num-test (tan 10/1234) 0.00810390511110)
(num-test (tan -10/1234) -0.00810390511110)
(num-test (tan 10/1234000000) 0.00000000810373)
(num-test (tan -10/1234000000) -0.00000000810373)
(num-test (tan 10/500029) 0.00001999884007)
(num-test (tan -10/500029) -0.00001999884007)
(num-test (tan 10/362880) 0.00002755731923)
(num-test (tan -10/362880) -0.00002755731923)
(num-test (tan 1234/1234) 1.55740772465490)
(num-test (tan -1234/1234) -1.55740772465490)
(num-test (tan 1234/1234000000) 0.00000100000000)
(num-test (tan -1234/1234000000) -0.00000100000000)
(num-test (tan 1234/500029) 0.00246786187432)
(num-test (tan -1234/500029) -0.00246786187432)
(num-test (tan 1234/362880) 0.00340058630026)
(num-test (tan -1234/362880) -0.00340058630026)
(num-test (tan 1234000000/1234000000) 1.55740772465490)
(num-test (tan -1234000000/1234000000) -1.55740772465490)
(num-test (tan 500029/1234000000) 0.00040520990873)
(num-test (tan -500029/1234000000) -0.00040520990873)
(num-test (tan 500029/500029) 1.55740772465490)
(num-test (tan -500029/500029) -1.55740772465490)
(num-test (tan 500029/362880) 5.12092171798848)
(num-test (tan -500029/362880) -5.12092171798848)
(num-test (tan 362880/1234000000) 0.00029406807979)
(num-test (tan -362880/1234000000) -0.00029406807979)
(num-test (tan 362880/500029) 0.88723539913856)
(num-test (tan -362880/500029) -0.88723539913856)
(num-test (tan 362880/362880) 1.55740772465490)
(num-test (tan -362880/362880) -1.55740772465490)
(num-test (tan 0.0) 0.0)
(num-test (tan 0.00000001) 0.00000001)
(num-test (tan -0.00000001) -0.00000001)
(num-test (tan 1.0) 1.55740772465490)
(num-test (tan -1.0) -1.55740772465490)
(num-test (tan 0.0+0.0i) 0.0)
(num-test (tan -0.0+0.0i) 0.0)
(num-test (tan 0.0-0.0i) 0.0)
(num-test (tan -0.0-0.0i) -0.0)
(num-test (tan 0.0+0.00000001i) 0.0+0.00000001i)
(num-test (tan -0.0+0.00000001i) 0.0+0.00000001i)
(num-test (tan 0.0-0.00000001i) 0.0-0.00000001i)
(num-test (tan -0.0-0.00000001i) -0.0-0.00000001i)
(num-test (tan 0.0+1.0i) 0.0+0.76159415595576i)
(num-test (tan -0.0+1.0i) 0.0+0.76159415595576i)
(num-test (tan 0.0-1.0i) 0.0-0.76159415595576i)
(num-test (tan -0.0-1.0i) -0.0-0.76159415595576i)
(num-test (tan 0.0+3.14159265358979i) 0.0+0.99627207622075i)
(num-test (tan -0.0+3.14159265358979i) 0.0+0.99627207622075i)
(num-test (tan 0.0-3.14159265358979i) 0.0-0.99627207622075i)
(num-test (tan -0.0-3.14159265358979i) -0.0-0.99627207622075i)
(num-test (tan 0.0+2.71828182845905i) 0.0+0.99132891580060i)
(num-test (tan -0.0+2.71828182845905i) 0.0+0.99132891580060i)
(num-test (tan 0.0-2.71828182845905i) 0.0-0.99132891580060i)
(num-test (tan -0.0-2.71828182845905i) -0.0-0.99132891580060i)
(num-test (tan 0.00000001+0.0i) 0.00000001)
(num-test (tan -0.00000001+0.0i) -0.00000001)
(num-test (tan 0.00000001-0.0i) 0.00000001)
(num-test (tan -0.00000001-0.0i) -0.00000001)
(num-test (tan 0.00000001+0.00000001i) 0.00000001+0.00000001i)
(num-test (tan -0.00000001+0.00000001i) -0.00000001+0.00000001i)
(num-test (tan 0.00000001-0.00000001i) 0.00000001-0.00000001i)
(num-test (tan -0.00000001-0.00000001i) -0.00000001-0.00000001i)
(num-test (tan 0.00000001+1.0i) 0.00000000419974+0.76159415595576i)
(num-test (tan -0.00000001+1.0i) -0.00000000419974+0.76159415595576i)
(num-test (tan 0.00000001-1.0i) 0.00000000419974-0.76159415595576i)
(num-test (tan -0.00000001-1.0i) -0.00000000419974-0.76159415595576i)
(num-test (tan 0.00000001+3.14159265358979i) 0.00000000007442+0.99627207622075i)
(num-test (tan -0.00000001+3.14159265358979i) -0.00000000007442+0.99627207622075i)
(num-test (tan 0.00000001-3.14159265358979i) 0.00000000007442-0.99627207622075i)
(num-test (tan -0.00000001-3.14159265358979i) -0.00000000007442-0.99627207622075i)
(num-test (tan 0.00000001+2.71828182845905i) 0.00000000017267+0.99132891580060i)
(num-test (tan -0.00000001+2.71828182845905i) -0.00000000017267+0.99132891580060i)
(num-test (tan 0.00000001-2.71828182845905i) 0.00000000017267-0.99132891580060i)
(num-test (tan -0.00000001-2.71828182845905i) -0.00000000017267-0.99132891580060i)
(num-test (tan 1.0+0.0i) 1.55740772465490)
(num-test (tan -1.0+0.0i) -1.55740772465490)
(num-test (tan 1.0-0.0i) 1.55740772465490)
(num-test (tan -1.0-0.0i) -1.55740772465490)
(num-test (tan 1.0+0.00000001i) 1.55740772465490+0.00000003425519i)
(num-test (tan -1.0+0.00000001i) -1.55740772465490+0.00000003425519i)
(num-test (tan 1.0-0.00000001i) 1.55740772465490-0.00000003425519i)
(num-test (tan -1.0-0.00000001i) -1.55740772465490-0.00000003425519i)
(num-test (tan 1.0+1.0i) 0.27175258531951+1.08392332733869i)
(num-test (tan -1.0+1.0i) -0.27175258531951+1.08392332733869i)
(num-test (tan 1.0-1.0i) 0.27175258531951-1.08392332733869i)
(num-test (tan -1.0-1.0i) -0.27175258531951-1.08392332733869i)
(num-test (tan 1.0+3.14159265358979i) 0.00340139653674+1.00154968930275i)
(num-test (tan -1.0+3.14159265358979i) -0.00340139653674+1.00154968930275i)
(num-test (tan 1.0-3.14159265358979i) 0.00340139653674-1.00154968930275i)
(num-test (tan -1.0-3.14159265358979i) -0.00340139653674-1.00154968930275i)
(num-test (tan 1.0+2.71828182845905i) 0.00794757997665+1.00359921084211i)
(num-test (tan -1.0+2.71828182845905i) -0.00794757997665+1.00359921084211i)
(num-test (tan 1.0-2.71828182845905i) 0.00794757997665-1.00359921084211i)
(num-test (tan -1.0-2.71828182845905i) -0.00794757997665-1.00359921084211i)
(num-test (tan 1234.0+1234.0i) 2.7e-20+1.0i)
(num-test (tan 1234.0-1234.0i) 2.7e-20-1.0i)
(num-test (tan 10/3) .1941255059835657) 
(num-test (tan 1234/3) -0.2186940320047828) 
(num-test (tan 1234/10) 1.204471256531804) 
(num-test (tan 1234000000/3) -18.78094727276203)              ; -18.78094815124064721190472 according to arprec's mathtool
(num-test (tan 1234000000/500029) -7.31654379832009) 
(num-test (tan 1234000000/362880) 4.911576750502133) 
(num-test (tan 500029/2) .2804673425353792) 
(num-test (tan 500029/3) -1.173139817032177) 
(num-test (tan 500029/10) 3.767116303516932) 
(num-test (tan 500029/1234) -0.05562302342803592) 
(num-test (tan 362880/1234) -2.927532635052267) 
(num-test (tan our-pi) 6.982889851335445E-15) 
(num-test (tan 2.71828182845905) -0.4505495340698621) 
(num-test (tan 0.00000001+1234.0i) +8.077935669463161E-28+1.0i) 
(num-test (tan 0.00000001+1234000000.0i) 0.0+1.0i) 
(num-test (tan 3.14159265358979+0.0i) 6.982889851335445E-15) 
(num-test (tan 3.14159265358979+0.00000001i) +6.982889851335444E-15+1.0E-8i) 
(num-test (tan 3.14159265358979+1.0i) +2.932634567877868E-15+0.7615941559557649i) 
(num-test (tan 3.14159265358979+3.14159265358979i) +5.196631812627532E-17+0.99627207622075i) 
(num-test (tan 3.14159265358979+2.71828182845905i) +1.205734242765375E-16+0.991328915800599i) 
(num-test (tan 3.14159265358979+1234.0i) 0.0+1.0i) 
(num-test (tan 3.14159265358979+1234000000.0i) -7.703719777548943E-34+1.0i) 
(num-test (tan 2.71828182845905+0.0i) -0.4505495340698621) 
(num-test (tan 2.71828182845905+0.00000001i) -0.4505495340698621+1.2029948826505699E-8i) 
(num-test (tan 2.71828182845905+1.0i) -0.1692870118766369+0.8196826057997404i) 
(num-test (tan 2.71828182845905+3.14159265358979i) -0.002790687681003331+0.9975247319761639i) 
(num-test (tan 2.71828182845905+2.71828182845905i) -0.00648578276962794+0.9942257438545914i) 
(num-test (tan 2.71828182845905+1234.0i) +2.710505431213761E-20+1.0i) 
(num-test (tan 2.71828182845905+1234000000.0i) +2.710505431213761E-20+1.0i) 
(num-test (tan 1234.0+0.00000001i) -0.7537751984442328+1.5681770497896427E-8i) 
(num-test (tan 1234.0+3.14159265358979i) -0.003586791196867043+0.9989656315245496i) 
(num-test (tan 1234.0+2.71828182845905i) -0.008351965390936033+0.9975698313220817i) 
(num-test (tan 1234000000.0+0.00000001i) -6.212941995900324+3.9600648244422054E-7i) 
(num-test (tan 1234000000.0+3.14159265358979i) -0.001176098307980411+1.003551866736695i) 
(num-test (tan 1234000000.0+2.71828182845905i) -0.002755390838840499+1.008299558244272i) 
;(num-test (tan 1.5707963259845+2.0630965522972e-18i) 1.234000079689074E+9+3.141593059344448i)
(num-test (tan -3.45266983001243932001e-04+0.0e+00i) -3.4526699672092183585e-4+0.0i)
(num-test (tan 3.45266983001243932001e-04+0.0e+00i) 3.4526699672092183585e-4+0.0i)
(num-test (tan -3.45266983001243932001e-04+1.19209289550781250e-07i) -3.4526699672091692931e-4+1.1920930376163652989e-7i)
(num-test (tan -3.45266983001243932001e-04-1.19209289550781250e-07i) -3.4526699672091692931e-4-1.1920930376163652989e-7i)
(num-test (tan 3.45266983001243932001e-04+1.19209289550781250e-07i) 3.4526699672091692931e-4+1.1920930376163652989e-7i)
(num-test (tan 3.45266983001243932001e-04-1.19209289550781250e-07i) 3.4526699672091692931e-4-1.1920930376163652989e-7i)
(num-test (tan -3.45266983001243932001e-04+5.0e-01i) -2.7153443992655805934e-4+4.6211720058436229979e-1i)
(num-test (tan -3.45266983001243932001e-04-5.0e-01i) -2.7153443992655805934e-4-4.6211720058436229979e-1i)
(num-test (tan 3.45266983001243932001e-04+5.0e-01i) 2.7153443992655805934e-4+4.6211720058436229979e-1i)
(num-test (tan 3.45266983001243932001e-04-5.0e-01i) 2.7153443992655805934e-4-4.6211720058436229979e-1i)
(num-test (tan -3.45266983001243932001e-04+1.0e+00i) -1.4500326960274960880e-4+7.6159419408485704836e-1i)
(num-test (tan -3.45266983001243932001e-04-1.0e+00i) -1.4500326960274960880e-4-7.6159419408485704836e-1i)
(num-test (tan 3.45266983001243932001e-04+1.0e+00i) 1.4500326960274960880e-4+7.6159419408485704836e-1i)
(num-test (tan 3.45266983001243932001e-04-1.0e+00i) 1.4500326960274960880e-4-7.6159419408485704836e-1i)
(num-test (tan -3.45266983001243932001e-04+2.0e+00i) -2.4393395410435306874e-5+9.6402758819508310556e-1i)
(num-test (tan -3.45266983001243932001e-04-2.0e+00i) -2.4393395410435306874e-5-9.6402758819508310556e-1i)
(num-test (tan 3.45266983001243932001e-04+2.0e+00i) 2.4393395410435306874e-5+9.6402758819508310556e-1i)
(num-test (tan 3.45266983001243932001e-04-2.0e+00i) 2.4393395410435306874e-5-9.6402758819508310556e-1i)
(num-test (tan 1.57045105981189525579e+00+0.0e+00i) 2.8963092606501007060e3+0.0i)
(num-test (tan -1.57045105981189525579e+00+0.0e+00i) -2.8963092606501007060e3+0.0i)
(num-test (tan 1.57045105981189525579e+00+1.19209289550781250e-07i) 2.8963089153831588642e3+9.9999992052646305569e-1i)
(num-test (tan 1.57045105981189525579e+00-1.19209289550781250e-07i) 2.8963089153831588642e3-9.9999992052646305569e-1i)
(num-test (tan -1.57045105981189525579e+00+1.19209289550781250e-07i) -2.8963089153831588642e3+9.9999992052646305569e-1i)
(num-test (tan -1.57045105981189525579e+00-1.19209289550781250e-07i) -2.8963089153831588642e3-9.9999992052646305569e-1i)
(num-test (tan 1.57045105981189525579e+00+5.0e-01i) 1.2715121175455623363e-3+2.1639524637389325996e0i)
(num-test (tan 1.57045105981189525579e+00-5.0e-01i) 1.2715121175455623363e-3-2.1639524637389325996e0i)
(num-test (tan -1.57045105981189525579e+00+5.0e-01i) -1.2715121175455623363e-3+2.1639524637389325996e0i)
(num-test (tan -1.57045105981189525579e+00-5.0e-01i) -1.2715121175455623363e-3-2.1639524637389325996e0i)
(num-test (tan 1.57045105981189525579e+00+1.0e+00i) 2.4999454374276273814e-4+1.3130351721648674823e0i)
(num-test (tan 1.57045105981189525579e+00-1.0e+00i) 2.4999454374276273814e-4-1.3130351721648674823e0i)
(num-test (tan -1.57045105981189525579e+00+1.0e+00i) -2.4999454374276273814e-4+1.3130351721648674823e0i)
(num-test (tan -1.57045105981189525579e+00-1.0e+00i) -2.4999454374276273814e-4-1.3130351721648674823e0i)
(num-test (tan 1.57045105981189525579e+00+2.0e+00i) 2.6247825506572821595e-5+1.0373147113268752620e0i)
(num-test (tan 1.57045105981189525579e+00-2.0e+00i) 2.6247825506572821595e-5-1.0373147113268752620e0i)
(num-test (tan -1.57045105981189525579e+00+2.0e+00i) -2.6247825506572821595e-5+1.0373147113268752620e0i)
(num-test (tan -1.57045105981189525579e+00-2.0e+00i) -2.6247825506572821595e-5-1.0373147113268752620e0i)
(num-test (tan 1.57114159377789786021e+00+0.0e+00i) -2.8963092606511280143e3+0.0i)
(num-test (tan -1.57114159377789786021e+00+0.0e+00i) 2.8963092606511280143e3+0.0i)
(num-test (tan 1.57114159377789786021e+00+1.19209289550781250e-07i) -2.8963089153841861720e3+9.9999992052717244672e-1i)
(num-test (tan 1.57114159377789786021e+00-1.19209289550781250e-07i) -2.8963089153841861720e3-9.9999992052717244672e-1i)
(num-test (tan -1.57114159377789786021e+00+1.19209289550781250e-07i) 2.8963089153841861720e3+9.9999992052717244672e-1i)
(num-test (tan -1.57114159377789786021e+00-1.19209289550781250e-07i) 2.8963089153841861720e3-9.9999992052717244672e-1i)
(num-test (tan 1.57114159377789786021e+00+5.0e-01i) -1.2715121175451113370e-3+2.1639524637389326002e0i)
(num-test (tan 1.57114159377789786021e+00-5.0e-01i) -1.2715121175451113370e-3-2.1639524637389326002e0i)
(num-test (tan -1.57114159377789786021e+00+5.0e-01i) 1.2715121175451113370e-3+2.1639524637389326002e0i)
(num-test (tan -1.57114159377789786021e+00-5.0e-01i) 1.2715121175451113370e-3-2.1639524637389326002e0i)
(num-test (tan 1.57114159377789786021e+00+1.0e+00i) -2.4999454374267406620e-4+1.3130351721648674824e0i)
(num-test (tan 1.57114159377789786021e+00-1.0e+00i) -2.4999454374267406620e-4-1.3130351721648674824e0i)
(num-test (tan -1.57114159377789786021e+00+1.0e+00i) 2.4999454374267406620e-4+1.3130351721648674824e0i)
(num-test (tan -1.57114159377789786021e+00-1.0e+00i) 2.4999454374267406620e-4-1.3130351721648674824e0i)
(num-test (tan 1.57114159377789786021e+00+2.0e+00i) -2.6247825506563511609e-5+1.0373147113268752620e0i)
(num-test (tan 1.57114159377789786021e+00-2.0e+00i) -2.6247825506563511609e-5-1.0373147113268752620e0i)
(num-test (tan -1.57114159377789786021e+00+2.0e+00i) 2.6247825506563511609e-5+1.0373147113268752620e0i)
(num-test (tan -1.57114159377789786021e+00-2.0e+00i) 2.6247825506563511609e-5-1.0373147113268752620e0i)
(num-test (tan 3.14124738660679181379e+00+0.0e+00i) -3.4526699672110257641e-4+0.0i)
(num-test (tan -3.14124738660679181379e+00+0.0e+00i) 3.4526699672110257641e-4+0.0i)
(num-test (tan 3.14124738660679181379e+00+1.19209289550781250e-07i) -3.4526699672109766987e-4+1.1920930376163652991e-7i)
(num-test (tan 3.14124738660679181379e+00-1.19209289550781250e-07i) -3.4526699672109766987e-4-1.1920930376163652991e-7i)
(num-test (tan -3.14124738660679181379e+00+1.19209289550781250e-07i) 3.4526699672109766987e-4+1.1920930376163652991e-7i)
(num-test (tan -3.14124738660679181379e+00-1.19209289550781250e-07i) 3.4526699672109766987e-4-1.1920930376163652991e-7i)
(num-test (tan 3.14124738660679181379e+00+5.0e-01i) -2.7153443992670020234e-4+4.6211720058436229984e-1i)
(num-test (tan 3.14124738660679181379e+00-5.0e-01i) -2.7153443992670020234e-4-4.6211720058436229984e-1i)
(num-test (tan -3.14124738660679181379e+00+5.0e-01i) 2.7153443992670020234e-4+4.6211720058436229984e-1i)
(num-test (tan -3.14124738660679181379e+00-5.0e-01i) 2.7153443992670020234e-4-4.6211720058436229984e-1i)
(num-test (tan 3.14124738660679181379e+00+1.0e+00i) -1.4500326960282551519e-4+7.6159419408485704840e-1i)
(num-test (tan 3.14124738660679181379e+00-1.0e+00i) -1.4500326960282551519e-4-7.6159419408485704840e-1i)
(num-test (tan -3.14124738660679181379e+00+1.0e+00i) 1.4500326960282551519e-4+7.6159419408485704840e-1i)
(num-test (tan -3.14124738660679181379e+00-1.0e+00i) 1.4500326960282551519e-4-7.6159419408485704840e-1i)
(num-test (tan 3.14124738660679181379e+00+2.0e+00i) -2.4393395410448076340e-5+9.6402758819508310557e-1i)
(num-test (tan 3.14124738660679181379e+00-2.0e+00i) -2.4393395410448076340e-5-9.6402758819508310557e-1i)
(num-test (tan -3.14124738660679181379e+00+2.0e+00i) 2.4393395410448076340e-5+9.6402758819508310557e-1i)
(num-test (tan -3.14124738660679181379e+00-2.0e+00i) 2.4393395410448076340e-5-9.6402758819508310557e-1i)
(num-test (tan 3.14193792057279441821e+00+0.0e+00i) 3.4526699672085764703e-4+0.0i)
(num-test (tan -3.14193792057279441821e+00+0.0e+00i) -3.4526699672085764703e-4+0.0i)
(num-test (tan 3.14193792057279441821e+00+1.19209289550781250e-07i) 3.4526699672085274049e-4+1.1920930376163652989e-7i)
(num-test (tan 3.14193792057279441821e+00-1.19209289550781250e-07i) 3.4526699672085274049e-4-1.1920930376163652989e-7i)
(num-test (tan -3.14193792057279441821e+00+1.19209289550781250e-07i) -3.4526699672085274049e-4+1.1920930376163652989e-7i)
(num-test (tan -3.14193792057279441821e+00-1.19209289550781250e-07i) -3.4526699672085274049e-4-1.1920930376163652989e-7i)
(num-test (tan 3.14193792057279441821e+00+5.0e-01i) 2.7153443992650757820e-4+4.6211720058436229978e-1i)
(num-test (tan 3.14193792057279441821e+00-5.0e-01i) 2.7153443992650757820e-4-4.6211720058436229978e-1i)
(num-test (tan -3.14193792057279441821e+00+5.0e-01i) -2.7153443992650757820e-4+4.6211720058436229978e-1i)
(num-test (tan -3.14193792057279441821e+00-5.0e-01i) -2.7153443992650757820e-4-4.6211720058436229978e-1i)
(num-test (tan 3.14193792057279441821e+00+1.0e+00i) 1.4500326960272265115e-4+7.6159419408485704835e-1i)
(num-test (tan 3.14193792057279441821e+00-1.0e+00i) 1.4500326960272265115e-4-7.6159419408485704835e-1i)
(num-test (tan -3.14193792057279441821e+00+1.0e+00i) -1.4500326960272265115e-4+7.6159419408485704835e-1i)
(num-test (tan -3.14193792057279441821e+00-1.0e+00i) -1.4500326960272265115e-4-7.6159419408485704835e-1i)
(num-test (tan 3.14193792057279441821e+00+2.0e+00i) 2.4393395410430771882e-5+9.6402758819508310556e-1i)
(num-test (tan 3.14193792057279441821e+00-2.0e+00i) 2.4393395410430771882e-5-9.6402758819508310556e-1i)
(num-test (tan -3.14193792057279441821e+00+2.0e+00i) -2.4393395410430771882e-5+9.6402758819508310556e-1i)
(num-test (tan -3.14193792057279441821e+00-2.0e+00i) -2.4393395410430771882e-5-9.6402758819508310556e-1i)
(num-test (tan 4.71204371340168837179e+00+0.0e+00i) 2.8963092606490733978e3+0.0i)
(num-test (tan -4.71204371340168837179e+00+0.0e+00i) -2.8963092606490733978e3+0.0i)
(num-test (tan 4.71204371340168837179e+00+1.19209289550781250e-07i) 2.8963089153821315563e3+9.9999992052575366466e-1i)
(num-test (tan 4.71204371340168837179e+00-1.19209289550781250e-07i) 2.8963089153821315563e3-9.9999992052575366466e-1i)
(num-test (tan -4.71204371340168837179e+00+1.19209289550781250e-07i) -2.8963089153821315563e3+9.9999992052575366466e-1i)
(num-test (tan -4.71204371340168837179e+00-1.19209289550781250e-07i) -2.8963089153821315563e3-9.9999992052575366466e-1i)
(num-test (tan 4.71204371340168837179e+00+5.0e-01i) 1.2715121175460133355e-3+2.1639524637389325989e0i)
(num-test (tan 4.71204371340168837179e+00-5.0e-01i) 1.2715121175460133355e-3-2.1639524637389325989e0i)
(num-test (tan -4.71204371340168837179e+00+5.0e-01i) -1.2715121175460133355e-3+2.1639524637389325989e0i)
(num-test (tan -4.71204371340168837179e+00-5.0e-01i) -1.2715121175460133355e-3-2.1639524637389325989e0i)
(num-test (tan 4.71204371340168837179e+00+1.0e+00i) 2.4999454374285141007e-4+1.3130351721648674822e0i)
(num-test (tan 4.71204371340168837179e+00-1.0e+00i) 2.4999454374285141007e-4-1.3130351721648674822e0i)
(num-test (tan -4.71204371340168837179e+00+1.0e+00i) -2.4999454374285141007e-4+1.3130351721648674822e0i)
(num-test (tan -4.71204371340168837179e+00-1.0e+00i) -2.4999454374285141007e-4-1.3130351721648674822e0i)
(num-test (tan 4.71204371340168837179e+00+2.0e+00i) 2.6247825506582131582e-5+1.0373147113268752620e0i)
(num-test (tan 4.71204371340168837179e+00-2.0e+00i) 2.6247825506582131582e-5-1.0373147113268752620e0i)
(num-test (tan -4.71204371340168837179e+00+2.0e+00i) -2.6247825506582131582e-5+1.0373147113268752620e0i)
(num-test (tan -4.71204371340168837179e+00-2.0e+00i) -2.6247825506582131582e-5-1.0373147113268752620e0i)
(num-test (tan 4.71273424736769097620e+00+0.0e+00i) -2.8963092606521553225e3+0.0i)
(num-test (tan -4.71273424736769097620e+00+0.0e+00i) 2.8963092606521553225e3+0.0i)
(num-test (tan 4.71273424736769097620e+00+1.19209289550781250e-07i) -2.8963089153852134799e3+9.9999992052788183776e-1i)
(num-test (tan 4.71273424736769097620e+00-1.19209289550781250e-07i) -2.8963089153852134799e3-9.9999992052788183776e-1i)
(num-test (tan -4.71273424736769097620e+00+1.19209289550781250e-07i) 2.8963089153852134799e3+9.9999992052788183776e-1i)
(num-test (tan -4.71273424736769097620e+00-1.19209289550781250e-07i) 2.8963089153852134799e3-9.9999992052788183776e-1i)
(num-test (tan 4.71273424736769097620e+00+5.0e-01i) -1.2715121175446603377e-3+2.1639524637389326009e0i)
(num-test (tan 4.71273424736769097620e+00-5.0e-01i) -1.2715121175446603377e-3-2.1639524637389326009e0i)
(num-test (tan -4.71273424736769097620e+00+5.0e-01i) 1.2715121175446603377e-3+2.1639524637389326009e0i)
(num-test (tan -4.71273424736769097620e+00-5.0e-01i) 1.2715121175446603377e-3-2.1639524637389326009e0i)
(num-test (tan 4.71273424736769097620e+00+1.0e+00i) -2.4999454374258539427e-4+1.3130351721648674825e0i)
(num-test (tan 4.71273424736769097620e+00-1.0e+00i) -2.4999454374258539427e-4-1.3130351721648674825e0i)
(num-test (tan -4.71273424736769097620e+00+1.0e+00i) 2.4999454374258539427e-4+1.3130351721648674825e0i)
(num-test (tan -4.71273424736769097620e+00-1.0e+00i) 2.4999454374258539427e-4-1.3130351721648674825e0i)
(num-test (tan 4.71273424736769097620e+00+2.0e+00i) -2.6247825506554201622e-5+1.0373147113268752620e0i)
(num-test (tan 4.71273424736769097620e+00-2.0e+00i) -2.6247825506554201622e-5-1.0373147113268752620e0i)
(num-test (tan -4.71273424736769097620e+00+2.0e+00i) 2.6247825506554201622e-5+1.0373147113268752620e0i)
(num-test (tan -4.71273424736769097620e+00-2.0e+00i) 2.6247825506554201622e-5-1.0373147113268752620e0i)
(num-test (tan 6.28284004019658492979e+00+0.0e+00i) -3.4526699672122504111e-4+0.0i)
(num-test (tan -6.28284004019658492979e+00+0.0e+00i) 3.4526699672122504111e-4+0.0i)
(num-test (tan 6.28284004019658492979e+00+1.19209289550781250e-07i) -3.4526699672122013457e-4+1.1920930376163652992e-7i)
(num-test (tan 6.28284004019658492979e+00-1.19209289550781250e-07i) -3.4526699672122013457e-4-1.1920930376163652992e-7i)
(num-test (tan -6.28284004019658492979e+00+1.19209289550781250e-07i) 3.4526699672122013457e-4+1.1920930376163652992e-7i)
(num-test (tan -6.28284004019658492979e+00-1.19209289550781250e-07i) 3.4526699672122013457e-4-1.1920930376163652992e-7i)
(num-test (tan 6.28284004019658492979e+00+5.0e-01i) -2.7153443992679651442e-4+4.6211720058436229987e-1i)
(num-test (tan 6.28284004019658492979e+00-5.0e-01i) -2.7153443992679651442e-4-4.6211720058436229987e-1i)
(num-test (tan -6.28284004019658492979e+00+5.0e-01i) 2.7153443992679651442e-4+4.6211720058436229987e-1i)
(num-test (tan -6.28284004019658492979e+00-5.0e-01i) 2.7153443992679651442e-4-4.6211720058436229987e-1i)
(num-test (tan 6.28284004019658492979e+00+1.0e+00i) -1.4500326960287694721e-4+7.6159419408485704843e-1i)
(num-test (tan 6.28284004019658492979e+00-1.0e+00i) -1.4500326960287694721e-4-7.6159419408485704843e-1i)
(num-test (tan -6.28284004019658492979e+00+1.0e+00i) 1.4500326960287694721e-4+7.6159419408485704843e-1i)
(num-test (tan -6.28284004019658492979e+00-1.0e+00i) 1.4500326960287694721e-4-7.6159419408485704843e-1i)
(num-test (tan 6.28284004019658492979e+00+2.0e+00i) -2.4393395410456728569e-5+9.6402758819508310558e-1i)
(num-test (tan 6.28284004019658492979e+00-2.0e+00i) -2.4393395410456728569e-5-9.6402758819508310558e-1i)
(num-test (tan -6.28284004019658492979e+00+2.0e+00i) 2.4393395410456728569e-5+9.6402758819508310558e-1i)
(num-test (tan -6.28284004019658492979e+00-2.0e+00i) 2.4393395410456728569e-5-9.6402758819508310558e-1i)
(num-test (tan 6.28353057416258753420e+00+0.0e+00i) 3.4526699672073518233e-4+0.0i)
(num-test (tan -6.28353057416258753420e+00+0.0e+00i) -3.4526699672073518233e-4+0.0i)
(num-test (tan 6.28353057416258753420e+00+1.19209289550781250e-07i) 3.4526699672073027579e-4+1.1920930376163652988e-7i)
(num-test (tan 6.28353057416258753420e+00-1.19209289550781250e-07i) 3.4526699672073027579e-4-1.1920930376163652988e-7i)
(num-test (tan -6.28353057416258753420e+00+1.19209289550781250e-07i) -3.4526699672073027579e-4+1.1920930376163652988e-7i)
(num-test (tan -6.28353057416258753420e+00-1.19209289550781250e-07i) -3.4526699672073027579e-4-1.1920930376163652988e-7i)
(num-test (tan 6.28353057416258753420e+00+5.0e-01i) 2.7153443992641126612e-4+4.6211720058436229974e-1i)
(num-test (tan 6.28353057416258753420e+00-5.0e-01i) 2.7153443992641126612e-4-4.6211720058436229974e-1i)
(num-test (tan -6.28353057416258753420e+00+5.0e-01i) -2.7153443992641126612e-4+4.6211720058436229974e-1i)
(num-test (tan -6.28353057416258753420e+00-5.0e-01i) -2.7153443992641126612e-4-4.6211720058436229974e-1i)
(num-test (tan 6.28353057416258753420e+00+1.0e+00i) 1.4500326960267121913e-4+7.6159419408485704832e-1i)
(num-test (tan 6.28353057416258753420e+00-1.0e+00i) 1.4500326960267121913e-4-7.6159419408485704832e-1i)
(num-test (tan -6.28353057416258753420e+00+1.0e+00i) -1.4500326960267121913e-4+7.6159419408485704832e-1i)
(num-test (tan -6.28353057416258753420e+00-1.0e+00i) -1.4500326960267121913e-4-7.6159419408485704832e-1i)
(num-test (tan 6.28353057416258753420e+00+2.0e+00i) 2.4393395410422119654e-5+9.6402758819508310555e-1i)
(num-test (tan 6.28353057416258753420e+00-2.0e+00i) 2.4393395410422119654e-5-9.6402758819508310555e-1i)
(num-test (tan -6.28353057416258753420e+00+2.0e+00i) -2.4393395410422119654e-5+9.6402758819508310555e-1i)
(num-test (tan -6.28353057416258753420e+00-2.0e+00i) -2.4393395410422119654e-5-9.6402758819508310555e-1i)
(num-test (tan 9.42443269378637893396e+00+0.0e+00i) -3.4526699672045932728e-4+0.0i)
(num-test (tan -9.42443269378637893396e+00+0.0e+00i) 3.4526699672045932728e-4+0.0i)
(num-test (tan 9.42443269378637893396e+00+1.19209289550781250e-07i) -3.4526699672045442074e-4+1.1920930376163652985e-7i)
(num-test (tan 9.42443269378637893396e+00-1.19209289550781250e-07i) -3.4526699672045442074e-4-1.1920930376163652985e-7i)
(num-test (tan -9.42443269378637893396e+00+1.19209289550781250e-07i) 3.4526699672045442074e-4+1.1920930376163652985e-7i)
(num-test (tan -9.42443269378637893396e+00-1.19209289550781250e-07i) 3.4526699672045442074e-4-1.1920930376163652985e-7i)
(num-test (tan 9.42443269378637893396e+00+5.0e-01i) -2.7153443992619432056e-4+4.6211720058436229968e-1i)
(num-test (tan 9.42443269378637893396e+00-5.0e-01i) -2.7153443992619432056e-4-4.6211720058436229968e-1i)
(num-test (tan -9.42443269378637893396e+00+5.0e-01i) 2.7153443992619432056e-4+4.6211720058436229968e-1i)
(num-test (tan -9.42443269378637893396e+00-5.0e-01i) 2.7153443992619432056e-4-4.6211720058436229968e-1i)
(num-test (tan 9.42443269378637893396e+00+1.0e+00i) -1.4500326960255536711e-4+7.6159419408485704826e-1i)
(num-test (tan 9.42443269378637893396e+00-1.0e+00i) -1.4500326960255536711e-4-7.6159419408485704826e-1i)
(num-test (tan -9.42443269378637893396e+00+1.0e+00i) 1.4500326960255536711e-4+7.6159419408485704826e-1i)
(num-test (tan -9.42443269378637893396e+00-1.0e+00i) 1.4500326960255536711e-4-7.6159419408485704826e-1i)
(num-test (tan 9.42443269378637893396e+00+2.0e+00i) -2.4393395410402630273e-5+9.6402758819508310554e-1i)
(num-test (tan 9.42443269378637893396e+00-2.0e+00i) -2.4393395410402630273e-5-9.6402758819508310554e-1i)
(num-test (tan -9.42443269378637893396e+00+2.0e+00i) 2.4393395410402630273e-5+9.6402758819508310554e-1i)
(num-test (tan -9.42443269378637893396e+00-2.0e+00i) 2.4393395410402630273e-5-9.6402758819508310554e-1i)
(num-test (tan 9.42512322775237976202e+00+0.0e+00i) 3.4526699671972453911e-4+0.0i)
(num-test (tan -9.42512322775237976202e+00+0.0e+00i) -3.4526699671972453911e-4+0.0i)
(num-test (tan 9.42512322775237976202e+00+1.19209289550781250e-07i) 3.4526699671971963257e-4+1.1920930376163652979e-7i)
(num-test (tan 9.42512322775237976202e+00-1.19209289550781250e-07i) 3.4526699671971963257e-4-1.1920930376163652979e-7i)
(num-test (tan -9.42512322775237976202e+00+1.19209289550781250e-07i) -3.4526699671971963257e-4+1.1920930376163652979e-7i)
(num-test (tan -9.42512322775237976202e+00-1.19209289550781250e-07i) -3.4526699671971963257e-4-1.1920930376163652979e-7i)
(num-test (tan 9.42512322775237976202e+00+5.0e-01i) 2.7153443992561644811e-4+4.6211720058436229949e-1i)
(num-test (tan 9.42512322775237976202e+00-5.0e-01i) 2.7153443992561644811e-4-4.6211720058436229949e-1i)
(num-test (tan -9.42512322775237976202e+00+5.0e-01i) -2.7153443992561644811e-4+4.6211720058436229949e-1i)
(num-test (tan -9.42512322775237976202e+00-5.0e-01i) -2.7153443992561644811e-4-4.6211720058436229949e-1i)
(num-test (tan 9.42512322775237976202e+00+1.0e+00i) 1.450032696022467750e-4+7.6159419408485704810e-1i)
(num-test (tan 9.42512322775237976202e+00-1.0e+00i) 1.450032696022467750e-4-7.6159419408485704810e-1i)
(num-test (tan -9.42512322775237976202e+00+1.0e+00i) -1.450032696022467750e-4+7.6159419408485704810e-1i)
(num-test (tan -9.42512322775237976202e+00-1.0e+00i) -1.450032696022467750e-4-7.6159419408485704810e-1i)
(num-test (tan 9.42512322775237976202e+00+2.0e+00i) 2.439339541035071690e-5+9.6402758819508310550e-1i)
(num-test (tan 9.42512322775237976202e+00-2.0e+00i) 2.439339541035071690e-5-9.6402758819508310550e-1i)
(num-test (tan -9.42512322775237976202e+00+2.0e+00i) -2.439339541035071690e-5+9.6402758819508310550e-1i)
(num-test (tan -9.42512322775237976202e+00-2.0e+00i) -2.439339541035071690e-5-9.6402758819508310550e-1i)


;; -------- asin
(num-test (asin 0) 0.0)
(num-test (asin 1) 1.57079632679490)
(num-test (asin -1) -1.57079632679490)
(num-test (asin 2) 1.57079632679490+1.31695789692482i)
(num-test (asin -2) -1.57079632679490+1.31695789692482i)
(num-test (asin 3) 1.57079632679490+1.76274717403909i)
(num-test (asin -3) -1.57079632679490+1.76274717403909i)
(num-test (asin 10) 1.57079632679490+2.99322284612638i)
(num-test (asin -10) -1.57079632679490+2.99322284612638i)
(num-test (asin 1234) 1.57079632679490+7.81116322068415i)
(num-test (asin -1234) -1.57079632679490+7.81116322068415i)
(num-test (asin 0/1) 0.0)
(num-test (asin 0/2) 0.0)
(num-test (asin 0/3) 0.0)
(num-test (asin 0/10) 0.0)
(num-test (asin 0/1234) 0.0)
(num-test (asin 0/1234000000) 0.0)
(num-test (asin 0/500029) 0.0)
(num-test (asin 0/362880) 0.0)
(num-test (asin 1/1) 1.57079632679490)
(num-test (asin -1/1) -1.57079632679490)
(num-test (asin 1/2) 0.52359877559830)
(num-test (asin -1/2) -0.52359877559830)
(num-test (asin 1/3) 0.33983690945412)
(num-test (asin -1/3) -0.33983690945412)
(num-test (asin 1/10) 0.10016742116156)
(num-test (asin -1/10) -0.10016742116156)
(num-test (asin 1/1234) 0.00081037286017)
(num-test (asin -1/1234) -0.00081037286017)
(num-test (asin 1/1234000000) 0.00000000081037)
(num-test (asin -1/1234000000) -0.00000000081037)
(num-test (asin 1/500029) 0.00000199988401)
(num-test (asin -1/500029) -0.00000199988401)
(num-test (asin 1/362880) 0.00000275573192)
(num-test (asin -1/362880) -0.00000275573192)
(num-test (asin 2/1) 1.57079632679490+1.31695789692482i)
(num-test (asin -2/1) -1.57079632679490+1.31695789692482i)
(num-test (asin 2/2) 1.57079632679490)
(num-test (asin -2/2) -1.57079632679490)
(num-test (asin 2/3) 0.72972765622697)
(num-test (asin -2/3) -0.72972765622697)
(num-test (asin 2/10) 0.20135792079033)
(num-test (asin -2/10) -0.20135792079033)
(num-test (asin 2/1234) 0.00162074625252)
(num-test (asin -2/1234) -0.00162074625252)
(num-test (asin 2/1234000000) 0.00000000162075)
(num-test (asin -2/1234000000) -0.00000000162075)
(num-test (asin 2/500029) 0.00000399976801)
(num-test (asin -2/500029) -0.00000399976801)
(num-test (asin 2/362880) 0.00000551146384)
(num-test (asin -2/362880) -0.00000551146384)
(num-test (asin 3/1) 1.57079632679490+1.76274717403909i)
(num-test (asin -3/1) -1.57079632679490+1.76274717403909i)
(num-test (asin 3/2) 1.57079632679490+0.96242365011921i)
(num-test (asin -3/2) -1.57079632679490+0.96242365011921i)
(num-test (asin 3/3) 1.57079632679490)
(num-test (asin -3/3) -1.57079632679490)
(num-test (asin 3/10) 0.30469265401540)
(num-test (asin -3/10) -0.30469265401540)
(num-test (asin 3/1234) 0.00243112070922)
(num-test (asin -3/1234) -0.00243112070922)
(num-test (asin 3/1234000000) 0.00000000243112)
(num-test (asin -3/1234000000) -0.00000000243112)
(num-test (asin 3/500029) 0.00000599965202)
(num-test (asin -3/500029) -0.00000599965202)
(num-test (asin 3/362880) 0.00000826719577+0.0i)
(num-test (asin -3/362880) -0.00000826719577+0.0i)
(num-test (asin 10/1) 1.57079632679490+2.99322284612638i)
(num-test (asin -10/1) -1.57079632679490+2.99322284612638i)
(num-test (asin 10/2) 1.57079632679490+2.29243166956117i)
(num-test (asin -10/2) -1.57079632679490+2.29243166956117i)
(num-test (asin 10/3) 1.57079632679490+1.87382024252741i)
(num-test (asin -10/3) -1.57079632679490+1.87382024252741i)
(num-test (asin 10/10) 1.57079632679490)
(num-test (asin -10/10) -1.57079632679490)
(num-test (asin 10/1234) 0.00810381641321)
(num-test (asin -10/1234) -0.00810381641321)
(num-test (asin 10/1234000000) 0.00000000810373)
(num-test (asin -10/1234000000) -0.00000000810373)
(num-test (asin 10/500029) 0.00001999884007)
(num-test (asin -10/500029) -0.00001999884007)
(num-test (asin 10/362880) 0.00002755731923)
(num-test (asin -10/362880) -0.00002755731923)
(num-test (asin 1234/1) 1.57079632679490+7.81116322068415i)
(num-test (asin -1234/1) -1.57079632679490+7.81116322068415i)
(num-test (asin 1234/2) 1.57079632679490+7.11801554770806i)
(num-test (asin -1234/2) -1.57079632679490+7.11801554770806i)
(num-test (asin 1234/3) 1.57079632679490+6.71254961876657i)
(num-test (asin -1234/3) -1.57079632679490+6.71254961876657i)
(num-test (asin 1234/10) 1.57079632679490+5.50856187402751i)
(num-test (asin -1234/10) -1.57079632679490+5.50856187402751i)
(num-test (asin 1234/1234) 1.57079632679490)
(num-test (asin -1234/1234) -1.57079632679490)
(num-test (asin 1234/1234000000) 0.00000100000000)
(num-test (asin -1234/1234000000) -0.00000100000000)
(num-test (asin 1234/500029) 0.00246785936931)
(num-test (asin -1234/500029) -0.00246785936931)
(num-test (asin 1234/362880) 0.00340057974625)
(num-test (asin -1234/362880) -0.00340057974625)
(num-test (asin 1234000000/1234000000) 1.57079632679490)
(num-test (asin -1234000000/1234000000) -1.57079632679490)
(num-test (asin 1234000000/500029) 1.57079632679490+8.50425252675189i)
(num-test (asin -1234000000/500029) -1.57079632679490+8.50425252675189i)
(num-test (asin 1234000000/362880) 1.57079632679490+8.82484644053502i)
(num-test (asin -1234000000/362880) -1.57079632679490+8.82484644053502i)
(num-test (asin 500029/3) 1.570796326794897-12.71695626760523i)
(num-test (asin -500029/3) -1.570796326794897+12.71695626760523i)
(num-test (asin 500029/10) 1.570796326794897-1.151298346318831e1i)
(num-test (asin -500029/10) -1.57079632679490+1.151298346318831e1i)
(num-test (asin 500029/1234) 1.57079632679490+6.69755082925184i)
(num-test (asin -500029/1234) -1.57079632679490+6.69755082925184i)
(num-test (asin 500029/1234000000) 0.00040520989764)
(num-test (asin -500029/1234000000) -0.00040520989764)
(num-test (asin 500029/500029) 1.57079632679490)
(num-test (asin -500029/500029) -1.57079632679490)
(num-test (asin 500029/362880) 1.57079632679490+0.84413377868249i)
(num-test (asin -500029/362880) -1.57079632679490+0.84413377868249i)
(num-test (asin 362880/2) 1.570796326794896619231321691639751442098E0-1.280182748007387555277950474671339683536E1i)
(num-test (asin -362880/2) -1.570796326794897+12.80182748007388i)
(num-test (asin 362880/3) 1.570796326794896619231321691639751442098E0-1.239636237195621859776598192721440899765E1i)
(num-test (asin -362880/3) -1.570796326794897+12.39636237195622i)
(num-test (asin 362880/10) 1.570796326794897-1.119238956745752e1i)
(num-test (asin -362880/10) -1.57079632679490+1.119238956745752e1i)
(num-test (asin 362880/1234) 1.57079632679490+6.37695556516894i)
(num-test (asin -362880/1234) -1.57079632679490+6.37695556516894i)
(num-test (asin 362880/1234000000) 0.00029406807555)
(num-test (asin -362880/1234000000) -0.00029406807555)
(num-test (asin 362880/500029) 0.81207730383580)
(num-test (asin -362880/500029) -0.81207730383580)
(num-test (asin 362880/362880) 1.57079632679490)
(num-test (asin -362880/362880) -1.57079632679490)
(num-test (asin 0.0) 0.0)
(num-test (asin 0.00000001) 0.00000001)
(num-test (asin -0.00000001) -0.00000001)
(num-test (asin 1.0) 1.57079632679490)
(num-test (asin -1.0) -1.57079632679490)
(num-test (asin our-pi) 1.57079632679490+1.81152627246085i)
(num-test (asin -3.14159265358979) -1.57079632679490+1.81152627246085i)
(num-test (asin 2.71828182845905) 1.57079632679490+1.65745445415308i)
(num-test (asin -2.71828182845905) -1.57079632679490+1.65745445415308i)
(num-test (asin 1234.0) 1.57079632679490+7.81116322068415i)
(num-test (asin -1234.0) -1.57079632679490+7.81116322068415i)
(num-test (asin 0.0+0.0i) 0.0)
(num-test (asin -0.0+0.0i) 0.0)
(num-test (asin 0.0-0.0i) 0.0)
(num-test (asin -0.0-0.0i) -0.0)
(num-test (asin 0.0+0.00000001i) 0.0+0.00000001i)
(num-test (asin -0.0+0.00000001i) 0.0+0.00000001i)
(num-test (asin 0.0-0.00000001i) 0.0-0.00000001i)
(num-test (asin -0.0-0.00000001i) -0.0-0.00000001i)
(num-test (asin 0.0+1.0i) 0.0+0.88137358701954i)
(num-test (asin -0.0+1.0i) 0.0+0.88137358701954i)
(num-test (asin 0.0-1.0i) 0.0-0.88137358701954i)
(num-test (asin -0.0-1.0i) -0.0-0.88137358701954i)
(num-test (asin 0.0+3.14159265358979i) 0.0+1.86229574331085i)
(num-test (asin -0.0+3.14159265358979i) 0.0+1.86229574331085i)
(num-test (asin 0.0-3.14159265358979i) 0.0-1.86229574331085i)
(num-test (asin -0.0-3.14159265358979i) -0.0-1.86229574331085i)
(num-test (asin 0.0+2.71828182845905i) 0.0+1.72538255885232i)
(num-test (asin -0.0+2.71828182845905i) 0.0+1.72538255885232i)
(num-test (asin 0.0-2.71828182845905i) 0.0-1.72538255885231i)
(num-test (asin -0.0-2.71828182845905i) -0.0-1.72538255885231i)
(num-test (asin 0.0+1234.0i) 0.0+7.81116354896171i)
(num-test (asin -0.0+1234.0i) 0.0+7.81116354896171i)
(num-test (asin 0.0-1234.0i) 0.0-7.81116354920125i)
(num-test (asin -0.0-1234.0i) -0.0-7.81116354920125i)
(num-test (asin 0.0-1234000000.0i) 0.0-21.62667394298955i)
(num-test (asin -0.0-1234000000.0i) -0.0-21.62667394298955i)
(num-test (asin 0.00000001+0.0i) 0.00000001)
(num-test (asin -0.00000001+0.0i) -0.00000001)
(num-test (asin 0.00000001-0.0i) 0.00000001)
(num-test (asin -0.00000001-0.0i) -0.00000001)
(num-test (asin 0.00000001+0.00000001i) 0.00000001+0.00000001i)
(num-test (asin -0.00000001+0.00000001i) -0.00000001+0.00000001i)
(num-test (asin 0.00000001-0.00000001i) 0.00000001-0.00000001i)
(num-test (asin -0.00000001-0.00000001i) -0.00000001-0.00000001i)
(num-test (asin 0.00000001+1.0i) 0.00000000707107+0.88137358701954i)
(num-test (asin -0.00000001+1.0i) -0.00000000707107+0.88137358701954i)
(num-test (asin 0.00000001-1.0i) 0.00000000707107-0.88137358701954i)
(num-test (asin -0.00000001-1.0i) -0.00000000707107-0.88137358701954i)
(num-test (asin 0.00000001+3.14159265358979i) 0.00000000303314+1.86229574331085i)
(num-test (asin -0.00000001+3.14159265358979i) -0.00000000303314+1.86229574331085i)
(num-test (asin 0.00000001-3.14159265358979i) 0.00000000303314-1.86229574331085i)
(num-test (asin -0.00000001-3.14159265358979i) -0.00000000303314-1.86229574331085i)
(num-test (asin 0.00000001+2.71828182845905i) 0.00000000345258+1.72538255885232i)
(num-test (asin -0.00000001+2.71828182845905i) -0.00000000345258+1.72538255885232i)
(num-test (asin 0.00000001-2.71828182845905i) 0.00000000345258-1.72538255885231i)
(num-test (asin -0.00000001-2.71828182845905i) -0.00000000345258-1.72538255885231i)
(num-test (asin 0.00000001+1234.0i) 0.00000000000810+7.81116354896171i)
(num-test (asin -0.00000001+1234.0i) -0.00000000000810+7.81116354896171i)
(num-test (asin 0.00000001-1234.0i) 0.00000000000810-7.81116354920125i)
(num-test (asin -0.00000001-1234.0i) -0.00000000000810-7.81116354920125i)
(num-test (asin 0.00000001-1234000000.0i) 0.0-21.62667394298955i)
(num-test (asin -0.00000001-1234000000.0i) -0.0-21.62667394298955i)
(num-test (asin 1.0+0.0i) 1.57079632679490)
(num-test (asin -1.0+0.0i) -1.57079632679490)
(num-test (asin 1.0-0.0i) 1.57079632679490)
(num-test (asin -1.0-0.0i) -1.57079632679490)
(num-test (asin 1.0+0.00000001i) 1.57069632679498+0.00010000000008i)
(num-test (asin -1.0+0.00000001i) -1.57069632679498+0.00010000000008i)
(num-test (asin 1.0-0.00000001i) 1.57069632679498-0.00010000000008i)
(num-test (asin -1.0-0.00000001i) -1.57069632679498-0.00010000000008i)
(num-test (asin 1.0+1.0i) 0.66623943249252+1.06127506190504i)
(num-test (asin -1.0+1.0i) -0.66623943249252+1.06127506190504i)
(num-test (asin 1.0-1.0i) 0.66623943249252-1.06127506190504i)
(num-test (asin -1.0-1.0i) -0.66623943249252-1.06127506190504i)
(num-test (asin 1.0+3.14159265358979i) 0.29558503421163+1.90462768697066i)
(num-test (asin -1.0+3.14159265358979i) -0.29558503421163+1.90462768697066i)
(num-test (asin 1.0-3.14159265358979i) 0.29558503421163-1.90462768697066i)
(num-test (asin -1.0-3.14159265358979i) -0.29558503421163-1.90462768697066i)
(num-test (asin 1.0+2.71828182845905i) 0.33444277187637+1.77905438385935i)
(num-test (asin -1.0+2.71828182845905i) -0.33444277187637+1.77905438385935i)
(num-test (asin 1.0-2.71828182845905i) 0.33444277187637-1.77905438385935i)
(num-test (asin -1.0-2.71828182845905i) -0.33444277187637-1.77905438385935i)
(num-test (asin 1.0+1234.0i) 0.00081037232806+7.81116387772663i)
(num-test (asin -1.0+1234.0i) -0.00081037232806+7.81116387772663i)
(num-test (asin 1.0-1234.0i) 0.00081037232800-7.81116387755283i)
(num-test (asin -1.0-1234.0i) -0.00081037232800-7.81116387755283i)
(num-test (asin 1.0-1234000000.0i) 0.00000000081037-21.62667394298955i)
(num-test (asin -1.0-1234000000.0i) -0.00000000081037-21.62667394298955i)
(num-test (asin 3.14159265358979+0.0i) 1.57079632679490+1.81152627246085i)
(num-test (asin -3.14159265358979+0.0i) -1.57079632679490+1.81152627246085i)
(num-test (asin 3.14159265358979-0.0i) 1.57079632679490+1.81152627246085i)
(num-test (asin -3.14159265358979-0.0i) -1.57079632679490+1.81152627246085i)
(num-test (asin 3.14159265358979+0.00000001i) 1.57079632343715+1.81152627246085i)
(num-test (asin -3.14159265358979+0.00000001i) -1.57079632343715+1.81152627246085i)
(num-test (asin 3.14159265358979-0.00000001i) 1.57079632343715-1.81152627246085i)
(num-test (asin -3.14159265358979-0.00000001i) -1.57079632343715-1.81152627246085i)
(num-test (asin 3.14159265358979+1.0i) 1.24854303281344+1.86711439316026i)
(num-test (asin -3.14159265358979+1.0i) -1.24854303281344+1.86711439316026i)
(num-test (asin 3.14159265358979-1.0i) 1.24854303281344-1.86711439316026i)
(num-test (asin -3.14159265358979-1.0i) -1.24854303281344-1.86711439316026i)
(num-test (asin 3.14159265358979+3.14159265358979i) 0.77273977912748+2.18469104082751i)
(num-test (asin -3.14159265358979+3.14159265358979i) -0.77273977912748+2.18469104082751i)
(num-test (asin 3.14159265358979-3.14159265358979i) 0.77273977912748-2.18469104082751i)
(num-test (asin -3.14159265358979-3.14159265358979i) -0.77273977912748-2.18469104082751i)
(num-test (asin 3.14159265358979+2.71828182845905i) 0.84309656416035+2.11552790803290i)
(num-test (asin -3.14159265358979+2.71828182845905i) -0.84309656416035+2.11552790803290i)
(num-test (asin 3.14159265358979-2.71828182845905i) 0.84309656416035-2.11552790803290i)
(num-test (asin -3.14159265358979-2.71828182845905i) -0.84309656416035-2.11552790803290i)
(num-test (asin 3.14159265358979+1234.0i) 0.00254585480900+7.81116678966949i)
(num-test (asin -3.14159265358979+1234.0i) -0.00254585480900+7.81116678966949i)
(num-test (asin 3.14159265358979-1234.0i) 0.00254585480937-7.81116678989204i)
(num-test (asin -3.14159265358979-1234.0i) -0.00254585480937-7.81116678989204i)
(num-test (asin 3.14159265358979-1234000000.0i) 0.00000000254586-21.62667394298955i)
(num-test (asin -3.14159265358979-1234000000.0i) -0.00000000254586-21.62667394298955i)
(num-test (asin 2.71828182845905+0.0i) 1.57079632679490+1.65745445415308i)
(num-test (asin -2.71828182845905+0.0i) -1.57079632679490+1.65745445415308i)
(num-test (asin 2.71828182845905-0.0i) 1.57079632679490+1.65745445415308i)
(num-test (asin -2.71828182845905-0.0i) -1.57079632679490+1.65745445415308i)
(num-test (asin 2.71828182845905+0.00000001i) 1.57079632283867+1.65745445415308i)
(num-test (asin -2.71828182845905+0.00000001i) -1.57079632283867+1.65745445415308i)
(num-test (asin 2.71828182845905-0.00000001i) 1.57079632283867-1.65745445415308i)
(num-test (asin -2.71828182845905-0.00000001i) -1.57079632283867-1.65745445415308i)
(num-test (asin 2.71828182845905+1.0i) 1.19757808518581+1.73375471569385i)
(num-test (asin -2.71828182845905+1.0i) -1.19757808518581+1.73375471569385i)
(num-test (asin 2.71828182845905-1.0i) 1.19757808518581-1.73375471569385i)
(num-test (asin -2.71828182845905-1.0i) -1.19757808518581-1.73375471569385i)
(num-test (asin 2.71828182845905+3.14159265358979i) 0.69904796994656+2.11968336368048i)
(num-test (asin -2.71828182845905+3.14159265358979i) -0.69904796994656+2.11968336368048i)
(num-test (asin 2.71828182845905-3.14159265358979i) 0.69904796994655-2.11968336368048i)
(num-test (asin -2.71828182845905-3.14159265358979i) -0.69904796994655-2.11968336368048i)
(num-test (asin 2.71828182845905+2.71828182845905i) 0.76849735588475+2.04014932880026i)
(num-test (asin -2.71828182845905+2.71828182845905i) -0.76849735588475+2.04014932880026i)
(num-test (asin 2.71828182845905-2.71828182845905i) 0.76849735588475-2.04014932880026i)
(num-test (asin -2.71828182845905-2.71828182845905i) -0.76849735588475-2.04014932880026i)
(num-test (asin 2.71828182845905+1234.0i) 0.00220281729236+7.81116597510553i)
(num-test (asin -2.71828182845905+1234.0i) -0.00220281729236+7.81116597510553i)
(num-test (asin 2.71828182845905-1234.0i) 0.00220281729269-7.81116597540442i)
(num-test (asin -2.71828182845905-1234.0i) -0.00220281729269-7.81116597540442i)
(num-test (asin 2.71828182845905-1234000000.0i) 0.00000000220282-21.62667394298955i)
(num-test (asin -2.71828182845905-1234000000.0i) -0.00000000220282-21.62667394298955i)
(num-test (asin 1234.0+0.0i) 1.57079632679490+7.81116322068415i)
(num-test (asin -1234.0+0.0i) -1.57079632679490+7.81116322068415i)
(num-test (asin 1234.0-0.0i) 1.57079632679490+7.81116322068415i)
(num-test (asin -1234.0-0.0i) -1.57079632679490+7.81116322068415i)
(num-test (asin 1234.0+0.00000001i) 1.57079632678679+7.81116322068415i)
(num-test (asin -1234.0+0.00000001i) -1.57079632678679+7.81116322068415i)
(num-test (asin 1234.0-0.00000001i) 1.57079632678679-7.81116322084923i)
(num-test (asin -1234.0-0.00000001i) -1.57079632678679-7.81116322084923i)
(num-test (asin 1234.0+1.0i) 1.56998595393442+7.81116354944842i)
(num-test (asin -1234.0+1.0i) -1.56998595393442+7.81116354944842i)
(num-test (asin 1234.0-1.0i) 1.56998595393473-7.81116354920146i)
(num-test (asin -1234.0-1.0i) -1.56998595393473-7.81116354920146i)
(num-test (asin 1234.0+3.14159265358979i) 1.56825047031506+7.81116646138554i)
(num-test (asin -1234.0+3.14159265358979i) -1.56825047031506+7.81116646138554i)
(num-test (asin 1234.0-3.14159265358979i) 1.56825047031367-7.81116646154641i)
(num-test (asin -1234.0-3.14159265358979i) -1.56825047031367-7.81116646154641i)
(num-test (asin 1234.0+2.71828182845905i) 1.56859350805543+7.81116564738434i)
(num-test (asin -1234.0+2.71828182845905i) -1.56859350805543+7.81116564738434i)
(num-test (asin 1234.0-2.71828182845905i) 1.56859350805562-7.81116564705719i)
(num-test (asin -1234.0-2.71828182845905i) -1.56859350805562-7.81116564705719i)
(num-test (asin 1234.0+1234.0i) 0.78539808146835+8.15773697538346i)
(num-test (asin -1234.0+1234.0i) -0.78539808146835+8.15773697538346i)
(num-test (asin 1234.0-1234.0i) 0.78539808130944-8.15773697530526i)
(num-test (asin -1234.0-1234.0i) -0.78539808130944-8.15773697530526i)
(num-test (asin 1234.0-1234000000.0i) 0.00000100000000-21.62667394299005i)
(num-test (asin -1234.0-1234000000.0i) -0.00000100000000-21.62667394299005i)
(num-test (asin 1234000000.0-0.00000001i) 1.57079632679490-21.62667394298955i)
(num-test (asin -1234000000.0-0.00000001i) -1.57079632679490-21.62667394298955i)
(num-test (asin 1234000000.0-1.0i) 1.57079632598452-21.62667394298955i)
(num-test (asin -1234000000.0-1.0i) -1.57079632598452-21.62667394298955i)
(num-test (asin 1234000000.0-3.14159265358979i) 1.57079632424904-21.62667394298955i)
(num-test (asin -1234000000.0-3.14159265358979i) -1.57079632424904-21.62667394298955i)
(num-test (asin 1234000000.0-2.71828182845905i) 1.57079632459208-21.62667394298955i)
(num-test (asin -1234000000.0-2.71828182845905i) -1.57079632459208-21.62667394298955i)
(num-test (asin 1234000000.0-1234.0i) 1.57079532679490-21.62667394299005i)
(num-test (asin -1234000000.0-1234.0i) -1.57079532679490-21.62667394299005i)
(num-test (asin 1234000000.0-1234000000.0i) 0.78539816339745-21.97324753326953i)
(num-test (asin -1234000000.0-1234000000.0i) -0.78539816339745-21.97324753326953i)
(num-test (asin 8.2729394e-17) 8.2729394e-17)
(num-test (asin 1.821832e-231) 1.821832e-231)
(num-test (asin 48983.30495194942-64983.97008730317i) 0.6459128607940432-12i)
(num-test (asin 1234000000/3) 1.570796326794897-20.52806165432143i) 
(num-test (asin 500029/2) 1.570796326794897-13.12242137571839i) 
(num-test (asin 0.00000001+1234000000.0i) +8.103725052149596E-18+21.62667380877375i) 
(num-test (asin 3.14159265358979+1234000000.0i) +2.545860611523387E-9+21.62667380877375i) 
(num-test (asin 2.71828182845905+1234000000.0i) +2.2028207814967068E-9+21.62667380877375i) 
(num-test (asin 1234000000.0+0.00000001i) +1.570796327200083+21.62667394298955i) 
(num-test (asin 1234000000.0+3.14159265358979i) +1.570796327200083+21.62667394298955i) 
(num-test (asin 0.0e+00+0.0e+00i) 0e0+0.0i)
(num-test (asin 0.0e+00+1.19209289550781250e-07i) 0+1.1920928955078096766e-7i)
(num-test (asin 0.0e+00-1.19209289550781250e-07i) 0-1.1920928955078096766e-7i)
(num-test (asin 0.0e+00+5.0e-01i) 0+4.8121182505960344750e-1i)
(num-test (asin 0.0e+00-5.0e-01i) 0-4.8121182505960344750e-1i)
(num-test (asin 0.0e+00+1.0e+00i) 0+8.8137358701954302523e-1i)
(num-test (asin 0.0e+00-1.0e+00i) 0-8.8137358701954302523e-1i)
(num-test (asin 0.0e+00+2.0e+00i) 0+1.4436354751788103425e0i)
(num-test (asin 0.0e+00-2.0e+00i) 0-1.4436354751788103425e0i)
(num-test (asin 0.0e+00+8.3886080e+06i) 0+1.6635532333438690979e1i)
(num-test (asin 0.0e+00-8.3886080e+06i) 0-1.6635532333438690979e1i)
(num-test (asin 1.19209289550781250e-07+0.0e+00i) 1.1920928955078153234e-7+0.0i)
(num-test (asin -1.19209289550781250e-07+0.0e+00i) -1.1920928955078153234e-7+0.0i)
(num-test (asin 1.19209289550781250e-07+1.19209289550781250e-07i) 1.1920928955078068531e-7+1.1920928955078181469e-7i)
(num-test (asin 1.19209289550781250e-07-1.19209289550781250e-07i) 1.1920928955078068531e-7-1.1920928955078181469e-7i)
(num-test (asin -1.19209289550781250e-07+1.19209289550781250e-07i) -1.1920928955078068531e-7+1.1920928955078181469e-7i)
(num-test (asin -1.19209289550781250e-07-1.19209289550781250e-07i) -1.1920928955078068531e-7-1.1920928955078181469e-7i)
(num-test (asin 1.19209289550781250e-07+5.0e-01i) 1.0662402999400097805e-7+4.8121182505960598961e-1i)
(num-test (asin 1.19209289550781250e-07-5.0e-01i) 1.0662402999400097805e-7-4.8121182505960598961e-1i)
(num-test (asin -1.19209289550781250e-07+5.0e-01i) -1.0662402999400097805e-7+4.8121182505960598961e-1i)
(num-test (asin -1.19209289550781250e-07-5.0e-01i) -1.0662402999400097805e-7-4.8121182505960598961e-1i)
(num-test (asin 1.19209289550781250e-07+1.0e+00i) 8.4293697021788013662e-8+8.8137358701954553738e-1i)
(num-test (asin 1.19209289550781250e-07-1.0e+00i) 8.4293697021788013662e-8-8.8137358701954553738e-1i)
(num-test (asin -1.19209289550781250e-07+1.0e+00i) -8.4293697021788013662e-8+8.8137358701954553738e-1i)
(num-test (asin -1.19209289550781250e-07-1.0e+00i) -8.4293697021788013662e-8-8.8137358701954553738e-1i)
(num-test (asin 1.19209289550781250e-07+2.0e+00i) 5.3312014997000413263e-8+1.4436354751788116136e0i)
(num-test (asin 1.19209289550781250e-07-2.0e+00i) 5.3312014997000413263e-8-1.4436354751788116136e0i)
(num-test (asin -1.19209289550781250e-07+2.0e+00i) -5.3312014997000413263e-8+1.4436354751788116136e0i)
(num-test (asin -1.19209289550781250e-07-2.0e+00i) -5.3312014997000413263e-8-1.4436354751788116136e0i)
(num-test (asin 1.19209289550781250e-07+8.3886080e+06i) 1.4210854715201902743e-14+1.6635532333438690979e1i)
(num-test (asin 1.19209289550781250e-07-8.3886080e+06i) 1.4210854715201902743e-14-1.6635532333438690979e1i)
(num-test (asin -1.19209289550781250e-07+8.3886080e+06i) -1.4210854715201902743e-14+1.6635532333438690979e1i)
(num-test (asin -1.19209289550781250e-07-8.3886080e+06i) -1.4210854715201902743e-14-1.6635532333438690979e1i)
(num-test (asin 5.0e-01+0.0e+00i) 5.2359877559829887308e-1+0.0i)
(num-test (asin -5.0e-01+0.0e+00i) -5.2359877559829887308e-1+0.0i)
(num-test (asin 5.0e-01+1.19209289550781250e-07i) 5.2359877559829340332e-1+1.3765103082409432364e-7i)
(num-test (asin 5.0e-01-1.19209289550781250e-07i) 5.2359877559829340332e-1-1.3765103082409432364e-7i)
(num-test (asin -5.0e-01+1.19209289550781250e-07i) -5.2359877559829340332e-1+1.3765103082409432364e-7i)
(num-test (asin -5.0e-01-1.19209289550781250e-07i) -5.2359877559829340332e-1-1.3765103082409432364e-7i)
(num-test (asin 5.0e-01+5.0e-01i) 4.5227844715119068206e-1+5.3063753095251782602e-1i)
(num-test (asin 5.0e-01-5.0e-01i) 4.5227844715119068206e-1-5.3063753095251782602e-1i)
(num-test (asin -5.0e-01+5.0e-01i) -4.5227844715119068206e-1+5.3063753095251782602e-1i)
(num-test (asin -5.0e-01-5.0e-01i) -4.5227844715119068206e-1-5.3063753095251782602e-1i)
(num-test (asin 5.0e-01+1.0e+00i) 3.4943906285721329363e-1+9.2613303135018242455e-1i)
(num-test (asin 5.0e-01-1.0e+00i) 3.4943906285721329363e-1-9.2613303135018242455e-1i)
(num-test (asin -5.0e-01+1.0e+00i) -3.4943906285721329363e-1+9.2613303135018242455e-1i)
(num-test (asin -5.0e-01-1.0e+00i) -3.4943906285721329363e-1-9.2613303135018242455e-1i)
(num-test (asin 5.0e-01+2.0e+00i) 2.2101863562288385890e-1+1.4657153519472905218e0i)
(num-test (asin 5.0e-01-2.0e+00i) 2.2101863562288385890e-1-1.4657153519472905218e0i)
(num-test (asin -5.0e-01+2.0e+00i) -2.2101863562288385890e-1+1.4657153519472905218e0i)
(num-test (asin -5.0e-01-2.0e+00i) -2.2101863562288385890e-1-1.4657153519472905218e0i)
(num-test (asin 5.0e-01+8.3886080e+06i) 5.9604644775390130897e-8+1.6635532333438692755e1i)
(num-test (asin 5.0e-01-8.3886080e+06i) 5.9604644775390130897e-8-1.6635532333438692755e1i)
(num-test (asin -5.0e-01+8.3886080e+06i) -5.9604644775390130897e-8+1.6635532333438692755e1i)
(num-test (asin -5.0e-01-8.3886080e+06i) -5.9604644775390130897e-8-1.6635532333438692755e1i)
(num-test (asin 1.0e+00+0.0e+00i) 1.5707963267948966192e0+0.0i)
(num-test (asin -1.0e+00+0.0e+00i) -1.5707963267948966192e0+0.0i)
(num-test (asin 1.0e+00+1.19209289550781250e-07i) 1.5704510598153252947e0+3.4526698643116312881e-4i)
(num-test (asin 1.0e+00-1.19209289550781250e-07i) 1.5704510598153252947e0-3.4526698643116312881e-4i)
(num-test (asin -1.0e+00+1.19209289550781250e-07i) -1.5704510598153252947e0+3.4526698643116312881e-4i)
(num-test (asin -1.0e+00-1.19209289550781250e-07i) -1.5704510598153252947e0-3.4526698643116312881e-4i)
(num-test (asin 1.0e+00+5.0e-01i) 8.9590748120889023907e-1+7.3285767597364526089e-1i)
(num-test (asin 1.0e+00-5.0e-01i) 8.9590748120889023907e-1-7.3285767597364526089e-1i)
(num-test (asin -1.0e+00+5.0e-01i) -8.9590748120889023907e-1+7.3285767597364526089e-1i)
(num-test (asin -1.0e+00-5.0e-01i) -8.9590748120889023907e-1-7.3285767597364526089e-1i)
(num-test (asin 1.0e+00+1.0e+00i) 6.6623943249251525510e-1+1.0612750619050356520e0i)
(num-test (asin 1.0e+00-1.0e+00i) 6.6623943249251525510e-1-1.0612750619050356520e0i)
(num-test (asin -1.0e+00+1.0e+00i) -6.6623943249251525510e-1+1.0612750619050356520e0i)
(num-test (asin -1.0e+00-1.0e+00i) -6.6623943249251525510e-1-1.0612750619050356520e0i)
(num-test (asin 1.0e+00+2.0e+00i) 4.2707858639247612548e-1+1.5285709194809981613e0i)
(num-test (asin 1.0e+00-2.0e+00i) 4.2707858639247612548e-1-1.5285709194809981613e0i)
(num-test (asin -1.0e+00+2.0e+00i) -4.2707858639247612548e-1+1.5285709194809981613e0i)
(num-test (asin -1.0e+00-2.0e+00i) -4.2707858639247612548e-1-1.5285709194809981613e0i)
(num-test (asin 1.0e+00+8.3886080e+06i) 1.1920928955077983828e-7+1.6635532333438698084e1i)
(num-test (asin 1.0e+00-8.3886080e+06i) 1.1920928955077983828e-7-1.6635532333438698084e1i)
(num-test (asin -1.0e+00+8.3886080e+06i) -1.1920928955077983828e-7+1.6635532333438698084e1i)
(num-test (asin -1.0e+00-8.3886080e+06i) -1.1920928955077983828e-7-1.6635532333438698084e1i)
(num-test (asin 2.0e+00+0.0e+00i) 1.5707963267948966192e0-1.3169578969248167086e0i)
(num-test (asin -2.0e+00+0.0e+00i) -1.5707963267948966192e0+1.3169578969248167086e0i)
(num-test (asin 2.0e+00+1.19209289550781250e-07i) 1.5707962579693812072e0+1.3169578969248194435e0i)
(num-test (asin 2.0e+00-1.19209289550781250e-07i) 1.5707962579693812072e0-1.3169578969248194435e0i)
(num-test (asin -2.0e+00+1.19209289550781250e-07i) -1.5707962579693812072e0+1.3169578969248194435e0i)
(num-test (asin -2.0e+00-1.19209289550781250e-07i) -1.5707962579693812072e0-1.3169578969248194435e0i)
(num-test (asin 2.0e+00+5.0e-01i) 1.2930420702371826591e0+1.3618009008578457882e0i)
(num-test (asin 2.0e+00-5.0e-01i) 1.2930420702371826591e0-1.3618009008578457882e0i)
(num-test (asin -2.0e+00+5.0e-01i) -1.2930420702371826591e0+1.3618009008578457882e0i)
(num-test (asin -2.0e+00-5.0e-01i) -1.2930420702371826591e0-1.3618009008578457882e0i)
(num-test (asin 2.0e+00+1.0e+00i) 1.0634400235777520562e0+1.4693517443681852733e0i)
(num-test (asin 2.0e+00-1.0e+00i) 1.0634400235777520562e0-1.4693517443681852733e0i)
(num-test (asin -2.0e+00+1.0e+00i) -1.0634400235777520562e0+1.4693517443681852733e0i)
(num-test (asin -2.0e+00-1.0e+00i) -1.0634400235777520562e0-1.4693517443681852733e0i)
(num-test (asin 2.0e+00+2.0e+00i) 7.5424914469804604071e-1+1.7343245214879664480e0i)
(num-test (asin 2.0e+00-2.0e+00i) 7.5424914469804604071e-1-1.7343245214879664480e0i)
(num-test (asin -2.0e+00+2.0e+00i) -7.5424914469804604071e-1+1.7343245214879664480e0i)
(num-test (asin -2.0e+00-2.0e+00i) -7.5424914469804604071e-1-1.7343245214879664480e0i)
(num-test (asin 2.0e+00+8.3886080e+06i) 2.3841857910155628843e-7+1.663553233343871940e1i)
(num-test (asin 2.0e+00-8.3886080e+06i) 2.3841857910155628843e-7-1.663553233343871940e1i)
(num-test (asin -2.0e+00+8.3886080e+06i) -2.3841857910155628843e-7+1.663553233343871940e1i)
(num-test (asin -2.0e+00-8.3886080e+06i) -2.3841857910155628843e-7-1.663553233343871940e1i)
(num-test (asin 8.3886080e+06+0.0e+00i) 1.5707963267948966192e0-1.6635532333438683873e1i)
(num-test (asin -8.3886080e+06+0.0e+00i) -1.5707963267948966192e0+1.6635532333438683873e1i)
(num-test (asin 8.3886080e+06+1.19209289550781250e-07i) 1.5707963267948824084e0+1.6635532333438683873e1i)
(num-test (asin 8.3886080e+06-1.19209289550781250e-07i) 1.5707963267948824084e0-1.6635532333438683873e1i)
(num-test (asin -8.3886080e+06+1.19209289550781250e-07i) -1.5707963267948824084e0+1.6635532333438683873e1i)
(num-test (asin -8.3886080e+06-1.19209289550781250e-07i) -1.5707963267948824084e0-1.6635532333438683873e1i)
(num-test (asin 8.3886080e+06+5.0e-01i) 1.5707962671902518438e0+1.6635532333438685650e1i)
(num-test (asin 8.3886080e+06-5.0e-01i) 1.5707962671902518438e0-1.6635532333438685650e1i)
(num-test (asin -8.3886080e+06+5.0e-01i) -1.5707962671902518438e0+1.6635532333438685650e1i)
(num-test (asin -8.3886080e+06-5.0e-01i) -1.5707962671902518438e0-1.6635532333438685650e1i)
(num-test (asin 8.3886080e+06+1.0e+00i) 1.5707962075856070684e0+1.6635532333438690979e1i)
(num-test (asin 8.3886080e+06-1.0e+00i) 1.5707962075856070684e0-1.6635532333438690979e1i)
(num-test (asin -8.3886080e+06+1.0e+00i) -1.5707962075856070684e0+1.6635532333438690979e1i)
(num-test (asin -8.3886080e+06-1.0e+00i) -1.5707962075856070684e0-1.6635532333438690979e1i)
(num-test (asin 8.3886080e+06+2.0e+00i) 1.5707960883763175177e0+1.6635532333438712295e1i)
(num-test (asin 8.3886080e+06-2.0e+00i) 1.5707960883763175177e0-1.6635532333438712295e1i)
(num-test (asin -8.3886080e+06+2.0e+00i) -1.5707960883763175177e0+1.6635532333438712295e1i)
(num-test (asin -8.3886080e+06-2.0e+00i) -1.5707960883763175177e0-1.6635532333438712295e1i)
(num-test (asin 8.3886080e+06+8.3886080e+06i) 7.8539816339744653326e-1+1.6982105923718660081e1i)
(num-test (asin 8.3886080e+06-8.3886080e+06i) 7.8539816339744653326e-1-1.6982105923718660081e1i)
(num-test (asin -8.3886080e+06+8.3886080e+06i) -7.8539816339744653326e-1+1.6982105923718660081e1i)
(num-test (asin -8.3886080e+06-8.3886080e+06i) -7.8539816339744653326e-1-1.6982105923718660081e1i)
(num-test (asin -1.0e+01) -1.5707963267948966192e0+2.9932228461263808979e0i)
(num-test (asin -2.0e+00) -1.5707963267948966192e0+1.3169578969248167086e0i)
(num-test (asin -1.0e+00) -1.5707963267948966192e0+0.0i)
(num-test (asin -7.50e-01) -8.4806207898148100805e-1+0.0i)
(num-test (asin -5.0e-01) -5.2359877559829887308e-1+0.0i)
(num-test (asin -1.250e-01) -1.2532783116806539687e-1+0.0i)
(num-test (asin -3.45266983001243932001e-04) -3.4526698986108292481e-4+0.0i)
(num-test (asin -1.19209289550781250e-07) -1.1920928955078153234e-7+0.0i)
(num-test (asin 0.0e+00) 0e0+0.0i)
(num-test (asin 1.19209289550781250e-07) 1.1920928955078153234e-7+0.0i)
(num-test (asin 3.45266983001243932001e-04) 3.4526698986108292481e-4+0.0i)
(num-test (asin 1.250e-01) 1.2532783116806539687e-1+0.0i)
(num-test (asin 5.0e-01) 5.2359877559829887308e-1+0.0i)
(num-test (asin 7.50e-01) 8.4806207898148100805e-1+0.0i)
(num-test (asin 1.0e+00) 1.5707963267948966192e0+0.0i)
(num-test (asin 2.0e+00) 1.5707963267948966192e0-1.3169578969248167086e0i)
(num-test (asin 1.0e+01) 1.5707963267948966192e0-2.9932228461263808979e0i)
(num-test (asin 0) 0.0)
(num-test (asin 2) 1.570796326794897-1.316957896924817i)
(num-test (asin 3.0+70000000i) 4.2857142400327436E-8+18.7571529895002i)
(num-test (asin 70000000+3i) 1.570796279536826+18.75715298057358i)
(num-test (asin 3.0-70000000i) 4.2857142400327436E-8-18.7571529895002i)
(num-test (asin -70000000+3i) -1.570796279536826+18.75715298057358i)

(let ((err 0.0))
  (do ((i 0 (+ i 1))
       (x -10.0 (+ x .1)))
      ((= i 200))
    (let ((y (magnitude (- x (sin (asin x))))))
      (if (> y err) (set! err y))))
  (if (> err 1e-12)
      (begin (display "(sin (asin ...)) error: ") (display err) (newline))))

(let ((err 0.0))
  (do ((i 0 (+ i 1))
       (x 1.0-i (+ x -.1+i)))
      ((= i 100))
    (let ((y (magnitude (- x (sin (asin x))))))
      (if (> y err) (set! err y))))
  (if (> err 1e-9)
      (begin (display "(sin (asin [complex] ...)) error: ") (display err) (newline))))



;; -------- acos
(num-test (acos 0) 1.57079632679490)
(num-test (acos 1) 0.0)
(num-test (acos -1) our-pi)
(num-test (acos 2) 0.0-1.31695789692482i)
(num-test (acos -2) 3.14159265358979-1.31695789692482i)
(num-test (acos 3) 0.0-1.76274717403909i)
(num-test (acos -3) 3.14159265358979-1.76274717403909i)
(num-test (acos 10) 0.0-2.99322284612638i)
(num-test (acos -10) 3.14159265358979-2.99322284612638i)
(num-test (acos 1234) 0.0-7.81116322068415i)
(num-test (acos -1234) 3.14159265358979-7.81116322068415i)
(num-test (acos 0/1) 1.57079632679490)
(num-test (acos 0/2) 1.57079632679490)
(num-test (acos 0/3) 1.57079632679490)
(num-test (acos 0/10) 1.57079632679490)
(num-test (acos 0/1234) 1.57079632679490)
(num-test (acos 0/1234000000) 1.57079632679490)
(num-test (acos 0/500029) 1.57079632679490)
(num-test (acos 0/362880) 1.57079632679490)
(num-test (acos 1/1) 0.0)
(num-test (acos -1/1) our-pi)
(num-test (acos 1/2) 1.04719755119660)
(num-test (acos -1/2) 2.09439510239320)
(num-test (acos 1/3) 1.23095941734077)
(num-test (acos -1/3) 1.91063323624902)
(num-test (acos 1/10) 1.47062890563334)
(num-test (acos -1/10) 1.67096374795646)
(num-test (acos 1/1234) 1.56998595393473)
(num-test (acos -1/1234) 1.57160669965507)
(num-test (acos 1/1234000000) 1.57079632598452)
(num-test (acos -1/1234000000) 1.57079632760527)
(num-test (acos 1/500029) 1.57079432691089)
(num-test (acos -1/500029) 1.57079832667890)
(num-test (acos 1/362880) 1.57079357106297)
(num-test (acos -1/362880) 1.57079908252682)
(num-test (acos 2/1) 0.0-1.31695789692482i)
(num-test (acos -2/1) 3.14159265358979-1.31695789692482i)
(num-test (acos 2/2) 0.0)
(num-test (acos -2/2) our-pi)
(num-test (acos 2/3) 0.84106867056793)
(num-test (acos -2/3) 2.30052398302186)
(num-test (acos 2/10) 1.36943840600457)
(num-test (acos -2/10) 1.77215424758523)
(num-test (acos 2/1234) 1.56917558054238)
(num-test (acos -2/1234) 1.57241707304741)
(num-test (acos 2/1234000000) 1.57079632517415)
(num-test (acos -2/1234000000) 1.57079632841564)
(num-test (acos 2/500029) 1.57079232702688)
(num-test (acos -2/500029) 1.57080032656291)
(num-test (acos 2/362880) 1.57079081533105)
(num-test (acos -2/362880) 1.57080183825874)
(num-test (acos 3/1) 0.0-1.76274717403909i)
(num-test (acos -3/1) 3.14159265358979-1.76274717403909i)
(num-test (acos 3/2) 0.0-0.96242365011921i)
(num-test (acos -3/2) 3.14159265358979-0.96242365011921i)
(num-test (acos 3/3) 0.0)
(num-test (acos -3/3) our-pi)
(num-test (acos 3/10) 1.26610367277950)
(num-test (acos -3/10) 1.87548898081029)
(num-test (acos 3/1234) 1.56836520608568)
(num-test (acos -3/1234) 1.57322744750412)
(num-test (acos 3/1234000000) 1.57079632436378)
(num-test (acos -3/1234000000) 1.57079632922601)
(num-test (acos 3/500029) 1.57079032714288)
(num-test (acos -3/500029) 1.57080232644692)
(num-test (acos 3/362880) 1.57078805959913-0.0i)
(num-test (acos -3/362880) 1.57080459399066-0.0i)
(num-test (acos 10/1) 0.0-2.99322284612638i)
(num-test (acos -10/1) 3.14159265358979-2.99322284612638i)
(num-test (acos 10/2) 0.0-2.29243166956117i)
(num-test (acos -10/2) 3.14159265358979-2.29243166956117i)
(num-test (acos 10/3) 0.0-1.87382024252741i)
(num-test (acos -10/3) 3.14159265358979-1.87382024252741i)
(num-test (acos 10/10) 0.0)
(num-test (acos -10/10) our-pi)
(num-test (acos 10/1234) 1.56269251038168)
(num-test (acos -10/1234) 1.57890014320811)
(num-test (acos 10/1234000000) 1.57079631869117)
(num-test (acos -10/1234000000) 1.57079633489862)
(num-test (acos 10/500029) 1.57077632795483)
(num-test (acos -10/500029) 1.57081632563497)
(num-test (acos 10/362880) 1.57076876947567)
(num-test (acos -10/362880) 1.57082388411412)
(num-test (acos 1234/1) 0.0-7.81116322068415i)
(num-test (acos -1234/1) 3.14159265358979-7.81116322068415i)
(num-test (acos 1234/2) 0.0-7.11801554770806i)
(num-test (acos -1234/2) 3.14159265358979-7.11801554770806i)
(num-test (acos 1234/3) 0.0-6.71254961876657i)
(num-test (acos -1234/3) 3.14159265358979-6.71254961876657i)
(num-test (acos 1234/10) 0.0-5.50856187402751i)
(num-test (acos -1234/10) 3.14159265358979-5.50856187402751i)
(num-test (acos 1234/1234) 0.0)
(num-test (acos -1234/1234) our-pi)
(num-test (acos 1234/1234000000) 1.57079532679490)
(num-test (acos -1234/1234000000) 1.57079732679490)
(num-test (acos 1234/500029) 1.56832846742558)
(num-test (acos -1234/500029) 1.57326418616421)
(num-test (acos 1234/362880) 1.56739574704864)
(num-test (acos -1234/362880) 1.57419690654115)
(num-test (acos 1234000000/1234000000) 0.0)
(num-test (acos -1234000000/1234000000) our-pi)
(num-test (acos 1234000000/500029) 0.0-8.50425252675189i)
(num-test (acos -1234000000/500029) 3.14159265358979-8.50425252675189i)
(num-test (acos 1234000000/362880) 0.0-8.82484644053502i)
(num-test (acos -1234000000/362880) 3.14159265358979-8.82484644053502i)
(num-test (acos 500029/3) 0+12.7169561400958i)
(num-test (acos -500029/3) 3.14159265358979-12.7169561400958i)
(num-test (acos 500029/10) 0.0+11.51298346318831i) ; maxima
(num-test (acos -500029/10) 3.14159265358979-11.51298333576987i)
(num-test (acos 500029/1234) 0.0-6.69755082925184i)
(num-test (acos -500029/1234) 3.14159265358979-6.69755082925184i)
(num-test (acos 500029/1234000000) 1.57039111689726)
(num-test (acos -500029/1234000000) 1.57120153669253)
(num-test (acos 500029/500029) 0.0)
(num-test (acos -500029/500029) our-pi)
(num-test (acos 500029/362880) 0.0-0.84413377868249i)
(num-test (acos -500029/362880) 3.14159265358979-0.84413377868249i)
(num-test (acos 362880/2) 0.0+1.280182724631953e1i)
(num-test (acos -362880/2) 3.141592653589793-12.80182748007388i)
(num-test (acos 362880/3) 0.0+12.39636237195622i)
(num-test (acos -362880/3) 3.141592653589793-12.39636237195622i)
(num-test (acos 362880/10) 0.0-11.19238956745752i)
(num-test (acos -362880/10) 3.14159265358979-11.19238956745752i)
(num-test (acos 362880/1234) 0.0-6.37695556516894i)
(num-test (acos -362880/1234) 3.14159265358979-6.37695556516894i)
(num-test (acos 362880/1234000000) 1.57050225871935)
(num-test (acos -362880/1234000000) 1.57109039487045)
(num-test (acos 362880/500029) 0.75871902295910)
(num-test (acos -362880/500029) 2.38287363063069)
(num-test (acos 362880/362880) 0.0)
(num-test (acos -362880/362880) our-pi)
(num-test (acos 0.0) 1.57079632679490)
(num-test (acos 0.00000001) 1.57079631679490)
(num-test (acos -0.00000001) 1.57079633679490)
(num-test (acos 1.0) 0.0)
(num-test (acos -1.0) our-pi)
(num-test (acos our-pi) 0.0-1.81152627246085i)
(num-test (acos -3.14159265358979) 3.14159265358979-1.81152627246085i)
(num-test (acos 2.71828182845905) 0.0-1.65745445415308i)
(num-test (acos -2.71828182845905) 3.14159265358979-1.65745445415308i)
(num-test (acos 1234.0) 0.0-7.81116322068415i)
(num-test (acos -1234.0) 3.14159265358979-7.81116322068415i)
(num-test (acos 0.0+0.0i) 1.57079632679490)
(num-test (acos -0.0+0.0i) 1.57079632679490)
(num-test (acos 0.0-0.0i) 1.57079632679490)
(num-test (acos -0.0-0.0i) 1.57079632679490)
(num-test (acos 0.0+0.00000001i) 1.57079632679490-0.00000001i)
(num-test (acos -0.0+0.00000001i) 1.57079632679490-0.00000001i)
(num-test (acos 0.0-0.00000001i) 1.57079632679490+0.00000001i)
(num-test (acos -0.0-0.00000001i) 1.57079632679490+0.00000001i)
(num-test (acos 0.0+1.0i) 1.57079632679490-0.88137358701954i)
(num-test (acos -0.0+1.0i) 1.57079632679490-0.88137358701954i)
(num-test (acos 0.0-1.0i) 1.57079632679490+0.88137358701954i)
(num-test (acos -0.0-1.0i) 1.57079632679490+0.88137358701954i)
(num-test (acos 0.0+3.14159265358979i) 1.57079632679490-1.86229574331085i)
(num-test (acos -0.0+3.14159265358979i) 1.57079632679490-1.86229574331085i)
(num-test (acos 0.0-3.14159265358979i) 1.57079632679490+1.86229574331085i)
(num-test (acos -0.0-3.14159265358979i) 1.57079632679490+1.86229574331085i)
(num-test (acos 0.0+2.71828182845905i) 1.57079632679490-1.72538255885232i)
(num-test (acos -0.0+2.71828182845905i) 1.57079632679490-1.72538255885232i)
(num-test (acos 0.0-2.71828182845905i) 1.57079632679490+1.72538255885231i)
(num-test (acos -0.0-2.71828182845905i) 1.57079632679490+1.72538255885231i)
(num-test (acos 0.0+1234.0i) 1.57079632679490-7.81116354896171i)
(num-test (acos -0.0+1234.0i) 1.57079632679490-7.81116354896171i)
(num-test (acos 0.0-1234.0i) 1.57079632679490+7.81116354920125i)
(num-test (acos -0.0-1234.0i) 1.57079632679490+7.81116354920125i)
(num-test (acos 0.0-1234000000.0i) 1.57079632679490+21.62667394298955i)
(num-test (acos -0.0-1234000000.0i) 1.57079632679490+21.62667394298955i)
(num-test (acos 0.00000001+0.0i) 1.57079631679490)
(num-test (acos -0.00000001+0.0i) 1.57079633679490)
(num-test (acos 0.00000001-0.0i) 1.57079631679490)
(num-test (acos -0.00000001-0.0i) 1.57079633679490)
(num-test (acos 0.00000001+0.00000001i) 1.57079631679490-0.00000001i)
(num-test (acos -0.00000001+0.00000001i) 1.57079633679490-0.00000001i)
(num-test (acos 0.00000001-0.00000001i) 1.57079631679490+0.00000001i)
(num-test (acos -0.00000001-0.00000001i) 1.57079633679490+0.00000001i)
(num-test (acos 0.00000001+1.0i) 1.57079631972383-0.88137358701954i)
(num-test (acos -0.00000001+1.0i) 1.57079633386596-0.88137358701954i)
(num-test (acos 0.00000001-1.0i) 1.57079631972383+0.88137358701954i)
(num-test (acos -0.00000001-1.0i) 1.57079633386596+0.88137358701954i)
(num-test (acos 0.00000001+3.14159265358979i) 1.57079632376175-1.86229574331085i)
(num-test (acos -0.00000001+3.14159265358979i) 1.57079632982804-1.86229574331085i)
(num-test (acos 0.00000001-3.14159265358979i) 1.57079632376175+1.86229574331085i)
(num-test (acos -0.00000001-3.14159265358979i) 1.57079632982804+1.86229574331085i)
(num-test (acos 0.00000001+2.71828182845905i) 1.57079632334232-1.72538255885232i)
(num-test (acos -0.00000001+2.71828182845905i) 1.57079633024747-1.72538255885232i)
(num-test (acos 0.00000001-2.71828182845905i) 1.57079632334232+1.72538255885231i)
(num-test (acos -0.00000001-2.71828182845905i) 1.57079633024747+1.72538255885231i)
(num-test (acos 0.00000001+1234.0i) 1.57079632678679-7.81116354896171i)
(num-test (acos -0.00000001+1234.0i) 1.57079632680300-7.81116354896171i)
(num-test (acos 0.00000001-1234.0i) 1.57079632678679+7.81116354920125i)
(num-test (acos -0.00000001-1234.0i) 1.57079632680300+7.81116354920125i)
(num-test (acos 0.00000001-1234000000.0i) 1.57079632679490+21.62667394298955i)
(num-test (acos -0.00000001-1234000000.0i) 1.57079632679490+21.62667394298955i)
(num-test (acos 1.0+0.0i) 0.0)
(num-test (acos -1.0+0.0i) our-pi)
(num-test (acos 1.0-0.0i) 0.0)
(num-test (acos -1.0-0.0i) our-pi)
(num-test (acos 1.0+0.00000001i) 0.00009999999992-0.00010000000008i)
(num-test (acos -1.0+0.00000001i) 3.14149265358988-0.00010000000008i)
(num-test (acos 1.0-0.00000001i) 0.00009999999992+0.00010000000008i)
(num-test (acos -1.0-0.00000001i) 3.14149265358988+0.00010000000008i)
(num-test (acos 1.0+1.0i) 0.90455689430238-1.06127506190504i)
(num-test (acos -1.0+1.0i) 2.23703575928741-1.06127506190504i)
(num-test (acos 1.0-1.0i) 0.90455689430238+1.06127506190504i)
(num-test (acos -1.0-1.0i) 2.23703575928741+1.06127506190504i)
(num-test (acos 1.0+3.14159265358979i) 1.27521129258327-1.90462768697066i)
(num-test (acos -1.0+3.14159265358979i) 1.86638136100653-1.90462768697066i)
(num-test (acos 1.0-3.14159265358979i) 1.27521129258327+1.90462768697066i)
(num-test (acos -1.0-3.14159265358979i) 1.86638136100653+1.90462768697066i)
(num-test (acos 1.0+2.71828182845905i) 1.23635355491853-1.77905438385935i)
(num-test (acos -1.0+2.71828182845905i) 1.90523909867127-1.77905438385935i)
(num-test (acos 1.0-2.71828182845905i) 1.23635355491853+1.77905438385935i)
(num-test (acos -1.0-2.71828182845905i) 1.90523909867127+1.77905438385935i)
(num-test (acos 1.0+1234.0i) 1.56998595446684-7.81116387772663i)
(num-test (acos -1.0+1234.0i) 1.57160669912296-7.81116387772663i)
(num-test (acos 1.0-1234.0i) 1.56998595446690+7.81116387755283i)
(num-test (acos -1.0-1234.0i) 1.57160669912289+7.81116387755283i)
(num-test (acos 1.0-1234000000.0i) 1.57079632598452+21.62667394298955i)
(num-test (acos -1.0-1234000000.0i) 1.57079632760527+21.62667394298955i)
(num-test (acos 3.14159265358979+0.0i) 0.0-1.81152627246085i)
(num-test (acos -3.14159265358979+0.0i) 3.14159265358979-1.81152627246085i)
(num-test (acos 3.14159265358979-0.0i) 0.0-1.81152627246085i)
(num-test (acos -3.14159265358979-0.0i) 3.14159265358979-1.81152627246085i)
(num-test (acos 3.14159265358979+0.00000001i) 0.00000000335775-1.81152627246085i)
(num-test (acos -3.14159265358979+0.00000001i) 3.14159265023205-1.81152627246085i)
(num-test (acos 3.14159265358979-0.00000001i) 0.00000000335775+1.81152627246085i)
(num-test (acos -3.14159265358979-0.00000001i) 3.14159265023205+1.81152627246085i)
(num-test (acos 3.14159265358979+1.0i) 0.32225329398146-1.86711439316026i)
(num-test (acos -3.14159265358979+1.0i) 2.81933935960833-1.86711439316026i)
(num-test (acos 3.14159265358979-1.0i) 0.32225329398146+1.86711439316026i)
(num-test (acos -3.14159265358979-1.0i) 2.81933935960833+1.86711439316026i)
(num-test (acos 3.14159265358979+3.14159265358979i) 0.79805654766741-2.18469104082751i)
(num-test (acos -3.14159265358979+3.14159265358979i) 2.34353610592238-2.18469104082751i)
(num-test (acos 3.14159265358979-3.14159265358979i) 0.79805654766741+2.18469104082751i)
(num-test (acos -3.14159265358979-3.14159265358979i) 2.34353610592238+2.18469104082751i)
(num-test (acos 3.14159265358979+2.71828182845905i) 0.72769976263454-2.11552790803290i)
(num-test (acos -3.14159265358979+2.71828182845905i) 2.41389289095525-2.11552790803290i)
(num-test (acos 3.14159265358979-2.71828182845905i) 0.72769976263454+2.11552790803290i)
(num-test (acos -3.14159265358979-2.71828182845905i) 2.41389289095525+2.11552790803290i)
(num-test (acos 3.14159265358979+1234.0i) 1.56825047198589-7.81116678966949i)
(num-test (acos -3.14159265358979+1234.0i) 1.57334218160390-7.81116678966949i)
(num-test (acos 3.14159265358979-1234.0i) 1.56825047198552+7.81116678989204i)
(num-test (acos -3.14159265358979-1234.0i) 1.57334218160427+7.81116678989204i)
(num-test (acos 3.14159265358979-1234000000.0i) 1.57079632424904+21.62667394298955i)
(num-test (acos -3.14159265358979-1234000000.0i) 1.57079632934076+21.62667394298955i)
(num-test (acos 2.71828182845905+0.0i) 0.0-1.65745445415308i)
(num-test (acos -2.71828182845905+0.0i) 3.14159265358979-1.65745445415308i)
(num-test (acos 2.71828182845905-0.0i) 0.0-1.65745445415308i)
(num-test (acos -2.71828182845905-0.0i) 3.14159265358979-1.65745445415308i)
(num-test (acos 2.71828182845905+0.00000001i) 0.00000000395623-1.65745445415308i)
(num-test (acos -2.71828182845905+0.00000001i) 3.14159264963356-1.65745445415308i)
(num-test (acos 2.71828182845905-0.00000001i) 0.00000000395623+1.65745445415308i)
(num-test (acos -2.71828182845905-0.00000001i) 3.14159264963356+1.65745445415308i)
(num-test (acos 2.71828182845905+1.0i) 0.37321824160908-1.73375471569385i)
(num-test (acos -2.71828182845905+1.0i) 2.76837441198071-1.73375471569385i)
(num-test (acos 2.71828182845905-1.0i) 0.37321824160908+1.73375471569385i)
(num-test (acos -2.71828182845905-1.0i) 2.76837441198071+1.73375471569385i)
(num-test (acos 2.71828182835905+3.14159265358979i) 0.87174835684834-2.11968336368048i)
(num-test (acos -2.71828182845905+3.14159265358979i) 2.26984429674145-2.11968336368048i)
(num-test (acos 2.71828182845905-3.14159265358979i) 0.87174835684834+2.11968336368048i)
(num-test (acos -2.71828182845905-3.14159265358979i) 2.26984429674145+2.11968336368048i)
(num-test (acos 2.71828182845905+2.71828182845905i) 0.80229897091015-2.04014932880026i)
(num-test (acos -2.71828182845905+2.71828182845905i) 2.33929368267965-2.04014932880026i)
(num-test (acos 2.71828182845905-2.71828182845905i) 0.80229897091014+2.04014932880026i)
(num-test (acos -2.71828182845905-2.71828182845905i) 2.33929368267965+2.04014932880026i)
(num-test (acos 2.71828182845905+1234.0i) 1.56859350950254-7.81116597510553i)
(num-test (acos -2.71828182845905+1234.0i) 1.57299914408725-7.81116597510553i)
(num-test (acos 2.71828182845905-1234.0i) 1.56859350950221+7.81116597540442i)
(num-test (acos -2.71828182845905-1234.0i) 1.57299914408758+7.81116597540442i)
(num-test (acos 2.71828182845905-1234000000.0i) 1.57079632459208+21.62667394298955i)
(num-test (acos -2.71828182845905-1234000000.0i) 1.57079632899772+21.62667394298955i)
(num-test (acos 1234.0+0.0i) 0.0-7.81116322068415i)
(num-test (acos -1234.0+0.0i) 3.14159265358979-7.81116322068415i)
(num-test (acos 1234.0-0.0i) 0.0-7.81116322068415i)
(num-test (acos -1234.0-0.0i) 3.14159265358979-7.81116322068415i)
(num-test (acos 1234.0+0.00000001i) 0.00000000000810-7.81116322068415i)
(num-test (acos -1234.0+0.00000001i) 3.14159265358169-7.81116322068415i)
(num-test (acos 1234.0-0.00000001i) 0.00000000000810+7.81116322084923i)
(num-test (acos -1234.0-0.00000001i) 3.14159265358169+7.81116322084923i)
(num-test (acos 1234.0+1.0i) 0.00081037286048-7.81116354944842i)
(num-test (acos -1234.0+1.0i) 3.14078228072931-7.81116354944842i)
(num-test (acos 1234.0-1.0i) 0.00081037286017+7.81116354920146i)
(num-test (acos -1234.0-1.0i) 3.14078228072962+7.81116354920146i)
(num-test (acos 1234.0+3.14159265358979i) 0.00254585647983-7.81116646138554i)
(num-test (acos -1234.0+3.14159265358979i) 3.13904679710996-7.81116646138554i)
(num-test (acos 1234.0-3.14159265358979i) 0.00254585648123+7.81116646154641i)
(num-test (acos -1234.0-3.14159265358979i) 3.13904679710856+7.81116646154641i)
(num-test (acos 1234.0+2.71828182845905i) 0.00220281873947-7.81116564738434i)
(num-test (acos -1234.0+2.71828182845905i) 3.13938983485033-7.81116564738434i)
(num-test (acos 1234.0-2.71828182845905i) 0.00220281873928+7.81116564705719i)
(num-test (acos -1234.0-2.71828182845905i) 3.13938983485052+7.81116564705719i)
(num-test (acos 1234.0+1234.0i) 0.78539824532655-8.15773697538346i)
(num-test (acos -1234.0+1234.0i) 2.35619440826324-8.15773697538346i)
(num-test (acos 1234.0-1234.0i) 0.78539824548545+8.15773697530526i)
(num-test (acos -1234.0-1234.0i) 2.35619440810434+8.15773697530526i)
(num-test (acos 1234.0-1234000000.0i) 1.57079532679490+21.62667394299005i)
(num-test (acos -1234.0-1234000000.0i) 1.57079732679490+21.62667394299005i)
(num-test (acos 1234000000.0-0.00000001i) 0.0+21.62667394298955i)
(num-test (acos -1234000000.0-0.00000001i) 3.14159265358979+21.62667394298955i)
(num-test (acos 1234000000.0-1.0i) 0.00000000081037+21.62667394298955i)
(num-test (acos -1234000000.0-1.0i) 3.14159265277942+21.62667394298955i)
(num-test (acos 1234000000.0-3.14159265358979i) 0.00000000254586+21.62667394298955i)
(num-test (acos -1234000000.0-3.14159265358979i) 3.14159265104393+21.62667394298955i)
(num-test (acos 1234000000.0-2.71828182845905i) 0.00000000220282+21.62667394298955i)
(num-test (acos -1234000000.0-2.71828182845905i) 3.14159265138697+21.62667394298955i)
(num-test (acos 1234000000.0-1234.0i) 0.00000100000000+21.62667394299005i)
(num-test (acos -1234000000.0-1234.0i) 3.14159165358979+21.62667394299005i)
(num-test (acos 1234000000.0-1234000000.0i) 0.78539816339745+21.97324753326953i)
(num-test (acos -1234000000.0-1234000000.0i) 2.35619449019234+21.97324753326953i)
(num-test (acos -2.0) 3.141592653589793-1.316957896924817i)
(num-test (acos 3.0+70000000i) 1.570796283937754-18.7571529895002i) ; C breaks near here
(num-test (acos 70000000+3i) 4.725807101202406E-8-18.75715298057358i)
(num-test (acos 3.0-70000000i) 1.570796283937754+18.7571529895002i)
(num-test (acos -70000000+3i) 3.141592606331722-18.75715298057358i)
(num-test (acos 1.5) 0.0+0.9624236501192069i)
(num-test (acos -1.5) 3.141592653589793-0.9624236501192069i)
(num-test (acos -1.0e+01) 3.1415926535897932385e0-2.9932228461263808979e0i)
(num-test (acos -2.0e+00) 3.1415926535897932385e0-1.3169578969248167086e0i)
(num-test (acos -1.0e+00) 3.1415926535897932385e0+0.0i)
(num-test (acos -7.50e-01) 2.4188584057763776273e0+0.0i)
(num-test (acos -5.0e-01) 2.0943951023931954923e0+0.0i)
(num-test (acos -1.250e-01) 1.6961241579629620161e0+0.0i)
(num-test (acos -3.45266983001243932001e-04) 1.5711415937847577022e0+0.0i)
(num-test (acos -1.19209289550781250e-07) 1.570796446004186170e0+0.0i)
(num-test (acos 0.0e+00) 1.5707963267948966192e0+0.0i)
(num-test (acos 1.19209289550781250e-07) 1.5707962075856070684e0+0.0i)
(num-test (acos 3.45266983001243932001e-04) 1.5704510598050355363e0+0.0i)
(num-test (acos 1.250e-01) 1.4454684956268312224e0+0.0i)
(num-test (acos 5.0e-01) 1.0471975511965977462e0+0.0i)
(num-test (acos 7.50e-01) 7.2273424781341561118e-1+0.0i)
(num-test (acos 1.0e+00) 0e0+0.0i)
(num-test (acos 2.0e+00) 0+1.3169578969248167086e0i)
(num-test (acos 1.0e+01) 0+2.9932228461263808979e0i)
(num-test (acos 0.0e+00+0.0e+00i) 1.5707963267948966192e0+0.0i)
(num-test (acos 0.0e+00+1.19209289550781250e-07i) 1.5707963267948966192e0-1.1920928955078096766e-7i)
(num-test (acos 0.0e+00-1.19209289550781250e-07i) 1.5707963267948966192e0+1.1920928955078096766e-7i)
(num-test (acos 0.0e+00+5.0e-01i) 1.5707963267948966192e0-4.8121182505960344750e-1i)
(num-test (acos 0.0e+00-5.0e-01i) 1.5707963267948966192e0+4.8121182505960344750e-1i)
(num-test (acos 0.0e+00+1.0e+00i) 1.5707963267948966192e0-8.8137358701954302523e-1i)
(num-test (acos 0.0e+00-1.0e+00i) 1.5707963267948966192e0+8.8137358701954302523e-1i)
(num-test (acos 0.0e+00+2.0e+00i) 1.5707963267948966192e0-1.4436354751788103425e0i)
(num-test (acos 0.0e+00-2.0e+00i) 1.5707963267948966192e0+1.4436354751788103425e0i)
(num-test (acos 0.0e+00+8.3886080e+06i) 1.5707963267948966192e0-1.6635532333438690979e1i)
(num-test (acos 0.0e+00-8.3886080e+06i) 1.5707963267948966192e0+1.6635532333438690979e1i)
(num-test (acos 1.19209289550781250e-07+0.0e+00i) 1.5707962075856070684e0+0.0i)
(num-test (acos -1.19209289550781250e-07+0.0e+00i) 1.570796446004186170e0+0.0i)
(num-test (acos 1.19209289550781250e-07+1.19209289550781250e-07i) 1.5707962075856070685e0-1.1920928955078181469e-7i)
(num-test (acos 1.19209289550781250e-07-1.19209289550781250e-07i) 1.5707962075856070685e0+1.1920928955078181469e-7i)
(num-test (acos -1.19209289550781250e-07+1.19209289550781250e-07i) 1.570796446004186170e0-1.1920928955078181469e-7i)
(num-test (acos -1.19209289550781250e-07-1.19209289550781250e-07i) 1.570796446004186170e0+1.1920928955078181469e-7i)
(num-test (acos 1.19209289550781250e-07+5.0e-01i) 1.5707962201708666252e0-4.8121182505960598961e-1i)
(num-test (acos 1.19209289550781250e-07-5.0e-01i) 1.5707962201708666252e0+4.8121182505960598961e-1i)
(num-test (acos -1.19209289550781250e-07+5.0e-01i) 1.5707964334189266132e0-4.8121182505960598961e-1i)
(num-test (acos -1.19209289550781250e-07-5.0e-01i) 1.5707964334189266132e0+4.8121182505960598961e-1i)
(num-test (acos 1.19209289550781250e-07+1.0e+00i) 1.5707962425011995974e0-8.8137358701954553738e-1i)
(num-test (acos 1.19209289550781250e-07-1.0e+00i) 1.5707962425011995974e0+8.8137358701954553738e-1i)
(num-test (acos -1.19209289550781250e-07+1.0e+00i) 1.5707964110885936410e0-8.8137358701954553738e-1i)
(num-test (acos -1.19209289550781250e-07-1.0e+00i) 1.5707964110885936410e0+8.8137358701954553738e-1i)
(num-test (acos 1.19209289550781250e-07+2.0e+00i) 1.5707962734828816222e0-1.4436354751788116136e0i)
(num-test (acos 1.19209289550781250e-07-2.0e+00i) 1.5707962734828816222e0+1.4436354751788116136e0i)
(num-test (acos -1.19209289550781250e-07+2.0e+00i) 1.5707963801069116162e0-1.4436354751788116136e0i)
(num-test (acos -1.19209289550781250e-07-2.0e+00i) 1.5707963801069116162e0+1.4436354751788116136e0i)
(num-test (acos 1.19209289550781250e-07+8.3886080e+06i) 1.5707963267948824084e0-1.6635532333438690979e1i)
(num-test (acos 1.19209289550781250e-07-8.3886080e+06i) 1.5707963267948824084e0+1.6635532333438690979e1i)
(num-test (acos -1.19209289550781250e-07+8.3886080e+06i) 1.5707963267949108301e0-1.6635532333438690979e1i)
(num-test (acos -1.19209289550781250e-07-8.3886080e+06i) 1.5707963267949108301e0+1.6635532333438690979e1i)
(num-test (acos 5.0e-01+0.0e+00i) 1.0471975511965977462e0+0.0i)
(num-test (acos -5.0e-01+0.0e+00i) 2.0943951023931954923e0+0.0i)
(num-test (acos 5.0e-01+1.19209289550781250e-07i) 1.0471975511966032159e0-1.3765103082409432364e-7i)
(num-test (acos 5.0e-01-1.19209289550781250e-07i) 1.0471975511966032159e0+1.3765103082409432364e-7i)
(num-test (acos -5.0e-01+1.19209289550781250e-07i) 2.0943951023931900225e0-1.3765103082409432364e-7i)
(num-test (acos -5.0e-01-1.19209289550781250e-07i) 2.0943951023931900225e0+1.3765103082409432364e-7i)
(num-test (acos 5.0e-01+5.0e-01i) 1.1185178796437059372e0-5.3063753095251782602e-1i)
(num-test (acos 5.0e-01-5.0e-01i) 1.1185178796437059372e0+5.3063753095251782602e-1i)
(num-test (acos -5.0e-01+5.0e-01i) 2.0230747739460873013e0-5.3063753095251782602e-1i)
(num-test (acos -5.0e-01-5.0e-01i) 2.0230747739460873013e0+5.3063753095251782602e-1i)
(num-test (acos 5.0e-01+1.0e+00i) 1.2213572639376833256e0-9.2613303135018242455e-1i)
(num-test (acos 5.0e-01-1.0e+00i) 1.2213572639376833256e0+9.2613303135018242455e-1i)
(num-test (acos -5.0e-01+1.0e+00i) 1.9202353896521099129e0-9.2613303135018242455e-1i)
(num-test (acos -5.0e-01-1.0e+00i) 1.9202353896521099129e0+9.2613303135018242455e-1i)
(num-test (acos 5.0e-01+2.0e+00i) 1.3497776911720127603e0-1.4657153519472905218e0i)
(num-test (acos 5.0e-01-2.0e+00i) 1.3497776911720127603e0+1.4657153519472905218e0i)
(num-test (acos -5.0e-01+2.0e+00i) 1.7918149624177804781e0-1.4657153519472905218e0i)
(num-test (acos -5.0e-01-2.0e+00i) 1.7918149624177804781e0+1.4657153519472905218e0i)
(num-test (acos 5.0e-01+8.3886080e+06i) 1.5707962671902518438e0-1.6635532333438692755e1i)
(num-test (acos 5.0e-01-8.3886080e+06i) 1.5707962671902518438e0+1.6635532333438692755e1i)
(num-test (acos -5.0e-01+8.3886080e+06i) 1.5707963863995413946e0-1.6635532333438692755e1i)
(num-test (acos -5.0e-01-8.3886080e+06i) 1.5707963863995413946e0+1.6635532333438692755e1i)
(num-test (acos 1.0e+00+0.0e+00i) 0e0+0.0i)
(num-test (acos -1.0e+00+0.0e+00i) 3.1415926535897932385e0+0.0i)
(num-test (acos 1.0e+00+1.19209289550781250e-07i) 3.4526697957132450399e-4-3.4526698643116312881e-4i)
(num-test (acos 1.0e+00-1.19209289550781250e-07i) 3.4526697957132450399e-4+3.4526698643116312881e-4i)
(num-test (acos -1.0e+00+1.19209289550781250e-07i) 3.1412473866102219140e0-3.4526698643116312881e-4i)
(num-test (acos -1.0e+00-1.19209289550781250e-07i) 3.1412473866102219140e0+3.4526698643116312881e-4i)
(num-test (acos 1.0e+00+5.0e-01i) 6.7488884558600638016e-1-7.3285767597364526089e-1i)
(num-test (acos 1.0e+00-5.0e-01i) 6.7488884558600638016e-1+7.3285767597364526089e-1i)
(num-test (acos -1.0e+00+5.0e-01i) 2.4667038080037868583e0-7.3285767597364526089e-1i)
(num-test (acos -1.0e+00-5.0e-01i) 2.4667038080037868583e0+7.3285767597364526089e-1i)
(num-test (acos 1.0e+00+1.0e+00i) 9.0455689430238136413e-1-1.0612750619050356520e0i)
(num-test (acos 1.0e+00-1.0e+00i) 9.0455689430238136413e-1+1.0612750619050356520e0i)
(num-test (acos -1.0e+00+1.0e+00i) 2.2370357592874118743e0-1.0612750619050356520e0i)
(num-test (acos -1.0e+00-1.0e+00i) 2.2370357592874118743e0+1.0612750619050356520e0i)
(num-test (acos 1.0e+00+2.0e+00i) 1.1437177404024204938e0-1.5285709194809981613e0i)
(num-test (acos 1.0e+00-2.0e+00i) 1.1437177404024204938e0+1.5285709194809981613e0i)
(num-test (acos -1.0e+00+2.0e+00i) 1.9978749131873727447e0-1.5285709194809981613e0i)
(num-test (acos -1.0e+00-2.0e+00i) 1.9978749131873727447e0+1.5285709194809981613e0i)
(num-test (acos 1.0e+00+8.3886080e+06i) 1.5707962075856070685e0-1.6635532333438698084e1i)
(num-test (acos 1.0e+00-8.3886080e+06i) 1.5707962075856070685e0+1.6635532333438698084e1i)
(num-test (acos -1.0e+00+8.3886080e+06i) 1.570796446004186170e0-1.6635532333438698084e1i)
(num-test (acos -1.0e+00-8.3886080e+06i) 1.570796446004186170e0+1.6635532333438698084e1i)
(num-test (acos 2.0e+00+0.0e+00i) 0+1.3169578969248167086e0i)
(num-test (acos -2.0e+00+0.0e+00i) 3.1415926535897932385e0-1.3169578969248167086e0i)
(num-test (acos 2.0e+00+1.19209289550781250e-07i) 6.8825515412047433504e-8-1.3169578969248194435e0i)
(num-test (acos 2.0e+00-1.19209289550781250e-07i) 6.8825515412047433504e-8+1.3169578969248194435e0i)
(num-test (acos -2.0e+00+1.19209289550781250e-07i) 3.1415925847642778264e0-1.3169578969248194435e0i)
(num-test (acos -2.0e+00-1.19209289550781250e-07i) 3.1415925847642778264e0+1.3169578969248194435e0i)
(num-test (acos 2.0e+00+5.0e-01i) 2.7775425655771396018e-1-1.3618009008578457882e0i)
(num-test (acos 2.0e+00-5.0e-01i) 2.7775425655771396018e-1+1.3618009008578457882e0i)
(num-test (acos -2.0e+00+5.0e-01i) 2.8638383970320792783e0-1.3618009008578457882e0i)
(num-test (acos -2.0e+00-5.0e-01i) 2.8638383970320792783e0+1.3618009008578457882e0i)
(num-test (acos 2.0e+00+1.0e+00i) 5.0735630321714456304e-1-1.4693517443681852733e0i)
(num-test (acos 2.0e+00-1.0e+00i) 5.0735630321714456304e-1+1.4693517443681852733e0i)
(num-test (acos -2.0e+00+1.0e+00i) 2.6342363503726486754e0-1.4693517443681852733e0i)
(num-test (acos -2.0e+00-1.0e+00i) 2.6342363503726486754e0+1.4693517443681852733e0i)
(num-test (acos 2.0e+00+2.0e+00i) 8.1654718209685057852e-1-1.7343245214879664480e0i)
(num-test (acos 2.0e+00-2.0e+00i) 8.1654718209685057852e-1+1.7343245214879664480e0i)
(num-test (acos -2.0e+00+2.0e+00i) 2.3250454714929426599e0-1.7343245214879664480e0i)
(num-test (acos -2.0e+00-2.0e+00i) 2.3250454714929426599e0+1.7343245214879664480e0i)
(num-test (acos 2.0e+00+8.3886080e+06i) 1.5707960883763175177e0-1.663553233343871940e1i)
(num-test (acos 2.0e+00-8.3886080e+06i) 1.5707960883763175177e0+1.663553233343871940e1i)
(num-test (acos -2.0e+00+8.3886080e+06i) 1.5707965652134757208e0-1.663553233343871940e1i)
(num-test (acos -2.0e+00-8.3886080e+06i) 1.5707965652134757208e0+1.663553233343871940e1i)
(num-test (acos 8.3886080e+06+0.0e+00i) 0+1.6635532333438683873e1i)
(num-test (acos -8.3886080e+06+0.0e+00i) 3.1415926535897932385e0-1.6635532333438683873e1i)
(num-test (acos 8.3886080e+06+1.19209289550781250e-07i) 1.4210854715202104692e-14-1.6635532333438683873e1i)
(num-test (acos 8.3886080e+06-1.19209289550781250e-07i) 1.4210854715202104692e-14+1.6635532333438683873e1i)
(num-test (acos -8.3886080e+06+1.19209289550781250e-07i) 3.1415926535897790276e0-1.6635532333438683873e1i)
(num-test (acos -8.3886080e+06-1.19209289550781250e-07i) 3.1415926535897790276e0+1.6635532333438683873e1i)
(num-test (acos 8.3886080e+06+5.0e-01i) 5.9604644775390977930e-8-1.6635532333438685650e1i)
(num-test (acos 8.3886080e+06-5.0e-01i) 5.9604644775390977930e-8+1.6635532333438685650e1i)
(num-test (acos -8.3886080e+06+5.0e-01i) 3.1415925939851484631e0-1.6635532333438685650e1i)
(num-test (acos -8.3886080e+06-5.0e-01i) 3.1415925939851484631e0+1.6635532333438685650e1i)
(num-test (acos 8.3886080e+06+1.0e+00i) 1.1920928955078153234e-7-1.6635532333438690979e1i)
(num-test (acos 8.3886080e+06-1.0e+00i) 1.1920928955078153234e-7+1.6635532333438690979e1i)
(num-test (acos -8.3886080e+06+1.0e+00i) 3.1415925343805036877e0-1.6635532333438690979e1i)
(num-test (acos -8.3886080e+06-1.0e+00i) 3.1415925343805036877e0+1.6635532333438690979e1i)
(num-test (acos 8.3886080e+06+2.0e+00i) 2.3841857910155967656e-7-1.6635532333438712295e1i)
(num-test (acos 8.3886080e+06-2.0e+00i) 2.3841857910155967656e-7+1.6635532333438712295e1i)
(num-test (acos -8.3886080e+06+2.0e+00i) 3.1415924151712141369e0-1.6635532333438712295e1i)
(num-test (acos -8.3886080e+06-2.0e+00i) 3.1415924151712141369e0+1.6635532333438712295e1i)
(num-test (acos 8.3886080e+06+8.3886080e+06i) 7.8539816339745008597e-1-1.6982105923718660081e1i)
(num-test (acos 8.3886080e+06-8.3886080e+06i) 7.8539816339745008597e-1+1.6982105923718660081e1i)
(num-test (acos -8.3886080e+06+8.3886080e+06i) 2.3561944901923431525e0-1.6982105923718660081e1i)
(num-test (acos -8.3886080e+06-8.3886080e+06i) 2.3561944901923431525e0+1.6982105923718660081e1i)
(num-test (acos 1234000000/3) 0+20.52806165432143i) 
(num-test (acos 0.00000001+1234000000.0i) 1.570796326794897-21.62667380877375i) 
(num-test (acos 3.14159265358979+1234000000.0i) 1.570796324249036-21.62667380877375i) 
(num-test (acos 2.71828182845905+1234000000.0i) 1.570796324592076-21.62667380877375i) 
(num-test (acos 1234000000.0+0.00000001i) -4.051865509779873E-10-21.62667394298955i) 
(num-test (acos 1234000000.0+3.14159265358979i) -4.051865509779873E-10-21.62667394298955i) 
(num-test (acos 1234000000.0+2.71828182845905i) -4.051865509779873E-10-21.62667394298955i) 
(num-test (acos -64983.97009220963-48983.30494825104i) 2.495679792792491+12.0i)

(let ((err 0.0))
  (do ((i 0 (+ i 1))
       (x -10.0 (+ x .1)))
      ((= i 200))
    (let ((y (magnitude (- x (cos (acos x))))))
      (if (> y err) (set! err y))))
  (if (> err 1e-12)
      (begin (display "(cos (acos ...)) error: ") (display err) (newline))))

(let ((err 0.0))
  (do ((i 0 (+ i 1))
       (x 1.0-i (+ x -0.1+0.1i)))
      ((= i 200))
    (let ((y (magnitude (- x (cos (acos x))))))
      (if (> y err) (set! err y))))
  (if (> err 1e-10)
      (begin (display "(cos (acos [complex] ...)) error: ") (display err) (newline))))


;; -------- atan
(num-test (atan 0) 0.0)
(num-test (atan 1) 0.78539816339745)
(num-test (atan -1) -0.78539816339745)
(num-test (atan 2) 1.10714871779409)
(num-test (atan -2) -1.10714871779409)
(num-test (atan 3) 1.24904577239825)
(num-test (atan -3) -1.24904577239825)
(num-test (atan 10) 1.47112767430373)
(num-test (atan -10) -1.47112767430373)
(num-test (atan 1234) 1.56998595420081)
(num-test (atan -1234) -1.56998595420081)
(num-test (atan 0/1) 0.0)
(num-test (atan 0/2) 0.0)
(num-test (atan 0/3) 0.0)
(num-test (atan 0/10) 0.0)
(num-test (atan 0/1234) 0.0)
(num-test (atan 0/1234000000) 0.0)
(num-test (atan 0/500029) 0.0)
(num-test (atan 0/362880) 0.0)
(num-test (atan 1/1) 0.78539816339745)
(num-test (atan -1/1) -0.78539816339745)
(num-test (atan 1/2) 0.46364760900081)
(num-test (atan -1/2) -0.46364760900081)
(num-test (atan 1/3) 0.32175055439664)
(num-test (atan -1/3) -0.32175055439664)
(num-test (atan 1/10) 0.09966865249116)
(num-test (atan -1/10) -0.09966865249116)
(num-test (atan 1/1234) 0.00081037259408)
(num-test (atan -1/1234) -0.00081037259408)
(num-test (atan 1/1234000000) 0.00000000081037)
(num-test (atan -1/1234000000) -0.00000000081037)
(num-test (atan 1/500029) 0.00000199988401)
(num-test (atan -1/500029) -0.00000199988401)
(num-test (atan 1/362880) 0.00000275573192)
(num-test (atan -1/362880) -0.00000275573192)
(num-test (atan 2/1) 1.10714871779409)
(num-test (atan -2/1) -1.10714871779409)
(num-test (atan 2/2) 0.78539816339745)
(num-test (atan -2/2) -0.78539816339745)
(num-test (atan 2/3) 0.58800260354757)
(num-test (atan -2/3) -0.58800260354757)
(num-test (atan 2/10) 0.19739555984988)
(num-test (atan -2/10) -0.19739555984988)
(num-test (atan 2/1234) 0.00162074412382)
(num-test (atan -2/1234) -0.00162074412382)
(num-test (atan 2/1234000000) 0.00000000162075)
(num-test (atan -2/1234000000) -0.00000000162075)
(num-test (atan 2/500029) 0.00000399976801)
(num-test (atan -2/500029) -0.00000399976801)
(num-test (atan 2/362880) 0.00000551146384)
(num-test (atan -2/362880) -0.00000551146384)
(num-test (atan 3/1) 1.24904577239825)
(num-test (atan -3/1) -1.24904577239825)
(num-test (atan 3/2) 0.98279372324733)
(num-test (atan -3/2) -0.98279372324733)
(num-test (atan 3/3) 0.78539816339745)
(num-test (atan -3/3) -0.78539816339745)
(num-test (atan 3/10) 0.29145679447787)
(num-test (atan -3/10) -0.29145679447787)
(num-test (atan 3/1234) 0.00243111352487)
(num-test (atan -3/1234) -0.00243111352487)
(num-test (atan 3/1234000000) 0.00000000243112)
(num-test (atan -3/1234000000) -0.00000000243112)
(num-test (atan 3/500029) 0.00000599965202)
(num-test (atan -3/500029) -0.00000599965202)
(num-test (atan 3/362880) 0.00000826719577)
(num-test (atan -3/362880) -0.00000826719577)
(num-test (atan 10/1) 1.47112767430373)
(num-test (atan -10/1) -1.47112767430373)
(num-test (atan 10/2) 1.37340076694502)
(num-test (atan -10/2) -1.37340076694502)
(num-test (atan 10/3) 1.27933953231703)
(num-test (atan -10/3) -1.27933953231703)
(num-test (atan 10/10) 0.78539816339745)
(num-test (atan -10/10) -0.78539816339745)
(num-test (atan 10/1234) 0.00810355033005)
(num-test (atan -10/1234) -0.00810355033005)
(num-test (atan 10/1234000000) 0.00000000810373)
(num-test (atan -10/1234000000) -0.00000000810373)
(num-test (atan 10/500029) 0.00001999884006)
(num-test (atan -10/500029) -0.00001999884006)
(num-test (atan 10/362880) 0.00002755731922)
(num-test (atan -10/362880) -0.00002755731922)
(num-test (atan 1234/1) 1.56998595420081)
(num-test (atan -1234/1) -1.56998595420081)
(num-test (atan 1234/2) 1.56917558267108)
(num-test (atan -1234/2) -1.56917558267108)
(num-test (atan 1234/3) 1.56836521327003)
(num-test (atan -1234/3) -1.56836521327003)
(num-test (atan 1234/10) 1.56269277646485)
(num-test (atan -1234/10) -1.56269277646485)
(num-test (atan 1234/1234) 0.78539816339745)
(num-test (atan -1234/1234) -0.78539816339745)
(num-test (atan 1234/1234000000) 0.00000100000000)
(num-test (atan -1234/1234000000) -0.00000100000000)
(num-test (atan 1234/500029) 0.00246785185431)
(num-test (atan -1234/500029) -0.00246785185431)
(num-test (atan 1234/362880) 0.00340056008437)
(num-test (atan -1234/362880) -0.00340056008437)
(num-test (atan 1234000000/1234000000) 0.78539816339745)
(num-test (atan -1234000000/1234000000) -0.78539816339745)
(num-test (atan 1234000000/500029) 1.57039111693053)
(num-test (atan -1234000000/500029) -1.57039111693053)
(num-test (atan 1234000000/362880) 1.57050225873206)
(num-test (atan -1234000000/362880) -1.57050225873206)
(num-test (atan 500029/2) 1.57079232702688)
(num-test (atan -500029/2) -1.57079232702688)
(num-test (atan 500029/3) 1.57079032714288)
(num-test (atan -500029/3) -1.57079032714288)
(num-test (atan 500029/10) 1.57077632795483)
(num-test (atan -500029/10) -1.57077632795483)
(num-test (atan 500029/1234) 1.56832847494059)
(num-test (atan -500029/1234) -1.56832847494059)
(num-test (atan 500029/1234000000) 0.00040520986437)
(num-test (atan -500029/1234000000) -0.00040520986437)
(num-test (atan 500029/500029) 0.78539816339745)
(num-test (atan -500029/500029) -0.78539816339745)
(num-test (atan 500029/362880) 0.94301772831206)
(num-test (atan -500029/362880) -0.94301772831206)
(num-test (atan 362880/2) 1.57079081533105)
(num-test (atan -362880/2) -1.57079081533105)
(num-test (atan 362880/3) 1.57078805959913)
(num-test (atan -362880/3) -1.57078805959913)
(num-test (atan 362880/10) 1.57076876947568)
(num-test (atan -362880/10) -1.57076876947568)
(num-test (atan 362880/1234) 1.56739576671053)
(num-test (atan -362880/1234) -1.56739576671053)
(num-test (atan 362880/1234000000) 0.00029406806284)
(num-test (atan -362880/1234000000) -0.00029406806284)
(num-test (atan 362880/500029) 0.62777859848283)
(num-test (atan -362880/500029) -0.62777859848283)
(num-test (atan 362880/362880) 0.78539816339745)
(num-test (atan -362880/362880) -0.78539816339745)
(num-test (atan 0.0) 0.0)
(num-test (atan 0.00000001) 0.00000001)
(num-test (atan -0.00000001) -0.00000001)
(num-test (atan 1.0) 0.78539816339745)
(num-test (atan -1.0) -0.78539816339745)
(num-test (atan our-pi) 1.26262725567891)
(num-test (atan -3.14159265358979) -1.26262725567891)
(num-test (atan 2.71828182845905) 1.21828290501728)
(num-test (atan -2.71828182845905) -1.21828290501728)
(num-test (atan 1234.0) 1.56998595420081)
(num-test (atan -1234.0) -1.56998595420081)
(num-test (atan 0.0+0.0i) 0.0)
(num-test (atan -0.0+0.0i) 0.0)
(num-test (atan 0.0-0.0i) 0.0)
(num-test (atan -0.0-0.0i) -0.0)
(num-test (atan 0.0+0.00000001i) 0.0+0.00000001i)
(num-test (atan -0.0+0.00000001i) 0.0+0.00000001i)
(num-test (atan 0.0-0.00000001i) 0.0-0.00000001i)
(num-test (atan -0.0-0.00000001i) -0.0-0.00000001i)
(num-test (atan 0.0+3.14159265358979i) 1.57079632679490+0.32976531495670i)
(num-test (atan -0.0+3.14159265358979i) 1.57079632679490+0.32976531495670i)
(num-test (atan 0.0-3.14159265358979i) 1.57079632679490-0.32976531495670i)
(num-test (atan -0.0-3.14159265358979i) -1.57079632679490-0.32976531495670i)
(num-test (atan 0.0+2.71828182845905i) 1.57079632679490+0.38596841645265i)
(num-test (atan -0.0+2.71828182845905i) 1.57079632679490+0.38596841645265i)
(num-test (atan 0.0-2.71828182845905i) 1.57079632679490-0.38596841645265i)
(num-test (atan -0.0-2.71828182845905i) -1.57079632679490-0.38596841645265i)
(num-test (atan 0.0+1234.0i) 1.57079632679490+0.00081037294887i)
(num-test (atan -0.0+1234.0i) 1.57079632679490+0.00081037294887i)
(num-test (atan 0.0-1234.0i) 1.57079632679490-0.00081037294887i)
(num-test (atan -0.0-1234.0i) -1.57079632679490-0.00081037294887i)
(num-test (atan 0.00000001+0.0i) 0.00000001)
(num-test (atan -0.00000001+0.0i) -0.00000001)
(num-test (atan 0.00000001-0.0i) 0.00000001)
(num-test (atan -0.00000001-0.0i) -0.00000001)
(num-test (atan 0.00000001+0.00000001i) 0.00000001+0.00000001i)
(num-test (atan -0.00000001+0.00000001i) -0.00000001+0.00000001i)
(num-test (atan 0.00000001-0.00000001i) 0.00000001-0.00000001i)
(num-test (atan -0.00000001-0.00000001i) -0.00000001-0.00000001i)
(num-test (atan 0.00000001+1.0i) 0.78539816589789+9.55691396225616i)
(num-test (atan -0.00000001+1.0i) -0.78539816589789+9.55691396225616i)
(num-test (atan 0.00000001-1.0i) 0.78539816589789-9.55691396225616i)
(num-test (atan -0.00000001-1.0i) -0.78539816589789-9.55691396225616i)
(num-test (atan 0.00000001+3.14159265358979i) 1.57079632566745+0.32976531495670i)
(num-test (atan -0.00000001+3.14159265358979i) -1.57079632566745+0.32976531495670i)
(num-test (atan 0.00000001-3.14159265358979i) 1.57079632566745-0.32976531495670i)
(num-test (atan -0.00000001-3.14159265358979i) -1.57079632566745-0.32976531495670i)
(num-test (atan 0.00000001+2.71828182845905i) 1.57079632522972+0.38596841645265i)
(num-test (atan -0.00000001+2.71828182845905i) -1.57079632522972+0.38596841645265i)
(num-test (atan 0.00000001-2.71828182845905i) 1.57079632522972-0.38596841645265i)
(num-test (atan -0.00000001-2.71828182845905i) -1.57079632522972-0.38596841645265i)
(num-test (atan 0.00000001+1234.0i) 1.57079632679489+0.00081037294887i)
(num-test (atan -0.00000001+1234.0i) -1.57079632679489+0.00081037294887i)
(num-test (atan 0.00000001-1234.0i) 1.57079632679489-0.00081037294887i)
(num-test (atan -0.00000001-1234.0i) -1.57079632679489-0.00081037294887i)
(num-test (atan 1.0+0.0i) 0.78539816339745)
(num-test (atan -1.0+0.0i) -0.78539816339745)
(num-test (atan 1.0-0.0i) 0.78539816339745)
(num-test (atan -1.0-0.0i) -0.78539816339745)
(num-test (atan 1.0+0.00000001i) 0.78539816339745+0.00000000500000i)
(num-test (atan -1.0+0.00000001i) -0.78539816339745+0.00000000500000i)
(num-test (atan 1.0-0.00000001i) 0.78539816339745-0.00000000500000i)
(num-test (atan -1.0-0.00000001i) -0.78539816339745-0.00000000500000i)
(num-test (atan 1.0+1.0i) 1.01722196789785+0.40235947810853i)
(num-test (atan -1.0+1.0i) -1.01722196789785+0.40235947810853i)
(num-test (atan 1.0-1.0i) 1.01722196789785-0.40235947810853i)
(num-test (atan -1.0-1.0i) -1.01722196789785-0.40235947810853i)
(num-test (atan 1.0+3.14159265358979i) 1.47082882591946+0.29462144034086i)
(num-test (atan -1.0+3.14159265358979i) -1.47082882591946+0.29462144034086i)
(num-test (atan 1.0-3.14159265358979i) 1.47082882591946-0.29462144034086i)
(num-test (atan -1.0-3.14159265358979i) -1.47082882591946-0.29462144034086i)
(num-test (atan 1.0+2.71828182845905i) 1.43862796047891+0.33050259272341i)
(num-test (atan -1.0+2.71828182845905i) -1.43862796047891+0.33050259272341i)
(num-test (atan 1.0-2.71828182845905i) 1.43862796047891-0.33050259272341i)
(num-test (atan -1.0-2.71828182845905i) -1.43862796047891-0.33050259272341i)
(num-test (atan 1.0+1234.0i) 1.57079567009087+0.00081037241669i)
(num-test (atan -1.0+1234.0i) -1.57079567009087+0.00081037241669i)
(num-test (atan 1.0-1234.0i) 1.57079567009087-0.00081037241669i)
(num-test (atan -1.0-1234.0i) -1.57079567009087-0.00081037241669i)
(num-test (atan 3.14159265358979+0.0i) 1.26262725567891)
(num-test (atan -3.14159265358979+0.0i) -1.26262725567891)
(num-test (atan 3.14159265358979-0.0i) 1.26262725567891)
(num-test (atan -3.14159265358979-0.0i) -1.26262725567891)
(num-test (atan 3.14159265358979+0.00000001i) 1.26262725567891+0.00000000092000i)
(num-test (atan -3.14159265358979+0.00000001i) -1.26262725567891+0.00000000092000i)
(num-test (atan 3.14159265358979-0.00000001i) 1.26262725567891-0.00000000092000i)
(num-test (atan -3.14159265358979-0.00000001i) -1.26262725567891-0.00000000092000i)
(num-test (atan 3.14159265358979+1.0i) 1.28734057432439+0.08505998507745i)
(num-test (atan -3.14159265358979+1.0i) -1.28734057432439+0.08505998507745i)
(num-test (atan 3.14159265358979-1.0i) 1.28734057432439-0.08505998507745i)
(num-test (atan -3.14159265358979-1.0i) -1.28734057432439-0.08505998507745i)
(num-test (atan 3.14159265358979+3.14159265358979i) 1.40903828502376+0.15638868878130i)
(num-test (atan -3.14159265358979+3.14159265358979i) -1.40903828502376+0.15638868878130i)
(num-test (atan 3.14159265358979-3.14159265358979i) 1.40903828502376-0.15638868878130i)
(num-test (atan -3.14159265358979-3.14159265358979i) -1.40903828502376-0.15638868878130i)
(num-test (atan 3.14159265358979+2.71828182845905i) 1.38641010899673+0.15352587926173i)
(num-test (atan -3.14159265358979+2.71828182845905i) -1.38641010899673+0.15352587926173i)
(num-test (atan 3.14159265358979-2.71828182845905i) 1.38641010899673-0.15352587926173i)
(num-test (atan -3.14159265358979-2.71828182845905i) -1.38641010899673-0.15352587926173i)
(num-test (atan 3.14159265358979+1234.0i) 1.57079426371036+0.00081036769654i)
(num-test (atan -3.14159265358979+1234.0i) -1.57079426371036+0.00081036769654i)
(num-test (atan 3.14159265358979-1234.0i) 1.57079426371036-0.00081036769654i)
(num-test (atan -3.14159265358979-1234.0i) -1.57079426371036-0.00081036769654i)
(num-test (atan 2.71828182845905+0.0i) 1.21828290501728)
(num-test (atan -2.71828182845905+0.0i) -1.21828290501728)
(num-test (atan 2.71828182845905-0.0i) 1.21828290501728)
(num-test (atan -2.71828182845905-0.0i) -1.21828290501728)
(num-test (atan 2.71828182845905+0.00000001i) 1.21828290501728+0.00000000119203i)
(num-test (atan -2.71828182845905+0.00000001i) -1.21828290501728+0.00000000119203i)
(num-test (atan 2.71828182845905-0.00000001i) 1.21828290501728-0.00000000119203i)
(num-test (atan -2.71828182845905-0.00000001i) -1.21828290501728-0.00000000119203i)
(num-test (atan 2.71828182845905+1.0i) 1.25363416718071+0.10816322574795i)
(num-test (atan -2.71828182845905+1.0i) -1.25363416718071+0.10816322574795i)
(num-test (atan 2.71828182845905-1.0i) 1.25363416718071-0.10816322574795i)
(num-test (atan -2.71828182845905-1.0i) -1.25363416718071-0.10816322574795i)
(num-test (atan 2.71828182845905+3.14159265358979i) 1.40945039787275+0.17937970703436i)
(num-test (atan -2.71828182845905+3.14159265358979i) -1.40945039787275+0.17937970703436i)
(num-test (atan 2.71828182845905-3.14159265358979i) 1.40945039787275-0.17937970703436i)
(num-test (atan -2.71828182845905-3.14159265358979i) -1.40945039787275-0.17937970703436i)
(num-test (atan 2.71828182845905+2.71828182845905i) 1.38288382352616+0.17963089485802i)
(num-test (atan -2.71828182845905+2.71828182845905i) -1.38288382352616+0.17963089485802i)
(num-test (atan 2.71828182845905-2.71828182845905i) 1.38288382352616-0.17963089485802i)
(num-test (atan -2.71828182845905-2.71828182845905i) -1.38288382352616-0.17963089485802i)
(num-test (atan 2.71828182845905+1234.0i) 1.57079454169576+0.00081036901661i)
(num-test (atan -2.71828182845905+1234.0i) -1.57079454169576+0.00081036901661i)
(num-test (atan 2.71828182845905-1234.0i) 1.57079454169576-0.00081036901661i)
(num-test (atan -2.71828182845905-1234.0i) -1.57079454169576-0.00081036901661i)
(num-test (atan 1234.0+0.0i) 1.56998595420081)
(num-test (atan -1234.0+0.0i) -1.56998595420081)
(num-test (atan 1234.0-0.0i) 1.56998595420081)
(num-test (atan -1234.0-0.0i) -1.56998595420081)
(num-test (atan 1234.0+0.00000001i) 1.56998595420081+0.00000000000001i)
(num-test (atan -1234.0+0.00000001i) -1.56998595420081+0.00000000000001i)
(num-test (atan 1234.0-0.00000001i) 1.56998595420081-0.00000000000001i)
(num-test (atan -1234.0-0.00000001i) -1.56998595420081-0.00000000000001i)
(num-test (atan 1234.0+1.0i) 1.56998595473299+0.00000065670317i)
(num-test (atan -1234.0+1.0i) -1.56998595473299+0.00000065670317i)
(num-test (atan 1234.0-1.0i) 1.56998595473299-0.00000065670317i)
(num-test (atan -1234.0-1.0i) -1.56998595473299-0.00000065670317i)
(num-test (atan 1234.0+3.14159265358979i) 1.56998595945313+0.00000206308183i)
(num-test (atan -1234.0+3.14159265358979i) -1.56998595945313+0.00000206308183i)
(num-test (atan 1234.0-3.14159265358979i) 1.56998595945313-0.00000206308183i)
(num-test (atan -1234.0-3.14159265358979i) -1.56998595945313-0.00000206308183i)
(num-test (atan 1234.0+2.71828182845905i) 1.56998595813306+0.00000178509679i)
(num-test (atan -1234.0+2.71828182845905i) -1.56998595813306+0.00000178509679i)
(num-test (atan 1234.0-2.71828182845905i) 1.56998595813306-0.00000178509679i)
(num-test (atan -1234.0-2.71828182845905i) -1.56998595813306-0.00000178509679i)
(num-test (atan 1234.0+1234.0i) 1.57039114036481+0.00040518634139i)
(num-test (atan -1234.0+1234.0i) -1.57039114036481+0.00040518634139i)
(num-test (atan 1234.0-1234.0i) 1.57039114036481-0.00040518634139i)
(num-test (atan -1234.0-1234.0i) -1.57039114036481-0.00040518634139i)
(num-test (atan 6.9279836e-17) 6.9279836e-17)
(num-test (atan 1.7976931e+308) 1.5707963267949)
(num-test (atan 0 0) 0.0)
(num-test (atan 0 1.0) 0.0)
(num-test (atan 0 -1.0) our-pi)
(num-test (atan 1.0 0) (/ our-pi 2))
(num-test (atan -1.0 0) (/ our-pi -2))
(num-test (atan 0.0 0.0) 0.0)
(num-test (atan 0.0 0.00000001) 0.0)
(num-test (atan 0.0 1.0) 0.0)
(num-test (atan 0.0 our-pi) 0.0)
(num-test (atan 0.0 2.71828182845905) 0.0)
(num-test (atan 0.0 1234.0) 0.0)
(num-test (atan 0.0 1234000000.0) 0.0)
(num-test (atan 0.00000001 0.0) 1.57079632679490)
(num-test (atan 0.00000001 0.00000001) 0.78539816339745)
(num-test (atan 0.00000001 1.0) 0.00000001)
(num-test (atan 0.00000001 our-pi) 0.00000000318310)
(num-test (atan 0.00000001 2.71828182845905) 0.00000000367879)
(num-test (atan 0.00000001 1234.0) 0.00000000000810)
(num-test (atan 0.00000001 1234000000.0) 0.0)
(num-test (atan 1.0 0.0) 1.57079632679490)
(num-test (atan 1.0 0.00000001) 1.57079631679490)
(num-test (atan 1.0 1.0) 0.78539816339745)
(num-test (atan 1.0 our-pi) 0.30816907111598)
(num-test (atan 1.0 2.71828182845905) 0.35251342177762)
(num-test (atan 1.0 1234.0) 0.00081037259408)
(num-test (atan 1.0 1234000000.0) 0.00000000081037)
(num-test (atan 3.14159265358979 0.0) 1.57079632679490)
(num-test (atan 3.14159265358979 0.00000001) 1.57079632361180)
(num-test (atan 3.14159265358979 1.0) 1.26262725567891)
(num-test (atan 3.14159265358979 our-pi) 0.78539816339745)
(num-test (atan 3.14159265358979 2.71828182845905) 0.85751178635585)
(num-test (atan 3.14159265358979 1234.0) 0.00254585564530)
(num-test (atan 3.14159265358979 1234000000.0) 0.00000000254586)
(num-test (atan 2.71828182845905 0.0) 1.57079632679490)
(num-test (atan 2.71828182845905 0.00000001) 1.57079632311610)
(num-test (atan 2.71828182845905 1.0) 1.21828290501728)
(num-test (atan 2.71828182845905 our-pi) 0.71328454043905)
(num-test (atan 2.71828182845905 2.71828182845905) 0.78539816339745)
(num-test (atan 2.71828182845905 1234.0) 0.00220281801598)
(num-test (atan 2.71828182845905 1234000000.0) 0.00000000220282)
(num-test (atan 1234.0 0.0) 1.57079632679490)
(num-test (atan 1234.0 0.00000001) 1.57079632678679)
(num-test (atan 1234.0 1.0) 1.56998595420081)
(num-test (atan 1234.0 our-pi) 1.56825047114960)
(num-test (atan 1234.0 2.71828182845905) 1.56859350877892)
(num-test (atan 1234.0 1234.0) 0.78539816339745)
(num-test (atan 1234.0 1234000000.0) 0.00000100000000)
(num-test (atan 1234000000.0 0.0) 1.57079632679490)
(num-test (atan 1234000000.0 0.00000001) 1.57079632679490)
(num-test (atan 1234000000.0 1.0) 1.57079632598452)
(num-test (atan 1234000000.0 our-pi) 1.57079632424904)
(num-test (atan 1234000000.0 2.71828182845905) 1.57079632459208)
(num-test (atan 1234000000.0 1234.0) 1.57079532679490)
(num-test (atan 1234000000.0 1234000000.0) 0.78539816339745)
(num-test (atan 1234000000/3) 1.570796324363778) 
(num-test (atan 0.00000001+1234000000.0i) +1.570796326794897+8.103727714748784E-10i) 
(num-test (atan 3.14159265358979+1234000000.0i) +1.570796326794897+8.103727714748784E-10i) 
(num-test (atan 2.71828182845905+1234000000.0i) +1.570796326794897+8.103727714748784E-10i) 
(num-test (atan 1234000000.0+0.00000001i) +1.570796325984524+6.567040287478756E-27i) 
(num-test (atan 1234000000.0+3.14159265358979i) +1.570796325984524+2.063096552297151E-18i) 
(num-test (atan 1234000000.0+2.71828182845905i) +1.570796325984524+1.785106628021167E-18i)
(num-test (atan 1) 0.7853981633974483)
(num-test (atan 0.0e+00+0.0e+00i) 0e0+0.0i)
(num-test (atan 0.0e+00+1.19209289550781250e-07i) 0+1.1920928955078181469e-7i)
(num-test (atan 0.0e+00-1.19209289550781250e-07i) 0-1.1920928955078181469e-7i)
(num-test (atan 0.0e+00+5.0e-01i) 0+5.4930614433405484570e-1i)
(num-test (atan 0.0e+00-5.0e-01i) 0-5.4930614433405484570e-1i)
(num-test (atan 0.0e+00+2.0e+00i) 1.5707963267948966192e0+5.4930614433405484570e-1i)
(num-test (atan 0.0e+00-2.0e+00i) -1.5707963267948966192e0-5.4930614433405484570e-1i)
(num-test (atan 0.0e+00+8.3886080e+06i) 1.5707963267948966192e0+1.1920928955078181469e-7i)
(num-test (atan 0.0e+00-8.3886080e+06i) -1.5707963267948966192e0-1.1920928955078181469e-7i)
(num-test (atan 1.19209289550781250e-07+0.0e+00i) 1.1920928955078068531e-7+0.0i)
(num-test (atan -1.19209289550781250e-07+0.0e+00i) -1.1920928955078068531e-7+0.0i)
(num-test (atan 1.19209289550781250e-07+1.19209289550781250e-07i) 1.1920928955078237938e-7+1.1920928955078012062e-7i)
(num-test (atan 1.19209289550781250e-07-1.19209289550781250e-07i) 1.1920928955078237938e-7-1.1920928955078012062e-7i)
(num-test (atan -1.19209289550781250e-07+1.19209289550781250e-07i) -1.1920928955078237938e-7+1.1920928955078012062e-7i)
(num-test (atan -1.19209289550781250e-07-1.19209289550781250e-07i) -1.1920928955078237938e-7-1.1920928955078012062e-7i)
(num-test (atan 1.19209289550781250e-07+5.0e-01i) 1.5894571940103932425e-7+5.4930614433404221383e-1i)
(num-test (atan 1.19209289550781250e-07-5.0e-01i) 1.5894571940103932425e-7-5.4930614433404221383e-1i)
(num-test (atan -1.19209289550781250e-07+5.0e-01i) -1.5894571940103932425e-7+5.4930614433404221383e-1i)
(num-test (atan -1.19209289550781250e-07-5.0e-01i) -1.5894571940103932425e-7-5.4930614433404221383e-1i)
(num-test (atan 1.19209289550781250e-07+1.0e+00i) 7.8539819319977069731e-1+8.3177661667193446012e0i)
(num-test (atan 1.19209289550781250e-07-1.0e+00i) 7.8539819319977069731e-1-8.3177661667193446012e0i)
(num-test (atan -1.19209289550781250e-07+1.0e+00i) -7.8539819319977069731e-1+8.3177661667193446012e0i)
(num-test (atan -1.19209289550781250e-07-1.0e+00i) -7.8539819319977069731e-1-8.3177661667193446012e0i)
(num-test (atan 1.19209289550781250e-07+2.0e+00i) 1.5707962870584667690e0+5.4930614433405168773e-1i)
(num-test (atan 1.19209289550781250e-07-2.0e+00i) 1.5707962870584667690e0-5.4930614433405168773e-1i)
(num-test (atan -1.19209289550781250e-07+2.0e+00i) -1.5707962870584667690e0+5.4930614433405168773e-1i)
(num-test (atan -1.19209289550781250e-07-2.0e+00i) -1.5707962870584667690e0-5.4930614433405168773e-1i)
(num-test (atan 1.19209289550781250e-07+8.3886080e+06i) 1.5707963267948966192e0+1.1920928955078181469e-7i)
(num-test (atan 1.19209289550781250e-07-8.3886080e+06i) 1.5707963267948966192e0-1.1920928955078181469e-7i)
(num-test (atan -1.19209289550781250e-07+8.3886080e+06i) -1.5707963267948966192e0+1.1920928955078181469e-7i)
(num-test (atan -1.19209289550781250e-07-8.3886080e+06i) -1.5707963267948966192e0-1.1920928955078181469e-7i)
(num-test (atan 5.0e-01+0.0e+00i) 4.6364760900080611621e-1+0.0i)
(num-test (atan -5.0e-01+0.0e+00i) -4.6364760900080611621e-1+0.0i)
(num-test (atan 5.0e-01+1.19209289550781250e-07i) 4.6364760900081066369e-1+9.5367431640625072280e-8i)
(num-test (atan 5.0e-01-1.19209289550781250e-07i) 4.6364760900081066369e-1-9.5367431640625072280e-8i)
(num-test (atan -5.0e-01+1.19209289550781250e-07i) -4.6364760900081066369e-1+9.5367431640625072280e-8i)
(num-test (atan -5.0e-01-1.19209289550781250e-07i) -4.6364760900081066369e-1-9.5367431640625072280e-8i)
(num-test (atan 5.0e-01+5.0e-01i) 5.5357435889704525151e-1+4.0235947810852509365e-1i)
(num-test (atan 5.0e-01-5.0e-01i) 5.5357435889704525151e-1-4.0235947810852509365e-1i)
(num-test (atan -5.0e-01+5.0e-01i) -5.5357435889704525151e-1+4.0235947810852509365e-1i)
(num-test (atan -5.0e-01-5.0e-01i) -5.5357435889704525151e-1-4.0235947810852509365e-1i)
(num-test (atan 5.0e-01+1.0e+00i) 9.0788749496088038670e-1+7.0830333601405402006e-1i)
(num-test (atan 5.0e-01-1.0e+00i) 9.0788749496088038670e-1-7.0830333601405402006e-1i)
(num-test (atan -5.0e-01+1.0e+00i) -9.0788749496088038670e-1+7.0830333601405402006e-1i)
(num-test (atan -5.0e-01-1.0e+00i) -9.0788749496088038670e-1-7.0830333601405402006e-1i)
(num-test (atan 5.0e-01+2.0e+00i) 1.4215468610018069803e0+5.0037000005253101744e-1i)
(num-test (atan 5.0e-01-2.0e+00i) 1.4215468610018069803e0-5.0037000005253101744e-1i)
(num-test (atan -5.0e-01+2.0e+00i) -1.4215468610018069803e0+5.0037000005253101744e-1i)
(num-test (atan -5.0e-01-2.0e+00i) -1.4215468610018069803e0-5.0037000005253101744e-1i)
(num-test (atan 5.0e-01+8.3886080e+06i) 1.5707963267948895138e0+1.1920928955078139117e-7i)
(num-test (atan 5.0e-01-8.3886080e+06i) 1.5707963267948895138e0-1.1920928955078139117e-7i)
(num-test (atan -5.0e-01+8.3886080e+06i) -1.5707963267948895138e0+1.1920928955078139117e-7i)
(num-test (atan -5.0e-01-8.3886080e+06i) -1.5707963267948895138e0-1.1920928955078139117e-7i)
(num-test (atan 1.0e+00+0.0e+00i) 7.8539816339744830962e-1+0.0i)
(num-test (atan -1.0e+00+0.0e+00i) -7.8539816339744830962e-1+0.0i)
(num-test (atan 1.0e+00+1.19209289550781250e-07i) 7.8539816339745186233e-1+5.9604644775390483828e-8i)
(num-test (atan 1.0e+00-1.19209289550781250e-07i) 7.8539816339745186233e-1-5.9604644775390483828e-8i)
(num-test (atan -1.0e+00+1.19209289550781250e-07i) -7.8539816339745186233e-1+5.9604644775390483828e-8i)
(num-test (atan -1.0e+00-1.19209289550781250e-07i) -7.8539816339745186233e-1-5.9604644775390483828e-8i)
(num-test (atan 1.0e+00+5.0e-01i) 8.4757566067082902713e-1+2.3887786125685909036e-1i)
(num-test (atan 1.0e+00-5.0e-01i) 8.4757566067082902713e-1-2.3887786125685909036e-1i)
(num-test (atan -1.0e+00+5.0e-01i) -8.4757566067082902713e-1+2.3887786125685909036e-1i)
(num-test (atan -1.0e+00-5.0e-01i) -8.4757566067082902713e-1-2.3887786125685909036e-1i)
(num-test (atan 1.0e+00+1.0e+00i) 1.0172219678978513677e0+4.0235947810852509365e-1i)
(num-test (atan 1.0e+00-1.0e+00i) 1.0172219678978513677e0-4.0235947810852509365e-1i)
(num-test (atan -1.0e+00+1.0e+00i) -1.0172219678978513677e0+4.0235947810852509365e-1i)
(num-test (atan -1.0e+00-1.0e+00i) -1.0172219678978513677e0-4.0235947810852509365e-1i)
(num-test (atan 1.0e+00+2.0e+00i) 1.3389725222944935611e0+4.0235947810852509365e-1i)
(num-test (atan 1.0e+00-2.0e+00i) 1.3389725222944935611e0-4.0235947810852509365e-1i)
(num-test (atan -1.0e+00+2.0e+00i) -1.3389725222944935611e0+4.0235947810852509365e-1i)
(num-test (atan -1.0e+00-2.0e+00i) -1.3389725222944935611e0-4.0235947810852509365e-1i)
(num-test (atan 1.0e+00+8.3886080e+06i) 1.5707963267948824084e0+1.1920928955078012062e-7i)
(num-test (atan 1.0e+00-8.3886080e+06i) 1.5707963267948824084e0-1.1920928955078012062e-7i)
(num-test (atan -1.0e+00+8.3886080e+06i) -1.5707963267948824084e0+1.1920928955078012062e-7i)
(num-test (atan -1.0e+00-8.3886080e+06i) -1.5707963267948824084e0-1.1920928955078012062e-7i)
(num-test (atan 2.0e+00+0.0e+00i) 1.1071487177940905030e0+0.0i)
(num-test (atan -2.0e+00+0.0e+00i) -1.1071487177940905030e0+0.0i)
(num-test (atan 2.0e+00+1.19209289550781250e-07i) 1.1071487177940916399e0+2.3841857910156200307e-8i)
(num-test (atan 2.0e+00-1.19209289550781250e-07i) 1.1071487177940916399e0-2.3841857910156200307e-8i)
(num-test (atan -2.0e+00+1.19209289550781250e-07i) -1.1071487177940916399e0+2.3841857910156200307e-8i)
(num-test (atan -2.0e+00-1.19209289550781250e-07i) -1.1071487177940916399e0-2.3841857910156200307e-8i)
(num-test (atan 2.0e+00+5.0e-01i) 1.1265564408348223487e0+9.6415620202996167238e-2i)
(num-test (atan 2.0e+00-5.0e-01i) 1.1265564408348223487e0-9.6415620202996167238e-2i)
(num-test (atan -2.0e+00+5.0e-01i) -1.1265564408348223487e0+9.6415620202996167238e-2i)
(num-test (atan -2.0e+00-5.0e-01i) -1.1265564408348223487e0-9.6415620202996167238e-2i)
(num-test (atan 2.0e+00+1.0e+00i) 1.1780972450961724644e0+1.7328679513998632735e-1i)
(num-test (atan 2.0e+00-1.0e+00i) 1.1780972450961724644e0-1.7328679513998632735e-1i)
(num-test (atan -2.0e+00+1.0e+00i) -1.1780972450961724644e0+1.7328679513998632735e-1i)
(num-test (atan -2.0e+00-1.0e+00i) -1.1780972450961724644e0-1.7328679513998632735e-1i)
(num-test (atan 2.0e+00+2.0e+00i) 1.3112232696716351433e0+2.3887786125685909036e-1i)
(num-test (atan 2.0e+00-2.0e+00i) 1.3112232696716351433e0-2.3887786125685909036e-1i)
(num-test (atan -2.0e+00+2.0e+00i) -1.3112232696716351433e0+2.3887786125685909036e-1i)
(num-test (atan -2.0e+00-2.0e+00i) -1.3112232696716351433e0-2.3887786125685909036e-1i)
(num-test (atan 2.0e+00+8.3886080e+06i) 1.5707963267948681975e0+1.1920928955077503843e-7i)
(num-test (atan 2.0e+00-8.3886080e+06i) 1.5707963267948681975e0-1.1920928955077503843e-7i)
(num-test (atan -2.0e+00+8.3886080e+06i) -1.5707963267948681975e0+1.1920928955077503843e-7i)
(num-test (atan -2.0e+00-8.3886080e+06i) -1.5707963267948681975e0-1.1920928955077503843e-7i)
(num-test (atan 8.3886080e+06+0.0e+00i) 1.5707962075856070685e0+0.0i)
(num-test (atan -8.3886080e+06+0.0e+00i) -1.5707962075856070685e0+0.0i)
(num-test (atan 8.3886080e+06+1.19209289550781250e-07i) 1.5707962075856070685e0+1.6940658945085766040e-21i)
(num-test (atan 8.3886080e+06-1.19209289550781250e-07i) 1.5707962075856070685e0-1.6940658945085766040e-21i)
(num-test (atan -8.3886080e+06+1.19209289550781250e-07i) -1.5707962075856070685e0+1.6940658945085766040e-21i)
(num-test (atan -8.3886080e+06-1.19209289550781250e-07i) -1.5707962075856070685e0-1.6940658945085766040e-21i)
(num-test (atan 8.3886080e+06+5.0e-01i) 1.5707962075856070685e0+7.1054273576008756410e-15i)
(num-test (atan 8.3886080e+06-5.0e-01i) 1.5707962075856070685e0-7.1054273576008756410e-15i)
(num-test (atan -8.3886080e+06+5.0e-01i) -1.5707962075856070685e0+7.1054273576008756410e-15i)
(num-test (atan -8.3886080e+06-5.0e-01i) -1.5707962075856070685e0-7.1054273576008756410e-15i)
(num-test (atan 8.3886080e+06+1.0e+00i) 1.5707962075856070685e0+1.4210854715201599821e-14i)
(num-test (atan 8.3886080e+06-1.0e+00i) 1.5707962075856070685e0-1.4210854715201599821e-14i)
(num-test (atan -8.3886080e+06+1.0e+00i) -1.5707962075856070685e0+1.4210854715201599821e-14i)
(num-test (atan -8.3886080e+06-1.0e+00i) -1.5707962075856070685e0-1.4210854715201599821e-14i)
(num-test (atan 8.3886080e+06+2.0e+00i) 1.5707962075856070685e0+2.8421709430401987951e-14i)
(num-test (atan 8.3886080e+06-2.0e+00i) 1.5707962075856070685e0-2.8421709430401987951e-14i)
(num-test (atan -8.3886080e+06+2.0e+00i) -1.5707962075856070685e0+2.8421709430401987951e-14i)
(num-test (atan -8.3886080e+06-2.0e+00i) -1.5707962075856070685e0-2.8421709430401987951e-14i)
(num-test (atan 8.3886080e+06+8.3886080e+06i) 1.5707962671902518438e0+5.9604644775390483828e-8i)
(num-test (atan 8.3886080e+06-8.3886080e+06i) 1.5707962671902518438e0-5.9604644775390483828e-8i)
(num-test (atan -8.3886080e+06+8.3886080e+06i) -1.5707962671902518438e0+5.9604644775390483828e-8i)
(num-test (atan -8.3886080e+06-8.3886080e+06i) -1.5707962671902518438e0-5.9604644775390483828e-8i)


(if with-hyperbolic-functions 
    (begin

      ;; -------- sinh
      (num-test (sinh 0) 0.0)
      (num-test (sinh 1) 1.17520119364380)
      (num-test (sinh -1) -1.17520119364380)
      (num-test (sinh 2) 3.62686040784702)
      (num-test (sinh -2) -3.62686040784702)
      (num-test (sinh 3) 10.01787492740990)
      (num-test (sinh -3) -10.01787492740990)
      (num-test (sinh 10) 11013.23287470339346)
      (num-test (sinh -10) -11013.23287470339346)
      (num-test (sinh 0/1) 0.0)
      (num-test (sinh 0/2) 0.0)
      (num-test (sinh 0/3) 0.0)
      (num-test (sinh 0/10) 0.0)
      (num-test (sinh 0/1234) 0.0)
      (num-test (sinh 0/1234000000) 0.0)
      (num-test (sinh 0/500029) 0.0)
      (num-test (sinh 0/362880) 0.0)
      (num-test (sinh 1/1) 1.17520119364380)
      (num-test (sinh -1/1) -1.17520119364380)
      (num-test (sinh 1/2) 0.52109530549375)
      (num-test (sinh -1/2) -0.52109530549375)
      (num-test (sinh 1/3) 0.33954055725615)
      (num-test (sinh -1/3) -0.33954055725615)
      (num-test (sinh 1/10) 0.10016675001984)
      (num-test (sinh -1/10) -0.10016675001984)
      (num-test (sinh 1/1234) 0.00081037286017)
      (num-test (sinh -1/1234) -0.00081037286017)
      (num-test (sinh 1/1234000000) 0.00000000081037)
      (num-test (sinh -1/1234000000) -0.00000000081037)
      (num-test (sinh 1/500029) 0.00000199988401)
      (num-test (sinh -1/500029) -0.00000199988401)
      (num-test (sinh 1/362880) 0.00000275573192)
      (num-test (sinh -1/362880) -0.00000275573192)
      (num-test (sinh 2/1) 3.62686040784702)
      (num-test (sinh -2/1) -3.62686040784702)
      (num-test (sinh 2/2) 1.17520119364380)
      (num-test (sinh -2/2) -1.17520119364380)
      (num-test (sinh 2/3) 0.71715846101104)
      (num-test (sinh -2/3) -0.71715846101104)
      (num-test (sinh 2/10) 0.20133600254109)
      (num-test (sinh -2/10) -0.20133600254109)
      (num-test (sinh 2/1234) 0.00162074625252)
      (num-test (sinh -2/1234) -0.00162074625252)
      (num-test (sinh 2/1234000000) 0.00000000162075)
      (num-test (sinh -2/1234000000) -0.00000000162075)
      (num-test (sinh 2/500029) 0.00000399976801)
      (num-test (sinh -2/500029) -0.00000399976801)
      (num-test (sinh 2/362880) 0.00000551146384)
      (num-test (sinh -2/362880) -0.00000551146384)
      (num-test (sinh 3/1) 10.01787492740990)
      (num-test (sinh -3/1) -10.01787492740990)
      (num-test (sinh 3/2) 2.12927945509482)
      (num-test (sinh -3/2) -2.12927945509482)
      (num-test (sinh 3/3) 1.17520119364380)
      (num-test (sinh -3/3) -1.17520119364380)
      (num-test (sinh 3/10) 0.30452029344714)
      (num-test (sinh -3/10) -0.30452029344714)
      (num-test (sinh 3/1234) 0.00243112070921)
      (num-test (sinh -3/1234) -0.00243112070921)
      (num-test (sinh 3/1234000000) 0.00000000243112)
      (num-test (sinh -3/1234000000) -0.00000000243112)
      (num-test (sinh 3/500029) 0.00000599965202)
      (num-test (sinh -3/500029) -0.00000599965202)
      (num-test (sinh 3/362880) 0.00000826719577)
      (num-test (sinh -3/362880) -0.00000826719577)
      (num-test (sinh 10/1) 11013.23287470339346)
      (num-test (sinh -10/1) -11013.23287470339346)
      (num-test (sinh 10/2) 74.20321057778875)
      (num-test (sinh -10/2) -74.20321057778875)
      (num-test (sinh 10/3) 13.99797545058944)
      (num-test (sinh -10/3) -13.99797545058944)
      (num-test (sinh 10/10) 1.17520119364380)
      (num-test (sinh -10/10) -1.17520119364380)
      (num-test (sinh 10/1234) 0.00810381641088)
      (num-test (sinh -10/1234) -0.00810381641088)
      (num-test (sinh 10/1234000000) 0.00000000810373)
      (num-test (sinh -10/1234000000) -0.00000000810373)
      (num-test (sinh 10/500029) 0.00001999884007)
      (num-test (sinh -10/500029) -0.00001999884007)
      (num-test (sinh 10/362880) 0.00002755731923)
      (num-test (sinh -10/362880) -0.00002755731923)
      (num-test (sinh 1234/1234) 1.17520119364380)
      (num-test (sinh -1234/1234) -1.17520119364380)
      (num-test (sinh 1234/1234000000) 0.00000100000000)
      (num-test (sinh -1234/1234000000) -0.00000100000000)
      (num-test (sinh 1234/500029) 0.00246785936931)
      (num-test (sinh -1234/500029) -0.00246785936931)
      (num-test (sinh 1234/362880) 0.00340057974622)
      (num-test (sinh -1234/362880) -0.00340057974622)
      (num-test (sinh 1234000000/1234000000) 1.17520119364380)
      (num-test (sinh -1234000000/1234000000) -1.17520119364380)
      (num-test (sinh 500029/1234000000) 0.00040520989764)
      (num-test (sinh -500029/1234000000) -0.00040520989764)
      (num-test (sinh 500029/500029) 1.17520119364380)
      (num-test (sinh -500029/500029) -1.17520119364380)
      (num-test (sinh 500029/362880) 1.85732460755603)
      (num-test (sinh -500029/362880) -1.85732460755603)
      (num-test (sinh 362880/1234000000) 0.00029406807555)
      (num-test (sinh -362880/1234000000) -0.00029406807555)
      (num-test (sinh 362880/500029) 0.79111846340564)
      (num-test (sinh -362880/500029) -0.79111846340564)
      (num-test (sinh 362880/362880) 1.17520119364380)
      (num-test (sinh -362880/362880) -1.17520119364380)
      (num-test (sinh 0.0) 0.0)
      (num-test (sinh 0.00000001) 0.00000001)
      (num-test (sinh -0.00000001) -0.00000001)
      (num-test (sinh 1.0) 1.17520119364380)
      (num-test (sinh -1.0) -1.17520119364380)
      (num-test (sinh our-pi) 11.54873935725775)
      (num-test (sinh -3.14159265358979) -11.54873935725775)
      (num-test (sinh 2.71828182845905) 7.54413710281697)
      (num-test (sinh -2.71828182845905) -7.54413710281697)
      (num-test (sinh 0.0+0.0i) 0.0)
      (num-test (sinh -0.0+0.0i) 0.0)
      (num-test (sinh 0.0-0.0i) 0.0)
      (num-test (sinh -0.0-0.0i) -0.0)
      (num-test (sinh 0.0+0.00000001i) 0.0+0.00000001i)
      (num-test (sinh -0.0+0.00000001i) 0.0+0.00000001i)
      (num-test (sinh 0.0-0.00000001i) 0.0-0.00000001i)
      (num-test (sinh -0.0-0.00000001i) -0.0-0.00000001i)
      (num-test (sinh 0.0+1.0i) 0.0+0.84147098480790i)
      (num-test (sinh -0.0+1.0i) 0.0+0.84147098480790i)
      (num-test (sinh 0.0-1.0i) 0.0-0.84147098480790i)
      (num-test (sinh -0.0-1.0i) -0.0-0.84147098480790i)
      (num-test (sinh 0.00000001+0.0i) 0.00000001)
      (num-test (sinh -0.00000001+0.0i) -0.00000001)
      (num-test (sinh 0.00000001-0.0i) 0.00000001)
      (num-test (sinh -0.00000001-0.0i) -0.00000001)
      (num-test (sinh 0.00000001+0.00000001i) 0.00000001+0.00000001i)
      (num-test (sinh -0.00000001+0.00000001i) -0.00000001+0.00000001i)
      (num-test (sinh 0.00000001-0.00000001i) 0.00000001-0.00000001i)
      (num-test (sinh -0.00000001-0.00000001i) -0.00000001-0.00000001i)
      (num-test (sinh 0.00000001+1.0i) 0.00000000540302+0.84147098480790i)
      (num-test (sinh -0.00000001+1.0i) -0.00000000540302+0.84147098480790i)
      (num-test (sinh 0.00000001-1.0i) 0.00000000540302-0.84147098480790i)
      (num-test (sinh -0.00000001-1.0i) -0.00000000540302-0.84147098480790i)
      (num-test (sinh 1.0+0.0i) 1.17520119364380)
      (num-test (sinh -1.0+0.0i) -1.17520119364380)
      (num-test (sinh 1.0-0.0i) 1.17520119364380)
      (num-test (sinh -1.0-0.0i) -1.17520119364380)
      (num-test (sinh 1.0+0.00000001i) 1.17520119364380+0.00000001543081i)
      (num-test (sinh -1.0+0.00000001i) -1.17520119364380+0.00000001543081i)
      (num-test (sinh 1.0-0.00000001i) 1.17520119364380-0.00000001543081i)
      (num-test (sinh -1.0-0.00000001i) -1.17520119364380-0.00000001543081i)
      (num-test (sinh 1.0+1.0i) 0.63496391478474+1.29845758141598i)
      (num-test (sinh -1.0+1.0i) -0.63496391478474+1.29845758141598i)
      (num-test (sinh 1.0-1.0i) 0.63496391478474-1.29845758141598i)
      (num-test (sinh -1.0-1.0i) -0.63496391478474-1.29845758141598i)
      (num-test (sinh 3.14159265358979+0.0i) 11.54873935725775)
      (num-test (sinh -3.14159265358979+0.0i) -11.54873935725775)
      (num-test (sinh 3.14159265358979-0.0i) 11.54873935725775)
      (num-test (sinh -3.14159265358979-0.0i) -11.54873935725775)
      (num-test (sinh 3.14159265358979+0.00000001i) 11.54873935725775+0.00000011591953i)
      (num-test (sinh -3.14159265358979+0.00000001i) -11.54873935725775+0.00000011591953i)
      (num-test (sinh 3.14159265358979-0.00000001i) 11.54873935725775-0.00000011591953i)
      (num-test (sinh -3.14159265358979-0.00000001i) -11.54873935725775-0.00000011591953i)
      (num-test (sinh 3.14159265358979+1.0i) 6.23981050459650+9.75429233860021i)
      (num-test (sinh -3.14159265358979+1.0i) -6.23981050459650+9.75429233860021i)
      (num-test (sinh 3.14159265358979-1.0i) 6.23981050459650-9.75429233860021i)
      (num-test (sinh -3.14159265358979-1.0i) -6.23981050459650-9.75429233860021i)
      (num-test (sinh 2.71828182845905+0.0i) 7.54413710281697)
      (num-test (sinh -2.71828182845905+0.0i) -7.54413710281697)
      (num-test (sinh 2.71828182845905-0.0i) 7.54413710281697)
      (num-test (sinh -2.71828182845905-0.0i) -7.54413710281697)
      (num-test (sinh 2.71828182845905+0.00000001i) 7.54413710281697+0.00000007610125i)
      (num-test (sinh -2.71828182845905+0.00000001i) -7.54413710281697+0.00000007610125i)
      (num-test (sinh 2.71828182845905-0.00000001i) 7.54413710281697-0.00000007610125i)
      (num-test (sinh -2.71828182845905-0.00000001i) -7.54413710281697-0.00000007610125i)
      (num-test (sinh 2.71828182845905+1.0i) 4.07611467243740+6.40369949494148i)
      (num-test (sinh -2.71828182845905+1.0i) -4.07611467243740+6.40369949494148i)
      (num-test (sinh 2.71828182845905-1.0i) 4.07611467243740-6.40369949494148i)
      (num-test (sinh -2.71828182845905-1.0i) -4.07611467243740-6.40369949494148i)
      (num-test (sinh 1234/3) 2.18155865313939E+178) 
      (num-test (sinh 1234/10) 1.953930316004457E+53) 
      (num-test (sinh 0.0+3.14159265358979i) 0.0-6.982889851335445E-15i) 
      (num-test (sinh 0.0+2.71828182845905i) 0.0+0.4107812905029501i) 
      (num-test (sinh 0.00000001+3.14159265358979i) -1.00000000000000003758922749678992050291E-8+3.231089148865173792463232707134864571569E-15i)
      (num-test (sinh 0.00000001+2.71828182845905i) -9.117339147869465E-9+0.4107812905029501i) 
      (num-test (sinh 0.00000001+1234.0i) -7.985506235875843E-9+0.6019276547624973i) 
      (num-test (sinh 0.00000001+1234000000.0i) 1.5890913089022285E-9-0.9872932128398908i) 
      (num-test (sinh 1.0+3.14159265358979i) -1.175201193643801-1.077516210464362E-14i) 
      (num-test (sinh 1.0+2.71828182845905i) -1.071470784943156+0.6338686545195173i) 
      (num-test (sinh 3.14159265358979+3.14159265358979i) -11.54873935725783-8.094533288479446E-14i) 
      (num-test (sinh 3.14159265358979+2.71828182845905i) -10.52937734504676+4.761757525968664i) 
      (num-test (sinh 3.14159265358979+1234.0i) -9.222253015388718+6.977517249251167i) 
      (num-test (sinh 3.14159265358979+1234000000.0i) 1.835200134139553-11.44465679247962i) 
      (num-test (sinh 2.71828182845905+3.14159265358979i) -7.54413710281663-5.314066559815525E-14i) 
      (num-test (sinh 2.71828182845905+2.71828182845905i) -6.878245654440458+3.126097025348496i) 
      (num-test (sinh 2.71828182845905+1234.0i) -6.024375387884452+4.58074477716391i) 
      (num-test (sinh 2.71828182845905+1234000000.0i) 1.198832270325275-7.513424898263172i) 
      (num-test (sinh 0.0e+00-3.45266983001243932001e-04i) 0-3.4526697614140534807e-4i)
      (num-test (sinh 0.0e+00+3.45266983001243932001e-04i) 0+3.4526697614140534807e-4i)
      (num-test (sinh 0.0e+00+1.57045105981189525579e+00i) 0+9.9999994039535581669e-1i)
      (num-test (sinh 0.0e+00-1.57045105981189525579e+00i) 0-9.9999994039535581669e-1i)
      (num-test (sinh 0.0e+00+1.57114159377789786021e+00i) 0+9.9999994039535581673e-1i)
      (num-test (sinh 0.0e+00-1.57114159377789786021e+00i) 0-9.9999994039535581673e-1i)
      (num-test (sinh 0.0e+00+3.14124738660679181379e+00i) 0+3.4526697614158608860e-4i)
      (num-test (sinh 0.0e+00-3.14124738660679181379e+00i) 0-3.4526697614158608860e-4i)
      (num-test (sinh 0.0e+00+3.14193792057279441821e+00i) 0-3.4526697614134115926e-4i)
      (num-test (sinh 0.0e+00-3.14193792057279441821e+00i) 0+3.4526697614134115926e-4i)
      (num-test (sinh 0.0e+00+4.71204371340168837179e+00i) 0-9.9999994039535581664e-1i)
      (num-test (sinh 0.0e+00-4.71204371340168837179e+00i) 0+9.9999994039535581664e-1i)
      (num-test (sinh 0.0e+00+4.71273424736769097620e+00i) 0-9.9999994039535581677e-1i)
      (num-test (sinh 0.0e+00-4.71273424736769097620e+00i) 0+9.9999994039535581677e-1i)
      (num-test (sinh 0.0e+00+6.28284004019658492979e+00i) 0-3.4526697614170855328e-4i)
      (num-test (sinh 0.0e+00-6.28284004019658492979e+00i) 0+3.4526697614170855328e-4i)
      (num-test (sinh 0.0e+00+6.28353057416258753420e+00i) 0+3.4526697614121869459e-4i)
      (num-test (sinh 0.0e+00-6.28353057416258753420e+00i) 0-3.4526697614121869459e-4i)
      (num-test (sinh 0.0e+00+9.42443269378637893396e+00i) 0+3.4526697614094283958e-4i)
      (num-test (sinh 0.0e+00-9.42443269378637893396e+00i) 0-3.4526697614094283958e-4i)
      (num-test (sinh 0.0e+00+9.42512322775237976202e+00i) 0-3.4526697614020805155e-4i)
      (num-test (sinh 0.0e+00-9.42512322775237976202e+00i) 0+3.4526697614020805155e-4i)
      (num-test (sinh 1.19209289550781250e-07-3.45266983001243932001e-04i) 1.1920928244535424533e-7-3.4526697614140780134e-4i)
      (num-test (sinh 1.19209289550781250e-07+3.45266983001243932001e-04i) 1.1920928244535424533e-7+3.4526697614140780134e-4i)
      (num-test (sinh -1.19209289550781250e-07-3.45266983001243932001e-04i) -1.1920928244535424533e-7-3.4526697614140780134e-4i)
      (num-test (sinh -1.19209289550781250e-07+3.45266983001243932001e-04i) -1.1920928244535424533e-7+3.4526697614140780134e-4i)
      (num-test (sinh 1.19209289550781250e-07+1.57045105981189525579e+00i) 4.1159030931177815679e-11+9.9999994039536292211e-1i)
      (num-test (sinh 1.19209289550781250e-07-1.57045105981189525579e+00i) 4.1159030931177815679e-11-9.9999994039536292211e-1i)
      (num-test (sinh -1.19209289550781250e-07+1.57045105981189525579e+00i) -4.1159030931177815679e-11+9.9999994039536292211e-1i)
      (num-test (sinh -1.19209289550781250e-07-1.57045105981189525579e+00i) -4.1159030931177815679e-11-9.9999994039536292211e-1i)
      (num-test (sinh 1.19209289550781250e-07+1.57114159377789786021e+00i) -4.1159030931163216752e-11+9.9999994039536292216e-1i)
      (num-test (sinh 1.19209289550781250e-07-1.57114159377789786021e+00i) -4.1159030931163216752e-11-9.9999994039536292216e-1i)
      (num-test (sinh -1.19209289550781250e-07+1.57114159377789786021e+00i) 4.1159030931163216752e-11+9.9999994039536292216e-1i)
      (num-test (sinh -1.19209289550781250e-07-1.57114159377789786021e+00i) 4.1159030931163216752e-11-9.9999994039536292216e-1i)
      (num-test (sinh 1.19209289550781250e-07+3.14124738660679181379e+00i) -1.1920928244535424532e-7+3.4526697614158854187e-4i)
      (num-test (sinh 1.19209289550781250e-07-3.14124738660679181379e+00i) -1.1920928244535424532e-7-3.4526697614158854187e-4i)
      (num-test (sinh -1.19209289550781250e-07+3.14124738660679181379e+00i) 1.1920928244535424532e-7+3.4526697614158854187e-4i)
      (num-test (sinh -1.19209289550781250e-07-3.14124738660679181379e+00i) 1.1920928244535424532e-7-3.4526697614158854187e-4i)
      (num-test (sinh 1.19209289550781250e-07+3.14193792057279441821e+00i) -1.1920928244535424533e-7-3.4526697614134361253e-4i)
      (num-test (sinh 1.19209289550781250e-07-3.14193792057279441821e+00i) -1.1920928244535424533e-7+3.4526697614134361253e-4i)
      (num-test (sinh -1.19209289550781250e-07+3.14193792057279441821e+00i) 1.1920928244535424533e-7-3.4526697614134361253e-4i)
      (num-test (sinh -1.19209289550781250e-07-3.14193792057279441821e+00i) 1.1920928244535424533e-7+3.4526697614134361253e-4i)
      (num-test (sinh 1.19209289550781250e-07+4.71204371340168837179e+00i) -4.1159030931192414605e-11-9.9999994039536292207e-1i)
      (num-test (sinh 1.19209289550781250e-07-4.71204371340168837179e+00i) -4.1159030931192414605e-11+9.9999994039536292207e-1i)
      (num-test (sinh -1.19209289550781250e-07+4.71204371340168837179e+00i) 4.1159030931192414605e-11-9.9999994039536292207e-1i)
      (num-test (sinh -1.19209289550781250e-07-4.71204371340168837179e+00i) 4.1159030931192414605e-11+9.9999994039536292207e-1i)
      (num-test (sinh 1.19209289550781250e-07+4.71273424736769097620e+00i) 4.1159030931148617825e-11-9.9999994039536292220e-1i)
      (num-test (sinh 1.19209289550781250e-07-4.71273424736769097620e+00i) 4.1159030931148617825e-11+9.9999994039536292220e-1i)
      (num-test (sinh -1.19209289550781250e-07+4.71273424736769097620e+00i) -4.1159030931148617825e-11-9.9999994039536292220e-1i)
      (num-test (sinh -1.19209289550781250e-07-4.71273424736769097620e+00i) -4.1159030931148617825e-11+9.9999994039536292220e-1i)
      (num-test (sinh 1.19209289550781250e-07+6.28284004019658492979e+00i) 1.1920928244535424532e-7-3.4526697614171100655e-4i)
      (num-test (sinh 1.19209289550781250e-07-6.28284004019658492979e+00i) 1.1920928244535424532e-7+3.4526697614171100655e-4i)
      (num-test (sinh -1.19209289550781250e-07+6.28284004019658492979e+00i) -1.1920928244535424532e-7-3.4526697614171100655e-4i)
      (num-test (sinh -1.19209289550781250e-07-6.28284004019658492979e+00i) -1.1920928244535424532e-7+3.4526697614171100655e-4i)
      (num-test (sinh 1.19209289550781250e-07+6.28353057416258753420e+00i) 1.1920928244535424534e-7+3.4526697614122114786e-4i)
      (num-test (sinh 1.19209289550781250e-07-6.28353057416258753420e+00i) 1.1920928244535424534e-7-3.4526697614122114786e-4i)
      (num-test (sinh -1.19209289550781250e-07+6.28353057416258753420e+00i) -1.1920928244535424534e-7+3.4526697614122114786e-4i)
      (num-test (sinh -1.19209289550781250e-07-6.28353057416258753420e+00i) -1.1920928244535424534e-7-3.4526697614122114786e-4i)
      (num-test (sinh 1.19209289550781250e-07+9.42443269378637893396e+00i) -1.1920928244535424535e-7+3.4526697614094529285e-4i)
      (num-test (sinh 1.19209289550781250e-07-9.42443269378637893396e+00i) -1.1920928244535424535e-7-3.4526697614094529285e-4i)
      (num-test (sinh -1.19209289550781250e-07+9.42443269378637893396e+00i) 1.1920928244535424535e-7+3.4526697614094529285e-4i)
      (num-test (sinh -1.19209289550781250e-07-9.42443269378637893396e+00i) 1.1920928244535424535e-7-3.4526697614094529285e-4i)
      (num-test (sinh 1.19209289550781250e-07+9.42512322775237976202e+00i) -1.1920928244535424538e-7-3.4526697614021050482e-4i)
      (num-test (sinh 1.19209289550781250e-07-9.42512322775237976202e+00i) -1.1920928244535424538e-7+3.4526697614021050482e-4i)
      (num-test (sinh -1.19209289550781250e-07+9.42512322775237976202e+00i) 1.1920928244535424538e-7-3.4526697614021050482e-4i)
      (num-test (sinh -1.19209289550781250e-07-9.42512322775237976202e+00i) 1.1920928244535424538e-7+3.4526697614021050482e-4i)
      (num-test (sinh 5.0e-01-3.45266983001243932001e-04i) 5.2109527443404709209e-1-3.8933200722534065172e-4i)
      (num-test (sinh 5.0e-01+3.45266983001243932001e-04i) 5.2109527443404709209e-1+3.8933200722534065172e-4i)
      (num-test (sinh -5.0e-01-3.45266983001243932001e-04i) -5.2109527443404709209e-1-3.8933200722534065172e-4i)
      (num-test (sinh -5.0e-01+3.45266983001243932001e-04i) -5.2109527443404709209e-1+3.8933200722534065172e-4i)
      (num-test (sinh 5.0e-01+1.57045105981189525579e+00i) 1.7991700040937027667e-4+1.1276258979946363572e0i)
      (num-test (sinh 5.0e-01-1.57045105981189525579e+00i) 1.7991700040937027667e-4-1.1276258979946363572e0i)
      (num-test (sinh -5.0e-01+1.57045105981189525579e+00i) -1.7991700040937027667e-4+1.1276258979946363572e0i)
      (num-test (sinh -5.0e-01-1.57045105981189525579e+00i) -1.7991700040937027667e-4-1.1276258979946363572e0i)
      (num-test (sinh 5.0e-01+1.57114159377789786021e+00i) -1.7991700040930646090e-4+1.1276258979946363573e0i)
      (num-test (sinh 5.0e-01-1.57114159377789786021e+00i) -1.7991700040930646090e-4-1.1276258979946363573e0i)
      (num-test (sinh -5.0e-01+1.57114159377789786021e+00i) 1.7991700040930646090e-4+1.1276258979946363573e0i)
      (num-test (sinh -5.0e-01-1.57114159377789786021e+00i) 1.7991700040930646090e-4-1.1276258979946363573e0i)
      (num-test (sinh 5.0e-01+3.14124738660679181379e+00i) -5.2109527443404709206e-1+3.8933200722554445944e-4i)
      (num-test (sinh 5.0e-01-3.14124738660679181379e+00i) -5.2109527443404709206e-1-3.8933200722554445944e-4i)
      (num-test (sinh -5.0e-01+3.14124738660679181379e+00i) 5.2109527443404709206e-1+3.8933200722554445944e-4i)
      (num-test (sinh -5.0e-01-3.14124738660679181379e+00i) 5.2109527443404709206e-1-3.8933200722554445944e-4i)
      (num-test (sinh 5.0e-01+3.14193792057279441821e+00i) -5.2109527443404709211e-1-3.8933200722526827075e-4i)
      (num-test (sinh 5.0e-01-3.14193792057279441821e+00i) -5.2109527443404709211e-1+3.8933200722526827075e-4i)
      (num-test (sinh -5.0e-01+3.14193792057279441821e+00i) 5.2109527443404709211e-1-3.8933200722526827075e-4i)
      (num-test (sinh -5.0e-01-3.14193792057279441821e+00i) 5.2109527443404709211e-1+3.8933200722526827075e-4i)
      (num-test (sinh 5.0e-01+4.71204371340168837179e+00i) -1.7991700040943409243e-4-1.1276258979946363572e0i)
      (num-test (sinh 5.0e-01-4.71204371340168837179e+00i) -1.7991700040943409243e-4+1.1276258979946363572e0i)
      (num-test (sinh -5.0e-01+4.71204371340168837179e+00i) 1.7991700040943409243e-4-1.1276258979946363572e0i)
      (num-test (sinh -5.0e-01-4.71204371340168837179e+00i) 1.7991700040943409243e-4+1.1276258979946363572e0i)
      (num-test (sinh 5.0e-01+4.71273424736769097620e+00i) 1.7991700040924264514e-4-1.1276258979946363573e0i)
      (num-test (sinh 5.0e-01-4.71273424736769097620e+00i) 1.7991700040924264514e-4+1.1276258979946363573e0i)
      (num-test (sinh -5.0e-01+4.71273424736769097620e+00i) -1.7991700040924264514e-4-1.1276258979946363573e0i)
      (num-test (sinh -5.0e-01-4.71273424736769097620e+00i) -1.7991700040924264514e-4+1.1276258979946363573e0i)
      (num-test (sinh 5.0e-01+6.28284004019658492979e+00i) 5.2109527443404709204e-1-3.8933200722568255379e-4i)
      (num-test (sinh 5.0e-01-6.28284004019658492979e+00i) 5.2109527443404709204e-1+3.8933200722568255379e-4i)
      (num-test (sinh -5.0e-01+6.28284004019658492979e+00i) -5.2109527443404709204e-1-3.8933200722568255379e-4i)
      (num-test (sinh -5.0e-01-6.28284004019658492979e+00i) -5.2109527443404709204e-1+3.8933200722568255379e-4i)
      (num-test (sinh 5.0e-01+6.28353057416258753420e+00i) 5.2109527443404709213e-1+3.8933200722513017641e-4i)
      (num-test (sinh 5.0e-01-6.28353057416258753420e+00i) 5.2109527443404709213e-1-3.8933200722513017641e-4i)
      (num-test (sinh -5.0e-01+6.28353057416258753420e+00i) -5.2109527443404709213e-1+3.8933200722513017641e-4i)
      (num-test (sinh -5.0e-01-6.28353057416258753420e+00i) -5.2109527443404709213e-1-3.8933200722513017641e-4i)
      (num-test (sinh 5.0e-01+9.42443269378637893396e+00i) -5.2109527443404709218e-1+3.8933200722481911514e-4i)
      (num-test (sinh 5.0e-01-9.42443269378637893396e+00i) -5.2109527443404709218e-1-3.8933200722481911514e-4i)
      (num-test (sinh -5.0e-01+9.42443269378637893396e+00i) 5.2109527443404709218e-1+3.8933200722481911514e-4i)
      (num-test (sinh -5.0e-01-9.42443269378637893396e+00i) 5.2109527443404709218e-1-3.8933200722481911514e-4i)
      (num-test (sinh 5.0e-01+9.42512322775237976202e+00i) -5.2109527443404709231e-1-3.8933200722399054908e-4i)
      (num-test (sinh 5.0e-01-9.42512322775237976202e+00i) -5.2109527443404709231e-1+3.8933200722399054908e-4i)
      (num-test (sinh -5.0e-01+9.42512322775237976202e+00i) 5.2109527443404709231e-1-3.8933200722399054908e-4i)
      (num-test (sinh -5.0e-01-9.42512322775237976202e+00i) 5.2109527443404709231e-1+3.8933200722399054908e-4i)
      (num-test (sinh 1.0e+00-3.45266983001243932001e-04i) 1.1752011235963524660e0-5.3277478472501939236e-4i)
      (num-test (sinh 1.0e+00+3.45266983001243932001e-04i) 1.1752011235963524660e0+5.3277478472501939236e-4i)
      (num-test (sinh -1.0e+00-3.45266983001243932001e-04i) -1.1752011235963524660e0-5.3277478472501939236e-4i)
      (num-test (sinh -1.0e+00+3.45266983001243932001e-04i) -1.1752011235963524660e0+5.3277478472501939236e-4i)
      (num-test (sinh 1.0e+00+1.57045105981189525579e+00i) 4.0575816248730593018e-4+1.5430805428404715942e0i)
      (num-test (sinh 1.0e+00-1.57045105981189525579e+00i) 4.0575816248730593018e-4-1.5430805428404715942e0i)
      (num-test (sinh -1.0e+00+1.57045105981189525579e+00i) -4.0575816248730593018e-4+1.5430805428404715942e0i)
      (num-test (sinh -1.0e+00-1.57045105981189525579e+00i) -4.0575816248730593018e-4-1.5430805428404715942e0i)
      (num-test (sinh 1.0e+00+1.57114159377789786021e+00i) -4.0575816248716200955e-4+1.5430805428404715942e0i)
      (num-test (sinh 1.0e+00-1.57114159377789786021e+00i) -4.0575816248716200955e-4-1.5430805428404715942e0i)
      (num-test (sinh -1.0e+00+1.57114159377789786021e+00i) 4.0575816248716200955e-4+1.5430805428404715942e0i)
      (num-test (sinh -1.0e+00-1.57114159377789786021e+00i) 4.0575816248716200955e-4-1.5430805428404715942e0i)
      (num-test (sinh 1.0e+00+3.14124738660679181379e+00i) -1.1752011235963524659e0+5.3277478472529828958e-4i)
      (num-test (sinh 1.0e+00-3.14124738660679181379e+00i) -1.1752011235963524659e0-5.3277478472529828958e-4i)
      (num-test (sinh -1.0e+00+3.14124738660679181379e+00i) 1.1752011235963524659e0+5.3277478472529828958e-4i)
      (num-test (sinh -1.0e+00-3.14124738660679181379e+00i) 1.1752011235963524659e0-5.3277478472529828958e-4i)
      (num-test (sinh 1.0e+00+3.14193792057279441821e+00i) -1.1752011235963524660e0-5.3277478472492034385e-4i)
      (num-test (sinh 1.0e+00-3.14193792057279441821e+00i) -1.1752011235963524660e0+5.3277478472492034385e-4i)
      (num-test (sinh -1.0e+00+3.14193792057279441821e+00i) 1.1752011235963524660e0-5.3277478472492034385e-4i)
      (num-test (sinh -1.0e+00-3.14193792057279441821e+00i) 1.1752011235963524660e0+5.3277478472492034385e-4i)
      (num-test (sinh 1.0e+00+4.71204371340168837179e+00i) -4.0575816248744985081e-4-1.5430805428404715941e0i)
      (num-test (sinh 1.0e+00-4.71204371340168837179e+00i) -4.0575816248744985081e-4+1.5430805428404715941e0i)
      (num-test (sinh -1.0e+00+4.71204371340168837179e+00i) 4.0575816248744985081e-4-1.5430805428404715941e0i)
      (num-test (sinh -1.0e+00-4.71204371340168837179e+00i) 4.0575816248744985081e-4+1.5430805428404715941e0i)
      (num-test (sinh 1.0e+00+4.71273424736769097620e+00i) 4.0575816248701808892e-4-1.5430805428404715943e0i)
      (num-test (sinh 1.0e+00-4.71273424736769097620e+00i) 4.0575816248701808892e-4+1.5430805428404715943e0i)
      (num-test (sinh -1.0e+00+4.71273424736769097620e+00i) -4.0575816248701808892e-4-1.5430805428404715943e0i)
      (num-test (sinh -1.0e+00-4.71273424736769097620e+00i) -4.0575816248701808892e-4+1.5430805428404715943e0i)
      (num-test (sinh 1.0e+00+6.28284004019658492979e+00i) 1.1752011235963524659e0-5.3277478472548726245e-4i)
      (num-test (sinh 1.0e+00-6.28284004019658492979e+00i) 1.1752011235963524659e0+5.3277478472548726245e-4i)
      (num-test (sinh -1.0e+00+6.28284004019658492979e+00i) -1.1752011235963524659e0-5.3277478472548726245e-4i)
      (num-test (sinh -1.0e+00-6.28284004019658492979e+00i) -1.1752011235963524659e0+5.3277478472548726245e-4i)
      (num-test (sinh 1.0e+00+6.28353057416258753420e+00i) 1.1752011235963524661e0+5.3277478472473137099e-4i)
      (num-test (sinh 1.0e+00-6.28353057416258753420e+00i) 1.1752011235963524661e0-5.3277478472473137099e-4i)
      (num-test (sinh -1.0e+00+6.28353057416258753420e+00i) -1.1752011235963524661e0+5.3277478472473137099e-4i)
      (num-test (sinh -1.0e+00-6.28353057416258753420e+00i) -1.1752011235963524661e0-5.3277478472473137099e-4i)
      (num-test (sinh 1.0e+00+9.42443269378637893396e+00i) -1.1752011235963524662e0+5.3277478472430570447e-4i)
      (num-test (sinh 1.0e+00-9.42443269378637893396e+00i) -1.1752011235963524662e0-5.3277478472430570447e-4i)
      (num-test (sinh -1.0e+00+9.42443269378637893396e+00i) 1.1752011235963524662e0+5.3277478472430570447e-4i)
      (num-test (sinh -1.0e+00-9.42443269378637893396e+00i) 1.1752011235963524662e0-5.3277478472430570447e-4i)
      (num-test (sinh 1.0e+00+9.42512322775237976202e+00i) -1.1752011235963524665e0-5.3277478472317186729e-4i)
      (num-test (sinh 1.0e+00-9.42512322775237976202e+00i) -1.1752011235963524665e0+5.3277478472317186729e-4i)
      (num-test (sinh -1.0e+00+9.42512322775237976202e+00i) 1.1752011235963524665e0-5.3277478472317186729e-4i)
      (num-test (sinh -1.0e+00-9.42512322775237976202e+00i) 1.1752011235963524665e0+5.3277478472317186729e-4i)
      (num-test (sinh 2.0e+00-3.45266983001243932001e-04i) 3.6268601916692946556e0-1.2989619299126701883e-3i)
      (num-test (sinh 2.0e+00+3.45266983001243932001e-04i) 3.6268601916692946556e0+1.2989619299126701883e-3i)
      (num-test (sinh -2.0e+00-3.45266983001243932001e-04i) -3.6268601916692946556e0-1.2989619299126701883e-3i)
      (num-test (sinh -2.0e+00+3.45266983001243932001e-04i) -3.6268601916692946556e0+1.2989619299126701883e-3i)
      (num-test (sinh 2.0e+00+1.57045105981189525579e+00i) 1.2522351259047577385e-3+3.7621954668392959445e0i)
      (num-test (sinh 2.0e+00-1.57045105981189525579e+00i) 1.2522351259047577385e-3-3.7621954668392959445e0i)
      (num-test (sinh -2.0e+00+1.57045105981189525579e+00i) -1.2522351259047577385e-3+3.7621954668392959445e0i)
      (num-test (sinh -2.0e+00-1.57045105981189525579e+00i) -1.2522351259047577385e-3-3.7621954668392959445e0i)
      (num-test (sinh 2.0e+00+1.57114159377789786021e+00i) -1.2522351259043135762e-3+3.7621954668392959447e0i)
      (num-test (sinh 2.0e+00-1.57114159377789786021e+00i) -1.2522351259043135762e-3-3.7621954668392959447e0i)
      (num-test (sinh -2.0e+00+1.57114159377789786021e+00i) 1.2522351259043135762e-3+3.7621954668392959447e0i)
      (num-test (sinh -2.0e+00-1.57114159377789786021e+00i) 1.2522351259043135762e-3-3.7621954668392959447e0i)
      (num-test (sinh 2.0e+00+3.14124738660679181379e+00i) -3.6268601916692946553e0+1.2989619299133501696e-3i)
      (num-test (sinh 2.0e+00-3.14124738660679181379e+00i) -3.6268601916692946553e0-1.2989619299133501696e-3i)
      (num-test (sinh -2.0e+00+3.14124738660679181379e+00i) 3.6268601916692946553e0+1.2989619299133501696e-3i)
      (num-test (sinh -2.0e+00-3.14124738660679181379e+00i) 3.6268601916692946553e0-1.2989619299133501696e-3i)
      (num-test (sinh 2.0e+00+3.14193792057279441821e+00i) -3.6268601916692946556e0-1.2989619299124286975e-3i)
      (num-test (sinh 2.0e+00-3.14193792057279441821e+00i) -3.6268601916692946556e0+1.2989619299124286975e-3i)
      (num-test (sinh -2.0e+00+3.14193792057279441821e+00i) 3.6268601916692946556e0-1.2989619299124286975e-3i)
      (num-test (sinh -2.0e+00-3.14193792057279441821e+00i) 3.6268601916692946556e0+1.2989619299124286975e-3i)
      (num-test (sinh 2.0e+00+4.71204371340168837179e+00i) -1.2522351259052019007e-3-3.7621954668392959444e0i)
      (num-test (sinh 2.0e+00-4.71204371340168837179e+00i) -1.2522351259052019007e-3+3.7621954668392959444e0i)
      (num-test (sinh -2.0e+00+4.71204371340168837179e+00i) 1.2522351259052019007e-3-3.7621954668392959444e0i)
      (num-test (sinh -2.0e+00-4.71204371340168837179e+00i) 1.2522351259052019007e-3+3.7621954668392959444e0i)
      (num-test (sinh 2.0e+00+4.71273424736769097620e+00i) 1.2522351259038694139e-3-3.7621954668392959448e0i)
      (num-test (sinh 2.0e+00-4.71273424736769097620e+00i) 1.2522351259038694139e-3+3.7621954668392959448e0i)
      (num-test (sinh -2.0e+00+4.71273424736769097620e+00i) -1.2522351259038694139e-3-3.7621954668392959448e0i)
      (num-test (sinh -2.0e+00-4.71273424736769097620e+00i) -1.2522351259038694139e-3+3.7621954668392959448e0i)
      (num-test (sinh 2.0e+00+6.28284004019658492979e+00i) 3.6268601916692946552e0-1.2989619299138109057e-3i)
      (num-test (sinh 2.0e+00-6.28284004019658492979e+00i) 3.6268601916692946552e0+1.2989619299138109057e-3i)
      (num-test (sinh -2.0e+00+6.28284004019658492979e+00i) -3.6268601916692946552e0-1.2989619299138109057e-3i)
      (num-test (sinh -2.0e+00-6.28284004019658492979e+00i) -3.6268601916692946552e0+1.2989619299138109057e-3i)
      (num-test (sinh 2.0e+00+6.28353057416258753420e+00i) 3.6268601916692946558e0+1.2989619299119679614e-3i)
      (num-test (sinh 2.0e+00-6.28353057416258753420e+00i) 3.6268601916692946558e0-1.2989619299119679614e-3i)
      (num-test (sinh -2.0e+00+6.28353057416258753420e+00i) -3.6268601916692946558e0+1.2989619299119679614e-3i)
      (num-test (sinh -2.0e+00-6.28353057416258753420e+00i) -3.6268601916692946558e0-1.2989619299119679614e-3i)
      (num-test (sinh 2.0e+00+9.42443269378637893396e+00i) -3.6268601916692946561e0+1.2989619299109301409e-3i)
      (num-test (sinh 2.0e+00-9.42443269378637893396e+00i) -3.6268601916692946561e0-1.2989619299109301409e-3i)
      (num-test (sinh -2.0e+00+9.42443269378637893396e+00i) 3.6268601916692946561e0+1.2989619299109301409e-3i)
      (num-test (sinh -2.0e+00-9.42443269378637893396e+00i) 3.6268601916692946561e0-1.2989619299109301409e-3i)
      (num-test (sinh 2.0e+00+9.42512322775237976202e+00i) -3.6268601916692946571e0-1.2989619299081657245e-3i)
      (num-test (sinh 2.0e+00-9.42512322775237976202e+00i) -3.6268601916692946571e0+1.2989619299081657245e-3i)
      (num-test (sinh -2.0e+00+9.42512322775237976202e+00i) 3.6268601916692946571e0-1.2989619299081657245e-3i)
      (num-test (sinh -2.0e+00-9.42512322775237976202e+00i) 3.6268601916692946571e0+1.2989619299081657245e-3i)
      (num-test (sinh 0) 0.0)


      ;; -------- cosh
      (num-test (cosh 0) 1.0)
      (num-test (cosh 1) 1.54308063481524)
      (num-test (cosh 2) 3.76219569108363)
      (num-test (cosh 3) 10.06766199577777)
      (num-test (cosh 10) 11013.23292010332261)
      (num-test (cosh 0/1) 1.0)
      (num-test (cosh 0/2) 1.0)
      (num-test (cosh 0/3) 1.0)
      (num-test (cosh 0/10) 1.0)
      (num-test (cosh 0/1234) 1.0)
      (num-test (cosh 0/1234000000) 1.0)
      (num-test (cosh 0/500029) 1.0)
      (num-test (cosh 0/362880) 1.0)
      (num-test (cosh 1/1) 1.54308063481524)
      (num-test (cosh 1/2) 1.12762596520638)
      (num-test (cosh -1/2) 1.12762596520638)
      (num-test (cosh 1/3) 1.05607186782994)
      (num-test (cosh -1/3) 1.05607186782994)
      (num-test (cosh 1/10) 1.00500416805580)
      (num-test (cosh -1/10) 1.00500416805580)
      (num-test (cosh 1/1234) 1.00000032835203)
      (num-test (cosh -1/1234) 1.00000032835203)
      (num-test (cosh 1/1234000000) 1.0)
      (num-test (cosh -1/1234000000) 1.0)
      (num-test (cosh 1/500029) 1.00000000000200)
      (num-test (cosh -1/500029) 1.00000000000200)
      (num-test (cosh 1/362880) 1.00000000000380)
      (num-test (cosh -1/362880) 1.00000000000380)
      (num-test (cosh 2/1) 3.76219569108363)
      (num-test (cosh 2/2) 1.54308063481524)
      (num-test (cosh 2/3) 1.23057558004363)
      (num-test (cosh 2/10) 1.02006675561908)
      (num-test (cosh -2/10) 1.02006675561908)
      (num-test (cosh 2/1234) 1.00000131340834)
      (num-test (cosh -2/1234) 1.00000131340834)
      (num-test (cosh 2/1234000000) 1.0)
      (num-test (cosh -2/1234000000) 1.0)
      (num-test (cosh 2/500029) 1.00000000000800)
      (num-test (cosh -2/500029) 1.00000000000800)
      (num-test (cosh 2/362880) 1.00000000001519)
      (num-test (cosh -2/362880) 1.00000000001519)
      (num-test (cosh 3/1) 10.06766199577777)
      (num-test (cosh 3/2) 2.35240961524325)
      (num-test (cosh 3/3) 1.54308063481524)
      (num-test (cosh 3/10) 1.04533851412886)
      (num-test (cosh -3/10) 1.04533851412886)
      (num-test (cosh 3/1234) 1.00000295516958)
      (num-test (cosh -3/1234) 1.00000295516958)
      (num-test (cosh 3/1234000000) 1.0)
      (num-test (cosh -3/1234000000) 1.0)
      (num-test (cosh 3/500029) 1.00000000001800)
      (num-test (cosh -3/500029) 1.00000000001800)
      (num-test (cosh 3/362880) 1.00000000003417)
      (num-test (cosh -3/362880) 1.00000000003417)
      (num-test (cosh 10/1) 11013.23292010332261)
      (num-test (cosh 10/2) 74.20994852478785)
      (num-test (cosh 10/3) 14.03364944393670)
      (num-test (cosh 10/10) 1.54308063481524)
      (num-test (cosh 10/1234) 1.00003283538113)
      (num-test (cosh -10/1234) 1.00003283538113)
      (num-test (cosh 10/1234000000) 1.0)
      (num-test (cosh -10/1234000000) 1.0)
      (num-test (cosh 10/500029) 1.00000000019998)
      (num-test (cosh -10/500029) 1.00000000019998)
      (num-test (cosh 10/362880) 1.00000000037970)
      (num-test (cosh -10/362880) 1.00000000037970)
      (num-test (cosh 1234/1234) 1.54308063481524)
      (num-test (cosh 1234/1234000000) 1.00000000000050)
      (num-test (cosh -1234/1234000000) 1.00000000000050)
      (num-test (cosh 1234/500029) 1.00000304516030)
      (num-test (cosh -1234/500029) 1.00000304516030)
      (num-test (cosh 1234/362880) 1.00000578195459)
      (num-test (cosh -1234/362880) 1.00000578195459)
      (num-test (cosh 1234000000/1234000000) 1.54308063481524)
      (num-test (cosh 500029/1234000000) 1.00000008209753)
      (num-test (cosh -500029/1234000000) 1.00000008209753)
      (num-test (cosh 500029/500029) 1.54308063481524)
      (num-test (cosh 500029/362880) 2.10942046492234)
      (num-test (cosh 362880/1234000000) 1.00000004323802)
      (num-test (cosh -362880/1234000000) 1.00000004323802)
      (num-test (cosh 362880/500029) 1.27509545648210)
      (num-test (cosh 362880/362880) 1.54308063481524)
      (num-test (cosh 0.0) 1.0)
      (num-test (cosh 0.00000001) 1.0)
      (num-test (cosh -0.00000001) 1.0)
      (num-test (cosh 1.0) 1.54308063481524)
      (num-test (cosh our-pi) 11.59195327552152)
      (num-test (cosh 2.71828182845905) 7.61012513866229)
      (num-test (cosh 0.0+0.0i) 1.0)
      (num-test (cosh -0.0+0.0i) 1.0)
      (num-test (cosh 0.0-0.0i) 1.0)
      (num-test (cosh -0.0-0.0i) 1.0)
      (num-test (cosh 0.0+0.00000001i) 1.0)
      (num-test (cosh -0.0+0.00000001i) 1.0)
      (num-test (cosh 0.0-0.00000001i) 1.0)
      (num-test (cosh -0.0-0.00000001i) 1.0)
      (num-test (cosh 0.0+1.0i) 0.54030230586814)
      (num-test (cosh -0.0+1.0i) 0.54030230586814)
      (num-test (cosh 0.0-1.0i) 0.54030230586814)
      (num-test (cosh 0.0+3.14159265358979i) -1.0)
      (num-test (cosh -0.0+3.14159265358979i) -1.0)
      (num-test (cosh 0.0-3.14159265358979i) -1.0)
      (num-test (cosh 0.0+2.71828182845905i) -0.91173391478697)
      (num-test (cosh -0.0+2.71828182845905i) -0.91173391478697)
      (num-test (cosh 0.0-2.71828182845905i) -0.91173391478697)
      (num-test (cosh 0.00000001+0.0i) 1.0)
      (num-test (cosh -0.00000001+0.0i) 1.0)
      (num-test (cosh 0.00000001-0.0i) 1.0)
      (num-test (cosh -0.00000001-0.0i) 1.0)
      (num-test (cosh 0.00000001+0.00000001i) 1.0+1e-16i) ; maxima
      (num-test (cosh -0.00000001+0.00000001i) 1.0-1e-16i)
      (num-test (cosh 0.00000001-0.00000001i) 1.0-1e-16i)
      (num-test (cosh -0.00000001-0.00000001i) 1.0+1e-16i)
      (num-test (cosh 0.00000001+1.0i) 0.54030230586814+0.00000000841471i)
      (num-test (cosh 0.00000001-1.0i) 0.54030230586814-0.00000000841471i)
      (num-test (cosh 0.00000001+3.14159265358979i) -1.0-6.982889851335445E-23i) ;maxima -- stopped here
      (num-test (cosh 0.00000001-3.14159265358979i) -1.0-0.0i)
      (num-test (cosh 0.00000001+2.71828182845905i) -0.91173391478697+0.00000000410781i)
      (num-test (cosh 0.00000001-2.71828182845905i) -0.91173391478697-0.00000000410781i)
      (num-test (cosh 1.0+0.0i) 1.54308063481524)
      (num-test (cosh 1.0-0.0i) 1.54308063481524)
      (num-test (cosh 1.0+0.00000001i) 1.54308063481524+0.00000001175201i)
      (num-test (cosh 1.0-0.00000001i) 1.54308063481524-0.00000001175201i)
      (num-test (cosh 1.0+1.0i) 0.83373002513115+0.98889770576287i)
      (num-test (cosh 1.0-1.0i) 0.83373002513115-0.98889770576287i)
      (num-test (cosh 1.0+3.14159265358979i) -1.54308063481524+0.0i)
      (num-test (cosh 1.0-3.14159265358979i) -1.54308063481524-0.0i)
      (num-test (cosh 1.0+2.71828182845905i) -1.40687894801206+0.48275066292556i)
      (num-test (cosh 1.0-2.71828182845905i) -1.40687894801206-0.48275066292556i)
      (num-test (cosh 3.14159265358979+0.0i) 11.59195327552152)
      (num-test (cosh 3.14159265358979-0.0i) 11.59195327552152)
      (num-test (cosh 3.14159265358979+0.00000001i) 11.59195327552152+0.00000011548739i)
      (num-test (cosh 3.14159265358979-0.00000001i) 11.59195327552152-0.00000011548739i)
      (num-test (cosh 3.14159265358979+1.0i) 6.26315908428001+9.71792908024139i)
      (num-test (cosh 3.14159265358979-1.0i) 6.26315908428001-9.71792908024139i)
      (num-test (cosh 3.14159265358979+3.14159265358979i) -11.59195327552152+0.0i)
      (num-test (cosh 3.14159265358979-3.14159265358979i) -11.59195327552152-0.0i)
      (num-test (cosh 3.14159265358979+2.71828182845905i) -10.56877693991882+4.74400605685607i)
      (num-test (cosh 3.14159265358979-2.71828182845905i) -10.56877693991882-4.74400605685607i)
      (num-test (cosh 2.71828182845905+0.0i) 7.61012513866229)
      (num-test (cosh 2.71828182845905-0.0i) 7.61012513866229)
      (num-test (cosh 2.71828182845905+0.00000001i) 7.61012513866229+0.00000007544137i)
      (num-test (cosh 2.71828182845905-0.00000001i) 7.61012513866229-0.00000007544137i)
      (num-test (cosh 2.71828182845905+1.0i) 4.11176816036433+6.34817247743319i)
      (num-test (cosh 2.71828182845905-1.0i) 4.11176816036433-6.34817247743319i)
      (num-test (cosh 2.71828182835905+3.14159265358979i) -7.61012513866229+0.0i)
      (num-test (cosh 2.71828182845905-3.14159265358979i) -7.61012513866229-0.0i)
      (num-test (cosh 2.71828182845905+2.71828182845905i) -6.93840918469126+3.09899037482603i)
      (num-test (cosh 2.71828182845905-2.71828182845905i) -6.93840918469126-3.09899037482603i)
      (num-test (cosh -2/3) 1.230575580043636) 
      (num-test (cosh -3/2) 2.352409615243247) 
      (num-test (cosh -10/3) 14.03364944393623) 
      (num-test (cosh 1234/3) 2.18155865313939E+178) 
      (num-test (cosh 1234/10) 1.953930316004457E+53) 
      (num-test (cosh 500029/1234) 4.77955809407816E+175) 
      (num-test (cosh -500029/362880) 2.109420464922257) 
      (num-test (cosh 362880/1234) 2.57697781296564E+127) 
      (num-test (cosh -362880/500029) 1.275095456482107) 
      (num-test (cosh -3.14159265358979) 11.5919532755216) 
      (num-test (cosh -2.71828182845905) 7.610125138661946) 
      (num-test (cosh 0.0+3.14159265358979i) -1.0) 
      (num-test (cosh 0.0+2.71828182845905i) -0.9117339147869464) 
      (num-test (cosh 0.00000001+1.0i) +0.5403023058681398+8.414709848078964E-9i) 
      (num-test (cosh 0.00000001+3.14159265358979i) -1.0-6.982889851335445E-23i) 
      (num-test (cosh 0.00000001+2.71828182845905i) -0.9117339147869464+4.1078129050295015E-9i) 
      (num-test (cosh 0.00000001+1234.0i) -0.7985506235875843+6.019276547624973E-9i) 
      (num-test (cosh 0.00000001+1234000000.0i) .1589091308902228-9.872932128398908E-9i) 
      (num-test (cosh 1.0+0.00000001i) +1.543080634815244+1.1752011936438014E-8i) 
      (num-test (cosh 1.0+3.14159265358979i) -1.543080634815244-8.206300488372603E-15i) 
      (num-test (cosh 1.0+2.71828182845905i) -1.406878948012029+0.4827506629256081i) 
      (num-test (cosh 3.14159265358979+0.0i) 11.5919532755216) 
      (num-test (cosh 3.14159265358979+0.00000001i) +11.5919532755216+1.154873935725783E-7i) 
      (num-test (cosh 3.14159265358979+1.0i) +6.263159084280057+9.71792908024146i) 
      (num-test (cosh 3.14159265358979+3.14159265358979i) -11.5919532755216-8.064357485351393E-14i) 
      (num-test (cosh 3.14159265358979+2.71828182845905i) -10.56877693991868+4.744006056856582i) 
      (num-test (cosh 3.14159265358979+1234.0i) -9.256761516765916+6.951505596777556i) 
      (num-test (cosh 3.14159265358979+1234000000.0i) 1.84206722033321-11.40199198427758i) 
      (num-test (cosh 2.71828182845905+0.0i) 7.610125138661946) 
      (num-test (cosh 2.71828182845905+0.00000001i) +7.610125138661945+7.54413710281663E-8i) 
      (num-test (cosh 2.71828182845905+1.0i) +4.111768160364146+6.348172477432901i) 
      (num-test (cosh 2.71828182845905+3.14159265358979i) -7.610125138661946-5.267987841234144E-14i) 
      (num-test (cosh 2.71828182845905+2.71828182845905i) -6.93840918469081+3.098990374826203i) 
      (num-test (cosh 2.71828182845905+1234.0i) -6.077070175058048+4.541024753505155i) 
      (num-test (cosh 2.71828182845905+1234000000.0i) 1.209318371750606-7.448275358344457i) 
      (num-test (cosh 0.0e+00-3.45266983001243932001e-04i) 9.9999994039535581673e-1+0.0i)
      (num-test (cosh 0.0e+00+3.45266983001243932001e-04i) 9.9999994039535581673e-1+0.0i)
      (num-test (cosh 0.0e+00+1.57045105981189525579e+00i) 3.4526697614152485627e-4+0.0i)
      (num-test (cosh 0.0e+00-1.57045105981189525579e+00i) 3.4526697614152485627e-4+0.0i)
      (num-test (cosh 0.0e+00+1.57114159377789786021e+00i) -3.4526697614140239160e-4+0.0i)
      (num-test (cosh 0.0e+00-1.57114159377789786021e+00i) -3.4526697614140239160e-4+0.0i)
      (num-test (cosh 0.0e+00+3.14124738660679181379e+00i) -9.9999994039535581667e-1+0.0i)
      (num-test (cosh 0.0e+00-3.14124738660679181379e+00i) -9.9999994039535581667e-1+0.0i)
      (num-test (cosh 0.0e+00+3.14193792057279441821e+00i) -9.9999994039535581675e-1+0.0i)
      (num-test (cosh 0.0e+00-3.14193792057279441821e+00i) -9.9999994039535581675e-1+0.0i)
      (num-test (cosh 0.0e+00+4.71204371340168837179e+00i) -3.4526697614164732094e-4+0.0i)
      (num-test (cosh 0.0e+00-4.71204371340168837179e+00i) -3.4526697614164732094e-4+0.0i)
      (num-test (cosh 0.0e+00+4.71273424736769097620e+00i) 3.4526697614127992692e-4+0.0i)
      (num-test (cosh 0.0e+00-4.71273424736769097620e+00i) 3.4526697614127992692e-4+0.0i)
      (num-test (cosh 0.0e+00+6.28284004019658492979e+00i) 9.9999994039535581662e-1+0.0i)
      (num-test (cosh 0.0e+00-6.28284004019658492979e+00i) 9.9999994039535581662e-1+0.0i)
      (num-test (cosh 0.0e+00+6.28353057416258753420e+00i) 9.9999994039535581679e-1+0.0i)
      (num-test (cosh 0.0e+00-6.28353057416258753420e+00i) 9.9999994039535581679e-1+0.0i)
      (num-test (cosh 0.0e+00+9.42443269378637893396e+00i) -9.9999994039535581689e-1+0.0i)
      (num-test (cosh 0.0e+00-9.42443269378637893396e+00i) -9.9999994039535581689e-1+0.0i)
      (num-test (cosh 0.0e+00+9.42512322775237976202e+00i) -9.9999994039535581714e-1+0.0i)
      (num-test (cosh 0.0e+00-9.42512322775237976202e+00i) -9.9999994039535581714e-1+0.0i)
      (num-test (cosh 1.19209289550781250e-07-3.45266983001243932001e-04i) 9.9999994039536292216e-1-4.1159030931163569191e-11i)
      (num-test (cosh 1.19209289550781250e-07+3.45266983001243932001e-04i) 9.9999994039536292216e-1+4.1159030931163569191e-11i)
      (num-test (cosh -1.19209289550781250e-07-3.45266983001243932001e-04i) 9.9999994039536292216e-1+4.1159030931163569191e-11i)
      (num-test (cosh -1.19209289550781250e-07+3.45266983001243932001e-04i) 9.9999994039536292216e-1-4.1159030931163569191e-11i)
      (num-test (cosh 1.19209289550781250e-07+1.57045105981189525579e+00i) 3.4526697614152730954e-4+1.1920928244535424532e-7i)
      (num-test (cosh 1.19209289550781250e-07-1.57045105981189525579e+00i) 3.4526697614152730954e-4-1.1920928244535424532e-7i)
      (num-test (cosh -1.19209289550781250e-07+1.57045105981189525579e+00i) 3.4526697614152730954e-4-1.1920928244535424532e-7i)
      (num-test (cosh -1.19209289550781250e-07-1.57045105981189525579e+00i) 3.4526697614152730954e-4+1.1920928244535424532e-7i)
      (num-test (cosh 1.19209289550781250e-07+1.57114159377789786021e+00i) -3.4526697614140484486e-4+1.1920928244535424533e-7i)
      (num-test (cosh 1.19209289550781250e-07-1.57114159377789786021e+00i) -3.4526697614140484486e-4-1.1920928244535424533e-7i)
      (num-test (cosh -1.19209289550781250e-07+1.57114159377789786021e+00i) -3.4526697614140484486e-4-1.1920928244535424533e-7i)
      (num-test (cosh -1.19209289550781250e-07-1.57114159377789786021e+00i) -3.4526697614140484486e-4+1.1920928244535424533e-7i)
      (num-test (cosh 1.19209289550781250e-07+3.14124738660679181379e+00i) -9.9999994039536292209e-1+4.1159030931185115142e-11i)
      (num-test (cosh 1.19209289550781250e-07-3.14124738660679181379e+00i) -9.9999994039536292209e-1-4.1159030931185115142e-11i)
      (num-test (cosh -1.19209289550781250e-07+3.14124738660679181379e+00i) -9.9999994039536292209e-1-4.1159030931185115142e-11i)
      (num-test (cosh -1.19209289550781250e-07-3.14124738660679181379e+00i) -9.9999994039536292209e-1+4.1159030931185115142e-11i)
      (num-test (cosh 1.19209289550781250e-07+3.14193792057279441821e+00i) -9.9999994039536292218e-1-4.1159030931155917289e-11i)
      (num-test (cosh 1.19209289550781250e-07-3.14193792057279441821e+00i) -9.9999994039536292218e-1+4.1159030931155917289e-11i)
      (num-test (cosh -1.19209289550781250e-07+3.14193792057279441821e+00i) -9.9999994039536292218e-1+4.1159030931155917289e-11i)
      (num-test (cosh -1.19209289550781250e-07-3.14193792057279441821e+00i) -9.9999994039536292218e-1-4.1159030931155917289e-11i)
      (num-test (cosh 1.19209289550781250e-07+4.71204371340168837179e+00i) -3.4526697614164977421e-4-1.1920928244535424532e-7i)
      (num-test (cosh 1.19209289550781250e-07-4.71204371340168837179e+00i) -3.4526697614164977421e-4+1.1920928244535424532e-7i)
      (num-test (cosh -1.19209289550781250e-07+4.71204371340168837179e+00i) -3.4526697614164977421e-4+1.1920928244535424532e-7i)
      (num-test (cosh -1.19209289550781250e-07-4.71204371340168837179e+00i) -3.4526697614164977421e-4-1.1920928244535424532e-7i)
      (num-test (cosh 1.19209289550781250e-07+4.71273424736769097620e+00i) 3.4526697614128238019e-4-1.1920928244535424533e-7i)
      (num-test (cosh 1.19209289550781250e-07-4.71273424736769097620e+00i) 3.4526697614128238019e-4+1.1920928244535424533e-7i)
      (num-test (cosh -1.19209289550781250e-07+4.71273424736769097620e+00i) 3.4526697614128238019e-4+1.1920928244535424533e-7i)
      (num-test (cosh -1.19209289550781250e-07-4.71273424736769097620e+00i) 3.4526697614128238019e-4-1.1920928244535424533e-7i)
      (num-test (cosh 1.19209289550781250e-07+6.28284004019658492979e+00i) 9.9999994039536292205e-1-4.1159030931199714069e-11i)
      (num-test (cosh 1.19209289550781250e-07-6.28284004019658492979e+00i) 9.9999994039536292205e-1+4.1159030931199714069e-11i)
      (num-test (cosh -1.19209289550781250e-07+6.28284004019658492979e+00i) 9.9999994039536292205e-1+4.1159030931199714069e-11i)
      (num-test (cosh -1.19209289550781250e-07-6.28284004019658492979e+00i) 9.9999994039536292205e-1-4.1159030931199714069e-11i)
      (num-test (cosh 1.19209289550781250e-07+6.28353057416258753420e+00i) 9.9999994039536292222e-1+4.1159030931141318362e-11i)
      (num-test (cosh 1.19209289550781250e-07-6.28353057416258753420e+00i) 9.9999994039536292222e-1-4.1159030931141318362e-11i)
      (num-test (cosh -1.19209289550781250e-07+6.28353057416258753420e+00i) 9.9999994039536292222e-1-4.1159030931141318362e-11i)
      (num-test (cosh -1.19209289550781250e-07-6.28353057416258753420e+00i) 9.9999994039536292222e-1+4.1159030931141318362e-11i)
      (num-test (cosh 1.19209289550781250e-07+9.42443269378637893396e+00i) -9.9999994039536292231e-1+4.1159030931108433883e-11i)
      (num-test (cosh 1.19209289550781250e-07-9.42443269378637893396e+00i) -9.9999994039536292231e-1-4.1159030931108433883e-11i)
      (num-test (cosh -1.19209289550781250e-07+9.42443269378637893396e+00i) -9.9999994039536292231e-1-4.1159030931108433883e-11i)
      (num-test (cosh -1.19209289550781250e-07-9.42443269378637893396e+00i) -9.9999994039536292231e-1+4.1159030931108433883e-11i)
      (num-test (cosh 1.19209289550781250e-07+9.42512322775237976202e+00i) -9.9999994039536292257e-1-4.1159030931020840323e-11i)
      (num-test (cosh 1.19209289550781250e-07-9.42512322775237976202e+00i) -9.9999994039536292257e-1+4.1159030931020840323e-11i)
      (num-test (cosh -1.19209289550781250e-07+9.42512322775237976202e+00i) -9.9999994039536292257e-1+4.1159030931020840323e-11i)
      (num-test (cosh -1.19209289550781250e-07-9.42512322775237976202e+00i) -9.9999994039536292257e-1-4.1159030931020840323e-11i)
      (num-test (cosh 5.0e-01-3.45266983001243932001e-04i) 1.1276258979946363573e0-1.7991700040930800151e-4i)
      (num-test (cosh 5.0e-01+3.45266983001243932001e-04i) 1.1276258979946363573e0+1.7991700040930800151e-4i)
      (num-test (cosh -5.0e-01-3.45266983001243932001e-04i) 1.1276258979946363573e0+1.7991700040930800151e-4i)
      (num-test (cosh -5.0e-01+3.45266983001243932001e-04i) 1.1276258979946363573e0-1.7991700040930800151e-4i)
      (num-test (cosh 5.0e-01+1.57045105981189525579e+00i) 3.8933200722547541227e-4+5.2109527443404709207e-1i)
      (num-test (cosh 5.0e-01-1.57045105981189525579e+00i) 3.8933200722547541227e-4-5.2109527443404709207e-1i)
      (num-test (cosh -5.0e-01+1.57045105981189525579e+00i) 3.8933200722547541227e-4-5.2109527443404709207e-1i)
      (num-test (cosh -5.0e-01-1.57045105981189525579e+00i) 3.8933200722547541227e-4+5.2109527443404709207e-1i)
      (num-test (cosh 5.0e-01+1.57114159377789786021e+00i) -3.8933200722533731792e-4+5.2109527443404709209e-1i)
      (num-test (cosh 5.0e-01-1.57114159377789786021e+00i) -3.8933200722533731792e-4-5.2109527443404709209e-1i)
      (num-test (cosh -5.0e-01+1.57114159377789786021e+00i) -3.8933200722533731792e-4-5.2109527443404709209e-1i)
      (num-test (cosh -5.0e-01-1.57114159377789786021e+00i) -3.8933200722533731792e-4+5.2109527443404709209e-1i)
      (num-test (cosh 5.0e-01+3.14124738660679181379e+00i) -1.1276258979946363572e0+1.7991700040940218455e-4i)
      (num-test (cosh 5.0e-01-3.14124738660679181379e+00i) -1.1276258979946363572e0-1.7991700040940218455e-4i)
      (num-test (cosh -5.0e-01+3.14124738660679181379e+00i) -1.1276258979946363572e0-1.7991700040940218455e-4i)
      (num-test (cosh -5.0e-01-3.14124738660679181379e+00i) -1.1276258979946363572e0+1.7991700040940218455e-4i)
      (num-test (cosh 5.0e-01+3.14193792057279441821e+00i) -1.1276258979946363573e0-1.7991700040927455302e-4i)
      (num-test (cosh 5.0e-01-3.14193792057279441821e+00i) -1.1276258979946363573e0+1.7991700040927455302e-4i)
      (num-test (cosh -5.0e-01+3.14193792057279441821e+00i) -1.1276258979946363573e0+1.7991700040927455302e-4i)
      (num-test (cosh -5.0e-01-3.14193792057279441821e+00i) -1.1276258979946363573e0-1.7991700040927455302e-4i)
      (num-test (cosh 5.0e-01+4.71204371340168837179e+00i) -3.8933200722561350661e-4-5.2109527443404709205e-1i)
      (num-test (cosh 5.0e-01-4.71204371340168837179e+00i) -3.8933200722561350661e-4+5.2109527443404709205e-1i)
      (num-test (cosh -5.0e-01+4.71204371340168837179e+00i) -3.8933200722561350661e-4+5.2109527443404709205e-1i)
      (num-test (cosh -5.0e-01-4.71204371340168837179e+00i) -3.8933200722561350661e-4-5.2109527443404709205e-1i)
      (num-test (cosh 5.0e-01+4.71273424736769097620e+00i) 3.8933200722519922358e-4-5.2109527443404709212e-1i)
      (num-test (cosh 5.0e-01-4.71273424736769097620e+00i) 3.8933200722519922358e-4+5.2109527443404709212e-1i)
      (num-test (cosh -5.0e-01+4.71273424736769097620e+00i) 3.8933200722519922358e-4+5.2109527443404709212e-1i)
      (num-test (cosh -5.0e-01-4.71273424736769097620e+00i) 3.8933200722519922358e-4-5.2109527443404709212e-1i)
      (num-test (cosh 5.0e-01+6.28284004019658492979e+00i) 1.1276258979946363572e0-1.7991700040946600032e-4i)
      (num-test (cosh 5.0e-01-6.28284004019658492979e+00i) 1.1276258979946363572e0+1.7991700040946600032e-4i)
      (num-test (cosh -5.0e-01+6.28284004019658492979e+00i) 1.1276258979946363572e0+1.7991700040946600032e-4i)
      (num-test (cosh -5.0e-01-6.28284004019658492979e+00i) 1.1276258979946363572e0-1.7991700040946600032e-4i)
      (num-test (cosh 5.0e-01+6.28353057416258753420e+00i) 1.1276258979946363574e0+1.7991700040921073725e-4i)
      (num-test (cosh 5.0e-01-6.28353057416258753420e+00i) 1.1276258979946363574e0-1.7991700040921073725e-4i)
      (num-test (cosh -5.0e-01+6.28353057416258753420e+00i) 1.1276258979946363574e0-1.7991700040921073725e-4i)
      (num-test (cosh -5.0e-01-6.28353057416258753420e+00i) 1.1276258979946363574e0+1.7991700040921073725e-4i)
      (num-test (cosh 5.0e-01+9.42443269378637893396e+00i) -1.1276258979946363575e0+1.7991700040906699050e-4i)
      (num-test (cosh 5.0e-01-9.42443269378637893396e+00i) -1.1276258979946363575e0-1.7991700040906699050e-4i)
      (num-test (cosh -5.0e-01+9.42443269378637893396e+00i) -1.1276258979946363575e0-1.7991700040906699050e-4i)
      (num-test (cosh -5.0e-01-9.42443269378637893396e+00i) -1.1276258979946363575e0+1.7991700040906699050e-4i)
      (num-test (cosh 5.0e-01+9.42512322775237976202e+00i) -1.1276258979946363577e0-1.7991700040868409591e-4i)
      (num-test (cosh 5.0e-01-9.42512322775237976202e+00i) -1.1276258979946363577e0+1.7991700040868409591e-4i)
      (num-test (cosh -5.0e-01+9.42512322775237976202e+00i) -1.1276258979946363577e0+1.7991700040868409591e-4i)
      (num-test (cosh -5.0e-01-9.42512322775237976202e+00i) -1.1276258979946363577e0-1.7991700040868409591e-4i)
      (num-test (cosh 1.0e+00-3.45266983001243932001e-04i) 1.5430805428404715942e0-4.057581624871654840e-4i)
      (num-test (cosh 1.0e+00+3.45266983001243932001e-04i) 1.5430805428404715942e0+4.057581624871654840e-4i)
      (num-test (cosh -1.0e+00-3.45266983001243932001e-04i) 1.5430805428404715942e0+4.057581624871654840e-4i)
      (num-test (cosh -1.0e+00+3.45266983001243932001e-04i) 1.5430805428404715942e0-4.057581624871654840e-4i)
      (num-test (cosh 1.0e+00+1.57045105981189525579e+00i) 5.3277478472520380315e-4+1.1752011235963524659e0i)
      (num-test (cosh 1.0e+00-1.57045105981189525579e+00i) 5.3277478472520380315e-4-1.1752011235963524659e0i)
      (num-test (cosh -1.0e+00+1.57045105981189525579e+00i) 5.3277478472520380315e-4-1.1752011235963524659e0i)
      (num-test (cosh -1.0e+00-1.57045105981189525579e+00i) 5.3277478472520380315e-4+1.1752011235963524659e0i)
      (num-test (cosh 1.0e+00+1.57114159377789786021e+00i) -5.3277478472501483029e-4+1.1752011235963524660e0i)
      (num-test (cosh 1.0e+00-1.57114159377789786021e+00i) -5.3277478472501483029e-4-1.1752011235963524660e0i)
      (num-test (cosh -1.0e+00+1.57114159377789786021e+00i) -5.3277478472501483029e-4-1.1752011235963524660e0i)
      (num-test (cosh -1.0e+00-1.57114159377789786021e+00i) -5.3277478472501483029e-4+1.1752011235963524660e0i)
      (num-test (cosh 1.0e+00+3.14124738660679181379e+00i) -1.5430805428404715941e0+4.0575816248737789049e-4i)
      (num-test (cosh 1.0e+00-3.14124738660679181379e+00i) -1.5430805428404715941e0-4.0575816248737789049e-4i)
      (num-test (cosh -1.0e+00+3.14124738660679181379e+00i) -1.5430805428404715941e0-4.0575816248737789049e-4i)
      (num-test (cosh -1.0e+00-3.14124738660679181379e+00i) -1.5430805428404715941e0+4.0575816248737789049e-4i)
      (num-test (cosh 1.0e+00+3.14193792057279441821e+00i) -1.5430805428404715943e0-4.0575816248709004923e-4i)
      (num-test (cosh 1.0e+00-3.14193792057279441821e+00i) -1.5430805428404715943e0+4.0575816248709004923e-4i)
      (num-test (cosh -1.0e+00+3.14193792057279441821e+00i) -1.5430805428404715943e0+4.0575816248709004923e-4i)
      (num-test (cosh -1.0e+00-3.14193792057279441821e+00i) -1.5430805428404715943e0-4.0575816248709004923e-4i)
      (num-test (cosh 1.0e+00+4.71204371340168837179e+00i) -5.3277478472539277601e-4-1.1752011235963524659e0i)
      (num-test (cosh 1.0e+00-4.71204371340168837179e+00i) -5.3277478472539277601e-4+1.1752011235963524659e0i)
      (num-test (cosh -1.0e+00+4.71204371340168837179e+00i) -5.3277478472539277601e-4+1.1752011235963524659e0i)
      (num-test (cosh -1.0e+00-4.71204371340168837179e+00i) -5.3277478472539277601e-4-1.1752011235963524659e0i)
      (num-test (cosh 1.0e+00+4.71273424736769097620e+00i) 5.3277478472482585742e-4-1.1752011235963524660e0i)
      (num-test (cosh 1.0e+00-4.71273424736769097620e+00i) 5.3277478472482585742e-4+1.1752011235963524660e0i)
      (num-test (cosh -1.0e+00+4.71273424736769097620e+00i) 5.3277478472482585742e-4+1.1752011235963524660e0i)
      (num-test (cosh -1.0e+00-4.71273424736769097620e+00i) 5.3277478472482585742e-4-1.1752011235963524660e0i)
      (num-test (cosh 1.0e+00+6.28284004019658492979e+00i) 1.5430805428404715941e0-4.0575816248752181112e-4i)
      (num-test (cosh 1.0e+00-6.28284004019658492979e+00i) 1.5430805428404715941e0+4.0575816248752181112e-4i)
      (num-test (cosh -1.0e+00+6.28284004019658492979e+00i) 1.5430805428404715941e0+4.0575816248752181112e-4i)
      (num-test (cosh -1.0e+00-6.28284004019658492979e+00i) 1.5430805428404715941e0-4.0575816248752181112e-4i)
      (num-test (cosh 1.0e+00+6.28353057416258753420e+00i) 1.5430805428404715943e0+4.0575816248694612861e-4i)
      (num-test (cosh 1.0e+00-6.28353057416258753420e+00i) 1.5430805428404715943e0-4.0575816248694612861e-4i)
      (num-test (cosh -1.0e+00+6.28353057416258753420e+00i) 1.5430805428404715943e0-4.0575816248694612861e-4i)
      (num-test (cosh -1.0e+00-6.28353057416258753420e+00i) 1.5430805428404715943e0+4.0575816248694612861e-4i)
      (num-test (cosh 1.0e+00+9.42443269378637893396e+00i) -1.5430805428404715945e0+4.0575816248662194348e-4i)
      (num-test (cosh 1.0e+00-9.42443269378637893396e+00i) -1.5430805428404715945e0-4.0575816248662194348e-4i)
      (num-test (cosh -1.0e+00+9.42443269378637893396e+00i) -1.5430805428404715945e0-4.0575816248662194348e-4i)
      (num-test (cosh -1.0e+00-9.42443269378637893396e+00i) -1.5430805428404715945e0+4.0575816248662194348e-4i)
      (num-test (cosh 1.0e+00+9.42512322775237976202e+00i) -1.5430805428404715949e0-4.0575816248575841970e-4i)
      (num-test (cosh 1.0e+00-9.42512322775237976202e+00i) -1.5430805428404715949e0+4.0575816248575841970e-4i)
      (num-test (cosh -1.0e+00+9.42512322775237976202e+00i) -1.5430805428404715949e0+4.0575816248575841970e-4i)
      (num-test (cosh -1.0e+00-9.42512322775237976202e+00i) -1.5430805428404715949e0-4.0575816248575841970e-4i)
      (num-test (cosh 2.0e+00-3.45266983001243932001e-04i) 3.7621954668392959447e0-1.2522351259043242989e-3i)
      (num-test (cosh 2.0e+00+3.45266983001243932001e-04i) 3.7621954668392959447e0+1.2522351259043242989e-3i)
      (num-test (cosh -2.0e+00-3.45266983001243932001e-04i) 3.7621954668392959447e0+1.2522351259043242989e-3i)
      (num-test (cosh -2.0e+00+3.45266983001243932001e-04i) 3.7621954668392959447e0-1.2522351259043242989e-3i)
      (num-test (cosh 2.0e+00+1.57045105981189525579e+00i) 1.2989619299131198016e-3+3.6268601916692946554e0i)
      (num-test (cosh 2.0e+00-1.57045105981189525579e+00i) 1.2989619299131198016e-3-3.6268601916692946554e0i)
      (num-test (cosh -2.0e+00+1.57045105981189525579e+00i) 1.2989619299131198016e-3-3.6268601916692946554e0i)
      (num-test (cosh -2.0e+00-1.57045105981189525579e+00i) 1.2989619299131198016e-3+3.6268601916692946554e0i)
      (num-test (cosh 2.0e+00+1.57114159377789786021e+00i) -1.2989619299126590655e-3+3.6268601916692946556e0i)
      (num-test (cosh 2.0e+00-1.57114159377789786021e+00i) -1.2989619299126590655e-3-3.6268601916692946556e0i)
      (num-test (cosh -2.0e+00+1.57114159377789786021e+00i) -1.2989619299126590655e-3-3.6268601916692946556e0i)
      (num-test (cosh -2.0e+00-1.57114159377789786021e+00i) -1.2989619299126590655e-3+3.6268601916692946556e0i)
      (num-test (cosh 2.0e+00+3.14124738660679181379e+00i) -3.7621954668392959444e0+1.2522351259049798196e-3i)
      (num-test (cosh 2.0e+00-3.14124738660679181379e+00i) -3.7621954668392959444e0-1.2522351259049798196e-3i)
      (num-test (cosh -2.0e+00+3.14124738660679181379e+00i) -3.7621954668392959444e0-1.2522351259049798196e-3i)
      (num-test (cosh -2.0e+00-3.14124738660679181379e+00i) -3.7621954668392959444e0+1.2522351259049798196e-3i)
      (num-test (cosh 2.0e+00+3.14193792057279441821e+00i) -3.7621954668392959448e0-1.2522351259040914950e-3i)
      (num-test (cosh 2.0e+00-3.14193792057279441821e+00i) -3.7621954668392959448e0+1.2522351259040914950e-3i)
      (num-test (cosh -2.0e+00+3.14193792057279441821e+00i) -3.7621954668392959448e0+1.2522351259040914950e-3i)
      (num-test (cosh -2.0e+00-3.14193792057279441821e+00i) -3.7621954668392959448e0-1.2522351259040914950e-3i)
      (num-test (cosh 2.0e+00+4.71204371340168837179e+00i) -1.2989619299135805376e-3-3.6268601916692946552e0i)
      (num-test (cosh 2.0e+00-4.71204371340168837179e+00i) -1.2989619299135805376e-3+3.6268601916692946552e0i)
      (num-test (cosh -2.0e+00+4.71204371340168837179e+00i) -1.2989619299135805376e-3+3.6268601916692946552e0i)
      (num-test (cosh -2.0e+00-4.71204371340168837179e+00i) -1.2989619299135805376e-3-3.6268601916692946552e0i)
      (num-test (cosh 2.0e+00+4.71273424736769097620e+00i) 1.2989619299121983294e-3-3.6268601916692946557e0i)
      (num-test (cosh 2.0e+00-4.71273424736769097620e+00i) 1.2989619299121983294e-3+3.6268601916692946557e0i)
      (num-test (cosh -2.0e+00+4.71273424736769097620e+00i) 1.2989619299121983294e-3+3.6268601916692946557e0i)
      (num-test (cosh -2.0e+00-4.71273424736769097620e+00i) 1.2989619299121983294e-3-3.6268601916692946557e0i)
      (num-test (cosh 2.0e+00+6.28284004019658492979e+00i) 3.7621954668392959443e0-1.2522351259054239819e-3i)
      (num-test (cosh 2.0e+00-6.28284004019658492979e+00i) 3.7621954668392959443e0+1.2522351259054239819e-3i)
      (num-test (cosh -2.0e+00+6.28284004019658492979e+00i) 3.7621954668392959443e0+1.2522351259054239819e-3i)
      (num-test (cosh -2.0e+00-6.28284004019658492979e+00i) 3.7621954668392959443e0-1.2522351259054239819e-3i)
      (num-test (cosh 2.0e+00+6.28353057416258753420e+00i) 3.7621954668392959449e0+1.2522351259036473328e-3i)
      (num-test (cosh 2.0e+00-6.28353057416258753420e+00i) 3.7621954668392959449e0-1.2522351259036473328e-3i)
      (num-test (cosh -2.0e+00+6.28353057416258753420e+00i) 3.7621954668392959449e0-1.2522351259036473328e-3i)
      (num-test (cosh -2.0e+00-6.28353057416258753420e+00i) 3.7621954668392959449e0+1.2522351259036473328e-3i)
      (num-test (cosh 2.0e+00+9.42443269378637893396e+00i) -3.7621954668392959453e0+1.2522351259026468452e-3i)
      (num-test (cosh 2.0e+00-9.42443269378637893396e+00i) -3.7621954668392959453e0-1.2522351259026468452e-3i)
      (num-test (cosh -2.0e+00+9.42443269378637893396e+00i) -3.7621954668392959453e0-1.2522351259026468452e-3i)
      (num-test (cosh -2.0e+00-9.42443269378637893396e+00i) -3.7621954668392959453e0+1.2522351259026468452e-3i)
      (num-test (cosh 2.0e+00+9.42512322775237976202e+00i) -3.7621954668392959462e0-1.2522351258999818715e-3i)
      (num-test (cosh 2.0e+00-9.42512322775237976202e+00i) -3.7621954668392959462e0+1.2522351258999818715e-3i)
      (num-test (cosh -2.0e+00+9.42512322775237976202e+00i) -3.7621954668392959462e0+1.2522351258999818715e-3i)
      (num-test (cosh -2.0e+00-9.42512322775237976202e+00i) -3.7621954668392959462e0-1.2522351258999818715e-3i)
      (num-test (cosh (log (/ (+ 1 (sqrt 5)) 2))) (/ (sqrt 5) 2))
      (num-test (cosh 0) 1.0)


      ;; -------- tanh
      (num-test (tanh 0) 0.0)
      (num-test (tanh 1) 0.76159415595576)
      (num-test (tanh -1) -0.76159415595576)
      (num-test (tanh 2) 0.96402758007582)
      (num-test (tanh -2) -0.96402758007582)
      (num-test (tanh 3) 0.99505475368673)
      (num-test (tanh -3) -0.99505475368673)
      (num-test (tanh 10) 0.99999999587769)
      (num-test (tanh -10) -0.99999999587769)
      (num-test (tanh 0/1) 0.0)
      (num-test (tanh 0/2) 0.0)
      (num-test (tanh 0/3) 0.0)
      (num-test (tanh 0/10) 0.0)
      (num-test (tanh 0/1234) 0.0)
      (num-test (tanh 0/1234000000) 0.0)
      (num-test (tanh 0/500029) 0.0)
      (num-test (tanh 0/362880) 0.0)
      (num-test (tanh 1/1) 0.76159415595576)
      (num-test (tanh -1/1) -0.76159415595576)
      (num-test (tanh 1/2) 0.46211715726001)
      (num-test (tanh -1/2) -0.46211715726001)
      (num-test (tanh 1/3) 0.32151273753163)
      (num-test (tanh -1/3) -0.32151273753163)
      (num-test (tanh 1/10) 0.09966799462496)
      (num-test (tanh -1/10) -0.09966799462496)
      (num-test (tanh 1/1234) 0.00081037259408)
      (num-test (tanh -1/1234) -0.00081037259408)
      (num-test (tanh 1/1234000000) 0.00000000081037)
      (num-test (tanh -1/1234000000) -0.00000000081037)
      (num-test (tanh 1/500029) 0.00000199988401)
      (num-test (tanh -1/500029) -0.00000199988401)
      (num-test (tanh 1/362880) 0.00000275573192)
      (num-test (tanh -1/362880) -0.00000275573192)
      (num-test (tanh 2/1) 0.96402758007582)
      (num-test (tanh -2/1) -0.96402758007582)
      (num-test (tanh 2/2) 0.76159415595576)
      (num-test (tanh -2/2) -0.76159415595576)
      (num-test (tanh 2/3) 0.58278294534791)
      (num-test (tanh -2/3) -0.58278294534791)
      (num-test (tanh 2/10) 0.19737532022490)
      (num-test (tanh -2/10) -0.19737532022490)
      (num-test (tanh 2/1234) 0.00162074412382)
      (num-test (tanh -2/1234) -0.00162074412382)
      (num-test (tanh 2/1234000000) 0.00000000162075)
      (num-test (tanh -2/1234000000) -0.00000000162075)
      (num-test (tanh 2/500029) 0.00000399976801)
      (num-test (tanh -2/500029) -0.00000399976801)
      (num-test (tanh 2/362880) 0.00000551146384)
      (num-test (tanh -2/362880) -0.00000551146384)
      (num-test (tanh 3/1) 0.99505475368673)
      (num-test (tanh -3/1) -0.99505475368673)
      (num-test (tanh 3/2) 0.90514825364487)
      (num-test (tanh -3/2) -0.90514825364487)
      (num-test (tanh 3/3) 0.76159415595576)
      (num-test (tanh -3/3) -0.76159415595576)
      (num-test (tanh 3/10) 0.29131261245159)
      (num-test (tanh -3/10) -0.29131261245159)
      (num-test (tanh 3/1234) 0.00243111352486)
      (num-test (tanh -3/1234) -0.00243111352486)
      (num-test (tanh 3/1234000000) 0.00000000243112)
      (num-test (tanh -3/1234000000) -0.00000000243112)
      (num-test (tanh 3/500029) 0.00000599965202)
      (num-test (tanh -3/500029) -0.00000599965202)
      (num-test (tanh 3/362880) 0.00000826719577)
      (num-test (tanh -3/362880) -0.00000826719577)
      (num-test (tanh 10/1) 0.99999999587769)
      (num-test (tanh -10/1) -0.99999999587769)
      (num-test (tanh 10/2) 0.99990920426260)
      (num-test (tanh -10/2) -0.99990920426260)
      (num-test (tanh 10/3) 0.99745796747384)
      (num-test (tanh -10/3) -0.99745796747384)
      (num-test (tanh 10/10) 0.76159415595576)
      (num-test (tanh -10/10) -0.76159415595576)
      (num-test (tanh 10/1234) 0.00810355032772)
      (num-test (tanh -10/1234) -0.00810355032772)
      (num-test (tanh 10/1234000000) 0.00000000810373)
      (num-test (tanh -10/1234000000) -0.00000000810373)
      (num-test (tanh 10/500029) 0.00001999884006)
      (num-test (tanh -10/500029) -0.00001999884006)
      (num-test (tanh 10/362880) 0.00002755731922)
      (num-test (tanh -10/362880) -0.00002755731922)
      (num-test (tanh 1234/10) 1.0)
      (num-test (tanh -1234/10) -1.0)
      (num-test (tanh 1234/1234) 0.76159415595576)
      (num-test (tanh -1234/1234) -0.76159415595576)
      (num-test (tanh 1234/1234000000) 0.00000100000000)
      (num-test (tanh -1234/1234000000) -0.00000100000000)
      (num-test (tanh 1234/500029) 0.00246785185430)
      (num-test (tanh -1234/500029) -0.00246785185430)
      (num-test (tanh 1234/362880) 0.00340056008434)
      (num-test (tanh -1234/362880) -0.00340056008434)
      (num-test (tanh 1234000000/1234000000) 0.76159415595576)
      (num-test (tanh -1234000000/1234000000) -0.76159415595576)
      (num-test (tanh 500029/1234000000) 0.00040520986437)
      (num-test (tanh -500029/1234000000) -0.00040520986437)
      (num-test (tanh 500029/500029) 0.76159415595576)
      (num-test (tanh -500029/500029) -0.76159415595576)
      (num-test (tanh 500029/362880) 0.88049046571870)
      (num-test (tanh -500029/362880) -0.88049046571870)
      (num-test (tanh 362880/1234) 1.0)
      (num-test (tanh -362880/1234) -1.0)
      (num-test (tanh 362880/1234000000) 0.00029406806284)
      (num-test (tanh -362880/1234000000) -0.00029406806284)
      (num-test (tanh 362880/500029) 0.62043861844530)
      (num-test (tanh -362880/500029) -0.62043861844530)
      (num-test (tanh 362880/362880) 0.76159415595576)
      (num-test (tanh -362880/362880) -0.76159415595576)
      (num-test (tanh 0.0) 0.0)
      (num-test (tanh 0.00000001) 0.00000001)
      (num-test (tanh -0.00000001) -0.00000001)
      (num-test (tanh 1.0) 0.76159415595576)
      (num-test (tanh -1.0) -0.76159415595576)
      (num-test (tanh our-pi) 0.99627207622075)
      (num-test (tanh -3.14159265358979) -0.99627207622075)
      (num-test (tanh 2.71828182845905) 0.99132891580060)
      (num-test (tanh -2.71828182845905) -0.99132891580060)
      (num-test (tanh 0.0+0.0i) 0.0)
      (num-test (tanh -0.0+0.0i) 0.0)
      (num-test (tanh 0.0-0.0i) 0.0)
      (num-test (tanh -0.0-0.0i) -0.0)
      (num-test (tanh 0.0+0.00000001i) 0.0+0.00000001i)
      (num-test (tanh -0.0+0.00000001i) 0.0+0.00000001i)
      (num-test (tanh 0.0-0.00000001i) 0.0-0.00000001i)
      (num-test (tanh -0.0-0.00000001i) -0.0-0.00000001i)
      (num-test (tanh 0.0+1.0i) 0.0+1.55740772465490i)
      (num-test (tanh -0.0+1.0i) 0.0+1.55740772465490i)
      (num-test (tanh 0.0-1.0i) 0.0-1.55740772465490i)
      (num-test (tanh -0.0-1.0i) -0.0-1.55740772465490i)
      (num-test (tanh 0.00000001+0.0i) 0.00000001)
      (num-test (tanh -0.00000001+0.0i) -0.00000001)
      (num-test (tanh 0.00000001-0.0i) 0.00000001)
      (num-test (tanh -0.00000001-0.0i) -0.00000001)
      (num-test (tanh 0.00000001+0.00000001i) 0.00000001+0.00000001i)
      (num-test (tanh -0.00000001+0.00000001i) -0.00000001+0.00000001i)
      (num-test (tanh 0.00000001-0.00000001i) 0.00000001-0.00000001i)
      (num-test (tanh -0.00000001-0.00000001i) -0.00000001-0.00000001i)
      (num-test (tanh 0.00000001+1.0i) 0.00000003425519+1.55740772465490i)
      (num-test (tanh -0.00000001+1.0i) -0.00000003425519+1.55740772465490i)
      (num-test (tanh 0.00000001-1.0i) 0.00000003425519-1.55740772465490i)
      (num-test (tanh -0.00000001-1.0i) -0.00000003425519-1.55740772465490i)
      (num-test (tanh 1.0+0.0i) 0.76159415595576)
      (num-test (tanh -1.0+0.0i) -0.76159415595576)
      (num-test (tanh 1.0-0.0i) 0.76159415595576)
      (num-test (tanh -1.0-0.0i) -0.76159415595576)
      (num-test (tanh 1.0+0.00000001i) 0.76159415595576+0.00000000419974i)
      (num-test (tanh -1.0+0.00000001i) -0.76159415595576+0.00000000419974i)
      (num-test (tanh 1.0-0.00000001i) 0.76159415595576-0.00000000419974i)
      (num-test (tanh -1.0-0.00000001i) -0.76159415595576-0.00000000419974i)
      (num-test (tanh 1.0+1.0i) 1.08392332733869+0.27175258531951i)
      (num-test (tanh -1.0+1.0i) -1.08392332733869+0.27175258531951i)
      (num-test (tanh 1.0-1.0i) 1.08392332733869-0.27175258531951i)
      (num-test (tanh -1.0-1.0i) -1.08392332733869-0.27175258531951i)
      (num-test (tanh 3.14159265358979+0.0i) 0.99627207622075)
      (num-test (tanh -3.14159265358979+0.0i) -0.99627207622075)
      (num-test (tanh 3.14159265358979-0.0i) 0.99627207622075)
      (num-test (tanh -3.14159265358979-0.0i) -0.99627207622075)
      (num-test (tanh 3.14159265358979+0.00000001i) 0.99627207622075+0.00000000007442i)
      (num-test (tanh -3.14159265358979+0.00000001i) -0.99627207622075+0.00000000007442i)
      (num-test (tanh 3.14159265358979-0.00000001i) 0.99627207622075-0.00000000007442i)
      (num-test (tanh -3.14159265358979-0.00000001i) -0.99627207622075-0.00000000007442i)
      (num-test (tanh 3.14159265358979+1.0i) 1.00154968930275+0.00340139653674i)
      (num-test (tanh -3.14159265358979+1.0i) -1.00154968930275+0.00340139653674i)
      (num-test (tanh 3.14159265358979-1.0i) 1.00154968930275-0.00340139653674i)
      (num-test (tanh -3.14159265358979-1.0i) -1.00154968930275-0.00340139653674i)
      (num-test (tanh 2.71828182845905+0.0i) 0.99132891580060)
      (num-test (tanh -2.71828182845905+0.0i) -0.99132891580060)
      (num-test (tanh 2.71828182845905-0.0i) 0.99132891580060)
      (num-test (tanh -2.71828182845905-0.0i) -0.99132891580060)
      (num-test (tanh 2.71828182845905+0.00000001i) 0.99132891580060+0.00000000017267i)
      (num-test (tanh -2.71828182845905+0.00000001i) -0.99132891580060+0.00000000017267i)
      (num-test (tanh 2.71828182845905-0.00000001i) 0.99132891580060-0.00000000017267i)
      (num-test (tanh -2.71828182845905-0.00000001i) -0.99132891580060-0.00000000017267i)
      (num-test (tanh 2.71828182845905+1.0i) 1.00359921084211+0.00794757997665i)
      (num-test (tanh -2.71828182845905+1.0i) -1.00359921084211+0.00794757997665i)
      (num-test (tanh 2.71828182845905-1.0i) 1.00359921084211-0.00794757997665i)
      (num-test (tanh -2.71828182845905-1.0i) -1.00359921084211-0.00794757997665i)
      (num-test (tanh 1234/3) 1.0) 
      (num-test (tanh 1234000000/3) 1.0) 
      (num-test (tanh 1234000000/500029) 1.0) 
      (num-test (tanh 1234000000/362880) 1.0) 
      (num-test (tanh 500029/2) 1.0) 
      (num-test (tanh 500029/3) 1.0) 
      (num-test (tanh 500029/10) 1.0) 
      (num-test (tanh 500029/1234) 1.0) 
      (num-test (tanh 0.0+3.14159265358979i) 0.0+6.982889851335445E-15i) 
      (num-test (tanh 0.0+2.71828182845905i) 0.0-0.4505495340698621i) 
      (num-test (tanh 0.00000001+3.14159265358979i) 1.0e-8+6.982889851335444E-15i) 
      (num-test (tanh 0.00000001+2.71828182845905i) 1.2029948826505699E-8-0.4505495340698621i) 
      (num-test (tanh 0.00000001+1234.0i) 1.5681770497896427E-8-0.7537751984442328i) 
      (num-test (tanh 0.00000001+1234000000.0i) 3.9600648244422054E-7-6.212941995900324i) 
      (num-test (tanh 1.0+3.14159265358979i) +0.7615941559557649+2.932634567877868E-15i) 
      (num-test (tanh 1.0+2.71828182845905i) .8196826057997404-0.1692870118766369i) 
      (num-test (tanh 3.14159265358979+3.14159265358979i) +0.99627207622075+5.196631812627532E-17i) 
      (num-test (tanh 3.14159265358979+2.71828182845905i) .9975247319761639-0.002790687681003331i) 
      (num-test (tanh 3.14159265358979+1234.0i) .9989656315245496-0.003586791196867043i) 
      (num-test (tanh 3.14159265358979+1234000000.0i) 1.003551866736695-0.001176098307980411i) 
      (num-test (tanh 2.71828182845905+3.14159265358979i) +0.991328915800599+1.205734242765375E-16i) 
      (num-test (tanh 2.71828182845905+2.71828182845905i) .9942257438545914-0.00648578276962794i) 
      (num-test (tanh 2.71828182845905+1234.0i) .9975698313220817-0.008351965390936033i) 
      (num-test (tanh 2.71828182845905+1234000000.0i) 1.008299558244272-0.002755390838840499i) 
      (num-test (tanh 1234.0+0.00000001i) +1.0+8.077935669463161E-28i) 
      (num-test (tanh 1234.0+3.14159265358979i) 1.0) 
      (num-test (tanh 1234.0+2.71828182845905i) +1.0+2.710505431213761E-20i) 
      (num-test (tanh 1234000000.0+0.00000001i) 1.0) 
      (num-test (tanh 1234000000.0+3.14159265358979i) 1.0-7.703719777548943E-34i) 
      (num-test (tanh 1234000000.0+2.71828182845905i) +1.0+2.710505431213761E-20i) 
      (num-test (tanh 0.0e+00-3.45266983001243932001e-04i) 0-3.4526699672092183585e-4i)
      (num-test (tanh 0.0e+00+3.45266983001243932001e-04i) 0+3.4526699672092183585e-4i)
      (num-test (tanh 0.0e+00+1.57045105981189525579e+00i) 0+2.8963092606501007060e3i)
      (num-test (tanh 0.0e+00-1.57045105981189525579e+00i) 0-2.8963092606501007060e3i)
      (num-test (tanh 0.0e+00+1.57114159377789786021e+00i) 0-2.8963092606511280143e3i)
      (num-test (tanh 0.0e+00-1.57114159377789786021e+00i) 0+2.8963092606511280143e3i)
      (num-test (tanh 0.0e+00+3.14124738660679181379e+00i) 0-3.4526699672110257641e-4i)
      (num-test (tanh 0.0e+00-3.14124738660679181379e+00i) 0+3.4526699672110257641e-4i)
      (num-test (tanh 0.0e+00+3.14193792057279441821e+00i) 0+3.4526699672085764703e-4i)
      (num-test (tanh 0.0e+00-3.14193792057279441821e+00i) 0-3.4526699672085764703e-4i)
      (num-test (tanh 0.0e+00+4.71204371340168837179e+00i) 0+2.8963092606490733978e3i)
      (num-test (tanh 0.0e+00-4.71204371340168837179e+00i) 0-2.8963092606490733978e3i)
      (num-test (tanh 0.0e+00+4.71273424736769097620e+00i) 0-2.8963092606521553225e3i)
      (num-test (tanh 0.0e+00-4.71273424736769097620e+00i) 0+2.8963092606521553225e3i)
      (num-test (tanh 0.0e+00+6.28284004019658492979e+00i) 0-3.4526699672122504111e-4i)
      (num-test (tanh 0.0e+00-6.28284004019658492979e+00i) 0+3.4526699672122504111e-4i)
      (num-test (tanh 0.0e+00+6.28353057416258753420e+00i) 0+3.4526699672073518233e-4i)
      (num-test (tanh 0.0e+00-6.28353057416258753420e+00i) 0-3.4526699672073518233e-4i)
      (num-test (tanh 0.0e+00+9.42443269378637893396e+00i) 0-3.4526699672045932728e-4i)
      (num-test (tanh 0.0e+00-9.42443269378637893396e+00i) 0+3.4526699672045932728e-4i)
      (num-test (tanh 0.0e+00+9.42512322775237976202e+00i) 0+3.4526699671972453911e-4i)
      (num-test (tanh 0.0e+00-9.42512322775237976202e+00i) 0-3.4526699671972453911e-4i)
      (num-test (tanh 1.19209289550781250e-07-3.45266983001243932001e-04i) 1.1920930376163652989e-7-3.4526699672091692931e-4i)
      (num-test (tanh 1.19209289550781250e-07+3.45266983001243932001e-04i) 1.1920930376163652989e-7+3.4526699672091692931e-4i)
      (num-test (tanh -1.19209289550781250e-07-3.45266983001243932001e-04i) -1.1920930376163652989e-7-3.4526699672091692931e-4i)
      (num-test (tanh -1.19209289550781250e-07+3.45266983001243932001e-04i) -1.1920930376163652989e-7+3.4526699672091692931e-4i)
      (num-test (tanh 1.19209289550781250e-07+1.57045105981189525579e+00i) 9.9999992052646305569e-1+2.8963089153831588642e3i)
      (num-test (tanh 1.19209289550781250e-07-1.57045105981189525579e+00i) 9.9999992052646305569e-1-2.8963089153831588642e3i)
      (num-test (tanh -1.19209289550781250e-07+1.57045105981189525579e+00i) -9.9999992052646305569e-1+2.8963089153831588642e3i)
      (num-test (tanh -1.19209289550781250e-07-1.57045105981189525579e+00i) -9.9999992052646305569e-1-2.8963089153831588642e3i)
      (num-test (tanh 1.19209289550781250e-07+1.57114159377789786021e+00i) 9.9999992052717244672e-1-2.8963089153841861720e3i)
      (num-test (tanh 1.19209289550781250e-07-1.57114159377789786021e+00i) 9.9999992052717244672e-1+2.8963089153841861720e3i)
      (num-test (tanh -1.19209289550781250e-07+1.57114159377789786021e+00i) -9.9999992052717244672e-1-2.8963089153841861720e3i)
      (num-test (tanh -1.19209289550781250e-07-1.57114159377789786021e+00i) -9.9999992052717244672e-1+2.8963089153841861720e3i)
      (num-test (tanh 1.19209289550781250e-07+3.14124738660679181379e+00i) 1.1920930376163652991e-7-3.4526699672109766987e-4i)
      (num-test (tanh 1.19209289550781250e-07-3.14124738660679181379e+00i) 1.1920930376163652991e-7+3.4526699672109766987e-4i)
      (num-test (tanh -1.19209289550781250e-07+3.14124738660679181379e+00i) -1.1920930376163652991e-7-3.4526699672109766987e-4i)
      (num-test (tanh -1.19209289550781250e-07-3.14124738660679181379e+00i) -1.1920930376163652991e-7+3.4526699672109766987e-4i)
      (num-test (tanh 1.19209289550781250e-07+3.14193792057279441821e+00i) 1.1920930376163652989e-7+3.4526699672085274049e-4i)
      (num-test (tanh 1.19209289550781250e-07-3.14193792057279441821e+00i) 1.1920930376163652989e-7-3.4526699672085274049e-4i)
      (num-test (tanh -1.19209289550781250e-07+3.14193792057279441821e+00i) -1.1920930376163652989e-7+3.4526699672085274049e-4i)
      (num-test (tanh -1.19209289550781250e-07-3.14193792057279441821e+00i) -1.1920930376163652989e-7-3.4526699672085274049e-4i)
      (num-test (tanh 1.19209289550781250e-07+4.71204371340168837179e+00i) 9.9999992052575366466e-1+2.8963089153821315563e3i)
      (num-test (tanh 1.19209289550781250e-07-4.71204371340168837179e+00i) 9.9999992052575366466e-1-2.8963089153821315563e3i)
      (num-test (tanh -1.19209289550781250e-07+4.71204371340168837179e+00i) -9.9999992052575366466e-1+2.8963089153821315563e3i)
      (num-test (tanh -1.19209289550781250e-07-4.71204371340168837179e+00i) -9.9999992052575366466e-1-2.8963089153821315563e3i)
      (num-test (tanh 1.19209289550781250e-07+4.71273424736769097620e+00i) 9.9999992052788183776e-1-2.8963089153852134799e3i)
      (num-test (tanh 1.19209289550781250e-07-4.71273424736769097620e+00i) 9.9999992052788183776e-1+2.8963089153852134799e3i)
      (num-test (tanh -1.19209289550781250e-07+4.71273424736769097620e+00i) -9.9999992052788183776e-1-2.8963089153852134799e3i)
      (num-test (tanh -1.19209289550781250e-07-4.71273424736769097620e+00i) -9.9999992052788183776e-1+2.8963089153852134799e3i)
      (num-test (tanh 1.19209289550781250e-07+6.28284004019658492979e+00i) 1.1920930376163652992e-7-3.4526699672122013457e-4i)
      (num-test (tanh 1.19209289550781250e-07-6.28284004019658492979e+00i) 1.1920930376163652992e-7+3.4526699672122013457e-4i)
      (num-test (tanh -1.19209289550781250e-07+6.28284004019658492979e+00i) -1.1920930376163652992e-7-3.4526699672122013457e-4i)
      (num-test (tanh -1.19209289550781250e-07-6.28284004019658492979e+00i) -1.1920930376163652992e-7+3.4526699672122013457e-4i)
      (num-test (tanh 1.19209289550781250e-07+6.28353057416258753420e+00i) 1.1920930376163652988e-7+3.4526699672073027579e-4i)
      (num-test (tanh 1.19209289550781250e-07-6.28353057416258753420e+00i) 1.1920930376163652988e-7-3.4526699672073027579e-4i)
      (num-test (tanh -1.19209289550781250e-07+6.28353057416258753420e+00i) -1.1920930376163652988e-7+3.4526699672073027579e-4i)
      (num-test (tanh -1.19209289550781250e-07-6.28353057416258753420e+00i) -1.1920930376163652988e-7-3.4526699672073027579e-4i)
      (num-test (tanh 1.19209289550781250e-07+9.42443269378637893396e+00i) 1.1920930376163652985e-7-3.4526699672045442074e-4i)
      (num-test (tanh 1.19209289550781250e-07-9.42443269378637893396e+00i) 1.1920930376163652985e-7+3.4526699672045442074e-4i)
      (num-test (tanh -1.19209289550781250e-07+9.42443269378637893396e+00i) -1.1920930376163652985e-7-3.4526699672045442074e-4i)
      (num-test (tanh -1.19209289550781250e-07-9.42443269378637893396e+00i) -1.1920930376163652985e-7+3.4526699672045442074e-4i)
      (num-test (tanh 1.19209289550781250e-07+9.42512322775237976202e+00i) 1.1920930376163652979e-7+3.4526699671971963257e-4i)
      (num-test (tanh 1.19209289550781250e-07-9.42512322775237976202e+00i) 1.1920930376163652979e-7-3.4526699671971963257e-4i)
      (num-test (tanh -1.19209289550781250e-07+9.42512322775237976202e+00i) -1.1920930376163652979e-7+3.4526699671971963257e-4i)
      (num-test (tanh -1.19209289550781250e-07-9.42512322775237976202e+00i) -1.1920930376163652979e-7-3.4526699671971963257e-4i)
      (num-test (tanh 5.0e-01-3.45266983001243932001e-04i) 4.6211720058436229979e-1-2.7153443992655805934e-4i)
      (num-test (tanh 5.0e-01+3.45266983001243932001e-04i) 4.6211720058436229979e-1+2.7153443992655805934e-4i)
      (num-test (tanh -5.0e-01-3.45266983001243932001e-04i) -4.6211720058436229979e-1-2.7153443992655805934e-4i)
      (num-test (tanh -5.0e-01+3.45266983001243932001e-04i) -4.6211720058436229979e-1+2.7153443992655805934e-4i)
      (num-test (tanh 5.0e-01+1.57045105981189525579e+00i) 2.1639524637389325996e0+1.2715121175455623363e-3i)
      (num-test (tanh 5.0e-01-1.57045105981189525579e+00i) 2.1639524637389325996e0-1.2715121175455623363e-3i)
      (num-test (tanh -5.0e-01+1.57045105981189525579e+00i) -2.1639524637389325996e0+1.2715121175455623363e-3i)
      (num-test (tanh -5.0e-01-1.57045105981189525579e+00i) -2.1639524637389325996e0-1.2715121175455623363e-3i)
      (num-test (tanh 5.0e-01+1.57114159377789786021e+00i) 2.1639524637389326002e0-1.2715121175451113370e-3i)
      (num-test (tanh 5.0e-01-1.57114159377789786021e+00i) 2.1639524637389326002e0+1.2715121175451113370e-3i)
      (num-test (tanh -5.0e-01+1.57114159377789786021e+00i) -2.1639524637389326002e0-1.2715121175451113370e-3i)
      (num-test (tanh -5.0e-01-1.57114159377789786021e+00i) -2.1639524637389326002e0+1.2715121175451113370e-3i)
      (num-test (tanh 5.0e-01+3.14124738660679181379e+00i) 4.6211720058436229984e-1-2.7153443992670020234e-4i)
      (num-test (tanh 5.0e-01-3.14124738660679181379e+00i) 4.6211720058436229984e-1+2.7153443992670020234e-4i)
      (num-test (tanh -5.0e-01+3.14124738660679181379e+00i) -4.6211720058436229984e-1-2.7153443992670020234e-4i)
      (num-test (tanh -5.0e-01-3.14124738660679181379e+00i) -4.6211720058436229984e-1+2.7153443992670020234e-4i)
      (num-test (tanh 5.0e-01+3.14193792057279441821e+00i) 4.6211720058436229978e-1+2.7153443992650757820e-4i)
      (num-test (tanh 5.0e-01-3.14193792057279441821e+00i) 4.6211720058436229978e-1-2.7153443992650757820e-4i)
      (num-test (tanh -5.0e-01+3.14193792057279441821e+00i) -4.6211720058436229978e-1+2.7153443992650757820e-4i)
      (num-test (tanh -5.0e-01-3.14193792057279441821e+00i) -4.6211720058436229978e-1-2.7153443992650757820e-4i)
      (num-test (tanh 5.0e-01+4.71204371340168837179e+00i) 2.1639524637389325989e0+1.2715121175460133355e-3i)
      (num-test (tanh 5.0e-01-4.71204371340168837179e+00i) 2.1639524637389325989e0-1.2715121175460133355e-3i)
      (num-test (tanh -5.0e-01+4.71204371340168837179e+00i) -2.1639524637389325989e0+1.2715121175460133355e-3i)
      (num-test (tanh -5.0e-01-4.71204371340168837179e+00i) -2.1639524637389325989e0-1.2715121175460133355e-3i)
      (num-test (tanh 5.0e-01+4.71273424736769097620e+00i) 2.1639524637389326009e0-1.2715121175446603377e-3i)
      (num-test (tanh 5.0e-01-4.71273424736769097620e+00i) 2.1639524637389326009e0+1.2715121175446603377e-3i)
      (num-test (tanh -5.0e-01+4.71273424736769097620e+00i) -2.1639524637389326009e0-1.2715121175446603377e-3i)
      (num-test (tanh -5.0e-01-4.71273424736769097620e+00i) -2.1639524637389326009e0+1.2715121175446603377e-3i)
      (num-test (tanh 5.0e-01+6.28284004019658492979e+00i) 4.6211720058436229987e-1-2.7153443992679651442e-4i)
      (num-test (tanh 5.0e-01-6.28284004019658492979e+00i) 4.6211720058436229987e-1+2.7153443992679651442e-4i)
      (num-test (tanh -5.0e-01+6.28284004019658492979e+00i) -4.6211720058436229987e-1-2.7153443992679651442e-4i)
      (num-test (tanh -5.0e-01-6.28284004019658492979e+00i) -4.6211720058436229987e-1+2.7153443992679651442e-4i)
      (num-test (tanh 5.0e-01+6.28353057416258753420e+00i) 4.6211720058436229974e-1+2.7153443992641126612e-4i)
      (num-test (tanh 5.0e-01-6.28353057416258753420e+00i) 4.6211720058436229974e-1-2.7153443992641126612e-4i)
      (num-test (tanh -5.0e-01+6.28353057416258753420e+00i) -4.6211720058436229974e-1+2.7153443992641126612e-4i)
      (num-test (tanh -5.0e-01-6.28353057416258753420e+00i) -4.6211720058436229974e-1-2.7153443992641126612e-4i)
      (num-test (tanh 5.0e-01+9.42443269378637893396e+00i) 4.6211720058436229968e-1-2.7153443992619432056e-4i)
      (num-test (tanh 5.0e-01-9.42443269378637893396e+00i) 4.6211720058436229968e-1+2.7153443992619432056e-4i)
      (num-test (tanh -5.0e-01+9.42443269378637893396e+00i) -4.6211720058436229968e-1-2.7153443992619432056e-4i)
      (num-test (tanh -5.0e-01-9.42443269378637893396e+00i) -4.6211720058436229968e-1+2.7153443992619432056e-4i)
      (num-test (tanh 5.0e-01+9.42512322775237976202e+00i) 4.6211720058436229949e-1+2.7153443992561644811e-4i)
      (num-test (tanh 5.0e-01-9.42512322775237976202e+00i) 4.6211720058436229949e-1-2.7153443992561644811e-4i)
      (num-test (tanh -5.0e-01+9.42512322775237976202e+00i) -4.6211720058436229949e-1+2.7153443992561644811e-4i)
      (num-test (tanh -5.0e-01-9.42512322775237976202e+00i) -4.6211720058436229949e-1-2.7153443992561644811e-4i)
      (num-test (tanh 1.0e+00-3.45266983001243932001e-04i) 7.6159419408485704836e-1-1.4500326960274960880e-4i)
      (num-test (tanh 1.0e+00+3.45266983001243932001e-04i) 7.6159419408485704836e-1+1.4500326960274960880e-4i)
      (num-test (tanh -1.0e+00-3.45266983001243932001e-04i) -7.6159419408485704836e-1-1.4500326960274960880e-4i)
      (num-test (tanh -1.0e+00+3.45266983001243932001e-04i) -7.6159419408485704836e-1+1.4500326960274960880e-4i)
      (num-test (tanh 1.0e+00+1.57045105981189525579e+00i) 1.3130351721648674823e0+2.4999454374276273814e-4i)
      (num-test (tanh 1.0e+00-1.57045105981189525579e+00i) 1.3130351721648674823e0-2.4999454374276273814e-4i)
      (num-test (tanh -1.0e+00+1.57045105981189525579e+00i) -1.3130351721648674823e0+2.4999454374276273814e-4i)
      (num-test (tanh -1.0e+00-1.57045105981189525579e+00i) -1.3130351721648674823e0-2.4999454374276273814e-4i)
      (num-test (tanh 1.0e+00+1.57114159377789786021e+00i) 1.3130351721648674824e0-2.4999454374267406620e-4i)
      (num-test (tanh 1.0e+00-1.57114159377789786021e+00i) 1.3130351721648674824e0+2.4999454374267406620e-4i)
      (num-test (tanh -1.0e+00+1.57114159377789786021e+00i) -1.3130351721648674824e0-2.4999454374267406620e-4i)
      (num-test (tanh -1.0e+00-1.57114159377789786021e+00i) -1.3130351721648674824e0+2.4999454374267406620e-4i)
      (num-test (tanh 1.0e+00+3.14124738660679181379e+00i) 7.6159419408485704840e-1-1.4500326960282551519e-4i)
      (num-test (tanh 1.0e+00-3.14124738660679181379e+00i) 7.6159419408485704840e-1+1.4500326960282551519e-4i)
      (num-test (tanh -1.0e+00+3.14124738660679181379e+00i) -7.6159419408485704840e-1-1.4500326960282551519e-4i)
      (num-test (tanh -1.0e+00-3.14124738660679181379e+00i) -7.6159419408485704840e-1+1.4500326960282551519e-4i)
      (num-test (tanh 1.0e+00+3.14193792057279441821e+00i) 7.6159419408485704835e-1+1.4500326960272265115e-4i)
      (num-test (tanh 1.0e+00-3.14193792057279441821e+00i) 7.6159419408485704835e-1-1.4500326960272265115e-4i)
      (num-test (tanh -1.0e+00+3.14193792057279441821e+00i) -7.6159419408485704835e-1+1.4500326960272265115e-4i)
      (num-test (tanh -1.0e+00-3.14193792057279441821e+00i) -7.6159419408485704835e-1-1.4500326960272265115e-4i)
      (num-test (tanh 1.0e+00+4.71204371340168837179e+00i) 1.3130351721648674822e0+2.4999454374285141007e-4i)
      (num-test (tanh 1.0e+00-4.71204371340168837179e+00i) 1.3130351721648674822e0-2.4999454374285141007e-4i)
      (num-test (tanh -1.0e+00+4.71204371340168837179e+00i) -1.3130351721648674822e0+2.4999454374285141007e-4i)
      (num-test (tanh -1.0e+00-4.71204371340168837179e+00i) -1.3130351721648674822e0-2.4999454374285141007e-4i)
      (num-test (tanh 1.0e+00+4.71273424736769097620e+00i) 1.3130351721648674825e0-2.4999454374258539427e-4i)
      (num-test (tanh 1.0e+00-4.71273424736769097620e+00i) 1.3130351721648674825e0+2.4999454374258539427e-4i)
      (num-test (tanh -1.0e+00+4.71273424736769097620e+00i) -1.3130351721648674825e0-2.4999454374258539427e-4i)
      (num-test (tanh -1.0e+00-4.71273424736769097620e+00i) -1.3130351721648674825e0+2.4999454374258539427e-4i)
      (num-test (tanh 1.0e+00+6.28284004019658492979e+00i) 7.6159419408485704843e-1-1.4500326960287694721e-4i)
      (num-test (tanh 1.0e+00-6.28284004019658492979e+00i) 7.6159419408485704843e-1+1.4500326960287694721e-4i)
      (num-test (tanh -1.0e+00+6.28284004019658492979e+00i) -7.6159419408485704843e-1-1.4500326960287694721e-4i)
      (num-test (tanh -1.0e+00-6.28284004019658492979e+00i) -7.6159419408485704843e-1+1.4500326960287694721e-4i)
      (num-test (tanh 1.0e+00+6.28353057416258753420e+00i) 7.6159419408485704832e-1+1.4500326960267121913e-4i)
      (num-test (tanh 1.0e+00-6.28353057416258753420e+00i) 7.6159419408485704832e-1-1.4500326960267121913e-4i)
      (num-test (tanh -1.0e+00+6.28353057416258753420e+00i) -7.6159419408485704832e-1+1.4500326960267121913e-4i)
      (num-test (tanh -1.0e+00-6.28353057416258753420e+00i) -7.6159419408485704832e-1-1.4500326960267121913e-4i)
      (num-test (tanh 1.0e+00+9.42443269378637893396e+00i) 7.6159419408485704826e-1-1.4500326960255536711e-4i)
      (num-test (tanh 1.0e+00-9.42443269378637893396e+00i) 7.6159419408485704826e-1+1.4500326960255536711e-4i)
      (num-test (tanh -1.0e+00+9.42443269378637893396e+00i) -7.6159419408485704826e-1-1.4500326960255536711e-4i)
      (num-test (tanh -1.0e+00-9.42443269378637893396e+00i) -7.6159419408485704826e-1+1.4500326960255536711e-4i)
      (num-test (tanh 1.0e+00+9.42512322775237976202e+00i) 7.6159419408485704810e-1+1.450032696022467750e-4i)
      (num-test (tanh 1.0e+00-9.42512322775237976202e+00i) 7.6159419408485704810e-1-1.450032696022467750e-4i)
      (num-test (tanh -1.0e+00+9.42512322775237976202e+00i) -7.6159419408485704810e-1+1.450032696022467750e-4i)
      (num-test (tanh -1.0e+00-9.42512322775237976202e+00i) -7.6159419408485704810e-1-1.450032696022467750e-4i)
      (num-test (tanh 2.0e+00-3.45266983001243932001e-04i) 9.6402758819508310556e-1-2.4393395410435306874e-5i)
      (num-test (tanh 2.0e+00+3.45266983001243932001e-04i) 9.6402758819508310556e-1+2.4393395410435306874e-5i)
      (num-test (tanh -2.0e+00-3.45266983001243932001e-04i) -9.6402758819508310556e-1-2.4393395410435306874e-5i)
      (num-test (tanh -2.0e+00+3.45266983001243932001e-04i) -9.6402758819508310556e-1+2.4393395410435306874e-5i)
      (num-test (tanh 2.0e+00+1.57045105981189525579e+00i) 1.0373147113268752620e0+2.6247825506572821595e-5i)
      (num-test (tanh 2.0e+00-1.57045105981189525579e+00i) 1.0373147113268752620e0-2.6247825506572821595e-5i)
      (num-test (tanh -2.0e+00+1.57045105981189525579e+00i) -1.0373147113268752620e0+2.6247825506572821595e-5i)
      (num-test (tanh -2.0e+00-1.57045105981189525579e+00i) -1.0373147113268752620e0-2.6247825506572821595e-5i)
      (num-test (tanh 2.0e+00+1.57114159377789786021e+00i) 1.0373147113268752620e0-2.6247825506563511609e-5i)
      (num-test (tanh 2.0e+00-1.57114159377789786021e+00i) 1.0373147113268752620e0+2.6247825506563511609e-5i)
      (num-test (tanh -2.0e+00+1.57114159377789786021e+00i) -1.0373147113268752620e0-2.6247825506563511609e-5i)
      (num-test (tanh -2.0e+00-1.57114159377789786021e+00i) -1.0373147113268752620e0+2.6247825506563511609e-5i)
      (num-test (tanh 2.0e+00+3.14124738660679181379e+00i) 9.6402758819508310557e-1-2.4393395410448076340e-5i)
      (num-test (tanh 2.0e+00-3.14124738660679181379e+00i) 9.6402758819508310557e-1+2.4393395410448076340e-5i)
      (num-test (tanh -2.0e+00+3.14124738660679181379e+00i) -9.6402758819508310557e-1-2.4393395410448076340e-5i)
      (num-test (tanh -2.0e+00-3.14124738660679181379e+00i) -9.6402758819508310557e-1+2.4393395410448076340e-5i)
      (num-test (tanh 2.0e+00+3.14193792057279441821e+00i) 9.6402758819508310556e-1+2.4393395410430771882e-5i)
      (num-test (tanh 2.0e+00-3.14193792057279441821e+00i) 9.6402758819508310556e-1-2.4393395410430771882e-5i)
      (num-test (tanh -2.0e+00+3.14193792057279441821e+00i) -9.6402758819508310556e-1+2.4393395410430771882e-5i)
      (num-test (tanh -2.0e+00-3.14193792057279441821e+00i) -9.6402758819508310556e-1-2.4393395410430771882e-5i)
      (num-test (tanh 2.0e+00+4.71204371340168837179e+00i) 1.0373147113268752620e0+2.6247825506582131582e-5i)
      (num-test (tanh 2.0e+00-4.71204371340168837179e+00i) 1.0373147113268752620e0-2.6247825506582131582e-5i)
      (num-test (tanh -2.0e+00+4.71204371340168837179e+00i) -1.0373147113268752620e0+2.6247825506582131582e-5i)
      (num-test (tanh -2.0e+00-4.71204371340168837179e+00i) -1.0373147113268752620e0-2.6247825506582131582e-5i)
      (num-test (tanh 2.0e+00+4.71273424736769097620e+00i) 1.0373147113268752620e0-2.6247825506554201622e-5i)
      (num-test (tanh 2.0e+00-4.71273424736769097620e+00i) 1.0373147113268752620e0+2.6247825506554201622e-5i)
      (num-test (tanh -2.0e+00+4.71273424736769097620e+00i) -1.0373147113268752620e0-2.6247825506554201622e-5i)
      (num-test (tanh -2.0e+00-4.71273424736769097620e+00i) -1.0373147113268752620e0+2.6247825506554201622e-5i)
      (num-test (tanh 2.0e+00+6.28284004019658492979e+00i) 9.6402758819508310558e-1-2.4393395410456728569e-5i)
      (num-test (tanh 2.0e+00-6.28284004019658492979e+00i) 9.6402758819508310558e-1+2.4393395410456728569e-5i)
      (num-test (tanh -2.0e+00+6.28284004019658492979e+00i) -9.6402758819508310558e-1-2.4393395410456728569e-5i)
      (num-test (tanh -2.0e+00-6.28284004019658492979e+00i) -9.6402758819508310558e-1+2.4393395410456728569e-5i)
      (num-test (tanh 2.0e+00+6.28353057416258753420e+00i) 9.6402758819508310555e-1+2.4393395410422119654e-5i)
      (num-test (tanh 2.0e+00-6.28353057416258753420e+00i) 9.6402758819508310555e-1-2.4393395410422119654e-5i)
      (num-test (tanh -2.0e+00+6.28353057416258753420e+00i) -9.6402758819508310555e-1+2.4393395410422119654e-5i)
      (num-test (tanh -2.0e+00-6.28353057416258753420e+00i) -9.6402758819508310555e-1-2.4393395410422119654e-5i)
      (num-test (tanh 2.0e+00+9.42443269378637893396e+00i) 9.6402758819508310554e-1-2.4393395410402630273e-5i)
      (num-test (tanh 2.0e+00-9.42443269378637893396e+00i) 9.6402758819508310554e-1+2.4393395410402630273e-5i)
      (num-test (tanh -2.0e+00+9.42443269378637893396e+00i) -9.6402758819508310554e-1-2.4393395410402630273e-5i)
      (num-test (tanh -2.0e+00-9.42443269378637893396e+00i) -9.6402758819508310554e-1+2.4393395410402630273e-5i)
      (num-test (tanh 2.0e+00+9.42512322775237976202e+00i) 9.6402758819508310550e-1+2.439339541035071690e-5i)
      (num-test (tanh 2.0e+00-9.42512322775237976202e+00i) 9.6402758819508310550e-1-2.439339541035071690e-5i)
      (num-test (tanh -2.0e+00+9.42512322775237976202e+00i) -9.6402758819508310550e-1+2.439339541035071690e-5i)
      (num-test (tanh -2.0e+00-9.42512322775237976202e+00i) -9.6402758819508310550e-1-2.439339541035071690e-5i)
      (num-test (tanh 50) 1.0)


      ;; -------- asinh
      (num-test (asinh 0) 0.0)
      (num-test (asinh 1) 0.88137358701954)
      (num-test (asinh -1) -0.88137358701954)
      (num-test (asinh 2) 1.44363547517881)
      (num-test (asinh -2) -1.44363547517881)
      (num-test (asinh 3) 1.81844645923207)
      (num-test (asinh -3) -1.81844645923207)
      (num-test (asinh 10) 2.99822295029797)
      (num-test (asinh -10) -2.99822295029798)
      (num-test (asinh 1234) 7.81116354920125)
      (num-test (asinh -1234) -7.81116354896171)
      (num-test (asinh 1234000000) 21.62667394298955)
      (num-test (asinh 500029) 13.81556855628334)
      (num-test (asinh 362880) 13.49497466064331)
      (num-test (asinh 0/1) 0.0)
      (num-test (asinh 0/2) 0.0)
      (num-test (asinh 0/3) 0.0)
      (num-test (asinh 0/10) 0.0)
      (num-test (asinh 0/1234) 0.0)
      (num-test (asinh 0/1234000000) 0.0)
      (num-test (asinh 0/500029) 0.0)
      (num-test (asinh 0/362880) 0.0)
      (num-test (asinh 1/1) 0.88137358701954)
      (num-test (asinh -1/1) -0.88137358701954)
      (num-test (asinh 1/2) 0.48121182505960)
      (num-test (asinh -1/2) -0.48121182505960)
      (num-test (asinh 1/3) 0.32745015023726)
      (num-test (asinh -1/3) -0.32745015023726)
      (num-test (asinh 1/10) 0.09983407889921)
      (num-test (asinh -1/10) -0.09983407889921)
      (num-test (asinh 1/1234) 0.00081037268278)
      (num-test (asinh -1/1234) -0.00081037268278)
      (num-test (asinh 1/1234000000) 0.00000000081037)
      (num-test (asinh -1/1234000000) -0.00000000081037)
      (num-test (asinh 1/500029) 0.00000199988401)
      (num-test (asinh -1/500029) -0.00000199988401)
      (num-test (asinh 1/362880) 0.00000275573192)
      (num-test (asinh -1/362880) -0.00000275573192)
      (num-test (asinh 2/1) 1.44363547517881)
      (num-test (asinh -2/1) -1.44363547517881)
      (num-test (asinh 2/2) 0.88137358701954)
      (num-test (asinh -2/2) -0.88137358701954)
      (num-test (asinh 2/3) 0.62514511725042)
      (num-test (asinh -2/3) -0.62514511725042)
      (num-test (asinh 2/10) 0.19869011034924)
      (num-test (asinh -2/10) -0.19869011034924)
      (num-test (asinh 2/1234) 0.00162074483338)
      (num-test (asinh -2/1234) -0.00162074483338)
      (num-test (asinh 2/1234000000) 0.00000000162075)
      (num-test (asinh -2/1234000000) -0.00000000162075)
      (num-test (asinh 2/500029) 0.00000399976801)
      (num-test (asinh -2/500029) -0.00000399976801)
      (num-test (asinh 2/362880) 0.00000551146384)
      (num-test (asinh -2/362880) -0.00000551146384)
      (num-test (asinh 3/1) 1.81844645923207)
      (num-test (asinh -3/1) -1.81844645923207)
      (num-test (asinh 3/2) 1.19476321728711)
      (num-test (asinh -3/2) -1.19476321728711)
      (num-test (asinh 3/3) 0.88137358701954)
      (num-test (asinh -3/3) -0.88137358701954)
      (num-test (asinh 3/10) 0.29567304756342)
      (num-test (asinh -3/10) -0.29567304756342)
      (num-test (asinh 3/1234) 0.00243111591964)
      (num-test (asinh -3/1234) -0.00243111591964)
      (num-test (asinh 3/1234000000) 0.00000000243112)
      (num-test (asinh -3/1234000000) -0.00000000243112)
      (num-test (asinh 3/500029) 0.00000599965202)
      (num-test (asinh -3/500029) -0.00000599965202)
      (num-test (asinh 3/362880) 0.00000826719577)
      (num-test (asinh -3/362880) -0.00000826719577)
      (num-test (asinh 10/1) 2.99822295029797)
      (num-test (asinh -10/1) -2.99822295029798)
      (num-test (asinh 10/2) 2.31243834127275)
      (num-test (asinh -10/2) -2.31243834127276)
      (num-test (asinh 10/3) 1.91889647209853)
      (num-test (asinh -10/3) -1.91889647209853)
      (num-test (asinh 10/10) 0.88137358701954)
      (num-test (asinh -10/10) -0.88137358701954)
      (num-test (asinh 10/1234) 0.00810363902153)
      (num-test (asinh -10/1234) -0.00810363902153)
      (num-test (asinh 10/1234000000) 0.00000000810373)
      (num-test (asinh -10/1234000000) -0.00000000810373)
      (num-test (asinh 10/500029) 0.00001999884007)
      (num-test (asinh -10/500029) -0.00001999884007)
      (num-test (asinh 10/362880) 0.00002755731922)
      (num-test (asinh -10/362880) -0.00002755731922)
      (num-test (asinh 1234/1) 7.81116354920125)
      (num-test (asinh -1234/1) -7.81116354896171)
      (num-test (asinh 1234/2) 7.11801686116871)
      (num-test (asinh -1234/2) -7.11801686109890)
      (num-test (asinh 1234/3) 6.71255257393796)
      (num-test (asinh -1234/3) -6.71255257391934)
      (num-test (asinh 1234/10) 5.50859470922766)
      (num-test (asinh -1234/10) -5.50859470922637)
      (num-test (asinh 1234/1234) 0.88137358701954)
      (num-test (asinh -1234/1234) -0.88137358701954)
      (num-test (asinh 1234/1234000000) 0.00000100000000)
      (num-test (asinh -1234/1234000000) -0.00000100000000)
      (num-test (asinh 1234/500029) 0.00246785435930)
      (num-test (asinh -1234/500029) -0.00246785435930)
      (num-test (asinh 1234/362880) 0.00340056663829)
      (num-test (asinh -1234/362880) -0.00340056663829)
      (num-test (asinh 1234000000/1) 21.62667394298955)
      (num-test (asinh 1234000000/2) 20.93352676242961)
      (num-test (asinh 1234000000/3) 20.52806165432144)
      (num-test (asinh 1234000000/10) 19.32408884999551)
      (num-test (asinh 1234000000/1234) 14.50865773852447)
      (num-test (asinh 1234000000/1234000000) 0.88137358701954)
      (num-test (asinh -1234000000/1234000000) -0.88137358701954)
      (num-test (asinh 1234000000/500029) 8.50425260831592)
      (num-test (asinh -1234000000/500029) -8.50425260755399)
      (num-test (asinh 1234000000/362880) 8.82484648452709)
      (num-test (asinh -1234000000/362880) -8.82484648383427)
      (num-test (asinh 500029/1) 13.81556855628334)
      (num-test (asinh 500029/2) 13.12242137572639)
      (num-test (asinh 500029/3) 12.71695626762323)
      (num-test (asinh -500029/3) -12.71695626762323)
      (num-test (asinh 500029/10) 11.51298346338828)
      (num-test (asinh -500029/10) -11.51298333576987)
      (num-test (asinh 500029/1234) 6.69755387439290)
      (num-test (asinh -500029/1234) -6.69755387437470)
      (num-test (asinh 500029/1234000000) 0.00040520987546)
      (num-test (asinh -500029/1234000000) -0.00040520987546)
      (num-test (asinh 500029/500029) 0.88137358701954)
      (num-test (asinh -500029/500029) -0.88137358701954)
      (num-test (asinh 500029/362880) 1.12509614482188)
      (num-test (asinh -500029/362880) -1.12509614482188)
      (num-test (asinh 362880/1) 13.49497466064331)
      (num-test (asinh 362880/2) 12.80182748008906)
      (num-test (asinh -362880/2) -12.80182748008906)
      (num-test (asinh 362880/3) 12.39636237199039)
      (num-test (asinh -362880/3) -12.39636125811213)
      (num-test (asinh 362880/10) 11.19238956783722)
      (num-test (asinh -362880/10) -11.19238933388543)
      (num-test (asinh 362880/1234) 6.37696134713805)
      (num-test (asinh -362880/1234) -6.37696134714268)
      (num-test (asinh 362880/1234000000) 0.00029406806707)
      (num-test (asinh -362880/1234000000) -0.00029406806707)
      (num-test (asinh 362880/500029) 0.67360821551802)
      (num-test (asinh -362880/500029) -0.67360821551802)
      (num-test (asinh 362880/362880) 0.88137358701954)
      (num-test (asinh -362880/362880) -0.88137358701954)
      (num-test (asinh 0.0) 0.0)
      (num-test (asinh 0.00000001) 0.00000001)
      (num-test (asinh -0.00000001) -0.00000001)
      (num-test (asinh 1.0) 0.88137358701954)
      (num-test (asinh -1.0) -0.88137358701954)
      (num-test (asinh our-pi) 1.86229574331085)
      (num-test (asinh -3.14159265358979) -1.86229574331085)
      (num-test (asinh 2.71828182845905) 1.72538255885231)
      (num-test (asinh -2.71828182845905) -1.72538255885232)
      (num-test (asinh 1234.0) 7.81116354920125)
      (num-test (asinh -1234.0) -7.81116354896171)
      (num-test (asinh 1234000000.0) 21.62667394298955)
      (num-test (asinh 0.0+0.0i) 0.0)
      (num-test (asinh -0.0+0.0i) 0.0)
      (num-test (asinh 0.0-0.0i) 0.0)
      (num-test (asinh -0.0-0.0i) -0.0)
      (num-test (asinh 0.0+0.00000001i) 0.0+0.00000001i)
      (num-test (asinh -0.0+0.00000001i) 0.0+0.00000001i)
      (num-test (asinh 0.0-0.00000001i) 0.0-0.00000001i)
      (num-test (asinh -0.0-0.00000001i) 0.0-0.00000001i)
      (num-test (asinh 0.0+1.0i) 0.0+1.57079632679490i)
      (num-test (asinh -0.0+1.0i) 0.0+1.57079632679490i)
      (num-test (asinh 0.0-1.0i) 0.0-1.57079632679490i)
      (num-test (asinh -0.0-1.0i) 0.0-1.57079632679490i)
      (num-test (asinh 0.0+3.14159265358979i) 1.81152627246085+1.57079632679490i)
      (num-test (asinh -0.0+3.14159265358979i) 1.81152627246085+1.57079632679490i)
      (num-test (asinh 0.0-3.14159265358979i) 1.81152627246085-1.57079632679490i)
      (num-test (asinh -0.0-3.14159265358979i) -1.81152627246085-1.57079632679490i)
      (num-test (asinh 0.0+2.71828182845905i) 1.65745445415308+1.57079632679490i)
      (num-test (asinh -0.0+2.71828182845905i) 1.65745445415308+1.57079632679490i)
      (num-test (asinh 0.0-2.71828182845905i) 1.65745445415308-1.57079632679490i)
      (num-test (asinh -0.0-2.71828182845905i) -1.65745445415308-1.57079632679490i)
      (num-test (asinh 0.0+1234.0i) 7.81116322084923+1.57079632679490i)
      (num-test (asinh -0.0+1234.0i) 7.81116322084923+1.57079632679490i)
      (num-test (asinh 0.0-1234.0i) 7.81116322084923-1.57079632679490i)
      (num-test (asinh -0.0-1234.0i) -7.81116322068415-1.57079632679490i)
      (num-test (asinh 0.0+1234000000.0i) 21.62667394298955+1.57079632679490i)
      (num-test (asinh -0.0+1234000000.0i) 21.62667394298955+1.57079632679490i)
      (num-test (asinh 0.0-1234000000.0i) 21.62667394298955-1.57079632679490i)
      (num-test (asinh 0.00000001+0.0i) 0.00000001)
      (num-test (asinh -0.00000001+0.0i) -0.00000001)
      (num-test (asinh 0.00000001-0.0i) 0.00000001)
      (num-test (asinh -0.00000001-0.0i) -0.00000001)
      (num-test (asinh 0.00000001+0.00000001i) 0.00000001+0.00000001i)
      (num-test (asinh -0.00000001+0.00000001i) -0.00000001+0.00000001i)
      (num-test (asinh 0.00000001-0.00000001i) 0.00000001-0.00000001i)
      (num-test (asinh -0.00000001-0.00000001i) -0.00000001-0.00000001i)
      (num-test (asinh 0.00000001+1.0i) 0.00010000000008+1.57069632679498i)
      (num-test (asinh -0.00000001+1.0i) -0.00010000000008+1.57069632679498i)
      (num-test (asinh 0.00000001-1.0i) 0.00010000000008-1.57069632679498i)
      (num-test (asinh -0.00000001-1.0i) -0.00010000000008-1.57069632679498i)
      (num-test (asinh 0.00000001+3.14159265358979i) 1.81152627246085+1.57079632343715i)
      (num-test (asinh -0.00000001+3.14159265358979i) -1.81152627246085+1.57079632343715i)
      (num-test (asinh 0.00000001-3.14159265358979i) 1.81152627246085-1.57079632343715i)
      (num-test (asinh -0.00000001-3.14159265358979i) -1.81152627246085-1.57079632343715i)
      (num-test (asinh 0.00000001+2.71828182845905i) 1.65745445415308+1.57079632283867i)
      (num-test (asinh -0.00000001+2.71828182845905i) -1.65745445415308+1.57079632283867i)
      (num-test (asinh 0.00000001-2.71828182845905i) 1.65745445415308-1.57079632283867i)
      (num-test (asinh -0.00000001-2.71828182845905i) -1.65745445415308-1.57079632283867i)
      (num-test (asinh 0.00000001+1234.0i) 7.81116322084923+1.57079632678679i)
      (num-test (asinh -0.00000001+1234.0i) -7.81116322068415+1.57079632678679i)
      (num-test (asinh 0.00000001-1234.0i) 7.81116322084923-1.57079632678679i)
      (num-test (asinh -0.00000001-1234.0i) -7.81116322068415-1.57079632678679i)
      (num-test (asinh 0.00000001+1234000000.0i) 21.62667394298955+1.57079632679490i)
      (num-test (asinh 0.00000001-1234000000.0i) 21.62667394298955-1.57079632679490i)
      (num-test (asinh 1.0+0.0i) 0.88137358701954)
      (num-test (asinh -1.0+0.0i) -0.88137358701954)
      (num-test (asinh 1.0-0.0i) 0.88137358701954)
      (num-test (asinh -1.0-0.0i) -0.88137358701954)
      (num-test (asinh 1.0+0.00000001i) 0.88137358701954+0.00000000707107i)
      (num-test (asinh -1.0+0.00000001i) -0.88137358701954+0.00000000707107i)
      (num-test (asinh 1.0-0.00000001i) 0.88137358701954-0.00000000707107i)
      (num-test (asinh -1.0-0.00000001i) -0.88137358701954-0.00000000707107i)
      (num-test (asinh 1.0+1.0i) 1.06127506190504+0.66623943249252i)
      (num-test (asinh -1.0+1.0i) -1.06127506190504+0.66623943249252i)
      (num-test (asinh 1.0-1.0i) 1.06127506190504-0.66623943249252i)
      (num-test (asinh -1.0-1.0i) -1.06127506190504-0.66623943249252i)
      (num-test (asinh 1.0+3.14159265358979i) 1.86711439316026+1.24854303281344i)
      (num-test (asinh -1.0+3.14159265358979i) -1.86711439316026+1.24854303281344i)
      (num-test (asinh 1.0-3.14159265358979i) 1.86711439316026-1.24854303281344i)
      (num-test (asinh -1.0-3.14159265358979i) -1.86711439316026-1.24854303281344i)
      (num-test (asinh 1.0+2.71828182845905i) 1.73375471569385+1.19757808518581i)
      (num-test (asinh -1.0+2.71828182845905i) -1.73375471569385+1.19757808518581i)
      (num-test (asinh 1.0-2.71828182845905i) 1.73375471569385-1.19757808518581i)
      (num-test (asinh -1.0-2.71828182845905i) -1.73375471569385-1.19757808518581i)
      (num-test (asinh 1.0+1234.0i) 7.81116354920146+1.56998595393473i)
      (num-test (asinh -1.0+1234.0i) -7.81116354944842+1.56998595393442i)
      (num-test (asinh 1.0-1234.0i) 7.81116354920146-1.56998595393473i)
      (num-test (asinh -1.0-1234.0i) -7.81116354944842-1.56998595393442i)
      (num-test (asinh 1.0+1234000000.0i) 21.62667394298955+1.57079632598452i)
      (num-test (asinh 1.0-1234000000.0i) 21.62667394298955-1.57079632598452i)
      (num-test (asinh 3.14159265358979+0.0i) 1.86229574331085)
      (num-test (asinh -3.14159265358979+0.0i) -1.86229574331085)
      (num-test (asinh 3.14159265358979-0.0i) 1.86229574331085)
      (num-test (asinh -3.14159265358979-0.0i) -1.86229574331085)
      (num-test (asinh 3.14159265358979+0.00000001i) 1.86229574331085+0.00000000303314i)
      (num-test (asinh -3.14159265358979+0.00000001i) -1.86229574331085+0.00000000303314i)
      (num-test (asinh 3.14159265358979-0.00000001i) 1.86229574331085-0.00000000303314i)
      (num-test (asinh -3.14159265358979-0.00000001i) -1.86229574331085-0.00000000303314i)
      (num-test (asinh 3.14159265358979+1.0i) 1.90462768697066+0.29558503421163i)
      (num-test (asinh -3.14159265358979+1.0i) -1.90462768697066+0.29558503421163i)
      (num-test (asinh 3.14159265358979-1.0i) 1.90462768697066-0.29558503421163i)
      (num-test (asinh -3.14159265358979-1.0i) -1.90462768697066-0.29558503421163i)
      (num-test (asinh 3.14159265358979+3.14159265358979i) 2.18469104082751+0.77273977912748i)
      (num-test (asinh -3.14159265358979+3.14159265358979i) -2.18469104082751+0.77273977912748i)
      (num-test (asinh 3.14159265358979-3.14159265358979i) 2.18469104082751-0.77273977912748i)
      (num-test (asinh -3.14159265358979-3.14159265358979i) -2.18469104082751-0.77273977912748i)
      (num-test (asinh 3.14159265358979+2.71828182845905i) 2.11968336368048+0.69904796994655i)
      (num-test (asinh -3.14159265358979+2.71828182845905i) -2.11968336368048+0.69904796994656i)
      (num-test (asinh 3.14159265358979-2.71828182845905i) 2.11968336368048-0.69904796994655i)
      (num-test (asinh -3.14159265358979-2.71828182845905i) -2.11968336368048-0.69904796994656i)
      (num-test (asinh 3.14159265358979+1234.0i) 7.81116646154641+1.56825047031367i)
      (num-test (asinh -3.14159265358979+1234.0i) -7.81116646138554+1.56825047031506i)
      (num-test (asinh 3.14159265358979-1234.0i) 7.81116646154641-1.56825047031367i)
      (num-test (asinh -3.14159265358979-1234.0i) -7.81116646138554-1.56825047031506i)
      (num-test (asinh 3.14159265358979+1234000000.0i) 21.62667394298955+1.57079632424904i)
      (num-test (asinh 3.14159265358979-1234000000.0i) 21.62667394298955-1.57079632424904i)
      (num-test (asinh 2.71828182845905+0.0i) 1.72538255885231)
      (num-test (asinh -2.71828182845905+0.0i) -1.72538255885232)
      (num-test (asinh 2.71828182845905-0.0i) 1.72538255885231)
      (num-test (asinh -2.71828182845905-0.0i) -1.72538255885232)
      (num-test (asinh 2.71828182845905+0.00000001i) 1.72538255885231+0.00000000345258i)
      (num-test (asinh -2.71828182845905+0.00000001i) -1.72538255885232+0.00000000345258i)
      (num-test (asinh 2.71828182845905-0.00000001i) 1.72538255885231-0.00000000345258i)
      (num-test (asinh -2.71828182845905-0.00000001i) -1.72538255885232-0.00000000345258i)
      (num-test (asinh 2.71828182845905+1.0i) 1.77905438385935+0.33444277187637i)
      (num-test (asinh -2.71828182845905+1.0i) -1.77905438385935+0.33444277187637i)
      (num-test (asinh 2.71828182845905-1.0i) 1.77905438385935-0.33444277187637i)
      (num-test (asinh -2.71828182845905-1.0i) -1.77905438385935-0.33444277187637i)
      (num-test (asinh 2.71828182845905+3.14159265358979i) 2.11552790803290+0.84309656416035i)
      (num-test (asinh -2.71828182845905+3.14159265358979i) -2.11552790803290+0.84309656416035i)
      (num-test (asinh 2.71828182845905-3.14159265358979i) 2.11552790803290-0.84309656416035i)
      (num-test (asinh -2.71828182845905-3.14159265358979i) -2.11552790803290-0.84309656416035i)
      (num-test (asinh 2.71828182845905+2.71828182845905i) 2.04014932880026+0.76849735588475i)
      (num-test (asinh -2.71828182845905+2.71828182845905i) -2.04014932880026+0.76849735588475i)
      (num-test (asinh 2.71828182845905-2.71828182845905i) 2.04014932880026-0.76849735588475i)
      (num-test (asinh -2.71828182845905-2.71828182845905i) -2.04014932880026-0.76849735588475i)
      (num-test (asinh 2.71828182845905+1234.0i) 7.81116564705719+1.56859350805562i)
      (num-test (asinh -2.71828182845905+1234.0i) -7.81116564738434+1.56859350805543i)
      (num-test (asinh 2.71828182845905-1234.0i) 7.81116564705719-1.56859350805562i)
      (num-test (asinh -2.71828182845905-1234.0i) -7.81116564738434-1.56859350805543i)
      (num-test (asinh 2.71828182845905+1234000000.0i) 21.62667394298955+1.57079632459208i)
      (num-test (asinh 2.71828182845905-1234000000.0i) 21.62667394298955-1.57079632459208i)
      (num-test (asinh 1234.0+0.0i) 7.81116354920125)
      (num-test (asinh -1234.0+0.0i) -7.81116354896171)
      (num-test (asinh 1234.0-0.0i) 7.81116354920125)
      (num-test (asinh -1234.0-0.0i) -7.81116354896171)
      (num-test (asinh 1234.0+0.00000001i) 7.81116354920125+0.00000000000810i)
      (num-test (asinh -1234.0+0.00000001i) -7.81116354896171+0.00000000000810i)
      (num-test (asinh 1234.0-0.00000001i) 7.81116354920125-0.00000000000810i)
      (num-test (asinh -1234.0-0.00000001i) -7.81116354896171-0.00000000000810i)
      (num-test (asinh 1234.0+1.0i) 7.81116387755283+0.00081037232800i)
      (num-test (asinh -1234.0+1.0i) -7.81116387772663+0.00081037232806i)
      (num-test (asinh 1234.0-1.0i) 7.81116387755283-0.00081037232800i)
      (num-test (asinh -1234.0-1.0i) -7.81116387772663-0.00081037232806i)
      (num-test (asinh 1234.0+3.14159265358979i) 7.81116678989204+0.00254585480937i)
      (num-test (asinh -1234.0+3.14159265358979i) -7.81116678966949+0.00254585480900i)
      (num-test (asinh 1234.0-3.14159265358979i) 7.81116678989204-0.00254585480937i)
      (num-test (asinh -1234.0-3.14159265358979i) -7.81116678966949-0.00254585480900i)
      (num-test (asinh 1234.0+2.71828182845905i) 7.81116597540442+0.00220281729269i)
      (num-test (asinh -1234.0+2.71828182845905i) -7.81116597510553+0.00220281729236i)
      (num-test (asinh 1234.0-2.71828182845905i) 7.81116597540442-0.00220281729269i)
      (num-test (asinh -1234.0-2.71828182845905i) -7.81116597510553-0.00220281729236i)
      (num-test (asinh 1234.0+1234.0i) 8.15773697530526+0.78539808130944i)
      (num-test (asinh -1234.0+1234.0i) -8.15773697538346+0.78539808146835i)
      (num-test (asinh 1234.0-1234.0i) 8.15773697530526-0.78539808130944i)
      (num-test (asinh -1234.0-1234.0i) -8.15773697538346-0.78539808146835i)
      (num-test (asinh 1234.0+1234000000.0i) 21.62667394299005+1.57079532679490i)
      (num-test (asinh 1234.0-1234000000.0i) 21.62667394299005-1.57079532679490i)
      (num-test (asinh 1234000000.0+0.0i) 21.62667394298955)
      (num-test (asinh 1234000000.0-0.0i) 21.62667394298955)
      (num-test (asinh 1234000000.0+0.00000001i) 21.62667394298955+0.0i)
      (num-test (asinh 1234000000.0-0.00000001i) 21.62667394298955-0.0i)
      (num-test (asinh 1234000000.0+1.0i) 21.62667394298955+0.00000000081037i)
      (num-test (asinh 1234000000.0-1.0i) 21.62667394298955-0.00000000081037i)
      (num-test (asinh 1234000000.0+3.14159265358979i) 21.62667394298955+0.00000000254586i)
      (num-test (asinh 1234000000.0-3.14159265358979i) 21.62667394298955-0.00000000254586i)
      (num-test (asinh 1234000000.0+2.71828182845905i) 21.62667394298955+0.00000000220282i)
      (num-test (asinh 1234000000.0-2.71828182845905i) 21.62667394298955-0.00000000220282i)
      (num-test (asinh 1234000000.0+1234.0i) 21.62667394299005+0.00000100000000i)
      (num-test (asinh 1234000000.0-1234.0i) 21.62667394299005-0.00000100000000i)
      (num-test (asinh 1234000000.0+1234000000.0i) 21.97324753326953+0.78539816339745i)
      (num-test (asinh 1234000000.0-1234000000.0i) 21.97324753326953-0.78539816339745i)
      (num-test (asinh -1234000000/3) -20.52806165432143) 
      (num-test (asinh -500029/2) -13.12242137572639) 
      (num-test (asinh 0.00000001+1234000000.0i) +21.62667394298955+1.570796327200083i) 
      (num-test (asinh 3.14159265358979+1234000000.0i) +21.62667394298955+1.570796327200083i) 
      (num-test (asinh 2.71828182845905+1234000000.0i) +21.62667394298955+1.570796327200083i) 
      (num-test (asinh 1234000000.0+0.00000001i) +21.62667380877375+8.103725052149596E-18i) 
      (num-test (asinh 1234000000.0+2.71828182845905i) +21.62667380877375+2.2028207814967068E-9i)
      (num-test (asinh 0.0e+00+0.0e+00i) 0e0+0.0i)
      (num-test (asinh 0.0e+00+1.19209289550781250e-07i) 0+1.1920928955078153234e-7i)
      (num-test (asinh 0.0e+00-1.19209289550781250e-07i) 0-1.1920928955078153234e-7i)
      (num-test (asinh 0.0e+00+5.0e-01i) 0+5.2359877559829887308e-1i)
      (num-test (asinh 0.0e+00-5.0e-01i) 0-5.2359877559829887308e-1i)
      (num-test (asinh 0.0e+00+1.0e+00i) 0+1.5707963267948966192e0i)
      (num-test (asinh 0.0e+00-1.0e+00i) 0-1.5707963267948966192e0i)
      (num-test (asinh 0.0e+00+2.0e+00i) 1.3169578969248167086e0+1.5707963267948966192e0i)
      (num-test (asinh 0.0e+00-2.0e+00i) -1.3169578969248167086e0-1.5707963267948966192e0i)
      (num-test (asinh 0.0e+00+8.3886080e+06i) 1.6635532333438683873e1+1.5707963267948966192e0i)
      (num-test (asinh 0.0e+00-8.3886080e+06i) -1.6635532333438683873e1-1.5707963267948966192e0i)
      (num-test (asinh 1.19209289550781250e-07+0.0e+00i) 1.1920928955078096766e-7+0.0i)
      (num-test (asinh -1.19209289550781250e-07+0.0e+00i) -1.1920928955078096766e-7+0.0i)
      (num-test (asinh 1.19209289550781250e-07+1.19209289550781250e-07i) 1.1920928955078181469e-7+1.1920928955078068531e-7i)
      (num-test (asinh 1.19209289550781250e-07-1.19209289550781250e-07i) 1.1920928955078181469e-7-1.1920928955078068531e-7i)
      (num-test (asinh -1.19209289550781250e-07+1.19209289550781250e-07i) -1.1920928955078181469e-7+1.1920928955078068531e-7i)
      (num-test (asinh -1.19209289550781250e-07-1.19209289550781250e-07i) -1.1920928955078181469e-7-1.1920928955078068531e-7i)
      (num-test (asinh 1.19209289550781250e-07+5.0e-01i) 1.3765103082409432364e-7+5.2359877559829340332e-1i)
      (num-test (asinh 1.19209289550781250e-07-5.0e-01i) 1.3765103082409432364e-7-5.2359877559829340332e-1i)
      (num-test (asinh -1.19209289550781250e-07+5.0e-01i) -1.3765103082409432364e-7+5.2359877559829340332e-1i)
      (num-test (asinh -1.19209289550781250e-07-5.0e-01i) -1.3765103082409432364e-7-5.2359877559829340332e-1i)
      (num-test (asinh 1.19209289550781250e-07+1.0e+00i) 3.4526698643116312881e-4+1.5704510598153252947e0i)
      (num-test (asinh 1.19209289550781250e-07-1.0e+00i) 3.4526698643116312881e-4-1.5704510598153252947e0i)
      (num-test (asinh -1.19209289550781250e-07+1.0e+00i) -3.4526698643116312881e-4+1.5704510598153252947e0i)
      (num-test (asinh -1.19209289550781250e-07-1.0e+00i) -3.4526698643116312881e-4-1.5704510598153252947e0i)
      (num-test (asinh 1.19209289550781250e-07+2.0e+00i) 1.3169578969248194435e0+1.5707962579693812072e0i)
      (num-test (asinh 1.19209289550781250e-07-2.0e+00i) 1.3169578969248194435e0-1.5707962579693812072e0i)
      (num-test (asinh -1.19209289550781250e-07+2.0e+00i) -1.3169578969248194435e0+1.5707962579693812072e0i)
      (num-test (asinh -1.19209289550781250e-07-2.0e+00i) -1.3169578969248194435e0-1.5707962579693812072e0i)
      (num-test (asinh 1.19209289550781250e-07+8.3886080e+06i) 1.6635532333438683873e1+1.5707963267948824084e0i)
      (num-test (asinh 1.19209289550781250e-07-8.3886080e+06i) 1.6635532333438683873e1-1.5707963267948824084e0i)
      (num-test (asinh -1.19209289550781250e-07+8.3886080e+06i) -1.6635532333438683873e1+1.5707963267948824084e0i)
      (num-test (asinh -1.19209289550781250e-07-8.3886080e+06i) -1.6635532333438683873e1-1.5707963267948824084e0i)
      (num-test (asinh 5.0e-01+0.0e+00i) 4.8121182505960344750e-1+0.0i)
      (num-test (asinh -5.0e-01+0.0e+00i) -4.8121182505960344750e-1+0.0i)
      (num-test (asinh 5.0e-01+1.19209289550781250e-07i) 4.8121182505960598961e-1+1.0662402999400097805e-7i)
      (num-test (asinh 5.0e-01-1.19209289550781250e-07i) 4.8121182505960598961e-1-1.0662402999400097805e-7i)
      (num-test (asinh -5.0e-01+1.19209289550781250e-07i) -4.8121182505960598961e-1+1.0662402999400097805e-7i)
      (num-test (asinh -5.0e-01-1.19209289550781250e-07i) -4.8121182505960598961e-1-1.0662402999400097805e-7i)
      (num-test (asinh 5.0e-01+5.0e-01i) 5.3063753095251782602e-1+4.5227844715119068206e-1i)
      (num-test (asinh 5.0e-01-5.0e-01i) 5.3063753095251782602e-1-4.5227844715119068206e-1i)
      (num-test (asinh -5.0e-01+5.0e-01i) -5.3063753095251782602e-1+4.5227844715119068206e-1i)
      (num-test (asinh -5.0e-01-5.0e-01i) -5.3063753095251782602e-1-4.5227844715119068206e-1i)
      (num-test (asinh 5.0e-01+1.0e+00i) 7.3285767597364526089e-1+8.9590748120889023907e-1i)
      (num-test (asinh 5.0e-01-1.0e+00i) 7.3285767597364526089e-1-8.9590748120889023907e-1i)
      (num-test (asinh -5.0e-01+1.0e+00i) -7.3285767597364526089e-1+8.9590748120889023907e-1i)
      (num-test (asinh -5.0e-01-1.0e+00i) -7.3285767597364526089e-1-8.9590748120889023907e-1i)
      (num-test (asinh 5.0e-01+2.0e+00i) 1.3618009008578457882e0+1.2930420702371826591e0i)
      (num-test (asinh 5.0e-01-2.0e+00i) 1.3618009008578457882e0-1.2930420702371826591e0i)
      (num-test (asinh -5.0e-01+2.0e+00i) -1.3618009008578457882e0+1.2930420702371826591e0i)
      (num-test (asinh -5.0e-01-2.0e+00i) -1.3618009008578457882e0-1.2930420702371826591e0i)
      (num-test (asinh 5.0e-01+8.3886080e+06i) 1.6635532333438685650e1+1.5707962671902518438e0i)
      (num-test (asinh 5.0e-01-8.3886080e+06i) 1.6635532333438685650e1-1.5707962671902518438e0i)
      (num-test (asinh -5.0e-01+8.3886080e+06i) -1.6635532333438685650e1+1.5707962671902518438e0i)
      (num-test (asinh -5.0e-01-8.3886080e+06i) -1.6635532333438685650e1-1.5707962671902518438e0i)
      (num-test (asinh 1.0e+00+0.0e+00i) 8.8137358701954302523e-1+0.0i)
      (num-test (asinh -1.0e+00+0.0e+00i) -8.8137358701954302523e-1+0.0i)
      (num-test (asinh 1.0e+00+1.19209289550781250e-07i) 8.8137358701954553738e-1+8.4293697021788013662e-8i)
      (num-test (asinh 1.0e+00-1.19209289550781250e-07i) 8.8137358701954553738e-1-8.4293697021788013662e-8i)
      (num-test (asinh -1.0e+00+1.19209289550781250e-07i) -8.8137358701954553738e-1+8.4293697021788013662e-8i)
      (num-test (asinh -1.0e+00-1.19209289550781250e-07i) -8.8137358701954553738e-1-8.4293697021788013662e-8i)
      (num-test (asinh 1.0e+00+5.0e-01i) 9.2613303135018242455e-1+3.4943906285721329363e-1i)
      (num-test (asinh 1.0e+00-5.0e-01i) 9.2613303135018242455e-1-3.4943906285721329363e-1i)
      (num-test (asinh -1.0e+00+5.0e-01i) -9.2613303135018242455e-1+3.4943906285721329363e-1i)
      (num-test (asinh -1.0e+00-5.0e-01i) -9.2613303135018242455e-1-3.4943906285721329363e-1i)
      (num-test (asinh 1.0e+00+1.0e+00i) 1.0612750619050356520e0+6.6623943249251525510e-1i)
      (num-test (asinh 1.0e+00-1.0e+00i) 1.0612750619050356520e0-6.6623943249251525510e-1i)
      (num-test (asinh -1.0e+00+1.0e+00i) -1.0612750619050356520e0+6.6623943249251525510e-1i)
      (num-test (asinh -1.0e+00-1.0e+00i) -1.0612750619050356520e0-6.6623943249251525510e-1i)
      (num-test (asinh 1.0e+00+2.0e+00i) 1.4693517443681852733e0+1.0634400235777520562e0i)
      (num-test (asinh 1.0e+00-2.0e+00i) 1.4693517443681852733e0-1.0634400235777520562e0i)
      (num-test (asinh -1.0e+00+2.0e+00i) -1.4693517443681852733e0+1.0634400235777520562e0i)
      (num-test (asinh -1.0e+00-2.0e+00i) -1.4693517443681852733e0-1.0634400235777520562e0i)
      (num-test (asinh 1.0e+00+8.3886080e+06i) 1.6635532333438690979e1+1.5707962075856070684e0i)
      (num-test (asinh 1.0e+00-8.3886080e+06i) 1.6635532333438690979e1-1.5707962075856070684e0i)
      (num-test (asinh -1.0e+00+8.3886080e+06i) -1.6635532333438690979e1+1.5707962075856070684e0i)
      (num-test (asinh -1.0e+00-8.3886080e+06i) -1.6635532333438690979e1-1.5707962075856070684e0i)
      (num-test (asinh 2.0e+00+0.0e+00i) 1.4436354751788103425e0+0.0i)
      (num-test (asinh -2.0e+00+0.0e+00i) -1.4436354751788103425e0+0.0i)
      (num-test (asinh 2.0e+00+1.19209289550781250e-07i) 1.4436354751788116136e0+5.3312014997000413263e-8i)
      (num-test (asinh 2.0e+00-1.19209289550781250e-07i) 1.4436354751788116136e0-5.3312014997000413263e-8i)
      (num-test (asinh -2.0e+00+1.19209289550781250e-07i) -1.4436354751788116136e0+5.3312014997000413263e-8i)
      (num-test (asinh -2.0e+00-1.19209289550781250e-07i) -1.4436354751788116136e0-5.3312014997000413263e-8i)
      (num-test (asinh 2.0e+00+5.0e-01i) 1.4657153519472905218e0+2.2101863562288385890e-1i)
      (num-test (asinh 2.0e+00-5.0e-01i) 1.4657153519472905218e0-2.2101863562288385890e-1i)
      (num-test (asinh -2.0e+00+5.0e-01i) -1.4657153519472905218e0+2.2101863562288385890e-1i)
      (num-test (asinh -2.0e+00-5.0e-01i) -1.4657153519472905218e0-2.2101863562288385890e-1i)
      (num-test (asinh 2.0e+00+1.0e+00i) 1.5285709194809981613e0+4.2707858639247612548e-1i)
      (num-test (asinh 2.0e+00-1.0e+00i) 1.5285709194809981613e0-4.2707858639247612548e-1i)
      (num-test (asinh -2.0e+00+1.0e+00i) -1.5285709194809981613e0+4.2707858639247612548e-1i)
      (num-test (asinh -2.0e+00-1.0e+00i) -1.5285709194809981613e0-4.2707858639247612548e-1i)
      (num-test (asinh 2.0e+00+2.0e+00i) 1.7343245214879664480e0+7.5424914469804604071e-1i)
      (num-test (asinh 2.0e+00-2.0e+00i) 1.7343245214879664480e0-7.5424914469804604071e-1i)
      (num-test (asinh -2.0e+00+2.0e+00i) -1.7343245214879664480e0+7.5424914469804604071e-1i)
      (num-test (asinh -2.0e+00-2.0e+00i) -1.7343245214879664480e0-7.5424914469804604071e-1i)
      (num-test (asinh 2.0e+00+8.3886080e+06i) 1.6635532333438712295e1+1.5707960883763175177e0i)
      (num-test (asinh 2.0e+00-8.3886080e+06i) 1.6635532333438712295e1-1.5707960883763175177e0i)
      (num-test (asinh -2.0e+00+8.3886080e+06i) -1.6635532333438712295e1+1.5707960883763175177e0i)
      (num-test (asinh -2.0e+00-8.3886080e+06i) -1.6635532333438712295e1-1.5707960883763175177e0i)
      (num-test (asinh 8.3886080e+06+0.0e+00i) 1.6635532333438690979e1+0.0i)
      (num-test (asinh -8.3886080e+06+0.0e+00i) -1.6635532333438690979e1+0.0i)
      (num-test (asinh 8.3886080e+06+1.19209289550781250e-07i) 1.6635532333438690979e1+1.4210854715201902743e-14i)
      (num-test (asinh 8.3886080e+06-1.19209289550781250e-07i) 1.6635532333438690979e1-1.4210854715201902743e-14i)
      (num-test (asinh -8.3886080e+06+1.19209289550781250e-07i) -1.6635532333438690979e1+1.4210854715201902743e-14i)
      (num-test (asinh -8.3886080e+06-1.19209289550781250e-07i) -1.6635532333438690979e1-1.4210854715201902743e-14i)
      (num-test (asinh 8.3886080e+06+5.0e-01i) 1.6635532333438692755e1+5.9604644775390130897e-8i)
      (num-test (asinh 8.3886080e+06-5.0e-01i) 1.6635532333438692755e1-5.9604644775390130897e-8i)
      (num-test (asinh -8.3886080e+06+5.0e-01i) -1.6635532333438692755e1+5.9604644775390130897e-8i)
      (num-test (asinh -8.3886080e+06-5.0e-01i) -1.6635532333438692755e1-5.9604644775390130897e-8i)
      (num-test (asinh 8.3886080e+06+1.0e+00i) 1.6635532333438698084e1+1.1920928955077983828e-7i)
      (num-test (asinh 8.3886080e+06-1.0e+00i) 1.6635532333438698084e1-1.1920928955077983828e-7i)
      (num-test (asinh -8.3886080e+06+1.0e+00i) -1.6635532333438698084e1+1.1920928955077983828e-7i)
      (num-test (asinh -8.3886080e+06-1.0e+00i) -1.6635532333438698084e1-1.1920928955077983828e-7i)
      (num-test (asinh 8.3886080e+06+2.0e+00i) 1.663553233343871940e1+2.3841857910155628843e-7i)
      (num-test (asinh 8.3886080e+06-2.0e+00i) 1.663553233343871940e1-2.3841857910155628843e-7i)
      (num-test (asinh -8.3886080e+06+2.0e+00i) -1.663553233343871940e1+2.3841857910155628843e-7i)
      (num-test (asinh -8.3886080e+06-2.0e+00i) -1.663553233343871940e1-2.3841857910155628843e-7i)
      (num-test (asinh 8.3886080e+06+8.3886080e+06i) 1.6982105923718660081e1+7.8539816339744653326e-1i)
      (num-test (asinh 8.3886080e+06-8.3886080e+06i) 1.6982105923718660081e1-7.8539816339744653326e-1i)
      (num-test (asinh -8.3886080e+06+8.3886080e+06i) -1.6982105923718660081e1+7.8539816339744653326e-1i)
      (num-test (asinh -8.3886080e+06-8.3886080e+06i) -1.6982105923718660081e1-7.8539816339744653326e-1i)


      ;; -------- acosh
      (num-test (acosh 0) 0.0+1.57079632679490i)
      (num-test (acosh 1) 0.0)
      (num-test (acosh -1) 0.0+3.14159265358979i)
      (num-test (acosh 2) 1.31695789692482)
      (num-test (acosh -2) 1.31695789692482+3.14159265358979i)
      (num-test (acosh 3) 1.76274717403909)
      (num-test (acosh -3) 1.76274717403909+3.14159265358979i)
      (num-test (acosh 10) 2.99322284612638)
      (num-test (acosh -10) 2.99322284612638+3.14159265358979i)
      (num-test (acosh 1234) 7.81116322084923)
      (num-test (acosh -1234) 7.81116322084923+3.14159265358979i)
      (num-test (acosh 1234000000) 21.62667394298955)
      (num-test (acosh -1234000000) 21.62667394298955+3.14159265358979i)
      (num-test (acosh 500029) 13.81556855628134)
      (num-test (acosh -500029) 13.81556855628134+3.14159265358979i)
      (num-test (acosh 362880) 13.49497466063952)
      (num-test (acosh -362880) 13.49497466063952+3.14159265358979i)
      (num-test (acosh 0/1) 0.0+1.57079632679490i)
      (num-test (acosh 0/2) 0.0+1.57079632679490i)
      (num-test (acosh 0/3) 0.0+1.57079632679490i)
      (num-test (acosh 0/10) 0.0+1.57079632679490i)
      (num-test (acosh 0/1234) 0.0+1.57079632679490i)
      (num-test (acosh 0/1234000000) 0.0+1.57079632679490i)
      (num-test (acosh 0/500029) 0.0+1.57079632679490i)
      (num-test (acosh 0/362880) 0.0+1.57079632679490i)
      (num-test (acosh 1/1) 0.0)
      (num-test (acosh -1/1) 0.0+3.14159265358979i)
      (num-test (acosh 1/2) 0.0+1.04719755119660i)
      (num-test (acosh -1/2) 0.0+2.09439510239320i)
      (num-test (acosh 1/3) 0.0+1.23095941734077i)
      (num-test (acosh -1/3) 0.0+1.91063323624902i)
      (num-test (acosh 1/10) 0.0+1.47062890563334i)
      (num-test (acosh -1/10) 0.0+1.67096374795646i)
      (num-test (acosh 1/1234) 0.0+1.56998595393473i)
      (num-test (acosh -1/1234) 0.0+1.57160669965507i)
      (num-test (acosh 1/1234000000) 0.0+1.57079632598452i)
      (num-test (acosh -1/1234000000) 0.0+1.57079632760527i)
      (num-test (acosh 1/500029) 0.0+1.57079432691089i)
      (num-test (acosh -1/500029) 0.0+1.57079832667890i)
      (num-test (acosh 1/362880) 0.0+1.57079357106297i)
      (num-test (acosh -1/362880) 0.0+1.57079908252682i)
      (num-test (acosh 2/1) 1.31695789692482)
      (num-test (acosh -2/1) 1.31695789692482+3.14159265358979i)
      (num-test (acosh 2/2) 0.0)
      (num-test (acosh -2/2) 0.0+3.14159265358979i)
      (num-test (acosh 2/3) 0.0+0.84106867056793i)
      (num-test (acosh -2/3) 0.0+2.30052398302186i)
      (num-test (acosh 2/10) 0.0+1.36943840600457i)
      (num-test (acosh -2/10) 0.0+1.77215424758523i)
      (num-test (acosh 2/1234) 0.0+1.56917558054238i)
      (num-test (acosh -2/1234) 0.0+1.57241707304741i)
      (num-test (acosh 2/1234000000) 0.0+1.57079632517415i)
      (num-test (acosh -2/1234000000) 0.0+1.57079632841564i)
      (num-test (acosh 2/500029) 0.0+1.57079232702688i)
      (num-test (acosh -2/500029) 0.0+1.57080032656291i)
      (num-test (acosh 2/362880) 0.0+1.57079081533105i)
      (num-test (acosh -2/362880) 0.0+1.57080183825874i)
      (num-test (acosh 3/1) 1.76274717403909)
      (num-test (acosh -3/1) 1.76274717403909+3.14159265358979i)
      (num-test (acosh 3/2) 0.96242365011921)
      (num-test (acosh -3/2) 0.96242365011921+3.14159265358979i)
      (num-test (acosh 3/3) 0.0)
      (num-test (acosh -3/3) 0.0+3.14159265358979i)
      (num-test (acosh 3/10) 0.0+1.26610367277950i)
      (num-test (acosh -3/10) 0.0+1.87548898081029i)
      (num-test (acosh 3/1234) 0.0+1.56836520608568i)
      (num-test (acosh -3/1234) 0.0+1.57322744750412i)
      (num-test (acosh 3/1234000000) 0.0+1.57079632436378i)
      (num-test (acosh -3/1234000000) 0.0+1.57079632922601i)
      (num-test (acosh 3/500029) 0.0+1.57079032714288i)
      (num-test (acosh -3/500029) 0.0+1.57080232644692i)
      (num-test (acosh 3/362880) 0.0-1.57078805959913i)
      (num-test (acosh -3/362880) 0.0-1.57080459399066i)
      (num-test (acosh 10/1) 2.99322284612638)
      (num-test (acosh -10/1) 2.99322284612638+3.14159265358979i)
      (num-test (acosh 10/2) 2.29243166956118)
      (num-test (acosh -10/2) 2.29243166956118+3.14159265358979i)
      (num-test (acosh 10/3) 1.87382024252741)
      (num-test (acosh -10/3) 1.87382024252741+3.14159265358979i)
      (num-test (acosh 10/10) 0.0)
      (num-test (acosh -10/10) 0.0+3.14159265358979i)
      (num-test (acosh 10/1234) 0.0+1.56269251038168i)
      (num-test (acosh -10/1234) 0.0+1.57890014320811i)
      (num-test (acosh 10/1234000000) 0.0+1.57079631869117i)
      (num-test (acosh -10/1234000000) 0.0+1.57079633489862i)
      (num-test (acosh 10/500029) 0.0+1.57077632795483i)
      (num-test (acosh -10/500029) 0.0+1.57081632563497i)
      (num-test (acosh 10/362880) 0.0+1.57076876947567i)
      (num-test (acosh -10/362880) 0.0+1.57082388411412i)
      (num-test (acosh 1234/1) 7.81116322084923)
      (num-test (acosh -1234/1) 7.81116322084923+3.14159265358979i)
      (num-test (acosh 1234/2) 7.11801554776066)
      (num-test (acosh -1234/2) 7.11801554776066+3.14159265358979i)
      (num-test (acosh 1234/3) 6.71254961876983)
      (num-test (acosh -1234/3) 6.71254961876983+3.14159265358979i)
      (num-test (acosh 1234/10) 5.50856187402619)
      (num-test (acosh -1234/10) 5.50856187402619+3.14159265358979i)
      (num-test (acosh 1234/1234) 0.0)
      (num-test (acosh -1234/1234) 0.0+3.14159265358979i)
      (num-test (acosh 1234/1234000000) 0.0+1.57079532679490i)
      (num-test (acosh -1234/1234000000) 0.0+1.57079732679490i)
      (num-test (acosh 1234/500029) 0.0+1.56832846742558i)
      (num-test (acosh -1234/500029) 0.0+1.57326418616421i)
      (num-test (acosh 1234/362880) 0.0+1.56739574704864i)
      (num-test (acosh -1234/362880) 0.0+1.57419690654115i)
      (num-test (acosh 1234000000/1) 21.62667394298955)
      (num-test (acosh -1234000000/1) 21.62667394298955+3.14159265358979i)
      (num-test (acosh 1234000000/2) 20.93352676242961)
      (num-test (acosh -1234000000/2) 20.93352676242961+3.14159265358979i)
      (num-test (acosh 1234000000/3) 20.52806165432144)
      (num-test (acosh -1234000000/3) 20.52806165432144+3.14159265358979i)
      (num-test (acosh 1234000000/10) 19.32408884999551)
      (num-test (acosh -1234000000/10) 19.32408884999551+3.14159265358979i)
      (num-test (acosh 1234000000/1234) 14.50865773852397)
      (num-test (acosh -1234000000/1234) 14.50865773852397+3.14159265358979i)
      (num-test (acosh 1234000000/1234000000) 0.0)
      (num-test (acosh -1234000000/1234000000) 0.0+3.14159265358979i)
      (num-test (acosh 1234000000/500029) 8.50425252621839)
      (num-test (acosh -1234000000/500029) 8.50425252621839+3.14159265358979i)
      (num-test (acosh 1234000000/362880) 8.82484644128907)
      (num-test (acosh -1234000000/362880) 8.82484644128907+3.14159265358979i)
      (num-test (acosh 500029/1) 13.81556855628134)
      (num-test (acosh -500029/1) 13.81556855628134+3.14159265358979i)
      (num-test (acosh 500029/2) 13.12242137571839)
      (num-test (acosh -500029/2) 13.12242137571839+3.14159265358979i)
      (num-test (acosh 500029/3) 12.71695626760523)
      (num-test (acosh -500029/3) 12.71695626760523+3.14159265358979i)
      (num-test (acosh 500029/10) 11.51298346318831)
      (num-test (acosh -500029/10) 11.51298346318831+3.14159265358979i)
      (num-test (acosh 500029/1234) 6.69755082923415)
      (num-test (acosh -500029/1234) 6.69755082923415+3.14159265358979i)
      (num-test (acosh 500029/1234000000) 0.0+1.57039111689726i)
      (num-test (acosh -500029/1234000000) 0.0+1.57120153669253i)
      (num-test (acosh 500029/500029) 0.0)
      (num-test (acosh -500029/500029) 0.0+3.14159265358979i)
      (num-test (acosh 500029/362880) 0.84413377868249)
      (num-test (acosh -500029/362880) 0.84413377868249+3.14159265358979i)
      (num-test (acosh 362880/1) 13.49497466063952)
      (num-test (acosh -362880/1) 13.49497466063952+3.14159265358979i)
      (num-test (acosh 362880/2) 12.80182748007388)
      (num-test (acosh -362880/2) 12.80182748007388+3.14159265358979i)
      (num-test (acosh 362880/3) 12.39636237195622)
      (num-test (acosh -362880/3) 12.39636237195622+3.14159265358979i)
      (num-test (acosh 362880/10) 11.19238956745752)
      (num-test (acosh -362880/10) 11.19238956745752+3.14159265358979i)
      (num-test (acosh 362880/1234) 6.37695556518904)
      (num-test (acosh -362880/1234) 6.37695556518904+3.14159265358979i)
      (num-test (acosh 362880/1234000000) 0.0+1.57050225871935i)
      (num-test (acosh -362880/1234000000) 0.0+1.57109039487045i)
      (num-test (acosh 362880/500029) 0.0+0.75871902295910i)
      (num-test (acosh -362880/500029) 0.0+2.38287363063069i)
      (num-test (acosh 362880/362880) 0.0)
      (num-test (acosh -362880/362880) 0.0+3.14159265358979i)
      (num-test (acosh 0.0) 0.0+1.57079632679490i)
      (num-test (acosh 0.00000001) 0.0+1.57079631679490i)
      (num-test (acosh -0.00000001) 0.0+1.57079633679490i)
      (num-test (acosh 1.0) 0.0)
      (num-test (acosh -1.0) 0.0+3.14159265358979i)
      (num-test (acosh our-pi) 1.81152627246085)
      (num-test (acosh -3.14159265358979) 1.81152627246085+3.14159265358979i)
      (num-test (acosh 2.71828182845905) 1.65745445415308)
      (num-test (acosh -2.71828182845905) 1.65745445415308+3.14159265358979i)
      (num-test (acosh 1234.0) 7.81116322084923)
      (num-test (acosh -1234.0) 7.81116322084923+3.14159265358979i)
      (num-test (acosh 1234000000.0) 21.62667394298955)
      (num-test (acosh -1234000000.0) 21.62667394298955+3.14159265358979i)
      (num-test (acosh 0.0+0.0i) 0.0+1.57079632679490i)
      (num-test (acosh -0.0+0.0i) 0.0+1.57079632679490i)
      (num-test (acosh 0.0-0.0i) 0.0+1.57079632679490i)
      (num-test (acosh -0.0-0.0i) 0.0+1.57079632679490i)
      (num-test (acosh 0.0+0.00000001i) 0.00000001+1.57079632679490i)
      (num-test (acosh -0.0+0.00000001i) 0.00000001+1.57079632679490i)
      (num-test (acosh 0.0-0.00000001i) 0.00000001-1.57079632679490i)
      (num-test (acosh -0.0-0.00000001i) 0.00000001-1.57079632679490i)
      (num-test (acosh 0.0+1.0i) 0.88137358701954+1.57079632679490i)
      (num-test (acosh -0.0+1.0i) 0.88137358701954+1.57079632679490i)
      (num-test (acosh 0.0-1.0i) 0.88137358701954-1.57079632679490i)
      (num-test (acosh -0.0-1.0i) 0.88137358701954-1.57079632679490i)
      (num-test (acosh 0.0+3.14159265358979i) 1.86229574331085+1.57079632679490i)
      (num-test (acosh -0.0+3.14159265358979i) 1.86229574331085+1.57079632679490i)
      (num-test (acosh 0.0-3.14159265358979i) 1.86229574331085-1.57079632679490i)
      (num-test (acosh -0.0-3.14159265358979i) 1.86229574331085-1.57079632679490i)
      (num-test (acosh 0.0+2.71828182845905i) 1.72538255885231+1.57079632679490i)
      (num-test (acosh -0.0+2.71828182845905i) 1.72538255885231+1.57079632679490i)
      (num-test (acosh 0.0-2.71828182845905i) 1.72538255885231-1.57079632679490i)
      (num-test (acosh -0.0-2.71828182845905i) 1.72538255885232-1.57079632679490i)
      (num-test (acosh 0.0+1234.0i) 7.81116354920125+1.57079632679490i)
      (num-test (acosh -0.0+1234.0i) 7.81116354920125+1.57079632679490i)
      (num-test (acosh 0.0-1234.0i) 7.81116354920125-1.57079632679490i)
      (num-test (acosh -0.0-1234.0i) 7.81116354896171-1.57079632679490i)
      (num-test (acosh 0.0+1234000000.0i) 21.62667394298955+1.57079632679490i)
      (num-test (acosh -0.0+1234000000.0i) 21.62667394298955+1.57079632679490i)
      (num-test (acosh 0.0-1234000000.0i) 21.62667394298955-1.57079632679490i)
      (num-test (acosh 0.00000001+0.0i) 0.0+1.57079631679490i)
      (num-test (acosh -0.00000001+0.0i) 0.0+1.57079633679490i)
      (num-test (acosh 0.00000001-0.0i) 0.0+1.57079631679490i)
      (num-test (acosh -0.00000001-0.0i) 0.0+1.57079633679490i)
      (num-test (acosh 0.00000001+0.00000001i) 0.00000001+1.57079631679490i)
      (num-test (acosh -0.00000001+0.00000001i) 0.00000001+1.57079633679490i)
      (num-test (acosh 0.00000001-0.00000001i) 0.00000001-1.57079631679490i)
      (num-test (acosh -0.00000001-0.00000001i) 0.00000001-1.57079633679490i)
      (num-test (acosh 0.00000001+1.0i) 0.88137358701954+1.57079631972383i)
      (num-test (acosh -0.00000001+1.0i) 0.88137358701954+1.57079633386596i)
      (num-test (acosh 0.00000001-1.0i) 0.88137358701954-1.57079631972383i)
      (num-test (acosh -0.00000001-1.0i) 0.88137358701954-1.57079633386596i)
      (num-test (acosh 0.00000001+3.14159265358979i) 1.86229574331085+1.57079632376175i)
      (num-test (acosh -0.00000001+3.14159265358979i) 1.86229574331085+1.57079632982804i)
      (num-test (acosh 0.00000001-3.14159265358979i) 1.86229574331085-1.57079632376175i)
      (num-test (acosh -0.00000001-3.14159265358979i) 1.86229574331085-1.57079632982804i)
      (num-test (acosh 0.00000001+2.71828182845905i) 1.72538255885231+1.57079632334232i)
      (num-test (acosh -0.00000001+2.71828182845905i) 1.72538255885231+1.57079633024747i)
      (num-test (acosh 0.00000001-2.71828182845905i) 1.72538255885231-1.57079632334232i)
      (num-test (acosh -0.00000001-2.71828182845905i) 1.72538255885231-1.57079633024747i)
      (num-test (acosh 0.00000001+1234.0i) 7.81116354920125+1.57079632678679i)
      (num-test (acosh -0.00000001+1234.0i) 7.81116354920125+1.57079632680300i)
      (num-test (acosh 0.00000001-1234.0i) 7.81116354920125-1.57079632678679i)
      (num-test (acosh -0.00000001-1234.0i) 7.81116354920125-1.57079632680300i)
      (num-test (acosh 0.00000001+1234000000.0i) 21.62667394298955+1.57079632679490i)
      (num-test (acosh -0.00000001+1234000000.0i) 21.62667394298955+1.57079632679490i)
      (num-test (acosh 0.00000001-1234000000.0i) 21.62667394298955-1.57079632679490i)
      (num-test (acosh -0.00000001-1234000000.0i) 21.62667394298955-1.57079632679490i)
      (num-test (acosh 1.0+0.0i) 0.0)
      (num-test (acosh -1.0+0.0i) 0.0+3.14159265358979i)
      (num-test (acosh 1.0-0.0i) 0.0)
      (num-test (acosh -1.0-0.0i) 0.0+3.14159265358979i)
      (num-test (acosh 1.0+0.00000001i) 0.00010000000008+0.00009999999992i)
      (num-test (acosh -1.0+0.00000001i) 0.00010000000008+3.14149265358988i)
      (num-test (acosh 1.0-0.00000001i) 0.00010000000008-0.00009999999992i)
      (num-test (acosh -1.0-0.00000001i) 0.00010000000008-3.14149265358988i)
      (num-test (acosh 1.0+1.0i) 1.06127506190504+0.90455689430238i)
      (num-test (acosh -1.0+1.0i) 1.06127506190504+2.23703575928741i)
      (num-test (acosh 1.0-1.0i) 1.06127506190504-0.90455689430238i)
      (num-test (acosh -1.0-1.0i) 1.06127506190504-2.23703575928741i)
      (num-test (acosh 1.0+3.14159265358979i) 1.90462768697066+1.27521129258327i)
      (num-test (acosh -1.0+3.14159265358979i) 1.90462768697066+1.86638136100653i)
      (num-test (acosh 1.0-3.14159265358979i) 1.90462768697066-1.27521129258327i)
      (num-test (acosh -1.0-3.14159265358979i) 1.90462768697066-1.86638136100653i)
      (num-test (acosh 1.0+2.71828182845905i) 1.77905438385935+1.23635355491853i)
      (num-test (acosh -1.0+2.71828182845905i) 1.77905438385935+1.90523909867127i)
      (num-test (acosh 1.0-2.71828182845905i) 1.77905438385935-1.23635355491853i)
      (num-test (acosh -1.0-2.71828182845905i) 1.77905438385935-1.90523909867127i)
      (num-test (acosh 1.0+1234.0i) 7.81116387755283+1.56998595446690i)
      (num-test (acosh -1.0+1234.0i) 7.81116387755283+1.57160669912289i)
      (num-test (acosh 1.0-1234.0i) 7.81116387755283-1.56998595446690i)
      (num-test (acosh -1.0-1234.0i) 7.81116387755283-1.57160669912289i)
      (num-test (acosh 1.0+1234000000.0i) 21.62667394298955+1.57079632598452i)
      (num-test (acosh -1.0+1234000000.0i) 21.62667394298955+1.57079632760527i)
      (num-test (acosh 1.0-1234000000.0i) 21.62667394298955-1.57079632598452i)
      (num-test (acosh -1.0-1234000000.0i) 21.62667394298955-1.57079632760527i)
      (num-test (acosh 3.14159265358979+0.0i) 1.81152627246085)
      (num-test (acosh -3.14159265358979+0.0i) 1.81152627246085+3.14159265358979i)
      (num-test (acosh 3.14159265358979-0.0i) 1.81152627246085)
      (num-test (acosh -3.14159265358979-0.0i) 1.81152627246085+3.14159265358979i)
      (num-test (acosh 3.14159265358979+0.00000001i) 1.81152627246085+0.00000000335775i)
      (num-test (acosh -3.14159265358979+0.00000001i) 1.81152627246085+3.14159265023205i)
      (num-test (acosh 3.14159265358979-0.00000001i) 1.81152627246085-0.00000000335775i)
      (num-test (acosh -3.14159265358979-0.00000001i) 1.81152627246085-3.14159265023205i)
      (num-test (acosh 3.14159265358979+1.0i) 1.86711439316026+0.32225329398146i)
      (num-test (acosh -3.14159265358979+1.0i) 1.86711439316026+2.81933935960833i)
      (num-test (acosh 3.14159265358979-1.0i) 1.86711439316026-0.32225329398146i)
      (num-test (acosh -3.14159265358979-1.0i) 1.86711439316026-2.81933935960833i)
      (num-test (acosh 3.14159265358979+3.14159265358979i) 2.18469104082751+0.79805654766741i)
      (num-test (acosh -3.14159265358979+3.14159265358979i) 2.18469104082751+2.34353610592238i)
      (num-test (acosh 3.14159265358979-3.14159265358979i) 2.18469104082751-0.79805654766741i)
      (num-test (acosh -3.14159265358979-3.14159265358979i) 2.18469104082751-2.34353610592238i)
      (num-test (acosh 3.14159265358979+2.71828182845905i) 2.11552790803290+0.72769976263454i)
      (num-test (acosh -3.14159265358979+2.71828182845905i) 2.11552790803290+2.41389289095525i)
      (num-test (acosh 3.14159265358979-2.71828182845905i) 2.11552790803290-0.72769976263454i)
      (num-test (acosh -3.14159265358979-2.71828182845905i) 2.11552790803290-2.41389289095525i)
      (num-test (acosh 3.14159265358979+1234.0i) 7.81116678989204+1.56825047198552i)
      (num-test (acosh -3.14159265358979+1234.0i) 7.81116678989204+1.57334218160427i)
      (num-test (acosh 3.14159265358979-1234.0i) 7.81116678989204-1.56825047198552i)
      (num-test (acosh -3.14159265358979-1234.0i) 7.81116678989204-1.57334218160427i)
      (num-test (acosh 3.14159265358979+1234000000.0i) 21.62667394298955+1.57079632424904i)
      (num-test (acosh -3.14159265358979+1234000000.0i) 21.62667394298955+1.57079632934076i)
      (num-test (acosh 3.14159265358979-1234000000.0i) 21.62667394298955-1.57079632424904i)
      (num-test (acosh -3.14159265358979-1234000000.0i) 21.62667394298955-1.57079632934076i)
      (num-test (acosh 2.71828182845905+0.0i) 1.65745445415308)
      (num-test (acosh -2.71828182845905+0.0i) 1.65745445415308+3.14159265358979i)
      (num-test (acosh 2.71828182845905-0.0i) 1.65745445415308)
      (num-test (acosh -2.71828182845905-0.0i) 1.65745445415308+3.14159265358979i)
      (num-test (acosh 2.71828182845905+0.00000001i) 1.65745445415308+0.00000000395623i)
      (num-test (acosh -2.71828182845905+0.00000001i) 1.65745445415308+3.14159264963356i)
      (num-test (acosh 2.71828182845905-0.00000001i) 1.65745445415308-0.00000000395623i)
      (num-test (acosh -2.71828182845905-0.00000001i) 1.65745445415308-3.14159264963356i)
      (num-test (acosh 2.71828182845905+1.0i) 1.73375471569385+0.37321824160908i)
      (num-test (acosh -2.71828182845905+1.0i) 1.73375471569385+2.76837441198071i)
      (num-test (acosh 2.71828182845905-1.0i) 1.73375471569385-0.37321824160908i)
      (num-test (acosh -2.71828182845905-1.0i) 1.73375471569385-2.76837441198071i)
      (num-test (acosh 2.71828182845905+3.14159265358979i) 2.11968336368048+0.87174835684834i)
      (num-test (acosh -2.71828182845905+3.14159265358979i) 2.11968336368048+2.26984429674145i)
      (num-test (acosh 2.71828182845905-3.14159265358979i) 2.11968336368048-0.87174835684834i)
      (num-test (acosh -2.71828182845905-3.14159265358979i) 2.11968336368048-2.26984429674145i)
      (num-test (acosh 2.71828182845905+2.71828182845905i) 2.04014932880026+0.80229897091014i)
      (num-test (acosh -2.71828182845905+2.71828182845905i) 2.04014932880026+2.33929368267965i)
      (num-test (acosh 2.71828182845905-2.71828182845905i) 2.04014932880026-0.80229897091014i)
      (num-test (acosh -2.71828182845905-2.71828182845905i) 2.04014932880026-2.33929368267965i)
      (num-test (acosh 2.71828182845905+1234.0i) 7.81116597540442+1.56859350950221i)
      (num-test (acosh -2.71828182845905+1234.0i) 7.81116597540442+1.57299914408758i)
      (num-test (acosh 2.71828182845905-1234.0i) 7.81116597540442-1.56859350950221i)
      (num-test (acosh -2.71828182845905-1234.0i) 7.81116597540442-1.57299914408758i)
      (num-test (acosh 2.71828182845905+1234000000.0i) 21.62667394298955+1.57079632459208i)
      (num-test (acosh -2.71828182845905+1234000000.0i) 21.62667394298955+1.57079632899772i)
      (num-test (acosh 2.71828182845905-1234000000.0i) 21.62667394298955-1.57079632459208i)
      (num-test (acosh -2.71828182845905-1234000000.0i) 21.62667394298955-1.57079632899772i)
      (num-test (acosh 1234.0+0.0i) 7.81116322084923)
      (num-test (acosh -1234.0+0.0i) 7.81116322084923+3.14159265358979i)
      (num-test (acosh 1234.0-0.0i) 7.81116322084923)
      (num-test (acosh -1234.0-0.0i) 7.81116322084923+3.14159265358979i)
      (num-test (acosh 1234.0+0.00000001i) 7.81116322084923+0.00000000000810i)
      (num-test (acosh -1234.0+0.00000001i) 7.81116322084923+3.14159265358169i)
      (num-test (acosh 1234.0-0.00000001i) 7.81116322084923-0.00000000000810i)
      (num-test (acosh -1234.0-0.00000001i) 7.81116322084923-3.14159265358169i)
      (num-test (acosh 1234.0+1.0i) 7.81116354920146+0.00081037286017i)
      (num-test (acosh -1234.0+1.0i) 7.81116354920146+3.14078228072962i)
      (num-test (acosh 1234.0-1.0i) 7.81116354920146-0.00081037286017i)
      (num-test (acosh -1234.0-1.0i) 7.81116354920146-3.14078228072962i)
      (num-test (acosh 1234.0+3.14159265358979i) 7.81116646154641+0.00254585648123i)
      (num-test (acosh -1234.0+3.14159265358979i) 7.81116646154641+3.13904679710856i)
      (num-test (acosh 1234.0-3.14159265358979i) 7.81116646154641-0.00254585648123i)
      (num-test (acosh -1234.0-3.14159265358979i) 7.81116646154641-3.13904679710856i)
      (num-test (acosh 1234.0+2.71828182845905i) 7.81116564705719+0.00220281873928i)
      (num-test (acosh -1234.0+2.71828182845905i) 7.81116564705719+3.13938983485052i)
      (num-test (acosh 1234.0-2.71828182845905i) 7.81116564705719-0.00220281873928i)
      (num-test (acosh -1234.0-2.71828182845905i) 7.81116564705719-3.13938983485052i)
      (num-test (acosh 1234.0+1234.0i) 8.15773697530526+0.78539824548545i)
      (num-test (acosh -1234.0+1234.0i) 8.15773697530526+2.35619440810434i)
      (num-test (acosh 1234.0-1234.0i) 8.15773697530526-0.78539824548545i)
      (num-test (acosh -1234.0-1234.0i) 8.15773697530526-2.35619440810434i)
      (num-test (acosh 1234.0+1234000000.0i) 21.62667394299005+1.57079532679490i)
      (num-test (acosh -1234.0+1234000000.0i) 21.62667394299005+1.57079732679490i)
      (num-test (acosh 1234.0-1234000000.0i) 21.62667394299005-1.57079532679490i)
      (num-test (acosh -1234.0-1234000000.0i) 21.62667394299005-1.57079732679490i)
      (num-test (acosh 1234000000.0+0.0i) 21.62667394298955)
      (num-test (acosh -1234000000.0+0.0i) 21.62667394298955+3.14159265358979i)
      (num-test (acosh 1234000000.0-0.0i) 21.62667394298955)
      (num-test (acosh -1234000000.0-0.0i) 21.62667394298955+3.14159265358979i)
      (num-test (acosh 1234000000.0+0.00000001i) 21.62667394298955+0.0i)
      (num-test (acosh -1234000000.0+0.00000001i) 21.62667394298955+3.14159265358979i)
      (num-test (acosh 1234000000.0-0.00000001i) 21.62667394298955-0.0i)
      (num-test (acosh -1234000000.0-0.00000001i) 21.62667394298955-3.14159265358979i)
      (num-test (acosh 1234000000.0+1.0i) 21.62667394298955+0.00000000081037i)
      (num-test (acosh -1234000000.0+1.0i) 21.62667394298955+3.14159265277942i)
      (num-test (acosh 1234000000.0-1.0i) 21.62667394298955-0.00000000081037i)
      (num-test (acosh -1234000000.0-1.0i) 21.62667394298955-3.14159265277942i)
      (num-test (acosh 1234000000.0+3.14159265358979i) 21.62667394298955+0.00000000254586i)
      (num-test (acosh -1234000000.0+3.14159265358979i) 21.62667394298955+3.14159265104393i)
      (num-test (acosh 1234000000.0-3.14159265358979i) 21.62667394298955-0.00000000254586i)
      (num-test (acosh -1234000000.0-3.14159265358979i) 21.62667394298955-3.14159265104393i)
      (num-test (acosh 1234000000.0+2.71828182845905i) 21.62667394298955+0.00000000220282i)
      (num-test (acosh -1234000000.0+2.71828182845905i) 21.62667394298955+3.14159265138697i)
      (num-test (acosh 1234000000.0-2.71828182845905i) 21.62667394298955-0.00000000220282i)
      (num-test (acosh -1234000000.0-2.71828182845905i) 21.62667394298955-3.14159265138697i)
      (num-test (acosh 1234000000.0+1234.0i) 21.62667394299005+0.00000100000000i)
      (num-test (acosh -1234000000.0+1234.0i) 21.62667394299005+3.14159165358979i)
      (num-test (acosh 1234000000.0-1234.0i) 21.62667394299005-0.00000100000000i)
      (num-test (acosh -1234000000.0-1234.0i) 21.62667394299005-3.14159165358979i)
      (num-test (acosh 1234000000.0+1234000000.0i) 21.97324753326953+0.78539816339745i)
      (num-test (acosh -1234000000.0+1234000000.0i) 21.97324753326953+2.35619449019234i)
      (num-test (acosh 1234000000.0-1234000000.0i) 21.97324753326953-0.78539816339745i)
      (num-test (acosh -1234000000.0-1234000000.0i) 21.97324753326953-2.35619449019234i)
      (num-test (acosh 0) 0+1.570796326794897i)
      (num-test (acosh 1) 0.0) ; or 0??
      (num-test (acosh -1) 0+3.141592653589793i)
      (num-test (acosh -1.0e+01) 2.9932228461263808979e0+3.1415926535897932385e0i)
      (num-test (acosh -2.0e+00) 1.3169578969248167086e0+3.1415926535897932385e0i)
      (num-test (acosh -1.0e+00) 0+3.1415926535897932385e0i)
      (num-test (acosh -7.50e-01) 0+2.4188584057763776273e0i)
      (num-test (acosh -5.0e-01) 0+2.0943951023931954923e0i)
      (num-test (acosh -1.250e-01) 0+1.6961241579629620161e0i)
      (num-test (acosh -3.45266983001243932001e-04) 0+1.5711415937847577022e0i)
      (num-test (acosh -1.19209289550781250e-07) 0+1.570796446004186170e0i)
      (num-test (acosh 0.0e+00) 0+1.5707963267948966192e0i)
      (num-test (acosh 1.19209289550781250e-07) 0+1.5707962075856070684e0i)
      (num-test (acosh 3.45266983001243932001e-04) 0+1.5704510598050355363e0i)
      (num-test (acosh 1.250e-01) 0+1.4454684956268312224e0i)
      (num-test (acosh 5.0e-01) 0+1.0471975511965977462e0i)
      (num-test (acosh 7.50e-01) 0+7.2273424781341561118e-1i)
      (num-test (acosh 1.0e+00) 0e0+0.0i)
      (num-test (acosh 2.0e+00) 1.3169578969248167086e0+0.0i)
      (num-test (acosh 1.0e+01) 2.9932228461263808979e0+0.0i)
      (num-test (acosh 0.0e+00+0.0e+00i) 0+1.5707963267948966192e0i)
      (num-test (acosh 0.0e+00+1.19209289550781250e-07i) 1.1920928955078096766e-7+1.5707963267948966192e0i)
      (num-test (acosh 0.0e+00-1.19209289550781250e-07i) 1.1920928955078096766e-7-1.5707963267948966192e0i)
      (num-test (acosh 0.0e+00+5.0e-01i) 4.8121182505960344750e-1+1.5707963267948966192e0i)
      (num-test (acosh 0.0e+00-5.0e-01i) 4.8121182505960344750e-1-1.5707963267948966192e0i)
      (num-test (acosh 0.0e+00+1.0e+00i) 8.8137358701954302523e-1+1.5707963267948966192e0i)
      (num-test (acosh 0.0e+00-1.0e+00i) 8.8137358701954302523e-1-1.5707963267948966192e0i)
      (num-test (acosh 0.0e+00+2.0e+00i) 1.4436354751788103425e0+1.5707963267948966192e0i)
      (num-test (acosh 0.0e+00-2.0e+00i) 1.4436354751788103425e0-1.5707963267948966192e0i)
      (num-test (acosh 0.0e+00+8.3886080e+06i) 1.6635532333438690979e1+1.5707963267948966192e0i)
      (num-test (acosh 0.0e+00-8.3886080e+06i) 1.6635532333438690979e1-1.5707963267948966192e0i)
      (num-test (acosh 1.19209289550781250e-07+0.0e+00i) 0+1.5707962075856070684e0i)
      (num-test (acosh -1.19209289550781250e-07+0.0e+00i) 0+1.570796446004186170e0i)
      (num-test (acosh 1.19209289550781250e-07+1.19209289550781250e-07i) 1.1920928955078181469e-7+1.5707962075856070685e0i)
      (num-test (acosh 1.19209289550781250e-07-1.19209289550781250e-07i) 1.1920928955078181469e-7-1.5707962075856070685e0i)
      (num-test (acosh -1.19209289550781250e-07+1.19209289550781250e-07i) 1.1920928955078181469e-7+1.570796446004186170e0i)
      (num-test (acosh -1.19209289550781250e-07-1.19209289550781250e-07i) 1.1920928955078181469e-7-1.570796446004186170e0i)
      (num-test (acosh 1.19209289550781250e-07+5.0e-01i) 4.8121182505960598961e-1+1.5707962201708666252e0i)
      (num-test (acosh 1.19209289550781250e-07-5.0e-01i) 4.8121182505960598961e-1-1.5707962201708666252e0i)
      (num-test (acosh -1.19209289550781250e-07+5.0e-01i) 4.8121182505960598961e-1+1.5707964334189266132e0i)
      (num-test (acosh -1.19209289550781250e-07-5.0e-01i) 4.8121182505960598961e-1-1.5707964334189266132e0i)
      (num-test (acosh 1.19209289550781250e-07+1.0e+00i) 8.8137358701954553738e-1+1.5707962425011995974e0i)
      (num-test (acosh 1.19209289550781250e-07-1.0e+00i) 8.8137358701954553738e-1-1.5707962425011995974e0i)
      (num-test (acosh -1.19209289550781250e-07+1.0e+00i) 8.8137358701954553738e-1+1.5707964110885936410e0i)
      (num-test (acosh -1.19209289550781250e-07-1.0e+00i) 8.8137358701954553738e-1-1.5707964110885936410e0i)
      (num-test (acosh 1.19209289550781250e-07+2.0e+00i) 1.4436354751788116136e0+1.5707962734828816222e0i)
      (num-test (acosh 1.19209289550781250e-07-2.0e+00i) 1.4436354751788116136e0-1.5707962734828816222e0i)
      (num-test (acosh -1.19209289550781250e-07+2.0e+00i) 1.4436354751788116136e0+1.5707963801069116162e0i)
      (num-test (acosh -1.19209289550781250e-07-2.0e+00i) 1.4436354751788116136e0-1.5707963801069116162e0i)
      (num-test (acosh 1.19209289550781250e-07+8.3886080e+06i) 1.6635532333438690979e1+1.5707963267948824084e0i)
      (num-test (acosh 1.19209289550781250e-07-8.3886080e+06i) 1.6635532333438690979e1-1.5707963267948824084e0i)
      (num-test (acosh -1.19209289550781250e-07+8.3886080e+06i) 1.6635532333438690979e1+1.5707963267949108301e0i)
      (num-test (acosh -1.19209289550781250e-07-8.3886080e+06i) 1.6635532333438690979e1-1.5707963267949108301e0i)
      (num-test (acosh 5.0e-01+0.0e+00i) 0+1.0471975511965977462e0i)
      (num-test (acosh -5.0e-01+0.0e+00i) 0+2.0943951023931954923e0i)
      (num-test (acosh 5.0e-01+1.19209289550781250e-07i) 1.3765103082409432364e-7+1.0471975511966032159e0i)
      (num-test (acosh 5.0e-01-1.19209289550781250e-07i) 1.3765103082409432364e-7-1.0471975511966032159e0i)
      (num-test (acosh -5.0e-01+1.19209289550781250e-07i) 1.3765103082409432364e-7+2.0943951023931900225e0i)
      (num-test (acosh -5.0e-01-1.19209289550781250e-07i) 1.3765103082409432364e-7-2.0943951023931900225e0i)
      (num-test (acosh 5.0e-01+5.0e-01i) 5.3063753095251782602e-1+1.1185178796437059372e0i)
      (num-test (acosh 5.0e-01-5.0e-01i) 5.3063753095251782602e-1-1.1185178796437059372e0i)
      (num-test (acosh -5.0e-01+5.0e-01i) 5.3063753095251782602e-1+2.0230747739460873013e0i)
      (num-test (acosh -5.0e-01-5.0e-01i) 5.3063753095251782602e-1-2.0230747739460873013e0i)
      (num-test (acosh 5.0e-01+1.0e+00i) 9.2613303135018242455e-1+1.2213572639376833256e0i)
      (num-test (acosh 5.0e-01-1.0e+00i) 9.2613303135018242455e-1-1.2213572639376833256e0i)
      (num-test (acosh -5.0e-01+1.0e+00i) 9.2613303135018242455e-1+1.9202353896521099129e0i)
      (num-test (acosh -5.0e-01-1.0e+00i) 9.2613303135018242455e-1-1.9202353896521099129e0i)
      (num-test (acosh 5.0e-01+2.0e+00i) 1.4657153519472905218e0+1.3497776911720127603e0i)
      (num-test (acosh 5.0e-01-2.0e+00i) 1.4657153519472905218e0-1.3497776911720127603e0i)
      (num-test (acosh -5.0e-01+2.0e+00i) 1.4657153519472905218e0+1.7918149624177804781e0i)
      (num-test (acosh -5.0e-01-2.0e+00i) 1.4657153519472905218e0-1.7918149624177804781e0i)
      (num-test (acosh 5.0e-01+8.3886080e+06i) 1.6635532333438692755e1+1.5707962671902518438e0i)
      (num-test (acosh 5.0e-01-8.3886080e+06i) 1.6635532333438692755e1-1.5707962671902518438e0i)
      (num-test (acosh -5.0e-01+8.3886080e+06i) 1.6635532333438692755e1+1.5707963863995413946e0i)
      (num-test (acosh -5.0e-01-8.3886080e+06i) 1.6635532333438692755e1-1.5707963863995413946e0i)
      (num-test (acosh 1.0e+00+0.0e+00i) 0e0+0.0i)
      (num-test (acosh -1.0e+00+0.0e+00i) 0+3.1415926535897932385e0i)
      (num-test (acosh 1.0e+00+1.19209289550781250e-07i) 3.4526698643116312881e-4+3.4526697957132450399e-4i)
      (num-test (acosh 1.0e+00-1.19209289550781250e-07i) 3.4526698643116312881e-4-3.4526697957132450399e-4i)
      (num-test (acosh -1.0e+00+1.19209289550781250e-07i) 3.4526698643116312881e-4+3.1412473866102219140e0i)
      (num-test (acosh -1.0e+00-1.19209289550781250e-07i) 3.4526698643116312881e-4-3.1412473866102219140e0i)
      (num-test (acosh 1.0e+00+5.0e-01i) 7.3285767597364526089e-1+6.7488884558600638016e-1i)
      (num-test (acosh 1.0e+00-5.0e-01i) 7.3285767597364526089e-1-6.7488884558600638016e-1i)
      (num-test (acosh -1.0e+00+5.0e-01i) 7.3285767597364526089e-1+2.4667038080037868583e0i)
      (num-test (acosh -1.0e+00-5.0e-01i) 7.3285767597364526089e-1-2.4667038080037868583e0i)
      (num-test (acosh 1.0e+00+1.0e+00i) 1.0612750619050356520e0+9.0455689430238136413e-1i)
      (num-test (acosh 1.0e+00-1.0e+00i) 1.0612750619050356520e0-9.0455689430238136413e-1i)
      (num-test (acosh -1.0e+00+1.0e+00i) 1.0612750619050356520e0+2.2370357592874118743e0i)
      (num-test (acosh -1.0e+00-1.0e+00i) 1.0612750619050356520e0-2.2370357592874118743e0i)
      (num-test (acosh 1.0e+00+2.0e+00i) 1.5285709194809981613e0+1.1437177404024204938e0i)
      (num-test (acosh 1.0e+00-2.0e+00i) 1.5285709194809981613e0-1.1437177404024204938e0i)
      (num-test (acosh -1.0e+00+2.0e+00i) 1.5285709194809981613e0+1.9978749131873727447e0i)
      (num-test (acosh -1.0e+00-2.0e+00i) 1.5285709194809981613e0-1.9978749131873727447e0i)
      (num-test (acosh 1.0e+00+8.3886080e+06i) 1.6635532333438698084e1+1.5707962075856070685e0i)
      (num-test (acosh 1.0e+00-8.3886080e+06i) 1.6635532333438698084e1-1.5707962075856070685e0i)
      (num-test (acosh -1.0e+00+8.3886080e+06i) 1.6635532333438698084e1+1.570796446004186170e0i)
      (num-test (acosh -1.0e+00-8.3886080e+06i) 1.6635532333438698084e1-1.570796446004186170e0i)
      (num-test (acosh 2.0e+00+0.0e+00i) 1.3169578969248167086e0+0.0i)
      (num-test (acosh -2.0e+00+0.0e+00i) 1.3169578969248167086e0+3.1415926535897932385e0i)
      (num-test (acosh 2.0e+00+1.19209289550781250e-07i) 1.3169578969248194435e0+6.8825515412047433504e-8i)
      (num-test (acosh 2.0e+00-1.19209289550781250e-07i) 1.3169578969248194435e0-6.8825515412047433504e-8i)
      (num-test (acosh -2.0e+00+1.19209289550781250e-07i) 1.3169578969248194435e0+3.1415925847642778264e0i)
      (num-test (acosh -2.0e+00-1.19209289550781250e-07i) 1.3169578969248194435e0-3.1415925847642778264e0i)
      (num-test (acosh 2.0e+00+5.0e-01i) 1.3618009008578457882e0+2.7775425655771396018e-1i)
      (num-test (acosh 2.0e+00-5.0e-01i) 1.3618009008578457882e0-2.7775425655771396018e-1i)
      (num-test (acosh -2.0e+00+5.0e-01i) 1.3618009008578457882e0+2.8638383970320792783e0i)
      (num-test (acosh -2.0e+00-5.0e-01i) 1.3618009008578457882e0-2.8638383970320792783e0i)
      (num-test (acosh 2.0e+00+1.0e+00i) 1.4693517443681852733e0+5.0735630321714456304e-1i)
      (num-test (acosh 2.0e+00-1.0e+00i) 1.4693517443681852733e0-5.0735630321714456304e-1i)
      (num-test (acosh -2.0e+00+1.0e+00i) 1.4693517443681852733e0+2.6342363503726486754e0i)
      (num-test (acosh -2.0e+00-1.0e+00i) 1.4693517443681852733e0-2.6342363503726486754e0i)
      (num-test (acosh 2.0e+00+2.0e+00i) 1.7343245214879664480e0+8.1654718209685057852e-1i)
      (num-test (acosh 2.0e+00-2.0e+00i) 1.7343245214879664480e0-8.1654718209685057852e-1i)
      (num-test (acosh -2.0e+00+2.0e+00i) 1.7343245214879664480e0+2.3250454714929426599e0i)
      (num-test (acosh -2.0e+00-2.0e+00i) 1.7343245214879664480e0-2.3250454714929426599e0i)
      (num-test (acosh 2.0e+00+8.3886080e+06i) 1.663553233343871940e1+1.5707960883763175177e0i)
      (num-test (acosh 2.0e+00-8.3886080e+06i) 1.663553233343871940e1-1.5707960883763175177e0i)
      (num-test (acosh -2.0e+00+8.3886080e+06i) 1.663553233343871940e1+1.5707965652134757208e0i)
      (num-test (acosh -2.0e+00-8.3886080e+06i) 1.663553233343871940e1-1.5707965652134757208e0i)
      (num-test (acosh 8.3886080e+06+0.0e+00i) 1.6635532333438683873e1+0.0i)
      (num-test (acosh -8.3886080e+06+0.0e+00i) 1.6635532333438683873e1+3.1415926535897932385e0i)
      (num-test (acosh 8.3886080e+06+1.19209289550781250e-07i) 1.6635532333438683873e1+1.4210854715202104692e-14i)
      (num-test (acosh 8.3886080e+06-1.19209289550781250e-07i) 1.6635532333438683873e1-1.4210854715202104692e-14i)
      (num-test (acosh -8.3886080e+06+1.19209289550781250e-07i) 1.6635532333438683873e1+3.1415926535897790276e0i)
      (num-test (acosh -8.3886080e+06-1.19209289550781250e-07i) 1.6635532333438683873e1-3.1415926535897790276e0i)
      (num-test (acosh 8.3886080e+06+5.0e-01i) 1.6635532333438685650e1+5.9604644775390977930e-8i)
      (num-test (acosh 8.3886080e+06-5.0e-01i) 1.6635532333438685650e1-5.9604644775390977930e-8i)
      (num-test (acosh -8.3886080e+06+5.0e-01i) 1.6635532333438685650e1+3.1415925939851484631e0i)
      (num-test (acosh -8.3886080e+06-5.0e-01i) 1.6635532333438685650e1-3.1415925939851484631e0i)
      (num-test (acosh 8.3886080e+06+1.0e+00i) 1.6635532333438690979e1+1.1920928955078153234e-7i)
      (num-test (acosh 8.3886080e+06-1.0e+00i) 1.6635532333438690979e1-1.1920928955078153234e-7i)
      (num-test (acosh -8.3886080e+06+1.0e+00i) 1.6635532333438690979e1+3.1415925343805036877e0i)
      (num-test (acosh -8.3886080e+06-1.0e+00i) 1.6635532333438690979e1-3.1415925343805036877e0i)
      (num-test (acosh 8.3886080e+06+2.0e+00i) 1.6635532333438712295e1+2.3841857910155967656e-7i)
      (num-test (acosh 8.3886080e+06-2.0e+00i) 1.6635532333438712295e1-2.3841857910155967656e-7i)
      (num-test (acosh -8.3886080e+06+2.0e+00i) 1.6635532333438712295e1+3.1415924151712141369e0i)
      (num-test (acosh -8.3886080e+06-2.0e+00i) 1.6635532333438712295e1-3.1415924151712141369e0i)
      (num-test (acosh 8.3886080e+06+8.3886080e+06i) 1.6982105923718660081e1+7.8539816339745008597e-1i)
      (num-test (acosh 8.3886080e+06-8.3886080e+06i) 1.6982105923718660081e1-7.8539816339745008597e-1i)
      (num-test (acosh -8.3886080e+06+8.3886080e+06i) 1.6982105923718660081e1+2.3561944901923431525e0i)
      (num-test (acosh -8.3886080e+06-8.3886080e+06i) 1.6982105923718660081e1-2.3561944901923431525e0i)


      ;; -------- atanh
      (num-test (atanh 0) 0.0)
      (num-test (atanh 2) 0.54930614433405+1.57079632679490i)
      (num-test (atanh -2) -0.54930614433405+1.57079632679490i)
      (num-test (atanh 3) 0.34657359027997+1.57079632679490i)
      (num-test (atanh -3) -0.34657359027997+1.57079632679490i)
      (num-test (atanh 10) 0.10033534773108+1.57079632679490i)
      (num-test (atanh -10) -0.10033534773108+1.57079632679490i)
      (num-test (atanh 1234) 0.00081037294887+1.57079632679490i)
      (num-test (atanh -1234) -0.00081037294887+1.57079632679490i)
      (num-test (atanh 500029) 0.00000199988401+1.57079632679490i)
      (num-test (atanh -500029) -0.00000199988401+1.57079632679490i)
      (num-test (atanh 362880) 0.00000275573192+1.57079632679490i)
      (num-test (atanh -362880) -0.00000275573192+1.57079632679490i)
      (num-test (atanh 0/1) 0.0)
      (num-test (atanh 0/2) 0.0)
      (num-test (atanh 0/3) 0.0)
      (num-test (atanh 0/10) 0.0)
      (num-test (atanh 0/1234) 0.0)
      (num-test (atanh 0/1234000000) 0.0)
      (num-test (atanh 0/500029) 0.0)
      (num-test (atanh 0/362880) 0.0)
      (num-test (atanh 1/2) 0.54930614433405)
      (num-test (atanh -1/2) -0.54930614433405)
      (num-test (atanh 1/3) 0.34657359027997)
      (num-test (atanh -1/3) -0.34657359027997)
      (num-test (atanh 1/10) 0.10033534773108)
      (num-test (atanh -1/10) -0.10033534773108)
      (num-test (atanh 1/1234) 0.00081037294887)
      (num-test (atanh -1/1234) -0.00081037294887)
      (num-test (atanh 1/1234000000) 0.00000000081037)
      (num-test (atanh -1/1234000000) -0.00000000081037)
      (num-test (atanh 1/500029) 0.00000199988401)
      (num-test (atanh -1/500029) -0.00000199988401)
      (num-test (atanh 1/362880) 0.00000275573192)
      (num-test (atanh -1/362880) -0.00000275573192)
      (num-test (atanh 2/1) 0.54930614433405+1.57079632679490i)
      (num-test (atanh -2/1) -0.54930614433405+1.57079632679490i)
      (num-test (atanh 2/3) 0.80471895621705)
      (num-test (atanh -2/3) -0.80471895621705)
      (num-test (atanh 2/10) 0.20273255405408)
      (num-test (atanh -2/10) -0.20273255405408)
      (num-test (atanh 2/1234) 0.00162074696209)
      (num-test (atanh -2/1234) -0.00162074696209)
      (num-test (atanh 2/1234000000) 0.00000000162075)
      (num-test (atanh -2/1234000000) -0.00000000162075)
      (num-test (atanh 2/500029) 0.00000399976801)
      (num-test (atanh -2/500029) -0.00000399976801)
      (num-test (atanh 2/362880) 0.00000551146384)
      (num-test (atanh -2/362880) -0.00000551146384)
      (num-test (atanh 3/1) 0.34657359027997+1.57079632679490i)
      (num-test (atanh -3/1) -0.34657359027997+1.57079632679490i)
      (num-test (atanh 3/2) 0.80471895621705+1.57079632679490i)
      (num-test (atanh -3/2) -0.80471895621705+1.57079632679490i)
      (num-test (atanh 3/10) 0.30951960420311)
      (num-test (atanh -3/10) -0.30951960420311)
      (num-test (atanh 3/1234) 0.00243112310402)
      (num-test (atanh -3/1234) -0.00243112310402)
      (num-test (atanh 3/1234000000) 0.00000000243112)
      (num-test (atanh -3/1234000000) -0.00000000243112)
      (num-test (atanh 3/500029) 0.00000599965202)
      (num-test (atanh -3/500029) -0.00000599965202)
      (num-test (atanh 3/362880) 0.00000826719577)
      (num-test (atanh -3/362880) -0.00000826719577)
      (num-test (atanh 10/1) 0.10033534773108+1.57079632679490i)
      (num-test (atanh -10/1) -0.10033534773108+1.57079632679490i)
      (num-test (atanh 10/2) 0.20273255405408+1.57079632679490i)
      (num-test (atanh -10/2) -0.20273255405408+1.57079632679490i)
      (num-test (atanh 10/3) 0.30951960420311+1.57079632679490i)
      (num-test (atanh -10/3) -0.30951960420311+1.57079632679490i)
      (num-test (atanh 10/1234) 0.00810390511343)
      (num-test (atanh -10/1234) -0.00810390511343)
      (num-test (atanh 10/1234000000) 0.00000000810373)
      (num-test (atanh -10/1234000000) -0.00000000810373)
      (num-test (atanh 10/500029) 0.00001999884007)
      (num-test (atanh -10/500029) -0.00001999884007)
      (num-test (atanh 10/362880) 0.00002755731923)
      (num-test (atanh -10/362880) -0.00002755731923)
      (num-test (atanh 1234/1) 0.00081037294887+1.57079632679490i)
      (num-test (atanh -1234/1) -0.00081037294887+1.57079632679490i)
      (num-test (atanh 1234/2) 0.00162074696209+1.57079632679490i)
      (num-test (atanh -1234/2) -0.00162074696209+1.57079632679490i)
      (num-test (atanh 1234/3) 0.00243112310402+1.57079632679490i)
      (num-test (atanh -1234/3) -0.00243112310402+1.57079632679490i)
      (num-test (atanh 1234/10) 0.00810390511343+1.57079632679490i)
      (num-test (atanh -1234/10) -0.00810390511343+1.57079632679490i)
      (num-test (atanh 1234/1234000000) 0.00000100000000)
      (num-test (atanh -1234/1234000000) -0.00000100000000)
      (num-test (atanh 1234/500029) 0.00246786187433)
      (num-test (atanh -1234/500029) -0.00246786187433)
      (num-test (atanh 1234/362880) 0.00340058630029)
      (num-test (atanh -1234/362880) -0.00340058630029)
      (num-test (atanh 1234000000/1234) 0.00000100000000+1.57079632679490i)
      (num-test (atanh -1234000000/1234) -0.00000100000000+1.57079632679490i)
      (num-test (atanh 1234000000/500029) 0.00040520990873+1.57079632679490i)
      (num-test (atanh -1234000000/500029) -0.00040520990873+1.57079632679490i)
      (num-test (atanh 1234000000/362880) 0.00029406807979+1.57079632679490i)
      (num-test (atanh -1234000000/362880) -0.00029406807979+1.57079632679490i)
      (num-test (atanh 500029/1) 0.00000199988401+1.57079632679490i)
      (num-test (atanh -500029/1) -0.00000199988401+1.57079632679490i)
      (num-test (atanh 500029/2) 0.00000399976801+1.57079632679490i)
      (num-test (atanh -500029/2) -0.00000399976801+1.57079632679490i)
      (num-test (atanh 500029/3) 0.00000599965202+1.57079632679490i)
      (num-test (atanh -500029/3) -0.00000599965202+1.57079632679490i)
      (num-test (atanh 500029/10) 0.00001999884007+1.57079632679490i)
      (num-test (atanh -500029/10) -0.00001999884007+1.57079632679490i)
      (num-test (atanh 500029/1234) 0.00246786187433+1.57079632679490i)
      (num-test (atanh -500029/1234) -0.00246786187433+1.57079632679490i)
      (num-test (atanh 500029/1234000000) 0.00040520990873)
      (num-test (atanh -500029/1234000000) -0.00040520990873)
      (num-test (atanh 500029/362880) 0.91962065666764+1.57079632679490i)
      (num-test (atanh -500029/362880) -0.91962065666764+1.57079632679490i)
      (num-test (atanh 362880/1) 0.00000275573192+1.57079632679490i)
      (num-test (atanh -362880/1) -0.00000275573192+1.57079632679490i)
      (num-test (atanh 362880/2) 0.00000551146384+1.57079632679490i)
      (num-test (atanh -362880/2) -0.00000551146385+1.57079632679490i)
      (num-test (atanh 362880/3) 0.00000826719577+1.57079632679490i)
      (num-test (atanh -362880/3) -0.00000826719577+1.57079632679490i)
      (num-test (atanh 362880/10) 0.00002755731923+1.57079632679490i)
      (num-test (atanh -362880/10) -0.00002755731923+1.57079632679490i)
      (num-test (atanh 362880/1234) 0.00340058630029+1.57079632679490i)
      (num-test (atanh -362880/1234) -0.00340058630029+1.57079632679490i)
      (num-test (atanh 362880/1234000000) 0.00029406807979)
      (num-test (atanh -362880/1234000000) -0.00029406807979)
      (num-test (atanh 362880/500029) 0.91962065666764)
      (num-test (atanh -362880/500029) -0.91962065666764)
      (num-test (atanh 0.0) 0.0)
      (num-test (atanh 0.00000001) 0.00000001)
      (num-test (atanh -0.00000001) -0.00000001)
      (num-test (atanh our-pi) 0.32976531495670+1.57079632679490i)
      (num-test (atanh -3.14159265358979) -0.32976531495670+1.57079632679490i)
      (num-test (atanh 2.71828182845905) 0.38596841645265+1.57079632679490i)
      (num-test (atanh -2.71828182845905) -0.38596841645265+1.57079632679490i)
      (num-test (atanh 1234.0) 0.00081037294887+1.57079632679490i)
      (num-test (atanh -1234.0) -0.00081037294887+1.57079632679490i)
      (num-test (atanh 0.0+0.0i) 0.0)
      (num-test (atanh -0.0+0.0i) 0.0)
      (num-test (atanh 0.0-0.0i) 0.0)
      (num-test (atanh -0.0-0.0i) -0.0)
      (num-test (atanh 0.0+0.00000001i) 0.0+0.00000001i)
      (num-test (atanh -0.0+0.00000001i) 0.0+0.00000001i)
      (num-test (atanh 0.0-0.00000001i) 0.0-0.00000001i)
      (num-test (atanh -0.0-0.00000001i) 0.0-0.00000001i)
      (num-test (atanh 0.0+1.0i) -0.0+0.78539816339745i)
      (num-test (atanh -0.0+1.0i) -0.0+0.78539816339745i)
      (num-test (atanh 0.0-1.0i) -0.0-0.78539816339745i)
      (num-test (atanh -0.0-1.0i) -0.0-0.78539816339745i)
      (num-test (atanh 0.0+3.14159265358979i) -0.0+1.26262725567891i)
      (num-test (atanh -0.0+3.14159265358979i) -0.0+1.26262725567891i)
      (num-test (atanh 0.0-3.14159265358979i) -0.0-1.26262725567891i)
      (num-test (atanh -0.0-3.14159265358979i) -0.0-1.26262725567891i)
      (num-test (atanh 0.0+2.71828182845905i) -0.0+1.21828290501728i)
      (num-test (atanh -0.0+2.71828182845905i) -0.0+1.21828290501728i)
      (num-test (atanh 0.0-2.71828182845905i) -0.0-1.21828290501728i)
      (num-test (atanh -0.0-2.71828182845905i) -0.0-1.21828290501728i)
      (num-test (atanh 0.0+1234.0i) 0.0+1.56998595420081i)
      (num-test (atanh -0.0+1234.0i) 0.0+1.56998595420081i)
      (num-test (atanh 0.0-1234.0i) 0.0-1.56998595420081i)
      (num-test (atanh -0.0-1234.0i) 0.0-1.56998595420081i)
      (num-test (atanh 0.00000001+0.0i) 0.00000001)
      (num-test (atanh -0.00000001+0.0i) -0.00000001)
      (num-test (atanh 0.00000001-0.0i) 0.00000001)
      (num-test (atanh -0.00000001-0.0i) -0.00000001)
      (num-test (atanh 0.00000001+0.00000001i) 0.00000001+0.00000001i)
      (num-test (atanh -0.00000001+0.00000001i) -0.00000001+0.00000001i)
      (num-test (atanh 0.00000001-0.00000001i) 0.00000001-0.00000001i)
      (num-test (atanh -0.00000001-0.00000001i) -0.00000001-0.00000001i)
      (num-test (atanh 0.00000001+1.0i) 0.00000000500000+0.78539816339745i)
      (num-test (atanh -0.00000001+1.0i) -0.00000000500000+0.78539816339745i)
      (num-test (atanh 0.00000001-1.0i) 0.00000000500000-0.78539816339745i)
      (num-test (atanh -0.00000001-1.0i) -0.00000000500000-0.78539816339745i)
      (num-test (atanh 0.00000001+3.14159265358979i) 0.00000000092000+1.26262725567891i)
      (num-test (atanh -0.00000001+3.14159265358979i) -0.00000000092000+1.26262725567891i)
      (num-test (atanh 0.00000001-3.14159265358979i) 0.00000000092000-1.26262725567891i)
      (num-test (atanh -0.00000001-3.14159265358979i) -0.00000000092000-1.26262725567891i)
      (num-test (atanh 0.00000001+2.71828182845905i) 0.00000000119203+1.21828290501728i)
      (num-test (atanh -0.00000001+2.71828182845905i) -0.00000000119203+1.21828290501728i)
      (num-test (atanh 0.00000001-2.71828182845905i) 0.00000000119203-1.21828290501728i)
      (num-test (atanh -0.00000001-2.71828182845905i) -0.00000000119203-1.21828290501728i)
      (num-test (atanh 0.00000001+1234.0i) 0.00000000000001+1.56998595420081i)
      (num-test (atanh -0.00000001+1234.0i) -0.00000000000001+1.56998595420081i)
      (num-test (atanh 0.00000001-1234.0i) 0.00000000000001-1.56998595420081i)
      (num-test (atanh -0.00000001-1234.0i) -0.00000000000001-1.56998595420081i)
      (num-test (atanh 1.0+0.00000001i) 9.55691396225616+0.78539816589745i)
      (num-test (atanh -1.0+0.00000001i) -9.55691396225615+0.78539816589745i)
      (num-test (atanh 1.0-0.00000001i) 9.55691396225616-0.78539816589745i)
      (num-test (atanh -1.0-0.00000001i) -9.55691396225615-0.78539816589745i)
      (num-test (atanh 1.0+1.0i) 0.40235947810853+1.01722196789785i)
      (num-test (atanh -1.0+1.0i) -0.40235947810853+1.01722196789785i)
      (num-test (atanh 1.0-1.0i) 0.40235947810853-1.01722196789785i)
      (num-test (atanh -1.0-1.0i) -0.40235947810853-1.01722196789785i)
      (num-test (atanh 1.0+3.14159265358979i) 0.08505998507745+1.28734057432439i)
      (num-test (atanh -1.0+3.14159265358979i) -0.08505998507745+1.28734057432439i)
      (num-test (atanh 1.0-3.14159265358979i) 0.08505998507745-1.28734057432439i)
      (num-test (atanh -1.0-3.14159265358979i) -0.08505998507745-1.28734057432439i)
      (num-test (atanh 1.0+2.71828182845905i) 0.10816322574795+1.25363416718071i)
      (num-test (atanh -1.0+2.71828182845905i) -0.10816322574795+1.25363416718071i)
      (num-test (atanh 1.0-2.71828182845905i) 0.10816322574795-1.25363416718071i)
      (num-test (atanh -1.0-2.71828182845905i) -0.10816322574795-1.25363416718071i)
      (num-test (atanh 1.0+1234.0i) 0.00000065670317+1.56998595473299i)
      (num-test (atanh -1.0+1234.0i) -0.00000065670317+1.56998595473299i)
      (num-test (atanh 1.0-1234.0i) 0.00000065670317-1.56998595473299i)
      (num-test (atanh -1.0-1234.0i) -0.00000065670317-1.56998595473299i)
      (num-test (atanh 3.14159265358979+0.0i) 0.32976531495670+1.57079632679490i)
      (num-test (atanh -3.14159265358979+0.0i) -0.32976531495670+1.57079632679490i)
      (num-test (atanh 3.14159265358979-0.0i) 0.32976531495670+1.57079632679490i)
      (num-test (atanh -3.14159265358979-0.0i) -0.32976531495670+1.57079632679490i)
      (num-test (atanh 3.14159265358979+0.00000001i) 0.32976531495670+1.57079632566745i)
      (num-test (atanh -3.14159265358979+0.00000001i) -0.32976531495670+1.57079632566745i)
      (num-test (atanh 3.14159265358979-0.00000001i) 0.32976531495670-1.57079632566745i)
      (num-test (atanh -3.14159265358979-0.00000001i) -0.32976531495670-1.57079632566745i)
      (num-test (atanh 3.14159265358979+1.0i) 0.29462144034086+1.47082882591946i)
      (num-test (atanh -3.14159265358979+1.0i) -0.29462144034086+1.47082882591946i)
      (num-test (atanh 3.14159265358979-1.0i) 0.29462144034086-1.47082882591946i)
      (num-test (atanh -3.14159265358979-1.0i) -0.29462144034086-1.47082882591946i)
      (num-test (atanh 3.14159265358979+3.14159265358979i) 0.15638868878130+1.40903828502376i)
      (num-test (atanh -3.14159265358979+3.14159265358979i) -0.15638868878130+1.40903828502376i)
      (num-test (atanh 3.14159265358979-3.14159265358979i) 0.15638868878130-1.40903828502376i)
      (num-test (atanh -3.14159265358979-3.14159265358979i) -0.15638868878130-1.40903828502376i)
      (num-test (atanh 3.14159265358979+2.71828182845905i) 0.17937970703436+1.40945039787275i)
      (num-test (atanh -3.14159265358979+2.71828182845905i) -0.17937970703436+1.40945039787275i)
      (num-test (atanh 3.14159265358979-2.71828182845905i) 0.17937970703436-1.40945039787275i)
      (num-test (atanh -3.14159265358979-2.71828182845905i) -0.17937970703436-1.40945039787275i)
      (num-test (atanh 3.14159265358979+1234.0i) 0.00000206308183+1.56998595945313i)
      (num-test (atanh -3.14159265358979+1234.0i) -0.00000206308183+1.56998595945313i)
      (num-test (atanh 3.14159265358979-1234.0i) 0.00000206308183-1.56998595945313i)
      (num-test (atanh -3.14159265358979-1234.0i) -0.00000206308183-1.56998595945313i)
      (num-test (atanh 2.71828182845905+0.0i) 0.38596841645265+1.57079632679490i)
      (num-test (atanh -2.71828182845905+0.0i) -0.38596841645265+1.57079632679490i)
      (num-test (atanh 2.71828182845905-0.0i) 0.38596841645265+1.57079632679490i)
      (num-test (atanh -2.71828182845905-0.0i) -0.38596841645265+1.57079632679490i)
      (num-test (atanh 2.71828182845905+0.00000001i) 0.38596841645265+1.57079632522972i)
      (num-test (atanh -2.71828182845905+0.00000001i) -0.38596841645265+1.57079632522972i)
      (num-test (atanh 2.71828182845905-0.00000001i) 0.38596841645265-1.57079632522972i)
      (num-test (atanh -2.71828182845905-0.00000001i) -0.38596841645265-1.57079632522972i)
      (num-test (atanh 2.71828182845905+1.0i) 0.33050259272341+1.43862796047891i)
      (num-test (atanh -2.71828182845905+1.0i) -0.33050259272341+1.43862796047891i)
      (num-test (atanh 2.71828182845905-1.0i) 0.33050259272341-1.43862796047891i)
      (num-test (atanh -2.71828182845905-1.0i) -0.33050259272341-1.43862796047891i)
      (num-test (atanh 2.71828182845905+3.14159265358979i) 0.15352587926173+1.38641010899673i)
      (num-test (atanh -2.71828182845905+3.14159265358979i) -0.15352587926173+1.38641010899673i)
      (num-test (atanh 2.71828182845905-3.14159265358979i) 0.15352587926173-1.38641010899673i)
      (num-test (atanh -2.71828182845905-3.14159265358979i) -0.15352587926173-1.38641010899673i)
      (num-test (atanh 2.71828182845905+2.71828182845905i) 0.17963089485802+1.38288382352616i)
      (num-test (atanh -2.71828182845905+2.71828182845905i) -0.17963089485802+1.38288382352616i)
      (num-test (atanh 2.71828182845905-2.71828182845905i) 0.17963089485802-1.38288382352616i)
      (num-test (atanh -2.71828182845905-2.71828182845905i) -0.17963089485802-1.38288382352616i)
      (num-test (atanh 2.71828182845905+1234.0i) 0.00000178509679+1.56998595813306i)
      (num-test (atanh -2.71828182845905+1234.0i) -0.00000178509679+1.56998595813306i)
      (num-test (atanh 2.71828182845905-1234.0i) 0.00000178509679-1.56998595813306i)
      (num-test (atanh -2.71828182845905-1234.0i) -0.00000178509679-1.56998595813306i)
      (num-test (atanh 1234.0+0.0i) 0.00081037294887+1.57079632679490i)
      (num-test (atanh -1234.0+0.0i) -0.00081037294887+1.57079632679490i)
      (num-test (atanh 1234.0-0.0i) 0.00081037294887+1.57079632679490i)
      (num-test (atanh -1234.0-0.0i) -0.00081037294887+1.57079632679490i)
      (num-test (atanh 1234.0+0.00000001i) 0.00081037294887+1.57079632679489i)
      (num-test (atanh -1234.0+0.00000001i) -0.00081037294887+1.57079632679489i)
      (num-test (atanh 1234.0-0.00000001i) 0.00081037294887-1.57079632679489i)
      (num-test (atanh -1234.0-0.00000001i) -0.00081037294887-1.57079632679489i)
      (num-test (atanh 1234.0+1.0i) 0.00081037241669+1.57079567009087i)
      (num-test (atanh -1234.0+1.0i) -0.00081037241669+1.57079567009087i)
      (num-test (atanh 1234.0-1.0i) 0.00081037241669-1.57079567009087i)
      (num-test (atanh -1234.0-1.0i) -0.00081037241669-1.57079567009087i)
      (num-test (atanh 1234.0+3.14159265358979i) 0.00081036769654+1.57079426371036i)
      (num-test (atanh -1234.0+3.14159265358979i) -0.00081036769654+1.57079426371036i)
      (num-test (atanh 1234.0-3.14159265358979i) 0.00081036769654-1.57079426371036i)
      (num-test (atanh -1234.0-3.14159265358979i) -0.00081036769654-1.57079426371036i)
      (num-test (atanh 1234.0+2.71828182845905i) 0.00081036901661+1.57079454169576i)
      (num-test (atanh -1234.0+2.71828182845905i) -0.00081036901661+1.57079454169576i)
      (num-test (atanh 1234.0-2.71828182845905i) 0.00081036901661-1.57079454169576i)
      (num-test (atanh -1234.0-2.71828182845905i) -0.00081036901661-1.57079454169576i)
      (num-test (atanh 1234.0+1234.0i) 0.00040518634139+1.57039114036481i)
      (num-test (atanh -1234.0+1234.0i) -0.00040518634139+1.57039114036481i)
      (num-test (atanh 1234.0-1234.0i) 0.00040518634139-1.57039114036481i)
      (num-test (atanh -1234.0-1234.0i) -0.00040518634139-1.57039114036481i)
      (num-test (atanh -1.0e+01) -1.0033534773107558064e-1+1.5707963267948966192e0i)
      (num-test (atanh -2.0e+00) -5.4930614433405484570e-1+1.5707963267948966192e0i)
      (num-test (atanh -7.50e-01) -9.7295507452765665255e-1+0.0i)
      (num-test (atanh -5.0e-01) -5.4930614433405484570e-1+0.0i)
      (num-test (atanh -1.250e-01) -1.2565721414045303884e-1+0.0i)
      (num-test (atanh -3.45266983001243932001e-04) -3.4526699672092216295e-4+0.0i)
      (num-test (atanh -1.19209289550781250e-07) -1.1920928955078181469e-7+0.0i)
      (num-test (atanh 0.0e+00) 0e0+0.0i)
      (num-test (atanh 1.19209289550781250e-07) 1.1920928955078181469e-7+0.0i)
      (num-test (atanh 3.45266983001243932001e-04) 3.4526699672092216295e-4+0.0i)
      (num-test (atanh 1.250e-01) 1.2565721414045303884e-1+0.0i)
      (num-test (atanh 5.0e-01) 5.4930614433405484570e-1+0.0i)
      (num-test (atanh 7.50e-01) 9.7295507452765665255e-1+0.0i)
      (num-test (atanh 2.0e+00) 5.4930614433405484570e-1-1.5707963267948966192e0i)
      (num-test (atanh 1.0e+01) 1.0033534773107558064e-1-1.5707963267948966192e0i)
      (num-test (atanh 2.8147497671066e+14) 3.552713678800501e-15-1.570796326794897i)
      (num-test (atanh 0.0e+00+0.0e+00i) 0e0+0.0i)
      (num-test (atanh 0.0e+00+1.19209289550781250e-07i) 0+1.1920928955078068531e-7i)
      (num-test (atanh 0.0e+00-1.19209289550781250e-07i) 0-1.1920928955078068531e-7i)
      (num-test (atanh 0.0e+00+5.0e-01i) 0+4.6364760900080611621e-1i)
      (num-test (atanh 0.0e+00-5.0e-01i) 0-4.6364760900080611621e-1i)
      (num-test (atanh 0.0e+00+1.0e+00i) 0+7.8539816339744830962e-1i)
      (num-test (atanh 0.0e+00-1.0e+00i) 0-7.8539816339744830962e-1i)
      (num-test (atanh 0.0e+00+2.0e+00i) 0+1.1071487177940905030e0i)
      (num-test (atanh 0.0e+00-2.0e+00i) 0-1.1071487177940905030e0i)
      (num-test (atanh 0.0e+00+8.3886080e+06i) 0+1.5707962075856070685e0i)
      (num-test (atanh 0.0e+00-8.3886080e+06i) 0-1.5707962075856070685e0i)
      (num-test (atanh 1.19209289550781250e-07+0.0e+00i) 1.1920928955078181469e-7+0.0i)
      (num-test (atanh -1.19209289550781250e-07+0.0e+00i) -1.1920928955078181469e-7+0.0i)
      (num-test (atanh 1.19209289550781250e-07+1.19209289550781250e-07i) 1.1920928955078012062e-7+1.1920928955078237938e-7i)
      (num-test (atanh 1.19209289550781250e-07-1.19209289550781250e-07i) 1.1920928955078012062e-7-1.1920928955078237938e-7i)
      (num-test (atanh -1.19209289550781250e-07+1.19209289550781250e-07i) -1.1920928955078012062e-7+1.1920928955078237938e-7i)
      (num-test (atanh -1.19209289550781250e-07-1.19209289550781250e-07i) -1.1920928955078012062e-7-1.1920928955078237938e-7i)
      (num-test (atanh 1.19209289550781250e-07+5.0e-01i) 9.5367431640625072280e-8+4.6364760900081066369e-1i)
      (num-test (atanh 1.19209289550781250e-07-5.0e-01i) 9.5367431640625072280e-8-4.6364760900081066369e-1i)
      (num-test (atanh -1.19209289550781250e-07+5.0e-01i) -9.5367431640625072280e-8+4.6364760900081066369e-1i)
      (num-test (atanh -1.19209289550781250e-07-5.0e-01i) -9.5367431640625072280e-8-4.6364760900081066369e-1i)
      (num-test (atanh 1.19209289550781250e-07+1.0e+00i) 5.9604644775390483828e-8+7.8539816339745186233e-1i)
      (num-test (atanh 1.19209289550781250e-07-1.0e+00i) 5.9604644775390483828e-8-7.8539816339745186233e-1i)
      (num-test (atanh -1.19209289550781250e-07+1.0e+00i) -5.9604644775390483828e-8+7.8539816339745186233e-1i)
      (num-test (atanh -1.19209289550781250e-07-1.0e+00i) -5.9604644775390483828e-8-7.8539816339745186233e-1i)
      (num-test (atanh 1.19209289550781250e-07+2.0e+00i) 2.3841857910156200307e-8+1.1071487177940916399e0i)
      (num-test (atanh 1.19209289550781250e-07-2.0e+00i) 2.3841857910156200307e-8-1.1071487177940916399e0i)
      (num-test (atanh -1.19209289550781250e-07+2.0e+00i) -2.3841857910156200307e-8+1.1071487177940916399e0i)
      (num-test (atanh -1.19209289550781250e-07-2.0e+00i) -2.3841857910156200307e-8-1.1071487177940916399e0i)
      (num-test (atanh 1.19209289550781250e-07+8.3886080e+06i) 1.6940658945085766040e-21+1.5707962075856070685e0i)
      (num-test (atanh 1.19209289550781250e-07-8.3886080e+06i) 1.6940658945085766040e-21-1.5707962075856070685e0i)
      (num-test (atanh -1.19209289550781250e-07+8.3886080e+06i) -1.6940658945085766040e-21+1.5707962075856070685e0i)
      (num-test (atanh -1.19209289550781250e-07-8.3886080e+06i) -1.6940658945085766040e-21-1.5707962075856070685e0i)
      (num-test (atanh 5.0e-01+0.0e+00i) 5.4930614433405484570e-1+0.0i)
      (num-test (atanh -5.0e-01+0.0e+00i) -5.4930614433405484570e-1+0.0i)
      (num-test (atanh 5.0e-01+1.19209289550781250e-07i) 5.4930614433404221383e-1+1.5894571940103932425e-7i)
      (num-test (atanh 5.0e-01-1.19209289550781250e-07i) 5.4930614433404221383e-1-1.5894571940103932425e-7i)
      (num-test (atanh -5.0e-01+1.19209289550781250e-07i) -5.4930614433404221383e-1+1.5894571940103932425e-7i)
      (num-test (atanh -5.0e-01-1.19209289550781250e-07i) -5.4930614433404221383e-1-1.5894571940103932425e-7i)
      (num-test (atanh 5.0e-01+5.0e-01i) 4.0235947810852509365e-1+5.5357435889704525151e-1i)
      (num-test (atanh 5.0e-01-5.0e-01i) 4.0235947810852509365e-1-5.5357435889704525151e-1i)
      (num-test (atanh -5.0e-01+5.0e-01i) -4.0235947810852509365e-1+5.5357435889704525151e-1i)
      (num-test (atanh -5.0e-01-5.0e-01i) -4.0235947810852509365e-1-5.5357435889704525151e-1i)
      (num-test (atanh 5.0e-01+1.0e+00i) 2.3887786125685909036e-1+8.4757566067082902713e-1i)
      (num-test (atanh 5.0e-01-1.0e+00i) 2.3887786125685909036e-1-8.4757566067082902713e-1i)
      (num-test (atanh -5.0e-01+1.0e+00i) -2.3887786125685909036e-1+8.4757566067082902713e-1i)
      (num-test (atanh -5.0e-01-1.0e+00i) -2.3887786125685909036e-1-8.4757566067082902713e-1i)
      (num-test (atanh 5.0e-01+2.0e+00i) 9.6415620202996167238e-2+1.1265564408348223487e0i)
      (num-test (atanh 5.0e-01-2.0e+00i) 9.6415620202996167238e-2-1.1265564408348223487e0i)
      (num-test (atanh -5.0e-01+2.0e+00i) -9.6415620202996167238e-2+1.1265564408348223487e0i)
      (num-test (atanh -5.0e-01-2.0e+00i) -9.6415620202996167238e-2-1.1265564408348223487e0i)
      (num-test (atanh 5.0e-01+8.3886080e+06i) 7.1054273576008756410e-15+1.5707962075856070685e0i)
      (num-test (atanh 5.0e-01-8.3886080e+06i) 7.1054273576008756410e-15-1.5707962075856070685e0i)
      (num-test (atanh -5.0e-01+8.3886080e+06i) -7.1054273576008756410e-15+1.5707962075856070685e0i)
      (num-test (atanh -5.0e-01-8.3886080e+06i) -7.1054273576008756410e-15-1.5707962075856070685e0i)
      (num-test (atanh 1.0e+00+1.19209289550781250e-07i) 8.3177661667193446012e0+7.8539819319977069731e-1i)
      (num-test (atanh 1.0e+00-1.19209289550781250e-07i) 8.3177661667193446012e0-7.8539819319977069731e-1i)
      (num-test (atanh -1.0e+00+1.19209289550781250e-07i) -8.3177661667193446012e0+7.8539819319977069731e-1i)
      (num-test (atanh -1.0e+00-1.19209289550781250e-07i) -8.3177661667193446012e0-7.8539819319977069731e-1i)
      (num-test (atanh 1.0e+00+5.0e-01i) 7.0830333601405402006e-1+9.0788749496088038670e-1i)
      (num-test (atanh 1.0e+00-5.0e-01i) 7.0830333601405402006e-1-9.0788749496088038670e-1i)
      (num-test (atanh -1.0e+00+5.0e-01i) -7.0830333601405402006e-1+9.0788749496088038670e-1i)
      (num-test (atanh -1.0e+00-5.0e-01i) -7.0830333601405402006e-1-9.0788749496088038670e-1i)
      (num-test (atanh 1.0e+00+1.0e+00i) 4.0235947810852509365e-1+1.0172219678978513677e0i)
      (num-test (atanh 1.0e+00-1.0e+00i) 4.0235947810852509365e-1-1.0172219678978513677e0i)
      (num-test (atanh -1.0e+00+1.0e+00i) -4.0235947810852509365e-1+1.0172219678978513677e0i)
      (num-test (atanh -1.0e+00-1.0e+00i) -4.0235947810852509365e-1-1.0172219678978513677e0i)
      (num-test (atanh 1.0e+00+2.0e+00i) 1.7328679513998632735e-1+1.1780972450961724644e0i)
      (num-test (atanh 1.0e+00-2.0e+00i) 1.7328679513998632735e-1-1.1780972450961724644e0i)
      (num-test (atanh -1.0e+00+2.0e+00i) -1.7328679513998632735e-1+1.1780972450961724644e0i)
      (num-test (atanh -1.0e+00-2.0e+00i) -1.7328679513998632735e-1-1.1780972450961724644e0i)
      (num-test (atanh 1.0e+00+8.3886080e+06i) 1.4210854715201599821e-14+1.5707962075856070685e0i)
      (num-test (atanh 1.0e+00-8.3886080e+06i) 1.4210854715201599821e-14-1.5707962075856070685e0i)
      (num-test (atanh -1.0e+00+8.3886080e+06i) -1.4210854715201599821e-14+1.5707962075856070685e0i)
      (num-test (atanh -1.0e+00-8.3886080e+06i) -1.4210854715201599821e-14-1.5707962075856070685e0i)
      (num-test (atanh 2.0e+00+0.0e+00i) 5.4930614433405484570e-1-1.5707963267948966192e0i)
      (num-test (atanh -2.0e+00+0.0e+00i) -5.4930614433405484570e-1+1.5707963267948966192e0i)
      (num-test (atanh 2.0e+00+1.19209289550781250e-07i) 5.4930614433405168773e-1+1.5707962870584667690e0i)
      (num-test (atanh 2.0e+00-1.19209289550781250e-07i) 5.4930614433405168773e-1-1.5707962870584667690e0i)
      (num-test (atanh -2.0e+00+1.19209289550781250e-07i) -5.4930614433405168773e-1+1.5707962870584667690e0i)
      (num-test (atanh -2.0e+00-1.19209289550781250e-07i) -5.4930614433405168773e-1-1.5707962870584667690e0i)
      (num-test (atanh 2.0e+00+5.0e-01i) 5.0037000005253101744e-1+1.4215468610018069803e0i)
      (num-test (atanh 2.0e+00-5.0e-01i) 5.0037000005253101744e-1-1.4215468610018069803e0i)
      (num-test (atanh -2.0e+00+5.0e-01i) -5.0037000005253101744e-1+1.4215468610018069803e0i)
      (num-test (atanh -2.0e+00-5.0e-01i) -5.0037000005253101744e-1-1.4215468610018069803e0i)
      (num-test (atanh 2.0e+00+1.0e+00i) 4.0235947810852509365e-1+1.3389725222944935611e0i)
      (num-test (atanh 2.0e+00-1.0e+00i) 4.0235947810852509365e-1-1.3389725222944935611e0i)
      (num-test (atanh -2.0e+00+1.0e+00i) -4.0235947810852509365e-1+1.3389725222944935611e0i)
      (num-test (atanh -2.0e+00-1.0e+00i) -4.0235947810852509365e-1-1.3389725222944935611e0i)
      (num-test (atanh 2.0e+00+2.0e+00i) 2.3887786125685909036e-1+1.3112232696716351433e0i)
      (num-test (atanh 2.0e+00-2.0e+00i) 2.3887786125685909036e-1-1.3112232696716351433e0i)
      (num-test (atanh -2.0e+00+2.0e+00i) -2.3887786125685909036e-1+1.3112232696716351433e0i)
      (num-test (atanh -2.0e+00-2.0e+00i) -2.3887786125685909036e-1-1.3112232696716351433e0i)
      (num-test (atanh 2.0e+00+8.3886080e+06i) 2.8421709430401987951e-14+1.5707962075856070685e0i)
      (num-test (atanh 2.0e+00-8.3886080e+06i) 2.8421709430401987951e-14-1.5707962075856070685e0i)
      (num-test (atanh -2.0e+00+8.3886080e+06i) -2.8421709430401987951e-14+1.5707962075856070685e0i)
      (num-test (atanh -2.0e+00-8.3886080e+06i) -2.8421709430401987951e-14-1.5707962075856070685e0i)
      (num-test (atanh 8.3886080e+06+0.0e+00i) 1.1920928955078181469e-7-1.5707963267948966192e0i)
      (num-test (atanh -8.3886080e+06+0.0e+00i) -1.1920928955078181469e-7+1.5707963267948966192e0i)
      (num-test (atanh 8.3886080e+06+1.19209289550781250e-07i) 1.1920928955078181469e-7+1.5707963267948966192e0i)
      (num-test (atanh 8.3886080e+06-1.19209289550781250e-07i) 1.1920928955078181469e-7-1.5707963267948966192e0i)
      (num-test (atanh -8.3886080e+06+1.19209289550781250e-07i) -1.1920928955078181469e-7+1.5707963267948966192e0i)
      (num-test (atanh -8.3886080e+06-1.19209289550781250e-07i) -1.1920928955078181469e-7-1.5707963267948966192e0i)
      (num-test (atanh 8.3886080e+06+5.0e-01i) 1.1920928955078139117e-7+1.5707963267948895138e0i)
      (num-test (atanh 8.3886080e+06-5.0e-01i) 1.1920928955078139117e-7-1.5707963267948895138e0i)
      (num-test (atanh -8.3886080e+06+5.0e-01i) -1.1920928955078139117e-7+1.5707963267948895138e0i)
      (num-test (atanh -8.3886080e+06-5.0e-01i) -1.1920928955078139117e-7-1.5707963267948895138e0i)
      (num-test (atanh 8.3886080e+06+1.0e+00i) 1.1920928955078012062e-7+1.5707963267948824084e0i)
      (num-test (atanh 8.3886080e+06-1.0e+00i) 1.1920928955078012062e-7-1.5707963267948824084e0i)
      (num-test (atanh -8.3886080e+06+1.0e+00i) -1.1920928955078012062e-7+1.5707963267948824084e0i)
      (num-test (atanh -8.3886080e+06-1.0e+00i) -1.1920928955078012062e-7-1.5707963267948824084e0i)
      (num-test (atanh 8.3886080e+06+2.0e+00i) 1.1920928955077503843e-7+1.5707963267948681975e0i)
      (num-test (atanh 8.3886080e+06-2.0e+00i) 1.1920928955077503843e-7-1.5707963267948681975e0i)
      (num-test (atanh -8.3886080e+06+2.0e+00i) -1.1920928955077503843e-7+1.5707963267948681975e0i)
      (num-test (atanh -8.3886080e+06-2.0e+00i) -1.1920928955077503843e-7-1.5707963267948681975e0i)
      (num-test (atanh 8.3886080e+06+8.3886080e+06i) 5.9604644775390483828e-8+1.5707962671902518438e0i)
      (num-test (atanh 8.3886080e+06-8.3886080e+06i) 5.9604644775390483828e-8-1.5707962671902518438e0i)
      (num-test (atanh -8.3886080e+06+8.3886080e+06i) -5.9604644775390483828e-8+1.5707962671902518438e0i)
      (num-test (atanh -8.3886080e+06-8.3886080e+06i) -5.9604644775390483828e-8-1.5707962671902518438e0i)

      (let ((err 0.0))
	(do ((i 0 (+ i 1))
	     (x 1.0 (+ x .1)))
	    ((= i 100))
	  (let ((y (abs (- x (cosh (acosh x))))))
	    (if (> y err) (set! err y))))
	(if (> err 1e-14)
	    (begin (display "(cosh (acosh ...)) error: ") (display err) (newline))))
      
      (let ((err 0.0))
	(do ((i 0 (+ i 1))
	     (x 1.0+i (+ x 0.1-0.1i)))
	    ((= i 100))
	  (let ((y (magnitude (- x (cosh (acosh x))))))
	    (if (> y err) (set! err y))))
	(if (> err 1e-14)
	    (begin (display "(cosh (acosh [complex] ...)) error: ") (display err) (newline))))
      
      (let ((err 0.0))
	(do ((i 0 (+ i 1))
	     (x 1.0 (+ x .1)))
	    ((= i 100))
	  (let ((y (abs (- x (sinh (asinh x))))))
	    (if (> y err) (set! err y))))
	(if (> err 1e-14)
	    (begin (display "(sinh (asinh ...)) error: ") (display err) (newline))))
      
      (let ((err 0.0))
	(do ((i 0 (+ i 1))
	     (x 1.0-i (+ x -0.1+i)))
	    ((= i 100))
	  (let ((y (magnitude (- x (sinh (asinh x))))))
	    (if (> y err) (set! err y))))
	(if (> err 1e-9)
	    (begin (display "(sinh (asinh [complex] ...)) error: ") (display err) (newline))))
      
      (let ((err 0.0))
	(do ((i 0 (+ i 1))
	     (x 0.0 (+ x .1)))
	    ((= i 100))
	  (let ((y (magnitude (- x (tanh (atanh x))))))
	    (if (> y err) (set! err y))))
	(if (> err 1e-12)
	    (begin (display "(tanh (atanh ...)) error: ") (display err) (newline))))

      ))


;; -------- sqrt
(num-test (sqrt 0) 0)
(num-test (sqrt 1) 1)
(num-test (sqrt -1) 0.0+1.0i)
(num-test (sqrt 2) 1.41421356237310)
(num-test (sqrt -2) 0.0+1.41421356237310i)
(num-test (sqrt 3) 1.73205080756888)
(num-test (sqrt -3) 0.0+1.73205080756888i)
(num-test (sqrt 10) 3.16227766016838)
(num-test (sqrt -10) 0.0+3.16227766016838i)
(num-test (sqrt 1234) 35.12833614050059)
(num-test (sqrt -1234) 0.0+35.12833614050059i)
(num-test (sqrt 1234000000) 35128.33614050059259)
(num-test (sqrt -1234000000) 0.0+35128.33614050059259i)
(num-test (sqrt 500029) 707.12728698587216)
(num-test (sqrt -500029) 0.0+707.12728698587216i)
(num-test (sqrt 362880) 602.39521910453436)
(num-test (sqrt -362880) 0.0+602.39521910453436i)
(num-test (sqrt 0/1) 0)
(num-test (sqrt 0/2) 0)
(num-test (sqrt 0/3) 0)
(num-test (sqrt 0/10) 0)
(num-test (sqrt 0/1234) 0)
(num-test (sqrt 0/1234000000) 0)
(num-test (sqrt 0/500029) 0)
(num-test (sqrt 0/362880) 0)
(num-test (sqrt 1/1) 1)
(num-test (sqrt -1/1) 0.0+1.0i)
(num-test (sqrt 1/2) 0.70710678118655)
(num-test (sqrt -1/2) 0.0+0.70710678118655i)
(num-test (sqrt 1/3) 0.57735026918963)
(num-test (sqrt -1/3) 0.0+0.57735026918963i)
(num-test (sqrt 1/10) 0.31622776601684)
(num-test (sqrt -1/10) 0.0+0.31622776601684i)
(num-test (sqrt 1/1234) 0.02846704711548)
(num-test (sqrt -1/1234) 0.0+0.02846704711548i)
(num-test (sqrt 1/1234000000) 0.00002846704712)
(num-test (sqrt -1/1234000000) 0.0+0.00002846704712i)
(num-test (sqrt 1/500029) 0.00141417255196)
(num-test (sqrt -1/500029) 0.0+0.00141417255196i)
(num-test (sqrt 1/362880) 0.00166003973519)
(num-test (sqrt -1/362880) 0.0+0.00166003973519i)
(num-test (sqrt 2/1) 1.41421356237310)
(num-test (sqrt -2/1) 0.0+1.41421356237310i)
(num-test (sqrt 2/2) 1)
(num-test (sqrt -2/2) 0.0+1.0i)
(num-test (sqrt 2/3) 0.81649658092773)
(num-test (sqrt -2/3) 0.0+0.81649658092773i)
(num-test (sqrt 2/10) 0.44721359549996)
(num-test (sqrt -2/10) 0.0+0.44721359549996i)
(num-test (sqrt 2/1234) 0.04025848411142)
(num-test (sqrt -2/1234) 0.0+0.04025848411142i)
(num-test (sqrt 2/1234000000) 0.00004025848411)
(num-test (sqrt -2/1234000000) 0.0+0.00004025848411i)
(num-test (sqrt 2/500029) 0.00199994200252)
(num-test (sqrt -2/500029) 0.0+0.00199994200252i)
(num-test (sqrt 2/362880) 0.00234765070758)
(num-test (sqrt -2/362880) 0.0+0.00234765070758i)
(num-test (sqrt 3/1) 1.73205080756888)
(num-test (sqrt -3/1) 0.0+1.73205080756888i)
(num-test (sqrt 3/2) 1.22474487139159)
(num-test (sqrt -3/2) 0.0+1.22474487139159i)
(num-test (sqrt 3/3) 1)
(num-test (sqrt -3/3) 0.0+1.0i)
(num-test (sqrt 3/10) 0.54772255750517)
(num-test (sqrt -3/10) 0.0+0.54772255750517i)
(num-test (sqrt 3/1234) 0.04930637194547)
(num-test (sqrt -3/1234) 0.0+0.04930637194547i)
(num-test (sqrt 3/1234000000) 0.00004930637195)
(num-test (sqrt -3/1234000000) 0.0+0.00004930637195i)
(num-test (sqrt 3/500029) 0.00244941871067)
(num-test (sqrt -3/500029) 0.0+0.00244941871067i)
(num-test (sqrt 3/362880) 0.00287527316393)
(num-test (sqrt -3/362880) 0.0+0.00287527316393i)
(num-test (sqrt 10/1) 3.16227766016838)
(num-test (sqrt -10/1) 0.0+3.16227766016838i)
(num-test (sqrt 10/2) 2.23606797749979)
(num-test (sqrt -10/2) 0.0+2.23606797749979i)
(num-test (sqrt 10/3) 1.82574185835055)
(num-test (sqrt -10/3) 0.0+1.82574185835055i)
(num-test (sqrt 10/10) 1)
(num-test (sqrt -10/10) 0.0+1.0i)
(num-test (sqrt 10/1234) 0.09002070714424)
(num-test (sqrt -10/1234) 0.0+0.09002070714424i)
(num-test (sqrt 10/1234000000) 0.00009002070714)
(num-test (sqrt -10/1234000000) 0.0+0.00009002070714i)
(num-test (sqrt 10/500029) 0.00447200626870)
(num-test (sqrt -10/500029) 0.0+0.00447200626870i)
(num-test (sqrt 10/362880) 0.00524950656957)
(num-test (sqrt -10/362880) 0.0+0.00524950656957i)
(num-test (sqrt 1234/1) 35.12833614050059)
(num-test (sqrt -1234/1) 0.0+35.12833614050059i)
(num-test (sqrt 1234/2) 24.83948469674844)
(num-test (sqrt -1234/2) 0.0+24.83948469674844i)
(num-test (sqrt 1234/3) 20.28135432690167)
(num-test (sqrt -1234/3) 0.0+20.28135432690167i)
(num-test (sqrt 1234/10) 11.10855526159905)
(num-test (sqrt -1234/10) 0.0+11.10855526159905i)
(num-test (sqrt 1234/1234) 1)
(num-test (sqrt -1234/1234) 0.0+1.0i)
(num-test (sqrt 1234/1234000000) 0.001)
(num-test (sqrt -1234/1234000000) 0.0+0.001i)
(num-test (sqrt 1234/500029) 0.04967752876605)
(num-test (sqrt -1234/500029) 0.0+0.04967752876605i)
(num-test (sqrt 1234/362880) 0.05831443382422)
(num-test (sqrt -1234/362880) 0.0+0.05831443382422i)
(num-test (sqrt 1234000000/1) 35128.33614050059259)
(num-test (sqrt -1234000000/1) 0.0+35128.33614050059259i)
(num-test (sqrt 1234000000/2) 24839.48469674844091)
(num-test (sqrt -1234000000/2) 0.0+24839.48469674844091i)
(num-test (sqrt 1234000000/3) 20281.35432690167727)
(num-test (sqrt -1234000000/3) 0.0+20281.35432690167727i)
(num-test (sqrt 1234000000/10) 11108.55526159905276)
(num-test (sqrt -1234000000/10) 0.0+11108.55526159905276i)
(num-test (sqrt 1234000000/1234) 1000)
(num-test (sqrt -1234000000/1234) 0.0+1000.0i)
(num-test (sqrt 1234000000/1234000000) 1)
(num-test (sqrt -1234000000/1234000000) 0.0+1.0i)
(num-test (sqrt 1234000000/500029) 49.67752876605147)
(num-test (sqrt -1234000000/500029) 0.0+49.67752876605147i)
(num-test (sqrt 1234000000/362880) 58.31443382422451)
(num-test (sqrt -1234000000/362880) 0.0+58.31443382422451i)
(num-test (sqrt 500029/1) 707.12728698587216)
(num-test (sqrt -500029/1) 0.0+707.12728698587216i)
(num-test (sqrt 500029/2) 500.01449978975609)
(num-test (sqrt -500029/2) 0.0+500.01449978975609i)
(num-test (sqrt 500029/3) 408.26012949262304)
(num-test (sqrt -500029/3) 0.0+408.26012949262304i)
(num-test (sqrt 500029/10) 223.61328225308978)
(num-test (sqrt -500029/10) 0.0+223.61328225308978i)
(num-test (sqrt 500029/1234) 20.12982579526738)
(num-test (sqrt -500029/1234) 0.0+20.12982579526738i)
(num-test (sqrt 500029/1234000000) 0.02012982579527)
(num-test (sqrt -500029/1234000000) 0.0+0.02012982579527i)
(num-test (sqrt 500029/500029) 1)
(num-test (sqrt -500029/500029) 0.0+1.0i)
(num-test (sqrt 500029/362880) 1.17385939423129)
(num-test (sqrt -500029/362880) 0.0+1.17385939423129i)
(num-test (sqrt 362880/1) 602.39521910453436)
(num-test (sqrt -362880/1) 0.0+602.39521910453436i)
(num-test (sqrt 362880/2) 425.95774438317238)
(num-test (sqrt -362880/2) 0.0+425.95774438317238i)
(num-test (sqrt 362880/3) 347.79304190854651)
(num-test (sqrt -362880/3) 0.0+347.79304190854651i)
(num-test (sqrt 362880/10) 190.49409439665052)
(num-test (sqrt -362880/10) 0.0+190.49409439665052i)
(num-test (sqrt 362880/1234) 17.14841308438784)
(num-test (sqrt -362880/1234) 0.0+17.14841308438784i)
(num-test (sqrt 362880/1234000000) 0.01714841308439)
(num-test (sqrt -362880/1234000000) 0.0+0.01714841308439i)
(num-test (sqrt 362880/500029) 0.85189078429181)
(num-test (sqrt -362880/500029) 0.0+0.85189078429181i)
(num-test (sqrt 362880/362880) 1)
(num-test (sqrt -362880/362880) 0.0+1.0i)
(num-test (sqrt 0.0) 0.0)
(num-test (sqrt 0.00000001) 0.00010000000000)
(num-test (sqrt -0.00000001) 0.0+0.00010000000000i)
(num-test (sqrt 1.0) 1.0)
(num-test (sqrt -1.0) 0.0+1.0i)
(num-test (sqrt our-pi) 1.77245385090552)
(num-test (sqrt -3.14159265358979) 0.0+1.77245385090552i)
(num-test (sqrt 2.71828182845905) 1.64872127070013)
(num-test (sqrt -2.71828182845905) 0.0+1.64872127070013i)
(num-test (sqrt 1234.0) 35.12833614050059)
(num-test (sqrt -1234.0) 0.0+35.12833614050059i)
(num-test (sqrt 1234000000.0) 35128.33614050059259)
(num-test (sqrt -1234000000.0) 0.0+35128.33614050059259i)
(num-test (sqrt 0.0+0.0i) 0.0)
(num-test (sqrt -0.0+0.0i) 0.0)
(num-test (sqrt 0.0-0.0i) 0.0)
(num-test (sqrt -0.0-0.0i) 0.0)
(num-test (sqrt 0.0+0.00000001i) 0.00007071067812+0.00007071067812i)
(num-test (sqrt -0.0+0.00000001i) 0.00007071067812+0.00007071067812i)
(num-test (sqrt 0.0-0.00000001i) 0.00007071067812-0.00007071067812i)
(num-test (sqrt -0.0-0.00000001i) 0.00007071067812-0.00007071067812i)
(num-test (sqrt 0.0+1.0i) 0.70710678118655+0.70710678118655i)
(num-test (sqrt -0.0+1.0i) 0.70710678118655+0.70710678118655i)
(num-test (sqrt 0.0-1.0i) 0.70710678118655-0.70710678118655i)
(num-test (sqrt -0.0-1.0i) 0.70710678118655-0.70710678118655i)
(num-test (sqrt 0.0+3.14159265358979i) 1.25331413731550+1.25331413731550i)
(num-test (sqrt -0.0+3.14159265358979i) 1.25331413731550+1.25331413731550i)
(num-test (sqrt 0.0-3.14159265358979i) 1.25331413731550-1.25331413731550i)
(num-test (sqrt -0.0-3.14159265358979i) 1.25331413731550-1.25331413731550i)
(num-test (sqrt 0.0+2.71828182845905i) 1.16582199079856+1.16582199079856i)
(num-test (sqrt -0.0+2.71828182845905i) 1.16582199079856+1.16582199079856i)
(num-test (sqrt 0.0-2.71828182845905i) 1.16582199079856-1.16582199079856i)
(num-test (sqrt -0.0-2.71828182845905i) 1.16582199079856-1.16582199079856i)
(num-test (sqrt 0.0+1234.0i) 24.83948469674844+24.83948469674844i)
(num-test (sqrt -0.0+1234.0i) 24.83948469674844+24.83948469674844i)
(num-test (sqrt 0.0-1234.0i) 24.83948469674844-24.83948469674844i)
(num-test (sqrt -0.0-1234.0i) 24.83948469674844-24.83948469674844i)
(num-test (sqrt 0.0+1234000000.0i) 24839.48469674844091+24839.48469674844091i)
(num-test (sqrt -0.0+1234000000.0i) 24839.48469674844091+24839.48469674844091i)
(num-test (sqrt 0.0-1234000000.0i) 24839.48469674844091-24839.48469674844091i)
(num-test (sqrt -0.0-1234000000.0i) 24839.48469674844091-24839.48469674844091i)
(num-test (sqrt 0.00000001+0.0i) 0.00010000000000)
(num-test (sqrt -0.00000001+0.0i) 0.0+0.00010000000000i)
(num-test (sqrt 0.00000001-0.0i) 0.00010000000000)
(num-test (sqrt -0.00000001-0.0i) 0.0+0.00010000000000i)
(num-test (sqrt 0.00000001+0.00000001i) 0.00010986841135+0.00004550898606i)
(num-test (sqrt -0.00000001+0.00000001i) 0.00004550898606+0.00010986841135i)
(num-test (sqrt 0.00000001-0.00000001i) 0.00010986841135-0.00004550898606i)
(num-test (sqrt -0.00000001-0.00000001i) 0.00004550898606-0.00010986841135i)
(num-test (sqrt 0.00000001+1.0i) 0.70710678472208+0.70710677765101i)
(num-test (sqrt -0.00000001+1.0i) 0.70710677765101+0.70710678472208i)
(num-test (sqrt 0.00000001-1.0i) 0.70710678472208-0.70710677765101i)
(num-test (sqrt -0.00000001-1.0i) 0.70710677765101-0.70710678472208i)
(num-test (sqrt 0.00000001+3.14159265358979i) 1.25331413931021+1.25331413532079i)
(num-test (sqrt -0.00000001+3.14159265358979i) 1.25331413532079+1.25331413931021i)
(num-test (sqrt 0.00000001-3.14159265358979i) 1.25331413931021-1.25331413532079i)
(num-test (sqrt -0.00000001-3.14159265358979i) 1.25331413532079-1.25331413931021i)
(num-test (sqrt 0.00000001+2.71828182845905i) 1.16582199294297+1.16582198865415i)
(num-test (sqrt -0.00000001+2.71828182845905i) 1.16582198865415+1.16582199294297i)
(num-test (sqrt 0.00000001-2.71828182845905i) 1.16582199294297-1.16582198865415i)
(num-test (sqrt -0.00000001-2.71828182845905i) 1.16582198865415-1.16582199294297i)
(num-test (sqrt 0.00000001+1234.0i) 24.83948469684909+24.83948469664779i)
(num-test (sqrt -0.00000001+1234.0i) 24.83948469664779+24.83948469684909i)
(num-test (sqrt 0.00000001-1234.0i) 24.83948469684909-24.83948469664779i)
(num-test (sqrt -0.00000001-1234.0i) 24.83948469664779-24.83948469684909i)
(num-test (sqrt 0.00000001+1234000000.0i) 24839.48469674844091+24839.48469674844091i)
(num-test (sqrt -0.00000001+1234000000.0i) 24839.48469674844091+24839.48469674844091i)
(num-test (sqrt 0.00000001-1234000000.0i) 24839.48469674844091-24839.48469674844091i)
(num-test (sqrt -0.00000001-1234000000.0i) 24839.48469674844091-24839.48469674844091i)
(num-test (sqrt 1.0+0.0i) 1.0)
(num-test (sqrt -1.0+0.0i) 0.0+1.0i)
(num-test (sqrt 1.0-0.0i) 1.0)
(num-test (sqrt -1.0-0.0i) 0.0+1.0i)
(num-test (sqrt 1.0+0.00000001i) 1.0+0.00000000500000i)
(num-test (sqrt -1.0+0.00000001i) 0.00000000500000+1.0i)
(num-test (sqrt 1.0-0.00000001i) 1.0-0.00000000500000i)
(num-test (sqrt -1.0-0.00000001i) 0.00000000500000-1.0i)
(num-test (sqrt 1.0+1.0i) 1.09868411346781+0.45508986056223i)
(num-test (sqrt -1.0+1.0i) 0.45508986056223+1.09868411346781i)
(num-test (sqrt 1.0-1.0i) 1.09868411346781-0.45508986056223i)
(num-test (sqrt -1.0-1.0i) 0.45508986056223-1.09868411346781i)
(num-test (sqrt 1.0+3.14159265358979i) 1.46576060621706+1.07165953303174i)
(num-test (sqrt -1.0+3.14159265358979i) 1.07165953303174+1.46576060621706i)
(num-test (sqrt 1.0-3.14159265358979i) 1.46576060621706-1.07165953303174i)
(num-test (sqrt -1.0-3.14159265358979i) 1.07165953303174-1.46576060621706i)
(num-test (sqrt 1.0+2.71828182845905i) 1.39577697566445+0.97375220964833i)
(num-test (sqrt -1.0+2.71828182845905i) 0.97375220964833+1.39577697566445i)
(num-test (sqrt 1.0-2.71828182845905i) 1.39577697566445-0.97375220964833i)
(num-test (sqrt -1.0-2.71828182845905i) 0.97375220964833-1.39577697566445i)
(num-test (sqrt 1.0+1234.0i) 24.84955135597340+24.82942211557006i)
(num-test (sqrt -1.0+1234.0i) 24.82942211557006+24.84955135597340i)
(num-test (sqrt 1.0-1234.0i) 24.84955135597340-24.82942211557006i)
(num-test (sqrt -1.0-1234.0i) 24.82942211557006-24.84955135597340i)
(num-test (sqrt 1.0+1234000000.0i) 24839.48470681306208+24839.48468668381975i)
(num-test (sqrt -1.0+1234000000.0i) 24839.48468668381975+24839.48470681306208i)
(num-test (sqrt 1.0-1234000000.0i) 24839.48470681306208-24839.48468668381975i)
(num-test (sqrt -1.0-1234000000.0i) 24839.48468668381975-24839.48470681306208i)
(num-test (sqrt 3.14159265358979+0.0i) 1.77245385090552)
(num-test (sqrt -3.14159265358979+0.0i) 0.0+1.77245385090552i)
(num-test (sqrt 3.14159265358979-0.0i) 1.77245385090552)
(num-test (sqrt -3.14159265358979-0.0i) 0.0+1.77245385090552i)
(num-test (sqrt 3.14159265358979+0.00000001i) 1.77245385090552+0.00000000282095i)
(num-test (sqrt -3.14159265358979+0.00000001i) 0.00000000282095+1.77245385090552i)
(num-test (sqrt 3.14159265358979-0.00000001i) 1.77245385090552-0.00000000282095i)
(num-test (sqrt -3.14159265358979-0.00000001i) 0.00000000282095-1.77245385090552i)
(num-test (sqrt 3.14159265358979+1.0i) 1.79422698718214+0.27867154132224i)
(num-test (sqrt -3.14159265358979+1.0i) 0.27867154132224+1.79422698718214i)
(num-test (sqrt 3.14159265358979-1.0i) 1.79422698718214-0.27867154132224i)
(num-test (sqrt -3.14159265358979-1.0i) 0.27867154132224-1.79422698718214i)
(num-test (sqrt 3.14159265358979+3.14159265358979i) 1.94736688784473+0.80662577586157i)
(num-test (sqrt -3.14159265358979+3.14159265358979i) 0.80662577586157+1.94736688784473i)
(num-test (sqrt 3.14159265358979-3.14159265358979i) 1.94736688784473-0.80662577586157i)
(num-test (sqrt -3.14159265358979-3.14159265358979i) 0.80662577586157-1.94736688784473i)
(num-test (sqrt 3.14159265358979+2.71828182845905i) 1.90996689184696+0.71160443672153i)
(num-test (sqrt -3.14159265358979+2.71828182845905i) 0.71160443672153+1.90996689184696i)
(num-test (sqrt 3.14159265358979-2.71828182845905i) 1.90996689184696-0.71160443672153i)
(num-test (sqrt -3.14159265358979-2.71828182845905i) 0.71160443672153-1.90996689184696i)
(num-test (sqrt 3.14159265358979+1234.0i) 24.87112373493049+24.80788590719961i)
(num-test (sqrt -3.14159265358979+1234.0i) 24.80788590719961+24.87112373493049i)
(num-test (sqrt 3.14159265358979-1234.0i) 24.87112373493049-24.80788590719961i)
(num-test (sqrt -3.14159265358979-1234.0i) 24.80788590719961-24.87112373493049i)
(num-test (sqrt 3.14159265358979+1234000000.0i) 24839.48472836738074+24839.48466512950108i)
(num-test (sqrt -3.14159265358979+1234000000.0i) 24839.48466512950108+24839.48472836738074i)
(num-test (sqrt 3.14159265358979-1234000000.0i) 24839.48472836738074-24839.48466512950108i)
(num-test (sqrt -3.14159265358979-1234000000.0i) 24839.48466512950108-24839.48472836738074i)
(num-test (sqrt 2.71828182845905+0.0i) 1.64872127070013)
(num-test (sqrt -2.71828182845905+0.0i) 0.0+1.64872127070013i)
(num-test (sqrt 2.71828182845905-0.0i) 1.64872127070013)
(num-test (sqrt -2.71828182845905-0.0i) 0.0+1.64872127070013i)
(num-test (sqrt 2.71828182845905+0.00000001i) 1.64872127070013+0.00000000303265i)
(num-test (sqrt -2.71828182845905+0.00000001i) 0.00000000303265+1.64872127070013i)
(num-test (sqrt 2.71828182845905-0.00000001i) 1.64872127070013-0.00000000303265i)
(num-test (sqrt -2.71828182845905-0.00000001i) 0.00000000303265-1.64872127070013i)
(num-test (sqrt 2.71828182845905+1.0i) 1.67551015515410+0.29841657387867i)
(num-test (sqrt -2.71828182845905+1.0i) 0.29841657387867+1.67551015515410i)
(num-test (sqrt 2.71828182845905-1.0i) 1.67551015515410-0.29841657387867i)
(num-test (sqrt -2.71828182845905-1.0i) 0.29841657387867-1.67551015515410i)
(num-test (sqrt 2.71828182845905+3.14159265358979i) 1.85373086379501+0.84737021833856i)
(num-test (sqrt -2.71828182845905+3.14159265358979i) 0.84737021833856+1.85373086379501i)
(num-test (sqrt 2.71828182845905-3.14159265358979i) 1.85373086379501-0.84737021833856i)
(num-test (sqrt -2.71828182845905-3.14159265358979i) 0.84737021833856-1.85373086379501i)
(num-test (sqrt 2.71828182845905+2.71828182845905i) 1.81142386765469+0.75031633318890i)
(num-test (sqrt -2.71828182845905+2.71828182845905i) 0.75031633318890+1.81142386765469i)
(num-test (sqrt 2.71828182845905-2.71828182845905i) 1.81142386765469-0.75031633318890i)
(num-test (sqrt -2.71828182845905-2.71828182845905i) 0.75031633318890-1.81142386765469i)
(num-test (sqrt 2.71828182845905+1234.0i) 24.86685822304219+24.81214130333015i)
(num-test (sqrt -2.71828182845905+1234.0i) 24.81214130333015+24.86685822304219i)
(num-test (sqrt 2.71828182845905-1234.0i) 24.86685822304219-24.81214130333015i)
(num-test (sqrt -2.71828182845905-1234.0i) 24.81214130333015-24.86685822304219i)
(num-test (sqrt 2.71828182845905+1234000000.0i) 24839.48472410691829+24839.48466938996353i)
(num-test (sqrt -2.71828182845905+1234000000.0i) 24839.48466938996353+24839.48472410691829i)
(num-test (sqrt 2.71828182845905-1234000000.0i) 24839.48472410691829-24839.48466938996353i)
(num-test (sqrt -2.71828182845905-1234000000.0i) 24839.48466938996353-24839.48472410691829i)
(num-test (sqrt 1234.0+0.0i) 35.12833614050059)
(num-test (sqrt -1234.0+0.0i) 0.0+35.12833614050059i)
(num-test (sqrt 1234.0-0.0i) 35.12833614050059)
(num-test (sqrt -1234.0-0.0i) 0.0+35.12833614050059i)
(num-test (sqrt 1234.0+0.00000001i) 35.12833614050059+0.00000000014234i)
(num-test (sqrt -1234.0+0.00000001i) 0.00000000014234+35.12833614050059i)
(num-test (sqrt 1234.0-0.00000001i) 35.12833614050059-0.00000000014234i)
(num-test (sqrt -1234.0-0.00000001i) 0.00000000014234-35.12833614050059i)
(num-test (sqrt 1234.0+1.0i) 35.12833902411499+0.01423352238934i)
(num-test (sqrt -1234.0+1.0i) 0.01423352238934+35.12833902411499i)
(num-test (sqrt 1234.0-1.0i) 35.12833902411499-0.01423352238934i)
(num-test (sqrt -1234.0-1.0i) 0.01423352238934-35.12833902411499i)
(num-test (sqrt 1234.0+3.14159265358979i) 35.12836460058208+0.04471589681601i)
(num-test (sqrt -1234.0+3.14159265358979i) 0.04471589681601+35.12836460058208i)
(num-test (sqrt 1234.0-3.14159265358979i) 35.12836460058208-0.04471589681601i)
(num-test (sqrt -1234.0-3.14159265358979i) 0.04471589681601-35.12836460058208i)
(num-test (sqrt 1234.0+2.71828182845905i) 35.12835744766116+0.03869070497402i)
(num-test (sqrt -1234.0+2.71828182845905i) 0.03869070497402+35.12835744766116i)
(num-test (sqrt 1234.0-2.71828182845905i) 35.12835744766116-0.03869070497402i)
(num-test (sqrt -1234.0-2.71828182845905i) 0.03869070497402-35.12835744766116i)
(num-test (sqrt 1234.0+1234.0i) 38.59494485012512+15.98654959596347i)
(num-test (sqrt -1234.0+1234.0i) 15.98654959596347+38.59494485012512i)
(num-test (sqrt 1234.0-1234.0i) 38.59494485012512-15.98654959596347i)
(num-test (sqrt -1234.0-1234.0i) 15.98654959596347-38.59494485012512i)
(num-test (sqrt 1234.0+1234000000.0i) 24839.49711649389428+24839.47227700919757i)
(num-test (sqrt -1234.0+1234000000.0i) 24839.47227700919757+24839.49711649389428i)
(num-test (sqrt 1234.0-1234000000.0i) 24839.49711649389428-24839.47227700919757i)
(num-test (sqrt -1234.0-1234000000.0i) 24839.47227700919757-24839.49711649389428i)
(num-test (sqrt 1234000000.0+0.0i) 35128.33614050059259)
(num-test (sqrt -1234000000.0+0.0i) 0.0+35128.33614050059259i)
(num-test (sqrt 1234000000.0-0.0i) 35128.33614050059259)
(num-test (sqrt -1234000000.0-0.0i) 0.0+35128.33614050059259i)
(num-test (sqrt 1234000000.0+0.00000001i) 35128.33614050059259+0.00000000000014i)
(num-test (sqrt -1234000000.0+0.00000001i) 0.00000000000014+35128.33614050059259i)
(num-test (sqrt 1234000000.0-0.00000001i) 35128.33614050059259-0.00000000000014i)
(num-test (sqrt -1234000000.0-0.00000001i) 0.00000000000014-35128.33614050059259i)
(num-test (sqrt 1234000000.0+1.0i) 35128.33614050059259+0.00001423352356i)
(num-test (sqrt -1234000000.0+1.0i) 0.00001423352356+35128.33614050059259i)
(num-test (sqrt 1234000000.0-1.0i) 35128.33614050059259-0.00001423352356i)
(num-test (sqrt -1234000000.0-1.0i) 0.00001423352356-35128.33614050059259i)
(num-test (sqrt 1234000000.0+3.14159265358979i) 35128.33614050059259+0.00004471593304i)
(num-test (sqrt -1234000000.0+3.14159265358979i) 0.00004471593304+35128.33614050059259i)
(num-test (sqrt 1234000000.0-3.14159265358979i) 35128.33614050059259-0.00004471593304i)
(num-test (sqrt -1234000000.0-3.14159265358979i) 0.00004471593304-35128.33614050059259i)
(num-test (sqrt 1234000000.0+2.71828182845905i) 35128.33614050059259+0.00003869072844i)
(num-test (sqrt -1234000000.0+2.71828182845905i) 0.00003869072844+35128.33614050059259i)
(num-test (sqrt 1234000000.0-2.71828182845905i) 35128.33614050059259-0.00003869072844i)
(num-test (sqrt -1234000000.0-2.71828182845905i) 0.00003869072844-35128.33614050059259i)
(num-test (sqrt 1234000000.0+1234.0i) 35128.33614050497999+0.01756416807025i)
(num-test (sqrt -1234000000.0+1234.0i) 0.01756416807025+35128.33614050497999i)
(num-test (sqrt 1234000000.0-1234.0i) 35128.33614050497999-0.01756416807025i)
(num-test (sqrt -1234000000.0-1234.0i) 0.01756416807025-35128.33614050497999i)
(num-test (sqrt 1234000000.0+1234000000.0i) 38594.94485012512450+15986.54959596346634i)
(num-test (sqrt -1234000000.0+1234000000.0i) 15986.54959596346634+38594.94485012512450i)
(num-test (sqrt 1234000000.0-1234000000.0i) 38594.94485012512450-15986.54959596346634i)
(num-test (sqrt -1234000000.0-1234000000.0i) 15986.54959596346634-38594.94485012512450i)
(num-test (sqrt 2.2250739e-308) 1.4916681e-154)
(num-test (sqrt 1.7976931e+308) 1.3407808e+154)
(num-test (sqrt 0.0e+00+0.0e+00i) 0e0+0.0i)
(num-test (sqrt 0.0e+00+1.19209289550781250e-07i) 2.44140625e-4+2.44140625e-4i)
(num-test (sqrt 0.0e+00-1.19209289550781250e-07i) 2.44140625e-4-2.44140625e-4i)
(num-test (sqrt 0.0e+00+5.0e-01i) 5e-1+5e-1i)
(num-test (sqrt 0.0e+00-5.0e-01i) 5e-1-5e-1i)
(num-test (sqrt 0.0e+00+1.0e+00i) 7.0710678118654752440e-1+7.0710678118654752440e-1i)
(num-test (sqrt 0.0e+00-1.0e+00i) 7.0710678118654752440e-1-7.0710678118654752440e-1i)
(num-test (sqrt 0.0e+00+2.0e+00i) 1+1i)
(num-test (sqrt 0.0e+00-2.0e+00i) 1-1i)
(num-test (sqrt 0.0e+00+8.3886080e+06i) 2048+2048i)
(num-test (sqrt 0.0e+00-8.3886080e+06i) 2048-2048i)
(num-test (sqrt 1.19209289550781250e-07+0.0e+00i) 3.4526698300124390840e-4+0.0i)
(num-test (sqrt -1.19209289550781250e-07+0.0e+00i) 0+3.4526698300124390840e-4i)
(num-test (sqrt 1.19209289550781250e-07+1.19209289550781250e-07i) 3.7933934912842707699e-4+1.5712750315077700799e-4i)
(num-test (sqrt 1.19209289550781250e-07-1.19209289550781250e-07i) 3.7933934912842707699e-4-1.5712750315077700799e-4i)
(num-test (sqrt -1.19209289550781250e-07+1.19209289550781250e-07i) 1.5712750315077700799e-4+3.7933934912842707699e-4i)
(num-test (sqrt -1.19209289550781250e-07-1.19209289550781250e-07i) 1.5712750315077700799e-4-3.7933934912842707699e-4i)
(num-test (sqrt 1.19209289550781250e-07+5.0e-01i) 5.0000005960464832810e-1+4.9999994039535877732e-1i)
(num-test (sqrt 1.19209289550781250e-07-5.0e-01i) 5.0000005960464832810e-1-4.9999994039535877732e-1i)
(num-test (sqrt -1.19209289550781250e-07+5.0e-01i) 4.9999994039535877732e-1+5.0000005960464832810e-1i)
(num-test (sqrt -1.19209289550781250e-07-5.0e-01i) 4.9999994039535877732e-1-5.0000005960464832810e-1i)
(num-test (sqrt 1.19209289550781250e-07+1.0e+00i) 7.0710682333339729137e-1+7.0710673903970026958e-1i)
(num-test (sqrt 1.19209289550781250e-07-1.0e+00i) 7.0710682333339729137e-1-7.0710673903970026958e-1i)
(num-test (sqrt -1.19209289550781250e-07+1.0e+00i) 7.0710673903970026958e-1+7.0710682333339729137e-1i)
(num-test (sqrt -1.19209289550781250e-07-1.0e+00i) 7.0710673903970026958e-1-7.0710682333339729137e-1i)
(num-test (sqrt 1.19209289550781250e-07+2.0e+00i) 1.0000000298023228318e0+9.9999997019767805639e-1i)
(num-test (sqrt 1.19209289550781250e-07-2.0e+00i) 1.0000000298023228318e0-9.9999997019767805639e-1i)
(num-test (sqrt -1.19209289550781250e-07+2.0e+00i) 9.9999997019767805639e-1+1.0000000298023228318e0i)
(num-test (sqrt -1.19209289550781250e-07-2.0e+00i) 9.9999997019767805639e-1-1.0000000298023228318e0i)
(num-test (sqrt 1.19209289550781250e-07+8.3886080e+06i) 2.0480000000000145519e3+2.0479999999999854481e3i)
(num-test (sqrt 1.19209289550781250e-07-8.3886080e+06i) 2.0480000000000145519e3-2.0479999999999854481e3i)
(num-test (sqrt -1.19209289550781250e-07+8.3886080e+06i) 2.0479999999999854481e3+2.0480000000000145519e3i)
(num-test (sqrt -1.19209289550781250e-07-8.3886080e+06i) 2.0479999999999854481e3-2.0480000000000145519e3i)
(num-test (sqrt 5.0e-01+0.0e+00i) 7.0710678118654752440e-1+0.0i)
(num-test (sqrt -5.0e-01+0.0e+00i) 0+7.0710678118654752440e-1i)
(num-test (sqrt 5.0e-01+1.19209289550781250e-07i) 7.0710678118655254870e-1+8.4293697021787464631e-8i)
(num-test (sqrt 5.0e-01-1.19209289550781250e-07i) 7.0710678118655254870e-1-8.4293697021787464631e-8i)
(num-test (sqrt -5.0e-01+1.19209289550781250e-07i) 8.4293697021787464631e-8+7.0710678118655254870e-1i)
(num-test (sqrt -5.0e-01-1.19209289550781250e-07i) 8.4293697021787464631e-8-7.0710678118655254870e-1i)
(num-test (sqrt 5.0e-01+5.0e-01i) 7.7688698701501865367e-1+3.2179712645279131237e-1i)
(num-test (sqrt 5.0e-01-5.0e-01i) 7.7688698701501865367e-1-3.2179712645279131237e-1i)
(num-test (sqrt -5.0e-01+5.0e-01i) 3.2179712645279131237e-1+7.7688698701501865367e-1i)
(num-test (sqrt -5.0e-01-5.0e-01i) 3.2179712645279131237e-1-7.7688698701501865367e-1i)
(num-test (sqrt 5.0e-01+1.0e+00i) 8.9945371997393363613e-1+5.5589297025142117199e-1i)
(num-test (sqrt 5.0e-01-1.0e+00i) 8.9945371997393363613e-1-5.5589297025142117199e-1i)
(num-test (sqrt -5.0e-01+1.0e+00i) 5.5589297025142117199e-1+8.9945371997393363613e-1i)
(num-test (sqrt -5.0e-01-1.0e+00i) 5.5589297025142117199e-1-8.9945371997393363613e-1i)
(num-test (sqrt 5.0e-01+2.0e+00i) 1.1317139242778694103e0+8.8361553087551326576e-1i)
(num-test (sqrt 5.0e-01-2.0e+00i) 1.1317139242778694103e0-8.8361553087551326576e-1i)
(num-test (sqrt -5.0e-01+2.0e+00i) 8.8361553087551326576e-1+1.1317139242778694103e0i)
(num-test (sqrt -5.0e-01-2.0e+00i) 8.8361553087551326576e-1-1.1317139242778694103e0i)
(num-test (sqrt 5.0e-01+8.3886080e+06i) 2.0480000610351571595e3+2.0479999389648446595e3i)
(num-test (sqrt 5.0e-01-8.3886080e+06i) 2.0480000610351571595e3-2.0479999389648446595e3i)
(num-test (sqrt -5.0e-01+8.3886080e+06i) 2.0479999389648446595e3+2.0480000610351571595e3i)
(num-test (sqrt -5.0e-01-8.3886080e+06i) 2.0479999389648446595e3-2.0480000610351571595e3i)
(num-test (sqrt 1.0e+00+0.0e+00i) 1e0+0.0i)
(num-test (sqrt -1.0e+00+0.0e+00i) 0+1i)
(num-test (sqrt 1.0e+00+1.19209289550781250e-07i) 1.0000000000000017764e0+5.9604644775390519121e-8i)
(num-test (sqrt 1.0e+00-1.19209289550781250e-07i) 1.0000000000000017764e0-5.9604644775390519121e-8i)
(num-test (sqrt -1.0e+00+1.19209289550781250e-07i) 5.9604644775390519121e-8+1.0000000000000017764e0i)
(num-test (sqrt -1.0e+00-1.19209289550781250e-07i) 5.9604644775390519121e-8-1.0000000000000017764e0i)
(num-test (sqrt 1.0e+00+5.0e-01i) 1.0290855136357461252e0+2.4293413587832283909e-1i)
(num-test (sqrt 1.0e+00-5.0e-01i) 1.0290855136357461252e0-2.4293413587832283909e-1i)
(num-test (sqrt -1.0e+00+5.0e-01i) 2.4293413587832283909e-1+1.0290855136357461252e0i)
(num-test (sqrt -1.0e+00-5.0e-01i) 2.4293413587832283909e-1-1.0290855136357461252e0i)
(num-test (sqrt 1.0e+00+1.0e+00i) 1.0986841134678099660e0+4.5508986056222734130e-1i)
(num-test (sqrt 1.0e+00-1.0e+00i) 1.0986841134678099660e0-4.5508986056222734130e-1i)
(num-test (sqrt -1.0e+00+1.0e+00i) 4.5508986056222734130e-1+1.0986841134678099660e0i)
(num-test (sqrt -1.0e+00-1.0e+00i) 4.5508986056222734130e-1-1.0986841134678099660e0i)
(num-test (sqrt 1.0e+00+2.0e+00i) 1.2720196495140689643e0+7.8615137775742328607e-1i)
(num-test (sqrt 1.0e+00-2.0e+00i) 1.2720196495140689643e0-7.8615137775742328607e-1i)
(num-test (sqrt -1.0e+00+2.0e+00i) 7.8615137775742328607e-1+1.2720196495140689643e0i)
(num-test (sqrt -1.0e+00-2.0e+00i) 7.8615137775742328607e-1-1.2720196495140689643e0i)
(num-test (sqrt 1.0e+00+8.3886080e+06i) 2.0480001220703161380e3+2.0479998779296911380e3i)
(num-test (sqrt 1.0e+00-8.3886080e+06i) 2.0480001220703161380e3-2.0479998779296911380e3i)
(num-test (sqrt -1.0e+00+8.3886080e+06i) 2.0479998779296911380e3+2.0480001220703161380e3i)
(num-test (sqrt -1.0e+00-8.3886080e+06i) 2.0479998779296911380e3-2.0480001220703161380e3i)
(num-test (sqrt 2.0e+00+0.0e+00i) 1.4142135623730950488e0+0.0i)
(num-test (sqrt -2.0e+00+0.0e+00i) 0+1.4142135623730950488e0i)
(num-test (sqrt 2.0e+00+1.19209289550781250e-07i) 1.4142135623730956768e0+4.2146848510894013070e-8i)
(num-test (sqrt 2.0e+00-1.19209289550781250e-07i) 1.4142135623730956768e0-4.2146848510894013070e-8i)
(num-test (sqrt -2.0e+00+1.19209289550781250e-07i) 4.2146848510894013070e-8+1.4142135623730956768e0i)
(num-test (sqrt -2.0e+00-1.19209289550781250e-07i) 4.2146848510894013070e-8-1.4142135623730956768e0i)
(num-test (sqrt 2.0e+00+5.0e-01i) 1.4250531240639470060e0+1.7543205637629383228e-1i)
(num-test (sqrt 2.0e+00-5.0e-01i) 1.4250531240639470060e0-1.7543205637629383228e-1i)
(num-test (sqrt -2.0e+00+5.0e-01i) 1.7543205637629383228e-1+1.4250531240639470060e0i)
(num-test (sqrt -2.0e+00-5.0e-01i) 1.7543205637629383228e-1-1.4250531240639470060e0i)
(num-test (sqrt 2.0e+00+1.0e+00i) 1.4553466902253548081e0+3.4356074972251246414e-1i)
(num-test (sqrt 2.0e+00-1.0e+00i) 1.4553466902253548081e0-3.4356074972251246414e-1i)
(num-test (sqrt -2.0e+00+1.0e+00i) 3.4356074972251246414e-1+1.4553466902253548081e0i)
(num-test (sqrt -2.0e+00-1.0e+00i) 3.4356074972251246414e-1-1.4553466902253548081e0i)
(num-test (sqrt 2.0e+00+2.0e+00i) 1.5537739740300373073e0+6.4359425290558262474e-1i)
(num-test (sqrt 2.0e+00-2.0e+00i) 1.5537739740300373073e0-6.4359425290558262474e-1i)
(num-test (sqrt -2.0e+00+2.0e+00i) 6.4359425290558262474e-1+1.5537739740300373073e0i)
(num-test (sqrt -2.0e+00-2.0e+00i) 6.4359425290558262474e-1-1.5537739740300373073e0i)
(num-test (sqrt 2.0e+00+8.3886080e+06i) 2.0480002441406395519e3+2.0479997558593895519e3i)
(num-test (sqrt 2.0e+00-8.3886080e+06i) 2.0480002441406395519e3-2.0479997558593895519e3i)
(num-test (sqrt -2.0e+00+8.3886080e+06i) 2.0479997558593895519e3+2.0480002441406395519e3i)
(num-test (sqrt -2.0e+00-8.3886080e+06i) 2.0479997558593895519e3-2.0480002441406395519e3i)
(num-test (sqrt 8.3886080e+06+0.0e+00i) 2.8963093757400986599e3+0.0i)
(num-test (sqrt -8.3886080e+06+0.0e+00i) 0+2.8963093757400986599e3i)
(num-test (sqrt 8.3886080e+06+1.19209289550781250e-07i) 2.8963093757400986599e3+2.0579515874459976458e-11i)
(num-test (sqrt 8.3886080e+06-1.19209289550781250e-07i) 2.8963093757400986599e3-2.0579515874459976458e-11i)
(num-test (sqrt -8.3886080e+06+1.19209289550781250e-07i) 2.0579515874459976458e-11+2.8963093757400986599e3i)
(num-test (sqrt -8.3886080e+06-1.19209289550781250e-07i) 2.0579515874459976458e-11-2.8963093757400986599e3i)
(num-test (sqrt 8.3886080e+06+5.0e-01i) 2.8963093757400999462e3+8.6316745750310938767e-5i)
(num-test (sqrt 8.3886080e+06-5.0e-01i) 2.8963093757400999462e3-8.6316745750310938767e-5i)
(num-test (sqrt -8.3886080e+06+5.0e-01i) 8.6316745750310938767e-5+2.8963093757400999462e3i)
(num-test (sqrt -8.3886080e+06-5.0e-01i) 8.6316745750310938767e-5-2.8963093757400999462e3i)
(num-test (sqrt 8.3886080e+06+1.0e+00i) 2.8963093757401038048e3+1.7263349150062164754e-4i)
(num-test (sqrt 8.3886080e+06-1.0e+00i) 2.8963093757401038048e3-1.7263349150062164754e-4i)
(num-test (sqrt -8.3886080e+06+1.0e+00i) 1.7263349150062164754e-4+2.8963093757401038048e3i)
(num-test (sqrt -8.3886080e+06-1.0e+00i) 1.7263349150062164754e-4-2.8963093757401038048e3i)
(num-test (sqrt 8.3886080e+06+2.0e+00i) 2.8963093757401192395e3+3.4526698300124145513e-4i)
(num-test (sqrt 8.3886080e+06-2.0e+00i) 2.8963093757401192395e3-3.4526698300124145513e-4i)
(num-test (sqrt -8.3886080e+06+2.0e+00i) 3.4526698300124145513e-4+2.8963093757401192395e3i)
(num-test (sqrt -8.3886080e+06-2.0e+00i) 3.4526698300124145513e-4-2.8963093757401192395e3i)
(num-test (sqrt 8.3886080e+06+8.3886080e+06i) 3.1821290988135164054e3+1.3180810299506332155e3i)
(num-test (sqrt 8.3886080e+06-8.3886080e+06i) 3.1821290988135164054e3-1.3180810299506332155e3i)
(num-test (sqrt -8.3886080e+06+8.3886080e+06i) 1.3180810299506332155e3+3.1821290988135164054e3i)
(num-test (sqrt -8.3886080e+06-8.3886080e+06i) 1.3180810299506332155e3-3.1821290988135164054e3i)
(num-test (sqrt -1.0e+01) 0+3.1622776601683793320e0i)
(num-test (sqrt -2.0e+00) 0+1.4142135623730950488e0i)
(num-test (sqrt -1.0e+00) 0+1i)
(num-test (sqrt -7.50e-01) 0+8.6602540378443864676e-1i)
(num-test (sqrt -5.0e-01) 0+7.0710678118654752440e-1i)
(num-test (sqrt -1.250e-01) 0+3.5355339059327376220e-1i)
(num-test (sqrt -3.45266983001243932001e-04) 0+1.8581361171917517303e-2i)
(num-test (sqrt -1.19209289550781250e-07) 0+3.4526698300124390840e-4i)
(num-test (sqrt 0.0e+00) 0e0+0.0i)
(num-test (sqrt 1.19209289550781250e-07) 3.4526698300124390840e-4+0.0i)
(num-test (sqrt 3.45266983001243932001e-04) 1.8581361171917517303e-2+0.0i)
(num-test (sqrt 1.250e-01) 3.5355339059327376220e-1+0.0i)
(num-test (sqrt 5.0e-01) 7.0710678118654752440e-1+0.0i)
(num-test (sqrt 7.50e-01) 8.6602540378443864676e-1+0.0i)
(num-test (sqrt 1.0e+00) 1e0+0.0i)
(num-test (sqrt 2.0e+00) 1.4142135623730950488e0+0.0i)
(num-test (sqrt 1.0e+01) 3.1622776601683793320e0+0.0i)
(num-test (sqrt 9) 3)
(num-test (sqrt -9.0) 0.0+3.0i)
(num-test (sqrt (sqrt (sqrt 256))) 2)
(num-test (sqrt (sqrt (sqrt 1/256))) 1/2)

(if (and (integer? (sqrt 4))
	 (exact? (sqrt 4)))
    (begin
      (for-each
       (lambda (n sqn)
	 (if (positive? n) ; in case 32 bit int
	     (let ((val (sqrt n)))
	       (if (or (not (integer? val))
		       (not (eqv? sqn val)))
		   (begin
		     (display "(sqrt ") (display n) (display ") expected ") 
		     (display sqn) (display " but got ") (display val) (newline))))))
       (list 9 491401 19439281 1248844921 235565593201)
       (list 3 701 4409 35339 485351))

      (for-each
       (lambda (n)
	 (if (positive? n)
	     (let ((val (sqrt n)))
	       (if (or (integer? val)
		       (> (abs (- (* val val) n)) .001))
		   (begin
		     (display "(sqrt ") (display n) (display ") expected ") 
		     (display (sqrt (exact->inexact n))) (display " but got ") (display val) (newline))))))
       (list 10 491400 19439282 1248844920 235565593200))

      (test (eqv? (expt 2 3) 8) #t)
      ;(test (eqv? (log 8 2) 3) #t) ; optimization in C (-O2) changes this
      (num-test (log 8 2) 3)
      (num-test (log 1/8 2) -3)
      (test (eqv? (expt 701 2) 491401) #t)
      (test (eqv? (log 491401 701) 2) #t)
      ))

(let ((err 0.0))
  (do ((i 0 (+ i 1))
       (x -10.0 (+ x .1)))
      ((= i 200))
    (let ((y (magnitude (- x (* (sqrt x) (sqrt x))))))
      (if (> y err) (set! err y))))
  (if (> err 1e-14)
      (begin (display "(sqr (sqrt ...)) error: ") (display err) (newline))))

(let ((err 0.0))
  (do ((i 0 (+ i 1))
       (x -10.0+i (+ x .1+i)))
      ((= i 200))
    (let ((y (magnitude (- x (* (sqrt x) (sqrt x))))))
      (if (> y err) (set! err y))))
  (if (> err 1e-12)
      (begin (display "(sqr (sqrt [complex] ...)) error: ") (display err) (newline))))



;; -------- exp
(num-test (exp 0) 1.0)
(num-test (exp 1) 2.71828182845905)
(num-test (exp -1) 0.36787944117144)
(num-test (exp 2) 7.38905609893065)
(num-test (exp -2) 0.13533528323661)
(num-test (exp 3) 20.08553692318767)
(num-test (exp -3) 0.04978706836786)
(num-test (exp 10) 22026.46579480671789)
(num-test (exp -10) 0.00004539992976)
(num-test (exp 0/1) 1.0)
(num-test (exp 0/2) 1.0)
(num-test (exp 0/3) 1.0)
(num-test (exp 0/10) 1.0)
(num-test (exp 0/1234) 1.0)
(num-test (exp 0/1234000000) 1.0)
(num-test (exp 0/500029) 1.0)
(num-test (exp 0/362880) 1.0)
(num-test (exp 1/1) 2.71828182845905)
(num-test (exp -1/1) 0.36787944117144)
(num-test (exp 1/2) 1.64872127070013)
(num-test (exp -1/2) 0.60653065971263)
(num-test (exp 1/3) 1.39561242508609)
(num-test (exp -1/3) 0.71653131057379)
(num-test (exp 1/10) 1.10517091807565)
(num-test (exp -1/10) 0.90483741803596)
(num-test (exp 1/1234) 1.00081070121220)
(num-test (exp -1/1234) 0.99918995549186)
(num-test (exp 1/1234000000) 1.00000000081037)
(num-test (exp -1/1234000000) 0.99999999918963)
(num-test (exp 1/500029) 1.00000199988601)
(num-test (exp -1/500029) 0.99999800011799)
(num-test (exp 1/362880) 1.00000275573572)
(num-test (exp -1/362880) 0.99999724427187)
(num-test (exp 2/1) 7.38905609893065)
(num-test (exp -2/1) 0.13533528323661)
(num-test (exp 2/2) 2.71828182845905)
(num-test (exp -2/2) 0.36787944117144)
(num-test (exp 2/3) 1.94773404105468)
(num-test (exp -2/3) 0.51341711903259)
(num-test (exp 2/10) 1.22140275816017)
(num-test (exp -2/10) 0.81873075307798)
(num-test (exp 2/1234) 1.00162205966086)
(num-test (exp -2/1234) 0.99838056715583)
(num-test (exp 2/1234000000) 1.00000000162075)
(num-test (exp -2/1234000000) 0.99999999837925)
(num-test (exp 2/500029) 1.00000399977601)
(num-test (exp -2/500029) 0.99999600023999)
(num-test (exp 2/362880) 1.00000551147903)
(num-test (exp -2/362880) 0.99999448855134)
(num-test (exp 3/1) 20.08553692318767)
(num-test (exp -3/1) 0.04978706836786)
(num-test (exp 3/2) 4.48168907033806)
(num-test (exp -3/2) 0.22313016014843)
(num-test (exp 3/3) 2.71828182845905)
(num-test (exp -3/3) 0.36787944117144)
(num-test (exp 3/10) 1.34985880757600)
(num-test (exp -3/10) 0.74081822068172)
(num-test (exp 3/1234) 1.00243407587880)
(num-test (exp -3/1234) 0.99757183446037)
(num-test (exp 3/1234000000) 1.00000000243112)
(num-test (exp -3/1234000000) 0.99999999756888)
(num-test (exp 3/500029) 1.00000599967002)
(num-test (exp -3/500029) 0.99999400036598)
(num-test (exp 3/362880) 1.00000826722994)
(num-test (exp -3/362880) 0.99999173283841)
(num-test (exp 10/1) 22026.46579480671789)
(num-test (exp -10/1) 0.00004539992976)
(num-test (exp 10/2) 148.41315910257660)
(num-test (exp -10/2) 0.00673794699909)
(num-test (exp 10/3) 28.03162489452614)
(num-test (exp -10/3) 0.03567399334725)
(num-test (exp 10/10) 2.71828182845905)
(num-test (exp -10/10) 0.36787944117144)
(num-test (exp 10/1234) 1.00813665179201)
(num-test (exp -10/1234) 0.99192901897025)
(num-test (exp 10/1234000000) 1.00000000810373)
(num-test (exp -10/1234000000) 0.99999999189627)
(num-test (exp 10/500029) 1.00001999904005)
(num-test (exp -10/500029) 0.99998000135991)
(num-test (exp 10/362880) 1.00002755769893)
(num-test (exp -10/362880) 0.99997244306048)
(num-test (exp 1234/1234) 2.71828182845905)
(num-test (exp -1234/1234) 0.36787944117144)
(num-test (exp 1234/1234000000) 1.00000100000050)
(num-test (exp -1234/1234000000) 0.99999900000050)
(num-test (exp 1234/500029) 1.00247090452960)
(num-test (exp -1234/500029) 0.99753518579099)
(num-test (exp 1234/362880) 1.00340636170081)
(num-test (exp -1234/362880) 0.99660520220837)
(num-test (exp 1234000000/1234000000) 2.71828182845905)
(num-test (exp -1234000000/1234000000) 0.36787944117144)
(num-test (exp 500029/1234000000) 1.00040529199516)
(num-test (exp -500029/1234000000) 0.99959487219989)
(num-test (exp 500029/500029) 2.71828182845905)
(num-test (exp -500029/500029) 0.36787944117144)
(num-test (exp 500029/362880) 3.96674507247837)
(num-test (exp -500029/362880) 0.25209585736630)
(num-test (exp 362880/1234000000) 1.00029411131357)
(num-test (exp -362880/1234000000) 0.99970597516246)
(num-test (exp 362880/500029) 2.06621391988774)
(num-test (exp -362880/500029) 0.48397699307646)
(num-test (exp 362880/362880) 2.71828182845905)
(num-test (exp -362880/362880) 0.36787944117144)
(num-test (exp 0.0) 1.0)
(num-test (exp 0.00000001) 1.00000001)
(num-test (exp -0.00000001) 0.99999999000000)
(num-test (exp 1.0) 2.71828182845905)
(num-test (exp -1.0) 0.36787944117144)
(num-test (exp our-pi) 23.14069263277927)
(num-test (exp -3.14159265358979) 0.04321391826377)
(num-test (exp 2.71828182845905) 15.15426224147926)
(num-test (exp -2.71828182845905) 0.06598803584531)
(num-test (exp 0.0+0.0i) 1.0)
(num-test (exp -0.0+0.0i) 1.0)
(num-test (exp 0.0-0.0i) 1.0)
(num-test (exp -0.0-0.0i) 1.0)
(num-test (exp 0.0+0.00000001i) 1.0+0.00000001i)
(num-test (exp -0.0+0.00000001i) 1.0+0.00000001i)
(num-test (exp 0.0-0.00000001i) 1.0-0.00000001i)
(num-test (exp -0.0-0.00000001i) 1.0-0.00000001i)
(num-test (exp 0.0+1.0i) 0.54030230586814+0.84147098480790i)
(num-test (exp -0.0+1.0i) 0.54030230586814+0.84147098480790i)
(num-test (exp 0.0-1.0i) 0.54030230586814-0.84147098480790i)
(num-test (exp -0.0-1.0i) 0.54030230586814-0.84147098480790i)
(num-test (exp 0.0+3.14159265358979i) -1.0+0.0i)
(num-test (exp -0.0+3.14159265358979i) -1.0+0.0i)
(num-test (exp 0.0-3.14159265358979i) -1.0-0.0i)
(num-test (exp -0.0-3.14159265358979i) -1.0-0.0i)
(num-test (exp 0.0+2.71828182845905i) -0.91173391478697+0.41078129050291i)
(num-test (exp -0.0+2.71828182845905i) -0.91173391478697+0.41078129050291i)
(num-test (exp 0.0-2.71828182845905i) -0.91173391478697-0.41078129050291i)
(num-test (exp -0.0-2.71828182845905i) -0.91173391478697-0.41078129050291i)
(num-test (exp 0.00000001+0.0i) 1.00000001)
(num-test (exp -0.00000001+0.0i) 0.99999999000000)
(num-test (exp 0.00000001-0.0i) 1.00000001)
(num-test (exp -0.00000001-0.0i) 0.99999999000000)
(num-test (exp 0.00000001+0.00000001i) 1.00000001+0.00000001i)
(num-test (exp -0.00000001+0.00000001i) 0.99999999000000+0.00000001i)
(num-test (exp 0.00000001-0.00000001i) 1.00000001-0.00000001i)
(num-test (exp -0.00000001-0.00000001i) 0.99999999000000-0.00000001i)
(num-test (exp 0.00000001+1.0i) 0.54030231127116+0.84147099322261i)
(num-test (exp -0.00000001+1.0i) 0.54030230046512+0.84147097639319i)
(num-test (exp 0.00000001-1.0i) 0.54030231127116-0.84147099322261i)
(num-test (exp -0.00000001-1.0i) 0.54030230046512-0.84147097639319i)
(num-test (exp 0.00000001+3.14159265358979i) -1.00000001+0.0i)
(num-test (exp -0.00000001+3.14159265358979i) -0.99999999000000+0.0i)
(num-test (exp 0.00000001-3.14159265358979i) -1.00000001-0.0i)
(num-test (exp -0.00000001-3.14159265358979i) -0.99999999000000-0.0i)
(num-test (exp 0.00000001+2.71828182845905i) -0.91173392390430+0.41078129461072i)
(num-test (exp -0.00000001+2.71828182845905i) -0.91173390566963+0.41078128639510i)
(num-test (exp 0.00000001-2.71828182845905i) -0.91173392390430-0.41078129461072i)
(num-test (exp -0.00000001-2.71828182845905i) -0.91173390566963-0.41078128639510i)
(num-test (exp 1.0+0.0i) 2.71828182845905)
(num-test (exp -1.0+0.0i) 0.36787944117144)
(num-test (exp 1.0-0.0i) 2.71828182845905)
(num-test (exp -1.0-0.0i) 0.36787944117144)
(num-test (exp 1.0+0.00000001i) 2.71828182845905+0.00000002718282i)
(num-test (exp -1.0+0.00000001i) 0.36787944117144+0.00000000367879i)
(num-test (exp 1.0-0.00000001i) 2.71828182845905-0.00000002718282i)
(num-test (exp -1.0-0.00000001i) 0.36787944117144-0.00000000367879i)
(num-test (exp 1.0+1.0i) 1.46869393991589+2.28735528717884i)
(num-test (exp -1.0+1.0i) 0.19876611034641+0.30955987565311i)
(num-test (exp 1.0-1.0i) 1.46869393991589-2.28735528717884i)
(num-test (exp -1.0-1.0i) 0.19876611034641-0.30955987565311i)
(num-test (exp 1.0+3.14159265358979i) -2.71828182845905+0.0i)
(num-test (exp -1.0+3.14159265358979i) -0.36787944117144+0.0i)
(num-test (exp 1.0-3.14159265358979i) -2.71828182845905-0.0i)
(num-test (exp -1.0-3.14159265358979i) -0.36787944117144-0.0i)
(num-test (exp 1.0+2.71828182845905i) -2.47834973295523+1.11661931744501i)
(num-test (exp -1.0+2.71828182845905i) -0.33540816306888+0.15111799159389i)
(num-test (exp 1.0-2.71828182845905i) -2.47834973295523-1.11661931744501i)
(num-test (exp -1.0-2.71828182845905i) -0.33540816306888-0.15111799159389i)
(num-test (exp 3.14159265358979+0.0i) 23.14069263277927)
(num-test (exp -3.14159265358979+0.0i) 0.04321391826377)
(num-test (exp 3.14159265358979-0.0i) 23.14069263277927)
(num-test (exp -3.14159265358979-0.0i) 0.04321391826377)
(num-test (exp 3.14159265358979+0.00000001i) 23.14069263277926+0.00000023140693i)
(num-test (exp -3.14159265358979+0.00000001i) 0.04321391826377+0.00000000043214i)
(num-test (exp 3.14159265358979-0.00000001i) 23.14069263277926-0.00000023140693i)
(num-test (exp -3.14159265358979-0.00000001i) 0.04321391826377-0.00000000043214i)
(num-test (exp 3.14159265358979+1.0i) 12.50296958887651+19.47222141884161i)
(num-test (exp -3.14159265358979+1.0i) 0.02334857968351+0.03636325835882i)
(num-test (exp 3.14159265358979-1.0i) 12.50296958887651-19.47222141884161i)
(num-test (exp -3.14159265358979-1.0i) 0.02334857968351-0.03636325835882i)
(num-test (exp 3.14159265358979+3.14159265358979i) -23.14069263277927+0.0i)
(num-test (exp -3.14159265358979+3.14159265358979i) -0.04321391826377+0.0i)
(num-test (exp 3.14159265358979-3.14159265358979i) -23.14069263277927-0.0i)
(num-test (exp -3.14159265358979-3.14159265358979i) -0.04321391826377-0.0i)
(num-test (exp 3.14159265358979+2.71828182845905i) -21.09815428496572+9.50576358282422i)
(num-test (exp -3.14159265358979+2.71828182845905i) -0.03939959487191+0.01775146911208i)
(num-test (exp 3.14159265358979-2.71828182845905i) -21.09815428496572-9.50576358282422i)
(num-test (exp -3.14159265358979-2.71828182845905i) -0.03939959487191-0.01775146911208i)
(num-test (exp 2.71828182845905+0.0i) 15.15426224147926)
(num-test (exp -2.71828182845905+0.0i) 0.06598803584531)
(num-test (exp 2.71828182845905-0.0i) 15.15426224147926)
(num-test (exp -2.71828182845905-0.0i) 0.06598803584531)
(num-test (exp 2.71828182845905+0.00000001i) 15.15426224147926+0.00000015154262i)
(num-test (exp -2.71828182845905+0.00000001i) 0.06598803584531+0.00000000065988i)
(num-test (exp 2.71828182845905-0.00000001i) 15.15426224147926-0.00000015154262i)
(num-test (exp -2.71828182845905-0.00000001i) 0.06598803584531-0.00000000065988i)
(num-test (exp 2.71828182845905+1.0i) 8.18788283280173+12.75187197237468i)
(num-test (exp -2.71828182845905+1.0i) 0.03565348792693+0.05552701750829i)
(num-test (exp 2.71828182845905-1.0i) 8.18788283280173-12.75187197237468i)
(num-test (exp -2.71828182845905-1.0i) 0.03565348792693-0.05552701750829i)
(num-test (exp 2.71828182845905+3.14159265358979i) -15.15426224147926+0.0i)
(num-test (exp -2.71828182845905+3.14159265358979i) -0.06598803584531+0.0i)
(num-test (exp 2.71828182845905-3.14159265358979i) -15.15426224147926-0.0i)
(num-test (exp -2.71828182845905-3.14159265358979i) -0.06598803584531-0.0i)
(num-test (exp 2.71828182845905+2.71828182845905i) -13.81665483913218+6.22508740017436i)
(num-test (exp -2.71828182845905+2.71828182845905i) -0.06016353025035+0.02710665052229i)
(num-test (exp 2.71828182845905-2.71828182845905i) -13.81665483913218-6.22508740017436i)
(num-test (exp -2.71828182845905-2.71828182845905i) -0.06016353025035-0.02710665052229i)
(num-test (exp -1234.0+0.0i) 0.0)
(num-test (exp -1234.0-0.0i) 0.0)
(num-test (exp -1234.0+0.00000001i) 0.0)
(num-test (exp -1234.0-0.00000001i) 0.0)
(num-test (exp -1234.0+3.14159265358979i) -0.0)
(num-test (exp -1234.0-3.14159265358979i) -0.0)
(num-test (exp -1234.0+2.71828182845905i) -0.0)
(num-test (exp -1234.0-2.71828182845905i) -0.0)
(num-test (exp -1234000000.0+0.0i) 0.0)
(num-test (exp -1234000000.0-0.0i) 0.0)
(num-test (exp -1234000000.0+0.00000001i) 0.0)
(num-test (exp -1234000000.0-0.00000001i) 0.0)
(num-test (exp -1234000000.0+3.14159265358979i) -0.0)
(num-test (exp -1234000000.0-3.14159265358979i) -0.0)
(num-test (exp -1234000000.0+2.71828182845905i) -0.0)
(num-test (exp -1234000000.0-2.71828182845905i) -0.0)
(num-test (exp 10.0) 22026.46579480672)
(num-test (exp 100.0) 2.688117141816135E+43)
(num-test (exp -10.0) 4.5399929762484853E-5)
(num-test (exp -100.0) 3.720075976020836E-44)
(num-test (exp -7.080000e+02) 3.307553e-308)
(num-test (exp 7.090000e+02) 8.218407461554972189241372386597816393254E307)
(num-test (exp 0.00000001+1234.0i) -0.7985506315730906+0.601927660781774i) 
(num-test (exp 0.00000001+1234000000.0i) .1589091324793142-0.987293222712823i) 
(num-test (exp 3.14159265358979+1234.0i) -18.47901453215463+13.92902284602872i) 
(num-test (exp 3.14159265358979+1234000000.0i) 3.677267354472762-22.8466487767572i) 
(num-test (exp 2.71828182845905+1234.0i) -12.1014455629425+9.121769530669065i) 
(num-test (exp 2.71828182845905+1234000000.0i) 2.408150642075881-14.96170025660763i)
(num-test (exp 0.0e+00-3.45266983001243932001e-04i) 9.9999994039535581673e-1-3.4526697614140534807e-4i)
(num-test (exp 0.0e+00+3.45266983001243932001e-04i) 9.9999994039535581673e-1+3.4526697614140534807e-4i)
(num-test (exp 0.0e+00+1.57045105981189525579e+00i) 3.4526697614152485627e-4+9.9999994039535581669e-1i)
(num-test (exp 0.0e+00-1.57045105981189525579e+00i) 3.4526697614152485627e-4-9.9999994039535581669e-1i)
(num-test (exp 0.0e+00+1.57114159377789786021e+00i) -3.4526697614140239160e-4+9.9999994039535581673e-1i)
(num-test (exp 0.0e+00-1.57114159377789786021e+00i) -3.4526697614140239160e-4-9.9999994039535581673e-1i)
(num-test (exp 0.0e+00+3.14124738660679181379e+00i) -9.9999994039535581667e-1+3.4526697614158608860e-4i)
(num-test (exp 0.0e+00-3.14124738660679181379e+00i) -9.9999994039535581667e-1-3.4526697614158608860e-4i)
(num-test (exp 0.0e+00+3.14193792057279441821e+00i) -9.9999994039535581675e-1-3.4526697614134115926e-4i)
(num-test (exp 0.0e+00-3.14193792057279441821e+00i) -9.9999994039535581675e-1+3.4526697614134115926e-4i)
(num-test (exp 0.0e+00+4.71204371340168837179e+00i) -3.4526697614164732094e-4-9.9999994039535581664e-1i)
(num-test (exp 0.0e+00-4.71204371340168837179e+00i) -3.4526697614164732094e-4+9.9999994039535581664e-1i)
(num-test (exp 0.0e+00+4.71273424736769097620e+00i) 3.4526697614127992692e-4-9.9999994039535581677e-1i)
(num-test (exp 0.0e+00-4.71273424736769097620e+00i) 3.4526697614127992692e-4+9.9999994039535581677e-1i)
(num-test (exp 0.0e+00+6.28284004019658492979e+00i) 9.9999994039535581662e-1-3.4526697614170855328e-4i)
(num-test (exp 0.0e+00-6.28284004019658492979e+00i) 9.9999994039535581662e-1+3.4526697614170855328e-4i)
(num-test (exp 0.0e+00+6.28353057416258753420e+00i) 9.9999994039535581679e-1+3.4526697614121869459e-4i)
(num-test (exp 0.0e+00-6.28353057416258753420e+00i) 9.9999994039535581679e-1-3.4526697614121869459e-4i)
(num-test (exp 0.0e+00+9.42443269378637893396e+00i) -9.9999994039535581689e-1+3.4526697614094283958e-4i)
(num-test (exp 0.0e+00-9.42443269378637893396e+00i) -9.9999994039535581689e-1-3.4526697614094283958e-4i)
(num-test (exp 0.0e+00+9.42512322775237976202e+00i) -9.9999994039535581714e-1-3.4526697614020805155e-4i)
(num-test (exp 0.0e+00-9.42512322775237976202e+00i) -9.9999994039535581714e-1+3.4526697614020805155e-4i)
(num-test (exp 1.19209289550781250e-07-3.45266983001243932001e-04i) 1.0000000596046453675e0-3.4526701730043873250e-4i)
(num-test (exp 1.19209289550781250e-07+3.45266983001243932001e-04i) 1.0000000596046453675e0+3.4526701730043873250e-4i)
(num-test (exp -1.19209289550781250e-07-3.45266983001243932001e-04i) 9.9999982118608047680e-1-3.4526693498237687017e-4i)
(num-test (exp -1.19209289550781250e-07+3.45266983001243932001e-04i) 9.9999982118608047680e-1+3.4526693498237687017e-4i)
(num-test (exp 1.19209289550781250e-07+1.57045105981189525579e+00i) 3.4526701730055824072e-4+1.0000000596046453675e0i)
(num-test (exp 1.19209289550781250e-07-1.57045105981189525579e+00i) 3.4526701730055824072e-4-1.0000000596046453675e0i)
(num-test (exp -1.19209289550781250e-07+1.57045105981189525579e+00i) 3.4526693498249637836e-4+9.9999982118608047676e-1i)
(num-test (exp -1.19209289550781250e-07-1.57045105981189525579e+00i) 3.4526693498249637836e-4-9.9999982118608047676e-1i)
(num-test (exp 1.19209289550781250e-07+1.57114159377789786021e+00i) -3.4526701730043577603e-4+1.0000000596046453675e0i)
(num-test (exp 1.19209289550781250e-07-1.57114159377789786021e+00i) -3.4526701730043577603e-4-1.0000000596046453675e0i)
(num-test (exp -1.19209289550781250e-07+1.57114159377789786021e+00i) -3.4526693498237391370e-4+9.9999982118608047680e-1i)
(num-test (exp -1.19209289550781250e-07-1.57114159377789786021e+00i) -3.4526693498237391370e-4-9.9999982118608047680e-1i)
(num-test (exp 1.19209289550781250e-07+3.14124738660679181379e+00i) -1.0000000596046453674e0+3.4526701730061947306e-4i)
(num-test (exp 1.19209289550781250e-07-3.14124738660679181379e+00i) -1.0000000596046453674e0-3.4526701730061947306e-4i)
(num-test (exp -1.19209289550781250e-07+3.14124738660679181379e+00i) -9.9999982118608047674e-1+3.4526693498255761069e-4i)
(num-test (exp -1.19209289550781250e-07-3.14124738660679181379e+00i) -9.9999982118608047674e-1-3.4526693498255761069e-4i)
(num-test (exp 1.19209289550781250e-07+3.14193792057279441821e+00i) -1.0000000596046453675e0-3.4526701730037454368e-4i)
(num-test (exp 1.19209289550781250e-07-3.14193792057279441821e+00i) -1.0000000596046453675e0+3.4526701730037454368e-4i)
(num-test (exp -1.19209289550781250e-07+3.14193792057279441821e+00i) -9.9999982118608047682e-1-3.4526693498231268137e-4i)
(num-test (exp -1.19209289550781250e-07-3.14193792057279441821e+00i) -9.9999982118608047682e-1+3.4526693498231268137e-4i)
(num-test (exp 1.19209289550781250e-07+4.71204371340168837179e+00i) -3.4526701730068070540e-4-1.0000000596046453674e0i)
(num-test (exp 1.19209289550781250e-07-4.71204371340168837179e+00i) -3.4526701730068070540e-4+1.0000000596046453674e0i)
(num-test (exp -1.19209289550781250e-07+4.71204371340168837179e+00i) -3.4526693498261884302e-4-9.9999982118608047672e-1i)
(num-test (exp -1.19209289550781250e-07-4.71204371340168837179e+00i) -3.4526693498261884302e-4+9.9999982118608047672e-1i)
(num-test (exp 1.19209289550781250e-07+4.71273424736769097620e+00i) 3.4526701730031331134e-4-1.0000000596046453676e0i)
(num-test (exp 1.19209289550781250e-07-4.71273424736769097620e+00i) 3.4526701730031331134e-4+1.0000000596046453676e0i)
(num-test (exp -1.19209289550781250e-07+4.71273424736769097620e+00i) 3.4526693498225144904e-4-9.9999982118608047684e-1i)
(num-test (exp -1.19209289550781250e-07-4.71273424736769097620e+00i) 3.4526693498225144904e-4+9.9999982118608047684e-1i)
(num-test (exp 1.19209289550781250e-07+6.28284004019658492979e+00i) 1.0000000596046453674e0-3.4526701730074193775e-4i)
(num-test (exp 1.19209289550781250e-07-6.28284004019658492979e+00i) 1.0000000596046453674e0+3.4526701730074193775e-4i)
(num-test (exp -1.19209289550781250e-07+6.28284004019658492979e+00i) 9.9999982118608047670e-1-3.4526693498268007535e-4i)
(num-test (exp -1.19209289550781250e-07-6.28284004019658492979e+00i) 9.9999982118608047670e-1+3.4526693498268007535e-4i)
(num-test (exp 1.19209289550781250e-07+6.28353057416258753420e+00i) 1.0000000596046453676e0+3.452670173002520790e-4i)
(num-test (exp 1.19209289550781250e-07-6.28353057416258753420e+00i) 1.0000000596046453676e0-3.452670173002520790e-4i)
(num-test (exp -1.19209289550781250e-07+6.28353057416258753420e+00i) 9.9999982118608047687e-1+3.4526693498219021671e-4i)
(num-test (exp -1.19209289550781250e-07-6.28353057416258753420e+00i) 9.9999982118608047687e-1-3.4526693498219021671e-4i)
(num-test (exp 1.19209289550781250e-07+9.42443269378637893396e+00i) -1.0000000596046453677e0+3.4526701729997622396e-4i)
(num-test (exp 1.19209289550781250e-07-9.42443269378637893396e+00i) -1.0000000596046453677e0-3.4526701729997622396e-4i)
(num-test (exp -1.19209289550781250e-07+9.42443269378637893396e+00i) -9.9999982118608047696e-1+3.4526693498191436174e-4i)
(num-test (exp -1.19209289550781250e-07-9.42443269378637893396e+00i) -9.9999982118608047696e-1-3.4526693498191436174e-4i)
(num-test (exp 1.19209289550781250e-07+9.42512322775237976202e+00i) -1.0000000596046453679e0-3.4526701729924143584e-4i)
(num-test (exp 1.19209289550781250e-07-9.42512322775237976202e+00i) -1.0000000596046453679e0+3.4526701729924143584e-4i)
(num-test (exp -1.19209289550781250e-07+9.42512322775237976202e+00i) -9.9999982118608047721e-1-3.4526693498117957380e-4i)
(num-test (exp -1.19209289550781250e-07-9.42512322775237976202e+00i) -9.9999982118608047721e-1+3.4526693498117957380e-4i)
(num-test (exp 5.0e-01-3.45266983001243932001e-04i) 1.6487211724286834494e0-5.6924900763464865323e-4i)
(num-test (exp 5.0e-01+3.45266983001243932001e-04i) 1.6487211724286834494e0+5.6924900763464865323e-4i)
(num-test (exp -5.0e-01-3.45266983001243932001e-04i) 6.0653062356058926519e-1-2.0941500681603265022e-4i)
(num-test (exp -5.0e-01+3.45266983001243932001e-04i) 6.0653062356058926519e-1+2.0941500681603265022e-4i)
(num-test (exp 5.0e-01+1.57045105981189525579e+00i) 5.6924900763484568894e-4+1.6487211724286834493e0i)
(num-test (exp 5.0e-01-1.57045105981189525579e+00i) 5.6924900763484568894e-4-1.6487211724286834493e0i)
(num-test (exp -5.0e-01+1.57045105981189525579e+00i) 2.0941500681610513560e-4+6.0653062356058926516e-1i)
(num-test (exp -5.0e-01-1.57045105981189525579e+00i) 2.0941500681610513560e-4-6.0653062356058926516e-1i)
(num-test (exp 5.0e-01+1.57114159377789786021e+00i) -5.6924900763464377883e-4+1.6487211724286834494e0i)
(num-test (exp 5.0e-01-1.57114159377789786021e+00i) -5.6924900763464377883e-4-1.6487211724286834494e0i)
(num-test (exp -5.0e-01+1.57114159377789786021e+00i) -2.0941500681603085702e-4+6.0653062356058926519e-1i)
(num-test (exp -5.0e-01-1.57114159377789786021e+00i) -2.0941500681603085702e-4-6.0653062356058926519e-1i)
(num-test (exp 5.0e-01+3.14124738660679181379e+00i) -1.6487211724286834493e0+5.6924900763494664399e-4i)
(num-test (exp 5.0e-01-3.14124738660679181379e+00i) -1.6487211724286834493e0-5.6924900763494664399e-4i)
(num-test (exp -5.0e-01+3.14124738660679181379e+00i) -6.0653062356058926515e-1+2.0941500681614227489e-4i)
(num-test (exp -5.0e-01-3.14124738660679181379e+00i) -6.0653062356058926515e-1-2.0941500681614227489e-4i)
(num-test (exp 5.0e-01+3.14193792057279441821e+00i) -1.6487211724286834494e0-5.6924900763454282377e-4i)
(num-test (exp 5.0e-01-3.14193792057279441821e+00i) -1.6487211724286834494e0+5.6924900763454282377e-4i)
(num-test (exp -5.0e-01+3.14193792057279441821e+00i) -6.0653062356058926520e-1-2.0941500681599371773e-4i)
(num-test (exp -5.0e-01-3.14193792057279441821e+00i) -6.0653062356058926520e-1+2.0941500681599371773e-4i)
(num-test (exp 5.0e-01+4.71204371340168837179e+00i) -5.6924900763504759905e-4-1.6487211724286834492e0i)
(num-test (exp 5.0e-01-4.71204371340168837179e+00i) -5.6924900763504759905e-4+1.6487211724286834492e0i)
(num-test (exp -5.0e-01+4.71204371340168837179e+00i) -2.0941500681617941418e-4-6.0653062356058926514e-1i)
(num-test (exp -5.0e-01-4.71204371340168837179e+00i) -2.0941500681617941418e-4+6.0653062356058926514e-1i)
(num-test (exp 5.0e-01+4.71273424736769097620e+00i) 5.6924900763444186872e-4-1.6487211724286834494e0i)
(num-test (exp 5.0e-01-4.71273424736769097620e+00i) 5.6924900763444186872e-4+1.6487211724286834494e0i)
(num-test (exp -5.0e-01+4.71273424736769097620e+00i) 2.0941500681595657844e-4-6.0653062356058926521e-1i)
(num-test (exp -5.0e-01-4.71273424736769097620e+00i) 2.0941500681595657844e-4+6.0653062356058926521e-1i)
(num-test (exp 5.0e-01+6.28284004019658492979e+00i) 1.6487211724286834492e0-5.6924900763514855410e-4i)
(num-test (exp 5.0e-01-6.28284004019658492979e+00i) 1.6487211724286834492e0+5.6924900763514855410e-4i)
(num-test (exp -5.0e-01+6.28284004019658492979e+00i) 6.0653062356058926512e-1-2.0941500681621655347e-4i)
(num-test (exp -5.0e-01-6.28284004019658492979e+00i) 6.0653062356058926512e-1+2.0941500681621655347e-4i)
(num-test (exp 5.0e-01+6.28353057416258753420e+00i) 1.6487211724286834495e0+5.6924900763434091366e-4i)
(num-test (exp 5.0e-01-6.28353057416258753420e+00i) 1.6487211724286834495e0-5.6924900763434091366e-4i)
(num-test (exp -5.0e-01+6.28353057416258753420e+00i) 6.0653062356058926523e-1+2.0941500681591943916e-4i)
(num-test (exp -5.0e-01-6.28353057416258753420e+00i) 6.0653062356058926523e-1-2.0941500681591943916e-4i)
(num-test (exp 5.0e-01+9.42443269378637893396e+00i) -1.6487211724286834496e0+5.6924900763388610565e-4i)
(num-test (exp 5.0e-01-9.42443269378637893396e+00i) -1.6487211724286834496e0-5.6924900763388610565e-4i)
(num-test (exp -5.0e-01+9.42443269378637893396e+00i) -6.0653062356058926528e-1+2.0941500681575212464e-4i)
(num-test (exp -5.0e-01-9.42443269378637893396e+00i) -6.0653062356058926528e-1-2.0941500681575212464e-4i)
(num-test (exp 5.0e-01+9.42512322775237976202e+00i) -1.6487211724286834501e0-5.6924900763267464498e-4i)
(num-test (exp 5.0e-01-9.42512322775237976202e+00i) -1.6487211724286834501e0+5.6924900763267464498e-4i)
(num-test (exp -5.0e-01+9.42512322775237976202e+00i) -6.0653062356058926544e-1-2.0941500681530645317e-4i)
(num-test (exp -5.0e-01-9.42512322775237976202e+00i) -6.0653062356058926544e-1+2.0941500681530645317e-4i)
(num-test (exp 1.0e+00-3.45266983001243932001e-04i) 2.7182816664368240602e0-9.3853294721218487636e-4i)
(num-test (exp 1.0e+00+3.45266983001243932001e-04i) 2.7182816664368240602e0+9.3853294721218487636e-4i)
(num-test (exp -1.0e+00-3.45266983001243932001e-04i) 3.6787941924411912823e-1-1.2701662223785390836e-4i)
(num-test (exp -1.0e+00+3.45266983001243932001e-04i) 3.6787941924411912823e-1+1.2701662223785390836e-4i)
(num-test (exp 1.0e+00+1.57045105981189525579e+00i) 9.3853294721250973333e-4+2.7182816664368240601e0i)
(num-test (exp 1.0e+00-1.57045105981189525579e+00i) 9.3853294721250973333e-4-2.7182816664368240601e0i)
(num-test (exp -1.0e+00+1.57045105981189525579e+00i) 1.2701662223789787297e-4+3.6787941924411912822e-1i)
(num-test (exp -1.0e+00-1.57045105981189525579e+00i) 1.2701662223789787297e-4-3.6787941924411912822e-1i)
(num-test (exp 1.0e+00+1.57114159377789786021e+00i) -9.3853294721217683983e-4+2.7182816664368240602e0i)
(num-test (exp 1.0e+00-1.57114159377789786021e+00i) -9.3853294721217683983e-4-2.7182816664368240602e0i)
(num-test (exp -1.0e+00+1.57114159377789786021e+00i) -1.2701662223785282074e-4+3.6787941924411912823e-1i)
(num-test (exp -1.0e+00-1.57114159377789786021e+00i) -1.2701662223785282074e-4-3.6787941924411912823e-1i)
(num-test (exp 1.0e+00+3.14124738660679181379e+00i) -2.718281666436824060e0+9.3853294721267618008e-4i)
(num-test (exp 1.0e+00-3.14124738660679181379e+00i) -2.718281666436824060e0-9.3853294721267618008e-4i)
(num-test (exp -1.0e+00+3.14124738660679181379e+00i) -3.6787941924411912821e-1+1.2701662223792039909e-4i)
(num-test (exp -1.0e+00-3.14124738660679181379e+00i) -3.6787941924411912821e-1-1.2701662223792039909e-4i)
(num-test (exp 1.0e+00+3.14193792057279441821e+00i) -2.7182816664368240603e0-9.3853294721201039309e-4i)
(num-test (exp 1.0e+00-3.14193792057279441821e+00i) -2.7182816664368240603e0+9.3853294721201039309e-4i)
(num-test (exp -1.0e+00+3.14193792057279441821e+00i) -3.6787941924411912824e-1-1.2701662223783029462e-4i)
(num-test (exp -1.0e+00-3.14193792057279441821e+00i) -3.6787941924411912824e-1+1.2701662223783029462e-4i)
(num-test (exp 1.0e+00+4.71204371340168837179e+00i) -9.3853294721284262682e-4-2.718281666436824060e0i)
(num-test (exp 1.0e+00-4.71204371340168837179e+00i) -9.3853294721284262682e-4+2.718281666436824060e0i)
(num-test (exp -1.0e+00+4.71204371340168837179e+00i) -1.2701662223794292521e-4-3.6787941924411912820e-1i)
(num-test (exp -1.0e+00-4.71204371340168837179e+00i) -1.2701662223794292521e-4+3.6787941924411912820e-1i)
(num-test (exp 1.0e+00+4.71273424736769097620e+00i) 9.3853294721184394634e-4-2.7182816664368240603e0i)
(num-test (exp 1.0e+00-4.71273424736769097620e+00i) 9.3853294721184394634e-4+2.7182816664368240603e0i)
(num-test (exp -1.0e+00+4.71273424736769097620e+00i) 1.2701662223780776850e-4-3.6787941924411912825e-1i)
(num-test (exp -1.0e+00-4.71273424736769097620e+00i) 1.2701662223780776850e-4+3.6787941924411912825e-1i)
(num-test (exp 1.0e+00+6.28284004019658492979e+00i) 2.7182816664368240599e0-9.3853294721300907357e-4i)
(num-test (exp 1.0e+00-6.28284004019658492979e+00i) 2.7182816664368240599e0+9.3853294721300907357e-4i)
(num-test (exp -1.0e+00+6.28284004019658492979e+00i) 3.6787941924411912819e-1-1.2701662223796545132e-4i)
(num-test (exp -1.0e+00-6.28284004019658492979e+00i) 3.6787941924411912819e-1+1.2701662223796545132e-4i)
(num-test (exp 1.0e+00+6.28353057416258753420e+00i) 2.7182816664368240604e0+9.3853294721167749959e-4i)
(num-test (exp 1.0e+00-6.28353057416258753420e+00i) 2.7182816664368240604e0-9.3853294721167749959e-4i)
(num-test (exp -1.0e+00+6.28353057416258753420e+00i) 3.6787941924411912825e-1+1.2701662223778524238e-4i)
(num-test (exp -1.0e+00-6.28353057416258753420e+00i) 3.6787941924411912825e-1-1.2701662223778524238e-4i)
(num-test (exp 1.0e+00+9.42443269378637893396e+00i) -2.7182816664368240606e0+9.3853294721092764795e-4i)
(num-test (exp 1.0e+00-9.42443269378637893396e+00i) -2.7182816664368240606e0-9.3853294721092764795e-4i)
(num-test (exp -1.0e+00+9.42443269378637893396e+00i) -3.6787941924411912829e-1+1.270166222376837610e-4i)
(num-test (exp -1.0e+00-9.42443269378637893396e+00i) -3.6787941924411912829e-1-1.270166222376837610e-4i)
(num-test (exp 1.0e+00+9.42512322775237976202e+00i) -2.7182816664368240613e0-9.3853294720893028698e-4i)
(num-test (exp 1.0e+00-9.42512322775237976202e+00i) -2.7182816664368240613e0+9.3853294720893028698e-4i)
(num-test (exp -1.0e+00+9.42512322775237976202e+00i) -3.6787941924411912838e-1-1.2701662223741344759e-4i)
(num-test (exp -1.0e+00-9.42512322775237976202e+00i) -3.6787941924411912838e-1+1.2701662223741344759e-4i)
(num-test (exp 2.0e+00-3.45266983001243932001e-04i) 7.3890556585085906002e0-2.5511970558169944872e-3i)
(num-test (exp 2.0e+00+3.45266983001243932001e-04i) 7.3890556585085906002e0+2.5511970558169944872e-3i)
(num-test (exp -2.0e+00-3.45266983001243932001e-04i) 1.3533527517000128913e-1-4.6726804008345889445e-5i)
(num-test (exp -2.0e+00+3.45266983001243932001e-04i) 1.3533527517000128913e-1+4.6726804008345889445e-5i)
(num-test (exp 2.0e+00+1.57045105981189525579e+00i) 2.551197055817877540e-3+7.3890556585085905999e0i)
(num-test (exp 2.0e+00-1.57045105981189525579e+00i) 2.551197055817877540e-3-7.3890556585085905999e0i)
(num-test (exp -2.0e+00+1.57045105981189525579e+00i) 4.6726804008362063122e-5+1.3533527517000128913e-1i)
(num-test (exp -2.0e+00-1.57045105981189525579e+00i) 4.6726804008362063122e-5-1.3533527517000128913e-1i)
(num-test (exp 2.0e+00+1.57114159377789786021e+00i) -2.5511970558169726417e-3+7.3890556585085906002e0i)
(num-test (exp 2.0e+00-1.57114159377789786021e+00i) -2.5511970558169726417e-3-7.3890556585085906002e0i)
(num-test (exp -2.0e+00+1.57114159377789786021e+00i) -4.6726804008345489330e-5+1.3533527517000128913e-1i)
(num-test (exp -2.0e+00-1.57114159377789786021e+00i) -4.6726804008345489330e-5-1.3533527517000128913e-1i)
(num-test (exp 2.0e+00+3.14124738660679181379e+00i) -7.3890556585085905998e0+2.5511970558183299892e-3i)
(num-test (exp 2.0e+00-3.14124738660679181379e+00i) -7.3890556585085905998e0-2.5511970558183299892e-3i)
(num-test (exp -2.0e+00+3.14124738660679181379e+00i) -1.3533527517000128912e-1+4.6726804008370350017e-5i)
(num-test (exp -2.0e+00-3.14124738660679181379e+00i) -1.3533527517000128912e-1-4.6726804008370350017e-5i)
(num-test (exp 2.0e+00+3.14193792057279441821e+00i) -7.3890556585085906004e0-2.5511970558165201925e-3i)
(num-test (exp 2.0e+00-3.14193792057279441821e+00i) -7.3890556585085906004e0+2.5511970558165201925e-3i)
(num-test (exp -2.0e+00+3.14193792057279441821e+00i) -1.3533527517000128914e-1-4.6726804008337202435e-5i)
(num-test (exp -2.0e+00-3.14193792057279441821e+00i) -1.3533527517000128914e-1+4.6726804008337202435e-5i)
(num-test (exp 2.0e+00+4.71204371340168837179e+00i) -2.5511970558187824384e-3-7.3890556585085905996e0i)
(num-test (exp 2.0e+00-4.71204371340168837179e+00i) -2.5511970558187824384e-3+7.3890556585085905996e0i)
(num-test (exp -2.0e+00+4.71204371340168837179e+00i) -4.6726804008378636913e-5-1.3533527517000128912e-1i)
(num-test (exp -2.0e+00-4.71204371340168837179e+00i) -4.6726804008378636913e-5+1.3533527517000128912e-1i)
(num-test (exp 2.0e+00+4.71273424736769097620e+00i) 2.5511970558160677434e-3-7.3890556585085906006e0i)
(num-test (exp 2.0e+00-4.71273424736769097620e+00i) 2.5511970558160677434e-3+7.3890556585085906006e0i)
(num-test (exp -2.0e+00+4.71273424736769097620e+00i) 4.6726804008328915539e-5-1.3533527517000128914e-1i)
(num-test (exp -2.0e+00-4.71273424736769097620e+00i) 4.6726804008328915539e-5+1.3533527517000128914e-1i)
(num-test (exp 2.0e+00+6.28284004019658492979e+00i) 7.3890556585085905995e0-2.5511970558192348875e-3i)
(num-test (exp 2.0e+00-6.28284004019658492979e+00i) 7.3890556585085905995e0+2.5511970558192348875e-3i)
(num-test (exp -2.0e+00+6.28284004019658492979e+00i) 1.3533527517000128912e-1-4.6726804008386923808e-5i)
(num-test (exp -2.0e+00-6.28284004019658492979e+00i) 1.3533527517000128912e-1+4.6726804008386923808e-5i)
(num-test (exp 2.0e+00+6.28353057416258753420e+00i) 7.3890556585085906007e0+2.5511970558156152942e-3i)
(num-test (exp 2.0e+00-6.28353057416258753420e+00i) 7.3890556585085906007e0-2.5511970558156152942e-3i)
(num-test (exp -2.0e+00+6.28353057416258753420e+00i) 1.3533527517000128914e-1+4.6726804008320628644e-5i)
(num-test (exp -2.0e+00-6.28353057416258753420e+00i) 1.3533527517000128914e-1-4.6726804008320628644e-5i)
(num-test (exp 2.0e+00+9.42443269378637893396e+00i) -7.3890556585085906014e0+2.5511970558135769861e-3i)
(num-test (exp 2.0e+00-9.42443269378637893396e+00i) -7.3890556585085906014e0-2.5511970558135769861e-3i)
(num-test (exp -2.0e+00+9.42443269378637893396e+00i) -1.3533527517000128916e-1+4.6726804008283295729e-5i)
(num-test (exp -2.0e+00-9.42443269378637893396e+00i) -1.3533527517000128916e-1-4.6726804008283295729e-5i)
(num-test (exp 2.0e+00+9.42512322775237976202e+00i) -7.3890556585085906033e0-2.5511970558081475961e-3i)
(num-test (exp 2.0e+00-9.42512322775237976202e+00i) -7.3890556585085906033e0+2.5511970558081475961e-3i)
(num-test (exp -2.0e+00+9.42512322775237976202e+00i) -1.3533527517000128919e-1-4.6726804008183852982e-5i)
(num-test (exp -2.0e+00-9.42512322775237976202e+00i) -1.3533527517000128919e-1+4.6726804008183852982e-5i)
(num-test (exp -1000) 0.0)
(num-test (exp -1000000) 0.0)
(num-test (exp (make-rectangular 0.0 (* 0.5 our-pi))) 0+i)
(num-test (exp (make-rectangular 0.0 our-pi)) -1)
(num-test (exp 100.0) 2.688117141816135e43)
(num-test (exp 500.0) 1.40359221785284E+217)
(num-test (exp -100.0) 3.720075976020836E-44)
(num-test (exp -500.0) 7.12457640674129E-218)


;; -------- log
(num-test (log (/ (+ 1 (sqrt 5)) 2)) (acosh (/ (sqrt 5) 2)))
(num-test (log 1) 0.0)
(num-test (log -1) 0.0+3.14159265358979i)
(num-test (log 2) 0.69314718055995)
(num-test (log -2) 0.69314718055995+3.14159265358979i)
(num-test (log 3) 1.09861228866811)
(num-test (log -3) 1.09861228866811+3.14159265358979i)
(num-test (log 10) 2.30258509299405)
(num-test (log -10) 2.30258509299405+3.14159265358979i)
(num-test (log 1234) 7.11801620446533)
(num-test (log -1234) 7.11801620446533+3.14159265358979i)
(num-test (log 1234000000) 20.93352676242961)
(num-test (log -1234000000) 20.93352676242961+3.14159265358979i)
(num-test (log 500029) 13.12242137572239)
(num-test (log -500029) 13.12242137572239+3.14159265358979i)
(num-test (log 362880) 12.80182748008147)
(num-test (log -362880) 12.80182748008147+3.14159265358979i)
(num-test (log 1/1) 0.0)
(num-test (log -1/1) 0.0+3.14159265358979i)
(num-test (log 1/2) -0.69314718055995)
(num-test (log -1/2) -0.69314718055995+3.14159265358979i)
(num-test (log 1/3) -1.09861228866811)
(num-test (log -1/3) -1.09861228866811+3.14159265358979i)
(num-test (log 1/10) -2.30258509299405)
(num-test (log -1/10) -2.30258509299405+3.14159265358979i)
(num-test (log 1/1234) -7.11801620446533)
(num-test (log -1/1234) -7.11801620446533+3.14159265358979i)
(num-test (log 1/1234000000) -20.93352676242961)
(num-test (log -1/1234000000) -20.93352676242961+3.14159265358979i)
(num-test (log 1/500029) -13.12242137572239)
(num-test (log -1/500029) -13.12242137572239+3.14159265358979i)
(num-test (log 1/362880) -12.80182748008147)
(num-test (log -1/362880) -12.80182748008147+3.14159265358979i)
(num-test (log 2/1) 0.69314718055995)
(num-test (log -2/1) 0.69314718055995+3.14159265358979i)
(num-test (log 2/2) 0.0)
(num-test (log -2/2) 0.0+3.14159265358979i)
(num-test (log 2/3) -0.40546510810816)
(num-test (log -2/3) -0.40546510810816+3.14159265358979i)
(num-test (log 2/10) -1.60943791243410)
(num-test (log -2/10) -1.60943791243410+3.14159265358979i)
(num-test (log 2/1234) -6.42486902390539)
(num-test (log -2/1234) -6.42486902390539+3.14159265358979i)
(num-test (log 2/1234000000) -20.24037958186966)
(num-test (log -2/1234000000) -20.24037958186966+3.14159265358979i)
(num-test (log 2/500029) -12.42927419516245)
(num-test (log -2/500029) -12.42927419516245+3.14159265358979i)
(num-test (log 2/362880) -12.10868029952152)
(num-test (log -2/362880) -12.10868029952152+3.14159265358979i)
(num-test (log 3/1) 1.09861228866811)
(num-test (log -3/1) 1.09861228866811+3.14159265358979i)
(num-test (log 3/2) 0.40546510810816)
(num-test (log -3/2) 0.40546510810816+3.14159265358979i)
(num-test (log 3/3) 0.0)
(num-test (log -3/3) 0.0+3.14159265358979i)
(num-test (log 3/10) -1.20397280432594)
(num-test (log -3/10) -1.20397280432594+3.14159265358979i)
(num-test (log 3/1234) -6.01940391579722)
(num-test (log -3/1234) -6.01940391579722+3.14159265358979i)
(num-test (log 3/1234000000) -19.83491447376150)
(num-test (log -3/1234000000) -19.83491447376150+3.14159265358979i)
(num-test (log 3/500029) -12.02380908705428)
(num-test (log -3/500029) -12.02380908705428+3.14159265358979i)
(num-test (log 3/362880) -11.70321519141336)
(num-test (log -3/362880) -11.70321519141336+3.14159265358979i)
(num-test (log 10/1) 2.30258509299405)
(num-test (log -10/1) 2.30258509299405+3.14159265358979i)
(num-test (log 10/2) 1.60943791243410)
(num-test (log -10/2) 1.60943791243410+3.14159265358979i)
(num-test (log 10/3) 1.20397280432594)
(num-test (log -10/3) 1.20397280432594+3.14159265358979i)
(num-test (log 10/10) 0.0)
(num-test (log -10/10) 0.0+3.14159265358979i)
(num-test (log 10/1234) -4.81543111147129)
(num-test (log -10/1234) -4.81543111147129+3.14159265358979i)
(num-test (log 10/1234000000) -18.63094166943556)
(num-test (log -10/1234000000) -18.63094166943556+3.14159265358979i)
(num-test (log 10/500029) -10.81983628272835)
(num-test (log -10/500029) -10.81983628272835+3.14159265358979i)
(num-test (log 10/362880) -10.49924238708742)
(num-test (log -10/362880) -10.49924238708742+3.14159265358979i)
(num-test (log 1234/1) 7.11801620446533)
(num-test (log -1234/1) 7.11801620446533+3.14159265358979i)
(num-test (log 1234/2) 6.42486902390539)
(num-test (log -1234/2) 6.42486902390539+3.14159265358979i)
(num-test (log 1234/3) 6.01940391579722)
(num-test (log -1234/3) 6.01940391579722+3.14159265358979i)
(num-test (log 1234/10) 4.81543111147129)
(num-test (log -1234/10) 4.81543111147129+3.14159265358979i)
(num-test (log 1234/1234) 0.0)
(num-test (log -1234/1234) 0.0+3.14159265358979i)
(num-test (log 1234/1234000000) -13.81551055796427)
(num-test (log -1234/1234000000) -13.81551055796427+3.14159265358979i)
(num-test (log 1234/500029) -6.00440517125706)
(num-test (log -1234/500029) -6.00440517125706+3.14159265358979i)
(num-test (log 1234/362880) -5.68381127561614)
(num-test (log -1234/362880) -5.68381127561614+3.14159265358979i)
(num-test (log 1234000000/1) 20.93352676242961)
(num-test (log -1234000000/1) 20.93352676242961+3.14159265358979i)
(num-test (log 1234000000/2) 20.24037958186966)
(num-test (log -1234000000/2) 20.24037958186966+3.14159265358979i)
(num-test (log 1234000000/3) 19.83491447376150)
(num-test (log -1234000000/3) 19.83491447376150+3.14159265358979i)
(num-test (log 1234000000/10) 18.63094166943556)
(num-test (log -1234000000/10) 18.63094166943556+3.14159265358979i)
(num-test (log 1234000000/1234) 13.81551055796427)
(num-test (log -1234000000/1234) 13.81551055796427+3.14159265358979i)
(num-test (log 1234000000/1234000000) 0.0)
(num-test (log -1234000000/1234000000) 0.0+3.14159265358979i)
(num-test (log 1234000000/500029) 7.81110538670721)
(num-test (log -1234000000/500029) 7.81110538670721+3.14159265358979i)
(num-test (log 1234000000/362880) 8.13169928234814)
(num-test (log -1234000000/362880) 8.13169928234814+3.14159265358979i)
(num-test (log 500029/1) 13.12242137572239)
(num-test (log -500029/1) 13.12242137572239+3.14159265358979i)
(num-test (log 500029/2) 12.42927419516245)
(num-test (log -500029/2) 12.42927419516245+3.14159265358979i)
(num-test (log 500029/3) 12.02380908705428)
(num-test (log -500029/3) 12.02380908705428+3.14159265358979i)
(num-test (log 500029/10) 10.81983628272835)
(num-test (log -500029/10) 10.81983628272835+3.14159265358979i)
(num-test (log 500029/1234) 6.00440517125706)
(num-test (log -500029/1234) 6.00440517125706+3.14159265358979i)
(num-test (log 500029/1234000000) -7.81110538670721)
(num-test (log -500029/1234000000) -7.81110538670721+3.14159265358979i)
(num-test (log 500029/500029) 0.0)
(num-test (log -500029/500029) 0.0+3.14159265358979i)
(num-test (log 500029/362880) 0.32059389564092)
(num-test (log -500029/362880) 0.32059389564092+3.14159265358979i)
(num-test (log 362880/1) 12.80182748008147)
(num-test (log -362880/1) 12.80182748008147+3.14159265358979i)
(num-test (log 362880/2) 12.10868029952152)
(num-test (log -362880/2) 12.10868029952152+3.14159265358979i)
(num-test (log 362880/3) 11.70321519141336)
(num-test (log -362880/3) 11.70321519141336+3.14159265358979i)
(num-test (log 362880/10) 10.49924238708742)
(num-test (log -362880/10) 10.49924238708742+3.14159265358979i)
(num-test (log 362880/1234) 5.68381127561614)
(num-test (log -362880/1234) 5.68381127561614+3.14159265358979i)
(num-test (log 362880/1234000000) -8.13169928234814)
(num-test (log -362880/1234000000) -8.13169928234814+3.14159265358979i)
(num-test (log 362880/500029) -0.32059389564092)
(num-test (log -362880/500029) -0.32059389564092+3.14159265358979i)
(num-test (log 362880/362880) 0.0)
(num-test (log -362880/362880) 0.0+3.14159265358979i)
(num-test (log 0.00000001) -18.42068074395237)
(num-test (log -0.00000001) -18.42068074395237+3.14159265358979i)
(num-test (log 1.0) 0.0)
(num-test (log -1.0) 0.0+3.14159265358979i)
(num-test (log our-pi) 1.14472988584940)
(num-test (log -3.14159265358979) 1.14472988584940+3.14159265358979i)
(num-test (log 2.71828182845905) 1.0)
(num-test (log -2.71828182845905) 1.0+3.14159265358979i)
(num-test (log 1234.0) 7.11801620446533)
(num-test (log -1234.0) 7.11801620446533+3.14159265358979i)
(num-test (log 1234000000.0) 20.93352676242961)
(num-test (log -1234000000.0) 20.93352676242961+3.14159265358979i)
(num-test (log 0.0+0.00000001i) -18.42068074395237+1.57079632679490i)
(num-test (log -0.0+0.00000001i) -18.42068074395237+1.57079632679490i)
(num-test (log 0.0-0.00000001i) -18.42068074395237-1.57079632679490i)
(num-test (log -0.0-0.00000001i) -18.42068074395237-1.57079632679490i)
(num-test (log 0.0+1.0i) 0.0+1.57079632679490i)
(num-test (log -0.0+1.0i) 0.0+1.57079632679490i)
(num-test (log 0.0-1.0i) 0.0-1.57079632679490i)
(num-test (log -0.0-1.0i) 0.0-1.57079632679490i)
(num-test (log 0.0+3.14159265358979i) 1.14472988584940+1.57079632679490i)
(num-test (log -0.0+3.14159265358979i) 1.14472988584940+1.57079632679490i)
(num-test (log 0.0-3.14159265358979i) 1.14472988584940-1.57079632679490i)
(num-test (log -0.0-3.14159265358979i) 1.14472988584940-1.57079632679490i)
(num-test (log 0.0+2.71828182845905i) 1.0+1.57079632679490i)
(num-test (log -0.0+2.71828182845905i) 1.0+1.57079632679490i)
(num-test (log 0.0-2.71828182845905i) 1.0-1.57079632679490i)
(num-test (log -0.0-2.71828182845905i) 1.0-1.57079632679490i)
(num-test (log 0.0+1234.0i) 7.11801620446533+1.57079632679490i)
(num-test (log -0.0+1234.0i) 7.11801620446533+1.57079632679490i)
(num-test (log 0.0-1234.0i) 7.11801620446533-1.57079632679490i)
(num-test (log -0.0-1234.0i) 7.11801620446533-1.57079632679490i)
(num-test (log 0.0+1234000000.0i) 20.93352676242961+1.57079632679490i)
(num-test (log -0.0+1234000000.0i) 20.93352676242961+1.57079632679490i)
(num-test (log 0.0-1234000000.0i) 20.93352676242961-1.57079632679490i)
(num-test (log -0.0-1234000000.0i) 20.93352676242961-1.57079632679490i)
(num-test (log 0.00000001+0.0i) -18.42068074395237)
(num-test (log -0.00000001+0.0i) -18.42068074395237+3.14159265358979i)
(num-test (log 0.00000001-0.0i) -18.42068074395237)
(num-test (log -0.00000001-0.0i) -18.42068074395237+3.14159265358979i)
(num-test (log 0.00000001+0.00000001i) -18.07410715367239+0.78539816339745i)
(num-test (log -0.00000001+0.00000001i) -18.07410715367239+2.35619449019234i)
(num-test (log 0.00000001-0.00000001i) -18.07410715367239-0.78539816339745i)
(num-test (log -0.00000001-0.00000001i) -18.07410715367239-2.35619449019234i)
(num-test (log 0.00000001+1.0i) 0.0+1.57079631679490i)
(num-test (log -0.00000001+1.0i) 0.0+1.57079633679490i)
(num-test (log 0.00000001-1.0i) 0.0-1.57079631679490i)
(num-test (log -0.00000001-1.0i) 0.0-1.57079633679490i)
(num-test (log 0.00000001+3.14159265358979i) 1.14472988584940+1.57079632361180i)
(num-test (log -0.00000001+3.14159265358979i) 1.14472988584940+1.57079632997800i)
(num-test (log 0.00000001-3.14159265358979i) 1.14472988584940-1.57079632361180i)
(num-test (log -0.00000001-3.14159265358979i) 1.14472988584940-1.57079632997800i)
(num-test (log 0.00000001+2.71828182845905i) 1.0+1.57079632311610i)
(num-test (log -0.00000001+2.71828182845905i) 1.0+1.57079633047369i)
(num-test (log 0.00000001-2.71828182845905i) 1.0-1.57079632311610i)
(num-test (log -0.00000001-2.71828182845905i) 1.0-1.57079633047369i)
(num-test (log 0.00000001+1234.0i) 7.11801620446533+1.57079632678679i)
(num-test (log -0.00000001+1234.0i) 7.11801620446533+1.57079632680300i)
(num-test (log 0.00000001-1234.0i) 7.11801620446533-1.57079632678679i)
(num-test (log -0.00000001-1234.0i) 7.11801620446533-1.57079632680300i)
(num-test (log 0.00000001+1234000000.0i) 20.93352676242961+1.57079632679490i)
(num-test (log -0.00000001+1234000000.0i) 20.93352676242961+1.57079632679490i)
(num-test (log 0.00000001-1234000000.0i) 20.93352676242961-1.57079632679490i)
(num-test (log -0.00000001-1234000000.0i) 20.93352676242961-1.57079632679490i)
(num-test (log 1.0+0.0i) 0.0)
(num-test (log -1.0+0.0i) 0.0+3.14159265358979i)
(num-test (log 1.0-0.0i) 0.0)
(num-test (log -1.0-0.0i) 0.0+3.14159265358979i)
(num-test (log 1.0+0.00000001i) 0.0+0.00000001i)
(num-test (log -1.0+0.00000001i) 0.0+3.14159264358979i)
(num-test (log 1.0-0.00000001i) 0.0-0.00000001i)
(num-test (log -1.0-0.00000001i) 0.0-3.14159264358979i)
(num-test (log 1.0+1.0i) 0.34657359027997+0.78539816339745i)
(num-test (log -1.0+1.0i) 0.34657359027997+2.35619449019234i)
(num-test (log 1.0-1.0i) 0.34657359027997-0.78539816339745i)
(num-test (log -1.0-1.0i) 0.34657359027997-2.35619449019234i)
(num-test (log 1.0+3.14159265358979i) 1.19298515341341+1.26262725567891i)
(num-test (log -1.0+3.14159265358979i) 1.19298515341341+1.87896539791088i)
(num-test (log 1.0-3.14159265358979i) 1.19298515341341-1.26262725567891i)
(num-test (log -1.0-3.14159265358979i) 1.19298515341341-1.87896539791088i)
(num-test (log 1.0+2.71828182845905i) 1.06346400552149+1.21828290501728i)
(num-test (log -1.0+2.71828182845905i) 1.06346400552149+1.92330974857252i)
(num-test (log 1.0-2.71828182845905i) 1.06346400552149-1.21828290501728i)
(num-test (log -1.0-2.71828182845905i) 1.06346400552149-1.92330974857252i)
(num-test (log 1.0+1234.0i) 7.11801653281724+1.56998595420081i)
(num-test (log -1.0+1234.0i) 7.11801653281724+1.57160669938898i)
(num-test (log 1.0-1234.0i) 7.11801653281724-1.56998595420081i)
(num-test (log -1.0-1234.0i) 7.11801653281724-1.57160669938898i)
(num-test (log 1.0+1234000000.0i) 20.93352676242961+1.57079632598452i)
(num-test (log -1.0+1234000000.0i) 20.93352676242961+1.57079632760527i)
(num-test (log 1.0-1234000000.0i) 20.93352676242961-1.57079632598452i)
(num-test (log -1.0-1234000000.0i) 20.93352676242961-1.57079632760527i)
(num-test (log 3.14159265358979+0.0i) 1.14472988584940)
(num-test (log -3.14159265358979+0.0i) 1.14472988584940+3.14159265358979i)
(num-test (log 3.14159265358979-0.0i) 1.14472988584940)
(num-test (log -3.14159265358979-0.0i) 1.14472988584940+3.14159265358979i)
(num-test (log 3.14159265358979+0.00000001i) 1.14472988584940+0.00000000318310i)
(num-test (log -3.14159265358979+0.00000001i) 1.14472988584940+3.14159265040669i)
(num-test (log 3.14159265358979-0.00000001i) 1.14472988584940-0.00000000318310i)
(num-test (log -3.14159265358979-0.00000001i) 1.14472988584940-3.14159265040669i)
(num-test (log 3.14159265358979+1.0i) 1.19298515341341+0.30816907111598i)
(num-test (log -3.14159265358979+1.0i) 1.19298515341341+2.83342358247381i)
(num-test (log 3.14159265358979-1.0i) 1.19298515341341-0.30816907111598i)
(num-test (log -3.14159265358979-1.0i) 1.19298515341341-2.83342358247381i)
(num-test (log 3.14159265358979+3.14159265358979i) 1.49130347612937+0.78539816339745i)
(num-test (log -3.14159265358979+3.14159265358979i) 1.49130347612937+2.35619449019234i)
(num-test (log 3.14159265358979-3.14159265358979i) 1.49130347612937-0.78539816339745i)
(num-test (log -3.14159265358979-3.14159265358979i) 1.49130347612937-2.35619449019234i)
(num-test (log 3.14159265358979+2.71828182845905i) 1.42415703773030+0.71328454043905i)
(num-test (log -3.14159265358979+2.71828182845905i) 1.42415703773030+2.42830811315074i)
(num-test (log 3.14159265358979-2.71828182845905i) 1.42415703773030-0.71328454043905i)
(num-test (log -3.14159265358979-2.71828182845905i) 1.42415703773030-2.42830811315074i)
(num-test (log 3.14159265358979+1234.0i) 7.11801944515932+1.56825047114960i)
(num-test (log -3.14159265358979+1234.0i) 7.11801944515932+1.57334218244020i)
(num-test (log 3.14159265358979-1234.0i) 7.11801944515932-1.56825047114960i)
(num-test (log -3.14159265358979-1234.0i) 7.11801944515932-1.57334218244020i)
(num-test (log 3.14159265358979+1234000000.0i) 20.93352676242961+1.57079632424904i)
(num-test (log -3.14159265358979+1234000000.0i) 20.93352676242961+1.57079632934076i)
(num-test (log 3.14159265358979-1234000000.0i) 20.93352676242961-1.57079632424904i)
(num-test (log -3.14159265358979-1234000000.0i) 20.93352676242961-1.57079632934076i)
(num-test (log 2.71828182845905+0.0i) 1.0)
(num-test (log -2.71828182845905+0.0i) 1.0+3.14159265358979i)
(num-test (log 2.71828182845905-0.0i) 1.0)
(num-test (log -2.71828182845905-0.0i) 1.0+3.14159265358979i)
(num-test (log 2.71828182845905+0.00000001i) 1.0+0.00000000367879i)
(num-test (log -2.71828182845905+0.00000001i) 1.0+3.14159264991100i)
(num-test (log 2.71828182845905-0.00000001i) 1.0-0.00000000367879i)
(num-test (log -2.71828182845905-0.00000001i) 1.0-3.14159264991100i)
(num-test (log 2.71828182845905+1.0i) 1.06346400552149+0.35251342177762i)
(num-test (log -2.71828182845905+1.0i) 1.06346400552149+2.78907923181217i)
(num-test (log 2.71828182845905-1.0i) 1.06346400552149-0.35251342177762i)
(num-test (log -2.71828182845905-1.0i) 1.06346400552149-2.78907923181217i)
(num-test (log 2.71828182845905+3.14159265358979i) 1.42415703773030+0.85751178635585i)
(num-test (log -2.71828182845905+3.14159265358979i) 1.42415703773030+2.28408086723395i)
(num-test (log 2.71828182845905-3.14159265358979i) 1.42415703773030-0.85751178635585i)
(num-test (log -2.71828182845905-3.14159265358979i) 1.42415703773030-2.28408086723395i)
(num-test (log 2.71828182845905+2.71828182845905i) 1.34657359027997+0.78539816339745i)
(num-test (log -2.71828182845905+2.71828182845905i) 1.34657359027997+2.35619449019234i)
(num-test (log 2.71828182845905-2.71828182845905i) 1.34657359027997-0.78539816339745i)
(num-test (log -2.71828182845905-2.71828182845905i) 1.34657359027997-2.35619449019234i)
(num-test (log 2.71828182845905+1234.0i) 7.11801863067090+1.56859350877892i)
(num-test (log -2.71828182845905+1234.0i) 7.11801863067090+1.57299914481088i)
(num-test (log 2.71828182845905-1234.0i) 7.11801863067090-1.56859350877892i)
(num-test (log -2.71828182845905-1234.0i) 7.11801863067090-1.57299914481088i)
(num-test (log 2.71828182845905+1234000000.0i) 20.93352676242961+1.57079632459208i)
(num-test (log -2.71828182845905+1234000000.0i) 20.93352676242961+1.57079632899772i)
(num-test (log 2.71828182845905-1234000000.0i) 20.93352676242961-1.57079632459208i)
(num-test (log -2.71828182845905-1234000000.0i) 20.93352676242961-1.57079632899772i)
(num-test (log 1234.0+0.0i) 7.11801620446533)
(num-test (log -1234.0+0.0i) 7.11801620446533+3.14159265358979i)
(num-test (log 1234.0-0.0i) 7.11801620446533)
(num-test (log -1234.0-0.0i) 7.11801620446533+3.14159265358979i)
(num-test (log 1234.0+0.00000001i) 7.11801620446533+0.00000000000810i)
(num-test (log -1234.0+0.00000001i) 7.11801620446533+3.14159265358169i)
(num-test (log 1234.0-0.00000001i) 7.11801620446533-0.00000000000810i)
(num-test (log -1234.0-0.00000001i) 7.11801620446533-3.14159265358169i)
(num-test (log 1234.0+1.0i) 7.11801653281724+0.00081037259408i)
(num-test (log -1234.0+1.0i) 7.11801653281724+3.14078228099571i)
(num-test (log 1234.0-1.0i) 7.11801653281724-0.00081037259408i)
(num-test (log -1234.0-1.0i) 7.11801653281724-3.14078228099571i)
(num-test (log 1234.0+3.14159265358979i) 7.11801944515932+0.00254585564530i)
(num-test (log -1234.0+3.14159265358979i) 7.11801944515932+3.13904679794449i)
(num-test (log 1234.0-3.14159265358979i) 7.11801944515932-0.00254585564530i)
(num-test (log -1234.0-3.14159265358979i) 7.11801944515932-3.13904679794449i)
(num-test (log 1234.0+2.71828182845905i) 7.11801863067090+0.00220281801598i)
(num-test (log -1234.0+2.71828182845905i) 7.11801863067090+3.13938983557381i)
(num-test (log 1234.0-2.71828182845905i) 7.11801863067090-0.00220281801598i)
(num-test (log -1234.0-2.71828182845905i) 7.11801863067090-3.13938983557381i)
(num-test (log 1234.0+1234.0i) 7.46458979474531+0.78539816339745i)
(num-test (log -1234.0+1234.0i) 7.46458979474531+2.35619449019234i)
(num-test (log 1234.0-1234.0i) 7.46458979474531-0.78539816339745i)
(num-test (log -1234.0-1234.0i) 7.46458979474531-2.35619449019234i)
(num-test (log 1234.0+1234000000.0i) 20.93352676243011+1.57079532679490i)
(num-test (log -1234.0+1234000000.0i) 20.93352676243011+1.57079732679490i)
(num-test (log 1234.0-1234000000.0i) 20.93352676243011-1.57079532679490i)
(num-test (log -1234.0-1234000000.0i) 20.93352676243011-1.57079732679490i)
(num-test (log 1234000000.0+0.0i) 20.93352676242961)
(num-test (log -1234000000.0+0.0i) 20.93352676242961+3.14159265358979i)
(num-test (log 1234000000.0-0.0i) 20.93352676242961)
(num-test (log -1234000000.0-0.0i) 20.93352676242961+3.14159265358979i)
(num-test (log 1234000000.0+0.00000001i) 20.93352676242961+0.0i)
(num-test (log -1234000000.0+0.00000001i) 20.93352676242961+3.14159265358979i)
(num-test (log 1234000000.0-0.00000001i) 20.93352676242961-0.0i)
(num-test (log -1234000000.0-0.00000001i) 20.93352676242961-3.14159265358979i)
(num-test (log 1234000000.0+1.0i) 20.93352676242961+0.00000000081037i)
(num-test (log -1234000000.0+1.0i) 20.93352676242961+3.14159265277942i)
(num-test (log 1234000000.0-1.0i) 20.93352676242961-0.00000000081037i)
(num-test (log -1234000000.0-1.0i) 20.93352676242961-3.14159265277942i)
(num-test (log 1234000000.0+3.14159265358979i) 20.93352676242961+0.00000000254586i)
(num-test (log -1234000000.0+3.14159265358979i) 20.93352676242961+3.14159265104393i)
(num-test (log 1234000000.0-3.14159265358979i) 20.93352676242961-0.00000000254586i)
(num-test (log -1234000000.0-3.14159265358979i) 20.93352676242961-3.14159265104393i)
(num-test (log 1234000000.0+2.71828182845905i) 20.93352676242961+0.00000000220282i)
(num-test (log -1234000000.0+2.71828182845905i) 20.93352676242961+3.14159265138697i)
(num-test (log 1234000000.0-2.71828182845905i) 20.93352676242961-0.00000000220282i)
(num-test (log -1234000000.0-2.71828182845905i) 20.93352676242961-3.14159265138697i)
(num-test (log 1234000000.0+1234.0i) 20.93352676243011+0.00000100000000i)
(num-test (log -1234000000.0+1234.0i) 20.93352676243011+3.14159165358979i)
(num-test (log 1234000000.0-1234.0i) 20.93352676243011-0.00000100000000i)
(num-test (log -1234000000.0-1234.0i) 20.93352676243011-3.14159165358979i)
(num-test (log 1234000000.0+1234000000.0i) 21.28010035270958+0.78539816339745i)
(num-test (log -1234000000.0+1234000000.0i) 21.28010035270958+2.35619449019234i)
(num-test (log 1234000000.0-1234000000.0i) 21.28010035270958-0.78539816339745i)
(num-test (log -1234000000.0-1234000000.0i) 21.28010035270958-2.35619449019234i)
(num-test (log .3678794411714423) -1.0)
(num-test (log 22026.46579480672) 10.0)
(num-test (log 2.688117141816135E+43) 100.0)
(num-test (log 4.5399929762484853E-5) -10.0)
(num-test (log 3.720075976020836E-44) -100.0)
(num-test (log 2.2250739e-308) -708.39641851362)
(num-test (log 1.7976931e+308) 709.78271287399)
(num-test (log 1.19209289550781250e-07+0.0e+00i) -1.5942385152878742117e1+0.0i)
(num-test (log -1.19209289550781250e-07+0.0e+00i) -1.5942385152878742117e1+3.1415926535897932385e0i)
(num-test (log 1.19209289550781250e-07+1.19209289550781250e-07i) -1.5595811562598769462e1+7.8539816339744830962e-1i)
(num-test (log 1.19209289550781250e-07-1.19209289550781250e-07i) -1.5595811562598769462e1-7.8539816339744830962e-1i)
(num-test (log -1.19209289550781250e-07+1.19209289550781250e-07i) -1.5595811562598769462e1+2.3561944901923449288e0i)
(num-test (log -1.19209289550781250e-07-1.19209289550781250e-07i) -1.5595811562598769462e1-2.3561944901923449288e0i)
(num-test (log 1.19209289550781250e-07+5.0e-01i) -6.9314718055991688771e-1+1.5707960883763175177e0i)
(num-test (log 1.19209289550781250e-07-5.0e-01i) -6.9314718055991688771e-1-1.5707960883763175177e0i)
(num-test (log -1.19209289550781250e-07+5.0e-01i) -6.9314718055991688771e-1+1.5707965652134757208e0i)
(num-test (log -1.19209289550781250e-07-5.0e-01i) -6.9314718055991688771e-1-1.5707965652134757208e0i)
(num-test (log 1.19209289550781250e-07+1.0e+00i) 7.1054273576009513716e-15+1.5707962075856070685e0i)
(num-test (log 1.19209289550781250e-07-1.0e+00i) 7.1054273576009513716e-15-1.5707962075856070685e0i)
(num-test (log -1.19209289550781250e-07+1.0e+00i) 7.1054273576009513716e-15+1.570796446004186170e0i)
(num-test (log -1.19209289550781250e-07-1.0e+00i) 7.1054273576009513716e-15-1.570796446004186170e0i)
(num-test (log 1.19209289550781250e-07+2.0e+00i) 6.9314718055994708577e-1+1.5707962671902518438e0i)
(num-test (log 1.19209289550781250e-07-2.0e+00i) 6.9314718055994708577e-1-1.5707962671902518438e0i)
(num-test (log -1.19209289550781250e-07+2.0e+00i) 6.9314718055994708577e-1+1.5707963863995413946e0i)
(num-test (log -1.19209289550781250e-07-2.0e+00i) 6.9314718055994708577e-1-1.5707963863995413946e0i)
(num-test (log 1.19209289550781250e-07+8.3886080e+06i) 1.5942385152878742117e1+1.5707963267948824084e0i)
(num-test (log 1.19209289550781250e-07-8.3886080e+06i) 1.5942385152878742117e1-1.5707963267948824084e0i)
(num-test (log -1.19209289550781250e-07+8.3886080e+06i) 1.5942385152878742117e1+1.5707963267949108301e0i)
(num-test (log -1.19209289550781250e-07-8.3886080e+06i) 1.5942385152878742117e1-1.5707963267949108301e0i)
(num-test (log 5.0e-01+0.0e+00i) -6.9314718055994530942e-1+0.0i)
(num-test (log -5.0e-01+0.0e+00i) -6.9314718055994530942e-1+3.1415926535897932385e0i)
(num-test (log 5.0e-01+1.19209289550781250e-07i) -6.9314718055991688771e-1+2.3841857910155798249e-7i)
(num-test (log 5.0e-01-1.19209289550781250e-07i) -6.9314718055991688771e-1-2.3841857910155798249e-7i)
(num-test (log -5.0e-01+1.19209289550781250e-07i) -6.9314718055991688771e-1+3.1415924151712141369e0i)
(num-test (log -5.0e-01-1.19209289550781250e-07i) -6.9314718055991688771e-1-3.1415924151712141369e0i)
(num-test (log 5.0e-01+5.0e-01i) -3.4657359027997265471e-1+7.8539816339744830962e-1i)
(num-test (log 5.0e-01-5.0e-01i) -3.4657359027997265471e-1-7.8539816339744830962e-1i)
(num-test (log -5.0e-01+5.0e-01i) -3.4657359027997265471e-1+2.3561944901923449288e0i)
(num-test (log -5.0e-01-5.0e-01i) -3.4657359027997265471e-1-2.3561944901923449288e0i)
(num-test (log 5.0e-01+1.0e+00i) 1.1157177565710487788e-1+1.1071487177940905030e0i)
(num-test (log 5.0e-01-1.0e+00i) 1.1157177565710487788e-1-1.1071487177940905030e0i)
(num-test (log -5.0e-01+1.0e+00i) 1.1157177565710487788e-1+2.0344439357957027354e0i)
(num-test (log -5.0e-01-1.0e+00i) 1.1157177565710487788e-1-2.0344439357957027354e0i)
(num-test (log 5.0e-01+2.0e+00i) 7.2345949146816273071e-1+1.3258176636680324651e0i)
(num-test (log 5.0e-01-2.0e+00i) 7.2345949146816273071e-1-1.3258176636680324651e0i)
(num-test (log -5.0e-01+2.0e+00i) 7.2345949146816273071e-1+1.8157749899217607734e0i)
(num-test (log -5.0e-01-2.0e+00i) 7.2345949146816273071e-1-1.8157749899217607734e0i)
(num-test (log 5.0e-01+8.3886080e+06i) 1.5942385152878743893e1+1.5707962671902518438e0i)
(num-test (log 5.0e-01-8.3886080e+06i) 1.5942385152878743893e1-1.5707962671902518438e0i)
(num-test (log -5.0e-01+8.3886080e+06i) 1.5942385152878743893e1+1.5707963863995413946e0i)
(num-test (log -5.0e-01-8.3886080e+06i) 1.5942385152878743893e1-1.5707963863995413946e0i)
(num-test (log 1.0e+00+0.0e+00i) 0e0+0.0i)
(num-test (log -1.0e+00+0.0e+00i) 0+3.1415926535897932385e0i)
(num-test (log 1.0e+00+1.19209289550781250e-07i) 7.1054273576009513716e-15+1.1920928955078068531e-7i)
(num-test (log 1.0e+00-1.19209289550781250e-07i) 7.1054273576009513716e-15-1.1920928955078068531e-7i)
(num-test (log -1.0e+00+1.19209289550781250e-07i) 7.1054273576009513716e-15+3.1415925343805036877e0i)
(num-test (log -1.0e+00-1.19209289550781250e-07i) 7.1054273576009513716e-15-3.1415925343805036877e0i)
(num-test (log 1.0e+00+5.0e-01i) 1.1157177565710487788e-1+4.6364760900080611621e-1i)
(num-test (log 1.0e+00-5.0e-01i) 1.1157177565710487788e-1-4.6364760900080611621e-1i)
(num-test (log -1.0e+00+5.0e-01i) 1.1157177565710487788e-1+2.6779450445889871222e0i)
(num-test (log -1.0e+00-5.0e-01i) 1.1157177565710487788e-1-2.6779450445889871222e0i)
(num-test (log 1.0e+00+1.0e+00i) 3.4657359027997265471e-1+7.8539816339744830962e-1i)
(num-test (log 1.0e+00-1.0e+00i) 3.4657359027997265471e-1-7.8539816339744830962e-1i)
(num-test (log -1.0e+00+1.0e+00i) 3.4657359027997265471e-1+2.3561944901923449288e0i)
(num-test (log -1.0e+00-1.0e+00i) 3.4657359027997265471e-1-2.3561944901923449288e0i)
(num-test (log 1.0e+00+2.0e+00i) 8.0471895621705018730e-1+1.1071487177940905030e0i)
(num-test (log 1.0e+00-2.0e+00i) 8.0471895621705018730e-1-1.1071487177940905030e0i)
(num-test (log -1.0e+00+2.0e+00i) 8.0471895621705018730e-1+2.0344439357957027354e0i)
(num-test (log -1.0e+00-2.0e+00i) 8.0471895621705018730e-1-2.0344439357957027354e0i)
(num-test (log 1.0e+00+8.3886080e+06i) 1.5942385152878749222e1+1.5707962075856070685e0i)
(num-test (log 1.0e+00-8.3886080e+06i) 1.5942385152878749222e1-1.5707962075856070685e0i)
(num-test (log -1.0e+00+8.3886080e+06i) 1.5942385152878749222e1+1.570796446004186170e0i)
(num-test (log -1.0e+00-8.3886080e+06i) 1.5942385152878749222e1-1.570796446004186170e0i)
(num-test (log 2.0e+00+0.0e+00i) 6.9314718055994530942e-1+0.0i)
(num-test (log -2.0e+00+0.0e+00i) 6.9314718055994530942e-1+3.1415926535897932385e0i)
(num-test (log 2.0e+00+1.19209289550781250e-07i) 6.9314718055994708577e-1+5.9604644775390554414e-8i)
(num-test (log 2.0e+00-1.19209289550781250e-07i) 6.9314718055994708577e-1-5.9604644775390554414e-8i)
(num-test (log -2.0e+00+1.19209289550781250e-07i) 6.9314718055994708577e-1+3.1415925939851484631e0i)
(num-test (log -2.0e+00-1.19209289550781250e-07i) 6.9314718055994708577e-1-3.1415925939851484631e0i)
(num-test (log 2.0e+00+5.0e-01i) 7.2345949146816273071e-1+2.4497866312686415417e-1i)
(num-test (log 2.0e+00-5.0e-01i) 7.2345949146816273071e-1-2.4497866312686415417e-1i)
(num-test (log -2.0e+00+5.0e-01i) 7.2345949146816273071e-1+2.8966139904629290843e0i)
(num-test (log -2.0e+00-5.0e-01i) 7.2345949146816273071e-1-2.8966139904629290843e0i)
(num-test (log 2.0e+00+1.0e+00i) 8.0471895621705018730e-1+4.6364760900080611621e-1i)
(num-test (log 2.0e+00-1.0e+00i) 8.0471895621705018730e-1-4.6364760900080611621e-1i)
(num-test (log -2.0e+00+1.0e+00i) 8.0471895621705018730e-1+2.6779450445889871222e0i)
(num-test (log -2.0e+00-1.0e+00i) 8.0471895621705018730e-1-2.6779450445889871222e0i)
(num-test (log 2.0e+00+2.0e+00i) 1.0397207708399179641e0+7.8539816339744830962e-1i)
(num-test (log 2.0e+00-2.0e+00i) 1.0397207708399179641e0-7.8539816339744830962e-1i)
(num-test (log -2.0e+00+2.0e+00i) 1.0397207708399179641e0+2.3561944901923449288e0i)
(num-test (log -2.0e+00-2.0e+00i) 1.0397207708399179641e0-2.3561944901923449288e0i)
(num-test (log 2.0e+00+8.3886080e+06i) 1.5942385152878770538e1+1.5707960883763175177e0i)
(num-test (log 2.0e+00-8.3886080e+06i) 1.5942385152878770538e1-1.5707960883763175177e0i)
(num-test (log -2.0e+00+8.3886080e+06i) 1.5942385152878770538e1+1.5707965652134757208e0i)
(num-test (log -2.0e+00-8.3886080e+06i) 1.5942385152878770538e1-1.5707965652134757208e0i)
(num-test (log 8.3886080e+06+0.0e+00i) 1.5942385152878742117e1+0.0i)
(num-test (log -8.3886080e+06+0.0e+00i) 1.5942385152878742117e1+3.1415926535897932385e0i)
(num-test (log 8.3886080e+06+1.19209289550781250e-07i) 1.5942385152878742117e1+1.4210854715202003717e-14i)
(num-test (log 8.3886080e+06-1.19209289550781250e-07i) 1.5942385152878742117e1-1.4210854715202003717e-14i)
(num-test (log -8.3886080e+06+1.19209289550781250e-07i) 1.5942385152878742117e1+3.1415926535897790276e0i)
(num-test (log -8.3886080e+06-1.19209289550781250e-07i) 1.5942385152878742117e1-3.1415926535897790276e0i)
(num-test (log 8.3886080e+06+5.0e-01i) 1.5942385152878743893e1+5.9604644775390554414e-8i)
(num-test (log 8.3886080e+06-5.0e-01i) 1.5942385152878743893e1-5.9604644775390554414e-8i)
(num-test (log -8.3886080e+06+5.0e-01i) 1.5942385152878743893e1+3.1415925939851484631e0i)
(num-test (log -8.3886080e+06-5.0e-01i) 1.5942385152878743893e1-3.1415925939851484631e0i)
(num-test (log 8.3886080e+06+1.0e+00i) 1.5942385152878749222e1+1.1920928955078068531e-7i)
(num-test (log 8.3886080e+06-1.0e+00i) 1.5942385152878749222e1-1.1920928955078068531e-7i)
(num-test (log -8.3886080e+06+1.0e+00i) 1.5942385152878749222e1+3.1415925343805036877e0i)
(num-test (log -8.3886080e+06-1.0e+00i) 1.5942385152878749222e1-3.1415925343805036877e0i)
(num-test (log 8.3886080e+06+2.0e+00i) 1.5942385152878770538e1+2.3841857910155798249e-7i)
(num-test (log 8.3886080e+06-2.0e+00i) 1.5942385152878770538e1-2.3841857910155798249e-7i)
(num-test (log -8.3886080e+06+2.0e+00i) 1.5942385152878770538e1+3.1415924151712141369e0i)
(num-test (log -8.3886080e+06-2.0e+00i) 1.5942385152878770538e1-3.1415924151712141369e0i)
(num-test (log 8.3886080e+06+8.3886080e+06i) 1.6288958743158714771e1+7.8539816339744830962e-1i)
(num-test (log 8.3886080e+06-8.3886080e+06i) 1.6288958743158714771e1-7.8539816339744830962e-1i)
(num-test (log -8.3886080e+06+8.3886080e+06i) 1.6288958743158714771e1+2.3561944901923449288e0i)
(num-test (log -8.3886080e+06-8.3886080e+06i) 1.6288958743158714771e1-2.3561944901923449288e0i)
(num-test (log 0.0e+00+1.19209289550781250e-07i) -1.5942385152878742117e1+1.5707963267948966192e0i)
(num-test (log 0.0e+00-1.19209289550781250e-07i) -1.5942385152878742117e1-1.5707963267948966192e0i)
(num-test (log 0.0e+00+5.0e-01i) -6.9314718055994530942e-1+1.5707963267948966192e0i)
(num-test (log 0.0e+00-5.0e-01i) -6.9314718055994530942e-1-1.5707963267948966192e0i)
(num-test (log 0.0e+00+1.0e+00i) 0+1.5707963267948966192e0i)
(num-test (log 0.0e+00-1.0e+00i) 0-1.5707963267948966192e0i)
(num-test (log 0.0e+00+2.0e+00i) 6.9314718055994530942e-1+1.5707963267948966192e0i)
(num-test (log 0.0e+00-2.0e+00i) 6.9314718055994530942e-1-1.5707963267948966192e0i)
(num-test (log 0.0e+00+8.3886080e+06i) 1.5942385152878742117e1+1.5707963267948966192e0i)
(num-test (log 0.0e+00-8.3886080e+06i) 1.5942385152878742117e1-1.5707963267948966192e0i)
(num-test (log 1.19209289550781250e-07+1.19209289550781250e-07i) -1.5595811562598769462e1+7.8539816339744830962e-1i)
(num-test (log 1.19209289550781250e-07-1.19209289550781250e-07i) -1.5595811562598769462e1-7.8539816339744830962e-1i)
(num-test (log -1.19209289550781250e-07+1.19209289550781250e-07i) -1.5595811562598769462e1+2.3561944901923449288e0i)
(num-test (log -1.19209289550781250e-07-1.19209289550781250e-07i) -1.5595811562598769462e1-2.3561944901923449288e0i)
(num-test (log 1.19209289550781250e-07+5.0e-01i) -6.9314718055991688771e-1+1.5707960883763175177e0i)
(num-test (log 1.19209289550781250e-07-5.0e-01i) -6.9314718055991688771e-1-1.5707960883763175177e0i)
(num-test (log -1.19209289550781250e-07+5.0e-01i) -6.9314718055991688771e-1+1.5707965652134757208e0i)
(num-test (log -1.19209289550781250e-07-5.0e-01i) -6.9314718055991688771e-1-1.5707965652134757208e0i)
(num-test (log 1.19209289550781250e-07+1.0e+00i) 7.1054273576009513716e-15+1.5707962075856070685e0i)
(num-test (log 1.19209289550781250e-07-1.0e+00i) 7.1054273576009513716e-15-1.5707962075856070685e0i)
(num-test (log -1.19209289550781250e-07+1.0e+00i) 7.1054273576009513716e-15+1.570796446004186170e0i)
(num-test (log -1.19209289550781250e-07-1.0e+00i) 7.1054273576009513716e-15-1.570796446004186170e0i)
(num-test (log 1.19209289550781250e-07+2.0e+00i) 6.9314718055994708577e-1+1.5707962671902518438e0i)
(num-test (log 1.19209289550781250e-07-2.0e+00i) 6.9314718055994708577e-1-1.5707962671902518438e0i)
(num-test (log -1.19209289550781250e-07+2.0e+00i) 6.9314718055994708577e-1+1.5707963863995413946e0i)
(num-test (log -1.19209289550781250e-07-2.0e+00i) 6.9314718055994708577e-1-1.5707963863995413946e0i)
(num-test (log 1.19209289550781250e-07+8.3886080e+06i) 1.5942385152878742117e1+1.5707963267948824084e0i)
(num-test (log 1.19209289550781250e-07-8.3886080e+06i) 1.5942385152878742117e1-1.5707963267948824084e0i)
(num-test (log -1.19209289550781250e-07+8.3886080e+06i) 1.5942385152878742117e1+1.5707963267949108301e0i)
(num-test (log -1.19209289550781250e-07-8.3886080e+06i) 1.5942385152878742117e1-1.5707963267949108301e0i)
(num-test (log 5.0e-01+1.19209289550781250e-07i) -6.9314718055991688771e-1+2.3841857910155798249e-7i)
(num-test (log 5.0e-01-1.19209289550781250e-07i) -6.9314718055991688771e-1-2.3841857910155798249e-7i)
(num-test (log -5.0e-01+1.19209289550781250e-07i) -6.9314718055991688771e-1+3.1415924151712141369e0i)
(num-test (log -5.0e-01-1.19209289550781250e-07i) -6.9314718055991688771e-1-3.1415924151712141369e0i)
(num-test (log 5.0e-01+5.0e-01i) -3.4657359027997265471e-1+7.8539816339744830962e-1i)
(num-test (log 5.0e-01-5.0e-01i) -3.4657359027997265471e-1-7.8539816339744830962e-1i)
(num-test (log -5.0e-01+5.0e-01i) -3.4657359027997265471e-1+2.3561944901923449288e0i)
(num-test (log -5.0e-01-5.0e-01i) -3.4657359027997265471e-1-2.3561944901923449288e0i)
(num-test (log 5.0e-01+1.0e+00i) 1.1157177565710487788e-1+1.1071487177940905030e0i)
(num-test (log 5.0e-01-1.0e+00i) 1.1157177565710487788e-1-1.1071487177940905030e0i)
(num-test (log -5.0e-01+1.0e+00i) 1.1157177565710487788e-1+2.0344439357957027354e0i)
(num-test (log -5.0e-01-1.0e+00i) 1.1157177565710487788e-1-2.0344439357957027354e0i)
(num-test (log 5.0e-01+2.0e+00i) 7.2345949146816273071e-1+1.3258176636680324651e0i)
(num-test (log 5.0e-01-2.0e+00i) 7.2345949146816273071e-1-1.3258176636680324651e0i)
(num-test (log -5.0e-01+2.0e+00i) 7.2345949146816273071e-1+1.8157749899217607734e0i)
(num-test (log -5.0e-01-2.0e+00i) 7.2345949146816273071e-1-1.8157749899217607734e0i)
(num-test (log 5.0e-01+8.3886080e+06i) 1.5942385152878743893e1+1.5707962671902518438e0i)
(num-test (log 5.0e-01-8.3886080e+06i) 1.5942385152878743893e1-1.5707962671902518438e0i)
(num-test (log -5.0e-01+8.3886080e+06i) 1.5942385152878743893e1+1.5707963863995413946e0i)
(num-test (log -5.0e-01-8.3886080e+06i) 1.5942385152878743893e1-1.5707963863995413946e0i)
(num-test (log 1.0e+00+1.19209289550781250e-07i) 7.1054273576009513716e-15+1.1920928955078068531e-7i)
(num-test (log 1.0e+00-1.19209289550781250e-07i) 7.1054273576009513716e-15-1.1920928955078068531e-7i)
(num-test (log -1.0e+00+1.19209289550781250e-07i) 7.1054273576009513716e-15+3.1415925343805036877e0i)
(num-test (log -1.0e+00-1.19209289550781250e-07i) 7.1054273576009513716e-15-3.1415925343805036877e0i)
(num-test (log 1.0e+00+5.0e-01i) 1.1157177565710487788e-1+4.6364760900080611621e-1i)
(num-test (log 1.0e+00-5.0e-01i) 1.1157177565710487788e-1-4.6364760900080611621e-1i)
(num-test (log -1.0e+00+5.0e-01i) 1.1157177565710487788e-1+2.6779450445889871222e0i)
(num-test (log -1.0e+00-5.0e-01i) 1.1157177565710487788e-1-2.6779450445889871222e0i)
(num-test (log 1.0e+00+1.0e+00i) 3.4657359027997265471e-1+7.8539816339744830962e-1i)
(num-test (log 1.0e+00-1.0e+00i) 3.4657359027997265471e-1-7.8539816339744830962e-1i)
(num-test (log -1.0e+00+1.0e+00i) 3.4657359027997265471e-1+2.3561944901923449288e0i)
(num-test (log -1.0e+00-1.0e+00i) 3.4657359027997265471e-1-2.3561944901923449288e0i)
(num-test (log 1.0e+00+2.0e+00i) 8.0471895621705018730e-1+1.1071487177940905030e0i)
(num-test (log 1.0e+00-2.0e+00i) 8.0471895621705018730e-1-1.1071487177940905030e0i)
(num-test (log -1.0e+00+2.0e+00i) 8.0471895621705018730e-1+2.0344439357957027354e0i)
(num-test (log -1.0e+00-2.0e+00i) 8.0471895621705018730e-1-2.0344439357957027354e0i)
(num-test (log 1.0e+00+8.3886080e+06i) 1.5942385152878749222e1+1.5707962075856070685e0i)
(num-test (log 1.0e+00-8.3886080e+06i) 1.5942385152878749222e1-1.5707962075856070685e0i)
(num-test (log -1.0e+00+8.3886080e+06i) 1.5942385152878749222e1+1.570796446004186170e0i)
(num-test (log -1.0e+00-8.3886080e+06i) 1.5942385152878749222e1-1.570796446004186170e0i)
(num-test (log 2.0e+00+1.19209289550781250e-07i) 6.9314718055994708577e-1+5.9604644775390554414e-8i)
(num-test (log 2.0e+00-1.19209289550781250e-07i) 6.9314718055994708577e-1-5.9604644775390554414e-8i)
(num-test (log -2.0e+00+1.19209289550781250e-07i) 6.9314718055994708577e-1+3.1415925939851484631e0i)
(num-test (log -2.0e+00-1.19209289550781250e-07i) 6.9314718055994708577e-1-3.1415925939851484631e0i)
(num-test (log 2.0e+00+5.0e-01i) 7.2345949146816273071e-1+2.4497866312686415417e-1i)
(num-test (log 2.0e+00-5.0e-01i) 7.2345949146816273071e-1-2.4497866312686415417e-1i)
(num-test (log -2.0e+00+5.0e-01i) 7.2345949146816273071e-1+2.8966139904629290843e0i)
(num-test (log -2.0e+00-5.0e-01i) 7.2345949146816273071e-1-2.8966139904629290843e0i)
(num-test (log 2.0e+00+1.0e+00i) 8.0471895621705018730e-1+4.6364760900080611621e-1i)
(num-test (log 2.0e+00-1.0e+00i) 8.0471895621705018730e-1-4.6364760900080611621e-1i)
(num-test (log -2.0e+00+1.0e+00i) 8.0471895621705018730e-1+2.6779450445889871222e0i)
(num-test (log -2.0e+00-1.0e+00i) 8.0471895621705018730e-1-2.6779450445889871222e0i)
(num-test (log 2.0e+00+2.0e+00i) 1.0397207708399179641e0+7.8539816339744830962e-1i)
(num-test (log 2.0e+00-2.0e+00i) 1.0397207708399179641e0-7.8539816339744830962e-1i)
(num-test (log -2.0e+00+2.0e+00i) 1.0397207708399179641e0+2.3561944901923449288e0i)
(num-test (log -2.0e+00-2.0e+00i) 1.0397207708399179641e0-2.3561944901923449288e0i)
(num-test (log 2.0e+00+8.3886080e+06i) 1.5942385152878770538e1+1.5707960883763175177e0i)
(num-test (log 2.0e+00-8.3886080e+06i) 1.5942385152878770538e1-1.5707960883763175177e0i)
(num-test (log -2.0e+00+8.3886080e+06i) 1.5942385152878770538e1+1.5707965652134757208e0i)
(num-test (log -2.0e+00-8.3886080e+06i) 1.5942385152878770538e1-1.5707965652134757208e0i)
(num-test (log 8.3886080e+06+1.19209289550781250e-07i) 1.5942385152878742117e1+1.4210854715202003717e-14i)
(num-test (log 8.3886080e+06-1.19209289550781250e-07i) 1.5942385152878742117e1-1.4210854715202003717e-14i)
(num-test (log -8.3886080e+06+1.19209289550781250e-07i) 1.5942385152878742117e1+3.1415926535897790276e0i)
(num-test (log -8.3886080e+06-1.19209289550781250e-07i) 1.5942385152878742117e1-3.1415926535897790276e0i)
(num-test (log 8.3886080e+06+5.0e-01i) 1.5942385152878743893e1+5.9604644775390554414e-8i)
(num-test (log 8.3886080e+06-5.0e-01i) 1.5942385152878743893e1-5.9604644775390554414e-8i)
(num-test (log -8.3886080e+06+5.0e-01i) 1.5942385152878743893e1+3.1415925939851484631e0i)
(num-test (log -8.3886080e+06-5.0e-01i) 1.5942385152878743893e1-3.1415925939851484631e0i)
(num-test (log 8.3886080e+06+1.0e+00i) 1.5942385152878749222e1+1.1920928955078068531e-7i)
(num-test (log 8.3886080e+06-1.0e+00i) 1.5942385152878749222e1-1.1920928955078068531e-7i)
(num-test (log -8.3886080e+06+1.0e+00i) 1.5942385152878749222e1+3.1415925343805036877e0i)
(num-test (log -8.3886080e+06-1.0e+00i) 1.5942385152878749222e1-3.1415925343805036877e0i)
(num-test (log 8.3886080e+06+2.0e+00i) 1.5942385152878770538e1+2.3841857910155798249e-7i)
(num-test (log 8.3886080e+06-2.0e+00i) 1.5942385152878770538e1-2.3841857910155798249e-7i)
(num-test (log -8.3886080e+06+2.0e+00i) 1.5942385152878770538e1+3.1415924151712141369e0i)
(num-test (log -8.3886080e+06-2.0e+00i) 1.5942385152878770538e1-3.1415924151712141369e0i)
(num-test (log 8.3886080e+06+8.3886080e+06i) 1.6288958743158714771e1+7.8539816339744830962e-1i)
(num-test (log 8.3886080e+06-8.3886080e+06i) 1.6288958743158714771e1-7.8539816339744830962e-1i)
(num-test (log -8.3886080e+06+8.3886080e+06i) 1.6288958743158714771e1+2.3561944901923449288e0i)
(num-test (log -8.3886080e+06-8.3886080e+06i) 1.6288958743158714771e1-2.3561944901923449288e0i)
(num-test (exp (log 8)) 8.0)
(num-test (exp (log 1000)) 1000.0)
(num-test (exp (log 1000000)) 1000000.0)
(num-test (exp (log 1000000000)) 1000000000.0)
(num-test (log (exp 0.1)) 0.1)
(num-test (log (exp 0.0001)) 0.0001)
(num-test (log (exp 0.0000001)) 0.0000001)
(num-test (log (exp 8)) 8.0)
(num-test (log 0+i) (make-rectangular 0.0 (* 0.5 our-pi)))
(num-test (log 0-i) (make-rectangular 0.0 (* -0.5 our-pi)))
(num-test (log 1.0e-8) -18.42068074395237)
(num-test (log 1.0e-12) -27.63102111592855)
(num-test (log (log (log -1))) 0.66457192224882+0.9410294873126i)

(let ((err 0.0))
  (do ((i 0 (+ i 1))
       (x -10.0 (+ x .1)))
      ((= i 200))
    (if (not (zero? x))
	(let ((y (magnitude (- x (exp (log x))))))
	  (if (> y err) (set! err y)))))
  (if (> err 1e-14)
      (begin (display "(exp (log ...)) error: ") (display err) (newline))))

(let ((err 0.0))
  (do ((i 0 (+ i 1))
       (x -10.0+i (+ x 0.1-0.1i)))
      ((= i 200))
    (if (not (zero? x))
	(let ((y (magnitude (- x (exp (log x))))))
	  (if (> y err) (set! err y)))))
  (if (> err 1e-14)
      (begin (display "(exp (log [complex] ...)) error: ") (display err) (newline))))

;; -------- expt
(num-test (expt 0 0) 1)
(num-test (expt 0 1) 0)
(num-test (expt 0 2) 0)
(num-test (expt 0 3) 0)
(num-test (expt 0 10) 0)
(num-test (expt 0 1234) 0)
(num-test (expt 1 0) 1)
(num-test (expt -1 0) 1)
(num-test (expt 1 0) 1)
(num-test (expt 1 1) 1)
(num-test (expt -1 1) -1)
(num-test (expt 1 -1) 1)
(num-test (expt 1 2) 1)
(num-test (expt -1 2) 1)
(num-test (expt 1 -2) 1)
(num-test (expt 1 3) 1)
(num-test (expt -1 3) -1)
(num-test (expt 1 -3) 1)
(num-test (expt 1 10) 1)
(num-test (expt -1 10) 1)
(num-test (expt 1 -10) 1)
(num-test (expt 1 1234) 1)
(num-test (expt -1 1234) 1)
(num-test (expt 1 -1234) 1)
(num-test (expt -0 -0) 1)
(num-test (expt -0.0 0) 0.0)
(num-test (expt 1 -0) 1)

(if with-large-powers 
    (begin
      (num-test (expt 0 1234000000) 0)
      (num-test (expt 0 500029) 0)
      (num-test (expt 0 362880) 0)
      (num-test (expt 0 (expt 2 32)) 0)
      (num-test (expt 0 (expt 2 31)) 0)
      (num-test (expt 0 (expt 2 33)) 0)
      (num-test (expt 0 (expt 2 63)) 0)
      (num-test (expt 0 (expt 2 64)) 0)
      (num-test (expt 0 (expt 2 65)) 0)
      (num-test (expt 0.0 (expt 2 32)) 0.0)
      (num-test (expt 0.0 (expt 2 31)) 0.0)
      (num-test (expt 0.0 (expt 2 33)) 0.0)
      (num-test (expt 0.0 (expt 2 63)) 0.0)
      (num-test (expt 0.0 (expt 2 64)) 0.0)
      (num-test (expt 0.0 (expt 2 65)) 0.0)
      (num-test (expt 1 (expt 2 32)) 1)
      (num-test (expt 1 (- (expt 2 32))) 1)
      (num-test (expt -1 (expt 2 32)) 1)
      (num-test (expt -1 (- (expt 2 32))) 1)
      (num-test (expt -1 (+ 1 (expt 2 32))) -1)
      (num-test (expt 1 1234000000) 1)
      (num-test (expt -1 1234000000) 1)
      (num-test (expt 1 -1234000000) 1)
      (num-test (expt 1 500029) 1)
      (num-test (expt -1 500029) -1)
      (num-test (expt 1 -500029) 1)
      (num-test (expt 1 362880) 1)
      (num-test (expt -1 362880) 1)
      (num-test (expt 1 -362880) 1)
      (num-test (expt 3 10) 59049)
      (num-test (expt -3 10) 59049)
      (num-test (expt 3 -10) 1/59049)
      (num-test (expt 1234 2) 1522756)
      (num-test (expt -1234 2) 1522756)
      (num-test (expt 1234 -2) 1/1522756)
      (num-test (expt 1234 -3) 1/1879080904)
      (num-test (expt 1234000000 -1) 1/1234000000)
      (num-test (expt 500029 0) 1)
      (num-test (expt -500029 0) 1)
      (num-test (expt 500029 0) 1)
      (num-test (expt 500029 1) 500029)
      (num-test (expt -500029 1) -500029)
      (num-test (expt 500029 -1) 1/500029)
      (if (or with-bignums with-64-bit-ints) (num-test (expt 500029 -2) 1/250029000841))
      (num-test (expt 362880 1) 362880)
      (num-test (expt -362880 1) -362880)
      (num-test (expt 362880 -1) 1/362880)
      (if (or with-bignums with-64-bit-ints) (num-test (expt 362880 -2) 1/131681894400))
      (num-test (expt 2/3 1234/362880) 0.99862213634996)
      (num-test (expt -2/3 1234/362880) 0.99856514997103+0.01066829281071i)
      (num-test (expt 2/10 1234000000/500029) 0.0)
      (num-test (expt -2/10 1234000000/500029) 0.0)
      (num-test (expt 2/1234 500029/1234000000) 0.99739996551176)
      (num-test (expt -2/1234 500029/1234000000) 0.99739915734849+0.00126969420446i)
      (num-test (expt 2/1234000000 362880/1234) 0.0)
      (num-test (expt -2/1234000000 362880/1234) 0.0)
      (num-test (expt 2/500029 0/10) 1)
      (num-test (expt -2/500029 0/10) 1)
      (num-test (expt 2/362880 1/3) 0.01766399721581)
      (test (or (< (magnitude (- (expt -2/362880 1/3) 0.00883199860790+0.01529747032127i)) 1e-6)
		(< (magnitude (- (expt -2/362880 1/3) 0.00883199860790-0.01529747032127i)) 1e-6)
		(< (magnitude (- (expt -2/362880 1/3) -0.017663997215805)) 1e-6)) #t)
      (num-test (expt 3/1 10/2) 243)
      (num-test (expt -3/1 10/2) -243)
      (num-test (expt 3/10 500029/362880) 0.19032743228093)
      (num-test (expt -3/10 500029/362880) -0.07120470796705-0.17650615015342i)
      (num-test (expt 3/1234 362880/500029) 0.01267163536282)
      (num-test (expt -3/1234 362880/500029) -0.00825127977478+0.00961700186370i)
      (num-test (expt 3/1234000000 0/1234000000) 1)
      (num-test (expt -3/1234000000 0/1234000000) 1)
      (num-test (expt 3/500029 1/1234) 0.99030354920325)
      (num-test (expt -3/500029 1/1234) 0.99030033992383+0.00252117260474i)
      (num-test (expt 3/362880 2/10) 0.09626571578282)
      (num-test (expt -3/362880 2/10) 0.07788060004397+0.05658356803852i)
      (num-test (expt 10/1234 0/362880) 1)
      (num-test (expt -10/1234 0/362880) 1)
      (num-test (expt 10/1234000000 1/500029) 0.99996274097186)
      (num-test (expt -10/1234000000 1/500029) 0.99996274095212+0.00000628258681i)
      (num-test (expt 10/500029 2/1234000000) 0.99999998246380)
      (num-test (expt -10/500029 2/1234000000) 0.99999998246380+0.00000000509172i)
      (num-test (expt 10/362880 3/1234) 0.97479810574733)
      (num-test (expt -10/362880 3/1234) 0.97476967459219+0.00744502948457i)
      (num-test (expt 1234/10 0/1) 1)
      (num-test (expt -1234/10 0/1) 1)
      (num-test (expt 1234/1234000000 2/362880) 0.99992385921192)
      (num-test (expt -1234/1234000000 2/362880) 0.99992385906203+0.00001731345596i)
      (num-test (expt 1234/500029 3/500029) 0.99996397630725)
      (num-test (expt -1234/500029 3/500029) 0.99996397612963+0.00001884778372i)
      (num-test (expt 1234/362880 10/1234000000) 0.99999995393994)
      (num-test (expt -1234/362880 10/1234000000) 0.99999995393994+0.00000002545861i)
      (num-test (expt 1234000000/3 0/3) 1)
      (num-test (expt -1234000000/3 0/3) 1)
      (num-test (expt 1234000000/10 1/2) 11108.55526159905094)
      (num-test (expt -1234000000/10 1/2) 0.00000000000068+11108.55526159905094i)
      (num-test (expt 1234000000/500029 10/362880) 1.00021527629325)
      (num-test (expt -1234000000/500029 10/362880) 1.00021527254493+0.00008659250882i)
      (num-test (expt 1234000000/362880 1234/500029) 1.02027058333165)
      (num-test (expt -1234000000/362880 1234/500029) 1.02023991975054+0.00791007960266i)
      (num-test (expt 500029/1 362880/1234000000) 1.00386634022855)
      (num-test (expt -500029/1 362880/1234000000) 1.00386591183654+0.00092741384842i)
      (num-test (expt 500029/2 0/1234) 1)
      (num-test (expt -500029/2 0/1234) 1)
      (num-test (expt 500029/3 1/10) 3.32803123591083)
      (num-test (expt -500029/3 1/10) 3.16514579334680+1.02841820970710i)
      (num-test (expt 500029/10 2/3) 1357.26128653075921)
      (num-test (expt -500029/10 2/3) -678.63064326537926+1175.42275370878747i)
      (num-test (expt 500029/1234 3/2) 8156.80442672750178)
      (num-test (expt -500029/1234 3/2) -0.00000000000150-8156.80442672750178i)
      (num-test (expt 500029/1234000000 10/1) 0.0)
      (num-test (expt -500029/1234000000 10/1) 0.0-0.0i)
      (num-test (expt 362880/1 0/500029) 1)
      (num-test (expt -362880/1 0/500029) 1)
      (num-test (expt 362880/2 1/1234000000) 1.00000000981254)
      (num-test (expt -362880/2 1/1234000000) 1.00000000981254+0.00000000254586i)
      (num-test (expt 362880/3 2/1234) 1.01914896791961)
      (num-test (expt -362880/3 2/1234) 1.01913575690562+0.00518920109559i)
      (num-test (expt 362880/10 3/10) 23.33076127248722)
      (num-test (expt -362880/10 3/10) 13.71347740072436+18.87498236114703i)
      (num-test (expt 362880/1234 10/3) 169106218.17807236313820)
      (num-test (expt -362880/1234 10/3) -84553109.08903615176678-146450280.88012450933456i)
      (num-test (expt 362880/1234000000 1234/2) 0.0)
      (num-test (expt -362880/1234000000 1234/2) -0.0)))

(num-test (expt 2 0) 1)
(num-test (expt -2 0) 1)
(num-test (expt 2 0) 1)
(num-test (expt 2 1) 2)
(num-test (expt -2 1) -2)
(num-test (expt 2 -1) 1/2)
(num-test (expt 2 2) 4)
(num-test (expt -2 2) 4)
(num-test (expt 2 -2) 1/4)
(num-test (expt 2 3) 8)
(num-test (expt -2 3) -8)
(num-test (expt 2 -3) 1/8)
(num-test (expt 2 10) 1024)
(num-test (expt -2 10) 1024)
(num-test (expt 2 -10) 1/1024)
(num-test (expt 3 0) 1)
(num-test (expt -3 0) 1)
(num-test (expt 3 0) 1)
(num-test (expt 3 1) 3)
(num-test (expt -3 1) -3)
(num-test (expt 3 -1) 1/3)
(num-test (expt 3 2) 9)
(num-test (expt -3 2) 9)
(num-test (expt 3 -2) 1/9)
(num-test (expt 3 3) 27)
(num-test (expt -3 3) -27)
(num-test (expt 3 -3) 1/27)
(num-test (expt 2.0 -1.0220000e+03) 2.225073858507201383090232717332404063624E-308)
(num-test (expt 2.0 1.0230000e+03) 8.98846567431157953864652595394512365232E307)
(num-test (expt 0+i 1/2) (make-rectangular (/ (sqrt 2)) (/ (sqrt 2))))
(num-test (expt (expt 0+i 1/3) 3) 0+i)
(num-test (expt (expt 0+i 1/4) 4) 0+i)
(num-test (expt (expt 1+i 1/2) 2) 1+i)
(num-test (expt (expt 1+i 1/3) 3) 1+i)
(num-test (expt (expt 1+i 1/4) 4) 1+i)
(num-test (expt (expt 1-i 1/2) 2) 1-i)
(num-test (expt (expt 1-i 1/3) 3) 1-i)
(num-test (expt (expt 1-i 1/4) 4) 1-i)
(num-test (expt (expt 0+i 1/10) 10) 0+i)
(num-test (expt (expt 0+i -1/2) -2) 0+i)
(num-test (expt (expt 0+i -1/3) -3) 0+i)
(num-test (expt (expt 0+i -1/4) -4) 0+i)
(num-test (expt 0+i 2) -1)
(num-test (expt 0+i 0+i) 2.078795763507619085469556198349787700342E-1)
(num-test (expt 0-i 2) -1)
(num-test (expt (expt 2 0+i) (/ 0+i)) 2)
(num-test (expt -1 1/2) 0+i)
(num-test (expt -1.0 1/2) 0+i)
(num-test (expt (expt -1/2 1/2) 2) -1/2)
(num-test (expt (expt -1/2 -1/2) -2) -1/2)
(num-test (expt (expt 1/2 -1/2) -2) 1/2)
(num-test (expt (expt -1/3 1/3) 3) -1/3)
(num-test (expt (expt -1/3 1/2) 2) -1/3)
(num-test (expt (expt -1/3 -1/2) -2) -1/3)
(num-test (expt (expt -2 1/2) 2) -2)
(num-test (expt (expt -2 -1/2) -2) -2)
(num-test (expt (expt -1 -1/2) -2) -1)
(num-test (expt (expt 1 -1/2) -2) 1)
(num-test (expt (expt 1 1/123) 123) 1)
(num-test (expt (expt -1 1/123) 123) -1)
(num-test (expt -1/8 -3) -512)

(test (= (make-rectangular 1.0 0.0) (make-polar 1.0 0.0) 1.0+0i (* -1.0 -1.0) (/ 1.0) 
	 (exp 0.0) (abs -1.0) (cos 0.0) (log (exp 1)) (magnitude 1.0+0i) (max 0.0 1.0) (min 1.0 2.0) )
      #t)

(let ((xs (list 2 3 4 1/2 1/3 1/4 2.5 1+i 2.5+1.5i 2.5-.5i))
      (ys  (list 2 3 4 -2 -3 1/2 1/3 1/4 -1/2 -1/3 -1/4 2.5 -3.5 1+i -1+2i 2.5+1.5i 2.5-.5i)))
  (for-each
   (lambda (x)
     (for-each 
      (lambda (y)
	(num-test (expt (expt x y) (/ y)) x)
	;;	(if (> (magnitude (- (expt (expt x y) (/ y)) x)) 1e-6)
	;;	    (format #t "(expt (expt ~A ~A) (/ ~A)) -> ~A (~A)~%" x y y (expt (expt x y) (/ y)) (magnitude (- (expt (expt x y) (/ y)) x))))
	)
      ys))
   xs))

(if with-bignums
    (begin
      (num-test (expt 4722366482869645213696 1/2) 68719476736)
      (num-test (expt 324518553658426726783156020576256 1/3) 68719476736)
      (num-test (expt 4722366482869645213696/6561 1/2) 68719476736/81)
      (num-test (expt 324518553658426726783156020576256/19683 1/3) 68719476736/27)

      (num-test (expt 4722366482869645213696/6561 -1/2) (/ 68719476736/81))
      (num-test (expt (expt -4722366482869645213696/6561 1/2) 2) -4722366482869645213696/6561)

      (num-test (numerator 1195068768795265792518361315725116351898245581/48889032896862784894921) 24444516448431392447461)
      (num-test (denominator 1195068768795265792518361315725116351898245581/48889032896862784894921) 1)
      (num-test (numerator 24444516448431392447461/1195068768795265792518361315725116351898245581) 1)
      (num-test (denominator 24444516448431392447461/1195068768795265792518361315725116351898245581) 48889032896862784894921)

      (if with-bigfloats
	  (begin
	    (num-test (expt 4722366482869645213696.0 1/2) 68719476736.0)
	    (num-test (expt 324518553658426726783156020576256.0 1/3) 68719476736.0)
	    (num-test (+ 12345678901234567890+12345678901234567890i 12345678901234567890-12345678901234567890i) 2.469135780246913578E19)
	    (num-test (+ 1.0 12345678901234567890) 1.2345678901234567891E19)
	    (num-test (* 2 12345678901234567890+12345678901234567890i) 2.469135780246913578E19+2.469135780246913578E19i)
	    (num-test (- 2 12345678901234567890+12345678901234567890i) -12345678901234567890-12345678901234567890i)
	    ))
      ))

(num-test (expt 10 0) 1)
(num-test (expt -10 0) 1)
(num-test (expt 10 0) 1)
(num-test (expt 10 1) 10)
(num-test (expt -10 1) -10)
(num-test (expt 10 -1) 1/10)
(num-test (expt 10 2) 100)
(num-test (expt -10 2) 100)
(num-test (expt 10 -2) 1/100)
(num-test (expt 10 3) 1000)
(num-test (expt -10 3) -1000)
(num-test (expt 10 -3) 1/1000)
(if (or with-bignums with-64-bit-ints) (num-test (expt 10 -10) 1/10000000000))
(num-test (expt 1234 0) 1)
(num-test (expt -1234 0) 1)
(num-test (expt 1234 0) 1)
(num-test (expt 1234 1) 1234)
(num-test (expt -1234 1) -1234)
(num-test (expt 1234 -1) 1/1234)
(num-test (expt 1234000000 0) 1)
(num-test (expt -1234000000 0) 1)
(num-test (expt 1234000000 0) 1)
(num-test (expt 0/500029 500029/2) 0.0)
(num-test (expt 0/362880 362880/1) 0)
(num-test (expt 1/2 3/362880) 0.99999426963298)
(num-test (expt -1/2 3/362880) 0.99999426929571+0.00002597201266i)
(num-test (expt 1/3 10/500029) 0.99997802926990)
(num-test (expt -1/3 10/500029) 0.99997802729625+0.00006282682861i)
(num-test (expt 1/10 1234/1234000000) 0.99999769741756)
(num-test (expt -1/10 1234/1234000000) 0.99999769741262+0.00000314158542i)
(num-test (expt 1/1234000000 500029/10) 0.0)
(num-test (expt -1/1234000000 500029/10) -0.0)
(num-test (expt 1/500029 362880/3) 0.0)
(num-test (expt -1/500029 362880/3) 0.0)
(num-test (expt 362880 0) 1)
(num-test (expt -362880 0) 1)
(num-test (expt 362880 0) 1)
(num-test (expt 0/1 1/362880) 0.0)
(num-test (expt 0/2 2/500029) 0.0)
(num-test (expt 0/3 3/1234000000) 0.0)
(num-test (expt 0/10 10/1234) 0.0)
(num-test (expt 0/1234 1234/10) 0.0)
(num-test (expt 1/362880 0/2) 1)
(num-test (expt -1/362880 0/2) 1)
(num-test (expt 2/1 3/1) 8)
(num-test (expt 1/4 -2) 16)
(num-test (expt 16 1/4) 2)
(num-test (expt -2/1 3/1) -8)
(num-test (expt 0 0.00000001) 0.0)
(num-test (expt 0 1.0) 0.0)
(num-test (expt 0 our-pi) 0.0)
(num-test (expt 0 2.71828182845905) 0.0)
(num-test (expt 0 1234.0) 0.0)
(num-test (expt 0 1234000000.0) 0.0)
(num-test (expt 1 0.0) 1.0)
(num-test (expt -1 0.0) 1.0)
(num-test (expt 1 -0.0) 1.0)
(num-test (expt 1 0.00000001) 1.0)
(num-test (expt -1 0.00000001) 1.0+0.00000003141593i)
(num-test (expt 1 -0.00000001) 1.0)
(num-test (expt 1 1.0) 1.0)
(num-test (expt -1 1.0) -1.0+0.0i)
(num-test (expt 1 -1.0) 1.0)
(num-test (expt 1 our-pi) 1.0)
(num-test (expt -1 our-pi) -0.90268536193307-0.43030121700009i)
(num-test (expt 1 -3.14159265358979) 1.0)
(num-test (expt 1 2.71828182845905) 1.0)
(num-test (expt -1 2.71828182845905) -0.63325565131482+0.77394268526671i)
(num-test (expt 1 -2.71828182845905) 1.0)
(num-test (expt 1 1234.0) 1.0)
(num-test (expt -1 1234.0) 1.0)
(num-test (expt 1 -1234.0) 1.0)
(num-test (expt 1 1234000000.0) 1.0)
(num-test (expt -1 1234000000.0) 1.0)
(num-test (expt 1 -1234000000.0) 1.0)
(num-test (expt 2 0.0) 1.0)
(num-test (expt -2 0.0) 1.0)
(num-test (expt 2 -0.0) 1.0)
(num-test (expt 2 0.00000001) 1.00000000693147)
(num-test (expt -2 0.00000001) 1.00000000693147+0.00000003141593i)
(num-test (expt 2 -0.00000001) 0.99999999306853)
(num-test (expt 2 1.0) 2.0)
(num-test (expt -2 1.0) -2.0+0.0i)
(num-test (expt 2 -1.0) 0.50000000000000)
(num-test (expt 2 our-pi) 8.82497782707629)
(num-test (expt -2 our-pi) -7.96617830388569-3.79739869898975i)
(num-test (expt 2 -3.14159265358979) 0.11331473229676)
(num-test (expt 2 2.71828182845905) 6.58088599101792)
(num-test (expt -2 2.71828182845905) -4.16738324447062+5.09322857532248i)
(num-test (expt 2 -2.71828182845905) 0.15195522325791)
(num-test (expt 3 0.0) 1.0)
(num-test (expt -3 0.0) 1.0)
(num-test (expt 3 -0.0) 1.0)
(num-test (expt 3 0.00000001) 1.00000001098612)
(num-test (expt -3 0.00000001) 1.00000001098612+0.00000003141593i)
(num-test (expt 3 -0.00000001) 0.99999998901388)
(num-test (expt 3 1.0) 3.0)
(num-test (expt -3 1.0) -3.0+0.0i)
(num-test (expt 3 -1.0) 0.33333333333333)
(num-test (expt 3 our-pi) 31.54428070019755)
(num-test (expt -3 our-pi) -28.47456044077623-13.57354237468751i)
(num-test (expt 3 -3.14159265358979) 0.03170146783514)
(num-test (expt 3 2.71828182845905) 19.81299074527465)
(num-test (expt -3 2.71828182845905) -12.54668835889338+15.33411926056231i)
(num-test (expt 3 -2.71828182845905) 0.05047193595639)
(num-test (expt 10 0.0) 1.0)
(num-test (expt -10 0.0) 1.0)
(num-test (expt 10 -0.0) 1.0)
(num-test (expt 10 0.00000001) 1.00000002302585)
(num-test (expt -10 0.00000001) 1.00000002302585+0.00000003141593i)
(num-test (expt 10 -0.00000001) 0.99999997697415)
(num-test (expt 10 1.0) 10.0)
(num-test (expt -10 1.0) -10.0+0.0i)
(num-test (expt 10 -1.0) 0.1)
(num-test (expt 10 our-pi) 1385.45573136701182)
(num-test (expt -10 our-pi) -1250.63060831127905-596.16328730697728i)
(num-test (expt 10 -3.14159265358979) 0.00072178415907)
(num-test (expt 10 2.71828182845905) 522.73529967043669)
(num-test (expt -10 2.71828182845905) -331.02508265804954+404.56716151063557i)
(num-test (expt 10 -2.71828182845905) 0.00191301410222)
(num-test (expt 1234 0.0) 1.0)
(num-test (expt -1234 0.0) 1.0)
(num-test (expt 1234 -0.0) 1.0)
(num-test (expt 1234 0.00000001) 1.00000007118016)
(num-test (expt -1234 0.00000001) 1.00000007118016+0.00000003141593i)
(num-test (expt 1234 -0.00000001) 0.99999992881984)
(num-test (expt 1234 1.0) 1234.00000000000045)
(num-test (expt -1234 1.0) -1234.00000000000045+0.00000000000015i)
(num-test (expt 1234 -1.0) 0.00081037277147)
(num-test (expt 1234 -3.14159265358979) 0.00000000019424)
(num-test (expt 1234 2.71828182845905) 252968138.32201290130615)
(num-test (expt -1234 2.71828182845905) -160193503.19500353932381+195782840.25985893607140i)
(num-test (expt 1234 -2.71828182845905) 0.00000000395307)
(num-test (expt 1234000000 0.0) 1.0)
(num-test (expt -1234000000 0.0) 1.0)
(num-test (expt 1234000000 -0.0) 1.0)
(num-test (expt 1234000000 0.00000001) 1.00000020933529)
(num-test (expt -1234000000 0.00000001) 1.00000020933529+0.00000003141593i)
(num-test (expt 1234000000 -0.00000001) 0.99999979066475)
(num-test (expt 1234000000 -1.0) 0.00000000081037)
(num-test (expt 500029 0.0) 1.0)
(num-test (expt -500029 0.0) 1.0)
(num-test (expt 500029 -0.0) 1.0)
(num-test (expt 500029 0.00000001) 1.00000013122422)
(num-test (expt -500029 0.00000001) 1.00000013122422+0.00000003141593i)
(num-test (expt 500029 -0.00000001) 0.99999986877579)
(num-test (expt 500029 1.0) 500029.00000000040745)
(num-test (expt -500029 1.0) -500029.00000000040745+0.00000000006123i)
(num-test (expt 500029 -1.0) 0.00000199988401)
(num-test (expt 362880 0.0) 1.0)
(num-test (expt -362880 0.0) 1.0)
(num-test (expt 362880 -0.0) 1.0)
(num-test (expt 362880 0.00000001) 1.00000012801828)
(num-test (expt -362880 0.00000001) 1.00000012801828+0.00000003141593i)
(num-test (expt 362880 -0.00000001) 0.99999987198173)
(num-test (expt 362880 1.0) 362879.99999999982538)
(num-test (expt -362880 1.0) -362879.99999999982538+0.00000000004444i)
(num-test (expt 362880 -1.0) 0.00000275573192)
(num-test (expt 0.0 0.00000001) 0.0)
(num-test (expt -0.0 0.00000001) 0.0)
(num-test (expt 0.0 1.0) 0.0)
(num-test (expt -0.0 1.0) 0.0)
(num-test (expt 0.0 our-pi) 0.0)
(num-test (expt -0.0 our-pi) 0.0)
(num-test (expt 0.0 2.71828182845905) 0.0)
(num-test (expt -0.0 2.71828182845905) 0.0)
(num-test (expt 0.0 1234.0) 0.0)
(num-test (expt -0.0 1234.0) 0.0)
(num-test (expt 0.0 1234000000.0) 0.0)
(num-test (expt -0.0 1234000000.0) 0.0)
(num-test (expt 0.00000001 0.0) 1.0)
(num-test (expt -0.00000001 0.0) 1.0)
(num-test (expt 0.00000001 -0.0) 1.0)
(num-test (expt -0.00000001 -0.0) 1.0)
(num-test (expt 0.00000001 0.00000001) 0.99999981579321)
(num-test (expt -0.00000001 0.00000001) 0.99999981579321+0.00000003141592i)
(num-test (expt 0.00000001 -0.00000001) 1.00000018420682)
(num-test (expt -0.00000001 -0.00000001) 1.00000018420682-0.00000003141593i)
(num-test (expt 0.00000001 1.0) 0.00000001)
(num-test (expt -0.00000001 1.0) -0.00000001+0.0i)
(num-test (expt 0.00000001 -1.0) 100000000.00000017881393)
(num-test (expt -0.00000001 -1.0) -100000000.00000017881393-0.00000001224606i)
(num-test (expt 0.00000001 our-pi) 0.0)
(num-test (expt -0.00000001 our-pi) -0.0-0.0i)
(num-test (expt 0.00000001 2.71828182845905) 0.0)
(num-test (expt -0.00000001 2.71828182845905) -0.0+0.0i)
(num-test (expt 0.00000001 1234.0) 0.0)
(num-test (expt -0.00000001 1234.0) 0.0)
(num-test (expt 0.00000001 1234000000.0) 0.0)
(num-test (expt -0.00000001 1234000000.0) 0.0)
(num-test (expt 1.0 0.0) 1.0)
(num-test (expt -1.0 0.0) 1.0)
(num-test (expt 1.0 -0.0) 1.0)
(num-test (expt -1.0 -0.0) 1.0)
(num-test (expt 1.0 0.00000001) 1.0)
(num-test (expt -1.0 0.00000001) 1.0+0.00000003141593i)
(num-test (expt 1.0 -0.00000001) 1.0)
(num-test (expt -1.0 -0.00000001) 1.0-0.00000003141593i)
(num-test (expt 1.0 1.0) 1.0)
(num-test (expt -1.0 1.0) -1.0+0.0i)
(num-test (expt 1.0 -1.0) 1.0)
(num-test (expt -1.0 -1.0) -1.0-0.0i)
(num-test (expt 1.0 our-pi) 1.0)
(num-test (expt -1.0 our-pi) -0.90268536193307-0.43030121700009i)
(num-test (expt 1.0 -3.14159265358979) 1.0)
(num-test (expt -1.0 -3.14159265358979) -0.90268536193307+0.43030121700009i)
(num-test (expt 1.0 2.71828182845905) 1.0)
(num-test (expt -1.0 2.71828182845905) -0.63325565131482+0.77394268526671i)
(num-test (expt 1.0 -2.71828182845905) 1.0)
(num-test (expt -1.0 -2.71828182845905) -0.63325565131482-0.77394268526671i)
(num-test (expt 1.0 1234.0) 1.0)
(num-test (expt -1.0 1234.0) 1.0)
(num-test (expt 1.0 -1234.0) 1.0)
(num-test (expt -1.0 -1234.0) 1.0)
(num-test (expt 1.0 1234000000.0) 1.0)
(num-test (expt -1.0 1234000000.0) 1.0)
(num-test (expt 1.0 -1234000000.0) 1.0)
(num-test (expt -1.0 -1234000000.0) 1.0)
(num-test (expt 3.14159265358979 0.0) 1.0)
(num-test (expt -3.14159265358979 0.0) 1.0)
(num-test (expt 3.14159265358979 -0.0) 1.0)
(num-test (expt -3.14159265358979 -0.0) 1.0)
(num-test (expt 3.14159265358979 0.00000001) 1.00000001144730)
(num-test (expt -3.14159265358979 0.00000001) 1.00000001144730+0.00000003141593i)
(num-test (expt 3.14159265358979 -0.00000001) 0.99999998855270)
(num-test (expt -3.14159265358979 -0.00000001) 0.99999998855270-0.00000003141593i)
(num-test (expt 3.14159265358979 1.0) our-pi)
(num-test (expt -3.14159265358979 1.0) -3.14159265358979+0.0i)
(num-test (expt 3.14159265358979 -1.0) 0.31830988618379)
(num-test (expt -3.14159265358979 -1.0) -0.31830988618379-0.0i)
(num-test (expt 3.14159265358979 our-pi) 36.46215960720790)
(num-test (expt -3.14159265358979 our-pi) -32.91385774189388-15.68971165343314i)
(num-test (expt 3.14159265358979 -3.14159265358979) 0.02742569312330)
(num-test (expt -3.14159265358979 -3.14159265358979) -0.02475677172327+0.01180130912803i)
(num-test (expt 3.14159265358979 2.71828182845905) 22.45915771836104)
(num-test (expt -3.14159265358979 2.71828182845905) -14.22238854892297+17.38210083337688i)
(num-test (expt 3.14159265358979 -2.71828182845905) 0.04452526726692)
(num-test (expt -3.14159265358979 -2.71828182845905) -0.02819587712308-0.03446000491078i)
(num-test (expt 2.71828182845905 0.0) 1.0)
(num-test (expt -2.71828182845905 0.0) 1.0)
(num-test (expt 2.71828182845905 -0.0) 1.0)
(num-test (expt -2.71828182845905 -0.0) 1.0)
(num-test (expt 2.71828182845905 0.00000001) 1.00000001)
(num-test (expt -2.71828182845905 0.00000001) 1.00000001+0.00000003141593i)
(num-test (expt 2.71828182845905 -0.00000001) 0.99999999000000)
(num-test (expt -2.71828182845905 -0.00000001) 0.99999999000000-0.00000003141593i)
(num-test (expt 2.71828182845905 1.0) 2.71828182845905)
(num-test (expt -2.71828182845905 1.0) -2.71828182845905+0.0i)
(num-test (expt 2.71828182845905 -1.0) 0.36787944117144)
(num-test (expt -2.71828182845905 -1.0) -0.36787944117144-0.0i)
(num-test (expt 2.71828182845905 our-pi) 23.14069263277927)
(num-test (expt -2.71828182845905 our-pi) -20.88876450460231-9.95746820210997i)
(num-test (expt 2.71828182845905 -3.14159265358979) 0.04321391826377)
(num-test (expt -2.71828182845905 -3.14159265358979) -0.03900857144848+0.01859500162024i)
(num-test (expt 2.71828182845905 2.71828182845905) 15.15426224147926)
(num-test (expt -2.71828182845905 2.71828182845905) -9.59652220592352+11.72853041240636i)
(num-test (expt 2.71828182845905 -2.71828182845905) 0.06598803584531)
(num-test (expt -2.71828182845905 -2.71828182845905) -0.04178729661821-0.05107095765760i)
(num-test (expt 1234.0 0.0) 1.0)
(num-test (expt -1234.0 0.0) 1.0)
(num-test (expt 1234.0 -0.0) 1.0)
(num-test (expt -1234.0 -0.0) 1.0)
(num-test (expt 1234.0 0.00000001) 1.00000007118016)
(num-test (expt -1234.0 0.00000001) 1.00000007118016+0.00000003141593i)
(num-test (expt 1234.0 -0.00000001) 0.99999992881984)
(num-test (expt -1234.0 -0.00000001) 0.99999992881984-0.00000003141592i)
(num-test (expt 1234.0 1.0) 1234.00000000000045)
(num-test (expt -1234.0 1.0) -1234.00000000000045+0.00000000000015i)
(num-test (expt 1234.0 -1.0) 0.00081037277147)
(num-test (expt -1234.0 -1.0) -0.00081037277147-0.0i)
(num-test (expt 1234.0 -3.14159265358979) 0.00000000019424)
(num-test (expt -1234.0 -3.14159265358979) -0.00000000017534+0.00000000008358i)
(num-test (expt 1234.0 2.71828182845905) 252968138.32201290130615)
(num-test (expt -1234.0 2.71828182845905) -160193503.19500353932381+195782840.25985893607140i)
(num-test (expt 1234.0 -2.71828182845905) 0.00000000395307)
(num-test (expt -1234.0 -2.71828182845905) -0.00000000250330-0.00000000305945i)
(num-test (expt 1234000000.0 0.0) 1.0)
(num-test (expt -1234000000.0 0.0) 1.0)
(num-test (expt 1234000000.0 -0.0) 1.0)
(num-test (expt -1234000000.0 -0.0) 1.0)
(num-test (expt 1234000000.0 0.00000001) 1.00000020933529)
(num-test (expt -1234000000.0 0.00000001) 1.00000020933529+0.00000003141593i)
(num-test (expt 1234000000.0 -0.00000001) 0.99999979066475)
(num-test (expt -1234000000.0 -0.00000001) 0.99999979066475-0.00000003141592i)
(num-test (expt 1234000000.0 -1.0) 0.00000000081037)
(num-test (expt -1234000000.0 -1.0) -0.00000000081037-0.0i)
(num-test (expt 0.00000001 0) 1.0)
(num-test (expt -0.00000001 0) 1.0)
(num-test (expt 0.00000001 0) 1.0)
(num-test (expt -0.00000001 0) 1.0)
(num-test (expt 0.00000001 1) 0.00000001)
(num-test (expt -0.00000001 1) -0.00000001+0.0i)
(num-test (expt 0.00000001 -1) 100000000.00000017881393)
(num-test (expt -0.00000001 -1) -100000000.00000017881393-0.00000001224606i)
(num-test (expt 0.00000001 2) 0.0)
(num-test (expt -0.00000001 2) 0.0-0.0i)
(num-test (expt 0.00000001 3) 0.0)
(num-test (expt -0.00000001 3) -0.0+0.0i)
(num-test (expt 0.00000001 10) 0.0)
(num-test (expt -0.00000001 10) 0.0-0.0i)
(num-test (expt 0.00000001 1234) 0.0)
(num-test (expt -0.00000001 1234) 0.0)
(num-test (expt 0.00000001 1234000000) 0.0)
(num-test (expt -0.00000001 1234000000) 0.0)
(num-test (expt 0.00000001 500029) 0.0)
(num-test (expt -0.00000001 500029) -0.0)
(num-test (expt 0.00000001 362880) 0.0)
(num-test (expt -0.00000001 362880) 0.0)
(num-test (expt 1.0 0) 1.0)
(num-test (expt -1.0 0) 1.0)
(num-test (expt 1.0 0) 1.0)
(num-test (expt -1.0 0) 1.0)
(num-test (expt 1.0 1) 1.0)
(num-test (expt -1.0 1) -1.0+0.0i)
(num-test (expt 1.0 -1) 1.0)
(num-test (expt -1.0 -1) -1.0-0.0i)
(num-test (expt 1.0 2) 1.0)
(num-test (expt -1.0 2) 1.0-0.0i)
(num-test (expt 1.0 -2) 1.0)
(num-test (expt -1.0 -2) 1.0+0.0i)
(num-test (expt 1.0 3) 1.0)
(num-test (expt -1.0 3) -1.0+0.0i)
(num-test (expt 1.0 -3) 1.0)
(num-test (expt -1.0 -3) -1.0-0.0i)
(num-test (expt 1.0 10) 1.0)
(num-test (expt -1.0 10) 1.0-0.0i)
(num-test (expt 1.0 -10) 1.0)
(num-test (expt -1.0 -10) 1.0+0.0i)
(num-test (expt 1.0 1234) 1.0)
(num-test (expt -1.0 1234) 1.0)
(num-test (expt 1.0 -1234) 1.0)
(num-test (expt -1.0 -1234) 1.0)
(num-test (expt 1.0 1234000000) 1.0)
(num-test (expt -1.0 1234000000) 1.0)
(num-test (expt 1.0 -1234000000) 1.0)
(num-test (expt -1.0 -1234000000) 1.0)
(num-test (expt 1.0 500029) 1.0)
(num-test (expt -1.0 500029) -1.0)
(num-test (expt 1.0 -500029) 1.0)
(num-test (expt -1.0 -500029) -1.0)
(num-test (expt 1.0 362880) 1.0)
(num-test (expt -1.0 362880) 1.0)
(num-test (expt 1.0 -362880) 1.0)
(num-test (expt -1.0 -362880) 1.0)
(num-test (expt 3.14159265358979 0) 1.0)
(num-test (expt -3.14159265358979 0) 1.0)
(num-test (expt 3.14159265358979 0) 1.0)
(num-test (expt -3.14159265358979 0) 1.0)
(num-test (expt 3.14159265358979 1) our-pi)
(num-test (expt -3.14159265358979 1) -3.14159265358979+0.0i)
(num-test (expt 3.14159265358979 -1) 0.31830988618379)
(num-test (expt -3.14159265358979 -1) -0.31830988618379-0.0i)
(num-test (expt 3.14159265358979 2) 9.86960440108936)
(num-test (expt -3.14159265358979 2) 9.86960440108936-0.0i)
(num-test (expt 3.14159265358979 -2) 0.10132118364234)
(num-test (expt -3.14159265358979 -2) 0.10132118364234+0.0i)
(num-test (expt 3.14159265358979 3) 31.00627668029983)
(num-test (expt -3.14159265358979 3) -31.00627668029983)
(num-test (expt 3.14159265358979 -3) 0.03225153443320)
(num-test (expt -3.14159265358979 -3) -0.03225153443320-0.0i)
(num-test (expt 3.14159265358979 10) 93648.04747608296748)
(num-test (expt -3.14159265358979 10) 93648.04747608296748)
(num-test (expt 3.14159265358979 -10) 0.00001067827923)
(num-test (expt -3.14159265358979 -10) 0.00001067827923+0.0i)
(num-test (expt 2.71828182845905 0) 1.0)
(num-test (expt -2.71828182845905 0) 1.0)
(num-test (expt 2.71828182845905 0) 1.0)
(num-test (expt -2.71828182845905 0) 1.0)
(num-test (expt 2.71828182845905 1) 2.71828182845905)
(num-test (expt -2.71828182845905 1) -2.71828182845905+0.0i)
(num-test (expt 2.71828182845905 -1) 0.36787944117144)
(num-test (expt -2.71828182845905 -1) -0.36787944117144-0.0i)
(num-test (expt 2.71828182845905 2) 7.38905609893065)
(num-test (expt -2.71828182845905 2) 7.38905609893065-0.0i)
(num-test (expt 2.71828182845905 -2) 0.13533528323661)
(num-test (expt -2.71828182845905 -2) 0.13533528323661+0.0i)
(num-test (expt 2.71828182845905 3) 20.08553692318767)
(num-test (expt -2.71828182845905 3) -20.08553692318767+0.00000000000001i)
(num-test (expt 2.71828182845905 -3) 0.04978706836786)
(num-test (expt -2.71828182845905 -3) -0.04978706836786-0.0i)
(num-test (expt 2.71828182845905 10) 22026.46579480671789)
(num-test (expt -2.71828182845905 10) 22026.46579480671789-0.00000000002697i)
(num-test (expt 2.71828182845905 -10) 0.00004539992976)
(num-test (expt -2.71828182845905 -10) 0.00004539992976+0.0i)
(num-test (expt 1234.0 0) 1.0)
(num-test (expt -1234.0 0) 1.0)
(num-test (expt 1234.0 0) 1.0)
(num-test (expt -1234.0 0) 1.0)
(num-test (expt 1234.0 1) 1234.00000000000045)
(num-test (expt -1234.0 1) -1234.00000000000045+0.00000000000015i)
(num-test (expt 1234.0 -1) 0.00081037277147)
(num-test (expt -1234.0 -1) -0.00081037277147-0.0i)
(num-test (expt 1234.0 2) 1522756.00000000093132)
(num-test (expt -1234.0 2) 1522756.00000000093132-0.00000000037296i)
(num-test (expt 1234.0 -2) 0.00000065670403)
(num-test (expt -1234.0 -2) 0.00000065670403+0.0i)
(num-test (expt 1234.0 -3) 0.00000000053218)
(num-test (expt -1234.0 -3) -0.00000000053218-0.0i)
(num-test (expt 1234000000.0 0) 1.0)
(num-test (expt -1234000000.0 0) 1.0)
(num-test (expt 1234000000.0 0) 1.0)
(num-test (expt -1234000000.0 0) 1.0)
(num-test (expt 1234000000.0 -1) 0.00000000081037)
(num-test (expt -1234000000.0 -1) -0.00000000081037-0.0i)
(num-test (expt 0.0-1234000000.0i -1234000000.0+0.00000001i) 0.0)
(num-test (expt -0.0-1234000000.0i -1234000000.0-0.00000001i) 0.0)
(num-test (expt 0.00000001+0.0i 0.00000001+0.00000001i) 0.99999981579319-0.00000018420677i)
(num-test (expt -0.00000001+0.0i 0.00000001-0.00000001i) 0.99999984720911+0.00000021562270i)
(num-test (expt 0.00000001-0.0i -0.00000001+0.00000001i) 1.00000018420681-0.00000018420684i)
(num-test (expt -0.00000001-0.0i -0.00000001-0.00000001i) 1.00000021562275+0.00000015279091i)
(num-test (expt 0.00000001+0.00000001i 1.0+0.0i) 0.00000001+0.00000001i)
(num-test (expt -0.00000001+0.00000001i 1.0-0.0i) -0.00000001+0.00000001i)
(num-test (expt 0.00000001-0.00000001i -1.0+0.0i) 50000000.00000004470348+50000000.00000003725290i)
(num-test (expt -0.00000001-0.00000001i -1.0-0.0i) -50000000.00000003725290+50000000.00000004470348i)
(num-test (expt 1.0+0.00000001i 3.14159265358979+0.00000001i) 1.0+0.00000003141593i)
(num-test (expt -1.0+0.00000001i 3.14159265358979-0.00000001i) -0.90268540381008-0.43030120215971i)
(num-test (expt 1.0-0.00000001i -3.14159265358979+0.00000001i) 1.0+0.00000003141593i)
(num-test (expt -1.0-0.00000001i -3.14159265358979-0.00000001i) -0.90268534709269-0.43030117512308i)
(num-test (expt 1.0+1.0i 2.71828182845905+0.0i) -1.37164508585166+2.16782742612896i)
(num-test (expt -1.0+1.0i 2.71828182845905-0.0i) 2.54637618158683+0.31121428769451i)
(num-test (expt 1.0-1.0i -2.71828182845905+0.0i) -0.20842863525121+0.32941270052205i)
(num-test (expt -1.0-1.0i -2.71828182845905-0.0i) 0.38693516117166+0.04729063656767i)
(num-test (expt 3.14159265358979-1.0i -1234.0+0.00000001i) -0.0)
(num-test (expt -3.14159265358979-1.0i -1234.0-0.00000001i) -0.0)
(num-test (expt 3.14159265358979-3.14159265358979i -1234000000.0+0.0i) 0.0)
(num-test (expt -3.14159265358979-3.14159265358979i -1234000000.0-0.0i) 0.0)
(num-test (expt 2.71828182845905+3.14159265358979i 0.0+0.00000001i) 0.99999999142488+0.00000001424157i)
(num-test (expt -2.71828182845905+3.14159265358979i 0.0-0.00000001i) 1.00000002284081-0.00000001424157i)
(num-test (expt 2.71828182845905-3.14159265358979i -0.0+0.00000001i) 1.00000000857512+0.00000001424157i)
(num-test (expt -2.71828182845905-3.14159265358979i -0.0-0.00000001i) 0.99999997715919-0.00000001424157i)
(num-test (expt 2.71828182845905+2.71828182845905i 0.00000001+0.0i) 1.00000001346574+0.00000000785398i)
(num-test (expt -2.71828182845905+2.71828182845905i 0.00000001-0.0i) 1.00000001346574+0.00000002356195i)
(num-test (expt 2.71828182845905-2.71828182845905i -0.00000001+0.0i) 0.99999998653426+0.00000000785398i)
(num-test (expt -2.71828182845905-2.71828182845905i -0.00000001-0.0i) 0.99999998653426+0.00000002356194i)
(num-test (expt 1234.0+2.71828182845905i 1.0+0.00000001i) 1233.99999977932634+2.71836966474906i)
(num-test (expt -1234.0+2.71828182845905i 1.0-0.00000001i) -1234.00003854658053+2.71836975014915i)
(num-test (expt 1234.0-2.71828182845905i -1.0+0.00000001i) 0.00081036883911+0.00000178515565i)
(num-test (expt -1234.0-2.71828182845905i -1.0-0.00000001i) -0.00081036881365+0.00000178515559i)
(num-test (expt 1234.0+1234.0i 3.14159265358979+0.0i) -11947544392.17545890808105+9547275530.50568199157715i)
(num-test (expt -1234.0+1234.0i 3.14159265358979-0.0i) 6676669154.05054950714111+13759228759.84499740600586i)
(num-test (expt 1234.0-1234.0i -3.14159265358979+0.0i) -5.108095859217296E-11+4.081876325659167E-11i)
(num-test (expt -1234.0-1234.0i -3.14159265358979-0.0i) 2.854567008891443E-11+5.882669873167984E-11i)
(num-test (expt 1234000000.0-1234.0i -2.71828182845905+0.00000001i) 0.0+0.0i)
(num-test (expt -1234000000.0-1234.0i -2.71828182845905-0.00000001i) -0.0+0.0i)
(num-test (expt 1234000000.0-1234000000.0i -1234.0+0.0i) 0.0)
(num-test (expt -1234000000.0-1234000000.0i -1234.0-0.0i) 0.0)
(num-test (expt 1.0e+00+0.0e+00i 0.0e+00+0.0e+00i) 1e0+0.0i)
(num-test (expt 1.0e+00+0.0e+00i 1.0e+00+0.0e+00i) 1e0+0.0i)
(num-test (expt 1.0e+00+0.0e+00i 0.0e+00+1.0e+00i) 1e0+0.0i)
(num-test (expt 1.0e+00+0.0e+00i -1.0e+00+0.0e+00i) 1e0+0.0i)
(num-test (expt 1.0e+00+0.0e+00i 0.0e+00-1.0e+00i) 1e0+0.0i)
(num-test (expt 1.0e+00+0.0e+00i 5.0e-01+1.00000000000000005551e-01i) 1e0+0.0i)
(num-test (expt 1.0e+00+0.0e+00i 5.0e-01-1.00000000000000005551e-01i) 1e0+0.0i)
(num-test (expt 0.0e+00+1.0e+00i 0.0e+00+0.0e+00i) 1e0+0.0i)
(num-test (expt 0.0e+00+1.0e+00i 1.0e+00+0.0e+00i) 0+1i)
(num-test (expt 0.0e+00+1.0e+00i 0.0e+00+1.0e+00i) 2.0787957635076190855e-1+0.0i)
(num-test (expt 0.0e+00+1.0e+00i -1.0e+00+0.0e+00i) 0-1i)
(num-test (expt 0.0e+00+1.0e+00i 0.0e+00-1.0e+00i) 4.8104773809653516555e0+0.0i)
(num-test (expt 0.0e+00+1.0e+00i 5.0e-01+1.00000000000000005551e-01i) 6.0431891044739184057e-1+6.0431891044739184057e-1i)
(num-test (expt 0.0e+00+1.0e+00i 5.0e-01-1.00000000000000005551e-01i) 8.2737771622906514822e-1+8.2737771622906514822e-1i)
(num-test (expt -1.0e+00+0.0e+00i 0.0e+00+0.0e+00i) 1e0+0.0i)
(num-test (expt -1.0e+00+0.0e+00i 1.0e+00+0.0e+00i) -1e0+0.0i)
(num-test (expt -1.0e+00+0.0e+00i 0.0e+00+1.0e+00i) 4.3213918263772249774e-2+0.0i)
(num-test (expt -1.0e+00+0.0e+00i -1.0e+00+0.0e+00i) -1e0+0.0i)
(num-test (expt -1.0e+00+0.0e+00i 0.0e+00-1.0e+00i) 2.3140692632779269006e1+0.0i)
(num-test (expt -1.0e+00+0.0e+00i 5.0e-01+1.00000000000000005551e-01i) 0+7.3040269104864559813e-1i)
(num-test (expt -1.0e+00+0.0e+00i 5.0e-01-1.00000000000000005551e-01i) 0+1.3691077706248469087e0i)
(num-test (expt 0.0e+00-1.0e+00i 0.0e+00+0.0e+00i) 1e0+0.0i)
(num-test (expt 0.0e+00-1.0e+00i 1.0e+00+0.0e+00i) 0-1i)
(num-test (expt 0.0e+00-1.0e+00i 0.0e+00+1.0e+00i) 4.8104773809653516555e0+0.0i)
(num-test (expt 0.0e+00-1.0e+00i -1.0e+00+0.0e+00i) 0+1i)
(num-test (expt 0.0e+00-1.0e+00i 0.0e+00-1.0e+00i) 2.0787957635076190855e-1+0.0i)
(num-test (expt 0.0e+00-1.0e+00i 5.0e-01+1.00000000000000005551e-01i) 8.2737771622906514822e-1-8.2737771622906514822e-1i)
(num-test (expt 0.0e+00-1.0e+00i 5.0e-01-1.00000000000000005551e-01i) 6.0431891044739184057e-1-6.0431891044739184057e-1i)
(num-test (expt 5.0e-01+1.00000000000000005551e-01i 0.0e+00+0.0e+00i) 1e0+0.0i)
(num-test (expt 5.0e-01+1.00000000000000005551e-01i 1.0e+00+0.0e+00i) 5e-1+1.0000000000000000555e-1i)
(num-test (expt 5.0e-01+1.00000000000000005551e-01i 0.0e+00+1.0e+00i) 6.4160554864378080418e-1-5.1201864456768275590e-1i)
(num-test (expt 5.0e-01+1.00000000000000005551e-01i -1.0e+00+0.0e+00i) 1.9230769230769230687e0-3.8461538461538463509e-1i)
(num-test (expt 5.0e-01+1.00000000000000005551e-01i 0.0e+00-1.0e+00i) 9.5219021866126714108e-1+7.5987364224031834571e-1i)
(num-test (expt 5.0e-01+1.00000000000000005551e-01i 5.0e-01+1.00000000000000005551e-01i) 6.9977300530987816719e-1+2.1940939105372143160e-2i)
(num-test (expt 5.0e-01+1.00000000000000005551e-01i 5.0e-01-1.00000000000000005551e-01i) 7.1829191470060938876e-1+1.2038189555821612762e-1i)
(num-test (expt 5.0e-01-1.00000000000000005551e-01i 0.0e+00+0.0e+00i) 1e0+0.0i)
(num-test (expt 5.0e-01-1.00000000000000005551e-01i 1.0e+00+0.0e+00i) 5e-1-1.0000000000000000555e-1i)
(num-test (expt 5.0e-01-1.00000000000000005551e-01i 0.0e+00+1.0e+00i) 9.5219021866126714108e-1-7.5987364224031834571e-1i)
(num-test (expt 5.0e-01-1.00000000000000005551e-01i -1.0e+00+0.0e+00i) 1.9230769230769230687e0+3.8461538461538463509e-1i)
(num-test (expt 5.0e-01-1.00000000000000005551e-01i 0.0e+00-1.0e+00i) 6.4160554864378080418e-1+5.1201864456768275590e-1i)
(num-test (expt 5.0e-01-1.00000000000000005551e-01i 5.0e-01+1.00000000000000005551e-01i) 7.1829191470060938876e-1-1.2038189555821612762e-1i)
(num-test (expt 5.0e-01-1.00000000000000005551e-01i 5.0e-01-1.00000000000000005551e-01i) 6.9977300530987816719e-1-2.1940939105372143160e-2i)

(num-test (expt 0 10) 0)
(num-test (expt 1 10) 1)
(num-test (expt 2 10) 1024)
(num-test (expt 3 10) 59049)
(num-test (expt 4 10) 1048576)
(num-test (expt 5 10) 9765625)
(num-test (expt 6 10) 60466176)
(num-test (expt 7 10) 282475249)
(num-test (expt 8 10) 1073741824)
(if (or with-bignums with-64-bit-ints) (num-test (expt 9 10) 3486784401))
(if (or with-bignums with-64-bit-ints) (num-test (expt 10 10) 10000000000))
(if (or with-bignums with-64-bit-ints) (num-test (expt 11 10) 25937424601))
(num-test (expt 0 1/10) 0)
(num-test (expt 1 1/10) 1)
(num-test (expt 1024 1/10) 2)
(num-test (expt 59049 1/10) 3)
(num-test (expt 1048576 1/10) 4)
(num-test (expt 9765625 1/10) 5)
(num-test (expt 60466176 1/10) 6)
(num-test (expt 282475249 1/10) 7)
(num-test (expt 1073741824 1/10) 8)
(if (or with-bignums with-64-bit-ints) (num-test (expt 3486784401 1/10) 9))
(if (or with-bignums with-64-bit-ints) (num-test (expt 10000000000 1/10) 10))
(if (or with-bignums with-64-bit-ints) (num-test (expt 25937424601 1/10) 11))
(num-test (expt 2 9) 512)
(num-test (expt 8 1/3) 2)
(num-test (expt 1024 1/10) 2)
(num-test (expt 512 1/9) 2)
(num-test (expt (expt 20 10) 1/10) 20)
(num-test (expt (expt 40 10) 1/10) 40)
(num-test (expt (expt 2 30) 1/30) 2)
(num-test (expt (expt 2 50) 1/50) 2)
(num-test (expt (expt 2 1/10) 10) 2)
(num-test (expt (expt 2 1/30) 30) 2)
(num-test (expt 64 1/6) 2)
(num-test (expt 64 1/3) 4)
(num-test (expt 64 2/3) 16)
(num-test (expt 64 1/2) 8)
(num-test (expt 64 3/2) 512)
(num-test (expt 1/2 3) 1/8)
(num-test (expt 1/64 1/2) 1/8)
(num-test (expt 1/64 1/3) 1/4)
(num-test (expt 1/64 3/2) 1/512)
(num-test (expt 1/64 2/3) 1/16)
(num-test (expt 1/2 10) 1/1024)
(num-test (expt 2/3 5) (* 2/3 2/3 2/3 2/3 2/3))
(num-test (expt 2 -10) 1/1024)
(num-test (expt 2 -9) 1/512)
(num-test (expt 64 -1/6) 1/2)
(num-test (expt 64 -1/3) 1/4)
(num-test (expt 64 -2/3) 1/16)
(num-test (expt 64 -1/2) 1/8)
(num-test (expt 64 -3/2) 1/512)
(num-test (expt 1/2 -3) 8)
(num-test (expt 1/64 -1/2) 8)
(num-test (expt 1/64 -1/3) 4)
(num-test (expt 1/64 -3/2) 512)
(num-test (expt 1/64 -2/3) 16)
(num-test (expt 1/2 -10) 1024)
(num-test (expt 2718/1000 617/5) 3.858179469787681136058424024656091858698003418770850904916305853631035158956514884526199288e53) ; not an int!
(num-test (expt 2/3 -5) (/ 1 (* 2/3 2/3 2/3 2/3 2/3)))
(num-test (expt 0 1.0+i) 0.0) ; ??
(num-test (expt 0 1.0) 0.0)
(test (expt 0 0) 1 )
(test (expt 0 1) 0 )
(test (expt 0 256) 0 )
(test (expt -1 256) 1 )
(test (expt -1 255) -1 )
(test (expt -1 -256) 1 )
(test (expt -1 -255) -1 )
(test (expt 256 0) 1 )
(test (expt -256 0) 1 )
(test (expt 256 1) 256 )
(test (expt -256 1) -256 )
(test (expt 2 3) 8 )
(test (expt -2 3) -8 )
(test (expt 3 2) 9 )
(test (expt -3 2) 9 )

;; a few picky ones
(num-test (expt 3 1) 3)
(num-test (expt 3/4 1) 3/4)
(num-test (expt 3.0 1) 3.0)
(num-test (expt 1+i 1) 1+i)
(num-test (expt 3 1.0) 3.0)
(num-test (expt 3/4 1.0) 0.75)
(num-test (expt 3.0 1.0) 3.0)
(num-test (expt 1+i 1.0) 1.0+i)
(num-test (expt 0 1) 0)
(num-test (expt 0.0 1) 0.0)
(num-test (expt 1 0) 1)
(num-test (expt 1.0 0) 1.0)
(num-test (expt 1 0.0) 1.0)
(num-test (expt 1.0 0.0) 1.0)
(num-test (expt 3/4 0) 1)
(num-test (expt 3/4 0.0) 1.0)
(num-test (expt 3.0 0) 1.0)
(num-test (expt 3.0 0.0) 1.0)
(num-test (expt 0 0) 1)
(num-test (expt 0.0 0) 0.0)
(num-test (expt 0 0.0) 0.0)
(num-test (expt 0.0 0.0) 0.0)


(let ((x-10 (lambda (n) (- (expt n 10) (* n n n n n n n n n n)))))
  (let ((happy #t)
	(lim (if with-bignums 100
		 (if with-64-bit-ints 74
		     8))))
    (do ((i 1 (+ i 2)))
	((or (not happy) (> i lim))) ; stop around 63 bits
      (let ((val (x-10 (/ i 2))))
	(if (not (= val 0))
	    (begin
	      (set! happy #f)
	      (display "(expt ") (display i) (display "/2 10) = ") (display (expt (/ i 2) 10))
	      (display " but (* ") (display i) (display "/2 ... 10x) = ")
	      (display (/ (* i i i i i i i i i i) 1024)) (newline)))))))

(let ((x-10 (lambda (n) (- (expt n -10) (/ 1 (* n n n n n n n n n n))))))
  (let ((happy #t)
	(lim (if with-bignums 100
		 (if with-64-bit-ints 74
		     8))))
    (do ((i 1 (+ i 2)))
	((or (not happy) (> i lim))) ; stop around 63 bits
      (let ((val (x-10 (/ i 2))))
	(if (not (= val 0))
	    (begin
	      (set! happy #f)
	      (display "(expt ") (display i) (display "/2 -10) = ") (display (expt (/ i 2) -10))
	      (display " but (* 1/(") (display i) (display "/2) ... 10x) = ")
	      (display (/ 1024 (* i i i i i i i i i i))) 
	      (display " [diff=") (display val) (display "]")
	      (newline)))))))

(let ((happy #t)
      (lim (if with-bignums 50
	       (if with-64-bit-ints 19
		   4))))
  (do ((i 1 (+ i 1)))
      ((or (not happy) (> i lim)))
    (let* ((val1 (expt 3 i))
	   (val2 (sqrt (* val1 val1))))
      (if (not (= val1 val2))
	  (begin
	    (set! happy #f)
	    (display "[3^i] (sqrt ") (display (* val1 val1)) (display " = ") (display val2)
	      (display " but should be ") (display val1) (newline))))))



;;; -------- rationalize

(if with-rationalize 
    (begin
      (num-test (rationalize 0.0 1.0001) 0)
      (num-test (rationalize -0.0 1.0001) 0)
      (num-test (rationalize 0.0 0.50000000000000) 0)
      (num-test (rationalize -0.0 0.50000000000000) 0)
      (num-test (rationalize 0.0 0.1) 0)
      (num-test (rationalize -0.0 0.1) 0)
      (num-test (rationalize 0.0 0.001) 0)
      (num-test (rationalize -0.0 0.001) 0)
      (num-test (rationalize 0.0 0.00300000000000) 0)
      (num-test (rationalize -0.0 0.00300000000000) 0)
      (num-test (rationalize 0.0 0.00002000000000) 0)
      (num-test (rationalize -0.0 0.00002000000000) 0)
      (num-test (rationalize 0.0 0.00000001) 0)
      (num-test (rationalize -0.0 0.00000001) 0)
      (num-test (rationalize 0.00000001 1.0) 0)
      (num-test (rationalize -0.00000001 1.0) 0)
      (num-test (rationalize 0.00000001 0.50000000000000) 0)
      (num-test (rationalize -0.00000001 0.50000000000000) 0)
      (num-test (rationalize 0.00000001 0.1) 0)
      (num-test (rationalize -0.00000001 0.1) 0)
      (num-test (rationalize 0.00000001 0.001) 0)
      (num-test (rationalize -0.00000001 0.001) 0)
      (num-test (rationalize 0.00000001 0.00300000000000) 0)
      (num-test (rationalize -0.00000001 0.00300000000000) 0)
      (num-test (rationalize 0.00000001 0.00002000000000) 0)
      (num-test (rationalize -0.00000001 0.00002000000000) 0)
      (num-test (rationalize 0.00000001 0.000000011) 0)
      (num-test (rationalize -0.00000001 0.000000011) 0)
      (num-test (rationalize 1.0 1.0001) 0)
      (num-test (rationalize -1.0 1.0001) 0)
      (num-test (rationalize 1.0 0.999) 1)
      (num-test (rationalize -1.0 0.999) -1)
      (num-test (rationalize 1.0 0.50000000000000) 1)
      (num-test (rationalize -1.0 0.50000000000000) -1)
      (num-test (rationalize 1.0 0.1) 1)
      (num-test (rationalize -1.0 0.1) -1)
      (num-test (rationalize 1.0 0.001) 1)
      (num-test (rationalize -1.0 0.001) -1)
      (num-test (rationalize 1.0 0.00300000000000) 1)
      (num-test (rationalize -1.0 0.00300000000000) -1)
      (num-test (rationalize 1.0 0.00002000000000) 1)
      (num-test (rationalize -1.0 0.00002000000000) -1)
      (num-test (rationalize 1.0 0.00000001) 1)
      (num-test (rationalize -1.0 0.00000001) -1)
      (num-test (rationalize 3.14159265358979 1.0) 3)
      (num-test (rationalize -3.14159265358979 1.0) -3)
      (num-test (rationalize 3.14159265358979 0.50000000000000) 3)
      (num-test (rationalize -3.14159265358979 0.50000000000000) -3)
      (num-test (rationalize 3.14159265358979 0.1) 16/5)
      (num-test (rationalize -3.14159265358979 0.1) -16/5)
      (num-test (rationalize 3.14159265358979 0.001) 201/64)
      (num-test (rationalize -3.14159265358979 0.001) -201/64)
      (num-test (rationalize 3.14159265358979 0.00300000000000) 22/7)
      (num-test (rationalize -3.14159265358979 0.00300000000000) -22/7)
      (num-test (rationalize 3.14159265358979 0.00002000000000) 355/113)
      (num-test (rationalize -3.14159265358979 0.00002000000000) -355/113)
      (num-test (rationalize 3.14159265358979 0.00000001) 100798/32085)
      (num-test (rationalize -3.14159265358979 0.00000001) -100798/32085)
      (num-test (rationalize 2.71828182845905 1.0) 2)
      (num-test (rationalize -2.71828182845905 1.0) -2)
      (num-test (rationalize 2.71828182845905 0.50000000000000) 3)
      (num-test (rationalize -2.71828182845905 0.50000000000000) -3)
      (num-test (rationalize 2.71828182845905 0.1) 8/3)
      (num-test (rationalize -2.71828182845905 0.1) -8/3)
      (num-test (rationalize 2.71828182845905 0.001) 87/32)
      (num-test (rationalize -2.71828182845905 0.001) -87/32)
      (num-test (rationalize 2.71828182845905 0.00300000000000) 68/25)
      (num-test (rationalize -2.71828182845905 0.00300000000000) -68/25)
      (num-test (rationalize 2.71828182845905 0.00002000000000) 878/323)
      (num-test (rationalize -2.71828182845905 0.00002000000000) -878/323)
      (num-test (rationalize 2.71828182845905 0.00000001) 23225/8544)
      (num-test (rationalize -2.71828182845905 0.00000001) -23225/8544)
      (num-test (rationalize 1234.12339999999995 1.0) 1234)
      (num-test (rationalize -1234.12339999999995 1.0) -1234)
      (num-test (rationalize 1234.12339999999995 0.50000000000000) 1234)
      (num-test (rationalize -1234.12339999999995 0.50000000000000) -1234)
      (num-test (rationalize 1234.12339999999995 0.1) 6171/5)
      (num-test (rationalize -1234.12339999999995 0.1) -6171/5)
      (num-test (rationalize 1234.12339999999995 0.001) 60472/49)
      (num-test (rationalize -1234.12339999999995 0.001) -60472/49)
      (num-test (rationalize 1234.12339999999995 0.00300000000000) 9873/8)
      (num-test (rationalize -1234.12339999999995 0.00300000000000) -9873/8)
      (num-test (rationalize 1234.12339999999995 0.00002000000000) 290019/235)
      (num-test (rationalize -1234.12339999999995 0.00002000000000) -290019/235)
      (num-test (rationalize 1234.12339999999995 0.00000001) 6170617/5000)
      (num-test (rationalize -1234.12339999999995 0.00000001) -6170617/5000)
      (num-test (rationalize 1234000000.01234006881714 1.0) 1234000000/1)
      (num-test (rationalize -1234000000.01234006881714 1.0) -1234000000/1)
      (num-test (rationalize 1234000000.01234006881714 0.50000000000000) 1234000000/1)
      (num-test (rationalize -1234000000.01234006881714 0.50000000000000) -1234000000/1)
      (num-test (rationalize 1234000000.01234006881714 0.1) 1234000000/1)
      (num-test (rationalize -1234000000.01234006881714 0.1) -1234000000/1)
      (num-test (rationalize 1234000000.01234006881714 0.001) 92550000001/75)
      (num-test (rationalize -1234000000.01234006881714 0.001) -92550000001/75)
      (num-test (rationalize 1234000000.01234006881714 0.00300000000000) 81444000001/66)
      (num-test (rationalize -1234000000.01234006881714 0.00300000000000) -81444000001/66)
      (num-test (rationalize 1234000000.01234006881714 0.00002000000000) 99954000001/81)
      (num-test (rationalize -1234000000.01234006881714 0.00002000000000) -99954000001/81)
      (num-test (rationalize 1234000000.01234006881714 0.000001) 2400130000024/1945)
      (num-test (rationalize -1234000000.01234006881714 0.000001) -2400130000024/1945)
      (num-test (rationalize 0.33 1.0) 0)
      (num-test (rationalize -0.33 1.0) 0)
      (num-test (rationalize 0.33 0.50000000000000) 0)
      (num-test (rationalize -0.33 0.50000000000000) 0)
      (num-test (rationalize 0.33 0.1) 1/3)
      (num-test (rationalize -0.33 0.1) -1/3)
      (num-test (rationalize 0.33 0.001) 26/79)
      (num-test (rationalize -0.33 0.001) -26/79)
      (num-test (rationalize 0.33 0.00300000000000) 18/55)
      (num-test (rationalize -0.33 0.00300000000000) -18/55)
      (num-test (rationalize 0.33 0.00002000000000) 33/100)
      (num-test (rationalize -0.33 0.00002000000000) -33/100)
      (num-test (rationalize 0.33 0.00000001) 33/100)
      (num-test (rationalize -0.33 0.00000001) -33/100)
      (num-test (rationalize 0.99990 1.0) 0)
      (num-test (rationalize -0.99990 1.0) 0)
      (num-test (rationalize 0.99990 0.50000000000000) 1)
      (num-test (rationalize -0.99990 0.50000000000000) -1)
      (num-test (rationalize 0.99990 0.1) 1)
      (num-test (rationalize -0.99990 0.1) -1)
      (num-test (rationalize 0.99990 0.001) 1)
      (num-test (rationalize -0.99990 0.001) -1)
      (num-test (rationalize 0.99990 0.00300000000000) 1)
      (num-test (rationalize -0.99990 0.00300000000000) -1)
      (num-test (rationalize 0.99990 0.00002000000000) 8333/8334)
      (num-test (rationalize -0.99990 0.00002000000000) -8333/8334)
      (num-test (rationalize 0.99990 0.00000001) 9999/10000)
      (num-test (rationalize -0.99990 0.00000001) -9999/10000)
      (num-test (rationalize 0.5010 1.0) 0)
      (num-test (rationalize -0.5010 1.0) 0)
      (num-test (rationalize 0.5010 0.50000000000000) 1)
      (num-test (rationalize -0.5010 0.50000000000000) -1)
      (num-test (rationalize 0.5010 0.1) 1/2)
      (num-test (rationalize -0.5010 0.1) -1/2)
      (num-test (rationalize 0.5010 0.00099) 127/253)
      (num-test (rationalize -0.5010 0.00099) -127/253)
      (num-test (rationalize 0.5010 0.00300000000000) 1/2)
      (num-test (rationalize -0.5010 0.00300000000000) -1/2)
      (num-test (rationalize 0.5010 0.00002000000000) 246/491)
      (num-test (rationalize -0.5010 0.00002000000000) -246/491)
      (num-test (rationalize 0.5010 0.00000001) 501/1000)
      (num-test (rationalize -0.5010 0.00000001) -501/1000)
      (num-test (rationalize 0.499 1.0) 0)
      (num-test (rationalize -0.499 1.0) 0)
      (num-test (rationalize 0.499 0.50000000000000) 0)
      (num-test (rationalize -0.499 0.50000000000000) 0)
      (num-test (rationalize 0.499 0.1) 1/2)
      (num-test (rationalize -0.499 0.1) -1/2)
      (num-test (rationalize 0.499 0.00099) 126/253)
      (num-test (rationalize -0.499 0.00099) -126/253)
      (num-test (rationalize 0.499 0.00300000000000) 1/2)
      (num-test (rationalize -0.499 0.00300000000000) -1/2)
      (num-test (rationalize 0.499 0.00002000000000) 245/491)
      (num-test (rationalize -0.499 0.00002000000000) -245/491)
      (num-test (rationalize 0.499 0.00000001) 499/1000)
      (num-test (rationalize -0.499 0.00000001) -499/1000)
      (num-test (rationalize 1.501 1.0) 1)
      (num-test (rationalize -1.501 1.0) -1)
      (num-test (rationalize 1.501 0.50000000000000) 2)
      (num-test (rationalize -1.501 0.50000000000000) -2)
      (num-test (rationalize 1.501 0.1) 3/2)
      (num-test (rationalize -1.501 0.1) -3/2)
      (num-test (rationalize 1.501 0.001) 3/2)
      (num-test (rationalize -1.501 0.001) -3/2)
      (num-test (rationalize 1.501 0.00300000000000) 3/2)
      (num-test (rationalize -1.501 0.00300000000000) -3/2)
      (num-test (rationalize 1.501 0.00002000000000) 737/491)
      (num-test (rationalize -1.501 0.00002000000000) -737/491)
      (num-test (rationalize 1.501 0.00000001) 1501/1000)
      (num-test (rationalize -1.501 0.00000001) -1501/1000)
      (num-test (rationalize 1.499 1.0) 1)
      (num-test (rationalize -1.499 1.0) -1)
      (num-test (rationalize 1.499 0.50000000000000) 1)
      (num-test (rationalize -1.499 0.50000000000000) -1)
      (num-test (rationalize 1.499 0.1) 3/2)
      (num-test (rationalize -1.499 0.1) -3/2)
      (num-test (rationalize 1.499 0.001) 3/2)
      (num-test (rationalize -1.499 0.001) -3/2)
      (num-test (rationalize 1.499 0.00300000000000) 3/2)
      (num-test (rationalize -1.499 0.00300000000000) -3/2)
      (num-test (rationalize 1.499 0.00002000000000) 736/491)
      (num-test (rationalize -1.499 0.00002000000000) -736/491)
      (num-test (rationalize 1.499 0.00000001) 1499/1000)
      (num-test (rationalize -1.499 0.00000001) -1499/1000)

      (num-test (rationalize 1.16 .2) 1)
      (num-test (rationalize 1.16 .1) 5/4)
      (num-test (rationalize 1.16 .041) 6/5)
      (num-test (rationalize 1.16 .039) 7/6)
      (num-test (rationalize 1.16 .007) 7/6)
      (num-test (rationalize 1.16 .006) 22/19)
      (num-test (rationalize 1.16 .0022) 22/19)
      (num-test (rationalize 1.16 .002) 29/25)
      (num-test (rationalize 1.16 .0000001) 29/25)

      (num-test (rationalize 23.1 22.0) 2)
      (num-test (rationalize 23.1 22) 2)
      (num-test (rationalize 23.1 .5) 23)
      (num-test (rationalize 23.1 1/2) 23)

      (num-test (rationalize 1/2 3/4) 0)
      (num-test (rationalize 1/2 1/4) 1/2)
      (num-test (rationalize 1 3) 0)
      (num-test (rationalize 11/10 1/5) 1)
      (num-test (rationalize 3/4 1/2) 1)
      (num-test (rationalize 1/4 1/3) 0)
      (num-test (rationalize 1/4 1/6) 1/3)
      (num-test (rationalize 2/3 1/4) 1/2)
      (num-test (rationalize 1/3 1/3) 0)
      (num-test (rationalize 1/3 1/4) 1/2)
      (num-test (rationalize 3/10 1/10) 1/3)

      (num-test (rationalize (exact->inexact 1/2) 3/4) 0)
      (num-test (rationalize (exact->inexact 1/2) 1/4) 1/2)
      (num-test (rationalize (exact->inexact 1) 3) 0)
      (num-test (rationalize (exact->inexact 11/10) 1/5) 1)
      (num-test (rationalize (exact->inexact 3/4) 1/2) 1)
      (num-test (rationalize (exact->inexact 1/4) 1/3) 0)
      (num-test (rationalize (exact->inexact 1/4) 1/6) 1/3)
      (num-test (rationalize (exact->inexact 2/3) 1/4) 1/2)
      (num-test (rationalize (exact->inexact 1/3) 1/4) 1/2)
      (num-test (rationalize (exact->inexact 3/10) 1/10) 1/3)

      (num-test (rationalize 1/2 (exact->inexact 3/4)) 0)
      (num-test (rationalize 1/2 (exact->inexact 1/4)) 1/2)
      (num-test (rationalize 1 (exact->inexact 3)) 0)
      (num-test (rationalize 11/10 (exact->inexact 1/5)) 1)
      (num-test (rationalize 3/4 (exact->inexact 1/2)) 1)
      (num-test (rationalize 1/4 (exact->inexact 1/3)) 0)
      (num-test (rationalize 1/4 (exact->inexact 1/6)) 1/3)
      (num-test (rationalize 2/3 (exact->inexact 1/4)) 1/2)
      (num-test (rationalize 1/3 (exact->inexact 1/4)) 1/2)
      (num-test (rationalize 3/10 (exact->inexact 1/10)) 1/3)

      (num-test (rationalize (exact->inexact 1/2) (exact->inexact 3/4)) 0)
      (num-test (rationalize (exact->inexact 1/2) (exact->inexact 1/4)) 1/2)
      (num-test (rationalize (exact->inexact 1) (exact->inexact 3)) 0)
      (num-test (rationalize (exact->inexact 11/10) (exact->inexact 1/5)) 1)
      (num-test (rationalize (exact->inexact 3/4) (exact->inexact 1/2)) 1)
      (num-test (rationalize (exact->inexact 1/4) (exact->inexact 1/3)) 0)
      (num-test (rationalize (exact->inexact 1/4) (exact->inexact 1/6)) 1/3)
      (num-test (rationalize (exact->inexact 2/3) (exact->inexact 1/4)) 1/2)
      (num-test (rationalize (exact->inexact 1/3) (exact->inexact 1/4)) 1/2)
      (num-test (rationalize (exact->inexact 3/10) (exact->inexact 1/10)) 1/3)

      (num-test (rationalize -1/2 3/4) 0)
      (num-test (rationalize -1/2 1/4) -1/2)
      (num-test (rationalize -1 3) 0)
      (num-test (rationalize -11/10 1/5) -1)
      (num-test (rationalize -3/4 1/2) -1)
      (num-test (rationalize -1/4 1/3) 0)
      (num-test (rationalize -1/4 1/6) -1/3)
      (num-test (rationalize -2/3 1/4) -1/2)
      (num-test (rationalize -1/3 1/4) -1/2)
      (num-test (rationalize -1/3 1/3) 0)
      (num-test (rationalize -3/10 1/10) -1/3)

      (num-test (rationalize .239 .0005) 11/46) ;baseball of course... the average .001 is the hardest to get: 1/667
      (num-test (rationalize .001 .0005) 1/667)
      (num-test (rationalize .334 .0005) 96/287)

      (num-test (rationalize 1.0000001 0.00000001) 9090911/9090910)
      (num-test (rationalize 0.000000015 0.000000001) 1/62500001)

      (num-test (rationalize -1 -1) 0) ;; spec says "differs by no more than", but that seems to imply a comparison
                                       ;; on either side, so a negative error doesn't change the result??
      (num-test (rationalize 1/4 -1/6) 1/3)
      (num-test (rationalize -3/10 -1/10) -1/3)
      (num-test (rationalize (exact->inexact 1/3) (exact->inexact -1/4)) 1/2)
      
      (if with-bigfloats
	  (begin
	    ;; can this sort of thing be handled with locally set precisions? 
	    (num-test (rationalize 385817946978768113605842402465609185854927496022065152.5) 771635893957536227211684804931218371709854992044130305/2)
	    ))

      (num-test (rationalize 0.5 0.02) 1/2)
      (num-test (rationalize 1073741824 1) 1073741823) ; perverse
      ))


;; -------- gcd and lcm

(num-test (gcd) 0)
(num-test (lcm) 1)
(num-test (gcd 1 0) 1)
(num-test (lcm 1 0) 0)
(num-test (gcd 0) 0)
(num-test (lcm 0) 0)
(num-test (gcd 0) 0)
(num-test (lcm 0) 0)
(num-test (gcd 1) 1)
(num-test (lcm 1) 1)
(num-test (gcd -1) 1)
(num-test (lcm -1) 1)
(num-test (gcd 2) 2)
(num-test (lcm 2) 2)
(num-test (gcd -2) 2)
(num-test (lcm -2) 2)
(num-test (gcd 3) 3)
(num-test (lcm 3) 3)
(num-test (gcd -3) 3)
(num-test (lcm -3) 3)
(num-test (gcd 10) 10)
(num-test (lcm 10) 10)
(num-test (gcd -10) 10)
(num-test (lcm -10) 10)
(num-test (gcd 1234) 1234)
(num-test (lcm 1234) 1234)
(num-test (gcd -1234) 1234)
(num-test (lcm -1234) 1234)
(num-test (gcd 1234000000) 1234000000)
(num-test (lcm 1234000000) 1234000000)
(num-test (gcd -1234000000) 1234000000)
(num-test (lcm -1234000000) 1234000000)
(num-test (gcd 500029) 500029)
(num-test (lcm 500029) 500029)
(num-test (gcd -500029) 500029)
(num-test (lcm -500029) 500029)
(num-test (gcd 362880) 362880)
(num-test (lcm 362880) 362880)
(num-test (gcd -362880) 362880)
(num-test (lcm -362880) 362880)
(num-test (gcd 0 0) 0)
(num-test (lcm 0 0) 0)
(num-test (gcd 0 0) 0)
(num-test (lcm 0 0) 0)
(num-test (gcd 0 0) 0)
(num-test (lcm 0 0) 0)
(num-test (gcd 0 0) 0)
(num-test (lcm 0 0) 0)
(num-test (gcd 0 1) 1)
(num-test (lcm 0 1) 0)
(num-test (gcd 0 1) 1)
(num-test (lcm 0 1) 0)
(num-test (gcd 0 -1) 1)
(num-test (lcm 0 -1) 0)
(num-test (gcd 0 -1) 1)
(num-test (lcm 0 -1) 0)
(num-test (gcd 0 2) 2)
(num-test (lcm 0 2) 0)
(num-test (gcd 0 2) 2)
(num-test (lcm 0 2) 0)
(num-test (gcd 0 -2) 2)
(num-test (lcm 0 -2) 0)
(num-test (gcd 0 -2) 2)
(num-test (lcm 0 -2) 0)
(num-test (gcd 0 3) 3)
(num-test (lcm 0 3) 0)
(num-test (gcd 0 3) 3)
(num-test (lcm 0 3) 0)
(num-test (gcd 0 -3) 3)
(num-test (lcm 0 -3) 0)
(num-test (gcd 0 -3) 3)
(num-test (lcm 0 -3) 0)
(num-test (gcd 0 10) 10)
(num-test (lcm 0 10) 0)
(num-test (gcd 0 10) 10)
(num-test (lcm 0 10) 0)
(num-test (gcd 0 -10) 10)
(num-test (lcm 0 -10) 0)
(num-test (gcd 0 -10) 10)
(num-test (lcm 0 -10) 0)
(num-test (gcd 0 1234) 1234)
(num-test (lcm 0 1234) 0)
(num-test (gcd 0 1234) 1234)
(num-test (lcm 0 1234) 0)
(num-test (gcd 0 -1234) 1234)
(num-test (lcm 0 -1234) 0)
(num-test (gcd 0 -1234) 1234)
(num-test (lcm 0 -1234) 0)
(num-test (gcd 0 1234000000) 1234000000)
(num-test (lcm 0 1234000000) 0)
(num-test (gcd 0 1234000000) 1234000000)
(num-test (lcm 0 1234000000) 0)
(num-test (gcd 0 -1234000000) 1234000000)
(num-test (lcm 0 -1234000000) 0)
(num-test (gcd 0 -1234000000) 1234000000)
(num-test (lcm 0 -1234000000) 0)
(num-test (gcd 0 500029) 500029)
(num-test (lcm 0 500029) 0)
(num-test (gcd 0 500029) 500029)
(num-test (lcm 0 500029) 0)
(num-test (gcd 0 -500029) 500029)
(num-test (lcm 0 -500029) 0)
(num-test (gcd 0 -500029) 500029)
(num-test (lcm 0 -500029) 0)
(num-test (gcd 0 362880) 362880)
(num-test (lcm 0 362880) 0)
(num-test (gcd 0 362880) 362880)
(num-test (lcm 0 362880) 0)
(num-test (gcd 0 -362880) 362880)
(num-test (lcm 0 -362880) 0)
(num-test (gcd 0 -362880) 362880)
(num-test (lcm 0 -362880) 0)
(num-test (gcd 1 0) 1)
(num-test (lcm 1 0) 0)
(num-test (gcd -1 0) 1)
(num-test (lcm -1 0) 0)
(num-test (gcd 1 0) 1)
(num-test (lcm 1 0) 0)
(num-test (gcd -1 0) 1)
(num-test (lcm -1 0) 0)
(num-test (gcd 1 1) 1)
(num-test (lcm 1 1) 1)
(num-test (gcd -1 1) 1)
(num-test (lcm -1 1) 1)
(num-test (gcd 1 -1) 1)
(num-test (lcm 1 -1) 1)
(num-test (gcd -1 -1) 1)
(num-test (lcm -1 -1) 1)
(num-test (gcd 1 2) 1)
(num-test (lcm 1 2) 2)
(num-test (gcd -1 2) 1)
(num-test (lcm -1 2) 2)
(num-test (gcd 1 -2) 1)
(num-test (lcm 1 -2) 2)
(num-test (gcd -1 -2) 1)
(num-test (lcm -1 -2) 2)
(num-test (gcd 1 3) 1)
(num-test (lcm 1 3) 3)
(num-test (gcd -1 3) 1)
(num-test (lcm -1 3) 3)
(num-test (gcd 1 -3) 1)
(num-test (lcm 1 -3) 3)
(num-test (gcd -1 -3) 1)
(num-test (lcm -1 -3) 3)
(num-test (gcd 1 10) 1)
(num-test (lcm 1 10) 10)
(num-test (gcd -1 10) 1)
(num-test (lcm -1 10) 10)
(num-test (gcd 1 -10) 1)
(num-test (lcm 1 -10) 10)
(num-test (gcd -1 -10) 1)
(num-test (lcm -1 -10) 10)
(num-test (gcd 1 1234) 1)
(num-test (lcm 1 1234) 1234)
(num-test (gcd -1 1234) 1)
(num-test (lcm -1 1234) 1234)
(num-test (gcd 1 -1234) 1)
(num-test (lcm 1 -1234) 1234)
(num-test (gcd -1 -1234) 1)
(num-test (lcm -1 -1234) 1234)
(num-test (gcd 1 1234000000) 1)
(num-test (lcm 1 1234000000) 1234000000)
(num-test (gcd -1 1234000000) 1)
(num-test (lcm -1 1234000000) 1234000000)
(num-test (gcd 1 -1234000000) 1)
(num-test (lcm 1 -1234000000) 1234000000)
(num-test (gcd -1 -1234000000) 1)
(num-test (lcm -1 -1234000000) 1234000000)
(num-test (gcd 1 500029) 1)
(num-test (lcm 1 500029) 500029)
(num-test (gcd -1 500029) 1)
(num-test (lcm -1 500029) 500029)
(num-test (gcd 1 -500029) 1)
(num-test (lcm 1 -500029) 500029)
(num-test (gcd -1 -500029) 1)
(num-test (lcm -1 -500029) 500029)
(num-test (gcd 1 362880) 1)
(num-test (lcm 1 362880) 362880)
(num-test (gcd -1 362880) 1)
(num-test (lcm -1 362880) 362880)
(num-test (gcd 1 -362880) 1)
(num-test (lcm 1 -362880) 362880)
(num-test (gcd -1 -362880) 1)
(num-test (lcm -1 -362880) 362880)
(num-test (gcd 2 0) 2)
(num-test (lcm 2 0) 0)
(num-test (gcd -2 0) 2)
(num-test (lcm -2 0) 0)
(num-test (gcd 2 0) 2)
(num-test (lcm 2 0) 0)
(num-test (gcd -2 0) 2)
(num-test (lcm -2 0) 0)
(num-test (gcd 2 1) 1)
(num-test (lcm 2 1) 2)
(num-test (gcd -2 1) 1)
(num-test (lcm -2 1) 2)
(num-test (gcd 2 -1) 1)
(num-test (lcm 2 -1) 2)
(num-test (gcd -2 -1) 1)
(num-test (lcm -2 -1) 2)
(num-test (gcd 2 2) 2)
(num-test (lcm 2 2) 2)
(num-test (gcd -2 2) 2)
(num-test (lcm -2 2) 2)
(num-test (gcd 2 -2) 2)
(num-test (lcm 2 -2) 2)
(num-test (gcd -2 -2) 2)
(num-test (lcm -2 -2) 2)
(num-test (gcd 2 3) 1)
(num-test (lcm 2 3) 6)
(num-test (gcd -2 3) 1)
(num-test (lcm -2 3) 6)
(num-test (gcd 2 -3) 1)
(num-test (lcm 2 -3) 6)
(num-test (gcd -2 -3) 1)
(num-test (lcm -2 -3) 6)
(num-test (gcd 2 10) 2)
(num-test (lcm 2 10) 10)
(num-test (gcd -2 10) 2)
(num-test (lcm -2 10) 10)
(num-test (gcd 2 -10) 2)
(num-test (lcm 2 -10) 10)
(num-test (gcd -2 -10) 2)
(num-test (lcm -2 -10) 10)
(num-test (gcd 2 1234) 2)
(num-test (lcm 2 1234) 1234)
(num-test (gcd -2 1234) 2)
(num-test (lcm -2 1234) 1234)
(num-test (gcd 2 -1234) 2)
(num-test (lcm 2 -1234) 1234)
(num-test (gcd -2 -1234) 2)
(num-test (lcm -2 -1234) 1234)
(num-test (gcd 2 1234000000) 2)
(num-test (lcm 2 1234000000) 1234000000)
(num-test (gcd -2 1234000000) 2)
(num-test (lcm -2 1234000000) 1234000000)
(num-test (gcd 2 -1234000000) 2)
(num-test (lcm 2 -1234000000) 1234000000)
(num-test (gcd -2 -1234000000) 2)
(num-test (lcm -2 -1234000000) 1234000000)
(num-test (gcd 2 500029) 1)
(num-test (lcm 2 500029) 1000058)
(num-test (gcd -2 500029) 1)
(num-test (lcm -2 500029) 1000058)
(num-test (gcd 2 -500029) 1)
(num-test (lcm 2 -500029) 1000058)
(num-test (gcd -2 -500029) 1)
(num-test (lcm -2 -500029) 1000058)
(num-test (gcd 2 362880) 2)
(num-test (lcm 2 362880) 362880)
(num-test (gcd -2 362880) 2)
(num-test (lcm -2 362880) 362880)
(num-test (gcd 2 -362880) 2)
(num-test (lcm 2 -362880) 362880)
(num-test (gcd -2 -362880) 2)
(num-test (lcm -2 -362880) 362880)
(num-test (gcd 3 0) 3)
(num-test (lcm 3 0) 0)
(num-test (gcd -3 0) 3)
(num-test (lcm -3 0) 0)
(num-test (gcd 3 0) 3)
(num-test (lcm 3 0) 0)
(num-test (gcd -3 0) 3)
(num-test (lcm -3 0) 0)
(num-test (gcd 3 1) 1)
(num-test (lcm 3 1) 3)
(num-test (gcd -3 1) 1)
(num-test (lcm -3 1) 3)
(num-test (gcd 3 -1) 1)
(num-test (lcm 3 -1) 3)
(num-test (gcd -3 -1) 1)
(num-test (lcm -3 -1) 3)
(num-test (gcd 3 2) 1)
(num-test (lcm 3 2) 6)
(num-test (gcd -3 2) 1)
(num-test (lcm -3 2) 6)
(num-test (gcd 3 -2) 1)
(num-test (lcm 3 -2) 6)
(num-test (gcd -3 -2) 1)
(num-test (lcm -3 -2) 6)
(num-test (gcd 3 3) 3)
(num-test (lcm 3 3) 3)
(num-test (gcd -3 3) 3)
(num-test (lcm -3 3) 3)
(num-test (gcd 3 -3) 3)
(num-test (lcm 3 -3) 3)
(num-test (gcd -3 -3) 3)
(num-test (lcm -3 -3) 3)
(num-test (gcd 3 10) 1)
(num-test (lcm 3 10) 30)
(num-test (gcd -3 10) 1)
(num-test (lcm -3 10) 30)
(num-test (gcd 3 -10) 1)
(num-test (lcm 3 -10) 30)
(num-test (gcd -3 -10) 1)
(num-test (lcm -3 -10) 30)
(num-test (gcd 3 1234) 1)
(num-test (lcm 3 1234) 3702)
(num-test (gcd -3 1234) 1)
(num-test (lcm -3 1234) 3702)
(num-test (gcd 3 -1234) 1)
(num-test (lcm 3 -1234) 3702)
(num-test (gcd -3 -1234) 1)
(num-test (lcm -3 -1234) 3702)
(num-test (gcd 3 1234000000) 1)
(num-test (lcm 3 1234000000) 3702000000)
(num-test (gcd -3 1234000000) 1)
(num-test (lcm -3 1234000000) 3702000000)
(num-test (gcd 3 -1234000000) 1)
(num-test (lcm 3 -1234000000) 3702000000)
(num-test (gcd -3 -1234000000) 1)
(num-test (lcm -3 -1234000000) 3702000000)
(num-test (gcd 3 500029) 1)
(num-test (lcm 3 500029) 1500087)
(num-test (gcd -3 500029) 1)
(num-test (lcm -3 500029) 1500087)
(num-test (gcd 3 -500029) 1)
(num-test (lcm 3 -500029) 1500087)
(num-test (gcd -3 -500029) 1)
(num-test (lcm -3 -500029) 1500087)
(num-test (gcd 3 362880) 3)
(num-test (lcm 3 362880) 362880)
(num-test (gcd -3 362880) 3)
(num-test (lcm -3 362880) 362880)
(num-test (gcd 3 -362880) 3)
(num-test (lcm 3 -362880) 362880)
(num-test (gcd -3 -362880) 3)
(num-test (lcm -3 -362880) 362880)
(num-test (gcd 10 0) 10)
(num-test (lcm 10 0) 0)
(num-test (gcd -10 0) 10)
(num-test (lcm -10 0) 0)
(num-test (gcd 10 0) 10)
(num-test (lcm 10 0) 0)
(num-test (gcd -10 0) 10)
(num-test (lcm -10 0) 0)
(num-test (gcd 10 1) 1)
(num-test (lcm 10 1) 10)
(num-test (gcd -10 1) 1)
(num-test (lcm -10 1) 10)
(num-test (gcd 10 -1) 1)
(num-test (lcm 10 -1) 10)
(num-test (gcd -10 -1) 1)
(num-test (lcm -10 -1) 10)
(num-test (gcd 10 2) 2)
(num-test (lcm 10 2) 10)
(num-test (gcd -10 2) 2)
(num-test (lcm -10 2) 10)
(num-test (gcd 10 -2) 2)
(num-test (lcm 10 -2) 10)
(num-test (gcd -10 -2) 2)
(num-test (lcm -10 -2) 10)
(num-test (gcd 10 3) 1)
(num-test (lcm 10 3) 30)
(num-test (gcd -10 3) 1)
(num-test (lcm -10 3) 30)
(num-test (gcd 10 -3) 1)
(num-test (lcm 10 -3) 30)
(num-test (gcd -10 -3) 1)
(num-test (lcm -10 -3) 30)
(num-test (gcd 10 10) 10)
(num-test (lcm 10 10) 10)
(num-test (gcd -10 10) 10)
(num-test (lcm -10 10) 10)
(num-test (gcd 10 -10) 10)
(num-test (lcm 10 -10) 10)
(num-test (gcd -10 -10) 10)
(num-test (lcm -10 -10) 10)
(num-test (gcd 10 1234) 2)
(num-test (lcm 10 1234) 6170)
(num-test (gcd -10 1234) 2)
(num-test (lcm -10 1234) 6170)
(num-test (gcd 10 -1234) 2)
(num-test (lcm 10 -1234) 6170)
(num-test (gcd -10 -1234) 2)
(num-test (lcm -10 -1234) 6170)
(num-test (gcd 10 1234000000) 10)
(num-test (lcm 10 1234000000) 1234000000)
(num-test (gcd -10 1234000000) 10)
(num-test (lcm -10 1234000000) 1234000000)
(num-test (gcd 10 -1234000000) 10)
(num-test (lcm 10 -1234000000) 1234000000)
(num-test (gcd -10 -1234000000) 10)
(num-test (lcm -10 -1234000000) 1234000000)
(num-test (gcd 10 500029) 1)
(num-test (lcm 10 500029) 5000290)
(num-test (gcd -10 500029) 1)
(num-test (lcm -10 500029) 5000290)
(num-test (gcd 10 -500029) 1)
(num-test (lcm 10 -500029) 5000290)
(num-test (gcd -10 -500029) 1)
(num-test (lcm -10 -500029) 5000290)
(num-test (gcd 10 362880) 10)
(num-test (lcm 10 362880) 362880)
(num-test (gcd -10 362880) 10)
(num-test (lcm -10 362880) 362880)
(num-test (gcd 10 -362880) 10)
(num-test (lcm 10 -362880) 362880)
(num-test (gcd -10 -362880) 10)
(num-test (lcm -10 -362880) 362880)
(num-test (gcd 1234 0) 1234)
(num-test (lcm 1234 0) 0)
(num-test (gcd -1234 0) 1234)
(num-test (lcm -1234 0) 0)
(num-test (gcd 1234 0) 1234)
(num-test (lcm 1234 0) 0)
(num-test (gcd -1234 0) 1234)
(num-test (lcm -1234 0) 0)
(num-test (gcd 1234 1) 1)
(num-test (lcm 1234 1) 1234)
(num-test (gcd -1234 1) 1)
(num-test (lcm -1234 1) 1234)
(num-test (gcd 1234 -1) 1)
(num-test (lcm 1234 -1) 1234)
(num-test (gcd -1234 -1) 1)
(num-test (lcm -1234 -1) 1234)
(num-test (gcd 1234 2) 2)
(num-test (lcm 1234 2) 1234)
(num-test (gcd -1234 2) 2)
(num-test (lcm -1234 2) 1234)
(num-test (gcd 1234 -2) 2)
(num-test (lcm 1234 -2) 1234)
(num-test (gcd -1234 -2) 2)
(num-test (lcm -1234 -2) 1234)
(num-test (gcd 1234 3) 1)
(num-test (lcm 1234 3) 3702)
(num-test (gcd -1234 3) 1)
(num-test (lcm -1234 3) 3702)
(num-test (gcd 1234 -3) 1)
(num-test (lcm 1234 -3) 3702)
(num-test (gcd -1234 -3) 1)
(num-test (lcm -1234 -3) 3702)
(num-test (gcd 1234 10) 2)
(num-test (lcm 1234 10) 6170)
(num-test (gcd -1234 10) 2)
(num-test (lcm -1234 10) 6170)
(num-test (gcd 1234 -10) 2)
(num-test (lcm 1234 -10) 6170)
(num-test (gcd -1234 -10) 2)
(num-test (lcm -1234 -10) 6170)
(num-test (gcd 1234 1234) 1234)
(num-test (lcm 1234 1234) 1234)
(num-test (gcd -1234 1234) 1234)
(num-test (lcm -1234 1234) 1234)
(num-test (gcd 1234 -1234) 1234)
(num-test (lcm 1234 -1234) 1234)
(num-test (gcd -1234 -1234) 1234)
(num-test (lcm -1234 -1234) 1234)
(num-test (gcd 1234 1234000000) 1234)
(num-test (lcm 1234 1234000000) 1234000000)
(num-test (gcd -1234 1234000000) 1234)
(num-test (lcm -1234 1234000000) 1234000000)
(num-test (gcd 1234 -1234000000) 1234)
(num-test (lcm 1234 -1234000000) 1234000000)
(num-test (gcd -1234 -1234000000) 1234)
(num-test (lcm -1234 -1234000000) 1234000000)
(num-test (gcd 1234 500029) 1)
(num-test (lcm 1234 500029) 617035786)
(num-test (gcd -1234 500029) 1)
(num-test (lcm -1234 500029) 617035786)
(num-test (gcd 1234 -500029) 1)
(num-test (lcm 1234 -500029) 617035786)
(num-test (gcd -1234 -500029) 1)
(num-test (lcm -1234 -500029) 617035786)
(num-test (gcd 1234 362880) 2)
(num-test (lcm 1234 362880) 223896960)
(num-test (gcd -1234 362880) 2)
(num-test (lcm -1234 362880) 223896960)
(num-test (gcd 1234 -362880) 2)
(num-test (lcm 1234 -362880) 223896960)
(num-test (gcd -1234 -362880) 2)
(num-test (lcm -1234 -362880) 223896960)
(num-test (gcd 1234000000 0) 1234000000)
(num-test (lcm 1234000000 0) 0)
(num-test (gcd -1234000000 0) 1234000000)
(num-test (lcm -1234000000 0) 0)
(num-test (gcd 1234000000 0) 1234000000)
(num-test (lcm 1234000000 0) 0)
(num-test (gcd -1234000000 0) 1234000000)
(num-test (lcm -1234000000 0) 0)
(num-test (gcd 1234000000 1) 1)
(num-test (lcm 1234000000 1) 1234000000)
(num-test (gcd -1234000000 1) 1)
(num-test (lcm -1234000000 1) 1234000000)
(num-test (gcd 1234000000 -1) 1)
(num-test (lcm 1234000000 -1) 1234000000)
(num-test (gcd -1234000000 -1) 1)
(num-test (lcm -1234000000 -1) 1234000000)
(num-test (gcd 1234000000 2) 2)
(num-test (lcm 1234000000 2) 1234000000)
(num-test (gcd -1234000000 2) 2)
(num-test (lcm -1234000000 2) 1234000000)
(num-test (gcd 1234000000 -2) 2)
(num-test (lcm 1234000000 -2) 1234000000)
(num-test (gcd -1234000000 -2) 2)
(num-test (lcm -1234000000 -2) 1234000000)
(num-test (gcd 1234000000 3) 1)
(num-test (lcm 1234000000 3) 3702000000)
(num-test (gcd -1234000000 3) 1)
(num-test (lcm -1234000000 3) 3702000000)
(num-test (gcd 1234000000 -3) 1)
(num-test (lcm 1234000000 -3) 3702000000)
(num-test (gcd -1234000000 -3) 1)
(num-test (lcm -1234000000 -3) 3702000000)
(num-test (gcd 1234000000 10) 10)
(num-test (lcm 1234000000 10) 1234000000)
(num-test (gcd -1234000000 10) 10)
(num-test (lcm -1234000000 10) 1234000000)
(num-test (gcd 1234000000 -10) 10)
(num-test (lcm 1234000000 -10) 1234000000)
(num-test (gcd -1234000000 -10) 10)
(num-test (lcm -1234000000 -10) 1234000000)
(num-test (gcd 1234000000 1234) 1234)
(num-test (lcm 1234000000 1234) 1234000000)
(num-test (gcd -1234000000 1234) 1234)
(num-test (lcm -1234000000 1234) 1234000000)
(num-test (gcd 1234000000 -1234) 1234)
(num-test (lcm 1234000000 -1234) 1234000000)
(num-test (gcd -1234000000 -1234) 1234)
(num-test (lcm -1234000000 -1234) 1234000000)
(num-test (gcd 1234000000 1234000000) 1234000000)
(num-test (lcm 1234000000 1234000000) 1234000000)
(num-test (gcd -1234000000 1234000000) 1234000000)
(num-test (lcm -1234000000 1234000000) 1234000000)
(num-test (gcd 1234000000 -1234000000) 1234000000)
(num-test (lcm 1234000000 -1234000000) 1234000000)
(num-test (gcd -1234000000 -1234000000) 1234000000)
(num-test (lcm -1234000000 -1234000000) 1234000000)
(num-test (gcd 1234000000 500029) 1)
(num-test (lcm 1234000000 500029) 617035786000000)
(num-test (gcd -1234000000 500029) 1)
(num-test (lcm -1234000000 500029) 617035786000000)
(num-test (gcd 1234000000 -500029) 1)
(num-test (lcm 1234000000 -500029) 617035786000000)
(num-test (gcd -1234000000 -500029) 1)
(num-test (lcm -1234000000 -500029) 617035786000000)
(num-test (gcd 1234000000 362880) 640)
(num-test (lcm 1234000000 362880) 699678000000)
(num-test (gcd -1234000000 362880) 640)
(num-test (lcm -1234000000 362880) 699678000000)
(num-test (gcd 1234000000 -362880) 640)
(num-test (lcm 1234000000 -362880) 699678000000)
(num-test (gcd -1234000000 -362880) 640)
(num-test (lcm -1234000000 -362880) 699678000000)
(num-test (gcd 500029 0) 500029)
(num-test (lcm 500029 0) 0)
(num-test (gcd -500029 0) 500029)
(num-test (lcm -500029 0) 0)
(num-test (gcd 500029 0) 500029)
(num-test (lcm 500029 0) 0)
(num-test (gcd -500029 0) 500029)
(num-test (lcm -500029 0) 0)
(num-test (gcd 500029 1) 1)
(num-test (lcm 500029 1) 500029)
(num-test (gcd -500029 1) 1)
(num-test (lcm -500029 1) 500029)
(num-test (gcd 500029 -1) 1)
(num-test (lcm 500029 -1) 500029)
(num-test (gcd -500029 -1) 1)
(num-test (lcm -500029 -1) 500029)
(num-test (gcd 500029 2) 1)
(num-test (lcm 500029 2) 1000058)
(num-test (gcd -500029 2) 1)
(num-test (lcm -500029 2) 1000058)
(num-test (gcd 500029 -2) 1)
(num-test (lcm 500029 -2) 1000058)
(num-test (gcd -500029 -2) 1)
(num-test (lcm -500029 -2) 1000058)
(num-test (gcd 500029 3) 1)
(num-test (lcm 500029 3) 1500087)
(num-test (gcd -500029 3) 1)
(num-test (lcm -500029 3) 1500087)
(num-test (gcd 500029 -3) 1)
(num-test (lcm 500029 -3) 1500087)
(num-test (gcd -500029 -3) 1)
(num-test (lcm -500029 -3) 1500087)
(num-test (gcd 500029 10) 1)
(num-test (lcm 500029 10) 5000290)
(num-test (gcd -500029 10) 1)
(num-test (lcm -500029 10) 5000290)
(num-test (gcd 500029 -10) 1)
(num-test (lcm 500029 -10) 5000290)
(num-test (gcd -500029 -10) 1)
(num-test (lcm -500029 -10) 5000290)
(num-test (gcd 500029 1234) 1)
(num-test (lcm 500029 1234) 617035786)
(num-test (gcd -500029 1234) 1)
(num-test (lcm -500029 1234) 617035786)
(num-test (gcd 500029 -1234) 1)
(num-test (lcm 500029 -1234) 617035786)
(num-test (gcd -500029 -1234) 1)
(num-test (lcm -500029 -1234) 617035786)
(num-test (gcd 500029 1234000000) 1)
(num-test (lcm 500029 1234000000) 617035786000000)
(num-test (gcd -500029 1234000000) 1)
(num-test (lcm -500029 1234000000) 617035786000000)
(num-test (gcd 500029 -1234000000) 1)
(num-test (lcm 500029 -1234000000) 617035786000000)
(num-test (gcd -500029 -1234000000) 1)
(num-test (lcm -500029 -1234000000) 617035786000000)
(num-test (gcd 500029 500029) 500029)
(num-test (lcm 500029 500029) 500029)
(num-test (gcd -500029 500029) 500029)
(num-test (lcm -500029 500029) 500029)
(num-test (gcd 500029 -500029) 500029)
(num-test (lcm 500029 -500029) 500029)
(num-test (gcd -500029 -500029) 500029)
(num-test (lcm -500029 -500029) 500029)
(num-test (gcd 500029 362880) 1)
(num-test (lcm 500029 362880) 181450523520)
(num-test (gcd -500029 362880) 1)
(num-test (lcm -500029 362880) 181450523520)
(num-test (gcd 500029 -362880) 1)
(num-test (lcm 500029 -362880) 181450523520)
(num-test (gcd -500029 -362880) 1)
(num-test (lcm -500029 -362880) 181450523520)
(num-test (gcd 362880 0) 362880)
(num-test (lcm 362880 0) 0)
(num-test (gcd -362880 0) 362880)
(num-test (lcm -362880 0) 0)
(num-test (gcd 362880 0) 362880)
(num-test (lcm 362880 0) 0)
(num-test (gcd -362880 0) 362880)
(num-test (lcm -362880 0) 0)
(num-test (gcd 362880 1) 1)
(num-test (lcm 362880 1) 362880)
(num-test (gcd -362880 1) 1)
(num-test (lcm -362880 1) 362880)
(num-test (gcd 362880 -1) 1)
(num-test (lcm 362880 -1) 362880)
(num-test (gcd -362880 -1) 1)
(num-test (lcm -362880 -1) 362880)
(num-test (gcd 362880 2) 2)
(num-test (lcm 362880 2) 362880)
(num-test (gcd -362880 2) 2)
(num-test (lcm -362880 2) 362880)
(num-test (gcd 362880 -2) 2)
(num-test (lcm 362880 -2) 362880)
(num-test (gcd -362880 -2) 2)
(num-test (lcm -362880 -2) 362880)
(num-test (gcd 362880 3) 3)
(num-test (lcm 362880 3) 362880)
(num-test (gcd -362880 3) 3)
(num-test (lcm -362880 3) 362880)
(num-test (gcd 362880 -3) 3)
(num-test (lcm 362880 -3) 362880)
(num-test (gcd -362880 -3) 3)
(num-test (lcm -362880 -3) 362880)
(num-test (gcd 362880 10) 10)
(num-test (lcm 362880 10) 362880)
(num-test (gcd -362880 10) 10)
(num-test (lcm -362880 10) 362880)
(num-test (gcd 362880 -10) 10)
(num-test (lcm 362880 -10) 362880)
(num-test (gcd -362880 -10) 10)
(num-test (lcm -362880 -10) 362880)
(num-test (gcd 362880 1234) 2)
(num-test (lcm 362880 1234) 223896960)
(num-test (gcd -362880 1234) 2)
(num-test (lcm -362880 1234) 223896960)
(num-test (gcd 362880 -1234) 2)
(num-test (lcm 362880 -1234) 223896960)
(num-test (gcd -362880 -1234) 2)
(num-test (lcm -362880 -1234) 223896960)
(num-test (gcd 362880 1234000000) 640)
(num-test (lcm 362880 1234000000) 699678000000)
(num-test (gcd -362880 1234000000) 640)
(num-test (lcm -362880 1234000000) 699678000000)
(num-test (gcd 362880 -1234000000) 640)
(num-test (lcm 362880 -1234000000) 699678000000)
(num-test (gcd -362880 -1234000000) 640)
(num-test (lcm -362880 -1234000000) 699678000000)
(num-test (gcd 362880 500029) 1)
(num-test (lcm 362880 500029) 181450523520)
(num-test (gcd -362880 500029) 1)
(num-test (lcm -362880 500029) 181450523520)
(num-test (gcd 362880 -500029) 1)
(num-test (lcm 362880 -500029) 181450523520)
(num-test (gcd -362880 -500029) 1)
(num-test (lcm -362880 -500029) 181450523520)
(num-test (gcd 362880 362880) 362880)
(num-test (lcm 362880 362880) 362880)
(num-test (gcd -362880 362880) 362880)
(num-test (lcm -362880 362880) 362880)
(num-test (gcd 362880 -362880) 362880)
(num-test (lcm 362880 -362880) 362880)
(num-test (gcd -362880 -362880) 362880)
(num-test (lcm -362880 -362880) 362880)
(num-test (gcd 60 42) 6)
(num-test (gcd 3333 -33 101) 1)
(num-test (gcd 3333 -33 1002001) 11)
(num-test (gcd 91 -49) 7)
(num-test (gcd 63 -42 35) 7)
(num-test (gcd 0 0 0 10) 10)
(num-test (lcm 0 0 0 10) 0)
(num-test (gcd 0 1 -1 362880) 1)
(num-test (gcd 0 2 2 -500029) 1)
(num-test (gcd 0 3 -3 -1234000000) 1)
(num-test (gcd 0 10 10 1234) 2)
(num-test (lcm 0 10 10 1234) 0)
(num-test (gcd 0 1234 -1234 10) 2)
(num-test (lcm 0 1234 -1234 10) 0)
(num-test (gcd 0 1234000000 1234000000 -3) 1)
(num-test (gcd 0 500029 -500029 -2) 1)
(num-test (gcd 0 362880 362880 1) 1)
(num-test (gcd 1 0 -1 1) 1)
(num-test (lcm 1 0 -1 1) 0)
(num-test (gcd 1 1 2 -10) 1)
(num-test (lcm 1 1 2 -10) 10)
(num-test (gcd 1 2 -3 -362880) 1)
(num-test (gcd 1 3 10 500029) 1)
(num-test (gcd 1 10 -1234 1234000000) 1)
(num-test (gcd 1 1234 1234000000 -1234) 1)
(num-test (gcd 1 1234000000 -500029 -10) 1)
(num-test (gcd 1 500029 362880 3) 1)
(num-test (gcd 1 362880 0 2) 1)
(num-test (gcd 2 0 2 -2) 2)
(num-test (lcm 2 0 2 -2) 0)
(num-test (gcd 2 1 -3 -1) 1)
(num-test (lcm 2 1 -3 -1) 6)
(num-test (gcd 2 2 10 10) 2)
(num-test (lcm 2 2 10 10) 10)
(num-test (gcd 2 3 -1234 362880) 1)
(num-test (gcd 2 10 1234000000 -500029) 1)
(num-test (gcd 2 1234 -500029 -1234000000) 1)
(num-test (gcd 2 1234000000 362880 1234) 2)
(num-test (gcd 2 500029 0 10) 1)
(num-test (gcd 2 362880 1 -3) 1)
(num-test (gcd 3 0 -3 -3) 3)
(num-test (lcm 3 0 -3 -3) 0)
(num-test (gcd 3 1 10 2) 1)
(num-test (lcm 3 1 10 2) 30)
(num-test (gcd 3 2 -1234 1) 1)
(num-test (lcm 3 2 -1234 1) 3702)
(num-test (gcd 3 3 1234000000 -10) 1)
(num-test (gcd 3 10 -500029 -362880) 1)
(num-test (gcd 3 1234 362880 500029) 1)
(num-test (gcd 3 1234000000 0 1234000000) 1)
(num-test (gcd 3 500029 1 -1234) 1)
(num-test (gcd 3 362880 -2 -10) 1)
(num-test (gcd 10 0 10 10) 10)
(num-test (lcm 10 0 10 10) 0)
(num-test (gcd 10 1 -1234 3) 1)
(num-test (lcm 10 1 -1234 3) 18510)
(num-test (gcd 10 2 1234000000 -2) 2)
(num-test (gcd 10 3 -500029 -1) 1)
(num-test (gcd 10 10 362880 10) 10)
(num-test (gcd 10 1234 0 362880) 2)
(num-test (gcd 10 1234000000 1 -500029) 1)
(num-test (gcd 10 500029 -2 -1234000000) 1)
(num-test (gcd 10 362880 3 1234) 1)
(num-test (gcd 1234 0 -1234 1234) 1234)
(num-test (lcm 1234 0 -1234 1234) 0)
(num-test (gcd 1234 1 1234000000 -10) 1)
(num-test (gcd 1234 2 -500029 -3) 1)
(num-test (gcd 1234 3 362880 2) 1)
(num-test (gcd 1234 10 0 1) 1)
(num-test (lcm 1234 10 0 1) 0)
(num-test (gcd 1234 1234 1 -10) 1)
(num-test (lcm 1234 1234 1 -10) 6170)
(num-test (gcd 1234 1234000000 -2 -362880) 2)
(num-test (gcd 1234 500029 3 500029) 1)
(num-test (gcd 1234 362880 -10 1234000000) 2)
(num-test (gcd 1234000000 0 1234000000 -1234000000) 1234000000)
(num-test (gcd 1234000000 1 -500029 -1234) 1)
(num-test (gcd 1234000000 2 362880 10) 2)
(num-test (gcd 1234000000 3 0 3) 1)
(num-test (gcd 1234000000 10 1 -2) 1)
(num-test (gcd 1234000000 1234 -2 -1) 1)
(num-test (gcd 1234000000 1234000000 3 10) 1)
(num-test (gcd 1234000000 500029 -10 362880) 1)
(num-test (gcd 1234000000 362880 1234 -500029) 1)
(num-test (gcd 500029 0 -500029 -500029) 500029)
(num-test (gcd 500029 1 362880 1234000000) 1)
(num-test (gcd 500029 2 0 1234) 1)
(num-test (gcd 500029 3 1 -10) 1)
(num-test (gcd 500029 10 -2 -3) 1)
(num-test (gcd 500029 1234 3 2) 1)
(num-test (gcd 500029 1234000000 -10 1) 1)
(num-test (gcd 500029 500029 1234 -10) 1)
(num-test (gcd 500029 362880 -1234000000 -362880) 1)
(num-test (gcd 362880 0 362880 362880) 362880)
(num-test (gcd 362880 1 0 500029) 1)
(num-test (gcd 362880 2 1 -1234000000) 1)
(num-test (gcd 362880 3 -2 -1234) 1)
(num-test (gcd 362880 10 3 10) 1)
(num-test (gcd 362880 1234 -10 3) 1)
(num-test (gcd 362880 1234000000 1234 -2) 2)
(num-test (gcd 362880 500029 -1234000000 -1) 1)
(num-test (gcd 362880 362880 500029 10) 1)
(num-test (gcd 323 28747 27113) 19)
(num-test (lcm 323 28747 27113) 41021969)
(num-test (lcm (* 512 500057) (* 128 500057) (* 2048 500057)) 1024116736)
(num-test (gcd (* 512 500057) (* 128 500057) (* 2048 500057)) 64007296)
(num-test (gcd 91 -49) 7)
(num-test (lcm 14 35) 70)
(num-test (gcd (- (expt 2 11) 1) (- (expt 2 19) 1)) (- (expt 2 (gcd 11 19)) 1))
(num-test (gcd (- (expt 2 11) 1) (- (expt 2 22) 1)) (- (expt 2 (gcd 11 22)) 1))
(num-test (gcd (- (expt 2 12) 1) (- (expt 2 18) 1)) (- (expt 2 (gcd 12 18)) 1))
(if (or with-bignums with-64-bit-ints) 
    (num-test (gcd (- (expt 2 52) 1) (- (expt 2 39) 1)) (- (expt 2 (gcd 52 39)) 1)))
(num-test (gcd (numerator 7/9) (denominator 7/9)) 1)
(num-test (gcd 0 4) 4 )
(num-test (gcd -4 0) 4 )
(num-test (gcd 32 -36) 4 )
(num-test (gcd) 0 )
(num-test (lcm 32 -36) 288 )
(num-test (lcm) 1 )
(num-test (gcd 2 0) 2)
(num-test (lcm 2 0) 0)


;; -------- real-part and imag-part
(num-test (real-part 1) 1)
(num-test (imag-part 1) 0.0)
(num-test (real-part 2.0) 2.0)
(num-test (imag-part -2.0) 0.0)
(num-test (real-part 2/3) 2/3)
(num-test (imag-part 1+i) 1.0)
(num-test (imag-part 0+i) 1.0)
(num-test (imag-part 1-i) -1.0)
(num-test (imag-part 2/3) 0.0)
(num-test (real-part 0.0+0.0i) 0.0)
(num-test (imag-part 0.0+0.0i) 0.0)
(num-test (real-part -0.0+0.00000001i) -0.0)
(num-test (imag-part -0.0+0.00000001i) 0.00000001)
(num-test (real-part 0.0-1.0i) 0.0)
(num-test (imag-part 0.0-1.0i) -1.0)
(num-test (real-part -0.0-3.14159265358979i) -0.0)
(num-test (imag-part -0.0-3.14159265358979i) -3.14159265358979)
(num-test (real-part 0.0+2.71828182845905i) 0.0)
(num-test (imag-part 0.0+2.71828182845905i) 2.71828182845905)
(num-test (real-part -0.0+1234.0i) -0.0)
(num-test (imag-part -0.0+1234.0i) 1234.0)
(num-test (real-part 0.0-1234000000.0i) 0.0)
(num-test (imag-part 0.0-1234000000.0i) -1234000000.0)
(num-test (real-part -0.00000001-0.0i) -0.00000001)
(num-test (imag-part -0.00000001-0.0i) -0.0)
(num-test (real-part 0.00000001+0.00000001i) 0.00000001)
(num-test (imag-part 0.00000001+0.00000001i) 0.00000001)
(num-test (real-part -0.00000001+1.0i) -0.00000001)
(num-test (imag-part -0.00000001+1.0i) 1.0)
(num-test (real-part 0.00000001-3.14159265358979i) 0.00000001)
(num-test (imag-part 0.00000001-3.14159265358979i) -3.14159265358979)
(num-test (real-part -0.00000001-2.71828182845905i) -0.00000001)
(num-test (imag-part -0.00000001-2.71828182845905i) -2.71828182845905)
(num-test (real-part 0.00000001+1234.0i) 0.00000001)
(num-test (imag-part 0.00000001+1234.0i) 1234.0)
(num-test (real-part -0.00000001+1234000000.0i) -0.00000001)
(num-test (imag-part -0.00000001+1234000000.0i) 1234000000.0)
(num-test (real-part 1.0-0.0i) 1.0)
(num-test (imag-part 1.0-0.0i) -0.0)
(num-test (real-part -1.0-0.00000001i) -1.0)
(num-test (imag-part -1.0-0.00000001i) -0.00000001)
(num-test (real-part 1.0+1.0i) 1.0)
(num-test (imag-part 1.0+1.0i) 1.0)
(num-test (real-part -1.0+3.14159265358979i) -1.0)
(num-test (imag-part -1.0+3.14159265358979i) 3.14159265358979)
(num-test (real-part 1.0-2.71828182845905i) 1.0)
(num-test (imag-part 1.0-2.71828182845905i) -2.71828182845905)
(num-test (real-part -1.0-1234.0i) -1.0)
(num-test (imag-part -1.0-1234.0i) -1234.0)
(num-test (real-part 1.0+1234000000.0i) 1.0)
(num-test (imag-part 1.0+1234000000.0i) 1234000000.0)
(num-test (real-part -3.14159265358979+0.0i) -3.14159265358979)
(num-test (imag-part -3.14159265358979+0.0i) 0.0)
(num-test (real-part 3.14159265358979-0.00000001i) 3.14159265358979)
(num-test (imag-part 3.14159265358979-0.00000001i) -0.00000001)
(num-test (real-part -3.14159265358979-1.0i) -3.14159265358979)
(num-test (imag-part -3.14159265358979-1.0i) -1.0)
(num-test (real-part 3.14159265358979+3.14159265358979i) 3.14159265358979)
(num-test (imag-part 3.14159265358979+3.14159265358979i) 3.14159265358979)
(num-test (real-part -3.14159265358979+2.71828182845905i) -3.14159265358979)
(num-test (imag-part -3.14159265358979+2.71828182845905i) 2.71828182845905)
(num-test (real-part 3.14159265358979-1234.0i) 3.14159265358979)
(num-test (imag-part 3.14159265358979-1234.0i) -1234.0)
(num-test (real-part -3.14159265358979-1234000000.0i) -3.14159265358979)
(num-test (imag-part -3.14159265358979-1234000000.0i) -1234000000.0)
(num-test (real-part 2.71828182845905+0.0i) 2.71828182845905)
(num-test (imag-part 2.71828182845905+0.0i) 0.0)
(num-test (real-part -2.71828182845905+0.00000001i) -2.71828182845905)
(num-test (imag-part -2.71828182845905+0.00000001i) 0.00000001)
(num-test (real-part 2.71828182845905-1.0i) 2.71828182845905)
(num-test (imag-part 2.71828182845905-1.0i) -1.0)
(num-test (real-part -2.71828182845905-3.14159265358979i) -2.71828182845905)
(num-test (imag-part -2.71828182845905-3.14159265358979i) -3.14159265358979)
(num-test (real-part 2.71828182845905+2.71828182845905i) 2.71828182845905)
(num-test (imag-part 2.71828182845905+2.71828182845905i) 2.71828182845905)
(num-test (real-part -2.71828182845905+1234.0i) -2.71828182845905)
(num-test (imag-part -2.71828182845905+1234.0i) 1234.0)
(num-test (real-part 2.71828182845905-1234000000.0i) 2.71828182845905)
(num-test (imag-part 2.71828182845905-1234000000.0i) -1234000000.0)
(num-test (real-part -1234.0-0.0i) -1234.0)
(num-test (imag-part -1234.0-0.0i) -0.0)
(num-test (real-part 1234.0+0.00000001i) 1234.0)
(num-test (imag-part 1234.0+0.00000001i) 0.00000001)
(num-test (real-part -1234.0+1.0i) -1234.0)
(num-test (imag-part -1234.0+1.0i) 1.0)
(num-test (real-part 1234.0-3.14159265358979i) 1234.0)
(num-test (imag-part 1234.0-3.14159265358979i) -3.14159265358979)
(num-test (real-part -1234.0-2.71828182845905i) -1234.0)
(num-test (imag-part -1234.0-2.71828182845905i) -2.71828182845905)
(num-test (real-part 1234.0+1234.0i) 1234.0)
(num-test (imag-part 1234.0+1234.0i) 1234.0)
(num-test (real-part -1234.0+1234000000.0i) -1234.0)
(num-test (imag-part -1234.0+1234000000.0i) 1234000000.0)
(num-test (real-part 1234000000.0-0.0i) 1234000000.0)
(num-test (imag-part 1234000000.0-0.0i) -0.0)
(num-test (real-part -1234000000.0-0.00000001i) -1234000000.0)
(num-test (imag-part -1234000000.0-0.00000001i) -0.00000001)
(num-test (real-part 1234000000.0+1.0i) 1234000000.0)
(num-test (imag-part 1234000000.0+1.0i) 1.0)
(num-test (real-part -1234000000.0+3.14159265358979i) -1234000000.0)
(num-test (imag-part -1234000000.0+3.14159265358979i) 3.14159265358979)
(num-test (real-part 1234000000.0-2.71828182845905i) 1234000000.0)
(num-test (imag-part 1234000000.0-2.71828182845905i) -2.71828182845905)
(num-test (real-part -1234000000.0-1234.0i) -1234000000.0)
(num-test (imag-part -1234000000.0-1234.0i) -1234.0)
(num-test (real-part 1234000000.0+1234000000.0i) 1234000000.0)
(num-test (imag-part 1234000000.0+1234000000.0i) 1234000000.0)
(num-test (real-part 5) 5)
(num-test (real-part 1.4+0.0i) 1.4)
(num-test (imag-part 5) 0.0)
(num-test (imag-part 1.4+0.0i) 0.0)


;; -------- numerator and denominator
(num-test (numerator 12/6000996) 1)
(num-test (denominator 12/6000996) 500083)
(num-test (numerator 1) 1)
(num-test (numerator 2/3) 2)
(num-test (numerator 2/4) 1)
(num-test (denominator 2/4) 2)
(num-test (numerator -2/6) -1)
(num-test (denominator -2/6) 3)
(num-test (numerator -2/3) -2)
(num-test (denominator -2/3) 3)
(num-test (denominator 1) 1)
(num-test (denominator 2/3) 3)
(num-test (numerator 0/1) 0)
(num-test (denominator 0/1) 1)
(num-test (numerator 0/2) 0)
(num-test (denominator 0/2) 1)
(num-test (numerator 0/3) 0)
(num-test (denominator 0/3) 1)
(num-test (numerator 0/10) 0)
(num-test (denominator 0/10) 1)
(num-test (numerator 0/1234) 0)
(num-test (denominator 0/1234) 1)
(num-test (numerator 0/1234000000) 0)
(num-test (denominator 0/1234000000) 1)
(num-test (numerator 0/500029) 0)
(num-test (denominator 0/500029) 1)
(num-test (numerator 0/362880) 0)
(num-test (denominator 0/362880) 1)
(num-test (numerator 1/1) 1)
(num-test (denominator 1/1) 1)
(num-test (numerator -1/2) -1)
(num-test (denominator -1/2) 2)
(num-test (numerator 1/3) 1)
(num-test (denominator 1/3) 3)
(num-test (numerator -1/10) -1)
(num-test (denominator -1/10) 10)
(num-test (numerator 1/1234) 1)
(num-test (denominator 1/1234) 1234)
(num-test (numerator -1/1234000000) -1)
(num-test (denominator -1/1234000000) 1234000000)
(num-test (numerator 1/500029) 1)
(num-test (denominator 1/500029) 500029)
(num-test (numerator -1/362880) -1)
(num-test (denominator -1/362880) 362880)
(num-test (numerator -2/1) -2)
(num-test (denominator -2/1) 1)
(num-test (numerator 2/2) 1)
(num-test (denominator 2/2) 1)
(num-test (numerator -2/3) -2)
(num-test (denominator -2/3) 3)
(num-test (numerator 2/10) 1)
(num-test (denominator 2/10) 5)
(num-test (numerator -2/1234) -1)
(num-test (denominator -2/1234) 617)
(num-test (numerator 2/1234000000) 1)
(num-test (denominator 2/1234000000) 617000000)
(num-test (numerator -2/500029) -2)
(num-test (denominator -2/500029) 500029)
(num-test (numerator 2/362880) 1)
(num-test (denominator 2/362880) 181440)
(num-test (numerator 3/1) 3)
(num-test (denominator 3/1) 1)
(num-test (numerator -3/2) -3)
(num-test (denominator -3/2) 2)
(num-test (numerator 3/3) 1)
(num-test (denominator 3/3) 1)
(num-test (numerator -3/10) -3)
(num-test (denominator -3/10) 10)
(num-test (numerator 3/1234) 3)
(num-test (denominator 3/1234) 1234)
(num-test (numerator -3/1234000000) -3)
(num-test (denominator -3/1234000000) 1234000000)
(num-test (numerator 3/500029) 3)
(num-test (denominator 3/500029) 500029)
(num-test (numerator -3/362880) -1)
(num-test (denominator -3/362880) 120960)
(num-test (numerator -10/1) -10)
(num-test (denominator -10/1) 1)
(num-test (numerator 10/2) 5)
(num-test (denominator 10/2) 1)
(num-test (numerator -10/3) -10)
(num-test (denominator -10/3) 3)
(num-test (numerator 10/10) 1)
(num-test (denominator 10/10) 1)
(num-test (numerator -10/1234) -5)
(num-test (denominator -10/1234) 617)
(num-test (numerator 10/1234000000) 1)
(num-test (denominator 10/1234000000) 123400000)
(num-test (numerator -10/500029) -10)
(num-test (denominator -10/500029) 500029)
(num-test (numerator 10/362880) 1)
(num-test (denominator 10/362880) 36288)
(num-test (numerator 1234/1) 1234)
(num-test (denominator 1234/1) 1)
(num-test (numerator -1234/2) -617)
(num-test (denominator -1234/2) 1)
(num-test (numerator 1234/3) 1234)
(num-test (denominator 1234/3) 3)
(num-test (numerator -1234/10) -617)
(num-test (denominator -1234/10) 5)
(num-test (numerator 1234/1234) 1)
(num-test (denominator 1234/1234) 1)
(num-test (numerator -1234/1234000000) -1)
(num-test (denominator -1234/1234000000) 1000000)
(num-test (numerator 1234/500029) 1234)
(num-test (denominator 1234/500029) 500029)
(num-test (numerator -1234/362880) -617)
(num-test (denominator -1234/362880) 181440)
(num-test (numerator -1234000000/1) -1234000000)
(num-test (denominator -1234000000/1) 1)
(num-test (numerator 1234000000/2) 617000000)
(num-test (denominator 1234000000/2) 1)
(num-test (numerator -1234000000/3) -1234000000)
(num-test (denominator -1234000000/3) 3)
(num-test (numerator 1234000000/10) 123400000)
(num-test (denominator 1234000000/10) 1)
(num-test (numerator -1234000000/1234) -1000000)
(num-test (denominator -1234000000/1234) 1)
(num-test (numerator 1234000000/1234000000) 1)
(num-test (denominator 1234000000/1234000000) 1)
(num-test (numerator -1234000000/500029) -1234000000)
(num-test (denominator -1234000000/500029) 500029)
(num-test (numerator 1234000000/362880) 1928125)
(num-test (denominator 1234000000/362880) 567)
(num-test (numerator 500029/1) 500029)
(num-test (denominator 500029/1) 1)
(num-test (numerator -500029/2) -500029)
(num-test (denominator -500029/2) 2)
(num-test (numerator 500029/3) 500029)
(num-test (denominator 500029/3) 3)
(num-test (numerator -500029/10) -500029)
(num-test (denominator -500029/10) 10)
(num-test (numerator 500029/1234) 500029)
(num-test (denominator 500029/1234) 1234)
(num-test (numerator -500029/1234000000) -500029)
(num-test (denominator -500029/1234000000) 1234000000)
(num-test (numerator 500029/500029) 1)
(num-test (denominator 500029/500029) 1)
(num-test (numerator -500029/362880) -500029)
(num-test (denominator -500029/362880) 362880)
(num-test (numerator -362880/1) -362880)
(num-test (denominator -362880/1) 1)
(num-test (numerator 362880/2) 181440)
(num-test (denominator 362880/2) 1)
(num-test (numerator -362880/3) -120960)
(num-test (denominator -362880/3) 1)
(num-test (numerator 362880/10) 36288)
(num-test (denominator 362880/10) 1)
(num-test (numerator -362880/1234) -181440)
(num-test (denominator -362880/1234) 617)
(num-test (numerator 362880/1234000000) 567)
(num-test (denominator 362880/1234000000) 1928125)
(num-test (numerator -362880/500029) -362880)
(num-test (denominator -362880/500029) 500029)
(num-test (numerator 362880/362880) 1)
(num-test (denominator 362880/362880) 1)
(num-test (numerator 5/2) 5)
(num-test (numerator (/ 8 -6)) -4)
(num-test (denominator 5/2) 2)
(num-test (denominator (/ 8 -6)) 3)


;; -------- modulo, remainder, quotient
;;; (modulo x 0) -> x?  I seem to be getting errors instead; maxima returns x and refers to Section 3.4, of "Concrete Mathematics," by Graham, Knuth, and Patashnik
;;; (mod x 1.0) = sawtooth
(num-test (modulo 0 1) 0)
(num-test (modulo 3 1) 0)
(num-test (modulo -3 1) 0)
(num-test (modulo 3 -1) 0)
(num-test (remainder 0 1) 0)
(num-test (quotient 0 1) 0)
(num-test (modulo 0 1) 0)
(num-test (remainder 0 1) 0)
(num-test (quotient 0 1) 0)
(num-test (modulo 0 -1) 0)
(num-test (remainder 0 -1) 0)
(num-test (quotient 0 -1) 0)
(num-test (modulo 0 -1) 0)
(num-test (remainder 0 -1) 0)
(num-test (quotient 0 -1) 0)
(num-test (modulo 0 2) 0)
(num-test (remainder 0 2) 0)
(num-test (quotient 0 2) 0)
(num-test (modulo 0 2) 0)
(num-test (remainder 0 2) 0)
(num-test (quotient 0 2) 0)
(num-test (modulo 0 -2) 0)
(num-test (remainder 0 -2) 0)
(num-test (quotient 0 -2) 0)
(num-test (modulo 0 -2) 0)
(num-test (remainder 0 -2) 0)
(num-test (quotient 0 -2) 0)
(num-test (modulo 0 3) 0)
(num-test (remainder 0 3) 0)
(num-test (quotient 0 3) 0)
(num-test (modulo 0 3) 0)
(num-test (remainder 0 3) 0)
(num-test (quotient 0 3) 0)
(num-test (modulo 0 -3) 0)
(num-test (remainder 0 -3) 0)
(num-test (quotient 0 -3) 0)
(num-test (modulo 0 -3) 0)
(num-test (remainder 0 -3) 0)
(num-test (quotient 0 -3) 0)
(num-test (modulo 0 10) 0)
(num-test (remainder 0 10) 0)
(num-test (quotient 0 10) 0)
(num-test (modulo 0 10) 0)
(num-test (remainder 0 10) 0)
(num-test (quotient 0 10) 0)
(num-test (modulo 0 -10) 0)
(num-test (remainder 0 -10) 0)
(num-test (quotient 0 -10) 0)
(num-test (modulo 0 -10) 0)
(num-test (remainder 0 -10) 0)
(num-test (quotient 0 -10) 0)
(num-test (modulo 0 1234) 0)
(num-test (remainder 0 1234) 0)
(num-test (quotient 0 1234) 0)
(num-test (modulo 0 1234) 0)
(num-test (remainder 0 1234) 0)
(num-test (quotient 0 1234) 0)
(num-test (modulo 0 -1234) 0)
(num-test (remainder 0 -1234) 0)
(num-test (quotient 0 -1234) 0)
(num-test (modulo 0 -1234) 0)
(num-test (remainder 0 -1234) 0)
(num-test (quotient 0 -1234) 0)
(num-test (modulo 0 1234000000) 0)
(num-test (remainder 0 1234000000) 0)
(num-test (quotient 0 1234000000) 0)
(num-test (modulo 0 1234000000) 0)
(num-test (remainder 0 1234000000) 0)
(num-test (quotient 0 1234000000) 0)
(num-test (modulo 0 -1234000000) 0)
(num-test (remainder 0 -1234000000) 0)
(num-test (quotient 0 -1234000000) 0)
(num-test (modulo 0 -1234000000) 0)
(num-test (remainder 0 -1234000000) 0)
(num-test (quotient 0 -1234000000) 0)
(num-test (modulo 0 500029) 0)
(num-test (remainder 0 500029) 0)
(num-test (quotient 0 500029) 0)
(num-test (modulo 0 500029) 0)
(num-test (remainder 0 500029) 0)
(num-test (quotient 0 500029) 0)
(num-test (modulo 0 -500029) 0)
(num-test (remainder 0 -500029) 0)
(num-test (quotient 0 -500029) 0)
(num-test (modulo 0 -500029) 0)
(num-test (remainder 0 -500029) 0)
(num-test (quotient 0 -500029) 0)
(num-test (modulo 0 362880) 0)
(num-test (remainder 0 362880) 0)
(num-test (quotient 0 362880) 0)
(num-test (modulo 0 362880) 0)
(num-test (remainder 0 362880) 0)
(num-test (quotient 0 362880) 0)
(num-test (modulo 0 -362880) 0)
(num-test (remainder 0 -362880) 0)
(num-test (quotient 0 -362880) 0)
(num-test (modulo 0 -362880) 0)
(num-test (remainder 0 -362880) 0)
(num-test (quotient 0 -362880) 0)
(num-test (modulo 1 1) 0)
(num-test (remainder 1 1) 0)
(num-test (quotient 1 1) 1)
(num-test (modulo -1 1) 0)
(num-test (remainder -1 1) 0)
(num-test (quotient -1 1) -1)
(num-test (modulo 1 -1) 0)
(num-test (remainder 1 -1) 0)
(num-test (quotient 1 -1) -1)
(num-test (modulo -1 -1) 0)
(num-test (remainder -1 -1) 0)
(num-test (quotient -1 -1) 1)
(num-test (modulo 1 2) 1)
(num-test (remainder 1 2) 1)
(num-test (quotient 1 2) 0)
(num-test (modulo -1 2) 1)
(num-test (remainder -1 2) -1)
(num-test (quotient -1 2) 0)
(num-test (modulo 1 -2) -1)
(num-test (remainder 1 -2) 1)
(num-test (quotient 1 -2) 0)
(num-test (modulo -1 -2) -1)
(num-test (remainder -1 -2) -1)
(num-test (quotient -1 -2) 0)
(num-test (modulo 1 3) 1)
(num-test (remainder 1 3) 1)
(num-test (quotient 1 3) 0)
(num-test (modulo -1 3) 2)
(num-test (remainder -1 3) -1)
(num-test (quotient -1 3) 0)
(num-test (modulo 1 -3) -2)
(num-test (remainder 1 -3) 1)
(num-test (quotient 1 -3) 0)
(num-test (modulo -1 -3) -1)
(num-test (remainder -1 -3) -1)
(num-test (quotient -1 -3) 0)
(num-test (modulo 1 10) 1)
(num-test (remainder 1 10) 1)
(num-test (quotient 1 10) 0)
(num-test (modulo -1 10) 9)
(num-test (remainder -1 10) -1)
(num-test (quotient -1 10) 0)
(num-test (modulo 1 -10) -9)
(num-test (remainder 1 -10) 1)
(num-test (quotient 1 -10) 0)
(num-test (modulo -1 -10) -1)
(num-test (remainder -1 -10) -1)
(num-test (quotient -1 -10) 0)
(num-test (modulo 1 1234) 1)
(num-test (remainder 1 1234) 1)
(num-test (quotient 1 1234) 0)
(num-test (modulo -1 1234) 1233)
(num-test (remainder -1 1234) -1)
(num-test (quotient -1 1234) 0)
(num-test (modulo 1 -1234) -1233)
(num-test (remainder 1 -1234) 1)
(num-test (quotient 1 -1234) 0)
(num-test (modulo -1 -1234) -1)
(num-test (remainder -1 -1234) -1)
(num-test (quotient -1 -1234) 0)
(num-test (modulo 1 1234000000) 1)
(num-test (remainder 1 1234000000) 1)
(num-test (quotient 1 1234000000) 0)
(num-test (modulo -1 1234000000) 1233999999)
(num-test (remainder -1 1234000000) -1)
(num-test (quotient -1 1234000000) 0)
(num-test (modulo 1 -1234000000) -1233999999)
(num-test (remainder 1 -1234000000) 1)
(num-test (quotient 1 -1234000000) 0)
(num-test (modulo -1 -1234000000) -1)
(num-test (remainder -1 -1234000000) -1)
(num-test (quotient -1 -1234000000) 0)
(num-test (modulo 1 500029) 1)
(num-test (remainder 1 500029) 1)
(num-test (quotient 1 500029) 0)
(num-test (modulo -1 500029) 500028)
(num-test (remainder -1 500029) -1)
(num-test (quotient -1 500029) 0)
(num-test (modulo 1 -500029) -500028)
(num-test (remainder 1 -500029) 1)
(num-test (quotient 1 -500029) 0)
(num-test (modulo -1 -500029) -1)
(num-test (remainder -1 -500029) -1)
(num-test (quotient -1 -500029) 0)
(num-test (modulo 1 362880) 1)
(num-test (remainder 1 362880) 1)
(num-test (quotient 1 362880) 0)
(num-test (modulo -1 362880) 362879)
(num-test (remainder -1 362880) -1)
(num-test (quotient -1 362880) 0)
(num-test (modulo 1 -362880) -362879)
(num-test (remainder 1 -362880) 1)
(num-test (quotient 1 -362880) 0)
(num-test (modulo -1 -362880) -1)
(num-test (remainder -1 -362880) -1)
(num-test (quotient -1 -362880) 0)
(num-test (modulo 2 1) 0)
(num-test (remainder 2 1) 0)
(num-test (quotient 2 1) 2)
(num-test (modulo -2 1) 0)
(num-test (remainder -2 1) 0)
(num-test (quotient -2 1) -2)
(num-test (modulo 2 -1) 0)
(num-test (remainder 2 -1) 0)
(num-test (quotient 2 -1) -2)
(num-test (modulo -2 -1) 0)
(num-test (remainder -2 -1) 0)
(num-test (quotient -2 -1) 2)
(num-test (modulo 2 2) 0)
(num-test (remainder 2 2) 0)
(num-test (quotient 2 2) 1)
(num-test (modulo -2 2) 0)
(num-test (remainder -2 2) 0)
(num-test (quotient -2 2) -1)
(num-test (modulo 2 -2) 0)
(num-test (remainder 2 -2) 0)
(num-test (quotient 2 -2) -1)
(num-test (modulo -2 -2) 0)
(num-test (remainder -2 -2) 0)
(num-test (quotient -2 -2) 1)
(num-test (modulo 2 3) 2)
(num-test (remainder 2 3) 2)
(num-test (quotient 2 3) 0)
(num-test (modulo -2 3) 1)
(num-test (remainder -2 3) -2)
(num-test (quotient -2 3) 0)
(num-test (modulo 2 -3) -1)
(num-test (remainder 2 -3) 2)
(num-test (quotient 2 -3) 0)
(num-test (modulo -2 -3) -2)
(num-test (remainder -2 -3) -2)
(num-test (quotient -2 -3) 0)
(num-test (modulo 2 10) 2)
(num-test (remainder 2 10) 2)
(num-test (quotient 2 10) 0)
(num-test (modulo -2 10) 8)
(num-test (remainder -2 10) -2)
(num-test (quotient -2 10) 0)
(num-test (modulo 2 -10) -8)
(num-test (remainder 2 -10) 2)
(num-test (quotient 2 -10) 0)
(num-test (modulo -2 -10) -2)
(num-test (remainder -2 -10) -2)
(num-test (quotient -2 -10) 0)
(num-test (modulo 2 1234) 2)
(num-test (remainder 2 1234) 2)
(num-test (quotient 2 1234) 0)
(num-test (modulo -2 1234) 1232)
(num-test (remainder -2 1234) -2)
(num-test (quotient -2 1234) 0)
(num-test (modulo 2 -1234) -1232)
(num-test (remainder 2 -1234) 2)
(num-test (quotient 2 -1234) 0)
(num-test (modulo -2 -1234) -2)
(num-test (remainder -2 -1234) -2)
(num-test (quotient -2 -1234) 0)
(num-test (modulo 2 1234000000) 2)
(num-test (remainder 2 1234000000) 2)
(num-test (quotient 2 1234000000) 0)
(num-test (modulo -2 1234000000) 1233999998)
(num-test (remainder -2 1234000000) -2)
(num-test (quotient -2 1234000000) 0)
(num-test (modulo 2 -1234000000) -1233999998)
(num-test (remainder 2 -1234000000) 2)
(num-test (quotient 2 -1234000000) 0)
(num-test (modulo -2 -1234000000) -2)
(num-test (remainder -2 -1234000000) -2)
(num-test (quotient -2 -1234000000) 0)
(num-test (modulo 2 500029) 2)
(num-test (remainder 2 500029) 2)
(num-test (quotient 2 500029) 0)
(num-test (modulo -2 500029) 500027)
(num-test (remainder -2 500029) -2)
(num-test (quotient -2 500029) 0)
(num-test (modulo 2 -500029) -500027)
(num-test (remainder 2 -500029) 2)
(num-test (quotient 2 -500029) 0)
(num-test (modulo -2 -500029) -2)
(num-test (remainder -2 -500029) -2)
(num-test (quotient -2 -500029) 0)
(num-test (modulo 2 362880) 2)
(num-test (remainder 2 362880) 2)
(num-test (quotient 2 362880) 0)
(num-test (modulo -2 362880) 362878)
(num-test (remainder -2 362880) -2)
(num-test (quotient -2 362880) 0)
(num-test (modulo 2 -362880) -362878)
(num-test (remainder 2 -362880) 2)
(num-test (quotient 2 -362880) 0)
(num-test (modulo -2 -362880) -2)
(num-test (remainder -2 -362880) -2)
(num-test (quotient -2 -362880) 0)
(num-test (modulo 3 1) 0)
(num-test (remainder 3 1) 0)
(num-test (quotient 3 1) 3)
(num-test (modulo -3 1) 0)
(num-test (remainder -3 1) 0)
(num-test (quotient -3 1) -3)
(num-test (modulo 3 -1) 0)
(num-test (remainder 3 -1) 0)
(num-test (quotient 3 -1) -3)
(num-test (modulo -3 -1) 0)
(num-test (remainder -3 -1) 0)
(num-test (quotient -3 -1) 3)
(num-test (modulo 3 2) 1)
(num-test (remainder 3 2) 1)
(num-test (quotient 3 2) 1)
(num-test (modulo -3 2) 1)
(num-test (remainder -3 2) -1)
(num-test (quotient -3 2) -1)
(num-test (modulo 3 -2) -1)
(num-test (remainder 3 -2) 1)
(num-test (quotient 3 -2) -1)
(num-test (modulo -3 -2) -1)
(num-test (remainder -3 -2) -1)
(num-test (quotient -3 -2) 1)
(num-test (modulo 3 3) 0)
(num-test (remainder 3 3) 0)
(num-test (quotient 3 3) 1)
(num-test (modulo -3 3) 0)
(num-test (remainder -3 3) 0)
(num-test (quotient -3 3) -1)
(num-test (modulo 3 -3) 0)
(num-test (remainder 3 -3) 0)
(num-test (quotient 3 -3) -1)
(num-test (modulo -3 -3) 0)
(num-test (remainder -3 -3) 0)
(num-test (quotient -3 -3) 1)
(num-test (modulo 3 10) 3)
(num-test (remainder 3 10) 3)
(num-test (quotient 3 10) 0)
(num-test (modulo -3 10) 7)
(num-test (remainder -3 10) -3)
(num-test (quotient -3 10) 0)
(num-test (modulo 3 -10) -7)
(num-test (remainder 3 -10) 3)
(num-test (quotient 3 -10) 0)
(num-test (modulo -3 -10) -3)
(num-test (remainder -3 -10) -3)
(num-test (quotient -3 -10) 0)
(num-test (modulo 3 1234) 3)
(num-test (remainder 3 1234) 3)
(num-test (quotient 3 1234) 0)
(num-test (modulo -3 1234) 1231)
(num-test (remainder -3 1234) -3)
(num-test (quotient -3 1234) 0)
(num-test (modulo 3 -1234) -1231)
(num-test (remainder 3 -1234) 3)
(num-test (quotient 3 -1234) 0)
(num-test (modulo -3 -1234) -3)
(num-test (remainder -3 -1234) -3)
(num-test (quotient -3 -1234) 0)
(num-test (modulo 3 1234000000) 3)
(num-test (remainder 3 1234000000) 3)
(num-test (quotient 3 1234000000) 0)
(num-test (modulo -3 1234000000) 1233999997)
(num-test (remainder -3 1234000000) -3)
(num-test (quotient -3 1234000000) 0)
(num-test (modulo 3 -1234000000) -1233999997)
(num-test (remainder 3 -1234000000) 3)
(num-test (quotient 3 -1234000000) 0)
(num-test (modulo -3 -1234000000) -3)
(num-test (remainder -3 -1234000000) -3)
(num-test (quotient -3 -1234000000) 0)
(num-test (modulo 3 500029) 3)
(num-test (remainder 3 500029) 3)
(num-test (quotient 3 500029) 0)
(num-test (modulo -3 500029) 500026)
(num-test (remainder -3 500029) -3)
(num-test (quotient -3 500029) 0)
(num-test (modulo 3 -500029) -500026)
(num-test (remainder 3 -500029) 3)
(num-test (quotient 3 -500029) 0)
(num-test (modulo -3 -500029) -3)
(num-test (remainder -3 -500029) -3)
(num-test (quotient -3 -500029) 0)
(num-test (modulo 3 362880) 3)
(num-test (remainder 3 362880) 3)
(num-test (quotient 3 362880) 0)
(num-test (modulo -3 362880) 362877)
(num-test (remainder -3 362880) -3)
(num-test (quotient -3 362880) 0)
(num-test (modulo 3 -362880) -362877)
(num-test (remainder 3 -362880) 3)
(num-test (quotient 3 -362880) 0)
(num-test (modulo -3 -362880) -3)
(num-test (remainder -3 -362880) -3)
(num-test (quotient -3 -362880) 0)
(num-test (modulo 10 1) 0)
(num-test (remainder 10 1) 0)
(num-test (quotient 10 1) 10)
(num-test (modulo -10 1) 0)
(num-test (remainder -10 1) 0)
(num-test (quotient -10 1) -10)
(num-test (modulo 10 -1) 0)
(num-test (remainder 10 -1) 0)
(num-test (quotient 10 -1) -10)
(num-test (modulo -10 -1) 0)
(num-test (remainder -10 -1) 0)
(num-test (quotient -10 -1) 10)
(num-test (modulo 10 2) 0)
(num-test (remainder 10 2) 0)
(num-test (quotient 10 2) 5)
(num-test (modulo -10 2) 0)
(num-test (remainder -10 2) 0)
(num-test (quotient -10 2) -5)
(num-test (modulo 10 -2) 0)
(num-test (remainder 10 -2) 0)
(num-test (quotient 10 -2) -5)
(num-test (modulo -10 -2) 0)
(num-test (remainder -10 -2) 0)
(num-test (quotient -10 -2) 5)
(num-test (modulo 10 3) 1)
(num-test (remainder 10 3) 1)
(num-test (quotient 10 3) 3)
(num-test (modulo -10 3) 2)
(num-test (remainder -10 3) -1)
(num-test (quotient -10 3) -3)
(num-test (modulo 10 -3) -2)
(num-test (remainder 10 -3) 1)
(num-test (quotient 10 -3) -3)
(num-test (modulo -10 -3) -1)
(num-test (remainder -10 -3) -1)
(num-test (quotient -10 -3) 3)
(num-test (modulo 10 10) 0)
(num-test (remainder 10 10) 0)
(num-test (quotient 10 10) 1)
(num-test (modulo -10 10) 0)
(num-test (remainder -10 10) 0)
(num-test (quotient -10 10) -1)
(num-test (modulo 10 -10) 0)
(num-test (remainder 10 -10) 0)
(num-test (quotient 10 -10) -1)
(num-test (modulo -10 -10) 0)
(num-test (remainder -10 -10) 0)
(num-test (quotient -10 -10) 1)
(num-test (modulo 10 1234) 10)
(num-test (remainder 10 1234) 10)
(num-test (quotient 10 1234) 0)
(num-test (modulo -10 1234) 1224)
(num-test (remainder -10 1234) -10)
(num-test (quotient -10 1234) 0)
(num-test (modulo 10 -1234) -1224)
(num-test (remainder 10 -1234) 10)
(num-test (quotient 10 -1234) 0)
(num-test (modulo -10 -1234) -10)
(num-test (remainder -10 -1234) -10)
(num-test (quotient -10 -1234) 0)
(num-test (modulo 10 1234000000) 10)
(num-test (remainder 10 1234000000) 10)
(num-test (quotient 10 1234000000) 0)
(num-test (modulo -10 1234000000) 1233999990)
(num-test (remainder -10 1234000000) -10)
(num-test (quotient -10 1234000000) 0)
(num-test (modulo 10 -1234000000) -1233999990)
(num-test (remainder 10 -1234000000) 10)
(num-test (quotient 10 -1234000000) 0)
(num-test (modulo -10 -1234000000) -10)
(num-test (remainder -10 -1234000000) -10)
(num-test (quotient -10 -1234000000) 0)
(num-test (modulo 10 500029) 10)
(num-test (remainder 10 500029) 10)
(num-test (quotient 10 500029) 0)
(num-test (modulo -10 500029) 500019)
(num-test (remainder -10 500029) -10)
(num-test (quotient -10 500029) 0)
(num-test (modulo 10 -500029) -500019)
(num-test (remainder 10 -500029) 10)
(num-test (quotient 10 -500029) 0)
(num-test (modulo -10 -500029) -10)
(num-test (remainder -10 -500029) -10)
(num-test (quotient -10 -500029) 0)
(num-test (modulo 10 362880) 10)
(num-test (remainder 10 362880) 10)
(num-test (quotient 10 362880) 0)
(num-test (modulo -10 362880) 362870)
(num-test (remainder -10 362880) -10)
(num-test (quotient -10 362880) 0)
(num-test (modulo 10 -362880) -362870)
(num-test (remainder 10 -362880) 10)
(num-test (quotient 10 -362880) 0)
(num-test (modulo -10 -362880) -10)
(num-test (remainder -10 -362880) -10)
(num-test (quotient -10 -362880) 0)
(num-test (modulo 1234 1) 0)
(num-test (remainder 1234 1) 0)
(num-test (quotient 1234 1) 1234)
(num-test (modulo -1234 1) 0)
(num-test (remainder -1234 1) 0)
(num-test (quotient -1234 1) -1234)
(num-test (modulo 1234 -1) 0)
(num-test (remainder 1234 -1) 0)
(num-test (quotient 1234 -1) -1234)
(num-test (modulo -1234 -1) 0)
(num-test (remainder -1234 -1) 0)
(num-test (quotient -1234 -1) 1234)
(num-test (modulo 1234 2) 0)
(num-test (remainder 1234 2) 0)
(num-test (quotient 1234 2) 617)
(num-test (modulo -1234 2) 0)
(num-test (remainder -1234 2) 0)
(num-test (quotient -1234 2) -617)
(num-test (modulo 1234 -2) 0)
(num-test (remainder 1234 -2) 0)
(num-test (quotient 1234 -2) -617)
(num-test (modulo -1234 -2) 0)
(num-test (remainder -1234 -2) 0)
(num-test (quotient -1234 -2) 617)
(num-test (modulo 1234 3) 1)
(num-test (remainder 1234 3) 1)
(num-test (quotient 1234 3) 411)
(num-test (modulo -1234 3) 2)
(num-test (remainder -1234 3) -1)
(num-test (quotient -1234 3) -411)
(num-test (modulo 1234 -3) -2)
(num-test (remainder 1234 -3) 1)
(num-test (quotient 1234 -3) -411)
(num-test (modulo -1234 -3) -1)
(num-test (remainder -1234 -3) -1)
(num-test (quotient -1234 -3) 411)
(num-test (modulo 1234 10) 4)
(num-test (remainder 1234 10) 4)
(num-test (quotient 1234 10) 123)
(num-test (modulo -1234 10) 6)
(num-test (remainder -1234 10) -4)
(num-test (quotient -1234 10) -123)
(num-test (modulo 1234 -10) -6)
(num-test (remainder 1234 -10) 4)
(num-test (quotient 1234 -10) -123)
(num-test (modulo -1234 -10) -4)
(num-test (remainder -1234 -10) -4)
(num-test (quotient -1234 -10) 123)
(num-test (modulo 1234 1234) 0)
(num-test (remainder 1234 1234) 0)
(num-test (quotient 1234 1234) 1)
(num-test (modulo -1234 1234) 0)
(num-test (remainder -1234 1234) 0)
(num-test (quotient -1234 1234) -1)
(num-test (modulo 1234 -1234) 0)
(num-test (remainder 1234 -1234) 0)
(num-test (quotient 1234 -1234) -1)
(num-test (modulo -1234 -1234) 0)
(num-test (remainder -1234 -1234) 0)
(num-test (quotient -1234 -1234) 1)
(num-test (modulo 1234 1234000000) 1234)
(num-test (remainder 1234 1234000000) 1234)
(num-test (quotient 1234 1234000000) 0)
(num-test (modulo -1234 1234000000) 1233998766)
(num-test (remainder -1234 1234000000) -1234)
(num-test (quotient -1234 1234000000) 0)
(num-test (modulo 1234 -1234000000) -1233998766)
(num-test (remainder 1234 -1234000000) 1234)
(num-test (quotient 1234 -1234000000) 0)
(num-test (modulo -1234 -1234000000) -1234)
(num-test (remainder -1234 -1234000000) -1234)
(num-test (quotient -1234 -1234000000) 0)
(num-test (modulo 1234 500029) 1234)
(num-test (remainder 1234 500029) 1234)
(num-test (quotient 1234 500029) 0)
(num-test (modulo -1234 500029) 498795)
(num-test (remainder -1234 500029) -1234)
(num-test (quotient -1234 500029) 0)
(num-test (modulo 1234 -500029) -498795)
(num-test (remainder 1234 -500029) 1234)
(num-test (quotient 1234 -500029) 0)
(num-test (modulo -1234 -500029) -1234)
(num-test (remainder -1234 -500029) -1234)
(num-test (quotient -1234 -500029) 0)
(num-test (modulo 1234 362880) 1234)
(num-test (remainder 1234 362880) 1234)
(num-test (quotient 1234 362880) 0)
(num-test (modulo -1234 362880) 361646)
(num-test (remainder -1234 362880) -1234)
(num-test (quotient -1234 362880) 0)
(num-test (modulo 1234 -362880) -361646)
(num-test (remainder 1234 -362880) 1234)
(num-test (quotient 1234 -362880) 0)
(num-test (modulo -1234 -362880) -1234)
(num-test (remainder -1234 -362880) -1234)
(num-test (quotient -1234 -362880) 0)
(num-test (modulo 1234000000 1) 0)
(num-test (remainder 1234000000 1) 0)
(num-test (quotient 1234000000 1) 1234000000)
(num-test (modulo -1234000000 1) 0)
(num-test (remainder -1234000000 1) 0)
(num-test (quotient -1234000000 1) -1234000000)
(num-test (modulo 1234000000 -1) 0)
(num-test (remainder 1234000000 -1) 0)
(num-test (quotient 1234000000 -1) -1234000000)
(num-test (modulo -1234000000 -1) 0)
(num-test (remainder -1234000000 -1) 0)
(num-test (quotient -1234000000 -1) 1234000000)
(num-test (modulo 1234000000 2) 0)
(num-test (remainder 1234000000 2) 0)
(num-test (quotient 1234000000 2) 617000000)
(num-test (modulo -1234000000 2) 0)
(num-test (remainder -1234000000 2) 0)
(num-test (quotient -1234000000 2) -617000000)
(num-test (modulo 1234000000 -2) 0)
(num-test (remainder 1234000000 -2) 0)
(num-test (quotient 1234000000 -2) -617000000)
(num-test (modulo -1234000000 -2) 0)
(num-test (remainder -1234000000 -2) 0)
(num-test (quotient -1234000000 -2) 617000000)
(num-test (modulo 1234000000 3) 1)
(num-test (remainder 1234000000 3) 1)
(num-test (quotient 1234000000 3) 411333333)
(num-test (modulo -1234000000 3) 2)
(num-test (remainder -1234000000 3) -1)
(num-test (quotient -1234000000 3) -411333333)
(num-test (modulo 1234000000 -3) -2)
(num-test (remainder 1234000000 -3) 1)
(num-test (quotient 1234000000 -3) -411333333)
(num-test (modulo -1234000000 -3) -1)
(num-test (remainder -1234000000 -3) -1)
(num-test (quotient -1234000000 -3) 411333333)
(num-test (modulo 1234000000 10) 0)
(num-test (remainder 1234000000 10) 0)
(num-test (quotient 1234000000 10) 123400000)
(num-test (modulo -1234000000 10) 0)
(num-test (remainder -1234000000 10) 0)
(num-test (quotient -1234000000 10) -123400000)
(num-test (modulo 1234000000 -10) 0)
(num-test (remainder 1234000000 -10) 0)
(num-test (quotient 1234000000 -10) -123400000)
(num-test (modulo -1234000000 -10) 0)
(num-test (remainder -1234000000 -10) 0)
(num-test (quotient -1234000000 -10) 123400000)
(num-test (modulo 1234000000 1234) 0)
(num-test (remainder 1234000000 1234) 0)
(num-test (quotient 1234000000 1234) 1000000)
(num-test (modulo -1234000000 1234) 0)
(num-test (remainder -1234000000 1234) 0)
(num-test (quotient -1234000000 1234) -1000000)
(num-test (modulo 1234000000 -1234) 0)
(num-test (remainder 1234000000 -1234) 0)
(num-test (quotient 1234000000 -1234) -1000000)
(num-test (modulo -1234000000 -1234) 0)
(num-test (remainder -1234000000 -1234) 0)
(num-test (quotient -1234000000 -1234) 1000000)
(num-test (modulo 1234000000 1234000000) 0)
(num-test (remainder 1234000000 1234000000) 0)
(num-test (quotient 1234000000 1234000000) 1)
(num-test (modulo -1234000000 1234000000) 0)
(num-test (remainder -1234000000 1234000000) 0)
(num-test (quotient -1234000000 1234000000) -1)
(num-test (modulo 1234000000 -1234000000) 0)
(num-test (remainder 1234000000 -1234000000) 0)
(num-test (quotient 1234000000 -1234000000) -1)
(num-test (modulo -1234000000 -1234000000) 0)
(num-test (remainder -1234000000 -1234000000) 0)
(num-test (quotient -1234000000 -1234000000) 1)
(num-test (modulo 1234000000 500029) 428457)
(num-test (remainder 1234000000 500029) 428457)
(num-test (quotient 1234000000 500029) 2467)
(num-test (modulo -1234000000 500029) 71572)
(num-test (remainder -1234000000 500029) -428457)
(num-test (quotient -1234000000 500029) -2467)
(num-test (modulo 1234000000 -500029) -71572)
(num-test (remainder 1234000000 -500029) 428457)
(num-test (quotient 1234000000 -500029) -2467)
(num-test (modulo -1234000000 -500029) -428457)
(num-test (remainder -1234000000 -500029) -428457)
(num-test (quotient -1234000000 -500029) 2467)
(num-test (modulo 1234000000 362880) 208000)
(num-test (remainder 1234000000 362880) 208000)
(num-test (quotient 1234000000 362880) 3400)
(num-test (modulo -1234000000 362880) 154880)
(num-test (remainder -1234000000 362880) -208000)
(num-test (quotient -1234000000 362880) -3400)
(num-test (modulo 1234000000 -362880) -154880)
(num-test (remainder 1234000000 -362880) 208000)
(num-test (quotient 1234000000 -362880) -3400)
(num-test (modulo -1234000000 -362880) -208000)
(num-test (remainder -1234000000 -362880) -208000)
(num-test (quotient -1234000000 -362880) 3400)
(num-test (modulo 500029 1) 0)
(num-test (remainder 500029 1) 0)
(num-test (quotient 500029 1) 500029)
(num-test (modulo -500029 1) 0)
(num-test (remainder -500029 1) 0)
(num-test (quotient -500029 1) -500029)
(num-test (modulo 500029 -1) 0)
(num-test (remainder 500029 -1) 0)
(num-test (quotient 500029 -1) -500029)
(num-test (modulo -500029 -1) 0)
(num-test (remainder -500029 -1) 0)
(num-test (quotient -500029 -1) 500029)
(num-test (modulo 500029 2) 1)
(num-test (remainder 500029 2) 1)
(num-test (quotient 500029 2) 250014)
(num-test (modulo -500029 2) 1)
(num-test (remainder -500029 2) -1)
(num-test (quotient -500029 2) -250014)
(num-test (modulo 500029 -2) -1)
(num-test (remainder 500029 -2) 1)
(num-test (quotient 500029 -2) -250014)
(num-test (modulo -500029 -2) -1)
(num-test (remainder -500029 -2) -1)
(num-test (quotient -500029 -2) 250014)
(num-test (modulo 500029 3) 1)
(num-test (remainder 500029 3) 1)
(num-test (quotient 500029 3) 166676)
(num-test (modulo -500029 3) 2)
(num-test (remainder -500029 3) -1)
(num-test (quotient -500029 3) -166676)
(num-test (modulo 500029 -3) -2)
(num-test (remainder 500029 -3) 1)
(num-test (quotient 500029 -3) -166676)
(num-test (modulo -500029 -3) -1)
(num-test (remainder -500029 -3) -1)
(num-test (quotient -500029 -3) 166676)
(num-test (modulo 500029 10) 9)
(num-test (remainder 500029 10) 9)
(num-test (quotient 500029 10) 50002)
(num-test (modulo -500029 10) 1)
(num-test (remainder -500029 10) -9)
(num-test (quotient -500029 10) -50002)
(num-test (modulo 500029 -10) -1)
(num-test (remainder 500029 -10) 9)
(num-test (quotient 500029 -10) -50002)
(num-test (modulo -500029 -10) -9)
(num-test (remainder -500029 -10) -9)
(num-test (quotient -500029 -10) 50002)
(num-test (modulo 500029 1234) 259)
(num-test (remainder 500029 1234) 259)
(num-test (quotient 500029 1234) 405)
(num-test (modulo -500029 1234) 975)
(num-test (remainder -500029 1234) -259)
(num-test (quotient -500029 1234) -405)
(num-test (modulo 500029 -1234) -975)
(num-test (remainder 500029 -1234) 259)
(num-test (quotient 500029 -1234) -405)
(num-test (modulo -500029 -1234) -259)
(num-test (remainder -500029 -1234) -259)
(num-test (quotient -500029 -1234) 405)
(num-test (modulo 500029 1234000000) 500029)
(num-test (remainder 500029 1234000000) 500029)
(num-test (quotient 500029 1234000000) 0)
(num-test (modulo -500029 1234000000) 1233499971)
(num-test (remainder -500029 1234000000) -500029)
(num-test (quotient -500029 1234000000) 0)
(num-test (modulo 500029 -1234000000) -1233499971)
(num-test (remainder 500029 -1234000000) 500029)
(num-test (quotient 500029 -1234000000) 0)
(num-test (modulo -500029 -1234000000) -500029)
(num-test (remainder -500029 -1234000000) -500029)
(num-test (quotient -500029 -1234000000) 0)
(num-test (modulo 500029 500029) 0)
(num-test (remainder 500029 500029) 0)
(num-test (quotient 500029 500029) 1)
(num-test (modulo -500029 500029) 0)
(num-test (remainder -500029 500029) 0)
(num-test (quotient -500029 500029) -1)
(num-test (modulo 500029 -500029) 0)
(num-test (remainder 500029 -500029) 0)
(num-test (quotient 500029 -500029) -1)
(num-test (modulo -500029 -500029) 0)
(num-test (remainder -500029 -500029) 0)
(num-test (quotient -500029 -500029) 1)
(num-test (modulo 500029 362880) 137149)
(num-test (remainder 500029 362880) 137149)
(num-test (quotient 500029 362880) 1)
(num-test (modulo -500029 362880) 225731)
(num-test (remainder -500029 362880) -137149)
(num-test (quotient -500029 362880) -1)
(num-test (modulo 500029 -362880) -225731)
(num-test (remainder 500029 -362880) 137149)
(num-test (quotient 500029 -362880) -1)
(num-test (modulo -500029 -362880) -137149)
(num-test (remainder -500029 -362880) -137149)
(num-test (quotient -500029 -362880) 1)
(num-test (modulo 362880 1) 0)
(num-test (remainder 362880 1) 0)
(num-test (quotient 362880 1) 362880)
(num-test (modulo -362880 1) 0)
(num-test (remainder -362880 1) 0)
(num-test (quotient -362880 1) -362880)
(num-test (modulo 362880 -1) 0)
(num-test (remainder 362880 -1) 0)
(num-test (quotient 362880 -1) -362880)
(num-test (modulo -362880 -1) 0)
(num-test (remainder -362880 -1) 0)
(num-test (quotient -362880 -1) 362880)
(num-test (modulo 362880 2) 0)
(num-test (remainder 362880 2) 0)
(num-test (quotient 362880 2) 181440)
(num-test (modulo -362880 2) 0)
(num-test (remainder -362880 2) 0)
(num-test (quotient -362880 2) -181440)
(num-test (modulo 362880 -2) 0)
(num-test (remainder 362880 -2) 0)
(num-test (quotient 362880 -2) -181440)
(num-test (modulo -362880 -2) 0)
(num-test (remainder -362880 -2) 0)
(num-test (quotient -362880 -2) 181440)
(num-test (modulo 362880 3) 0)
(num-test (remainder 362880 3) 0)
(num-test (quotient 362880 3) 120960)
(num-test (modulo -362880 3) 0)
(num-test (remainder -362880 3) 0)
(num-test (quotient -362880 3) -120960)
(num-test (modulo 362880 -3) 0)
(num-test (remainder 362880 -3) 0)
(num-test (quotient 362880 -3) -120960)
(num-test (modulo -362880 -3) 0)
(num-test (remainder -362880 -3) 0)
(num-test (quotient -362880 -3) 120960)
(num-test (modulo 362880 10) 0)
(num-test (remainder 362880 10) 0)
(num-test (quotient 362880 10) 36288)
(num-test (modulo -362880 10) 0)
(num-test (remainder -362880 10) 0)
(num-test (quotient -362880 10) -36288)
(num-test (modulo 362880 -10) 0)
(num-test (remainder 362880 -10) 0)
(num-test (quotient 362880 -10) -36288)
(num-test (modulo -362880 -10) 0)
(num-test (remainder -362880 -10) 0)
(num-test (quotient -362880 -10) 36288)
(num-test (modulo 362880 1234) 84)
(num-test (remainder 362880 1234) 84)
(num-test (quotient 362880 1234) 294)
(num-test (modulo -362880 1234) 1150)
(num-test (remainder -362880 1234) -84)
(num-test (quotient -362880 1234) -294)
(num-test (modulo 362880 -1234) -1150)
(num-test (remainder 362880 -1234) 84)
(num-test (quotient 362880 -1234) -294)
(num-test (modulo -362880 -1234) -84)
(num-test (remainder -362880 -1234) -84)
(num-test (quotient -362880 -1234) 294)
(num-test (modulo 362880 1234000000) 362880)
(num-test (remainder 362880 1234000000) 362880)
(num-test (quotient 362880 1234000000) 0)
(num-test (modulo -362880 1234000000) 1233637120)
(num-test (remainder -362880 1234000000) -362880)
(num-test (quotient -362880 1234000000) 0)
(num-test (modulo 362880 -1234000000) -1233637120)
(num-test (remainder 362880 -1234000000) 362880)
(num-test (quotient 362880 -1234000000) 0)
(num-test (modulo -362880 -1234000000) -362880)
(num-test (remainder -362880 -1234000000) -362880)
(num-test (quotient -362880 -1234000000) 0)
(num-test (modulo 362880 500029) 362880)
(num-test (remainder 362880 500029) 362880)
(num-test (quotient 362880 500029) 0)
(num-test (modulo -362880 500029) 137149)
(num-test (remainder -362880 500029) -362880)
(num-test (quotient -362880 500029) 0)
(num-test (modulo 362880 -500029) -137149)
(num-test (remainder 362880 -500029) 362880)
(num-test (quotient 362880 -500029) 0)
(num-test (modulo -362880 -500029) -362880)
(num-test (remainder -362880 -500029) -362880)
(num-test (quotient -362880 -500029) 0)
(num-test (modulo 362880 362880) 0)
(num-test (remainder 362880 362880) 0)
(num-test (quotient 362880 362880) 1)
(num-test (modulo -362880 362880) 0)
(num-test (remainder -362880 362880) 0)
(num-test (quotient -362880 362880) -1)
(num-test (modulo 362880 -362880) 0)
(num-test (remainder 362880 -362880) 0)
(num-test (quotient 362880 -362880) -1)
(num-test (modulo -362880 -362880) 0)
(num-test (remainder -362880 -362880) 0)
(num-test (quotient -362880 -362880) 1)

(if with-generic-modulo
    (begin
      (num-test (modulo 3.1 2.0) 1.1)
      (num-test (modulo 3.1 2) 1.1)
      (num-test (modulo -3.1 2.0) 0.9) ; parallels (modulo -3 2) -> 1
      (num-test (modulo -3.1 2.5) 1.9)
      (num-test (modulo 3.1 -2.0) -0.9)
      (num-test (modulo -3.1 -2.0) -1.1)
      (num-test (modulo 3 2.5) 0.5)
      (num-test (modulo 19439282 4409.5) 2206.0)
      (num-test (modulo 3/2 1/4) 0)
      (num-test (modulo 3/2 1/3) 1/6)
      (num-test (modulo 3/2 0.3) 0.0)
      (num-test (modulo 1.5 1/4) 0.0)
      (num-test (modulo 1.5 1/3) 0.16666666666667)
      (num-test (modulo 1.5 0.3) 0.0)
      (num-test (modulo 1/2 2) 1/2)
      (num-test (modulo 0.5 2.0) 0.5)
      (num-test (modulo 1/2 -2) -3/2)
      (num-test (modulo 0.5 -2.0) -1.5)
      (num-test (modulo 0.5 -2) -1.5)
      (num-test (modulo 1 -4) -3)
      (num-test (modulo -1/2 -1) -1/2)
      (num-test (modulo -0.5 -1) -0.5)
      (num-test (modulo -1/2 -1.0) -0.5)
      (num-test (modulo -3/2 2) 1/2)
      (num-test (modulo -1.5 2) 0.5)
      (num-test (modulo -3/2 -2) -3/2)
      (num-test (modulo -1.5 -2) -1.5)

      (num-test (remainder 3.1 2.0) 1.1)
      (num-test (remainder 3.1 2) 1.1)
      (num-test (remainder -3.1 2.0) -1.1)
      (num-test (remainder -3.1 2.5) -0.6)
      (num-test (remainder 3.1 -2.0) 1.1)
      (num-test (remainder -3.1 -2.0) -1.1)
      (num-test (remainder 3 2.5) 0.5)
      (num-test (remainder 19439282 4409.5) 2206.0)
      (num-test (remainder 3/2 1/4) 0)
      (num-test (remainder 3/2 1/3) 1/6)
      (num-test (remainder 3/2 0.3) 0.0)
      (num-test (remainder 1.5 1/4) 0.0)
      (num-test (remainder 1.5 1/3) 0.16666666666667)
      (num-test (remainder 1.5 0.3) 0.0)
      (num-test (remainder 1/2 2) 1/2)
      (num-test (remainder 0.5 2.0) 0.5)
      (num-test (remainder 1/2 -2) 1/2)
      (num-test (remainder 0.5 -2.0) 0.5)
      (num-test (remainder 0.5 -2) 0.5)
      (num-test (remainder 1 -4) 1)
      (num-test (remainder -1/2 -1) -1/2)
      (num-test (remainder -0.5 -1) -0.5)
      (num-test (remainder -1/2 -1.0) -0.5)
      (num-test (remainder -3/2 2) -3/2)
      (num-test (remainder -1.5 2) -1.5)
      (num-test (remainder -3/2 -2) -3/2)
      (num-test (remainder -1.5 -2) -1.5)

      (num-test (quotient 3.1 2.0) 1)
      (num-test (quotient 3.1 2) 1)
      (num-test (quotient -3.1 2.0) -1)
      (num-test (quotient -3.1 2.5) -1)
      (num-test (quotient 3.1 -2.0) -1)
      (num-test (quotient -3.1 -2.0) 1)
      (num-test (quotient 3 2.5) 1)
      (num-test (quotient 19439282 4409.5) 4408)
      (num-test (quotient 3/2 1/4) 6)
      (num-test (quotient 3/2 1/3) 4)
      (num-test (quotient 3/2 0.3) 5)
      (num-test (quotient 1.5 1/4) 6)
      (num-test (quotient 1.5 1/3) 4)
      (num-test (quotient 1.5 0.3) 5)
      (num-test (quotient 1/2 2) 0)
      (num-test (quotient 0.5 2.0) 0)
      (num-test (quotient 1/2 -2) 0)
      (num-test (quotient 0.5 -2.0) 0)
      (num-test (quotient 0.5 -2) 0)
      (num-test (quotient 1 -4) 0)
      (num-test (quotient -1/2 -1) 0)
      (num-test (quotient -0.5 -1) 0)
      (num-test (quotient -1/2 -1.0) 0)
      (num-test (quotient -3/2 2) 0)
      (num-test (quotient -1.5 2) 0)
      (num-test (quotient -3/2 -2) 0)
      (num-test (quotient -1.5 -2) 0)

      (num-test (lcm 1/2 2) 2)
      (num-test (lcm 1/3 3/4) 3)
      (num-test (lcm 2 3/4) 6)
      (num-test (lcm 3/4 2) 6)
      (num-test (lcm 1/3 3/4 5/8) 15)
      (num-test (lcm 1/3 2/3) 2/3)
      (num-test (lcm 1/3 1/6 5/12) 5/3)
      (num-test (lcm 1/3 1/6 5/12 2) 10)

      (num-test (gcd 1/2 1/3) 1/6)
      (num-test (gcd 1/2 2) 1/2)
      (num-test (gcd 1/3 3/4) 1/12)
      (num-test (gcd 2 3/4) 1/4)
      (num-test (gcd 3/4 2) 1/4)
      (num-test (gcd 1/3 3/4 5/8) 1/24)
      (num-test (gcd 1/3 2/3) 1/3)
      (num-test (gcd 1/3 1/6 5/12) 1/12)
      (num-test (gcd 1/3 1/6 5/12 2) 1/12)
      ))

(num-test (modulo 13 4) 1)
(num-test (modulo -13 4) 3)
(num-test (quotient 35 7) 5 )
(num-test (quotient -35 7) -5)
(num-test (quotient 35 -7) -5)
(num-test (quotient -35 -7) 5)
(num-test (modulo 13 4) 1)
(num-test (remainder 13 4) 1)
(num-test (modulo -13 4) 3)
(num-test (remainder -13 4) -1)
(num-test (modulo 13 -4) -3)
(num-test (remainder 13 -4) 1)
(num-test (modulo -13 -4) -1)
(num-test (remainder -13 -4) -1)
(num-test (modulo 0 86400) 0)
(num-test (modulo 0 -86400) 0)

(let ((top-exp 60))
  (let ((happy #t))
    (do ((i 2 (+ i 1)))
	((or (not happy) (> i top-exp)))
      (let* ((val1 (inexact->exact (expt 2 i)))
	     (val2 (+ val1 1))
	     (mv (modulo val2 2))
	     (qv (quotient val2 2))
	     (rv (remainder val2 2)))
	(if (not (= mv 1))
	    (begin (set! happy #f) (display "(modulo ") (display val2) (display " 2) = ") (display mv) (display "?") (newline)))
	(if (not (= qv (/ val1 2)))
	    (begin (set! happy #f) (display "(quotient ") (display val2) (display " 2) = ") (display qv) (display "?") (newline)))
	(if (not (= rv 1))
	    (begin (set! happy #f) (display "(remainder ") (display val2) (display " 2) = ") (display rv) (display "?") (newline))))))
  
  (if with-generic-modulo
      (let ((happy #t))
	(do ((i 2 (+ i 1)))
	    ((or (not happy) (> i top-exp)))
	  (let* ((val1 (inexact->exact (expt 2 i)))
		 (val2 (/ (+ val1 1) 2))
		 (mv (modulo val2 2))
		 (qv (quotient val2 2))
		 (rv (remainder val2 2)))
	    (if (not (= mv 1/2))
		(begin (set! happy #f) (display "(modulo ") (display val2) (display " 2) = ") (display mv) (display "?") (newline)))
	    (if (not (= qv (/ val1 4)))
		(begin (set! happy #f) (display "(quotient ") (display val2) (display " 2) = ") (display qv) (display "?") (newline)))
	    (if (not (= rv 1/2))
		(begin (set! happy #f) (display "(remainder ") (display val2) (display " 2) = ") (display rv) (display "?") (newline))))))
      ))
  

;; -------- abs and magnitude

(num-test (abs 0) 0)
(num-test (abs -0) 0)
(num-test (magnitude 0) 0)
(num-test (magnitude -0) 0)
(num-test (abs 0/1) 0/1)
(num-test (abs -0/1) 0/1)
(num-test (magnitude 0/1) 0/1)
(num-test (magnitude -0/1) 0/1)
(num-test (abs 0/2) 0/2)
(num-test (abs -0/2) 0/2)
(num-test (magnitude 0/2) 0/2)
(num-test (magnitude -0/2) 0/2)
(num-test (abs 0/3) 0/3)
(num-test (abs -0/3) 0/3)
(num-test (magnitude 0/3) 0/3)
(num-test (magnitude -0/3) 0/3)
(num-test (abs 0/10) 0/10)
(num-test (abs -0/10) 0/10)
(num-test (magnitude 0/10) 0/10)
(num-test (magnitude -0/10) 0/10)
(num-test (abs 0/1234) 0/1234)
(num-test (abs -0/1234) 0/1234)
(num-test (magnitude 0/1234) 0/1234)
(num-test (magnitude -0/1234) 0/1234)
(num-test (abs 0/1234000000) 0/1234000000)
(num-test (abs -0/1234000000) 0/1234000000)
(num-test (magnitude 0/1234000000) 0/1234000000)
(num-test (magnitude -0/1234000000) 0/1234000000)
(num-test (abs 1) 1)
(num-test (abs -1) 1)
(num-test (magnitude 1) 1)
(num-test (magnitude -1) 1)
(num-test (abs 0/362880) 0/362880)
(num-test (abs -0/362880) 0/362880)
(num-test (magnitude 0/362880) 0/362880)
(num-test (magnitude -0/362880) 0/362880)
(num-test (abs 0/500029) 0/500029)
(num-test (abs -0/500029) 0/500029)
(num-test (magnitude 0/500029) 0/500029)
(num-test (magnitude -0/500029) 0/500029)
(num-test (abs 1/1) 1/1)
(num-test (abs -1/1) 1/1)
(num-test (magnitude 1/1) 1/1)
(num-test (magnitude -1/1) 1/1)
(num-test (abs 1/2) 1/2)
(num-test (abs -1/2) 1/2)
(num-test (magnitude 1/2) 1/2)
(num-test (magnitude -1/2) 1/2)
(num-test (abs 1/3) 1/3)
(num-test (abs -1/3) 1/3)
(num-test (magnitude 1/3) 1/3)
(num-test (magnitude -1/3) 1/3)
(num-test (abs 1/10) 1/10)
(num-test (abs -1/10) 1/10)
(num-test (magnitude 1/10) 1/10)
(num-test (magnitude -1/10) 1/10)
(num-test (abs 1/1234) 1/1234)
(num-test (abs -1/1234) 1/1234)
(num-test (magnitude 1/1234) 1/1234)
(num-test (magnitude -1/1234) 1/1234)
(num-test (abs 1/1234000000) 1/1234000000)
(num-test (abs -1/1234000000) 1/1234000000)
(num-test (magnitude 1/1234000000) 1/1234000000)
(num-test (magnitude -1/1234000000) 1/1234000000)
(num-test (abs 1/500029) 1/500029)
(num-test (abs -1/500029) 1/500029)
(num-test (magnitude 1/500029) 1/500029)
(num-test (magnitude -1/500029) 1/500029)
(num-test (abs 1/362880) 1/362880)
(num-test (abs -1/362880) 1/362880)
(num-test (magnitude 1/362880) 1/362880)
(num-test (magnitude -1/362880) 1/362880)
(num-test (abs 2) 2)
(num-test (abs -2) 2)
(num-test (magnitude 2) 2)
(num-test (magnitude -2) 2)
(num-test (abs 2/1) 2/1)
(num-test (abs -2/1) 2/1)
(num-test (magnitude 2/1) 2/1)
(num-test (magnitude -2/1) 2/1)
(num-test (abs 2/2) 2/2)
(num-test (abs -2/2) 2/2)
(num-test (magnitude 2/2) 2/2)
(num-test (magnitude -2/2) 2/2)
(num-test (abs 2/3) 2/3)
(num-test (abs -2/3) 2/3)
(num-test (magnitude 2/3) 2/3)
(num-test (magnitude -2/3) 2/3)
(num-test (abs 2/10) 2/10)
(num-test (abs -2/10) 2/10)
(num-test (magnitude 2/10) 2/10)
(num-test (magnitude -2/10) 2/10)
(num-test (abs 2/1234) 2/1234)
(num-test (abs -2/1234) 2/1234)
(num-test (magnitude 2/1234) 2/1234)
(num-test (magnitude -2/1234) 2/1234)
(num-test (abs 2/1234000000) 2/1234000000)
(num-test (abs -2/1234000000) 2/1234000000)
(num-test (magnitude 2/1234000000) 2/1234000000)
(num-test (magnitude -2/1234000000) 2/1234000000)
(num-test (abs 2/500029) 2/500029)
(num-test (abs -2/500029) 2/500029)
(num-test (magnitude 2/500029) 2/500029)
(num-test (magnitude -2/500029) 2/500029)
(num-test (abs 2/362880) 2/362880)
(num-test (abs -2/362880) 2/362880)
(num-test (magnitude 2/362880) 2/362880)
(num-test (magnitude -2/362880) 2/362880)
(num-test (abs 3) 3)
(num-test (abs -3) 3)
(num-test (magnitude 3) 3)
(num-test (magnitude -3) 3)
(num-test (abs 3/1) 3/1)
(num-test (abs -3/1) 3/1)
(num-test (magnitude 3/1) 3/1)
(num-test (magnitude -3/1) 3/1)
(num-test (abs 3/2) 3/2)
(num-test (abs -3/2) 3/2)
(num-test (magnitude 3/2) 3/2)
(num-test (magnitude -3/2) 3/2)
(num-test (abs 3/3) 3/3)
(num-test (abs -3/3) 3/3)
(num-test (magnitude 3/3) 3/3)
(num-test (magnitude -3/3) 3/3)
(num-test (abs 3/10) 3/10)
(num-test (abs -3/10) 3/10)
(num-test (magnitude 3/10) 3/10)
(num-test (magnitude -3/10) 3/10)
(num-test (abs 3/1234) 3/1234)
(num-test (abs -3/1234) 3/1234)
(num-test (magnitude 3/1234) 3/1234)
(num-test (magnitude -3/1234) 3/1234)
(num-test (abs 3/1234000000) 3/1234000000)
(num-test (abs -3/1234000000) 3/1234000000)
(num-test (magnitude 3/1234000000) 3/1234000000)
(num-test (magnitude -3/1234000000) 3/1234000000)
(num-test (abs 3/500029) 3/500029)
(num-test (abs -3/500029) 3/500029)
(num-test (magnitude 3/500029) 3/500029)
(num-test (magnitude -3/500029) 3/500029)
(num-test (abs 3/362880) 3/362880)
(num-test (abs -3/362880) 3/362880)
(num-test (magnitude 3/362880) 3/362880)
(num-test (magnitude -3/362880) 3/362880)
(num-test (abs 10) 10)
(num-test (abs -10) 10)
(num-test (magnitude 10) 10)
(num-test (magnitude -10) 10)
(num-test (abs 10/1) 10/1)
(num-test (abs -10/1) 10/1)
(num-test (magnitude 10/1) 10/1)
(num-test (magnitude -10/1) 10/1)
(num-test (abs 10/2) 10/2)
(num-test (abs -10/2) 10/2)
(num-test (magnitude 10/2) 10/2)
(num-test (magnitude -10/2) 10/2)
(num-test (abs 10/3) 10/3)
(num-test (abs -10/3) 10/3)
(num-test (magnitude 10/3) 10/3)
(num-test (magnitude -10/3) 10/3)
(num-test (abs 10/10) 10/10)
(num-test (abs -10/10) 10/10)
(num-test (magnitude 10/10) 10/10)
(num-test (magnitude -10/10) 10/10)
(num-test (abs 10/1234) 10/1234)
(num-test (abs -10/1234) 10/1234)
(num-test (magnitude 10/1234) 10/1234)
(num-test (magnitude -10/1234) 10/1234)
(num-test (abs 10/1234000000) 10/1234000000)
(num-test (abs -10/1234000000) 10/1234000000)
(num-test (magnitude 10/1234000000) 10/1234000000)
(num-test (magnitude -10/1234000000) 10/1234000000)
(num-test (abs 10/500029) 10/500029)
(num-test (abs -10/500029) 10/500029)
(num-test (magnitude 10/500029) 10/500029)
(num-test (magnitude -10/500029) 10/500029)
(num-test (abs 10/362880) 10/362880)
(num-test (abs -10/362880) 10/362880)
(num-test (magnitude 10/362880) 10/362880)
(num-test (magnitude -10/362880) 10/362880)
(num-test (abs 1234) 1234)
(num-test (abs -1234) 1234)
(num-test (magnitude 1234) 1234)
(num-test (magnitude -1234) 1234)
(num-test (abs 1234/1) 1234/1)
(num-test (abs -1234/1) 1234/1)
(num-test (magnitude 1234/1) 1234/1)
(num-test (magnitude -1234/1) 1234/1)
(num-test (abs 1234/2) 1234/2)
(num-test (abs -1234/2) 1234/2)
(num-test (magnitude 1234/2) 1234/2)
(num-test (magnitude -1234/2) 1234/2)
(num-test (abs 1234/3) 1234/3)
(num-test (abs -1234/3) 1234/3)
(num-test (magnitude 1234/3) 1234/3)
(num-test (magnitude -1234/3) 1234/3)
(num-test (abs 1234/10) 1234/10)
(num-test (abs -1234/10) 1234/10)
(num-test (magnitude 1234/10) 1234/10)
(num-test (magnitude -1234/10) 1234/10)
(num-test (abs 1234/1234) 1234/1234)
(num-test (abs -1234/1234) 1234/1234)
(num-test (magnitude 1234/1234) 1234/1234)
(num-test (magnitude -1234/1234) 1234/1234)
(num-test (abs 1234/1234000000) 1234/1234000000)
(num-test (abs -1234/1234000000) 1234/1234000000)
(num-test (magnitude 1234/1234000000) 1234/1234000000)
(num-test (magnitude -1234/1234000000) 1234/1234000000)
(num-test (abs 1234/500029) 1234/500029)
(num-test (abs -1234/500029) 1234/500029)
(num-test (magnitude 1234/500029) 1234/500029)
(num-test (magnitude -1234/500029) 1234/500029)
(num-test (abs 1234/362880) 1234/362880)
(num-test (abs -1234/362880) 1234/362880)
(num-test (magnitude 1234/362880) 1234/362880)
(num-test (magnitude -1234/362880) 1234/362880)
(num-test (abs 1234000000) 1234000000)
(num-test (abs -1234000000) 1234000000)
(num-test (magnitude 1234000000) 1234000000)
(num-test (magnitude -1234000000) 1234000000)
(num-test (abs 1234000000/1) 1234000000/1)
(num-test (abs -1234000000/1) 1234000000/1)
(num-test (magnitude 1234000000/1) 1234000000/1)
(num-test (magnitude -1234000000/1) 1234000000/1)
(num-test (abs 1234000000/2) 1234000000/2)
(num-test (abs -1234000000/2) 1234000000/2)
(num-test (magnitude 1234000000/2) 1234000000/2)
(num-test (magnitude -1234000000/2) 1234000000/2)
(num-test (abs 1234000000/3) 1234000000/3)
(num-test (abs -1234000000/3) 1234000000/3)
(num-test (magnitude 1234000000/3) 1234000000/3)
(num-test (magnitude -1234000000/3) 1234000000/3)
(num-test (abs 1234000000/10) 1234000000/10)
(num-test (abs -1234000000/10) 1234000000/10)
(num-test (magnitude 1234000000/10) 1234000000/10)
(num-test (magnitude -1234000000/10) 1234000000/10)
(num-test (abs 1234000000/1234) 1234000000/1234)
(num-test (abs -1234000000/1234) 1234000000/1234)
(num-test (magnitude 1234000000/1234) 1234000000/1234)
(num-test (magnitude -1234000000/1234) 1234000000/1234)
(num-test (abs 1234000000/1234000000) 1234000000/1234000000)
(num-test (abs -1234000000/1234000000) 1234000000/1234000000)
(num-test (magnitude 1234000000/1234000000) 1234000000/1234000000)
(num-test (magnitude -1234000000/1234000000) 1234000000/1234000000)
(num-test (abs 1234000000/500029) 1234000000/500029)
(num-test (abs -1234000000/500029) 1234000000/500029)
(num-test (magnitude 1234000000/500029) 1234000000/500029)
(num-test (magnitude -1234000000/500029) 1234000000/500029)
(num-test (abs 1234000000/362880) 1234000000/362880)
(num-test (abs -1234000000/362880) 1234000000/362880)
(num-test (magnitude 1234000000/362880) 1234000000/362880)
(num-test (magnitude -1234000000/362880) 1234000000/362880)
(num-test (abs 500029) 500029)
(num-test (abs -500029) 500029)
(num-test (magnitude 500029) 500029)
(num-test (magnitude -500029) 500029)
(num-test (abs 500029/1) 500029/1)
(num-test (abs -500029/1) 500029/1)
(num-test (magnitude 500029/1) 500029/1)
(num-test (magnitude -500029/1) 500029/1)
(num-test (abs 500029/2) 500029/2)
(num-test (abs -500029/2) 500029/2)
(num-test (magnitude 500029/2) 500029/2)
(num-test (magnitude -500029/2) 500029/2)
(num-test (abs 500029/3) 500029/3)
(num-test (abs -500029/3) 500029/3)
(num-test (magnitude 500029/3) 500029/3)
(num-test (magnitude -500029/3) 500029/3)
(num-test (abs 500029/10) 500029/10)
(num-test (abs -500029/10) 500029/10)
(num-test (magnitude 500029/10) 500029/10)
(num-test (magnitude -500029/10) 500029/10)
(num-test (abs 500029/1234) 500029/1234)
(num-test (abs -500029/1234) 500029/1234)
(num-test (magnitude 500029/1234) 500029/1234)
(num-test (magnitude -500029/1234) 500029/1234)
(num-test (abs 500029/1234000000) 500029/1234000000)
(num-test (abs -500029/1234000000) 500029/1234000000)
(num-test (magnitude 500029/1234000000) 500029/1234000000)
(num-test (magnitude -500029/1234000000) 500029/1234000000)
(num-test (abs 500029/500029) 500029/500029)
(num-test (abs -500029/500029) 500029/500029)
(num-test (magnitude 500029/500029) 500029/500029)
(num-test (magnitude -500029/500029) 500029/500029)
(num-test (abs 500029/362880) 500029/362880)
(num-test (abs -500029/362880) 500029/362880)
(num-test (magnitude 500029/362880) 500029/362880)
(num-test (magnitude -500029/362880) 500029/362880)
(num-test (abs 362880) 362880)
(num-test (abs -362880) 362880)
(num-test (magnitude 362880) 362880)
(num-test (magnitude -362880) 362880)
(num-test (abs 362880/1) 362880/1)
(num-test (abs -362880/1) 362880/1)
(num-test (magnitude 362880/1) 362880/1)
(num-test (magnitude -362880/1) 362880/1)
(num-test (abs 362880/2) 362880/2)
(num-test (abs -362880/2) 362880/2)
(num-test (magnitude 362880/2) 362880/2)
(num-test (magnitude -362880/2) 362880/2)
(num-test (abs 362880/3) 362880/3)
(num-test (abs -362880/3) 362880/3)
(num-test (magnitude 362880/3) 362880/3)
(num-test (magnitude -362880/3) 362880/3)
(num-test (abs 362880/10) 362880/10)
(num-test (abs -362880/10) 362880/10)
(num-test (magnitude 362880/10) 362880/10)
(num-test (magnitude -362880/10) 362880/10)
(num-test (abs 362880/1234) 362880/1234)
(num-test (abs -362880/1234) 362880/1234)
(num-test (magnitude 362880/1234) 362880/1234)
(num-test (magnitude -362880/1234) 362880/1234)
(num-test (abs 362880/1234000000) 362880/1234000000)
(num-test (abs -362880/1234000000) 362880/1234000000)
(num-test (magnitude 362880/1234000000) 362880/1234000000)
(num-test (magnitude -362880/1234000000) 362880/1234000000)
(num-test (abs 362880/500029) 362880/500029)
(num-test (abs -362880/500029) 362880/500029)
(num-test (magnitude 362880/500029) 362880/500029)
(num-test (magnitude -362880/500029) 362880/500029)
(num-test (abs 362880/362880) 362880/362880)
(num-test (abs -362880/362880) 362880/362880)
(num-test (magnitude 362880/362880) 362880/362880)
(num-test (magnitude -362880/362880) 362880/362880)
(num-test (abs 0.0) 0.0)
(num-test (abs -0.0) 0.0)
(num-test (magnitude 0.0) 0.0)
(num-test (magnitude -0.0) 0.0)
(num-test (abs 0.00000001) 0.00000001)
(num-test (abs -0.00000001) 0.00000001)
(num-test (magnitude 0.00000001) 0.00000001)
(num-test (magnitude -0.00000001) 0.00000001)
(num-test (abs 1.0) 1.0)
(num-test (abs -1.0) 1.0)
(num-test (magnitude 1.0) 1.0)
(num-test (magnitude -1.0) 1.0)
(num-test (abs our-pi) our-pi)
(num-test (abs -3.14159265358979) our-pi)
(num-test (magnitude our-pi) our-pi)
(num-test (magnitude -3.14159265358979) our-pi)
(num-test (abs 2.71828182845905) 2.71828182845905)
(num-test (abs -2.71828182845905) 2.71828182845905)
(num-test (magnitude 2.71828182845905) 2.71828182845905)
(num-test (magnitude -2.71828182845905) 2.71828182845905)
(num-test (abs 1234.0) 1234.0)
(num-test (abs -1234.0) 1234.0)
(num-test (magnitude 1234.0) 1234.0)
(num-test (magnitude -1234.0) 1234.0)
(num-test (abs 1234000000.0) 1234000000.0)
(num-test (abs -1234000000.0) 1234000000.0)
(num-test (magnitude 1234000000.0) 1234000000.0)
(num-test (magnitude -1234000000.0) 1234000000.0)
(num-test (abs 0) 0)
(num-test (abs 2) 2)
(num-test (abs -2) 2)
(num-test (abs 10) 10)
(num-test (abs -10) 10)
(num-test (abs 1234000000) 1234000000)
(num-test (abs -1234000000) 1234000000)
(num-test (abs 362880) 362880)
(num-test (abs -362880) 362880)
(num-test (abs 0/1) 0/1)
(num-test (abs 0/1) 0/1)
(num-test (abs 2/2) 2/2)
(num-test (abs -2/2) 2/2)
(num-test (abs 10/3) 10/3)
(num-test (abs -10/3) 10/3)
(num-test (abs 1234000000/10) 1234000000/10)
(num-test (abs -1234000000/10) 1234000000/10)
(num-test (abs 362880/1234) 362880/1234)
(num-test (abs -362880/1234) 362880/1234)
(num-test (abs 0.0) 0.0)
(num-test (abs -0.0) 0.0)
(num-test (abs 1.0) 1.0)
(num-test (abs -1.0) 1.0)
(num-test (abs 2.71828182845905) 2.71828182845905)
(num-test (abs -2.71828182845905) 2.71828182845905)
(num-test (abs 1234000000.0) 1234000000.0)
(num-test (abs -1234000000.0) 1234000000.0)
(num-test (abs 6) 6)
(num-test (abs -6) 6)
(num-test (abs -123456789) 123456789)
(num-test (abs -1234567890) 1234567890)
(num-test (abs -7) 7)
(num-test (abs 7) 7)
(num-test (abs 0) 0)
(num-test (magnitude 0.0+0.00000001i) 0.00000001)
(num-test (magnitude 0.0e+00+0.0e+00i) 0e0+0.0i)
(num-test (magnitude 0.0e+00+1.19209289550781250e-07i) 1.1920928955078125e-7+0.0i)
(num-test (magnitude 0.0e+00-1.19209289550781250e-07i) 1.1920928955078125e-7+0.0i)
(num-test (magnitude 0.0e+00+5.0e-01i) 5e-1+0.0i)
(num-test (magnitude 0.0e+00-5.0e-01i) 5e-1+0.0i)
(num-test (magnitude 0.0e+00+1.0e+00i) 1e0+0.0i)
(num-test (magnitude 0.0e+00-1.0e+00i) 1e0+0.0i)
(num-test (magnitude 0.0e+00+2.0e+00i) 2e0+0.0i)
(num-test (magnitude 0.0e+00-2.0e+00i) 2e0+0.0i)
(num-test (magnitude 0.0e+00+8.3886080e+06i) 8.388608e6+0.0i)
(num-test (magnitude 0.0e+00-8.3886080e+06i) 8.388608e6+0.0i)
(num-test (magnitude 1.19209289550781250e-07+0.0e+00i) 1.1920928955078125e-7+0.0i)
(num-test (magnitude -1.19209289550781250e-07+0.0e+00i) 1.1920928955078125e-7+0.0i)
(num-test (magnitude 1.19209289550781250e-07+1.19209289550781250e-07i) 1.6858739404357612715e-7+0.0i)
(num-test (magnitude 1.19209289550781250e-07-1.19209289550781250e-07i) 1.6858739404357612715e-7+0.0i)
(num-test (magnitude -1.19209289550781250e-07+1.19209289550781250e-07i) 1.6858739404357612715e-7+0.0i)
(num-test (magnitude -1.19209289550781250e-07-1.19209289550781250e-07i) 1.6858739404357612715e-7+0.0i)
(num-test (magnitude 1.19209289550781250e-07+5.0e-01i) 5.0000000000001421085e-1+0.0i)
(num-test (magnitude 1.19209289550781250e-07-5.0e-01i) 5.0000000000001421085e-1+0.0i)
(num-test (magnitude -1.19209289550781250e-07+5.0e-01i) 5.0000000000001421085e-1+0.0i)
(num-test (magnitude -1.19209289550781250e-07-5.0e-01i) 5.0000000000001421085e-1+0.0i)
(num-test (magnitude 1.19209289550781250e-07+1.0e+00i) 1.0000000000000071054e0+0.0i)
(num-test (magnitude 1.19209289550781250e-07-1.0e+00i) 1.0000000000000071054e0+0.0i)
(num-test (magnitude -1.19209289550781250e-07+1.0e+00i) 1.0000000000000071054e0+0.0i)
(num-test (magnitude -1.19209289550781250e-07-1.0e+00i) 1.0000000000000071054e0+0.0i)
(num-test (magnitude 1.19209289550781250e-07+2.0e+00i) 2.0000000000000035527e0+0.0i)
(num-test (magnitude 1.19209289550781250e-07-2.0e+00i) 2.0000000000000035527e0+0.0i)
(num-test (magnitude -1.19209289550781250e-07+2.0e+00i) 2.0000000000000035527e0+0.0i)
(num-test (magnitude -1.19209289550781250e-07-2.0e+00i) 2.0000000000000035527e0+0.0i)
(num-test (magnitude 1.19209289550781250e-07+8.3886080e+06i) 8.3886080e6+0.0i)
(num-test (magnitude 1.19209289550781250e-07-8.3886080e+06i) 8.3886080e6+0.0i)
(num-test (magnitude -1.19209289550781250e-07+8.3886080e+06i) 8.3886080e6+0.0i)
(num-test (magnitude -1.19209289550781250e-07-8.3886080e+06i) 8.3886080e6+0.0i)
(num-test (magnitude 5.0e-01+0.0e+00i) 5e-1+0.0i)
(num-test (magnitude -5.0e-01+0.0e+00i) 5e-1+0.0i)
(num-test (magnitude 5.0e-01+1.19209289550781250e-07i) 5.0000000000001421085e-1+0.0i)
(num-test (magnitude 5.0e-01-1.19209289550781250e-07i) 5.0000000000001421085e-1+0.0i)
(num-test (magnitude -5.0e-01+1.19209289550781250e-07i) 5.0000000000001421085e-1+0.0i)
(num-test (magnitude -5.0e-01-1.19209289550781250e-07i) 5.0000000000001421085e-1+0.0i)
(num-test (magnitude 5.0e-01+5.0e-01i) 7.0710678118654752440e-1+0.0i)
(num-test (magnitude 5.0e-01-5.0e-01i) 7.0710678118654752440e-1+0.0i)
(num-test (magnitude -5.0e-01+5.0e-01i) 7.0710678118654752440e-1+0.0i)
(num-test (magnitude -5.0e-01-5.0e-01i) 7.0710678118654752440e-1+0.0i)
(num-test (magnitude 5.0e-01+1.0e+00i) 1.1180339887498948482e0+0.0i)
(num-test (magnitude 5.0e-01-1.0e+00i) 1.1180339887498948482e0+0.0i)
(num-test (magnitude -5.0e-01+1.0e+00i) 1.1180339887498948482e0+0.0i)
(num-test (magnitude -5.0e-01-1.0e+00i) 1.1180339887498948482e0+0.0i)
(num-test (magnitude 5.0e-01+2.0e+00i) 2.0615528128088302749e0+0.0i)
(num-test (magnitude 5.0e-01-2.0e+00i) 2.0615528128088302749e0+0.0i)
(num-test (magnitude -5.0e-01+2.0e+00i) 2.0615528128088302749e0+0.0i)
(num-test (magnitude -5.0e-01-2.0e+00i) 2.0615528128088302749e0+0.0i)
(num-test (magnitude 5.0e-01+8.3886080e+06i) 8.3886080000000149012e6+0.0i)
(num-test (magnitude 5.0e-01-8.3886080e+06i) 8.3886080000000149012e6+0.0i)
(num-test (magnitude -5.0e-01+8.3886080e+06i) 8.3886080000000149012e6+0.0i)
(num-test (magnitude -5.0e-01-8.3886080e+06i) 8.3886080000000149012e6+0.0i)
(num-test (magnitude 1.0e+00+0.0e+00i) 1e0+0.0i)
(num-test (magnitude -1.0e+00+0.0e+00i) 1e0+0.0i)
(num-test (magnitude 1.0e+00+1.19209289550781250e-07i) 1.0000000000000071054e0+0.0i)
(num-test (magnitude 1.0e+00-1.19209289550781250e-07i) 1.0000000000000071054e0+0.0i)
(num-test (magnitude -1.0e+00+1.19209289550781250e-07i) 1.0000000000000071054e0+0.0i)
(num-test (magnitude -1.0e+00-1.19209289550781250e-07i) 1.0000000000000071054e0+0.0i)
(num-test (magnitude 1.0e+00+5.0e-01i) 1.1180339887498948482e0+0.0i)
(num-test (magnitude 1.0e+00-5.0e-01i) 1.1180339887498948482e0+0.0i)
(num-test (magnitude -1.0e+00+5.0e-01i) 1.1180339887498948482e0+0.0i)
(num-test (magnitude -1.0e+00-5.0e-01i) 1.1180339887498948482e0+0.0i)
(num-test (magnitude 1.0e+00+1.0e+00i) 1.4142135623730950488e0+0.0i)
(num-test (magnitude 1.0e+00-1.0e+00i) 1.4142135623730950488e0+0.0i)
(num-test (magnitude -1.0e+00+1.0e+00i) 1.4142135623730950488e0+0.0i)
(num-test (magnitude -1.0e+00-1.0e+00i) 1.4142135623730950488e0+0.0i)
(num-test (magnitude 1.0e+00+2.0e+00i) 2.2360679774997896964e0+0.0i)
(num-test (magnitude 1.0e+00-2.0e+00i) 2.2360679774997896964e0+0.0i)
(num-test (magnitude -1.0e+00+2.0e+00i) 2.2360679774997896964e0+0.0i)
(num-test (magnitude -1.0e+00-2.0e+00i) 2.2360679774997896964e0+0.0i)
(num-test (magnitude 1.0e+00+8.3886080e+06i) 8.3886080000000596046e6+0.0i)
(num-test (magnitude 1.0e+00-8.3886080e+06i) 8.3886080000000596046e6+0.0i)
(num-test (magnitude -1.0e+00+8.3886080e+06i) 8.3886080000000596046e6+0.0i)
(num-test (magnitude -1.0e+00-8.3886080e+06i) 8.3886080000000596046e6+0.0i)
(num-test (magnitude 2.0e+00+0.0e+00i) 2e0+0.0i)
(num-test (magnitude -2.0e+00+0.0e+00i) 2e0+0.0i)
(num-test (magnitude 2.0e+00+1.19209289550781250e-07i) 2.0000000000000035527e0+0.0i)
(num-test (magnitude 2.0e+00-1.19209289550781250e-07i) 2.0000000000000035527e0+0.0i)
(num-test (magnitude -2.0e+00+1.19209289550781250e-07i) 2.0000000000000035527e0+0.0i)
(num-test (magnitude -2.0e+00-1.19209289550781250e-07i) 2.0000000000000035527e0+0.0i)
(num-test (magnitude 2.0e+00+5.0e-01i) 2.0615528128088302749e0+0.0i)
(num-test (magnitude 2.0e+00-5.0e-01i) 2.0615528128088302749e0+0.0i)
(num-test (magnitude -2.0e+00+5.0e-01i) 2.0615528128088302749e0+0.0i)
(num-test (magnitude -2.0e+00-5.0e-01i) 2.0615528128088302749e0+0.0i)
(num-test (magnitude 2.0e+00+1.0e+00i) 2.2360679774997896964e0+0.0i)
(num-test (magnitude 2.0e+00-1.0e+00i) 2.2360679774997896964e0+0.0i)
(num-test (magnitude -2.0e+00+1.0e+00i) 2.2360679774997896964e0+0.0i)
(num-test (magnitude -2.0e+00-1.0e+00i) 2.2360679774997896964e0+0.0i)
(num-test (magnitude 2.0e+00+2.0e+00i) 2.8284271247461900976e0+0.0i)
(num-test (magnitude 2.0e+00-2.0e+00i) 2.8284271247461900976e0+0.0i)
(num-test (magnitude -2.0e+00+2.0e+00i) 2.8284271247461900976e0+0.0i)
(num-test (magnitude -2.0e+00-2.0e+00i) 2.8284271247461900976e0+0.0i)
(num-test (magnitude 2.0e+00+8.3886080e+06i) 8.3886080000002384186e6+0.0i)
(num-test (magnitude 2.0e+00-8.3886080e+06i) 8.3886080000002384186e6+0.0i)
(num-test (magnitude -2.0e+00+8.3886080e+06i) 8.3886080000002384186e6+0.0i)
(num-test (magnitude -2.0e+00-8.3886080e+06i) 8.3886080000002384186e6+0.0i)
(num-test (magnitude 8.3886080e+06+0.0e+00i) 8.388608e6+0.0i)
(num-test (magnitude -8.3886080e+06+0.0e+00i) 8.388608e6+0.0i)
(num-test (magnitude 8.3886080e+06+1.19209289550781250e-07i) 8.3886080e6+0.0i)
(num-test (magnitude 8.3886080e+06-1.19209289550781250e-07i) 8.3886080e6+0.0i)
(num-test (magnitude -8.3886080e+06+1.19209289550781250e-07i) 8.3886080e6+0.0i)
(num-test (magnitude -8.3886080e+06-1.19209289550781250e-07i) 8.3886080e6+0.0i)
(num-test (magnitude 8.3886080e+06+5.0e-01i) 8.3886080000000149012e6+0.0i)
(num-test (magnitude 8.3886080e+06-5.0e-01i) 8.3886080000000149012e6+0.0i)
(num-test (magnitude -8.3886080e+06+5.0e-01i) 8.3886080000000149012e6+0.0i)
(num-test (magnitude -8.3886080e+06-5.0e-01i) 8.3886080000000149012e6+0.0i)
(num-test (magnitude 8.3886080e+06+1.0e+00i) 8.3886080000000596046e6+0.0i)
(num-test (magnitude 8.3886080e+06-1.0e+00i) 8.3886080000000596046e6+0.0i)
(num-test (magnitude -8.3886080e+06+1.0e+00i) 8.3886080000000596046e6+0.0i)
(num-test (magnitude -8.3886080e+06-1.0e+00i) 8.3886080000000596046e6+0.0i)
(num-test (magnitude 8.3886080e+06+2.0e+00i) 8.3886080000002384186e6+0.0i)
(num-test (magnitude 8.3886080e+06-2.0e+00i) 8.3886080000002384186e6+0.0i)
(num-test (magnitude -8.3886080e+06+2.0e+00i) 8.3886080000002384186e6+0.0i)
(num-test (magnitude -8.3886080e+06-2.0e+00i) 8.3886080000002384186e6+0.0i)
(num-test (magnitude 8.3886080e+06+8.3886080e+06i) 1.1863283203031444111e7+0.0i)
(num-test (magnitude 8.3886080e+06-8.3886080e+06i) 1.1863283203031444111e7+0.0i)
(num-test (magnitude -8.3886080e+06+8.3886080e+06i) 1.1863283203031444111e7+0.0i)
(num-test (magnitude -8.3886080e+06-8.3886080e+06i) 1.1863283203031444111e7+0.0i)
(num-test (magnitude 0.0-0.00000001i) 0.00000001)
(num-test (magnitude -0.0+0.00000001i) 0.00000001)
(num-test (magnitude -0.0-0.00000001i) 0.00000001)
(num-test (magnitude 0.0+1.0i) 1.0)
(num-test (magnitude -0.0+1.0i) 1.0)
(num-test (magnitude 0.0-1.0i) 1.0)
(num-test (magnitude -0.0-1.0i) 1.0)
(num-test (magnitude 0.0+3.14159265358979i) 3.14159265358979)
(num-test (magnitude -0.0+3.14159265358979i) 3.14159265358979)
(num-test (magnitude 0.0-3.14159265358979i) 3.14159265358979)
(num-test (magnitude -0.0-3.14159265358979i) 3.14159265358979)
(num-test (magnitude 0.0+2.71828182845905i) 2.71828182845905)
(num-test (magnitude -0.0+2.71828182845905i) 2.71828182845905)
(num-test (magnitude 0.0-2.71828182845905i) 2.71828182845905)
(num-test (magnitude -0.0-2.71828182845905i) 2.71828182845905)
(num-test (magnitude 0.0+1234.0i) 1234.0)
(num-test (magnitude -0.0+1234.0i) 1234.0)
(num-test (magnitude 0.0-1234.0i) 1234.0)
(num-test (magnitude -0.0-1234.0i) 1234.0)
(num-test (magnitude 0.0+1234000000.0i) 1234000000.0)
(num-test (magnitude -0.0+1234000000.0i) 1234000000.0)
(num-test (magnitude 0.0-1234000000.0i) 1234000000.0)
(num-test (magnitude -0.0-1234000000.0i) 1234000000.0)
(num-test (magnitude 0.00000001+0.00000001i) 0.00000001414214)
(num-test (magnitude -0.00000001+0.00000001i) 0.00000001414214)
(num-test (magnitude 0.00000001-0.00000001i) 0.00000001414214)
(num-test (magnitude -0.00000001-0.00000001i) 0.00000001414214)
(num-test (magnitude 0.00000001+1.0i) 1.0)
(num-test (magnitude -0.00000001+1.0i) 1.0)
(num-test (magnitude 0.00000001-1.0i) 1.0)
(num-test (magnitude -0.00000001-1.0i) 1.0)
(num-test (magnitude 0.00000001+3.14159265358979i) our-pi)
(num-test (magnitude -0.00000001+3.14159265358979i) our-pi)
(num-test (magnitude 0.00000001-3.14159265358979i) our-pi)
(num-test (magnitude -0.00000001-3.14159265358979i) our-pi)
(num-test (magnitude 0.00000001+2.71828182845905i) 2.71828182845905)
(num-test (magnitude -0.00000001+2.71828182845905i) 2.71828182845905)
(num-test (magnitude 0.00000001-2.71828182845905i) 2.71828182845905)
(num-test (magnitude -0.00000001-2.71828182845905i) 2.71828182845905)
(num-test (magnitude 0.00000001+1234.0i) 1234.0)
(num-test (magnitude -0.00000001+1234.0i) 1234.0)
(num-test (magnitude 0.00000001-1234.0i) 1234.0)
(num-test (magnitude -0.00000001-1234.0i) 1234.0)
(num-test (magnitude 0.00000001+1234000000.0i) 1234000000.0)
(num-test (magnitude -0.00000001+1234000000.0i) 1234000000.0)
(num-test (magnitude 0.00000001-1234000000.0i) 1234000000.0)
(num-test (magnitude -0.00000001-1234000000.0i) 1234000000.0)
(num-test (magnitude 1.0+0.00000001i) 1.0)
(num-test (magnitude -1.0+0.00000001i) 1.0)
(num-test (magnitude 1.0-0.00000001i) 1.0)
(num-test (magnitude -1.0-0.00000001i) 1.0)
(num-test (magnitude 1.0+1.0i) 1.41421356237310)
(num-test (magnitude -1.0+1.0i) 1.41421356237310)
(num-test (magnitude 1.0-1.0i) 1.41421356237310)
(num-test (magnitude -1.0-1.0i) 1.41421356237310)
(num-test (magnitude 1.0+3.14159265358979i) 3.29690830947562)
(num-test (magnitude -1.0+3.14159265358979i) 3.29690830947562)
(num-test (magnitude 1.0-3.14159265358979i) 3.29690830947562)
(num-test (magnitude -1.0-3.14159265358979i) 3.29690830947562)
(num-test (magnitude 1.0+2.71828182845905i) 2.89638673159001)
(num-test (magnitude -1.0+2.71828182845905i) 2.89638673159001)
(num-test (magnitude 1.0-2.71828182845905i) 2.89638673159001)
(num-test (magnitude -1.0-2.71828182845905i) 2.89638673159001)
(num-test (magnitude 1.0+1234.0i) 1234.00040518631931)
(num-test (magnitude -1.0+1234.0i) 1234.00040518631931)
(num-test (magnitude 1.0-1234.0i) 1234.00040518631931)
(num-test (magnitude -1.0-1234.0i) 1234.00040518631931)
(num-test (magnitude 1.0+1234000000.0i) 1234000000.0)
(num-test (magnitude -1.0+1234000000.0i) 1234000000.0)
(num-test (magnitude 1.0-1234000000.0i) 1234000000.0)
(num-test (magnitude -1.0-1234000000.0i) 1234000000.0)
(num-test (magnitude 3.14159265358979+0.00000001i) our-pi)
(num-test (magnitude -3.14159265358979+0.00000001i) our-pi)
(num-test (magnitude 3.14159265358979-0.00000001i) our-pi)
(num-test (magnitude -3.14159265358979-0.00000001i) our-pi)
(num-test (magnitude 3.14159265358979+1.0i) 3.29690830947562)
(num-test (magnitude -3.14159265358979+1.0i) 3.29690830947562)
(num-test (magnitude 3.14159265358979-1.0i) 3.29690830947562)
(num-test (magnitude -3.14159265358979-1.0i) 3.29690830947562)
(num-test (magnitude 3.14159265358979+3.14159265358979i) 4.44288293815837)
(num-test (magnitude -3.14159265358979+3.14159265358979i) 4.44288293815837)
(num-test (magnitude 3.14159265358979-3.14159265358979i) 4.44288293815837)
(num-test (magnitude -3.14159265358979-3.14159265358979i) 4.44288293815837)
(num-test (magnitude 3.14159265358979+2.71828182845905i) 4.15435440231331)
(num-test (magnitude -3.14159265358979+2.71828182845905i) 4.15435440231331)
(num-test (magnitude 3.14159265358979-2.71828182845905i) 4.15435440231331)
(num-test (magnitude -3.14159265358979-2.71828182845905i) 4.15435440231331)
(num-test (magnitude 3.14159265358979+1234.0i) 1234.00399902285608)
(num-test (magnitude -3.14159265358979+1234.0i) 1234.00399902285608)
(num-test (magnitude 3.14159265358979-1234.0i) 1234.00399902285608)
(num-test (magnitude -3.14159265358979-1234.0i) 1234.00399902285608)
(num-test (magnitude 3.14159265358979+1234000000.0i) 1234000000.0)
(num-test (magnitude -3.14159265358979+1234000000.0i) 1234000000.0)
(num-test (magnitude 3.14159265358979-1234000000.0i) 1234000000.0)
(num-test (magnitude -3.14159265358979-1234000000.0i) 1234000000.0)
(num-test (magnitude 2.71828182845905+0.00000001i) 2.71828182845905)
(num-test (magnitude -2.71828182845905+0.00000001i) 2.71828182845905)
(num-test (magnitude 2.71828182845905-0.00000001i) 2.71828182845905)
(num-test (magnitude -2.71828182845905-0.00000001i) 2.71828182845905)
(num-test (magnitude 2.71828182845905+1.0i) 2.89638673159001)
(num-test (magnitude -2.71828182845905+1.0i) 2.89638673159001)
(num-test (magnitude 2.71828182845905-1.0i) 2.89638673159001)
(num-test (magnitude -2.71828182845905-1.0i) 2.89638673159001)
(num-test (magnitude 2.71828182845905+3.14159265358979i) 4.15435440231331)
(num-test (magnitude -2.71828182845905+3.14159265358979i) 4.15435440231331)
(num-test (magnitude 2.71828182845905-3.14159265358979i) 4.15435440231331)
(num-test (magnitude -2.71828182845905-3.14159265358979i) 4.15435440231331)
(num-test (magnitude 2.71828182845905+2.71828182845905i) 3.84423102815912)
(num-test (magnitude -2.71828182845905+2.71828182845905i) 3.84423102815912)
(num-test (magnitude 2.71828182845905-2.71828182845905i) 3.84423102815912)
(num-test (magnitude -2.71828182845905-2.71828182845905i) 3.84423102815912)
(num-test (magnitude 2.71828182845905+1234.0i) 1234.00299394130275)
(num-test (magnitude -2.71828182845905+1234.0i) 1234.00299394130275)
(num-test (magnitude 2.71828182845905-1234.0i) 1234.00299394130275)
(num-test (magnitude -2.71828182845905-1234.0i) 1234.00299394130275)
(num-test (magnitude 2.71828182845905+1234000000.0i) 1234000000.0)
(num-test (magnitude -2.71828182845905+1234000000.0i) 1234000000.0)
(num-test (magnitude 2.71828182845905-1234000000.0i) 1234000000.0)
(num-test (magnitude -2.71828182845905-1234000000.0i) 1234000000.0)
(num-test (magnitude 1234.0+0.00000001i) 1234.0)
(num-test (magnitude -1234.0+0.00000001i) 1234.0)
(num-test (magnitude 1234.0-0.00000001i) 1234.0)
(num-test (magnitude -1234.0-0.00000001i) 1234.0)
(num-test (magnitude 1234.0+1.0i) 1234.00040518631931)
(num-test (magnitude -1234.0+1.0i) 1234.00040518631931)
(num-test (magnitude 1234.0-1.0i) 1234.00040518631931)
(num-test (magnitude -1234.0-1.0i) 1234.00040518631931)
(num-test (magnitude 1234.0+3.14159265358979i) 1234.00399902285608)
(num-test (magnitude -1234.0+3.14159265358979i) 1234.00399902285608)
(num-test (magnitude 1234.0-3.14159265358979i) 1234.00399902285608)
(num-test (magnitude -1234.0-3.14159265358979i) 1234.00399902285608)
(num-test (magnitude 1234.0+2.71828182845905i) 1234.00299394130275)
(num-test (magnitude -1234.0+2.71828182845905i) 1234.00299394130275)
(num-test (magnitude 1234.0-2.71828182845905i) 1234.00299394130275)
(num-test (magnitude -1234.0-2.71828182845905i) 1234.00299394130275)
(num-test (magnitude 1234.0+1234.0i) 1745.13953596839929)
(num-test (magnitude -1234.0+1234.0i) 1745.13953596839929)
(num-test (magnitude 1234.0-1234.0i) 1745.13953596839929)
(num-test (magnitude -1234.0-1234.0i) 1745.13953596839929)
(num-test (magnitude 1234.0+1234000000.0i) 1234000000.00061702728271)
(num-test (magnitude -1234.0+1234000000.0i) 1234000000.00061702728271)
(num-test (magnitude 1234.0-1234000000.0i) 1234000000.00061702728271)
(num-test (magnitude -1234.0-1234000000.0i) 1234000000.00061702728271)
(num-test (magnitude 1234000000.0+0.00000001i) 1234000000.0)
(num-test (magnitude -1234000000.0+0.00000001i) 1234000000.0)
(num-test (magnitude 1234000000.0-0.00000001i) 1234000000.0)
(num-test (magnitude -1234000000.0-0.00000001i) 1234000000.0)
(num-test (magnitude 1234000000.0+1.0i) 1234000000.0)
(num-test (magnitude -1234000000.0+1.0i) 1234000000.0)
(num-test (magnitude 1234000000.0-1.0i) 1234000000.0)
(num-test (magnitude -1234000000.0-1.0i) 1234000000.0)
(num-test (magnitude 1234000000.0+3.14159265358979i) 1234000000.0)
(num-test (magnitude -1234000000.0+3.14159265358979i) 1234000000.0)
(num-test (magnitude 1234000000.0-3.14159265358979i) 1234000000.0)
(num-test (magnitude -1234000000.0-3.14159265358979i) 1234000000.0)
(num-test (magnitude 1234000000.0+2.71828182845905i) 1234000000.0)
(num-test (magnitude -1234000000.0+2.71828182845905i) 1234000000.0)
(num-test (magnitude 1234000000.0-2.71828182845905i) 1234000000.0)
(num-test (magnitude -1234000000.0-2.71828182845905i) 1234000000.0)
(num-test (magnitude 1234000000.0+1234.0i) 1234000000.00061702728271)
(num-test (magnitude -1234000000.0+1234.0i) 1234000000.00061702728271)
(num-test (magnitude 1234000000.0-1234.0i) 1234000000.00061702728271)
(num-test (magnitude -1234000000.0-1234.0i) 1234000000.00061702728271)
(num-test (magnitude 1234000000.0+1234000000.0i) 1745139535.96839928627014)
(num-test (magnitude -1234000000.0+1234000000.0i) 1745139535.96839928627014)
(num-test (magnitude 1234000000.0-1234000000.0i) 1745139535.96839928627014)
(num-test (magnitude -1234000000.0-1234000000.0i) 1745139535.96839928627014)
(num-test (magnitude 0.0+0.00000001i) 0.00000001)
(num-test (magnitude -0.0+0.00000001i) 0.00000001)
(num-test (magnitude 1.0+1.0i) 1.41421356237310)
(num-test (magnitude -1.0+1.0i) 1.41421356237310)
(num-test (magnitude 2.71828182845905+3.14159265358979i) 4.15435440231331)
(num-test (magnitude -2.71828182845905+3.14159265358979i) 4.15435440231331)
(num-test (magnitude 1234000000.0+2.71828182845905i) 1234000000.0)
(num-test (magnitude -1234000000.0+2.71828182845905i) 1234000000.0)
(num-test (magnitude 0) 0)


;; -------- make-polar and make-rectangular

(num-test (make-rectangular 0 1) 0.0+1.0i)
(num-test (make-rectangular -0 1) -0.0+1.0i)
(num-test (make-rectangular 0 -1) 0.0-1.0i)
(num-test (make-rectangular -0 -1) -0.0-1.0i)
(num-test (make-polar 0 1) 0.0)
(num-test (make-polar -0 1) -0.0)
(num-test (make-polar 0 -1) 0.0)
(num-test (make-polar -0 -1) 0.0)
(num-test (make-rectangular 0 2) 0.0+2.0i)
(num-test (make-rectangular -0 2) -0.0+2.0i)
(num-test (make-rectangular 0 -2) 0.0-2.0i)
(num-test (make-rectangular -0 -2) -0.0-2.0i)
(num-test (make-polar 0 2) -0.0)
(num-test (make-polar -0 2) 0.0)
(num-test (make-polar 0 -2) 0.0)
(num-test (make-polar -0 -2) 0.0)
(num-test (make-rectangular 0 3) 0.0+3.0i)
(num-test (make-rectangular -0 3) -0.0+3.0i)
(num-test (make-rectangular 0 -3) 0.0-3.0i)
(num-test (make-rectangular -0 -3) -0.0-3.0i)
(num-test (make-polar 0 3) -0.0)
(num-test (make-polar -0 3) 0.0)
(num-test (make-polar 0 -3) 0.0)
(num-test (make-polar -0 -3) 0.0)
(num-test (make-rectangular 0 10) 0.0+10.0i)
(num-test (make-rectangular -0 10) -0.0+10.0i)
(num-test (make-rectangular 0 -10) 0.0-10.0i)
(num-test (make-rectangular -0 -10) -0.0-10.0i)
(num-test (make-polar 0 10) 0.0)
(num-test (make-polar -0 10) 0.0)
(num-test (make-polar 0 -10) -0.0)
(num-test (make-polar -0 -10) 0.0)
(num-test (make-rectangular 0 1234) 0.0+1234.0i)
(num-test (make-rectangular -0 1234) -0.0+1234.0i)
(num-test (make-rectangular 0 -1234) 0.0-1234.0i)
(num-test (make-rectangular -0 -1234) -0.0-1234.0i)
(num-test (make-polar 0 1234) -0.0)
(num-test (make-polar -0 1234) 0.0)
(num-test (make-polar 0 -1234) 0.0)
(num-test (make-polar -0 -1234) 0.0)
(num-test (make-rectangular 0 1234000000) 0.0+1234000000.0i)
(num-test (make-rectangular -0 1234000000) -0.0+1234000000.0i)
(num-test (make-rectangular 0 -1234000000) 0.0-1234000000.0i)
(num-test (make-rectangular -0 -1234000000) -0.0-1234000000.0i)
(num-test (make-polar 0 1234000000) 0.0)
(num-test (make-polar -0 1234000000) 0.0)
(num-test (make-polar 0 -1234000000) 0.0)
(num-test (make-polar -0 -1234000000) -0.0)
(num-test (make-rectangular 0 500029) 0.0+500029.0i)
(num-test (make-rectangular -0 500029) -0.0+500029.0i)
(num-test (make-rectangular 0 -500029) 0.0-500029.0i)
(num-test (make-rectangular -0 -500029) -0.0-500029.0i)
(num-test (make-polar 0 500029) 0.0)
(num-test (make-polar -0 500029) -0.0)
(num-test (make-polar 0 -500029) 0.0)
(num-test (make-polar -0 -500029) 0.0)
(num-test (make-rectangular 0 362880) 0.0+362880.0i)
(num-test (make-rectangular -0 362880) -0.0+362880.0i)
(num-test (make-rectangular 0 -362880) 0.0-362880.0i)
(num-test (make-rectangular -0 -362880) -0.0-362880.0i)
(num-test (make-polar 0 362880) 0.0)
(num-test (make-polar -0 362880) -0.0)
(num-test (make-polar 0 -362880) 0.0)
(num-test (make-polar -0 -362880) 0.0)
(num-test (make-rectangular 1 1) 1.0+1.0i)
(num-test (make-rectangular -1 1) -1.0+1.0i)
(num-test (make-rectangular 1 -1) 1.0-1.0i)
(num-test (make-rectangular -1 -1) -1.0-1.0i)
(num-test (make-polar 1 1) 0.54030230586814+0.84147098480790i)
(num-test (make-polar -1 1) -0.54030230586814-0.84147098480790i)
(num-test (make-polar 1 -1) 0.54030230586814-0.84147098480790i)
(num-test (make-polar -1 -1) -0.54030230586814+0.84147098480790i)
(num-test (make-rectangular 1 2) 1.0+2.0i)
(num-test (make-rectangular -1 2) -1.0+2.0i)
(num-test (make-rectangular 1 -2) 1.0-2.0i)
(num-test (make-rectangular -1 -2) -1.0-2.0i)
(num-test (make-polar 1 2) -0.41614683654714+0.90929742682568i)
(num-test (make-polar -1 2) 0.41614683654714-0.90929742682568i)
(num-test (make-polar 1 -2) -0.41614683654714-0.90929742682568i)
(num-test (make-polar -1 -2) 0.41614683654714+0.90929742682568i)
(num-test (make-rectangular 1 3) 1.0+3.0i)
(num-test (make-rectangular -1 3) -1.0+3.0i)
(num-test (make-rectangular 1 -3) 1.0-3.0i)
(num-test (make-rectangular -1 -3) -1.0-3.0i)
(num-test (make-polar 1 3) -0.98999249660045+0.14112000805987i)
(num-test (make-polar -1 3) 0.98999249660045-0.14112000805987i)
(num-test (make-polar 1 -3) -0.98999249660045-0.14112000805987i)
(num-test (make-polar -1 -3) 0.98999249660045+0.14112000805987i)
(num-test (make-rectangular 1 10) 1.0+10.0i)
(num-test (make-rectangular -1 10) -1.0+10.0i)
(num-test (make-rectangular 1 -10) 1.0-10.0i)
(num-test (make-rectangular -1 -10) -1.0-10.0i)
(num-test (make-polar 1 10) -0.83907152907645-0.54402111088937i)
(num-test (make-polar -1 10) 0.83907152907645+0.54402111088937i)
(num-test (make-polar 1 -10) -0.83907152907645+0.54402111088937i)
(num-test (make-polar -1 -10) 0.83907152907645-0.54402111088937i)
(num-test (make-rectangular 1 1234) 1.0+1234.0i)
(num-test (make-rectangular -1 1234) -1.0+1234.0i)
(num-test (make-rectangular 1 -1234) 1.0-1234.0i)
(num-test (make-rectangular -1 -1234) -1.0-1234.0i)
(num-test (make-polar 1 1234) -0.79855062358758+0.60192765476250i)
(num-test (make-polar -1 1234) 0.79855062358758-0.60192765476250i)
(num-test (make-polar 1 -1234) -0.79855062358758-0.60192765476250i)
(num-test (make-polar -1 -1234) 0.79855062358758+0.60192765476250i)
(num-test (make-rectangular 1 1234000000) 1.0+1234000000.0i)
(num-test (make-rectangular -1 1234000000) -1.0+1234000000.0i)
(num-test (make-rectangular 1 -1234000000) 1.0-1234000000.0i)
(num-test (make-rectangular -1 -1234000000) -1.0-1234000000.0i)
(num-test (make-polar 1 1234000000) 0.15890913095152-0.98729321283003i)
(num-test (make-polar -1 1234000000) -0.15890913095152+0.98729321283003i)
(num-test (make-polar 1 -1234000000) 0.15890913095152+0.98729321283003i)
(num-test (make-polar -1 -1234000000) -0.15890913095152-0.98729321283003i)
(num-test (make-rectangular 1 500029) 1.0+500029.0i)
(num-test (make-rectangular -1 500029) -1.0+500029.0i)
(num-test (make-rectangular 1 -500029) 1.0-500029.0i)
(num-test (make-rectangular -1 -500029) -1.0-500029.0i)
(num-test (make-polar 1 500029) 0.85414905629947+0.52002825848479i)
(num-test (make-polar -1 500029) -0.85414905629947-0.52002825848479i)
(num-test (make-polar 1 -500029) 0.85414905629947-0.52002825848479i)
(num-test (make-polar -1 -500029) -0.85414905629947+0.52002825848479i)
(num-test (make-rectangular 1 362880) 1.0+362880.0i)
(num-test (make-rectangular -1 362880) -1.0+362880.0i)
(num-test (make-rectangular 1 -362880) 1.0-362880.0i)
(num-test (make-rectangular -1 -362880) -1.0-362880.0i)
(num-test (make-polar 1 362880) 0.60918079547638+0.79303137291204i)
(num-test (make-polar -1 362880) -0.60918079547638-0.79303137291204i)
(num-test (make-polar 1 -362880) 0.60918079547638-0.79303137291204i)
(num-test (make-polar -1 -362880) -0.60918079547638+0.79303137291204i)
(num-test (make-rectangular 2 1) 2.0+1.0i)
(num-test (make-rectangular -2 1) -2.0+1.0i)
(num-test (make-rectangular 2 -1) 2.0-1.0i)
(num-test (make-rectangular -2 -1) -2.0-1.0i)
(num-test (make-polar 2 1) 1.08060461173628+1.68294196961579i)
(num-test (make-polar -2 1) -1.08060461173628-1.68294196961579i)
(num-test (make-polar 2 -1) 1.08060461173628-1.68294196961579i)
(num-test (make-polar -2 -1) -1.08060461173628+1.68294196961579i)
(num-test (make-rectangular 2 2) 2.0+2.0i)
(num-test (make-rectangular -2 2) -2.0+2.0i)
(num-test (make-rectangular 2 -2) 2.0-2.0i)
(num-test (make-rectangular -2 -2) -2.0-2.0i)
(num-test (make-polar 2 2) -0.83229367309428+1.81859485365136i)
(num-test (make-polar -2 2) 0.83229367309428-1.81859485365136i)
(num-test (make-polar 2 -2) -0.83229367309428-1.81859485365136i)
(num-test (make-polar -2 -2) 0.83229367309428+1.81859485365136i)
(num-test (make-rectangular 2 3) 2.0+3.0i)
(num-test (make-rectangular -2 3) -2.0+3.0i)
(num-test (make-rectangular 2 -3) 2.0-3.0i)
(num-test (make-rectangular -2 -3) -2.0-3.0i)
(num-test (make-polar 2 3) -1.97998499320089+0.28224001611973i)
(num-test (make-polar -2 3) 1.97998499320089-0.28224001611973i)
(num-test (make-polar 2 -3) -1.97998499320089-0.28224001611973i)
(num-test (make-polar -2 -3) 1.97998499320089+0.28224001611973i)
(num-test (make-rectangular 2 10) 2.0+10.0i)
(num-test (make-rectangular -2 10) -2.0+10.0i)
(num-test (make-rectangular 2 -10) 2.0-10.0i)
(num-test (make-rectangular -2 -10) -2.0-10.0i)
(num-test (make-polar 2 10) -1.67814305815290-1.08804222177874i)
(num-test (make-polar -2 10) 1.67814305815290+1.08804222177874i)
(num-test (make-polar 2 -10) -1.67814305815290+1.08804222177874i)
(num-test (make-polar -2 -10) 1.67814305815290-1.08804222177874i)
(num-test (make-rectangular 2 1234) 2.0+1234.0i)
(num-test (make-rectangular -2 1234) -2.0+1234.0i)
(num-test (make-rectangular 2 -1234) 2.0-1234.0i)
(num-test (make-rectangular -2 -1234) -2.0-1234.0i)
(num-test (make-polar 2 1234) -1.59710124717517+1.20385530952499i)
(num-test (make-polar -2 1234) 1.59710124717517-1.20385530952499i)
(num-test (make-polar 2 -1234) -1.59710124717517-1.20385530952499i)
(num-test (make-polar -2 -1234) 1.59710124717517+1.20385530952499i)
(num-test (make-rectangular 2 1234000000) 2.0+1234000000.0i)
(num-test (make-rectangular -2 1234000000) -2.0+1234000000.0i)
(num-test (make-rectangular 2 -1234000000) 2.0-1234000000.0i)
(num-test (make-rectangular -2 -1234000000) -2.0-1234000000.0i)
(num-test (make-polar 2 1234000000) 0.31781826190303-1.97458642566005i)
(num-test (make-polar -2 1234000000) -0.31781826190303+1.97458642566005i)
(num-test (make-polar 2 -1234000000) 0.31781826190303+1.97458642566005i)
(num-test (make-polar -2 -1234000000) -0.31781826190303-1.97458642566005i)
(num-test (make-rectangular 2 500029) 2.0+500029.0i)
(num-test (make-rectangular -2 500029) -2.0+500029.0i)
(num-test (make-rectangular 2 -500029) 2.0-500029.0i)
(num-test (make-rectangular -2 -500029) -2.0-500029.0i)
(num-test (make-polar 2 500029) 1.70829811259895+1.04005651696957i)
(num-test (make-polar -2 500029) -1.70829811259895-1.04005651696957i)
(num-test (make-polar 2 -500029) 1.70829811259895-1.04005651696957i)
(num-test (make-polar -2 -500029) -1.70829811259895+1.04005651696957i)
(num-test (make-rectangular 2 362880) 2.0+362880.0i)
(num-test (make-rectangular -2 362880) -2.0+362880.0i)
(num-test (make-rectangular 2 -362880) 2.0-362880.0i)
(num-test (make-rectangular -2 -362880) -2.0-362880.0i)
(num-test (make-polar 2 362880) 1.21836159095277+1.58606274582408i)
(num-test (make-polar -2 362880) -1.21836159095277-1.58606274582408i)
(num-test (make-polar 2 -362880) 1.21836159095277-1.58606274582408i)
(num-test (make-polar -2 -362880) -1.21836159095277+1.58606274582408i)
(num-test (make-rectangular 3 1) 3.0+1.0i)
(num-test (make-rectangular -3 1) -3.0+1.0i)
(num-test (make-rectangular 3 -1) 3.0-1.0i)
(num-test (make-rectangular -3 -1) -3.0-1.0i)
(num-test (make-polar 3 1) 1.62090691760442+2.52441295442369i)
(num-test (make-polar -3 1) -1.62090691760442-2.52441295442369i)
(num-test (make-polar 3 -1) 1.62090691760442-2.52441295442369i)
(num-test (make-polar -3 -1) -1.62090691760442+2.52441295442369i)
(num-test (make-rectangular 3 2) 3.0+2.0i)
(num-test (make-rectangular -3 2) -3.0+2.0i)
(num-test (make-rectangular 3 -2) 3.0-2.0i)
(num-test (make-rectangular -3 -2) -3.0-2.0i)
(num-test (make-polar 3 2) -1.24844050964143+2.72789228047704i)
(num-test (make-polar -3 2) 1.24844050964143-2.72789228047704i)
(num-test (make-polar 3 -2) -1.24844050964143-2.72789228047704i)
(num-test (make-polar -3 -2) 1.24844050964143+2.72789228047704i)
(num-test (make-rectangular 3 3) 3.0+3.0i)
(num-test (make-rectangular -3 3) -3.0+3.0i)
(num-test (make-rectangular 3 -3) 3.0-3.0i)
(num-test (make-rectangular -3 -3) -3.0-3.0i)
(num-test (make-polar 3 3) -2.96997748980134+0.42336002417960i)
(num-test (make-polar -3 3) 2.96997748980134-0.42336002417960i)
(num-test (make-polar 3 -3) -2.96997748980134-0.42336002417960i)
(num-test (make-polar -3 -3) 2.96997748980134+0.42336002417960i)
(num-test (make-rectangular 3 10) 3.0+10.0i)
(num-test (make-rectangular -3 10) -3.0+10.0i)
(num-test (make-rectangular 3 -10) 3.0-10.0i)
(num-test (make-rectangular -3 -10) -3.0-10.0i)
(num-test (make-polar 3 10) -2.51721458722936-1.63206333266811i)
(num-test (make-polar -3 10) 2.51721458722936+1.63206333266811i)
(num-test (make-polar 3 -10) -2.51721458722936+1.63206333266811i)
(num-test (make-polar -3 -10) 2.51721458722936-1.63206333266811i)
(num-test (make-rectangular 3 1234) 3.0+1234.0i)
(num-test (make-rectangular -3 1234) -3.0+1234.0i)
(num-test (make-rectangular 3 -1234) 3.0-1234.0i)
(num-test (make-rectangular -3 -1234) -3.0-1234.0i)
(num-test (make-polar 3 1234) -2.39565187076275+1.80578296428749i)
(num-test (make-polar -3 1234) 2.39565187076275-1.80578296428749i)
(num-test (make-polar 3 -1234) -2.39565187076275-1.80578296428749i)
(num-test (make-polar -3 -1234) 2.39565187076275+1.80578296428749i)
(num-test (make-rectangular 3 1234000000) 3.0+1234000000.0i)
(num-test (make-rectangular -3 1234000000) -3.0+1234000000.0i)
(num-test (make-rectangular 3 -1234000000) 3.0-1234000000.0i)
(num-test (make-rectangular -3 -1234000000) -3.0-1234000000.0i)
(num-test (make-polar 3 1234000000) 0.47672739285455-2.96187963849008i)
(num-test (make-polar -3 1234000000) -0.47672739285455+2.96187963849008i)
(num-test (make-polar 3 -1234000000) 0.47672739285455+2.96187963849008i)
(num-test (make-polar -3 -1234000000) -0.47672739285455-2.96187963849008i)
(num-test (make-rectangular 3 500029) 3.0+500029.0i)
(num-test (make-rectangular -3 500029) -3.0+500029.0i)
(num-test (make-rectangular 3 -500029) 3.0-500029.0i)
(num-test (make-rectangular -3 -500029) -3.0-500029.0i)
(num-test (make-polar 3 500029) 2.56244716889842+1.56008477545436i)
(num-test (make-polar -3 500029) -2.56244716889842-1.56008477545436i)
(num-test (make-polar 3 -500029) 2.56244716889842-1.56008477545436i)
(num-test (make-polar -3 -500029) -2.56244716889842+1.56008477545436i)
(num-test (make-rectangular 3 362880) 3.0+362880.0i)
(num-test (make-rectangular -3 362880) -3.0+362880.0i)
(num-test (make-rectangular 3 -362880) 3.0-362880.0i)
(num-test (make-rectangular -3 -362880) -3.0-362880.0i)
(num-test (make-polar 3 362880) 1.82754238642915+2.37909411873613i)
(num-test (make-polar -3 362880) -1.82754238642915-2.37909411873613i)
(num-test (make-polar 3 -362880) 1.82754238642915-2.37909411873613i)
(num-test (make-polar -3 -362880) -1.82754238642915+2.37909411873613i)
(num-test (make-rectangular 10 1) 10.0+1.0i)
(num-test (make-rectangular -10 1) -10.0+1.0i)
(num-test (make-rectangular 10 -1) 10.0-1.0i)
(num-test (make-rectangular -10 -1) -10.0-1.0i)
(num-test (make-polar 10 1) 5.40302305868140+8.41470984807897i)
(num-test (make-polar -10 1) -5.40302305868140-8.41470984807897i)
(num-test (make-polar 10 -1) 5.40302305868140-8.41470984807897i)
(num-test (make-polar -10 -1) -5.40302305868140+8.41470984807897i)
(num-test (make-rectangular 10 2) 10.0+2.0i)
(num-test (make-rectangular -10 2) -10.0+2.0i)
(num-test (make-rectangular 10 -2) 10.0-2.0i)
(num-test (make-rectangular -10 -2) -10.0-2.0i)
(num-test (make-polar 10 2) -4.16146836547142+9.09297426825682i)
(num-test (make-polar -10 2) 4.16146836547142-9.09297426825682i)
(num-test (make-polar 10 -2) -4.16146836547142-9.09297426825682i)
(num-test (make-polar -10 -2) 4.16146836547142+9.09297426825682i)
(num-test (make-rectangular 10 3) 10.0+3.0i)
(num-test (make-rectangular -10 3) -10.0+3.0i)
(num-test (make-rectangular 10 -3) 10.0-3.0i)
(num-test (make-rectangular -10 -3) -10.0-3.0i)
(num-test (make-polar 10 3) -9.89992496600445+1.41120008059867i)
(num-test (make-polar -10 3) 9.89992496600445-1.41120008059867i)
(num-test (make-polar 10 -3) -9.89992496600445-1.41120008059867i)
(num-test (make-polar -10 -3) 9.89992496600445+1.41120008059867i)
(num-test (make-rectangular 10 10) 10.0+10.0i)
(num-test (make-rectangular -10 10) -10.0+10.0i)
(num-test (make-rectangular 10 -10) 10.0-10.0i)
(num-test (make-rectangular -10 -10) -10.0-10.0i)
(num-test (make-polar 10 10) -8.39071529076452-5.44021110889370i)
(num-test (make-polar -10 10) 8.39071529076452+5.44021110889370i)
(num-test (make-polar 10 -10) -8.39071529076452+5.44021110889370i)
(num-test (make-polar -10 -10) 8.39071529076452-5.44021110889370i)
(num-test (make-rectangular 10 1234) 10.0+1234.0i)
(num-test (make-rectangular -10 1234) -10.0+1234.0i)
(num-test (make-rectangular 10 -1234) 10.0-1234.0i)
(num-test (make-rectangular -10 -1234) -10.0-1234.0i)
(num-test (make-polar 10 1234) -7.98550623587584+6.01927654762497i)
(num-test (make-polar -10 1234) 7.98550623587584-6.01927654762497i)
(num-test (make-polar 10 -1234) -7.98550623587584-6.01927654762497i)
(num-test (make-polar -10 -1234) 7.98550623587584+6.01927654762497i)
(num-test (make-rectangular 10 1234000000) 10.0+1234000000.0i)
(num-test (make-rectangular -10 1234000000) -10.0+1234000000.0i)
(num-test (make-rectangular 10 -1234000000) 10.0-1234000000.0i)
(num-test (make-rectangular -10 -1234000000) -10.0-1234000000.0i)
(num-test (make-polar 10 1234000000) 1.58909130951517-9.87293212830025i)
(num-test (make-polar -10 1234000000) -1.58909130951517+9.87293212830025i)
(num-test (make-polar 10 -1234000000) 1.58909130951517+9.87293212830025i)
(num-test (make-polar -10 -1234000000) -1.58909130951517-9.87293212830025i)
(num-test (make-rectangular 10 500029) 10.0+500029.0i)
(num-test (make-rectangular -10 500029) -10.0+500029.0i)
(num-test (make-rectangular 10 -500029) 10.0-500029.0i)
(num-test (make-rectangular -10 -500029) -10.0-500029.0i)
(num-test (make-polar 10 500029) 8.54149056299473+5.20028258484786i)
(num-test (make-polar -10 500029) -8.54149056299473-5.20028258484786i)
(num-test (make-polar 10 -500029) 8.54149056299473-5.20028258484786i)
(num-test (make-polar -10 -500029) -8.54149056299473+5.20028258484786i)
(num-test (make-rectangular 10 362880) 10.0+362880.0i)
(num-test (make-rectangular -10 362880) -10.0+362880.0i)
(num-test (make-rectangular 10 -362880) 10.0-362880.0i)
(num-test (make-rectangular -10 -362880) -10.0-362880.0i)
(num-test (make-polar 10 362880) 6.09180795476385+7.93031372912042i)
(num-test (make-polar -10 362880) -6.09180795476385-7.93031372912042i)
(num-test (make-polar 10 -362880) 6.09180795476385-7.93031372912042i)
(num-test (make-polar -10 -362880) -6.09180795476385+7.93031372912042i)
(num-test (make-rectangular 1234 1) 1234.0+1.0i)
(num-test (make-rectangular -1234 1) -1234.0+1.0i)
(num-test (make-rectangular 1234 -1) 1234.0-1.0i)
(num-test (make-rectangular -1234 -1) -1234.0-1.0i)
(num-test (make-polar 1234 1) 666.73304544128450+1038.37519525294420i)
(num-test (make-polar -1234 1) -666.73304544128450-1038.37519525294420i)
(num-test (make-polar 1234 -1) 666.73304544128450-1038.37519525294420i)
(num-test (make-polar -1234 -1) -666.73304544128450+1038.37519525294420i)
(num-test (make-rectangular 1234 2) 1234.0+2.0i)
(num-test (make-rectangular -1234 2) -1234.0+2.0i)
(num-test (make-rectangular 1234 -2) 1234.0-2.0i)
(num-test (make-rectangular -1234 -2) -1234.0-2.0i)
(num-test (make-polar 1234 2) -513.52519629917379+1122.07302470289119i)
(num-test (make-polar -1234 2) 513.52519629917379-1122.07302470289119i)
(num-test (make-polar 1234 -2) -513.52519629917379-1122.07302470289119i)
(num-test (make-polar -1234 -2) 513.52519629917379+1122.07302470289119i)
(num-test (make-rectangular 1234 3) 1234.0+3.0i)
(num-test (make-rectangular -1234 3) -1234.0+3.0i)
(num-test (make-rectangular 1234 -3) 1234.0-3.0i)
(num-test (make-rectangular -1234 -3) -1234.0-3.0i)
(num-test (make-polar 1234 3) -1221.65074080494969+174.14208994587614i)
(num-test (make-polar -1234 3) 1221.65074080494969-174.14208994587614i)
(num-test (make-polar 1234 -3) -1221.65074080494969-174.14208994587614i)
(num-test (make-polar -1234 -3) 1221.65074080494969+174.14208994587614i)
(num-test (make-rectangular 1234 10) 1234.0+10.0i)
(num-test (make-rectangular -1234 10) -1234.0+10.0i)
(num-test (make-rectangular 1234 -10) 1234.0-10.0i)
(num-test (make-rectangular -1234 -10) -1234.0-10.0i)
(num-test (make-polar 1234 10) -1035.41426688034221-671.32205083748227i)
(num-test (make-polar -1234 10) 1035.41426688034221+671.32205083748227i)
(num-test (make-polar 1234 -10) -1035.41426688034221+671.32205083748227i)
(num-test (make-polar -1234 -10) 1035.41426688034221-671.32205083748227i)
(num-test (make-rectangular 1234 1234) 1234.0+1234.0i)
(num-test (make-rectangular -1234 1234) -1234.0+1234.0i)
(num-test (make-rectangular 1234 -1234) 1234.0-1234.0i)
(num-test (make-rectangular -1234 -1234) -1234.0-1234.0i)
(num-test (make-polar 1234 1234) -985.41146950707912+742.77872597692169i)
(num-test (make-polar -1234 1234) 985.41146950707912-742.77872597692169i)
(num-test (make-polar 1234 -1234) -985.41146950707912-742.77872597692169i)
(num-test (make-polar -1234 -1234) 985.41146950707912+742.77872597692169i)
(num-test (make-rectangular 1234 1234000000) 1234.0+1234000000.0i)
(num-test (make-rectangular -1234 1234000000) -1234.0+1234000000.0i)
(num-test (make-rectangular 1234 -1234000000) 1234.0-1234000000.0i)
(num-test (make-rectangular -1234 -1234000000) -1234.0-1234000000.0i)
(num-test (make-polar 1234 1234000000) 196.09386759417183-1218.31982463225131i)
(num-test (make-polar -1234 1234000000) -196.09386759417183+1218.31982463225131i)
(num-test (make-polar 1234 -1234000000) 196.09386759417183+1218.31982463225131i)
(num-test (make-polar -1234 -1234000000) -196.09386759417183-1218.31982463225131i)
(num-test (make-rectangular 1234 500029) 1234.0+500029.0i)
(num-test (make-rectangular -1234 500029) -1234.0+500029.0i)
(num-test (make-rectangular 1234 -500029) 1234.0-500029.0i)
(num-test (make-rectangular -1234 -500029) -1234.0-500029.0i)
(num-test (make-polar 1234 500029) 1054.01993547355005+641.71487097022577i)
(num-test (make-polar -1234 500029) -1054.01993547355005-641.71487097022577i)
(num-test (make-polar 1234 -500029) 1054.01993547355005-641.71487097022577i)
(num-test (make-polar -1234 -500029) -1054.01993547355005+641.71487097022577i)
(num-test (make-rectangular 1234 362880) 1234.0+362880.0i)
(num-test (make-rectangular -1234 362880) -1234.0+362880.0i)
(num-test (make-rectangular 1234 -362880) 1234.0-362880.0i)
(num-test (make-rectangular -1234 -362880) -1234.0-362880.0i)
(num-test (make-polar 1234 362880) 751.72910161785899+978.60071417346035i)
(num-test (make-polar -1234 362880) -751.72910161785899-978.60071417346035i)
(num-test (make-polar 1234 -362880) 751.72910161785899-978.60071417346035i)
(num-test (make-polar -1234 -362880) -751.72910161785899+978.60071417346035i)
(num-test (make-rectangular 1234000000 1) 1234000000.0+1.0i)
(num-test (make-rectangular -1234000000 1) -1234000000.0+1.0i)
(num-test (make-rectangular 1234000000 -1) 1234000000.0-1.0i)
(num-test (make-rectangular -1234000000 -1) -1234000000.0-1.0i)
(num-test (make-polar 1234000000 1) 666733045.44128441810608+1038375195.25294423103333i)
(num-test (make-polar -1234000000 1) -666733045.44128441810608-1038375195.25294423103333i)
(num-test (make-polar 1234000000 -1) 666733045.44128441810608-1038375195.25294423103333i)
(num-test (make-polar -1234000000 -1) -666733045.44128441810608+1038375195.25294423103333i)
(num-test (make-rectangular 1234000000 2) 1234000000.0+2.0i)
(num-test (make-rectangular -1234000000 2) -1234000000.0+2.0i)
(num-test (make-rectangular 1234000000 -2) 1234000000.0-2.0i)
(num-test (make-rectangular -1234000000 -2) -1234000000.0-2.0i)
(num-test (make-polar 1234000000 2) -513525196.29917371273041+1122073024.70289111137390i)
(num-test (make-polar -1234000000 2) 513525196.29917371273041-1122073024.70289111137390i)
(num-test (make-polar 1234000000 -2) -513525196.29917371273041-1122073024.70289111137390i)
(num-test (make-polar -1234000000 -2) 513525196.29917371273041+1122073024.70289111137390i)
(num-test (make-rectangular 1234000000 3) 1234000000.0+3.0i)
(num-test (make-rectangular -1234000000 3) -1234000000.0+3.0i)
(num-test (make-rectangular 1234000000 -3) 1234000000.0-3.0i)
(num-test (make-rectangular -1234000000 -3) -1234000000.0-3.0i)
(num-test (make-polar 1234000000 3) -1221650740.80494976043701+174142089.94587615132332i)
(num-test (make-polar -1234000000 3) 1221650740.80494976043701-174142089.94587615132332i)
(num-test (make-polar 1234000000 -3) -1221650740.80494976043701-174142089.94587615132332i)
(num-test (make-polar -1234000000 -3) 1221650740.80494976043701+174142089.94587615132332i)
(num-test (make-rectangular 1234000000 10) 1234000000.0+10.0i)
(num-test (make-rectangular -1234000000 10) -1234000000.0+10.0i)
(num-test (make-rectangular 1234000000 -10) 1234000000.0-10.0i)
(num-test (make-rectangular -1234000000 -10) -1234000000.0-10.0i)
(num-test (make-polar 1234000000 10) -1035414266.88034236431122-671322050.83748233318329i)
(num-test (make-polar -1234000000 10) 1035414266.88034236431122+671322050.83748233318329i)
(num-test (make-polar 1234000000 -10) -1035414266.88034236431122+671322050.83748233318329i)
(num-test (make-polar -1234000000 -10) 1035414266.88034236431122-671322050.83748233318329i)
(num-test (make-rectangular 1234000000 1234) 1234000000.0+1234.0i)
(num-test (make-rectangular -1234000000 1234) -1234000000.0+1234.0i)
(num-test (make-rectangular 1234000000 -1234) 1234000000.0-1234.0i)
(num-test (make-rectangular -1234000000 -1234) -1234000000.0-1234.0i)
(num-test (make-polar 1234000000 1234) -985411469.50707900524139+742778725.97692167758942i)
(num-test (make-polar -1234000000 1234) 985411469.50707900524139-742778725.97692167758942i)
(num-test (make-polar 1234000000 -1234) -985411469.50707900524139-742778725.97692167758942i)
(num-test (make-polar -1234000000 -1234) 985411469.50707900524139+742778725.97692167758942i)
(num-test (make-rectangular 1234000000 1234000000) 1234000000.0+1234000000.0i)
(num-test (make-rectangular -1234000000 1234000000) -1234000000.0+1234000000.0i)
(num-test (make-rectangular 1234000000 -1234000000) 1234000000.0-1234000000.0i)
(num-test (make-rectangular -1234000000 -1234000000) -1234000000.0-1234000000.0i)
(num-test (make-polar 1234000000 1234000000) 196093867.59417182207108-1218319824.63225126266479i)
(num-test (make-polar -1234000000 1234000000) -196093867.59417182207108+1218319824.63225126266479i)
(num-test (make-polar 1234000000 -1234000000) 196093867.59417182207108+1218319824.63225126266479i)
(num-test (make-polar -1234000000 -1234000000) -196093867.59417182207108-1218319824.63225126266479i)
(num-test (make-rectangular 1234000000 500029) 1234000000.0+500029.0i)
(num-test (make-rectangular -1234000000 500029) -1234000000.0+500029.0i)
(num-test (make-rectangular 1234000000 -500029) 1234000000.0-500029.0i)
(num-test (make-rectangular -1234000000 -500029) -1234000000.0-500029.0i)
(num-test (make-polar 1234000000 500029) 1054019935.47354996204376+641714870.97022569179535i)
(num-test (make-polar -1234000000 500029) -1054019935.47354996204376-641714870.97022569179535i)
(num-test (make-polar 1234000000 -500029) 1054019935.47354996204376-641714870.97022569179535i)
(num-test (make-polar -1234000000 -500029) -1054019935.47354996204376+641714870.97022569179535i)
(num-test (make-rectangular 1234000000 362880) 1234000000.0+362880.0i)
(num-test (make-rectangular -1234000000 362880) -1234000000.0+362880.0i)
(num-test (make-rectangular 1234000000 -362880) 1234000000.0-362880.0i)
(num-test (make-rectangular -1234000000 -362880) -1234000000.0-362880.0i)
(num-test (make-polar 1234000000 362880) 751729101.61785900592804+978600714.17346036434174i)
(num-test (make-polar -1234000000 362880) -751729101.61785900592804-978600714.17346036434174i)
(num-test (make-polar 1234000000 -362880) 751729101.61785900592804-978600714.17346036434174i)
(num-test (make-polar -1234000000 -362880) -751729101.61785900592804+978600714.17346036434174i)
(num-test (make-rectangular 500029 1) 500029.0+1.0i)
(num-test (make-rectangular -500029 1) -500029.0+1.0i)
(num-test (make-rectangular 500029 -1) 500029.0-1.0i)
(num-test (make-rectangular -500029 -1) -500029.0-1.0i)
(num-test (make-polar 500029 1) 270166.82170094008325+420759.89506250765407i)
(num-test (make-polar -500029 1) -270166.82170094008325-420759.89506250765407i)
(num-test (make-polar 500029 -1) 270166.82170094008325-420759.89506250765407i)
(num-test (make-polar -500029 -1) -270166.82170094008325+420759.89506250765407i)
(num-test (make-rectangular 500029 2) 500029.0+2.0i)
(num-test (make-rectangular -500029 2) -500029.0+2.0i)
(num-test (make-rectangular 500029 -2) 500029.0-2.0i)
(num-test (make-rectangular -500029 -2) -500029.0-2.0i)
(num-test (make-polar 500029 2) -208085.48653183106217+454675.08303821878508i)
(num-test (make-polar -500029 2) 208085.48653183106217-454675.08303821878508i)
(num-test (make-polar 500029 -2) -208085.48653183106217-454675.08303821878508i)
(num-test (make-polar -500029 -2) 208085.48653183106217+454675.08303821878508i)
(num-test (make-rectangular 500029 3) 500029.0+3.0i)
(num-test (make-rectangular -500029 3) -500029.0+3.0i)
(num-test (make-rectangular 500029 -3) 500029.0-3.0i)
(num-test (make-rectangular -500029 -3) -500029.0-3.0i)
(num-test (make-polar 500029 3) -495024.95808262412902+70564.09651016735006i)
(num-test (make-polar -500029 3) 495024.95808262412902-70564.09651016735006i)
(num-test (make-polar 500029 -3) -495024.95808262412902-70564.09651016735006i)
(num-test (make-polar -500029 -3) 495024.95808262412902+70564.09651016735006i)
(num-test (make-rectangular 500029 10) 500029.0+10.0i)
(num-test (make-rectangular -500029 10) -500029.0+10.0i)
(num-test (make-rectangular 500029 -10) 500029.0-10.0i)
(num-test (make-rectangular -500029 -10) -500029.0-10.0i)
(num-test (make-polar 500029 10) -419560.09761256945785-272026.33205690066097i)
(num-test (make-polar -500029 10) 419560.09761256945785+272026.33205690066097i)
(num-test (make-polar 500029 -10) -419560.09761256945785+272026.33205690066097i)
(num-test (make-polar -500029 -10) 419560.09761256945785-272026.33205690066097i)
(num-test (make-rectangular 500029 1234) 500029.0+1234.0i)
(num-test (make-rectangular -500029 1234) -500029.0+1234.0i)
(num-test (make-rectangular 500029 -1234) 500029.0-1234.0i)
(num-test (make-rectangular -500029 -1234) -500029.0-1234.0i)
(num-test (make-polar 500029 1234) -399298.46976187621476+300981.28328323678579i)
(num-test (make-polar -500029 1234) 399298.46976187621476-300981.28328323678579i)
(num-test (make-polar 500029 -1234) -399298.46976187621476-300981.28328323678579i)
(num-test (make-polar -500029 -1234) 399298.46976187621476+300981.28328323678579i)
(num-test (make-rectangular 500029 1234000000) 500029.0+1234000000.0i)
(num-test (make-rectangular -500029 1234000000) -500029.0+1234000000.0i)
(num-test (make-rectangular 500029 -1234000000) 500029.0-1234000000.0i)
(num-test (make-rectangular -500029 -1234000000) -500029.0-1234000000.0i)
(num-test (make-polar 500029 1234000000) 79459.17384055603179-493675.23791818472091i)
(num-test (make-polar -500029 1234000000) -79459.17384055603179+493675.23791818472091i)
(num-test (make-polar 500029 -1234000000) 79459.17384055603179+493675.23791818472091i)
(num-test (make-polar -500029 -1234000000) -79459.17384055603179-493675.23791818472091i)
(num-test (make-rectangular 500029 500029) 500029.0+500029.0i)
(num-test (make-rectangular -500029 500029) -500029.0+500029.0i)
(num-test (make-rectangular 500029 -500029) 500029.0-500029.0i)
(num-test (make-rectangular -500029 -500029) -500029.0-500029.0i)
(num-test (make-polar 500029 500029) 427099.29847236932255+260029.21006188896718i)
(num-test (make-polar -500029 500029) -427099.29847236932255-260029.21006188896718i)
(num-test (make-polar 500029 -500029) 427099.29847236932255-260029.21006188896718i)
(num-test (make-polar -500029 -500029) -427099.29847236932255+260029.21006188896718i)
(num-test (make-rectangular 500029 362880) 500029.0+362880.0i)
(num-test (make-rectangular -500029 362880) -500029.0+362880.0i)
(num-test (make-rectangular 500029 -362880) 500029.0-362880.0i)
(num-test (make-rectangular -500029 -362880) -500029.0-362880.0i)
(num-test (make-polar 500029 362880) 304608.06398126127897+396538.68436583562288i)
(num-test (make-polar -500029 362880) -304608.06398126127897-396538.68436583562288i)
(num-test (make-polar 500029 -362880) 304608.06398126127897-396538.68436583562288i)
(num-test (make-polar -500029 -362880) -304608.06398126127897+396538.68436583562288i)
(num-test (make-rectangular 362880 1) 362880.0+1.0i)
(num-test (make-rectangular -362880 1) -362880.0+1.0i)
(num-test (make-rectangular 362880 -1) 362880.0-1.0i)
(num-test (make-rectangular -362880 -1) -362880.0-1.0i)
(num-test (make-polar 362880 1) 196064.90075343055651+305352.99096708948491i)
(num-test (make-polar -362880 1) -196064.90075343055651-305352.99096708948491i)
(num-test (make-polar 362880 -1) 196064.90075343055651-305352.99096708948491i)
(num-test (make-polar -362880 -1) -196064.90075343055651+305352.99096708948491i)
(num-test (make-rectangular 362880 2) 362880.0+2.0i)
(num-test (make-rectangular -362880 2) -362880.0+2.0i)
(num-test (make-rectangular 362880 -2) 362880.0-2.0i)
(num-test (make-rectangular -362880 -2) -362880.0-2.0i)
(num-test (make-polar 362880 2) -151011.36404622704140+329965.85024650336709i)
(num-test (make-polar -362880 2) 151011.36404622704140-329965.85024650336709i)
(num-test (make-polar 362880 -2) -151011.36404622704140-329965.85024650336709i)
(num-test (make-polar -362880 -2) 151011.36404622704140+329965.85024650336709i)
(num-test (make-rectangular 362880 3) 362880.0+3.0i)
(num-test (make-rectangular -362880 3) -362880.0+3.0i)
(num-test (make-rectangular 362880 -3) 362880.0-3.0i)
(num-test (make-rectangular -362880 -3) -362880.0-3.0i)
(num-test (make-polar 362880 3) -359248.47716636961559+51209.62852476461558i)
(num-test (make-polar -362880 3) 359248.47716636961559-51209.62852476461558i)
(num-test (make-polar 362880 -3) -359248.47716636961559-51209.62852476461558i)
(num-test (make-polar -362880 -3) 359248.47716636961559+51209.62852476461558i)
(num-test (make-rectangular 362880 10) 362880.0+10.0i)
(num-test (make-rectangular -362880 10) -362880.0+10.0i)
(num-test (make-rectangular 362880 -10) 362880.0-10.0i)
(num-test (make-rectangular -362880 -10) -362880.0-10.0i)
(num-test (make-polar 362880 10) -304482.27647126308875-197414.38071953449980i)
(num-test (make-polar -362880 10) 304482.27647126308875+197414.38071953449980i)
(num-test (make-polar 362880 -10) -304482.27647126308875+197414.38071953449980i)
(num-test (make-polar -362880 -10) 304482.27647126308875-197414.38071953449980i)
(num-test (make-rectangular 362880 1234) 362880.0+1234.0i)
(num-test (make-rectangular -362880 1234) -362880.0+1234.0i)
(num-test (make-rectangular 362880 -1234) 362880.0-1234.0i)
(num-test (make-rectangular -362880 -1234) -362880.0-1234.0i)
(num-test (make-polar 362880 1234) -289778.05028746259632+218427.50736021503690i)
(num-test (make-polar -362880 1234) 289778.05028746259632-218427.50736021503690i)
(num-test (make-polar 362880 -1234) -289778.05028746259632-218427.50736021503690i)
(num-test (make-polar -362880 -1234) 289778.05028746259632+218427.50736021503690i)
(num-test (make-rectangular 362880 1234000000) 362880.0+1234000000.0i)
(num-test (make-rectangular -362880 1234000000) -362880.0+1234000000.0i)
(num-test (make-rectangular 362880 -1234000000) 362880.0-1234000000.0i)
(num-test (make-rectangular -362880 -1234000000) -362880.0-1234000000.0i)
(num-test (make-polar 362880 1234000000) 57664.94543968644575-358268.96107175957877i)
(num-test (make-polar -362880 1234000000) -57664.94543968644575+358268.96107175957877i)
(num-test (make-polar 362880 -1234000000) 57664.94543968644575+358268.96107175957877i)
(num-test (make-polar -362880 -1234000000) -57664.94543968644575-358268.96107175957877i)
(num-test (make-rectangular 362880 500029) 362880.0+500029.0i)
(num-test (make-rectangular -362880 500029) -362880.0+500029.0i)
(num-test (make-rectangular 362880 -500029) 362880.0-500029.0i)
(num-test (make-rectangular -362880 -500029) -362880.0-500029.0i)
(num-test (make-polar 362880 500029) 309953.60954995284555+188707.85443895909702i)
(num-test (make-polar -362880 500029) -309953.60954995284555-188707.85443895909702i)
(num-test (make-polar 362880 -500029) 309953.60954995284555-188707.85443895909702i)
(num-test (make-polar -362880 -500029) -309953.60954995284555+188707.85443895909702i)
(num-test (make-rectangular 362880 362880) 362880.0+362880.0i)
(num-test (make-rectangular -362880 362880) -362880.0+362880.0i)
(num-test (make-rectangular 362880 -362880) 362880.0-362880.0i)
(num-test (make-rectangular -362880 -362880) -362880.0-362880.0i)
(num-test (make-polar 362880 362880) 221059.52706247055903+287775.22460232191952i)
(num-test (make-polar -362880 362880) -221059.52706247055903-287775.22460232191952i)
(num-test (make-polar 362880 -362880) 221059.52706247055903-287775.22460232191952i)
(num-test (make-polar -362880 -362880) -221059.52706247055903+287775.22460232191952i)
(num-test (make-rectangular 0.0 0.00000001) 0.0+0.00000001i)
(num-test (make-rectangular -0.0 0.00000001) -0.0+0.00000001i)
(num-test (make-rectangular 0.0 -0.00000001) 0.0-0.00000001i)
(num-test (make-rectangular -0.0 -0.00000001) -0.0-0.00000001i)
(num-test (make-polar 0.0 0.00000001) 0.0)
(num-test (make-polar -0.0 0.00000001) -0.0)
(num-test (make-polar 0.0 -0.00000001) 0.0)
(num-test (make-polar -0.0 -0.00000001) 0.0)
(num-test (make-rectangular 0.0 1.0) 0.0+1.0i)
(num-test (make-rectangular -0.0 1.0) -0.0+1.0i)
(num-test (make-rectangular 0.0 -1.0) 0.0-1.0i)
(num-test (make-rectangular -0.0 -1.0) -0.0-1.0i)
(num-test (make-polar 0.0 1.0) 0.0)
(num-test (make-polar -0.0 1.0) -0.0)
(num-test (make-polar 0.0 -1.0) 0.0)
(num-test (make-polar -0.0 -1.0) 0.0)
(num-test (make-rectangular 0.0 our-pi) 0.0+3.14159265358979i)
(num-test (make-rectangular -0.0 our-pi) -0.0+3.14159265358979i)
(num-test (make-rectangular 0.0 -3.14159265358979) 0.0-3.14159265358979i)
(num-test (make-rectangular -0.0 -3.14159265358979) -0.0-3.14159265358979i)
(num-test (make-polar 0.0 our-pi) -0.0)
(num-test (make-polar -0.0 our-pi) 0.0)
(num-test (make-polar 0.0 -3.14159265358979) 0.0)
(num-test (make-polar -0.0 -3.14159265358979) 0.0)
(num-test (make-rectangular 0.0 2.71828182845905) 0.0+2.71828182845905i)
(num-test (make-rectangular -0.0 2.71828182845905) -0.0+2.71828182845905i)
(num-test (make-rectangular 0.0 -2.71828182845905) 0.0-2.71828182845905i)
(num-test (make-rectangular -0.0 -2.71828182845905) -0.0-2.71828182845905i)
(num-test (make-polar 0.0 2.71828182845905) -0.0)
(num-test (make-polar -0.0 2.71828182845905) 0.0)
(num-test (make-polar 0.0 -2.71828182845905) 0.0)
(num-test (make-polar -0.0 -2.71828182845905) 0.0)
(num-test (make-rectangular 0.0 1234.0) 0.0+1234.0i)
(num-test (make-rectangular -0.0 1234.0) -0.0+1234.0i)
(num-test (make-rectangular 0.0 -1234.0) 0.0-1234.0i)
(num-test (make-rectangular -0.0 -1234.0) -0.0-1234.0i)
(num-test (make-polar 0.0 1234.0) -0.0)
(num-test (make-polar -0.0 1234.0) 0.0)
(num-test (make-polar 0.0 -1234.0) 0.0)
(num-test (make-polar -0.0 -1234.0) 0.0)
(num-test (make-rectangular 0.0 1234000000.0) 0.0+1234000000.0i)
(num-test (make-rectangular -0.0 1234000000.0) -0.0+1234000000.0i)
(num-test (make-rectangular 0.0 -1234000000.0) 0.0-1234000000.0i)
(num-test (make-rectangular -0.0 -1234000000.0) -0.0-1234000000.0i)
(num-test (make-polar 0.0 1234000000.0) 0.0)
(num-test (make-polar -0.0 1234000000.0) 0.0)
(num-test (make-polar 0.0 -1234000000.0) 0.0)
(num-test (make-polar -0.0 -1234000000.0) -0.0)
(num-test (make-rectangular 0.00000001 0.00000001) 0.00000001+0.00000001i)
(num-test (make-rectangular -0.00000001 0.00000001) -0.00000001+0.00000001i)
(num-test (make-rectangular 0.00000001 -0.00000001) 0.00000001-0.00000001i)
(num-test (make-rectangular -0.00000001 -0.00000001) -0.00000001-0.00000001i)
(num-test (make-polar 0.00000001 0.00000001) 0.00000001+1e-16i)
(num-test (make-polar -0.00000001 0.00000001) -0.00000001-1e-16i)
(num-test (make-polar 0.00000001 -0.00000001) 0.00000001-1e-16i)
(num-test (make-polar -0.00000001 -0.00000001) -0.00000001+1e-16i)
(num-test (make-rectangular 0.00000001 1.0) 0.00000001+1.0i)
(num-test (make-rectangular -0.00000001 1.0) -0.00000001+1.0i)
(num-test (make-rectangular 0.00000001 -1.0) 0.00000001-1.0i)
(num-test (make-rectangular -0.00000001 -1.0) -0.00000001-1.0i)
(num-test (make-polar 0.00000001 1.0) 0.00000000540302+0.00000000841471i)
(num-test (make-polar -0.00000001 1.0) -0.00000000540302-0.00000000841471i)
(num-test (make-polar 0.00000001 -1.0) 0.00000000540302-0.00000000841471i)
(num-test (make-polar -0.00000001 -1.0) -0.00000000540302+0.00000000841471i)
(num-test (make-rectangular 0.00000001 our-pi) 0.00000001+3.14159265358979i)
(num-test (make-rectangular -0.00000001 our-pi) -0.00000001+3.14159265358979i)
(num-test (make-rectangular 0.00000001 -3.14159265358979) 0.00000001-3.14159265358979i)
(num-test (make-rectangular -0.00000001 -3.14159265358979) -0.00000001-3.14159265358979i)
(num-test (make-polar 0.00000001 our-pi) -0.00000001+0.0i)
(num-test (make-polar -0.00000001 our-pi) 0.00000001-0.0i)
(num-test (make-polar 0.00000001 -3.14159265358979) -0.00000001-0.0i)
(num-test (make-polar -0.00000001 -3.14159265358979) 0.00000001+0.0i)
(num-test (make-rectangular 0.00000001 2.71828182845905) 0.00000001+2.71828182845905i)
(num-test (make-rectangular -0.00000001 2.71828182845905) -0.00000001+2.71828182845905i)
(num-test (make-rectangular 0.00000001 -2.71828182845905) 0.00000001-2.71828182845905i)
(num-test (make-rectangular -0.00000001 -2.71828182845905) -0.00000001-2.71828182845905i)
(num-test (make-polar 0.00000001 2.71828182845905) -0.00000000911734+0.00000000410781i)
(num-test (make-polar -0.00000001 2.71828182845905) 0.00000000911734-0.00000000410781i)
(num-test (make-polar 0.00000001 -2.71828182845905) -0.00000000911734-0.00000000410781i)
(num-test (make-polar -0.00000001 -2.71828182845905) 0.00000000911734+0.00000000410781i)
(num-test (make-rectangular 0.00000001 1234.0) 0.00000001+1234.0i)
(num-test (make-rectangular -0.00000001 1234.0) -0.00000001+1234.0i)
(num-test (make-rectangular 0.00000001 -1234.0) 0.00000001-1234.0i)
(num-test (make-rectangular -0.00000001 -1234.0) -0.00000001-1234.0i)
(num-test (make-polar 0.00000001 1234.0) -0.00000000798551+0.00000000601928i)
(num-test (make-polar -0.00000001 1234.0) 0.00000000798551-0.00000000601928i)
(num-test (make-polar 0.00000001 -1234.0) -0.00000000798551-0.00000000601928i)
(num-test (make-polar -0.00000001 -1234.0) 0.00000000798551+0.00000000601928i)
(num-test (make-rectangular 0.00000001 1234000000.0) 0.00000001+1234000000.0i)
(num-test (make-rectangular -0.00000001 1234000000.0) -0.00000001+1234000000.0i)
(num-test (make-rectangular 0.00000001 -1234000000.0) 0.00000001-1234000000.0i)
(num-test (make-rectangular -0.00000001 -1234000000.0) -0.00000001-1234000000.0i)
(num-test (make-polar 0.00000001 1234000000.0) 0.00000000158909-0.00000000987293i)
(num-test (make-polar -0.00000001 1234000000.0) -0.00000000158909+0.00000000987293i)
(num-test (make-polar 0.00000001 -1234000000.0) 0.00000000158909+0.00000000987293i)
(num-test (make-polar -0.00000001 -1234000000.0) -0.00000000158909-0.00000000987293i)
(num-test (make-rectangular 1.0 0.00000001) 1.0+0.00000001i)
(num-test (make-rectangular -1.0 0.00000001) -1.0+0.00000001i)
(num-test (make-rectangular 1.0 -0.00000001) 1.0-0.00000001i)
(num-test (make-rectangular -1.0 -0.00000001) -1.0-0.00000001i)
(num-test (make-polar 1.0 0.00000001) 1.0+0.00000001i)
(num-test (make-polar -1.0 0.00000001) -1.0-0.00000001i)
(num-test (make-polar 1.0 -0.00000001) 1.0-0.00000001i)
(num-test (make-polar -1.0 -0.00000001) -1.0+0.00000001i)
(num-test (make-rectangular 1.0 1.0) 1.0+1.0i)
(num-test (make-rectangular -1.0 1.0) -1.0+1.0i)
(num-test (make-rectangular 1.0 -1.0) 1.0-1.0i)
(num-test (make-rectangular -1.0 -1.0) -1.0-1.0i)
(num-test (make-polar 1.0 1.0) 0.54030230586814+0.84147098480790i)
(num-test (make-polar -1.0 1.0) -0.54030230586814-0.84147098480790i)
(num-test (make-polar 1.0 -1.0) 0.54030230586814-0.84147098480790i)
(num-test (make-polar -1.0 -1.0) -0.54030230586814+0.84147098480790i)
(num-test (make-rectangular 1.0 our-pi) 1.0+3.14159265358979i)
(num-test (make-rectangular -1.0 our-pi) -1.0+3.14159265358979i)
(num-test (make-rectangular 1.0 -3.14159265358979) 1.0-3.14159265358979i)
(num-test (make-rectangular -1.0 -3.14159265358979) -1.0-3.14159265358979i)
(num-test (make-polar 1.0 our-pi) -1.0+0.0i)
(num-test (make-polar -1.0 our-pi) 1.0-0.0i)
(num-test (make-polar 1.0 -3.14159265358979) -1.0-0.0i)
(num-test (make-polar -1.0 -3.14159265358979) 1.0+0.0i)
(num-test (make-rectangular 1.0 2.71828182845905) 1.0+2.71828182845905i)
(num-test (make-rectangular -1.0 2.71828182845905) -1.0+2.71828182845905i)
(num-test (make-rectangular 1.0 -2.71828182845905) 1.0-2.71828182845905i)
(num-test (make-rectangular -1.0 -2.71828182845905) -1.0-2.71828182845905i)
(num-test (make-polar 1.0 2.71828182845905) -0.91173391478697+0.41078129050291i)
(num-test (make-polar -1.0 2.71828182845905) 0.91173391478697-0.41078129050291i)
(num-test (make-polar 1.0 -2.71828182845905) -0.91173391478697-0.41078129050291i)
(num-test (make-polar -1.0 -2.71828182845905) 0.91173391478697+0.41078129050291i)
(num-test (make-rectangular 1.0 1234.0) 1.0+1234.0i)
(num-test (make-rectangular -1.0 1234.0) -1.0+1234.0i)
(num-test (make-rectangular 1.0 -1234.0) 1.0-1234.0i)
(num-test (make-rectangular -1.0 -1234.0) -1.0-1234.0i)
(num-test (make-polar 1.0 1234.0) -0.79855062358758+0.60192765476250i)
(num-test (make-polar -1.0 1234.0) 0.79855062358758-0.60192765476250i)
(num-test (make-polar 1.0 -1234.0) -0.79855062358758-0.60192765476250i)
(num-test (make-polar -1.0 -1234.0) 0.79855062358758+0.60192765476250i)
(num-test (make-rectangular 1.0 1234000000.0) 1.0+1234000000.0i)
(num-test (make-rectangular -1.0 1234000000.0) -1.0+1234000000.0i)
(num-test (make-rectangular 1.0 -1234000000.0) 1.0-1234000000.0i)
(num-test (make-rectangular -1.0 -1234000000.0) -1.0-1234000000.0i)
(num-test (make-polar 1.0 1234000000.0) 0.15890913095152-0.98729321283003i)
(num-test (make-polar -1.0 1234000000.0) -0.15890913095152+0.98729321283003i)
(num-test (make-polar 1.0 -1234000000.0) 0.15890913095152+0.98729321283003i)
(num-test (make-polar -1.0 -1234000000.0) -0.15890913095152-0.98729321283003i)
(num-test (make-rectangular 3.14159265358979 0.00000001) 3.14159265358979+0.00000001i)
(num-test (make-rectangular -3.14159265358979 0.00000001) -3.14159265358979+0.00000001i)
(num-test (make-rectangular 3.14159265358979 -0.00000001) 3.14159265358979-0.00000001i)
(num-test (make-rectangular -3.14159265358979 -0.00000001) -3.14159265358979-0.00000001i)
(num-test (make-polar 3.14159265358979 0.00000001) 3.14159265358979+0.00000003141593i)
(num-test (make-polar -3.14159265358979 0.00000001) -3.14159265358979-0.00000003141593i)
(num-test (make-polar 3.14159265358979 -0.00000001) 3.14159265358979-0.00000003141593i)
(num-test (make-polar -3.14159265358979 -0.00000001) -3.14159265358979+0.00000003141593i)
(num-test (make-rectangular 3.14159265358979 1.0) 3.14159265358979+1.0i)
(num-test (make-rectangular -3.14159265358979 1.0) -3.14159265358979+1.0i)
(num-test (make-rectangular 3.14159265358979 -1.0) 3.14159265358979-1.0i)
(num-test (make-rectangular -3.14159265358979 -1.0) -3.14159265358979-1.0i)
(num-test (make-polar 3.14159265358979 1.0) 1.69740975483297+2.64355906408146i)
(num-test (make-polar -3.14159265358979 1.0) -1.69740975483297-2.64355906408146i)
(num-test (make-polar 3.14159265358979 -1.0) 1.69740975483297-2.64355906408146i)
(num-test (make-polar -3.14159265358979 -1.0) -1.69740975483297+2.64355906408146i)
(num-test (make-rectangular 3.14159265358979 our-pi) 3.14159265358979+3.14159265358979i)
(num-test (make-rectangular -3.14159265358979 our-pi) -3.14159265358979+3.14159265358979i)
(num-test (make-rectangular 3.14159265358979 -3.14159265358979) 3.14159265358979-3.14159265358979i)
(num-test (make-rectangular -3.14159265358979 -3.14159265358979) -3.14159265358979-3.14159265358979i)
(num-test (make-polar 3.14159265358979 our-pi) -3.14159265358979+0.0i)
(num-test (make-polar -3.14159265358979 our-pi) 3.14159265358979-0.0i)
(num-test (make-polar 3.14159265358979 -3.14159265358979) -3.14159265358979-0.0i)
(num-test (make-polar -3.14159265358979 -3.14159265358979) 3.14159265358979+0.0i)
(num-test (make-rectangular 3.14159265358979 2.71828182845905) 3.14159265358979+2.71828182845905i)
(num-test (make-rectangular -3.14159265358979 2.71828182845905) -3.14159265358979+2.71828182845905i)
(num-test (make-rectangular 3.14159265358979 -2.71828182845905) 3.14159265358979-2.71828182845905i)
(num-test (make-rectangular -3.14159265358979 -2.71828182845905) -3.14159265358979-2.71828182845905i)
(num-test (make-polar 3.14159265358979 2.71828182845905) -2.86429656872339+1.29050748447607i)
(num-test (make-polar -3.14159265358979 2.71828182845905) 2.86429656872339-1.29050748447607i)
(num-test (make-polar 3.14159265358979 -2.71828182845905) -2.86429656872339-1.29050748447607i)
(num-test (make-polar -3.14159265358979 -2.71828182845905) 2.86429656872339+1.29050748447607i)
(num-test (make-rectangular 3.14159265358979 1234.0) 3.14159265358979+1234.0i)
(num-test (make-rectangular -3.14159265358979 1234.0) -3.14159265358979+1234.0i)
(num-test (make-rectangular 3.14159265358979 -1234.0) 3.14159265358979-1234.0i)
(num-test (make-rectangular -3.14159265358979 -1234.0) -3.14159265358979-1234.0i)
(num-test (make-polar 3.14159265358979 1234.0) -2.50872077258230+1.89101149819439i)
(num-test (make-polar -3.14159265358979 1234.0) 2.50872077258230-1.89101149819439i)
(num-test (make-polar 3.14159265358979 -1234.0) -2.50872077258230-1.89101149819439i)
(num-test (make-polar -3.14159265358979 -1234.0) 2.50872077258230+1.89101149819439i)
(num-test (make-rectangular 3.14159265358979 1234000000.0) 3.14159265358979+1234000000.0i)
(num-test (make-rectangular -3.14159265358979 1234000000.0) -3.14159265358979+1234000000.0i)
(num-test (make-rectangular 3.14159265358979 -1234000000.0) 3.14159265358979-1234000000.0i)
(num-test (make-rectangular -3.14159265358979 -1234000000.0) -3.14159265358979-1234000000.0i)
(num-test (make-polar 3.14159265358979 1234000000.0) 0.49922775838562-3.10167310436587i)
(num-test (make-polar -3.14159265358979 1234000000.0) -0.49922775838562+3.10167310436587i)
(num-test (make-polar 3.14159265358979 -1234000000.0) 0.49922775838562+3.10167310436587i)
(num-test (make-polar -3.14159265358979 -1234000000.0) -0.49922775838562-3.10167310436587i)
(num-test (make-rectangular 2.71828182845905 0.00000001) 2.71828182845905+0.00000001i)
(num-test (make-rectangular -2.71828182845905 0.00000001) -2.71828182845905+0.00000001i)
(num-test (make-rectangular 2.71828182845905 -0.00000001) 2.71828182845905-0.00000001i)
(num-test (make-rectangular -2.71828182845905 -0.00000001) -2.71828182845905-0.00000001i)
(num-test (make-polar 2.71828182845905 0.00000001) 2.71828182845905+0.00000002718282i)
(num-test (make-polar -2.71828182845905 0.00000001) -2.71828182845905-0.00000002718282i)
(num-test (make-polar 2.71828182845905 -0.00000001) 2.71828182845905-0.00000002718282i)
(num-test (make-polar -2.71828182845905 -0.00000001) -2.71828182845905+0.00000002718282i)
(num-test (make-rectangular 2.71828182845905 1.0) 2.71828182845905+1.0i)
(num-test (make-rectangular -2.71828182845905 1.0) -2.71828182845905+1.0i)
(num-test (make-rectangular 2.71828182845905 -1.0) 2.71828182845905-1.0i)
(num-test (make-rectangular -2.71828182845905 -1.0) -2.71828182845905-1.0i)
(num-test (make-polar 2.71828182845905 1.0) 1.46869393991589+2.28735528717884i)
(num-test (make-polar -2.71828182845905 1.0) -1.46869393991589-2.28735528717884i)
(num-test (make-polar 2.71828182845905 -1.0) 1.46869393991589-2.28735528717884i)
(num-test (make-polar -2.71828182845905 -1.0) -1.46869393991589+2.28735528717884i)
(num-test (make-rectangular 2.71828182845905 our-pi) 2.71828182845905+3.14159265358979i)
(num-test (make-rectangular -2.71828182845905 our-pi) -2.71828182845905+3.14159265358979i)
(num-test (make-rectangular 2.71828182845905 -3.14159265358979) 2.71828182845905-3.14159265358979i)
(num-test (make-rectangular -2.71828182845905 -3.14159265358979) -2.71828182845905-3.14159265358979i)
(num-test (make-polar 2.71828182845905 our-pi) -2.71828182845905+0.0i)
(num-test (make-polar -2.71828182845905 our-pi) 2.71828182845905-0.0i)
(num-test (make-polar 2.71828182845905 -3.14159265358979) -2.71828182845905-0.0i)
(num-test (make-polar -2.71828182845905 -3.14159265358979) 2.71828182845905+0.0i)
(num-test (make-rectangular 2.71828182845905 2.71828182845905) 2.71828182845905+2.71828182845905i)
(num-test (make-rectangular -2.71828182845905 2.71828182845905) -2.71828182845905+2.71828182845905i)
(num-test (make-rectangular 2.71828182845905 -2.71828182845905) 2.71828182845905-2.71828182845905i)
(num-test (make-rectangular -2.71828182845905 -2.71828182845905) -2.71828182845905-2.71828182845905i)
(num-test (make-polar 2.71828182845905 2.71828182845905) -2.47834973295523+1.11661931744501i)
(num-test (make-polar -2.71828182845905 2.71828182845905) 2.47834973295523-1.11661931744501i)
(num-test (make-polar 2.71828182845905 -2.71828182845905) -2.47834973295523-1.11661931744501i)
(num-test (make-polar -2.71828182845905 -2.71828182845905) 2.47834973295523+1.11661931744501i)
(num-test (make-rectangular 2.71828182845905 1234.0) 2.71828182845905+1234.0i)
(num-test (make-rectangular -2.71828182845905 1234.0) -2.71828182845905+1234.0i)
(num-test (make-rectangular 2.71828182845905 -1234.0) 2.71828182845905-1234.0i)
(num-test (make-rectangular -2.71828182845905 -1234.0) -2.71828182845905-1234.0i)
(num-test (make-polar 2.71828182845905 1234.0) -2.17068564920277+1.63620900598787i)
(num-test (make-polar -2.71828182845905 1234.0) 2.17068564920277-1.63620900598787i)
(num-test (make-polar 2.71828182845905 -1234.0) -2.17068564920277-1.63620900598787i)
(num-test (make-polar -2.71828182845905 -1234.0) 2.17068564920277+1.63620900598787i)
(num-test (make-rectangular 2.71828182845905 1234000000.0) 2.71828182845905+1234000000.0i)
(num-test (make-rectangular -2.71828182845905 1234000000.0) -2.71828182845905+1234000000.0i)
(num-test (make-rectangular 2.71828182845905 -1234000000.0) 2.71828182845905-1234000000.0i)
(num-test (make-rectangular -2.71828182845905 -1234000000.0) -2.71828182845905-1234000000.0i)
(num-test (make-polar 2.71828182845905 1234000000.0) 0.43195980304173-2.68374119979681i)
(num-test (make-polar -2.71828182845905 1234000000.0) -0.43195980304173+2.68374119979681i)
(num-test (make-polar 2.71828182845905 -1234000000.0) 0.43195980304173+2.68374119979681i)
(num-test (make-polar -2.71828182845905 -1234000000.0) -0.43195980304173-2.68374119979681i)
(num-test (make-rectangular 1234.0 0.00000001) 1234.0+0.00000001i)
(num-test (make-rectangular -1234.0 0.00000001) -1234.0+0.00000001i)
(num-test (make-rectangular 1234.0 -0.00000001) 1234.0-0.00000001i)
(num-test (make-rectangular -1234.0 -0.00000001) -1234.0-0.00000001i)
(num-test (make-polar 1234.0 0.00000001) 1234.0+0.00001234000000i)
(num-test (make-polar -1234.0 0.00000001) -1234.0-0.00001234000000i)
(num-test (make-polar 1234.0 -0.00000001) 1234.0-0.00001234000000i)
(num-test (make-polar -1234.0 -0.00000001) -1234.0+0.00001234000000i)
(num-test (make-rectangular 1234.0 1.0) 1234.0+1.0i)
(num-test (make-rectangular -1234.0 1.0) -1234.0+1.0i)
(num-test (make-rectangular 1234.0 -1.0) 1234.0-1.0i)
(num-test (make-rectangular -1234.0 -1.0) -1234.0-1.0i)
(num-test (make-polar 1234.0 1.0) 666.73304544128450+1038.37519525294420i)
(num-test (make-polar -1234.0 1.0) -666.73304544128450-1038.37519525294420i)
(num-test (make-polar 1234.0 -1.0) 666.73304544128450-1038.37519525294420i)
(num-test (make-polar -1234.0 -1.0) -666.73304544128450+1038.37519525294420i)
(num-test (make-rectangular 1234.0 our-pi) 1234.0+3.14159265358979i)
(num-test (make-rectangular -1234.0 our-pi) -1234.0+3.14159265358979i)
(num-test (make-rectangular 1234.0 -3.14159265358979) 1234.0-3.14159265358979i)
(num-test (make-rectangular -1234.0 -3.14159265358979) -1234.0-3.14159265358979i)
(num-test (make-polar 1234.0 our-pi) -1234.0+0.00000000000015i)
(num-test (make-polar -1234.0 our-pi) 1234.0-0.00000000000015i)
(num-test (make-polar 1234.0 -3.14159265358979) -1234.0-0.00000000000015i)
(num-test (make-polar -1234.0 -3.14159265358979) 1234.0+0.00000000000015i)
(num-test (make-rectangular 1234.0 2.71828182845905) 1234.0+2.71828182845905i)
(num-test (make-rectangular -1234.0 2.71828182845905) -1234.0+2.71828182845905i)
(num-test (make-rectangular 1234.0 -2.71828182845905) 1234.0-2.71828182845905i)
(num-test (make-rectangular -1234.0 -2.71828182845905) -1234.0-2.71828182845905i)
(num-test (make-polar 1234.0 2.71828182845905) -1125.07965084711486+506.90411248058950i)
(num-test (make-polar -1234.0 2.71828182845905) 1125.07965084711486-506.90411248058950i)
(num-test (make-polar 1234.0 -2.71828182845905) -1125.07965084711486-506.90411248058950i)
(num-test (make-polar -1234.0 -2.71828182845905) 1125.07965084711486+506.90411248058950i)
(num-test (make-rectangular 1234.0 1234.0) 1234.0+1234.0i)
(num-test (make-rectangular -1234.0 1234.0) -1234.0+1234.0i)
(num-test (make-rectangular 1234.0 -1234.0) 1234.0-1234.0i)
(num-test (make-rectangular -1234.0 -1234.0) -1234.0-1234.0i)
(num-test (make-polar 1234.0 1234.0) -985.41146950707912+742.77872597692169i)
(num-test (make-polar -1234.0 1234.0) 985.41146950707912-742.77872597692169i)
(num-test (make-polar 1234.0 -1234.0) -985.41146950707912-742.77872597692169i)
(num-test (make-polar -1234.0 -1234.0) 985.41146950707912+742.77872597692169i)
(num-test (make-rectangular 1234.0 1234000000.0) 1234.0+1234000000.0i)
(num-test (make-rectangular -1234.0 1234000000.0) -1234.0+1234000000.0i)
(num-test (make-rectangular 1234.0 -1234000000.0) 1234.0-1234000000.0i)
(num-test (make-rectangular -1234.0 -1234000000.0) -1234.0-1234000000.0i)
(num-test (make-polar 1234.0 1234000000.0) 196.09386759417183-1218.31982463225131i)
(num-test (make-polar -1234.0 1234000000.0) -196.09386759417183+1218.31982463225131i)
(num-test (make-polar 1234.0 -1234000000.0) 196.09386759417183+1218.31982463225131i)
(num-test (make-polar -1234.0 -1234000000.0) -196.09386759417183-1218.31982463225131i)
(num-test (make-rectangular 1234000000.0 0.00000001) 1234000000.0+0.00000001i)
(num-test (make-rectangular -1234000000.0 0.00000001) -1234000000.0+0.00000001i)
(num-test (make-rectangular 1234000000.0 -0.00000001) 1234000000.0-0.00000001i)
(num-test (make-rectangular -1234000000.0 -0.00000001) -1234000000.0-0.00000001i)
(num-test (make-polar 1234000000.0 0.00000001) 1234000000.0+12.34000000000000i)
(num-test (make-polar -1234000000.0 0.00000001) -1234000000.0-12.34000000000000i)
(num-test (make-polar 1234000000.0 -0.00000001) 1234000000.0-12.34000000000000i)
(num-test (make-polar -1234000000.0 -0.00000001) -1234000000.0+12.34000000000000i)
(num-test (make-rectangular 1234000000.0 1.0) 1234000000.0+1.0i)
(num-test (make-rectangular -1234000000.0 1.0) -1234000000.0+1.0i)
(num-test (make-rectangular 1234000000.0 -1.0) 1234000000.0-1.0i)
(num-test (make-rectangular -1234000000.0 -1.0) -1234000000.0-1.0i)
(num-test (make-polar 1234000000.0 1.0) 666733045.44128441810608+1038375195.25294423103333i)
(num-test (make-polar -1234000000.0 1.0) -666733045.44128441810608-1038375195.25294423103333i)
(num-test (make-polar 1234000000.0 -1.0) 666733045.44128441810608-1038375195.25294423103333i)
(num-test (make-polar -1234000000.0 -1.0) -666733045.44128441810608+1038375195.25294423103333i)
(num-test (make-rectangular 1234000000.0 our-pi) 1234000000.0+3.14159265358979i)
(num-test (make-rectangular -1234000000.0 our-pi) -1234000000.0+3.14159265358979i)
(num-test (make-rectangular 1234000000.0 -3.14159265358979) 1234000000.0-3.14159265358979i)
(num-test (make-rectangular -1234000000.0 -3.14159265358979) -1234000000.0-3.14159265358979i)
(num-test (make-polar 1234000000.0 our-pi) -1234000000.0+0.00000015111642i)
(num-test (make-polar -1234000000.0 our-pi) 1234000000.0-0.00000015111642i)
(num-test (make-polar 1234000000.0 -3.14159265358979) -1234000000.0-0.00000015111642i)
(num-test (make-polar -1234000000.0 -3.14159265358979) 1234000000.0+0.00000015111642i)
(num-test (make-rectangular 1234000000.0 2.71828182845905) 1234000000.0+2.71828182845905i)
(num-test (make-rectangular -1234000000.0 2.71828182845905) -1234000000.0+2.71828182845905i)
(num-test (make-rectangular 1234000000.0 -2.71828182845905) 1234000000.0-2.71828182845905i)
(num-test (make-rectangular -1234000000.0 -2.71828182845905) -1234000000.0-2.71828182845905i)
(num-test (make-polar 1234000000.0 2.71828182845905) -1125079650.84711480140686+506904112.48058950901031i)
(num-test (make-polar -1234000000.0 2.71828182845905) 1125079650.84711480140686-506904112.48058950901031i)
(num-test (make-polar 1234000000.0 -2.71828182845905) -1125079650.84711480140686-506904112.48058950901031i)
(num-test (make-polar -1234000000.0 -2.71828182845905) 1125079650.84711480140686+506904112.48058950901031i)
(num-test (make-rectangular 1234000000.0 1234.0) 1234000000.0+1234.0i)
(num-test (make-rectangular -1234000000.0 1234.0) -1234000000.0+1234.0i)
(num-test (make-rectangular 1234000000.0 -1234.0) 1234000000.0-1234.0i)
(num-test (make-rectangular -1234000000.0 -1234.0) -1234000000.0-1234.0i)
(num-test (make-polar 1234000000.0 1234.0) -985411469.50707900524139+742778725.97692167758942i)
(num-test (make-polar -1234000000.0 1234.0) 985411469.50707900524139-742778725.97692167758942i)
(num-test (make-polar 1234000000.0 -1234.0) -985411469.50707900524139-742778725.97692167758942i)
(num-test (make-polar -1234000000.0 -1234.0) 985411469.50707900524139+742778725.97692167758942i)
(num-test (make-rectangular 1234000000.0 1234000000.0) 1234000000.0+1234000000.0i)
(num-test (make-rectangular -1234000000.0 1234000000.0) -1234000000.0+1234000000.0i)
(num-test (make-rectangular 1234000000.0 -1234000000.0) 1234000000.0-1234000000.0i)
(num-test (make-rectangular -1234000000.0 -1234000000.0) -1234000000.0-1234000000.0i)
(num-test (make-polar 1234000000.0 1234000000.0) 196093867.59417182207108-1218319824.63225126266479i)
(num-test (make-polar -1234000000.0 1234000000.0) -196093867.59417182207108+1218319824.63225126266479i)
(num-test (make-polar 1234000000.0 -1234000000.0) 196093867.59417182207108+1218319824.63225126266479i)
(num-test (make-polar -1234000000.0 -1234000000.0) -196093867.59417182207108-1218319824.63225126266479i)
(num-test (make-polar 0 0) 0)
(num-test (make-polar 0.0 0.0) 0.0)
(num-test (make-rectangular 0 0) 0)
(num-test (make-rectangular 0.0 0.0) 0.0)


;; -------- angle
(num-test (angle 1) 0)
(num-test (angle -1) our-pi)
(num-test (angle 1/1) 0)
(num-test (angle -1/1) our-pi)
(num-test (angle 1/2) 0)
(num-test (angle -1/2) our-pi)
(num-test (angle 1/3) 0)
(num-test (angle -1/3) our-pi)
(num-test (angle 1/10) 0)
(num-test (angle -1/10) our-pi)
(num-test (angle 1/1234) 0)
(num-test (angle -1/1234) our-pi)
(num-test (angle 1/1234000000) 0)
(num-test (angle -1/1234000000) our-pi)
(num-test (angle 1/500029) 0)
(num-test (angle -1/500029) our-pi)
(num-test (angle 1/362880) 0)
(num-test (angle -1/362880) our-pi)
(num-test (angle 2) 0)
(num-test (angle -2) our-pi)
(num-test (angle 2/1) 0)
(num-test (angle -2/1) our-pi)
(num-test (angle 2/2) 0)
(num-test (angle -2/2) our-pi)
(num-test (angle 2/3) 0)
(num-test (angle -2/3) our-pi)
(num-test (angle 2/10) 0)
(num-test (angle -2/10) our-pi)
(num-test (angle 2/1234) 0)
(num-test (angle -2/1234) our-pi)
(num-test (angle 2/1234000000) 0)
(num-test (angle -2/1234000000) our-pi)
(num-test (angle 2/500029) 0)
(num-test (angle -2/500029) our-pi)
(num-test (angle 2/362880) 0)
(num-test (angle -2/362880) our-pi)
(num-test (angle 3) 0)
(num-test (angle -3) our-pi)
(num-test (angle 3/1) 0)
(num-test (angle -3/1) our-pi)
(num-test (angle 3/2) 0)
(num-test (angle -3/2) our-pi)
(num-test (angle 3/3) 0)
(num-test (angle -3/3) our-pi)
(num-test (angle 3/10) 0)
(num-test (angle -3/10) our-pi)
(num-test (angle 3/1234) 0)
(num-test (angle -3/1234) our-pi)
(num-test (angle 3/1234000000) 0)
(num-test (angle -3/1234000000) our-pi)
(num-test (angle 3/500029) 0)
(num-test (angle -3/500029) our-pi)
(num-test (angle 3/362880) 0)
(num-test (angle -3/362880) our-pi)
(num-test (angle 10) 0)
(num-test (angle -10) our-pi)
(num-test (angle 10/1) 0)
(num-test (angle -10/1) our-pi)
(num-test (angle 10/2) 0)
(num-test (angle -10/2) our-pi)
(num-test (angle 10/3) 0)
(num-test (angle -10/3) our-pi)
(num-test (angle 10/10) 0)
(num-test (angle -10/10) our-pi)
(num-test (angle 10/1234) 0)
(num-test (angle -10/1234) our-pi)
(num-test (angle 10/1234000000) 0)
(num-test (angle -10/1234000000) our-pi)
(num-test (angle 10/500029) 0)
(num-test (angle -10/500029) our-pi)
(num-test (angle 10/362880) 0)
(num-test (angle -10/362880) our-pi)
(num-test (angle 1234) 0)
(num-test (angle -1234) our-pi)
(num-test (angle 1234/1) 0)
(num-test (angle -1234/1) our-pi)
(num-test (angle 1234/2) 0)
(num-test (angle -1234/2) our-pi)
(num-test (angle 1234/3) 0)
(num-test (angle -1234/3) our-pi)
(num-test (angle 1234/10) 0)
(num-test (angle -1234/10) our-pi)
(num-test (angle 1234/1234) 0)
(num-test (angle -1234/1234) our-pi)
(num-test (angle 1234/1234000000) 0)
(num-test (angle -1234/1234000000) our-pi)
(num-test (angle 1234/500029) 0)
(num-test (angle -1234/500029) our-pi)
(num-test (angle 1234/362880) 0)
(num-test (angle -1234/362880) our-pi)
(num-test (angle 1234000000) 0)
(num-test (angle -1234000000) our-pi)
(num-test (angle 1234000000/1) 0)
(num-test (angle -1234000000/1) our-pi)
(num-test (angle 1234000000/2) 0)
(num-test (angle -1234000000/2) our-pi)
(num-test (angle 1234000000/3) 0)
(num-test (angle -1234000000/3) our-pi)
(num-test (angle 1234000000/10) 0)
(num-test (angle -1234000000/10) our-pi)
(num-test (angle 1234000000/1234) 0)
(num-test (angle -1234000000/1234) our-pi)
(num-test (angle 1234000000/1234000000) 0)
(num-test (angle -1234000000/1234000000) our-pi)
(num-test (angle 1234000000/500029) 0)
(num-test (angle -1234000000/500029) our-pi)
(num-test (angle 1234000000/362880) 0)
(num-test (angle -1234000000/362880) our-pi)
(num-test (angle 500029) 0)
(num-test (angle -500029) our-pi)
(num-test (angle 500029/1) 0)
(num-test (angle -500029/1) our-pi)
(num-test (angle 500029/2) 0)
(num-test (angle -500029/2) our-pi)
(num-test (angle 500029/3) 0)
(num-test (angle -500029/3) our-pi)
(num-test (angle 500029/10) 0)
(num-test (angle -500029/10) our-pi)
(num-test (angle 500029/1234) 0)
(num-test (angle -500029/1234) our-pi)
(num-test (angle 500029/1234000000) 0)
(num-test (angle -500029/1234000000) our-pi)
(num-test (angle 500029/500029) 0)
(num-test (angle -500029/500029) our-pi)
(num-test (angle 500029/362880) 0)
(num-test (angle -500029/362880) our-pi)
(num-test (angle 362880) 0)
(num-test (angle -362880) our-pi)
(num-test (angle 362880/1) 0)
(num-test (angle -362880/1) our-pi)
(num-test (angle 362880/2) 0)
(num-test (angle -362880/2) our-pi)
(num-test (angle 362880/3) 0)
(num-test (angle -362880/3) our-pi)
(num-test (angle 362880/10) 0)
(num-test (angle -362880/10)