/*
 * BufferReader.cc -- Buffered Input
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <iostream>

#include "BufferReader.h"

/** Create a buffer reader of the specified size (in bytes). Note that this
	class is abstract and doesn't do anything without the Read method being
	implemented.
*/

BufferReader::BufferReader( int _size ) :
	size( _size ),
	used( 0 )
{
	buffer = new uint8_t[ _size ];
	pthread_mutex_init( &mutex, NULL );
}

/** Destructor for the read buffer.
*/

BufferReader::~BufferReader( )
{
	delete buffer;
	pthread_mutex_destroy( &mutex );
}

/** Read n bytes from the buffer.
*/

int BufferReader::GetBuffer( uint8_t *data, int length )
{
	int required = length;
	int error = 0;
	
	pthread_mutex_lock( &mutex );

	do	
	{
		// Make sure the buffer is kept full
		if ( used != size )
		{
			int bytes = Read( buffer + used, size - used );
			if ( bytes > 0 )
				used += bytes;
			else
				error = 1;
		}
		
		// Take what's available
		if ( data != NULL )
		{
			if ( required < used )
			{
				memcpy( data, buffer, required );
				memmove( buffer, buffer + required, used - required );
				used -= required;
				required = 0;
			}
			else
			{
				memcpy( data, buffer, used );
				required -= used;
				used = 0;
			}
		}
	}
	while( !error && required != 0 );

	pthread_mutex_unlock( &mutex );
	
	return length - required;
}

/** Get a single byte from the buffer.
*/

int BufferReader::GetBuffer( uint8_t &byte )
{
	return GetBuffer( &byte, 1 );
}

/** Get an uint32_t from the buffer.
*/

int BufferReader::GetBuffer( uint32_t &value )
{
	uint8_t data[ 4 ];
	int error = GetBuffer( data, 4 );
	value = data[ 0 ] | ( data[ 1 ] << 8 ) | ( data[ 2 ] << 16 ) | ( data[ 3 ] << 24 );
	return error;
}

/** Get a short value from the buffer.
*/

int BufferReader::GetBuffer( int16_t &value )
{
	uint8_t data[ 2 ];
	int error = GetBuffer( data, 2 );
	value = data[ 0 ] | ( data[ 1 ] << 8 );
	return error;
}

/** Get an array of shorts from the buffer.
*/

int BufferReader::GetBuffer( int16_t *values, int length )
{
	int index = 0;
	uint8_t *ptr = buffer;
	int error = GetBuffer( buffer, length * 2 );
	for ( index = 0; index < length; ptr += 2 )
		values[ index ++ ] = *ptr | ( *( ptr + 1 ) << 8 );
	return error;
}
