/***************************************************************************
    smb4kbookmark  -  This is the bookmark container for Smb4K (next
    generation).
                             -------------------
    begin                : So Jun 8 2008
    copyright            : (C) 2008 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KBOOKMARK_H
#define SMB4KBOOKMARK_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <QString>

// KDE includes
#include <kdemacros.h>

// forward declarations
class Smb4KShare;

/**
 * This is the container class for bookmarks in Smb4K. It is a complete
 * rewrite of the previous class and comes with several improvements.
 *
 * @author Alexander Reinholdt <dustpuppy@users.berlios.de>
 */

class KDE_EXPORT Smb4KBookmark
{
  public:
    /**
     * The constructor.
     *
     * @param share           The share for which the bookmark should
     *                        be created.
     *
     * @param label           The optional bookmark label.
     */
    Smb4KBookmark( Smb4KShare *share,
                   const QString &label = QString() );

    /**
     * The copy constructor.
     *
     * @param bookmark        The bookmark that should be copied.
     */
    Smb4KBookmark( const Smb4KBookmark &bookmark );

    /**
     * The empty constructor.
     */
    Smb4KBookmark();

    /**
     * The destructor
     */
    ~Smb4KBookmark();

    /**
     * Set the workgroup name.
     *
     * @param workgroup       The workgroup where the share is located.
     */
    void setWorkgroup( const QString &workgroup );

    /**
     * Returns the workgroup/domain name.
     *
     * @returns the workgroup/domain name.
     */
    const QString &workgroup() const { return m_workgroup; }

    /**
     * Set the host name.
     *
     * Note: This function will also rewrite the UNC, so that it will
     * look like this: //NEW_HOST/SHARE. If there was a user defined in the
     * UNC, it will be lost. Use the setUNC() function when you want to get
     * sure nothing gets overwritten.
     *
     * @param host            The host where the share is located.
     */
    void setHost( const QString &host );

    /**
     * Returns the host name.
     *
     * @returns the host name.
     */
    const QString &host() const { return m_host; }

    /**
     * Set the share name.
     *
     * Note: This function will also rewrite the UNC, so that it will
     * look like this: //HOST/NEW_SHARE. If there was a user defined in the
     * UNC, it will be lost. Use the setUNC() function when you want to get
     * sure nothing gets overwritten.
     *
     * @param share           The share name
     */
    void setShare( const QString &share );

    /**
     * Returns the share name.
     *
     * @returns the share name.
     */
    const QString &share() const { return m_share; }

    /**
     * Set the host's IP address.
     *
     * @param ip              The host's IP address
     */
    void setHostIP( const QString &ip );

    /**
     * Returns the host's IP address.
     *
     * @returns the host's IP address.
     */
    const QString &hostIP() const { return m_ip; }

    /**
     * Set the share's type.
     *
     * @param type            The type of the share.
     */
    void setType( const QString &type );

    /**
     * Returns the share's type.
     *
     * @returns the type of the share.
     */
    const QString &type() const { return m_type; }

    /**
     * Set the UNC. The host and the share will also be set. Always use
     * this function if you need to store a UNC that contains a user.
     *
     * @param unc             The UNC
     */
    void setUNC( const QString &unc );

    /**
     * Returns the UNC of the bookmark.
     *
     * @returns the UNC.
     */
    const QString &unc() const { return m_unc; }

    /**
     * Set the (optional) the bookmark's label.
     *
     * @param label           The bookmark's label
     */
    void setLabel( const QString &label );

    /**
     * Returns the bookmark's label.
     *
     * @returns the bookmark's label.
     */
    const QString &label() const { return m_label; }

    /**
     * Set the profile for which this custom options are to be used.
     *
     * @param name              The name of the profile.
     */
    void setProfile( const QString &name );

    /**
     * Return the name of the profile for which the options are to be
     * used.
     *
     * @returns the name of the profile for which the options should be
     * used.
     */
    const QString &profile() const { return m_profile; }

  private:
    /**
     * Workgroup
     */
    QString m_workgroup;

    /**
     * Host
     */
    QString m_host;

    /**
     * Share name
     */
    QString m_share;

    /**
     * Host IP address
     */
    QString m_ip;

    /**
     * Type
     */
    QString m_type;

    /**
     * UNC
     */
    QString m_unc;

    /**
     * Label
     */
    QString m_label;

    /**
     * Profile
     */
    QString m_profile;

    /**
     * This function checks if the given IP address is either
     * compatible with IPv4 or IPv6. If it is not, an empty string
     * is returned.
     *
     * @param ip              The IP address that needs to be checked.
     *
     * @returns the IP address or an empty string if the IP address
     * is not compatible with either IPv4 or IPv6.
     */
    const QString &ipIsValid( const QString &ip );
};

#endif
