package SliMP3::History;

# SliMP3 Server Copyright (C) 2001 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License, 
# version 2.

use strict;
use File::Spec::Functions qw(:ALL);
use POSIX;
use SliMP3::Misc;

use SliMP3::Strings qw(string);

# a cache of songs played
my @history = ();

sub get_history {
	return @history;
}

# Clear can be used to clear the history cache.
sub clear {
	@history =();
	unlink(catfile(SliMP3::Prefs::get('playlistdir'),'__history.m3u'));
	undef;
};

# recount processes the history list by counting the number of times a song appears, and then
# sorting the list based on that number.  The sorted list is returned.  The returned list 
# will be a two dimensional array (N by 2).  Each element contains the song name, and the number
# of times it appeared in the history.
sub recount {
	my @outlist;
	my $listcount = 0;
	my $i=1;
	my $item;
	my $title;
	my $count;
	
	@outlist = ();
	
    if (scalar(@history)) {        
        
	    # Cycle through the history and count songs. 
        my %outhash;
	foreach $item (@history) {
		if (!exists($outhash{$item})) {
			$outhash{$item}[0] = $item;
			$outhash{$item}[2] = $listcount;
		}
		$outhash{$item}[1]++;
		$listcount++;
	}
        # Sort array by song count descending and (for ties) last played ascending
	@outlist = sort {$b->[1] <=> $a->[1] || $a->[2] <=> $b->[2]} values %outhash;
    	#return @outlist[sort {$outlist[$b][1] <=> $outlist[$a][1] || $outlist[$a][2] <=> $outlist[$b][2]} 0..$#outlist];
    }
    return @outlist;
};

#
#	load the track history from an M3U file
#	(don't worry if the file doesn't exist)
#
sub load {
    @history = ();
    return undef unless SliMP3::Prefs::get('savehistory');

    my $filename = catfile(SliMP3::Prefs::get('playlistdir'),'__history.m3u');

    open (FILE,$filename) or return undef;
    @history = SliMP3::Parse::M3U(
	\*FILE,
	SliMP3::Prefs::get('mp3dir'),
    );
    close FILE;
    undef;
}

# Record takes a song name and stores it at the first position of an array.  The max 
sub record {
    my $song = shift;
    return unless SliMP3::Prefs::get('historylength');
    # Add the newest song to the font of the list, so that the most recent song is at the top.
    unshift @history,$song;
    
    if (scalar(@history) > SliMP3::Prefs::get('historylength')) { pop @history; }
    if (SliMP3::Prefs::get('savehistory') && SliMP3::Prefs::get('playlistdir')) {
	SliMP3::Parse::writeM3U(
	    \@history,
	    catfile(SliMP3::Prefs::get('playlistdir'),'__history.m3u'),
	);
    }
};

#shrink history array if historylength is modified to be smaller than the current history array
sub adjustHistoryLength {
	my $newlen = SliMP3::Prefs::get('historylength');
	if (!$newlen) {
		clear();
	} elsif ($newlen < scalar(@history)) {
		splice @history,$newlen;
		if (SliMP3::Prefs::get('savehistory') && SliMP3::Prefs::get('playlistdir')) {
			SliMP3::Parse::writeM3U(\@history,catfile(SliMP3::Prefs::get('playlistdir'),'__history.m3u'));
		}
	}
}


# Histlist fills variables for populating an html file. 
sub hitlist {
	my($client, $paramref) = @_;
	my $output = "";
	my @items = ();
	my %list_form;
	my $i;
	my $song;
	my $itemnumber=0;
	my $maxplayed=0;

	@items = recount();
	if (scalar(@items)) {
		for ($i = 0; $i < scalar(@items); $i++) {
		    if ($maxplayed==0) { $maxplayed = $items[$i][1]; }
    		%list_form = ();
            $song = $items[$i][0];

			$list_form{'title'} 	= SliMP3::Info::standardTitle(undef,$song);
			$list_form{'artist'} 	= SliMP3::Info::artist($song);
			$list_form{'album'} 	= SliMP3::Info::album($song);
			$list_form{'itempath'} = $song;
			$list_form{'odd'}	  = ($itemnumber + 1) % 2;
			$list_form{'song_bar'} = hitlist_bar( $items[$i][1], $maxplayed );
			$list_form{'player'} = $$paramref{'player'};
			$itemnumber++;

			$list_form{'skinOverride'} = $$paramref{'skinOverride'};
			$list_form{'song_count'} = $items[$i][1];
			$$paramref{'browse_list'} .= &SliMP3::HTTP::filltemplatefile("hitlist_list.html", \%list_form);
		}
	}

	$$paramref{'total_song_count'} = SliMP3::Info::songCount([],[],[],[]);
	$$paramref{'genre_count'} = SliMP3::Info::genreCount([],[],[],[]);
	$$paramref{'artist_count'} = SliMP3::Info::artistCount([],[],[],[]);
	$$paramref{'album_count'} = SliMP3::Info::albumCount([],[],[],[]);

	$output .= &SliMP3::HTTP::filltemplatefile("hitlist.html", $paramref);
}

sub hitlist_bar {
    my $curr = shift;
    my $max = shift;
    my $returnval="";
    
    my %list_form=();
    
    $list_form{'cell_full'} = (($curr*100)/$max) > 9;
    $returnval .= &SliMP3::HTTP::filltemplatefile("hitlist_bar.html", \%list_form);
    $list_form{'cell_full'} = (($curr*100)/$max) > 19;
    $returnval .= &SliMP3::HTTP::filltemplatefile("hitlist_bar.html", \%list_form);
    $list_form{'cell_full'} = (($curr*100)/$max) > 29;
    $returnval .= &SliMP3::HTTP::filltemplatefile("hitlist_bar.html", \%list_form);
    $list_form{'cell_full'} = (($curr*100)/$max) > 39;
    $returnval .= &SliMP3::HTTP::filltemplatefile("hitlist_bar.html", \%list_form);
    $list_form{'cell_full'} = (($curr*100)/$max) > 49;
    $returnval .= &SliMP3::HTTP::filltemplatefile("hitlist_bar.html", \%list_form);
    $list_form{'cell_full'} = (($curr*100)/$max) > 59;
    $returnval .= &SliMP3::HTTP::filltemplatefile("hitlist_bar.html", \%list_form);
    $list_form{'cell_full'} = (($curr*100)/$max) > 69;
    $returnval .= &SliMP3::HTTP::filltemplatefile("hitlist_bar.html", \%list_form);
    $list_form{'cell_full'} = (($curr*100)/$max) > 79;
    $returnval .= &SliMP3::HTTP::filltemplatefile("hitlist_bar.html", \%list_form);
    $list_form{'cell_full'} = (($curr*100)/$max) > 89;
    $returnval .= &SliMP3::HTTP::filltemplatefile("hitlist_bar.html", \%list_form);
    $list_form{'cell_full'} = ($curr == $max);
    $returnval .= &SliMP3::HTTP::filltemplatefile("hitlist_bar.html", \%list_form);
    
    return $returnval;
}



	
1;
	
